import json
import logging
import sys
from abc import ABCMeta
from abc import abstractmethod
from threading import Thread

import pandas as pd
import six
from six import add_metaclass

from dataiku.base.socket_block_link import JavaLink
from dataiku.core import schema_handling
from dataiku.core.dataset import CSV_DOUBLE_QUOTE
from dataiku.core.dataset import CSV_QUOTE_CHAR
from dataiku.core.dataset import CSV_SEP
from dataiku.core.dataset import Dataset
from dataiku.core.dataset_write import AbstractStreamingAPI
from dataiku.core.dataset_write import RemoteStreamWriter
from dataiku.core.dataset_write import write_dataframe
from dataiku.core.intercom import jek_or_backend_json_call

logger = logging.getLogger(__name__)


def prepare_dataframe(
        df,
        steps,
        requested_output_schema,
        context_project_key,
        requested_input_schema=None,
        columns=None,
        parse_dates=True,
        infer_with_pandas=True,
        bool_as_str=False,
        use_nullable_integers=False,
        float_precision=None,
        na_values=None,
        skip_blank_lines=True,
        keep_default_na=True):
    """
    :param pd.DataFrame df: input dataframe to process
    :param dict steps:
    :param dict requested_output_schema:
    :param str context_project_key:
    :param dict or None requested_input_schema:

    :param list[dict] or None columns:
    :param bool parse_dates:
    :param bool infer_with_pandas:
    :param bool bool_as_str:
    :param use_nullable_integers:

    :param float_precision:
    :param na_values:
    :param bool skip_blank_lines:
    :param bool keep_default_na:

    :return:
    :rtype: pd.DataFrame
    """
    if not hasattr(df, "to_csv"):
        raise ValueError("Method write_from_dataframe expects a "
                         "dataframe as argument. You gave a %s" %
                         (df is None and "None" or df.__class__))

    (names, dtypes, parse_date_columns) = Dataset.get_dataframe_schema_st(
        requested_output_schema["columns"],
        columns,
        parse_dates,
        infer_with_pandas,
        bool_as_str,
        use_nullable_integers=use_nullable_integers)

    if requested_input_schema is None:
        requested_input_schema = {'columns': schema_handling.get_schema_from_df(df)}

    class PandasStreamProcessor(StreamHandle):

        def __init__(self):
            self._df_from_stream = None

        def get_processed_df(self):
            if self._df_from_stream is None:
                raise ValueError("Dataframe hasn't been processed yet")

            return self._df_from_stream

        def process_stream(self, stream_raw_df):
            with stream_raw_df as raw_df:
                self._df_from_stream = pd.read_table(
                    raw_df,
                    names=names,
                    dtype=dtypes,
                    header=None,
                    sep=CSV_SEP,
                    doublequote=CSV_DOUBLE_QUOTE,
                    quotechar=CSV_QUOTE_CHAR,
                    parse_dates=parse_date_columns,
                    float_precision=float_precision,
                    skip_blank_lines=skip_blank_lines,
                    na_values=na_values,
                    keep_default_na=keep_default_na)
            if self._df_from_stream is None:
                raise ValueError("Converting to pandas df failed for some unknown reason, check logs")

    stream_processor = PandasStreamProcessor()

    with PreparationStreamingAPI(stream_processor) as streaming_api:
        with DataframePreparation(steps, requested_input_schema, requested_output_schema,
                                  context_project_key, streaming_api) as processor:
            processor.prepare(df)

    return stream_processor.get_processed_df()


@add_metaclass(ABCMeta)
class StreamHandle(object):
    @abstractmethod
    def process_stream(self, stream):
        pass


class RaisableThread(Thread):

    def __init__(self, target):
        super(RaisableThread, self).__init__()
        self.target = target
        self._exc_info = None

    def run(self):
        try:
            self.target()
        except:  # Error not swallowed, will be reraised later on
            self._exc_info = sys.exc_info()

    def raise_error_if_any(self):
        if self._exc_info is not None:
            six.reraise(*self._exc_info)


class PreparationLinkException(Exception):
    pass


class PreparationStreamingAPI(AbstractStreamingAPI):
    def __init__(self, stream_handle):
        """
        :param StreamHandle stream_handle:
        """
        self.stream_handle = stream_handle
        self.link = None

    def init_session(self, request):
        logger.info("Initializing dataframe preparation streaming")
        java_link_info = jek_or_backend_json_call("datasets/init-preparation-session/",
                                                  data={"request": json.dumps(request)})

        logger.info("Connecting to java link")
        self.link = JavaLink(java_link_info["port"], java_link_info["secret"], java_link_info.get("serverCert", "NA"))
        # initiate connection
        self.link.connect()
        logger.info("Managed to connect to link")
        return java_link_info["secret"]

    def push_data(self, id_, generator):
        logger.info("Start sending data and receive prepared data")

        def send_data_thread_fn():
            logger.info("Start sending data to be prepared")
            with self.link.send_stream() as output_:
                for elem in generator:
                    output_.write(elem)
            logger.info("Done sending data to be prepared")

        def read_prepared_data_thread_fn():
            logger.info("Start receiving prepared data")
            with self.link.read_stream() as input_:
                self.stream_handle.process_stream(input_)
            logger.info("Done receiving prepared data")

        send_data_thread = RaisableThread(target=send_data_thread_fn)
        read_prepared_data_thread = RaisableThread(target=read_prepared_data_thread_fn)

        send_data_thread.start()
        read_prepared_data_thread.start()

        send_data_thread.join()
        read_prepared_data_thread.join()

        done_message = self.link.read_json()
        self.link.send_string("ack")

        if "error" in done_message and "stackTrace" in done_message:
            raise PreparationLinkException("Something went wrong with the"
                                           " preparation '%s'\nwith stack trace: %s" % (done_message["error"],
                                                                                        done_message["stackTrace"]))

        # Raise errors that might have occurred during the streaming (low probability as they would be in the done
        # message most of the time)
        read_prepared_data_thread.raise_error_if_any()
        send_data_thread.raise_error_if_any()

        logger.info("Done sending data and receive prepared data")

    def close(self):
        logger.info("Closing dataframe preparation streaming")
        if self.link is not None:
            self.link.close()

    def __enter__(self, ):
        return self

    def __exit__(self, type_, value, traceback):
        self.close()


class DataframePreparation(object):
    def __init__(self, preparation_steps, requested_input_schema, requested_output_schema,
                 context_project_key, streaming_api):
        self.preparation_steps = preparation_steps
        self.requested_input_schema = requested_input_schema
        self.requested_output_schema = requested_output_schema
        self.context_project_key = context_project_key
        self.streaming_api = streaming_api

        self.remote_writer = None

    def _get_init_data(self):
        return {
            "script": {"steps": self.preparation_steps},
            "requestedInputSchema": self.requested_input_schema,
            "requestedOutputSchema": self.requested_output_schema,
            "contextProjectKey": self.context_project_key,
        }

    def _start_once(self):
        if not self.remote_writer:
            logger.info("Initializing write session")
            id_ = self.streaming_api.init_session(self._get_init_data())
            self.remote_writer = RemoteStreamWriter(id_, self.streaming_api)
        else:
            pass

    def prepare(self, df):
        self._start_once()
        write_dataframe(df, self.remote_writer) # this is used in a pandas df->pandas df conversion, so no need to bother with a dataset schema

    def close(self):
        if not self.remote_writer:
            raise Exception("Remote writer not initialized, something went wrong")
        self.remote_writer.flush()
        self.remote_writer.close()
        self.remote_writer.join()
        if self.remote_writer.error_message:
            raise Exception(self.remote_writer.error_message)

    def __enter__(self, ):
        return self

    def __exit__(self, type_, value, traceback):
        self.close()
