(function(){
    'use strict';

    var app = angular.module('dataiku.common.lists', []);

    app.factory('ListFilter', function() {

        function preprocessQuery(queryString) {
            let regex = queryString.match(/^\/(.+)\/$/);
            return regex ? new RegExp(regex[1], 'i') : queryString.toLowerCase().split(/\s+/)
        }

        function scoreTokens(input, tokens) {
            return tokens.reduce((score, token) => score + scoreToken(input, token), 0);
        }

        function scoreToken(input, token) {
            let index = input.indexOf(token);
            if (index < 0) {
                return 0;
            }
            const prevChar = index === 0 ? ' ' : input.charAt(index - 1);
            const nextChar = index + token.length >= input.length ? ' ' : input.charAt(index + token.length);
            return 50 + (prevChar === ' ' ? 20 : 0) + ((nextChar === ' ' || nextChar === '.') ? 20 : 0) + (index === 0 ? 10 : 0);
        }

        function scoreQuery(input, query) {
            if (query instanceof RegExp) {
                return query.test(input.replace(/(<([^>]+)>)/ig, '')) ? 100 : 0;
            } else {
                return scoreTokens(input.replace(/(<([^>]+)>)/ig, '').toLowerCase(), query);
            }
        }

        function computeMatchScore(input, queryString) {
            let query = preprocessQuery(queryString);
            return scoreQuery(input, query);
        }

        function objectMatchesQuery(query, object) {
            for (let key in object) {
                if (typeof object[key] === 'object' ) {
                    if (objectMatchesQuery(query, object[key])) {
                        return true;
                    }
                } else if (query instanceof RegExp) {
                    if (query.test(object[key])) {
                        return true;
                    }
                } else {
                    if (String(object[key]).toLowerCase().replace(/(<([^>]+)>)/ig, '').includes(query.toLowerCase())) {
                        return true;
                    }
                }
            }
            return false;
        }

        function objectMatchesQueryWeighted(query, object, keyProperties) {
            // if no match : score = 0
            // if match on non key property : 0 < score <= 100
            // if match on key property : 100 < score <= 200
            let matchScore = 0;
            for (let key in object) {
                let item = object[key];
                if (typeof item === 'object') {
                    matchScore = Math.max(matchScore, objectMatchesQueryWeighted(query, item, keyProperties));
                } else {
                    let isKeyProperty = keyProperties.includes(key);
                    let tmpScore = scoreQuery(("" + item).toLowerCase(), query);
                    if (tmpScore>0) {
                        if (isKeyProperty) {
                            matchScore = Math.max(matchScore, 100 + tmpScore);
                        } else {
                            matchScore = Math.max(matchScore, tmpScore);
                        }
                    }
                }
            }
            return matchScore;
        }

        function filterList(list, query, keyProperties, isFilterPropertyDisciminant) {
            // query is either A. an ["array", "of", "strings"],
            // or B. a single /regexp/
            const threshold = isFilterPropertyDisciminant ? 100 : 0;
            return keyProperties && keyProperties.length > 0
                   ? filterListWeighted(list, query, keyProperties, threshold)
                   : filterListNonWeighted(list, query)
        }

        function filterListNonWeighted(list, query){
            return Array.isArray(query)
                    ? query.reduce(filterList, list)    // A
                    : list.filter(objectMatchesQuery.bind(null, query));  // B
        }

        function matchingQualityList(list, query, keyProperties) {
            return list.map(object => objectMatchesQueryWeighted(query, object, keyProperties));
        }

        function filterListWeighted(list, query, keyProperties, threshold) {
            let matchQualityList = matchingQualityList(list, query, keyProperties);
            return list.map(function(object, i){
                object.matchQuality = matchQualityList[i];
                return object;
            }).filter(object => object.matchQuality > threshold);
        }

        function sortByMatchingQuality(objectsList) {
            objectsList.sort(function(o1, o2) {
                if (o1.matchQuality && o2.matchQuality) {
                    if (o1.matchQuality < o2.matchQuality) {
                        return 1;
                    } else if (o1.matchQuality > o2.matchQuality) {
                        return -1;
                    } else {
                        return 0;
                    }
                } else if (o1.matchQuality) {
                    return -1;
                } else if (o2.matchQuality) {
                    return 1
                } else {
                    return 0;
                }
            });
        }

        function Pagination(list, perPage) {
            this.list = list;
            this.page = 1;  // /!\ 1-based
            this.perPage = perPage || 100;
            this.update();
        }
        Pagination.prototype.update = function updatePagination() {
            if (!this.list || !this.list.length || this.page <= 0) {
                this.slice = [];
                this.from = this.to = this.size = 0;
                return;
            } else {
                this.size = this.list.length;
            }
            this.maxPage = Math.ceil(this.size / this.perPage);
            this.page = Math.min(this.page, this.maxPage);
            this.from = (this.page - 1) * this.perPage;
            this.to = Math.min(this.list.length, this.from + this.perPage) - 1;
            this.slice = this.list.slice(this.from, this.to + 1);
            // page controls, e.g. when on page 6: [1, null, 4, 5, 6, 7, 8, null, 16]
            this.controls = Array(this.maxPage);
            for(var i = this.controls.length; i > 0; i--) { this.controls[i-1] = i; }
            if (this.maxPage > 10) { // keep 2 before, 2 after, and both ends
                if (this.page > 5) {
                    i = this.controls.splice(1, this.page - 4, null).length - 1;
                } // else i = 0; // already from the loop
                if (this.maxPage - this.page > 4) {
                    this.controls.splice(this.page - i + 2, this.maxPage - this.page - 3, null);
                }
            }
        };
        Pagination.prototype.go = function(p) { switch (p) {
            case 0 : this.page = Math.min(this.maxPage, this.page + 1); break;
            case -1: this.page = Math.max(           1, this.page - 1); break;
            default: this.page = p;
        } };    // NB: doesn't call update, this.page is probably $watch()ed
        Pagination.prototype.next = function() { this.go( 0); };
        Pagination.prototype.prev = function() { this.go(-1); };

        Pagination.prototype.nextPage = function(){
            this.page = Math.min(this.maxPage, this.page + 1);
            return this;
        }
        Pagination.prototype.prevPage = function(){
            this.page = Math.max(1, this.page - 1);
            return this;
        }
        // No bounds check is performed. Pages are 1-indexed
        Pagination.prototype.goToPage = function(page){
            this.page = page;
            return this;
        }

        Pagination.prototype.updateAndGetSlice = function(list){
            this.list = list;
            this.update();
            return this.slice;
        }

        return {
            /**
             * Filters a list (array) of Objects according to a queryString.
             * If queryString looks like a /regex/i, filter will use a regex,
             * otherwise it will use whitespace-separated literal tokens.
             *
             * keyProperties If set, objects that matches on one of the key properties are ranked first
             */
            filter: function filter(list, queryString, keyProperties = [], isFilterPropertyDisciminant = false) {
                queryString = queryString && queryString.trim();
                if (!list) return [];
                if (!queryString) return list.concat();  // no-filter fast-pass
                let query = preprocessQuery(queryString)
                return filterList(list, query, keyProperties, isFilterPropertyDisciminant)
            },
            /**
             * Handles pagination.
             * Invoke with `new`, then set its `page` and call `update()`.
             */
            Pagination: Pagination,
            /**
             * Sorts a list of projects by matchQuality property if the list objects have the property.
             */
            sortByMatchingQuality: sortByMatchingQuality,
            /**
             * computes a matching score between a query and a string input.
             * query can be either a regex or a list of tokens
             * No match : score = 0
             * basic match : score = 50
             * match + there is a space before or after : score = 70
             * match + there is a space before and after : score = 90
             * perfect match : score = 100
             */
            computeMatchScore: computeMatchScore
        };
    });

})();
