(function(){
'use strict';

const app = angular.module('dataiku.ml.gpuexecution', []);

// mirrors structure of com.dataiku.dip.analysis.model.core.GpuConfig.GpuSupportingCapability
// must be kept up to date
app.constant("GPU_SUPPORTING_CAPABILITY", {
    KERAS: "KERAS",
    GLUONTS: "GLUONTS",
    DEEP_HUB: "DEEP_HUB",
    DEEP_NN: "DEEP_NN",
    XGBOOST: "XGBOOST",
    SENTENCE_EMBEDDING: "SENTENCE_EMBEDDING"
});

app.service("GpuUsageService", function($filter, DataikuAPI, $stateParams, GPU_SUPPORTING_CAPABILITY, $q) {
    this.getEnvSupportsGpu = getEnvSupportsGpu;
    this.checkGpuUsability = checkGpuUsability;
    this.getCurrentEnvName = getCurrentEnvName;
    this.getAvailableGpuCapabilities = getAvailableGpuCapabilities;
    this.allowChangingMode = allowChangingMode;

    ////////

    const CAPABILITIES = {
        // unfortunately, despite looking weird, instantiation like this is necessary if you want to construct the dict in 1 statement
        // https://stackoverflow.com/a/10640182
        [GPU_SUPPORTING_CAPABILITY.KERAS]: {
            supportsMultiple: true,
            name: "Deep Learning Training",
            supportsDeviceTypeChange: true
        },
        [GPU_SUPPORTING_CAPABILITY.GLUONTS]: {
            supportsMultiple: false,
            name: "Time Series Training",
            supportsDeviceTypeChange: false
        },
        [GPU_SUPPORTING_CAPABILITY.DEEP_HUB]: {
            supportsMultiple: true,  // deephub supports multiple gpus for training, but will only ever use 1 when scoring
            name: "Computer Vision Training",
            supportsDeviceTypeChange: true
        },
        [GPU_SUPPORTING_CAPABILITY.DEEP_NN]: {
            supportsMultiple: false,
            name: "Deep Neural Network Training",
            supportsDeviceTypeChange: false,
        },
        [GPU_SUPPORTING_CAPABILITY.XGBOOST]: {
            supportsMultiple: false,
            name: "XGBoost Training",
            supportsDeviceTypeChange: false,
        },
        [GPU_SUPPORTING_CAPABILITY.SENTENCE_EMBEDDING]: {
            supportsMultiple: false,
            name: "Text Embedding (Code env based)",
            supportsDeviceTypeChange: false,
        }
    };
    this.CAPABILITIES = CAPABILITIES;

    /**
     * Returns a list of all possible capabilities for an MLTask given a backend type/prediction type combo
     * @param {string} backendType
     * @param {string} predictionType
     * @returns {Array<GPU_SUPPORTING_CAPABILITY>}
     */
    function getAvailableGpuCapabilities(backendType, predictionType) {
        const availableGpuCapabilities = [];

        if (backendType === "KERAS") {
            availableGpuCapabilities.push(GPU_SUPPORTING_CAPABILITY.KERAS, GPU_SUPPORTING_CAPABILITY.SENTENCE_EMBEDDING);
        } else if (backendType === "DEEP_HUB") {
            availableGpuCapabilities.push(GPU_SUPPORTING_CAPABILITY.DEEP_HUB);
        } else if (backendType === "PY_MEMORY") {
            if (predictionType === "TIMESERIES_FORECAST") {
                availableGpuCapabilities.push(GPU_SUPPORTING_CAPABILITY.GLUONTS);
            } else { // todo is this correct or do we need to be more specific ?
                availableGpuCapabilities.push(GPU_SUPPORTING_CAPABILITY.DEEP_NN);
                availableGpuCapabilities.push(GPU_SUPPORTING_CAPABILITY.XGBOOST);
            }

            availableGpuCapabilities.push(GPU_SUPPORTING_CAPABILITY.SENTENCE_EMBEDDING);
        }

        return availableGpuCapabilities;
    }

    function allowChangingMode(isScoreOrEval, gpuCapabilities) {
        if (isScoreOrEval) {
            return gpuCapabilities.every(capability => CAPABILITIES[capability].supportsDeviceTypeChange);
        }
        return true;
    }

    /**
     * Returns either the name of the code env that will be used for the training, or null if the dss built-in env will be used
     * @param {string} selectedEnvName The mltask envSelection env name
     * @param {string} envMode The mltask envSelection env mode - will be null outside of analysis
     * @param {string} defaultEnvName The name of the project default code env - can be null in analysis, will be null in recipes
     * @returns {string|null}
     */
    function getCurrentEnvName(selectedEnvName, envMode, defaultEnvName) {
        let currentEnvName = null

        if (envMode) { // analysis
            if(envMode === "INHERIT") {
                // this is only set when the inherited env is not the default dss env, so can still be null
                currentEnvName = defaultEnvName;
            } else if(envMode === "EXPLICIT_ENV") {
                currentEnvName = selectedEnvName;
            }
        } else { // recipes
            currentEnvName = selectedEnvName; // can still be null
        }

        return currentEnvName;
    }

    /**
     * Checks whether we can use multiple gpus, just a single gpu, or whether we have a mix
     *
     * @param {Array<GPU_SUPPORTING_CAPABILITY>} gpuCapabilities - List of enabled gpu features
     * @param {boolean} forceSingleGpu - Whether we want to force only allowing a single gpu
     * @returns {{singleGpuOnly: (boolean), possibleUsageWarning: string}} - Whether we should force selection to a single gpu, and an optional warning
     */
    function checkGpuUsability(gpuCapabilities, forceSingleGpu) {
        const singleGpuCapabilities = [];
        const multipleGpuCapabilities = [];
        let possibleUsageWarning = "";

        if (gpuCapabilities.length === 0) { // don't warn if no capabilities selected
            return {
                singleGpuOnly:false,
                possibleUsageWarning
            }
        }

        gpuCapabilities.forEach(key => {
            const multiple = CAPABILITIES[key].supportsMultiple;
            if (multiple) {
                multipleGpuCapabilities.push(CAPABILITIES[key].name)
            } else {
                singleGpuCapabilities.push(CAPABILITIES[key].name)
            }
        });

        // if we force single gpu, we don't care about the warning
        if (singleGpuCapabilities.length && multipleGpuCapabilities.length && !forceSingleGpu) {
            const multiString = $filter('andList')(multipleGpuCapabilities);
            const singleString = $filter('andList')(singleGpuCapabilities);
            possibleUsageWarning = `Multiple GPU's are only supported for ${multiString}. ${singleString} will use the first selected GPU.`;
        }

        return {
            singleGpuOnly: forceSingleGpu || !multipleGpuCapabilities.length,
            possibleUsageWarning
        }
    }

    /**
     * Checks on a design node, whether the current env has the required packages to execute on gpu
     * By default, we assume that any env supports gpu
     *
     * @param {boolean} isAutomationNode
     * @param {Array<GPU_SUPPORTING_CAPABILITY>} gpuCapabilities - List of gpu supporting capabilities
     * @param {string} currentEnvName - Name of the 'in use' code env (selected or inherited, or null if dss default env)
     * @returns {Promise<boolean>}
     */
    function getEnvSupportsGpu(isAutomationNode, gpuCapabilities, currentEnvName) {
        const deferred = $q.defer();

        // We only check if env supports GPU:
        // * On Design node, as the logic to retrieve the information relies on being on the design node
        // * For tensorflow-based env (i.e. for KERAS backend), because tensorflow has a CPU & a GPU package.
        // * For timeseries env (i.e. using mxnet), because mxnet has a CPU & a GPU package.
        // * For others (e.g. DEEP_HUB), we always consider that the env supports GPU.
        const isKeras = gpuCapabilities.includes(GPU_SUPPORTING_CAPABILITY.KERAS);
        const isTimeseries = gpuCapabilities.includes(GPU_SUPPORTING_CAPABILITY.GLUONTS);
        const isXgboost = gpuCapabilities.includes(GPU_SUPPORTING_CAPABILITY.XGBOOST);

        if (isAutomationNode) {
            deferred.resolve(true);
            return deferred.promise;
        }

        if (isKeras || isTimeseries) {
            if (currentEnvName) {
                DataikuAPI.codeenvs.listWithVisualMlPackages($stateParams.projectKey)
                    .then(function ({data}) {
                        let envSupportsGpu = true;
                        const env = data.envs.find(el => el.envName === currentEnvName);
                        if (isKeras) {
                            envSupportsGpu = env && env.keras.supportsGpu;
                        } else if (isTimeseries) {
                            envSupportsGpu = env && (env.timeseries.supportsGpu || env.mxnetTimeseries.supportsGpu || env.torchTimeseries.supportsGpu);
                        }
                        deferred.resolve(envSupportsGpu);
                    });
            } else {
                // default dss env does not support keras/timeseries with gpu
                deferred.resolve(false);
            }
        } else {
            deferred.resolve(true);
        }

        return deferred.promise;
    }
})

app.component("gpuCapabilityForm", {
    templateUrl:'/templates/analysis/mlcommon/settings/gpu-capability-form.html',
    bindings: {
        availableGpuCapabilities: '<',
        usedGpuCapabilities: '<',
        gpuConfig: '=',
        onSelectedChange:"&?"
    },
    controller : function($scope, GpuUsageService) {
        const $ctrl = this;
        $ctrl.GpuUsageService = GpuUsageService;

        $ctrl.toggleSelection = function(item) {
            const disabledCapabilitiesSet = new Set($ctrl.gpuConfig.disabledCapabilities);

            if (disabledCapabilitiesSet.has(item)) {
                disabledCapabilitiesSet.delete(item);
            } else {
                disabledCapabilitiesSet.add(item);
            }

            $ctrl.gpuConfig.disabledCapabilities = [...disabledCapabilitiesSet];

            if (angular.isFunction($ctrl.onSelectedChange)) {
                $ctrl.onSelectedChange();
            }
        }

        $ctrl.capabilityToId = function(gpuCapability) {
            let idString = "gpu_control_" + gpuCapability.toLowerCase();
            idString.replace("_", "-")
            return idString;
        };

        $ctrl.isUsed = function(gpuCapability) {
            const usedGpuCapabilitiesSet = new Set($ctrl.usedGpuCapabilities);
            return usedGpuCapabilitiesSet.has(gpuCapability);
        }
    }
})

})();
