// @ts-check
(function() {
    'use strict';
    /** @typedef {import('../types').AxisSpecs } AxisSpecs */
    /** @typedef {import("../types").GeneratedSources.DimensionDef} DimensionDef */
    /** @typedef {import("../types").GeneratedSources.MeasureDef} MeasureDef */
    /** @typedef {import("../types").GeneratedSources.ChartFilter} ChartFilter */
    /** @typedef {import("../types").GeneratedSources.ChartDef} ChartDef */
    /** @typedef {import("../../../../../../../../server//src/frontend/node_modules/ag-grid-enterprise/").GridOptions} GridOptions */
    /** @typedef {import("../../../../../../../../server//src/frontend/node_modules/ag-grid-enterprise/").GridApi} GridApi */


    angular.module('dataiku.charts').service('ChartStoreFactory', chartStoreFactory);

    /**
     * ChartSpecs is a structure which stores charts by frame and facets:
     * @type {
     *  [frameIndex: number]: {
     *      [facetIndex: number]: {
     *          dimensionDefToId: Map<DimensionDef, string>,
     *          measureDefToId: Map<MeasureDef, string>,
     *          axisSpecs: AxisSpecs,
     *          gridOptions: GridOptions,
     *          requestOptions: Record<string, any>
     *      }
     *  }
     * }
     * chartSpecs = {
     *      0: {
     *          0: {
     *              axisSpecs: {...}
     *          }
     *          1: {
     *              axisSpecs: {...}
     *          }
     *      },
     *      1: {
     *          0: {
     *              axisSpecs: {...}
     *          }
     *      }
     * }
     *
     * In order to retrieve a property (like axisSpecs) linked to a specific chart, you can use a convenience method like "get"
     * or retrieve it by using its related frameIndex and facetIndex:
     * const axisSpec = chartSpecs[frameIndex][facetIndex]['axisSpec'];
     *
     * ChartSpecs should register every property related to a chart instead of "chartHandler"
     */
    function ChartSpecs() { }

    /**
     * Insert is an util method to insert properties from a chart in the ChartSpecs structure
     * It automatically creates entries for the related frameIndex, facetIndex and key property
     * @param {number} frameIndex
     * @param {number} facetIndex
     * @param {string} key
     * @param {any} value
     */
    ChartSpecs.prototype.insert = function(frameIndex, facetIndex, key, value) {
        if (!this[frameIndex]) {
            this[frameIndex] = {};
        }

        if (!this[frameIndex][facetIndex]) {
            this[frameIndex][facetIndex] = {};
        }

        this[frameIndex][facetIndex][key] = value;

        return this[frameIndex][facetIndex][key];
    };

    /**
     * Get is an util method to get properties from a chart in the ChartSpecs structure
     * It automatically checks if the related frameIndex and facetIndex exist in the structure before returning the key
     * @param {number} frameIndex
     * @param {number} facetIndex
     * @param {string} key
     * @returns value related to key property in given chart (identified by frame and facet)
     */
    ChartSpecs.prototype.get = function(frameIndex, facetIndex, key) {
        return this[frameIndex] && this[frameIndex][facetIndex] && this[frameIndex][facetIndex][key];
    };

    /**
     * Remove is an util method to remove a property from a chart in the ChartSpecs structure
     * It automatically checks if the related frameIndex and facetIndex exist and deletes associated key
     * @param {number} frameIndex
     * @param {number} facetIndex
     * @param {string} key
     * @returns true if deleted, else false
     */
    ChartSpecs.prototype.remove = function(frameIndex, facetIndex, key) {
        if (this[frameIndex] && this[frameIndex][facetIndex] && this[frameIndex][facetIndex][key]) {
            delete this[frameIndex][facetIndex][key];
            return true;
        }
        return false;
    };

    /**
     * Flatten is an util method to retrieve all charts as a flat array,
     * adding frameIndex and facetIndex as properties of a chart object
     * @returns an array of charts
     */
    ChartSpecs.prototype.flatten = function() {
        return Object.entries(this).reduce((accByFrames, frameEntry) => {
            return Object.entries(frameEntry[1]).reduce((accByFacets, facetEntry) => {
                const value = facetEntry[1];
                value.frame = frameEntry[0];
                value.facet = facetEntry[0];
                accByFacets.push(value);
                return accByFacets;
            }, accByFrames);
        }, []);
    };

    /**
     * ChartStore
     * Store charts information to be shared across the codebase.
     * One chart store = one chart container (chart editor/insight/dashboard tile)
     * Each chart store owns a ChartSpecs structure which registers properties of each chart in the container (can be a solo chart or multiple subcharts)
     */
    function chartStore(CHART_AXIS_TYPES) {
        const chartSpecs = new ChartSpecs();

        return {
            get: function(key, frameIndex = 0, facetIndex = 0) {
                return chartSpecs.get(frameIndex, facetIndex, key);
            },
            set: function(key, value, frameIndex = 0, facetIndex = 0) {
                chartSpecs.insert(frameIndex, facetIndex, key, value);
            },
            /**
             *
             * @param {AxisSpecs} axisSpecs
             * @param {number} frameIndex
             * @param {number} facetIndex
             */
            setAxisSpecs: function(axisSpecs, frameIndex = 0, facetIndex = 0) {
                chartSpecs.insert(frameIndex, facetIndex, 'axisSpecs', axisSpecs);
            },
            getAxisSpecs: function(frameIndex = 0, facetIndex = 0) {
                return chartSpecs.get(frameIndex, facetIndex, 'axisSpecs');
            },
            getAxisSpec: function(axisId, frameIndex = 0, facetIndex = 0) {
                const axisSpecs = chartSpecs.get(frameIndex, facetIndex, 'axisSpecs');
                return axisSpecs && axisSpecs[axisId];
            },
            /**
             * Get the dimension id of `dimension`.
             * @param   {DimensionDef}  dimension
             * @param   {number}        frameIndex
             * @param   {number}        facetIndex
             * @return  {string}        dimension id
             */
            getDimensionId(dimension, frameIndex = 0, facetIndex = 0) {
                let dimensionDefToId = chartSpecs.get(frameIndex, facetIndex, 'dimensionDefToId');

                if (dimensionDefToId === undefined) {
                    dimensionDefToId = chartSpecs.insert(frameIndex, facetIndex, 'dimensionDefToId', new Map());
                }

                return dimensionDefToId.get(dimension);
            },
            /**
             * Set the dimension id of `dimension`.
             * @param {DimensionDef}    dimension
             * @param {number}          frameIndex
             * @param {number}          facetIndex
             */
            setDimensionId(dimension, id, frameIndex = 0, facetIndex = 0) {
                let dimensionDefToId = chartSpecs.get(frameIndex, facetIndex, 'dimensionDefToId');

                if (dimensionDefToId === undefined) {
                    dimensionDefToId = chartSpecs.insert(frameIndex, facetIndex, 'dimensionDefToId', new Map());
                }

                dimensionDefToId.set(dimension, id);
            },
            /**
             * Purge dimension ids
             * @param {number}  frameIndex
             * @param {number}  facetIndex
             */
            purgeDimensionIds(frameIndex = 0, facetIndex = 0) {
                chartSpecs.remove(frameIndex, facetIndex, 'dimensionDefToId');
            },
            /**
             * Get the measure id of `measure`.
             * @param   {MeasureDef}    measure
             * @return  {string}        measure id
             */
            /**
             * Get the measure id of `measure`.
             * @param   {MeasureDef}    measure
             * @param   {number}        frameIndex
             * @param   {number}        facetIndex
             * @return  {string}        measure id
             */
            getMeasureId(measure, frameIndex = 0, facetIndex = 0) {
                let measureDefToId = chartSpecs.get(frameIndex, facetIndex, 'measureDefToId');

                if (measureDefToId === undefined) {
                    measureDefToId = chartSpecs.insert(frameIndex, facetIndex, 'measureDefToId', new Map());
                }

                return measureDefToId.get(measure);
            },
            /**
             * Set the measure id of `measure`.
             * @param {MeasureDef}      measure
             * @param {number}          frameIndex
             * @param {number}          facetIndex
             */
            setMeasureId(measure, id, frameIndex = 0, facetIndex = 0) {
                let measureDefToId = chartSpecs.get(frameIndex, facetIndex, 'measureDefToId');

                if (measureDefToId === undefined) {
                    measureDefToId = chartSpecs.insert(frameIndex, facetIndex, 'measureDefToId', new Map());
                }

                measureDefToId.set(measure, id);
            },
            /**
             * Purge measure ids
             * @param {number}  frameIndex
             * @param {number}  facetIndex
             */
            purgeMeasureIds(frameIndex = 0, facetIndex = 0) {
                chartSpecs.remove(frameIndex, facetIndex, 'measureDefToId');
            },
            /**
             *
             * @param {number} frameIndex
             * @param {number} facetIndex
             * @returns {ChartFilter[]}
             */
            getAppliedDashboardFilters: function(frameIndex = 0, facetIndex = 0) {
                let dashboardFilters = chartSpecs.get(frameIndex, facetIndex, 'dashboardFilters');

                if (dashboardFilters === undefined) {
                    dashboardFilters = chartSpecs.insert(frameIndex, facetIndex, 'dashboardFilters', {});
                }

                return dashboardFilters;
            },
            /**
             *
             * @param {ChartFilter[]} dashboardFilters
             * @param {number} frameIndex
             * @param {number} facetIndex
             */
            setAppliedDashboardFilters: function(dashboardFilters, frameIndex = 0, facetIndex = 0) {
                chartSpecs.insert(frameIndex, facetIndex, 'dashboardFilters', dashboardFilters);
            },
            /**
             *
             * @param {number} frameIndex
             * @param {number} facetIndex
             * @returns {GridApi}
             */
            getGridApi: function(frameIndex = 0, facetIndex = 0) {
                let gridApi = chartSpecs.get(frameIndex, facetIndex, 'gridApi');

                if (gridApi === undefined) {
                    gridApi = chartSpecs.insert(frameIndex, facetIndex, 'gridApi', null);
                }

                return gridApi;
            },
            /**
             *
             * @param {GridApi} gridApi
             * @param {number} frameIndex
             * @param {number} facetIndex
             */
            setGridApi: function(gridApi, frameIndex = 0, facetIndex = 0) {
                chartSpecs.insert(frameIndex, facetIndex, 'gridApi', gridApi);
            },
            /**
             *
             * @param {{ [hierarchyName: string]: string[]}} hierarchyMissingColumns
             * @param {number} frameIndex
             * @param {number} facetIndex
             */
            setHierarchyMissingColumns: function(hierarchyMissingColumns, frameIndex = 0, facetIndex = 0) {
                chartSpecs.insert(frameIndex, facetIndex, 'hierarchyMissingColumns', hierarchyMissingColumns);
            },
            /**
             *
             * @param {number} frameIndex
             * @param {number} facetIndex
             * @returns {{ [hierarchyName: string]: string[]}}
             */
            getHierarchyMissingColumns: function(frameIndex = 0, facetIndex = 0) {
                let hierarchyMissingColumns = chartSpecs.get(frameIndex, facetIndex, 'hierarchyMissingColumns');

                if (hierarchyMissingColumns === undefined) {
                    hierarchyMissingColumns = chartSpecs.insert(frameIndex, facetIndex, 'hierarchyMissingColumns', {});
                }

                return hierarchyMissingColumns;
            },
            /**
             * Returns server request options
             * @param {number} frameIndex
             * @param {number} facetIndex
             * @returns {Record<string, any>} requestOptions
             */
            getRequestOptions: function(frameIndex = 0, facetIndex = 0) {
                let requestOptions = chartSpecs.get(frameIndex, facetIndex, 'requestOptions');

                if (requestOptions === undefined) {
                    requestOptions = chartSpecs.insert(frameIndex, facetIndex, 'requestOptions', {});
                }

                return requestOptions;
            },
            /**
             * Saves server request options
             * @param {Record<string, any>} requestOptions
             */
            setRequestOptions: function(requestOptions, frameIndex = 0, facetIndex = 0) {
                const currentRequestOptions = chartSpecs.get(frameIndex, facetIndex, 'requestOptions') || {};
                chartSpecs.insert(frameIndex, facetIndex, 'requestOptions', { ...currentRequestOptions, ...requestOptions });
            },

            getAllAxisSpecs: function(axisId) {
                const axisSpecs = chartSpecs.flatten().filter(chartSpec => chartSpec.axisSpecs).map(chartSpec => chartSpec.axisSpecs);
                return axisSpecs && axisSpecs.map(axisSpec => axisSpec[axisId]);
            },

            /**
             * Retrieves axis type of an axis (null if not found or not coherent between each subchart)
             * @param {string} axisId
             * @returns {CHART_AXIS_TYPES} axisType
             */
            getAxisType: function(axisId) {
                const axisSpecs = this.getAllAxisSpecs(axisId);
                const uniqueAxisTypes = Array.from(new Set(axisSpecs.map(spec => spec && spec.type)));
                return uniqueAxisTypes.length === 1 ? uniqueAxisTypes[0] : null;
            },

            /**
             * Retrieves dimension type (null if not found or not coherent between each subchart)
             * @param {string} axisId
             */
            getAxisDimensionOrUADimensionType: function(axisId) {
                const axisSpecs = this.getAllAxisSpecs(axisId);
                const uniqueDimensionTypes = Array.from(new Set(axisSpecs.map(spec => spec && spec.dimension && spec.dimension.type)));
                return uniqueDimensionTypes.length === 1 ? uniqueDimensionTypes[0] : null;
            },

            /**
             * Retrieves dimension num params mode if it exists (null if not found or not coherent between each subchart)
             * @param {string} axisId id of an y axis
             */
            getAxisDimensionOrUADimensionNumParamsMode: function(axisId) {
                const axisSpecs = this.getAllAxisSpecs(axisId);
                const uniqueNumParamsMode = Array.from(new Set(axisSpecs.map(spec => spec && spec.dimension && spec.dimension.type && spec.dimension.numParams && spec.dimension.numParams.mode)));
                return uniqueNumParamsMode.length === 1 ? uniqueNumParamsMode[0] : null;
            },

            /**
             * Stores the measures' & dimensions' Ids into the ChartStore
             * @param {ChartDef} chartDef
             */
            updateChartStoreMeasureIDsAndDimensionIDs(chartDef) {
                this.purgeMeasureIds();
                this.purgeDimensionIds();
                chartDef.genericMeasures.forEach((measure, i) => this.setMeasureId(measure, this.getMeasureUniqueId(measure, i)));
                chartDef.yDimension.forEach((yDim, i) => this.setDimensionId(yDim, this.getDimensionUniqueId(yDim, i, 'yDimension')));
                chartDef.xDimension.forEach((xDim, i) => this.setDimensionId(xDim, this.getDimensionUniqueId(xDim, i, 'xDimension')));

                if (chartDef.xHierarchyDimension && chartDef.xHierarchyDimension.length) {
                    chartDef.xHierarchyDimension[0].dimensions.forEach((dim, i) => this.setDimensionId(dim, this.getDimensionUniqueId(dim, i, 'xHierarchyDimension')));
                }
                if (chartDef.yHierarchyDimension && chartDef.yHierarchyDimension.length) {
                    chartDef.yHierarchyDimension[0].dimensions.forEach((dim, i) => this.setDimensionId(dim, this.getDimensionUniqueId(dim, i, 'yHierarchyDimension')));
                }
            },

            /**
             * Generates a unique ID for a dimensions
             * @param {DimensionDef} dimension
             * @param {number} index
             * @param {string} type
             */
            getDimensionUniqueId(dimension, index, type) {
                // Replaces points by underscore because dimension ids are used to build column ids with AGGrid which can't have dots.
                return (`${type}_${index}_${dimension.column}`).replace('.', '_');
            },

            /**
             * Generates a unique ID for a dimensions
             * @param {MeasureDef} measure
             * @param {number} index
             */
            getMeasureUniqueId(measure, index) {
                // Replaces points by underscore because measure ids are used to build column ids with AGGrid which can't have dots.
                return (`measure_${index}_${measure.column}`).replace('.', '_');
            }
        };
    }

    function chartStoreFactory(CHART_AXIS_TYPES) {
        let latestId = 1;
        const chartStores = {};

        const generateId = () => {
            return latestId++;
        };

        const svc = {
            create: () => {
                const id = generateId();
                const store = chartStore(CHART_AXIS_TYPES);
                if (!chartStores[id]) {
                    chartStores[id] = store;
                } else {
                    throw new Error(`${id} already exists, cannot create chart store`);
                }

                return { id, store };
            },
            get: (id) => {
                return chartStores[id];
            },
            getOrCreate: (id) => {
                if (id && chartStores[id]) {
                    return { id, store: svc.get(id) };
                }

                return svc.create();
            }
        };

        return svc;
    }
})();
