(function() {
    'use strict';
    /** @typedef {import('../../../types').AGGridConverterDFSRowData} AGGridConverterDFSRowData */
    /** @typedef {import('../../../types').AGGridConverterDFSColumnData} AGGridConverterDFSColumnData */
    /** @typedef {import('../../../types').EndColumnData} EndColumnData */
    /** @typedef {import('../../../types').SubTotalColumnData} SubTotalColumnData */
    /** @typedef {import('../../../types').ChartDef} ChartDef */
    /** @typedef {import('../../../types').GeneratedSources.MeasureDef} MeasureDef */
    /** @typedef {import('../../../types').GeneratedSources.DimensionDef} DimensionDef */
    /** @typedef {import('../../../types').GeneratedSources.PivotTableTensorResponse} PivotTableTensorResponse */
    /** @typedef {import('../../../types').PivotTableCellContent} PivotTableCellContent */
    /** @typedef {import('../../../types').PivotTableRowTreeCellContent} PivotTableRowTreeCellContent */
    /** @typedef {import('../../../types').PivotTableCellMeasure} PivotTableCellMeasure */
    /** @typedef {import('../../../types').AgGridRow} AgGridRow */
    /** @typedef {import('../../../types').colorScale} colorScale */
    /** @typedef {import('../../../types').GeneratedSources.DSSVisualizationTheme} DSSVisualizationTheme */

    /** @typedef {import('../../../../../../../../../../server/src/frontend/node_modules/ag-grid-enterprise/').GridOptions} GridOptions */
    /** @typedef {import('../../../../../../../../../../server/src/frontend/node_modules/ag-grid-enterprise/').RowClassParams} RowClassParams */
    /** @typedef {import('../../../../../../../../../../server/src/frontend/node_modules/ag-grid-enterprise/').RowStyle} RowStyle */
    /** @typedef {import('../../../../../../../../../../server/src/frontend/node_modules/ag-grid-enterprise/').ICellRendererParams} ICellRendererParams */
    /** @typedef {import('../../../../../../../../../../server/src/frontend/node_modules/ag-grid-enterprise/').ICellRendererComp} ICellRendererComp */
    /** @typedef {import('../../../../../../../../../../server/src/frontend/node_modules/ag-grid-community/').RowGroupOpenedEvent} RowGroupOpenedEvent */
    /** @typedef {import('../../../../../../../../../../server/src/frontend/node_modules/ag-grid-enterprise/').RowNode} RowNode */
    /** @typedef {import('../../../../../../../../../../server/src/frontend/node_modules/ag-grid-enterprise/').ColDef} ColDef */
    /** @typedef {import('../../../../../../../../../../server/src/frontend/node_modules/ag-grid-enterprise/').ColGroupDef} ColGroupDef */

    angular.module('dataiku.charts')
        .factory('AgGridConverter', function(ChartLabels, HierarchicalChartsUtils, ChartFormatting, ChartColorScales, ChartLegendUtils, ExpandCollapseToolPanel, DisplayTotalsToolPanel, LinkedList, ChartDimension, PivotTableUtils, ChartFeatures, ChartsStaticData, ClipboardUtils, ConditionalFormattingOptions, ColorUtils, ChartHierarchyDimension, translate) {
            const LIGHT_GREY_COLUMN = '#ededed';
            const LIGHT_GREY_ROW = '#f2f7fa';
            const SUBTOTAL_LEAF_ROW_KEY = '__Subtotal__Leaf__Row__';
            const DEFAULT_ROW_HEIGHT = 25;
            // 16 (chevron) + 16 (chevron padding right)
            const CHEVRON_OFFSET_WIDTH = 32;
            // 12 (row header cell padding left)
            const ROW_OFFSET_WIDTH = 12;
            const AUTOGROUP_COLUMN_KEY = 'ag-Grid-AutoColumn';
            const COLOR_FLAG_KEY = '__Color__Flag__';
            const GRAND_TOTAL_KEY = '__Grand__Total__';
            const GRAND_TOTAL_LABEL = 'Grand Total';
            const SUB_COLUMN_SEPARATOR = '__';
            const MEASURE_COLUMN_KEY = '__MEASURE__LIST__';
            const MEASURE_COLUMN_LABEL = 'Value';
            const UNIQUE_ROW_KEY = '__UNIQUE__ROW__';

            const DEFAULT_FONT_SIZE = '12px';
            const DEFAULT_FONT_COLOR = '#333';

            /** CSS properties of SubTotal column */
            const SUBTOTAL_COLUMN_STYLE = {
                backgroundColor: LIGHT_GREY_COLUMN,
                padding: 0
            };

            /** CSS properties of SubTotal row */
            const SUB_TOTAL_ROW_STYLE = {
                backgroundColor: LIGHT_GREY_ROW,
                padding: 0
            };

            /** CSS properties of leaf row */
            const LEAF_ROW_STYLE = {
                padding: 0
            };

            /** CSS properties of grand total column */
            const GRAND_TOTAL_COLUMN_STYLE = {
                backgroundColor: LIGHT_GREY_COLUMN,
                fontWeight: 'bold',
                padding: 0
            };

            /** the CSS properties of grand total row */
            const GRAND_TOTAL_ROW_STYLE = {
                backgroundColor: LIGHT_GREY_ROW,
                fontWeight: 'bold',
                padding: 0
            };

            /** the CSS properties of measure label column */
            const MEASURE_COLUMN_STYLE = {
                padding: 0
            };

            class AbstractAGGridPivotTableChartConverter {
                /**
                 * @param {ChartDef} chartDef
                 * @param {*} chartStore
                 * @param {*} chartData
                 * @param {PivotTableTensorResponse} data
                 * @param {*} chartHandler
                 * @param {*} tooltips
                 * @param {*} contextualMenu
                 * @param {JQuery<HTMLElement>} $rootElement
                 */
                constructor(chartDef, chartStore, chartData, data, chartHandler, tooltips, contextualMenu, $rootElement) {
                    /** @type ChartDef */
                    this.chartDef = chartDef;
                    this.chartStore = chartStore;
                    this.chartData = chartData;
                    /** @type PivotTableTensorResponse */
                    this.data = data;
                    this.chartHandler = chartHandler;
                    this.tooltips = tooltips;
                    this.contextualMenu = contextualMenu;
                    /** @type JQuery<HTMLElement> */
                    this.$rootElement = $rootElement;

                    /** @type {EndColumnData []} */
                    this.endColumns = [];
                    /** @type {SubTotalColumnData []} */
                    this.subtotalColumns = [];
                    /** @type {Map<string, number>} */
                    this.grandTotalColumnMaxCellWidth = new Map();
                    /** @type {Map<string[], PivotTableRowTreeCellContent>} */
                    this.mapGroupColumnCellIdToHeaderName = new Map();
                    /** @type {Set<RowNode>} */
                    this.initializedRowNode = new Set();

                    const yHierarchyDimension = ChartHierarchyDimension.getCurrentHierarchyDimension(chartDef, 'y');
                    /** @type {DimensionDef[]} */
                    this.yDimensions = yHierarchyDimension ? [yHierarchyDimension] : this.chartDef.yDimension;

                    const xHierarchyDimension = ChartHierarchyDimension.getCurrentHierarchyDimension(chartDef, 'x');
                    /** @type {DimensionDef[]} */
                    this.xDimensions = xHierarchyDimension ? [xHierarchyDimension] : this.chartDef.xDimension;

                    const cellFormatterProperties = HierarchicalChartsUtils.computeFormatterProperties(this.chartDef);

                    /** @type {((value: any, count: number) => string)[]} */
                    this.cellFormatters = cellFormatterProperties.cellFormatters;
                    /** @type {boolean} */
                    this.isPercentageOnly = cellFormatterProperties.isPercentageOnly;

                    this.colorGroupByMeasureId = {};
                    if (chartDef.colorMode === 'COLOR_GROUPS') {
                        this.chartHandler.legendsWrapper.deleteLegends();

                        // index of measure group colors are based of in data.aggregations
                        let basedOnMeasureIndexCounter = chartDef.genericMeasures.length - 1;
                        // object with rules and scale of each value that currently belongs to a group
                        this.colorGroupByMeasureId = chartDef.colorGroups ? chartDef.colorGroups
                            .filter(({ appliedColumns }) => !!appliedColumns)
                            .reduce((acc, { appliedColumns, rules, colorOptions, colorGroupMode, colorMeasure }) => {
                                let groupColorMeasure; // for scale based on another column

                                if (colorMeasure && colorMeasure.length > 0) {
                                    basedOnMeasureIndexCounter+=1;
                                    groupColorMeasure = colorMeasure[0];
                                }

                                appliedColumns.forEach(column => {
                                    const columnAggIndex = ConditionalFormattingOptions.getColorColumnIndex(
                                        column,
                                        chartDef.genericMeasures,
                                        chartDef.colorMode
                                    ); // for scale based on the appliedColumn itself

                                    const currColorScaleColumnIndex = groupColorMeasure ? basedOnMeasureIndexCounter : columnAggIndex;

                                    const hasColorProperties = colorOptions &&
                                        currColorScaleColumnIndex < this.chartData.data.aggregations.length // check if aggregation is present on the request response
                                    ;

                                    // call computeColorTableProperties here to avoid it being called for every getColorScale
                                    const colorScale = hasColorProperties ?
                                        this.computeColorTableProperties(
                                            colorOptions,
                                            currColorScaleColumnIndex,
                                            groupColorMeasure || chartDef.genericMeasures[currColorScaleColumnIndex]
                                        ).colorScale :
                                        undefined
                                    ;

                                    acc[ConditionalFormattingOptions.getMeasureId(column)] = {
                                        rules,
                                        colorGroupMode,
                                        colorProperties: hasColorProperties ? {
                                            getColorScale: () => colorScale,
                                            getBin: (cellCoords) => this.chartData.getAggrLoc(currColorScaleColumnIndex, cellCoords)
                                        } : undefined,
                                        // if is based on another column is set, use it as basedOnMeasure
                                        basedOnMeasure: groupColorMeasure,
                                        basedOnMeasureIndex: groupColorMeasure ? basedOnMeasureIndexCounter : -1
                                    };
                                });

                                return acc;
                            }, {}) : {}
                        ;
                    } else {
                        const colorProperties = this.computeColorTableProperties(chartDef.colorOptions);
                        /** @type {boolean} */
                        this.hasColorMeasure = colorProperties.hasColorMeasure;
                        /** @type {number} */
                        this.colorMeasureIndex = colorProperties.colorMeasureIndex;
                        /** @type {colorScale} */
                        this.colorScale = colorProperties.colorScale;
                    }

                    /** @type {Set<string>} */
                    this.rowDimensionId = new Set();
                    this.yDimensions.map(dimension => this.rowDimensionId.add(this.chartStore.getDimensionId(dimension)));

                    /** @type {ChartDef['pivotTableOptions']['tableFormatting']} */
                    const { rowMainHeaders, rowSubheaders, columnMainHeaders, columnSubheaders, values } = this.chartDef.pivotTableOptions.tableFormatting;

                    const defaultFontFamily = 'SourceSansPro';
                    const theme = this.chartHandler.getChartTheme();
                    const fontFamily = (theme && theme.generalFormatting && theme.generalFormatting.fontFamily) ? `${theme.generalFormatting.fontFamily}, ${defaultFontFamily}` : defaultFontFamily;

                    this.rowMainHeadersFormatting = {
                        fontFamily,
                        fontSize: rowMainHeaders ? `${rowMainHeaders.fontSize}px`: DEFAULT_FONT_SIZE,
                        color: rowMainHeaders ? rowMainHeaders.fontColor : DEFAULT_FONT_COLOR
                    };

                    this.rowSubheadersFormatting = {
                        fontFamily,
                        fontSize: rowSubheaders ? `${rowSubheaders.fontSize}px` : DEFAULT_FONT_SIZE,
                        color: rowSubheaders ? rowSubheaders.fontColor : DEFAULT_FONT_COLOR
                    };

                    this.columnMainHeadersFormatting = {
                        fontFamily,
                        fontSize: `${columnMainHeaders.fontSize}px`,
                        color: columnMainHeaders.fontColor
                    };

                    this.columnSubheadersFormatting = {
                        fontFamily,
                        fontSize: `${columnSubheaders.fontSize}px`,
                        color: columnSubheaders.fontColor
                    };

                    this.valuesFormatting = {
                        fontFamily,
                        fontSize: `${values.fontSize}px`,
                        color: ChartFeatures.isColoredPivotTable(chartDef) ? DEFAULT_FONT_COLOR : values.fontColor
                    };
                }

                /**
                 * Render cell with chevron to navigate inside row dimensions
                 * @param       {ICellRendererParams} params
                 * @return      {HTMLElement}
                 * @memberof    AbstractAGGridPivotTableChartConverter
                 */
                cellWithChevronRenderer(params) {
                    const container = document.createElement('div');

                    if (params.value !== undefined) {
                        let node = params.node;
                        for (let i = 0; i < params.value.depth; i++) {
                            node = node.parent;
                        }
                        return this.buildCellWithChevron(params, node);
                    }
                    return container;
                }

                /**
                 * Build tree row cell with chevron
                 * @param       {ICellRendererParams}               params
                 * @param       {RowNode}                           node
                 * @return      {HTMLElement}
                 * @memberof    AbstractAGGridPivotTableChartConverter
                 */
                buildCellWithChevron(params, node) {
                    const value = params.value;
                    const container = document.createElement('div');
                    container.classList.add('pivot-table__group');
                    const span = document.createElement('span');
                    span.classList.add('pivot-table__group-content');
                    span.innerText = value.label;
                    const hasChevron = value.depth > 0 || value.depth == 0 && node.hasChildren();
                    if (hasChevron) {
                        container.classList.add('pivot-table__group--with-chevron');
                        const chevronContainer = document.createElement('span');
                        const chevron = document.createElement('span');
                        chevronContainer.appendChild(chevron);
                        if (node.expanded) {
                            chevronContainer.className = 'ag-group-expanded pivot-table__chevron-container';
                            chevron.className = 'ag-icon ag-icon-tree-open';
                        } else {
                            chevronContainer.className = 'ag-group-contracted pivot-table__chevron-container';
                            chevron.className = 'ag-icon ag-icon-contracted';
                        }
                        if (!this.initializedRowNode.has(node)) {
                            this.initializedRowNode.add(node);
                            node.setExpanded(this.computeRowExpandedStatus(node.id));
                        }
                        container.appendChild(chevronContainer);
                        container.addEventListener('click', () => this.switchFoldStatus(node));
                        container.addEventListener('dblclick', () => this.switchFoldStatus(node));
                    }
                    if (value.hasOffset) {
                        container.classList.add('pivot-table__cell--total-label');
                    }
                    container.classList.add(PivotTableUtils.ROW_SUBHEADER_CLASSNAME);
                    this.contextualMenu.addContextualMenuHandler(container, undefined, PivotTableUtils.getRowHeaderContextualMenuActions(this.chartDef, PivotTableUtils.getToggleRowHeadersCallbacks(this.chartDef, this.chartHandler.openSection), value.label), 'pivotTableHeader' );
                    container.appendChild(span);
                    return container;
                }

                /**
                 * Switch row node status, fold or unfolded
                 * @param       {RowNode}   node
                 * @memberof    AbstractAGGridPivotTableChartConverter
                 */
                switchFoldStatus(node) {
                    const expanded = !node.expanded;
                    node.setExpanded(expanded);
                    if (expanded != this.chartDef.pivotTableOptions.areRowsExpandedByDefault) {
                        this.chartDef.pivotTableOptions.rowIdByCustomExpandedStatus[node.id] = expanded;
                    } else {
                        delete this.chartDef.pivotTableOptions.rowIdByCustomExpandedStatus[node.id];
                    }
                }

                /**
                 * Build AG Grid options
                 * @return      {GridOptions}
                 * @memberof    AbstractAGGridPivotTableChartConverter
                 */
                convert() {
                    const columnTree = this.buildColumnTree();
                    const rowTree = this.buildRowTree();
                    const { rowData, subtotalRowData, grandTotalRowData } = this.filterRowTreeAndConvertToRowData(rowTree);
                    const { columnDefs, grandTotalColumnDef } = this.filterColumnTreeAndConvertToColumnDefs(columnTree);

                    const fullRowData = [...rowData];
                    if (this.chartDef.pivotTableOptions.displayTotals.subTotals.rows) {
                        fullRowData.push(...subtotalRowData);
                    }
                    if (grandTotalRowData && this.chartDef.pivotTableOptions.displayTotals.grandTotal.row) {
                        fullRowData.push(grandTotalRowData);
                    }

                    if (grandTotalColumnDef) {
                        columnDefs.push(grandTotalColumnDef);
                    }

                    return {
                        defaultColDef: {
                            sortable: false,
                            resizable: true,
                            filter: false
                        },
                        treeData: this.chartDef.pivotTableOptions.tableFormatting.showRowHeaders ? true : false,
                        icons: {
                            groupExpanded: ' ',
                            groupContracted: ' '
                        },
                        debug: false,
                        rowHeight: this.getRowHeight(),
                        groupHeaderHeight: this.getGroupHeaderHeight(),
                        headerHeight: this.getHeaderHeight(),
                        enableCellTextSelection: true,
                        suppressFieldDotNotation: true,
                        rowData: fullRowData,
                        columnDefs: columnDefs,
                        columnTypes: {
                            rowNavigationColumn: {
                                suppressHeaderMenuButton: true,
                                cellRenderer: 'cellWithChevronRenderer',
                                cellRendererParams: {
                                    tooltips: this.tooltips,
                                    contextualMenu: this.contextualMenu,
                                    genericMeasures: this.chartDef.genericMeasures,
                                    colorGroupByMeasureId: this.colorGroupByMeasureId,
                                    colorMode: this.chartDef.colorMode,
                                    rowHeadersCallbacks: PivotTableUtils.getToggleRowHeadersCallbacks(this.chartDef, this.chartHandler.openSection)
                                },
                                cellStyle: this.rowSubheadersFormatting
                            },
                            subtotalColumn: {
                                suppressHeaderMenuButton: true,
                                cellStyle: {
                                    ...SUBTOTAL_COLUMN_STYLE,
                                    ...this.valuesFormatting
                                },
                                cellRenderer: 'cellWithTooltipRenderer',
                                cellRendererParams: {
                                    tooltips: this.tooltips,
                                    contextualMenu: this.contextualMenu,
                                    rowDimensionId: this.rowDimensionId,
                                    colorGroupByMeasureId: this.colorGroupByMeasureId,
                                    theme: this.chartHandler.getChartTheme(),
                                    chartDef: this.chartDef
                                }
                            },
                            grandTotalGroupColumn: {
                                suppressHeaderMenuButton: true
                            },
                            grandTotalColumn: {
                                suppressHeaderMenuButton: true,
                                cellRenderer: 'cellWithTooltipRenderer',
                                cellRendererParams: {
                                    tooltips: this.tooltips,
                                    contextualMenu: this.contextualMenu,
                                    rowDimensionId: this.rowDimensionId,
                                    colorGroupByMeasureId: this.colorGroupByMeasureId,
                                    theme: this.chartHandler.getChartTheme(),
                                    chartDef: this.chartDef
                                },
                                cellStyle: {
                                    ...GRAND_TOTAL_COLUMN_STYLE,
                                    ...this.valuesFormatting
                                }
                            },
                            dimensionLabelColumn: {
                                suppressHeaderMenuButton: true,
                                cellStyle: {
                                    ...this.columnSubheadersFormatting
                                }
                            },
                            groupAggregationColumn: {
                                suppressHeaderMenuButton: true,
                                cellStyle: {
                                    ...this.valuesFormatting
                                }
                            },
                            leafAggregationColumn: {
                                suppressHeaderMenuButton: true,
                                cellRenderer: 'cellWithTooltipRenderer',
                                cellRendererParams: {
                                    tooltips: this.tooltips,
                                    contextualMenu: this.contextualMenu,
                                    rowDimensionId: this.rowDimensionId,
                                    colorGroupByMeasureId: this.colorGroupByMeasureId,
                                    theme: this.chartHandler.getChartTheme(),
                                    chartDef: this.chartDef
                                },
                                cellStyle: {
                                    ...LEAF_ROW_STYLE,
                                    ...this.valuesFormatting
                                }
                            },
                            measureLabelColumn: {
                                suppressHeaderMenuButton: true,
                                cellRenderer: 'cellWithTooltipRenderer',
                                cellRendererParams: {
                                    tooltips: this.tooltips,
                                    contextualMenu: this.contextualMenu,
                                    rowDimensionId: this.rowDimensionId,
                                    colorGroupByMeasureId: this.colorGroupByMeasureId,
                                    canHideHeaders: this.yDimensions.length <= 1,
                                    rowHeadersCallbacks: PivotTableUtils.getToggleRowHeadersCallbacks(this.chartDef, this.chartHandler.openSection),
                                    theme: this.chartHandler.getChartTheme(),
                                    chartDef: this.chartDef
                                },
                                cellStyle: {
                                    ...MEASURE_COLUMN_STYLE,
                                    ...this.rowSubheadersFormatting
                                }
                            }
                        },
                        suppressContextMenu: true,
                        getRowId: (row) => {
                            if (row != null && row.data != null) {
                                return row.data.id;
                            }
                        },
                        sideBar: {
                            toolPanels: [
                                {
                                    id: ExpandCollapseToolPanel.id,
                                    labelDefault: ExpandCollapseToolPanel.label,
                                    labelKey: ExpandCollapseToolPanel.id,
                                    iconKey: '',
                                    toolPanel: ExpandCollapseToolPanel,
                                    toolPanelParams: {
                                        $rootElement: this.$rootElement,
                                        chartDef: this.chartDef
                                    }
                                },
                                {
                                    id: DisplayTotalsToolPanel.id,
                                    labelDefault: DisplayTotalsToolPanel.label,
                                    labelKey: DisplayTotalsToolPanel.id,
                                    iconKey: '',
                                    toolPanel: DisplayTotalsToolPanel,
                                    toolPanelParams: {
                                        $rootElement: this.$rootElement,
                                        pivotDisplayTotals: this.chartDef.pivotTableOptions.displayTotals,
                                        hasSubtotalRows: this.yDimensions.length > 1,
                                        hasSubtotalColumns: this.xDimensions.length > 1,
                                        hasGrandTotalRow: grandTotalRowData !== undefined && !ChartFeatures.isPivotTableWithNoDimension(this.chartDef), // can't disable grand total row when using pivot table with no dimension
                                        hasGrandTotalColumn: grandTotalColumnDef !== undefined,
                                        subtotalColumnType: 'subtotalColumn',
                                        rowData,
                                        subtotalRowData,
                                        grandTotalRowData,
                                        grandTotalColumnDef,
                                        recomputeColorScale: () => {
                                            const colorProperties = this.computeColorTableProperties(this.chartDef.colorOptions);
                                            this.colorScale = colorProperties.colorScale;
                                        }

                                    }
                                }
                            ]
                        },
                        getDataPath: row => row[this.getRowGroupKey()],
                        autoGroupColumnDef: {
                            // To force the tree data to work with groupHideOpenParents option
                            showRowGroup: '',
                            headerName: this.chartDef.pivotTableOptions.tableFormatting.showRowMainHeaders ? ChartLabels.getDimensionLabel(this.yDimensions[0]) : '',
                            headerComponentParams: { formatting: this.rowMainHeadersFormatting, isRowMainHeader: true },
                            lockPosition: true,
                            cellRendererParams: {
                                suppressCount: true,
                                suppressPadding: true,
                                tooltips: this.tooltips,
                                contextualMenu: this.contextualMenu,
                                innerRenderer: 'cellWithChevronRenderer'
                            },
                            cellStyle: this.rowSubheadersFormatting,
                            width: this.getRowDimensionsColumnWidth(
                                AUTOGROUP_COLUMN_KEY,
                                ChartLabels.getDimensionLabel(this.yDimensions[0]),
                                fullRowData.map(row => {
                                    const autoGroupColumnCellContent = this.switchRowPathIdToAutoGroupColumnCellContent(row);
                                    return autoGroupColumnCellContent ? autoGroupColumnCellContent.label : '';
                                }),
                                this.yDimensions.length > 1 ? CHEVRON_OFFSET_WIDTH + ROW_OFFSET_WIDTH : ROW_OFFSET_WIDTH
                            ),
                            pinned: this.chartDef.pivotTableOptions.tableFormatting.freezeRowHeaders ? 'left' : null,
                            valueGetter: (params) => {
                                return this.switchRowPathIdToAutoGroupColumnCellContent(params.data);
                            }
                        },
                        components: {
                            cellWithTooltipRenderer: CellWithTooltipRenderer,
                            cellWithChevronRenderer: (params) => this.cellWithChevronRenderer(params),
                            agColumnGroupHeader: CustomHeader,
                            agColumnHeader: CustomHeader
                        },
                        onRowGroupOpened: (e) => {
                            return this.toggleSubTotalDisplayOnParentRow(e);
                        },
                        getRowStyle: (params) => {
                            return this.addSubtotalRowStyle(params);
                        },
                        onColumnResized: (params) => {
                            if (_.isNil(params.columns) || !params.finished || params.source !== 'uiColumnResized') {
                                return;
                            }
                            params.columns.forEach(column => {
                                const newCustomWidth = parseInt(column.getActualWidth());
                                const colId = column.getId();
                                if (!_.isNil(colId) && !_.isNil(newCustomWidth) &&
                                    this.chartDef.pivotTableOptions.columnIdByCustomWidth[colId] !== newCustomWidth) {
                                    this.chartDef.pivotTableOptions.columnIdByCustomWidth[colId] = newCustomWidth;
                                }
                            });
                        },
                        ensureDomOrder: true,
                        sendToClipboard: () => {
                            ClipboardUtils.copyToClipboard((getSelection() || '').toString());
                        },
                        // Hide unfoleded parents. It should not work with TreeData, but it does anyway.
                        groupHideOpenParents: true
                    };
                }

                /**
                 * Compute pivot table color properties.
                 * @param   {ChartDef['colorOptions']} colorOptions
                 * @returns {{hasColorMeasure: boolean, colorMeasureIndex: number, colorScale: colorScale}}
                 */
                computeColorTableProperties(colorOptions, colorMeasureIndex = -1, measure = undefined) {
                    let hasColorMeasure = false;
                    let colorScale;
                    let extentFormatter;

                    if (this.data.aggregations.length > 0 && (this.chartDef.colorMeasure.length > 0 || colorMeasureIndex >= 0) && !ChartFeatures.isPivotTableWithNoDimension(this.chartDef)) {
                        hasColorMeasure = true;
                        if (colorMeasureIndex < 0) { // for unique scale
                            colorMeasureIndex = this.data.aggregations.length - 1;
                        } else {
                            // get formatter for min and max of colorGroups
                            extentFormatter = ChartFormatting.getForIsolatedNumber(measure);
                        }
                        const yDimensionIds = this.yDimensions.map(yDim => this.chartStore.getDimensionId(yDim));
                        const xDimensionIds = this.xDimensions.map(xDim => this.chartStore.getDimensionId(xDim));
                        /** @type {ChartColorContext} */
                        const colorContext = {
                            chartData: this.chartData,
                            colorOptions: colorOptions,
                            defaultLegendDimension: this.chartDef.genericMeasures,
                            colorSpec: {
                                type: 'MEASURE',
                                measureIdx: colorMeasureIndex,
                                withRgba: true,
                                colorAggrFn: HierarchicalChartsUtils.getColorMeasureAggFunction(this.chartDef, colorMeasureIndex),
                                binsToInclude: HierarchicalChartsUtils.getBinsToIncludeInColorScale(
                                    this.chartData,
                                    xDimensionIds,
                                    yDimensionIds,
                                    colorMeasureIndex,
                                    this.chartDef,
                                    this.chartDef.pivotTableOptions.displayTotals.subTotals.rows,
                                    this.chartDef.pivotTableOptions.displayTotals.subTotals.columns
                                )
                            },
                            theme: this.chartHandler.getChartTheme()
                        };
                        colorScale = ChartColorScales.createColorScale(colorContext);
                        colorScale.extentFormatter = extentFormatter;

                        ChartLegendUtils.initLegend(this.chartDef, this.chartData, this.chartHandler.legendsWrapper, colorScale);
                        this.chartDef.colorMeasure['$mIdx'] = colorMeasureIndex;
                    }
                    return { hasColorMeasure, colorMeasureIndex, colorScale };
                }

                /**
                 * Build AG Grid Column definitions filtering empty columns
                 * Explore column tree using BFS algorithm with a queue.
                 * @param       {AGGridConverterDFSColumnData[]} columnTree
                 * @return      {{columnDefs:(ColDef | ColGroupDef)[], grandTotalColumnDef?: ColDef | ColGroupDef}} column tree filtered
                 * @memberof    AbstractAGGridPivotTableChartConverter
                 */
                filterColumnTreeAndConvertToColumnDefs(columnTree) {
                    /** @type {((ColDef | ColGroupDef)[])} */
                    const columnDefs = [];

                    const queue = new LinkedList(columnTree.map(column => column));
                    while (queue.length) {
                        const curr = queue.shift();
                        if (!curr.isEmpty) {
                            if (curr.parent === null) {
                                columnDefs.push(curr.agGridColumn);
                            } else {
                                if (curr.parent.agGridColumn.children === undefined) {
                                    curr.parent.agGridColumn.children = [];
                                }
                                curr.parent.agGridColumn.children.push(curr.agGridColumn);
                            }
                            if (curr.children && curr.children.length) {
                                curr.children.forEach(child => {
                                    if (!child.isEmpty) {
                                        queue.push(child);
                                    }
                                });
                            }
                        }
                    }
                    if (this.hasGrandTotalColumn()) {
                        return { columnDefs, grandTotalColumnDef: this.getGrandTotalColumn() };
                    } else {
                        return { columnDefs };
                    }
                }

                /**
                 * Add style to subtotal rows.
                 * @param       {RowClassParams} params
                 * @return      {RowStyle}
                 * @memberof    AbstractAGGridPivotTableChartConverter
                 */
                addSubtotalRowStyle(params) {
                    const isSubtotalRow = params.data[COLOR_FLAG_KEY];
                    const rowPath = params.data[this.getRowGroupKey()];
                    let isGrandTotalRow = false;
                    if (rowPath && rowPath.length > 0) {
                        isGrandTotalRow = rowPath[0] === GRAND_TOTAL_KEY;
                    }
                    if (isGrandTotalRow) {
                        return {
                            ...GRAND_TOTAL_ROW_STYLE,
                            ...this.valuesFormatting
                        };
                    } else if (isSubtotalRow) {
                        return {
                            ...SUB_TOTAL_ROW_STYLE,
                            ...this.valuesFormatting
                        };
                    }
                }

                /**
                 * Returns the row header label corresponding to a particular row path
                 * @param       {any}                                   row
                 * @return      {PivotTableRowTreeCellContent}          the row header label corresponding to a particular row path
                 * @memberof    AbstractAGGridPivotTableChartConverter
                 */
                switchRowPathIdToAutoGroupColumnCellContent(row) {
                    const nodePath = row[this.getRowGroupKey()];
                    if (this.mapGroupColumnCellIdToHeaderName.has(nodePath)) {
                        return this.mapGroupColumnCellIdToHeaderName.get(nodePath);
                    }
                }

                /**
                 * Toggle display of subtotal when expand/collapsing a grouping row
                 * @param       {RowGroupOpenedEvent} e
                 * @memberof    AbstractAGGridPivotTableChartConverter
                 */
                toggleSubTotalDisplayOnParentRow(e) {
                    for (const key in e.node.data) {
                        if (Object.hasOwnProperty.call(e.node.data, key)) {
                            if (typeof e.node.data[key] === 'object') {
                                e.node.data[key].hidden = e.node.expanded;
                            }
                        }
                    }
                    e.node.setData(e.node.data);
                }

                /**
                 * Build AG Grid row flat tree using DFS (Depth First Seach) using a stack.
                 * It returns a flat table where the tree structure is defined by the rowPath.
                 * @return      {AGGridConverterDFSRowData[]}
                 * @memberof    AbstractAGGridPivotTableChartConverter
                 */
                buildRowTree() {
                    const [head, ...tail] = this.yDimensions;
                    const headIndex = 0;
                    /** @type {AGGridConverterDFSRowData[]} datastructure to navigate the tree*/
                    const stack = [];
                    /** @type {AGGridConverterDFSRowData[]} datastructure that will conserve the copy of the tree */
                    const rowTree = [];
                    const dimensionId = this.chartStore.getDimensionId(head);
                    const axisLabels = this.chartData.getAxisLabels(dimensionId);
                    const minValue = this.chartData.getMinValue(dimensionId);
                    const maxValue = this.chartData.getMaxValue(dimensionId);
                    const numValues = this.chartData.getNumValues(dimensionId);
                    const numberFormattingOptions = ChartDimension.getNumberFormattingOptions(head);

                    for (let i = 0; i < axisLabels.length; i++) {
                        const axisLabel = axisLabels[i];
                        if (i !== this.chartData.getSubtotalLabelIndex(dimensionId)) {
                            /** @type {Record<string, number>} */
                            const yCoordDict = {};
                            yCoordDict[dimensionId] = i;
                            const rowData = {
                                rowPath: [dimensionId + SUB_COLUMN_SEPARATOR + axisLabel.label],
                                parent: null,
                                children: [],
                                isFirstChild: i === 0,
                                depth: 0,
                                headerName: ChartLabels.getFormattedLabel(axisLabel, numberFormattingOptions, minValue, maxValue, numValues),
                                isEmpty: true,
                                yCoordDict,
                                head,
                                tail,
                                headIndex
                            };

                            rowTree.push(rowData);
                            stack.push(rowData);
                        }
                    }

                    while (stack.length !== 0) {
                        const curr = stack.pop();
                        curr.agGridRow = { id: curr.rowPath.join(SUB_COLUMN_SEPARATOR) };
                        this.fillRowNavigationColumnsInRow(curr);

                        if (curr.tail && curr.tail.length > 0) {
                            this.fillSubTotalsInRow(curr, true);
                            this.fillGrandTotalsInRow(curr);

                            const [childHead, ...childTail] = curr.tail;
                            const childHeadIndex = curr.headIndex + 1;
                            const childDimensionId = this.chartStore.getDimensionId(childHead);
                            const childValues = this.chartData.getAxisLabels(childDimensionId);
                            const childMinValue = this.chartData.getMinValue(childDimensionId);
                            const childMaxValue = this.chartData.getMaxValue(childDimensionId);
                            const childNumValues = this.chartData.getNumValues(childDimensionId);
                            const childNumberFormattingOptions = ChartDimension.getNumberFormattingOptions(childHead);

                            for (let j = 0; j < childValues.length; j++) {
                                const childValue = childValues[j];
                                if (j !== this.chartData.getSubtotalLabelIndex(childDimensionId)) {
                                    /** @type {Record<string, number>} */
                                    const childYCoordDict = { ...curr.yCoordDict };
                                    childYCoordDict[childDimensionId] = j;
                                    const childRowData = {
                                        rowPath: [...curr.rowPath, childDimensionId + SUB_COLUMN_SEPARATOR + childValue.label],
                                        parent: curr,
                                        children: [],
                                        yCoordDict: childYCoordDict,
                                        tail: childTail,
                                        head: childHead,
                                        headIndex: childHeadIndex,
                                        headerName: ChartLabels.getFormattedLabel(childValue, childNumberFormattingOptions, childMinValue, childMaxValue, childNumValues),
                                        isFirstChild: j === 0,
                                        depth: curr.depth + 1,
                                        isEmpty: true
                                    };
                                    curr.children.push(childRowData);
                                    stack.push(childRowData);
                                }
                            }
                        } else {
                            const isEmpty = this.fillMeasuresInRow(curr);
                            if (!isEmpty) {
                                this.fillSubTotalsInRow(curr, false);
                                this.fillGrandTotalsInRow(curr);
                            }
                        }
                    }

                    return rowTree;
                }

                /**
                 * @param       {AGGridConverterDFSRowData} rowData
                 * @memberof    AbstractAGGridPivotTableChartConverter
                 */
                setNotEmptyFlagsToRowBranch(rowData) {
                    rowData.isEmpty = false;
                    let parent = rowData.parent;
                    while (parent && parent.isEmpty) {
                        parent.isEmpty = false;
                        parent = parent.parent;
                    }
                }

                /**
                 * @param       {AGGridConverterDFSColumnData} columnData
                 * @memberof    AbstractAGGridPivotTableChartConverter
                 */
                setNotEmptyFlagsToColumnBranch(columnData) {
                    columnData.isEmpty = false;
                    let parent = columnData.parent;
                    while (parent && parent.isEmpty) {
                        parent.isEmpty = false;
                        parent = parent.parent;
                    }
                }

                /**
                 * Build AG Grid row flat tree using DFS (Depth First Seach) using a stack.
                 * It returns a flat table where the tree structure is defined by the rowPath.
                 * @param       {AGGridConverterDFSRowData[]} rowTree
                 * @return      {{rowData: AgGridRow[], subtotalRowData: AgGridRow[], grandTotalRowData?: AgGridRow}}
                 * @memberof    AbstractAGGridPivotTableChartConverter
                 */
                filterRowTreeAndConvertToRowData(rowTree) {
                    /** @type {AgGridRow[]} */
                    const rowData = [];
                    /** @type {AgGridRow[]} */
                    const subtotalRowData = [];
                    /** @type {AGGridConverterDFSRowData[]} datastructure to navigate the tree*/
                    const stack = [...rowTree.reverse()]; // Reverse because of the stack
                    while (stack.length) {
                        const curr = stack.pop();
                        if (!curr.isEmpty) {
                            rowData.push(curr.agGridRow);
                            this.fillRowNavigationColumnsInRow(curr);
                            if (curr.children.length) {
                                let hasFoundFirstNotEmpty = false;
                                curr.children.map(child => {
                                    if (!child.isEmpty) {
                                        if (!hasFoundFirstNotEmpty) {
                                            child.isFirstChild = true;
                                            hasFoundFirstNotEmpty = true;
                                        }
                                        stack.unshift(child);
                                    }
                                });
                                subtotalRowData.push(this.getSubtotalLeafRow(curr));
                            }
                        }
                    }
                    if (this.hasGrandTotalRow()) {
                        return { rowData, subtotalRowData, grandTotalRowData: this.getGrandTotalRow() };
                    } else {
                        return { rowData, subtotalRowData };
                    }
                }

                /**
                 * Add the measures in AgGrid Row tree.
                 * @param       {AGGridConverterDFSRowData} currentRow
                 * @memberof    AbstractAGGridPivotTableChartConverter
                 */
                fillRowNavigationColumnsInRow(currentRow) {
                    const isSubtotalRow = currentRow.rowPath[currentRow.rowPath.length - 1] === SUBTOTAL_LEAF_ROW_KEY;
                    if (isSubtotalRow) {
                        return;
                    }
                    currentRow.agGridRow[this.getRowGroupKey()] = currentRow.rowPath;
                    const cellContent = { label: currentRow.headerName, depth: 0 };
                    if (this.yDimensions[0] === currentRow.head) {
                        this.mapGroupColumnCellIdToHeaderName.set(currentRow.rowPath, cellContent);
                    } else {
                        currentRow.agGridRow[this.chartStore.getDimensionId(currentRow.head)] = cellContent;
                    }
                    if (currentRow.isFirstChild && currentRow.headIndex > 0) {
                        let currParent = currentRow.parent;
                        let currDepth = 1;
                        let currRowDimIndex = currentRow.headIndex - 1;
                        let currDimensionId = this.chartStore.getDimensionId(this.yDimensions[currRowDimIndex]);
                        while (currRowDimIndex >= 0 && currParent && currParent.agGridRow) {
                            if (currRowDimIndex === 0 && this.mapGroupColumnCellIdToHeaderName.has(currParent.rowPath)) {
                                const parentCellContent = this.mapGroupColumnCellIdToHeaderName.get(currParent.rowPath);
                                this.mapGroupColumnCellIdToHeaderName.set(currentRow.rowPath, { label: parentCellContent.label, depth: currDepth });
                            } else {
                                currentRow.agGridRow[currDimensionId] = { label: currParent.agGridRow[currDimensionId].label, depth: currDepth };
                            }
                            --currRowDimIndex;
                            currDepth++;
                            if (currParent && !currParent.isFirstChild) {
                                break;
                            }
                            currParent = currParent.parent;
                            currDimensionId = this.chartStore.getDimensionId(this.yDimensions[currRowDimIndex]);
                        }
                    }
                }

                /**
                 * Add the measures in AgGRid Row tree.
                 * @param       {AGGridConverterDFSRowData} currentRow
                 * @return      {boolean}                   true if is empty.
                 * @memberof    AbstractAGGridPivotTableChartConverter
                 */
                fillMeasuresInRow(currentRow) {
                    throw new Error('Method not implemented');
                }

                /**
                 * Returns a leaf row with the subtotals contained in the parent row.
                 * @param       {AGGridConverterDFSRowData} currentRow
                 * @return      {AgGridRow}                 sub-total leaf row
                 * @memberof    AbstractAGGridPivotTableChartConverter
                 */
                getSubtotalLeafRow(currentRow) {
                    const deepCopy = Object.keys(currentRow.agGridRow).filter(colId => !this.rowDimensionId.has(colId)).reduce((acc, key) => {
                        if (typeof currentRow.agGridRow[key] === 'object') {
                            // @ts-ignore
                            acc[key] = { ...currentRow.agGridRow[key] };
                        } else {
                            acc[key] = currentRow.agGridRow[key];
                        }
                        return acc;
                    }, {});
                    /** @type {AgGridRow} */
                    const aggridSubtotalLeafRow = deepCopy;
                    const parentRowPath = currentRow.agGridRow[this.getRowGroupKey()];
                    // @ts-ignore
                    const subtotalLeafRowPath = [...parentRowPath, SUBTOTAL_LEAF_ROW_KEY];
                    aggridSubtotalLeafRow[this.getRowGroupKey()] = subtotalLeafRowPath;
                    aggridSubtotalLeafRow.id = subtotalLeafRowPath.join(SUB_COLUMN_SEPARATOR);
                    if (currentRow.head === this.yDimensions[0]) {
                        this.mapGroupColumnCellIdToHeaderName.set(subtotalLeafRowPath, { label: this.chartDef.pivotTableOptions.tableFormatting.showRowHeaders ? 'Total ' + currentRow.headerName : '', depth: 0, hasOffset: true });
                    } else {
                        aggridSubtotalLeafRow[this.chartStore.getDimensionId(currentRow.head)] = { label: this.chartDef.pivotTableOptions.tableFormatting.showRowHeaders ? 'Total ' + currentRow.headerName : '', depth: 0, hasOffset: true };
                    }
                    currentRow.agGridRow[COLOR_FLAG_KEY] = true;
                    return aggridSubtotalLeafRow;
                }

                /**
                 * Fill the subtotals of a row.
                 * @param       {AGGridConverterDFSRowData}             currrentRow
                 * @param       {boolean}                               isGroupRow
                 * @memberof    AbstractAGGridPivotTableChartConverter
                 */
                fillSubTotalsInRow(currrentRow, isGroupRow) {
                    throw new Error('Method not implemented');
                }

                /**
                 * Fill the grand totals inside the row grand total column.
                 * @param       {AGGridConverterDFSRowData}         currrentRow
                 * @memberof    AbstractAGGridPivotTableChartConverter
                 */
                fillGrandTotalsInRow(currrentRow) {
                    throw new Error('Method not implemented');
                }

                /**
                 * Returns true if has Grand Total row
                 * @return      {boolean} true if has Grand Total row
                 * @memberof    AbstractAGGridPivotTableChartConverter
                 */
                hasGrandTotalRow() {
                    return this.chartDef.genericMeasures.filter(measure => measure.computeMode === 'NORMAL').length > 0;
                }

                /**
                 * Returns Grand Total row
                 * @return      {AgGridRow} agGridRowTree
                 * @memberof    AbstractAGGridPivotTableChartConverter
                 */
                getGrandTotalRow() {
                    throw new Error('Method not implemented');
                }

                /**
                 * Build Column tree iteratively using DFS (Depth First Search) with a stack.
                 * @returns     {AGGridConverterDFSColumnData[]} Columns
                 * @memberof    AbstractAGGridPivotTableChartConverter
                 */
                buildColumnTree() {

                    /** @type {AGGridConverterDFSColumnData[]} data structure that will conserve the copy of the tree */
                    const columnTree = [...this.getRowNavigationColumns()];

                    const [head, ...tail] = this.xDimensions;
                    const headIndex = 0;
                    /** @type {AGGridConverterDFSColumnData[]} */
                    const stack = [];
                    const isMeasureTreatedAsColumn = this.chartDef.pivotTableOptions.measureDisplayMode !== ChartsStaticData.pivotTableMeasureDisplayMode.ROWS && PivotTableUtils.hasRows(this.chartDef) && PivotTableUtils.hasColumns(this.chartDef);

                    // Add parent column
                    const parentDimensionId = this.chartStore.getDimensionId(head);
                    const minValue = this.chartData.getMinValue(parentDimensionId);
                    const maxValue = this.chartData.getMaxValue(parentDimensionId);
                    const numValues = this.chartData.getNumValues(parentDimensionId);
                    const numberFormattingOptions = ChartDimension.getNumberFormattingOptions(head);
                    const showColumnHeaders = this.chartDef.pivotTableOptions.tableFormatting.showColumnHeaders;
                    const showColumnMainHeaders = showColumnHeaders && this.chartDef.pivotTableOptions.tableFormatting.showColumnMainHeaders;

                    const dimensionLabelHeaderName = ChartLabels.getDimensionLabel(head);
                    const dimensionLabelColumnData = {
                        agGridColumn: {
                            groupId: parentDimensionId,
                            type: 'dimensionLabelColumn',
                            headerName: dimensionLabelHeaderName,
                            headerGroupComponentParams: { formatting: this.columnMainHeadersFormatting, isColumnMainHeader: true },
                            columnGroupShow: 'open',
                            children: [],
                            width: this.computeColumnWidth(dimensionLabelHeaderName, this.columnMainHeadersFormatting)
                        },
                        parent: null,
                        isEmpty: false,
                        children: []
                    };

                    if (showColumnMainHeaders) {
                        columnTree.push(dimensionLabelColumnData);
                    }

                    const axisLabels = this.chartData.getAxisLabels(parentDimensionId);
                    const headerComponentConfig = {
                        headerComponentParams: { formatting: this.columnSubheadersFormatting }
                    };

                    for (let i = 0; i < axisLabels.length; i++) {
                        const axisLabel = axisLabels[i];
                        if (i !== this.chartData.getSubtotalLabelIndex(parentDimensionId)) {
                            const headerName = this.chartDef.pivotTableOptions.tableFormatting.showColumnHeaders ? ChartLabels.getFormattedLabel(axisLabel, numberFormattingOptions, minValue, maxValue, numValues) : '';
                            /** @type {Record<string, number>} */
                            const xCoordDict = {};
                            xCoordDict[parentDimensionId] = i;
                            const groupId = `${dimensionLabelColumnData.agGridColumn.groupId}-${xCoordDict[parentDimensionId]}`;
                            const columnData = {
                                parent: showColumnMainHeaders ? dimensionLabelColumnData : null,
                                isEmpty: true,
                                xCoordDict,
                                head,
                                tail,
                                headIndex,
                                agGridColumn: {
                                    groupId,
                                    headerName: headerName,
                                    columnGroupShow: 'open',
                                    openByDefault: this.computeColumnExpandedStatus(groupId),
                                    type: 'groupAggregationColumn',
                                    width: this.computeColumnWidth(headerName, this.columnMainHeadersFormatting, !!tail.length)
                                },
                                children: []
                            };
                            const isLeafColumn = !isMeasureTreatedAsColumn && tail.length === 0;
                            if (isLeafColumn) {
                                columnData.agGridColumn = { ...columnData.agGridColumn, ...headerComponentConfig };
                            } else {
                                columnData.agGridColumn = { ...columnData.agGridColumn, headerGroupComponentParams: {
                                    ...headerComponentConfig.headerComponentParams,
                                    hasChevron: !!tail.length,
                                    chartDef: this.chartDef
                                } };
                            };
                            stack.push(columnData);
                            if (showColumnMainHeaders) {
                                // Dimension label is the root node
                                dimensionLabelColumnData.children.push(columnData);
                            } else {
                                // Dimension values are the root nodes
                                columnTree.push(columnData);
                            }
                        }
                    }

                    while (stack.length !== 0) {
                        const curr = stack.pop();
                        const dimensionId = this.chartStore.getDimensionId(curr.head);

                        if (curr.tail && curr.tail.length > 0) {
                            curr.agGridColumn.cellStyle = SUBTOTAL_COLUMN_STYLE;

                            const [childHead, ...childTail] = curr.tail;
                            const childHeadIndex = curr.headIndex + 1;
                            const childDimensionId = this.chartStore.getDimensionId(childHead);
                            const childMinValue = this.chartData.getMinValue(childDimensionId);
                            const childMaxValue = this.chartData.getMaxValue(childDimensionId);
                            const childNumValues = this.chartData.getNumValues(childDimensionId);
                            const childNumberFormattingOptions = ChartDimension.getNumberFormattingOptions(childHead);
                            const childValues = this.chartData.getAxisLabels(childDimensionId);

                            const subDimensionLabelHeaderName = ChartLabels.getDimensionLabel(childHead);
                            const groupId = curr.agGridColumn.groupId + SUB_COLUMN_SEPARATOR + childDimensionId;
                            const subDimensionLabelColumnData = {
                                agGridColumn: {
                                    groupId,
                                    headerName: subDimensionLabelHeaderName,
                                    type: 'dimensionLabelColumn',
                                    columnGroupShow: 'open',
                                    children: [],
                                    width: this.computeColumnWidth(subDimensionLabelHeaderName, this.columnMainHeadersFormatting),
                                    headerGroupComponentParams: { formatting: this.columnMainHeadersFormatting, isColumnMainHeader: true }
                                },
                                parent: showColumnMainHeaders ? curr : null,
                                isEmpty: true,
                                xCoordDict: curr.xCoordDict,
                                head: curr.head,
                                tail: curr.tail,
                                headIndex: childHeadIndex,
                                children: []
                            };

                            if (showColumnHeaders && showColumnMainHeaders) {
                                curr.children.push(subDimensionLabelColumnData);
                            }

                            for (let j = 0; j < childValues.length; j++) {
                                const childValue = childValues[j];
                                if (j !== this.chartData.getSubtotalLabelIndex(childDimensionId)) {
                                    const childXCoordDict = { ...curr.xCoordDict };
                                    childXCoordDict[childDimensionId] = j;
                                    const childHeaderName = showColumnHeaders ? ChartLabels.getFormattedLabel(childValue, childNumberFormattingOptions, childMinValue, childMaxValue, childNumValues) : '';
                                    const groupId = `${curr.agGridColumn.groupId}-${childXCoordDict[childDimensionId]}`;
                                    const childColumnData = {
                                        isEmpty: true,
                                        parent: (showColumnHeaders && showColumnMainHeaders) ? subDimensionLabelColumnData : curr,
                                        xCoordDict: childXCoordDict,
                                        head: childHead,
                                        tail: childTail,
                                        headIndex: childHeadIndex,
                                        agGridColumn: {
                                            groupId,
                                            headerName: childHeaderName,
                                            columnGroupShow: 'open',
                                            openByDefault: this.computeColumnExpandedStatus(groupId),
                                            type: 'groupAggregationColumn',
                                            width: this.computeColumnWidth(childHeaderName, this.columnSubheadersFormatting,!!childTail.length)
                                        },
                                        children: []
                                    };

                                    const isLeafColumn = !isMeasureTreatedAsColumn && childTail.length === 0;

                                    if (isLeafColumn) {
                                        childColumnData.agGridColumn = { ...childColumnData.agGridColumn, ...headerComponentConfig };
                                    } else {
                                        childColumnData.agGridColumn = { ...childColumnData.agGridColumn, headerGroupComponentParams: {
                                            formatting: this.columnSubheadersFormatting,
                                            hasChevron: !!childTail.length,
                                            chartDef: this.chartDef
                                        } };
                                    };

                                    if (showColumnHeaders && showColumnMainHeaders) {
                                        // Dimension label is the parent node
                                        subDimensionLabelColumnData.children.push(childColumnData);
                                    } else {
                                        // Previous dimension values are the parent nodes
                                        curr.children.push(childColumnData);
                                    }

                                    stack.push(childColumnData);
                                }
                            }
                            this.addSubTotalParentColumnsInColumnTree(curr, dimensionId);
                        } else {
                            this.addMeasuresInColumnTree(curr);
                        }
                    }
                    return columnTree;
                }

                /**
                 * Add row tree columns to display dimension values set as rows
                 * @return      {AGGridConverterDFSColumnData[]} agGridColumns
                 * @memberof    AbstractAGGridPivotTableChartConverter
                 */
                getRowNavigationColumns() {
                    const columns = [];

                    this.yDimensions.map((dimension, i) => {
                        // The first dimension will be skipped as it is the group column
                        if (i > 0) {
                            const headerName = this.chartDef.pivotTableOptions.tableFormatting.showRowMainHeaders ? ChartLabels.getDimensionLabel(dimension) : '';
                            const dimensionId = this.chartStore.getDimensionId(dimension);

                            const numberFormattingOptions = ChartDimension.getNumberFormattingOptions(dimension);
                            const minValue = this.chartData.getMinValue(dimensionId);
                            const maxValue = this.chartData.getMaxValue(dimensionId);
                            const numValues = this.chartData.getNumValues(dimensionId);
                            const labels = this.chartData.getAxisLabels(dimensionId)
                                .filter((_, i) => i !== this.chartData.getSubtotalLabelIndex(dimensionId))
                                .map(labelObj => ChartLabels.getFormattedLabel(labelObj, numberFormattingOptions, minValue, maxValue, numValues));
                            const isLast = i === this.yDimensions.length - 1;
                            const colId = dimensionId;
                            const width = this.getRowDimensionsColumnWidth(colId, headerName, labels, isLast ? 0 : CHEVRON_OFFSET_WIDTH);
                            columns.push({
                                parent: null,
                                isEmpty: false,
                                agGridColumn: {
                                    colId,
                                    field: colId,
                                    headerName,
                                    headerComponentParams: { formatting: this.rowMainHeadersFormatting, isRowMainHeader: true },
                                    type: 'rowNavigationColumn',
                                    width,
                                    lockPosition: true,
                                    pinned: this.chartDef.pivotTableOptions.tableFormatting.freezeRowHeaders ? 'left' : null
                                }
                            });
                        }
                    });
                    return columns;
                }

                /**
                 * Fulfills endColumns datastructure to find back aggregation position in tensor.
                 * If measures are treated as columns, it add sub-columns.
                 * @param       {AGGridConverterDFSColumnData}  currColumn
                 * @memberof    AbstractAGGridPivotTableChartConverter
                 */
                addMeasuresInColumnTree(currColumn) {
                    throw new Error('Method not implemented');
                }

                /**
                 * Adds the Sub-total parent columns. The column(s) appearing when the column-node is collapsed.
                 * @param       {AGGridConverterDFSColumnData}  currColumn
                 * @param       {string}                        dimensionId
                 * @memberof    AbstractAGGridPivotTableChartConverter
                 */
                addSubTotalParentColumnsInColumnTree(currColumn, dimensionId) {
                    throw new Error('Method not implemented');
                }

                /**
                 * Returns true if has Grand Total column
                 * @return      {boolean} true if has Grand Total column
                 * @memberof    AbstractAGGridPivotTableChartConverter
                 */
                hasGrandTotalColumn() {
                    return this.chartDef.genericMeasures.filter(measure => measure.computeMode === 'NORMAL').length > 0;
                }

                /**
                 * Returns Grand Total column
                 * @return      {ColGroupDef | ColDef} grand total column
                 * @memberof    AbstractAGGridPivotTableChartConverter
                 */
                getGrandTotalColumn() {
                    throw new Error('Method not implemented');
                }

                /**
                 * Returns the ID of leaf columns
                 * @param       {string}        parentColumnId
                 * @param       {string}        measureId
                 * @returns     {string}        leaf column ID
                 * @memberof    AbstractAGGridPivotTableChartConverter
                 */
                getLeafColumnId(parentColumnId, measureId) {
                    return parentColumnId === '' ? measureId : parentColumnId + SUB_COLUMN_SEPARATOR + measureId;
                }

                /**
                 * Returns measure label
                 * @param       {MeasureDef}    measure
                 * @returns     {string}        measure label
                 * @memberof    AbstractAGGridPivotTableChartConverter
                 */
                getMeasureLabel(measure) {
                    return ChartLabels.getLongMeasureLabel(measure);
                }

                /**
                 * Sub-column separator to build nested column ID
                 * @return      {string}
                 * @memberof    AbstractAGGridPivotTableChartConverter
                 */
                getRowGroupKey() {
                    return this.chartStore.getDimensionId(this.yDimensions[0]);
                }

                /**
                 * Returns row height
                 * @memberof AbstractAGGridPivotTableChartConverter
                 */
                getRowHeight() {
                    return DEFAULT_ROW_HEIGHT;
                }

                getHeaderHeight() {
                    const showColumnHeaders = this.chartDef.pivotTableOptions.tableFormatting.showColumnHeaders;
                    const showRowHeaders = this.chartDef.pivotTableOptions.tableFormatting.showRowHeaders;
                    const showRowMainHeaders = this.chartDef.pivotTableOptions.tableFormatting.showRowMainHeaders;
                    const noRowMainHeaders = !showRowHeaders || !showRowMainHeaders;
                    const noColumnMainHeaders = !this.chartDef.pivotTableOptions.tableFormatting.showColumnMainHeaders;
                    const noRowDimension = this.yDimensions.length === 0;
                    const noColumnDimension = this.xDimensions.length === 0;
                    const onlyOneMeasure = this.chartDef.genericMeasures.length === 1;

                    switch (showColumnHeaders) {
                        case false:
                            if ((noRowDimension && onlyOneMeasure) || noRowMainHeaders) {
                                return 0;
                            }
                            break;
                        case true:
                            if (noColumnDimension && noColumnMainHeaders && noRowMainHeaders) {
                                return 0;
                            }
                            break;
                    }
                }

                getGroupHeaderHeight() {
                    /*
                     * Prevent (n-1) empty rows to appear when hiding the column headers and there's n (>1) columns dimensions.
                     * The later still need to be included in the pivot to have the tree structure work, so they need to be hidden if the column headers are hidden but not the row headers.
                     */
                    if (!this.chartDef.pivotTableOptions.tableFormatting.showColumnHeaders && this.chartDef.pivotTableOptions.tableFormatting.showRowMainHeaders) {
                        return 0;
                    }
                }

                /**
                 * Compute cell content that will be passed to HTML Renderers
                 * @param       {number}                    aggregationIndex
                 * @param       {MeasureDef}                measure
                 * @param       {Record<string, number>?}   cellCoords
                 * @param       {boolean}                   isSubOrGrandTotal
                 * @return      {PivotTableCellMeasure}
                 * @memberof    AbstractAGGridPivotTableChartConverter
                 */
                computeCellValue(aggregationIndex, measure, cellCoords, isSubOrGrandTotal, hasTooltipAndContextualMenu = true) {
                    if (measure.computeMode !== 'NORMAL' && isSubOrGrandTotal) {
                        return {
                            value: 'NA',
                            originalValue: '',
                            cellCoords,
                            colorMeasureValue: '',
                            colorMeasureOriginalValue: '',
                            hasTooltipAndContextualMenu: false
                        };
                    }
                    let value = this.chartData.getSubtotalPoint(aggregationIndex, cellCoords);

                    let colorMeasureValue;
                    let colorMeasureOriginalValue;
                    const colorGroup = this.colorGroupByMeasureId ? this.colorGroupByMeasureId[ConditionalFormattingOptions.getMeasureId(measure)] : undefined;
                    const groupColorMeasure = colorGroup ? colorGroup.basedOnMeasure : undefined;
                    const groupColorMeasureAggregationIndex = colorGroup ? colorGroup.basedOnMeasureIndex : undefined;

                    // check if groupColorMeasureAggregationIndex is in aggregations, because the render happens before the new request
                    if (groupColorMeasure && groupColorMeasureAggregationIndex < this.chartData.data.aggregations.length) {
                        // do something like computeFormatterProperties for colorMeasure
                        const displayEmpty = this.chartDef.pivotTableOptions && this.chartDef.pivotTableOptions.displayEmptyValues;
                        const formatter = ChartFormatting.getForIsolatedNumber({ ...groupColorMeasure, displayEmpty });

                        colorMeasureOriginalValue = this.chartData.getSubtotalPoint(groupColorMeasureAggregationIndex, cellCoords);
                        if (!this.chartData.isBinMeaningful(colorGroup.basedOnMeasure.function, cellCoords, groupColorMeasureAggregationIndex)) {
                            colorMeasureOriginalValue = '';
                        }
                        colorMeasureValue = formatter(colorMeasureOriginalValue);
                    }

                    const count = isSubOrGrandTotal || measure.computeMode === 'DIFFERENCE' ? 1 : this.chartData.getNonNullCount(cellCoords, aggregationIndex);
                    const formattedValue = this.cellFormatters[aggregationIndex] ? this.cellFormatters[aggregationIndex](value, count) : value;

                    // set originalValue as empty string to avoid formatting with color rules empty cells
                    value = count > 0 ? value : '';
                    return { value: formattedValue, originalValue: value, colorMeasureValue, colorMeasureOriginalValue, cellCoords, hasTooltipAndContextualMenu };
                }

                /**
                 * Compute cell color that will be passed to HTML Renderers (when using unique scale)
                 * @param       {Record<string, number>?}            cellCoords
                 * @param       {boolean}                            isSubTotal
                 * @param       {number[]}                           aggrIdx indexes of aggregations displayed in this cell
                 * @return      {PivotTableCellContent['colorProperties'] | undefined}
                 * @memberof    AbstractAGGridPivotTableChartConverter
                 */
                computeCellColorProperties(cellCoords, isSubTotal = false, aggrIdx = []) {
                    let measure;
                    if (this.hasColorMeasure) {
                        if (isSubTotal) {
                            measure = this.chartData.getSubtotalPoint(this.colorMeasureIndex, cellCoords);
                        } else if (!PivotTableUtils.hasCellValueForCoords(this.chartData, aggrIdx, cellCoords, this.chartDef.pivotTableOptions.displayEmptyValues)) {
                            return undefined;
                        } else {
                            measure = this.chartData.getAggrPoint(this.colorMeasureIndex, cellCoords);
                        }
                        return { getColorScale: () => this.colorScale, measure, bin: this.chartData.getAggrLoc(this.colorMeasureIndex, cellCoords) };
                    } else {
                        return undefined;
                    }
                }

                computeColumnExpandedStatus(columnGroupId) {
                    return !_.isNil(this.chartDef.pivotTableOptions.columnIdByCustomExpandedStatus) && !_.isNil(this.chartDef.pivotTableOptions.columnIdByCustomExpandedStatus[columnGroupId]) ?
                        this.chartDef.pivotTableOptions.columnIdByCustomExpandedStatus[columnGroupId]:
                        this.chartDef.pivotTableOptions.areColumnExpandedByDefault;
                }

                computeRowExpandedStatus(rowId) {
                    return !_.isNil(this.chartDef.pivotTableOptions.rowIdByCustomExpandedStatus) && !_.isNil(this.chartDef.pivotTableOptions.rowIdByCustomExpandedStatus[rowId]) ?
                        this.chartDef.pivotTableOptions.rowIdByCustomExpandedStatus[rowId]:
                        this.chartDef.pivotTableOptions.areRowsExpandedByDefault;
                }

                /**
                 *
                 *
                 * @param {string} label
                 * @param {{fontSize: string; color: any;}} formatting
                 * @param {boolean} [hasChevron=false]
                 * @return {*}
                 * @memberof AbstractAGGridPivotTableChartConverter
                 */
                computeColumnWidth(label, formatting, hasChevron = false) {
                    const labelWidth = PivotTableUtils.getTextWidth(label, formatting.fontSize, formatting.fontFamily);
                    return hasChevron ? labelWidth + CHEVRON_OFFSET_WIDTH : labelWidth;
                }

                /**
                 * Returns true if column with {colId} has a user defined custom width
                 * @param   {string} colId
                 * @returns {boolean} true if column with {colId} has a user defined custom width
                 */
                hasCustomWidth(colId) {
                    return !_.isNil(colId) && !_.isNil(this.chartDef.pivotTableOptions.columnIdByCustomWidth) && !_.isNil(this.chartDef.pivotTableOptions.columnIdByCustomWidth[colId]);
                }

                /**
                 * Returns the custom width of column with {colId}
                 * @param   {string} colId
                 * @returns {number} The custom width of column with {colId}
                 */
                getCustomWidth(colId) {
                    return this.chartDef.pivotTableOptions.columnIdByCustomWidth[colId];
                }

                /**
                 * Adjust column width with cell value
                 * @param       {string | number}                 cellValue
                 * @param       {AGGridConverterDFSColumnData}    columnData
                 * @memberof    AbstractAGGridPivotTableChartConverter
                 */
                adjustColumnWidthFromCell(cellValue, columnData) {
                    if (this.hasCustomWidth(columnData.agGridColumn.colId)) {
                        return;
                    }
                    let width = this.computeColumnWidth(cellValue, this.valuesFormatting);
                    /** @type {AGGridConverterDFSColumnData | undefined} */
                    let currCol = columnData;
                    while (currCol) {
                        if (currCol.agGridColumn.width > width) {
                            width = currCol.agGridColumn.width;
                        }
                        currCol = currCol.parent;
                    }
                    columnData.agGridColumn.width = width;
                }

                /**
                 * Adjust column width with cell value
                 * @param       {string | number}   cellValue
                 * @param       {string}            colId
                 * @memberof    AbstractAGGridPivotTableChartConverter
                 */
                adjustGrandTotalColumnWidthFromCell(cellValue, colId) {
                    if (this.hasCustomWidth(colId)) {
                        return;
                    }
                    const textWidth = PivotTableUtils.getTextWidth(cellValue, this.valuesFormatting.fontSize, this.valuesFormatting.fontFamily);
                    if (this.grandTotalColumnMaxCellWidth.has(colId)) {
                        this.grandTotalColumnMaxCellWidth.set(colId, Math.max(this.grandTotalColumnMaxCellWidth.get(colId), textWidth));
                    } else {
                        this.grandTotalColumnMaxCellWidth.set(colId, textWidth);
                    }
                }

                /**
                 * Get row dimension column width
                 * @param   {string} colId
                 * @param   {string} label
                 * @param   {string []} cellLabels
                 * @param   {number} offset
                 * @returns {number} Row navigation column width
                 */
                getRowDimensionsColumnWidth(colId, label, cellLabels, offset = 0) {
                    if (this.hasCustomWidth(colId)) {
                        return this.getCustomWidth(colId);
                    }
                    let width = this.computeColumnWidth(label, this.rowMainHeadersFormatting);
                    width = cellLabels.reduce((acc, label) => Math.max(acc, PivotTableUtils.getTextWidth(label, this.rowSubheadersFormatting.fontSize, this.rowSubheadersFormatting.fontFamily)), width);
                    return width + offset;
                }

                /**
                 * Get leaf measure column width
                 * @param   {string} colId
                 * @param   {string} label
                 * @param {{fontSize: string; color: any;}} formatting
                 * @returns {number} Leaf measure column width
                 */
                getLeafColumnCustomOrAutoSizedWidth(colId, label, formatting) {
                    if (this.hasCustomWidth(colId)) {
                        return this.getCustomWidth(colId);
                    } else {
                        return this.computeColumnWidth(label, formatting);
                    }
                }

                /**
                 * Get grand total column width
                 * @param   {string} colId
                 * @param   {string} label
                 * @returns {number} Grand total column width
                 */
                getGrandTotalColumnCustomOrAutoSizedWidth(colId, leafColumnLabel = '') {
                    if (this.hasCustomWidth(colId)) {
                        return this.getCustomWidth(colId);
                    } else {
                        return Math.max(this.computeColumnWidth(GRAND_TOTAL_LABEL, this.columnSubheadersFormatting), this.computeColumnWidth(leafColumnLabel, this.columnSubheadersFormatting), this.grandTotalColumnMaxCellWidth.get(colId));
                    }
                }
            }

            // All converters could be services
            class RowAndColumnDimensionsWithMeasureAsRowConverter extends AbstractAGGridPivotTableChartConverter {

                /**
                 * Build AG Grid row flat tree using DFS (Depth First Search) using a stack.
                 * It returns a flat table where the tree structure is defined by the rowPath.
                 * @param       {AGGridConverterDFSRowData[]} rowTree
                 * @return      {{rowData: AgGridRow[], subtotalRowData: AgGridRow [], grandTotalRowData?: AgGridRow}}
                 * @memberof    AbstractAGGridPivotTableChartConverter
                 */
                filterRowTreeAndConvertToRowData(rowTree) {
                    const { rowData, subtotalRowData, grandTotalRowData } = super.filterRowTreeAndConvertToRowData(rowTree);
                    this.addMeasureLabelsInMeasureColumn([...rowData, ...subtotalRowData, grandTotalRowData]);
                    return { rowData, subtotalRowData, grandTotalRowData };
                }

                /**
                 * Add Measures Label list for measure column situated after grouping column
                 * @param    {AgGridRow[]} rowTree
                 * @memberof AbstractAGGridPivotTableChartConverter
                 */
                addMeasureLabelsInMeasureColumn(rowTree) {
                    const measureLabels = this.chartDef.genericMeasures.map(measure => {
                        return {
                            value: this.getMeasureLabel(measure)
                        };
                    });
                    for (let i = 0; i < rowTree.length; i++) {
                        const row = rowTree[i];
                        if (row) {
                            row[MEASURE_COLUMN_KEY] = { measures: measureLabels, isMeasureColumn: true };
                        }
                    }
                }

                fillMeasuresInRow(currentRow) {
                    let isEmpty = true;
                    currentRow.agGridRow.group = false;
                    if (this.chartDef.genericMeasures && this.chartDef.genericMeasures.length > 0) {
                        this.endColumns.map(endColumn => {
                            const cellCoords = { ...endColumn.xCoordDict, ...currentRow.yCoordDict };
                            const measures = this.chartDef.genericMeasures.map((measure, aggregationIndex) => {
                                const cellValue = this.computeCellValue(aggregationIndex, measure, cellCoords, false);
                                const isTupleExistingInDataset = this.chartData.getCount(cellCoords) !== 0;
                                if (isTupleExistingInDataset) {
                                    isEmpty = false;
                                    this.adjustColumnWidthFromCell(cellValue.value, endColumn.columnData);
                                    this.setNotEmptyFlagsToColumnBranch(endColumn.columnData);
                                }
                                return cellValue;
                            });
                            currentRow.agGridRow[endColumn.id] = {
                                measures: measures,
                                cellCoords: cellCoords,
                                colorProperties: this.computeCellColorProperties(cellCoords, false, this.chartDef.genericMeasures.map((_, aggrIdx) => aggrIdx))
                            };
                        });
                    }

                    if (!isEmpty) {
                        this.setNotEmptyFlagsToRowBranch(currentRow);
                    }

                    return isEmpty;
                }

                fillSubTotalsInRow(currentRow, isGroupRow) {
                    this.subtotalColumns.map(subtotalColumn => {
                        const cellCoords = { ...subtotalColumn.xCoordDict, ...currentRow.yCoordDict };
                        const measures = this.chartDef.genericMeasures.map((measure, aggregationIndex) => {
                            const cellValue = this.computeCellValue(aggregationIndex, measure, cellCoords, true);
                            this.adjustColumnWidthFromCell(cellValue.value, subtotalColumn.columnData);
                            return cellValue;
                        });
                        currentRow.agGridRow[subtotalColumn.id] = {
                            measures: measures,
                            cellCoords: cellCoords,
                            colorProperties: this.computeCellColorProperties(cellCoords, true)
                        };
                    });

                    if (isGroupRow) {
                        this.endColumns.forEach(endColumn => {
                            const cellCoords = { ...endColumn.xCoordDict, ...currentRow.yCoordDict };
                            const measures = this.chartDef.genericMeasures.map((measure, aggregationIndex) => {
                                const cellValue = this.computeCellValue(aggregationIndex, measure, cellCoords, true);
                                this.adjustColumnWidthFromCell(cellValue.value, endColumn.columnData);
                                return cellValue;
                            });
                            currentRow.agGridRow[endColumn.id] = {
                                measures: measures,
                                cellCoords: cellCoords,
                                colorProperties: this.computeCellColorProperties(cellCoords, true)
                            };
                        });
                        currentRow.agGridRow[COLOR_FLAG_KEY] = true;
                    }
                }

                fillGrandTotalsInRow(currentRow) {
                    if (this.chartDef.genericMeasures && this.chartDef.genericMeasures.length > 0) {
                        const measures = [];
                        this.chartDef.genericMeasures.map((measure, aggregationIndex) => {
                            const hasTooltipAndContextualMenu = Object.keys(currentRow.yCoordDict).length > 0;
                            const cellContent = this.computeCellValue(aggregationIndex, measure, currentRow.yCoordDict, true, hasTooltipAndContextualMenu);
                            this.adjustGrandTotalColumnWidthFromCell(cellContent.value, GRAND_TOTAL_KEY);
                            measures.push(cellContent);
                        });
                        currentRow.agGridRow[GRAND_TOTAL_KEY] = { measures: measures };
                    }
                }

                getGrandTotalRow() {
                    /** @type {AgGridRow} */
                    const grandTotalRow = {};
                    const rowPath = [GRAND_TOTAL_KEY];
                    this.mapGroupColumnCellIdToHeaderName.set(rowPath, { label: this.chartDef.pivotTableOptions.tableFormatting.showRowHeaders ? GRAND_TOTAL_LABEL : '', depth: 0 });
                    grandTotalRow[this.getRowGroupKey()] = rowPath;
                    grandTotalRow.id = rowPath.join(SUB_COLUMN_SEPARATOR);
                    grandTotalRow.group = false;
                    // Leaf columns
                    this.endColumns.forEach((endColumn) => {
                        const measures = this.chartDef.genericMeasures.map((measure, aggregationIndex) => {
                            if (endColumn.dimensionId !== undefined) {
                                const cellContent = this.computeCellValue(aggregationIndex, measure, endColumn.xCoordDict, true);
                                this.adjustColumnWidthFromCell(cellContent.value, endColumn.columnData);
                                return cellContent;
                            }
                        });
                        grandTotalRow[endColumn.id] = { measures: measures };
                    });

                    // Grand Total column
                    const grandTotalMeasureColId = GRAND_TOTAL_KEY;
                    const grandTotalMeasures = this.chartDef.genericMeasures.map((measure, aggregationIndex) => {
                        const cellContent = this.computeCellValue(aggregationIndex, measure, {}, true, false);
                        this.adjustGrandTotalColumnWidthFromCell(cellContent.value, GRAND_TOTAL_KEY);
                        return cellContent;
                    });
                    grandTotalRow[grandTotalMeasureColId] = { measures: grandTotalMeasures };

                    // Sub-Total columns
                    this.subtotalColumns.forEach(subtotalColumn => {
                        const measures = this.chartDef.genericMeasures.map((measure, aggregationIndex) => {
                            const cellContent = this.computeCellValue(aggregationIndex, measure, subtotalColumn.xCoordDict, true);
                            this.adjustColumnWidthFromCell(cellContent.value, subtotalColumn.columnData);
                            return cellContent;
                        });
                        grandTotalRow[subtotalColumn.id] = { measures: measures };
                    });
                    grandTotalRow[COLOR_FLAG_KEY] = true;
                    return grandTotalRow;
                }

                /**
                 * Build Column tree iteratively using DFS (Depth First Seach) using a stack.
                 * @returns     {AGGridConverterDFSColumnData[]} Columns
                 * @memberof    AbstractAGGridPivotTableChartConverter
                 */
                buildColumnTree() {
                    const columnTree = super.buildColumnTree();
                    // Add a column with the measure name list only if there is more than one measure
                    if (this.chartDef.genericMeasures.length > 1) {
                        const colId = MEASURE_COLUMN_KEY;

                        const width = this.getRowDimensionsColumnWidth(
                            colId,
                            MEASURE_COLUMN_LABEL,
                            this.chartDef.genericMeasures.map(measure => this.getMeasureLabel(measure))
                        );

                        columnTree.splice(this.yDimensions.length - 1, 0, {
                            isEmpty: false,
                            parent: null,
                            agGridColumn: {
                                colId,
                                field: colId,
                                headerName: this.chartDef.pivotTableOptions.tableFormatting.showRowMainHeaders ? MEASURE_COLUMN_LABEL : '',
                                pinned: this.chartDef.pivotTableOptions.tableFormatting.freezeRowHeaders ? 'left' : null,
                                type: 'measureLabelColumn',
                                width,
                                lockPosition: true,
                                headerComponentParams: { formatting: this.rowMainHeadersFormatting, isRowMainHeader: true }
                            }
                        });
                    }
                    return columnTree;
                }

                /**
                 * Fulfills endColumns datastructure to find back aggregation position in tensor.
                 * If measures are treated as columns, it add sub-columns.
                 * @param       {AGGridConverterDFSColumnData}      currColumn
                 * @memberof    AbstractAGGridPivotTableChartConverter
                 */
                addMeasuresInColumnTree(currColumn) {
                    const dimensionId = this.chartStore.getDimensionId(currColumn.head);
                    // Adding column id of leafs who contain generic measures as children
                    this.endColumns.push({ id: currColumn.agGridColumn.groupId, xCoordDict: currColumn.xCoordDict, dimensionId, columnData: currColumn });
                    const colId = currColumn.agGridColumn.groupId;
                    currColumn.agGridColumn.colId = colId;
                    currColumn.agGridColumn.field = colId;
                    currColumn.agGridColumn.cellRenderer = 'cellWithTooltipRenderer';

                    currColumn.agGridColumn.cellRendererParams = {
                        tooltips: this.tooltips,
                        contextualMenu: this.contextualMenu,
                        colorGroupByMeasureId: this.colorGroupByMeasureId,
                        chartDef: this.chartDef,
                        theme: this.chartHandler.getChartTheme()
                    };
                    currColumn.agGridColumn.width = this.getLeafColumnCustomOrAutoSizedWidth(colId, currColumn.agGridColumn.headerName, this.columnSubheadersFormatting);
                }

                /**
                 * Adds the Sub-total parent columns. The column(s) appearing when the column-node is collapsed.
                 * @param       {AGGridConverterDFSColumnData}  currColumn
                 * @param       {string}                        dimensionId
                 * @memberof     AbstractAGGridPivotTableChartConverter
                 */
                addSubTotalParentColumnsInColumnTree(currColumn, dimensionId) {
                    const colId = currColumn.agGridColumn.groupId + '__Total';
                    const headerName = this.chartDef.pivotTableOptions.tableFormatting.showColumnHeaders ? 'Total ' + currColumn.agGridColumn.headerName : '';

                    /** @type {AGGridConverterDFSColumnData} */
                    const subtotalColumnData = {
                        isEmpty: false,
                        parent: currColumn,
                        agGridColumn: {
                            colId,
                            field: colId,
                            type: 'subtotalColumn',
                            columnGroupShow: this.chartDef.pivotTableOptions.displayTotals.subTotals.columns ? undefined : 'closed',
                            headerName,
                            width: this.getLeafColumnCustomOrAutoSizedWidth(colId, headerName, this.columnSubheadersFormatting),
                            headerComponentParams: { formatting: this.columnSubheadersFormatting }
                        }
                    };
                    this.subtotalColumns.push({ id: colId, xCoordDict: currColumn.xCoordDict, dimensionId, columnData: subtotalColumnData });
                    currColumn.children.push(subtotalColumnData);
                }

                getGrandTotalColumn() {
                    const grandTotalColumn = {};
                    const headerName = this.chartDef.pivotTableOptions.tableFormatting.showColumnHeaders ? GRAND_TOTAL_LABEL : '';
                    grandTotalColumn.colId = GRAND_TOTAL_KEY;
                    grandTotalColumn.field = GRAND_TOTAL_KEY;
                    grandTotalColumn.headerName = headerName;
                    grandTotalColumn.type = 'grandTotalColumn';
                    grandTotalColumn.headerComponentParams = { formatting: this.columnSubheadersFormatting };
                    grandTotalColumn.initialHide = !this.chartDef.pivotTableOptions.displayTotals.grandTotal.column;
                    grandTotalColumn.width = this.getGrandTotalColumnCustomOrAutoSizedWidth(GRAND_TOTAL_KEY);
                    grandTotalColumn.suppressMovable = true;
                    return grandTotalColumn;
                }

                /**
                 * Returns row height - 25px for each measures
                 * @memberof AbstractAGGridPivotTableChartConverter
                 */
                getRowHeight() {
                    return this.chartDef.genericMeasures ? this.chartDef.genericMeasures.length * DEFAULT_ROW_HEIGHT : DEFAULT_ROW_HEIGHT;
                }
            }

            class OnlyColumnDimensionWithMeasureAsRowConverter extends RowAndColumnDimensionsWithMeasureAsRowConverter {

                /**
                 * Build AG Grid options
                 * @return      {GridOptions}
                 * @memberof    AbstractAGGridPivotTableChartConverter
                 */
                convert() {
                    const gridOptions = super.convert();
                    return { ...gridOptions, treeData: false };
                }

                /**
                 * Build AG Grid row list, when no fields placed on rows, measures displayed as rows.
                 * @return      {AGGridConverterDFSRowData[]}
                 * @memberof    AbstractAGGridPivotTableChartConverter
                 */
                buildRowTree() {
                    const uniqueRowData = { rowPath: [], yCoordDict: {}, headerName: '', isEmpty: false, agGridRow: { id: UNIQUE_ROW_KEY } };
                    return [uniqueRowData];
                }

                /**
                 * Build AG Grid row flat tree using DFS (Depth First Seach) using a stack.
                 * It returns a flat table where the tree structure is defined by the rowPath.
                 * @param       {AGGridConverterDFSRowData[]}       rowTree
                 * @return      {{rowData: AgGridRow[], subtotalRowData: AgGridRow[], grandTotalRowData?: AgGridRow}}
                 * @memberof    AbstractAGGridPivotTableChartConverter
                 */
                filterRowTreeAndConvertToRowData(rowTree) {
                    const uniqueAgGridRow = rowTree[0].agGridRow;
                    uniqueAgGridRow[this.getRowGroupKey()] = [''];
                    this.fillMeasuresInRow(rowTree[0]);
                    this.fillSubTotalsInRow(rowTree[0], false);
                    this.fillGrandTotalsInRow(rowTree[0]);
                    this.addMeasureLabelsInMeasureColumn([uniqueAgGridRow]);
                    return { rowData: [uniqueAgGridRow], subtotalRowData: [] };
                }

                hasGrandTotalRow() {
                    return false;
                }

                hasGrandTotalColumn() {
                    // No grand total column
                    return true;
                }
            }

            class RowAndColumnDimensionsWithMeasureAsColumnConverter extends AbstractAGGridPivotTableChartConverter {

                fillMeasuresInRow(currentRow) {
                    let isEmpty = true;
                    currentRow.agGridRow.group = false;
                    if (this.chartDef.genericMeasures && this.chartDef.genericMeasures.length > 0) {
                        this.chartDef.genericMeasures.map((measure, aggregationIndex) => {
                            const measureId = this.chartStore.getMeasureId(measure);
                            this.endColumns.forEach(endColumn => {
                                const cellCoords = { ...endColumn.xCoordDict, ...currentRow.yCoordDict };
                                const cellValue = this.computeCellValue(aggregationIndex, measure, cellCoords, false);
                                const isTupleExistingInDataset = this.chartData.getCount(cellCoords) != 0;
                                if (isTupleExistingInDataset) {
                                    isEmpty = false;
                                    this.adjustColumnWidthFromCell(cellValue.value, endColumn.columnData);
                                    this.setNotEmptyFlagsToColumnBranch(endColumn.columnData);
                                }
                                currentRow.agGridRow[this.getLeafColumnId(endColumn.id, measureId)] = {
                                    measure: cellValue,
                                    measureIndex: aggregationIndex,
                                    cellCoords: cellCoords,
                                    colorProperties: this.computeCellColorProperties(cellCoords, false, [aggregationIndex])
                                };
                            });
                        });
                    }

                    if (!isEmpty) {
                        this.setNotEmptyFlagsToRowBranch(currentRow);
                    }

                    return isEmpty;
                }

                fillSubTotalsInRow(currentRow, isGroupRow) {
                    this.subtotalColumns.map(subtotalColumn => {
                        const cellCoords = { ...subtotalColumn.xCoordDict, ...currentRow.yCoordDict };
                        const cellValue = this.computeCellValue(subtotalColumn.aggregationIndex, subtotalColumn.measure, cellCoords, true);
                        this.adjustColumnWidthFromCell(cellValue.value, subtotalColumn.columnData);
                        currentRow.agGridRow[subtotalColumn.id] = {
                            measure: cellValue,
                            measureIndex: subtotalColumn.aggregationIndex,
                            cellCoords: cellCoords,
                            colorProperties: this.computeCellColorProperties(cellCoords, true)
                        };
                    });
                    if (isGroupRow) {
                        this.chartDef.genericMeasures.map((measure, aggregationIndex) => {
                            this.endColumns.forEach(endColumn => {
                                const cellCoords = { ...endColumn.xCoordDict, ...currentRow.yCoordDict };
                                const measureId = this.chartStore.getMeasureId(measure);
                                const cellValue = this.computeCellValue(aggregationIndex, measure, cellCoords, true);
                                this.adjustColumnWidthFromCell(cellValue.value, endColumn.columnData);
                                currentRow.agGridRow[this.getLeafColumnId(endColumn.id, measureId)] = {
                                    measure: cellValue,
                                    measureIndex: aggregationIndex,
                                    cellCoords: cellCoords,
                                    colorProperties: this.computeCellColorProperties(cellCoords, true)
                                };
                            });
                        });
                        currentRow.agGridRow[COLOR_FLAG_KEY] = true;
                    }
                }

                fillGrandTotalsInRow(currentRow) {
                    if (this.chartDef.genericMeasures && this.chartDef.genericMeasures.length > 0) {
                        this.chartDef.genericMeasures.map((measure, aggregationIndex) => {
                            const measureId = this.chartStore.getMeasureId(measure);
                            const grandTotalMeasureColId = this.getLeafColumnId(GRAND_TOTAL_KEY, measureId);
                            const cellContent = this.computeCellValue(aggregationIndex, measure, currentRow.yCoordDict, true);
                            this.adjustGrandTotalColumnWidthFromCell(cellContent.value, grandTotalMeasureColId);
                            currentRow.agGridRow[grandTotalMeasureColId] = {
                                measure: cellContent,
                                measureIndex: aggregationIndex
                            };
                        });
                    }
                }

                getGrandTotalRow() {
                    /** @type {AgGridRow} */
                    const grandTotalRow = {};
                    const rowPath = [GRAND_TOTAL_KEY];
                    this.mapGroupColumnCellIdToHeaderName.set(rowPath, { label: GRAND_TOTAL_LABEL, depth: 0 });
                    grandTotalRow[this.getRowGroupKey()] = rowPath;
                    grandTotalRow.group = false;
                    grandTotalRow.id = rowPath.join(SUB_COLUMN_SEPARATOR);
                    this.endColumns.forEach((endColumn) => {
                        this.chartDef.genericMeasures.forEach((measure, aggregationIndex) => {
                            const measureId = this.chartStore.getMeasureId(measure);
                            if (endColumn.dimensionId !== undefined) {
                                const cellContent = {
                                    measure: this.computeCellValue(aggregationIndex, measure, endColumn.xCoordDict, true),
                                    measureIndex: aggregationIndex
                                };
                                grandTotalRow[this.getLeafColumnId(endColumn.id, measureId)] = cellContent;
                                this.adjustColumnWidthFromCell(cellContent.measure.value, endColumn.columnData);
                            }

                            // Grand Total column contains
                            const grandTotalMeasureColId = this.getLeafColumnId(GRAND_TOTAL_KEY, measureId);

                            const grandTotalValue = {
                                measure: this.computeCellValue(aggregationIndex, measure, {}, true, false),
                                measureIndex: aggregationIndex
                            };

                            // No column special case
                            if (endColumn.id === '') {
                                grandTotalRow[this.getLeafColumnId(endColumn.id, measureId)] = grandTotalValue;
                                this.adjustColumnWidthFromCell(grandTotalValue.measure.value, endColumn.columnData);
                            } else {
                                grandTotalRow[grandTotalMeasureColId] = grandTotalValue;
                                this.adjustGrandTotalColumnWidthFromCell(grandTotalValue.measure.value, this.getLeafColumnId(GRAND_TOTAL_KEY, measureId));
                            }
                        });
                    });

                    this.subtotalColumns.forEach(subtotalColumn => {
                        const subtotalValue = {
                            measure: this.computeCellValue(subtotalColumn.aggregationIndex, subtotalColumn.measure, subtotalColumn.xCoordDict, true),
                            measureIndex: subtotalColumn.aggregationIndex
                        };
                        grandTotalRow[subtotalColumn.id] = subtotalValue;
                        this.adjustColumnWidthFromCell(subtotalValue.measure.value, subtotalColumn.columnData);
                    });

                    grandTotalRow[COLOR_FLAG_KEY] = true;

                    return grandTotalRow;
                }

                /**
                 * Fulfills endColumns datastructure to find back aggregation position in tensor.
                 * If measures are treated as columns, it add sub-columns.
                 * @param       {AGGridConverterDFSColumnData}  currColumn
                 * @memberof    AbstractAGGridPivotTableChartConverter
                 */
                addMeasuresInColumnTree(currColumn) {
                    const dimensionId = this.chartStore.getDimensionId(currColumn.head);
                    const areMeasuresTreatedAsRows = this.chartDef.pivotTableOptions.measureDisplayMode === ChartsStaticData.pivotTableMeasureDisplayMode.ROWS;

                    // Adding column id of leafs who contain generic measures as children
                    currColumn.children = this.chartDef.genericMeasures.map((measure, i) => {
                        const measureId = this.chartStore.getMeasureId(measure);
                        const colId = this.getLeafColumnId(currColumn.agGridColumn.groupId, measureId);
                        let headerName = '';
                        if ((areMeasuresTreatedAsRows && this.chartDef.pivotTableOptions.tableFormatting.showRowMainHeaders) || this.chartDef.pivotTableOptions.tableFormatting.showColumnHeaders) {
                            headerName = this.getMeasureLabel(measure);
                        }
                        const leafColumnData = {
                            isEmpty: true,
                            parent: currColumn,
                            agGridColumn: {
                                colId,
                                field: colId,
                                headerName: headerName,
                                type: 'leafAggregationColumn',
                                width: this.getLeafColumnCustomOrAutoSizedWidth(colId, this.getMeasureLabel(measure), this.columnSubheadersFormatting),
                                headerComponentParams: { formatting: this.columnSubheadersFormatting }
                            }
                        };
                        this.endColumns.push({ id: currColumn.agGridColumn.groupId, xCoordDict: currColumn.xCoordDict, dimensionId, aggregationIndex: i, columnData: leafColumnData });

                        return leafColumnData;
                    });
                }

                /**
                 * Adds the Sub-total parent columns. The column(s) appearing when the column-node is collapsed.
                 * @param       {AGGridConverterDFSColumnData}          currColumn
                 * @param       {string}                                dimensionId
                 * @memberof    AbstractAGGridPivotTableChartConverter
                 */
                addSubTotalParentColumnsInColumnTree(currColumn, dimensionId) {
                    this.chartDef.genericMeasures.map((measure, i) => {
                        const colId = currColumn.agGridColumn.groupId + this.chartStore.getMeasureId(measure) + '__Total';
                        const headerName = this.chartDef.pivotTableOptions.tableFormatting.showColumnHeaders ? this.getMeasureLabel(measure) : '';
                        const subtotalColumnData = {
                            isEmpty: false,
                            parent: currColumn,
                            agGridColumn: {
                                colId,
                                field: colId,
                                type: 'subtotalColumn',
                                headerName,
                                columnGroupShow: this.chartDef.pivotTableOptions.displayTotals.subTotals.columns ? undefined : 'closed',
                                width: this.getLeafColumnCustomOrAutoSizedWidth(colId, headerName, this.columnSubheadersFormatting),
                                headerComponentParams: { formatting: this.columnSubheadersFormatting }
                            }
                        };
                        this.subtotalColumns.push({ id: colId, xCoordDict: currColumn.xCoordDict, dimensionId, aggregationIndex: i, measure, columnData: subtotalColumnData });
                        currColumn.children && currColumn.children.push(subtotalColumnData);
                    });
                }

                getGrandTotalColumn() {
                    const grandTotalColumn = {};
                    const headerName = this.chartDef.pivotTableOptions.tableFormatting.showColumnHeaders ? GRAND_TOTAL_LABEL : '';
                    grandTotalColumn.colId = GRAND_TOTAL_KEY;
                    grandTotalColumn.field = GRAND_TOTAL_KEY;
                    grandTotalColumn.headerName = headerName;
                    grandTotalColumn.initialHide = !this.chartDef.pivotTableOptions.displayTotals.grandTotal.column;
                    grandTotalColumn.type = 'grandTotalGroupColumn';
                    grandTotalColumn.headerGroupComponentParams = { formatting: this.columnSubheadersFormatting };
                    grandTotalColumn.suppressMovable = true;
                    grandTotalColumn.children = [];
                    this.chartDef.genericMeasures.map((measure) => {
                        const grandTotalLeaf = {};
                        const leafLabel = this.getMeasureLabel(measure);
                        const measureId = this.chartStore.getMeasureId(measure);
                        const colId = this.getLeafColumnId(grandTotalColumn.colId, measureId);
                        grandTotalLeaf.colId = colId;
                        grandTotalLeaf.field = colId;
                        grandTotalLeaf.initialHide = !this.chartDef.pivotTableOptions.displayTotals.grandTotal.column;
                        grandTotalLeaf.headerName = this.chartDef.pivotTableOptions.tableFormatting.showColumnHeaders ? this.getMeasureLabel(measure) : '';
                        grandTotalLeaf.type = 'grandTotalColumn';
                        grandTotalLeaf.suppressMovable = true;
                        grandTotalLeaf.headerComponentParams = { formatting: this.columnSubheadersFormatting };
                        grandTotalLeaf.width = this.getGrandTotalColumnCustomOrAutoSizedWidth(colId, leafLabel);
                        grandTotalColumn.children.push(grandTotalLeaf);
                    });
                    return grandTotalColumn;
                }
            }

            class OnlyRowDimensionWithMeasureAsColumnConverter extends RowAndColumnDimensionsWithMeasureAsColumnConverter {

                buildColumnTree() {
                    /** @type {AGGridConverterDFSColumnData[]} */
                    const columnTree = [];

                    columnTree.push(...this.getRowNavigationColumns());

                    this.chartDef.genericMeasures.forEach((measure) => {
                        const headerName = (this.chartDef.pivotTableOptions.tableFormatting.showColumnHeaders && this.chartDef.pivotTableOptions.tableFormatting.showColumnMainHeaders) ? this.getMeasureLabel(measure) : '';
                        const colId = this.chartStore.getMeasureId(measure);
                        const leafData = {
                            parent: null,
                            isEmpty: false,
                            agGridColumn: {
                                colId,
                                field: colId,
                                headerName,
                                type: 'leafAggregationColumn',
                                width: this.getLeafColumnCustomOrAutoSizedWidth(colId, headerName, this.columnMainHeadersFormatting),
                                lockPosition: true,
                                headerComponentParams: { formatting: this.columnMainHeadersFormatting }
                            }
                        };
                        this.endColumns.push({ id: '', xCoordDict: {}, columnData: leafData });
                        columnTree.push(leafData);
                    });

                    return columnTree;
                }

                hasGrandTotalRow() {
                    return true;
                }

                hasGrandTotalColumn() {
                    return false;
                }
            }

            class NoDimensionWithMeasureOnlyConverter extends OnlyRowDimensionWithMeasureAsColumnConverter{

                buildRowTree() {
                    return [];
                }

                getGrandTotalRow() {
                    // Force display of grand total's rows, in case it was removed before
                    this.chartDef.pivotTableOptions.displayTotals.grandTotal.row = true;

                    /** @type {AgGridRow} */
                    const grandTotalRow = {};
                    const rowPath = [GRAND_TOTAL_KEY];
                    this.mapGroupColumnCellIdToHeaderName.set(rowPath, { label: GRAND_TOTAL_LABEL, depth: 0 });
                    grandTotalRow[this.getRowGroupKey()] = rowPath;
                    grandTotalRow.group = false;

                    /** @type {AgGridRow} */
                    this.chartDef.genericMeasures.forEach((measure, aggregationIndex) => {
                        const measureId = this.chartStore.getMeasureId(measure);

                        const value = this.chartData.data.aggregations[aggregationIndex].tensor[0];
                        const formattedValue = this.cellFormatters[aggregationIndex] ? this.cellFormatters[aggregationIndex](value, 1) : value;
                        this.adjustColumnWidthFromCell(formattedValue, this.endColumns[aggregationIndex].columnData);

                        const grandTotalValue = {
                            measure: {
                                value: formattedValue,
                                originalValue: value,
                                cellCoords: {}
                            },
                            measureIndex: aggregationIndex
                        };
                        grandTotalRow[this.getLeafColumnId('', measureId)] = grandTotalValue;
                    });
                    return grandTotalRow;
                }

                hasGrandTotalColumn() {
                    return false;
                }
            }

            class CustomHeader {

                init(agParams) {
                    this.agParams = agParams;
                    this.formatting = this.agParams.formatting;
                    this.eGui = document.createElement('div');
                    this.eGui.classList.add('pivot-table-header');
                    if (this.agParams.displayName) {
                        const span = document.createElement('span');
                        span.classList.add('pivot-table-header__label');
                        span.innerText = this.agParams.displayName;
                        if (this.formatting) {
                            span.style.fontSize = this.formatting.fontSize;
                            span.style.color = this.formatting.color;
                        }
                        this.eGui.appendChild(span);

                        if (this.agParams.hasChevron) {
                            this.eGui.classList.add('pivot-table-header--with-chevron');
                            const chevronContainer = document.createElement('span');
                            chevronContainer.classList.add('ag-header-icon', 'ag-header-expand-icon', 'ag-ltr');
                            const chevron = document.createElement('span');
                            chevronContainer.appendChild(chevron);
                            this.eGui.appendChild(chevronContainer);
                            this.switchFoldStatusEventListener = this.switchFoldStatus.bind(this);
                            this.eGui.addEventListener('click', this.switchFoldStatusEventListener);
                            this.eGui.addEventListener('dblclick', this.switchFoldStatusEventListener);
                            this.updateChevronState();
                        }

                        if (this.agParams.isColumnMainHeader) {
                            this.eGui.classList.add(PivotTableUtils.COLUMN_MAIN_HEADER_CLASSNAME);
                        } else if (this.agParams.isRowMainHeader) {
                            this.eGui.classList.add(PivotTableUtils.ROW_MAIN_HEADER_CLASSNAME);
                        } else {
                            this.eGui.classList.add(PivotTableUtils.COLUMN_SUBHEADER_CLASSNAME);
                        }
                    }

                }

                updateChevronState() {
                    const isExpanded = this.agParams.columnGroup.isExpanded();
                    if (this.eGui == null) {
                        return;
                    }
                    const chevronContainer = this.eGui.querySelector('.ag-header-icon.ag-header-expand-icon.ag-ltr');
                    if (chevronContainer == null) {
                        return;
                    }
                    const chevron = chevronContainer.querySelector('span');
                    if (chevron == null) {
                        return;
                    }
                    if (isExpanded) {
                        chevronContainer.classList.remove('ag-header-expand-icon-expanded-collapsed');
                        chevronContainer.classList.add('ag-header-expand-icon-expanded');
                        chevron.className = 'ag-icon ag-icon-expanded';
                    } else {
                        chevronContainer.classList.remove('ag-header-expand-icon-expanded');
                        chevronContainer.classList.add('ag-header-expand-icon-expanded-collapsed');
                        chevron.className = 'ag-icon ag-icon-contracted';
                    }
                }

                switchFoldStatus() {
                    const isExpanded = !this.agParams.columnGroup.isExpanded();
                    const groupId = this.agParams.columnGroup.groupId;
                    this.agParams.api.setColumnGroupOpened(groupId, isExpanded);
                    if (isExpanded != this.agParams.chartDef.pivotTableOptions.areColumnExpandedByDefault) {
                        this.agParams.chartDef.pivotTableOptions.columnIdByCustomExpandedStatus[groupId] = isExpanded;
                    } else {
                        delete this.agParams.chartDef.pivotTableOptions.columnIdByCustomExpandedStatus[groupId];
                    }
                    this.updateChevronState();
                }

                getGui() {
                    return this.eGui;
                }

                destroy() {
                    if (this.switchFoldStatusEventListener && this.eGui) {
                        this.eGui.removeEventListener('click', this.switchFoldStatusEventListener);
                        this.eGui.removeEventListener('dblclick', this.switchFoldStatusEventListener);
                    }
                }
            }

            /**
             * Rendering pivot table cell content
             * @implements {ICellRendererComp}
             * @class CellWithTooltipRenderer
             */
            class CellWithTooltipRenderer {
                /**
                 * Build the HTML cell to render
                 * @param       {ICellRendererParams & {tooltips: any, value: PivotTableCellContent, genericMeasures: ChartDef['genericMeasures'],
                 *  rowDimensionId: Set<string>, colorGroupByMeasureId: any, contextualMenu: any, colorMode: ChartDef['colorMode']}} params
                 * @memberof    CellWithTooltipRenderer
                 */
                init(params) {
                    const { colorMode, genericMeasures } = params.chartDef;
                    this.eGui = document.createElement('div');
                    this.eGui.classList.add('pivot-table__cell');

                    if (params.value == null || params.value.hidden) {
                        return;
                    }

                    // Only apply formatting to value cells, skip measures and grand totals
                    const colType = (params.colDef && params.colDef.type && typeof params.colDef.type === 'string' && params.colDef.type) || '';
                    const canApplyColors = !['grandTotalColumn', 'rowNavigationColumn', 'measureLabelColumn'].includes(colType)
                        && ![MEASURE_COLUMN_KEY, GRAND_TOTAL_KEY].includes(params.node.id || '')
                    ;
                    const colorProperties = params.value.colorProperties;
                    const hasColorDim = colorProperties !== undefined;
                    const isUsingColorGroups = colorMode === 'COLOR_GROUPS';

                    this.contextualMenu = params.contextualMenu;
                    this.tooltips = params.tooltips;
                    this.theme = params.theme;
                    this.chartDef = params.chartDef;
                    let color;

                    if (!params.value.isMeasureColumn) {
                        this.eGui.classList.add('pivot-table__cell--numerical');
                    }
                    if (params.value.measure != null && params.value.measureIndex != null) { // single value in one cell
                        const measure = params.value.measure;
                        const genericMeasure = genericMeasures[params.value.measureIndex];
                        const measureId = ConditionalFormattingOptions.getMeasureId(genericMeasure);
                        const shouldApplyGroupColors = canApplyColors &&
                                params.colorGroupByMeasureId &&
                                (measureId in params.colorGroupByMeasureId) &&
                                isUsingColorGroups
                            ;

                        const colorGroupDetails = params.colorGroupByMeasureId[measureId];

                        const measureElement = document.createElement('div');
                        measureElement.classList.add('pivot-table__cell-content');
                        this.eGui && this.eGui.appendChild(measureElement);


                        if (shouldApplyGroupColors && colorGroupDetails.colorGroupMode === 'RULES') {
                            const measureRules = colorGroupDetails.rules;
                            const ruleBaseValue = measure.colorMeasureOriginalValue !== undefined ? measure.colorMeasureOriginalValue : measure.originalValue;
                            const colorMeasure = colorGroupDetails.basedOnMeasure || genericMeasure;

                            const ruleClass = ConditionalFormattingOptions.getColorRuleClass(ruleBaseValue, measureRules, colorMeasure, this.theme);
                            ruleClass.class && measureElement.classList.add(ruleClass.class);

                            measureElement.style.backgroundColor = (ruleClass.customColors && ruleClass.customColors.customBackgroundColor) || '';
                            measureElement.style.color = (ruleClass.customColors && ruleClass.customColors.customFontColor) || '';
                        }

                        // add colorScale colors
                        if (hasColorDim || (shouldApplyGroupColors && colorGroupDetails.colorGroupMode !== 'RULES')) {
                            measureElement.classList.add('colored-content-wrapper');
                            let colorScale;
                            let colorMeasureValue;
                            let bin;

                            // if colorScale is not set by group, unique scale is selected
                            if (!isUsingColorGroups) {
                                colorScale = colorProperties.getColorScale();
                                colorMeasureValue = colorProperties.measure;
                                bin = colorProperties.bin;
                            } else if (shouldApplyGroupColors && colorGroupDetails.colorGroupMode !== 'RULES' && colorGroupDetails && colorGroupDetails.colorProperties) {
                                // case where the column itself is the colorMeasure
                                colorScale = colorGroupDetails.colorProperties.getColorScale();
                                colorMeasureValue = measure.colorMeasureOriginalValue !== undefined ? measure.colorMeasureOriginalValue : measure.originalValue;
                                bin = colorGroupDetails.colorProperties.getBin(params.value.cellCoords);
                            }
                            if (!colorScale) {
                                this.fillElementWithMeasure(measureElement, measure);
                                return;
                            }
                            color = { value: colorScale(colorMeasureValue, bin), measure: colorMeasureValue };
                            measureElement.style.backgroundColor = color.value;
                            measureElement.style.color = ColorUtils.getFontContrastColor(color.value, this.theme && this.theme.generalFormatting.fontColor);
                        }
                        this.fillElementWithMeasure(measureElement, measure, color);
                    } else { // multiple values in one cell
                        let colorMeasureValue;

                        if (hasColorDim) {
                            this.eGui.classList.add('colored-content-wrapper');
                            const colorScale = colorProperties.getColorScale();
                            colorMeasureValue = colorProperties.measure;
                            color = { value: colorScale(colorMeasureValue, colorProperties.bin), measure: colorMeasureValue };
                        }

                        params.value.measures.forEach((measure, i) => {
                            const measureElement = document.createElement('div');
                            measureElement.classList.add('pivot-table__cell-content');

                            let shouldApplyGroupColors = false;
                            let measureId;
                            let genericMeasure;

                            // on the first render after removing a value, the new request is not yet executed, so params.value.measures still has the value but params.genericMeasures doesn't
                            if (i < genericMeasures.length) {
                                genericMeasure = genericMeasures[i];
                                measureId = ConditionalFormattingOptions.getMeasureId(genericMeasure);

                                shouldApplyGroupColors = canApplyColors &&
                                        params.colorGroupByMeasureId &&
                                        (measureId in params.colorGroupByMeasureId)&&
                                        isUsingColorGroups
                                ;
                            }

                            let colorGroupDetails;

                            if (shouldApplyGroupColors) {
                                colorGroupDetails = params.colorGroupByMeasureId[measureId];

                                if (colorGroupDetails.colorGroupMode === 'RULES') {
                                    const measureRules = colorGroupDetails.rules;
                                    const ruleBaseValue = measure.colorMeasureOriginalValue !== undefined ? measure.colorMeasureOriginalValue : measure.originalValue;
                                    const colorMeasure = colorGroupDetails.basedOnMeasure || genericMeasure;

                                    const ruleClass = ConditionalFormattingOptions.getColorRuleClass(ruleBaseValue, measureRules, colorMeasure, this.theme);
                                    ruleClass.class && measureElement.classList.add(ruleClass.class);

                                    measureElement.style.backgroundColor = (ruleClass.customColors && ruleClass.customColors.customBackgroundColor) || '';
                                    measureElement.style.color = (ruleClass.customColors && ruleClass.customColors.customFontColor) || '';
                                }
                            }

                            if (hasColorDim || (shouldApplyGroupColors && colorGroupDetails.colorGroupMode !== 'RULES')) {
                                let cellColor;

                                if (!isUsingColorGroups) {
                                    cellColor = color;
                                } else if (shouldApplyGroupColors && colorGroupDetails.colorGroupMode !== 'RULES' && colorGroupDetails && colorGroupDetails.colorProperties) {
                                    // if colorGroup is set to scale mode and the chart is using color groups, set cell color based on group ColorScale
                                    const colorScale = colorGroupDetails.colorProperties.getColorScale();
                                    colorMeasureValue = measure.colorMeasureOriginalValue !== undefined ? measure.colorMeasureOriginalValue : measure.originalValue;
                                    const bin = colorGroupDetails.colorProperties.getBin(params.value.cellCoords);

                                    cellColor = { value: colorScale(colorMeasureValue, bin), measure: colorMeasureValue };
                                }

                                if (cellColor) {
                                    measureElement.style.backgroundColor = cellColor.value;
                                    measureElement.style.color = ColorUtils.getFontContrastColor(cellColor.value, this.theme && this.theme.generalFormatting.fontColor);
                                }
                            }

                            if (colType === 'measureLabelColumn') {
                                measureElement.classList.add(PivotTableUtils.ROW_SUBHEADER_CLASSNAME);
                                this.contextualMenu.addContextualMenuHandler(measureElement, undefined, PivotTableUtils.getRowHeaderContextualMenuActions(this.chartDef, params.canHideHeaders, params.rowHeadersCallbacks, measure.value), 'pivotTableHeader' );
                            }

                            this.fillElementWithMeasure(measureElement, measure, color);
                            this.eGui && this.eGui.appendChild(measureElement);
                        });
                    }
                }

                refresh() {
                    return false;
                }

                /**
                 * Fill html element with measure information & tooltip handler
                 * @param       {HTMLElement}                                   el
                 * @param       {PivotTableCellMeasure}                         measure
                 * @param       {{value: string, measure: number} | undefined}    color
                 * @memberof    CellWithTooltipRenderer
                 */
                fillElementWithMeasure(el, measure, color) {
                    const label = `${measure.value || ''}`;
                    const labelElement = document.createElement('div');
                    el.appendChild(labelElement);
                    labelElement.classList.add('pivot-table__cell-content-measure');
                    labelElement.innerHTML = label !== '' ? window.sanitize(label) : '&nbsp;';
                    if (measure.cellCoords != null) {
                        el.classList.add('tar');
                        if (measure.hasTooltipAndContextualMenu) {
                            this.tooltips.addTooltipHandlers(el, measure.cellCoords, color);
                            const copyAction = { label: translate('CHARTS.CONTEXTUAL_MENU.PIVOT_TABLE.COPY_TO_CLIPBOARD', 'Copy to clipboard'), icon: 'dku-icon-copy-step-16', onMouseDown: () => {
                                const selectedText = (getSelection() || '').toString();
                                ClipboardUtils.copyToClipboard(selectedText === '' ? label : selectedText);
                            } };
                            this.contextualMenu.addContextualMenuHandler(el, measure.cellCoords, [copyAction] );
                        }
                    }
                }

                /**
                 * Remove contextual menu and tooltip handlers event listeners
                 * @memberof CellWithTooltipRenderer
                 */
                destroy() {
                    if (this.eGui && this.tooltips && this.contextualMenu) {
                        const measureElements = this.eGui.querySelectorAll('.pivot-table__cell-content');
                        if (measureElements && measureElements.length) {
                            measureElements.forEach(element => {
                                this.contextualMenu.removeContextualMenuHandler(element);
                                this.tooltips.removeTooltipsHandlers(element);
                            });
                        } else {
                            this.contextualMenu.removeContextualMenuHandler(this.eGui);
                            this.tooltips.removeTooltipsHandlers(this.eGui);
                        }
                    }
                }

                /**
                 * Returns the HTML cell to render
                 * @return {HTMLElement}
                 * @memberof CellWithTooltipRenderer
                 */
                getGui() {
                    return this.eGui;
                }
            }

            /**
             * @param {ChartDef} chartDef
             * @param {ChartStore} chartStore
             * @param {ChartTensorDataWrapper} chartData
             * @param {PivotTableTensorResponse} data
             * @param {*} chartHandler
             * @param {*} tooltips
             * @param {JQuery<HTMLElement>} $rootElement
             * @returns
             */
            function getAgGridConverter(chartDef, chartStore, chartData, data, chartHandler, tooltips, contextualMenu, $rootElement) {
                const isMeasureTreatedAsRow = chartDef.pivotTableOptions.measureDisplayMode === 'ROWS';
                /** @type typeof AbstractAGGridPivotTableChartConverter */
                let Converter;
                if (PivotTableUtils.hasRows(chartDef) && PivotTableUtils.hasColumns(chartDef)) {
                    Converter = isMeasureTreatedAsRow ? RowAndColumnDimensionsWithMeasureAsRowConverter : RowAndColumnDimensionsWithMeasureAsColumnConverter;
                } else if (!PivotTableUtils.hasRows(chartDef) && PivotTableUtils.hasColumns(chartDef)) {
                    Converter = OnlyColumnDimensionWithMeasureAsRowConverter;
                } else if (PivotTableUtils.hasRows(chartDef) && !PivotTableUtils.hasColumns(chartDef)) {
                    Converter = OnlyRowDimensionWithMeasureAsColumnConverter;
                } else {
                    // No row, no column => line of totals
                    Converter = NoDimensionWithMeasureOnlyConverter;
                }
                return new Converter(chartDef, chartStore, chartData, data, chartHandler, tooltips, contextualMenu, $rootElement);
            }

            return {
                getAgGridConverter
            };
        });
})();
