(function() {
    "use strict";
    const app = angular.module("dataiku.datasets");
    /**
     * Self controlled promotion banner for promoting plugins in dataset upload screens
     * For now the component is specifically designed for Excel Plugin Importer promotion but
     * it is easily made generalized in the future if needed
     * @template dataset-plugin-promotion-banner
     * @param {string} context context key to detect if the banner is permanently hidden
     * @param {Set<string>} datasetFormatTypes extensions of files in current context
     */

    /**
     * @typedef DatasetPluginPromotionBannerController
     * @param {string} context context key to detect if the banner is permanently hidden
     * @param {Set<string>} datasetFormatTypes extensions of files in current context
     * @param {boolean} neverShowAgainLocalStorageKey whether the user has toggled 'Never show again'. Synced with LocalStorage flag
     * @param {boolean} isClosedByUser whether the banner has been manually closed by the user. If true then the banner will never be shown again until the next component render
     * @param {boolean} fileFormatMatchesPluginPromotion whether the file formats need a plugin promotion
     * @param {(datasetFormatTypes: Set<string>) => void} updatePluginPromotionVisibility toggle banner display based on the file formats
     * @param {() => void} close hide the banner. It will not be shown again until the next component render
     * @param {() => void} triggerNeverShowAgain toggle 'Never show again' option. The banner will be permanently hidden
     */
    app.component('datasetPluginPromotionBanner', {
        templateUrl: '/static/dataiku/js/datasets/components/dataset-plugin-promotion-banner/dataset-plugin-promotion-banner.component.html',
        transclude: true,
        bindings: {
            datasetFormatTypes: '<',
            context: '@'
        },
        controller: function(LocalStorage, Logger, FileExtensionService) {
            var $ctrl = this;
            
            $ctrl.$onInit = () => {
                if (!$ctrl.context) {
                    throw new Error('Dataset Plugin Promotion Banner Context unset');
                }
                $ctrl.neverShowAgainLocalStorageKey = `dss.datasets.${$ctrl.context}.pluginPromotionBanner.neverShowAgain`;
                $ctrl.isClosedByUser = false;
                // OR assignment because maybe there is an excel file and this flag has been toggled by $onChanges
                $ctrl.fileFormatMatchesPluginPromotion |= false;
                try {
                    $ctrl.neverShowAgain = LocalStorage.get($ctrl.neverShowAgainLocalStorageKey) || false;
                } catch (e) {
                    Logger.warn('Could not fetch dataset plugin promotion banner display preference from local storage. Showing banner by default', e);
                    $ctrl.neverShowAgain = false;
                }
            }

            $ctrl.$onChanges = ({datasetFormatTypes}) => {
                if (datasetFormatTypes) {
                    $ctrl.updatePluginPromotionVisibility(datasetFormatTypes.currentValue);
                }
            }

            $ctrl.updatePluginPromotionVisibility = (datasetFormatTypes) => {
                $ctrl.fileFormatMatchesPluginPromotion = datasetFormatTypes && FileExtensionService.hasExcel(datasetFormatTypes);
            }

            $ctrl.close = () => {
                $ctrl.isClosedByUser = true;
            }

            $ctrl.triggerNeverShowAgain = () => {
                LocalStorage.set($ctrl.neverShowAgainLocalStorageKey, true)
                $ctrl.isClosedByUser = true;
            }
        }
    });    
})();
