#' Returns various information about the user currently authenticated using
#' this process
#'
#' This method returns a dict that may contain the following names (may also contain others):
#'
#' -  authIdentifier: login for a user, id for an API key
#' - groups: list of group names (if  context is an user)
#' - secrets: list of dicts containing user secrets (if context is an user)
#'
#' @param with_secrets: A boolean, Return user secrets
#' @return a dict
#' @export
dkuGetAuthInfo <- function(with_secrets=FALSE) {
    .dku_publicapi__request(GET, "/auth/info", query = list(withSecrets = with_secrets))
}

#' Returns various information about the DSS user authenticated by the dictionary of
#' HTTP headers provided in headers_dict.
#'
#' This method returns a dict that may contain the following names (may also contain others):
#'
#' -  authIdentifier: login for a user, id for an API key
#' - groups: list of group names (if  context is an user)
#' - secrets: list of dicts containing user secrets (if context is an user)
#'
#' @param headers_dict: named list of HTTP headers
#' @param with_secrets: A boolean, Return user secrets
#' @return a dict
#' @export
dkuGetAuthInfoFromBrowserHeaders <- function(headers_dict, with_secrets=FALSE) {
    # don't use dku_publicapi__request to avoid looping infinitely in the get_auth_headers
    location_data <- dku_intercom__get_location_info()
    prefix <- "/dip/publicapi"

    url <- paste0(location_data$backend_url, prefix, "/auth/info-from-browser-headers")

    resp = POST(url, body = RJSONIO::toJSON(headers_dict), query=list(withSecrets=with_secrets), add_headers(.headers = dku_intercom__get_auth_headers_base()), content_type("application/json"), dku_intercom__get_httr_config())
    dku__check_api_error(resp, paste("API call failed on", "/auth/info-from-browser-headers"))
    return (content(resp))
}

#' Returns an API ticket to use to impersonate the user  authenticated by the dictionary of
#' HTTP headers provided in headers_dict.
#'
#' This method returns the secret of an API ticker
#'
#' @param headers_dict: named list of HTTP headers
#' @return a string
#' @export
dkuGetTicketFromBrowserHeaders <- function(headers_dict) {
    # don't use dku_publicapi__request to avoid looping infinitely in the get_auth_headers
    location_data <- dku_intercom__get_location_info()
    prefix <- "/dip/publicapi"

    url <- paste0(location_data$backend_url, prefix, "/auth/ticket-from-browser-headers")

    resp = POST(url, body = RJSONIO::toJSON(headers_dict), query=list(), add_headers(.headers = dku_intercom__get_auth_headers_base()), content_type("application/json"), dku_intercom__get_httr_config())
    dku__check_api_error(resp, paste("API call failed on", "/auth/ticket-from-browser-headers"))
    return (content(resp)$msg)
}

#' Returns various information about the DSS user authenticated by the headers of a Rook request
# provided in request
#'
#' This method returns a dict that may contain the following names (may also contain others):
#'
#' -  authIdentifier: login for a user, id for an API key
#' - groups: list of group names (if  context is an user)
#' - secrets: list of dicts containing user secrets (if context is an user)
#'
#' @param request: a Rook request
#' @param with_secrets: A boolean, Return user secrets
#' @return a dict
#' @export
dkuGetAuthInfoFromRookRequest <- function(request, with_secrets=FALSE) {
  headers <- list()
  headers$Cookie <- request$HTTP_COOKIE
  dkuGetAuthInfoFromBrowserHeaders(headers, with_secrets)
}
