/*
 * Decompiled with CFR 0.152.
 */
package com.dataiku.dip.sanitycheck.detectors;

import com.dataiku.dip.coremodel.InfoMessage;
import com.dataiku.dip.dao.GeneralSettingsDAO;
import com.dataiku.dip.sanitycheck.SanityCheckDetectorBase;
import com.dataiku.dip.sanitycheck.SanityCheckInfoMessages;
import com.dataiku.dip.server.services.TransactionService;
import com.dataiku.dip.transactions.ifaces.Transaction;
import com.dataiku.dip.utils.DKULogger;
import com.dataiku.dip.utils.OperatingSystemInformation;
import java.io.IOException;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

@Component
public class CGroupsConfigurationDetector
implements SanityCheckDetectorBase {
    private static final String CGROUP_PATH_USER_PLACEHOLDER = "${user}";
    private static final DKULogger logger = DKULogger.getLogger((String)"dip.sanitycheck.cgroupsdetector");
    private static final Pattern MEMORY_LIMIT_PATTERN = Pattern.compile("^(\\d+)([kKmMgG]?)$");
    private final TransactionService transactionService;
    private final GeneralSettingsDAO generalSettingsDAO;
    private final OperatingSystemInformation operatingSystemInformation;

    @Autowired
    public CGroupsConfigurationDetector(TransactionService transactionService, GeneralSettingsDAO generalSettingsDAO, OperatingSystemInformation operatingSystemInformation) {
        this.transactionService = transactionService;
        this.generalSettingsDAO = generalSettingsDAO;
        this.operatingSystemInformation = operatingSystemInformation;
    }

    public List<InfoMessage.MessageCode> getCodes() {
        return Arrays.asList(Codes.values());
    }

    public SanityCheckInfoMessages runAnalysis(Set<String> exclusionList) {
        SanityCheckInfoMessages messages = new SanityCheckInfoMessages();
        if (!this.operatingSystemInformation.isLinux()) {
            return messages;
        }
        try {
            this.generateInfoMessages().filter(infoMessage -> !exclusionList.contains(infoMessage.code)).forEach(arg_0 -> ((SanityCheckInfoMessages)messages).addMessage(arg_0));
        }
        catch (Exception e) {
            logger.warn((Object)"Sanity check failed", (Throwable)e);
            messages.addMessage(this.createFatalMessage(e));
        }
        return messages;
    }

    private Stream<InfoMessage> generateInfoMessages() throws IOException {
        try (Transaction t = this.transactionService.beginRead();){
            GeneralSettingsDAO.LocalProcessesCgroupsSettings cgroupsSettings = this.generalSettingsDAO.getUnsafe().cgroupSettings;
            logger.debug((Object)"check cgroups enablement");
            if (!cgroupsSettings.enabled) {
                Stream<InfoMessage> stream = Stream.of(new InfoMessage(InfoMessage.Severity.WARNING, (InfoMessage.MessageCode)Codes.WARN_SECURITY_NO_CGROUPS, "It is recommended to enable and configure cgroups in your DSS instance. See <a href=\"https://doc.dataiku.com/dss/latest/operations/cgroups.html\" target=\"_blank\">the documentation</a> for more details."));
                return stream;
            }
            if (this.isMemoryControllerDisabledInV2(cgroupsSettings)) {
                Stream<InfoMessage> stream = Stream.of(new InfoMessage(InfoMessage.Severity.WARNING, (InfoMessage.MessageCode)Codes.WARN_SECURITY_NO_MEMORY_CONTROLLER, "It is recommended to enable memory controller when using cgroups V2. See <a href=\"https://doc.dataiku.com/dss/latest/operations/cgroups.html\" target=\"_blank\">the documentation</a> for more details."));
                return stream;
            }
            Stream<InfoMessage> stream = this.generateDetailedInfoMessages(cgroupsSettings);
            return stream;
        }
    }

    private boolean isMemoryControllerDisabledInV2(GeneralSettingsDAO.LocalProcessesCgroupsSettings cgroupsSettings) {
        List<String> v2ControllersList = Arrays.asList(cgroupsSettings.cgroupsV2Controllers.split(","));
        return cgroupsSettings.cgroupsVersion == GeneralSettingsDAO.LocalProcessesCgroupsSettings.CGroupsVersion.CGROUPS_V2 && !v2ControllersList.contains("memory");
    }

    private Stream<InfoMessage> generateDetailedInfoMessages(GeneralSettingsDAO.LocalProcessesCgroupsSettings cgroupsSettings) {
        Stream<InfoMessage> possibleNoUserIndependentMemoryLimitMessage = !this.containsUserIndependentMemoryLimit(cgroupsSettings) ? Stream.of(new InfoMessage(InfoMessage.Severity.WARNING, (InfoMessage.MessageCode)Codes.WARN_SECURITY_NO_USER_INDEPENDENT_MEMORY_LIMIT, "It is recommended to have an user independent memory limit. See <a href=\"https://doc.dataiku.com/dss/latest/operations/cgroups.html\" target=\"_blank\">the documentation</a> for more details.")) : Stream.of(new InfoMessage[0]);
        Stream<InfoMessage> possibleMemoryLimitIsTooHighMessage = this.findHighestMemorySetting(cgroupsSettings).filter(cGroupSettings -> this.isMemorySettingTooHigh(this.findHighestMemorySetting((GeneralSettingsDAO.CGroupSettings)cGroupSettings, cgroupsSettings.cgroupsVersion))).stream().map(cGroupSettings -> {
            long limit = this.findHighestMemorySetting((GeneralSettingsDAO.CGroupSettings)cGroupSettings, cgroupsSettings.cgroupsVersion);
            long total = this.operatingSystemInformation.getTotalMemoryInBytes();
            String limitStr = this.humanReadableBytes(limit);
            String totalStr = this.humanReadableBytes(total);
            String reason = this.explainTooHigh(limit, total);
            return new InfoMessage(InfoMessage.Severity.WARNING, (InfoMessage.MessageCode)Codes.WARN_SECURITY_MEMORY_LIMIT_TOO_HIGH, String.format("The memory limit for cgroup path <b>\"%s\"</b> is set to <b>%s</b>, which is considered too high given the host total memory (<b>%s</b>).<br>Reason: %s.<br>See <a href=\"https://doc.dataiku.com/dss/latest/operations/cgroups.html\" target=\"_blank\">the documentation</a> for more details.", cGroupSettings.cgroupPathTemplate, limitStr, totalStr, reason));
        });
        Set<String> allCgroupsWithMemorySettings = this.computeAllCgroupsWithMemorySettings(cgroupsSettings);
        Stream possibleProcessTypeWithoutLimitMessages = Arrays.stream(ProcessTypeWithMemoryChecks.values()).flatMap(processType -> this.generateProcessTypeWithoutLimitMessageIfApplicable((ProcessTypeWithMemoryChecks)((Object)processType), cgroupsSettings, allCgroupsWithMemorySettings));
        return Stream.of(possibleNoUserIndependentMemoryLimitMessage, possibleMemoryLimitIsTooHighMessage, possibleProcessTypeWithoutLimitMessages).flatMap(i -> i);
    }

    private Stream<InfoMessage> generateProcessTypeWithoutLimitMessageIfApplicable(ProcessTypeWithMemoryChecks processType, GeneralSettingsDAO.LocalProcessesCgroupsSettings cgroupsSettings, Set<String> allCgroupsWithMemorySettings) {
        Set allPossibleCgroupHierarchyMatches = processType.cgroupSettingsExtractor.apply((GeneralSettingsDAO.LocalProcessesCgroupsSettings)cgroupsSettings).targets.stream().map(processTypeCGroupTarget -> processTypeCGroupTarget.cgroupPathTemplate).flatMap(this::computeAllLevelsForPath).collect(Collectors.toSet());
        boolean isCgroupMatchWithMemorySettingFound = allPossibleCgroupHierarchyMatches.stream().anyMatch(allCgroupsWithMemorySettings::contains);
        return !isCgroupMatchWithMemorySettingFound ? Stream.of(new InfoMessage(InfoMessage.Severity.WARNING, (InfoMessage.MessageCode)Codes.WARN_SECURITY_PROCESS_TYPE_WITHOUT_MEMORY_LIMIT, "No memory limit is applicable to process type \"" + processType.nameInSettingsUI + "\" See <a href=\"https://doc.dataiku.com/dss/latest/operations/cgroups.html\" target=\"_blank\">the documentation</a> for more details.")) : Stream.of(new InfoMessage[0]);
    }

    private Stream<String> computeAllLevelsForPath(String path) {
        return this.computeRecursivelyAllLevelsForPath(Arrays.asList(path.split("/"))).map(pathElementList -> String.join((CharSequence)"/", pathElementList));
    }

    private Stream<List<String>> computeRecursivelyAllLevelsForPath(List<String> pathElementsList) {
        if (pathElementsList.size() == 1) {
            return Stream.of(pathElementsList);
        }
        List<String> parentPath = pathElementsList.subList(0, pathElementsList.size() - 1);
        return Stream.concat(Stream.of(pathElementsList), this.computeRecursivelyAllLevelsForPath(parentPath));
    }

    private Set<String> computeAllCgroupsWithMemorySettings(GeneralSettingsDAO.LocalProcessesCgroupsSettings cgroupsSettings) {
        return cgroupsSettings.cgroups.stream().filter(cGroupSettings -> this.hasMemoryLimit((GeneralSettingsDAO.CGroupSettings)cGroupSettings, cgroupsSettings.cgroupsVersion)).map(cGroupSettings -> cGroupSettings.cgroupPathTemplate).collect(Collectors.toSet());
    }

    private boolean containsUserIndependentMemoryLimit(GeneralSettingsDAO.LocalProcessesCgroupsSettings cgroupsSettings) {
        return cgroupsSettings.cgroups.stream().filter(this::isUserIndependent).anyMatch(cGroupSettings -> this.hasMemoryLimit((GeneralSettingsDAO.CGroupSettings)cGroupSettings, cgroupsSettings.cgroupsVersion));
    }

    private boolean isUserIndependent(GeneralSettingsDAO.CGroupSettings cGroupSettings) {
        return !cGroupSettings.cgroupPathTemplate.contains(CGROUP_PATH_USER_PLACEHOLDER);
    }

    private boolean hasMemoryLimit(GeneralSettingsDAO.CGroupSettings cGroupSettings, GeneralSettingsDAO.LocalProcessesCgroupsSettings.CGroupsVersion cGroupsVersion) {
        String memoryLimitKey = this.getMemoryLimitKey(cGroupsVersion);
        return cGroupSettings.limits.stream().anyMatch(simpleKeyValue -> simpleKeyValue.key.equals(memoryLimitKey) && (cGroupsVersion == GeneralSettingsDAO.LocalProcessesCgroupsSettings.CGroupsVersion.CGROUPS_V2 || cGroupSettings.cgroupPathTemplate.startsWith("memory/")));
    }

    private Optional<GeneralSettingsDAO.CGroupSettings> findHighestMemorySetting(GeneralSettingsDAO.LocalProcessesCgroupsSettings cgroupsSettings) {
        return cgroupsSettings.cgroups.stream().filter(cGroupSettings -> this.hasMemoryLimit((GeneralSettingsDAO.CGroupSettings)cGroupSettings, cgroupsSettings.cgroupsVersion)).max((cgroupSetting1, cgroupSettings2) -> Long.signum(this.findHighestMemorySetting((GeneralSettingsDAO.CGroupSettings)cgroupSetting1, cgroupsSettings.cgroupsVersion) - this.findHighestMemorySetting((GeneralSettingsDAO.CGroupSettings)cgroupSettings2, cgroupsSettings.cgroupsVersion)));
    }

    private long findHighestMemorySetting(GeneralSettingsDAO.CGroupSettings cGroupSettings, GeneralSettingsDAO.LocalProcessesCgroupsSettings.CGroupsVersion cGroupsVersion) {
        String memoryLimitKey = this.getMemoryLimitKey(cGroupsVersion);
        return cGroupSettings.limits.stream().filter(simpleKeyValue -> simpleKeyValue.key.equals(memoryLimitKey)).map(simpleKeyValue -> this.parseMemorySetting(simpleKeyValue.value)).max(Comparator.naturalOrder()).orElse(0L);
    }

    private long parseMemorySetting(String memorySetting) {
        Matcher matcher = MEMORY_LIMIT_PATTERN.matcher(memorySetting);
        if (matcher.matches()) {
            try {
                long numericValue = Long.parseLong(matcher.group(1));
                long multiplier = (long)Math.pow(1024.0, this.powerForMemoryUnit(matcher.group(2)));
                return numericValue * multiplier;
            }
            catch (NumberFormatException e) {
                return 0L;
            }
        }
        return 0L;
    }

    private long powerForMemoryUnit(String unit) {
        switch (unit.toLowerCase()) {
            case "k": {
                return 1L;
            }
            case "m": {
                return 2L;
            }
            case "g": {
                return 3L;
            }
        }
        return 0L;
    }

    private boolean isMemorySettingTooHigh(long memorySettingInBytes) {
        long SIZE_15GB_IN_BYTES = 0x3C0000000L;
        long SIZE_24GB_IN_BYTES = 0x600000000L;
        long SIZE_32GB_IN_BYTES = 0x800000000L;
        long SIZE_40GB_IN_BYTES = 0xA00000000L;
        long SIZE_64GB_IN_BYTES = 0x1000000000L;
        long SIZE_96GB_IN_BYTES = 0x1800000000L;
        long totalMemoryInBytes = this.operatingSystemInformation.getTotalMemoryInBytes();
        if (totalMemoryInBytes < SIZE_32GB_IN_BYTES) {
            return memorySettingInBytes > SIZE_15GB_IN_BYTES;
        }
        if (totalMemoryInBytes <= SIZE_64GB_IN_BYTES) {
            return memorySettingInBytes > SIZE_40GB_IN_BYTES;
        }
        if (totalMemoryInBytes <= SIZE_96GB_IN_BYTES) {
            return memorySettingInBytes > totalMemoryInBytes - SIZE_24GB_IN_BYTES;
        }
        return (double)memorySettingInBytes > (double)totalMemoryInBytes * 0.75;
    }

    private String getMemoryLimitKey(GeneralSettingsDAO.LocalProcessesCgroupsSettings.CGroupsVersion cGroupsVersion) {
        switch (cGroupsVersion) {
            case CGROUPS_V1: {
                return "memory.limit_in_bytes";
            }
            case CGROUPS_V2: {
                return "memory.max";
            }
        }
        return null;
    }

    private String humanReadableBytes(long bytes) {
        if (bytes < 1024L) {
            return bytes + " B";
        }
        int exp = (int)(Math.log(bytes) / Math.log(1024.0));
        String pre = "" + "KMGTPE".charAt(exp - 1);
        return String.format("%.1f %sB", (double)bytes / Math.pow(1024.0, exp), pre);
    }

    private String explainTooHigh(long memorySettingInBytes, long totalMemoryInBytes) {
        long SIZE_15GB = 0x3C0000000L;
        long SIZE_24GB = 0x600000000L;
        long SIZE_32GB = 0x800000000L;
        long SIZE_40GB = 0xA00000000L;
        long SIZE_64GB = 0x1000000000L;
        long SIZE_96GB = 0x1800000000L;
        if (totalMemoryInBytes < SIZE_32GB && memorySettingInBytes > SIZE_15GB) {
            return "limit > 15 GB on a host with less than 32 GB of RAM";
        }
        if (totalMemoryInBytes <= SIZE_64GB && memorySettingInBytes > SIZE_40GB) {
            return "limit > 40 GB on a host with \u2264 64 GB of RAM";
        }
        if (totalMemoryInBytes <= SIZE_96GB && memorySettingInBytes > totalMemoryInBytes - SIZE_24GB) {
            return "limit within 24 GB of total memory (\u2264 96 GB host)";
        }
        if (totalMemoryInBytes > SIZE_96GB && (double)memorySettingInBytes > (double)totalMemoryInBytes * 0.75) {
            return "limit exceeds 75% of total system memory";
        }
        return "limit too close to total available memory";
    }

    static enum Codes implements InfoMessage.MessageCode
    {
        WARN_SECURITY_NO_CGROUPS("Resources Control - Control groups support are disabled", InfoMessage.FixabilityCategory.ADMIN_TROUBLESHOOTING),
        WARN_SECURITY_NO_MEMORY_CONTROLLER("Resources Control - The memory controller is not enabled", InfoMessage.FixabilityCategory.ADMIN_TROUBLESHOOTING),
        WARN_SECURITY_NO_USER_INDEPENDENT_MEMORY_LIMIT("Resources Control - There is no user independent memory limit", InfoMessage.FixabilityCategory.ADMIN_TROUBLESHOOTING),
        WARN_SECURITY_MEMORY_LIMIT_TOO_HIGH("Resources Control - One of the memory limits is too high", InfoMessage.FixabilityCategory.ADMIN_TROUBLESHOOTING),
        WARN_SECURITY_PROCESS_TYPE_WITHOUT_MEMORY_LIMIT("Resources Control - The process type has no memory limit", InfoMessage.FixabilityCategory.ADMIN_TROUBLESHOOTING);

        private final String title;
        private final InfoMessage.FixabilityCategory fixability;

        private Codes(String title, InfoMessage.FixabilityCategory fixability) {
            this.title = title;
            this.fixability = fixability;
        }

        public String getCode() {
            return this.name();
        }

        public String getCodeTitle() {
            return this.title;
        }

        public InfoMessage.FixabilityCategory getFixability() {
            return this.fixability;
        }
    }

    private static enum ProcessTypeWithMemoryChecks {
        IN_MEMORY_MACHINE_LEARNING(cgroupsSettings -> cgroupsSettings.mlKernels, "In-memory machine learning & Generative AI"),
        PYTHON_PLUS_R_RECIPES(cgroupsSettings -> cgroupsSettings.pythonRRecipes, "Python + R recipes"),
        PYSPARK_PLUS_SPARKR_PLUS_SPARKLYR_RECIPES(cgroupsSettings -> cgroupsSettings.pythonRSparkRecipes, "PySpark + SparkR + Sparklyr recipes"),
        PYTHON_SCENARIOS(cgroupsSettings -> cgroupsSettings.pythonScenarios, "Scenarios (Python steps & triggers)"),
        JUPYTER_KERNELS(cgroupsSettings -> cgroupsSettings.jupyterKernels, "Jupyter kernels (Python, R, Scala)"),
        IN_MEMORY_ML_RECIPES(cgroupsSettings -> cgroupsSettings.mlRecipes, "In-memory ML recipes (train, score)"),
        PYTHON_MACROS(cgroupsSettings -> cgroupsSettings.pythonMacros, "Python macros"),
        RMARKDOWN_BUILDERS(cgroupsSettings -> cgroupsSettings.rmarkdownBuilders, "RMarkdown builders"),
        WEBAPP_BACKENDS(cgroupsSettings -> cgroupsSettings.webappDevBackends, "Webapp backends"),
        METRICS_AND_CHECKS(cgroupsSettings -> cgroupsSettings.metricsChecks, "Metrics & Checks"),
        INTERACTIVE_STATISTICS(cgroupsSettings -> cgroupsSettings.eda, "Interactive statistics"),
        IN_MEMORY_STATISTICS_RECIPES(cgroupsSettings -> cgroupsSettings.edaRecipes, "In-memory statistics recipes (incl. PCA)"),
        DEPLOYMENT_HOOKS(cgroupsSettings -> cgroupsSettings.deploymentHooks, "Deployment hooks"),
        DEV_LAMBDA_SERVER(cgroupsSettings -> cgroupsSettings.devLambdaServer, "Dev lambda server"),
        PROJECT_STANDARDS(cgroupsSettings -> cgroupsSettings.projectStandards, "Project standards"),
        CUSTOM_PYTHON_DATA_ACCESS_COMPONENTS(cgroupsSettings -> cgroupsSettings.customPythonDataAccessComponents, "Custom Python data access components (FS providers, datasets, exporters, formats)");

        final Function<GeneralSettingsDAO.LocalProcessesCgroupsSettings, GeneralSettingsDAO.ProcessTypeCGroupSettings> cgroupSettingsExtractor;
        final String nameInSettingsUI;

        private ProcessTypeWithMemoryChecks(Function<GeneralSettingsDAO.LocalProcessesCgroupsSettings, GeneralSettingsDAO.ProcessTypeCGroupSettings> cgroupSettingsExtractor, String nameInSettingsUI) {
            this.cgroupSettingsExtractor = cgroupSettingsExtractor;
            this.nameInSettingsUI = nameInSettingsUI;
        }
    }
}

