(function() {
    'use strict';
    
    const app = angular.module('dataiku.analysis.mlcore');

    app.controller('TimeseriesResidualsController', function($scope, DataikuAPI, FutureProgressModal, PerTimeseriesService, ResidualsPlottingService, SINGLE_TIMESERIES_IDENTIFIER, ExportUtils, Debounce, CreateExportModal, $stateParams, Dialogs, $filter) {
        const BLUE_COLOR = "#1F77B4";
        $scope.badgeIconSettings = {
            style: 'outline',
            label: 'info-circle',
        }
        $scope.timeseriesGraphFilters = {};
        $scope.loading = true;

        $scope.computeResiduals = function(fullModelId) {
            DataikuAPI.ml.prediction.computePerTimeseriesResiduals(fullModelId)
                .then(function (res) {
                    FutureProgressModal.show($scope, res.data, "Compute timeseries residuals").then(function (residuals) {
                        if (residuals) {
                            $scope.residuals = residuals;
                            $scope.false;
                        }
                    });
                }).catch(setErrorInScope.bind($scope)).finally(() => {
                $scope.loading = false;
            });
        }

        $scope.exportResiduals = function() {
            const featureOptions = {
                isDownloadable: true,
                hideAdvancedParameters: true
            };

            CreateExportModal($scope, { title: "Residuals for " }, featureOptions).then(function(params) {
                params.contextProjectKey = $stateParams.projectKey;
                DataikuAPI.ml.exportTimeseriesResiduals($scope.modelData.fullModelId, params).then(function(response) {
                    ExportUtils.defaultHandleExportResult($scope, params, response.data)
                }).catch(function(response) {
                    Dialogs.error($scope, "An error occurred while exporting the model residuals data.", response.data.detailedMessageHTML);
                });
            });
        }

        $scope.exportStats = function () {
            const columns = [
                {name: "jarqueBera", type: "double"},
                {name: "jarqueBeraPValue", type: "double"},
                {name: "skew", type: "double"},
                {name: "kurtosis", type: "double"},
                {name: "ljungBox", type: "double"},
                {name: "ljungBoxPValue", type: "double"},

            ];
            let identifierKeys = [];
            if (!(Object.keys($scope.residuals)[0] === SINGLE_TIMESERIES_IDENTIFIER)) {
                const oneIdentifierParsedKey = JSON.parse(Object.keys($scope.residuals)[0])

                identifierKeys = Object.keys(oneIdentifierParsedKey).map(k => {
                    return {name: k, type: "string"}
                });
            }

            const data = [];
            Object.entries($scope.residuals).forEach(([unparsedIdentifier, residualsObj]) => {
                const parsedIdentifier = identifierKeys.length ? JSON.parse(unparsedIdentifier) : {};
                const row = [
                    residualsObj["stats"]["jarqueBera"],
                    residualsObj["stats"]["jarqueBeraPValue"],
                    residualsObj["stats"]["skew"],
                    residualsObj["stats"]["kurtosis"],
                    residualsObj["stats"]["ljungBox"],
                    residualsObj["stats"]["ljungBoxPValue"],
                ];
                identifierKeys.forEach(id => row.push(parsedIdentifier[id["name"]]));
                data.push(row);
            });

            ExportUtils.exportUIData($scope, {
                name: "Computed stats for: " + $scope.modelData.userMeta.name,
                columns: columns.concat(identifierKeys),
                data: data,
            }, "Export computed stats");
        }

        $scope.getResiduals = function(fullModelId, identifiers) {
            DataikuAPI.ml.prediction.getPerTimeseriesResiduals(fullModelId, identifiers)
                .then((res) => {
                    $scope.residuals = res.data;
                }).finally(() => {
                $scope.loading = false;
            });
        }

        $scope.initView = function() {
            $scope.allDisplayedTimeseriesIdentifiers = [];
            $scope.residualsHistograms = {};
            $scope.residualsOverTime = {};
            $scope.qqplots = {};
            $scope.correlograms = {};
            const isDashboardTile = !!$stateParams.dashboardId;
            $scope.objectId = isDashboardTile ? `insightId.${$scope.insight.id}` : `fullModelId.${$scope.modelData.fullModelId}`;

            const unparsedTimeseriesArray = Object.keys($scope.residuals);

            if (unparsedTimeseriesArray.length > 1) {
                $scope.allTimeseriesIdentifierValuesMap = PerTimeseriesService.initTimeseriesIdentifiersValues($scope.modelData.coreParams.timeseriesIdentifiers);
                unparsedTimeseriesArray.forEach(function(unparsedTimeseriesIdentifier) {
                    const parsedTimeseriesIdentifier = JSON.parse(unparsedTimeseriesIdentifier);
                    PerTimeseriesService.addIdentifierValues(parsedTimeseriesIdentifier, $scope.modelData.coreParams.timeseriesIdentifiers, $scope.allTimeseriesIdentifierValuesMap);
                });
                PerTimeseriesService.removeDuplicatesAndSortIdentifierValuesForFilterDropdowns($scope.allTimeseriesIdentifierValuesMap);
            } else {
                // there's no time series identifiers filter to trigger this so we do it manually here
                $scope.updateDisplayedTimeseries();
            }
        }

        function getResidualOverTime(timeseriesResiduals) {
            const padToLengthTwo = dateElem => dateElem.toString().padStart(2, '0');
            const startMaxNumberOfPoints = 500;
            return {
                grid: { top: 32, bottom: 64, right: 8, left: 48 },
                title: {
                    text: "Standardized residuals over time",
                    textAlign: "left",
                    textStyle: { fontWeight: 'bold', fontSize: 12 }
                },

                xAxis: {
                    name: "Standardized residual value",
                    type: 'time',
                    nameLocation: "start",
                    nameGap: 32,
                    nameRotate: 90,
                    position: "bottom",
                },
                yAxis: {
                    type: 'value',
                },
                tooltip: { trigger: 'axis', confine: true },
                dataZoom: [
                    {
                        type: 'inside',
                        startValue: new Date(timeseriesResiduals["dates"][timeseriesResiduals["dates"].length - startMaxNumberOfPoints]),
                        endValue: new Date(timeseriesResiduals["dates"][timeseriesResiduals["dates"].length - 1]),
                    },
                    {
                        type: "slider",
                        startValue: new Date(timeseriesResiduals["dates"][timeseriesResiduals["dates"].length - startMaxNumberOfPoints]),
                        endValue: new Date(timeseriesResiduals["dates"][timeseriesResiduals["dates"].length - 1]),
                        textStyle: { fontSize: 11 },
                        labelFormatter: function(timestamp) {
                            const date = new Date(timestamp);
                            const year = date.getFullYear();
                            const month = padToLengthTwo(date.getMonth() + 1);
                            const day = padToLengthTwo(date.getDate());
                            const hour = padToLengthTwo(date.getHours());
                            const minute = padToLengthTwo(date.getMinutes());
                            const second = padToLengthTwo(date.getSeconds());
                            return `${year}-${month}-${day}\n${hour}:${minute}:${second}`;
                        },
                        left: 64,
                        right: 64,
                    }
                ],
                series: [
                    {
                        name: "Residuals",
                        data: timeseriesResiduals["stdResiduals"].map((r, idx) => [timeseriesResiduals["dates"][idx], r]),
                        type: 'line',
                        symbolSize: 4,
                        color: BLUE_COLOR,
                        sampling: 'lttb',
                    },
                ]
            }
        }

        function getCorrelogramConfig(timeseriesResiduals) {
            const series = "acf" in timeseriesResiduals["stats"] ?
                [
                    {
                        type: 'bar',
                        data: timeseriesResiduals["stats"]["acf"]["x"].map((v, idx) => [idx, v]),
                        barWidth: 2,
                        color: BLUE_COLOR
                    },
                    {
                        type: 'scatter',
                        data: timeseriesResiduals["stats"]["acf"]["x"].map((v, idx) => [idx, v]),
                        color: BLUE_COLOR
                    }
                ] : [];
            const title = "acf" in timeseriesResiduals["stats"] ?
                {
                    text: "Correlogram (up to 10 lags)",
                    textAlign: "left",
                    textStyle: { fontWeight: 'bold', fontSize: 12 }
                } :
                {
                    text: "All residuals are 0",
                    left: 'center',
                    top: 'center',
                    textStyle: { fontWeight: 'bold', fontSize: 12 }
                }
            return {
                grid: { top: 32, bottom: 40 , right: 8, left: 48 },
                title,
                yAxis: {
                    type: 'value',
                    name: 'Correlation',
                    nameLocation: "center",
                    nameGap: 32
                },
                xAxis:{
                    type: 'value',
                    name: 'Lag',
                    nameLocation: "center"
                },
                tooltip: {
                    trigger: 'item', confine: true
                },
                series
            }
        }
        
        $scope.nicePrecision = n => $filter('nicePrecision')(n, 4);
        
        $scope.getTitle = (unparsedTsIdentifier) => $filter('displayTimeseriesName')(unparsedTsIdentifier);

        $scope.updateDisplayedTimeseries = function () {
            if (!$scope.residuals) return;
            const newlyDisplayedTimeseries = [];
            for (const unparsedTimeseriesIdentifier in $scope.residuals) {
                if (PerTimeseriesService.shouldDisplayTimeseries(unparsedTimeseriesIdentifier, $scope.timeseriesGraphFilters)) {
                    newlyDisplayedTimeseries.push(unparsedTimeseriesIdentifier);
                }
            }
            $scope.allDisplayedTimeseriesIdentifiers = newlyDisplayedTimeseries.sort();
            $scope.allDisplayedTimeseriesIdentifiers.forEach(unparsedTimeseriesIdentifier => {
                const currentResiduals = $scope.residuals[unparsedTimeseriesIdentifier];
                if (!(unparsedTimeseriesIdentifier in $scope.residualsHistograms)) {
                    $scope.residualsHistograms[unparsedTimeseriesIdentifier] = ResidualsPlottingService.createResidualsBarChartConfig(currentResiduals.stdResiduals, true, null, null);
                }
                if (!(unparsedTimeseriesIdentifier in $scope.residualsOverTime)) {
                    $scope.residualsOverTime[unparsedTimeseriesIdentifier] = getResidualOverTime(currentResiduals);
                }
                if (!(unparsedTimeseriesIdentifier in $scope.qqplots)) {
                    $scope.qqplots[unparsedTimeseriesIdentifier] = ResidualsPlottingService.getQQPlotConfig(currentResiduals.stdResiduals, currentResiduals.theoreticalQuantiles);

                }
                if (!(unparsedTimeseriesIdentifier in $scope.correlograms)) {
                    $scope.correlograms[unparsedTimeseriesIdentifier] = getCorrelogramConfig(currentResiduals);
                }
            });
        };

        $scope.$on('timeseriesIdentifiersFiltersUpdated', function(event, filters) {
            $scope.timeseriesGraphFilters = filters;
            $scope.updateDisplayedTimeseries();
        });

    });

    app.component("residualsStatsTable", {
        bindings: {
            tableTitle: '@',
            stats: '<'
        },
        templateUrl: "/templates/ml/prediction-model/timeseries/stats-table.html",
        controller: function() {
            const $ctrl = this;
        }
    });

})();
