(function() {
    'use strict';

    angular.module('dataiku.dashboards').component('dashboardGrid', {
        bindings: {
            editable: '<', // boolean
            minHeight: '<', // number
            minWidth: '<', // number
            minNumberOfRows: '<', // number
            padding: '<', // number
            tileSpacing: '<', // number
            gridColor: '<', // string
            addExtraGridSpaceBelow: '<', // boolean
            backgroundColor: '<', // string
            columnNumber: '<', // number
            showGrid: '<', // boolean
            isInGroupTile: '<', // boolean
            tiles: '<', // DashboardTile[]
            page: '<', // DashboardPage
            preselectedTile: '<', // DashboardTile
            autoStackUp: '<', // boolean
            canExportDatasets: '<', // boolean
            insightsMap: '<', // { [insightId: string]: Insight }
            accessMap: '<', // { [insightId: string]: string }
            activeFilters: '<',
            filters: '<',
            dashboardTheme: '<',
            canEditGrid: '<',
            loadingStateChange: '&', // ({ $isLoading }) => void
            tilesChange: '&', // ({ $tiles }) => void
            filtersChange: '&', // ({ $filters }) => void
            filtersParamsChange: '&', // ({ $filtersParams }) => void,
            raiseError: '&', // ({ $errorData }) => void,
            isDeactivatedChange: '&', // ({ $isDeactivated }) => void
            preselectedTileChange: '&', // ({ $preselectedTile }) => void
            gridInit: '&', // ({$api: { reload: () => void }}),
            toggleGroupTileScrollBarDisplay: '&', // ({ $forceShow: boolean, $scrollTopOnHide: boolean }) => void
            onEditGridClick: '&', // () => void
            onAddElementClick: '&' // () => void
        },
        templateUrl: '/static/dataiku/js/dashboards/components/dashboard-grid/dashboard-grid.component.html',
        controller: function(DashboardFilters, $timeout, $rootScope, DashboardUtils, DashboardPageUtils, TileLoadingState, TileLoadingBehavior, $element, $scope, $q, ContextualMenu, TileUtils, LinkedList, DetectUtils, Debounce, WT1, Logger, translate) {
            const ctrl = this;

            /*
             * Tile IDs are used to track the tiles, so they need to be defined before they are used by the ng-repeat.
             * To be sure it's the case, we keep a copy of the tile array (still referencing the original tiles) with their IDs and update it when the tiles change.
             */
            ctrl.tileWithIds = [];
            ctrl.gridContainer;
            ctrl.gridListRendered = false;
            ctrl.appConfig = $rootScope.appConfig;
            ctrl.isErrorMap = {};
            ctrl.hook = null;
            ctrl.isLoading = false;
            ctrl.cellWidth = null;
            ctrl.cellHeight = null;
            ctrl.currentGroupTileIdInEditMode = null;
            ctrl.gridMinHeight = null;
            ctrl.TileUtils = TileUtils;
            ctrl.tileMinSizeByTileId = {};
            ctrl.translate = translate;
            ctrl.editButtonLabel = translate('DASHBOARD.EDIT.BUTTON', 'Edit');
            ctrl.emptyStateViewModeDescription = translate('DASHBOARD.PAGE.EMPTY_STATE.VIEW.DESCRIPTION','Switch to edit mode to add tiles.');


            let isTryingToDragLockedTile = false;
            let blockTileDeselection;
            let tileVisibilityObserver = null;
            const isInExport = getCookie('dku_graphics_export') === 'true';
            const nbSimultaneousLoading = $rootScope.appConfig.nbSimultaneousInsightLoading;
            let deferredLoadTileInProgress = false;
            let lockGridResize = false;
            let pendingResizeRequest = false;
            const selectedTileIds = new Set();
            const initializedTileIds = new Set();
            let scrollingContainer = null;
            let resizeObserver = null;
            let grid = null;
            // Flags used to track the selection mode of the tiles for WT1 events
            let hasUsedRectangleSelection = false;
            let hasUsedMultiSelectKey = false;

            /*
             * In export mode we wait for all the tiles to be loaded independently of their visibility
             * In normal mode we only load the visible tiles
             */
            if (isInExport) {
                $scope.$watch('$ctrl.hook.loadStates', triggerIncommingTileLoading, true);
            } else {
                tileVisibilityObserver = getTileVisibilityObserver();
            }

            ctrl.$onInit = function() {
                ctrl.hook = {
                    loadPromises : {},
                    reloadPromises : {},
                    loadStates: {},
                    visibleStates: {},
                    adjacentStates: {},
                    isErrorMap : ctrl.isErrorMap,
                    setErrorInDashboardPageScope : function(data, status, headers, config, statusText) {
                        ctrl.raiseError({ $errorData : { data, status, headers, config, statusText } });
                    },
                    editable: ctrl.editable
                };
                ctrl.gridInit({ $api: { reload: ctrl.reloadTiles, resizeGrid: resizeGridTree } });

                // The parent grid is in charge of resizing itself and then the child grid in group tiles
                if (!ctrl.isInGroupTile) {
                    $(window).on('resize.dashboard', handleResizeEvent);
                }
                initGrid();
            };

            ctrl.$onChanges = function(changes) {
                if (changes.tiles && ctrl.tiles != null) {
                    for (const tile of ctrl.tiles) {
                        tile.$tileId = tile.$tileId != null ? tile.$tileId : TileUtils.getUniqueTileId();
                    }
                    ctrl.tileWithIds = [...ctrl.tiles];
                    if (ctrl.gridListRendered) {
                        syncGridFromTiles();
                    }
                    ctrl.tileMinSizeByTileId = ctrl.tiles.reduce((acc, tile) => {
                        acc[tile.$tileId] = TileUtils.getTileMinSizeThreshold(tile);
                        return acc;
                    }, {});
                }
                if (changes.tileSpacing && ctrl.tileSpacing != null && selectedTileIds.size) {
                    updateSelectedTilesResizeHandles();
                }
                if (changes.preselectedTile &&
                    ctrl.preselectedTile != null &&
                    ctrl.tiles != null &&
                    ctrl.gridListRendered) {
                    if (ctrl.tiles.includes(ctrl.preselectedTile)) {
                        selectedTileIds.clear();
                        if (ctrl.isGroupTile && !ctrl.editable) {
                            $scope.$emit('dashboardEnterGroupTileEditMode', { grid });
                        }
                        $timeout(() => {
                            selectedTileIds.add(ctrl.preselectedTile.$tileId);
                            updateSelectedTilesResizeHandles();
                            ctrl.preselectedTileChange({ $preselectedTile: null });
                        });
                    } else if (ctrl.isInGroupTile && ctrl.editable) {
                        $scope.$emit('dashboardExitGroupTileEditMode', { grid });
                    }
                }
                if (changes.editable && !changes.editable.isFirstChange()) {
                    if (ctrl.gridListRendered) {
                        grid.setIsReadOnly(!ctrl.editable);
                    }
                    if (ctrl.hook != null) {
                        ctrl.hook.editable = ctrl.editable;
                    }
                    if (!ctrl.editable) {
                        selectedTileIds.clear();
                    }
                }
                if (ctrl.gridListRendered &&
                        ((changes.minHeight && !changes.minHeight.isFirstChange()) ||
                        (changes.tileSpacing && !changes.tileSpacing.isFirstChange()) ||
                        (changes.padding && !changes.padding.isFirstChange()) ||
                        (changes.minNumberOfRows && !changes.minNumberOfRows.isFirstChange()))) {
                    grid.options.cellSizeMatchCustomHeight = ctrl.minHeight + ctrl.tileSpacing - ctrl.padding * 2;
                    grid.options.numberOfCellsMatchingCustomHeight = ctrl.minNumberOfRows;
                    grid.options.offsetHeight = ctrl.isInGroupTile ? -ctrl.tileSpacing : 0;
                    grid.options.itemSpacing = ctrl.tileSpacing;
                    resizeGridTree();
                }

                if (changes.columnNumber && !changes.columnNumber.isFirstChange() && ctrl.gridListRendered) {
                    grid.resize(ctrl.columnNumber);
                    $timeout(function() {
                        angular.element(ctrl.gridContainer).scope().$broadcast('resize');
                        syncTilesFromGridDeletionsAndMoves();
                    }, 200);
                }

                if (changes.autoStackUp && !changes.autoStackUp.isFirstChange() && ctrl.gridListRendered) {
                    grid.toggleAutoStackUp(ctrl.autoStackUp);
                    syncTilesFromGridDeletionsAndMoves();
                }
            };

            $scope.$on('dashboardToggleTileGrouping', () => {
                if (ctrl.isInGroupTile || !ctrl.editable) {
                    return;
                }
                if (canGroupSelectedTiles()) {
                    groupSelectedTiles();
                } else if (hasSingleGroupTileSelected()) {
                    ungroupTile(getSelectedTiles()[0]);
                }
            });

            $scope.$on('dashboardUngroupTile', (_, { tile }) => {
                if (ctrl.isInGroupTile || !ctrl.editable) {
                    return;
                }
                ungroupTile(tile);
            });

            $scope.$on('dashboardStackUpGrid', (_, params) => {
                if (params.grid == null || params.grid !== grid) {
                    return;
                }
                stackTilesUp(params.isFromGroupTile || false);
            });

            $scope.$on('dashboardEnterGroupTileEditMode', (_, { grid }) => {
                if (ctrl.isInGroupTile || !ctrl.editable) {
                    return;
                }
                const groupTile = getParentGroupTileFromElement(grid.$element);
                if (groupTile != null) {
                    const element = DashboardPageUtils.getTileElementFromTileId($element, groupTile.$tileId);
                    selectTiles([groupTile], [element], false);
                    enterGroupTileEditModeIfInactive(groupTile.$tileId);
                }
            });

            $scope.$on('dashboardExitGroupTileEditMode', (_, { grid }) => {
                if (ctrl.isInGroupTile) {
                    return;
                }
                const groupTile = getParentGroupTileFromElement(grid.$element);
                if (groupTile != null) {
                    exitGroupTileEditModeIfActive(groupTile.$tileId);
                }
            });

            $scope.$on('dashboardSelectParentGroupTile', (_, { grid }) => {
                const groupTile = getParentGroupTileFromElement(grid.$element);
                if (groupTile != null) {
                    const element = DashboardPageUtils.getTileElementFromTileId($element, groupTile.$tileId);
                    selectTiles([groupTile], [element], false);
                    exitGroupTileEditModeIfActive(groupTile.$tileId);
                }
            });

            $scope.$on('dashboardLockResizeGridTree', () => {
                lockGridResize = true;
            });

            $scope.$on('dashboardUnlockResizeGridTree', () => {
                lockGridResize = false;
            });

            const unregisterDashboardSelectTilesEvent = $rootScope.$on('dashboardSelectTiles', (_, params) => {
                if (params.grid !== grid) {
                    // Unselect tiles from other grids
                    selectedTileIds.clear();
                }
            });

            const unregisterDashboardUnselectTilesEvent = $rootScope.$on('dashboardUnselectTiles', () => {
                exitGroupTileEditModeIfActive();
                selectedTileIds.clear();
                hasUsedRectangleSelection = false;
                hasUsedMultiSelectKey = false;
            });

            $scope.$on('dashboardRemoveFilterTile', () => {
                const filterTile = DashboardFilters.findFiltersTileList(ctrl.tiles);
                if (filterTile) {
                    ctrl.deleteTile(filterTile);
                }
            });

            $scope.$on('dashboardSaved', () => {
                if (ctrl.gridListRendered) {
                    grid.captureItemSizeAndPositionSnapshot();
                }
            });

            ctrl.isTileSelected = (tile) => selectedTileIds.has(tile.$tileId);

            Mousetrap.bind('backspace', function() {
                deleteSelectedTiles();
            });

            ctrl.handleIsTileLockedChange = function(tile) {
                toggleTilesLockedState([tile]);
            };

            ctrl.deleteTile = function(tile) {
                deleteTiles([tile]);
            };

            ctrl.reloadTiles = function() {
                Object.keys(ctrl.hook.loadStates).forEach(key => {
                    ctrl.hook.loadStates[key] = TileLoadingState.WAITING_FOR_RELOAD;
                    ctrl.isErrorMap[key] = false;
                });
                ctrl.loadTiles(DashboardPageUtils.getVisibleTilesToLoadOrReloadPromiseByTileId(ctrl.hook));
            };

            ctrl.loadTiles = function(tileLoadingPromiseByTileId) {
                // Compute how many tiles we can load given the number of already loading tiles
                const nbLoading = DashboardPageUtils.getNumberOfLoadingTiles(ctrl.hook);
                if (nbLoading > nbSimultaneousLoading) {
                    Logger.warn('More tiles than allowed are loading at the same time. There might be an issue.');
                }
                const remainingLoadingSlots = Math.max(0, nbSimultaneousLoading - nbLoading);

                // Build the list of tiles that we should load now.
                const tilesToLoad = DashboardPageUtils.getTilesToLoadOrderedList(tileLoadingPromiseByTileId, ctrl.hook, getTilesById());

                // If there are too many loading requests. Only keeping the first ones
                if (tilesToLoad.length > remainingLoadingSlots) {
                    tilesToLoad.length = remainingLoadingSlots;
                }

                if (tilesToLoad.length > 0) {
                    updateLoadingState(true);
                    for (const i in tilesToLoad) {
                        const tileId = tilesToLoad[i];
                        const d = $q.defer();
                        ctrl.hook.loadStates[tileId] = TileLoadingState.LOADING;
                        const markComplete = tileLoadingPromiseByTileId[tileId](d.resolve, d.reject) !== TileLoadingBehavior.DELAYED_COMPLETE;
                        const tileLoadingCallback = getTileLoadingCallback(tileLoadingPromiseByTileId, tileId, markComplete);
                        d.promise.then(tileLoadingCallback, tileLoadingCallback); // Don't try to reload a tile if an error appear the first time for now.
                    }
                } else {
                    completeTileLoading(tileLoadingPromiseByTileId, 1000);
                }
            };

            ctrl.openGridMenu = function($event) {
                if (!canStackGridTilesUp()) {
                    return;
                }

                const menu = new ContextualMenu({
                    template: '/templates/dashboards/grid-contextual-menu.html'
                });
                const newScope = $rootScope.$new();
                newScope.stackTilesUp = () => stackTilesUp(ctrl.isInGroupTile);
                menu.scope = newScope;
                menu.openAtXY($event.pageX, $event.pageY);
            };

            ctrl.openTileMenu = function($event, tile) {
                $event.stopPropagation();
                const menu = new ContextualMenu({
                    template: '/templates/dashboards/tile-contextual-menu.html'
                });
                // Don't display group tile context menu when the group tile is in edit mode
                if (TileUtils.isGroupTile(tile) && ctrl.currentGroupTileIdInEditMode && tile.$tileId === ctrl.currentGroupTileIdInEditMode) {
                    return;
                }
                const newScope = $rootScope.$new();
                newScope.tileCopy = angular.copy(tile);
                newScope.tile = tile;
                if (ctrl.isTileSelected(tile) && hasMultipleTilesSelected()) {
                    newScope.isMultiSelectionMenu = true;
                    newScope.openMoveCopyTileModal = openMoveCopyTileModal;
                    newScope.deleteSelectedTiles = deleteSelectedTiles;
                    newScope.canLockSelectedTiles = canLockSelectedTiles();
                    newScope.lockSelectedTiles = lockSelectedTiles;
                    newScope.canUnlockSelectedTiles = canUnlockSelectedTiles();
                    newScope.unlockSelectedTiles = unlockSelectedTiles;
                    newScope.canGroupSelectedTiles = canGroupSelectedTiles();
                    newScope.groupSelectedTiles = groupSelectedTiles;
                } else {
                    newScope.isMultiSelectionMenu = false;
                    newScope.openMoveCopyTileModal = openMoveCopyTileModal;
                    newScope.deleteTile = ctrl.deleteTile;
                    newScope.handleIsTileLockedChange = ctrl.handleIsTileLockedChange;
                    newScope.ungroupTile = ungroupTile;
                    newScope.canStackGroupTileChildrenUp = canStackGroupTileChildrenUp(tile);
                    newScope.stackGroupTileChildrenUp = stackGroupTileChildrenUp;
                }

                menu.scope = newScope;
                menu.openAtXY($event.pageX, $event.pageY);
            };

            ctrl.handleMainGridClick = function(event) {
                if (!ignoreMainGridClick(event)) {
                    deselectTiles();
                }
            };

            ctrl.handleTileClick = function(tile, event) {
                if (!ignoreTileSelectionClick(event)) {
                    const multiSelectKeyPressed = isMultiSelectKeyPressed(event);
                    if (isGroupTileEditionClick(tile, multiSelectKeyPressed)) {
                        handleGroupTileEditionClick(tile, event);
                    } else {
                        handleTileSelectionClick(tile, event, multiSelectKeyPressed);
                    }
                    event.stopPropagation();
                }
            };

            ctrl.handleTileMouseDown = function(tile, event) {
                if (!tile.locked || DashboardPageUtils.isPerformingDradAngDropWithinTile(event.target)) {
                    return;
                }
                event.preventDefault();
                const overlayTimeout = $timeout(() => {
                    isTryingToDragLockedTile = true;
                }, 50);

                const onMouseMove = () => {
                    if (isTryingToDragLockedTile) {
                        toggleLockedOverlayOnTile(tile, true);
                    }
                };

                const onMouseUp = () => {
                    toggleLockedOverlayOnTile(tile, false);
                    $timeout.cancel(overlayTimeout);
                    isTryingToDragLockedTile = false;
                    document.removeEventListener('mousemove', onMouseMove);
                    document.removeEventListener('mouseup', onMouseUp);
                };
                document.addEventListener('mousemove', onMouseMove);
                document.addEventListener('mouseup', onMouseUp);
            };

            ctrl.startRectangleSelection = (e) => {
                if (e == null) {
                    return;
                }

                // Allow lasso only in edit mode, on the main grid, when shift key is pressed
                if (!ctrl.editable
                    || ctrl.isInGroupTile
                    || !e.shiftKey
                    || e.target == null
                ) {
                    return;
                }
                // Only respond to left mouse button
                if (e.button !== 0) {
                    return;
                }

                // block text selection
                e.preventDefault();

                const gridElement = ctrl.gridContainer[0];
                const bounds = gridElement.getBoundingClientRect();
                const startX = e.clientX - bounds.left;
                const startY = e.clientY - bounds.top;
                const startScrollY = scrollingContainer.scrollTop;

                let selectionBox = document.createElement('div');
                selectionBox.className = 'selection-box';
                selectionBox.style.left = `${startX}px`;
                selectionBox.style.top = `${startY}px`;
                selectionBox.style.position = 'absolute';
                selectionBox.style.border = '2px dashed #3b99fc';
                selectionBox.style.backgroundColor = 'rgba(59, 153, 252, 0.2)';
                selectionBox.style.pointerEvents = 'none';
                selectionBox.style.zIndex = '100000';

                gridElement.appendChild(selectionBox);
                blockTileDeselection = true;

                const onMouseMove = (evt) => {
                    evt.preventDefault();
                    const x = evt.clientX - bounds.left;
                    const scrollY = scrollingContainer.scrollTop;
                    const y = evt.clientY - bounds.top + scrollY - startScrollY;

                    const width = Math.abs(x - startX);
                    const height = Math.abs(y - startY);

                    selectionBox.style.width = `${width}px`;
                    selectionBox.style.height = `${height}px`;
                    selectionBox.style.left = `${Math.min(x, startX)}px`;
                    selectionBox.style.top = `${Math.min(y, startY)}px`;
                };

                const onMouseUp = (evt) => {
                    document.removeEventListener('mousemove', onMouseMove);
                    document.removeEventListener('mouseup', onMouseUp);
                    evt.preventDefault();
                    if (!selectionBox) {
                        return;
                    }

                    // Determine selection area
                    const rect = selectionBox.getBoundingClientRect();
                    const tileWrappers = gridElement.querySelectorAll(':scope > .tile-wrapper');
                    if (tileWrappers && tileWrappers.length) {
                        const selectedTileWrappers = [...tileWrappers].filter(tileWrapper => {
                            const itemRect = tileWrapper.getBoundingClientRect();
                            // To select what is touching the selection area
                            return (
                                itemRect.right > rect.left &&
                                itemRect.left < rect.right &&
                                itemRect.bottom > rect.top &&
                                itemRect.top < rect.bottom
                            );
                            // To select only what is inside the selection area
                            /*
                             * return (
                             *     itemRect.right < rect.right &&
                             *     itemRect.left > rect.left &&
                             *     itemRect.top > rect.top &&
                             *     itemRect.bottom < rect.bottom
                             * );
                             */
                        });
                        const selectedTiles = selectedTileWrappers.map(selectedTileWrapper => DashboardPageUtils.getTileByElement(ctrl.tiles, selectedTileWrapper));

                        // Remove selection box
                        gridElement.removeChild(selectionBox);
                        selectionBox = null;

                        hasUsedRectangleSelection = true;
                        selectTiles(selectedTiles, selectedTileWrappers, false).then(() => {
                            blockTileDeselection = false;
                        });
                    }
                };

                // Add event listeners
                document.addEventListener('mousemove', onMouseMove);
                document.addEventListener('mouseup', onMouseUp);
            };

            ctrl.scrollToTile = function(tile) {
                return ctrl.editable && tile.$added && ctrl.isTileSelected(tile) && ctrl.currentGroupTileIdInEditMode == null ;
            };

            function stackGroupTileChildrenUp(groupTile) {
                const tileElement = DashboardPageUtils.getTileElementFromTileId($element, groupTile.$tileId);
                const groupTileGridElement = tileElement.find('.dashboard-grid');
                $scope.$broadcast('dashboardStackUpGrid', { grid: groupTileGridElement.getGridListInstance(), isFromGroupTile: true });
            }

            function canStackGroupTileChildrenUp(groupTile) {
                if (ctrl.autoStackUp || !TileUtils.isGroupTile(groupTile)) {
                    return false;
                }
                const tileElement = DashboardPageUtils.getTileElementFromTileId($element, groupTile.$tileId);
                const groupTileGridElement = tileElement.find('.dashboard-grid');
                const groupTileGrid = groupTileGridElement.getGridListInstance();
                return groupTileGrid != null && groupTileGrid.canStackItemsUp();
            }

            function canStackGridTilesUp() {
                if (ctrl.autoStackUp) {
                    return false;
                }
                return grid.canStackItemsUp();
            }

            function stackTilesUp(isFromGroupTile) {
                if (isFromGroupTile) {
                    WT1.event('dashboard-stack-up-group-tile');
                } else {
                    WT1.event('dashboard-stack-up-dashboard-page');
                }
                grid.stackItemsUp();
                syncTilesFromGridDeletionsAndMoves();
            }

            function isGroupTileEditionClick(tile, isMultiSelectKeyPressed) {
                return !isMultiSelectKeyPressed &&
                    ctrl.isTileSelected(tile) &&
                    TileUtils.isGroupTile(tile) &&
                    selectedTileIds.size === 1 &&
                    !blockTileDeselection; // To avoid entering group tile edition mode after resizing it
            }

            function handleTileSelectionClick(tile, event, isMultiSelectKeyPressed) {
                exitGroupTileEditModeIfActive();
                const element = DashboardPageUtils.getTileElementFromTileId($element, tile.$tileId);
                selectTiles([tile], [element], ctrl.isInGroupTile ? false : isMultiSelectKeyPressed);
            }

            function handleGroupTileEditionClick(tile, event) {
                if (isGroupTileInEditMode(tile.$tileId)) {
                    /**
                     * Here the click is on an empty space of a group tile grid while the group tile is in edit mode
                     * - unselect its child tiles
                     * - exit its edit mode
                     * - select the group tile itself
                     */
                    exitGroupTileEditModeIfActive();
                    const element = DashboardPageUtils.getTileElementFromTileId($element, tile.$tileId);
                    selectTiles([tile], [element], false);
                } else {
                    /*
                     * Here the group tile is already selected but not in edit mode
                     * 1. If the click is on one of its child tiles:
                     *   - Enter group tile edit mode
                     *   - Select the child tile
                     * 2. If the click is on an empty space of a group tile grid
                     *   - Only enter group tile edit mode
                     */
                    enterGroupTileEditModeIfInactive(tile.$tileId);
                    const childTileWrapper = DashboardPageUtils.getTileWrapperFromElement(event.target);
                    const childTile = DashboardPageUtils.getTileByElement(tile.grid.tiles, childTileWrapper);
                    if (childTile != null) {
                        ctrl.preselectedTileChange({ $preselectedTile: childTile });
                    }
                }
            }

            function syncGridFromTiles() {
                return $timeout(() => {

                    if ($scope.$$destroyed || grid === null) {
                        return;
                    }

                    grid.synchronizeWithDOM();
                    resetTileVisibilityObserver();
                    if (ctrl.tiles == null || !ctrl.tiles.length) {
                        updateLoadingState(false);
                        return;
                    }
                    const hasUninitializedTiles = initializeTiles();
                    // Grid might have attributed a position to newly added tiles
                    if (hasUninitializedTiles) {
                        syncTilesFromGridDeletionsAndMoves();
                    }
                    updateSelectedTilesResizeHandles();
                });
            }

            function initializeTiles() {
                let hasUninitializedTiles = false;

                ctrl.tiles.forEach(tile => {
                    if (!initializedTileIds.has(tile.$tileId)) {
                        tile.$added = true;
                        hasUninitializedTiles = true;
                        initializedTileIds.add(tile.$tileId);
                    }

                    if (tileVisibilityObserver) {
                        const tileElement = DashboardPageUtils.getTileElementFromTileId($element, tile.$tileId);
                        observeTileVisibilityOnceItsLoadingDataIsReady(tileVisibilityObserver, tile, tileElement);
                    }
                });

                return hasUninitializedTiles;
            }

            function resetTileVisibilityObserver() {
                if (tileVisibilityObserver) {
                    tileVisibilityObserver.disconnect();
                }
            }

            function syncTilesFromGridDeletionsAndMoves() {
                return $timeout(() => {
                    if (grid != null && !$scope.$$destroyed) {
                        const items = grid.gridList.items;
                        const itemById = new Map(items.map(item => [item.id, item]));

                        updateTilesPosition(itemById);
                        const tileIdsToRemoveSet = getTileIdsToRemoveSet(itemById);
                        cleanRemovedTilesFromHookIndexes(tileIdsToRemoveSet);
                        const remainingTiles = ctrl.tiles.filter(tile => !tileIdsToRemoveSet.has(tile.$tileId));
                        ctrl.tilesChange({ $tiles: remainingTiles });
                        $scope.$emit('dashboardSyncModelsDone');
                    } else {
                        cleanHookIndexes();
                    }
                });
            };

            function updateTilesPosition(itemById) {
                ctrl.tiles.forEach((tile) => {
                    if (itemById.has(tile.$tileId)) {
                        const item = itemById.get(tile.$tileId);
                        updateTilePositionFromGridItem(tile, item);
                    }
                });
            }

            function updateTilePositionFromGridItem(tile, item) {
                tile.box.left = item.x;
                tile.box.top = item.y;
                tile.box.width = item.w;
                tile.box.height = item.h;
            }

            function getTileIdsToRemoveSet(itemById) {
                const tileIdsToRemove = new Set();
                ctrl.tiles.forEach((tile) => {
                    if (!tile.$new && !itemById.has(tile.$tileId)) {
                        tileIdsToRemove.add(tile.$tileId);
                    }
                    delete tile.$new;
                });
                return tileIdsToRemove;
            }

            function cleanRemovedTilesFromHookIndexes(tileIdsToRemoveSet) {
                tileIdsToRemoveSet.forEach((tileId) => {
                    initializedTileIds.delete(tileId);
                    delete ctrl.hook.loadPromises[tileId];
                    delete ctrl.hook.reloadPromises[tileId];
                    delete ctrl.hook.loadStates[tileId];
                    delete ctrl.hook.visibleStates[tileId];
                    delete ctrl.hook.adjacentStates[tileId];
                });
            }

            function cleanHookIndexes() {
                if (ctrl.hook == null) {
                    return;
                }
                /*
                 * Manually clear hook indexes to help the garbage collector.
                 *
                 * Tiles and the grid keep circular references through ctrl.hook.
                 * Combined with pending $timeout callbacks, this can prevent
                 * proper cleanup after the component is destroyed.
                 *
                 * To avoid leaks, we explicitly reset these collections (assigning {})
                 * in case any child tile still holds a reference to them.
                 */
                ctrl.hook.loadPromises = {};
                ctrl.hook.reloadPromises = {};
                ctrl.hook.loadStates = {};
                ctrl.hook.visibleStates = {};
                ctrl.hook.adjacentStates = {};
            }

            function updateSelectedTilesResizeHandles() {
                selectedTileIds.forEach(tileId => {
                    const tileElement = DashboardPageUtils.getTileElementFromTileId($element, tileId);
                    const tileWrapper = DashboardPageUtils.getTileWrapperFromElement(tileElement);
                    DashboardPageUtils.positionResizeHandles(tileWrapper, ctrl.tileSpacing);
                });
            }

            function isGroupTileInEditMode(tileId) {
                return ctrl.currentGroupTileIdInEditMode === tileId;
            }

            function hasGroupTileInEditMode() {
                return ctrl.currentGroupTileIdInEditMode != null;
            }

            function enterGroupTileEditModeIfInactive(tileId) {
                if (isGroupTileInEditMode(tileId)) {
                    return;
                }
                if (hasGroupTileInEditMode()) {
                    exitGroupTileEditModeIfActive();
                }
                const tileElement = DashboardPageUtils.getTileElementFromTileId($element, tileId);
                const tileWrapper = DashboardPageUtils.getTileWrapperFromElement(tileElement);
                tileWrapper.addClass('tile-wrapper--no-resize-handle');
                ctrl.currentGroupTileIdInEditMode = tileId;
                grid.blockDragging(tileElement);
            }

            function exitGroupTileEditModeIfActive() {
                if (ctrl.currentGroupTileIdInEditMode == null) {
                    return;
                }
                const groupTile = getTilesById()[ctrl.currentGroupTileIdInEditMode];
                if (groupTile != null && !groupTile.locked) {
                    const tileElement = DashboardPageUtils.getTileElementFromTileId($element, ctrl.currentGroupTileIdInEditMode);
                    const tileWrapper = DashboardPageUtils.getTileWrapperFromElement(tileElement);
                    tileWrapper.removeClass('tile-wrapper--no-resize-handle');
                    grid.unblockDragging(DashboardPageUtils.getTileElementFromTileId($element, ctrl.currentGroupTileIdInEditMode));
                }
                ctrl.currentGroupTileIdInEditMode = null;
            }

            function canLockSelectedTiles() {
                return getSelectedTiles().some(tile => !tile.locked);

            }
            function lockSelectedTiles() {
                toggleTilesLockedState(getSelectedTiles(), true);
            }
            function canUnlockSelectedTiles() {
                return getSelectedTiles().some(tile => tile.locked);

            }
            function unlockSelectedTiles() {
                toggleTilesLockedState(getSelectedTiles(), false);
            }

            function toggleTilesLockedState(tiles, forcedLockedState = null) {
                tiles.forEach(tile => {
                    const el = DashboardPageUtils.getTileElementFromTileId($element, tile.$tileId);
                    const locked = forcedLockedState != null ? forcedLockedState : !tile.locked;
                    tile.locked = locked;
                    if (locked) {
                        grid.lockItem(el);
                    } else {
                        grid.unlockItem(el);
                        // Need to reposition resize handles on unlock
                        const tileWrapper = DashboardPageUtils.getTileWrapperFromElement(el);
                        DashboardPageUtils.positionResizeHandles(tileWrapper, ctrl.tileSpacing);
                    }
                });
                syncTilesFromGridDeletionsAndMoves();
            }

            function hasSingleGroupTileSelected() {
                return selectedTileIds.size === 1 && TileUtils.isGroupTile(getSelectedTiles()[0]);
            }

            function canGroupSelectedTiles() {
                return selectedTileIds.size > 0 && TileUtils.canGroupTiles(getSelectedTiles());
            }

            function groupSelectedTiles() {

                const tilesToGroup = getSelectedTiles();
                const groupTile = TileUtils.createGroupTile(tilesToGroup, { useDashboardSpacing: true, tileSpacing: ctrl.tileSpacing, backgroundColor: ctrl.backgroundColor });

                DashboardUtils.sendWT1TileCreation(groupTile.tileType, {
                    triggeredFrom: 'dashboard-multi-selection',
                    hasUsedRectangleSelection: hasUsedRectangleSelection,
                    hasUsedMultiSelectKey: hasUsedMultiSelectKey,
                    childTileCount: selectedTileIds.size
                });

                deselectTiles();
                tilesToGroup.forEach(tile => {
                    const el = $('[data-id=' + tile.$tileId + ']');
                    grid.deleteItem($(el));
                });
                ctrl.tiles.push(groupTile);
                syncTilesFromGridDeletionsAndMoves();
                if (ctrl.preselectedTileChange) {
                    ctrl.preselectedTileChange({ $preselectedTile: groupTile });
                }
            }

            function ungroupTile(groupTile) {
                if (!TileUtils.isGroupTile(groupTile)) {
                    return;
                }

                if (groupTile.locked) {
                    const el = DashboardPageUtils.getTileElementFromTileId($element, groupTile.$tileId);
                    groupTile.locked = false;
                    grid.unlockItem(el);
                }

                // remove group tile
                const el = $('[data-id=' + groupTile.$tileId + ']');
                grid.deleteItem($(el));

                // move up child tiles to parent
                const newTiles = TileUtils.adaptChildTilesToMainGrid(groupTile);
                ctrl.tiles.push(...newTiles);
                syncTilesFromGridDeletionsAndMoves();
            };

            function openMoveCopyTileModal(tile) {
                $scope.$emit('dashboardOpenMoveCopyTileModal', { tile });
            }

            function hasMultipleTilesSelected(){
                return selectedTileIds.size > 1;
            }

            function getSelectedTiles() {
                return ctrl.tiles.filter(tile => selectedTileIds.has(tile.$tileId));
            }

            function deleteTiles(tiles) {
                tiles.forEach(tile => {
                    const el = $('[data-id=' + tile.$tileId + ']');
                    grid.deleteItem($(el));
                });
                syncTilesFromGridDeletionsAndMoves();
                deselectTiles();
            };

            function deleteSelectedTiles() {
                deleteTiles(getSelectedTiles());
            }

            ctrl.getBackgroundColor = (opacity, backgroundColor) => {
                if (backgroundColor == null) {
                    backgroundColor = '#ffffff';
                }
                const opacityHex = Math.round((opacity != null ? opacity : 1) * 255).toString(16).padStart(2, '0');
                return `${backgroundColor}${opacityHex}`;
            };

            $scope.$on('$destroy', function() {
                $(window).off('resize.dashboard', handleResizeEvent);
                Mousetrap.unbind('backspace');

                if (tileVisibilityObserver != null) {
                    tileVisibilityObserver.disconnect();
                }
                if (resizeObserver != null) {
                    resizeObserver.disconnect();
                }
                if (unregisterDashboardUnselectTilesEvent != null) {
                    unregisterDashboardUnselectTilesEvent();
                }
                if (unregisterDashboardSelectTilesEvent != null) {
                    unregisterDashboardSelectTilesEvent();
                }
                if (scrollingContainer != null) {
                    scrollingContainer.removeEventListener('wheel', handleShiftWheelScroll, { passive: false });
                }
                cleanHookIndexes();
            });

            $element.on('$destroy', function() {
                if (ctrl.gridListRendered) {
                    grid.destroy();
                    ctrl.gridListRendered = false;
                    grid = null;
                }
            });

            function handleResizeEvent(event) {
                if (event.detail && event.detail.skipInDashboards || ctrl.gridContainer == null) {
                    return;
                }
                if (!event.originalEvent || !event.originalEvent.data || !event.originalEvent.data.propagate) {
                    event.stopPropagation();
                    event.stopImmediatePropagation();
                    resizeGridTree();
                }
            }

            function resizeGridTree() {
                if (lockGridResize) {
                    pendingResizeRequest = true;
                    return;
                }
                lockGridResize = true;

                // block css transition on tile size change
                DashboardPageUtils.getTileWrappers($element).addClass('tile-wrapper--no-animation');
                return reflowGridTreeBFSAsync().then(() => {
                    const customEvent = new Event('resize');
                    customEvent.data = { propagate: true };

                    window.dispatchEvent(customEvent);

                    // set back css transition on tile size change
                    DashboardPageUtils.getTileWrappers($element).removeClass('tile-wrapper--no-animation');
                    lockGridResize = false;
                    if (pendingResizeRequest) {
                        pendingResizeRequest = false;
                        resizeGridTree();
                    }
                });
            }

            function reflowGridTreeBFSAsync() {
                if (ctrl.gridContainer == null) {
                    return Promise.resolve();
                }

                const queue = new LinkedList([ctrl.gridContainer]);

                function processNext() {
                    if (!queue.length) {
                        return Promise.resolve();
                    }

                    const currentGridElement = queue.shift();

                    return $timeout(() => {
                        currentGridElement.getGridListInstance().reflow();
                    }).then(() => {
                        currentGridElement.find('.dashboard-grid:not(:has(.dashboard-grid))').each((_, element) => {
                            queue.push($(element));
                        });

                        return processNext();
                    });
                }

                return processNext();
            }

            function selectTiles(tiles, tileElements, isMultiSelectKeyPressed = false) {
                if (tiles == null || tiles.length === 0 || tileElements == null || tileElements.length !== tiles.length) {
                    return Promise.resolve();
                }
                tiles.forEach((tile, i) => {
                    const tileElement = tileElements[i];
                    const tileWrapper = DashboardPageUtils.getTileWrapperFromElement(tileElement);
                    if (ctrl.isTileSelected(tile)) {
                        if (tileWrapper != null && !tileWrapper.hasClass('ui-draggable-dragging') && !tileWrapper.hasClass('ui-resizable-resizing')) {
                            if (!tile.$showLockedOverlay) {
                                tile.$showEditOverlay = true;
                            }
                        }
                    }
                    DashboardPageUtils.positionResizeHandles(tileWrapper, ctrl.tileSpacing);
                });
                hasUsedMultiSelectKey = isMultiSelectKeyPressed;
                const tilesById = getTilesById();
                if (tiles.length === 1) {
                    if (isMultiSelectKeyPressed) {
                        // it might unselect tile if already selected
                        if (selectedTileIds.has(tiles[0].$tileId)) {
                            selectedTileIds.delete(tiles[0].$tileId);
                        } else {
                            selectedTileIds.add(tiles[0].$tileId);
                        }
                    } else {
                        // Here one single tile is selected by a single click
                        hasUsedRectangleSelection = false;
                        selectedTileIds.clear();
                        selectedTileIds.add(tiles[0].$tileId);
                    }
                    if (selectedTileIds.size === 0) {
                        return $timeout(() => {
                            $rootScope.$emit('dashboardUnselectTiles');
                        });
                    } else {
                        const selectedTiles = Array.from(selectedTileIds).map(id => tilesById[id]);
                        return $timeout(() => {
                            $rootScope.$emit('dashboardSelectTiles', { tiles: selectedTiles, grid });
                        });
                    }
                } else {
                    selectedTileIds.clear();
                    tiles.forEach(tile => selectedTileIds.add(tile.$tileId));
                    return $timeout(() => {
                        $rootScope.$emit('dashboardSelectTiles', { tiles, grid });
                    });
                }
            };

            function deselectTiles() {
                if (blockTileDeselection) {
                    return;
                }
                $rootScope.$emit('dashboardUnselectTiles');
            };

            function ignoreTileSelectionClick(event){
                return (!ctrl.editable && ctrl.isInGroupTile) ||
                    event.target.hasAttribute('dashboard-no-select') ||
                    event.target.closest('[dashboard-no-select]') !== null;
            }

            function isMultiSelectKeyPressed(event) {
                return DetectUtils.getOS() === 'macos' ? !!event.metaKey : !!event.ctrlKey;
            }

            function ignoreMainGridClick(event){
                return ctrl.isInGroupTile || isMultiSelectKeyPressed(event) ||
                    (!event.target.classList.contains('grid-display') && !event.target.classList.contains('dashboard-grid'));
            }

            function toggleLockedOverlayOnAllLockedTiles(showLockedOverlay, ignoreGroupTiles = false) {
                $timeout(() => {
                    ctrl.tiles.forEach(tile => {
                        if (!tile.locked) {
                            return;
                        }
                        if (TileUtils.isGroupTile(tile) && ignoreGroupTiles) {
                            return;
                        }
                        if (showLockedOverlay) {
                            tile.$showEditOverlay = false;
                        }
                        tile.$showLockedOverlay = showLockedOverlay;
                    });
                });
            }

            function toggleLockedOverlayOnTile(tile, showLockedOverlay) {
                if (showLockedOverlay === tile.$showLockedOverlay) {
                    return;
                }
                $timeout(() => {
                    if (showLockedOverlay) {
                        tile.$showEditOverlay = false;
                    }
                    tile.$showLockedOverlay = showLockedOverlay;
                });
            }

            function observeTileVisibilityOnceItsLoadingDataIsReady(tileVisibilityObserver, tile, el) {
                if (ctrl.hook.loadPromises && ctrl.hook.loadStates[tile.$tileId] != null) {
                    tileVisibilityObserver.observe(el.get(0));
                } else {
                    const unregisterLoadPromiseWatcher = $scope.$watch('$ctrl.hook.loadStates', function(newValue) {
                        if (newValue && newValue[tile.$tileId] != null) {
                            tileVisibilityObserver.observe(el.get(0));
                            unregisterLoadPromiseWatcher();
                        }
                    }, true);
                }
            }

            function getScrollingContainer() {
                if (ctrl.isInGroupTile) {
                    return $element.closest('.dashboard-tile__group-tile-container')[0];
                } else {
                    return document.querySelector('.dashboard-export-page-wrapper');
                }
            }

            function handleShiftWheelScroll(event) {
                if (event.shiftKey && scrollingContainer != null) {
                    event.preventDefault(); // Prevent horizontal scrolling
                    scrollingContainer.scrollBy(0, event.deltaY); // Scroll vertically
                }
            }

            const dispatchDebouncedResize = Debounce().withDelay(50, 50).wrap(() => {
                $timeout(() => {
                    window.dispatchEvent(new Event('resize'));
                });
            });

            function initGrid() {
                ctrl.gridContainer = $($element.find('.dashboard-grid').addBack('.dashboard-grid')); //addBack to add self
                /*
                 * Instantiating gridList
                 */
                const flashItems = function(items) {
                    // Hack to flash changed items visually
                    for (let i = 0; i < items.length; i++) {
                        (function(el) {
                            el.addClass('changed');
                            setTimeout(function() {
                                el.removeClass('changed');
                            }, 0);
                        })(items[i].$element);
                    }
                };
                $timeout(() => {
                    if ($scope.$$destroyed) {
                        return;
                    }

                    scrollingContainer = getScrollingContainer();
                    grid = ctrl.gridContainer.gridList(
                        {
                            lanes: ctrl.columnNumber,
                            cellSizeMatchCustomHeight: ctrl.minHeight != null ? ctrl.minHeight + ctrl.tileSpacing - ctrl.padding * 2: false,
                            numberOfCellsMatchingCustomHeight: ctrl.minNumberOfRows != null ? ctrl.minNumberOfRows : false,
                            addExtraGridSpaceBelow: ctrl.addExtraGridSpaceBelow,
                            autoStackUp: ctrl.autoStackUp,
                            offsetHeight: ctrl.isInGroupTile ? - ctrl.tileSpacing : 0,
                            // In group tiles the drop zone is the group tile container and not the group tile grid as its scrolling zone might overflow on the main grid
                            dropZone: ctrl.isInGroupTile ? DashboardPageUtils.getTileWrapperFromElement($element) : null,
                            onCellSizeChange: (cellWidth, cellHeight) => {
                                ctrl.cellWidth = cellWidth;
                                ctrl.cellHeight = cellHeight;
                            },
                            onChange: function(changedItems) {
                                flashItems(changedItems);
                            },
                            canDragItemOut: (draggedItem) => {
                                const tile = DashboardPageUtils.getTileByElement(ctrl.tiles, draggedItem.$element);
                                return !TileUtils.isGroupTile(tile) && !TileUtils.isFilterTile(tile);
                            },
                            onBeforeDragItemOut: (_, targetGrid) => {
                                targetGrid.setIsReadOnly(false);
                            },
                            onAfterDragItemOut: () => {
                                if (ctrl.isInGroupTile) {
                                    grid.setIsReadOnly(true);
                                    ctrl.toggleGroupTileScrollBarDisplay({ $forceShow: false, $scrollTopOnHide: true });
                                }
                                disableTileDropIntoGroupTiles();
                            },
                            onBeforeDragItemIn: (draggedItem, sourceGrid) => {
                                // The tile spacing of target grid can be different from the source grid
                                DashboardPageUtils.positionResizeHandles(draggedItem.$element, ctrl.tileSpacing);
                                draggedItem.$element.css('padding', `${ctrl.tileSpacing / 2}px`);

                                if (ctrl.isInGroupTile) {
                                    ctrl.toggleGroupTileScrollBarDisplay({ $forceShow: true });
                                }

                                const sourceGroupTile = getParentGroupTileFromElement(sourceGrid.$element);
                                if (sourceGroupTile == null) {
                                    return;
                                }
                                const groupTileElement = DashboardPageUtils.getTileElementFromTileId($element, sourceGroupTile.$tileId);

                                // Need to lock source group tile to prevent it from moving when positionning element dragged from it
                                if (!sourceGroupTile.locked) {
                                    grid.lockItem(groupTileElement);
                                }
                            },
                            onAfterDragItemIn: (_, sourceGrid) => {
                                const sourceGroupTile = getParentGroupTileFromElement(sourceGrid.$element);
                                if (sourceGroupTile == null) {
                                    return;
                                }
                                const groupTileElement = DashboardPageUtils.getTileElementFromTileId($element, sourceGroupTile.$tileId);

                                // Can unlock back the source group tile
                                if (!sourceGroupTile.locked) {
                                    grid.unlockItem(groupTileElement);
                                }
                                enableTileDropIntoGroupTiles();
                            },
                            onDropItemOut: (draggedItem) => {
                                const tile = DashboardPageUtils.getTileByElement(ctrl.tiles, draggedItem.$element);
                                selectedTileIds.clear();
                                // Cleaning work and tile removal done in drag stop callback method below
                                return { tile };
                            },
                            onDropItemIn: (draggedItem, _, dropOutData) => {
                                disableTileDropIntoGroupTiles();

                                if (ctrl.isInGroupTile) {
                                    ctrl.toggleGroupTileScrollBarDisplay({ $forceShow: false });
                                }

                                // Need the tile to have a new object reference so it gets selected after Angular rendering by calling preselectedTileChange
                                const tile = { ...dropOutData.tile };
                                tile.box.left = draggedItem.x;
                                tile.box.top = draggedItem.y;
                                tile.box.width = draggedItem.w;
                                tile.box.height = draggedItem.h;

                                // AngularJS will recreate the tile element and add it to the DOM itself so it can be tracked by its digest cycle
                                draggedItem.$element.remove();
                                ctrl.tilesChange({ $tiles: [...ctrl.tiles, tile] });
                                $timeout(() => {
                                    if (ctrl.isInGroupTile) {
                                        $scope.$emit('dashboardEnterGroupTileEditMode', { grid });
                                    } else {
                                        exitGroupTileEditModeIfActive(ctrl.isInGroupTile);
                                    }
                                    ctrl.preselectedTileChange({ $preselectedTile: tile });
                                });

                            },
                            direction: 'vertical',
                            readOnly: !ctrl.editable,
                            isChildGrid: ctrl.isInGroupTile,
                            itemSpacing: ctrl.tileSpacing,
                            scrollingContainer: $(scrollingContainer)
                        },
                        {
                            start: function(event) {
                                $rootScope.$broadcast('dashboardLockResizeGridTree');
                                if (event.shiftKey || DashboardPageUtils.isPerformingDradAngDropWithinTile(event.target)) {
                                    return false;
                                }
                                $(event.target).addClass('tile-wrapper--no-animation');
                                const draggedTile = DashboardPageUtils.getTileByElement(ctrl.tiles, event.target);

                                exitGroupTileEditModeIfActive();
                                selectTiles([draggedTile], [event.target], false);

                                // Avoid group tile to enter in edit mode after performing the D&D
                                event.target.setAttribute('dashboard-no-select','');

                                // Avoid deselecting child group tile when cursor is moving out of the group tile
                                blockTileDeselection = true;


                                let ingoreLockedOverlayOnLockedGroupTiles = false;

                                if (!TileUtils.isGroupTile(draggedTile) && !TileUtils.isFilterTile(draggedTile)) {
                                    enableTileDropIntoGroupTiles();
                                    ingoreLockedOverlayOnLockedGroupTiles = true;
                                }

                                if (ctrl.isInGroupTile) {
                                    ctrl.toggleGroupTileScrollBarDisplay({ $forceShow: true });
                                }

                                toggleLockedOverlayOnAllLockedTiles(true, ingoreLockedOverlayOnLockedGroupTiles);

                                $timeout(function() {
                                    draggedTile.$showEditOverlay = false;
                                });

                            },
                            stop: function(event) {
                                if (ctrl.isInGroupTile) {
                                    ctrl.toggleGroupTileScrollBarDisplay({ $forceShow: false });
                                }

                                $rootScope.$broadcast('dashboardUnlockResizeGridTree');
                                if (pendingResizeRequest) {
                                    pendingResizeRequest = false;
                                    resizeGridTree();
                                }
                                $(event.target).removeClass('tile-wrapper--no-animation');
                                toggleLockedOverlayOnAllLockedTiles(false);
                                const draggedTile = DashboardPageUtils.getTileByElement(ctrl.tiles, event.target);

                                if (!TileUtils.isGroupTile(draggedTile)) {
                                    disableTileDropIntoGroupTiles();
                                }
                                syncTilesFromGridDeletionsAndMoves();
                                $timeout(() => {
                                    event.target.removeAttribute('dashboard-no-select');
                                    blockTileDeselection = false;
                                });
                            },
                            distance: 5,
                            scrollSensitivity: ctrl.isGroupTile ? 60 : 20,
                            scroll: true,
                            containment: $('[data-dashboard-grid="main"]')
                        },
                        {
                            resize: function(event) {
                                $rootScope.$broadcast('dashboardLockResizeGridTree');
                                event.stopPropagation();
                                toggleLockedOverlayOnAllLockedTiles(true);
                            },
                            stop: function(event) {
                                $rootScope.$broadcast('dashboardUnlockResizeGridTree');
                                if (pendingResizeRequest) {
                                    pendingResizeRequest = false;
                                    resizeGridTree();
                                }
                                toggleLockedOverlayOnAllLockedTiles(false);
                                syncTilesFromGridDeletionsAndMoves();
                                // For group tiles, its inner-grid will handle itself the resize of its child tiles
                                const resizedTile = DashboardPageUtils.getTileByElement(ctrl.tiles, event.target);
                                if (TileUtils.isGroupTile(resizedTile)) {
                                    return;
                                }
                                // For other tile types, trigger the resize of the tile content
                                blockTileDeselection = true;
                                $timeout(function() {
                                    if (TileUtils.isGroupTile) {
                                        angular.element(event.target).scope().$broadcast('resize');
                                    }
                                    blockTileDeselection = false;
                                }, 200);
                            }
                        }
                    );
                }).then(() => {
                    if (!$scope.$$destroyed && grid !== null) {
                        $rootScope.spinnerPosition = undefined;
                        ctrl.gridListRendered = true;
                        $rootScope.$broadcast('gridListRendered');
                    }
                }).then(() => {
                    syncGridFromTiles();
                }).then(() => {
                    /*
                     * Main dashboard grid resizes itself of scroll bar display change
                     * For inner grid it is the group tile component in charge of that
                     */

                    if (!ctrl.isInGroupTile && !$scope.$$destroyed) {
                        scrollingContainer.addEventListener('wheel', handleShiftWheelScroll, { passive: false });
                        resizeObserver = new ResizeObserver(() => dispatchDebouncedResize(), { box: 'border-box' });
                        resizeObserver.observe(scrollingContainer);
                    }
                    return syncTilesFromGridDeletionsAndMoves();
                });
            };

            function enableTileDropIntoGroupTiles() {
                ctrl.tiles.forEach(currentTile => {
                    if (TileUtils.isGroupTile(currentTile) && !currentTile.locked) {
                        grid.blockItemWhenHoveredByAnotherItem(DashboardPageUtils.getTileElementFromTileId($element, currentTile.$tileId));
                    }
                });
            }

            function disableTileDropIntoGroupTiles() {
                ctrl.tiles.forEach(currentTile => {
                    if (TileUtils.isGroupTile(currentTile) && !currentTile.locked) {
                        grid.unblockItemWhenHoveredByAnotherItem(DashboardPageUtils.getTileElementFromTileId($element, currentTile.$tileId));
                    }
                });
            }

            function getParentGroupTileFromElement(element) {
                const groupTileWrapper = DashboardPageUtils.getTileWrapperFromElement(element);
                return DashboardPageUtils.getTileByElement(ctrl.tiles, groupTileWrapper);
            }

            function updateLoadingState(newVal) {
                ctrl.isLoading = newVal;
                ctrl.loadingStateChange({ $isLoading: newVal });
            }

            /**
             * Return tile loading callback
             * @param {Record<string, Promise<void>>} tileLoadingPromiseByTileId
             * @param {string} tileId
             * @param {boolean} markComplete
             * @returns
             */
            function getTileLoadingCallback(tileLoadingPromiseByTileId, tileId, markComplete = true) {
                return function() {
                    /*
                     * For simple tiles, mark them as fully loaded (i.e. complete). For tiles with delayed
                     * complete events, just mark them as loaded. This way we are still waiting for them but
                     * in the meantime we can load more tiles.
                     */
                    if (ctrl.hook.loadStates[tileId] !== TileLoadingState.COMPLETE) {
                        ctrl.hook.loadStates[tileId] = markComplete ? TileLoadingState.COMPLETE : TileLoadingState.LOADED;
                    }
                    completeTileLoading(tileLoadingPromiseByTileId);
                };
            };

            /**
             * Complete tile loading
             * @param {Record<string, Promise<void>>} tileLoadingPromiseByTileId
             * @param {number} timeout
             */
            function completeTileLoading(tileLoadingPromiseByTileId, timeout) {
                updateLoadingState(DashboardPageUtils.hasLoadingTiles(tileLoadingPromiseByTileId, ctrl.hook));
                if (ctrl.isLoading && !deferredLoadTileInProgress) {
                    // If there are pending loading insights, call us back later
                    deferredLoadTileInProgress = true;
                    const loadTiles = () => {
                        deferredLoadTileInProgress = false;
                        ctrl.loadTiles(tileLoadingPromiseByTileId);
                    };
                    timeout ? $timeout(loadTiles, timeout) : loadTiles();
                }
            }

            /**
             * Returns tile visibility observer
             * @return {IntersectionObserver} tile visibility observer
             */
            function getTileVisibilityObserver() {
                return new IntersectionObserver((entries) => {
                    $timeout(() => {
                        const newVisibleStates = {};
                        entries.forEach((entry) => {
                            // In export mode we set all the tiles as visible
                            if (entry.isIntersecting || isInExport) {
                                newVisibleStates[entry.target.dataset.id] = true;
                            } else {
                                newVisibleStates[entry.target.dataset.id] = false;
                            }
                        });

                        ctrl.hook.visibleStates = {
                            ...ctrl.hook.visibleStates,
                            ...newVisibleStates
                        };

                        ctrl.hook.adjacentStates = DashboardPageUtils.getIsTileAdjacentToVisibleTilesByTileId(ctrl.hook.visibleStates, getTilesById());
                        ctrl.loadTiles(DashboardPageUtils.getVisibleTilesToLoadOrReloadPromiseByTileId(ctrl.hook));
                    });

                }, {
                    root: document.querySelector('.dashboard-export-page-wrapper')
                });
            }

            /**
             * Returns tile by id index
             * @return {Record<string, Tile>} tile by id index
             */
            function getTilesById(){
                return Object.fromEntries(ctrl.tiles.map(tile => [tile.$tileId, tile]));
            }

            /**
             * Watcher callback to trigger tile loading when scope.hook.loadStates is updated.
             * @param {Record<string, TileLoadingState>} nv
             * @param {Record<string, TileLoadingState>} ov
             */
            function triggerIncommingTileLoading(nv, ov) {
                // Waiting for newcomers
                if (!angular.equals(nv, ov)) {
                    $timeout(function() {
                        if (!ctrl.isLoading) {
                            ctrl.loadTiles(ctrl.hook.loadPromises);
                        }
                    });
                }
            }
        }
    });
})();
