(function(){
    'use strict';

    const app = angular.module('dataiku.dashboards');

    /**
     * Was previously located here: src/main/platypus/static/dataiku/js/dashboards/edit.js
     */
    app.factory('TileUtils', function(DashboardUtils, CreateModalFromTemplate, $rootScope, $stateParams, StateUtils, translate) {

        const DEFAULT_BORDER_OPTIONS = {
            color: '#CCCCCC',
            radius: 0,
            size: 1
        };
        const DEFAULT_TITLE_OPTIONS = {
            fontColor: '#000000',
            fontSize: 13
        };

        const getDefaultShowTitleMode = function(insight) {
            return DashboardUtils.getInsightHandler(insight.type).defaultTileShowTitleMode || 'YES';
        };

        const getDefaultTileParams = function(insight) {
            switch(insight.type) {
                case 'text':
                    return {
                        textAlign: 'LEFT',
                        verticalAlign: 'TOP'
                    };
                case 'image':
                case 'iframe':
                case 'group':
                    return {};
                default:
                    var handler = DashboardUtils.getInsightHandler(insight.type);
                    if (handler.getDefaultTileParams) {
                        return handler.getDefaultTileParams(insight);
                    }
                    return angular.copy(handler.defaultTileParams || {});
            }
        };

        const getDefaultTileBox = function(insight) {
            let dimensions;
            switch (insight.type) {
                case 'text':
                    dimensions = [6, 6];
                    break;
                case 'image':
                    dimensions = [9, 9];
                    break;
                case 'iframe':
                    dimensions = [18, 12];
                    break;
                case 'group':
                    dimensions = [18, 12];
                    break;
                default:
                    var handler = DashboardUtils.getInsightHandler(insight.type);
                    if (handler != null && handler.getDefaultTileDimensions) {
                        dimensions = handler.getDefaultTileDimensions(insight);
                    } else {
                        dimensions = handler != null && handler.defaultTileDimensions || [9, 9];
                    }
            }

            return { width: dimensions[0], height: dimensions[1] };
        };

        const getTileMinSizeThreshold = function(tile) {
            const dimensions = getDefaultTileBox(
                tile.insight != null ?
                    tile.insight :
                    tile.tileType === 'INSIGHT' && tile.insightType != null ?
                        { type: tile.insightType } :
                        { type: tile.tileType.toLowerCase() }
            );
            return { minWidth: Math.ceil(dimensions.width / 1.5), minHeight: Math.ceil(dimensions.height / 1.5) };
        };

        const openUploadPictureDialog = function(tile) {
            CreateModalFromTemplate(
                '/templates/widgets/image-uploader-dialog.html',
                $rootScope,
                null,
                function(newScope) {
                    newScope.projectKey = $stateParams.projectKey;
                    newScope.objectType = 'DASHBOARD_TILE';
                    newScope.objectId = tile.imageId;
                    newScope.translate = translate;
                },
                'image-uploader-dialog'
            )
                .then(function(id) {
                    tile.imageId = id;
                });
        };

        // TODO @dashboards move to insight handlers
        const getDefaultTileClickAction = function(insight) {
            switch (insight.type) {
                case 'metrics':
                case 'scenario_last_runs':
                case 'eda':
                    return 'OPEN_INSIGHT';
                case 'chart':
                case 'dataset_table':
                default:
                    return 'DO_NOTHING';
            }
        };

        const copyNonInsightTile = function(tile) {
            if (tile.tileType != 'INSIGHT') {
                const copy = angular.copy(tile);
                copy.box.left = -1;
                copy.box.top = -1;
                copy.clickAction = 'DO_NOTHING';
                return copy;
            }
        };

        const getIsTileInsightEditable = function(insight, editable) {
            if (!insight) {
                return false;
            }
            const canModerateDashboards = $rootScope.projectSummary && $rootScope.projectSummary.canModerateDashboards;
            return editable &&
                DashboardUtils.canEditInsight(insight, canModerateDashboards) &&
                DashboardUtils.hasEditTab(insight);
        };

        const getTargetUiSref = (tile, insight, editable, targetInsight, tabName, section) => {
            if (!insight) {
                return;
            }

            const options = {
                originDashboard: {
                    id: $stateParams.dashboardId,
                    name: $stateParams.dashboardName,
                    pageId: $stateParams.pageId,
                    edit: editable
                }
            };

            if (getIsTileInsightEditable(insight, !!editable)) {
                options.tab = 'edit';
            }
            if (tabName) {
                options.tabSelect = tabName;
            }
            if (section) {
                options.sections = JSON.stringify([section]);
            }

            const insightUiSref = (insightId) => DashboardUtils.isDashboardEmbeddedInAWorkspace()
                ? StateUtils.uiSref.workspaceDashboardInsight($stateParams.workspaceKey, $stateParams.projectKey, $stateParams.objectId, insightId)
                : StateUtils.uiSref.insight(insightId, $stateParams.projectKey, options);

            switch (tile.clickAction) {
                case 'DO_NOTHING':
                case 'OPEN_INSIGHT':
                    options.name = insight.name;
                    return insightUiSref(tile.insightId);
                case 'OPEN_OTHER_INSIGHT':
                    if (!targetInsight) {
                        return null;
                    }
                    options.name = targetInsight.name;
                    return insightUiSref(targetInsight.id || undefined);
            }
        };

        const isGroupTile = (tile) => tile != null && tile.tileType === 'GROUP';

        const isFilterTile = (tile) => tile != null && tile.insightType === 'filters';

        function canConfigureTileDisplay(tile) {
            return !isFilterTile(tile);
        }

        function canConfigureTileTitleAndBorder(tile) {
            return !isFilterTile(tile);
        }

        function canConfigureTileBackgroundOpacity(tile) {
            return tile.tileType === 'TEXT' || tile.insightType === 'chart' || tile.insightType === 'metrics' || isGroupTile(tile);
        }

        function canConfigureTileBackgroundColor(tile) {
            return isGroupTile(tile);
        }

        function canLink(tile) {
            if (isFilterTile(tile) || isGroupTile(tile)) {
                return false;
            } else if (tile.insightType != 'scenario_run_button') {
                return true;
            }
            return tile.displayMode != 'INSIGHT';
        }

        function canGroupTiles(tiles) {
            return tiles != null && tiles.every(tile => !isGroupTile(tile) && !isFilterTile(tile));
        }

        function createGroupTile(childTiles, { useDashboardSpacing, tileSpacing, backgroundColor }) {
            let groupBox = getDefaultTileBox({ type: 'group' });
            if (childTiles && childTiles.length) {
                groupBox = computeGroupTileBox(childTiles);
            }
            const groupTile = {
                tileType: 'GROUP',
                tileParams: getDefaultTileParams({ type: 'group' }),
                box: groupBox,
                backgroundOpacity: 1,
                backgroundColor: backgroundColor,
                useDashboardSpacing: useDashboardSpacing,
                tileSpacing: tileSpacing,
                padding: 0,
                locked: false,
                $new: true,
                autoLoad:true,
                borderOptions: angular.copy(DEFAULT_BORDER_OPTIONS),
                titleOptions: {
                    ...angular.copy(DEFAULT_TITLE_OPTIONS),
                    title: 'Group tile',
                    showTitle: 'NO'
                },
                grid: {
                    tiles: getChildTilesWithWithCoordinatesRelativeToGroupBox(childTiles, groupBox)
                }
            };
            return groupTile;
        }

        function getChildTilesWithWithCoordinatesRelativeToGroupBox(childTiles, groupBox) {
            if (childTiles == null) {
                return [];
            }
            return childTiles.map(childTile => {
                return {
                    ...childTile,
                    box: {
                        width: childTile.box.width,
                        height: childTile.box.height,
                        left: childTile.box.left - groupBox.left,
                        top: childTile.box.top - groupBox.top
                    }
                };
            });
        }

        function computeGroupTileBox(childTiles) {
            const initialBox = {
                left: Infinity,
                top: Infinity,
                right: -Infinity,
                bottom: -Infinity
            };

            // Reduce to find the boundaries
            const { left, top, right, bottom } = childTiles.reduce((acc, tile) => {
                const { left, top, width, height } = tile.box;
                const tileRight = left + width;
                const tileBottom = top + height;

                return {
                    left: Math.min(acc.left, left),
                    top: Math.min(acc.top, top),
                    right: Math.max(acc.right, tileRight),
                    bottom: Math.max(acc.bottom, tileBottom)
                };
            }, initialBox);

            // Calculate the width and height of the bounding box
            const width = right - left;
            const height = bottom - top;

            // Return the group tile box object
            return {
                left,
                top,
                width,
                height
            };
        }

        function adaptChildTilesToMainGrid(groupTile) {
            if (groupTile.grid.tiles == null || groupTile.grid.tiles.length === 0) {
                return [];
            }
            const newTiles = groupTile.grid.tiles.map(childTile => {
                return {
                    ...childTile,
                    $tileId: getUniqueTileId(),
                    $new: true,
                    box: computeUnGroupedChildTileBox(groupTile, childTile)
                };
            });

            // Sort tiles so those with defined `top` and `left` come first so that it get positionned first
            newTiles.sort((a, b) => {
                const aHasPosition = a.box.top !== undefined && a.box.left !== undefined;
                const bHasPosition = b.box.top !== undefined && b.box.left !== undefined;
                return bHasPosition - aHasPosition;
            });
            return newTiles;
        }

        function computeUnGroupedChildTileBox(groupTile, childTile) {
            const childBox = childTile.box;
            const groupBox = groupTile.box;

            // Determine if the child is outside the group's bounds due to scrolling
            const isOutOfGroupVertically =
                        childBox.top > groupBox.height ||
                        childBox.top + childBox.height > groupBox.height;

            // If it's outside the group, reset position properties
            if (isOutOfGroupVertically) {
                return {
                    width: childBox.width,
                    height: childBox.height
                };
            }

            // Otherwise, adjust position relative to the parent grid
            return {
                top: groupBox.top + childBox.top,
                left: groupBox.left + childBox.left,
                width: childBox.width,
                height: childBox.height
            };
        }

        function isChildTileVerticallyOverflowingGroupTile(tile, groupTileHeight) {
            return tile.box.top + tile.box.height > groupTileHeight;
        }

        function getUniqueTileId() {
            return _.uniqueId(`${Date.now().toString()}_`);
        }


        /**
         * Returns a flat list of tiles, including those nested within group tiles
         */
        function getFlattenedTileList(tiles) {
            const flatList = [];
            const stack = [...tiles];

            while (stack.length) {
                const tile = stack.pop();
                flatList.push(tile);

                if (isGroupTile(tile) &&
                    tile.grid?.tiles?.length) {
                    stack.push(...tile.grid.tiles);
                }
            }

            return flatList;
        }

        return {
            DEFAULT_BORDER_OPTIONS,
            DEFAULT_TITLE_OPTIONS,
            newInsightTile : function(insight) {
                return {
                    tileType: 'INSIGHT',
                    insightId: insight.id,
                    insightType: insight.type,
                    tileParams: getDefaultTileParams(insight),
                    displayMode: 'INSIGHT',
                    box: getDefaultTileBox(insight),
                    clickAction: getDefaultTileClickAction(insight),
                    titleOptions: { showTitle: getDefaultShowTitleMode(insight) },
                    resizeImageMode: 'FIT_SIZE',
                    autoLoad: true
                };
            },
            getDefaultTileParams,
            getDefaultTileBox,
            getTileMinSizeThreshold,
            getDefaultTileClickAction,
            openUploadPictureDialog,
            copyNonInsightTile,
            getIsTileInsightEditable,
            getTargetUiSref,
            isGroupTile,
            isFilterTile,
            canConfigureTileDisplay,
            canConfigureTileTitleAndBorder,
            canConfigureTileBackgroundOpacity,
            canConfigureTileBackgroundColor,
            canLink,
            canGroupTiles,
            createGroupTile,
            getUniqueTileId,
            adaptChildTilesToMainGrid,
            isChildTileVerticallyOverflowingGroupTile,
            getFlattenedTileList
        };
    });

})();
