(function() {
'use strict';

const app = angular.module('dataiku.flow.tools', []);

// Magic trick to work around dependency injection order
// because ng2 ToolBridgeService is not initialized yet when bootstrapping ng1 services
app.factory('Ng1ToolBridgeService', function($injector) {
    return () => $injector.get('ToolBridgeService');
});

app.service('FlowToolsRegistry', function() {
    const flowViews = {};
    const flowTools = {};

    this.registerView = function(service) {
        let def = service.getDefinition();
        def.isTool = false;
        flowViews[def.getName()] = def;
    };
    this.registerFlowTool = function(service) {
        let def = service.getDefinition();
        def.isTool = true;
        flowTools[def.getName()] = def;
    };

    this.getDef = function(name) {
        return flowViews[name] || flowTools[name];
    };
    this.getFlowViews = function() {
        return Object.values(flowViews);
    };
});


app.service('FlowToolsLoader', function(FlowToolsRegistry,
    FlowZonesView, TagsView, CustomFieldsView, ConnectionsView, FileformatsView, RecipesEnginesView, RecipesCodeEnvsView,
    PipelinesView, ImpalaWriteModeView, HiveModeView, SparkConfigView,
    PartitioningView, PartitionsView, ScenariosView, WatchView, CountOfRecordsView, FilesizeView,
    CreationView, LastModifiedView, LastBuildView, LastBuildDurationView,
    RecentActivityView, DataQualityView,
    CopyFlowTool, PropagateSchemaFlowTools, CheckConsistencyFlowTool, ColumnUsageView) {

    FlowToolsRegistry.registerView(FlowZonesView);
    FlowToolsRegistry.registerView(TagsView);
    FlowToolsRegistry.registerView(CustomFieldsView);

    FlowToolsRegistry.registerView(ConnectionsView);
    FlowToolsRegistry.registerView(RecipesEnginesView);
    FlowToolsRegistry.registerView(RecipesCodeEnvsView);

    FlowToolsRegistry.registerView(ImpalaWriteModeView);
    FlowToolsRegistry.registerView(HiveModeView);
    FlowToolsRegistry.registerView(SparkConfigView);
    FlowToolsRegistry.registerView(PipelinesView.getService("SPARK_PIPELINES"));
    FlowToolsRegistry.registerView(PipelinesView.getService("SQL_PIPELINES"));

    FlowToolsRegistry.registerView(CreationView);
    FlowToolsRegistry.registerView(LastModifiedView);
    FlowToolsRegistry.registerView(LastBuildView);
    FlowToolsRegistry.registerView(LastBuildDurationView);
    FlowToolsRegistry.registerView(RecentActivityView);

    FlowToolsRegistry.registerView(PartitioningView);
    FlowToolsRegistry.registerView(PartitionsView);
    FlowToolsRegistry.registerView(ScenariosView);

    FlowToolsRegistry.registerView(DataQualityView);
    FlowToolsRegistry.registerView(CountOfRecordsView);
    FlowToolsRegistry.registerView(FilesizeView);
    FlowToolsRegistry.registerView(FileformatsView);

    FlowToolsRegistry.registerView(WatchView);

    FlowToolsRegistry.registerView(ColumnUsageView);

    FlowToolsRegistry.registerFlowTool(CopyFlowTool);
    FlowToolsRegistry.registerFlowTool(CheckConsistencyFlowTool);
    FlowToolsRegistry.registerFlowTool(PropagateSchemaFlowTools);
});


/*
* Note that for now, flow views are simply flow tools
*/
app.service('FlowTool', function($rootScope, Assert, Logger, FlowToolsRegistry, $injector) {
    const svc = this;
    let currentTool = {};

    this.setCurrent = function(tool)  {
        currentTool = tool;
    };

    this.getCurrent = function()  {
        return currentTool;
    };

    this.unactivateCurrentTool = function(redraw = true) {
        if (!currentTool.def) return; // None active
        if (currentTool.def.destroyFlowTool) {
            currentTool.def.destroyFlowTool();
        }
        svc.setCurrent({});
        
        if (redraw) {
            $rootScope.$emit('drawGraph', false, false);
        }
        return currentTool;
    };

    this.activateTool = function(currentToolSession, viewData) {
        Assert.trueish(currentToolSession, 'no currentToolSession');

        Logger.info("Activating tool", currentToolSession, FlowToolsRegistry.registry);

        svc.unactivateCurrentTool(false);

        let def = FlowToolsRegistry.getDef(currentToolSession.type);
        Assert.trueish(def, 'no tool def');
        svc.setCurrent({
            drawHooks: {},
            actionHooks: {},
            type: currentToolSession.type,
            currentSession: currentToolSession,
            def: def
        });

        return def.initFlowTool(currentTool, viewData);
    };


    /*
    * Refresh the flow state only for some views
    * (or all is not specified)
    */
    this.refreshFlowStateWhenViewIsActive = function(viewNames) {
        const currentTool = svc.getCurrent();
        if (currentTool.def && (!viewNames || !viewNames.length || viewNames.includes(currentTool.def.getName()))) {
            currentTool.refreshState();
        }
    };

    $rootScope.$on('flowDisplayUpdated', function() {
        if (currentTool.drawHooks && currentTool.drawHooks.updateFlowToolDisplay) {
            currentTool.drawHooks.updateFlowToolDisplay();
        }
    });

    $rootScope.$on('flowItemClicked', function(evt, evt2, item) {
        if (currentTool.actionHooks && currentTool.actionHooks.onItemClick) {
            currentTool.actionHooks.onItemClick(item, evt2);
        }
    });
});


app.service('FlowViewsUtils', function($stateParams, WT1, DataikuAPI, Logger, MonoFuture, ProgressStackMessageBuilder,
    FlowGraphSelection, FlowGraphFiltering, FlowGraphFolding) {

    this.addFocusBehavior = function(tool) {


        function isItemSelectedbyId(itemId) {

            let val = tool.user.state.valueByNode[itemId];
            if (val === undefined) {
                return false;
            }
            // Multi-valued view (tags, scenarios)
            if (angular.isArray(tool.getRepr(val))) {
                for (let v of tool.getRepr(val)) {
                    if (tool.user.state.focusMap[v]) {
                        return true;
                    }
                }
                return false;
            }
            // Single-valued view
            return tool.user.state.focusMap[tool.getRepr(val)];
        }

        function isItemSelected(item) {
            return isItemSelectedbyId(item.realId);
        }

        function getSelectedIdsList() {
            const selectedIdslist = [];
            Object.keys(tool.user.state.valueByNode).forEach (itemId => {
                if (isItemSelectedbyId(itemId))  selectedIdslist.push(itemId);
            });
            return selectedIdslist;
        }

        tool.user.isFocused = function(val) {
            // Disable focus when in continuous mode
            if (typeof tool.colorScale == 'function' && tool.colorScale().continuous) {
                return true;
            }

            const repr = tool.getRepr(val);
            if (angular.isArray(repr)) {
                let any = false;
                repr.forEach(function(it) {
                    if (tool.user.state.focusMap[it]) {
                        any = true;
                    }
                });
                return any;
            } else {
                return tool.user.state.focusMap[repr];
            }
        };

        tool.user.getFocusedAsList = function() {
            return [];
        };

        //TODO @flow deprecated
        tool.user.getSingleFocused = function() {
            for (let val in tool.user.state.focusMap) {
                if(tool.user.state.focusMap[val]) {
                    return val;
                }
            }
        };

        tool.user.zoomToFocused = function() {
            FlowGraphFolding.ensureNodesNotFolded(getSelectedIdsList());
            let scope = $('#flow-graph').scope();
            if($('#flow-graph svg .focus').length) {
                scope.zoomToBbox(FlowGraphFiltering.getBBoxFromSelector(scope.svg, '.focus'), 1.2);
            } else {
                scope.zoomToBbox(FlowGraphFiltering.getBBoxFromSelector(scope.svg, '.node'), 1.2);
            }
        };

        tool.user.selectFocused = function() {
            WT1.event("flow-view-select-focused", {tool: tool.def.name});
            FlowGraphFolding.ensureNodesNotFolded(getSelectedIdsList());
            FlowGraphSelection.select(isItemSelected);
        };
    };

    //TODO @flow move or rename service
    this.addAsynchronousStateComputationBehavior = function(tool) {
        tool.user.update = function(scope) {
            tool.user.updateStatus.updating = true;
            tool.user.firstUpdateDone = true;
            return MonoFuture(scope).wrap(DataikuAPI.flow.tools.startUpdate)($stateParams.projectKey, tool.def.getName(), tool.user.updateOptions)
            .success(function(data) {
                tool.user.state = data.result;
                tool.drawHooks.updateFlowToolDisplay();
                tool.user.updateStatus.updating = false;
            }).error(function(a,b,c) {
                tool.user.updateStatus.updating = false;
                setErrorInScope.bind(scope)(a,b,c);
            }).update(function(data) {
                tool.user.updateStatus.progress = data.progress;
                tool.user.updateStatus.totalPercent = ProgressStackMessageBuilder.getPercentage(data.progress);
            });
        };
    };

});


app.service('FlowToolsUtils', function(Logger, FlowGraph) {
    const svc = this;

    this.notSoGrey = function(node, elt) {
        svc.colorNode(node, elt, '#ACACAC');
    },

    this.greyOutNode = function(node, elt) {
        svc.colorNode(node, elt, '#DADADA');
    },

    /**
     * 
     * @param {FlowGraphNode} node node object from DSS `FlowGraph.node()`
     * @param {import('./project_flow').D3Node} elt D3 node object
     * @param {?string} color `''` or `undefined` means default color
     * @returns 
     */
    this.colorNode = function(node, elt, color) {
        try {
            if(elt === undefined) {
                return;
            }
            if (node.nodeType == 'LOCAL_DATASET' || node.nodeType == 'FOREIGN_DATASET') {
                elt.style('fill', color);
                if (node.neverBuilt) {
                    elt.select('.never-built-computable .main-dataset-rectangle').style('stroke', color);
                    elt.select('.never-built-computable .nodeicon').style('color', color);
                    elt.select('.never-built-computable .nodelabel-wrapper').style('color', color);
                }
            } else if (node.nodeType == 'LOCAL_MANAGED_FOLDER'  || node.nodeType == 'FOREIGN_MANAGED_FOLDER') {
                elt.style('fill', color);
            } else if (node.nodeType == 'LOCAL_STREAMING_ENDPOINT') {
                elt.style('fill', color);
            } else if (node.nodeType == 'LOCAL_SAVEDMODEL' || node.nodeType == 'FOREIGN_SAVEDMODEL') {
                elt.style('fill', color);
            } else if (node.nodeType == 'LOCAL_MODELEVALUATIONSTORE' || node.nodeType == 'FOREIGN_MODELEVALUATIONSTORE') {
                elt.style('fill', color);
            } else if (node.nodeType == 'LOCAL_GENAIEVALUATIONSTORE' || node.nodeType == 'FOREIGN_GENAIEVALUATIONSTORE') {
                elt.style('fill', color);
            } else if (node.nodeType == 'LOCAL_RETRIEVABLE_KNOWLEDGE' || node.nodeType == 'FOREIGN_RETRIEVABLE_KNOWLEDGE') {
                elt.style('fill', color);
                if (node.neverBuilt) {
                    elt.select('.never-built-computable .main-retrievable-knowledge-rectangle').style('stroke', color);
                    elt.select('.never-built-computable .nodeicon').style('color', color);
                    elt.select('.never-built-computable .nodelabel-wrapper').style('color', color);
                }
            } else if (node.nodeType == 'RECIPE') {
                elt.select('.bzicon').style('fill', color);
            } else if (node.nodeType == 'LABELING_TASK') {
                elt.select('.bzicon').style('fill', color);
            } else if (node.nodeType == 'ZONE') {
                elt.style('background-color', color);
                elt.style('stroke', color);
                const rgbColor = d3.rgb(color);
                const titleColor = (rgbColor.r*0.299 + rgbColor.g*0.587 + rgbColor.b*0.114) >= 128 ? "#000" : "#FFF";
                elt.style('color', titleColor);
            } else {
                Logger.warn("Cannot color node", node);
            }
            elt.select("g, rect").attr("color", color); //text color
        } catch (e) {
            Logger.error("Failed to color node", e);
        }
    }

    // Bottom right colored indicator
    // There might be several (Ex: tags, so there is an index)
    const RADIUS = 6;
    this.addViewValueIndicator = function(elt, color='rgba(0,0,0,0)', idx = 0, onClick) {
        let tsz = elt.select(".tool-simple-zone");

        if (!tsz.empty()) {
            if (idx == 0) {
                tsz.selectAll("*").remove();
            }
            let tszHeight = tsz.attr("data-height");
            tsz.append("circle")
                .attr("cx", RADIUS + 2)
                .attr("cy", tszHeight - RADIUS - idx * (RADIUS*2 + 2))
                .attr("r", RADIUS)
                .attr("fill", color)
                .on("click", onClick)
                ;
        }
    }
});

app.directive('flowToolSupport', function($rootScope, $stateParams, Assert, WT1, Logger, DataikuAPI, FlowToolsRegistry, FlowTool, FlowGraph, ToolBridgeService, $injector) {
    return {
        restrict: 'A',
        link : function(scope, element, attrs) {
            function activateFromStateIfNeeded() {
                Assert.trueish(scope.toolsState, 'no tool state');
                scope.toolsState.otherActive = {}
                $.each(scope.toolsState.active, function(k, v) {
                    if (k != scope.toolsState.currentId) {
                        scope.toolsState.hasOtherActive = true;
                        scope.toolsState.otherActive[k] = v;
                    }
                });

                if (scope.toolsState.currentId) {
                    scope.tool = FlowTool.activateTool(scope.toolsState.active[scope.toolsState.currentId]);
                }
            }

            scope.refreshToolsState = function() {
                DataikuAPI.flow.tools.getSessions($stateParams.projectKey).success(function(data) {
                    scope.toolsState = data;
                    activateFromStateIfNeeded();
                }).error(setErrorInScope.bind(scope));
            };

            /**
             * Register a flow tool start record associated to the project.
             * @param {string} type tool name.
             * @param {Object} data configuration and properties of the tool.
             * @returns {Promise<void>} a promise that will be resolved when the tool has been started successfully and activated.
             */
            scope.startTool = function(type, data) {
                WT1.event("flow-tool-start", {tool: type});
                if (!scope.drawZones.drawZones) {
                    scope.showZones();
                }
                return DataikuAPI.flow.tools.start($stateParams.projectKey, type, data).success(function(data) {
                    scope.toolsState = data;
                    activateFromStateIfNeeded();
                }).error(setErrorInScope.bind(scope));
            };

            scope.$on('projectTagsUpdated', function (e, args) {
                if (scope.tool && scope.tool.type=="TAGS") {
                    scope.tool.refreshState(false, args.updateGraphTags);
                }
            });

            scope.stopAction = function() {
                if (!scope.toolsState || !scope.toolsState.currentId) {
                    Logger.warn('no active tool, cannot stop');
                    return;
                }

                $.each(FlowGraph.get().nodes, function (nodeId) {
                    const nodeElt = FlowGraph.d3NodeWithId(nodeId);
                    nodeElt.classed('focus', false).classed('out-of-focus', false);
                });
                if (scope.tool.type == "PROPAGATE_SCHEMA") { //reset paths colors applied on
                    FlowGraph.getSvg().find('.grey-out-path').each(function () {
                        d3.select(this).classed('grey-out-path', false);
                    });
                }
                scope.tool = FlowTool.unactivateCurrentTool();
                /* if last view was a flow action, we want to refresh the search to get the coloration that matches the query */
                $rootScope.$emit('flowDisplayUpdated', true);
                DataikuAPI.flow.tools.stop($stateParams.projectKey, scope.toolsState.currentId)
                    .success(function (data) {
                        scope.toolsState = data;
                        if (!scope.drawZones.drawZones) {
                            scope.showZones();
                        }
                        activateFromStateIfNeeded();
                    })
                    .error(setErrorInScope.bind(scope));
            };

            scope.activateTool = function(toolId) {
                 DataikuAPI.flow.tools.setActive($stateParams.projectKey, toolId).success(function(data) {
                    scope.toolsState = data;
                    activateFromStateIfNeeded();
                }).error(setErrorInScope.bind(scope));
            };

            const h = $rootScope.$on('stopAction', scope.stopAction);
            scope.$on('$destroy', h);

            scope.refreshToolsState();

            // show zones
            scope.showZones = function () {
                ToolBridgeService.emitShouldDrawZones(true);
            };
            
            scope.startView = function (view) {
                const tool = FlowTool.getCurrent();
                if (tool && tool.def && tool.action) {
                    scope.stopAction();
                } 
                const params = view.getViewParams();
                const options = params ? params.toOptions() : {};
                FlowTool.activateTool(
                    {
                        type: view.getName(),
                        options: options
                    },
                    view.latestData
                );
                // Reset draw zone except for zone view
                if (
                    !scope.drawZones.drawZones &&
                    view.getName() !== 'FLOW_ZONES'
                ) {
                    scope.showZones();
                }
                scope.tool = FlowTool.getCurrent();
                ToolBridgeService.emitViewTool(scope.tool);
            };

            scope.stopView = function () {
                const tool = FlowTool.getCurrent();
                if (!tool || !tool.def || tool.action) {
                    // don't stop an empty tool
                    return;
                }
                scope.tool = FlowTool.unactivateCurrentTool();
                if (!scope.drawZones.drawZones) {
                    scope.showZones();
                }
                ToolBridgeService.emitViewTool(scope.tool);
            };

            var toolBridgeSubscriptions = [];

            /* SUBSCRIPTION AND SIGNALS */
            toolBridgeSubscriptions.push(
                ToolBridgeService.viewActivation$.subscribe(view => {
                    if (view) {
                        scope.startView(view);
                    } else {
                        scope.stopView();
                    }
                })
            );

            toolBridgeSubscriptions.push(
                ToolBridgeService.shouldDrawZones$.subscribe(shouldDrawZones => {
                    if (scope.drawZones.drawZones === shouldDrawZones) {
                        return;
                    }
                    scope.drawZones.drawZones = shouldDrawZones;
                    scope.redrawZone();
                })
            );

            // Cleanup when the scope is destroyed
            scope.$on('$destroy', function() {
                if (toolBridgeSubscriptions) {
                    toolBridgeSubscriptions.forEach(subs => subs.unsubscribe());
                }
                FlowTool.unactivateCurrentTool();
            });
        }
    }
});

app.directive("flowToolFacetElt", function() {
    return {
        template:
            `<label class="horizontal-flex" ng-class="{'single-focused': states[key]}">
                <input type="checkbox" ng-if="!singleFocused" ng-model="states[key]" ng-click="$event.stopPropagation()"/>
                <span class="dib flex horizontal-flex" ng-click="click(key, $event)">
                    <span class="bullet noflex" style="background-color: {{color}};" />
                    <span class="text flex">
                        <span ng-if="!displayGlobalTags">{{displayName ? displayName : (isNumber(key) ? (key | number) : key) }}</span>
                        <span ng-if="displayGlobalTags" ui-global-tag="displayName ? displayName : (isNumber(key) ? (key | number) : key)" object-type="'TAGGABLE_OBJECT'"/>
                    </span>
                    <span class="number noflex">{{number}}</span>
                </span>
            </label>`,
        scope: {
            color: '=',
            key: '=',
            displayName: '=',
            number: '=',
            singleFocused: '=',
            states: '=',
            displayGlobalTags: '='
        },
        link: function(scope, element, attr) {
            scope.click = function(key, evt) {
                if (!scope.states) return;
                $.each(scope.states, function(k) {
                    scope.states[k] = false;
                });
                scope.states[key] = true;
                evt.preventDefault();
                evt.stopPropagation();
            };
            scope.isNumber = n  => angular.isNumber(n);
        }
    }
});
})();
