(function () {
    'use strict';

    const app = angular.module('dataiku.flow.runtime');

    app.service('JobFlowGraphSelection', function (FlowGraph) {

        /* node selection model */

        let selectedNodesByRealId = {}; // real ID is the preferred identifier
        let selectedRecipesByName = {}; // activities refer to recipes by name
        let selectedDatasetsByName = {}; // activities refer to datasets by name

        const select = (node, unselect = false) => {
            selectedNodesByRealId[node.realId] = !selectedNodesByRealId[node.realId] || !unselect;
            if (node.nodeType === 'RECIPE') selectedRecipesByName[node.name] = !selectedRecipesByName[node.name] || !unselect;
            if (node.nodeType === 'LOCAL_DATASET') selectedDatasetsByName[node.name] = !selectedDatasetsByName[node.name] || !unselect;
            return selectedNodesByRealId[node.realId];
        };

        const reset = () => {
            selectedNodesByRealId = {};
            selectedRecipesByName = {};
            selectedDatasetsByName = {};
        };

        const selectSingle = node => {
            reset();
            return select(node);
        };

        const selectMulti = (node) => {
            return select(node, true);
        };

        const selectRect = nodes => {
            nodes.forEach(node => select(node));
            return nodes.length > 0;
        };

        const hasSelection = () => Object.values(selectedNodesByRealId).includes(true);
        const countSelection = () => Object.values(selectedNodesByRealId).reduce((c, s) => c + (s ? 1 : 0), 0);
        const hasSingleSelection = () => countSelection() === 1;

        const getSelectedRecipes = () => Object.keys(selectedRecipesByName).filter(recipeName => selectedRecipesByName[recipeName]);
        const getSelectedDatasets = () => Object.keys(selectedDatasetsByName).filter(datasetName => selectedDatasetsByName[datasetName]);

        /* node selection view */

        this.clearSelection = scope => {
            reset();
            d3.selectAll('#flow-graph .selected:not(.zone_cluster)').classed('selected', false);
            this.clearActivitySelection(scope);
        };

        this.singleSelectionStrategy = scope => ({
            onItemClick: item => {
                this.clearSelection(scope);
                if (item.nodeType === 'ZONE') return;
                const selected = selectSingle(item);
                d3.selectAll(`#flow-graph [data-id="${item.id}"]:not(.zone_cluster), #flow-graph [data-node-id="${item.realId}"]`).classed('selected', selected); // select current item
                this.activitySelectionStrategy(scope);
            }
        });

        this.multiSelectionStrategy = scope => ({
            onItemClick: item => {
                if (item.nodeType === 'ZONE') return;
                const selected = selectMulti(item);
                d3.selectAll(`#flow-graph [data-id="${item.id}"]:not(.zone_cluster), #flow-graph [data-node-id="${item.realId}"]`).classed('selected', selected); // select or unselect current item
                this.activitySelectionStrategy(scope);
            }
        });

        this.rectangularSelectionStrategy = scope => items => {
            this.clearSelection(scope);
            const selected = selectRect(items.filter(item => item.nodeType !== 'ZONE'));
            items.forEach(item => d3.selectAll(`#flow-graph [data-id="${item.id}"]:not(.zone_cluster), #flow-graph [data-node-id="${item.realId}"]`).classed('selected', selected));
            this.activitySelectionStrategy(scope);
        };

        /* activity selection model */

        const selectActivityHook = scope => activity => scope.setSelectedActivity(activity);
        const highlightActivity = activity => activity.highlighted = true;

        const getActivityRecipesName = activity => {
            const recipesName = [];
            if (activity.recipeName) recipesName.push(activity.recipeName);
            if (activity.pipelineRecipes) recipesName.push(...activity.pipelineRecipes.map(recipe => recipe.recipeName));
            return recipesName;
        };

        /* activity selection view */

        this.clearActivitySelection = scope => {
            const activities = scope.getActivities();
            if (!activities) return;
            if (scope.shouldClearSelectedActivity()) selectActivityHook(scope)(null);
            activities.forEach(activity => activity.highlighted = false);
        };

        this.activitySelectionStrategy = scope => {
            const activities = scope.getActivities();
            if (!activities) return;
            // compute activities relates to recipes and datasets included in flow graph selection
            const selectedActivitiesFromRecipes = [];
            const selectedActivitiesFromDatasets = [];
            getSelectedRecipes().forEach(recipeName => Object.values(activities).filter(activity => getActivityRecipesName(activity).includes(recipeName)).forEach(activity => selectedActivitiesFromRecipes.push(activity)));
            getSelectedDatasets().forEach(datasetName => Object.values(activities).filter(activity => activity.targets.filter(target => datasetName === (target.id ? target.id.split('.')[1] : target.datasetName)).length > 0).forEach(activity => selectedActivitiesFromDatasets.push(activity)));
            const selectedActivities = new Set([...selectedActivitiesFromRecipes, ...selectedActivitiesFromDatasets]);
            this.clearActivitySelection(scope);
            if (selectedActivities.size === 1) selectActivityHook(scope)(selectedActivities.values().next().value);
            else selectedActivities.forEach(activity => highlightActivity(activity));
            const firstSelectedActivityIndex = Math.min(selectedActivitiesFromRecipes.length > 0 ? activities.indexOf(selectedActivitiesFromRecipes[0]) : Infinity, selectedActivitiesFromDatasets.length > 0 ? activities.indexOf(selectedActivitiesFromDatasets[0]) : Infinity);
            if (firstSelectedActivityIndex !== Infinity) scope.scrollToActivity(firstSelectedActivityIndex);
            if (scope.showSelectedActivityLog && scope.getSelectedActivity()) scope.showSelectedActivityLog();
        };

        this.zoomOnSelectedActivities = function(factor = 1) {
            FlowGraph.zoomOnSelectedActivities(hasSingleSelection(), factor);
        }

        this.activityRecipesSelectionStrategy = scope => selectedActivity => {
            this.clearSelection(scope);
            if (!FlowGraph.ready() || !selectedActivity) return;
            const activities = Object.values(scope.getActivities());
            getActivityRecipesName(selectedActivity)
                .map(recipeName => {
                    // highlight each activity related to the selected activity
                    activities.filter(activity => getActivityRecipesName(activity).includes(recipeName)).forEach(activity => highlightActivity(activity));
                    return FlowGraph.recipeNodeFromName(recipeName);
                })
                .filter(Boolean)
                .forEach(node => {
                    const selected = selectMulti(node);
                    d3.selectAll(`#flow-graph [data-id="${node.id}"]:not(.zone_cluster), #flow-graph [data-node-id="${node.realId}"]`).classed('selected', selected); // select or unselect current item
                });

        };

    });
})();
