(function() {
    'use strict';

    angular.module('dataiku.charts')
        .factory('GroupedColumnsUtils', GroupedColumnsUtils);

    /**
     * (!) This service previously was in static/dataiku/js/simple_report/column-bars/column.js
     */
    function GroupedColumnsUtils(SVGUtils, ChartMeasure, ChartAxesUtils, ChartsStaticData, CHART_VARIANTS, UnaggregatedMeasureComputeModes) {
        const computeUAValueData = function(genericData, chartData, measure, measureIndex, hasLogScale) {
            const getBarBase = (total) => {
                if (total === 0 && hasLogScale) {
                    return 1;
                }
                return total;
            };
            const result = {
                ...genericData,
                isUnaggregated: true,
                uaComputeMode: measure.uaComputeMode
            };
            const uaValues = chartData.aggr(measureIndex).get();
            if (measure.uaComputeMode === UnaggregatedMeasureComputeModes.STACK) {
                let positiveTotal = 0;
                let negativeTotal = 0;
                result.data = uaValues.map(function(uaValue, idx) {
                    let base = uaValue >= 0 ? positiveTotal : negativeTotal;
                    let top = base + uaValue;

                    if (uaValue >= 0) {
                        base = getBarBase(base);
                        top = getBarBase(top);
                        positiveTotal = top;
                    } else {
                        negativeTotal = top;
                    }
                    return {
                        isUnaggregated: true,
                        isStacked: true,
                        ...genericData,
                        base,
                        top,
                        uaValueIdx: idx
                    };
                });
            } else {
                // overlay mode
                const positiveData = [];
                const negativeData = [];
                uaValues.forEach(function(uaValue, idx) {
                    const commonData = {
                        isUnaggregated: true,
                        base: hasLogScale ? 1 : 0,
                        top: uaValue,
                        ...genericData,
                        uaValueIdx: idx
                    };
                    if (uaValue >= 0) {
                        positiveData.push(commonData);
                    } else {
                        negativeData.push(commonData);
                    }
                });
                result.data = [
                    /*
                     * in Overlay mode, we display one bar per ua value, to handle tooltip and hover effect
                     * but we color only the biggest bar for positive values and for negative values, other bars are transparent by default
                     */
                    ...positiveData.map(function(data, idx) {
                        return {
                            ...data,
                            transparent: idx > 0
                        };
                    }),
                    ...negativeData.reverse().map(function(data, idx) {
                        return {
                            ...data,
                            transparent: idx > 0
                        };
                    })
                ];
            }

            return result;
        };


        return {
            prepareData: function(chartDef, chartData, measureFilter, ignoreLabels = new Set()) {
                const xLabels = chartData.getAxisLabels('x');
                const colorLabels = chartData.getAxisLabels('color') || [null];
                const facetLabels = chartData.getAxisLabels('facet') || [null];
                const animationLabels = chartData.getAxisLabels('animation') || [null];
                const hasLogScale = chartDef.xAxisFormatting.isLogScale || ChartAxesUtils.isYAxisLogScale(chartDef.yAxesFormatting, ChartsStaticData.LEFT_AXIS_ID);
                const getBarBase = (total) => {
                    if (total === 0 && hasLogScale) {
                        return 1;
                    }
                    return total;
                };

                const animationData = { frames: [], maxTotal: 0, minTotal: 0 };
                animationLabels.forEach(function(animationLabel, a) {
                    chartData.fixAxis('animation', a);

                    const frameData = { facets: [], maxTotal: 0, minTotal: 0 };
                    facetLabels.forEach(function(facetLabel, f) {
                        chartData.fixAxis('facet', f);

                        const facetData = { groups: [], maxTotal: 0, minTotal: 0 };
                        let runningTotal = 0; // only usefull for waterfall
                        xLabels.forEach(function(xLabel, x) {
                            chartData.fixAxis('x', x);
                            const columns = [];
                            if (xLabel && ignoreLabels.has(xLabel.label)) {
                                return;
                            }
                            const isTotalBar = chartData.isTotalBar(x);
                            if (isTotalBar) {
                                // for total bar, reset the total to start from the x axis
                                runningTotal = 0;
                            }

                            let cIndex = 0;
                            colorLabels.forEach(function(colorLabel, c) {
                                chartData.fixAxis('color', c);
                                if (colorLabel && ignoreLabels.has(colorLabel.label)) {
                                    return;
                                }
                                chartDef.genericMeasures.forEach(function(measure, measureIndex) {
                                    if (measureFilter && !measureFilter(measure)) {
                                        return;
                                    }
                                    const measureData = {
                                        color: chartData.getColorOfValue(c, x),
                                        measure: measureIndex,
                                        x: x,
                                        colorScaleIndex: chartData.getColorOfValue(cIndex, x)
                                    };

                                    let subTextElementsData = [];
                                    if (chartData.isBinMeaningful(measure.function, null, measureIndex)) {
                                        if (ChartMeasure.isRealUnaggregatedMeasure(measure)) {
                                            // in that case the value is a double array instead of a single value
                                            Object.assign(measureData, computeUAValueData(measureData, chartData, measure, measureIndex, hasLogScale));
                                        } else {
                                            subTextElementsData = SVGUtils.getLabelSubTexts(chartDef, measureIndex, measure.valuesInChartDisplayOptions, false).map(function(subTextElementData) {
                                                return {
                                                    ...subTextElementData,
                                                    // add measure data at sub text level also, to retrieve it easily
                                                    ...measureData
                                                };
                                            });
                                            const value = chartData.aggr(measureIndex).get();
                                            measureData.data = [{ ...measureData, base: getBarBase(runningTotal), top: getBarBase(runningTotal + value) }];
                                            if (chartDef.variant === CHART_VARIANTS.waterfall && !isTotalBar) {
                                                runningTotal += value;
                                            }
                                        }
                                    } else {
                                        // we don't want to display a bar for non meaningful values
                                        measureData.data = [{ ...measureData, base: getBarBase(runningTotal), top: getBarBase(runningTotal) }];
                                    }

                                    columns.push({
                                        ...measureData,
                                        textsElements: subTextElementsData,
                                        valuesInChartDisplayOptions: measure.valuesInChartDisplayOptions
                                    });
                                });
                                cIndex++;
                            });
                            facetData.groups.push({ x: x, columns: columns });
                        });
                        frameData.facets.push(facetData);
                    });
                    animationData.frames.push(frameData);
                });

                return animationData;
            }
        };
    }
})();
