import logging
import sys
import traceback
import json

from dataiku.base.socket_block_link import JavaLink, parse_javalink_args
from dataiku.base.utils import watch_stdin, get_json_friendly_error
from dataiku.doctor.docgen.renderer.tree_renderer import TreeRenderer
from dataiku.doctor.docgen.common.docx_io import DocxIO
from io import open

# socket-based connection to backend
# the goal of this server is to replace the placeholders inside the docx by their values.
def serve(port, secret, server_cert=None):
    link = JavaLink(port, secret, server_cert=server_cert)
    # initiate connection
    link.connect()
    # get work to do
    command = link.read_json()
    try:
        task = command.get("task", "")
        template_path = command.get("templatePath", "")
        resolved_result_path = command.get("resolvedResultPath", "")
        output_file = command.get("outputFile", "")
        project_key = command.get("projectKey", "")
        print("Command received from DSS ", task, template_path, resolved_result_path, output_file, project_key)

        docx_io = DocxIO()
        document = docx_io.load(template_path)
        renderer = TreeRenderer(project_key)


        with open(resolved_result_path, encoding="utf-8") as json_file:
            # This file was generated by the java class "ResolvedResult".
            resolved_placeholders = json.load(json_file)

        # Replace the placeholders
        document = renderer.resolve_placeholders(document, resolved_placeholders)

        # Puppeteer placeholders are not supported for now
        
        docx_io.to_folder_file(document, output_file)

        # send ack
        link.send_json({'ok': True, 'count': 0})
    except:
        link.send_string('') # mark failure
        traceback.print_exc()
        link.send_json(get_json_friendly_error())
    finally:
        # done
        link.close()


if __name__ == "__main__":
    logging.basicConfig(level=logging.DEBUG, format='%(asctime)s %(levelname)s %(message)s')
    watch_stdin()
    port, secret, server_cert = parse_javalink_args()
    serve(port, secret, server_cert=server_cert)
