(function () {
    "use strict";

    const filterEditor = {
        bindings: {
            filterDesc: '=',
            schema: '=',
            mustRunInDatabase: '=',
            dataset: '=',
            filterUpdateCallback: '=',
            recipeAdditionalParams: '=',
            mainRecipeInput: '=',
            modelLabel: '@',
            modelClass: '@',
            recipe: "<?",
        },
        templateUrl: "/static/dataiku/nested-filters/filter-editor/filter-editor.component.html",
        controller: function filterEditorController($scope, $element, $attrs, $stateParams, $timeout, FiltersService, FilterEditorService,
                                                    Expressions, CodeMirrorSettingService, ElasticSearchQueryUtils, $state, translate) {

            /*
            Initializations
            */
            const ctrl = this;
            let uiData;
            let lastUiData;
            const manuallyEdited = {}; //save the expression if it was manually edited to restore it if necessary
            var timeout;//buffer validate expression AJAX calls
            ctrl.$onInit = function () {
                $scope.translate = translate;
                $scope.regenInternalFields = function() {
                    // Read attributes and save them into scope
                    let defaults = {
                        modelLabel: translate("FILTER.KEEP_ONLY_ROWS" , "Keep only rows that satisfy"),
                        modelClass: "dku-text",
                        mustRunInDatabase: false,
                        grelExpressionEmpty: true,
                        filterUpdateCallback: null,
                        dataset: null,
                        schema: null,
                        filterDesc: null,
                        mainRecipeInput: null,
                        recipeAdditionalParams: null,
                        recipe: null,
                    };
                    $.each(defaults, function(param, value) {
                        if (param in ctrl) {
                            $scope[param] = ctrl[param] || value;
                        } else {
                            $scope[param] = value;
                        }
                    });
                    FilterEditorService.internalFieldsErrorLog($scope.filterDesc, $scope.schema, $scope.dataset, $attrs);

                    $scope.recipeStatus = $scope.$parent.recipeStatus;
                }
                $scope.regenInternalFields();
    
                if ( $attrs.hideSqlFilter ) {
                    $scope.hideSqlFilter = $scope.$eval($attrs.hideSqlFilter);
                }
    
                $scope.initEditorOptions = function() {
                    // Initialize editorOptions
                    $scope.editorOptions = {
                            onLoad: function(cm) {
                                cm.on("keyup", function(cm, evt) {
                                    if (evt.type === 'keyup') {
                                        /* Ignore tab, esc, and navigation/arrow keys */
                                        if (evt.keyCode === 9 || evt.keyCode === 27 || (evt.keyCode>= 33 && evt.keyCode <= 40)) {
                                            return;
                                        } else {
                                            let options = {
                                                columns: columns,
                                                completeSingle: false
                                            }
                                            CodeMirror.commands.autocomplete(cm, null, options);
                                        }
                                    }
                                });
                            },
                            mode:'text/grel',
                            theme:'elegant',
                            variables: columns,
                            lineNumbers : false,
                            lineWrapping : true,
                            indentUnit: 4,
                            autofocus: true
                    };
    
    
                    var columns = function(){
                        return FiltersService.getColumns($scope).map(function(c){return c.name});
                    };
                    $scope.columns = FiltersService.getColumns($scope);
    
                    // Initialize sqlEditorOptions
                    $scope.sqlEditorOptions = CodeMirrorSettingService.get('text/x-sql');
                    $scope.sqlEditorOptions.variables = columns;
                    $scope.sqlEditorOptions.autofocus = true;
                };
                $scope.initEditorOptions();
    
                $scope.initUiData = function() {
                    // Initialize UiData
                    $scope.filterDesc.uiData = $scope.filterDesc.uiData || {};
                    var uiData = $scope.filterDesc.uiData;
                    uiData.conditions = uiData.conditions || [{}];
    
                    // Coming from interactive search
                    if ($scope.recipeAdditionalParams && !!$scope.recipeAdditionalParams.elasticSearchQuery) {
                        manuallyEdited["ES_QUERY_STRING"] = $scope.recipeAdditionalParams.elasticSearchQuery;
                        $scope.filterDesc.expression = $scope.recipeAdditionalParams.elasticSearchQuery;
                        uiData.mode = "ES_QUERY_STRING";  // set to query string by default
                        $scope.recipeAdditionalParams.elasticSearchQuery = null; // query was used, now discard it
                    }
    
                    uiData.mode = uiData.mode || '&&';
                    uiData.$filterOptions = uiData.mode || 'rules';
                    if (uiData.mode === '&&'){
                        uiData.$filterOptions = 'rules';
                    }
                    if (uiData.mode === '||'){
                        uiData.$filterOptions = 'rules';
                    }
                    if(uiData.mode === 'SQL'){
                        uiData.$filterOptions = 'SQL';
                    }
                    if(uiData.mode === 'CUSTOM'){
                        uiData.$filterOptions = 'CUSTOM';
                    }
                    if (uiData.$filterOptions==="rules") {
                        uiData.$latestOperator = uiData.mode;
                    } else if (!("$latestOperator" in uiData)) {
                        uiData.$latestOperator = "&&";
                    }
                    let lastUiData = angular.copy(uiData);
                    $scope.conditions = uiData.conditions; //short name
    
                    return {uiData, lastUiData};
                }

                let initData = $scope.initUiData();
                uiData = initData.uiData;
                lastUiData = initData.lastUiData;
                $scope.filterDesc.$status = $scope.filterDesc.$status || {}; //status is not saved on server
    
                FilterEditorService.getVariables($scope, $scope.recipe, $stateParams);
    
                // At load time a validation will be triggered. In order to prevent flickering between different messages, we have to initialize validationPending to true
                $scope.validationPending = true;
                $scope.validateExpression = function() {
                    if (!$scope.filterDesc.enabled || !$scope.filterDesc.expression || !FiltersService.hasSchema($scope)) {
                        return;
                    }
                    $scope.validationPending = false;
                    $scope.validationInProgress = $scope.validationInProgress ? $scope.validationInProgress+1 : 1;
                    Expressions.validateExpression($scope.filterDesc.expression, FiltersService.getSchema($scope), $scope.recipe)
                        .success(function(data) {
                            if (data.ok && $scope.mustRunInDatabase && !data.fullyTranslated) {
                                data.ok = false;
                                data.message = translate("FILTER.SQL.CANNOT_BE_TRANSLATED", "this expression cannot be translated to SQL." )
                            }
                            $scope.filterDesc.$status = $scope.filterDesc.$status || {};
                            $.extend($scope.filterDesc.$status, data);
                            $scope.filterDesc.$status.validated = true;
                            $scope.validationInProgress--;
                            manuallyEdited["SQL"] = null;
                        })
                        .error(function(data) {
                            $scope.validationInProgress--;
                            setErrorInScope.bind($scope);
                        });
                };
    
                if ($scope.filterDesc.expression && $scope.filterDesc.uiData.$filterOptions !== "ES_QUERY_STRING") {
                    $scope.validateExpression();
                }
                if ($scope.filterUpdateCallback) {
                    // This watcher needs to be defined in the onInit because it depends on bindings of the component
                    $scope.$watch('[filterDesc.expression, filterDesc.enabled, filterDesc.uiData]',
                        function(nv, ov){
                            if(nv !== ov) $scope.filterUpdateCallback($scope.filterDesc);
                        },
                        true
                    );
                }
            }       
            /*
            Scope methods
            */
            $scope.updateUiData = function(){
                // dirty rules to update mode when we update filterOptions as well.
                // latestOperator is used to remember if the user had set "And" or "Or" last time he was in rules mode.
                if (["SQL", "CUSTOM", "ES_QUERY_STRING"].includes(uiData.$filterOptions)) {
                    uiData.mode = uiData.$filterOptions;
                } else if (lastUiData.$filterOptions !== "rules") {
                    uiData.mode = lastUiData.$latestOperator;
                }
                if (uiData.$filterOptions==="rules") {
                    uiData.$latestOperator = uiData.mode;
                }
                delete $scope.filterDesc.language;
                // handle mode shifts
                if (uiData.$filterOptions !== lastUiData.$filterOptions) {
                    if (uiData.$filterOptions === "ES_QUERY_STRING") {
                        $scope.filterDesc.expression = manuallyEdited["ES_QUERY_STRING"]; // If we had something, set it
                    } else if (lastUiData.$filterOptions === "ES_QUERY_STRING") {
                        manuallyEdited["ES_QUERY_STRING"] = $scope.filterDesc.expression; // Keep previous expression, useful if page has just been (re)loaded
                        // And reset old data
                        if (["SQL", "CUSTOM"].includes(uiData.$filterOptions) && manuallyEdited[uiData.$filterOptions]) {
                            $scope.filterDesc.expression = manuallyEdited[uiData.$filterOptions];
                        } else {
                            $scope.filterDesc.expression = null; // clear expression when coming from ES_QUERY_STRING
                            $scope.filterDesc.$status.ok = true; // and set status OK because expression is empty
                        }
                    } else if (lastUiData.$filterOptions === 'SQL') {
                        // no keeping the expression for the other modes
                        if (uiData.$filterOptions==="CUSTOM") {
                            $scope.filterDesc.expression = manuallyEdited["CUSTOM"];
                        }
                    } else if (lastUiData.$filterOptions === 'CUSTOM') {
                        // if moving to SQL try putting the converted GREL expression
                        let setTranslatedExpression = null;
                        if (uiData.$filterOptions === 'SQL' && !manuallyEdited["SQL"]) {
                            setTranslatedExpression = function(data) {
                                if (data.ok) {
                                    $scope.filterDesc.expression = data.sql;
                                    manuallyEdited["SQL"] = null;
                                }
                            };
                        }
                        if (setTranslatedExpression !== null) {
                            Expressions.validateExpression($scope.filterDesc.expression, FiltersService.getSchema($scope), $scope.recipe)
                            .success(setTranslatedExpression)
                            .error(setErrorInScope.bind($scope));
                        }
                    } else {
                        let setTranslatedExpression = null;
                        if (uiData.$filterOptions=== 'SQL' && !manuallyEdited["SQL"]) {
                            setTranslatedExpression = function(data) {
                                if (data.ok) {
                                    $scope.filterDesc.expression = data.sql;
                                    manuallyEdited["SQL"] = null;
                                }
                            };
                        } else if (uiData.$filterOptions === 'CUSTOM' && !manuallyEdited["CUSTOM"]) {
                            setTranslatedExpression = function(data) {
                                if (data.ok) {
                                    $scope.filterDesc.expression = data.grel;
                                    manuallyEdited["CUSTOM"] = null;
                                }
                            };
                        }
                        if (setTranslatedExpression !== null) {
                            let lastFilterDesc = angular.copy($scope.filterDesc);
                            lastFilterDesc.uiData = lastUiData;
                            FilterEditorService.validateAST($scope, $stateParams, lastFilterDesc, setTranslatedExpression)
                        }
                    }
                }
                lastUiData = angular.copy(uiData);
            }
            let columns; // In order to avoid infinite digest loop, the columns need to be stored in the same object. See https://code.angularjs.org/1.6.10/docs/error/$rootScope/infdig
            $scope.getColumns = function() {
                if(!columns) {
                    $scope.updateColumns();
                }
                return columns;
            };
            $scope.updateColumns = function() {
                columns = FiltersService.getColumns($scope);
            }

            $scope.hasQuotingCharacters = function() {
                const recipeStatus = $scope.recipeStatus;
                return (recipeStatus && recipeStatus.selectedEngine && recipeStatus.selectedEngine.stringQuotingCharacter) ? true : false;
            }

            $scope.onExpressionChange = function(){
                $scope.revalidateExpression();
                manuallyEdited["CUSTOM"] = $scope.filterDesc.expression;
            };

            $scope.onFilterModeChange = function() {
                $scope.updateUiData();
                $scope.revalidateExpression();
            };

            $scope.onSQLChange = function(){
                manuallyEdited["SQL"] = $scope.filterDesc.expression;
            };

            $scope.onESQueryStringChange = function(){
                manuallyEdited["ES_QUERY_STRING"] = $scope.filterDesc.expression;
            };
            
            $scope.revalidateExpression = function(){
                $scope.filterDesc.$status.validated = false;
                $timeout.cancel(timeout);
                if (!$scope.filterDesc.expression) return;
                $scope.validationPending = true;
                timeout = $timeout($scope.validateExpression, 400);
            };
            
            $scope.clickedOnColumnName = function(colName) {
                let valueToAdd;
                if ($scope.filterDesc.uiData.$filterOptions === "ES_QUERY_STRING") {
                    valueToAdd = ElasticSearchQueryUtils.escapeColumn(colName);
                } else if (colName.match(/^[a-z0-9_]+$/i)) {
                    valueToAdd = colName;
                } else {
                    valueToAdd = `val('${colName}')`;
                }
                $scope.addFormulaElement(valueToAdd);
            };

            $scope.clickedOnVariableName = function(varName) {
                if ($scope.filterDesc.uiData.$filterOptions === "ES_QUERY_STRING") {
                    $scope.addFormulaElement("${" + varName + "}");
                } else {
                    $scope.addFormulaElement('variables["' + varName + '"]');
                }
            };

            $scope.addFormulaElement = function(code) {
                // replace selection and focuses editor
                var cm = $('.CodeMirror', $element).get(0).CodeMirror;
                cm.replaceSelection(code);
                cm.focus();
            };

            $scope.goToInputDataset = function() {
                if ($scope.mainRecipeInput.localProject) { // local dataset
                    $state.go("projects.project.datasets.dataset.search", {projectKey: $stateParams.projectKey, datasetName: $scope.mainRecipeInput.smartName, queryString: $scope.filterDesc.expression})
                } else { // foreign dataset
                    $state.go("projects.project.foreigndatasets.dataset.search", {projectKey: $stateParams.projectKey, datasetFullName: $scope.mainRecipeInput.smartName, queryString: $scope.filterDesc.expression})
                }
            }

            /*
            Watchers
            */
            $scope.$parent.$watch($attrs.schema, function(nv) {
                $scope.schema = nv;
                $scope.updateColumns();
            });

            $scope.$watch(
                'filterDesc.uiData',
                function(nv, ov){
                    if(nv !== ov) $scope.updateUiData();
                },
                true
            );

            $scope.$watch(
                'filterDesc.enabled',
                $scope.updateUiData,
                true
            );

            $scope.$watch(
                'mustRunInDatabase',
                $scope.validateExpression
            );

            $scope.$watch('recipeStatus', function() {
                // in case this filter is in a visual recipe, make sure to update the internal
                // flags when the engine or something else changes
                $scope.regenInternalFields();
            },true);
        },
    };

    angular.module("dataiku.nestedFilters").component("filterEditor", filterEditor);
})();
