import os, json, re, logging
from dataiku.code_studio import CodeStudioBlock, get_dataiku_user_uid_gid

from block_utils import LibLocationPathReplacer, generate_python_codenv

from distutils.version import LooseVersion

class DashCodeStudioBlock(CodeStudioBlock):
    def __init__(self, config, plugin_config):
        self.config = config
        self.plugin_config = plugin_config

    _ENTRYPOINT_FILE = "dash-entrypoint.sh"
        
    def _get_entrypoint_path(self):
        entrypoint_path = self.config.get("startScript", "/opt/dataiku")
        if entrypoint_path.endswith("/") or not entrypoint_path.endswith(".sh"):
            entrypoint_path = os.path.join(entrypoint_path, self._ENTRYPOINT_FILE)
        return entrypoint_path

    def _get_port(self):
        return self.config.get("port", 8282)
        
    def build_spec(self, spec, env, template):
        port = self._get_port()
        entrypoint_path = self._get_entrypoint_path()
        start_file = self.config.get("startFile", "__CODE_STUDIO_VERSIONED__/dash/app.py")

        # replace the lib locations in open_in_path
        replacer = LibLocationPathReplacer(spec)
        start_file = replacer.replace_variable_by_path(start_file)

        # get code env stuff

        default_packages = "dash==2.18.1 pandas"
        generate_codenv, pyenv_path = generate_python_codenv("DASH", self.config, template, default_packages, "/opt/dataiku/python-code-envs/pyenv-dash", "python3.9", env.get("globalCodeEnvsExtraSettings"))

        # add the entrypoint script in the buildir
        entrypoint_script = """
#! /bin/bash

USER=dataiku
HOME=/home/dataiku

# CREATE THE START APP.PY IF NEEDED

if [ ! -f {start_file} ]; then
    mkdir -p $(dirname {start_file})
    cat << 'EOF' > {start_file}
from dash import Dash, html, dcc, callback, Output, Input
import plotly.express as px
import pandas as pd

df = pd.read_csv('https://raw.githubusercontent.com/plotly/datasets/master/gapminder_unfiltered.csv')

app = Dash()

app.layout = [
    html.H1(children='Title of Dash App', style={{'textAlign':'center'}}),
    dcc.Dropdown(df.country.unique(), 'Canada', id='dropdown-selection'),
    dcc.Graph(id='graph-content')
]

@callback(
    Output('graph-content', 'figure'),
    Input('dropdown-selection', 'value')
)
def update_graph(value):
    dff = df[df.country==value]
    return px.line(dff, x='year', y='pop')

# see https://flask.palletsprojects.com/en/stable/api/#flask.Flask.run and 
# https://werkzeug.palletsprojects.com/en/stable/serving/#werkzeug.serving.run_simple for options
app.run(debug=True, port={port}, threaded=True, processes=0)
EOF
fi

# START DASH
export DASH_URL_BASE_PATHNAME=$(eval echo "$DKU_CODE_STUDIO_BROWSER_PATH_{port}/")
LC_ALL=en_US.utf8 {pyenv_path}/bin/python {start_file}
""".format(pyenv_path=pyenv_path, start_file=start_file, port=port)
        with open(os.path.join(env["buildDir"], self._ENTRYPOINT_FILE), "wb") as f:
            f.write(entrypoint_script.encode("utf8"))
            
        # the dockerfile addition
        spec["dockerfile"] = spec.get("dockerfile", "") + """

##### DASH BLOCK #####

USER root
WORKDIR /opt/dataiku

RUN yum install -y gettext

{generate_codenv}

# entrypoint.sh
COPY {entrypoint_file} {entrypoint_path}
RUN chown dataiku:root {entrypoint_path} && chmod +x {entrypoint_path}

# USER dataiku
USER {uid_gid}
WORKDIR /home/dataiku
""".format(pyenv_path=pyenv_path, start_file=start_file, port=port,
           entrypoint_path=entrypoint_path, entrypoint_file=self._ENTRYPOINT_FILE,
           generate_codenv=generate_codenv,
           uid_gid=get_dataiku_user_uid_gid())
        return spec

    def build_launch(self, spec, env):
        if env['launchedFrom'] == 'WEBAPP' and not self.config.get("useInWebapps", False):
            return spec
        port = self._get_port()
        spec['entrypoints'] = spec.get('entrypoints', []) + [self._get_entrypoint_path()]
        readiness_probe_url = "http://localhost:" + str(port) + "${baseUrlPort" + str(port) + "}/" # baseUrlPort should be replaced by actual url in BlockBasedCodeStudioMeta/buildYaml
        if spec.get('readinessProbeUrl', "") == "":
            spec['readinessProbeUrl'] = readiness_probe_url
        exposed_port = {
            "label": "Dash", 
            "proxiedUrlSuffix": "$uri$is_args$args",
            "exposeHtml": True, 
            "port": port,
            "readinessProbeUrl": readiness_probe_url
        }
        spec['exposedPorts'] = spec.get('exposedPorts', []) + [exposed_port]
        return spec
        
    def build_creation(self, spec, env):
        return spec
        
