(function() {
    'use strict';

    angular.module('dataiku.charts', ['dataiku.constants']);
    angular.module('dataiku.directives.simple_report', ['dataiku.charts', 'dataiku.shared']);

})();

;
(function() {
    'use strict';

    /**
     * <axis-range-subform
     *      axis='x'
     *      chart-type='chart.def.type'
     *      custom-extent='chart.def.xAxisFormatting.customExtent'>
     * </axis-range-subform>
     *
     * @param { String }    axis                        - 'x' or 'y'
     * @param { Object }    customExtent
     * @param { String }    customExtent.editMode       - ('AUTO' or 'MANUAL'); // the mode to use to define extent (auto computed or manually defined)
     * @param { Array }     customExtent.$autoExtent     - [minExtent, maxExtent]; // array of 2 floats: initial min and max values auto-detected (used in AUTO editMode)
     * @param { Array }     customExtent.manualExtent   - [minExtent, maxExtent]; // array of 2 floats or null: user custom min and max values.
     *                                                  If null, min or max are auto computed (used in MANUAL editMode)
     */
    angular.module('dataiku.charts').component('axisRangeSubform', {
        templateUrl: '/static/dataiku/js/simple_report/components/axis-range-subform/axis-range-subform.component.html',
        bindings: {
            axis: '@',
            chartType: '<',
            customExtent: '=',
            includeZero: '=?',
            editModeDisabled: '<'
        },
        controller: function(
            ChartsStaticData,
            ChartAxesUtils,
            ChartFeatures,
            $scope,
            $element
        ) {
            const ctrl = this;
            let minInput, maxInput;
            const NG_INVALID = 'ng-invalid';

            ctrl.extentEditModes = ChartsStaticData.extentEditModes;
            ctrl.getZeroDisabledMessage = getZeroDisabledMessage;
            ctrl.isManualMode = isManualMode;
            ctrl.onEditModeChange = onEditModeChange;
            ctrl.onMaxChange = onMaxChange;
            ctrl.onMinChange = onMinChange;
            ctrl.setExtentMax = setExtentMax;
            ctrl.setExtentMin = setExtentMin;
            ctrl.canChartIncludeZero = canChartIncludeZero;

            //////////

            // On blur, fix invalid or format null input
            function setExtentMin() {
                if (ctrl.customExtent.manualExtent[0] !== null) {
                    const currentMax = ChartAxesUtils.getManualExtentMax(ctrl.customExtent);
                    ctrl.customExtent.manualExtent[0] = validateExtentMin(currentMax, ctrl.customExtent.manualExtent[0]);
                    validateMinInput();
                } else {
                    $scope.$broadcast('formatManualExtent');
                }
            }

            function setExtentMax() {
                if (ctrl.customExtent.manualExtent[1] !== null) {
                    const currentMin = ChartAxesUtils.getManualExtentMin(ctrl.customExtent);
                    ctrl.customExtent.manualExtent[1] = validateExtentMax(currentMin, ctrl.customExtent.manualExtent[1]);
                    validateMaxInput();
                } else {
                    $scope.$broadcast('formatManualExtent');
                }
            }

            // On extent input change, check for invalid values
            function onMinChange(newMin) {
                const currentMax = ChartAxesUtils.getManualExtentMax(ctrl.customExtent);

                if (newMin !== null && parseFloat(newMin) > parseFloat(currentMax)) {
                    invalidateMinInput();
                } else {
                    validateMinInput();
                }
            }

            function onMaxChange(newMax) {
                const currentMin = ChartAxesUtils.getManualExtentMin(ctrl.customExtent);

                if (newMax !== null && parseFloat(newMax) < parseFloat(currentMin)) {
                    invalidateMaxInput();
                } else {
                    validateMaxInput();
                }
            }

            function isManualMode() {
                return ChartAxesUtils.isManualMode(ctrl.customExtent);
            }

            function canChartIncludeZero() {
                return ChartFeatures.canIncludeZero(ctrl.chartType);
            }

            function getZeroDisabledMessage() {
                if (ctrl.isManualMode()) {
                    return 'This option is ignored in manual range edition.';
                }
            }

            function onEditModeChange() {
                if (ctrl.isManualMode()) {
                    $scope.$broadcast('formatManualExtent');
                }
            }

            function validateExtentMin(currentMax, newMin) {
                const min = parseFloat(newMin) > parseFloat(currentMax) ? currentMax : newMin;
                return parseFloat(min);
            }

            function validateExtentMax(currentMin, newMax) {
                const max = parseFloat(newMax) < parseFloat(currentMin) ? currentMin : newMax;
                return parseFloat(max);
            }

            const getMinInput = () => {
                if (!minInput) {
                    minInput = $element.find('input[name=\'min\']');
                }
                return minInput;
            };

            const getMaxInput = () => {
                if (!maxInput) {
                    maxInput = $element.find('input[name=\'max\']');
                }
                return maxInput;
            };

            const validateMinInput = () => getMinInput().removeClass(NG_INVALID);
            const validateMaxInput = () => getMaxInput().removeClass(NG_INVALID);
            const invalidateMinInput = () => getMinInput().addClass(NG_INVALID);
            const invalidateMaxInput = () => getMaxInput().addClass(NG_INVALID);
        }
    });
})();

;
(function () {
    'use strict';

    angular
        .module('dataiku.charts')
        .component('dashboardPageMenu', {
            templateUrl: '/static/dataiku/js/simple_report/components/dashboard-page-menu/dashboard-page-menu.component.html',
            bindings: {
                canEdit: '<',
                urlToCopy: '<',
                onDuplicate: '&',
                onEdit: '&',
                onRemove: '&',
            },
            controller: function (DashboardUtils) {
                const ctrl = this;
                ctrl.copyURL = () => {
                    DashboardUtils.copyToClipboard(ctrl.urlToCopy);
                }
            }
        });
})();

;
(function() {
    'use strict';

    angular.module('dataiku.charts').component('discreteColorMenu', {
        templateUrl: 'static/dataiku/js/simple_report/components/discrete-color-menu/discrete-color-menu.component.html',
        bindings: {
            geoLayer: '<?',
            geoIndex: '<?',
            chartDef: '<',
            legends: '<',
            usableColumns: '<'
        },
        controller: function($scope, $timeout, ChartCustomColors, ChartColorSelection, CHART_TYPES, DKU_PALETTE_NAMES) {
            let subscription;

            const ctrl = this;

            ctrl.getColorOptions = () => {
                return angular.copy(ChartColorSelection.getOptions(ctrl.chartDef, ctrl.geoLayer));
            };

            ctrl.setColorOptions = () => {
                return ChartColorSelection.setOptions(angular.copy(ctrl.colorOptions), ctrl.chartDef, ctrl.geoLayer);
            };

            ctrl.isMeaningPalette = () => {
                return ctrl.colorOptions.colorPalette === DKU_PALETTE_NAMES.MEANING;
            };

            ctrl.isCustomPalette = () => {
                return ctrl.colorOptions.colorPalette === DKU_PALETTE_NAMES.CUSTOM;
            };

            ctrl.$onInit = () => {
                ctrl.colorOptions = ctrl.getColorOptions();
                let geoLayerId;
                if (!_.isNil($scope.geoIndex)) {
                    geoLayerId = `geo-${$scope.geoIndex}`;
                }
                const { id, colorOptions$ } = ChartColorSelection.getOrCreateCustomColorsOptions(geoLayerId);
                ctrl.id = id;

                subscription = colorOptions$.subscribe((newColorOptions) => {
                    ctrl.colorOptions = angular.copy(newColorOptions);
                    ctrl.setColorOptions();
                    $timeout(() => $scope.$apply());
                });

                ChartCustomColors.updateInputColorOptions(id, angular.copy(ctrl.colorOptions));
            };

            ctrl.$onDestroy = () => {
                if (subscription) {
                    subscription.unsubscribe();
                }
                ChartColorSelection.removeCustomColorsOptions([ctrl.id]);
            };

            ctrl.getColorDimensionOrMeasure = () => {
                return ChartColorSelection.getColorDimensionOrMeasure(ctrl.chartDef, ctrl.geoLayer);
            };

            ctrl.onColorOptionsChange = (value) => {
                ChartCustomColors.updateInputColorOptions(ctrl.id, angular.copy(value));
            };

            ctrl.getLegend = (legends, geometryIndex) => {
                const chartDef = ctrl.chartDef;
                if (!legends) {
                    return;
                }
                if (chartDef.type !== CHART_TYPES.GEOMETRY_MAP || (chartDef.geoLayers.length < 3 && geometryIndex === undefined)) {
                    return legends[0];
                } else {
                    return legends[geometryIndex];
                }
            };
        }
    });
})();

;
(function() {
    'use strict';

    /**
     * <display-label-subform
     *      measure="measure"
     *      chart-def="chart.def"
     *      // OR
     *      dimension="dimension"
     * </display-label-subform>
     *
     * @param { com.dataiku.dip.pivot.frontend.model.MeasureDef | undefined }       measure
     * @param { com.dataiku.dip.pivot.frontend.model.DimensionDef | undefined }     dimension
     * @param { com.dataiku.dip.pivot.frontend.model.chartDef | undefined }        chartDef
     */
    angular.module('dataiku.charts').component('displayLabelSubform', {
        templateUrl: '/static/dataiku/js/simple_report/components/display-label-subform/display-label-subform.component.html',
        bindings: {
            measure: '<',
            dimension: '<',
            chartDef: '<',
            helperText: '<'
        },
        controller: function(ChartLabels, ChartDefinitionChangeHandler, $scope, translate) {
            const ctrl = this;
            $scope.translate = translate;
            ctrl.placeholder = '';
            ctrl.measureOrNumericalDimension = null;

            ctrl.$onChanges = function() {
                ctrl.measureOrNumericalDimension = ctrl.measure || ctrl.dimension;
                ctrl.type = ctrl.measure ? 'measure' : 'dimension';
                ctrl.updatePlaceholder(ctrl.measureOrNumericalDimension, ctrl.type);
                ctrl.updatePossibleDimensionSorts(ctrl.type);
            };

            /**
             * Updates the placeholder value based on the given measure or dimension.
             *
             * @param {com.dataiku.dip.pivot.frontend.model.MeasureDef | com.dataiku.dip.pivot.frontend.model.DimensionDef}    measureOrNumericalDimension    the measure or dimension from which to compute the placeholder
             * @param {'measure' | 'dimension'}                                                                                type                           the type of the first parameter
             */
            ctrl.updatePlaceholder = function(measureOrNumericalDimension, type) {
                if (!measureOrNumericalDimension) {
                    ctrl.placeholder = '';
                    return;
                }
                let getPlaceholder;
                if (measureOrNumericalDimension.isA === 'ua') {
                    getPlaceholder = ChartLabels.getUaLabel;
                } else {
                    getPlaceholder = type === 'measure' ? ChartLabels.getLongMeasureLabel : ChartLabels.getDimensionLabel;
                }

                ctrl.placeholder = getPlaceholder(measureOrNumericalDimension);
            };

            ctrl.updatePossibleDimensionSorts = function(type) {
                if (ctrl.chartDef && type === 'measure') {
                    // sorts depend only on measures, and should be updated when displayLabel changes
                    ctrl.chartDef.possibleDimensionSorts = ChartDefinitionChangeHandler.getStdDimensionPossibleSorts(ctrl.chartDef);
                }
            };
        }
    });
})();

;
(function() {
    'use strict';

    /**
     * <dropdown-number-formatting-button
     *    open-section="openSection"
     *    chart-def-key="chartDefKey"
     *    list-index="listIndex"
     *    toggle-contextual-menu="toggleContextualMenu"
     * >
     * </dropdown-number-formatting-button>
     *
     */
    angular.module('dataiku.charts').component('dropdownNumberFormattingButton', {
        templateUrl: 'static/dataiku/js/simple_report/components/dropdown-number-formatting-button/dropdown-number-formatting-button.html',
        bindings: {
            openSection: '=',
            chartDefKey: '=',
            toggleContextualMenu: '=',
            listIndex: '='
        },
        controller: function(ChartFormattingPaneSections, CHART_FORMATTING_PANE_PROPERTIES, FormattingPaneValues, $scope, translate) {
            $scope.translate = translate;
            const ctrl = this;

            ctrl.openFormattingPane = function(event) {
                const { openSection, chartDefKey, listIndex, toggleContextualMenu } = ctrl;
                toggleContextualMenu(event);

                const measureOrDimensionId = FormattingPaneValues.computeMeasureOrDimensionUniqueId(chartDefKey, listIndex);
                openSection(ChartFormattingPaneSections.DATA_VALUES, {
                    [CHART_FORMATTING_PANE_PROPERTIES.DATA_VALUES_SELECTED_MEASURE_DIMENSION_KEY]: measureOrDimensionId
                });
                if (chartDefKey === 'genericMeasures' && FormattingPaneValues.valuesDisplayed$.getValue() === false) {
                    // then open also the values in chart section if this section is displayed
                    openSection(ChartFormattingPaneSections.VALUES, {
                        [CHART_FORMATTING_PANE_PROPERTIES.VALUES_DISPLAY_SELECTED_MEASURE_KEY]: measureOrDimensionId
                    });
                }
            };

        }
    });
})();

;
(function() {
    angular.module('dataiku.charts').component('invalidChartPlaceholder', {
        templateUrl: '/static/dataiku/js/simple_report/components/invalid-chart-placeholder/invalid-chart-placeholder.component.html',
        bindings: {
            validity: '<',
            chartType: '<',
            chartVariant: '<',
            isInDashboard: '<',
            isInPredicted: '<',
            canEdit: '<',
            computeChartPreview: '<',
            computeChartOptionsPreview: '<',
            revertToLinoEngineAndReload: '<',
            forceExecute: '<',
            rebuildSampling: '<'
        },
        controller: function() {
            const ctrl = this;
            ctrl.canErrorBeDisplayedBothInChartsAndDashboards = false;

            ctrl.$onChanges = function() {
                if (!_.isNil(ctrl.validity)) {
                    ctrl.canErrorBeDisplayedBothInChartsAndDashboards = ctrl.validity.type === 'PIVOT_TABLE_TOO_MUCH_DATA' || ctrl.validity.type === 'CORRUPTED_LINO_CACHE' || ctrl.validity.type === 'EMPTY_CUSTOM_BINS';
                }
            };

        }
    });
})();

;
(function() {
    'use strict';

    angular.module('dataiku.charts').component('kpiHolder', {
        templateUrl: '/static/dataiku/js/simple_report/components/kpi-holder/kpi-holder.component.html',
        bindings: {
            chartDef: '<',
            theme: '<',
            data: '<',
            loadedCallback: '&'
        },

        controller: function($scope, $element, $timeout, ChartDataWrapperFactory, ChartFormatting, ChartLabels, ChartsStaticData, ConditionalFormattingOptions, CHART_TYPES) {
            const ctrl = this;
            const VALUE_SELECTOR = 'kpi-holder-measure__value';
            const LABEL_SELECTOR = 'kpi-holder-measure__label';
            const MEASURE_SELECTOR = 'kpi-holder-measure';
            const ROUNDING = 0.9;
            const VALUE_HEIGHT_OVERFLOW = 0.17;
            const MIN_FONT_BEFORE_HIDDING_LABEL = 30;
            const MAX_MARGINS = 4;
            let valuePadding = 0;

            ctrl.el = $element[0];

            const init = (theme = ctrl.theme) => {
                // init can be triggered for a non-kpi chart because of the way we handle displays.
                if (!ctrl.data || !ctrl.chartDef || ctrl.chartDef.type !== CHART_TYPES.KPI) {
                    return;
                }
                const chartData = ChartDataWrapperFactory.chartTensorDataWrapper(ctrl.data);
                ctrl.chartDef.colorMode = 'COLOR_GROUPS';

                // index of measure group colors are based of in data.aggregations
                let basedOnMeasureIndexCounter = ctrl.chartDef.genericMeasures.length - 1;
                ctrl.colorGroupByMeasureId = ctrl.chartDef.colorGroups ? ctrl.chartDef.colorGroups
                    .filter(({ appliedColumns }) => !!appliedColumns)
                    .reduce((acc, { appliedColumns, colorMeasure, rules }) => {
                        let groupColorMeasure;

                        if (colorMeasure && colorMeasure.length > 0) {
                            basedOnMeasureIndexCounter += 1;
                            groupColorMeasure = colorMeasure[0];
                        }

                        appliedColumns.forEach(column => (
                            acc[ConditionalFormattingOptions.getMeasureId(column)] = {
                                rules,
                                basedOnMeasure: groupColorMeasure,
                                basedOnMeasureIndex: groupColorMeasure ? basedOnMeasureIndexCounter : -1
                            }
                        ));
                        return acc;
                    }, {}) : []
                ;


                ctrl.colorRulesClasses = ctrl.chartDef.genericMeasures.map((m, i) => {
                    const colorGroup = ctrl.colorGroupByMeasureId[ConditionalFormattingOptions.getMeasureId(m)];
                    if (!colorGroup) {
                        return ConditionalFormattingOptions.getColorRuleClass('', [], m, theme);
                    }

                    const measureRules = colorGroup.rules;

                    // use "base on another column" value if defined, if not use the value of the column itself
                    const basedOnMeasureIndex = colorGroup.basedOnMeasureIndex >= 0 ? colorGroup.basedOnMeasureIndex : i;
                    const ruleBaseValue = chartData.getAggrExtent(basedOnMeasureIndex)[0];

                    const colorMeasure = colorGroup.basedOnMeasure || m;

                    return ConditionalFormattingOptions.getColorRuleClass(ruleBaseValue, measureRules, colorMeasure, theme);
                });

                ctrl.multipleKPIs = ctrl.chartDef.genericMeasures.length > 1;
                if (ctrl.multipleKPIs) {
                    valuePadding = MAX_MARGINS * 2; // 2 sides
                }

                // sanitize all dom measures
                for (let i = 0; i < ctrl.chartDef.genericMeasures.length; i++) {
                    ctrl.sanitizeKpi(i);
                }

                // take all dimensions generated by flexbox and thank him for his loyal services
                const dimensions = [];
                for (let i = 0; i < ctrl.chartDef.genericMeasures.length; i++) {
                    dimensions[i] = ctrl.getKpiBoxSize(i);
                }

                // replace the sanitized kpi value and label if displayed & find the best fitting fontSize for all kpi and keep the minimum one
                const computeMinValue = (hideAllLabels = false) => {
                    let minValueFontSize = Number.MAX_VALUE;
                    for (let i = 0; i < ctrl.chartDef.genericMeasures.length; i++) {
                        const hideLabel = hideAllLabels || !ctrl.chartDef.genericMeasures[i].showDisplayLabel;
                        minValueFontSize = Math.min(minValueFontSize, ctrl.applyKpiValue(i, dimensions[i], hideLabel));
                    }
                    return minValueFontSize;
                };

                // first compute the minFontSize we can display with labels
                let minValueFontSize = computeMinValue();
                const hideAllLabels = minValueFontSize <= MIN_FONT_BEFORE_HIDDING_LABEL;

                // if we must hide labels, recompute the new minFontSize, as it can now be greater
                if (hideAllLabels) {
                    minValueFontSize = computeMinValue(true);
                }

                for (let i = 0; i < ctrl.chartDef.genericMeasures.length; i++) {
                    ctrl.applyKpiLabel(i, dimensions[i], hideAllLabels);
                }

                // apply the minimum fontSize found to all kpis
                for (let i = 0; i < ctrl.chartDef.genericMeasures.length; i++) {
                    ctrl.applyFontSize(
                        ctrl.el.getElementsByClassName(VALUE_SELECTOR)[i],
                        ctrl.chartDef.genericMeasures[i].kpiValueFontSizeMode === 'RESPONSIVE' ? minValueFontSize * ctrl.chartDef.genericMeasures[i].responsiveTextAreaFill / 100 : ctrl.chartDef.genericMeasures[i].kpiValueFontSize
                    );
                }

                ctrl.loadedCallback();
            };

            /*
             * Re-init on resize
             * Add a timeout to make sure the scope is ready, as now chartDef.genericMeasures can be modified by Angular forms in formatting pane
             */
            $scope.$on('window-resized-kpi', (_, { theme }) => $timeout(() => init(theme), 0));

            /*
             * On first init wait for the dom to be rendered
             * https://stackoverflow.com/questions/18646756
             */
            $scope.$watch('$viewContentLoaded', () => $timeout(() => init(), 0));

            /**
             * Sanitize the dom by setting default values and minimal font-size.
             * This step is necessary to render a default dom skeletton and
             * allow flexbox to render symmetrical boxes for each measure.
             *
             * A default value like a whitespace must be set for the value and label
             * as the flexbox rendering is different with and without.
             *
             * As we keep a default font-size for the label, the latter is not concerned by sanitization.
             *
             * @param {number} index of the measure in dom
             */
            this.sanitizeKpi = function(index) {

                // sanitize container
                const container = ctrl.el.getElementsByClassName(MEASURE_SELECTOR)[index];
                if (ctrl.multipleKPIs) {
                    container.style.margin = `${MAX_MARGINS}px`;
                    container.style.padding = `${MAX_MARGINS}px`;
                }

                // sanitize value div
                const valueDom = ctrl.el.getElementsByClassName(VALUE_SELECTOR)[index];
                valueDom.textContent = ' ';
                valueDom.style.fontSize = '1px';
                valueDom.style.lineHeight = '1px';
                valueDom.style.width = '';

                // sanitize label div
                const labelDom = ctrl.el.getElementsByClassName(LABEL_SELECTOR)[index];
                labelDom.style.display = '';
                labelDom.textContent = ' ';
                labelDom.style.width = '';
            };

            this.applyKpiValue = function(index, containerDim, hidelabel) {
                const chartData = ChartDataWrapperFactory.chartTensorDataWrapper(ctrl.data);
                const kpiValueDom = ctrl.el.getElementsByClassName(VALUE_SELECTOR)[index];
                kpiValueDom.textContent = ChartFormatting.getForIsolatedNumber(ctrl.chartDef.genericMeasures[index])(chartData.getAggrExtent(index)[0]);
                const labelFontSize = hidelabel ? 0 : this.getRawLabelFontSize(index);
                return ctrl.fitKpiValueToContainer(kpiValueDom, containerDim, labelFontSize);
            };

            this.applyKpiLabel = function(index, containerDim, hideAllLabels) {
                const measure = ctrl.chartDef.genericMeasures[index];
                const kpiLabelDom = ctrl.el.getElementsByClassName(LABEL_SELECTOR)[index];

                if (!measure.showDisplayLabel || hideAllLabels) {
                    kpiLabelDom.style.display = 'none';
                    return;
                }

                const kpiValueDom = ctrl.el.getElementsByClassName(VALUE_SELECTOR)[index];
                const boxWidth = Math.ceil(containerDim.width * (ROUNDING + 0.05)) + 'px'; // hack to make sure we don't ellipsis on responsive mode

                kpiLabelDom.style.width = boxWidth;
                kpiValueDom.style.width = boxWidth;
                kpiLabelDom.style.display = '';

                if (measure.labelPosition === ChartsStaticData.LABEL_POSITIONS.BOTTOM.value) {
                    kpiLabelDom.parentNode.insertBefore(kpiValueDom, kpiLabelDom);
                } else {
                    kpiValueDom.parentNode.insertBefore(kpiLabelDom, kpiValueDom);
                }

                kpiLabelDom.textContent = ChartLabels.getLongMeasureLabel(ctrl.chartDef.genericMeasures[index]);
            };

            this.getKpiBoxSize = function(index) {
                const kpiDiv = ctrl.el.getElementsByClassName(MEASURE_SELECTOR)[index];
                return { height: kpiDiv.clientHeight - valuePadding, width: kpiDiv.clientWidth - valuePadding };
            };

            /**
             * Perform a binary search to find the best fontSize fit.
             * O(log(n))
             * @param {HTMLElement} kpiValue the dom of the displayed value
             * @param {{width: number, height: number}} containerDimensions original dimension where the span must fit
             */
            this.fitKpiValueToContainer = function(kpiValue, containerDimensions, labelFontSize) {
                let left = 0;
                let right = 500;
                let mid = 0;
                while (left <= right) {
                    mid = (right + left) >> 1;
                    ctrl.applyFontSize(kpiValue, mid);
                    const kpiSpanClientRect = kpiValue.getBoundingClientRect();
                    if (kpiSpanClientRect.width <= containerDimensions.width * ROUNDING && kpiSpanClientRect.height <= containerDimensions.height * ROUNDING - labelFontSize) {
                        left = mid + 1;
                    } else {
                        right = mid - 1;
                    }
                }

                return mid;
            };

            this.applyFontSize = function(kpiValue, size) {
                /*
                 * some characters like ',' or 'p' are overflowing span bounding rect
                 * we add an extra margin to lineHeight to remedy this
                 */
                kpiValue.style.fontSize = size + 'px';
                kpiValue.style.lineHeight = Math.round(size + size * VALUE_HEIGHT_OVERFLOW) + 'px';
            };

            this.getAlignmentClass = (index) => {
                const measure = this.chartDef.genericMeasures[index];
                if (measure) {
                    switch (measure.kpiTextAlign) {
                        case 'LEFT':
                            return 'kpi-holder-measure--left';
                        case 'RIGHT':
                            return 'kpi-holder-measure--right';
                        default:
                            return 'kpi-holder-measure--center';
                    }
                }
            };

            this.getRawValueFontSize = (index) => {
                return this.chartDef.genericMeasures && this.chartDef.genericMeasures.length && this.chartDef.genericMeasures[index] && this.chartDef.genericMeasures[index].kpiValueFontSize;
            };

            this.getRawLabelFontSize = (index) => {
                return this.chartDef.genericMeasures && this.chartDef.genericMeasures.length && this.chartDef.genericMeasures[index] && this.chartDef.genericMeasures[index].labelTextFormatting && this.chartDef.genericMeasures[index].labelTextFormatting.fontSize;
            };

            this.getValueFontSize = (index) => {
                return `${this.getRawValueFontSize(index)}px`;
            };

            this.getLabelFontSize = (index) => {
                return `${this.getRawLabelFontSize(index)}px`;
            };

            this.getLabelFontColor = (index) => {
                const hasMeasureConditionalFormatting = this.chartDef.genericMeasures && this.chartDef.genericMeasures.length && ctrl.colorGroupByMeasureId && ctrl.colorGroupByMeasureId[ConditionalFormattingOptions.getMeasureId(this.chartDef.genericMeasures[index])];
                return !hasMeasureConditionalFormatting && this.chartDef.genericMeasures && this.chartDef.genericMeasures.length && this.chartDef.genericMeasures[index] && this.chartDef.genericMeasures[index].labelTextFormatting && this.chartDef.genericMeasures[index].labelTextFormatting.fontColor;
            };
        }
    });
})();

;
(function() {
    'use strict';

    angular.module('dataiku.charts').component('legendOptzone', {
        templateUrl: '/static/dataiku/js/simple_report/components/legend-optzone/legend-optzone.component.html',
        bindings: {
            canPlaceInSidebar: '<',
            legendPlacement: '=',
            legendFormatting: '=',
            displayLegendFormatting: '<',
            theme: '<'
        },

        controller: function($scope, $timeout, ColorUtils, translate, DSSVisualizationThemeUtils) {
            const ctrl = this;
            $scope.translate = translate;

            ctrl.$onInit = () => {
                ctrl.themeColors = ColorUtils.getThemeColorsWithBlackWhite(ctrl.theme);
                ctrl.foregroundColors = ColorUtils.generateThemePaletteColors(DSSVisualizationThemeUtils.getThemeOrDefault(ctrl.theme).colors, ctrl.themeColors.length > 0).foregroundColors;
            };

            $scope.$watch('$ctrl.theme', (newTheme, oldTheme) => {
                if (!newTheme) {
                    return;
                }
                ctrl.defaultFormatting = { fontColor: newTheme.legendFormatting.fontColor, fontSize: newTheme.legendFormatting.fontSize };
                ctrl.themeColors = ColorUtils.getThemeColorsWithBlackWhite(newTheme);
                if (!ctrl.foregroundColors || !oldTheme || !_.isEqual(newTheme.colors, oldTheme.colors)) {
                    ctrl.themeColors = ColorUtils.getThemeColorsWithBlackWhite(newTheme);
                    const paletteColors = ColorUtils.generateThemePaletteColors(newTheme.colors, ctrl.themeColors.length > 0);
                    if (paletteColors) {
                        ctrl.foregroundColors = paletteColors.foregroundColors;
                    }
                }
            });

            ctrl.categories = {
                'OUTER': ['OUTER_RIGHT', 'OUTER_LEFT', 'OUTER_TOP', 'OUTER_BOTTOM'],
                'INNER': ['INNER_TOP_RIGHT', 'INNER_TOP_LEFT', 'INNER_BOTTOM_LEFT', 'INNER_BOTTOM_RIGHT']
            };

            $scope.$watch('$ctrl.legendPlacementCategory', function(nv, ov) {
                if (!nv) {
                    return;
                }

                if (nv === 'SIDEBAR') {
                    ctrl.legendPlacement = 'SIDEBAR';
                } else {
                    if (ctrl.categories[nv].indexOf(ctrl.legendPlacement) === -1) {
                        ctrl.legendPlacement = ctrl.categories[nv][0];
                    }
                }
            });

            $scope.$watch('$ctrl.legendPlacement', function(nv, ov) {
                if (!nv) {
                    return;
                }

                if (nv === 'SIDEBAR') {
                    ctrl.legendPlacementCategory = 'SIDEBAR';
                } else {
                    for (const cat in ctrl.categories) {
                        if (ctrl.categories[cat].indexOf(nv) > -1) {
                            ctrl.legendPlacementCategory = cat;
                            break;
                        }
                    }
                }
            });

            ctrl.onLegendFormattingChange = function(value) {
                $timeout(() => ctrl.legendFormatting = value);
            };
        }
    });
})();

;
(function() {
    'use strict';

    /**
     * <number-formatting-subform
     *     multiplier="multiplier"
     *     decimalPlaces="decimalPlaces"
     *     prefix="prefix"
     *     suffix="suffix"
     *     computeMode="computeMode">
     * </number-formatting-subform>
     *
     * @param { string }                                                        multiplier
     * @param { number }                                                        decimalPlaces
     * @param { string }                                                        prefix
     * @param { string }                                                        suffix
     * @param { com.dataiku.dip.pivot.backend.model.Aggregation.ComputeMode }   computeMode
     */
    angular.module('dataiku.charts').component('numberFormattingSubform', {
        templateUrl: '/static/dataiku/js/simple_report/components/number-formatting-subform/number-formatting-subform.component.html',
        bindings: {
            shouldFormatInPercentage: '=',
            multiplier: '=',
            decimalPlaces: '=',
            hideTrailingZeros: '=',
            digitGrouping: '=',
            useParenthesesForNegativeValues: '=',
            prefix: '=',
            suffix: '=',
            computeMode: '<?'
        },
        controller: function(ChartsStaticData) {
            const ctrl = this;
            ctrl.availableMultipliers = ChartsStaticData.availableMultipliers;
            ctrl.availableDigitGrouping = ChartsStaticData.availableDigitGrouping;

            ctrl.isComputedAsPercentage = () => {
                return ctrl.computeMode === 'PERCENTAGE' || ctrl.computeMode === 'CUMULATIVE_PERCENTAGE';
            };

            ctrl.$onInit = function() {
                ctrl.multiplier = ctrl.multiplier || 'Auto';
            };
        }
    });
})();

;
(function() {
    'use strict';

    /**
     * <text-align-subform value="value"></text-align-subform>
     *
     * @param { 'LEFT' | 'RIGHT' | 'CENTER'}    value
     */
    angular.module('dataiku.charts').component('textAlignSubform', {
        templateUrl: '/static/dataiku/js/simple_report/components/text-align-subform/text-align-subform.component.html',
        bindings: {
            value: '='
        },
        controller: function($timeout, translate) {
            const ctrl = this;

            this.translate = translate;

            ctrl.setAlignment = (alignment) => {
                $timeout(() => ctrl.value = alignment);
            };
        }
    });
})();

;
(function() {
    'use strict';

    const CHART_AXIS_TYPES = {
        DIMENSION: 'DIMENSION',
        MEASURE: 'MEASURE',
        UNAGGREGATED: 'UNAGGREGATED'
    };

    angular.module('dataiku.charts')
        .constant('CHART_AXIS_TYPES', CHART_AXIS_TYPES);

}());

;
(function() {
    'use strict';

    const ECHART_AXIS_TYPES = {
        CATEGORY: 'category',
        VALUE: 'value',
        TIME: 'time',
        LOG: 'log'
    };

    angular.module('dataiku.charts')
        .constant('ECHART_AXIS_TYPES', ECHART_AXIS_TYPES);

}());

;
(function() {
    'use strict';

    const DKU_PALETTE_NAMES = {
        MEANING: '__dku_meaning__',
        CUSTOM: '__dku_custom__',
        THEME: 'default_theme'
    };

    angular.module('dataiku.charts')
        .constant('DKU_PALETTE_NAMES', DKU_PALETTE_NAMES);

}());

;
(function() {
    'use strict';

    function buildDateDisplay(mainDateFormat, dateFormat, dateFilterOption, dateFilterOptionTimezone = 'UTC') {
        return {
            mainDateFormat,
            dateFormat,
            dateFilterOption,
            dateFilterOptionTimezone,
            formatDateFn: function(timestamp, formatToApply) {
                return d3.time.format.utc(formatToApply)(new Date(timestamp));
            }
        };
    }

    const CHART_DATES_LABELS = {
        DATE_DISPLAY_UNIT_DEFAULT: buildDateDisplay(undefined, '%Y-%m-%d', 'MMM d, y'),
        DATE_DISPLAY_UNIT_DAY_AND_MINUTES: buildDateDisplay(undefined, '%Y-%m-%d %H:%M', 'MMM d, y HH:mm'),
        DATE_DISPLAY_UNIT_MINUTES: buildDateDisplay('%Y-%m-%d', '%H:%M', 'HH:mm'),
        DATE_DISPLAY_UNIT_SECONDS: buildDateDisplay('%Y-%m-%d', '%H:%M:%S', 'HH:mm:ss'),
        DATE_DISPLAY_UNIT_MILLISECONDS: buildDateDisplay('%Y-%m-%d', '%H:%M:%S:%L', 'HH:mm:ss:sss')
    };

    const CHART_LABELS = {
        SUBTOTAL_BIN_LABEL: '___dku_total_value___',
        NO_RECORDS: 'No records'
    };

    angular.module('dataiku.charts')
        .constant('CHART_DATES_LABELS', CHART_DATES_LABELS)
        .constant('CHART_LABELS', CHART_LABELS);

}());

;
(function() {
    'use strict';

    const CHART_MODES = {
        COLUMNS: 'COLUMNS',
        POINTS: 'POINTS'
    };

    angular.module('dataiku.charts')
        .constant('CHART_MODES', CHART_MODES);

}());

;
(function() {
    'use strict';

    const CHART_SORT_TYPES = {
        NATURAL: 'NATURAL',
        AGGREGATION: 'AGGREGATION'
    };

    angular.module('dataiku.charts')
        .constant('CHART_SORT_TYPES', CHART_SORT_TYPES);

}());

;
(function() {
    'use strict';

    // ChartVariant.java
    const CHART_VARIANTS = {
        normal: 'normal',
        stacked100: 'stacked_100',
        binnedXYRectangle: 'binned_xy_rect',
        binnedXYHexagon: 'binned_xy_hex',
        filledMap: 'filled_map',
        donut: 'donut',
        colored: 'colored',
        waterfall: 'waterfall'
    };

    angular.module('dataiku.charts')
        .constant('CHART_VARIANTS', CHART_VARIANTS);

}());

;
(function() {
    'use strict';

    window.dkuDragType = (
        window.navigator.userAgent.indexOf('Trident') >= 0) ? 'Text' :
        (window.navigator.userAgent.indexOf('Edge') >= 0 ? 'text/plain' : 'json');

    const app = angular.module('dataiku.directives.simple_report');

    /**
     * Container required for common logic of drag and drop directives (from simple_report/directives/drag-drop/).
     * This code previously was in static/dataiku/js/simple_report/chart_dragdrop.js
     */
    app.controller('ChartDragDropController', ($scope) => {

        $scope.setDragActive = () => {
            $('.chart-configuration-wrapper').addClass('drag-active');
        };

        $scope.setDragInactive = () => {
            $('.chart-configuration-wrapper').removeClass('drag-active');
        };

        $scope.addClassHereAndThere = (element, clazz) => {
            $(element).addClass(clazz);
            $(element).parent().parent().addClass(clazz);
        };

        $scope.removeClassHereAndThere = (element, clazz) => {
            $(element).removeClass(clazz);
            $(element).parent().parent().removeClass(clazz);
        };
    });

})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.directives.simple_report');

    /**
     * Controller handling sliders used by range filters.
     * (!) This controller previously was in static/dataiku/js/simple_report/chart_logic.js
     */
    app.controller('ChartSliderController', function($scope, $timeout) {
        $scope.slideEnd = function() {
            $timeout(function() {
                const filterData = $scope.filterTmpData[$scope.$index];
                const facetUiState = $scope.facetUiStates[$scope.$index];
                if (!filterData || !facetUiState) {
                    return;
                } // when a filter is added, you have to wait for the response from the backend before filterTmpData is here
                filterData.minValue = facetUiState.sliderModelMin;
                filterData.maxValue = facetUiState.sliderModelMax;
            });
        };
    });

})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.charts');

    // (!) This controller previously was in static/dataiku/js/simple_report/config_ui.js
    app.controller('ColorSelectorController', function($scope, ChartColorSelection) {
        $scope.getUaColor = function(measure) {
            return ChartColorSelection.getUaColor($scope.chart.def, measure);
        };
        $scope.getColorMeasure = function() {
            return ChartColorSelection.getColorMeasure($scope.chart.def);
        }
    });
})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.charts');

    // (!) This controller previously was in static/dataiku/js/simple_report/config_ui.js
    app.controller('ContinuousColorSelectorController', function($scope, ChartColorSelection) {
        $scope.getColorOptions = function(geoLayer) {
            return ChartColorSelection.getOptions($scope.chart.def, geoLayer);
        };

        $scope.getColorDimensionOrMeasure = function(geoLayer) {
            return ChartColorSelection.getColorDimensionOrMeasure($scope.chart.def, geoLayer);
        };
    });
})();

;


(function() {
    'use strict';

    angular.module('dataiku.charts')
        .controller('EditCustomPaletteModalController', EditCustomPaletteModalController);

    /**
     * (!) This controller previously was in static/dataiku/js/simple_report/common/colors.js
     */
    function EditCustomPaletteModalController($scope, DataikuAPI, $filter, $timeout, StateUtils, FutureProgressModal, ColorUtils, WT1, translate) {
        let originalPalette;

        $scope.uiState = {};
        $scope.exportOptions = {};

        $scope.paletteInvalidMessage = translate('COMPONENT.COLOR_PALETTE.MODAL.INVALID_MESSAGE', 'Colors are required and must be valid hexadecimal, rgb(a) or CSS values.');
        $scope.noChangeMessage = translate('COMPONENT.COLOR_PALETTE.MODAL.NO_CHANGE_MESSAGE', 'No changes to be saved');

        $scope.init = function(palette, paletteType, specifyValuesEnabled = true) {
            $scope.palette = angular.copy(palette);
            $scope.defaultColors = ColorUtils.getDiscretePaletteColors('dku_font');
            originalPalette = angular.copy(palette);
            $scope.paletteType = paletteType;
            $scope.specifyValuesEnabled = specifyValuesEnabled;
            $scope.isDirty = false;
            $scope.isPaletteColorsFormValid = true;
        };

        $scope.save = function() {
            $scope.resolveModal($scope.palette);
        };

        $scope.$watch('palette', function(nv) {
            $scope.isDirty = originalPalette && !_.isEqual(nv, originalPalette);
        }, _.isEqual);

        $scope.codeMirrorOptions = {
            mode: 'application/javascript',
            lineNumbers: false,
            readOnly: true,
            onLoad: function(instance) {
                instance.on('focus', function() {
                    instance.execCommand('selectAll');
                });
            }
        };

        $scope.handlePaletteChange = function(palette) {
            $timeout(() => {
                $scope.palette = palette;
            });
        };

        $scope.handleValidityChange = function(validity) {
            $timeout(() => {
                $scope.isPaletteColorsFormValid = validity === 'VALID';
            });
        };

        const getJsSnippet = function(type, id, name, colors, values) {
            let clippedValues;
            if (values && values.length) {
                clippedValues = values.concat();
                clippedValues.length = colors.length;
            }

            return 'dkuColorPalettes.add' + $filter('capitalize')(type.toLowerCase()) + '({'
                + '\n    "id": ' + JSON.stringify(id) + ','
                + '\n    "name": ' + JSON.stringify(name) + ','
                + '\n    "category": "Plugin palettes",'
                + '\n    "colors": ' + JSON.stringify(colors)
                + (clippedValues ? (',\n    "values": ' + JSON.stringify(clippedValues)) : '')
                + '\n});';
        };

        $scope.updateSnippet = function() {
            $scope.jsSnippet = getJsSnippet($scope.paletteType, $scope.exportOptions.paletteId, $scope.exportOptions.paletteName, $scope.palette.colors, $scope.palette.values);
        };

        $scope.prepareExport = function() {
            $scope.updateSnippet();
            $scope.uiState.exporting = true;
        };

        $scope.export = function() {
            DataikuAPI.plugindev.create($scope.exportOptions.pluginId, 'EMPTY')
                .error(setErrorInScope.bind($scope))
                .success(function(data) {
                    FutureProgressModal.show($scope, data, 'Creating plugin').then(function(result) {
                        if (result) {
                            WT1.event('plugin-dev-create');
                            DataikuAPI.plugindev.createContent($scope.exportOptions.pluginId, '/js', true)
                                .error(setErrorInScope.bind($scope))
                                .success(function() {
                                    DataikuAPI.plugindev.createContent($scope.exportOptions.pluginId, '/js/palette.js', false)
                                        .error(setErrorInScope.bind($scope))
                                        .success(function() {
                                            DataikuAPI.plugindev.setContent($scope.exportOptions.pluginId, '/js/palette.js', $scope.jsSnippet)
                                                .error(setErrorInScope.bind($scope))
                                                .success(function() {
                                                    $scope.dismiss();
                                                    StateUtils.go.pluginDefinition($scope.exportOptions.pluginId);
                                                });
                                        });
                                });
                        }
                    });
                });
        };
    }
})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.charts');

    app.controller('GeometryMapColorController', function($scope) {
        $scope.geoLayers = [];

        $scope.onGeoLayerChange = function(index) {
            $scope.index = index;
            $scope.geoLayer = $scope.chart.def.geoLayers[index];
        };

        $scope.hasColor = function() {
            return $scope.geoLayer.uaColor.length;
        };

        $scope.$watch('chart.def.geoLayers', function(newValue) {
            //if a geolayer was selected already, we need to map it to the correct index (in case of reorder)
            if (!_.isNil($scope.geoLayer)) {
                const selectedGeoIndex = newValue.findIndex(layer => _.isEqual(layer, $scope.geoLayer));
                if (selectedGeoIndex >= 0) {
                    $scope.index = selectedGeoIndex;
                }
            } else {
                const displayedLayerIndex = newValue.findIndex(geoLayer => geoLayer.geometry && geoLayer.geometry.length);
                if (displayedLayerIndex >= 0) {
                    $scope.geoLayer = newValue[displayedLayerIndex];
                    $scope.index = displayedLayerIndex;
                }
            }

            const geoLayers = [];
            newValue.forEach((layer, index) => {
                if (layer.geometry && layer.geometry.length) {
                    let name = layer.geometry[0].column;
                    if (layer.uaColor && layer.uaColor.length) {
                        name += ` / ${layer.uaColor[0].column}`;
                    }
                    geoLayers.push({ id: index, name });
                }
            });
            $scope.geoLayers = geoLayers;
            $scope.hasGeoLayers = newValue && newValue.some(geoLayer => geoLayer.geometry && geoLayer.geometry.length);
        }, true);
    });
})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.charts');

    // (!) This controller previously was in static/dataiku/js/simple_report/config_ui.js
    app.controller('SingleColorSelectorController', function($scope, ChartColorSelection) {
        $scope.colors = [
            '#F03334', '#FF7703', '#F6C762', '#ECD941', '#82D96B', '#63E9C3',
            '#69CEF0', '#1EA8FC', '#2678B1', '#7638AF', '#BE66BF', '#EA3596',
            '#000000', '#8A8A8A', '#BABBBB', '#D2D2D2', '#E8E8E8', '#FFFFFF'
        ];

        $scope.grayBlock = [0, 1, 2, 3, 4, 5, 6, 7, 8, 9].map(function(x) {
            const c = x / 10 * 255;
            return d3.rgb(c, c, c).toString();
        });

        $scope.getColorOptions = function(geoLayer) {
            return ChartColorSelection.getOptions($scope.chart.def, geoLayer);
        };
    });
})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.charts');

    // (!) This controller previously was in static/dataiku/js/simple_report/maps/maps_base.js
    app.controller('MapBackgroundPickerController', function($scope, BuiltinMapBackgrounds) {
        $scope.backgrounds = BuiltinMapBackgrounds.getBackgrounds();
        $scope.categories = {};
        $scope.backgrounds.forEach(function(background) {
            if (!$scope.categories[background.category]) {
                $scope.categories[background.category] = [background];
            } else {
                $scope.categories[background.category].push(background);
            }
        });
    });

})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.directives.simple_report');

    /**
     * Controller for common logic to handle std aggregated measures in the dropdown.
     */
    app.controller('StdAggregatedMeasureController', ($scope, ChartLabels, ChartFeatures, ChartCustomMeasures, ColumnAvailability) => {
        $scope.ChartFeatures = ChartFeatures;
        $scope.ChartLabels = ChartLabels;
        $scope.getAggregationDescription = ColumnAvailability.getAggregationDescription;
        $scope.countOfRecordsLabel = ChartLabels.COUNT_OF_RECORDS_LABEL;
        $scope.getAvailableAggregations = (measureType, chartType) => ChartLabels.getAvailableAggregationsLabels(measureType, chartType, $scope.contextualMenuMeasureType);
        $scope.getCustomMeasure = ChartCustomMeasures.getCustomMeasure;
        $scope.getAvailableUnaggregatedModes = (measure) => ChartLabels.getAvailableUnaggregatedModesLabels(measure, $scope.chart.def, $scope.contextualMenuMeasureType);
        $scope.onUnaggregatedChange = function(isMeasureUnaggregated, measure) {
            if (isMeasureUnaggregated) {
                const compatibleUaComputeModes = Object.keys($scope.getAvailableUnaggregatedModes(measure));
                if (!compatibleUaComputeModes.includes(measure.uaComputeMode)) {
                    measure.uaComputeMode = compatibleUaComputeModes[0];
                }
            }
            if (!$scope.$$phase) {
                $scope.$apply();
            }
        };
    });

})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.charts');

    // (!) This controller previously was in static/dataiku/js/simple_report/config_ui.js
    app.controller('AdminMapChartController', function($scope, ChartTypeChangeHandler) {
        $scope.acceptGeo = function(data) {
            return ChartTypeChangeHandler.acceptGeo(data);
        };
        $scope.acceptMeasure = function(data) {
            return ChartTypeChangeHandler.stdAggregatedAcceptMeasure(data);
        };
    });
})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.charts');

    // (!) This controller previously was in static/dataiku/js/simple_report/config_ui.js
    app.controller('BinnedXYChartDefController', function($scope, ChartTypeChangeHandler, CHART_VARIANTS, ChartColumnTypeUtils) {
        $scope.acceptMeasure = function(data) {
            return ChartTypeChangeHandler.stdAggregatedAcceptMeasure(data);
        };
        $scope.acceptDimensionOrHierarchy = function(data) {
            return ChartTypeChangeHandler.binnedXYAcceptDimensionOrHierarchy($scope.chart.def.variant, data);
        };
        $scope.hasSizeMenu = function() {
            return $scope.chart.def.variant === CHART_VARIANTS.binnedXYHexagon;
        };
        $scope.getXDimList = (data) => ChartColumnTypeUtils.isHierarchy(data) ? $scope.chart.def.xHierarchyDimension : $scope.chart.def.xDimension;
        $scope.getYDimList = (data) => ChartColumnTypeUtils.isHierarchy(data) ? $scope.chart.def.yHierarchyDimension : $scope.chart.def.yDimension;
    });
})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.charts');

    // (!) This controller previously was in static/dataiku/js/simple_report/config_ui.js
    app.controller('BoxplotsChartDefController', function($scope, ChartTypeChangeHandler, ChartColumnTypeUtils) {
        $scope.acceptMeasure = function(data) {
            return ChartTypeChangeHandler.boxplotsAcceptMeasure(data);
        };
        $scope.acceptBreakdownDimension = function(data) {
            return ChartTypeChangeHandler.boxplotsAcceptBreakdown(data);
        };

        $scope.acceptColorDimension = function(data) {
            return ChartTypeChangeHandler.boxplotsAcceptColorDimension(data);
        };

        $scope.getBreakdownDimList = (data) => ChartColumnTypeUtils.isHierarchy(data) ? $scope.chart.def.boxplotBreakdownHierarchyDimension : $scope.chart.def.boxplotBreakdownDim;
    });
})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.charts');

    // (!) This controller previously was in static/dataiku/js/simple_report/config_ui.js
    app.controller('Density2DChartDefController', function($scope, ChartTypeChangeHandler) {
        $scope.accept = function(data) {
            return ChartTypeChangeHandler.density2dAccept(data);
        };
    });
})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.charts');

    // (!) This controller previously was in static/dataiku/js/simple_report/config_ui.js
    app.controller('DensityHeatMapChartController', function($scope, ChartTypeChangeHandler) {
        $scope.acceptGeo = function(data) {
            return ChartTypeChangeHandler.acceptGeo(data);
        };
        $scope.acceptScaleMeasure = function(data) {
            return ChartTypeChangeHandler.densityMapAcceptScaleMeasure(data);
        };
    });
})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.charts');

    // (!) This controller previously was in static/dataiku/js/simple_report/config_ui.js
    app.controller('GeometryMapChartController', function($scope, ChartTypeChangeHandler, ChartColorSelection, ChartsStaticData) {

        $scope.$watch('chart.def.geoLayers', function(newValue) {
            const lastGeoLayer = newValue[newValue.length - 1];
            $scope.generateEmptyPlaceholder(lastGeoLayer);
            prepareGeoLayers(newValue);
        }, true);


        $scope.acceptMeasure = data => ChartTypeChangeHandler.scatterAccept(data);

        $scope.acceptGeo = data => ChartTypeChangeHandler.acceptGeo(data);

        $scope.getUaColor = measure => ChartColorSelection.getUaColor($scope.chart.def, measure);

        $scope.removeGeometry = function(index) {
            if (!isLastGeoLayer(index)) {
                $scope.chart.def.geoLayers.splice(index, 1);
            }
        }

        $scope.sortableOptions = {
            update: function(e, ui) {
                const sortableElement = ui.item.sortable;
                preventTheLastItemFromMoving(sortableElement);
            },
            handle: '.handle'
        };

        $scope.isEmptyPlaceholder = index => isLastGeoLayer(index);

        function isLastGeoLayer(index) {
            return (index === $scope.chart.def.geoLayers.length - 1);
        }

        function preventTheLastItemFromMoving(sortableElement) {
            const lastItemIndex = sortableElement.sourceModel.length - 1;
            if ((sortableElement.index === lastItemIndex) || (sortableElement.dropindex === lastItemIndex)) {
                sortableElement.cancel();
            }
        }

        function isLayerEmpty(geoLayer) {
            return (geoLayer.geometry.length === 0 && geoLayer.uaColor.length === 0);
        }

        $scope.generateEmptyPlaceholder = function(lastGeoLayer) {
            if (!isLayerEmpty(lastGeoLayer)) {
                $scope.chart.def.geoLayers.push(ChartTypeChangeHandler.newEmptyGeoPlaceholder($scope.chart.def.geoLayers.length));
            }
        }

        function prepareGeoLayers(geoLayers) {
            for (let i = 0; i < geoLayers.length - 1; ++i) {
                if (isLayerEmpty(geoLayers[i])) {
                    geoLayers.splice(i, 1);
                } else {
                    autoCompleteGeoLayers(geoLayers[i])
                }
            }
        }

        function autoCompleteGeoLayers(geoLayer) {
            if (geoLayer.geometry.length > 0) {
                ChartTypeChangeHandler.autocompleteUA(geoLayer.geometry[0]);
            }
            if (geoLayer.uaColor.length > 0) {
                ChartTypeChangeHandler.autocompleteUA(geoLayer.uaColor[0]);
                if (geoLayer.geometry.length > 0) {
                    geoLayer.geometry[0].aggregationFunction = ChartsStaticData.GEOM_AGGREGATIONS.DEFAULT;
                }
            }
        }
    });
})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.charts');

    app.controller('KpiChartDefController', function($scope, ChartTypeChangeHandler) {
        $scope.acceptMeasure = function(data) {
            return ChartTypeChangeHandler.stdAggregatedAcceptMeasureWithAlphanumResults(data);
        };
    });
})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.charts');

    app.controller('PivotChartDefController', function($scope, ChartTypeChangeHandler, ChartColumnTypeUtils) {
        $scope.acceptMeasure = function(data) {
            return ChartTypeChangeHandler.stdAggregatedAcceptMeasureWithAlphanumResults(data);
        };
        $scope.acceptDimensionOrHierarchy = function(data) {
            return ChartTypeChangeHandler.stdAggregatedAcceptDimensionOrHierarchy(data);
        };
        $scope.acceptColor = function(data) {
            return ChartTypeChangeHandler.stdAggregatedAcceptMeasure(data);
        };
        $scope.getXDimList = (data) => ChartColumnTypeUtils.isHierarchy(data) ? $scope.chart.def.xHierarchyDimension : $scope.chart.def.xDimension;
        $scope.getYDimList = (data) => ChartColumnTypeUtils.isHierarchy(data) ? $scope.chart.def.yHierarchyDimension : $scope.chart.def.yDimension;
    });
})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.charts');

    // (!) This controller previously was in static/dataiku/js/simple_report/config_ui.js
    app.controller('ScatterChartController', function($scope, ChartTypeChangeHandler) {
        $scope.scatterAcceptDrop = function(data) {
            return ChartTypeChangeHandler.scatterAccept(data);
        };
        $scope.scatterAcceptScaleMeasure = function(data) {
            return ChartTypeChangeHandler.scatterAcceptScaleMeasure(data);
        };
    });
})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.charts');

    // (!) This controller previously was in static/dataiku/js/simple_report/config_ui.js
    app.controller('ScatterMapChartController', function($scope, ChartTypeChangeHandler) {
        $scope.acceptGeo = function(data) {
            return ChartTypeChangeHandler.acceptGeo(data);
        };

        $scope.acceptMeasure = function(data) {
            return ChartTypeChangeHandler.scatterAccept(data);
        };

        $scope.acceptScaleMeasure = function(data) {
            return ChartTypeChangeHandler.scatterAcceptScaleMeasure(data);
        };
    });
})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.charts');

    app.controller('ScatterMultiplePairsChartController', function($scope, ChartTypeChangeHandler, ChartUADimension, ChartAxesUtils, ChartsStaticData, translate) {
        $scope.dimensionPairs = _.clone($scope.chart.def.uaDimensionPair);
        const lastPair = $scope.dimensionPairs[$scope.dimensionPairs.length - 1];
        generateEmptyPlaceholder(lastPair);

        $scope.$watch('dimensionPairs', function(newValue) {
            let lastPair = newValue[newValue.length - 1];
            generateEmptyPlaceholder(lastPair);

            newValue.forEach((pair, i) => {
                autocompletePair(pair);
                if (ChartUADimension.isPairEmpty(pair) && !isLastPair(i)) {
                    $scope.dimensionPairs.splice(i, 1);
                }
            });

            const lastIndex = $scope.dimensionPairs.length - 1;
            lastPair = $scope.dimensionPairs[lastIndex];
            const newDimensionPairs = ChartUADimension.isPairEmpty(lastPair) ? $scope.dimensionPairs.slice(0, lastIndex) : $scope.dimensionPairs;
            const yAxesFormatting = [];
            const pairIds = ChartUADimension.createUaDimPairIds(newDimensionPairs);
            newDimensionPairs.forEach((pair, index) => {
                pair.id = pairIds[index];
                if (pair.id) {
                    let formatting = $scope.chart.def.yAxesFormatting.find(v => v.id === pair.id);
                    if (!formatting) {
                        formatting = ChartAxesUtils.initYAxesFormatting(pair.id);
                    }
                    yAxesFormatting.push(formatting);
                }
            });
            const basicChartsFormatting = $scope.chart.def.yAxesFormatting.filter(v => v.id === ChartsStaticData.LEFT_AXIS_ID || v.id === ChartsStaticData.RIGHT_AXIS_ID);
            $scope.chart.def.uaDimensionPair = [...newDimensionPairs];
            $scope.chart.def.yAxesFormatting = [...basicChartsFormatting, ...yAxesFormatting];
        }, true);

        $scope.scatterAcceptUaXDrop = function(data, pairIndex) {
            return ChartTypeChangeHandler.scatterMPAccept(data, $scope.chart.def, 'x', pairIndex);
        };

        $scope.scatterAcceptUaYDrop = function(data, pairIndex) {
            return ChartTypeChangeHandler.scatterMPAccept(data, $scope.chart.def, 'y', pairIndex);
        };

        $scope.getXDropzonePlaceholder = function(pairIndex) {
            const pair = $scope.dimensionPairs[pairIndex];
            if (pair.uaYDimension && pair.uaYDimension.length) {
                const uaXDimension = ChartUADimension.getPairUaXDimension($scope.dimensionPairs, pair);
                if (uaXDimension && uaXDimension.length) {
                    return uaXDimension[0].column;
                }
            }
            return translate('CHARTS.DEFINITION.SHARED.X_AXIS', 'Drop to set the X axis');
        };

        $scope.sortableOptions = {
            handle: '.handle'
        };

        function generateEmptyPlaceholder(lastPair) {
            if ((!lastPair || !ChartUADimension.isPairEmpty(lastPair)) && $scope.dimensionPairs.length < 5) {
                $scope.dimensionPairs.push({ uaXDimension: [], uaYDimension: [] });
            }
        };

        function autocompletePair(pair) {
            if (pair.uaXDimension.length) {
                ChartTypeChangeHandler.autocompleteUA(pair.uaXDimension[0]);
            }
            if (pair.uaYDimension.length) {
                ChartTypeChangeHandler.autocompleteUA(pair.uaYDimension[0]);
            }
        }

        $scope.removePair = function(index) {
            $scope.dimensionPairs.splice(index, 1);
        };

        $scope.isEmptyPlaceholder = index => ChartUADimension.isPairEmpty($scope.dimensionPairs[index]);

        function isLastPair(index) {
            return index === $scope.dimensionPairs.length - 1;
        }
    });
})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.charts');

    // (!) This controller previously was in static/dataiku/js/simple_report/config_ui.js
    app.controller('StdAggregatedChartDefController', function($scope, ChartTypeChangeHandler, ChartColumnTypeUtils, ChartFeatures) {

        $scope.ChartFeatures = ChartFeatures;
        $scope.acceptMeasure = function(data) {
            return ChartTypeChangeHandler.stdAggregatedAcceptMeasure(data);
        };

        $scope.acceptDimensionOrHierarchy = function(data) {
            return ChartTypeChangeHandler.stdAggregatedAcceptDimensionOrHierarchy(data);
        };

        $scope.acceptDimension = function(data) {
            return ChartTypeChangeHandler.stdAggregatedAcceptDimension(data);
        };

        $scope.getDim0List = (data) => ChartColumnTypeUtils.isHierarchy(data) ? $scope.chart.def.genericHierarchyDimension : $scope.chart.def.genericDimension0;
    });
})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.charts');

    // (!) This controller previously was in static/dataiku/js/simple_report/config_ui.js
    app.controller('StdGroupedChartDefController', function($scope, ChartTypeChangeHandler, ChartColumnTypeUtils) {

        $scope.acceptMeasure = function(data) {
            return ChartTypeChangeHandler.stdAggregatedAcceptMeasure(data);
        };

        $scope.acceptDimensionOrHierarchy = function(data) {
            return ChartTypeChangeHandler.stdAggregatedAcceptDimensionOrHierarchy(data);
        };

        $scope.getGroupDimList = (data) => ChartColumnTypeUtils.isHierarchy(data) ? $scope.chart.def.groupHierarchyDimension : $scope.chart.def.groupDimension;
    });
})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.charts');

    app.controller('TreemapChartDefController', function($scope, ChartTypeChangeHandler, ChartColumnTypeUtils) {

        $scope.acceptMeasure = function(data) {
            return ChartTypeChangeHandler.stdAggregatedAcceptMeasure(data);
        };

        $scope.acceptDimensionOrHierarchy = function(data) {
            return ChartTypeChangeHandler.stdAggregatedAcceptDimensionOrHierarchy(data);
        };

        $scope.getYDimList = (data) => ChartColumnTypeUtils.isHierarchy(data) ? $scope.chart.def.yHierarchyDimension : $scope.chart.def.yDimension;
    });
})();

;
(function() {
    'use strict';

    angular.module('dataiku.charts')
        .directive('animatedChartSlider', animatedChartSlider);

    /**
     * (!) This directive previously was in static/dataiku/js/simple_report/common/animation.js
     */
    function animatedChartSlider(ChartDimension, ChartUADimension) {
        return {
            scope: {
                labels: '=',
                currentFrame: '=',
                dimension: '='
            },
            template: '<div class="horizontal-flex animated-chart-slider" style="align-items: center;">'
                + '<div class="noflex">{{firstValue}}</div>'
                + '<div class="progress flex">'
                + '<div class="current" style="left:{{cursorLeft}}%; width: {{cursorWidth}}%;" ng-mousedown="startSliding($event)" ng-mouseup="stopSliding()"></div>'
                + '</div>'
                + '<div class="noflex">{{lastValue}}</div>'
                + '</div>',
            link: function($scope, $el) {

                let labelPositions;

                const findClosestIdx = function(x, arr) {
                    const indexArr = arr.map(function(k) {
                        return Math.abs(k.center - x)
                    });
                    const min = Math.min.apply(Math, indexArr);
                    return indexArr.indexOf(min);
                };

                $scope.$watch('labels', function(nv) {
                    if (!nv) {
                        return;
                    }

                    if (ChartDimension.isUngroupedNumerical($scope.dimension)) {
                        $scope.firstValue = $scope.labels[0].sortValue;
                        $scope.lastValue = $scope.labels[$scope.labels.length - 1].sortValue;
                        const scale = d3.scale.linear()
                            .domain([$scope.labels[0].sortValue, $scope.labels[$scope.labels.length - 1].sortValue])
                            .range([0, 100]);
                        labelPositions = $scope.labels.map(function(label) {
                            return {
                                center: scale(label.sortValue),
                                start: scale(label.sortValue) - 1,
                                width: 2
                            };
                        });
                    } else if (ChartDimension.isGroupedNumerical($scope.dimension)) {
                        $scope.firstValue = $scope.labels[0].min;
                        $scope.lastValue = $scope.labels[$scope.labels.length - 1].max;
                        const linearScale = d3.scale.linear()
                            .domain([$scope.labels[0].min, $scope.labels[$scope.labels.length - 1].max])
                            .range([0, 100]);
                        labelPositions = $scope.labels.map(function(label) {
                            return {
                                center: linearScale(label.sortValue),
                                start: linearScale(label.min),
                                width: linearScale(label.max) - linearScale(label.min)
                            };
                        });
                    } else if (ChartDimension.isAlphanumLike($scope.dimension) || ChartUADimension.isDiscreteDate($scope.dimension)) {
                        $scope.firstValue = null;
                        $scope.lastValue = null;
                        const ordinalScale = d3.scale.ordinal()
                            .domain($scope.labels.map(function(d, i) {
                                return i;
                            }))
                            .rangeBands([0, 100]);

                        labelPositions = $scope.labels.map(function(label, i) {
                            return {
                                start: ordinalScale(i),
                                width: ordinalScale.rangeBand(),
                                center: ordinalScale(i) + ordinalScale.rangeBand() / 2
                            };
                        });
                    }

                    if ($scope.currentFrame !== null) {
                        $scope.cursorLeft = labelPositions[$scope.currentFrame].start;
                        $scope.cursorWidth = labelPositions[$scope.currentFrame].width;
                    }
                });

                const slideCursor = function($evt) {
                    $evt.preventDefault(); // useful to avoid selecting content while sliding
                    const sliderPosition = $el.offset().left;
                    const xPosition = ($evt.pageX - sliderPosition) / $el.width() * 100;
                    $scope.$apply(function() {
                        $scope.currentFrame = findClosestIdx(xPosition, labelPositions);
                    });
                };

                $scope.startSliding = function($evt) {
                    $scope.sliding = true;
                    $(window).on('mouseup.chart-animation.' + $scope.$id, $scope.stopSliding);
                    $(window).on('mousemove.chart-animation' + $scope.$id, slideCursor);
                    $('body').css('cursor', 'move');
                };

                $scope.stopSliding = function() {
                    $scope.sliding = false;
                    $(window).off('mouseup.chart-animation.' + $scope.$id);
                    $(window).off('mousemove.chart-animation' + $scope.$id);
                    $('body').css('cursor', 'auto');
                };

                $scope.$watch('currentFrame', function(nv) {
                    if (nv == null) {
                        return;
                    }
                    $scope.cursorLeft = labelPositions[nv].start;
                    $scope.cursorWidth = labelPositions[nv].width;
                });
            }
        }
    }

})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.directives.simple_report');

    /*
     * (!) This directive previously was in static/dataiku/js/simple_report/chart_logic.js
     *
     * Need in the scope :
     * - response (containing the pivotResponse - among other things)
     * - "chart" : the chart object, which must contain at least
     *       - data, a ChartSpec object
     *       - summary
     * - 'getExecutePromise(request)' : a function that returns a promise
     * DO NOT USE AN ISOLATE SCOPE as there is some communication with drag-drop
     * stuff
     */
    app.directive('chartConfiguration', function(MonoFuture, Debounce, DataikuAPI, translate,
        ChartDimension, ChartLabels, ChartRequestComputer, $state, $timeout, PluginsService, Logger, ActivityIndicator,
        ChartTypeChangeHandler, ChartDefinitionChangeHandler, ChartUADimension, _MapCharts, ChartsStaticData, ChartConfigurationCopyPaste, DetectUtils, ChartIconUtils, ChartSetErrorInScope, ChartFeatures, PolygonSources, ChartAxesUtils,
        ChartZoomControlAdapter, ChartDataUtils, WT1, CHART_TYPES, CHART_VARIANTS, ChartStoreFactory, ColorUtils, $rootScope, $stateParams, ChartsAvailableTypes, RegressionTypes, ChartFilterUtils, ChartFormattingPane, ChartFormattingPaneSections,
        DataGroupingMode, ChartColorSelection, ConditionalFormattingOptions, ChartDataWrapperFactory, DSSVisualizationThemeUtils, ChartDrilldown, ChartHierarchyDimension, ChartHierarchies, DRILL_UP_SOURCE) {
        return {
            restrict: 'AE',
            templateUrl: '/static/dataiku/js/simple_report/directives/chart-configuration/chart-configuration.directive.html',
            link: function(scope, element, attrs) {
                // We can't isolate the scope, so we watch the attribute for changes and update the scope variable
                scope.$watch(attrs['dropdownMaxHeightPx'], function(newVal) {
                    scope.dropdownMaxHeightPx = newVal;
                });
                //needs to be put in the scope so these functions are accessible in the dashboards
                scope.canHaveZoomControls = ChartFeatures.canHaveZoomControls;
                scope.canDisplayLegend = ChartFeatures.canDisplayLegend;
                scope.isStackedLegend = ChartFeatures.isStackedLegend;

                ChartSetErrorInScope.defineInScope(scope);
                scope.isInAnalysis = $state.current.name.indexOf('analysis') != -1;
                scope.isInPredicted = $state.current.name.indexOf('predicted') != -1;
                scope.isInInsight = $state.current.name.indexOf('insight') != -1;
                scope.isInNoteBooks = $state.current.name.indexOf('notebooks') != -1;

                // Notebooks can actually have multiple charts, but you can only configure one at a time.
                scope.canConfigureMaxOneChart = scope.isInInsight || scope.isInNoteBooks;

                scope.shouldDisplaySamplingStatus = shouldDisplaySamplingStatus;
                scope.RegressionTypes = RegressionTypes;
                scope.PolygonSources = PolygonSources;

                scope.hierarchyDimsInChart = [];
                scope.breadcrumbs = [];
                scope.breadcrumbTooltips = {};

                scope.hasLegacyBadge = ChartFeatures.hasLegacyBadge;
                scope.hasBetaBadge = ChartFeatures.hasBetaBadge;

                scope.legacyBadge = {
                    tooltip: translate('CHARTS.HEADER.BADGE.LEGACY.TOOLTIP', 'You are using the legacy engine. Click to go back to the new one.')
                };
                scope.editingTitle = false;

                scope.warningBadge = ChartsStaticData.WARNING_BADGE;

                scope.chartTypePickerOptions = {
                    chartVariant: scope.chart.def.variant,
                    chartType: scope.chart.def.type,
                    webappType: scope.chart.def.webAppType
                };

                scope.availableChartTypes = ChartsAvailableTypes.getAvailableChartTypes();
                scope.chartTypePickerOptions = { ...scope.chartTypePickerOptions, chartTypes: scope.availableChartTypes };

                if (!scope.isInPredicted && !scope.isInAnalysis && !scope.isInNoteBooks) {
                    DataikuAPI.explores.listPluginChartDescs($stateParams.projectKey)
                        .success((data) => {
                            scope.webapps = parseWebapps(data);
                            ChartConfigurationCopyPaste.setWebApps(scope.webapps);
                            scope.chartTypePickerOptions = { ...scope.chartTypePickerOptions, webappTypes: scope.webapps };
                        }).error(setErrorInScope.bind(scope));
                };

                if (!scope.chartBottomOffset) {
                    scope.chartBottomOffset = 0;
                }

                scope.optionsFolds = {
                    legend: true,
                    chartMode: true,
                    showTopBar: true
                };
                scope.PluginsService = PluginsService;

                scope.getPivotResponseOptions = () => ({
                    projectKey: $stateParams.projectKey,
                    dataSpec: scope.getDataSpec(),
                    requestedSampleId: scope.chart.summary.requiredSampleId,
                    onError: onPivotRequestError(scope),
                    visualAnalysisFullModelId: $stateParams.fullModelId
                });

                Mousetrap.bind('s h h', function() {
                    scope.$apply(function() {
                        scope.bigChartSwitch();
                    });
                });

                Mousetrap.bind('s v', function() {
                    scope.$apply(function() {
                        scope.switchCharts();
                    });
                });

                scope.$on('$destroy', function() {
                    Mousetrap.unbind('s h h');
                    Mousetrap.unbind('s v');
                });

                scope.switchCharts = function() {
                    if (ChartFeatures.hasEChartsDefinition(scope.chart.def.type) && ChartFeatures.hasD3Definition(scope.chart.def.type)) {
                        if (ChartFeatures.hasDefaultEChartsDisplay(scope.chart.def.type)) {
                            scope.chart.def.displayWithEChartsByDefault = !scope.chart.def.displayWithEChartsByDefault;
                        } else {
                            scope.chart.def.displayWithECharts = !scope.chart.def.displayWithECharts;
                        }
                    } else if (!ChartFeatures.hasD3Definition(scope.chart.def.type)) {
                        Logger.warn('No d3 definition for this chart');
                    } else {
                        Logger.warn('No echarts definition for this chart');
                    }
                };

                scope.openSection = function(section, properties) {
                    const readOnly = $state.current.name === 'projects.project.dashboards.insights.insight.view';
                    if (!readOnly && !(section === ChartFormattingPaneSections.MISC && scope.isInPredicted)) {
                        ChartFormattingPane.open(section, true, true);
                        if (properties && typeof properties === 'object') {
                            Object.keys(properties).forEach(propertyName => {
                                ChartFormattingPane.setData(propertyName, properties[propertyName]);
                            });
                        }
                    }
                };

                scope.onFoldableSectionLoaded = function(section) {
                    let sections = [];

                    if ($stateParams.sections) {
                        sections = JSON.parse($stateParams.sections);
                    }

                    sections.forEach(sectionId => {
                        if (section.getId() === sectionId) {
                            section.open();
                            setTimeout(() => section.scrollIntoView());
                        }
                    });
                };

                scope._MapCharts = _MapCharts;

                const addBigChartClass = (addClass) => {
                    if (addClass) {
                        $('.charts-container').addClass('big-chart');
                    } else {
                        $('.charts-container').removeClass('big-chart');
                    }
                };

                scope.fixupCurrentChart = function() {
                    ChartTypeChangeHandler.fixupChart(scope.chart.def, scope.chart.theme);
                };

                addBigChartClass(false); // remove class in case user is switching from a big chart
                scope.bigChart = false;
                scope.chartHeaderLabel = translate('CHARTS.HEADER.EXPAND', 'Expand chart');
                scope.bigChartSwitch = function() {
                    scope.bigChart = !scope.bigChart;
                    scope.chartHeaderLabel = scope.bigChart ? translate('CHARTS.HEADER.REDUCE', 'Reduce chart') : translate('CHARTS.HEADER.EXPAND', 'Expand chart');
                    scope.isBigChartSwitch = true;
                    $('.graphWrapper').fadeTo(0, 0);
                    addBigChartClass(scope.bigChart);
                    //waiting for the css transition to finish (0.25s, we use 300ms, extra 50ms is fore safety)
                    $timeout(function() {
                        //for binned_xy_hex we need to recompute because width and height are taken into account in chart data computing
                        if (scope.chart.def.type == CHART_TYPES.BINNED_XY && scope.chart.def.variant == CHART_VARIANTS.binnedXYHexagon) {
                            scope.recomputeAndUpdateData();
                            scope.executeIfValid();
                        } else {
                            scope.redraw();
                        }
                        $('.graphWrapper').fadeTo(0, 1);
                    }, 250);
                };

                scope.chartSpecific = {};

                scope.droppedData = [];

                scope.ChartTypeChangeHandler = ChartTypeChangeHandler;
                scope.ChartsStaticData = ChartsStaticData;
                scope.ChartFormattingPane = ChartFormattingPane;
                scope.ChartFormattingPaneSections = ChartFormattingPaneSections;

                let frontImportantChangeTimeoutId;
                let noRedrawDelayedChangeTimeoutId;
                let invalidChangeTimeoutId;
                let themeChangeTimeoutId;

                /**
                 * A hack to redraw the scatter MP when a new dimension pair placeholder is added.
                 * We do not recompute the chart until the pair is complete but we do add a new placeholder
                 * resulting in the chart decreasing in size but not redrawing. This launches a redraw.
                 */
                const resizeObserver = new ResizeObserver(function(entries) {
                    const newHeight = entries[0].contentRect.height;
                    const newWidth = entries[0].contentRect.width;
                    const deltaHeight = Math.abs(scope.chartParamBarHeight - newHeight);
                    const deltaWidth = Math.abs(scope.chartParamBarWidth - newWidth);
                    // 30 is a treshold to avoid redrawing too frequently. Can be increased if needed.
                    if ((scope.chartParamBarHeight || scope.chartParamBarWidth) && Math.max(deltaHeight, deltaWidth) > 30) {
                        if (!scope.isBigChartSwitch) {
                            handleResize(scope);
                        } else {
                            scope.isBigChartSwitch = false;
                        }
                    }
                    scope.chartParamBarHeight = newHeight;
                    scope.chartParamBarWidth = newWidth;
                });

                /*
                 * ------------------------------------------------------
                 * only trigger this code once, when the chart is initialized
                 */
                const unregister = scope.$watch('chart', function(nv) {
                    if (nv == null) {
                        return;
                    }
                    unregister();

                    scope.executedOnce = false;

                    if (angular.isUndefined(scope.chart.def)) {
                        Logger.warn('!! BAD CHART !!');
                    }

                    // scope.chart.spec.unregisterWatch = 1;

                    scope.fixupCurrentChart();

                    // STATIC DATA
                    scope.staticData = {};

                    scope.chartTypes = [
                        {
                            type: CHART_TYPES.GROUPED_COLUMNS,
                            title: 'Grouped columns',
                            description: 'Use to create a grouped bar chart.<br/> Break down once to create one group of bars per category. Measures provide bars.<br/> Break down twice to create one group of bars per category and one bar for each subcategory.'
                        },
                        {
                            type: CHART_TYPES.STACKED_COLUMNS,
                            title: 'Stacked columns',
                            description: 'Use to display data that can be summed.<br/> Break down once with several measures to stack the measures.<br/>  Break down twice to create one stack element per value of the second dimension.'
                        },
                        {
                            type: CHART_TYPES.STACKED_AREA,
                            title: 'Stacked area',
                            description: 'Use to display data that can be summed.<br/> Break down once with several measures to stack the measures.<br/>  Break down twice to create one stack element per value of the second dimension.'
                        },
                        {
                            type: CHART_TYPES.LINES,
                            title: 'Lines',
                            description: 'Use to compare evolutions.<br/> Break down once with several measures to create one line per measure.<br/>  Break down twice to create one line per value of the second dimension.'
                        },
                        {
                            type: CHART_TYPES.SCATTER,
                            title: 'Scatter plot',
                            description: 'Scatterize'
                        }
                    ];
                    if (PluginsService.isPluginLoaded('geoadmin')) {
                        scope.chartTypes.push({
                            type: CHART_TYPES.MAP,
                            title: 'World map (BETA)',
                            description: 'Use to plot and aggregate geo data'
                        });
                    } else {
                        scope.chartTypes.push({
                            type: CHART_TYPES.MAP,
                            title: 'World map (BETA)',
                            description: 'Use to plot and aggregate geo data',
                            disabled: true,
                            disabledReason: 'You need to install the \'geoadmin\' plugin. Please see documentation'
                        });
                    }

                    scope.dataGroupingModes = DataGroupingMode;
                    scope.dataGroupingLabels = ChartLabels.DATA_GROUPING_LABELS;
                    scope.allYAxisModes = {
                        'NORMAL': { value: 'NORMAL', label: 'Normal', shortLabel: 'Normal' },
                        'LOG': { value: 'LOG', label: 'Logarithmic scale', shortLabel: 'Log' },
                        'PERCENTAGE_STACK': { value: 'PERCENTAGE_STACK', label: 'Normalize stacks at 100%', shortLabel: '100% stack' }
                    };
                    scope.allXAxisModes = {
                        'NORMAL': { value: 'NORMAL', label: 'Normal', shortLabel: 'Normal' },
                        'CUMULATIVE': { value: 'CUMULATIVE', label: 'Cumulative values', shortLabel: 'Cumulative' },
                        'DIFFERENCE': {
                            value: 'DIFFERENCE', label: 'Difference (replace each value by the diff to the previous one)',
                            shortLabel: 'Difference'
                        }
                    };
                    scope.allComputeModes = {
                        'NONE': { value: 'NONE', label: 'No computation', shortLabel: 'None' },
                        'LIFT_AVG': {
                            value: 'LIFT_AVG', shortLabel: 'Ratio to AVG',
                            label: 'Compute ratio of each value relative to average of values'
                        },
                        'AB_RATIO': {
                            value: 'AB_RATIO', shortLabel: 'a/b ratio',
                            label: 'Compute ratio of measure 1 / measure 2'
                        },
                        'AB_RATIO_PCT': {
                            value: 'AB_RATIO_PCT', shortLabel: 'a/b ratio (%)',
                            label: 'Compute ratio of measure 1 / measure 2, as percentage'
                        }
                    };

                    scope.initChartCommonScopeConfig(scope);
                    scope.graphError = { error: null };

                    scope.dateFilterParts = ChartFilterUtils.getDateChartFilterParts();
                    scope.dateFilterTypes = ChartFilterUtils.getDateFilterTypes();

                    scope.temporalBinningModes = ChartsStaticData.dateModes.concat(ChartsStaticData.UNBINNED_TREAT_AS_ALPHANUM);

                    scope.familyToTypeMap = {
                        'basic': [CHART_TYPES.GROUPED_COLUMNS, CHART_TYPES.STACKED_BARS, CHART_TYPES.STACKED_COLUMNS, CHART_TYPES.MULTI_COLUMNS_LINES, CHART_TYPES.LINES, CHART_TYPES.STACKED_AREA, CHART_TYPES.PIE],
                        'table': [CHART_TYPES.PIVOT_TABLE],
                        'scatter': [CHART_TYPES.SCATTER, CHART_TYPES.GROUPED_XY, CHART_TYPES.BINNED_XY],
                        'map': [CHART_TYPES.SCATTER_MAP, CHART_TYPES.ADMINISTRATIVE_MAP, CHART_TYPES.GRID_MAP, CHART_TYPES.GEOMETRY_MAP, CHART_TYPES.DENSITY_HEAT_MAP],
                        'other': [CHART_TYPES.BOXPLOTS, CHART_TYPES.LIFT, CHART_TYPES.DENSITY_2D, CHART_TYPES.TREEMAP, CHART_TYPES.KPI, CHART_TYPES.RADAR, CHART_TYPES.SANKEY, CHART_TYPES.GAUGE],
                        'webapp': [CHART_TYPES.WEBAPP]
                    };

                    scope.canExportToExcel = ChartFeatures.canExportToExcel;
                    scope.canExportToImage = ChartFeatures.canExportToImage;
                    scope.appVersion = $rootScope.appConfig.version.product_version;

                    scope.getDownloadDisabledReason = function() {
                        return ChartFeatures.getExportDisabledReason(scope.chart.def);
                    };

                    scope.canDownloadChart = function() {
                        return scope.validity.valid && (scope.canExportToExcel(scope.chart.def) || scope.canExportToImage(scope.chart.def));
                    };

                    scope.typeAndVariantToImageMap = ChartIconUtils.typeAndVariantToImageMap;

                    scope.computeChartOptionsPreview = function() {
                        const macos = DetectUtils.getOS() === 'macos';
                        return macos ? '/static/dataiku/images/charts/previews/cmd.svg' : '/static/dataiku/images/charts/previews/ctrl.svg';
                    };

                    scope.computeChartPreview = function(type, variant) {
                        let imageName = '';
                        if (typeof (variant) === 'undefined') {
                            variant = 'normal';
                        }
                        if (typeof (scope.typeAndVariantToImageMap[type]) !== 'undefined'
                            && typeof (scope.typeAndVariantToImageMap[type][variant]) !== 'undefined'
                            && typeof (scope.typeAndVariantToImageMap[type][variant].preview) !== 'undefined') {
                            imageName = scope.typeAndVariantToImageMap[type][variant].preview;
                        }
                        if (imageName != '') {
                            return '/static/dataiku/images/charts/previews/' + imageName + '.png';
                        }
                        return false;
                    };

                    scope.request = {};

                    /*
                     * ------------------------------------------------------
                     * Property accessors and helpers
                     */

                    scope.ChartDimension = ChartDimension;
                    scope.isGroupedNumericalDimension = ChartDimension.isGroupedNumerical.bind(ChartDimension);
                    scope.isTimelineable = ChartDimension.isTimelineable.bind(ChartDimension);
                    scope.isUngroupedNumericalDimension = ChartDimension.isUngroupedNumerical.bind(ChartDimension);
                    scope.isAlphanumLikeDimension = ChartDimension.isAlphanumLike.bind(ChartDimension);
                    scope.isNumericalDimension = ChartDimension.isTrueNumerical.bind(ChartDimension);

                    scope.ChartUADimension = ChartUADimension;

                    scope.setDimensionSort = (dimension, newSortId) => {
                        dimension.sort = angular.copy(scope.chart.def.possibleDimensionSorts.find(
                            sort => sort.sortId === newSortId
                        ));
                    };

                    scope.acceptStdAggrTooltipMeasure = function(data) {
                        return ChartTypeChangeHandler.stdAggregatedAcceptMeasureWithAlphanumResults(data);
                    };

                    scope.acceptUaTooltip = function(data) {
                        return ChartTypeChangeHandler.uaTooltipAccept(data);
                    };

                    scope.acceptFilter = function(data) {
                        const ret = ChartTypeChangeHandler.stdAggregatedAcceptDimension(data);
                        if (!ret.accept || !data) {
                            return ret;
                        }
                        if (data.type == 'GEOMETRY' || data.type == 'GEOPOINT') {
                            return {
                                accept: false,
                                message: 'Cannot filter on Geo dimensions'
                            };
                        }
                        return ret;
                    };

                    scope.dimensionBinDescription = function(dimension) {
                        return ChartDimension.getDimensionBinDescription(dimension, scope.chart.def);
                    };

                    scope.dateModeSuffix = function(mode) {
                        return `(${ChartDimension.getDateModeDescription(mode)})`;
                    };

                    scope.geoDimDescription = function(dim) {
                        for (let i = 0; i < ChartsStaticData.mapAdminLevels.length; i++) {
                            if (dim.adminLevel == ChartsStaticData.mapAdminLevels[i][0]) {
                                return ChartsStaticData.mapAdminLevels[i][2];
                            }
                        }
                        return translate('CHARTS.DROPPED_DIMENSION.UNKNOWN', 'Unknown');
                    };

                    scope.isDateRangeFilter = ChartFilterUtils.isDateRangeFilter.bind(ChartFilterUtils);
                    scope.isDatePartFilter = ChartFilterUtils.isDatePartFilter.bind(ChartFilterUtils);

                    /*
                     * ------------------------------------------------------
                     * Response handling / Facets stuff
                     */

                    scope.filterTmpDataWatchDeregister = null;

                    scope.onResponse = function() {
                        scope.setValidity({ valid: true });
                        scope.uiDisplayState = scope.uiDisplayState || {};
                        scope.uiDisplayState.chartTopRightLabel = ChartDataUtils.computeRecordsStatusLabel(
                            scope.response.result.pivotResponse.beforeFilterRecords,
                            scope.response.result.pivotResponse.afterFilterRecords,
                            ChartDimension.getComputedMainAutomaticBinningModeLabel(scope.response.result.pivotResponse, scope.chart.def),
                            undefined,
                            scope.chart.def.type
                        );

                        scope.uiDisplayState.chartRecordsFinalCountTooltip = ChartDataUtils.getRecordsFinalCountTooltip(
                            scope.chart.def.type,
                            scope.response.result.pivotResponse.afterFilterRecords,
                            scope.response.result.pivotResponse.axesPairs,
                            undefined
                        );

                        scope.uiDisplayState.samplingSummaryMessage = ChartDataUtils.getSamplingSummaryMessage(scope.response.result.pivotResponse, scope.chart.def.type, scope.readOnly ? null : translate('CHARTS.HEADER.DATA_SAMPLING.CLICK_TO_OPEN', 'Click to open sampling settings'), undefined);

                        if (scope.chart.summary && scope.response.result.updatedSampleId) {
                            scope.chart.summary.requiredSampleId = scope.response.result.updatedSampleId;
                        }

                        // For the facet indexes to match the filter indexes, we introduce null entries in front of the filters using the minimal UI, as they don't have facets.
                        const responseFacetsStack = scope.response.result.pivotResponse.filterFacets.reverse();
                        scope.filterFacets = scope.chart.def.filters.map((filter) => {
                            if (filter.useMinimalUi) {
                                return null;
                            }
                            return responseFacetsStack.pop();
                        });
                        /*
                         * Filters need to compare the previous and current values of engineType and refreshableSelection to detect sampling changes.
                         * Because they are mutated on the chart object, we need to duplicate them for the change detection to work properly.
                         */
                        scope.samplingParams = {
                            engineType: scope.chart.engineType || 'LINO',
                            refreshableSelection: angular.copy(scope.chart.refreshableSelection)
                        };
                        scope.recordsMetadata = ChartFeatures.getRecordsMetadata(scope.chart.def, scope.response.result.pivotResponse.sampleMetadata, scope.response.result.pivotResponse.beforeFilterRecords, scope.response.result.pivotResponse.afterFilterRecords);
                        scope.display0Warning = ChartFeatures.shouldDisplay0Warning(scope.chart.def.type, scope.response.result.pivotResponse);
                        scope.colorColumn = ChartColorSelection.getColorDimensionOrMeasure(scope.chart.def, scope.chart.def.geoLayers[0]);

                        const axesDef = ChartDimension.getAxesDef(scope.chart.def, ChartDimension.getChartDimensions(scope.chart.def));
                        scope.chartData = ChartDataWrapperFactory.chartTensorDataWrapper(scope.response.result.pivotResponse, axesDef);
                    };

                    scope.resetResponseErrorsInScope();

                    /*
                     * Wraps scope.getExecutePromise
                     * and add supports for automatic abortion
                     */
                    const executePivotRequest = MonoFuture(scope).wrap(scope.getExecutePromise);

                    scope.executeIfValid = function() {
                        const validity = ChartTypeChangeHandler.getValidity(scope.chart);
                        scope.setValidity(validity);
                        /*
                         * clear the response as well, otherwise when changing the chart, it will
                         * first run once with the new settings and the old response, producing
                         * js errors when something drastic (like chart type) changes
                         */
                        scope.previousResponseHadResult = (scope.response && scope.response.hasResult);
                        scope.response = null;

                        if (validity.valid) {
                            Logger.info('Chart is OK, executing');
                            scope.execute();
                        } else {
                            scope.resetResponseErrorsInScope();
                            Logger.info('Chart is NOK, not executing', scope.validity);
                        }
                    };

                    // fetch the response
                    scope.execute = Debounce()
                        .withDelay(1, 300)
                        .withScope(scope)
                        .withSpinner(true)
                        .wrap(function() {

                            Logger.info('Debounced, executing');
                            scope.executedOnce = true;

                            let request = null;

                            try {
                                const wrapper = element.find('.chart-zone');
                                const width = wrapper.width();
                                const height = wrapper.height();
                                scope.chartSpecific = {
                                    ...scope.chartSpecific,
                                    datasetProjectKey: scope.getDataSpec().datasetProjectKey,
                                    datasetName: scope.getDataSpec().datasetName,
                                    context: scope.getCurrentChartsContext()
                                };
                                request = ChartRequestComputer.compute(scope.chart.def, width, height, scope.chartSpecific);
                                request.useLiveProcessingIfAvailable = scope.chart.def.useLiveProcessingIfAvailable;
                                Logger.info('Request is', request);
                                scope.graphError.error = null;
                            } catch (error) {
                                Logger.info('Not executing, chart is not ready', error);
                                scope.graphError.error = error;
                            }
                            // We are sure that request is valid so we can generate the name
                            if (!scope.chart.def.userEditedName) {
                                const newName = ChartTypeChangeHandler.computeAutoName(scope.chart.def);
                                if (newName.length > 0) {
                                    scope.chart.def.name = newName;
                                }
                            }

                            scope.filter = { 'query': undefined };
                            resetErrorInScope(scope);

                            scope.excelExportableChart = undefined;
                            const chartDefCopy = angular.copy(scope.chart.def);

                            executePivotRequest(request, undefined, false).update(function(data) {
                                scope.request = request;
                                scope.response = data;

                            }).success(function(data) {
                                // For Excel export
                                scope.excelExportableChart = {
                                    pivotResponse: data.result.pivotResponse,
                                    chartDef: chartDefCopy
                                };

                                scope.request = request;
                                scope.response = data;
                                scope.resetResponseErrorsInScope();
                                scope.onResponse();

                            }).error(function(data, status, headers) {
                                onPivotRequestError(scope)(data, status, headers);
                            });
                        });

                    function getChangedDefinitionMessage(changedDefinition) {
                        const { name, nv, ov } = changedDefinition;
                        return `${name} \nbefore: ${JSON.stringify(ov)} \nafter: ${JSON.stringify(nv)}`;
                    }

                    function onChartImportantDefinitionChanged(changedDefinition, newChartDef) {
                        Logger.info(`Chart important change: ${getChangedDefinitionMessage(changedDefinition)}`);

                        if (newChartDef) {
                            const oldChartDef = angular.copy(scope.chart.def);
                            scope.recomputeAndUpdateData(changedDefinition);

                            const newImportantChangedDefinition = ChartDefinitionChangeHandler.getRecomputeChange(formatDefinition(scope.chart.def), formatDefinition(oldChartDef));
                            if (newImportantChangedDefinition) {
                                Logger.info('Data has been modified, not executing --> will execute at next cycle');
                                return;
                            }
                            Logger.info('Triggering executeIfValid');
                            scope.executeIfValid();
                        }
                    }

                    function onChartFrontImportantDefinitionChanged(changedDefinition, newChartDef) {
                        Logger.info(`Chart front important change: ${getChangedDefinitionMessage(changedDefinition)}`);
                        if (newChartDef) {
                            if (!scope.insight) {
                                scope.saveChart();
                            }
                            scope.redraw({ updateThumbnail: true });
                        }
                    }

                    function onChartFrontImportantNoRedrawDefinitionChanged(changedDefinition, newChartDef) {
                        Logger.info(`Chart front important no redraw change: ${getChangedDefinitionMessage(changedDefinition)}`);
                        if (newChartDef && !scope.insight) {
                            scope.saveChart();
                        }
                    }

                    function formatDefinition(chartDef) {
                        if (chartDef.type === CHART_TYPES.GEOMETRY_MAP) {
                            const flattenedChartDef = angular.copy(chartDef);
                            const listPropertiesToFlatten = ['geometry', 'uaColor'];
                            for (const property of listPropertiesToFlatten) {
                                flattenedChartDef[property] = [];
                                for (const geoLayer of chartDef.geoLayers) {
                                    flattenedChartDef[property].push(getFirstValue(geoLayer[property]));
                                }
                            }

                            flattenedChartDef['colorOptions'] = [];
                            for (const geoLayer of chartDef.geoLayers) {
                                flattenedChartDef['colorOptions'].push(geoLayer['colorOptions']);
                            }
                            return flattenedChartDef;
                        }
                        return chartDef;
                    }

                    function getFirstValue(array) {
                        if (array.length > 0) {
                            return array[0];
                        }
                        return {};
                    }

                    scope.$watch('reusableDimensions.length', () => {
                        scope.updateHierarchyMissingColumns();
                    });


                    //This watch needs to be set before the scope.chart.def watch
                    scope.$watch('chart.theme', (nv, ov) => {
                        Logger.info('Chart theme change');
                        scope.immutableTheme = nv ? angular.copy(nv) : undefined;
                        $timeout.cancel(themeChangeTimeoutId);
                        if (!scope.insight) {
                            themeChangeTimeoutId = $timeout(() => {
                                scope.saveChart();
                            }, 600);
                        }
                        /*
                         * Updating the theme palettes doesn't trigger a change detection cycle on
                         * chart.def as only the theme changes. The chart still needs to be redrawn.
                         */
                        if (nv && (!ov || !angular.equals(nv.themePalettes, ov.themePalettes))) {
                            scope.redraw({ updateThumbnail: true });
                        }

                        if (scope.chart && scope.chart.theme && scope.chart.theme.generalFormatting && scope.chart.theme.generalFormatting.fontFamily) {
                            element.css('--visualization-font-family', scope.chart.theme.generalFormatting.fontFamily);
                            element.css('--visualization-font-color', scope.chart.theme.generalFormatting.fontColor);
                        }
                    });

                    // thumbnailData changes are handled by a dedicated watcher
                    const ignoredChartDefFields = ['thumbnailData'];

                    scope.$watch(() => _.omit(scope.chart.def, ignoredChartDefFields), function(nv, ov) {
                        if (!nv) {
                            return;
                        }
                        if (!ov) {
                            onChartImportantDefinitionChanged({ name: 'initial', nv, ov }, nv);
                        }

                        if (nv.tooltipOptions && nv.tooltipOptions.display == false && ov.tooltipOptions.display == true) {
                            $rootScope.$emit('unfixTooltip');
                        }

                        $timeout.cancel(invalidChangeTimeoutId);

                        const invalidMessage = ChartDefinitionChangeHandler.getInvalidChangeMessage(nv);
                        if (invalidMessage !== null) {
                            // Timeouting to prevent displaying warning while user is still typing in a custom input.
                            invalidChangeTimeoutId = $timeout(() => {
                                ActivityIndicator.hide();
                                ActivityIndicator.warning('Not refreshing: ' + invalidMessage);
                            }, ChartsStaticData.SAVE_DEBOUNCE_DURATION);
                            return;
                        }

                        scope.immutableChartDef = angular.copy(nv);

                        const newValue = formatDefinition(nv);
                        const oldValue = formatDefinition(ov);

                        // Check for a change which triggers save + recompute + redraw
                        const importantChangedDefinition = ChartDefinitionChangeHandler.getRecomputeChange(newValue, oldValue);
                        if (importantChangedDefinition) {
                            $timeout.cancel(themeChangeTimeoutId);
                            $timeout.cancel(frontImportantChangeTimeoutId); // apply immediate change, prevail on delayed changes
                            onChartImportantDefinitionChanged(importantChangedDefinition, newValue);
                            return;
                        }
                        // Check for a change which triggers save + redraw
                        const frontChangedDefinition = ChartDefinitionChangeHandler.getFrontImportantChange(newValue, oldValue);
                        if (frontChangedDefinition) {
                            $timeout.cancel(themeChangeTimeoutId);
                            $timeout.cancel(frontImportantChangeTimeoutId); // apply immediate change, prevail on delayed changes
                            onChartFrontImportantDefinitionChanged(frontChangedDefinition, newValue);
                            return;
                        }
                        // Check for a change which triggers save + redraw after a timeout
                        const delayedChangedDefinition = ChartDefinitionChangeHandler.getDelayedFrontImportantChange(newValue, oldValue);
                        if (delayedChangedDefinition) {
                            $timeout.cancel(themeChangeTimeoutId);
                            // Timeouting to prevent excessive refresh while user is still typing in a custom input.
                            $timeout.cancel(frontImportantChangeTimeoutId);
                            frontImportantChangeTimeoutId = $timeout(() => {
                                onChartFrontImportantDefinitionChanged(delayedChangedDefinition, newValue);
                            }, ChartsStaticData.SAVE_DEBOUNCE_DURATION);
                            return;
                        }
                        // Check for a change which triggers save
                        const noRedrawChangedDefinition = ChartDefinitionChangeHandler.getNoRedrawChange(newValue, oldValue);
                        if (noRedrawChangedDefinition) {
                            $timeout.cancel(themeChangeTimeoutId);
                            $timeout.cancel(noRedrawDelayedChangeTimeoutId); // apply immediate change, prevail on delayed changes
                            onChartFrontImportantNoRedrawDefinitionChanged(noRedrawChangedDefinition, newValue);
                            return;
                        }
                        // Check for a change which triggers save after a timeout
                        const noRedrawDelayedChangedDefinition = ChartDefinitionChangeHandler.getDelayedNoRedrawChange(newValue, oldValue);
                        if (noRedrawDelayedChangedDefinition) {
                            $timeout.cancel(themeChangeTimeoutId);
                            // Timeouting to prevent excessive refresh while user is still typing in a custom input.
                            $timeout.cancel(noRedrawDelayedChangeTimeoutId);
                            noRedrawDelayedChangeTimeoutId = $timeout(() => {
                                onChartFrontImportantNoRedrawDefinitionChanged(noRedrawDelayedChangedDefinition, newValue);
                            }, ChartsStaticData.SAVE_DEBOUNCE_DURATION);
                            return;
                        }
                    }, true);

                    const resetImmutableChartDef = function(nv, ov) {
                        if (!angular.equals(nv, ov)) {
                            scope.immutableChartDef = angular.copy(scope.chart.def);
                        }
                    };

                    scope.$watch('chart.summary', () => {
                        scope.chartUsableColumns = angular.copy(scope.chart.summary ? scope.chart.summary.usableColumns : []);
                    });

                    scope.$watch('chart.def.$axisSpecs', resetImmutableChartDef);

                    scope.$watch('chart.def.xCustomExtent.$autoExtent', resetImmutableChartDef);

                    //  Checks only the first y axis formatting options for $autoExtent, if one is updated, others are.
                    scope.$watch('chart.def.yAxesFormatting[0].customExtent.$autoExtent', resetImmutableChartDef);

                    scope.$watch('chart.def.$zoomControlInstanceId', resetImmutableChartDef);

                    scope.$watch('chart.def.thumbnailData', function(nv, ov) {
                        if (nv !== ov && !scope.insight) {
                            scope.saveChart(true);
                        }
                    });

                    $(window).on('resize.chart_logic', function(e) {
                        if (e.detail && e.detail.skipInCharts) {
                            return;
                        }
                        handleResize(scope);
                    });
                    scope.$on('$destroy', function() {
                        $(window).off('resize.chart_logic');
                        resizeObserver.disconnect();
                    });
                    const chartParamBar = document.querySelector('.chart-param-bar');
                    scope.chartParamBarHeight = chartParamBar.offsetHeight;
                    scope.chartParamBarWidth = chartParamBar.offsetWidth;
                    resizeObserver.observe(chartParamBar);

                    scope.forceExecute = function(options) {
                        if (options !== undefined) {
                            const { store, id } = ChartStoreFactory.getOrCreate(scope.chart.def.$chartStoreId);
                            scope.chart.def.$chartStoreId = id;
                            store.setRequestOptions(options);
                        }
                        scope.recomputeAndUpdateData();
                        scope.executeIfValid();
                    };

                    scope.rebuildSampling = function() {
                        const refreshTrigger = new Date().getTime();
                        if (scope.chart.copySelectionFromScript) {
                            scope.shaker.explorationSampling._refreshTrigger = refreshTrigger;
                        } else {
                            scope.chart.refreshableSelection._refreshTrigger = refreshTrigger;
                        }
                        scope.$emit('chartSamplingChanged', { chart: scope.chart });
                    };

                    scope.$on('forceExecuteChart', function() {
                        scope.forceExecute();
                    });

                    scope.$emit('listeningToForceExecuteChart'); // inform datasetChartBase directive that forceExecute() can be triggered through broadcast

                    scope.redraw = function(options) {
                        scope.$broadcast('redraw', options);
                    };

                    scope.revertToLinoEngineAndReload = function() {
                        scope.chart.engineType = 'LINO';
                        scope.forceExecute();
                    };

                    scope.craftBinningFormChart = function(params) {
                        $rootScope.globallyOpenContextualMenu = undefined; // close currently opened one
                        scope.$emit('craftBinningFormChart', { dimension: angular.copy(params.dimension), dimensionRef: params.dimension, isEditMode: params.isEditMode, hideOneTickPerBin: params.hideOneTickPerBin, customBinningOnly: params.customBinningOnly });
                    };

                    scope.createReusableDimensionFromChart = function(params) {
                        $rootScope.globallyOpenContextualMenu = undefined; // close currently opened one
                        scope.$emit('createReusableDimensionFromChart', { ...params, dimension: angular.copy(params.dimension), dimensionRef: params.dimension });
                    };

                    /*
                     * ------------------------------------------------------
                     * Recompute/Update handlers
                     */
                    scope.recomputeAndUpdateData = function(changedDefinition) {
                        const { store, id } = ChartStoreFactory.getOrCreate(scope.chart.def.$chartStoreId);
                        scope.chart.def.$chartStoreId = id;

                        ChartTypeChangeHandler.fixupSpec(scope.chart, scope.chart.theme, changedDefinition);

                        scope.canHasTooltipMeasures = [
                            CHART_TYPES.MULTI_COLUMNS_LINES,
                            CHART_TYPES.GROUPED_COLUMNS,
                            CHART_TYPES.STACKED_COLUMNS,
                            CHART_TYPES.STACKED_BARS,
                            CHART_TYPES.GRID_MAP,
                            CHART_TYPES.LINES,
                            CHART_TYPES.STACKED_AREA,
                            CHART_TYPES.ADMINISTRATIVE_MAP,
                            CHART_TYPES.PIE,
                            CHART_TYPES.BINNED_XY
                        ].includes(scope.chart.def.type);

                        scope.canAnimate = ChartFeatures.canAnimate(scope.chart.def.type, scope.chart.def.variant);
                        scope.canFacet = ChartFeatures.canFacet(scope.chart.def.type, scope.chart.def.webAppType, scope.chart.def.variant);
                        scope.canFilter = ChartFeatures.canFilter(scope.chart.def.type, scope.chart.def.webAppType);
                        scope.updateHierarchyInfo();

                        return;
                    };

                    scope.setChartType = function(chartId) {
                        let newChartType;
                        if (scope.availableChartTypes || scope.webapps) {
                            newChartType = [...(scope.availableChartTypes || []), ...(scope.webapps || [])].find(item => item.id === chartId);
                        }
                        if (!newChartType) {
                            return;
                        }
                        WT1.event('chart-type-change', {
                            chartId: `${$stateParams.projectKey.dkuHashCode()}.${getWT1ContextName().dkuHashCode()}.${scope.chart.def.name.dkuHashCode()}`,
                            chartType: newChartType.type,
                            chartVariant: newChartType.variant
                        });

                        const oldChartType = scope.chart.def.type;
                        element.find('.pivot-charts .mainzone').remove(); // avoid flickering
                        Logger.info('Set chart type');
                        ChartTypeChangeHandler.onChartTypeChange(scope.chart.def, newChartType.type, newChartType.variant, newChartType.webappType);
                        Logger.info('AFTER chart type', scope.chart.def);
                        scope.chart.def.type = newChartType.type;
                        scope.chart.def.variant = newChartType.variant;
                        scope.chart.def.webAppType = newChartType.webappType;
                        if (!scope.validity.valid) {
                            // If chart is not ready and we changed, we can apply things related to new chart type
                            DSSVisualizationThemeUtils.applyToChart({ chart: scope.chart.def, theme: scope.chart.theme });
                        }

                        if (scope.chart.def.$zoomControlInstanceId) {
                            ChartZoomControlAdapter.clear(scope.chart.def.$zoomControlInstanceId);
                            scope.chart.def.$zoomControlInstanceId = null;
                        }

                        onChartImportantDefinitionChanged({ name: 'type', nv: newChartType.type, ov: oldChartType }, scope.chart.def);
                        if (!scope.$$phase) {
                            scope.$apply(); // sc-115878
                        }
                    };
                });

                scope.exportToImage = function() {
                    scope.$broadcast('export-chart');
                    WT1.event('chart-download', {
                        chartId: `${$stateParams.projectKey.dkuHashCode()}.${getWT1ContextName().dkuHashCode()}.${scope.chart.def.name.dkuHashCode()}`,
                        format: 'image',
                        chartType: scope.chart.def.type,
                        chartVariant: scope.chart.def.variant
                    });
                    if (scope.displayChartOptionsMenu) {
                        scope.toggleChartOptionsMenu();
                    }
                };

                scope.exportToExcel = function() {
                    if (scope.excelExportableChart) {
                        let animationFrameIdx;
                        const chartDef = scope.chart.def;
                        const colorMapsArray = scope.getColorMap();
                        const pivotResponse = scope.excelExportableChart.pivotResponse;

                        if (chartDef.animationDimension.length) {
                            animationFrameIdx = scope.animation.currentFrame;
                        }

                        DataikuAPI.shakers.charts.exportToExcel(
                            chartDef,
                            pivotResponse,
                            animationFrameIdx,
                            colorMapsArray
                        ).success(data => {
                            WT1.event('chart-download', {
                                chartId: `${$stateParams.projectKey.dkuHashCode()}.${getWT1ContextName().dkuHashCode()}.${scope.chart.def.name.dkuHashCode()}`,
                                format: 'excel',
                                chartType: chartDef.type,
                                chartVariant: chartDef.variant
                            });
                            downloadURL(DataikuAPI.shakers.charts.downloadExcelUrl(data.id));
                        }).error(setErrorInScope.bind(scope));
                        if (scope.displayChartOptionsMenu) {
                            scope.switchchartOptionsMenu();
                        }
                    }
                };

                scope.getColorMap = function() {
                    const colorMapsArray = [];

                    if (scope.chart.def.colorMode === 'COLOR_GROUPS') {
                        scope.getColorFromConditionalFormatting(colorMapsArray);
                    } else {
                        scope.getColorFromColorScale(colorMapsArray);
                    }
                    return colorMapsArray;
                },

                scope.getColorFromColorScale = function(colorMapArray) {
                    const pivotResponse = scope.excelExportableChart.pivotResponse;
                    if (scope.chart.def.colorMeasure.length && pivotResponse.aggregations.length > 1) {
                        const colorScale = scope.legendsWrapper.getLegend(0).scale;
                        const backgroundColorMap = {};
                        const fontColorMap = {};
                        pivotResponse.aggregations[pivotResponse.aggregations.length - 1].tensor.forEach((value, index) => {
                            const color = colorScale(value, index);

                            if (!(value in backgroundColorMap) && color) {
                                backgroundColorMap[value] = ColorUtils.toHex(color);
                                fontColorMap[value] = ColorUtils.getFontContrastColor(color, scope.chart.theme && scope.chart.theme.generalFormatting.fontColor);
                            }
                        });
                        colorMapArray.push({ backgroundColorMap, fontColorMap });
                    }
                },

                scope.getColorFromConditionalFormatting = function(colorMapsArray) {
                    scope.chart.def.genericMeasures.forEach((measure, measureIndex) => {
                        colorMapsArray.push(scope.retrieveColorMaps(scope.chart.def.colorGroups, measure, measureIndex));
                    });
                    return colorMapsArray;
                },

                scope.retrieveColorMaps = function(colorGroups, measure, measureIndex) {
                    const backgroundColorMap = {};
                    const fontColorMap = {};
                    const pivotResponse = scope.excelExportableChart.pivotResponse;
                    const measureId = ConditionalFormattingOptions.getMeasureId(measure);
                    const measureGroup = colorGroups.find(group => group.appliedColumns && group.appliedColumns.map(column => ConditionalFormattingOptions.getMeasureId(column)).includes(measureId));

                    pivotResponse.aggregations[measureIndex].tensor.forEach((value, valueIndex) => {
                        if (!_.isNil(measureGroup)) {
                            value = scope.chartData.getNonNullCount(valueIndex, measureIndex) > 0 ? value : '';

                            const ruleClass = ConditionalFormattingOptions.getColorRuleClass(value, measureGroup.rules, measure, scope.chart.theme);
                            const colorFormattingClass = ruleClass.class;

                            let fontColor = '#000000';
                            let backgroundColor = '#FFFFFF';
                            if (colorFormattingClass.includes('text')) {
                                fontColor = ConditionalFormattingOptions.getStripeColor(colorFormattingClass, {});
                            } else if (colorFormattingClass.includes('background')) {
                                fontColor = '#FFFFFF'; // fontcolor always white with background color.
                                backgroundColor = ConditionalFormattingOptions.getStripeColor(colorFormattingClass, {});
                            } else if (colorFormattingClass.includes('custom')) {
                                fontColor = ruleClass.customColors.customFontColor || fontColor;
                                backgroundColor = ruleClass.customColors.customBackgroundColor || backgroundColor;
                            }
                            scope.retrieveColor(fontColorMap, fontColor, value);
                            scope.retrieveColor(backgroundColorMap, backgroundColor, value);
                        }
                    });
                    return { backgroundColorMap, fontColorMap };
                };

                scope.retrieveColor = function(colorMap, color, measureValue) {
                    if (!(measureValue in colorMap && color)) {
                        colorMap[measureValue] = color;
                    }
                },

                scope.openPasteModalFromKeydown = function(data) {
                    !scope.readOnly && ChartConfigurationCopyPaste.pasteChartFromClipboard({ valid: scope.validity.valid, isInInsight: scope.isInInsight }, data).then(res => scope.pasteChart(res, scope.currentChart.index));
                };

                scope.keydownCopy = function() {
                    // copy chart only if the selection is empty
                    if (window.getSelection().toString() === '') {
                        ChartConfigurationCopyPaste.copyChartToClipboard(scope.chart.def, scope.appVersion, scope.chart.theme);
                    }
                };

                scope.deleteCurrentChart = function() {
                    scope.deleteChart(scope.currentChart.index);
                };

                scope.duplicateChart = function() {
                    scope.addChart({
                        chart: {
                            ...scope.chart,
                            def: {
                                ...scope.chart.def,
                                id: window.crypto.getRandomValues(new Uint32Array(1))[0].toString(16)
                            }
                        },
                        index: scope.currentChart.index,
                        copyOfName: true,
                        wt1Event: 'chart-duplicate'
                    });
                };

                scope.displayChartOptionsMenu = false;
                scope.toggleChartOptionsMenu = function() {
                    scope.displayChartOptionsMenu = !scope.displayChartOptionsMenu;
                    if (scope.displayChartOptionsMenu) {
                        $timeout(function() {
                            $(window).on('click', scope.switchChartOptionsMenuOnClick);
                        });
                    } else {
                        $(window).off('click', scope.switchChartOptionsMenuOnClick);
                    }
                };

                scope.switchChartOptionsMenuOnClick = function(e) {
                    const clickedEl = e.target;
                    if ($(clickedEl).closest('.chart-options-wrapper').length <= 0 && scope.displayChartOptionsMenu) {
                        scope.toggleChartOptionsMenu();
                        scope.$apply();
                    }
                };

                scope.blurElement = function(inputId) {
                    $timeout(function() {
                        $(inputId).blur();
                    });
                };

                scope.blurTitleEdition = function() {
                    scope.editingTitle = false;
                    scope.chart.def.userEditedName = true;
                    $timeout(scope.saveChart);
                    if (scope.excelExportableChart) {
                        scope.excelExportableChart.chartDef.name = scope.chart.def.name;
                    }
                };

                scope.editChartTitle = function() {
                    scope.editingTitle = !(scope.isInInsight && $rootScope.topNav.tab === 'view');
                };

                scope.openChartSamplingTab = function() {
                    $rootScope.$broadcast('tabSelect', 'sampling-engine');
                };

                function getWT1ContextName() {
                    const context = scope.analysisCoreParams || scope.insight || scope.dataset || { name: 'unknown' };
                    return context.name;
                }

                function shouldDisplaySamplingStatus() {
                    const pivotResponse = scope.response && scope.response.result && scope.response.result.pivotResponse;
                    const sampleMetadata = pivotResponse && pivotResponse.sampleMetadata;
                    return (sampleMetadata && !sampleMetadata.sampleIsWholeDataset);
                }

                scope.shouldDisplayPointRangeWarning = () => {
                    return scope.chart.def.type === CHART_TYPES.SCATTER && scope.uiDisplayState?.lowPointSizeRangeWarning;
                };

                scope.hasBetaFormattingPane = function(chartType) {
                    const charts = [
                        CHART_TYPES.STACKED_COLUMNS,
                        CHART_TYPES.GROUPED_COLUMNS,
                        CHART_TYPES.STACKED_BARS,
                        CHART_TYPES.STACKED_AREA,
                        CHART_TYPES.PIVOT_TABLE,
                        CHART_TYPES.SCATTER,
                        CHART_TYPES.SCATTER_MULTIPLE_PAIRS,
                        CHART_TYPES.LINES,
                        CHART_TYPES.TREEMAP,
                        CHART_TYPES.GEOMETRY_MAP,
                        CHART_TYPES.SCATTER_MAP,
                        CHART_TYPES.ADMINISTRATIVE_MAP,
                        CHART_TYPES.GRID_MAP,
                        CHART_TYPES.DENSITY_HEAT_MAP,
                        CHART_TYPES.MULTI_COLUMNS_LINES,
                        CHART_TYPES.PIE,
                        CHART_TYPES.SANKEY,
                        CHART_TYPES.GROUPED_XY,
                        CHART_TYPES.BINNED_XY,
                        CHART_TYPES.LIFT,
                        CHART_TYPES.BOXPLOTS,
                        CHART_TYPES.DENSITY_2D,
                        CHART_TYPES.RADAR,
                        CHART_TYPES.KPI,
                        CHART_TYPES.GAUGE
                    ];
                    return charts.includes(chartType);
                };

                scope.assignToChartDef = function(propertyPath, value, merge) {
                    let assignedValue = value;

                    //  If we use the merge strategy, we assign properties from the incoming value to the existing object, otherwise, we just replace it
                    if (merge) {
                        assignedValue = _.assign(_.get(scope.chart.def, propertyPath), value);
                    }

                    _.set(scope.chart.def, propertyPath, assignedValue);
                    $timeout(() => scope.$apply());
                };

                scope.onChartDefPropertyChange = function(event) {
                    if (event.id) {
                        //  If we provide an id, it targets yAxesFormatting
                        scope.updateYAxisFormatting(event.key, event.value, event.id);
                    } else {
                        scope.assignToChartDef(event.key, event.value, event.merge);
                    }
                };

                scope.onFormattingPaneTabChange = function() {
                    $rootScope.$broadcast('reflow');
                };

                scope.onFetchColumnsSummaryForPalette = function() {
                    if ($rootScope.fetchColumnsSummaryForCurrentChart) {
                        $rootScope.fetchColumnsSummaryForCurrentChart(true).then($rootScope.redraw);
                    } else {
                        $rootScope.fetchColumnsSummary().then($rootScope.redraw);
                    }
                };

                scope.updateYAxisFormatting = (key, value, id) => {
                    const yAxesFormatting = _.clone(scope.chart.def.yAxesFormatting);
                    const axisFormatting = ChartAxesUtils.getFormattingForYAxis(yAxesFormatting, id);
                    _.set(axisFormatting, key, value);
                    scope.assignToChartDef('yAxesFormatting', yAxesFormatting);
                };

                scope.headerDrillUp = (dimension) => {
                    ChartDrilldown.handleDrillUp(scope.chart.def, dimension, DRILL_UP_SOURCE.CHART_HEADER);
                };

                /**
                 * Sets values in scope for:
                 * hierarchiesInChart - hierarchy dimensions used in chart
                 * hierarchyDimsInChart - current dimensions of the hierarchies used in chart
                 * hierarchiesMissingColumns - columns missing per hierarchy (and saves them in chartStore)
                 * breadcrumbs - breadcrumbs for hierarchies used in chart
                 * breadcrumbTooltips - text version of breadcrumbs displayed as tooltip for the dimensions in the chart header
                 */
                scope.updateHierarchyInfo = () => {
                    scope.hierarchiesInChart = ChartHierarchyDimension.getHierarchyDimensionsForChartType(scope.chart.def) || [];
                    scope.updateHierarchyMissingColumns();
                    scope.hierarchyDimsInChart = ChartHierarchyDimension.getHierarchyDimensionsForChartType(scope.chart.def).map(hierarchy => hierarchy.dimensions[hierarchy.level]);
                    scope.breadcrumbs = ChartDrilldown.getBreadcrumbs(scope.chart.def);

                    scope.breadcrumbTooltips = {};
                    scope.hierarchyDimsInChart.forEach(dimension => {
                        scope.breadcrumbTooltips[dimension.hierarchyId] = ChartDrilldown.getBreadcrumbTooltipForHierarchyDim(scope.breadcrumbs, dimension);
                    });
                };

                scope.updateHierarchyMissingColumns = () => {
                    const { store, id } = ChartStoreFactory.getOrCreate(scope.chart.def.$chartStoreId);
                    scope.chart.def.$chartStoreId = id;
                    scope.hierarchiesMissingColumns = ChartHierarchies.getHierarchiesMissingColumns(scope.hierarchiesInChart, scope.usableColumns, scope.reusableDimensions);
                    store.setHierarchyMissingColumns(scope.hierarchiesMissingColumns);
                };

                scope.canDimensionDrillUp = (dimension) => {
                    return ChartHierarchyDimension.canDimensionDrillUp(scope.chart.def, dimension);
                };

                scope.resetResponseErrorsInScope = () => {
                    scope.allFilteredOut = false;
                    scope.display0Warning = false;
                    scope.hasUnusableHierarchy = false;
                };
            }
        };

        function onPivotRequestError(scope) {
            return (data, status, headers) => {
                scope.response = undefined;
                scope.resetResponseErrorsInScope();

                if (data.code === 'FILTERED_OUT') {
                    scope.allFilteredOut = true;
                } else if (data.code === 'CANNOT_DISPLAY_WITH_EMPTY_AXES') {
                    scope.display0Warning = true;
                } else if ((data.code === 'MISSING_COLUMN' || data.errorType === 'com.dataiku.dip.exceptions.PivotMissingColumnException')
                    && !_.isEmpty(scope.hierarchiesMissingColumns)) {
                    const canDisplayChart = scope.hierarchiesInChart.every(hierarchy => ChartHierarchyDimension.hasHierarchyUsableColumns(scope.chart.def.$chartStoreId, hierarchy));
                    if (canDisplayChart) {
                        scope.hierarchiesInChart.forEach((hierarchy) => ChartHierarchyDimension.resetHierarchyToUsableDimension(scope.chart.def.$chartStoreId, hierarchy));
                    } else {
                        scope.hasUnusableHierarchy = true;
                    }
                } else {
                    if (ChartTypeChangeHandler.hasRequestResponseWarning(scope.chart.def, data)) {
                        scope.setValidity(ChartTypeChangeHandler.getRequestResponseWarning(scope.chart.def, data));
                    } else if (ChartDataUtils.isPivotRequestAborted(data)) {
                        // Manually aborted => do not report as error
                    } else {
                        scope.chartSetErrorInScope(data, status, headers);
                    }
                }
            };
        }

        function parseWebapps(data) {
            const webapps = [];
            data.forEach(w => {
                webapps.push({
                    id: w.desc.id,
                    displayName: w.desc.meta.label || w.desc.id,
                    type: CHART_TYPES.WEBAPP,
                    variant: CHART_VARIANTS.normal,
                    webappType: w.webappType,
                    isWebapp: true
                });
            });
            if (webapps) {
                webapps.sort((a, b) => a.webappType.localeCompare(b.webappType));
            }
            return webapps;
        }

        function handleResize(scope) {
            const debouncedRedrawAfterResize = Debounce()
                .withDelay(1, 300)
                .withScope(scope)
                .withSpinner(false)
                .wrap(f => {
                    Logger.debug('Redrawing chart after resize (debounced)');
                    scope.redraw();
                });

            Logger.debug('Window was resized, will redraw chart a bit later');
            if (scope.chart.def.type == CHART_TYPES.BINNED_XY && scope.chart.def.variant == CHART_VARIANTS.binnedXYHexagon) {
                scope.recomputeAndUpdateData();
                scope.executeIfValid();
            } else {
                debouncedRedrawAfterResize();
            }
            scope.$apply();
        }
    });

})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.charts');

    // (!) This directive previously was in static/dataiku/js/simple_report/config_ui.js
    app.directive('contextualMenu', function($rootScope, $window, $compile, $timeout, CHART_FILTERS, ChartStoreFactory, ChartFeatures, ChartAxesUtils, ChartsStaticData) {
        let isCurrentlyEdited = false;

        function getContextualMenuClassName(axis) {
            return `${axis}-axis-contextual-menu qa_charts_${axis}-axis-contextual-menu`;
        }

        function targetIsSelectInput(e) {
            return e.target.closest('.bootstrap-select') !== null;
        }

        function targetShouldNotCloseInput(e) {
            return e.target.closest('.no-global-contextual-menu-close') !== null;
        }

        $($window).on('click', function(e) {
            if (targetIsSelectInput(e) || targetShouldNotCloseInput(e) || isCurrentlyEdited) {
                e.stopPropagation();
                isCurrentlyEdited = false;
            } else if (!e.isDefaultPrevented() && !e.target.hasAttribute('no-global-contextual-menu-close')) {
                $rootScope.globallyOpenContextualMenu = undefined;
                $rootScope.$apply();
            }
        });
        return {
            scope: true,
            compile: function(element, attrs) {
                const popoverTemplate = element.find('.contextualMenu').detach();
                return function($scope, element, attrs) {
                    let popover = null;
                    let popoverScope = null;
                    $scope.contextualMenu = false;
                    $scope.ChartFeatures = ChartFeatures;
                    $scope.getXAxisTitle = ChartAxesUtils.getXAxisTitle;
                    $scope.getYAxisTitle = ChartAxesUtils.getYAxisTitle;
                    $scope.COLUMN_TYPES = CHART_FILTERS.COLUMN_TYPES;

                    $scope.getXAxisHint = () => {
                        return ChartFeatures.hasMultipleXDim($scope.chart.def.type) ? '(settings will be common to all X dimensions)' : '';
                    };

                    function hide() {
                        if (popover) {
                            if (popoverScope) {
                                popoverScope.$destroy();
                                popoverScope = null;
                            }
                            popover.hide().remove();
                            popover = null;
                        }
                    }
                    function show() {
                        if (popover === null) {
                            /*
                             * Since Angular 1.6, in a <select>, ng-model is set to null when the corresponding <option> is removed.
                             *
                             * Here is what happens when a contextualMenu containing a select is removed (using hide()):
                             * - The selected <option> is removed from DOM (like the others) triggering its $destroy callback.
                             * - This callback removes the value from the optionsMap and set a digest's call back (ie: $$postDigest function).
                             * - $$postDigest is triggered after angular's digest and checks if its scope (popoverScope in our case) is destroyed.
                             * - If yes it does nothing (return)
                             * - If not, $$postDigest set the select's ngModel to null, because its current value is no longer in optionsMap
                             *
                             * popoverScope prevents any nested <select>'s ngModel to get set to null when a contextualMenu is closed.
                             * This fix work because we destroy popoverScope (where the select lives), before deleting the DOM containing it (along with the <option> elements).
                             * So when $$postDigest positively checks if its scope is $destroyed, it just returns without setting the select's ngModel to null.
                             */
                            popoverScope = $scope.$new();
                            /*
                             * We may need the original scope in some context, e.g. modals opened from a contextualMenu
                             * because clicking on the modal will close the menu and destroyed its scope
                             */
                            popoverScope.$contextScope = $scope;
                            popover = $compile(popoverTemplate.get(0).cloneNode(true))(popoverScope);
                        }
                        popover.appendTo('body');

                        const position = attrs.cepPosition || 'align-left-bottom';
                        const mainZone = element;
                        const mzOff = element.offset();

                        /* Fairly ugly ... */
                        if (element.parent().parent().hasClass('chartdef-dropped')) {
                            mzOff.top -= 4;
                            mzOff.left -= 10;
                        }

                        switch (position) {
                            case 'align-left-bottom':
                                popover.css({ left: mzOff.left, top: mzOff.top + mainZone.height() });
                                break;
                            case 'align-right-bottom':
                                popover.css({
                                    top: mzOff.top + mainZone.height(),
                                    left: mzOff.left + mainZone.innerWidth() - popover.innerWidth()
                                });
                                break;
                            case 'align-right-top':
                                popover.css({
                                    top: mzOff.top,
                                    left: mzOff.left + mainZone.innerWidth()
                                });
                                break;
                            case 'smart':
                                var offset = { left: 'auto', right: 'auto', top: 'auto', bottom: 'auto' };
                                if (mzOff.left * 2 < window.innerWidth) {
                                    offset.left = mzOff.left;
                                } else {
                                    offset.right = window.innerWidth - mzOff.left - mainZone.innerWidth();
                                }
                                if (mzOff.top * 2 < window.innerHeight) {
                                    offset.top = mzOff.top + mainZone.height();
                                } else {
                                    offset.bottom = window.innerHeight - mzOff.top;
                                }
                                popover.css(offset);
                                break;
                            case 'smart-left-bottom':
                                $timeout(function() {
                                    // Left-bottom position, except if the menu would overflow the window, then left-top
                                    const offset = { left: mzOff.left, right: 'auto', top: 'auto', bottom: 'auto' };

                                    if (mzOff.top + mainZone.height() + popover.outerHeight() > window.innerHeight) {
                                        offset.bottom = window.innerHeight - mzOff.top;
                                    } else {
                                        offset.top = mzOff.top + mainZone.height();
                                    }
                                    popover.css(offset);
                                });
                                break;
                        }
                        if (attrs.cepWidth === 'fit-main') {
                            popover.css('width', mainZone.innerWidth());
                        }
                        popover.show();

                        popover.on('click', function(e) {
                            isCurrentlyEdited = false;
                            if (!targetIsSelectInput(e)) {
                                e.stopPropagation();
                            }
                        });

                        popover.on('mouseleave', function(e) {
                            // eslint-disable-next-line no-undef
                            if (isLeftClickPressed(e)) {
                                isCurrentlyEdited = true;
                            }
                        });
                    }

                    $scope.$watch('contextualMenu', function(nv, ov) {
                        if (nv) {
                            show();
                        } else {
                            hide();
                        }
                    });

                    $scope.toggleContextualMenu = function(e) {
                        if ($scope.globallyOpenContextualMenu && $scope.globallyOpenContextualMenu[0] === element[0]) {
                            $rootScope.globallyOpenContextualMenu = undefined;
                        } else {
                            $rootScope.globallyOpenContextualMenu = element;
                        }
                        e.preventDefault();
                    };

                    $scope.getMeasureClassName = function(chartType) {
                        const measureAxis = chartType === 'stacked_bars' ? 'x' : 'y';
                        return getContextualMenuClassName(measureAxis);
                    };

                    $scope.getDimensionClassName = function(chartType) {
                        const dimensionAxis = chartType === 'stacked_bars' ? 'y' : 'x';
                        return getContextualMenuClassName(dimensionAxis);
                    };

                    $scope.$on('$destroy', function() {
                        hide();
                    });

                    const globallyOpenContextualMenuWatcherUnsubscribe = $rootScope.$watch('globallyOpenContextualMenu', function(nv, ov) {
                        $scope.contextualMenu = ($rootScope.globallyOpenContextualMenu && $rootScope.globallyOpenContextualMenu[0] === element[0]);
                    });
                    $scope.$on('$destroy', globallyOpenContextualMenuWatcherUnsubscribe);

                    $scope.getYAxisFormatting = function(formatting, id) {
                        return ChartAxesUtils.getFormattingForYAxis(formatting, id);
                    };

                    if ($scope.chart) {
                        const { id, store } = ChartStoreFactory.getOrCreate($scope.chart.def.$chartStoreId);
                        $scope.chart.def.$chartStoreId = id;
                        $scope.$watch(() => store.get('axisSpecs'), (axisSpecs, oldAxisSpecs) => {
                            if (axisSpecs != null && (!angular.equals(axisSpecs, oldAxisSpecs) || _.isNil($scope.xAxisSpec) || _.isNil($scope.yAxisSpec))) {
                                const isXAxisPercentScale = !!(axisSpecs.xSpec && axisSpecs.xSpec.isPercentScale);
                                const isYAxisPercentScale = !!(axisSpecs.ySpec && axisSpecs.ySpec.isPercentScale);
                                $scope.xAxisComputeMode = isXAxisPercentScale ? 'PERCENTAGE' : 'NORMAL';
                                $scope.yAxisComputeMode = isYAxisPercentScale ? 'PERCENTAGE' : 'NORMAL';
                                $scope.ySpecId = attrs.ySpecId;
                                $scope.xAxisSpec = store.getAxisSpec('x');
                                $scope.yAxisSpec = store.getAxisSpec($scope.ySpecId || ChartsStaticData.LEFT_AXIS_ID);
                                if ($scope.yAxisSpec) {
                                    $scope.yAxisFormatting = $scope.getYAxisFormatting($scope.chart.def.yAxesFormatting, $scope.yAxisSpec.id);
                                }
                            }
                        });
                    }
                };
            }
        };
    });
})();

;
(function(){
    'use strict';

    const app = angular.module('dataiku.charts');

    /**
     * (!) This directive previously was in static/dataiku/js/simple_report/chart_dragdrop.js
     */
    app.directive('chartDragCopySource', () => {
        return {
            controller: 'ChartDragDropController',
            link : function($scope, element, attrs) {

                const el = element[0];

                $scope.$watch(attrs.chartDragDisable, function(nv) {
                    element[0].draggable = nv !== true;
                });

                el.addEventListener('dragstart', function(e) {
                    $scope.$apply(function() {
                        $scope.activeDragDrop.active = true;
                        $scope.setDragActive();
                        $scope.activeDragDrop.data = $scope.$eval(attrs.chartDragCopySource);
                    });
                    e.dataTransfer.effectAllowed = 'copy';
                    e.dataTransfer.setData(window.dkuDragType, JSON.stringify($scope.activeDragDrop.data));
                    // FIXME highlight droppable
                    this.classList.add('dragging');
                    return false;
                }, false);

                el.addEventListener('dragend', function(e) {
                    this.classList.remove('dragging');
                    return false;
                }, false);
            }
        };
    });
})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.charts');

    /**
     * (!) This directive previously was in static/dataiku/js/simple_report/chart_dragdrop.js
     * ---- Additonal API ----
     * Ability to specify which element inside the chart-drag-drop-list-item directive should be draggable
     * - Add the additional `specify-drag-trigger` attribute with the directive
     * - Set the `trigger-drag` attribute to all elements that should trigger a dragstart
     * As a result all nodes inside your `specify-drag-trigger` element will be dragged
     * But only those with a `trigger-drag` attribute will trigger a dragstart
     */
    app.directive('chartDragDropListItem', () => {
        let shouldBeDragged = true;
        return {
            controller: 'ChartDragDropController',
            link: function($scope, element, attrs) {

                $(element).attr('draggable', 'true');

                // if specifics trigger-drags were declared, check if the currently pressed element can trigger a dragstart
                const specifyDragTrigger = attrs.specifyDragTrigger !== undefined;
                if (specifyDragTrigger) {
                    element[0].addEventListener('mousedown', function(e) {
                        let node = e.target;
                        while (node) {
                            // found the trigger-drag attr. in one of the parent, we can drag
                            if (node.hasAttribute('trigger-drag')) {
                                shouldBeDragged = true;
                                break;
                            }
                            // reached the final node w/o finding the 'trigger-drag' attr. we can't drag
                            if (node.hasAttribute('specify-drag-trigger')) {
                                shouldBeDragged = false;
                                break;
                            }
                            node = node.parentElement;
                        }
                    });
                }

                element[0].addEventListener('dragstart', function(e) {
                    // if currently pressed element shouldn't trigger a drag we stop
                    if (specifyDragTrigger && !shouldBeDragged) {
                        e.preventDefault();
                        return;
                    }

                    const draggedElement = $(e.target);

                    $scope.$apply(function() {
                        $scope.activeDragDrop.active = true;
                        $scope.setDragActive();
                        $scope.activeDragDrop.moveFromList = $scope.$eval(attrs.chartDragDropListItem);
                        $scope.activeDragDrop.moveFromListIndex = draggedElement.index();
                        $scope.activeDragDrop.data = $scope.activeDragDrop.moveFromList[$scope.activeDragDrop.moveFromListIndex];
                        $scope.activeDragDrop.draggedElementToHide = draggedElement;
                    });

                    e.dataTransfer.effectAllowed = 'move';
                    e.dataTransfer.setData(window.dkuDragType, JSON.stringify($scope.activeDragDrop.data));

                    this.classList.add('dragging');
                    return false;
                }, false);

                element[0].addEventListener('dragend', function(e) {
                    this.classList.remove('dragging');
                    return false;
                }, false);
            }
        };
    });
})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.charts');

    /**
     * Mono-valued list drop zone. No placeholder since drop replaces.
     * (!) This directive previously was in static/dataiku/js/simple_report/chart_dragdrop.js
     */
    app.directive('chartDragDropListReplace', function($parse, Assert) {
        return {
            scope: true,
            controller: 'ChartDragDropController',
            link: function($scope, element, attrs) {

                let acceptFunc = function(data) {
                    return {
                        accept: true,
                        message: 'Drop here'
                    };
                };

                if (attrs.acceptDrop) {
                    const parsed = $parse(attrs.acceptDrop);
                    acceptFunc = function(data) {
                        return parsed($scope.$parent || $scope, { 'data': data });
                    };
                }
                const parsed = attrs.getChartDragDropListReplace && $parse(attrs.getChartDragDropListReplace);
                $scope.getChartDragDropListReplace = function(data) {
                    const parsedFn = parsed && parsed($scope.$parent || $scope);
                    if (!parsedFn) {
                        return $scope.$eval(attrs.chartDragDropListReplace);
                    }
                    return parsedFn(data);
                };

                const onDragOverOrEnter = function(e) {
                    this.classList.add('over');
                    $(this).parent().parent().addClass('over');

                    if ($scope.activeDragDrop.draggedElementToHide) {
                        $scope.activeDragDrop.draggedElementToHide.hide();
                    }

                    // Do we accept this payload ?
                    const accepted = acceptFunc($scope.activeDragDrop.data);
                    if (accepted.accept) {
                        e.dataTransfer.dropEffect = 'copyMove';
                        e.preventDefault();
                    } else {
                        $scope.$apply(function() {
                            $scope.validity.tempError = {};
                            $scope.validity.tempError.type = 'MEASURE_REJECTED';
                            $scope.validity.tempError.message = accepted.message;
                        });
                    }
                };

                element[0].addEventListener('dragover', onDragOverOrEnter, false);
                element[0].addEventListener('dragenter', onDragOverOrEnter, false);

                element[0].addEventListener('dragleave', function(e) {
                    this.classList.remove('over');
                    $(this).parent().parent().removeClass('over');
                    $scope.$apply(function() {
                        delete $scope.validity.tempError;
                    });
                    return false;
                }, false);

                /*
                 * This is triggered as soon as a drag becomes active on the page
                 * and highlights the drop zone if it's accepted
                 */
                $scope.$watch('activeDragDrop.active', function(nv, ov) {
                    if (nv) {
                        const accepted = acceptFunc($scope.activeDragDrop.data);

                        if (accepted.accept) {
                            $scope.addClassHereAndThere(element, 'drop-accepted');
                        } else {
                            $scope.addClassHereAndThere(element, 'drop-rejected');
                        }
                    } else {
                        $scope.removeClassHereAndThere(element, 'drop-accepted');
                        $scope.removeClassHereAndThere(element, 'drop-rejected');
                    }
                }, true);

                element[0].addEventListener('drop', function(e) {
                    Assert.trueish($scope.activeDragDrop.active, 'no active drag and drop');

                    // Stops some browsers from redirecting.
                    if (e.stopPropagation) {
                        e.stopPropagation();
                    }

                    this.classList.remove('over');
                    $(this).parent().parent().removeClass('over');

                    // call the passed drop function
                    $scope.$apply(function($scope) {
                        const newData = angular.copy($scope.activeDragDrop.data);
                        // in some cases the list can be dynamic, based on the data we're dropping (hierarchies vs columns)
                        const targetList = $scope.getChartDragDropListReplace(newData);
                        delete newData.$$hashKey;
                        newData.__justDragDropped = true;

                        if ($scope.activeDragDrop.moveFromList && $scope.activeDragDrop.moveFromList === targetList) {
                            // DO nothing ...

                        } else if ($scope.activeDragDrop.moveFromList) {
                            targetList.splice(0, targetList.length);
                            targetList.push(newData);
                            $scope.activeDragDrop.moveFromList.splice($scope.activeDragDrop.moveFromListIndex, 1);
                        } else {
                            targetList.splice(0, targetList.length);
                            targetList.push(newData);
                        }

                        // Force remove placeholder right now
                        element.removeClass('drop-accepted');
                        element.removeClass('drop-rejected');

                        $scope.onDragEnd(newData);

                        $scope.$emit('dragDropSuccess');
                    });
                    return false;
                }, false);
            }
        };
    });
})();

;
(function() {
    'use strict';
    const app = angular.module('dataiku.charts');

    /**
     * (!) This directive previously was in static/dataiku/js/simple_report/chart_dragdrop.js
     */
    app.directive('chartDragDropList', function($parse, Assert) {
        return {
            scope: true,
            controller: 'ChartDragDropController',
            link: function($scope, element, attrs) {

                let acceptFunc = function(data) {
                    return {
                        accept: true,
                        message: 'Drop here'
                    };
                };
                if (attrs.acceptDrop) {
                    const parsed = $parse(attrs.acceptDrop);
                    acceptFunc = function(data) {
                        return parsed($scope.$parent || $scope, { 'data': data });
                    };
                }

                const parsed = attrs.getChartDragDropList && $parse(attrs.getChartDragDropList);
                $scope.getChartDragDropList = function(data) {
                    const parsedFn = parsed && parsed($scope.$parent || $scope);
                    if (!parsedFn) {
                        return $scope.$eval(attrs.chartDragDropList);
                    }
                    return parsedFn(data);
                };

                const placeholderPos = attrs.placeholderPos || 'end';
                const direction = attrs.direction || 'horizontal';

                const placeholder = $('<li class="sortable-placeholder" />');
                let placeholderAttachedOnce = false;

                const onDragOverOrEnter = function(e) {
                    this.classList.add('over');
                    $scope.addClassHereAndThere(this, 'over');

                    const dropLi = $(e.target).closest('li');

                    if ($scope.activeDragDrop.draggedElementToHide) {
                        $scope.activeDragDrop.draggedElementToHide.hide();
                    }

                    // Do we accept this payload ?
                    const accepted = acceptFunc($scope.activeDragDrop.data);
                    if (accepted.accept) {
                        e.dataTransfer.dropEffect = 'copyMove';

                        if (dropLi.length !== 0) {

                            // Determine the insertion point for the placeholder based on the list direction.
                            if (direction === 'horizontal') {
                                const mouseX = e.clientX;
                                const dropLiWidth = dropLi.outerWidth();
                                const dropLiLeft = dropLi.offset().left;
                                const isLeft = mouseX < dropLiLeft + dropLiWidth / 2;
                                if (isLeft) {
                                    dropLi.before(placeholder);
                                } else {
                                    dropLi.after(placeholder);
                                }
                            } else {
                                const mouseY = e.clientY;
                                const dropLiTop = dropLi.offset().top;
                                const dropLiHeight = dropLi.outerHeight();
                                const isTop = mouseY < dropLiTop + dropLiHeight / 2;
                                if (isTop) {
                                    dropLi.before(placeholder);
                                } else {
                                    dropLi.after(placeholder);
                                }
                            }
                        }
                        e.preventDefault();
                    } else {
                        $scope.$apply(function() {
                            $scope.validity.tempError = {};
                            $scope.validity.tempError.type = 'MEASURE_REJECTED';
                            $scope.validity.tempError.message = accepted.message;
                        });
                    }
                };
                element[0].addEventListener('dragover', onDragOverOrEnter, false);
                element[0].addEventListener('dragenter', onDragOverOrEnter, false);

                element[0].addEventListener('dragleave', function(e) {
                    $scope.removeClassHereAndThere(this, 'over');
                    $scope.$apply(function() {
                        if ($scope.validity && $scope.validity.tempError) {
                            delete $scope.validity.tempError;
                        }
                    });
                    return false;
                }, false);

                /*
                 * This is triggered as soon as a drag becomes active on the page
                 * and highlights the drop zone if it's accepted
                 */
                $scope.$watch('activeDragDrop.active', function(nv, ov) {
                    if (nv) {
                        const accepted = acceptFunc($scope.activeDragDrop.data);
                        if (accepted.accept) {
                            $scope.addClassHereAndThere(element, 'drop-accepted');
                            window.setTimeout(function() {
                                // With the new component pivot-filter-list.component.js the directive is no longer directly set on the ul element but on the component itself.
                                const listElement = element[0].tagName === 'UL' ? element : element.find('ul').first();
                                if (placeholderPos == 'end') {
                                    listElement.append(placeholder);
                                } else {
                                    listElement.prepend(placeholder);
                                }
                                placeholderAttachedOnce = true;
                            }, 10);
                        } else {
                            $scope.addClassHereAndThere(element, 'drop-rejected');
                        }
                    } else {
                        $scope.removeClassHereAndThere(element, 'drop-accepted');
                        $scope.removeClassHereAndThere(element, 'drop-rejected');

                        if (placeholderAttachedOnce) {
                            window.setTimeout(function() {
                                placeholder.detach();
                            }, 10);
                        }
                    }
                }, true);

                element[0].addEventListener('drop', function(e) {
                    Assert.trueish($scope.activeDragDrop.active, 'no active drag and drop');

                    // Stops some browsers from redirecting.
                    if (e.stopPropagation) {
                        e.stopPropagation();
                    }

                    $scope.removeClassHereAndThere(this, 'over');

                    // At which index are we dropping ?
                    let dropIndex = $(e.target).index();

                    // If dropping on the ul element, try dropping it on the displayed placeholder, if not found drop it at the end of the list
                    if ($(e.target).is('ul')) {
                        dropIndex = Array.from(e.target.children).findIndex(item => item === placeholder[0]);
                        if (dropIndex === -1) {
                            dropIndex = e.target.children.length;
                        }
                    }

                    // call the passed drop function
                    $scope.$apply(function($scope) {
                        const newData = angular.copy($scope.activeDragDrop.data);
                        delete newData.$$hashKey;
                        newData.__justDragDropped = true;

                        const targetList = $scope.getChartDragDropList(newData);
                        if ($scope.activeDragDrop.moveFromList && $scope.activeDragDrop.moveFromList === targetList) {
                            const oldIdx = $scope.activeDragDrop.moveFromListIndex;

                            if (dropIndex > oldIdx && dropIndex > 0) {
                                dropIndex--;
                            }

                            targetList.splice(dropIndex, 0, targetList.splice(oldIdx, 1)[0]);

                        } else if ($scope.activeDragDrop.moveFromList) {
                            targetList.splice(dropIndex, 0, newData);
                            if ($scope.activeDragDrop.moveFromList) {
                                $scope.activeDragDrop.moveFromList.splice($scope.activeDragDrop.moveFromListIndex, 1);
                            }
                        } else {
                            targetList.splice(dropIndex, 0, newData);
                        }

                        // Force remove placeholder right now
                        $scope.removeClassHereAndThere(element, 'drop-accepted');
                        $scope.removeClassHereAndThere(element, 'drop-rejected');
                        placeholder.detach();

                        $scope.onDragEnd(newData);

                        $scope.$emit('dragDropSuccess');
                    });
                    return false;
                }, false);
            }
        };
    });
})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.charts');

    /**
     * (!) This directive previously was in static/dataiku/js/simple_report/chart_dragdrop.js
     */
    app.directive('chartMultiDragDropZones', function(translate, ChartColumnTypeUtils, WT1) {
        return {
            controller: 'ChartDragDropController',
            link: function($scope, element) {

                $scope.activeDragDrop = {};
                $scope.translate = translate;

                $scope.onDragEnd = function(data) {
                    // Unhide the moved element, as ng-repeat will reuse it
                    if ($scope.activeDragDrop.draggedElementToHide) {
                        $scope.activeDragDrop.draggedElementToHide.show();
                    }
                    clear($scope.activeDragDrop);
                    $scope.setDragInactive();

                    const currentChartIndex = $scope.currentChart && $scope.currentChart.index;
                    if (ChartColumnTypeUtils.isHierarchy(data) && !_.isNil(currentChartIndex)) {
                        const chartType = $scope.charts && $scope.charts[currentChartIndex] && $scope.charts[currentChartIndex].def.type;
                        WT1.event('hierarchy-added-to-chart', { chartType });
                    }
                };

                element[0].addEventListener('dragend', function(e) {
                    $scope.$apply($scope.onDragEnd);
                });
            }
        };
    });
})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.charts');

    // (!) This directive previously was in static/dataiku/js/simple_report/config_ui.js
    app.directive('aggregatedGeoZone', function($parse, ChartsStaticData, translate, Ng2MenuOrchestratorService) {
        return {
            templateUrl: '/static/dataiku/js/simple_report/directives/drag-drop/drop-zones/aggregated-geo-zone/aggregated-geo-zone.directive.html',
            scope: true,
            link: function($scope, element, attrs) {
                const defaultAggregationOptions = [
                    { value: ChartsStaticData.GEOM_AGGREGATIONS.DEFAULT, label: translate('CHARTS.DROPPED_DIMENSION.KEEP_DUPLICATES', 'Keep duplicates') },
                    { value: ChartsStaticData.GEOM_AGGREGATIONS.DISTINCT, label: translate('CHARTS.DROPPED_DIMENSION.MAKE_UNIQUE', 'Make unique') }
                ];
                $scope.$watch(attrs.list, newList => $scope.list = newList);
                $scope.chartDefKey = attrs.chartDefKey;
                $scope.acceptCallback = $parse(attrs.acceptCallback)($scope);
                $scope.aggregationOptions = defaultAggregationOptions;
                $scope.$watch(attrs.uaColor, function(newUaColor) {
                    $scope.showAggregate = (newUaColor.length === 0);
                    if ($scope.showAggregate) {
                        $scope.aggregationOptions = defaultAggregationOptions;
                    } else {
                        $scope.aggregationOptions = [];
                    }
                }, true);

                $scope.getLabel = function(aggregationFunction) {
                    if (aggregationFunction === ChartsStaticData.GEOM_AGGREGATIONS.DISTINCT) {
                        return translate('CHARTS.DROPPED_DIMENSION.UNIQUE', '(UNIQUE)');
                    } else {
                        return '';
                    }
                };

                $scope.openMatMenu = function(menuIdx, event) {
                    Ng2MenuOrchestratorService.triggerMenuOpen($scope.chartDefKey, menuIdx, event);
                };
            }
        };
    });
})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.charts');

    // (!) This directive previously was in static/dataiku/js/simple_report/config_ui.js
    app.directive('geoAdminZone', function($parse, Ng2MenuOrchestratorService) {
        return {
            templateUrl: '/static/dataiku/js/simple_report/directives/drag-drop/drop-zones/geo-admin-zone/geo-admin-zone.directive.html',
            scope: true,
            link: function($scope, element, attrs) {
                $scope.$watch(attrs.list, newList => $scope.list = newList);
                $scope.id = attrs.id;
                $scope.chartDefKey = attrs.chartDefKey;
                $scope.acceptCallback = $parse(attrs.acceptCallback)($scope);

                $scope.openMatMenu = function(menuIdx, event) {
                    Ng2MenuOrchestratorService.triggerMenuOpen($scope.chartDefKey, menuIdx, event);
                };
            }
        };
    });
})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.charts');

    // (!) This directive previously was in static/dataiku/js/simple_report/config_ui.js
    app.directive('geoNoOptsZone', function($parse) {
        return {
            templateUrl: '/static/dataiku/js/simple_report/directives/drag-drop/drop-zones/geo-no-opts-zone/geo-no-opts-zone.directive.html',
            scope: true,
            link: function($scope, element, attrs) {
                $scope.$watch(attrs.list, newList => $scope.list = newList);
                $scope.acceptCallback = $parse(attrs.acceptCallback)($scope);
            }
        }
    });
})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.charts');

    // (!) This directive previously was in static/dataiku/js/simple_report/config_ui.js
    app.directive('monoUaZoneNoOpts', function($parse) {
        return {
            templateUrl: '/static/dataiku/js/simple_report/directives/drag-drop/drop-zones/mono-ua-zone-no-opts/mono-ua-zone-no-opts.directive.html',
            scope: true,
            link: function($scope, element, attrs) {
                $scope.$watch(attrs.list, newList => $scope.list = newList);
                $scope.acceptCallback = $parse(attrs.acceptCallback)($scope);
            }
        };
    });
})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.charts');

    // (!) This directive previously was in static/dataiku/js/simple_report/config_ui.js
    app.directive('monoUaZone', function($parse, ChartFeatures, ChartLabels, Ng2MenuOrchestratorService) {
        return {
            templateUrl: '/static/dataiku/js/simple_report/directives/drag-drop/drop-zones/mono-ua-zone/mono-ua-zone.directive.html',
            scope: true,
            link: function($scope, _element, attrs) {
                $scope.$watch(attrs.list, newList => $scope.list = newList);
                $scope.chartDefKey = attrs.chartDefKey;
                $scope.acceptCallback = $parse(attrs.acceptCallback)($scope);
                $scope.ChartFeatures = ChartFeatures;
                $scope.ChartLabels = ChartLabels;

                $scope.openMatMenu = function(menuIdx, event) {
                    Ng2MenuOrchestratorService.triggerMenuOpen($scope.chartDefKey, menuIdx, event);
                };
            }
        };
    });
})();

;
(function() {
    'use strict';


    const app = angular.module('dataiku.charts');

    // (!) This directive previously was in static/dataiku/js/simple_report/config_ui.js
    app.directive('monovaluedStdAggrDimensionZoneNoOpts', function($parse) {
        return {
            templateUrl: '/static/dataiku/js/simple_report/directives/drag-drop/drop-zones/monovalued-std-aggr-dimension-zone-no-opts/monovalued-std-aggr-dimension-zone-no-opts.directive.html',
            scope: true,
            link: function($scope, element, attrs) {
                $scope.$watch(attrs.list, newList => $scope.list = newList);
                $scope.acceptCallback = $parse(attrs.acceptCallback)($scope);
            }
        }
    });
})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.charts');

    // (!) This directive previously was in static/dataiku/js/simple_report/config_ui.js
    app.directive('monovaluedStdAggrDimensionZone', function($parse, ChartsStaticData, ChartLabels, ChartColumnTypeUtils, ChartHierarchyDimension, Ng2MenuOrchestratorService) {
        return {
            templateUrl: '/static/dataiku/js/simple_report/directives/drag-drop/drop-zones/monovalued-std-aggr-dimension-zone/monovalued-std-aggr-dimension-zone.directive.html',
            scope: true,
            link: function($scope, element, attrs) {
                $scope.$watch(attrs.list, newList => $scope.list = newList);
                $scope.getList = $parse(attrs.getList)($scope);
                attrs.$observe('chartStoreId', (nv) => {
                    $scope.chartStoreId = nv;
                });
                attrs.$observe('chartDefKey', (nv) => {
                    $scope.chartDefKey = nv;
                });
                $scope.isSecondDimension = $parse(attrs.isSecondDimension)($scope);
                $scope.hideOneTickPerBin = $scope.isSecondDimension;
                $scope.acceptCallback = $parse(attrs.acceptCallback)($scope);
                $scope.dateModes = ChartsStaticData.dateModes;
                $scope.ChartLabels = ChartLabels;
                $scope.isHierarchy = ChartColumnTypeUtils.isHierarchy;
                $scope.hasHierarchyWarning = ChartHierarchyDimension.hasHierarchyMissingColumns;
                $scope.getHierarchyWarningTooltip = ChartHierarchyDimension.getMissingColumnsTooltipForHierarchy;

                $scope.openMatMenu = function(menuIdx, event) {
                    Ng2MenuOrchestratorService.triggerMenuOpen($scope.chartDefKey, menuIdx, event);
                };
            }
        };
    });
})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.charts');

    // (!) This directive previously was in static/dataiku/js/simple_report/config_ui.js
    app.directive('monovaluedStdAggrMeasureZone', function($parse, ChartFeatures, ChartLabels, ChartCustomMeasures, ColorUtils, DefaultDSSVisualizationTheme, Ng2MenuOrchestratorService) {
        return {
            templateUrl: '/static/dataiku/js/simple_report/directives/drag-drop/drop-zones/monovalued-std-aggr-measure-zone/monovalued-std-aggr-measure-zone.directive.html',
            scope: true,
            controller: 'StdAggregatedMeasureController',
            link: function($scope, _element, attrs) {
                $scope.$watch(attrs.list, newList => $scope.list = newList);
                $scope.chartDefKey = attrs.chartDefKey;
                $scope.acceptCallback = $parse(attrs.acceptCallback)($scope);
                $scope.foregroundColors = [];
                $scope.themeColors = [];

                $scope.$watch(attrs.theme, (nv, ov) => {
                    $scope.theme = nv;
                    $scope.themeColors = ColorUtils.getThemeColorsWithBlackWhite($scope.theme);
                    if (nv && (!ov || !_.isEqual(nv.colors, ov.colors) || !$scope.foregroundColors.length)) {
                        const paletteColors = ColorUtils.generateThemePaletteColors(nv.colors, $scope.themeColors.length > 0);
                        if (paletteColors) {
                            $scope.foregroundColors = paletteColors.foregroundColors;
                        }
                    } else if (!nv) {
                        $scope.foregroundColors = ColorUtils.generateThemePaletteColors(DefaultDSSVisualizationTheme.colors, false).foregroundColors;
                    }
                    $scope.defaultFormatting = $scope.theme ? { fontColor: $scope.theme.generalFormatting.fontColor } : {};
                });

                $scope.openMatMenu = function(menuIdx, event) {
                    Ng2MenuOrchestratorService.triggerMenuOpen($scope.chartDefKey, menuIdx, event);
                };
            }
        };
    });
})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.charts');

    // (!) This directive previously was in static/dataiku/js/simple_report/config_ui.js
    app.directive('multiUaZoneNoOpts', function($parse) {
        return {
            templateUrl: '/static/dataiku/js/simple_report/directives/drag-drop/drop-zones/multi-ua-zone-no-opts/multi-ua-zone-no-opts.directive.html',
            scope: true,
            link: function($scope, element, attrs) {
                $scope.$watch(attrs.list, newList => $scope.list = newList);
                $scope.acceptCallback = $parse(attrs.acceptCallback)($scope);
            }
        };
    });
})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.charts');

    // (!) This directive previously was in static/dataiku/js/simple_report/config_ui.js
    app.directive('multiUaZone', function($parse, ChartLabels, Ng2MenuOrchestratorService) {
        return {
            templateUrl: '/static/dataiku/js/simple_report/directives/drag-drop/drop-zones/multi-ua-zone/multi-ua-zone.directive.html',
            scope: true,
            link: function($scope, element, attrs) {
                $scope.$watch(attrs.list, newList => $scope.list = newList);
                $scope.chartDefKey = attrs.chartDefKey;
                $scope.acceptCallback = $parse(attrs.acceptCallback)($scope);
                $scope.ChartLabels = ChartLabels;

                $scope.openMatMenu = function(menuIdx, event) {
                    Ng2MenuOrchestratorService.triggerMenuOpen($scope.chartDefKey, menuIdx, event);
                };
            }
        };
    });
})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.charts');

    // (!) This directive previously was in static/dataiku/js/simple_report/config_ui.js
    app.directive('multivaluedStdAggrDimensionZone', function($parse, ChartsStaticData, ChartLabels, ChartColumnTypeUtils, Ng2MenuOrchestratorService) {
        return {
            templateUrl: '/static/dataiku/js/simple_report/directives/drag-drop/drop-zones/multivalued-std-aggr-dimension-zone/multivalued-std-aggr-dimension-zone.directive.html',
            scope: true,
            link: function($scope, element, attrs) {
                $scope.$watch(attrs.list, newList => $scope.list = newList);
                $scope.getList = $parse(attrs.getList)($scope);
                $scope.chartDefKey = attrs.chartDefKey;
                $scope.acceptCallback = $parse(attrs.acceptCallback)($scope);
                $scope.dateModes = ChartsStaticData.dateModes;
                $scope.ChartLabels = ChartLabels;
                $scope.isHierarchy = ChartColumnTypeUtils.isHierarchy;

                $scope.openMatMenu = function(menuIdx, event) {
                    Ng2MenuOrchestratorService.triggerMenuOpen($scope.chartDefKey, menuIdx, event);
                };
            }
        };
    });
})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.charts');

    // (!) This directive previously was in static/dataiku/js/simple_report/config_ui.js
    app.directive('multivaluedStdAggrMeasureZone', function($parse, ChartFeatures, Ng2MenuOrchestratorService, ChartColorUtils) {
        return {
            templateUrl: '/static/dataiku/js/simple_report/directives/drag-drop/drop-zones/multivalued-std-aggr-measure-zone/multivalued-std-aggr-measure-zone.directive.html',
            scope: true,
            controller: 'StdAggregatedMeasureController',
            link: function($scope, _element, attrs) {
                $scope.$watch(attrs.list, newList => $scope.list = newList);
                $scope.chartDefKey = attrs.chartDefKey;
                $scope.acceptCallback = $parse(attrs.acceptCallback)($scope);
                $scope.direction = attrs.direction;
                $scope.shouldDisplayMultiPlotDisplayMode = function(measure) {
                    return ChartFeatures.canSetMultiPlotDisplayMode($scope.chart.def.type)
                    && measure && measure.displayType === 'line'
                    && $scope.chart.def.genericMeasures.some(m => m.displayType === 'column')
                    && !!ChartColorUtils.getColorDimensionOrMeasure($scope.chart.def);
                };
                $scope.$watch(attrs.theme, (nv) => {
                    $scope.theme = nv;
                });

                $scope.openMatMenu = function(menuIdx, event) {
                    Ng2MenuOrchestratorService.triggerMenuOpen($scope.chartDefKey, menuIdx, event);
                };
            }
        };
    });
})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.charts');

    // (!) This directive previously was in static/dataiku/js/simple_report/config_ui.js
    app.directive('scatterAxisZone', function($parse, ChartLabels, Ng2MenuOrchestratorService) {
        return {
            templateUrl: '/static/dataiku/js/simple_report/directives/drag-drop/drop-zones/scatter-axis-zone/scatter-axis-zone.directive.html',
            scope: true,
            link: function($scope, element, attrs) {
                $scope.$watch(attrs.list, newList => $scope.list = newList);
                $scope.chartDefKey = attrs.chartDefKey;
                $scope.acceptCallback = $parse(attrs.acceptCallback)($scope);
                $scope.ChartLabels = ChartLabels;
                $scope.pairIndex = attrs.pairIndex;

                $scope.openMatMenu = function(menuIdx, event) {
                    Ng2MenuOrchestratorService.triggerMenuOpen($scope.chartDefKey, menuIdx, event);
                };
            }
        };
    });
})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.charts');

    // (!) This directive previously was in static/dataiku/js/simple_report/config_ui.js
    app.directive('scatterDetailZone', function($parse, ChartLabels, ChartUADimension, Ng2MenuOrchestratorService) {
        return {
            templateUrl: '/static/dataiku/js/simple_report/directives/drag-drop/drop-zones/scatter-detail-zone/scatter-detail-zone.directive.html',
            scope: true,
            link: function($scope, element, attrs) {
                $scope.$watch(attrs.list, newList => $scope.list = newList);
                $scope.chartDefKey = attrs.chartDefKey;
                $scope.acceptCallback = $parse(attrs.acceptCallback)($scope);
                $scope.ChartLabels = ChartLabels;
                $scope.dateModes = ChartUADimension.getDateModes().map(({ value, label }) => ([value, label]));

                $scope.openMatMenu = function(menuIdx, event) {
                    Ng2MenuOrchestratorService.triggerMenuOpen($scope.chartDefKey, menuIdx, event);
                };
            }
        };
    });
})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.charts');

    // (!) This directive previously was in static/dataiku/js/simple_report/config_ui.js
    app.directive('extentFormatter', function() {
        return {
            restrict: 'A',
            require: 'ngModel',
            scope: {
                defaultExtentValue: '<'
            },
            link: function($scope, _element, attrs, ngModelController) {
                if ($scope.defaultExtentValue === null) {
                    return;
                }

                // On null input use default $autoExtent value
                const formatExtent = (newValue) => {
                    return newValue === null ? $scope.defaultExtentValue : newValue;
                }

                ngModelController.$formatters.push(formatExtent);

                $scope.$on('formatManualExtent', () => {
                    let viewValue = ngModelController.$modelValue;
                    ngModelController.$formatters.forEach(formatter => {
                        viewValue = formatter(viewValue);
                    });
                    ngModelController.$viewValue = viewValue;
                    ngModelController.$render();
                });
            }
        }
    });
})();

;
(function() {
    'use strict';

    angular.module('dataiku.charts')
        .directive('continuousColorLegend', continuousColorLegend);

    /**
     * (!) This directive previously was in static/dataiku/js/simple_report/common/legends.js
     */
    function continuousColorLegend(Fn, D3ChartAxes) {
        return {
            scope: true,
            templateUrl: '/static/dataiku/js/simple_report/directives/legends/continuous-color-legend/continuous-color-legend.directive.html',
            link: function($scope, element, attrs) {

                const placement = element.closest('.pivot-charts').attr('legend-placement');

                $scope.gradientError = false;
                $scope.$watch(attrs.legend, function(nv, ov) {
                    $scope.draw($scope.$eval(attrs.legend));
                });

                const svg = d3.select(element[0]).select('svg'),
                    $svg = element.find('svg'),
                    gradient = svg.select('linearGradient');

                let vertical, orient,
                    barWidth = Math.max(0, $svg.width() - 10),
                    barHeight = Math.max(0, $svg.height() - 20),
                    axisX = 5,
                    axisY = 10,
                    rectX = 0;
                switch (placement) {
                    case 'OUTER_RIGHT':
                        vertical = true;
                        barWidth = 15;
                        axisX = 15;
                        orient = 'right';
                        break;
                    case 'OUTER_LEFT':
                        vertical = true;
                        barWidth = 15;
                        orient = 'left';
                        break;
                    case 'OUTER_TOP':
                    case 'OUTER_BOTTOM':
                    default: // sidebar or inner
                        vertical = false;
                        $svg.height(45);
                        orient = 'bottom';
                        axisY = 15;
                        barHeight = 15;
                        break;
                }

                if (vertical) {
                    gradient.attr('x2', '0%').attr('y2', '100%');
                }

                svg.select('rect')
                    .attr('width', barWidth)
                    .attr('height', barHeight)
                    .attr('y', vertical ? 10 : 0)
                    .attr('x', vertical ? 0 : 5);

                const setStyle = function(style, ticks) {
                    const fontSize = style && style.fontSize ? `${style.fontSize}px` : '13px';
                    const fontColor = style && style.fontColor ? style.fontColor : 'inherit';
                    ticks.forEach(tick => {
                        tick.style.fontSize = fontSize;
                        tick.style.fill = fontColor;
                    });
                };

                $scope.draw = function(legend) {
                    if (!legend) {
                        if (svg.select('rect').attr('width') == 0) {
                            $scope.gradientError = true;
                        }
                        return;
                    }

                    $scope.gradientError = false;
                    const isCondFormatting = legend.type === 'CONDITIONAL_FORMATTING';

                    if (isCondFormatting) {
                        barWidth = 185;
                        // set width && height here bc legend can be initialized with empty attrs.legend
                        svg.select('rect')
                            .attr('width', barWidth)
                            .attr('height', 10)
                        ;
                    }

                    const axisScale = legend.scale.innerScale.copy();
                    if (legend.scale.diverging) {
                        axisScale.domain([axisScale.invert(0), axisScale.invert(1)]).range([0, 1]);
                    }

                    axisScale.range(axisScale.range().map(x => vertical ? (barHeight - x * barHeight) : x * barWidth)).interpolate(d3.interpolate);
                    const axis = d3.svg.axis().orient(orient).scale(axisScale);

                    // Arbitrary value to convert pixels to tick number
                    const pixelDivisor = 50;
                    axis.tickValues(axisScale.ticks(Math.min(10, vertical ? barHeight / pixelDivisor : barWidth / pixelDivisor)).concat(axisScale.domain()));
                    axis.tickFormat(legend.formatter);
                    const axisG = svg.select('g.axis');
                    axisG.selectAll('*').remove();
                    axisG.call(axis).select('path.domain').remove();

                    const ticks = axisG.selectAll('g.tick')[0];
                    ticks.sort((a, b) => a.__data__ - b.__data__);
                    setStyle($scope.chart ? $scope.chart.def.legendFormatting : {}, ticks);

                    // Anchor to start & end mostleft & mostright labels
                    if (!vertical) {
                        d3.select(ticks[0]).select('text').style('text-anchor', 'start');
                        d3.select(ticks[ticks.length - 1]).select('text').style('text-anchor', 'end');
                    }

                    if (isCondFormatting) {
                        // keep only first and last tick for conditional formatting
                        ticks.length > 1 && ticks.slice(1, -1).forEach(tick => tick.remove());
                        d3.select(ticks[0]).select('line').remove();
                        d3.select(ticks[0]).select('text').text('Min');

                        d3.select(ticks[ticks.length - 1]).select('line').remove();
                        d3.select(ticks[ticks.length - 1]).select('text').text('Max');
                    } else {
                        D3ChartAxes.sanitizeTicksDisplay(ticks);
                    }

                    const colors = legend.scale.outerScale.range();
                    const colorStops = [];
                    const numStops = legend.scale.quantizationMode === 'NONE' ? colors.length - 1 : colors.length;

                    if (legend.scale.quantizationMode !== 'QUANTILES') {
                        colors.forEach(function(c, i) {
                            colorStops.push({
                                color: c,
                                offset: i * 100 / numStops
                            });

                            if (legend.scale.quantizationMode !== 'NONE') {
                                colorStops.push({
                                    color: c,
                                    offset: (i + 1) * (100 / numStops)
                                });
                            }
                        });
                    } else {
                        const thresholds = legend.scale.outerScale.quantiles();
                        colors.forEach(function(c, i) {
                            colorStops.push({
                                color: c,
                                offset: (i === 0 ? 0 : thresholds[i - 1] * 100)
                            });
                            colorStops.push({
                                color: c,
                                offset: (i === colors.length - 1 ? 100 : thresholds[i] * 100)
                            });
                        });
                    }

                    // In the vertical scale, we want the first stop at the bottom
                    if (vertical) {
                        colorStops.forEach(function(stop) {
                            stop.offset = (100 - (stop.offset));
                        });
                        colorStops.reverse();
                    }

                    /*
                     * This was used to display the color palette with a log/square/square root gradient instead of a linear gradient,
                     * but instead we display a linear gradient and let d3 put the ticks at the right places
                     * if (scale.mode == 'LINEAR') {
                     *  points = legend.scale.domain();
                     * } else {
                     *  var NUM_STOPS = 100;
                     *  var range = axisScale.range();
                     *  var step = (domain[domain.length-1] - domain[0])/NUM_STOPS;
                     *  for (var i = 0; i < NUM_STOPS; i++) {
                     *      points.push(domain[0] + step*i);
                     *  }
                     * }
                     */

                    const padding = 5;
                    const margin = 15;
                    gradient[0][0].replaceChildren([]);
                    gradient.selectAll('stop').data(colorStops)
                        .enter().append('stop')
                        .attr('offset', stop => stop.offset + '%')
                        .attr('stop-color', Fn.prop('color'))
                        .attr('stop-opacity', 1);

                    if (vertical) {
                        const maxWidth = d3.max(axisG.selectAll('g.tick')[0].map(function(itm) {
                            return itm.getBoundingClientRect().width;
                        })) || 0;

                        $svg.css('width', maxWidth + margin);
                    } else {
                        const collapsedColorGroupHeight = 22;
                        const defaultHeight = isCondFormatting ? collapsedColorGroupHeight : 0;

                        const maxHeight = d3.max(axisG.selectAll('g.tick')[0].map(function(itm) {
                            return itm.getBoundingClientRect().height;
                        })) || defaultHeight;

                        $svg.css('height', maxHeight + margin + padding);
                        $svg.css('background-color', 'rgb(255, 255, 255, 0.6)');
                    }

                    if (placement == 'OUTER_LEFT') {
                        rectX = $svg.width() - 15;
                        axisX = rectX;
                    }

                    const axisGY = (vertical || isCondFormatting) ? axisY : axisY + padding;
                    const svgY = vertical ? 0 : padding;
                    axisG.attr('transform', 'translate(' + axisX + ',' + axisGY + ')');
                    svg.select('rect').attr('transform', 'translate(' + rectX + ', ' + svgY + ')');
                };
            }
        }
    }

})();

;
(function() {
    'use strict';

    angular.module('dataiku.charts')
        .directive('discreteColorLegend', discreteColorLegend);

    /**
     * (!) This directive previously was in static/dataiku/js/simple_report/common/legends.js
     */
    function discreteColorLegend() {
        return {
            scope: true,
            templateUrl: '/static/dataiku/js/simple_report/directives/legends/discrete-color-legend/discrete-color-legend.directive.html',
            link: function($scope, element, attrs) {
                $scope.$watch(attrs.legend, function(nv, ov) {
                    $scope.legend = $scope.$eval(attrs.legend);
                });
                const defaultMaxValue = 101; // Because by default dimension.maxValues is 100 (+ room for the others bin)
                $scope.maxValues = angular.isDefined(attrs.maxValues) ? $scope.$eval(attrs.maxValues) : defaultMaxValue;

                const setStyle = function(style) {
                    const lineHeightRatio = 1.3; //  To avoid a text to be cropped in its container, we should set a lineHeight which is, in average, 33% bigger
                    const defaultFontSize = style && style.fontSize ? `${style.fontSize}px` : '13px';
                    const defaultHeight = style && style.fontSize ? `${Math.round(style.fontSize * lineHeightRatio)}px` : '13px';

                    const defaultStyle = {
                        height: defaultHeight,
                        lineHeight: defaultHeight,
                        fontSize: defaultFontSize
                    };

                    $scope.itemStyle = {
                        minHeight: defaultHeight
                    };

                    $scope.legendShapeStyle = {
                        ...defaultStyle,
                        width: defaultHeight,
                        minWidth: defaultHeight
                    };

                    $scope.legendStyle = {
                        ...defaultStyle,
                        color: style && style.fontColor ? style.fontColor : 'inherit'
                    };
                };

                setStyle($scope.chart.def.legendFormatting);

                $scope.hasFocused = false;

                $scope.showMore = function(more = defaultMaxValue) {
                    $scope.maxValues += more;
                };

                const unfocusAll = function() {
                    if ($scope.tooltips.resetColors) {
                        $scope.tooltips.resetColors();
                    }
                    if ($scope.legend.unfocusFnList) {
                        $scope.legend.unfocusFnList.forEach(currentUnFocusFn => currentUnFocusFn());
                    }
                    $scope.legend.items.forEach(function(it) {
                        if (it.focused && it.unfocusFn) {
                            it.unfocusFn();
                        }
                        it.focused = false;
                    });
                };

                let lastHoveredIndex = null;

                $scope.onLegendMouseMove = function($event) {
                    const hoveredEl = document.elementFromPoint($event.clientX, $event.clientY);
                    const legendItemEl = hoveredEl?.closest('[data-legend-index]');
                    if (!legendItemEl) {
                        return;
                    }

                    const index = parseInt(legendItemEl.getAttribute('data-legend-index'));
                    if (index === lastHoveredIndex) {
                        return;
                    }

                    lastHoveredIndex = index;
                    unfocusAll();

                    const item = $scope.legend.items[index];
                    if ($scope.tooltips.focusColor) {
                        $scope.tooltips.focusColor(index);
                    }
                    item.focused = true;
                    item.focusFn?.();
                    $scope.legend.focusFnList?.forEach(currentFocusFn => currentFocusFn(index, { displayPoints: false }));
                    $scope.hasFocused = true;
                };

                $scope.clearLegendFocus = function() {
                    lastHoveredIndex = null;
                    unfocusAll();
                    $scope.hasFocused = false;
                    if ($scope.tooltips.resetColors) {
                        $scope.tooltips.resetColors();
                    }
                };
            }
        };
    }
})();

;
(function() {
    'use strict';

    const THUMBNAIL_WIDTH = 120;
    const THUMBNAIL_HEIGHT = 80;

    const app = angular.module('dataiku.directives.insights', ['dataiku.filters', 'dataiku.charts']);

    /**
     * Service responsible for displaying charts based on a pivot response.
     * (!) This directive previously was in static/dataiku/js/simple_report/chart_views.js
     */
    app.directive('pivotChartResult', function($timeout, $q, $state, CHART_TYPES, DKUPivotCharts, Logger, ChartFeatures, ChartDimension, ChartZoomControlAdapter, CanvasUtils, EChartsManager, Fn, ChartStoreFactory, ChartFormattingPane, ChartFormattingPaneSections, ChartActivityIndicator, ChartDefinitionChangeHandler, DSSVisualizationThemeUtils, ChartDrilldown, DRILL_UP_SOURCE) {

        return {
            templateUrl: '/static/dataiku/js/simple_report/directives/pivot-chart-result/pivot-chart-result.directive.html',
            scope: true,
            link: function(scope, element) {

                //these functions need to be put in the scope to be accessible in dashboards
                scope.canHaveZoomControls = ChartFeatures.canHaveZoomControls;
                scope.hasScatterZoomControlActivated = ChartFeatures.hasScatterZoomControlActivated;
                scope.isEChart = ChartFeatures.isEChart;
                scope.canDisplayLegend = ChartFeatures.canDisplayLegend;

                scope.chartActivityIndicator = ChartActivityIndicator.buildDefaultActivityIndicator();

                scope.openSection = function(section) {
                    const readOnly = $state.current.name === 'projects.project.dashboards.insights.insight.view';
                    if (!readOnly && !(section === ChartFormattingPaneSections.MISC && scope.isInPredicted)) {
                        ChartFormattingPane.open(section, true, true);
                    }
                };

                scope.breadcrumbDrillUp = (targetDimension) => {
                    ChartDrilldown.handleMultiLevelDrillUp(scope.chart.def, targetDimension, DRILL_UP_SOURCE.BREADCRUMB);
                };

                // Chart lazy loading causes issues with dashboard export, so we disable it when in an export flow.
                scope.disableLazyLoading = getCookie('dku_unattended') === 'true';

                const loadChart = function(axesDef, echartDef, d3Chart, chartActivityIndicator, hideLegend = false) {
                    if (ChartFeatures.isEChart(scope.chart.def)) {
                        element.find('.mainzone').remove();

                        const data = scope.response.result.pivotResponse;

                        if (scope.chart.theme && !DSSVisualizationThemeUtils.isThemeFontLoaded(scope.chart.theme)) {
                            // Force loading of fonts for gauge so we have the font loaded and can measure text width on first paint
                            DSSVisualizationThemeUtils.loadThemeFont(scope.chart.theme)
                                .finally(() => EChartsManager.initEcharts(scope, element, data, axesDef, echartDef, chartActivityIndicator, hideLegend, scope.chart.theme));
                        } else {
                            EChartsManager.initEcharts(scope, element, data, axesDef, echartDef, chartActivityIndicator, hideLegend, scope.chart.theme);
                        }
                    } else {
                        EChartsManager.disposeEcharts(scope);
                        d3Chart(element.find('.pivot-charts').css('display', ''), scope.chart.def, scope, axesDef, scope.response.result.pivotResponse);
                    }
                };

                const redrawChart = function() {
                    scope.uiDisplayState = scope.uiDisplayState || {};
                    scope.uiDisplayState.displayBrush = false;
                    scope.uiDisplayState.brushData = {};

                    // Make sure the new state of the chart is not built from obsolete settings.
                    ChartDefinitionChangeHandler.clearAllPrivateMembers(scope.chart.def);

                    scope.chart.def.hasEchart = ChartFeatures.hasEChartsDefinition(scope.chart.def.type);
                    scope.chart.def.hasD3 = ChartFeatures.hasD3Definition(scope.chart.def.type);
                    if (ChartFeatures.isEChart(scope.chart.def)) {
                        scope.onChartInit = EChartsManager.onInit(scope, element);
                        scope.onMetaChartInit = EChartsManager.onMetaInit();
                    } else {
                        scope.onChartInit = null;
                        scope.onMetaChartInit = null;
                        EChartsManager.disposeEcharts(scope);
                    }

                    if (!scope.echart && !scope.chart.def.type === CHART_TYPES.ADMINISTRATIVE_MAP) {
                        element.find('.legend-zone').remove();
                    }

                    if (scope.chart.def.$zoomControlInstanceId) {
                        ChartZoomControlAdapter.clear(scope.chart.def.$zoomControlInstanceId);
                        scope.chart.def.$zoomControlInstanceId = null;
                    }

                    const { store, id } = ChartStoreFactory.getOrCreate(scope.chart.def.$chartStoreId);
                    scope.chart.def.$chartStoreId = id;

                    const displayBreadcrumb = ChartFeatures.shouldDisplayBreadcrumb(scope.chart.def)
                        && scope.breadcrumbs && scope.breadcrumbs.some(b => !!b.elements.length)
                        && !scope.noBreadcrumb;
                    if (displayBreadcrumb) {
                        element.find('[data-qa-chart-breadcrumb]').css('display', '');
                    }

                    const rootElement = getRootElement(scope.chart.def);
                    const dims = ChartDimension.getChartDimensions(scope.chart.def);
                    const axesDef = ChartDimension.getAxesDef(scope.chart.def, dims, store);

                    // no rootElement, means we probably have a echarts and font is handle in their part.
                    if (rootElement && scope.chart && scope.chart.theme && scope.chart.theme.generalFormatting && scope.chart.theme.generalFormatting.fontFamily) {
                        element.css('--visualization-font-family', scope.chart.theme.generalFormatting.fontFamily);
                    }

                    switch (scope.chart.def.type) {
                        case CHART_TYPES.GROUPED_COLUMNS: {
                            DKUPivotCharts.GroupedColumnsChart(rootElement, scope.chart.def, scope, axesDef, scope.response.result.pivotResponse);
                            break;
                        }
                        case CHART_TYPES.MULTI_COLUMNS_LINES: {
                            DKUPivotCharts.MultiplotChart(rootElement, scope.chart.def, scope, axesDef, scope.response.result.pivotResponse);
                            break;
                        }
                        case CHART_TYPES.STACKED_COLUMNS: {
                            DKUPivotCharts.StackedColumnsChart(rootElement, scope.chart.def, scope, axesDef, scope.response.result.pivotResponse);
                            break;
                        }
                        case CHART_TYPES.LINES: {
                            DKUPivotCharts.LinesChart(rootElement, scope.chart.def, scope, axesDef, scope.response.result.pivotResponse, scope.getExecutePromise, scope.uiDisplayState, scope.chartActivityIndicator);
                            break;
                        }
                        case CHART_TYPES.STACKED_BARS: {
                            DKUPivotCharts.StackedBarsChart(rootElement, scope.chart.def, scope, axesDef, scope.response.result.pivotResponse);
                            break;
                        }
                        case CHART_TYPES.STACKED_AREA: {
                            DKUPivotCharts.StackedAreaChart(rootElement, scope.chart.def, scope, axesDef, scope.response.result.pivotResponse);
                            break;
                        }
                        case CHART_TYPES.BINNED_XY: {
                            DKUPivotCharts.BinnedXYChart(rootElement, scope.chart.def, scope, axesDef, scope.response.result.pivotResponse);
                            break;
                        }
                        case CHART_TYPES.GROUPED_XY: {
                            DKUPivotCharts.GroupedXYChart(rootElement, scope.chart.def, scope, axesDef, scope.response.result.pivotResponse);
                            break;
                        }
                        case CHART_TYPES.PIE: {
                            loadChart(axesDef, DKUPivotCharts.PieEChartDef, DKUPivotCharts.PieChart, scope.chartActivityIndicator);
                            break;
                        }
                        case CHART_TYPES.LIFT: {
                            DKUPivotCharts.LiftChart(rootElement, scope.chart.def, scope, axesDef, scope.response.result.pivotResponse);
                            break;
                        }
                        case CHART_TYPES.SCATTER: {
                            DKUPivotCharts.ScatterPlotChart(rootElement, scope.chart.def, scope, axesDef, scope.response.result.pivotResponse, scope.uiDisplayState);
                            break;
                        }
                        case CHART_TYPES.SCATTER_MULTIPLE_PAIRS: {
                            DKUPivotCharts.ScatterPlotMultiplePairsChart.create(rootElement, scope.chart.def, scope, axesDef, scope.response.result.pivotResponse, scope.uiDisplayState);
                            break;
                        }
                        case CHART_TYPES.PIVOT_TABLE: {
                            DKUPivotCharts.PivotTableChart(rootElement, scope.chart.def, scope, axesDef, scope.response.result.pivotResponse);
                            break;
                        }
                        case CHART_TYPES.BOXPLOTS: {
                            rootElement.show();
                            DKUPivotCharts.BoxplotsChart(rootElement, scope.chart.def, scope.response.result.pivotResponse, scope, axesDef);
                            break;
                        }
                        case CHART_TYPES.ADMINISTRATIVE_MAP: {
                            DKUPivotCharts.AdministrativeMapChart(rootElement, scope.chart.def, scope.response.result.pivotResponse, scope);
                            break;
                        }
                        case CHART_TYPES.GRID_MAP: {
                            DKUPivotCharts.GridMapChart(rootElement, scope.chart.def, scope.response.result.pivotResponse, scope);
                            break;
                        }
                        case CHART_TYPES.SCATTER_MAP: {
                            DKUPivotCharts.ScatterMapChart(rootElement, scope.chart.def, scope.response.result.pivotResponse, scope);
                            break;
                        }
                        case CHART_TYPES.DENSITY_HEAT_MAP: {
                            DKUPivotCharts.DensityHeatMapChart(rootElement, scope.chart.def, scope.response.result.pivotResponse, scope);
                            break;
                        }
                        case CHART_TYPES.GEOMETRY_MAP: {
                            DKUPivotCharts.GeometryMapChart(rootElement, scope.chart.def, scope.response.result.pivotResponse, scope);
                            break;
                        }
                        case CHART_TYPES.DENSITY_2D: {
                            rootElement.show();
                            DKUPivotCharts.Density2DChart(rootElement.get(0), scope.chart.def, scope.response.result.pivotResponse, scope);
                            break;
                        }
                        case CHART_TYPES.KPI: {
                            rootElement.show();
                            DKUPivotCharts.KpiChart(rootElement.get(0), scope, scope.chart.def, scope.getChartTheme());
                            break;
                        }
                        case CHART_TYPES.RADAR: {
                            loadChart(axesDef, DKUPivotCharts.RadarEChartDef, Fn.NOOP, scope.chartActivityIndicator);
                            break;
                        }
                        case CHART_TYPES.SANKEY: {
                            loadChart(axesDef, DKUPivotCharts.SankeyEChartDef, Fn.NOOP, scope.chartActivityIndicator, true);
                            break;
                        }
                        case CHART_TYPES.GAUGE: {
                            loadChart(axesDef, DKUPivotCharts.GaugeEChartDef, Fn.NOOP, scope.chartActivityIndicator);
                            break;
                        }
                        case CHART_TYPES.WEBAPP: {
                            rootElement.show();
                            DKUPivotCharts.WebappChart(rootElement, scope.chart.def, scope.response.result.pivotResponse, scope);
                            break;
                        }
                        case CHART_TYPES.TREEMAP: {
                            loadChart(axesDef, DKUPivotCharts.TreemapEChartDef, Fn.NOOP, scope.chartActivityIndicator, true);
                            break;
                        }
                        default:
                            throw new Error('Unknown chart type: ' + scope.chart.def.type);
                    }
                };

                function getRootElement(def) {
                    switch (def.type) {
                        case CHART_TYPES.GROUPED_COLUMNS:
                        case CHART_TYPES.MULTI_COLUMNS_LINES:
                        case CHART_TYPES.STACKED_COLUMNS:
                        case CHART_TYPES.LINES:
                        case CHART_TYPES.STACKED_BARS:
                        case CHART_TYPES.STACKED_AREA:
                        case CHART_TYPES.BINNED_XY:
                        case CHART_TYPES.GROUPED_XY:
                        case CHART_TYPES.LIFT:
                        case CHART_TYPES.SCATTER:
                        case CHART_TYPES.SCATTER_MULTIPLE_PAIRS:
                        case CHART_TYPES.ADMINISTRATIVE_MAP:
                        case CHART_TYPES.GRID_MAP:
                        case CHART_TYPES.SCATTER_MAP:
                        case CHART_TYPES.DENSITY_HEAT_MAP:
                        case CHART_TYPES.GEOMETRY_MAP:
                            return element.find('.pivot-charts').css('display', '');
                        case CHART_TYPES.PIE:
                        case CHART_TYPES.RADAR:
                        case CHART_TYPES.SANKEY:
                        case CHART_TYPES.GAUGE:
                        case CHART_TYPES.TREEMAP:
                            return element;
                        case CHART_TYPES.PIVOT_TABLE:
                            return element.find('.pivot-table-container').css('display', '');
                        case CHART_TYPES.BOXPLOTS:
                            return element.find('.boxplots-container');
                        case CHART_TYPES.DENSITY_2D:
                            return element.find('.direct-svg');
                        case CHART_TYPES.KPI:
                            return element.find('.kpi-container');
                        case CHART_TYPES.WEBAPP:
                            return element.find('.webapp-charts-container');
                        default:
                            throw new Error('Unknown chart type: ' + scope.chart.def.type);
                    }
                }

                function subscribeThumbnailToRedraw() {
                    if (scope.abortThumbnail) {
                        scope.abortThumbnail();
                    }
                    const previousLoadedCallback = scope.loadedCallback;
                    /*
                     * create an AbortController so we can cancel the promise and
                     * avoid unnecessary thumbnail computation
                     */
                    const controller = new AbortController();
                    scope.abortThumbnail = () => {
                        controller.abort();
                        scope.loadedCallback = previousLoadedCallback;
                    };
                    scope.loadedCallback = () => {
                        window.requestAnimationFrame(() => scope.updateThumbnail(controller.signal));
                        scope.loadedCallback = previousLoadedCallback;
                        if (typeof (previousLoadedCallback) === 'function') {
                            previousLoadedCallback();
                        }
                    };
                }

                function redraw(options = {}) {
                    if (!scope.response || !scope.response.hasResult || !scope.isInitialDrawReady) {
                        return;
                    }

                    element.children().children().each(function() {
                        if (this.tagName !== 'ACTIVITY-INDICATOR') {
                            $(this).hide();
                        }
                    });

                    if (scope.validity && !scope.validity.valid) {
                        scope.validity.valid = true;
                    }

                    // for debug
                    element.attr('chart-type', scope.chart.def.type);

                    if (scope.timing) {
                        scope.timing.drawStart = new Date().getTime();
                    }

                    try {
                        Logger.info('Start draw chart', scope.chart.def.type);
                        if (options.updateThumbnail) {
                            subscribeThumbnailToRedraw();
                        }
                        redrawChart();
                    } catch (err) {
                        if (err instanceof ChartIAE) {
                            Logger.warn('CHART IAE', err);
                            if (scope.validity) {
                                scope.validity.valid = false;
                                scope.validity.type = 'DRAW_ERROR';
                                scope.validity.message = err.message;
                            }
                        } else {
                            throw err;
                        }
                    }
                }

                function getChartElementThumbnail() {
                    return $q((resolve) => {
                        let setup;
                        let canvas;

                        if (!scope.isEChart(scope.chart.def)) {
                            const canvasParent = document.querySelector('foreignObject');
                            let margins;

                            switch (scope.chart.def.type) {
                                case CHART_TYPES.BOXPLOTS:
                                    return scope.exportBoxPlots(true).then(canvas => resolve({ canvas: CanvasUtils.resize(canvas, THUMBNAIL_WIDTH, THUMBNAIL_HEIGHT) }));
                                case CHART_TYPES.SCATTER:
                                case CHART_TYPES.SCATTER_MULTIPLE_PAIRS:
                                    margins = { x: canvasParent?.getAttribute('x'), y: canvasParent?.getAttribute('y') };
                                    return resolve({
                                        canvas: CanvasUtils.resize(
                                            document.querySelector('.chart-svg canvas'),
                                            THUMBNAIL_WIDTH,
                                            THUMBNAIL_HEIGHT,
                                            false,
                                            document.querySelectorAll('.chart-svg .reference-line'),
                                            margins
                                        )
                                    });
                                case CHART_TYPES.GEOMETRY_MAP:
                                case CHART_TYPES.SCATTER_MAP:
                                case CHART_TYPES.ADMINISTRATIVE_MAP:
                                case CHART_TYPES.GRID_MAP:
                                    return resolve({ canvas: CanvasUtils.resize(document.querySelector('canvas.leaflet-zoom-animated'), THUMBNAIL_WIDTH, THUMBNAIL_HEIGHT) });
                                case CHART_TYPES.DENSITY_2D:
                                    return resolve({ svg: document.querySelector('svg.direct-svg') });
                                case CHART_TYPES.LIFT:
                                case CHART_TYPES.LINES:
                                case CHART_TYPES.MULTI_COLUMNS_LINES:
                                    // Thicker strokes
                                    setup = clonedSvg =>
                                        clonedSvg.querySelectorAll('path.visible').forEach(line =>
                                            line.setAttribute('stroke-width', line.getAttribute('stroke-width') * 3));
                                // falls through
                                default:
                                    canvas = document.querySelector('canvas');

                                    if (canvas) {
                                        return resolve({ canvas: CanvasUtils.resize(canvas, THUMBNAIL_WIDTH, THUMBNAIL_HEIGHT) });
                                    } else {
                                        return resolve({ svg: document.querySelector('svg.chart-svg'), setup });
                                    }
                            }
                        } else if (scope.echart && scope.echart.echartDefInstance){
                            const thumbnailOptions = scope.echart.echartDefInstance.getThumbnailOptions(scope.echart.options);
                            const canvasURL = EChartsManager.getThumbnailCanvasUrl(thumbnailOptions, THUMBNAIL_WIDTH, THUMBNAIL_HEIGHT);
                            return resolve({ canvasURL });
                        }
                    });
                }

                const isTooSmall = () => {
                    const chart = document.querySelector('.chart-wrapper') || document.querySelector('.pivot-chart');
                    if (chart) {
                        const ratio = chart.clientHeight / chart.clientWidth;
                        return chart.clientWidth < 150 || chart.clientHeight < 150 || ratio < 0.2 || ratio > 1.8;
                    }
                    return false;
                };

                /*
                 * If we are using webgl in order for toDataURL to return something
                 * we need to be in the drawing function and have nothing async
                 */
                scope.getThumbnailForWebgl = origCanvas => {
                    if (ChartFeatures.canHaveThumbnail(scope.chart.def) && !scope.noThumbnail) {
                        if (isTooSmall()) {
                            return;
                        }

                        return CanvasUtils.resize(origCanvas, THUMBNAIL_WIDTH, THUMBNAIL_HEIGHT).toDataURL();
                    }
                };

                scope.updateThumbnailWithData = dataURL => {
                    if (dataURL) {
                        scope.chart.def.thumbnailData = dataURL;
                    } else {
                        delete scope.chart.def.thumbnailData;
                    }
                };

                scope.updateThumbnail = (signal) => $q(resolve => {
                    if (ChartFeatures.canHaveThumbnail(scope.chart.def) && !scope.noThumbnail) {
                        Logger.info('Computing thumbnail');

                        const isAborted = () => signal && signal.aborted;

                        if (isAborted() || isTooSmall()) {
                            return resolve();
                        }

                        getChartElementThumbnail()
                            .then(({ canvas, svg, setup, canvasURL }) => {
                                if (svg && !isAborted()) {
                                    return CanvasUtils.svgToCanvas(svg, THUMBNAIL_WIDTH, THUMBNAIL_HEIGHT, { filters: ['text', '.axis', '.hlines', '.vlines', '.legend', '.background'], setup });
                                // ECharts charts are all using canvas. But so is our pure canvas scatter, that does not require canvasURL but canvas so it must go to the final return
                                } else if (canvasURL && (scope.chart.def.displayWithECharts || scope.chart.def.displayWithEChartsByDefault)) {
                                    scope.updateThumbnailWithData(canvasURL);
                                    return;
                                }
                                return canvas;
                            })
                            .then(canvas => {
                                if (canvas && !isAborted()) {
                                    scope.chart.def.thumbnailData = canvas.toDataURL();
                                    Logger.info('Thumbnail Done');
                                }
                                resolve();
                            });
                    } else {
                        delete scope.chart.def.thumbnailData;
                        resolve();
                    }
                });

                scope.$on('resize', () => redraw());
                scope.$on('redraw', (e, opts) => redraw(opts));

                scope.$on('export-chart', function() {
                    scope.export();
                });

                scope.$on('$destroy', function() {
                    if (scope.chart.def.$zoomControlInstanceId) {
                        ChartZoomControlAdapter.clear(scope.chart.def.$zoomControlInstanceId);
                        scope.chart.def.$zoomControlInstanceId = null;
                    }
                });

                scope.export = function() {
                    if (scope.chart.def.type === CHART_TYPES.BOXPLOTS) {
                        scope.exportBoxPlots().then(function(canvas) {
                            CanvasUtils.downloadCanvas(canvas, scope.chart.def.name + '.png');
                        });
                        return;
                    }

                    let width;
                    let height;
                    let $svg;

                    if (scope.chart.def.type === CHART_TYPES.DENSITY_2D) {
                        $svg = element.find('svg.direct-svg');
                    } else {
                        $svg = element.find('svg.chart-svg');
                    }

                    if ($svg.length) {
                        width = $svg.width();
                        height = $svg.height();
                    } else {
                        const echart = element.find('div.main-echarts-zone');
                        width = echart.width();
                        height = echart.height();
                    }
                    scope.exportData(width, height).then(function(canvas) {
                        CanvasUtils.downloadCanvas(canvas, scope.chart.def.name + '.png');
                    });
                };

                /**
                 * @returns A style element containing all the CSS rules relative to charts
                 */
                function getChartStyleRules(forCanvg) {
                    const svgNS = 'http://www.w3.org/2000/svg';
                    const style = document.createElementNS(svgNS, 'style');
                    for (let i = 0; i < document.styleSheets.length; i++) {
                        const str = document.styleSheets[i].href;
                        if (str != null && str.substr(str.length - 10) === 'charts.css') {
                            const rules = document.styleSheets[i].cssRules;
                            for (let j = 0; j < rules.length; j++) {
                                style.textContent += (rules[j].cssText);
                                style.textContent += '\n';
                            }
                            break;
                        }
                    }
                    if (forCanvg) {
                        // Yes it's ugly
                        style.textContent = `<![CDATA[ .totallyFakeClassBecauseCanvgParserIsBuggy  {}\n${style.textContent}{]]>`;
                        // "{" is here to workaround CanVG parser brokenness
                    }
                    return style;
                }

                /**
                 * Add the passed title to the passed canvas
                 * @param canvas: canvas that we want to add a title to
                 * @param title: title that will be added to the canvas
                 * @params scale: the scaling coefficient that we want to apply to the title
                 */
                function addTitleToCanvas(canvas, title, titleHeight, scale) {
                    const ctx = canvas.getContext('2d');
                    ctx.textAlign = 'center';
                    ctx.textBaseline = 'middle';
                    ctx.font = 'normal normal 100 ' + 18 * scale + 'px sans-serif';
                    ctx.fillStyle = '#777';
                    ctx.fillText(title, canvas.width / 2, titleHeight * scale / 2);
                }

                /**
                 * Compute a multiplier coefficient enabling to scale passed dimensions to reach an image containing the same amount of pixels as in a 720p image.
                 * @param w: width of original image that we'd like to scale to HD
                 * @param h: height of original image that we'd like to scale to HD
                 * @returns c so that (w * c) * (h * c) = 921600, the number of pixels contained in a 720p image
                 */
                function getCoeffToHD(w, h) {
                    const nbPixelsHD = 921600; // nb pixels contained in a 720p image
                    const multiplier = Math.sqrt(nbPixelsHD / (w * h)); // so that (w * multiplier) * (h * multiplier) = nbPixelsHD
                    return multiplier;
                }


                scope.exportData = function(w, h, simplified, svgEl, noTitle) {
                    /**
                     * @returns a canvas that fits the passed dimensions
                     */
                    function generateCanvas(w, h) {
                        const canvas = document.createElement('canvas');
                        canvas.setAttribute('width', w);
                        canvas.setAttribute('height', h);
                        return canvas;
                    }

                    /**
                     * @returns the svg that contains the chart.
                     */
                    function getChartSVG() {
                        let svg;
                        if (angular.isDefined(svgEl)) {
                            svg = svgEl.get(0);
                        } else if (scope.chart.def.type === CHART_TYPES.DENSITY_2D) {
                            svg = element.find('svg.direct-svg').get(0);
                        } else {
                            svg = element.find('svg.chart-svg').get(0);
                        }
                        return svg;
                    }

                    /**
                     * Adapted from https://code.google.com/p/canvg/issues/detail?id=143
                     * @param svg: the SVG to get cloned
                     * @returns a clone of the passed SVG
                     */
                    function cloneSVG(svg) {
                        const clonedSVG = svg.cloneNode(true);
                        const $clonedSVG = $(clonedSVG);
                        const $svg = $(svg);
                        $clonedSVG.width($svg.width());
                        $clonedSVG.height($svg.height());
                        const customFontFamily = getComputedStyle(svg).getPropertyValue('--visualization-font-family');
                        if (customFontFamily) {
                            $clonedSVG.css('font-family', `${customFontFamily}, 'SourceSansPro'`);
                        }
                        return clonedSVG;
                    }

                    function fillOldCanvasInNewCanvas(oldCanvas, newCanvas, horizontalOffset, verticalOffset) {

                        const context = newCanvas.getContext('2d');

                        const oldWidth = oldCanvas.width;
                        const oldHeight = oldCanvas.height;
                        const proportion = oldWidth / oldHeight;

                        const availableWidth = newCanvas.width - horizontalOffset;
                        const availableHeight = newCanvas.height - verticalOffset;

                        // scale image using proportionally smallest measurement
                        const shouldScaleWidth = availableWidth / oldWidth > availableHeight / oldHeight;
                        const newWidth = shouldScaleWidth ? availableHeight * proportion : availableWidth;
                        const newHeight = shouldScaleWidth ? availableHeight : availableWidth / proportion;

                        // apply the old canvas to the new one
                        context.drawImage(oldCanvas, 0, 0, oldWidth, oldHeight, horizontalOffset, verticalOffset, newWidth, newHeight);

                        // return the new canvas
                        return newCanvas;
                    }

                    /**
                     * Looks for a canvas hosted in a foreignObject element in the passed svg, scale it, and add it to the passed canvas
                     * @params svg: the svg that might contain a canvas in a foreignObject
                     * @param canvas: the canvas that we want to add the scatter canvas to
                     * @params scale: the scaling coefficient that we want to apply to the scatter canvas
                     */
                    function addInnerCanvasToCanvas(svg, canvas, scale, horizontalOffset, verticalOffset) {
                        const $svg = $(svg);
                        const $foreignObject = $svg.find('foreignObject'),
                            x = parseFloat($foreignObject.attr('x')),
                            y = parseFloat($foreignObject.attr('y')),
                            width = parseFloat($foreignObject.attr('width')),
                            height = parseFloat($foreignObject.attr('height'));
                        const origCanvas = $foreignObject.find('.canvas-export').get(0);
                        canvas.getContext('2d').drawImage(origCanvas, (x + horizontalOffset) * scale, (y + verticalOffset) * scale, width * scale, height * scale);
                    }

                    /**
                     * @param chartElement: canvas or svg which contains the original chart
                     * @param canvas: the canvas that we want to add the legend to
                     * @params scale: the scaling coefficient that we want to apply to the DOM's legend
                     * @returns A promise that will resolve when the legend is added to the canvas
                     */
                    function addLegendToCanvas(chartElement, canvas, scale, verticalOffset) {
                        const d = $q.defer();
                        const $legendDiv = element.find('.legend-zone');
                        if ($legendDiv.size() === 0 || $legendDiv.width() === 0) {
                            d.resolve();
                        } else {
                            const legendOffset = $legendDiv.offset();
                            const wrapperOffset = element.offset();
                            const legendX = legendOffset.left - wrapperOffset.left;
                            const legendY = legendOffset.top - wrapperOffset.top;
                            const legendHeight = $legendDiv[0].clientHeight;
                            let chartHorizontalOffset = legendX * scale;
                            let chartVerticalOffset = legendY * scale;

                            switch(scope.chart.def.legendPlacement) {
                                case 'OUTER_RIGHT':
                                    chartHorizontalOffset = chartElement.clientWidth * scale;
                                    //  To avoid legend and title overlap, create a offset, as the title is centered, we need to divide the height by 2.
                                    chartVerticalOffset = verticalOffset * scale / 2;
                                    break;
                                case 'OUTER_BOTTOM':
                                    //  Consider chart and title height
                                    chartVerticalOffset = (chartElement.clientHeight + verticalOffset) * scale;
                                    break;
                                case 'OUTER_LEFT':
                                    chartVerticalOffset = verticalOffset * scale / 2;
                                    break;
                                case 'OUTER_TOP':
                                    //  We remove legendHeight to get the title height, also, as the title is centered, we need to divide the total height by 2 to center the legend too.
                                    chartVerticalOffset = (verticalOffset + legendHeight) * scale / 2;
                                    break;
                                case 'INNER_BOTTOM_LEFT':
                                    chartVerticalOffset = ((legendY + (verticalOffset * scale / 2)) * scale);
                                    break;
                                case 'INNER_BOTTOM_RIGHT':
                                    chartVerticalOffset = ((legendY + (verticalOffset * scale / 2)) * scale);
                                    break;
                                case 'INNER_TOP_LEFT':
                                    chartVerticalOffset = ((legendY + (verticalOffset * scale / 2)) * scale);
                                    break;
                                case 'INNER_TOP_RIGHT':
                                    chartVerticalOffset = ((legendY + (verticalOffset * scale / 2)) * scale);
                                    break;
                            }

                            CanvasUtils.htmlToCanvas($legendDiv, scale).then(function(legendCanvas) {
                                canvas.getContext('2d').drawImage(legendCanvas, chartHorizontalOffset, chartVerticalOffset, legendCanvas.width, legendCanvas.height);
                                d.resolve();
                            });
                        }
                        return d.promise;
                    }

                    // -- BEGINNING OF FUNCTION --

                    const deferred = $q.defer();
                    const chartTitle = simplified ? false : scope.chart.def.name;
                    const titleHeight = 52;

                    const horizontalOffset = 0;
                    const verticalOffset = chartTitle ? titleHeight : 0;
                    let verticalOffsetWithLegends = verticalOffset;
                    let horizontalOffsetWithLegends = horizontalOffset;

                    let legendWidth = 0;
                    let legendHeight = 0;

                    const hasLegend = !simplified && scope.chart.def.legendPlacement !== 'SIDEBAR' && ChartFeatures.canDisplayLegend(scope.chart.def.type);

                    if (hasLegend) {
                        const $legendDiv = element.find('.legend-zone');
                        switch(scope.chart.def.legendPlacement) {
                            case 'OUTER_LEFT':
                                legendWidth = $legendDiv[0].clientWidth;
                                horizontalOffsetWithLegends += legendWidth;
                                break;
                            case 'OUTER_RIGHT':
                                //  Otherwise, it's set apart to be added to the canvas' width only
                                legendWidth = $legendDiv[0].clientWidth;
                                break;
                            case 'OUTER_TOP':
                                legendHeight = $legendDiv[0].clientHeight;
                                verticalOffsetWithLegends += legendHeight;
                                break;
                            case 'OUTER_BOTTOM':
                                //  Otherwise, it's set apart to be added to the canvas' height only
                                legendHeight = $legendDiv[0].clientHeight;
                                break;
                        }
                    }

                    const isScatter = scope.chart.def.type === CHART_TYPES.SCATTER || scope.chart.def.type === CHART_TYPES.SCATTER_MULTIPLE_PAIRS;
                    let referenceLines;
                    const dimensions = { w: w + horizontalOffset + legendWidth + 5, h: h + verticalOffset + legendHeight + 5 };
                    // Creating a HD canvas that will "receive" the svg element
                    const scale = getCoeffToHD(dimensions.w, dimensions.h);
                    let canvas = generateCanvas(dimensions.w * scale, dimensions.h * scale);

                    if (!simplified) {
                        CanvasUtils.fill(canvas, 'white');
                    }

                    // Getting a clone SVG to inject in the canvas
                    const svg = getChartSVG();
                    const oldCanvas = document.getElementsByTagName('canvas')[0];
                    const clonedSVG = svg ? cloneSVG(svg) : undefined;

                    if (!svg) {
                        // Likely echart canvas
                        canvas = fillOldCanvasInNewCanvas(oldCanvas, canvas, horizontalOffsetWithLegends * scale, verticalOffsetWithLegends * scale);
                    } else {
                        clonedSVG.insertBefore(getChartStyleRules(true), clonedSVG.firstChild); //adding css rules

                        // For scatter, we remove the references lines because they should be draw on top on the points
                        if (isScatter) {
                            referenceLines = d3.select(clonedSVG).selectAll('.reference-line');
                            referenceLines.remove();
                        }
                        clonedSVG.setAttribute('transform', 'scale(' + scale + ')'); // scaling the svg samely as we scaled the canvas
                        if (simplified) {
                            d3.select(clonedSVG).selectAll('text').remove();
                            d3.select(clonedSVG).selectAll('.axis').remove();
                            d3.select(clonedSVG).selectAll('.hlines').remove();
                            d3.select(clonedSVG).selectAll('.vlines').remove();
                            d3.select(clonedSVG).selectAll('.legend').remove();
                        }
                        // Filling the canvas element that we created with the svg
                        const svgText = new XMLSerializer().serializeToString(clonedSVG);
                        canvg(canvas, svgText, { offsetY: verticalOffsetWithLegends, offsetX: horizontalOffsetWithLegends, ignoreDimensions: true, ignoreClear: true, renderCallback: function() {
                            $timeout(canvas.svg.stop);
                        } });
                    }


                    // In the case of scatter chart, the all chart content is already a canvas hosted in a foreignObject. Yet canvg doesn't handle foreignObjects, we'll manually copy the scatter canvas in the canvg canvas
                    if (isScatter && svg) {
                        addInnerCanvasToCanvas(svg, canvas, scale, horizontalOffsetWithLegends, verticalOffsetWithLegends);
                        // Put back only the reference lines, trash everything else.
                        if (clonedSVG && referenceLines && referenceLines.size() > 0) {
                            d3.select(clonedSVG).selectAll('g.chart').remove();
                            d3.select(clonedSVG).selectAll('foreignObject').remove();
                            referenceLines.forEach(referenceLine => clonedSVG.appendChild(referenceLine[0]));
                            const svgText = new XMLSerializer().serializeToString(clonedSVG);
                            canvg(canvas, svgText, { offsetY: verticalOffsetWithLegends, offsetX: horizontalOffsetWithLegends, ignoreDimensions: true, ignoreClear: true, renderCallback: function() {
                                $timeout(canvas.svg.stop);
                            } });
                        }
                    }

                    // Adding chart's title
                    if (chartTitle && !noTitle) {
                        addTitleToCanvas(canvas, chartTitle, titleHeight, scale);
                    }

                    // Adding chart's legend
                    if (hasLegend) {
                        addLegendToCanvas(svg || oldCanvas, canvas, scale, verticalOffset).then(() => deferred.resolve(canvas));
                    } else {
                        deferred.resolve(canvas);
                    }

                    return deferred.promise;
                };

                scope.exportBoxPlots = (thumbnail) => $q(resolve => {
                    const container = element.find('.boxplots-container');
                    const svg1 = container.find('svg.noflex');
                    const svg2 = container.find('div.flex.oa > svg');
                    const title = scope.chart.def.name;
                    const titleHeight = 52;
                    let verticalOffset = title ? titleHeight : 0;
                    const options = {
                        setup: (cloneSvg) => {
                            if (thumbnail) {
                                // Thicker strokes
                                cloneSvg.style.strokeWidth = 6;
                            } else {
                                cloneSvg.insertBefore(getChartStyleRules(), cloneSvg.firstChild);
                            }
                        },
                        filters: thumbnail ? ['.axis', '.hline'] : null
                    };

                    const allCanvas = [CanvasUtils.svgToCanvas(svg1.get(0), svg1.width(), svg1.height(), options)];
                    if (scope.chart.def.boxplotBreakdownDim.length || scope.chart.def.boxplotBreakdownHierarchyDimension.length) {
                        allCanvas.push(CanvasUtils.svgToCanvas(svg2.get(0), svg2.width(), svg2.height(), options));
                    }
                    $q.all(allCanvas).then(subcanvas => {
                        const canvas = document.createElement('canvas');
                        let horizontalOffset = 0;
                        let legendWidth = 0;
                        let legendHeight = 0;
                        const legendOffset = { top: 0, left: 0 };
                        const hasLegend = scope.chart.def.legendPlacement !== 'SIDEBAR' && ChartFeatures.canDisplayLegend(scope.chart.def.type) && !thumbnail;
                        function getCanvasProps(canvasToCheck, prop) {
                            if (canvasToCheck == null) {
                                return 0;
                            }
                            return canvasToCheck[prop];
                        }
                        if (hasLegend) {
                            const $legendDiv = container.find('.legend-zone');
                            switch(scope.chart.def.legendPlacement) {
                                case 'OUTER_LEFT':
                                    //  We add the legend width to the offset directly
                                    horizontalOffset += $legendDiv[0].clientWidth;
                                    legendOffset.top = verticalOffset;
                                    break;
                                case 'OUTER_RIGHT':
                                    //  Otherwise, it's set apart to be added to the canvas' width only
                                    legendWidth = $legendDiv[0].clientWidth;
                                    legendOffset.left = subcanvas[0].width + getCanvasProps(subcanvas[1], 'width');
                                    legendOffset.top = verticalOffset;
                                    break;
                                case 'OUTER_TOP':
                                    legendOffset.top = verticalOffset;
                                    //  We add the legend height to the offset directly
                                    verticalOffset += $legendDiv[0].clientHeight;
                                    break;
                                case 'OUTER_BOTTOM':
                                    //  Otherwise, it's set apart to be added to the canvas' height only
                                    legendHeight = $legendDiv[0].clientHeight;
                                    legendOffset.top = subcanvas[0].height + verticalOffset;
                                    break;
                                case 'INNER_BOTTOM_LEFT':
                                    legendOffset.top = subcanvas[0].height - $legendDiv[0].clientHeight;
                                    legendOffset.left = 30;
                                    break;
                                case 'INNER_BOTTOM_RIGHT':
                                    legendOffset.top = subcanvas[0].height - $legendDiv[0].clientHeight;;
                                    legendOffset.left = subcanvas[0].width + getCanvasProps(subcanvas[1], 'width') - $legendDiv[0].clientWidth;
                                    break;
                                case 'INNER_TOP_RIGHT':
                                    legendOffset.top = verticalOffset;
                                    legendOffset.left = subcanvas[0].width + getCanvasProps(subcanvas[1], 'width') - $legendDiv[0].clientWidth;
                                    break;
                                case 'INNER_TOP_LEFT':
                                    legendOffset.top = verticalOffset;
                                    legendOffset.left = 30;
                                    break;
                            }
                        }
                        const dimensions = { w: subcanvas[0].width + getCanvasProps(subcanvas[1], 'width') + horizontalOffset + legendWidth, h: subcanvas[0].height + verticalOffset + legendHeight };
                        const scale = getCoeffToHD(dimensions.w, dimensions.h);
                        canvas.setAttribute('width', dimensions.w);
                        canvas.setAttribute('height', dimensions.h);
                        if (!thumbnail) {
                            CanvasUtils.fill(canvas, 'white');
                        }
                        const ctx = canvas.getContext('2d');
                        ctx.drawImage(subcanvas[0], horizontalOffset, verticalOffset);
                        // 14 is a magic value to glue the first and second canvas to avoid cut in h lines
                        if (subcanvas.length > 1) {
                            ctx.drawImage(subcanvas[1], subcanvas[0].width + horizontalOffset - 14, verticalOffset + scale);
                        }

                        if (title && !thumbnail) {
                            addTitleToCanvas(canvas, title, titleHeight, 1);
                        }

                        if (hasLegend) {
                            CanvasUtils.htmlToCanvas(container.find('.legend-zone'), 1).then(function(legendCanvas) {
                                canvas.getContext('2d').drawImage(legendCanvas, legendOffset.left, legendOffset.top, legendCanvas.width, legendCanvas.height);
                                resolve(canvas);
                            });
                        } else {
                            resolve(canvas);
                        }
                    });
                });

                scope.$watch('response', function(nv) {
                    if (nv == null) {
                        return;
                    }
                    if (!scope.response.hasResult) {
                        return;
                    }
                    $timeout(() => {
                        scope.isInitialDrawReady = true;
                        redraw({ updateThumbnail: true });
                    });
                });
            }
        };
    });
})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.directives.simple_report');

    app.directive('selectAll', function() {
        return {
          restrict: 'A',
          link: function(scope, element) {
            element.on('keydown', function(event) {
              // Check if Ctrl+A is pressed
              if ((event.ctrlKey || event.metaKey) && event.which === 65) { // ASCII code for 'A'
                event.preventDefault();
                if (event.target) {
                  event.target.select();
                }                
                scope.$apply(); // Update the AngularJS scope if necessary
              }
            });
            
            // Clean up the event listener when the directive is destroyed
            scope.$on('$destroy', function() {
              element.off('keydown');
            });
          }
        };
      });
})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.charts');

    /**
     * This filters is intended to improve readability in arrays containing large numbers (such as status display)
     * - a thousand separator is introduce to make differences between large integers, such as counters, more distinguishable.
     * - the thousand separator used is short space, to make it more neutral across regions than a US style comma
     * - the number of digits displayed is controlled by lowering accordingly the number decimals (maximum 9, with a progressive reduction to a minimum of 2).
     *
     * (!) This filter previously was in static/dataiku/js/simple_report/chart_view_commons.js
     *
     * Used for metrics and probs
     */
    app.filter('longReadableNumber', function(NumberFormatter) {
        return NumberFormatter.longReadableNumberFilter();
    });

    /**
     * This method is now defined in number-formatter.service.ts
     */
    app.filter('longSmartNumber', function(NumberFormatter) {
        return NumberFormatter.longSmartNumberFilter();
    });

    /**
     * (!) This filter previously was in static/dataiku/js/simple_report/chart_view_commons.js
     * This method is now defined in number-formatter.service.ts
     */
    app.filter('smartNumber', function(NumberFormatter) {
        return NumberFormatter.smartNumberFilter();
    });

    /**
     * (!) This filter previously was in static/dataiku/js/simple_report/chart_view_commons.js
     * This method is now defined in number-formatter.service.ts
     */
    app.filter('percentageNumber', (NumberFormatter) => {
        return NumberFormatter.percentageNumberFilter();
    });
})();

;
(function() {
    'use strict';

    angular.module('dataiku.charts')
        .factory('AnimatedChartsUtils', AnimatedChartsUtils);

    /**
     * (!) This service previously was in static/dataiku/js/simple_report/common/animation.js
     */
    function AnimatedChartsUtils($interval, ChartFormatting) {
        const unwatchers = {},
            intervals = {};

        return {
            /**
             * Setup chartHandler.animation (used by the animation widget) for the given chart
             * @param {$scope} chartHandler
             * @param {ChartTensorDataWrapper} chartData
             * @param {ChartDef} chartDef
             * @param {function} drawFrame: drawing callback
             */
            initAnimation: function(chartHandler, chartData, chartDef, drawFrame, ignoreLabels = new Set()) {
                if (unwatchers[chartHandler.$id]) {
                    unwatchers[chartHandler.$id]();
                    delete unwatchers[chartHandler.$id];
                }

                if (intervals[chartHandler.$id]) {
                    $interval.cancel(intervals[chartHandler.$id]);
                    delete intervals[chartHandler.$id];
                }

                const animation = chartHandler.animation;

                animation.labelify = function(label) {
                    return ChartFormatting.getForOrdinalAxis(label);
                };
                animation.labels = chartData.getAxisLabels('animation', ignoreLabels);

                animation.playing = false;

                animation.drawFrame = function(frameIdx) {
                    /*
                     * Put back currentFrame to zero when necessary. For instance, after unselecting
                     * "Group extra values in a 'Others' category".
                     */
                    animation.currentFrame = animation.currentFrame >= animation.labels.length ? 0 : frameIdx;
                };
                animation.chartData = chartData;

                animation.hasNext = function() {
                    return animation.currentFrame < animation.labels.length - 1;
                };

                animation.play = function() {
                    if (animation.playing) {
                        return;
                    }

                    if (animation.currentFrame === animation.labels.length - 1) {
                        animation.currentFrame = 0;
                    }
                    animation.playing = true;
                    intervals[chartHandler.$id] = $interval(function() {
                        animation.drawFrame((animation.currentFrame + 1) % animation.labels.length);
                        if (!chartDef.animationRepeat && !animation.hasNext()) {
                            animation.pause();
                        }
                    }, (chartDef.animationFrameDuration || 3000));
                };

                animation.dimension = chartDef.animationDimension[0];

                animation.pause = function() {
                    animation.playing = false;
                    $interval.cancel(intervals[chartHandler.$id]);
                };

                unwatchers[chartHandler.$id] = chartHandler.$watch('animation.currentFrame', function(nv) {
                    if (nv == null) {
                        return;
                    }
                    drawFrame(animation.labels[nv].$tensorIndex);
                });

                chartHandler.$watch('chart.def.animationFrameDuration', function(nv) {
                    if (!nv) {
                        return;
                    }
                    if (animation.playing) {
                        animation.pause();
                        animation.play();
                    }
                });
            },

            unregisterAnimation: function(chartHandler) {
                if (unwatchers[chartHandler.$id]) {
                    unwatchers[chartHandler.$id]();
                    delete unwatchers[chartHandler.$id];
                }

                if (intervals[chartHandler.$id]) {
                    $interval.cancel(intervals[chartHandler.$id]);
                    delete intervals[chartHandler.$id];
                }
            },

            getAnimationCoord(animation){
                return animation.labels ? animation.labels[animation.currentFrame].$tensorIndex : animation.currentFrame;
            },

            getAnimationContext: function(animationDimension, animation) {
                return {
                    animation: animationDimension && animation.currentFrame || 0,
                    numberOfFrames: animationDimension && animation.labels.length || 1
                };
            }
        };
    }

})();

;
// @ts-check
(function() {
    'use strict';

    // @ts-ignore
    angular.module('dataiku.directives.simple_report').factory('ChartAxesUtils', chartAxesUtils);

    /**
     * ChartAxesUtils service
     * Utils to compute chart axes (agnostic, used for both d3 and echarts)
     * (!) This service previously was in static/dataiku/js/simple_report/common/axes.js
     * @typedef {import("../../../../../../../../../server/src/frontend/src/app/features/simple-report/services/chart-axis/chart-axis.type").ChartAxesUtilsReturnType} ChartAxesUtilsReturnType
     * @returns {ChartAxesUtilsReturnType}
     */
    function chartAxesUtils(ChartDataUtils, ChartUADimension, ChartDimension, ChartsStaticData, ChartFeatures, ChartYAxisPosition, AxisTicksConfigMode, AxisTicksFormatting, AxisTitleFormatting, ColumnAvailability, translate, CHART_VARIANTS) {

        const AUTO_EXTENT_MODE = 'AUTO';
        const MANUAL_EXTENT_MODE = 'MANUAL';

        function getManualExtentAtIndex(customExtent, index) {
            if (customExtent == null || customExtent.$autoExtent === undefined || (index < 0 && index > 1)) {
                return null;
            } // extent index can only be 0 or 1
            return customExtent.manualExtent[index] === null ? customExtent.$autoExtent[index] : customExtent.manualExtent[index];
        }

        function getManualExtentMin(customExtent) {
            return getManualExtentAtIndex(customExtent, 0);
        }

        function getManualExtentMax(customExtent) {
            return getManualExtentAtIndex(customExtent, 1);
        }

        /**
         * Get manual extent to apply (non-null) from customExtent
         * @param customExtent
         * @returns {[number, number]} [min, max]
         */
        function getManualExtent(customExtent) {
            return [getManualExtentMin(customExtent), getManualExtentMax(customExtent)];
        }

        /**
         * Checks if the user defined custom extent crops the initial extent
         * @param axesFormatting
         * @returns Returns True if initial extent is cropped, False otherwise.
         */
        const isCroppedExtent = axesFormatting => {
            const isCropped = axesFormatting.some(formatting => {
                if (formatting.customExtent.$autoExtent && formatting.customExtent.editMode === MANUAL_EXTENT_MODE) {
                    const manualExtent = getManualExtent(formatting.customExtent);
                    return formatting.customExtent.$autoExtent[0] < manualExtent[0] || formatting.customExtent.$autoExtent[1] > manualExtent[1];
                }
            });
            return isCropped;

        };

        /** @type {ChartAxesUtilsReturnType} */
        const svc = {
            shouldIncludeZero(chartDef, yAxisId) {
                const yAxisFormatting = this.getFormattingForYAxis(chartDef.yAxesFormatting, yAxisId);
                return ChartFeatures.canIncludeZero(chartDef.type) && yAxisFormatting && !svc.isManualMode(yAxisFormatting.customExtent) && yAxisFormatting.includeZero;
            },

            isNumerical(axisSpec) {
                return axisSpec.dimension && axisSpec.dimension.isA === 'ua' ? ChartUADimension.isTrueNumerical(axisSpec.dimension) : ChartDimension.isTrueNumerical(axisSpec.dimension);
            },

            isContinuousDate(axisSpec) {
                return axisSpec.dimension && axisSpec.dimension.isA === 'ua' ? ChartUADimension.isDate(axisSpec.dimension) : ChartDimension.isTimeline(axisSpec.dimension);
            },

            /**
             * Initialize CustomExtent with given initial extent, and compute the extent to apply in return
             * @param customExtent
             * customExtent (for user to define custom y and x axis range), has below parameters:
             *   - editMode: ("AUTO" or "MANUAL"); // the mode to use to define extent (auto computed or manually defined)
             *   - $autoExtent: [minExtent, maxExtent]; // array of 2 floats: initial min and max values auto-detected (used in AUTO editMode)
             *   - manualExtent: [minExtent, maxExtent]; // array of 2 floats or null: user custom min and max values, if null min or max are auto computed (used in MANUAL editMode)
             * @param initialExtent
             * @param isPercentScale
             * @returns new extent [minValue, maxValue] to apply based on user custom extent
             */
            initCustomAxisExtent: function(customExtent, initialExtent, isPercentScale) {
                const coeff = isPercentScale ? 100 : 1; // to format input between [0, 100] when in percentScale
                // @ts-ignore
                if (_.isFinite(initialExtent[0]) && _.isFinite(initialExtent[1])) {
                    customExtent.$autoExtent = [initialExtent[0] * coeff, initialExtent[1] * coeff];
                }

                if (customExtent.editMode === ChartsStaticData.AUTO_EXTENT_MODE) {
                    customExtent.manualExtent = [null, null];
                    return initialExtent;
                } else {
                    const manualExtent = getManualExtent(customExtent);
                    return [manualExtent[0] / coeff, manualExtent[1] / coeff];
                }
            },

            getDimensionExtent(chartData, axisSpec, ignoreLabels) {
                let extent = axisSpec.extent;

                if (!extent) {
                    if (axisSpec.type === 'UNAGGREGATED') {
                        extent = axisSpec.extent || ChartDataUtils.getUnaggregatedAxisExtent(axisSpec.dimension, axisSpec.data);
                    } else {
                        extent = ChartDataUtils.getAxisExtent(chartData, axisSpec.name, axisSpec.dimension, { ignoreLabels, initialExtent: axisSpec.customExtent && axisSpec.customExtent.$autoExtent });
                    }
                }

                if (axisSpec.customExtent) {
                    const transientExtent = svc.initCustomAxisExtent(axisSpec.customExtent, [extent.min, extent.max], axisSpec.isPercentScale);
                    extent = Object.assign(extent, { min: transientExtent[0], max: transientExtent[1] });
                }

                // Override min and max with pre-defined interval if requested.
                if (axisSpec.initialInterval) {
                    extent.min = axisSpec.initialInterval.min;
                    extent.max = axisSpec.initialInterval.max;
                }

                return extent;
            },

            getMeasureExtent(chartData, axisSpec, isLogScale, includeZero, computePercentScale, otherAxesIndexes = []) {
                let extent = axisSpec.extent;

                if (!extent) {
                    if (axisSpec.measureIdx === undefined) {
                        return null;
                    }
                    extent = ChartDataUtils.getMeasureExtent(chartData, axisSpec.measureIdx, true, null, axisSpec.measure[0]?.function, otherAxesIndexes);
                }

                if (extent[0] == Infinity) {
                    return null; // No values -> no axis
                }

                // Adjust extent if needed
                if (axisSpec.customExtent) {
                    extent = svc.initCustomAxisExtent(axisSpec.customExtent, extent, computePercentScale ? axisSpec.isPercentScale : false);
                }

                if (includeZero) {
                    const isZeroInRange = extent[0] > 0 != extent[1] > 0;
                    if (!isZeroInRange) {
                        extent[0] = Math.min(extent[0], 0);
                        extent[1] = Math.max(extent[1], 0);
                    }
                }

                if (isLogScale) {
                    const logScaleExtent = svc.fixNumericalLogScaleExtent(axisSpec, { min: extent[0] }, includeZero);
                    extent[0] = logScaleExtent.min;
                }

                return extent;
            },

            /**
             * A log axis scale cannot contain 0.
             * For user convenience, when includeZero is checked in auto range mode we allow the zero to be replaced by 1 automatically.
             *
             * @param   axisSpec        - Axis specification
             * @param {Boolean}     includeZero     - True to force inclusion of zero in extent
             * @param {Number}      currentMinValue        - Current known minimal value for the axis range.
             */
            getProperMinValueWhenInLogScale(axisSpec, includeZero, currentMinValue) {
                const shouldReplace = !svc.isManualMode(axisSpec.customExtent) && includeZero && currentMinValue === 0;
                return shouldReplace ? 1 : currentMinValue;
            },

            fixNumericalLogScaleExtent(axisSpec, extent, includeZero) {
                extent.min = svc.getProperMinValueWhenInLogScale(axisSpec, includeZero, extent.min);
                if (extent.min <= 0) {
                    // @ts-ignore
                    throw new ChartIAE('Cannot represent value 0 nor negative values on a log scale. Please disable log scale.');
                }
                return extent;
            },

            fixUnbinnedNumericalExtent(axisSpec, extent) {
                /*
                 * If the data is based on range bands AND we are going to use the linear scale
                 * to place the bands, then the linear scale must be refitted to give space for the bands
                 * (this corresponds to the COLUMN charts with 'Use raw values' mode)
                 * Same thing for scatter plot
                 */
                if (ChartDimension.isUngroupedNumerical(axisSpec.dimension) || axisSpec.type === 'UNAGGREGATED') {
                    const nbVals = extent.values.length;
                    const interval = extent.max - extent.min;
                    // Add 10% margin when not many bars, 5% margin else
                    const additionalPct = nbVals > 10 ? 5.0 : 10.0;
                    extent.min = extent.min - (interval * additionalPct) / 100;
                    extent.max = extent.max + (interval * additionalPct) / 100;
                }
                return extent;
            },

            includeZero(extent) {
                extent.min = Math.min(extent.min, 0);
                extent.max = Math.max(extent.max, 0);
                return extent;
            },

            computeYAxisID(orientation = 'left', index = 0, dimension) {
                let id = `y_${orientation}_${index}`;
                if (dimension) {
                    id += `_${dimension}`;
                }
                return id;
            },

            getFormattingForYAxis(yAxesFormatting, id = ChartsStaticData.LEFT_AXIS_ID) {
                return yAxesFormatting.find(v => v.id === id);
            },

            isYAxisLogScale(yAxesFormatting, id = ChartsStaticData.LEFT_AXIS_ID) {
                const formatting = this.getFormattingForYAxis(yAxesFormatting, id);
                return (formatting || {}).isLogScale;
            },

            getYAxisNumberFormatting(yAxesFormatting, id = ChartsStaticData.LEFT_AXIS_ID) {
                const formatting = this.getFormattingForYAxis(yAxesFormatting, id);
                return ((formatting || {}).axisValuesFormatting || {}).numberFormatting;
            },

            getYAxisCustomExtent(yAxesFormatting, id = ChartsStaticData.LEFT_AXIS_ID) {
                const formatting = this.getFormattingForYAxis(yAxesFormatting, id);
                return (formatting || {}).customExtent;
            },

            setYAxisCustomExtent(yAxesFormatting, newCustomExtent, id = ChartsStaticData.LEFT_AXIS_ID) {
                const index = yAxesFormatting.map(x => x.id).indexOf(id);
                yAxesFormatting[index].customExtent = newCustomExtent;
            },

            contatenateMeasuresNames(measures) {
                return measures.map(m => ColumnAvailability.getAggregatedLabel(m)).join(', ');
            },

            /**
             * getXAxisTitle is used to get correct x axis title (it is used by d3 and echarts, so it should stay agnostic)
             * @param xAxisSpec
             * @param chartDef
             */
            getXAxisTitle: function(xAxisSpec, chartDef) {
                if (!xAxisSpec || !chartDef || !ChartFeatures.canShowXAxisTitle(chartDef)) {
                    return;
                }

                let titleText = null;

                if (!!chartDef.xAxisFormatting.axisTitle && chartDef.xAxisFormatting.axisTitle.length > 0) {
                    titleText = chartDef.xAxisFormatting.axisTitle;
                } else if (xAxisSpec.dimension !== undefined) {
                    titleText = xAxisSpec.dimension.column;
                } else if (xAxisSpec.measure !== undefined) {
                    return this.contatenateMeasuresNames(xAxisSpec.measure);
                } else if (chartDef.genericDimension0.length === 1 || chartDef.genericHierarchyDimension.length === 1) {
                    titleText = (ChartDimension.getGenericDimension(chartDef) || {}).column;
                }

                return titleText;
            },

            /**
             * getYAxisTitle is used to get correct y axis title (it is used by d3 and echarts, so it should stay agnostic)
             * @param yAxisSpec
             * @param chartDef
             */
            getYAxisTitle: function(yAxisSpec, chartDef, axisFormatting) {
                if (!yAxisSpec) {
                    return;
                }

                let titleText = null;
                // @ts-ignore
                if (axisFormatting && !_.isNil(axisFormatting.axisTitle) && axisFormatting.axisTitle.length > 0) {
                    titleText = axisFormatting.axisTitle;
                } else if (yAxisSpec.dimension !== undefined) {
                    titleText = yAxisSpec.dimension.column;
                } else if (yAxisSpec.measure !== undefined) {
                    return ColumnAvailability.getAggregatedLabel(yAxisSpec.measure[0]);
                } else if (chartDef != null) {
                    const measuresOnAxis = svc.getMeasuresDisplayedOnAxis(yAxisSpec.position, chartDef.genericMeasures);
                    if (measuresOnAxis.length >= 1) {
                        return this.contatenateMeasuresNames(measuresOnAxis);
                    }
                }

                return titleText;
            },

            getMeasuresDisplayedOnAxis: function(axisPosition, measures) {
                const displayAxis = axisPosition === ChartYAxisPosition.LEFT ? 'axis1' : 'axis2';
                return measures.filter(measure => measure.displayAxis === displayAxis);
            },

            getYAxisSpecs: function(axisSpecs) {
                const ySpecs = axisSpecs && Object.keys(axisSpecs)
                    .filter((key) => key.toLowerCase().includes('y'))
                    .reduce((object, key) => {
                        return Object.assign(object, {
                            [key]: axisSpecs[key]
                        });
                    }, {});
                return ySpecs;
            },

            getYAxisName: function(id) {
                if (id.includes('right')) {
                    return translate('CHARTS.SHARED.RIGHT_AXIS', 'Right axis');
                } else {
                    return translate('CHARTS.SHARED.LEFT_AXIS', 'Left axis');
                }
            },

            isAxisDisplayed: function(genericMeasures, id) {
                const isAxisDisplayed = id.includes('right') ? svc.isRightAxisDisplayed : svc.isLeftAxisDisplayed;
                return id != null ? isAxisDisplayed(genericMeasures) : false;
            },

            isLeftAxisDisplayed: function(genericMeasures) {
                return genericMeasures != null && genericMeasures.some(measure => measure.displayAxis === 'axis1');
            },

            isRightAxisDisplayed: function(genericMeasures) {
                return genericMeasures != null && genericMeasures.some(measure => measure.displayAxis === 'axis2');
            },

            setNumberOfBinsToDimensions: function(chartData, chartDef, axisSpecs) {
                const possibleAxes = { x: 'x', y: 'y_left_0' };
                for (const [dataId, specId] of Object.entries(possibleAxes)) {
                    const axisLabels = chartData.getAxisLabels(dataId);
                    if (axisLabels && axisSpecs[specId] && axisSpecs[specId].dimension) {
                        axisSpecs[specId].dimension.$numberOfBins = axisLabels.length;
                        axisSpecs[specId].dimension.$isInteractiveChart = ChartDimension.isInteractiveChart(chartDef);
                        axisSpecs[specId].dimension.$forceOneTickPerBin = chartDef.variant === CHART_VARIANTS.waterfall;
                    }
                }
            },

            /**
             * Checks if a chart is cropped compared to its initial x and y extents
             * @param   chartDef
             * @returns  True if chart is cropped, False otherwise
             */
            isCroppedChart: function(chartDef) {
                return isCroppedExtent(chartDef.yAxesFormatting) || isCroppedExtent([chartDef.xAxisFormatting]);
            },

            /**
             * Computes the ratio: custom extent / initial extent
             * `customExtent.$initialExtent` keep track of the initial range of an axis before it was altered by another component (reference line e.g.)
             * @param   customExtent
             * @returns  result of initialExtentWidth / manualExtentWidth
             */
            getCustomExtentRatio: function(customExtent) {
                const activeExtent = customExtent.$autoExtent && customExtent.$initialExtent ? customExtent.$initialExtent : customExtent.$autoExtent;
                if (activeExtent) {
                    const initialExtentWidth = activeExtent[1] - activeExtent[0];
                    if (customExtent.editMode === ChartsStaticData.MANUAL_EXTENT_MODE) {
                        const manualExtent = getManualExtent(customExtent);
                        const manualExtentWidth = manualExtent[1] - manualExtent[0];
                        return parseFloat('' + initialExtentWidth / manualExtentWidth);
                    } else if (customExtent.$autoExtent && customExtent.$initialExtent) {
                        const autoExtentWidth = customExtent.$autoExtent[1] - customExtent.$autoExtent[0];
                        return parseFloat('' + initialExtentWidth / autoExtentWidth);
                    }
                }

                return 1;
            },

            /**
             * Check if custom extent is valid
             * @param axesFormatting
             * @returns {boolean} True if valid, False otherwise
             */
            isCustomExtentValid: function(axesFormatting) {
                return axesFormatting.every(formatting => {
                    if (formatting.customExtent && formatting.customExtent.editMode === MANUAL_EXTENT_MODE) {
                        const [min, max] = getManualExtent(formatting.customExtent);
                        return min <= max;
                    }
                    return true;
                });
            },

            resetCustomExtents: function(axesFormatting) {
                if (!axesFormatting) {
                    return;
                }
                axesFormatting.forEach(formatting => {
                    if (formatting && formatting.customExtent) {
                        formatting.customExtent.editMode = AUTO_EXTENT_MODE;
                    }
                });
            },

            isManualMode: function(customExtent) {
                return customExtent && customExtent.editMode
                    && customExtent.editMode === ChartsStaticData.MANUAL_EXTENT_MODE;
            },

            getLeftYAxis: function(yAxes) {
                const leftAxis = yAxes.filter(axis => svc.isLeftYAxis(axis));
                return leftAxis.length ? leftAxis[0] : null;
            },

            getRightYAxis: function(yAxes) {
                const rightAxis = yAxes.filter(axis => svc.isRightYAxis(axis));
                return rightAxis.length ? rightAxis[0] : null;
            },

            isLeftYAxis: function(axis) {
                return !!axis && axis.position === ChartYAxisPosition.LEFT;
            },

            isRightYAxis: function(axis) {
                return !!axis && axis.position === ChartYAxisPosition.RIGHT;
            },

            initYAxesFormatting: function(id) {
                return {
                    id,
                    axisTitleFormatting: { ...AxisTitleFormatting },
                    showAxisTitle: true,
                    displayAxis: true,
                    axisValuesFormatting: {
                        axisTicksFormatting: { ...AxisTicksFormatting },
                        numberFormatting: { ...ChartsStaticData.DEFAULT_NUMBER_FORMATTING_OPTIONS }
                    },
                    ticksConfig: { mode: AxisTicksConfigMode.INTERVAL },
                    customExtent: { ...ChartsStaticData.DEFAULT_CUSTOM_EXTENT },
                    isLogScale: false
                };
            },

            getManualExtentMin,
            getManualExtentMax,
            getManualExtent
        };

        return svc;
    }
})();

;
// @ts-check
(function() {
    'use strict';

    // @ts-ignore
    angular.module('dataiku.directives.simple_report')
        .factory('ChartFeatures', chartFeatures);

    /**
     * ChartFeatures service
     * Defines which UI option is available based on chart type.
     * (!) This service previously was in static/dataiku/js/simple_report/services/chart-features.service.js
     * @typedef {import("../../../../../../../../server/src/frontend/src/app/features/simple-report/services/chart-features/chart-features.type").ChartFeatureReturnType} ChartFeatureReturnType
     * @returns {ChartFeatureReturnType}
     */
    function chartFeatures(WebAppsService, ChartStoreFactory, ChartDimension, ChartsStaticData, ChartUADimension, ChartTypeChangeUtils, D3ChartZoomControl, CHART_TYPES, CHART_VARIANTS, CHART_AXIS_TYPES, NumberFormatter, AxisTicksConfigMode, VALUES_DISPLAY_MODES, translate, $rootScope) {

        function isChartTypeInList(chartType, chartTypeList) {
            return chartTypeList.includes(chartType);
        }

        function canDisplayTotalValues(chartDef) {
            return chartDef.type === CHART_TYPES.STACKED_COLUMNS && chartDef.variant !== CHART_VARIANTS.stacked100;
        }

        const canSetCustomAxisRange = chartDef => chartDef && chartDef.variant !== CHART_VARIANTS.binnedXYHexagon;
        const axisTicksModes = [
            { label: translate('CHARTS.SETTINGS.FORMAT.AXIS_TICKS.NUMBER', 'Number'), value: AxisTicksConfigMode.NUMBER },
            { label: translate('CHARTS.SETTINGS.FORMAT.AXIS_TICKS.INTERVAL', 'Interval'), value: AxisTicksConfigMode.INTERVAL }
        ];

        /** @type {ChartFeatureReturnType} */
        const svc = {

            canDisplayTotalValues,

            canExportToExcel: chartDef => {
                if (chartDef.facetDimension.length) {
                    return false;
                }
                return (isChartTypeInList(chartDef.type, [
                    CHART_TYPES.STACKED_COLUMNS, CHART_TYPES.STACKED_AREA,
                    CHART_TYPES.LINES, CHART_TYPES.PIVOT_TABLE, CHART_TYPES.GROUPED_XY, CHART_TYPES.KPI
                ]) || (chartDef.type === CHART_TYPES.GROUPED_COLUMNS && chartDef.variant !== CHART_VARIANTS.waterfall));
            },

            canExportToImage: chartDef => {
                if (chartDef.facetDimension.length) {
                    return false;
                }
                return !isChartTypeInList(chartDef.type, [
                    CHART_TYPES.PIVOT_TABLE, CHART_TYPES.KPI, CHART_TYPES.SCATTER_MAP, CHART_TYPES.ADMINISTRATIVE_MAP,
                    CHART_TYPES.GRID_MAP, CHART_TYPES.GEOMETRY_MAP, CHART_TYPES.WEBAPP, CHART_TYPES.DENSITY_HEAT_MAP
                ]);
            },

            isMap: chartType => {
                return isChartTypeInList(chartType, [
                    CHART_TYPES.GEOMETRY_MAP, CHART_TYPES.GRID_MAP, CHART_TYPES.SCATTER_MAP, CHART_TYPES.ADMINISTRATIVE_MAP, CHART_TYPES.DENSITY_HEAT_MAP
                ]);
            },

            canDisplayDimensionValuesInChart: chartType => {
                return chartType === CHART_TYPES.TREEMAP;
            },

            getExportDisabledReason: chartDef => {
                if (chartDef.facetDimension.length) {
                    return 'Download is not available for subcharts.';
                }
            },

            canDrillHierarchy: chartType => isChartTypeInList(chartType, [
                CHART_TYPES.GROUPED_COLUMNS, CHART_TYPES.STACKED_COLUMNS, CHART_TYPES.STACKED_BARS, CHART_TYPES.RADAR,
                CHART_TYPES.LINES, CHART_TYPES.MULTI_COLUMNS_LINES, CHART_TYPES.PIE, CHART_TYPES.STACKED_AREA, CHART_TYPES.BOXPLOTS,
                CHART_TYPES.BINNED_XY, CHART_TYPES.GROUPED_XY, CHART_TYPES.LIFT, CHART_TYPES.PIVOT_TABLE, CHART_TYPES.TREEMAP
            ]),

            canDisplayBreadcrumb: chartType => svc.canDrillHierarchy(chartType) && chartType !== CHART_TYPES.PIVOT_TABLE,

            shouldDisplayBreadcrumb: chartDef => svc.canDisplayBreadcrumb(chartDef.type) && chartDef.drilldownOptions.displayBreadcrumb,

            canAnimate: (chartType, chartVariant) => isChartTypeInList(chartType, [
                CHART_TYPES.MULTI_COLUMNS_LINES, CHART_TYPES.GROUPED_COLUMNS, CHART_TYPES.STACKED_COLUMNS,
                CHART_TYPES.STACKED_BARS, CHART_TYPES.LINES, CHART_TYPES.STACKED_AREA, CHART_TYPES.PIE,
                CHART_TYPES.BINNED_XY, CHART_TYPES.GROUPED_XY, CHART_TYPES.LIFT
            ]) && chartVariant !== CHART_VARIANTS.waterfall,

            canFacet: (chartType, webAppType, chartVariant) => {
                if (chartType === CHART_TYPES.GROUPED_COLUMNS) {
                    return chartVariant !== CHART_VARIANTS.waterfall;
                }
                if (chartType === CHART_TYPES.WEBAPP) {
                    const loadedDesc = WebAppsService.getWebAppLoadedDesc(webAppType) || {};
                    const pluginChartDesc = (loadedDesc.desc && loadedDesc.desc.chart) || {};
                    return pluginChartDesc.canFacet === true;
                } else {
                    return isChartTypeInList(chartType, [
                        CHART_TYPES.MULTI_COLUMNS_LINES, CHART_TYPES.STACKED_COLUMNS,
                        CHART_TYPES.STACKED_BARS, CHART_TYPES.LINES, CHART_TYPES.STACKED_AREA, CHART_TYPES.PIE,
                        CHART_TYPES.BINNED_XY, CHART_TYPES.GROUPED_XY, CHART_TYPES.LIFT
                    ]);
                }
            },

            canSetSubchartsCommonXAxis: chartType => chartType !== CHART_TYPES.PIE,

            canFilter: (chartType, webAppType) => {
                if (chartType === CHART_TYPES.WEBAPP) {
                    const loadedDesc = WebAppsService.getWebAppLoadedDesc(webAppType) || {};
                    const pluginChartDesc = (loadedDesc.desc && loadedDesc.desc.chart) || {};
                    return pluginChartDesc.canFilter === true;
                } else {
                    return true;
                }
            },

            canHaveZoomControls: (chartDef, zoomControlInstanceId) => {
                // @ts-ignore
                return !_.isNil(chartDef) && !_.isNil(zoomControlInstanceId) && (svc.canHaveEChartsZoomControls(chartDef) || svc.hasScatterZoomControlEnabled(chartDef.type, zoomControlInstanceId) || svc.canHaveMapsZoomControls(chartDef));
            },

            canHaveEChartsZoomControls: chartDef => chartDef.type === CHART_TYPES.TREEMAP && !!ChartDimension.getYDimensions(chartDef).length && !!chartDef.genericMeasures.length,

            canHaveMapsZoomControls: chartDef => svc.isMap(chartDef.type) && (chartDef.geometry.length > 0 || chartDef.geoLayers.length > 1),

            isScatterZoomed: (chartType, zoomControlInstanceId) => {
                return svc.isUnaggregated(chartType) && D3ChartZoomControl.isZoomed(zoomControlInstanceId);
            },

            isUnaggregated: (chartType) => {
                return chartType === CHART_TYPES.SCATTER || chartType === CHART_TYPES.SCATTER_MULTIPLE_PAIRS;
            },

            hasScatterZoomControlEnabled: (chartType, zoomControlInstanceId) => {
                return svc.isUnaggregated(chartType) && D3ChartZoomControl.isEnabled(zoomControlInstanceId);
            },

            hasScatterZoomControlActivated: (chartType, zoomControlInstanceId) => {
                return svc.isUnaggregated(chartType) && D3ChartZoomControl.isActivated(zoomControlInstanceId);
            },

            canSmooth: chartType => isChartTypeInList(chartType, [CHART_TYPES.LINES, CHART_TYPES.STACKED_AREA, CHART_TYPES.MULTI_COLUMNS_LINES]),

            canSetStrokeWidth: chartType => isChartTypeInList(chartType, [
                CHART_TYPES.LINES, CHART_TYPES.MULTI_COLUMNS_LINES, CHART_TYPES.GEOMETRY_MAP
            ]),

            canSetFillOpacity: chartDef => {
                return (chartDef.type === CHART_TYPES.GEOMETRY_MAP || (chartDef.type === CHART_TYPES.ADMINISTRATIVE_MAP && chartDef.variant === CHART_VARIANTS.filledMap));
            },

            canHideXAxis: chartType => chartType === CHART_TYPES.STACKED_BARS,

            canDisplayValuesInChart: chartType => isChartTypeInList(chartType, [
                CHART_TYPES.GAUGE, CHART_TYPES.GROUPED_COLUMNS, CHART_TYPES.STACKED_COLUMNS, CHART_TYPES.PIE, CHART_TYPES.RADAR, CHART_TYPES.SANKEY, CHART_TYPES.STACKED_BARS, CHART_TYPES.TREEMAP
            ]),

            canDisplayBrush: chartDef => {
                return chartDef && chartDef.linesZoomOptions && chartDef.linesZoomOptions.enabled && ChartDimension.isInteractiveChart(chartDef);
            },

            canDrawIdentityLine: chartDef => {
                const yAxisFormatting = chartDef.yAxesFormatting.find(formatting => formatting.id === ChartsStaticData.LEFT_AXIS_ID);
                return chartDef && chartDef.type === CHART_TYPES.SCATTER && !(chartDef.xAxisFormatting.isLogScale || (yAxisFormatting && yAxisFormatting.isLogScale));
            },

            canCustomizeValuesInChart: chartDef => {
                return isChartTypeInList(chartDef.type, [
                    CHART_TYPES.GROUPED_COLUMNS, CHART_TYPES.RADAR, CHART_TYPES.SANKEY, CHART_TYPES.GAUGE, CHART_TYPES.STACKED_COLUMNS, CHART_TYPES.STACKED_BARS, CHART_TYPES.TREEMAP])
                    || (chartDef.type === CHART_TYPES.PIE && svc.isEChart(chartDef));
            },

            canCustomizeLabelsInChart: chartDef => chartDef.type === CHART_TYPES.PIE && svc.isEChart(chartDef),

            canCustomizeValuesInChartOverlap: chartDef => isChartTypeInList(chartDef.type, [
                CHART_TYPES.GROUPED_COLUMNS, CHART_TYPES.RADAR, CHART_TYPES.SANKEY, CHART_TYPES.STACKED_COLUMNS, CHART_TYPES.STACKED_BARS
            ]) && svc.canCustomizeValuesInChart(chartDef),

            canAddMeasuresToLabelsInChart: chartDef => isChartTypeInList(chartDef.type, [
                CHART_TYPES.GROUPED_COLUMNS, CHART_TYPES.PIE, CHART_TYPES.RADAR, CHART_TYPES.SANKEY, CHART_TYPES.STACKED_COLUMNS, CHART_TYPES.STACKED_BARS, CHART_TYPES.TREEMAP, CHART_TYPES.LINES, CHART_TYPES.MULTI_COLUMNS_LINES]) && chartDef.variant !== CHART_VARIANTS.stacked100,

            shouldDisplayTotalValues: chartDef => canDisplayTotalValues(chartDef) && chartDef.valuesInChartDisplayOptions
                && chartDef.valuesInChartDisplayOptions.displayValues
                && chartDef.valuesInChartDisplayOptions.displayMode === VALUES_DISPLAY_MODES.VALUES_AND_TOTALS,

            shouldDisplay0Warning: (chartType, pivotResponse) => [CHART_TYPES.SANKEY, CHART_TYPES.TREEMAP].includes(chartType)
                && (pivotResponse && pivotResponse.aggregations.every(agg => agg.tensor.every(el => el === 0)))
            ,

            canDisplayLabelsInChart: chartType => isChartTypeInList(chartType, [CHART_TYPES.PIE, CHART_TYPES.GAUGE]),

            shouldDrawRegressionLine: (chartDef) => {
                return ChartUADimension.areAllNumericalOrDate(chartDef) && chartDef.type === CHART_TYPES.SCATTER && chartDef.scatterOptions
                && chartDef.scatterOptions.regression && chartDef.scatterOptions.regression.show;
            },

            canSelectPivotRowOrColumnDisplayMode: chartDef => chartDef != null && chartDef.type === CHART_TYPES.PIVOT_TABLE &&
                !!ChartDimension.getYDimensions(chartDef).length && !!ChartDimension.getXDimensions(chartDef).length,

            canCustomizePivotFormatting: chartDef => chartDef != null && chartDef.type === CHART_TYPES.PIVOT_TABLE &&
                (!!ChartDimension.getYDimensions(chartDef).length || !!ChartDimension.getXDimensions(chartDef).length)
                && (chartDef.genericMeasures && chartDef.genericMeasures.length),

            canCustomizePivotRowHeaders: chartDef => svc.canCustomizePivotFormatting(chartDef) &&
                !!ChartDimension.getYDimensions(chartDef).length || (chartDef.genericMeasures && chartDef.genericMeasures.length > 1),

            isColoredPivotTable: chartDef => chartDef != null && chartDef.type === CHART_TYPES.PIVOT_TABLE && (
                chartDef.colorMeasure && chartDef.colorMeasure.length ||
                chartDef.colorMode === 'COLOR_GROUPS'
            ),

            canSetMeasureColumnsTooltips: chartType => isChartTypeInList(chartType, [
                CHART_TYPES.MULTI_COLUMNS_LINES, CHART_TYPES.GROUPED_COLUMNS,
                CHART_TYPES.STACKED_COLUMNS, CHART_TYPES.STACKED_BARS, CHART_TYPES.STACKED_AREA,
                CHART_TYPES.GRID_MAP, CHART_TYPES.LINES, CHART_TYPES.ADMINISTRATIVE_MAP, CHART_TYPES.PIE, CHART_TYPES.BINNED_XY
            ]),

            canSetUnaggregatedColumnsTooltips: chartType => isChartTypeInList(chartType, [
                CHART_TYPES.SCATTER, CHART_TYPES.SCATTER_MULTIPLE_PAIRS, CHART_TYPES.SCATTER_MAP, CHART_TYPES.GEOMETRY_MAP, CHART_TYPES.DENSITY_HEAT_MAP
            ]),

            canDisplayTooltips: chartType => !isChartTypeInList(chartType, [
                CHART_TYPES.KPI, CHART_TYPES.GAUGE, CHART_TYPES.DENSITY_2D, CHART_TYPES.WEBAPP
            ]),

            shouldDisplayTooltips: (noTooltips, tooltipOptions, fixed) => {
                // @ts-ignore
                const isInADashboard = !_.isNil(noTooltips);

                // If the Show tooltip checkbox is unchecked at chart level, it can still be forced for this chart in a dashboard with the one set at dashboard level
                return (isInADashboard ? noTooltips === false : tooltipOptions.display) && !fixed;
            },

            canHandleEmptyBins: chartType => isChartTypeInList(chartType, [CHART_TYPES.LINES, CHART_TYPES.MULTI_COLUMNS_LINES]),

            canSetAxisTextColor: chartType => chartType !== CHART_TYPES.SCATTER_MULTIPLE_PAIRS,

            chartSupportOneTickPerBin: chartDef =>
                !(isChartTypeInList(chartDef.type,
                    [CHART_TYPES.PIE, CHART_TYPES.PIVOT_TABLE, CHART_TYPES.RADAR, CHART_TYPES.SANKEY, CHART_TYPES.TREEMAP, CHART_TYPES.SCATTER, CHART_TYPES.SCATTER_MULTIPLE_PAIRS, CHART_TYPES.BOXPLOTS, CHART_TYPES.GROUPED_XY]) || chartDef.hexbin)
                && chartDef.variant !== CHART_VARIANTS.binnedXYHexagon,

            chartOnlySupportsOneTickPerBin: (chartDef) => {
                return chartDef?.type === CHART_TYPES.GROUPED_COLUMNS && chartDef?.variant === CHART_VARIANTS.waterfall;
            },

            canModifyAxisRange: (chartDef, axis) => {
                if (!canSetCustomAxisRange(chartDef)) {
                    return false;
                }

                let dimensionsAndMeasures;

                if (axis === 'x') {
                    dimensionsAndMeasures = ChartTypeChangeUtils.takeAllInX(chartDef);
                } else if (axis === 'y') {
                    dimensionsAndMeasures = ChartTypeChangeUtils.takeAllInY(chartDef);
                } else {
                    return false;
                }

                const hasOneTickPerBin = svc.hasOneTickPerBinSelected(chartDef.$chartStoreId, axis);
                /*
                 * Only check the first value as the others are guaranteed to be of the same type.
                 * Dimensions always have one value, but measures can have multiple.
                 */
                return dimensionsAndMeasures.some(([firstOne]) =>
                    firstOne && (firstOne.isA === 'measure' || (ChartDimension.isTrueNumerical(firstOne) && !hasOneTickPerBin)));
            },

            canIncludeZero: chartType => isChartTypeInList(chartType, [
                CHART_TYPES.GROUPED_COLUMNS, CHART_TYPES.LINES, CHART_TYPES.MULTI_COLUMNS_LINES,
                CHART_TYPES.GROUPED_XY
            ]),

            isPivotTableWithNoDimension: (chartDef) => {
                return chartDef.type === CHART_TYPES.PIVOT_TABLE && !ChartDimension.getXDimensions(chartDef).length && !ChartDimension.getYDimensions(chartDef).length;
            },

            canHaveConditionalFormatting: chartType => isChartTypeInList(chartType, [
                CHART_TYPES.PIVOT_TABLE, CHART_TYPES.KPI
            ]),

            hasLogarithmicRegressionWarning: (chartDef) => {
                if (chartDef == null) {
                    return false;
                }

                const areValuesLessOrEqual0 = chartDef.xAxisFormatting && chartDef.xAxisFormatting.customExtent && chartDef.xAxisFormatting.customExtent.$autoExtent && chartDef.xAxisFormatting.customExtent.$autoExtent[0] <= 0;
                const isAxisMinAbove0 = chartDef.xAxisFormatting && chartDef.xAxisFormatting.customExtent && chartDef.xAxisFormatting.customExtent.manualExtent && chartDef.xAxisFormatting.customExtent.manualExtent[0] > 0;

                return areValuesLessOrEqual0 && !isAxisMinAbove0 && chartDef.type === CHART_TYPES.SCATTER;
            },

            hasExponentialRegressionWarning: (chartDef, yAxisID = ChartsStaticData.LEFT_AXIS_ID) => {
                if (chartDef == null) {
                    return false;
                }

                const yAxisFormatting = chartDef.yAxesFormatting.find(v => v.id === yAxisID);
                const areValuesLessOrEqual0 = yAxisFormatting && yAxisFormatting.customExtent && yAxisFormatting.customExtent.$autoExtent && yAxisFormatting.customExtent.$autoExtent[0] <= 0;
                const isAxisMinAbove0 = yAxisFormatting && yAxisFormatting.customExtent && yAxisFormatting.customExtent.manualExtent && yAxisFormatting.customExtent.manualExtent[0] > 0;

                return areValuesLessOrEqual0 && !isAxisMinAbove0 && chartDef.type === CHART_TYPES.SCATTER;
            },

            canEditAxisRange: (chartDef) => {
                return chartDef != null && !(ChartUADimension.areAllNumericalOrDate(chartDef) && chartDef.type === CHART_TYPES.SCATTER
                    && chartDef.scatterOptions && chartDef.scatterOptions.equalScales);
            },

            canShowXAxisTitle: (chartDef) => {
                const pairs = chartDef.uaDimensionPair.filter(pair => pair.uaXDimension && pair.uaXDimension.length && pair.uaXDimension[0].column && pair.uaYDimension && pair.uaYDimension.length && pair.uaYDimension[0].column);
                return !((chartDef.type === CHART_TYPES.SCATTER_MULTIPLE_PAIRS && pairs.length > 1) || chartDef.type === CHART_TYPES.BOXPLOTS);
            },

            hasMultipleXDim: chartType => chartType === CHART_TYPES.SCATTER_MULTIPLE_PAIRS,

            canSetLogScale: (chartDef, axisName, id) => {
                if (chartDef == null) {
                    return false;
                }

                switch (axisName) {
                    case 'x':
                        if (chartDef.type === CHART_TYPES.SCATTER) {
                            return ChartUADimension.isTrueNumerical(chartDef.uaXDimension[0]);
                        } else if (chartDef.type === CHART_TYPES.SCATTER_MULTIPLE_PAIRS) {
                            return chartDef.uaDimensionPair.every(pair => pair.uaXDimension && ChartUADimension.isTrueNumerical(pair.uaXDimension[0]));
                        }
                        return chartDef.type === CHART_TYPES.STACKED_BARS && chartDef.variant !== CHART_VARIANTS.stacked100;
                    case 'y':
                        if (chartDef.type === CHART_TYPES.SCATTER) {
                            return ChartUADimension.isTrueNumerical(chartDef.uaYDimension[0]);
                        } else if (chartDef.type === CHART_TYPES.BINNED_XY) {
                            return chartDef.variant !== CHART_VARIANTS.binnedXYHexagon
                                && chartDef.yDimension[0]
                                && !ChartDimension.hasOneTickPerBin(chartDef.yDimension[0])
                                && ChartDimension.isTrueNumerical(chartDef.yDimension[0]);
                        } else if (chartDef.type === CHART_TYPES.SCATTER_MULTIPLE_PAIRS) {
                            const axisPair = chartDef.uaDimensionPair.find(pair => pair.id === id);
                            return axisPair && ChartUADimension.isTrueNumerical(axisPair.uaYDimension[0]);
                        }
                        return chartDef.type !== CHART_TYPES.STACKED_BARS
                            && chartDef.variant !== CHART_VARIANTS.stacked100 && chartDef.variant !== CHART_VARIANTS.waterfall;
                    default:
                        return false;
                }
            },

            canSetLeftRightYAxes: chartType => isChartTypeInList(chartType, [
                CHART_TYPES.MULTI_COLUMNS_LINES, CHART_TYPES.GROUPED_COLUMNS, CHART_TYPES.LINES
            ]),

            canSelectGridlinesAxis: chartType => svc.canSetLeftRightYAxes(chartType) || chartType === CHART_TYPES.SCATTER_MULTIPLE_PAIRS,

            canHaveVerticalGridlines: chartDef => !(chartDef.facetDimension.length && chartDef.singleXAxis)
                && !(chartDef.type === CHART_TYPES.BOXPLOTS && !chartDef.boxplotBreakdownDim.length && !chartDef.boxplotBreakdownHierarchyDimension.length),

            getVLinesDisabledTooltip: chartType => chartType === CHART_TYPES.BOXPLOTS
                ? translate('CHARTS.SETTINGS.FORMAT.GRIDLINES.NO_X_DIMENSION', 'Add an X dimension to be able to display vertical gridlines')
                : translate('CHARTS.SETTINGS.FORMAT.GRIDLINES.COMMON_X_AXIS_NOT_SUPPORTED', 'Vertical gridlines cannot be drawn on subcharts using the common X axis option.'),

            canSetMultiPlotDisplayMode: chartType => chartType === CHART_TYPES.MULTI_COLUMNS_LINES,

            isKPIChart: chartType => chartType === CHART_TYPES.KPI,

            canUseUnaggregatedMeasures: (chartDef, column, measureType, measureDisplayType, contextualMenuMeasureType) => {
                return $rootScope?.featureFlagEnabled?.('unaggregatedMeasures') && !!column // can't use unaggregated mode for count of records
                    && ([
                        // CHART_TYPES.MULTI_COLUMNS_LINES, CHART_TYPES.LINES,
                        CHART_TYPES.GROUPED_COLUMNS].includes(chartDef.type)
                        || (chartDef.type === CHART_TYPES.MULTI_COLUMNS_LINES && measureDisplayType === 'column')
                        || (chartDef.type === CHART_TYPES.STACKED_COLUMNS && chartDef.variant !== CHART_VARIANTS.stacked100)
                    )
                    && measureType === 'NUMERICAL' && (!contextualMenuMeasureType || contextualMenuMeasureType === 'generic' || contextualMenuMeasureType === 'valuesInChart' || contextualMenuMeasureType === 'tooltip');
            }, // for now unaggregated mode available only in chart def measures

            isGaugeChart: chartType => chartType === CHART_TYPES.GAUGE,

            isKPILikeChart: chartType => isChartTypeInList(chartType, [
                CHART_TYPES.KPI, CHART_TYPES.GAUGE
            ]),

            canShowTextFormattingFromDropdown: chartType => isChartTypeInList(chartType, [
                CHART_TYPES.GAUGE
            ]),

            canDisplayNumberFormatting: measure => {
                return !((measure.type === 'CUSTOM' && measure.inferredType !== 'NUMERICAL')
                    || (measure.type === 'DATE' && !ChartsStaticData.ANY_AND_NUMERICAL_RESULT_AGGREGATION.includes(measure.function))
                    || ChartsStaticData.ALPHANUM_AND_ALPHANUMERICAL_RESULT_AGGREGATION.includes(measure.function));
            },

            canHaveThumbnail: chartDef => {
                if (chartDef.facetDimension.length) {
                    return false;
                }
                return !isChartTypeInList(chartDef.type, [CHART_TYPES.PIVOT_TABLE, CHART_TYPES.KPI, CHART_TYPES.WEBAPP]);
            },

            getMeasureComputationModes: chartDef => {
                const measureComputationModes = [];

                function addMeasureComputationMode(str) {
                    measureComputationModes.push(ChartsStaticData.stdAggrMeasureComputeModes[str]);
                }

                addMeasureComputationMode('NORMAL');

                const nbGDims = chartDef.genericDimension0.length + chartDef.genericHierarchyDimension.length + chartDef.genericDimension1.length;

                switch (chartDef.type) {
                    case CHART_TYPES.GROUPED_COLUMNS:
                    case CHART_TYPES.LINES:
                    case CHART_TYPES.MULTI_COLUMNS_LINES:
                        addMeasureComputationMode('PERCENTAGE');
                        addMeasureComputationMode('AVG_RATIO');
                        addMeasureComputationMode('CUMULATIVE');
                        addMeasureComputationMode('CUMULATIVE_PERCENTAGE');

                        if (nbGDims === 1) {
                            addMeasureComputationMode('DIFFERENCE');
                        }
                        break;
                    case CHART_TYPES.STACKED_COLUMNS:
                    case CHART_TYPES.STACKED_BARS:
                    case CHART_TYPES.STACKED_AREA:
                        addMeasureComputationMode('PERCENTAGE');
                        addMeasureComputationMode('CUMULATIVE');
                        addMeasureComputationMode('CUMULATIVE_PERCENTAGE');
                        break;
                    case CHART_TYPES.PIVOT_TABLE:
                        if (ChartDimension.getXDimensions(chartDef).length <= 1 && ChartDimension.getYDimensions(chartDef).length <= 1) {
                            addMeasureComputationMode('PERCENTAGE');
                            addMeasureComputationMode('AVG_RATIO');
                            addMeasureComputationMode('CUMULATIVE');
                            addMeasureComputationMode('CUMULATIVE_PERCENTAGE');
                            addMeasureComputationMode('DIFFERENCE');
                        }
                        break;
                    case CHART_TYPES.PIE:
                        addMeasureComputationMode('PERCENTAGE');
                        break;
                    case CHART_TYPES.TREEMAP:
                        if (ChartDimension.getYDimensions(chartDef).length <= 2) {
                            addMeasureComputationMode('PERCENTAGE');
                        }
                        break;
                }

                return measureComputationModes;
            },

            canSetAlongDimension: function(chartDef, computeMode) {
                const genericDimension0 = ChartDimension.getGenericDimension(chartDef);
                const xDimension = ChartDimension.getXDimension(chartDef);
                const yDimensions = ChartDimension.getYDimensions(chartDef);
                if (((genericDimension0 && chartDef.genericDimension1.length) || (xDimension && yDimensions.length) || (chartDef.type == CHART_TYPES.TREEMAP && yDimensions.length == 2))
                    && (computeMode == 'PERCENTAGE' || computeMode == 'AVG_RATIO' || computeMode == 'CUMULATIVE_PERCENTAGE')) {
                    return true;
                }
                return false;
            },

            getDimensionsForComputeAlong: chartDef => {
                if (chartDef.type === CHART_TYPES.PIVOT_TABLE) {
                    const xDimension = ChartDimension.getXDimension(chartDef);
                    const yDimension = ChartDimension.getYDimension(chartDef);
                    if (!xDimension || !yDimension) {
                        return [];
                    }
                    // PIVOT order of insertion of the axes is defined as follows: [...xDimension, ...yDimension]
                    return [[0, xDimension.column], [1, yDimension.column]];
                }
                if (chartDef.type === CHART_TYPES.SANKEY || chartDef.type === CHART_TYPES.TREEMAP) {
                    const yDimensions = ChartDimension.getYDimensions(chartDef);
                    if (yDimensions.length < 2) {
                        return [];
                    }
                    return [[0, yDimensions[0].column], [1, yDimensions[1].column]];
                }
                const genericDimension = ChartDimension.getGenericDimension(chartDef);
                if (!genericDimension || !chartDef.genericDimension1 || !chartDef.genericDimension1[0]) {
                    return [];
                }
                // default order of insertion of the axes is defined as follows: [...genericDimension0, ...genericDimension1]
                return [[0, genericDimension.column], [1, chartDef.genericDimension1[0].column]];
            },

            isEChart: (chartDef) => {
                const displayEchartsByDefault = svc.hasDefaultEChartsDisplay(chartDef.type) && chartDef.displayWithEChartsByDefault;
                const displayEchartsByFeatureFlag = !svc.hasDefaultEChartsDisplay(chartDef.type) && chartDef.displayWithECharts;

                return svc.hasEChartsDefinition(chartDef.type) && (displayEchartsByDefault || displayEchartsByFeatureFlag || !svc.hasD3Definition(chartDef.type));
            },

            isAxisNumericalAndFormattable: function(chartDef, axisId) {
                if (chartDef && chartDef.$chartStoreId && axisId) {
                    const chartStore = ChartStoreFactory.get(chartDef.$chartStoreId);
                    const axisType = chartStore.getAxisType(axisId);

                    if (axisType === CHART_AXIS_TYPES.MEASURE) {
                        return true;
                    } else {
                        const dimensionType = chartStore.getAxisDimensionOrUADimensionType(axisId);
                        const numParamsMode = chartStore.getAxisDimensionOrUADimensionNumParamsMode(axisId);
                        return dimensionType === 'NUMERICAL' && numParamsMode !== 'TREAT_AS_ALPHANUM';
                    }
                }

                return false;
            },

            getAxisTicksModes: () => {
                return axisTicksModes;
            },

            getAxisTicksDisabledMessage: (type, zoomControlInstanceId) => {
                if (svc.isScatterZoomed(type, zoomControlInstanceId)) {
                    return translate(
                        'CHARTS.SETTINGS.FORMAT.AXIS_TICKS.SCATTER_ZOOMED_TOOLTIP',
                        'Option is disabled as it is not compatible with the current zoom state of the chart. To enable it, reset the zoom state.'
                    );
                } else {
                    return translate(
                        'CHARTS.SETTINGS.FORMAT.AXIS_TICKS.DEFAULT_DISABLED_TOOLTIP',
                        'Option is disabled as "{{generate_one_tick_per_bin}}" is selected in the dimension',
                        { generate_one_tick_per_bin: translate('CHARTS.SHARED.GENERATE_ONE_TICK_PER_BIN', 'Generate one tick per bin') }
                    );
                }
            },

            canSetTicksConfig: (axisFormatting, chartDef, axisId) => {
                return axisFormatting && axisFormatting.displayAxis && svc.isAxisNumericalAndFormattable(chartDef, axisId);
            },

            hasOneTickPerBinSelected: (chartStoreId, axisId) => {
                if (chartStoreId) {
                    const chartStore = ChartStoreFactory.get(chartStoreId);
                    const axisSpec = chartStore.getAxisSpec(axisId);
                    return ChartDimension.hasOneTickPerBin(axisSpec && axisSpec.dimension) || false;
                }
            },

            supportSorting: (dimension, chartDef, isSecondDimension) => {
                if (!dimension) {
                    return false;
                }

                if (ChartDimension.isTrueNumerical(dimension)) {
                    return false;
                }

                if (svc.chartSupportOneTickPerBin(chartDef)) {
                    return !ChartDimension.supportOneTickPerBin(dimension) || ChartDimension.isAlphanumLike(dimension) || ChartDimension.isSortableDateDimension(dimension, isSecondDimension);
                }

                return true;
            },

            shouldRemoveOverlappingTicks: (axisFormatting, chartDef, axisId) => {
                const ticksConfig = axisFormatting.ticksConfig;
                return ticksConfig.number !== undefined && !svc.hasOneTickPerBinSelected(chartDef && chartDef.$chartStoreId, axisId) && svc.canSetTicksConfig(axisFormatting, chartDef, axisId);
            },

            canRealiasMeasuresAndDimensions: chartDef => {
                return chartDef.type !== 'boxplots';
            },

            canHaveBinningOptions: chartDef => {
                return chartDef.variant !== CHART_VARIANTS.binnedXYHexagon;
            },

            hasDefaultEChartsDisplay: chartType => {
                return chartType != null && [
                    CHART_TYPES.PIE
                ].includes(chartType);
            },

            hasEChartsDefinition: chartType => {
                return chartType != null && [
                    CHART_TYPES.GAUGE,
                    CHART_TYPES.PIE,
                    CHART_TYPES.RADAR,
                    CHART_TYPES.SANKEY,
                    CHART_TYPES.TREEMAP
                ].includes(chartType);
            },

            hasD3Definition: chartType => {
                return chartType != null && [
                    CHART_TYPES.GROUPED_COLUMNS,
                    CHART_TYPES.STACKED_BARS,
                    CHART_TYPES.STACKED_COLUMNS,
                    CHART_TYPES.MULTI_COLUMNS_LINES,
                    CHART_TYPES.LINES,
                    CHART_TYPES.STACKED_AREA,
                    CHART_TYPES.PIVOT_TABLE,
                    CHART_TYPES.SCATTER,
                    CHART_TYPES.GROUPED_XY,
                    CHART_TYPES.BINNED_XY,
                    CHART_TYPES.DENSITY_2D,
                    CHART_TYPES.SCATTER_MAP,
                    CHART_TYPES.DENSITY_HEAT_MAP,
                    CHART_TYPES.GEOMETRY_MAP,
                    CHART_TYPES.ADMINISTRATIVE_MAP,
                    CHART_TYPES.GRID_MAP,
                    CHART_TYPES.BOXPLOTS,
                    CHART_TYPES.PIE,
                    CHART_TYPES.LIFT,
                    CHART_TYPES.WEBAPP
                ].includes(chartType);

            },

            hasLegacyBadge: (chartType, isEChartsToggled) => {
                return svc.hasEChartsDefinition(chartType) && svc.hasD3Definition(chartType) && svc.hasDefaultEChartsDisplay(chartType) && isEChartsToggled === false;
            },

            hasBetaBadge: (chartType, isEChartsToggled) => {
                return svc.hasEChartsDefinition(chartType) && svc.hasD3Definition(chartType) && !svc.hasDefaultEChartsDisplay(chartType) && !!isEChartsToggled;
            },

            canDisplayLegend: (chartType) => {
                return !isChartTypeInList(chartType, [
                    CHART_TYPES.PIVOT_TABLE,
                    CHART_TYPES.LIFT,
                    CHART_TYPES.DENSITY_2D,
                    CHART_TYPES.KPI,
                    CHART_TYPES.GAUGE,
                    CHART_TYPES.SANKEY,
                    CHART_TYPES.DENSITY_HEAT_MAP
                ]);
            },

            isStackedLegend: (chartDef) => {
                return chartDef.type === CHART_TYPES.GEOMETRY_MAP && chartDef.geoLayers.length > 2;
            },

            shouldComputeLegend: (chartType) => {
                // compute sankey legend to have color menu, and compute pivot table legend for excel export
                return svc.canDisplayLegend(chartType) || isChartTypeInList(chartType, [CHART_TYPES.PIVOT_TABLE, CHART_TYPES.SANKEY]);
            },

            canConfigureForceLastPositionOthers: (chartType, dimension) => {
                return isChartTypeInList(chartType, [
                    CHART_TYPES.DONUT,
                    CHART_TYPES.PIE
                ]) && ChartDimension.isAlphanumLike(dimension) && dimension.generateOthersCategory;
            },

            canDisplayAxes: chartType => {
                return !isChartTypeInList(chartType, [
                    CHART_TYPES.PIE,
                    CHART_TYPES.PIVOT_TABLE,
                    CHART_TYPES.KPI,
                    CHART_TYPES.GAUGE,
                    CHART_TYPES.TREEMAP,
                    CHART_TYPES.SANKEY,
                    CHART_TYPES.RADAR,
                    CHART_TYPES.SCATTER_MAP,
                    CHART_TYPES.DENSITY_HEAT_MAP,
                    CHART_TYPES.GEOMETRY_MAP,
                    CHART_TYPES.ADMINISTRATIVE_MAP,
                    CHART_TYPES.GRID_MAP
                ]);
            },

            canHideAxes: chartType => {
                return svc.canDisplayAxes(chartType) && !isChartTypeInList(chartType, [
                    CHART_TYPES.DENSITY_2D,
                    CHART_TYPES.BOXPLOTS
                ]);
            },

            canUseSQL: chartDef => {
                return !chartDef.hexbin;
            },

            canShowTooltips: chartType => {
                return !isChartTypeInList(chartType, [CHART_TYPES.GAUGE, CHART_TYPES.KPI]);
            },

            canPinTooltip: (chartType) => {
                return chartType !== CHART_TYPES.SCATTER && chartType !== CHART_TYPES.SCATTER_MULTIPLE_PAIRS;
            },

            canFilterInTooltip: (chartDef) => {
                return ![CHART_TYPES.SCATTER, CHART_TYPES.SCATTER_MULTIPLE_PAIRS].includes(chartDef.type);
            },

            canIgnoreEmptyBinsForColorDimension: (chartDef) => {
                return [CHART_TYPES.GROUPED_COLUMNS, CHART_TYPES.MULTI_COLUMNS_LINES].includes(chartDef.type);
            },

            getRecordsMetadata: (chartDef, sampleMetadata, beforeFilterRecords, afterFilterRecords) => {
                if ([CHART_TYPES.SCATTER, CHART_TYPES.SCATTER_MULTIPLE_PAIRS].includes(chartDef.type)) {
                    const datasetRecordCount = sampleMetadata ? sampleMetadata.datasetRecordCount : beforeFilterRecords;
                    const sampleRecordCount = sampleMetadata ? sampleMetadata.sampleRecordCount : beforeFilterRecords;
                    const scatterOptions = chartDef.type === CHART_TYPES.SCATTER ? chartDef.scatterOptions : chartDef.scatterMPOptions;
	                const numberOfRecords = scatterOptions.numberOfRecords || 1000000;
                    const numberOfRecordsPerPair = Math.floor(numberOfRecords / (chartDef.uaDimensionPair.length || 1));

                    // By default, we compare it to the full dataset
                    let hasLessRecordsThanSampling = numberOfRecordsPerPair < datasetRecordCount;

                    // If we're using a sampling, datasetRecordCount equals -1, so we compare it to the sample
                    if (datasetRecordCount < 0) {
                        hasLessRecordsThanSampling = numberOfRecordsPerPair < sampleRecordCount;
                    }

                    if (afterFilterRecords < numberOfRecordsPerPair) {
                        // If there are less points after filtering than the authorized number of points per pair, we don't display the warning
                        hasLessRecordsThanSampling = false;
                    }


                    return {
                        hasLessRecordsThanSampling,
                        message: translate(
                            'CHARTS.HEADER.BADGE.TRUNCATED.TOOLTIP',
                            `${NumberFormatter.longSmartNumberFilter()(numberOfRecords)} points limit reached. You can modify the limit in the "Misc" section of your chart.`,
                            { maxNbPoints: NumberFormatter.longSmartNumberFilter()(numberOfRecords) }
                        )
                    };
                } else {
                    return null;
                }
            },

            shouldHideFormatTab: (chartType, readOnly) => {
                return readOnly;
            },

            canUseSecondGenericDimension: (chartDef) => {
                return (!!chartDef.genericDimension0?.length || !!chartDef.genericHierarchyDimension?.length || !!chartDef.genericDimension1?.length)
                && (chartDef.type !== CHART_TYPES.GROUPED_COLUMNS || chartDef.variant !== CHART_VARIANTS.waterfall);
            },

            canSupportMultipleGenericMeasures: (chartDef) => {
                return ([
                    CHART_TYPES.STACKED_COLUMNS,
                    CHART_TYPES.STACKED_BARS,
                    CHART_TYPES.LINES,
                    CHART_TYPES.STACKED_AREA
                ].includes(chartDef.type) && !chartDef.genericDimension1?.length)
                || (chartDef.type === CHART_TYPES.GROUPED_COLUMNS && chartDef.variant !== CHART_VARIANTS.waterfall && !chartDef.genericDimension1?.length)
                || ([
                    CHART_TYPES.KPI,
                    CHART_TYPES.RADAR,
                    CHART_TYPES.MULTI_COLUMNS_LINES,
                    CHART_TYPES.PIVOT_TABLE
                ].includes(chartDef.type));
            }
        };

        return svc;
    }
})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.directives.simple_report');

    /**
     * Service responsible for retrieving appropriate icon for a given chart type and DSS location.
     * (!) This service previously was in static/dataiku/js/simple_report/chart_type_picker.js
     */
    app.factory('ChartIconUtils', function(WebAppsService) {
        const ret = {
            computeChartIcon: function(type, variant, isInAnalysis, webAppType) {
                if (!ret.typeAndVariantToImageMap) {
                    return '';
                }
                if (typeof (type) !== 'undefined') {
                    if (type == 'webapp') {
                        const loadedDesc = WebAppsService.getWebAppLoadedDesc(webAppType) || {};
                        return loadedDesc && loadedDesc.desc && loadedDesc.desc.meta && loadedDesc.desc.meta.icon ? loadedDesc.desc.meta.icon : 'icon-puzzle-piece';
                    }
                    let imageName = 'basic_graphs';
                    if (typeof (variant) === 'undefined') {
                        variant = 'normal';
                    }
                    if (typeof (ret.typeAndVariantToImageMap[type]) !== 'undefined'
                        && typeof (ret.typeAndVariantToImageMap[type][variant]) !== 'undefined'
                        && typeof (ret.typeAndVariantToImageMap[type][variant]).icon !== 'undefined') {
                        imageName = ret.typeAndVariantToImageMap[type][variant].icon;
                    }
                    let uri = '/static/dataiku/images/charts/icons/';
                    if (isInAnalysis) {
                        uri += 'Chart_Icon_Analysis_';
                    } else {
                        uri += 'Chart_Icon_Dataset_';
                    }
                    return uri + imageName + '.svg';
                }
            },

            computeScatterLegendIcon(type) {
                return `/static/dataiku/images/charts/icons/scatter-legend/symbol_${type}.svg`;
            },

            typeAndVariantToImageMap: {
                'grouped_columns': {
                    'normal': {
                        'icon': 'vertical_bars',
                        'preview': 'grouped_columns'
                    },
                    'waterfall': {
                        'icon': 'waterfall',
                        'preview': 'waterfall'
                    }
                },
                'stacked_bars': {
                    'normal': {
                        'icon': 'horizontal_stacked_bars',
                        'preview': 'bar_graph'
                    },
                    'stacked_100': {
                        'icon': 'bar_stacked_100',
                        'preview': 'bar_graph'
                    }
                },
                'stacked_columns': {
                    'normal': {
                        'icon': 'stacked_color',
                        'preview': 'stacked_columns'
                    },
                    'stacked_100': {
                        'icon': 'stacked_100',
                        'preview': 'stacked_columns'
                    }
                },
                'multi_columns_lines': {
                    'normal': {
                        'icon': 'column__lines',
                        'preview': 'column__lines'
                    }
                },
                'lines': {
                    'normal': {
                        'icon': 'lines',
                        'preview': 'lines'
                    }
                },
                'stacked_area': {
                    'normal': {
                        'icon': 'stacked_areas',
                        'preview': 'stacked_areas'
                    },
                    'stacked_100': {
                        'icon': 'stacked_areas_100',
                        'preview': 'stacked_areas_100'
                    }
                },
                'pivot_table': {
                    'normal': {
                        'icon': 'table',
                        'preview': 'table'
                    }
                },
                'scatter': {
                    'normal': {
                        'icon': 'scatter',
                        'preview': 'scatter'
                    }
                },
                'scatter_multiple_pairs': {
                    'normal': {
                        'icon': 'scatter_multiple_pairs',
                        'preview': 'scatter_multiple_pairs'
                    }
                },
                'grouped_xy': {
                    'normal': {
                        'icon': 'grouped_scatter',
                        'preview': 'grouped_scatter'
                    }
                },
                'binned_xy': {
                    'normal': {
                        'icon': 'bubble',
                        'preview': 'bubble'
                    },
                    'binned_xy_rect': {
                        'icon': 'rectangles',
                        'preview': 'rectangles'
                    },
                    'binned_xy_hex': {
                        'icon': 'hexagons',
                        'preview': 'hexagons'
                    }
                },
                'density_2d': {
                    'normal': {
                        'icon': 'heatmap',
                        'preview': 'heatmap'
                    }
                },
                'kpi': {
                    'normal': {
                        'icon': 'kpi',
                        'preview': 'kpi'
                    }
                },
                'radar': {
                    'normal': {
                        'icon': 'radar_web',
                        'preview': 'radar_web'
                    }
                },
                'gauge': {
                    'normal': {
                        'icon': 'gauge',
                        'preview': 'gauge'
                    }
                },
                'sankey': {
                    'normal': {
                        'icon': 'sankey',
                        'preview': 'sankey'
                    }
                },
                'scatter_map': {
                    'normal': {
                        'icon': 'scatter_map',
                        'preview': 'scatter_map'
                    }
                },
                'density_heat_map': {
                    'normal': {
                        'icon': 'density_heat_map',
                        'preview': 'density_heat_map'
                    }
                },
                'geom_map': {
                    'normal': {
                        'icon': 'geom_map',
                        'preview': 'geom_map'
                    }
                },
                'admin_map': {
                    'normal': {
                        'icon': 'administrative_map',
                        'preview': 'administrative_map'
                    },
                    'filled_map': {
                        'icon': 'administrative_map',
                        'preview': 'administrative_map'
                    }
                },
                'grid_map': {
                    'normal': {
                        'icon': 'grid_map',
                        'preview': 'grid_map'
                    }
                },
                'treemap': {
                    'normal': {
                        'icon': 'tree_map',
                        'preview': 'tree_map'
                    }
                },
                'boxplots': {
                    'normal': {
                        'icon': 'box_plot',
                        'preview': 'box_plot'
                    }
                },
                'pie': {
                    'normal': {
                        'icon': 'pie',
                        'preview': 'pie'
                    },
                    'donut': {
                        'icon': 'donut',
                        'preview': 'donut'
                    }
                },
                'lift': {
                    'normal': {
                        'icon': 'diminishing_return_charts',
                        'preview': 'diminishing-reduction'
                    }
                }
            }
        };

        return ret;
    });

})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.charts');

    /**
     * Initialize a chart implemented in d3.
     * (!) This service previously was in static/dataiku/js/simple_report/chart_view_commons.js
     */
    app.factory('ChartManager', function(ChartDimension, ChartTooltips, ChartContextualMenu, AnimatedChartsUtils, $timeout, ChartLegendUtils,
        ChartFormatting, ChartColorScales, D3ChartAxes, ChartStoreFactory, ChartAxesUtils, ChartColorUtils, CHART_AXIS_TYPES) {

        const svc = {};

        /**
         * Create svg(s) in $container for the given chart
         * @param $container
         * @param chartData
         * @param chartDef
         * @return {*|jQuery|HTMLElement}
         */
        const createSVGs = function($container, chartData, chartDef, isInteractiveChart, ignoreLabels = new Set()) {
            $container.find('.mainzone').remove();
            const mainzone = $('<div class="mainzone" data-qa-screenshot-scope__chart>').prependTo($container);

            if (isInteractiveChart && chartDef.linesZoomOptions.displayBrush && chartDef.linesZoomOptions.enabled) {
                if (chartDef.legendPlacement === 'OUTER_TOP' || chartDef.legendPlacement === 'OUTER_BOTTOM') {
                    mainzone.addClass('mainzone--with-brush-and-horizontal-legend');
                } else {
                    mainzone.addClass('mainzone--with-brush');
                }
            }

            const $chartsContainer = $('<div class="charts">').appendTo(mainzone),
                isFacetted = chartData.axesDef.facet != undefined,
                facetLabels = chartData.getAxisLabels('facet', ignoreLabels) || [null];


            if (isFacetted) {
                $chartsContainer.addClass('facetted');
                $chartsContainer.height(facetLabels.length * (1 + chartDef.chartHeight) - 1);
            }

            let $svgs = $();

            const $chartsTable = $('<div class="charts-table">').appendTo($chartsContainer);

            facetLabels.forEach(function(facetLabel) {
                const $div = $('<div class="chart">');

                $div.appendTo($chartsTable);
                if (facetLabel) {
                    const $facetInfo = $('<div class="facet-info">').appendTo($div);
                    $('<h2>').text(ChartFormatting.getForOrdinalAxis(facetLabel.label)).appendTo($facetInfo);
                }

                const $wrapper = $('<div class="chart-wrapper">').appendTo($div);
                if (isFacetted) {
                    $wrapper.css('height', chartDef.chartHeight);
                }

                $svgs = $svgs.add($('<svg style="width: 100%; height: 100%;" class="chart-svg">').appendTo($wrapper));
            });

            return $svgs;
        };

        /**
         * Do all the initial setup surrounding the actual drawing area of a chart, this includes:
         *     - Create scales (x, y, y2, color)
         *     - Draw color legend
         *     - Adjust margins based on axis sizes
         *     - Create svgs (only one if chart is not facetted)
         *     - Draw axes
         *     - Create measure formatters
         *     - Initialize tooltips
         *
         * @param {ChartDef.java}                               chartDef
         * @param {a $scope object}                             chartHandler
         * @param {ChartTensorDataWrapper}                      chartData
         * @param {jQuery}                                      $container
         * @param {function}                                    drawFrame           a function(frameIdx, chartBase) that will be called every time a new animation frame is requested (only once if the chart has no animation dimension) to draw the actual chart
         * @param {[axisId: string]: AxisSpec}                  axisSpecs           an object containing axisSpec for all the axes
         * @param {AxisSpec}                                    colorSpec           an AxisSpec object for the color axis (nullable)
         * @param {Function}                                    [handleZoom]        an optional function to setup zoom
         * @param {Object}                                      [zoomUtils]         an optional object containing zoom information
         * AxisSpec:
         *     - type ('DIMENSION', 'MEASURE' or 'UNAGGREGATED') : whether the axis represents a dimension, one (or several) measures, or an unaggregated column
         *
         *     DIMENSION only attributes:
         *     - name : the name of the corresponding dimension as set in chartData
         *     - mode (either 'POINTS' or 'COLUMNS') : whether we sould use rangeRoundPoints or rangeRoundBands for the d3 ordinal scale
         *
         *     MEASURE only attributes:
         *     - measureIdx : the index of the measure if it is the axis only shows one measure
         *     - extent : the extent of the axis, will default to the extent of measure measureIdx if not provided
         *     - values : the list of values for the measure. Used for color spec to compute quantile scales.
         *
         *     UNAGGREGATED only attributes:
         *     - data : the data for this column (ScatterAxis.java object)
         *
         *     COMMON
         *     - dimension : the DimensionDef.java/NADimensionDef.java object for this column
         *
         *
         *     - withRgba (for ColorSpec only) : weather the color scale should include the chart's transparency setting with rgba or not
         *
         * @returns {ChartBase} : a ChartBase object, with the following properties:
         *      - $svgs {jQuery} $svgs,
         *      - colorScale {*} : a scale instance as returned by ChartColorScales.createColorScale
         *      - margins {{top: number, bottom: number, left: number, right: number}} the final, adjusted margins of the chart
         *      - vizWidth {number} : the width of the drawing area (full container width - legend with - margin width)
         *      - vizHeight {number} : the height of the drawing area
         *      - xAxis {d3 axis}
         *      - yAxes {d3 axis[]} (nullable)
         *      - tooltips {*} : a tooltip instance as returned by ChartTooltips.create
         *      - measureFormatters {array} : a list of measure formatters for all chart measures
         *      - isPercentChart {boolean} : is the chart a '*_100' variant
         *      - chartData {ChartTensorDataWrapper} : the same chartData that was given as input (for convenience)
         *
         */
        svc.initChart = function(chartDef, chartHandler, chartData, $container, drawFrame, axisSpecs, colorSpec, zoomContext, initLinesZoom, ignoreLabels = new Set()) {
            // Store the axis specs so they can be reused (for example in the axis configuration menus).
            const chartStoreMeta = ChartStoreFactory.getOrCreate(chartDef.$chartStoreId);
            chartDef.$chartStoreId = chartStoreMeta.id;

            chartStoreMeta.store.setAxisSpecs(axisSpecs);
            //  Retrieve axis specs in chart def for future migration (used for formatting pane)
            chartDef.$axisSpecs = axisSpecs;
            let chartBase = {};

            // Create color scale
            /** type: ChartColorContext */
            const colorContext = {
                chartData,
                colorOptions: ChartColorUtils.getChartColorOptions(chartDef),
                defaultLegendDimension: ChartColorUtils.getDefaultLegendDimension(chartDef, chartData),
                colorSpec,
                chartHandler,
                ignoreLabels,
                theme: chartHandler.getChartTheme()
            };
            const colorScale = ChartColorScales.createColorScale(colorContext);

            // Create axis
            chartBase.isPercentChart = chartDef.variant && chartDef.variant.endsWith('_100');

            const ySpecs = ChartAxesUtils.getYAxisSpecs(axisSpecs);

            const xAxis = D3ChartAxes.createAxis(
                chartData,
                axisSpecs?.['x'],
                chartBase.isPercentChart,
                chartDef.xAxisFormatting.isLogScale,
                undefined,
                chartDef.xAxisFormatting.axisValuesFormatting.numberFormatting,
                chartDef,
                ignoreLabels,
                (axisSpecs?.['x']?.type === CHART_AXIS_TYPES.MEASURE) && Object.keys(ySpecs || []).map((_, i) => i+1) // y axes indexes
            );

            const yAxes = [];
            ySpecs && Object.keys(ySpecs).forEach(key => {
                const spec = ySpecs[key];
                const includeZero = ChartAxesUtils.shouldIncludeZero(chartDef, key);
                const yAxis = D3ChartAxes.createAxis(
                    chartData,
                    spec,
                    chartBase.isPercentChart,
                    ChartAxesUtils.isYAxisLogScale(chartDef.yAxesFormatting, spec.id),
                    includeZero,
                    ChartAxesUtils.getYAxisNumberFormatting(chartDef.yAxesFormatting, spec.id),
                    chartDef,
                    ignoreLabels,
                    (spec?.type === CHART_AXIS_TYPES.MEASURE) && [0]
                );
                if (yAxis) {
                    yAxes.push(yAxis);
                }
            });

            ChartLegendUtils.createLegend($container, chartDef, chartData, chartHandler, colorSpec, colorScale, { ignoreLabels }).then(function() {
                $timeout(() => {
                    const $svgs = createSVGs($container, chartData, chartDef, zoomContext && ChartDimension.isInteractiveChart(chartDef), ignoreLabels);

                    if ($container.data('previous-tooltip')) {
                        $container.data('previous-tooltip').destroy();
                    }

                    const yAxesOptions = [];
                    yAxes.forEach(axis => {
                        yAxesOptions.push({
                            id: axis.id,
                            tickValues: axis && axis.tickValues(),
                            tickFormat: axis && axis.tickFormat()
                        });
                    });

                    // Draw axes in every svg
                    const axisOptions = {
                        x: {
                            tickValues: xAxis && xAxis.tickValues(),
                            tickFormat: xAxis && xAxis.tickFormat()
                        },
                        y: yAxesOptions
                    };

                    const yAxesColors = chartData.getYAxesColors(ySpecs, chartDef, chartHandler.getChartTheme());
                    const d3DrawContext = {
                        $svgs,
                        chartDef,
                        chartHandler,
                        ySpecs,
                        xAxis,
                        yAxes,
                        axisOptions,
                        yAxesColors
                    };
                    const { margins, vizWidth, vizHeight } = D3ChartAxes.drawAxes(d3DrawContext);

                    /*
                     * If the legend placement was INNER_TOP_LEFT or INNER_BOTTOM_*, its position depends on the margins (to not overlap with the axes)
                     * Now that all axes have been positioned, we can adjust its placement
                     */
                    ChartLegendUtils.adjustLegendPlacement(chartDef, $container, margins);

                    const measureFormatters = ChartFormatting.createMeasureFormatters(chartDef, chartData, Math.max(vizHeight, vizWidth));

                    const tooltips = ChartTooltips.create($container, chartHandler, chartData, chartDef, measureFormatters, undefined, ignoreLabels);

                    const { store, id } = ChartStoreFactory.getOrCreate(chartDef.$chartStoreId);
                    chartDef.$chartStoreId = id;
                    const contextualMenu = ChartContextualMenu.create(chartData, chartDef, store, chartHandler.animation);


                    // Everything that the chart might need
                    chartBase = {
                        $svgs: $svgs,
                        colorScale: colorScale,
                        margins: margins,
                        vizWidth: vizWidth,
                        vizHeight: vizHeight,
                        xAxis: xAxis,
                        yAxes: yAxes,
                        axisOptions,
                        tooltips: tooltips,
                        contextualMenu,
                        measureFormatters: measureFormatters,
                        chartData: chartData,
                        xSpec: axisSpecs && axisSpecs['x'],
                        ySpecs: ySpecs,
                        colorSpec
                    };

                    $container.data('previous-tooltip', tooltips);

                    angular.extend(chartHandler.tooltips, tooltips);

                    if (chartData.axesDef.animation != undefined) {
                        AnimatedChartsUtils.initAnimation(chartHandler, chartData, chartDef, function(frameIdx) {
                            chartBase.tooltips.setAnimationFrame(frameIdx);
                            return drawFrame(frameIdx, chartBase);
                        }, ignoreLabels);
                    } else {
                        AnimatedChartsUtils.unregisterAnimation(chartHandler);
                    }

                    if (zoomContext && initLinesZoom) {
                        zoomContext = { ...zoomContext, chartBase, svgs: chartBase.$svgs, xAxis };
                        chartDef.$zoomControlInstanceId = initLinesZoom(zoomContext);
                    }

                    $svgs.on('click', function(evt) {
                        if (evt.target.hasAttribute('data-legend') || evt.target.hasAttribute('tooltip-el')) {
                            return;
                        }
                        chartBase.tooltips.resetColors();
                        chartBase.tooltips.unfix();
                    });

                    if (chartData.axesDef.animation != undefined) {
                        // Draw first frame
                        chartHandler.animation.drawFrame(chartHandler.animation.currentFrame || 0, chartBase);
                        if (chartHandler.autoPlayAnimation) {
                            chartHandler.animation.play();
                        }
                    } else {
                        drawFrame(0, chartBase);
                    }
                    /*
                     * Signal to the callee handler that the chart has been loaded.
                     * Dashboards use it to determine when all insights are completely loaded.
                     */
                    if (typeof(chartHandler.loadedCallback) === 'function') {
                        chartHandler.loadedCallback();
                    }
                }, 100);
            });

            return chartBase;
        };

        return svc;
    });

})();

;
// @ts-check
(function() {
    'use strict';
    /** @typedef {import('../types').AxisSpecs } AxisSpecs */
    /** @typedef {import("../types").GeneratedSources.DimensionDef} DimensionDef */
    /** @typedef {import("../types").GeneratedSources.MeasureDef} MeasureDef */
    /** @typedef {import("../types").GeneratedSources.ChartFilter} ChartFilter */
    /** @typedef {import("../types").GeneratedSources.ChartDef} ChartDef */
    /** @typedef {import("../../../../../../../../server//src/frontend/node_modules/ag-grid-enterprise/").GridOptions} GridOptions */
    /** @typedef {import("../../../../../../../../server//src/frontend/node_modules/ag-grid-enterprise/").GridApi} GridApi */


    angular.module('dataiku.charts').service('ChartStoreFactory', chartStoreFactory);

    /**
     * ChartSpecs is a structure which stores charts by frame and facets:
     * @type {
     *  [frameIndex: number]: {
     *      [facetIndex: number]: {
     *          dimensionDefToId: Map<DimensionDef, string>,
     *          measureDefToId: Map<MeasureDef, string>,
     *          axisSpecs: AxisSpecs,
     *          gridOptions: GridOptions,
     *          requestOptions: Record<string, any>
     *      }
     *  }
     * }
     * chartSpecs = {
     *      0: {
     *          0: {
     *              axisSpecs: {...}
     *          }
     *          1: {
     *              axisSpecs: {...}
     *          }
     *      },
     *      1: {
     *          0: {
     *              axisSpecs: {...}
     *          }
     *      }
     * }
     *
     * In order to retrieve a property (like axisSpecs) linked to a specific chart, you can use a convenience method like "get"
     * or retrieve it by using its related frameIndex and facetIndex:
     * const axisSpec = chartSpecs[frameIndex][facetIndex]['axisSpec'];
     *
     * ChartSpecs should register every property related to a chart instead of "chartHandler"
     */
    function ChartSpecs() { }

    /**
     * Insert is an util method to insert properties from a chart in the ChartSpecs structure
     * It automatically creates entries for the related frameIndex, facetIndex and key property
     * @param {number} frameIndex
     * @param {number} facetIndex
     * @param {string} key
     * @param {any} value
     */
    ChartSpecs.prototype.insert = function(frameIndex, facetIndex, key, value) {
        if (!this[frameIndex]) {
            this[frameIndex] = {};
        }

        if (!this[frameIndex][facetIndex]) {
            this[frameIndex][facetIndex] = {};
        }

        this[frameIndex][facetIndex][key] = value;

        return this[frameIndex][facetIndex][key];
    };

    /**
     * Get is an util method to get properties from a chart in the ChartSpecs structure
     * It automatically checks if the related frameIndex and facetIndex exist in the structure before returning the key
     * @param {number} frameIndex
     * @param {number} facetIndex
     * @param {string} key
     * @returns value related to key property in given chart (identified by frame and facet)
     */
    ChartSpecs.prototype.get = function(frameIndex, facetIndex, key) {
        return this[frameIndex] && this[frameIndex][facetIndex] && this[frameIndex][facetIndex][key];
    };

    /**
     * Remove is an util method to remove a property from a chart in the ChartSpecs structure
     * It automatically checks if the related frameIndex and facetIndex exist and deletes associated key
     * @param {number} frameIndex
     * @param {number} facetIndex
     * @param {string} key
     * @returns true if deleted, else false
     */
    ChartSpecs.prototype.remove = function(frameIndex, facetIndex, key) {
        if (this[frameIndex] && this[frameIndex][facetIndex] && this[frameIndex][facetIndex][key]) {
            delete this[frameIndex][facetIndex][key];
            return true;
        }
        return false;
    };

    /**
     * Flatten is an util method to retrieve all charts as a flat array,
     * adding frameIndex and facetIndex as properties of a chart object
     * @returns an array of charts
     */
    ChartSpecs.prototype.flatten = function() {
        return Object.entries(this).reduce((accByFrames, frameEntry) => {
            return Object.entries(frameEntry[1]).reduce((accByFacets, facetEntry) => {
                const value = facetEntry[1];
                value.frame = frameEntry[0];
                value.facet = facetEntry[0];
                accByFacets.push(value);
                return accByFacets;
            }, accByFrames);
        }, []);
    };

    /**
     * ChartStore
     * Store charts information to be shared across the codebase.
     * One chart store = one chart container (chart editor/insight/dashboard tile)
     * Each chart store owns a ChartSpecs structure which registers properties of each chart in the container (can be a solo chart or multiple subcharts)
     */
    function chartStore(CHART_AXIS_TYPES) {
        const chartSpecs = new ChartSpecs();

        return {
            get: function(key, frameIndex = 0, facetIndex = 0) {
                return chartSpecs.get(frameIndex, facetIndex, key);
            },
            set: function(key, value, frameIndex = 0, facetIndex = 0) {
                chartSpecs.insert(frameIndex, facetIndex, key, value);
            },
            /**
             *
             * @param {AxisSpecs} axisSpecs
             * @param {number} frameIndex
             * @param {number} facetIndex
             */
            setAxisSpecs: function(axisSpecs, frameIndex = 0, facetIndex = 0) {
                chartSpecs.insert(frameIndex, facetIndex, 'axisSpecs', axisSpecs);
            },
            getAxisSpecs: function(frameIndex = 0, facetIndex = 0) {
                return chartSpecs.get(frameIndex, facetIndex, 'axisSpecs');
            },
            getAxisSpec: function(axisId, frameIndex = 0, facetIndex = 0) {
                const axisSpecs = chartSpecs.get(frameIndex, facetIndex, 'axisSpecs');
                return axisSpecs && axisSpecs[axisId];
            },
            /**
             * Get the dimension id of `dimension`.
             * @param   {DimensionDef}  dimension
             * @param   {number}        frameIndex
             * @param   {number}        facetIndex
             * @return  {string}        dimension id
             */
            getDimensionId(dimension, frameIndex = 0, facetIndex = 0) {
                let dimensionDefToId = chartSpecs.get(frameIndex, facetIndex, 'dimensionDefToId');

                if (dimensionDefToId === undefined) {
                    dimensionDefToId = chartSpecs.insert(frameIndex, facetIndex, 'dimensionDefToId', new Map());
                }

                return dimensionDefToId.get(dimension);
            },
            /**
             * Set the dimension id of `dimension`.
             * @param {DimensionDef}    dimension
             * @param {number}          frameIndex
             * @param {number}          facetIndex
             */
            setDimensionId(dimension, id, frameIndex = 0, facetIndex = 0) {
                let dimensionDefToId = chartSpecs.get(frameIndex, facetIndex, 'dimensionDefToId');

                if (dimensionDefToId === undefined) {
                    dimensionDefToId = chartSpecs.insert(frameIndex, facetIndex, 'dimensionDefToId', new Map());
                }

                dimensionDefToId.set(dimension, id);
            },
            /**
             * Purge dimension ids
             * @param {number}  frameIndex
             * @param {number}  facetIndex
             */
            purgeDimensionIds(frameIndex = 0, facetIndex = 0) {
                chartSpecs.remove(frameIndex, facetIndex, 'dimensionDefToId');
            },
            /**
             * Get the measure id of `measure`.
             * @param   {MeasureDef}    measure
             * @return  {string}        measure id
             */
            /**
             * Get the measure id of `measure`.
             * @param   {MeasureDef}    measure
             * @param   {number}        frameIndex
             * @param   {number}        facetIndex
             * @return  {string}        measure id
             */
            getMeasureId(measure, frameIndex = 0, facetIndex = 0) {
                let measureDefToId = chartSpecs.get(frameIndex, facetIndex, 'measureDefToId');

                if (measureDefToId === undefined) {
                    measureDefToId = chartSpecs.insert(frameIndex, facetIndex, 'measureDefToId', new Map());
                }

                return measureDefToId.get(measure);
            },
            /**
             * Set the measure id of `measure`.
             * @param {MeasureDef}      measure
             * @param {number}          frameIndex
             * @param {number}          facetIndex
             */
            setMeasureId(measure, id, frameIndex = 0, facetIndex = 0) {
                let measureDefToId = chartSpecs.get(frameIndex, facetIndex, 'measureDefToId');

                if (measureDefToId === undefined) {
                    measureDefToId = chartSpecs.insert(frameIndex, facetIndex, 'measureDefToId', new Map());
                }

                measureDefToId.set(measure, id);
            },
            /**
             * Purge measure ids
             * @param {number}  frameIndex
             * @param {number}  facetIndex
             */
            purgeMeasureIds(frameIndex = 0, facetIndex = 0) {
                chartSpecs.remove(frameIndex, facetIndex, 'measureDefToId');
            },
            /**
             *
             * @param {number} frameIndex
             * @param {number} facetIndex
             * @returns {ChartFilter[]}
             */
            getAppliedDashboardFilters: function(frameIndex = 0, facetIndex = 0) {
                let dashboardFilters = chartSpecs.get(frameIndex, facetIndex, 'dashboardFilters');

                if (dashboardFilters === undefined) {
                    dashboardFilters = chartSpecs.insert(frameIndex, facetIndex, 'dashboardFilters', {});
                }

                return dashboardFilters;
            },
            /**
             *
             * @param {ChartFilter[]} dashboardFilters
             * @param {number} frameIndex
             * @param {number} facetIndex
             */
            setAppliedDashboardFilters: function(dashboardFilters, frameIndex = 0, facetIndex = 0) {
                chartSpecs.insert(frameIndex, facetIndex, 'dashboardFilters', dashboardFilters);
            },
            /**
             *
             * @param {number} frameIndex
             * @param {number} facetIndex
             * @returns {GridApi}
             */
            getGridApi: function(frameIndex = 0, facetIndex = 0) {
                let gridApi = chartSpecs.get(frameIndex, facetIndex, 'gridApi');

                if (gridApi === undefined) {
                    gridApi = chartSpecs.insert(frameIndex, facetIndex, 'gridApi', null);
                }

                return gridApi;
            },
            /**
             *
             * @param {GridApi} gridApi
             * @param {number} frameIndex
             * @param {number} facetIndex
             */
            setGridApi: function(gridApi, frameIndex = 0, facetIndex = 0) {
                chartSpecs.insert(frameIndex, facetIndex, 'gridApi', gridApi);
            },
            /**
             *
             * @param {{ [hierarchyName: string]: string[]}} hierarchyMissingColumns
             * @param {number} frameIndex
             * @param {number} facetIndex
             */
            setHierarchyMissingColumns: function(hierarchyMissingColumns, frameIndex = 0, facetIndex = 0) {
                chartSpecs.insert(frameIndex, facetIndex, 'hierarchyMissingColumns', hierarchyMissingColumns);
            },
            /**
             *
             * @param {number} frameIndex
             * @param {number} facetIndex
             * @returns {{ [hierarchyName: string]: string[]}}
             */
            getHierarchyMissingColumns: function(frameIndex = 0, facetIndex = 0) {
                let hierarchyMissingColumns = chartSpecs.get(frameIndex, facetIndex, 'hierarchyMissingColumns');

                if (hierarchyMissingColumns === undefined) {
                    hierarchyMissingColumns = chartSpecs.insert(frameIndex, facetIndex, 'hierarchyMissingColumns', {});
                }

                return hierarchyMissingColumns;
            },
            /**
             * Returns server request options
             * @param {number} frameIndex
             * @param {number} facetIndex
             * @returns {Record<string, any>} requestOptions
             */
            getRequestOptions: function(frameIndex = 0, facetIndex = 0) {
                let requestOptions = chartSpecs.get(frameIndex, facetIndex, 'requestOptions');

                if (requestOptions === undefined) {
                    requestOptions = chartSpecs.insert(frameIndex, facetIndex, 'requestOptions', {});
                }

                return requestOptions;
            },
            /**
             * Saves server request options
             * @param {Record<string, any>} requestOptions
             */
            setRequestOptions: function(requestOptions, frameIndex = 0, facetIndex = 0) {
                const currentRequestOptions = chartSpecs.get(frameIndex, facetIndex, 'requestOptions') || {};
                chartSpecs.insert(frameIndex, facetIndex, 'requestOptions', { ...currentRequestOptions, ...requestOptions });
            },

            getAllAxisSpecs: function(axisId) {
                const axisSpecs = chartSpecs.flatten().filter(chartSpec => chartSpec.axisSpecs).map(chartSpec => chartSpec.axisSpecs);
                return axisSpecs && axisSpecs.map(axisSpec => axisSpec[axisId]);
            },

            /**
             * Retrieves axis type of an axis (null if not found or not coherent between each subchart)
             * @param {string} axisId
             * @returns {CHART_AXIS_TYPES} axisType
             */
            getAxisType: function(axisId) {
                const axisSpecs = this.getAllAxisSpecs(axisId);
                const uniqueAxisTypes = Array.from(new Set(axisSpecs.map(spec => spec && spec.type)));
                return uniqueAxisTypes.length === 1 ? uniqueAxisTypes[0] : null;
            },

            /**
             * Retrieves dimension type (null if not found or not coherent between each subchart)
             * @param {string} axisId
             */
            getAxisDimensionOrUADimensionType: function(axisId) {
                const axisSpecs = this.getAllAxisSpecs(axisId);
                const uniqueDimensionTypes = Array.from(new Set(axisSpecs.map(spec => spec && spec.dimension && spec.dimension.type)));
                return uniqueDimensionTypes.length === 1 ? uniqueDimensionTypes[0] : null;
            },

            /**
             * Retrieves dimension num params mode if it exists (null if not found or not coherent between each subchart)
             * @param {string} axisId id of an y axis
             */
            getAxisDimensionOrUADimensionNumParamsMode: function(axisId) {
                const axisSpecs = this.getAllAxisSpecs(axisId);
                const uniqueNumParamsMode = Array.from(new Set(axisSpecs.map(spec => spec && spec.dimension && spec.dimension.type && spec.dimension.numParams && spec.dimension.numParams.mode)));
                return uniqueNumParamsMode.length === 1 ? uniqueNumParamsMode[0] : null;
            },

            /**
             * Stores the measures' & dimensions' Ids into the ChartStore
             * @param {ChartDef} chartDef
             */
            updateChartStoreMeasureIDsAndDimensionIDs(chartDef) {
                this.purgeMeasureIds();
                this.purgeDimensionIds();
                chartDef.genericMeasures.forEach((measure, i) => this.setMeasureId(measure, this.getMeasureUniqueId(measure, i)));
                chartDef.yDimension.forEach((yDim, i) => this.setDimensionId(yDim, this.getDimensionUniqueId(yDim, i, 'yDimension')));
                chartDef.xDimension.forEach((xDim, i) => this.setDimensionId(xDim, this.getDimensionUniqueId(xDim, i, 'xDimension')));

                if (chartDef.xHierarchyDimension && chartDef.xHierarchyDimension.length) {
                    chartDef.xHierarchyDimension[0].dimensions.forEach((dim, i) => this.setDimensionId(dim, this.getDimensionUniqueId(dim, i, 'xHierarchyDimension')));
                }
                if (chartDef.yHierarchyDimension && chartDef.yHierarchyDimension.length) {
                    chartDef.yHierarchyDimension[0].dimensions.forEach((dim, i) => this.setDimensionId(dim, this.getDimensionUniqueId(dim, i, 'yHierarchyDimension')));
                }
            },

            /**
             * Generates a unique ID for a dimensions
             * @param {DimensionDef} dimension
             * @param {number} index
             * @param {string} type
             */
            getDimensionUniqueId(dimension, index, type) {
                // Replaces points by underscore because dimension ids are used to build column ids with AGGrid which can't have dots.
                return (`${type}_${index}_${dimension.column}`).replace('.', '_');
            },

            /**
             * Generates a unique ID for a dimensions
             * @param {MeasureDef} measure
             * @param {number} index
             */
            getMeasureUniqueId(measure, index) {
                // Replaces points by underscore because measure ids are used to build column ids with AGGrid which can't have dots.
                return (`measure_${index}_${measure.column}`).replace('.', '_');
            }
        };
    }

    function chartStoreFactory(CHART_AXIS_TYPES) {
        let latestId = 1;
        const chartStores = {};

        const generateId = () => {
            return latestId++;
        };

        const svc = {
            create: () => {
                const id = generateId();
                const store = chartStore(CHART_AXIS_TYPES);
                if (!chartStores[id]) {
                    chartStores[id] = store;
                } else {
                    throw new Error(`${id} already exists, cannot create chart store`);
                }

                return { id, store };
            },
            get: (id) => {
                return chartStores[id];
            },
            getOrCreate: (id) => {
                if (id && chartStores[id]) {
                    return { id, store: svc.get(id) };
                }

                return svc.create();
            }
        };

        return svc;
    }
})();

;
(function() {
    'use strict';

    angular.module('dataiku.charts')
        .factory('ChartTypeChangeHandler', chartTypeChangeHandler);

    /**
     * ChartTypeChangeHandler service
     * Defines what to do when switching from a chart type to another.
     * (!) This file previously was in static/dataiku/js/simple_report/chart_change.js
     */
    function chartTypeChangeHandler(Assert, LoggerProvider, ChartsStaticData, ChartAxesUtils, ChartTypeChangeUtils, ChartColumnTypeUtils, ChartColorUtils, VALUES_DISPLAY_MODES, DKU_PALETTE_NAMES, translate,
        ChartDimension, ChartUADimension, ChartFeatures, ChartMeasure, WebAppsService, PluginConfigUtils, ChartLabels, $rootScope, CHART_TYPES, CHART_VARIANTS, ChartColorSelection, ValuesInChartPlacementMode, ChartHierarchyDimension, TotalBarDisplayModes,
        ChartFilters, RegressionTypes, ValuesInChartOverlappingStrategy, PolygonSources, LineStyleTypes, AxisTicksConfigMode, AxisTicksFormatting, AxisTitleFormatting, GridlinesAxisType, ValueType, DSSVisualizationThemeUtils, ChartDefinitionChangeHandler) {

        const Logger = LoggerProvider.getLogger('charts');
        const COLUMN_TYPE_ORDER = {
            color: ['colorMeasure', 'uaColor'],
            size: ['sizeMeasure', 'uaSize'],
            tooltip: ['tooltipMeasures', 'uaTooltip']
        };

        const DEFAULT_FONT_FORMATTING = {
            fontSize: 11,
            fontColor: '#333',
            hasBackground: false
        };

        const DEFAULT_TABLE_HEADER_FONT_FORMATTING = {
            ...DEFAULT_FONT_FORMATTING,
            fontSize: 12
        };

        const DEFAULT_AUTO_FONT_FORMATTING = {
            fontSize: 11,
            fontColor: 'AUTO'
        };

        const DEFAULT_VALUES_DISPLAY_IN_CHART_TEXT_FORMATTING = {
            ...DEFAULT_AUTO_FONT_FORMATTING,
            hasBackground: false,
            backgroundColor: '#D9D9D9BF'
        };

        const DEFAULT_GAUGE_MIN_MAX = {
            sourceType: ValueType.Constant,
            aggregatedColumn: null,
            datasetColumn: null,
            customAggregation: null,
            aggregation: null
        };

        const accept = function(message) {
            return {
                accept: true,
                message: message
            };
        };

        const reject = function(message) {
            return {
                accept: false,
                message: message
            };
        };

        function setSingleIfHasEnough(srcArr, tgtArr, srcIdx) {
            tgtArr.length = 0;
            if (srcArr.length > srcIdx) {
                tgtArr.push(srcArr[srcIdx]);
            }
        };

        function retrieveGeoFromGeomMap(chartDef) {
            if (chartDef.geometry.length === 0 && chartDef.geoLayers && chartDef.geoLayers.length > 1) {
                chartDef.geometry = angular.copy(chartDef.geoLayers[0].geometry);
            }
        };

        function migrateToGeomMap(chartDef) {
            const uaColor = angular.copy(chartDef.uaColor);
            const colorOptions = angular.copy(chartDef.colorOptions);
            if (chartDef.geometry.length > 0) {
                chartDef.geoLayers.unshift({
                    geometry: angular.copy(chartDef.geometry),
                    uaColor: uaColor,
                    colorOptions: colorOptions
                });
                chartDef.geometry = [];
            } else {
                chartDef.geoLayers[0].colorOptions = colorOptions;
                chartDef.geoLayers[0].uaColor = uaColor;
            }
        }

        function retrieveOptionsFromGeomMap(chartDef) {
            if (chartDef.geoLayers && chartDef.geoLayers.length > 0) {
                chartDef.colorOptions = angular.copy(chartDef.geoLayers[0].colorOptions);
                if (chartDef.geoLayers[0].uaColor) {
                    chartDef.uaColor = angular.copy(chartDef.geoLayers[0].uaColor);
                }
            }
        }

        function computeGenericMeasureInitialValuesInChartDisplayTextFormattingOptions(chartDef) {
            // retrieve all common text formatting properties among displayed measures
            const commonProperties = {};

            const properties = ['fontSize', 'fontColor', 'hasBackground', 'backgroundColor'];
            if (chartDef.genericMeasures) {
                // filter on already initialized measures
                chartDef.genericMeasures.filter(m => m.isA === 'measure').forEach(m => {
                    const initializedProperties = Object.keys(commonProperties);
                    properties.forEach(prop => {
                        if (commonProperties[prop] !== null
                            && m.valuesInChartDisplayOptions
                            && m.valuesInChartDisplayOptions.displayValues
                            && m.valuesInChartDisplayOptions.textFormatting
                            && !_.isNil(m.valuesInChartDisplayOptions.textFormatting[prop])) {
                            if (!initializedProperties.includes(prop)) {
                                commonProperties[prop] = m.valuesInChartDisplayOptions.textFormatting[prop];
                            } else if (commonProperties[prop] !== m.valuesInChartDisplayOptions.textFormatting[prop]) {
                                // set it to null if at least 2 different values
                                commonProperties[prop] = null;
                            }
                        }
                    });
                });
            }

            return {
                ...DEFAULT_VALUES_DISPLAY_IN_CHART_TEXT_FORMATTING,
                ..._.omitBy(commonProperties, _.isNil)
            };
        }

        function computeGenericMeasureInitialValuesInChartDisplayOptions(chartDef) {
            // retrieve all common text formatting properties among displayed measures
            const properties = ['placementMode', 'spacing'];
            const commonProperties = {};

            if (!chartDef.genericMeasures) {
                return commonProperties;
            }

            for (const measure of chartDef.genericMeasures) {
                if (measure.isA !== 'measure' || !measure.valuesInChartDisplayOptions) {
                    continue;
                }

                for (const prop of properties) {
                    const value = measure.valuesInChartDisplayOptions[prop];

                    if (_.isNil(value)) {
                        continue;
                    }
                    if (!(prop in commonProperties)) {
                        commonProperties[prop] = value;
                    } else if (commonProperties[prop] !== value) {
                        // set it to null if at least 2 different values
                        commonProperties[prop] = null;
                    }
                }
            }

            return _.omitBy(commonProperties, _.isNil);
        }

        /**
         * Return one or more attribute(s) based on the chartDefAttributeName and removes them from the "chartDefAttributes" array
         * We use the following method because all non-returned measures are moved in tooltip
         * @param {object[]} chartDefAttributes
         * @param {string[]} chartDefAttributeNames
         * @param {boolean} multiple
         * @returns {*[]}
         */
        function getAttributeAndRemove(chartDefAttributes, chartDefAttributeNames = [], multiple = false) {
            if (Array.isArray(chartDefAttributes) && Array.isArray(chartDefAttributeNames)) {
                const response = [];
                const indexes = [];

                for (const chartDefAttributeName of chartDefAttributeNames) {
                    for (const [index, column] of chartDefAttributes.entries()) {
                        if (column && column.chartDefAttributeName === chartDefAttributeName) {
                            response.push(column);
                            indexes.push(index);
                            if (!multiple) {
                                break;
                            }
                        }
                    }
                }

                // Mandatory to delete several array element without indexes problems
                indexes.sort(descendingNumericSort).forEach((index) => chartDefAttributes.splice(index, 1));

                return response;
            }
            return [];
        }

        function autocompleteUA(ua) {
            if (ua.isA !== 'ua') {
                ua.sortBy = 'NATURAL';
                ua.isA = 'ua';
                ua.multiplier = ChartsStaticData.DEFAULT_MULTIPLIER;
                if (ua.type === 'DATE') {
                    ua.dateMode = 'RANGE';
                }
                if (ua.type === 'NUMERICAL') {
                    ua.digitGrouping = ChartsStaticData.DEFAULT_DIGIT_GROUPING;
                    ua.useParenthesesForNegativeValues = false;
                    ua.shouldFormatInPercentage = false;
                    ua.hideTrailingZeros = getDefaultTrailingZeros(ua);
                }
            }
            if (!ua.$id) {
                ua.$id = generateUniqueId();
            }
        }

        function newColorOptions(index) {
            const colorOptions = getDefaultColorOption(index);
            initializeColorPalettes(colorOptions);
            return colorOptions;
        }

        function getDefaultColorOption(index) {
            const circularColorPalette = window.dkuColorPalettes.discrete[0].sample;
            return { singleColor: circularColorPalette[index % circularColorPalette.length], transparency: 0.75 };
        }

        function initializeColorPalettes(colorOptions) {
            colorOptions.customPalette = ChartColorUtils.getDefaultCustomPalette();
            colorOptions.ccScaleMode = 'NORMAL';
            colorOptions.paletteType = 'CONTINUOUS';
            colorOptions.quantizationMode = 'NONE';
            colorOptions.numQuantizeSteps = 5;
            colorOptions.paletteMiddleValue = 0;
            if (colorOptions.paletteMiddleValue <= 0 && colorOptions.paletteType === 'DIVERGING' && colorOptions.ccScaleMode === 'LOG') {
                colorOptions.paletteMiddleValue = 1;
            }
            colorOptions.customColors = {};
        }

        // Initializing the necessary properties
        function fixupAxes(chartDef, theme) {

            // x axis
            if (_.isNil(chartDef.xAxisFormatting) || !Object.keys(chartDef.xAxisFormatting).length) {
                chartDef.xAxisFormatting = {
                    axisTitleFormatting: { ...AxisTitleFormatting },
                    customExtent: { ...ChartsStaticData.DEFAULT_CUSTOM_EXTENT }
                };
                if (theme) {
                    DSSVisualizationThemeUtils.applyAxisTitleFormatting(theme, chartDef.xAxisFormatting.axisTitleFormatting);
                }
            }
            if (_.isNil(chartDef.xAxisFormatting.displayAxis)) {
                chartDef.xAxisFormatting.displayAxis = true;
            }
            if (_.isNil(chartDef.xAxisFormatting.showAxisTitle)) {
                chartDef.xAxisFormatting.showAxisTitle = true;
            }
            if (_.isNil(chartDef.xAxisFormatting.axisValuesFormatting)) {
                chartDef.xAxisFormatting.axisValuesFormatting = {
                    axisTicksFormatting: { ...AxisTicksFormatting },
                    numberFormatting: { ...ChartsStaticData.DEFAULT_NUMBER_FORMATTING_OPTIONS }
                };
                if (theme) {
                    DSSVisualizationThemeUtils.applyAxisValueFormatting(theme, chartDef.xAxisFormatting.axisValuesFormatting);
                }
            }
            if (_.isNil(chartDef.xAxisFormatting.axisValuesFormatting.axisTicksFormatting)) {
                chartDef.xAxisFormatting.axisValuesFormatting.axisTicksFormatting = { ...AxisTicksFormatting };
                if (theme) {
                    DSSVisualizationThemeUtils.applyAxisValueFormatting(theme, chartDef.xAxisFormatting.axisValuesFormatting);
                }
            }
            if (_.isNil(chartDef.xAxisFormatting.axisValuesFormatting.numberFormatting)) {
                chartDef.xAxisFormatting.axisValuesFormatting.numberFormatting = { ...ChartsStaticData.DEFAULT_NUMBER_FORMATTING_OPTIONS };
            }
            if (_.isNil(chartDef.xAxisFormatting.axisValuesFormatting.numberFormatting.digitGrouping)) {
                chartDef.xAxisFormatting.axisValuesFormatting.numberFormatting.digitGrouping = ChartsStaticData.DEFAULT_DIGIT_GROUPING;
            }
            if (_.isNil(chartDef.xAxisFormatting.axisValuesFormatting.numberFormatting.useParenthesesForNegativeValues)) {
                chartDef.xAxisFormatting.axisValuesFormatting.numberFormatting.useParenthesesForNegativeValues = false;
            }
            if (_.isNil(chartDef.xAxisFormatting.axisValuesFormatting.numberFormatting.shouldFormatInPercentage)) {
                chartDef.xAxisFormatting.axisValuesFormatting.numberFormatting.shouldFormatInPercentage = false;
            }
            if (_.isNil(chartDef.xAxisFormatting.axisValuesFormatting.numberFormatting.hideTrailingZeros)) {
                chartDef.xAxisFormatting.axisValuesFormatting.numberFormatting.hideTrailingZeros = getDefaultTrailingZeros(chartDef.xAxisFormatting.axisValuesFormatting.numberFormatting);
            }
            if (_.isNil(chartDef.xAxisFormatting.axisTitleFormatting)) {
                chartDef.xAxisFormatting.axisTitleFormatting = { ...AxisTitleFormatting };
                if (theme) {
                    DSSVisualizationThemeUtils.applyAxisTitleFormatting(theme, chartDef.xAxisFormatting.axisTitleFormatting);
                }
            }
            if (_.isNil(chartDef.xAxisFormatting.ticksConfig)) {
                chartDef.xAxisFormatting.ticksConfig = { mode: AxisTicksConfigMode.INTERVAL, number: null };
            }
            if (_.isNil(chartDef.xAxisFormatting.customExtent) || !Object.keys(chartDef.xAxisFormatting.customExtent).length) {
                chartDef.xAxisFormatting.customExtent = { ...ChartsStaticData.DEFAULT_CUSTOM_EXTENT };
            }


            // y axis
            if (_.isNil(chartDef.yAxesFormatting) || !chartDef.yAxesFormatting.length) {
                chartDef.yAxesFormatting = [];
            }

            const leftAxisFormatting = chartDef.yAxesFormatting.find(formatting => formatting.id === ChartsStaticData.LEFT_AXIS_ID || _.isNil(formatting.id));
            const rightAxisFormatting = chartDef.yAxesFormatting.find(formatting => formatting.id === ChartsStaticData.RIGHT_AXIS_ID);
            if (!leftAxisFormatting || !Object.keys(leftAxisFormatting).length) {
                chartDef.yAxesFormatting.push({
                    id: ChartsStaticData.LEFT_AXIS_ID
                });
            }
            if (!rightAxisFormatting || !Object.keys(rightAxisFormatting).length) {
                chartDef.yAxesFormatting.push({
                    id: ChartsStaticData.RIGHT_AXIS_ID
                });
            }

            chartDef.yAxesFormatting.forEach(axisFormatting => {
                if (_.isNil(axisFormatting.id)) {
                    // left axis is the only one that can have no id at initialization due to mapping from old charts
                    axisFormatting.id = ChartsStaticData.LEFT_AXIS_ID;
                }
                if (_.isNil(axisFormatting.displayAxis)) {
                    axisFormatting.displayAxis = true;
                }
                if (_.isNil(axisFormatting.showAxisTitle)) {
                    axisFormatting.showAxisTitle = true;
                }
                if (_.isNil(axisFormatting.axisValuesFormatting)) {
                    axisFormatting.axisValuesFormatting = {
                        axisTicksFormatting: { ...AxisTicksFormatting },
                        numberFormatting: { ...ChartsStaticData.DEFAULT_NUMBER_FORMATTING_OPTIONS }
                    };
                    if (theme) {
                        DSSVisualizationThemeUtils.applyAxisValueFormatting(theme, axisFormatting.axisValuesFormatting);
                    }

                }
                if (_.isNil(axisFormatting.axisValuesFormatting.axisTicksFormatting)) {
                    axisFormatting.axisValuesFormatting.axisTicksFormatting = { ...AxisTicksFormatting };
                    if (theme) {
                        DSSVisualizationThemeUtils.applyAxisValueFormatting(theme, axisFormatting.axisValuesFormatting);
                    }
                }
                if (_.isNil(axisFormatting.axisValuesFormatting.numberFormatting)) {
                    axisFormatting.axisValuesFormatting.numberFormatting = { ...ChartsStaticData.DEFAULT_NUMBER_FORMATTING_OPTIONS };
                }
                if (_.isNil(axisFormatting.axisValuesFormatting.numberFormatting.digitGrouping)) {
                    axisFormatting.axisValuesFormatting.numberFormatting.digitGrouping = ChartsStaticData.DEFAULT_DIGIT_GROUPING;
                }
                if (_.isNil(axisFormatting.axisValuesFormatting.numberFormatting.useParenthesesForNegativeValues)) {
                    axisFormatting.axisValuesFormatting.numberFormatting.useParenthesesForNegativeValues = false;
                }
                if (_.isNil(axisFormatting.axisValuesFormatting.numberFormatting.shouldFormatInPercentage)) {
                    axisFormatting.axisValuesFormatting.numberFormatting.shouldFormatInPercentage = false;
                }
                if (_.isNil(axisFormatting.axisValuesFormatting.numberFormatting.hideTrailingZeros)) {
                    axisFormatting.axisValuesFormatting.numberFormatting.hideTrailingZeros = getDefaultTrailingZeros(axisFormatting.axisValuesFormatting.numberFormatting);
                }
                if (_.isNil(axisFormatting.axisTitleFormatting)) {
                    axisFormatting.axisTitleFormatting = { ...AxisTitleFormatting };
                    if (theme) {
                        DSSVisualizationThemeUtils.applyAxisTitleFormatting(theme, axisFormatting.axisTitleFormatting);
                    }
                }
                if (_.isNil(axisFormatting.ticksConfig)) {
                    axisFormatting.ticksConfig = { mode: AxisTicksConfigMode.INTERVAL, number: null };
                }
                if (_.isNil(axisFormatting.customExtent) || !Object.keys(axisFormatting.customExtent).length) {
                    axisFormatting.customExtent = { ...ChartsStaticData.DEFAULT_CUSTOM_EXTENT };
                }
                if (_.isNil(axisFormatting.includeZero)) {
                    axisFormatting.includeZero = true;
                }
            });


            // radial axis
            if (!chartDef.radialAxisFormatting || !chartDef.radialAxisFormatting.axisValuesFormatting || !chartDef.radialAxisFormatting.axisValuesFormatting.axisTicksFormatting) {
                // before we were storing the values formatting in the titles formatting
                const legacyFormatting = chartDef.radialAxisFormatting && chartDef.radialAxisFormatting.axisTitleFormatting && _.cloneDeep(chartDef.radialAxisFormatting.axisTitleFormatting);
                chartDef.radialAxisFormatting = {
                    axisValuesFormatting: {
                        axisTicksFormatting: {
                            ...AxisTitleFormatting,
                            ...(legacyFormatting || {})
                        }
                    }
                };
                if (theme && !legacyFormatting) {
                    DSSVisualizationThemeUtils.applyAxisValueFormatting(theme, chartDef.radialAxisFormatting.axisValuesFormatting);
                };
            }
            if (_.isNil(chartDef.tooltipOptions)) {
                chartDef.tooltipOptions = { display: true };
            }

            // gauge options
            if (_.isNil(chartDef.gaugeOptions)) {
                chartDef.gaugeOptions = {
                    min: { ...DEFAULT_GAUGE_MIN_MAX },
                    max: { ...DEFAULT_GAUGE_MIN_MAX },
                    displayPointer: false
                };
            }
            if (_.isNil(chartDef.gaugeOptions.min)) {
                chartDef.gaugeOptions.min = { ...DEFAULT_GAUGE_MIN_MAX };
            }
            if (_.isNil(chartDef.gaugeOptions.max)) {
                chartDef.gaugeOptions.max = { ...DEFAULT_GAUGE_MIN_MAX };
            }
            if (_.isNil(chartDef.gaugeOptions.min.aggregatedColumn)) {
                chartDef.gaugeOptions.min.aggregatedColumn = null;
            }
            if (_.isNil(chartDef.gaugeOptions.min.datasetColumn)) {
                chartDef.gaugeOptions.min.datasetColumn = null;
            }
            if (_.isNil(chartDef.gaugeOptions.min.customAggregation)) {
                chartDef.gaugeOptions.min.customAggregation = null;
            }
            if (_.isNil(chartDef.gaugeOptions.min.aggregation)) {
                chartDef.gaugeOptions.min.aggregation = null;
            }
            if (_.isNil(chartDef.gaugeOptions.max.aggregatedColumn)) {
                chartDef.gaugeOptions.max.aggregatedColumn = null;
            }
            if (_.isNil(chartDef.gaugeOptions.max.datasetColumn)) {
                chartDef.gaugeOptions.max.datasetColumn = null;
            }
            if (_.isNil(chartDef.gaugeOptions.max.customAggregation)) {
                chartDef.gaugeOptions.max.customAggregation = null;
            }
            if (_.isNil(chartDef.gaugeOptions.max.aggregation)) {
                chartDef.gaugeOptions.max.aggregation = null;
            }
            if (_.isNil(chartDef.gaugeOptions.axis)) {
                chartDef.gaugeOptions.axis = {
                    ticksConfig: {
                        mode: AxisTicksConfigMode.INTERVAL,
                        number: null
                    },
                    axisValuesFormatting: {
                        axisTicksFormatting: {
                            ...AxisTicksFormatting,
                            fontSize: 16
                        },
                        numberFormatting: { ...ChartsStaticData.DEFAULT_NUMBER_FORMATTING_OPTIONS }
                    },
                    thickness: 30
                };
                if (theme) {
                    DSSVisualizationThemeUtils.applyAxisValueFormatting(theme, chartDef.gaugeOptions.axis?.axisValuesFormatting);
                }
            }
            if (_.isNil(chartDef.gaugeOptions.axis.ticksConfig)) {
                chartDef.gaugeOptions.axis.ticksConfig = { mode: AxisTicksConfigMode.INTERVAL, number: null };
            }
            if (_.isNil(chartDef.gaugeOptions.axis.ticksConfigMode)) {
                chartDef.gaugeOptions.axis.ticksConfigMode = AxisTicksConfigMode.INTERVAL;
            }
            if (_.isNil(chartDef.gaugeOptions.axis.axisValuesFormatting)) {
                chartDef.gaugeOptions.axis.axisValuesFormatting = {
                    axisTicksFormatting: {
                        ...AxisTicksFormatting,
                        fontSize: 12
                    },
                    numberFormatting: { ...ChartsStaticData.DEFAULT_NUMBER_FORMATTING_OPTIONS }
                };
                if (theme) {
                    DSSVisualizationThemeUtils.applyAxisValueFormatting(theme, chartDef.gaugeOptions.axis.axisValuesFormatting);
                }

            }
            if (_.isNil(chartDef.gaugeOptions.axis.axisValuesFormatting.numberFormatting)) {
                chartDef.gaugeOptions.axis.axisValuesFormatting.numberFormatting = { ...ChartsStaticData.DEFAULT_NUMBER_FORMATTING_OPTIONS };
            }
            if (_.isNil(chartDef.gaugeOptions.axis.axisValuesFormatting.numberFormatting.digitGrouping)) {
                chartDef.gaugeOptions.axis.axisValuesFormatting.numberFormatting.digitGrouping = ChartsStaticData.DEFAULT_DIGIT_GROUPING;
            }
            if (_.isNil(chartDef.gaugeOptions.axis.axisValuesFormatting.numberFormatting.useParenthesesForNegativeValues)) {
                chartDef.gaugeOptions.axis.axisValuesFormatting.numberFormatting.useParenthesesForNegativeValues = false;
            }
            if (_.isNil(chartDef.gaugeOptions.axis.axisValuesFormatting.numberFormatting.shouldFormatInPercentage)) {
                chartDef.gaugeOptions.axis.axisValuesFormatting.numberFormatting.shouldFormatInPercentage = false;
            }
            if (_.isNil(chartDef.gaugeOptions.axis.axisValuesFormatting.numberFormatting.hideTrailingZeros)) {
                chartDef.gaugeOptions.axis.axisValuesFormatting.numberFormatting.hideTrailingZeros = getDefaultTrailingZeros(chartDef.gaugeOptions.axis.axisValuesFormatting.numberFormatting);
            }
        }

        function fixupNumberFormatting(formattableOptions) {
            if (!_.isNil(formattableOptions)) {
                formattableOptions.forEach(formattableOption => {

                    if (_.isNil(formattableOption.numberFormatting)) {
                        formattableOption.numberFormatting = { ...ChartsStaticData.DEFAULT_REFERENCE_LINES_NUMBER_FORMATTING_OPTIONS };
                    }

                    if (!_.isNil(formattableOption.prefix)) {
                        formattableOption.numberFormatting.prefix = formattableOption.prefix;
                        delete formattableOption.prefix;
                    }

                    if (!_.isNil(formattableOption.suffix)) {
                        formattableOption.numberFormatting.suffix = formattableOption.suffix;
                        delete formattableOption.suffix;
                    }

                    if (!_.isNil(formattableOption.multiplier)) {
                        formattableOption.numberFormatting.multiplier = formattableOption.multiplier;
                        delete formattableOption.multiplier;
                    }
                });
            }
        }

        function ensureReferenceLinesCompatibility(chartType, chartVariant, referenceLines) {
            if (!_.isNil(referenceLines)) {
                referenceLines.forEach(referenceLine => {
                    // revert type to constant for reference line of type AggregatedColumn on Unnaggregated charts
                    if ((chartType === CHART_TYPES.SCATTER || chartType === CHART_TYPES.SCATTER_MULTIPLE_PAIRS || chartVariant === CHART_VARIANTS.waterfall)
                        && referenceLine.sourceType === ValueType.AggregatedColumn) {
                        referenceLine.sourceType = ValueType.Constant;
                        referenceLine.constantValue = undefined;
                    }
                });
            }
        }

        function fixupReferenceLines(chartDef) {
            // Number Formatting in reference lines was not using NumberFormattingOptions before 13.2.0
            fixupNumberFormatting(chartDef.referenceLines);
        }

        function fixupGaugeTargets(chartDef) {
            // Number Formatting in gauge targets was not using NumberFormattingOptions before 13.2.0
            fixupNumberFormatting(chartDef.gaugeOptions.targets);
        }

        function fixupDynamicMeasure(dynamicMeasure) {
            if (!dynamicMeasure) {
                return;
            }

            dynamicMeasure.aggregatedColumn && fixPercentile(dynamicMeasure.aggregatedColumn);
            dynamicMeasure.datasetColumn && fixPercentile(dynamicMeasure.datasetColumn);
            dynamicMeasure.customAggregation && fixPercentile(dynamicMeasure.customAggregation);
        }

        function fixPercentile(value) {
            if (_.isNil(value.percentile) || value.percentile === 0) {
                value.percentile = 50;
            }
        }

        function fixupPivotTableFormatting(tableFormatting, theme) {

            if (!_.isNil(tableFormatting.rowSubheaders)) {
                return; // Fixup already done, 13.4 structure already there
            }

            // Headers formatting has been splitted into main headers and subheaders in 13.4.0
            if (!_.isNil(tableFormatting.rowHeaders)) {
                tableFormatting.rowSubheaders = { ...tableFormatting.rowHeaders };
                // Conserve < 13.4.0 row headers unfreezed status
                tableFormatting.freezeRowHeaders = false;
            } else {
                tableFormatting.rowSubheaders = { ...DEFAULT_TABLE_HEADER_FONT_FORMATTING, fontColor: theme ? theme.generalFormatting.fontColor : DEFAULT_TABLE_HEADER_FONT_FORMATTING.fontColor };
                tableFormatting.freezeRowHeaders = true; // Starting 13.4.0, row headers will be frozen by default
            }

            if (!_.isNil(tableFormatting.columnSubheaders)) {
                return; // Fixup already done, 13.4 structure already there
            }

            if (!_.isNil(tableFormatting.columnHeaders)) {
                tableFormatting.columnMainHeaders = { ...tableFormatting.columnHeaders };
                tableFormatting.columnSubheaders = { ...tableFormatting.columnHeaders };
            } else {
                tableFormatting.rowMainHeaders = { ...DEFAULT_TABLE_HEADER_FONT_FORMATTING, fontColor: theme ? theme.generalFormatting.fontColor : DEFAULT_TABLE_HEADER_FONT_FORMATTING.fontColor };
                tableFormatting.columnMainHeaders = { ...DEFAULT_TABLE_HEADER_FONT_FORMATTING, fontColor: theme ? theme.generalFormatting.fontColor : DEFAULT_TABLE_HEADER_FONT_FORMATTING.fontColor };
                tableFormatting.columnSubheaders = { ...DEFAULT_TABLE_HEADER_FONT_FORMATTING, fontColor: theme ? theme.generalFormatting.fontColor : DEFAULT_TABLE_HEADER_FONT_FORMATTING.fontColor };
            }
        }

        // By default, we trail zeros if no decimal places are specified, else we do
        function getDefaultTrailingZeros(numberFormattingOptions) {
            return !_.isNil(numberFormattingOptions.decimalPlaces) && numberFormattingOptions.decimalPlaces > 0 ? false : true;
        }

        function autocompleteGenericMeasure({ contextualMenuMeasureType, chartDef, defaultTextFormatting, theme }, measure, index, array) {
            const chartType = chartDef.type;
            if (!ChartLabels.aggregationIsSupported(measure.function, measure.type, chartType, contextualMenuMeasureType)) {
                measure.function = 'COUNT';
            }

            if (measure.isUnaggregated && !ChartFeatures.canUseUnaggregatedMeasures(chartDef, measure.column, measure.type, measure.displayType, contextualMenuMeasureType)) {
                measure.isUnaggregated = false;
            }

            if (measure.isA !== 'measure') {
                Assert.trueish(measure.type, 'no measure type');
                const col = measure.column;
                const type = measure.type;
                const measureFn = measure.function;
                const measureInferredType = measure.inferredType;
                clear(measure);
                if (col === '__COUNT__') {
                    measure.column = null;
                    measure.function = 'COUNT';
                } else if (type === 'ALPHANUM') {
                    measure.column = col;
                    measure.function = 'COUNTD';
                } else if (type === 'DATE') {
                    measure.column = col;
                    measure.function = 'COUNT';
                } else if (type === 'CUSTOM') {
                    measure.column = col;
                    measure.function = 'CUSTOM';
                    measure.customFunction = measureFn;
                    measure.inferredType = measureInferredType;
                } else {
                    measure.column = col;
                    measure.function = chartDef.variant === CHART_VARIANTS.waterfall ? 'SUM' : 'AVG';
                }
                measure.type = type;
                measure.displayed = true;
                measure.displayAxis = 'axis1';
                measure.displayType = chartDef.type == CHART_TYPES.MULTI_COLUMNS_LINES && array.some(m => m.displayType === 'column') ? 'line' : 'column';

                measure.isA = 'measure';
                measure.percentile = 50;
                measure.valueTextFormatting = { ...DEFAULT_FONT_FORMATTING };
                measure.labelTextFormatting = { fontColor: '#333333', fontSize: 15 };
                measure.prefix = '';
                measure.suffix = '';
                measure.decimalPlaces = null;

                measure.digitGrouping = ChartsStaticData.DEFAULT_DIGIT_GROUPING;
                measure.useParenthesesForNegativeValues = false;
                measure.shouldFormatInPercentage = false;
                measure.hideTrailingZeros = getDefaultTrailingZeros(measure);
                measure.multiplier = ChartsStaticData.DEFAULT_MULTIPLIER;
                const textFormatting = defaultTextFormatting || computeGenericMeasureInitialValuesInChartDisplayTextFormattingOptions(chartDef);
                let spacing = 12;
                switch (chartDef.type) {
                    case CHART_TYPES.GROUPED_COLUMNS :
                        spacing = 2;
                        break;
                    case CHART_TYPES.MULTI_COLUMNS_LINES :
                        spacing = 5;
                        break;
                    case CHART_TYPES.LINES :
                        spacing = 12;
                        break;

                    default:
                        break;
                }
                const otherComputedProperties = computeGenericMeasureInitialValuesInChartDisplayOptions(chartDef);

                measure.valuesInChartDisplayOptions = {
                    displayValues: chartType !== CHART_TYPES.TREEMAP,
                    textFormatting,
                    additionalMeasures: [],
                    addDetails: false,
                    placementMode: ValuesInChartPlacementMode.AUTO,
                    spacing,
                    ...otherComputedProperties
                };
                if (theme) {
                    DSSVisualizationThemeUtils.applyToMeasure(chartType, measure, theme);
                }
            }

            if (_.isNil(measure.isUnaggregated)) {
                measure.isUnaggregated = (chartDef.type === CHART_TYPES.GROUPED_COLUMNS && chartDef.variant === CHART_VARIANTS.waterfall);
            }
            const availableModes = Object.keys(ChartLabels.getAvailableUnaggregatedModesLabels(measure, chartDef, contextualMenuMeasureType));
            if (_.isNil(measure.uaComputeMode) || (availableModes.length > 0 && !availableModes.includes(measure.uaComputeMode))) {
                measure.uaComputeMode = availableModes[0];
            }
            if (!measure.$id) {
                measure.$id = generateUniqueId();
            }

            fixPercentile(measure);

            // Measure-compute modes
            const measureComputationModes = ChartFeatures.getMeasureComputationModes(chartDef);
            const modeEnabled = measureComputationModes.some(m => m[0] === measure.computeMode);

            // Revert to 'NORMAL' if the compute mode is invalid
            if (!modeEnabled) {
                measure.computeMode = 'NORMAL';
            }
        }

        function fixupStackedColumnsOptions(chartDef, theme) {
            if (ChartFeatures.canDisplayTotalValues(chartDef)) {
                if (_.isNil(chartDef.stackedColumnsOptions)) {
                    chartDef.stackedColumnsOptions = {
                        totalsInChartDisplayOptions: {
                            textFormatting: chartDef.valuesInChartDisplayOptions.textFormatting,
                            additionalMeasures: [],
                            addDetails: false
                        }
                    };
                }
                if (_.isNil(chartDef.stackedColumnsOptions.totalsInChartDisplayOptions)) {
                    chartDef.stackedColumnsOptions.totalsInChartDisplayOptions = {
                        textFormatting: chartDef.valuesInChartDisplayOptions.textFormatting,
                        additionalMeasures: [],
                        addDetails: false
                    };
                }
                if (_.isNil(chartDef.stackedColumnsOptions.totalsInChartDisplayOptions.additionalMeasures)) {
                    chartDef.stackedColumnsOptions.totalsInChartDisplayOptions.additionalMeasures = [];
                }

                if (_.isNil(chartDef.stackedColumnsOptions.totalsInChartDisplayOptions.spacing)) {
                    chartDef.stackedColumnsOptions.totalsInChartDisplayOptions.spacing = 5;
                }
                chartDef.stackedColumnsOptions.totalsInChartDisplayOptions.additionalMeasures.forEach(autocompleteGenericMeasure.bind(this, { contextualMenuMeasureType: 'valuesInChart', chartDef, defaultTextFormatting: chartDef.stackedColumnsOptions.totalsInChartDisplayOptions.textFormatting, theme }));
            }
        }

        function autocompleteHierarchyDimensions(chartDef, hierarchy) {
            if (!hierarchy.id) {
                hierarchy.id = _.uniqueId(`${Date.now().toString()}_`);
            }
            if (_.isNil(hierarchy.level)) {
                hierarchy.level = ChartHierarchyDimension.getHierarchyFirstUsableDimensionIdx(chartDef.$chartStoreId, hierarchy);
            }
            if (!hierarchy.selectedValues) {
                hierarchy.selectedValues = {};
            }
            if (!hierarchy.filterIds) {
                hierarchy.filterIds = {};
            }
            hierarchy.dimensions.forEach((dimension) => {
                dimension.hierarchyId = hierarchy.id;
                autocompleteGenericDimension(chartDef, dimension);
            });
        };

        function clearGenericOrHierarchyDimension(chartDef, changedDefinition, dimensionPropName, hierarchyPropName) {
            if (changedDefinition && changedDefinition.name === dimensionPropName && changedDefinition.nv.length) {
                chartDef[hierarchyPropName] = [];
            }
            if (changedDefinition && changedDefinition.name === hierarchyPropName && changedDefinition.nv.length) {
                chartDef[dimensionPropName] = [];
            }
        };

        function purgeHierarchyFilters(chartDef) {
            const hierarchyProps = ChartsStaticData.HIERARCHY_DIMENSIONS_PROPERTIES;
            let existingHierarchyFilters = [];
            hierarchyProps.forEach(prop => {
                chartDef[prop].forEach(hierarchy => {
                    const hierarchyFilters = Object.keys(hierarchy.filterIds).map(key => hierarchy.filterIds[key]);
                    existingHierarchyFilters = [...existingHierarchyFilters, ...hierarchyFilters];
                });
            });
            chartDef.filters = chartDef.filters.filter(filter => !filter.hierarchyId || existingHierarchyFilters.includes(filter.id));
        }

        function autocompleteGenericDimension(chartDef, dimension, listKey) {
            if (dimension.isA !== 'dimension') {
                Assert.trueish(dimension.column, 'no dimension column');
                Assert.trueish(dimension.type, 'no dimension type');
                const col = dimension.column;
                const type = dimension.type;
                clear(dimension);
                dimension.column = col;
                dimension.type = type;
                if (dimension.type === 'DATE') {
                    dimension.numParams = { emptyBinsMode: 'AVERAGE' };
                    dimension.dateParams = { mode: ChartsStaticData.defaultDateMode.value };
                } else if (dimension.type === 'NUMERICAL') {
                    /*
                     * Compute a proper number of bins :
                     * - Second dimension of a stack or group : 5
                     * - First dimension of an area or lines : 30
                     * - Scatter or first dimension of an histogram: 10
                     */
                    const cdt = chartDef.type;
                    const isMainDimension = [CHART_TYPES.GROUPED_COLUMNS, CHART_TYPES.STACKED_BARS, CHART_TYPES.STACKED_COLUMNS].includes(cdt)
                        && (dimension === chartDef.genericDimension0[0] || (chartDef.genericHierarchyDimension[0] && chartDef.genericHierarchyDimension[0].dimensions.includes(dimension)));
                    const nbBins = ChartDimension.getNumericalBinNumber(cdt, isMainDimension);
                    dimension.numParams = {
                        mode: 'FIXED_NB',
                        nbBins: nbBins,
                        binSize: 100,
                        emptyBinsMode: 'ZEROS'
                    };
                    dimension.maxValues = 100; // Will only be used if no binning
                } else if (dimension.type === 'ALPHANUM') {
                    dimension.numParams = { emptyBinsMode: 'ZEROS' };
                    dimension.maxValues = 20;
                }
                dimension.generateOthersCategory = true;
                dimension.filters = [];
                dimension.isA = 'dimension';

                if (dimension.type === 'GEOPOINT' && chartDef.type.find('_map') < 0) {
                    chartDef.type = CHART_TYPES.GRID_MAP;
                }
                if (dimension.type === 'GEOMETRY' && chartDef.type.find('_map') < 0) {
                    chartDef.type = CHART_TYPES.GRID_MAP;
                }

                if (listKey === 'genericDimension1'
                    && ChartFeatures.canIgnoreEmptyBinsForColorDimension(chartDef)
                    && (chartDef.genericDimension0.length === 1 || chartDef.genericHierarchyDimension.length === 1)
                    && ChartDimension.getGenericDimension(chartDef).column === dimension.column) {
                    dimension.ignoreEmptyBins = true;
                }

                dimension.digitGrouping = ChartsStaticData.DEFAULT_DIGIT_GROUPING;
                dimension.useParenthesesForNegativeValues = false;
                dimension.shouldFormatInPercentage = false;
                dimension.hideTrailingZeros = getDefaultTrailingZeros(dimension);
                dimension.multiplier = ChartsStaticData.DEFAULT_MULTIPLIER;
                dimension.oneTickPerBin = 'AUTO';
            }
            if (!dimension.$id) {
                dimension.$id = generateUniqueId();
            }
        };


        const svc = {
            autocompleteUA,
            autocompleteGenericDimension: autocompleteGenericDimension,
            onChartTypeChange: function(chartDef, newType, newVariant, newWebAppType) {
                Logger.info('Start type change:' + JSON.stringify(chartDef));

                let existingMeasures;
                let existingDimensions;
                let existingPrimaryDimensions;
                let existingSecondaryDimensions;
                let allMeasures;
                let allDimensions;
                let allUA;
                const oldType = chartDef.type;

                /*
                 * Reset user defined custom range on chart type change:
                 * Based on chart type, dimensions can move from x or y axes or other measures and break the intented range presentation (ex: from Scatter Plot to Grouped Bubbles)
                 */
                ChartAxesUtils.resetCustomExtents([chartDef.xAxisFormatting]);
                ChartAxesUtils.resetCustomExtents(chartDef.yAxesFormatting);

                if (oldType === CHART_TYPES.GEOMETRY_MAP) {
                    retrieveOptionsFromGeomMap(chartDef);
                    retrieveGeoFromGeomMap(chartDef);
                    chartDef.geoLayers = [svc.newEmptyGeoPlaceholder(0)];
                }

                ensureReferenceLinesCompatibility(newType, newVariant, chartDef.referenceLines);

                switch (newType) {
                    case CHART_TYPES.MULTI_COLUMNS_LINES:
                    case CHART_TYPES.GROUPED_COLUMNS:
                    case CHART_TYPES.STACKED_COLUMNS:
                    case CHART_TYPES.STACKED_BARS:
                    case CHART_TYPES.LINES:
                    case CHART_TYPES.STACKED_AREA:
                        existingPrimaryDimensions = ChartTypeChangeUtils.takeAllExistingPrimaryDimensions(chartDef);
                        existingSecondaryDimensions = ChartTypeChangeUtils.takeAllExistingSecondaryDimensions(chartDef);

                        ChartTypeChangeUtils.setDimensionsOrHierarachies(chartDef,
                            [['genericDimension0', 'genericHierarchyDimension']],
                            existingPrimaryDimensions, existingSecondaryDimensions);

                        existingDimensions = [...existingPrimaryDimensions, ...existingSecondaryDimensions];
                        if (existingDimensions.length && ChartFeatures.canUseSecondGenericDimension({ ...chartDef, type: newType, variant: newVariant, webAppType: newWebAppType })) {
                            chartDef.genericDimension1 = existingDimensions.splice(0, 1);
                        }

                        existingMeasures = ChartTypeChangeUtils.takeAllMeasuresWithUA(chartDef).filter(m => svc.stdAggregatedAcceptMeasure(m).accept);
                        chartDef.tooltipMeasures = getAttributeAndRemove(existingMeasures, COLUMN_TYPE_ORDER.tooltip, true);
                        // eslint-disable-next-line no-undef
                        if (has(chartDef.genericDimension0) || has(chartDef.genericHierarchyDimension)) {
                            // eslint-disable-next-line no-undef
                            if (ChartFeatures.canSupportMultipleGenericMeasures({ ...chartDef, type: newType, variant: newVariant, webAppType: newWebAppType })) {
                                chartDef.genericMeasures = existingMeasures;
                            } else {
                                chartDef.genericMeasures = existingMeasures.slice(0, 1);
                            }
                        }
                        break;
                    case CHART_TYPES.PIE:
                        existingMeasures = ChartTypeChangeUtils.takeAllMeasuresWithUA(chartDef).filter(m => svc.stdAggregatedAcceptMeasure(m).accept);
                        existingPrimaryDimensions = ChartTypeChangeUtils.takeAllExistingPrimaryDimensions(chartDef);
                        existingSecondaryDimensions = ChartTypeChangeUtils.takeAllExistingSecondaryDimensions(chartDef);

                        if (existingMeasures.length >= 1) {
                            chartDef.genericMeasures = existingMeasures.slice(0, 1);
                        }

                        ChartTypeChangeUtils.setDimensionsOrHierarachies(chartDef,
                            [['genericDimension0', 'genericHierarchyDimension']],
                            existingPrimaryDimensions, existingSecondaryDimensions);
                        break;
                    case CHART_TYPES.SCATTER:
                    case CHART_TYPES.HEATMAP:
                        existingDimensions = ChartTypeChangeUtils.takeAllExistingDimensions(chartDef);
                        setSingleIfHasEnough(existingDimensions, chartDef.uaXDimension, 0);
                        setSingleIfHasEnough(existingDimensions, chartDef.uaYDimension, 1);

                        //we don't need them, but we need to purge them
                        ChartTypeChangeUtils.takeAllExistingHierarchies(chartDef);

                        allUA = ChartTypeChangeUtils.takeAllUAWithMeasures(chartDef);
                        // Using the strict assignment just on the same kind of chart
                        if ([CHART_TYPES.SCATTER, CHART_TYPES.HEATMAP].includes(chartDef.type)) {
                            chartDef.uaColor = getAttributeAndRemove(allUA, COLUMN_TYPE_ORDER.color);
                            chartDef.uaSize = getAttributeAndRemove(allUA, COLUMN_TYPE_ORDER.size);
                            chartDef.uaTooltip = allUA;
                        } else {
                            // puts the dimensions that can be accepted as uaColor at the front of the list for easier selection
                            const acceptedScaleFirst = allUA.sort((a, b) => svc.scatterAcceptScaleMeasure(b).accept - svc.scatterAcceptScaleMeasure(a).accept);
                            if (acceptedScaleFirst.length && svc.scatterAcceptScaleMeasure(acceptedScaleFirst[0]).accept) {
                                setSingleIfHasEnough(acceptedScaleFirst, chartDef.uaSize, 0);
                                setSingleIfHasEnough(acceptedScaleFirst, chartDef.uaColor, 1);
                            } else {
                                setSingleIfHasEnough(allUA, chartDef.uaColor, 0);
                            }

                            chartDef.uaTooltip = allUA.slice(2);
                        }
                        break;
                    case CHART_TYPES.SCATTER_MULTIPLE_PAIRS:
                        existingDimensions = ChartTypeChangeUtils.takeAllExistingDimensions(chartDef);
                        if (existingDimensions.length) {
                            chartDef.uaDimensionPair = [
                                {
                                    uaXDimension: [],
                                    uaYDimension: []
                                }
                            ];
                            const possibleXDimensions = existingDimensions.filter(d => svc.scatterMPAccept(d, chartDef, 'x', 0).accept);
                            const possibleYDimensions = existingDimensions.filter(d => svc.scatterMPAccept(d, chartDef, 'y', 0).accept
                                && (!possibleXDimensions[0] || possibleXDimensions[0].column !== d.column));
                            setSingleIfHasEnough(possibleXDimensions, chartDef.uaDimensionPair[0].uaXDimension, 0);
                            setSingleIfHasEnough(possibleYDimensions, chartDef.uaDimensionPair[0].uaYDimension, 0);
                        }
                        //we don't need them, but we need to purge them
                        ChartTypeChangeUtils.takeAllUAWithMeasures(chartDef);
                        ChartTypeChangeUtils.takeAllExistingHierarchies(chartDef);
                        break;
                    case CHART_TYPES.GROUPED_XY:
                        /*
                         * TODO: If we come from scatter, maybe we could make a smarter
                         * choice by taking dimX -> AVG -> measX, dimY -> AVG -> measY
                         */
                        existingPrimaryDimensions = ChartTypeChangeUtils.takeAllExistingPrimaryDimensions(chartDef);
                        existingSecondaryDimensions = ChartTypeChangeUtils.takeAllExistingSecondaryDimensions(chartDef);

                        ChartTypeChangeUtils.setDimensionsOrHierarachies(chartDef,
                            [['groupDimension', 'groupHierarchyDimension']],
                            existingPrimaryDimensions, existingSecondaryDimensions);

                        allMeasures = ChartTypeChangeUtils.takeAllMeasuresWithUA(chartDef).filter(m => svc.stdAggregatedAcceptMeasure(m).accept);

                        setSingleIfHasEnough(allMeasures, chartDef.xMeasure, 0);
                        setSingleIfHasEnough(allMeasures, chartDef.yMeasure, 1);
                        setSingleIfHasEnough(allMeasures, chartDef.sizeMeasure, 2);
                        setSingleIfHasEnough(allMeasures, chartDef.colorMeasure, 3);
                        break;

                    case CHART_TYPES.BINNED_XY:
                        // xDim, yDim, colorMeasure, sizeM
                        existingPrimaryDimensions = ChartTypeChangeUtils.takeAllExistingPrimaryDimensions(chartDef).filter(d => svc.binnedXYAcceptDimensionOrHierarchy(newVariant, d).accept);
                        existingSecondaryDimensions = ChartTypeChangeUtils.takeAllExistingSecondaryDimensions(chartDef).filter(d => svc.binnedXYAcceptDimensionOrHierarchy(newVariant, d).accept);

                        ChartTypeChangeUtils.setDimensionsOrHierarachies(chartDef,
                            [['xDimension', 'xHierarchyDimension'], ['yDimension', 'yHierarchyDimension']],
                            existingPrimaryDimensions, existingSecondaryDimensions,
                            newVariant !== CHART_VARIANTS.binnedXYHexagon);

                        allMeasures = ChartTypeChangeUtils.takeAllMeasuresWithUA(chartDef).filter(m => svc.stdAggregatedAcceptMeasure(m).accept);
                        // Using the strict assignment just on the same kind of chart
                        if (newType === oldType) {
                            chartDef.colorMeasure = getAttributeAndRemove(allMeasures, COLUMN_TYPE_ORDER.color);
                            chartDef.sizeMeasure = getAttributeAndRemove(allMeasures, COLUMN_TYPE_ORDER.size);
                            chartDef.tooltipMeasures = allMeasures;
                        } else {
                            setSingleIfHasEnough(allMeasures, chartDef.colorMeasure, 0);
                            setSingleIfHasEnough(allMeasures, chartDef.sizeMeasure, 1);
                            chartDef.tooltipMeasures = allMeasures.slice(2);
                        }
                        break;

                    case CHART_TYPES.PIVOT_TABLE:
                        // xDim, yDim, genericMeasures
                        existingPrimaryDimensions = ChartTypeChangeUtils.takeAllExistingPrimaryDimensions(chartDef);
                        existingSecondaryDimensions = ChartTypeChangeUtils.takeAllExistingSecondaryDimensions(chartDef);

                        ChartTypeChangeUtils.setDimensionsOrHierarachies(chartDef,
                            [['xDimension', 'xHierarchyDimension'], ['yDimension', 'yHierarchyDimension']],
                            existingPrimaryDimensions, existingSecondaryDimensions);

                        allMeasures = ChartTypeChangeUtils.takeAllMeasuresWithUA(chartDef).filter(m => svc.stdAggregatedAcceptMeasureWithAlphanumResults(m).accept);
                        chartDef.genericMeasures = allMeasures;
                        break;

                    case CHART_TYPES.SCATTER_MAP:
                    case CHART_TYPES.DENSITY_HEAT_MAP:
                    case CHART_TYPES.HEATMAP_MAP:
                        allUA = ChartTypeChangeUtils.takeAllUAWithMeasures(chartDef).filter(m => svc.stdAggregatedAcceptMeasure(m).accept);
                        // Using the strict assignment just on the same kind of chart
                        if ([CHART_TYPES.SCATTER_MAP, CHART_TYPES.DENSITY_HEAT_MAP, CHART_TYPES.HEATMAP_MAP, CHART_TYPES.GEOMETRY_MAP, CHART_TYPES.GRID_MAP].includes(oldType)) {
                            chartDef.uaColor = getAttributeAndRemove(allUA, COLUMN_TYPE_ORDER.color);
                            chartDef.uaSize = getAttributeAndRemove(allUA, COLUMN_TYPE_ORDER.size);
                            chartDef.uaTooltip = allUA;
                        } else {
                            setSingleIfHasEnough(allUA, chartDef.uaSize, 0);
                            setSingleIfHasEnough(allUA, chartDef.uaColor, 1);
                            chartDef.uaTooltip = allUA.slice(2);
                        }

                        //we don't need them, but we need to purge them
                        ChartTypeChangeUtils.takeAllExistingHierarchies(chartDef);
                        break;
                    case CHART_TYPES.GEOMETRY_MAP:
                        allUA = ChartTypeChangeUtils.takeAllUAWithMeasures(chartDef).filter(m => svc.stdAggregatedAcceptMeasure(m).accept);
                        chartDef.uaColor = getAttributeAndRemove(allUA, COLUMN_TYPE_ORDER.color);
                        chartDef.uaSize = getAttributeAndRemove(allUA, COLUMN_TYPE_ORDER.size);
                        chartDef.uaTooltip = allUA;
                        migrateToGeomMap(chartDef);

                        //we don't need them, but we need to purge them
                        ChartTypeChangeUtils.takeAllExistingHierarchies(chartDef);
                        break;
                    case CHART_TYPES.GROUPED_SCATTER_MAP:
                        // eslint-disable-next-line no-undef
                        if (!has(chartDef.groupDimension)) {
                            allDimensions = ChartTypeChangeUtils.takeAllExistingDimensions(chartDef);
                            setSingleIfHasEnough(allDimensions, chartDef.groupDimension, 0);
                        }
                        allMeasures = ChartTypeChangeUtils.takeAllMeasuresWithUA(chartDef).filter(m => svc.stdAggregatedAcceptMeasure(m).accept);
                        // Using the strict assignment just on the same kind of chart
                        if (newType === oldType) {
                            chartDef.colorMeasure = getAttributeAndRemove(allMeasures, COLUMN_TYPE_ORDER.color);
                            chartDef.sizeMeasure = getAttributeAndRemove(allMeasures, COLUMN_TYPE_ORDER.size);
                            chartDef.tooltipMeasures = allMeasures;
                        } else {
                            setSingleIfHasEnough(allMeasures, chartDef.sizeMeasure, 0);
                            setSingleIfHasEnough(allMeasures, chartDef.colorMeasure, 1);
                            chartDef.tooltipMeasures = allMeasures.slice(2);
                        }
                        //we don't need them, but we need to purge them
                        ChartTypeChangeUtils.takeAllExistingHierarchies(chartDef);
                        break;
                    case CHART_TYPES.GRID_MAP:
                        allMeasures = ChartTypeChangeUtils.takeAllMeasuresWithUA(chartDef).filter(m => svc.stdAggregatedAcceptMeasure(m).accept);
                        setSingleIfHasEnough(allMeasures, chartDef.colorMeasure, 0);
                        chartDef.tooltipMeasures = allMeasures.slice(1);
                        //we don't need them, but we need to purge them
                        ChartTypeChangeUtils.takeAllExistingHierarchies(chartDef);
                        break;
                    case CHART_TYPES.ADMINISTRATIVE_MAP:
                        /*
                         * TODO: for admin map if we are in FILLED mode then we only
                         * have a color measure, no size, so we should probably inverse
                         */
                        allMeasures = ChartTypeChangeUtils.takeAllMeasuresWithUA(chartDef).filter(m => svc.stdAggregatedAcceptMeasure(m).accept);
                        // Using the strict assignment just on the same kind of chart
                        if (newType === oldType) {
                            chartDef.colorMeasure = getAttributeAndRemove(allMeasures, COLUMN_TYPE_ORDER.color);
                            chartDef.sizeMeasure = getAttributeAndRemove(allMeasures, COLUMN_TYPE_ORDER.size);
                            chartDef.tooltipMeasures = allMeasures;
                        } else {
                            setSingleIfHasEnough(allMeasures, chartDef.sizeMeasure, 0);
                            setSingleIfHasEnough(allMeasures, chartDef.colorMeasure, 1);
                            chartDef.tooltipMeasures = allMeasures.slice(2);
                        }
                        //we don't need them, but we need to purge them
                        ChartTypeChangeUtils.takeAllExistingHierarchies(chartDef);
                        break;
                    case CHART_TYPES.TREEMAP:
                        allMeasures = ChartTypeChangeUtils.takeAllMeasuresWithUA(chartDef).filter(m => svc.stdAggregatedAcceptMeasure(m).accept);
                        setSingleIfHasEnough(allMeasures, chartDef.genericMeasures, 0);
                        setSingleIfHasEnough(allMeasures, chartDef.colorMeasure, 1);

                        existingPrimaryDimensions = ChartTypeChangeUtils.takeAllExistingPrimaryDimensions(chartDef);
                        existingSecondaryDimensions = ChartTypeChangeUtils.takeAllExistingSecondaryDimensions(chartDef);
                        ChartTypeChangeUtils.setDimensionsOrHierarachies(chartDef,
                            [['yDimension', 'yHierarchyDimension']],
                            existingPrimaryDimensions, existingSecondaryDimensions, true, true);
                        break;
                    case CHART_TYPES.DENSITY_2D:
                        existingDimensions = ChartTypeChangeUtils.takeAllExistingDimensions(chartDef).filter(d => svc.density2dAccept(d).accept);
                        setSingleIfHasEnough(existingDimensions, chartDef.xDimension, 0);
                        setSingleIfHasEnough(existingDimensions, chartDef.yDimension, 1);
                        //we don't need them, but we need to purge them
                        ChartTypeChangeUtils.takeAllExistingHierarchies(chartDef);
                        break;

                    case CHART_TYPES.GAUGE:
                        ChartTypeChangeUtils.takeAllExistingDimensions(chartDef);
                        existingMeasures = ChartTypeChangeUtils.takeAllMeasuresWithUA(chartDef);
                        existingMeasures = existingMeasures.filter(m => svc.stdAggregatedAcceptMeasureWithAlphanumResults(m).accept);
                        setSingleIfHasEnough(existingMeasures, chartDef.genericMeasures, 0);
                        //we don't need them, but we need to purge them
                        ChartTypeChangeUtils.takeAllExistingHierarchies(chartDef);
                        break;

                    case CHART_TYPES.KPI:
                        ChartTypeChangeUtils.takeAllExistingDimensions(chartDef);
                        existingMeasures = ChartTypeChangeUtils.takeAllMeasuresWithUA(chartDef);
                        chartDef.genericMeasures = existingMeasures.filter(m => svc.stdAggregatedAcceptMeasureWithAlphanumResults(m).accept);
                        //we don't need them, but we need to purge them
                        ChartTypeChangeUtils.takeAllExistingHierarchies(chartDef);
                        break;

                    case CHART_TYPES.RADAR:
                        existingMeasures = ChartTypeChangeUtils.takeAllMeasuresWithUA(chartDef).filter(m => svc.stdAggregatedAcceptMeasureWithAlphanumResults(m).accept);
                        chartDef.genericMeasures = existingMeasures;

                        existingPrimaryDimensions = ChartTypeChangeUtils.takeAllExistingPrimaryDimensions(chartDef);
                        existingSecondaryDimensions = ChartTypeChangeUtils.takeAllExistingSecondaryDimensions(chartDef);
                        ChartTypeChangeUtils.setDimensionsOrHierarachies(chartDef,
                            [['genericDimension0', 'genericHierarchyDimension']],
                            existingPrimaryDimensions, existingSecondaryDimensions);
                        break;

                    case CHART_TYPES.SANKEY:
                        existingMeasures = ChartTypeChangeUtils.takeAllMeasuresWithUA(chartDef).filter(m => svc.stdAggregatedAcceptMeasureWithAlphanumResults(m).accept);
                        setSingleIfHasEnough(existingMeasures, chartDef.genericMeasures, 0);

                        existingDimensions = ChartTypeChangeUtils.takeAllExistingDimensions(chartDef);
                        chartDef.yDimension = existingDimensions;
                        //we don't need them, but we need to purge them
                        ChartTypeChangeUtils.takeAllExistingHierarchies(chartDef);
                        break;

                    case CHART_TYPES.BOXPLOTS:
                        existingPrimaryDimensions = ChartTypeChangeUtils.takeAllExistingPrimaryDimensions(chartDef);
                        existingSecondaryDimensions = ChartTypeChangeUtils.takeAllExistingSecondaryDimensions(chartDef);

                        ChartTypeChangeUtils.setDimensionsOrHierarachies(chartDef,
                            [['boxplotBreakdownDim', 'boxplotBreakdownHierarchyDimension']],
                            existingPrimaryDimensions, existingSecondaryDimensions);

                        existingDimensions = [...existingPrimaryDimensions, ...existingSecondaryDimensions];
                        if (existingDimensions.length) {
                            chartDef.genericDimension1 = existingDimensions.splice(0, 1);
                        }

                        allUA = ChartTypeChangeUtils.takeAllUAWithMeasures(chartDef).filter(m => svc.boxplotsAcceptMeasure(m).accept);
                        setSingleIfHasEnough(allUA, chartDef.boxplotValue, 0);
                        break;
                    case CHART_TYPES.LIFT:
                        existingPrimaryDimensions = ChartTypeChangeUtils.takeAllExistingPrimaryDimensions(chartDef);
                        existingSecondaryDimensions = ChartTypeChangeUtils.takeAllExistingSecondaryDimensions(chartDef);

                        ChartTypeChangeUtils.setDimensionsOrHierarachies(chartDef,
                            [['groupDimension', 'groupHierarchyDimension']],
                            existingPrimaryDimensions, existingSecondaryDimensions);

                        allMeasures = ChartTypeChangeUtils.takeAllMeasuresWithUA(chartDef).filter(m => svc.stdAggregatedAcceptMeasureWithAlphanumResults(m).accept);
                        setSingleIfHasEnough(allMeasures, chartDef.xMeasure, 0);
                        setSingleIfHasEnough(allMeasures, chartDef.yMeasure, 1);
                        break;

                    case CHART_TYPES.WEBAPP:
                        // all custom
                        break;

                    case CHART_TYPES.NUMERICAL_HEATMAP:
                    default:
                        throw Error('unimplemented chart type : ' + newType);
                }

                if (!ChartFeatures.canAnimate(newType, newVariant)) {
                    chartDef.animationDimension.length = 0;
                }

                if (!ChartFeatures.canFacet(newType, newWebAppType, newVariant)) {
                    chartDef.facetDimension.length = 0;
                }
            },

            /* ********************** ACCEPT / REJECT drops per type ********************* */

            stdAggregatedAcceptDimensionOrHierarchy: function(data) {
                if (ChartColumnTypeUtils.isNonColumnData(data)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.NON_COL_DATA);
                }
                // Reject count of records
                if (ChartColumnTypeUtils.isCount(data)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.NO_AGGR_ON_COUNT_REC);
                }
                if (ChartColumnTypeUtils.isGeoColumnType(data.type)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.GEO_DATA);
                }
                if (ChartColumnTypeUtils.isCustomColumnType(data.type)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.NON_CUSTOM_COL);
                }
                return accept();
            },

            stdAggregatedAcceptDimension: function(data) {
                if (ChartColumnTypeUtils.isHierarchy(data)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.HIERARCHY);
                }
                return svc.stdAggregatedAcceptDimensionOrHierarchy(data);
            },

            stdAggregatedAcceptMeasure: function(data) {
                if (ChartColumnTypeUtils.isNonColumnData(data)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.NON_COL_DATA);
                }
                if (ChartColumnTypeUtils.isHierarchy(data)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.HIERARCHY);
                }
                if (ChartColumnTypeUtils.isRBND(data)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.NON_RBND);
                }
                if (ChartColumnTypeUtils.isGeoColumnType(data.type)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.GEO_DATA);
                }
                if (ChartColumnTypeUtils.isDateColumnType(data.type)) {
                    return accept();
                }
                if (ChartColumnTypeUtils.isCustomMeasureAndNonNumerical(data)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.NON_NUM_CUSTOM_COL);
                }
                if (!ChartColumnTypeUtils.isNumericalColumnType(data.type) && !ChartColumnTypeUtils.isAlphanumColumnType(data.type) && !ChartColumnTypeUtils.isCustomColumnType(data.type)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.NON_NUM_ALPHA);
                }
                return accept();
            },

            stdAggregatedAcceptMeasureWithAlphanumResults: function(data) {
                if (ChartColumnTypeUtils.isNonColumnData(data)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.NON_COL_DATA);
                }
                if (ChartColumnTypeUtils.isHierarchy(data)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.HIERARCHY);
                }
                if (ChartColumnTypeUtils.isRBND(data)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.NON_RBND);
                }
                if (ChartColumnTypeUtils.isGeoColumnType(data.type)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.GEO_DATA);
                }
                if (ChartColumnTypeUtils.isDateColumnType(data.type)) {
                    return accept();
                }
                if (!ChartColumnTypeUtils.isNumericalColumnType(data.type) && !ChartColumnTypeUtils.isAlphanumColumnType(data.type) && !ChartColumnTypeUtils.isCustomColumnType(data.type)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.NON_NUM_ALPHA);
                }
                if (ChartMeasure.isRealUnaggregatedMeasure(data)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.NON_AGGR_MEASURE);
                }
                return accept();
            },

            uaTooltipAccept: function(data) {
                if (ChartColumnTypeUtils.isCount(data)) {
                    return reject(`${ChartLabels.CHART_ERROR_MESSAGES.COUNT_REC} in tooltips`);
                }
                if (ChartColumnTypeUtils.isHierarchy(data)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.HIERARCHY);
                }
                if (ChartColumnTypeUtils.isRBND(data)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.NON_RBND);
                }
                if (ChartColumnTypeUtils.isGeoColumnType(data.type)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.GEO_DATA);
                }
                if (ChartColumnTypeUtils.isCustomColumnType(data.type)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.NON_CUSTOM_COL);
                }
                return accept();
            },


            scatterAccept: function(data) {
                if (ChartColumnTypeUtils.isNonColumnData(data)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.NON_COL_DATA);
                }
                if (ChartColumnTypeUtils.isHierarchy(data)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.HIERARCHY);
                }
                if (ChartColumnTypeUtils.isRBND(data)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.NON_RBND);
                }
                if (ChartColumnTypeUtils.isCount(data)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.COUNT_REC);
                }
                if (ChartColumnTypeUtils.isGeoColumnType(data.type)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.GEO_DATA);
                }
                if (ChartColumnTypeUtils.isCustomColumnType(data.type)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.NON_CUSTOM_COL);
                }
                return accept();
            },

            scatterMPAccept: function(data, chartDef, dimension, pairIndex) {
                const index = parseInt(pairIndex);
                const { firstXDim, firstXDimIndex } = svc.getScatterFirstXDims(chartDef.uaDimensionPair, index);
                if (dimension === 'x') {
                    if (!_.isNil(firstXDim) && firstXDimIndex !== index) {
                        const sameType = ChartColumnTypeUtils.isSameColumnType(data, firstXDim);
                        if (!sameType) {
                            return reject(ChartLabels.CHART_ERROR_MESSAGES.COL_TYPE_X);
                        }
                    }
                    if (ChartColumnTypeUtils.isAlphanumColumnType(data.type)) {
                        return reject(ChartLabels.CHART_ERROR_MESSAGES.SCATTER_MP_ALPHANUM_X);
                    }
                    return svc.scatterAccept(data);
                }

                return svc.scatterAccept(data);
            },

            scatterAcceptScaleMeasure: function(data) {
                if (ChartColumnTypeUtils.isAlphanumColumnType(data.type)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.SCATTER_SCALE_NON_NUM);
                }
                return svc.scatterAccept(data);
            },

            densityMapAcceptScaleMeasure: function(data) {
                return svc.scatterAcceptScaleMeasure(data);
            },

            acceptGeo: function(data) {
                if (ChartColumnTypeUtils.isNonColumnData(data)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.NON_COL_DATA);
                }
                if (ChartColumnTypeUtils.isGeoColumnType(data.type)) {
                    return accept();
                } else {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.ONLY_GEO);
                }
            },

            density2dAccept: function(data) {
                if (ChartColumnTypeUtils.isNonColumnData(data)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.NON_COL_DATA);
                }
                if (ChartColumnTypeUtils.isHierarchy(data)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.HIERARCHY);
                }
                if (ChartColumnTypeUtils.isRBND(data)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.NON_RBND);
                }
                if (ChartColumnTypeUtils.isCount(data)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.COUNT_REC);
                }
                if (ChartColumnTypeUtils.isNumericalColumnType(data.type)
                    || ChartColumnTypeUtils.isDateColumnType(data.type)) {
                    return accept();
                } else {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.DENSITY_NUM_OR_DATE);
                }
            },

            boxplotsAcceptBreakdown: function(data) {
                if (ChartColumnTypeUtils.isNonColumnData(data)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.NON_COL_DATA);
                }
                if (ChartColumnTypeUtils.isCount(data)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.COUNT_REC);
                }
                if (ChartColumnTypeUtils.isGeoColumnType(data.type)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.GEO_DATA);
                }
                if (ChartColumnTypeUtils.isCustomColumnType(data.type)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.NON_CUSTOM_COL);
                }
                if (ChartColumnTypeUtils.isRBND(data)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.NON_RBND);
                }
                return accept();
            },
            boxplotsAcceptColorDimension: function(data) {
                if (ChartColumnTypeUtils.isHierarchy(data)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.HIERARCHY);
                }
                return svc.boxplotsAcceptBreakdown(data);
            },
            boxplotsAcceptMeasure: function(data) {
                if (ChartColumnTypeUtils.isNonColumnData(data)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.NON_COL_DATA);
                }
                if (ChartColumnTypeUtils.isHierarchy(data)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.HIERARCHY);
                }
                if (ChartColumnTypeUtils.isCustomColumnType(data.type)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.NON_CUSTOM_COL);
                }
                if (!ChartColumnTypeUtils.isNumericalColumnType(data.type)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.NON_NUM_COL);
                }
                if (ChartColumnTypeUtils.isCount(data)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.COUNT_REC);
                }
                if (ChartColumnTypeUtils.isRBND(data)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.NON_RBND);
                }
                return accept();
            },
            binnedXYAcceptDimensionOrHierarchy: function(chartVariant, data) {
                Assert.trueish(chartVariant, 'no chartDef variant');
                if (ChartColumnTypeUtils.isNonColumnData(data)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.NON_COL_DATA);
                }
                if (chartVariant === CHART_VARIANTS.binnedXYHexagon && ChartColumnTypeUtils.isHierarchy(data)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.HIERARCHY);
                }
                if (chartVariant === CHART_VARIANTS.binnedXYHexagon && !ChartColumnTypeUtils.isNumericalColumnType(data.type)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.HEX_BINNING_DIM);
                }
                if (ChartColumnTypeUtils.isCustomColumnType(data.type)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.NON_CUSTOM_COL);
                }
                return svc.stdAggregatedAcceptDimensionOrHierarchy(data);
            },

            getScatterFirstXDims(pairs, pairIndex) {
                let firstXDim;
                let firstXDimIndex;
                for (let i = 0; i < pairs.length; i++) {
                    if (i !== pairIndex) {
                        if (pairs[i].uaXDimension && pairs[i].uaXDimension.length && _.isNil(firstXDim)) {
                            firstXDim = pairs[i].uaXDimension[0];
                            firstXDimIndex = i;
                        }
                        if (!_.isNil(firstXDim)) {
                            return { firstXDim, firstXDimIndex };
                        }
                    }
                }
                return { firstXDim, firstXDimIndex };
            },

            computeAutoName: function(chartDef) {
                // If returns null or empty, don't use the new name.

                function getAutoNameMeasureLabel(measure) {
                    return measure.function === 'COUNT' ? 'Count' : ChartLabels.getShortMeasureLabel(measure, false);
                }

                function addGM0L() {
                    // eslint-disable-next-line no-undef
                    if (has(chartDef.genericMeasures)) {
                        newName += getAutoNameMeasureLabel(chartDef.genericMeasures[0]);
                    }
                }

                function sdl(d) {
                    return d.column;
                }

                function sml(m) {
                    return getAutoNameMeasureLabel(m[0]);
                }

                let dimensionPairs;
                let newName = '';
                let genericDimension, xDimension, yDimension, groupDimension, breakdownDimension;
                switch (chartDef.type) {
                    case CHART_TYPES.MULTI_COLUMNS_LINES:
                    case CHART_TYPES.GROUPED_COLUMNS:
                    case CHART_TYPES.STACKED_COLUMNS:
                    case CHART_TYPES.STACKED_BARS:
                    case CHART_TYPES.LINES:
                    case CHART_TYPES.STACKED_AREA:
                    case CHART_TYPES.PIE:
                        // eslint-disable-next-line no-undef
                        genericDimension = ChartDimension.getGenericDimension(chartDef);
                        // eslint-disable-next-line no-undef
                        if (genericDimension && !has(chartDef.genericDimension1)) {
                            addGM0L();
                            newName += ' by ';
                            newName += ChartLabels.getDimensionLabel(genericDimension, false);
                            // eslint-disable-next-line no-undef
                        } else if (genericDimension && has(chartDef.genericDimension1)) {
                            addGM0L();
                            newName += ' by ';
                            newName += ChartLabels.getDimensionLabel(genericDimension, false);
                            newName += ' and ';
                            newName += ChartLabels.getDimensionLabel(chartDef.genericDimension1[0], false);
                        }
                        break;
                    case CHART_TYPES.PIVOT_TABLE:
                        // eslint-disable-next-line no-undef
                        xDimension = ChartDimension.getXDimension(chartDef, 'x');
                        // eslint-disable-next-line no-undef
                        yDimension = ChartDimension.getYDimension(chartDef, 'y');
                        // eslint-disable-next-line no-undef
                        if ((xDimension || yDimension) && has(chartDef.genericMeasures)) {
                            addGM0L();
                            let i = 0;
                            [xDimension, yDimension].forEach(dimension => {
                                if (dimension) {
                                    newName += [' by ', ' and '][i];
                                    newName += sdl(dimension);
                                    i++;
                                }
                            });
                        }
                        break;
                    case CHART_TYPES.RADAR:
                        // eslint-disable-next-line no-undef
                        genericDimension = ChartDimension.getGenericDimension(chartDef);

                        // eslint-disable-next-line no-undef
                        if (genericDimension && has(chartDef.genericMeasures)) {
                            chartDef.genericMeasures.forEach((dim, i) => {
                                newName += dim.column;
                                if (i === chartDef.genericMeasures.length - 2) {
                                    newName += ' and ';
                                } else if (i < chartDef.genericMeasures.length - 1) {
                                    newName += ', ';
                                }
                            });
                            newName += ' by ' + ChartLabels.getDimensionLabel(genericDimension, false);
                        }
                        break;
                    case CHART_TYPES.TREEMAP:
                        yDimension = ChartHierarchyDimension.getCurrentHierarchyDimension(chartDef, 'y');
                        // eslint-disable-next-line no-undef
                        yDimension = yDimension ? [yDimension] : has(chartDef.yDimension) ? chartDef.yDimension : [];
                        // eslint-disable-next-line no-undef
                        if (yDimension.length && has(chartDef.genericMeasures)) {
                            addGM0L();
                            newName += ' by ';
                            yDimension.forEach((dim, i) => {
                                newName += dim.column;
                                if (i === yDimension.length - 2) {
                                    newName += ' and ';
                                } else if (i < yDimension.length - 1) {
                                    newName += ', ';
                                }
                            });
                        }
                        break;
                    case CHART_TYPES.BINNED_XY:
                        // eslint-disable-next-line no-undef
                        xDimension = ChartDimension.getXDimension(chartDef);
                        // eslint-disable-next-line no-undef
                        yDimension = ChartDimension.getYDimension(chartDef);
                        if (xDimension && yDimension) {
                            newName += sdl(xDimension) + ' vs ' + sdl(yDimension) + ' (aggregated)';
                        }
                        break;
                    case CHART_TYPES.GROUPED_XY:
                        // eslint-disable-next-line no-undef
                        groupDimension = ChartDimension.getGroupDimension(chartDef);
                        // eslint-disable-next-line no-undef
                        if (has(chartDef.xMeasure) && has(chartDef.yMeasure) && groupDimension) {
                            newName += sml(chartDef.xMeasure) + ' / ' + sml(chartDef.yMeasure) + ' by ' +
                                sdl(groupDimension);
                        }
                        break;
                    case CHART_TYPES.SCATTER:
                        newName = `${sdl(chartDef.uaXDimension[0])} vs ${sdl(chartDef.uaYDimension[0])}`;
                        break;
                    case CHART_TYPES.SCATTER_MULTIPLE_PAIRS:
                        dimensionPairs = ChartUADimension.getDisplayableDimensionPairs(chartDef.uaDimensionPair);
                        newName = `${sdl(ChartUADimension.getPairUaXDimension(dimensionPairs, dimensionPairs[0])[0])} vs ${sdl(dimensionPairs[0].uaYDimension[0])}`;
                        break;
                    case CHART_TYPES.GAUGE:
                    case CHART_TYPES.KPI:
                        addGM0L();
                        break;
                    case CHART_TYPES.WEBAPP:
                        newName = chartDef.$loadedDesc.desc.meta.label || CHART_TYPES.WEBAPP;
                        break;
                    case CHART_TYPES.BOXPLOTS:
                        newName += ChartLabels.getDimensionLabel(chartDef.boxplotValue[0], false);
                        // eslint-disable-next-line no-undef
                        breakdownDimension = ChartDimension.getBreakdownDimension(chartDef);
                        // eslint-disable-next-line no-undef
                        if (breakdownDimension && !has(chartDef.genericDimension1)) {
                            newName += ' by ';
                            newName += ChartLabels.getDimensionLabel(breakdownDimension, false);
                            // eslint-disable-next-line no-undef
                        } else if (breakdownDimension && has(chartDef.genericDimension1)) {
                            addGM0L();
                            newName += ' by ';
                            newName += ChartLabels.getDimensionLabel(breakdownDimension, false);
                            newName += ' and ';
                            newName += ChartLabels.getDimensionLabel(chartDef.genericDimension1[0], false);
                        }
                        break;
                }
                return newName;
            },

            /* ********************** Indicates chart validity ********************* */

            getValidity: function(chart) {
                const chartDef = chart.def;

                function ok() {
                    return {
                        valid: true
                    };
                }

                function incomplete(message) {
                    return {
                        valid: false,
                        type: 'INCOMPLETE',
                        message: message
                    };
                }

                function invalid(message) {
                    return {
                        valid: false,
                        type: 'INVALID',
                        message: message
                    };
                }

                function isGeometryMapComplete(geoLayers) {
                    return geoLayers.some(geoLayer => geoLayer.geometry && geoLayer.geometry.length > 0);
                }

                switch (chartDef.type) {
                    case CHART_TYPES.MULTI_COLUMNS_LINES:
                    case CHART_TYPES.GROUPED_COLUMNS:
                    case CHART_TYPES.STACKED_COLUMNS:
                    case CHART_TYPES.STACKED_BARS:
                    case CHART_TYPES.LINES:
                    case CHART_TYPES.STACKED_AREA:
                    case CHART_TYPES.PIE:
                        /* Minimal validity condition: first dimension, 1 measure */
                        // eslint-disable-next-line no-undef
                        if (!has(chartDef.genericDimension0) && !has(chartDef.genericHierarchyDimension)) {
                            return incomplete(ChartLabels.INCOMPLETE_CHART_LABELS.GIVE_GROUP);
                        }
                        // eslint-disable-next-line no-undef
                        if (!has(chartDef.genericMeasures)) {
                            return incomplete(ChartLabels.INCOMPLETE_CHART_LABELS.WHAT_TO_SHOW);
                        }
                        /* Check for invalidities */
                        if (chartDef.type === CHART_TYPES.STACKED_COLUMNS || chartDef.type === CHART_TYPES.STACKED_BARS) {
                            if (chartDef.variant === 'stacked_100') {
                                // Stack 100% needs two dimensions to be meaningful
                                // eslint-disable-next-line no-undef
                                if (!has(chartDef.genericDimension1) && chartDef.genericMeasures.length === 1) {
                                    return invalid(ChartLabels.CHART_ERROR_MESSAGES.STACKED_100_COLS);
                                }
                            }
                        }
                        return ok();

                    case CHART_TYPES.PIVOT_TABLE:
                        // eslint-disable-next-line no-undef
                        if (!has(chartDef.genericMeasures)) {
                            return incomplete(ChartLabels.INCOMPLETE_CHART_LABELS.WHAT_TO_SHOW);
                        }
                        return ok();

                    case CHART_TYPES.BINNED_XY:
                        // eslint-disable-next-line no-undef
                        if (!has(chartDef.xDimension) && !has(chartDef.xHierarchyDimension)) {
                            return incomplete(ChartLabels.INCOMPLETE_CHART_LABELS.GIVE_X_AXIS);
                        }
                        // eslint-disable-next-line no-undef
                        if (!has(chartDef.yDimension) && !has(chartDef.yHierarchyDimension)) {
                            return incomplete(ChartLabels.INCOMPLETE_CHART_LABELS.GIVE_Y_AXIS);
                        }

                        if (chartDef.variant === CHART_VARIANTS.binnedXYHexagon) {
                            if (!ChartDimension.isTrueNumerical(chartDef.xDimension[0]) ||
                                !ChartDimension.isTrueNumerical(chartDef.yDimension[0])) {
                                return invalid(ChartLabels.CHART_ERROR_MESSAGES.HEX_BINNING_DIM);
                            }
                            if (!ChartDimension.isTrueNumerical(chartDef.xDimension[0]) ||
                                !ChartDimension.isTrueNumerical(chartDef.yDimension[0])) {
                                return invalid(ChartLabels.CHART_ERROR_MESSAGES.HEX_BINNING_DIM);
                            }
                        }
                        return ok();

                    case CHART_TYPES.GROUPED_XY:
                    case CHART_TYPES.LIFT:
                        // eslint-disable-next-line no-undef
                        if (!has(chartDef.groupDimension) && !has(chartDef.groupHierarchyDimension)) {
                            return incomplete(ChartLabels.INCOMPLETE_CHART_LABELS.GIVE_GROUP);
                        }
                        // eslint-disable-next-line no-undef
                        if (!has(chartDef.xMeasure)) {
                            return incomplete(ChartLabels.INCOMPLETE_CHART_LABELS.LIFT_GIVE_X_AXIS);
                        }
                        // eslint-disable-next-line no-undef
                        if (!has(chartDef.yMeasure)) {
                            return incomplete(ChartLabels.INCOMPLETE_CHART_LABELS.LIFT_GIVE_Y_AXIS);
                        }
                        return ok();


                    case CHART_TYPES.DENSITY_2D:
                        // eslint-disable-next-line no-undef
                        if (!has(chartDef.xDimension)) {
                            return incomplete(ChartLabels.INCOMPLETE_CHART_LABELS.GIVE_X_AXIS);
                        }
                        // eslint-disable-next-line no-undef
                        if (!has(chartDef.yDimension)) {
                            return incomplete(ChartLabels.INCOMPLETE_CHART_LABELS.GIVE_Y_AXIS);
                        }
                        return ok();

                    case CHART_TYPES.SCATTER:
                        // eslint-disable-next-line no-undef
                        if (!has(chartDef.uaXDimension)) {
                            return incomplete(ChartLabels.INCOMPLETE_CHART_LABELS.GIVE_X_AXIS);
                        }
                        // eslint-disable-next-line no-undef
                        if (!has(chartDef.uaYDimension)) {
                            return incomplete(ChartLabels.INCOMPLETE_CHART_LABELS.GIVE_Y_AXIS);
                        }
                        return ok();

                    case CHART_TYPES.SCATTER_MULTIPLE_PAIRS:
                        if (!ChartUADimension.getDisplayableDimensionPairs(chartDef.uaDimensionPair).length) {
                            // eslint-disable-next-line no-undef
                            if (!chartDef.uaDimensionPair || !chartDef.uaDimensionPair.length || !has(chartDef.uaDimensionPair[0].uaXDimension)) {
                                return incomplete(ChartLabels.INCOMPLETE_CHART_LABELS.GIVE_X_AXIS);
                            }
                            // eslint-disable-next-line no-undef
                            if (!has(chartDef.uaDimensionPair[0].uaYDimension)) {
                                return incomplete(ChartLabels.INCOMPLETE_CHART_LABELS.GIVE_Y_AXIS);
                            }
                        }
                        return ok();

                    case CHART_TYPES.BOXPLOTS:
                        // eslint-disable-next-line no-undef
                        if (!has(chartDef.boxplotValue)) {
                            return incomplete(ChartLabels.INCOMPLETE_CHART_LABELS.WHAT_TO_SHOW);
                        }
                        return ok();

                    case CHART_TYPES.GAUGE:
                        // eslint-disable-next-line no-undef
                        if (!has(chartDef.genericMeasures)) {
                            return incomplete(ChartLabels.INCOMPLETE_CHART_LABELS.WHAT_TO_SHOW);
                        }
                        // eslint-disable-next-line no-undef
                        if (chartDef.genericMeasures.length > 1) {
                            return incomplete(ChartLabels.INCOMPLETE_CHART_LABELS.GAUGE_ONLY_ONE);
                        }
                        return ok();
                    case CHART_TYPES.KPI:
                        // eslint-disable-next-line no-undef
                        if (!has(chartDef.genericMeasures)) {
                            return incomplete(ChartLabels.INCOMPLETE_CHART_LABELS.WHAT_TO_SHOW);
                        }
                        return ok();

                    case CHART_TYPES.RADAR:
                        /* Minimal validity condition: first dimension, 1 measure */
                        // eslint-disable-next-line no-undef
                        if (!has(chartDef.genericDimension0) && !has(chartDef.genericHierarchyDimension)) {
                            return incomplete(ChartLabels.INCOMPLETE_CHART_LABELS.GIVE_GROUP);
                        }
                        // eslint-disable-next-line no-undef
                        if (!has(chartDef.genericMeasures)) {
                            return incomplete(ChartLabels.INCOMPLETE_CHART_LABELS.WHAT_TO_SHOW);
                        }
                        return ok();

                    case CHART_TYPES.SANKEY:
                        /* Minimal validity condition: 1 dimension, 2 measures */
                        // eslint-disable-next-line no-undef
                        if (!has(chartDef.yDimension) || chartDef.yDimension.length < 2) {
                            return incomplete(ChartLabels.INCOMPLETE_CHART_LABELS.TWO_LEVELS);
                        }
                        // eslint-disable-next-line no-undef
                        if (!has(chartDef.genericMeasures)) {
                            return incomplete(ChartLabels.INCOMPLETE_CHART_LABELS.SANKEY_GIVE_COL);
                        }
                        return ok();

                    case CHART_TYPES.SCATTER_MAP:
                    case CHART_TYPES.DENSITY_HEAT_MAP:
                    case CHART_TYPES.ADMINISTRATIVE_MAP:
                    case CHART_TYPES.GRID_MAP:
                        // eslint-disable-next-line no-undef
                        if (!has(chartDef.geometry)) {
                            return incomplete(ChartLabels.INCOMPLETE_CHART_LABELS.GIVE_GEO_COL);
                        }
                        return ok();
                    case CHART_TYPES.GEOMETRY_MAP:
                        if (!isGeometryMapComplete(chartDef.geoLayers)) {
                            return incomplete(ChartLabels.INCOMPLETE_CHART_LABELS.GIVE_GEO_COL);
                        }
                        return ok();
                    case CHART_TYPES.TREEMAP:
                        // eslint-disable-next-line no-undef
                        if (!has(chartDef.yDimension) && !has(chartDef.yHierarchyDimension)) {
                            return incomplete(ChartLabels.INCOMPLETE_CHART_LABELS.GIVE_GROUP);
                        }
                        // eslint-disable-next-line no-undef
                        if (!has(chartDef.genericMeasures)) {
                            return incomplete(ChartLabels.INCOMPLETE_CHART_LABELS.TREEMAP_GIVE_NUM_COL);
                        }
                        return ok();
                    case CHART_TYPES.WEBAPP:
                        // all custom
                        return ok();

                    default:
                        throw Error('unimplemented handling of ' + chartDef.type);
                }
            },

            /************** Indicates charts warnings **************/

            /**
             * Indicates if pivot table threshold has been reached
             */
            hasPivotTableReachThreshold: function(chartDef, data) {
                return chartDef.type === CHART_TYPES.PIVOT_TABLE && data && data.errorType === 'com.dataiku.dip.pivot.backend.model.SecurityAbortedException';
            },

            /**
             * Indicates if pivot table threshold has been reached
             */
            hasCorruptedLinoCacheWarning: function(data) {
                return data && data.errorType === 'com.dataiku.dip.pivot.backend.model.CorruptedDataException';
            },

            /**
             * Indicates if pivot table threshold has been reached
             */
            hasEmptyCustomBins: function(data) {
                return data && data.errorType === 'com.dataiku.dip.pivot.backend.model.EmptyCustomBinsException';
            },

            /**
             * Returns true if there is a waning to display
             */
            hasRequestResponseWarning: function(chartDef, data) {
                return this.hasPivotTableReachThreshold(chartDef, data) || this.hasCorruptedLinoCacheWarning(data) || this.hasEmptyCustomBins(data);
            },

            /**
             * Returns validity warning
             */
            getRequestResponseWarning: function(chartDef, data) {
                if (this.hasPivotTableReachThreshold(chartDef, data)) {
                    return {
                        valid: false,
                        type: 'PIVOT_TABLE_TOO_MUCH_DATA',
                        message: data.message
                    };
                } else if (this.hasCorruptedLinoCacheWarning(data)) {
                    return {
                        valid: false,
                        type: 'CORRUPTED_LINO_CACHE',
                        message: data.message
                    };
                } else if (this.hasEmptyCustomBins(data)) {
                    return {
                        valid: false,
                        type: 'EMPTY_CUSTOM_BINS',
                        message: data.message
                    };
                }
            },

            /* ********************** Fixup, autocomplete, handle sort ********************* */

            /* Fixup everything that needs to */
            fixupSpec: function(chart, theme, changedDefinition) {
                const chartDef = chart.def;
                const usableColumns = chart.summary && chart.summary.usableColumns;
                Logger.info('Fixing up the spec: ' + JSON.stringify(chartDef));

                if (chartDef.type === CHART_TYPES.WEBAPP) {
                    chartDef.webAppConfig = chartDef.webAppConfig || {};
                    chartDef.$loadedDesc = WebAppsService.getWebAppLoadedDesc(chartDef.webAppType) || {};
                    chartDef.$pluginDesc = WebAppsService.getOwnerPluginDesc(chartDef.webAppType);
                    chartDef.$pluginChartDesc = chartDef.$loadedDesc.desc.chart;
                    PluginConfigUtils.setDefaultValues(chartDef.$pluginChartDesc.leftBarParams, chartDef.webAppConfig);
                    PluginConfigUtils.setDefaultValues(chartDef.$pluginChartDesc.topBarParams, chartDef.webAppConfig);
                } else if (chartDef.variant === CHART_VARIANTS.waterfall) {
                    chartDef.colorOptions.colorPalette = 'dku_waterfall_palette';
                    chartDef.colorOptions.customColors = { ...ChartColorUtils.getWaterfallDefaultColors(), ...chartDef.colorOptions.customColors };
                } else {
                    chartDef.$loadedDesc = null;
                    chartDef.$pluginDesc = null;
                    chartDef.$pluginChartDesc = null;
                    if (chartDef.colorOptions && chartDef.colorOptions.colorPalette === 'dku_waterfall_palette') {
                        chartDef.colorOptions.colorPalette = theme.palettes.discrete ?? ChartsStaticData.DEFAULT_DISCRETE_PALETTE;
                    }
                }

                if (ChartFeatures.canDisplayDimensionValuesInChart(chartDef.type)) {
                    const hierarchyDimensions = (chartDef.yHierarchyDimension[0] || {}).dimensions || [];

                    [...chartDef.yDimension, ...hierarchyDimensions].forEach((dim, i) => {
                        if (dim.showDimensionValuesInChart !== false) {
                            dim.showDimensionValuesInChart = true;
                        }
                        /*
                         * Last dimension's values need to be displayed and cannot be hidden
                         * There can only be one hierarchy dimension so the values need to be displayed as well
                         */
                        if (chartDef.yDimension.length - 1 === i || dim.hierarchyId) {
                            dim.$canChangeShowValuesInChart = false;
                            dim.showDimensionValuesInChart = true;
                        } else {
                            dim.$canChangeShowValuesInChart = true;
                        }
                        if (!dim.textFormatting) {
                            dim.textFormatting = { ...DEFAULT_FONT_FORMATTING };
                        }
                    });
                }


                // resetting values display mode to default if selected mode is unsuported
                const valuesDisplayMode = chartDef.valuesInChartDisplayOptions.displayMode;
                if (chartDef.type === CHART_TYPES.STACKED_COLUMNS && (valuesDisplayMode === VALUES_DISPLAY_MODES.LABELS || valuesDisplayMode === VALUES_DISPLAY_MODES.VALUES_AND_LABELS)
                    || chartDef.type === CHART_TYPES.PIE && valuesDisplayMode === VALUES_DISPLAY_MODES.VALUES_AND_TOTALS) {
                    chartDef.valuesInChartDisplayOptions.displayMode = VALUES_DISPLAY_MODES.VALUES;
                }

                /* If log scale is enabled on X axis but not possible, disable it */
                if (chartDef.xAxisFormatting.isLogScale && !ChartFeatures.canSetLogScale(chartDef, 'x')) {
                    chartDef.xAxisFormatting.isLogScale = false;
                }

                chartDef.yAxesFormatting.forEach(axisFormatting => {
                    /* If log scale is enabled on Y axis but not possible, disable it */
                    if (axisFormatting.isLogScale && !ChartFeatures.canSetLogScale(chartDef, 'y')) {
                        axisFormatting.isLogScale = false;
                    }
                });

                if (chartDef.gridlinesOptions.$default) {
                    const isOnlyRightAxis = chartDef.genericMeasures && chartDef.genericMeasures.length === 1 && chartDef.genericMeasures[0].displayAxis === 'axis2';
                    chartDef.gridlinesOptions.vertical.show = chartDef.type === CHART_TYPES.STACKED_BARS;
                    chartDef.gridlinesOptions.horizontal.show = ![CHART_TYPES.SCATTER, CHART_TYPES.SCATTER_MULTIPLE_PAIRS, CHART_TYPES.BINNED_XY, CHART_TYPES.STACKED_BARS].includes(chartDef.type);
                    chartDef.gridlinesOptions.horizontal.displayAxis.type = isOnlyRightAxis ? GridlinesAxisType.RIGHT_Y_AXIS : GridlinesAxisType.LEFT_Y_AXIS;
                }

                // chart cannot have given dimension & corresponding hierarchy dimension at the same time
                clearGenericOrHierarchyDimension(chartDef, changedDefinition, 'genericDimension0', 'genericHierarchyDimension');
                clearGenericOrHierarchyDimension(chartDef, changedDefinition, 'xDimension', 'xHierarchyDimension');
                clearGenericOrHierarchyDimension(chartDef, changedDefinition, 'yDimension', 'yHierarchyDimension');
                clearGenericOrHierarchyDimension(chartDef, changedDefinition, 'groupDimension', 'groupHierarchyDimension');
                clearGenericOrHierarchyDimension(chartDef, changedDefinition, 'boxplotBreakdownDim', 'boxplotBreakdownHierarchyDimension');

                chartDef.genericDimension0.forEach(dimension => autocompleteGenericDimension(chartDef, dimension));
                chartDef.genericDimension1.forEach(dimension => autocompleteGenericDimension(chartDef, dimension, 'genericDimension1'));
                chartDef.facetDimension.forEach(dimension => autocompleteGenericDimension(chartDef, dimension));
                chartDef.animationDimension.forEach(dimension => autocompleteGenericDimension(chartDef, dimension));
                chartDef.boxplotBreakdownDim.forEach(dimension => autocompleteGenericDimension(chartDef, dimension));
                chartDef.xDimension.forEach(dimension => autocompleteGenericDimension(chartDef, dimension));
                chartDef.yDimension.forEach(dimension => autocompleteGenericDimension(chartDef, dimension));
                chartDef.groupDimension.forEach(dimension => autocompleteGenericDimension(chartDef, dimension));
                chartDef.genericHierarchyDimension.forEach(hierarchy => autocompleteHierarchyDimensions(chartDef, hierarchy, 'generic'));
                chartDef.xHierarchyDimension.forEach(hierarchy => autocompleteHierarchyDimensions(chartDef, hierarchy, 'x'));
                chartDef.yHierarchyDimension.forEach(hierarchy => autocompleteHierarchyDimensions(chartDef, hierarchy, 'y'));
                chartDef.groupHierarchyDimension.forEach(hierarchy => autocompleteHierarchyDimensions(chartDef, hierarchy, 'group'));
                chartDef.boxplotBreakdownHierarchyDimension.forEach(hierarchy => autocompleteHierarchyDimensions(chartDef, hierarchy, 'boxplotBreakdown'));

                // Reset custom range settings when there are no dimensions
                // eslint-disable-next-line no-undef
                if (areAllEmpty(ChartTypeChangeUtils.takeAllInX(chartDef))) {
                    ChartAxesUtils.resetCustomExtents([chartDef.xAxisFormatting]);
                }
                // eslint-disable-next-line no-undef
                if (areAllEmpty(ChartTypeChangeUtils.takeAllInY(chartDef))) {
                    ChartAxesUtils.resetCustomExtents(chartDef.yAxesFormatting);
                }

                if (!chartDef.filters) {
                    chartDef.filters = [];
                }

                chartDef.filters.forEach(filter => ChartFilters.autocompleteFilter(filter, usableColumns, false));
                // if a hierarchy was removed, remove all the related filters
                purgeHierarchyFilters(chartDef, changedDefinition);

                chartDef.genericMeasures.forEach(autocompleteGenericMeasure.bind(this, { contextualMenuMeasureType: 'generic', chartDef, theme }));
                chartDef.genericMeasures.forEach(measure => {
                    if (measure.valuesInChartDisplayOptions && measure.valuesInChartDisplayOptions.additionalMeasures) {
                        measure.valuesInChartDisplayOptions.additionalMeasures.forEach(autocompleteGenericMeasure.bind(this, { contextualMenuMeasureType: 'valuesInChart', chartDef, defaultTextFormatting: measure.valuesInChartDisplayOptions.textFormatting, theme }));
                    }
                });
                chartDef.sizeMeasure.forEach(autocompleteGenericMeasure.bind(this, { contextualMenuMeasureType: 'size', chartDef, theme }));
                chartDef.colorMeasure.forEach(autocompleteGenericMeasure.bind(this, { contextualMenuMeasureType: 'color', chartDef, theme }));
                if (Array.isArray(chartDef.colorGroups)) {
                    chartDef.colorGroups.forEach(colorGroup => {
                        if (_.isNil(colorGroup.colorMeasure)) {
                            colorGroup.colorMeasure = [];
                        } else {
                            colorGroup.colorMeasure.forEach(autocompleteGenericMeasure.bind(this, { contextualMenuMeasureType: 'colorGroup', chartDef, theme }));
                        }

                        if (chartDef.type === CHART_TYPES.KPI) {
                            colorGroup.colorGroupMode = 'RULES';
                        }
                    });
                } else {
                    chartDef.colorGroups = [];
                }
                chartDef.tooltipMeasures.forEach(autocompleteGenericMeasure.bind(this, { contextualMenuMeasureType: 'tooltip', chartDef, theme }));
                chartDef.xMeasure.forEach(autocompleteGenericMeasure.bind(this, { contextualMenuMeasureType: 'x', chartDef, theme }));
                chartDef.yMeasure.forEach(autocompleteGenericMeasure.bind(this, { contextualMenuMeasureType: 'y', chartDef, theme }));
                chartDef.uaXDimension.forEach(autocompleteUA);
                chartDef.uaYDimension.forEach(autocompleteUA);
                chartDef.uaDimensionPair.forEach(pair => {
                    pair.uaXDimension.forEach(autocompleteUA);
                    pair.uaYDimension.forEach(autocompleteUA);
                });
                chartDef.uaYDimension.forEach(autocompleteUA);
                chartDef.uaSize.forEach(autocompleteUA);
                chartDef.uaColor.forEach(autocompleteUA);
                chartDef.uaTooltip.forEach(autocompleteUA);
                chartDef.geometry.forEach(autocompleteUA);
                chartDef.boxplotValue.forEach(autocompleteUA);

                /* Handle dimension sorting options */
                const possibleStdDimensionSorts = ChartDefinitionChangeHandler.getStdDimensionPossibleSorts(chartDef);
                chartDef.genericDimension0.forEach(dim => ChartDefinitionChangeHandler.handleDimensionSort(dim, possibleStdDimensionSorts, chartDef));
                chartDef.facetDimension.forEach(dim => ChartDefinitionChangeHandler.handleDimensionSort(dim, possibleStdDimensionSorts, chartDef));
                chartDef.animationDimension.forEach(dim => ChartDefinitionChangeHandler.handleDimensionSort(dim, possibleStdDimensionSorts, chartDef));
                chartDef.xDimension.forEach(dim => ChartDefinitionChangeHandler.handleDimensionSort(dim, possibleStdDimensionSorts, chartDef));
                chartDef.yDimension.forEach(dim => ChartDefinitionChangeHandler.handleDimensionSort(dim, possibleStdDimensionSorts, chartDef));
                chartDef.groupDimension.forEach(dim => ChartDefinitionChangeHandler.handleDimensionSort(dim, possibleStdDimensionSorts, chartDef));
                chartDef.genericHierarchyDimension.forEach(hierarchy => hierarchy.dimensions.forEach(dim => ChartDefinitionChangeHandler.handleDimensionSort(dim, possibleStdDimensionSorts, chartDef)));
                chartDef.xHierarchyDimension.forEach(hierarchy => hierarchy.dimensions.forEach(dim => ChartDefinitionChangeHandler.handleDimensionSort(dim, possibleStdDimensionSorts, chartDef)));
                chartDef.yHierarchyDimension.forEach(hierarchy => hierarchy.dimensions.forEach(dim => ChartDefinitionChangeHandler.handleDimensionSort(dim, possibleStdDimensionSorts, chartDef)));
                chartDef.groupHierarchyDimension.forEach(hierarchy => hierarchy.dimensions.forEach(dim => ChartDefinitionChangeHandler.handleDimensionSort(dim, possibleStdDimensionSorts, chartDef)));

                if (chartDef.type === CHART_TYPES.BOXPLOTS) {
                    const boxplotPossibleSorts = [
                        { type: 'NATURAL', sortId: 'NATURAL', label: translate('CHARTS.SORTING_METHOD.NATURAL_ORDERING', 'Natural ordering'), sortAscending: true },
                        { type: 'COUNT', sortId: 'count desc', label: ChartLabels.DESCENDING_COUNT_OF_RECORDS_LABEL, sortAscending: false },
                        { type: 'COUNT', sortId: 'count asc', label: ChartLabels.ASCENDING_COUNT_OF_RECORDS_LABEL, sortAscending: true }
                    ];

                    if ($rootScope.featureFlagEnabled('unaggregatedMeasures')) {
                        boxplotPossibleSorts.unshift({ type: 'ORIGINAL', sortId: 'ORIGINAL', label: translate('CHARTS.SORTING_METHOD.NO_ORDERING', 'No sorting'), sortAscending: true });
                    }

                    chartDef.possibleDimensionSorts = boxplotPossibleSorts;
                    chartDef.boxplotBreakdownDim.forEach(dim => ChartDefinitionChangeHandler.handleDimensionSort(dim, boxplotPossibleSorts, chartDef, true));
                    chartDef.boxplotBreakdownHierarchyDimension.forEach(hierarchy => hierarchy.dimensions.forEach(dim => ChartDefinitionChangeHandler.handleDimensionSort(dim, boxplotPossibleSorts, chartDef, true)));
                    chartDef.genericDimension1.forEach(dim => ChartDefinitionChangeHandler.handleDimensionSort(dim, boxplotPossibleSorts, chartDef, true));
                } else {
                    chartDef.possibleDimensionSorts = possibleStdDimensionSorts;
                    chartDef.genericDimension1.forEach(dim => ChartDefinitionChangeHandler.handleDimensionSort(dim, possibleStdDimensionSorts, chartDef));
                }

                let colorColumnData;
                const currColorColumn = ChartColorSelection.getColorDimensionOrMeasure(chartDef, chartDef.geoLayers[0]);
                if (usableColumns && currColorColumn) {
                    colorColumnData = usableColumns.find(c => c.column === currColorColumn.column);
                }
                // rollback to default palette if there is no color column or no meaning and the chart is using a meaning palette
                if ((!colorColumnData || !colorColumnData.meaningInfo) && (chartDef.colorOptions.colorPalette === DKU_PALETTE_NAMES.MEANING)) {
                    chartDef.colorOptions.colorPalette = 'default';
                }

                let resetAllColumns = false;
                let indexChanged = -1;
                if (changedDefinition && chartDef.type == CHART_TYPES.MULTI_COLUMNS_LINES) {
                    if (changedDefinition.name === 'genericMeasures' && chartDef.genericDimension1.length) {
                        indexChanged = changedDefinition.nv.findIndex((v, index) => !_.isEqual(v, changedDefinition.ov[index]));
                    }
                    if (changedDefinition.name === 'genericDimension1' && changedDefinition.nv.length) {
                        resetAllColumns = true;
                    }
                }

                let columnFound = false;
                chartDef.genericMeasures.forEach((gm, index, array) => {
                    if (gm.displayed && gm.displayAxis == null) {
                        gm.displayAxis = 'axis1';
                    }
                    if (gm.displayed && gm.displayType == null) {
                        /*
                         * TODO: Reuse the logic that makes the default
                         * display type according to the dimension type
                         */
                        gm.displayType = chartDef.type == CHART_TYPES.MULTI_COLUMNS_LINES && array.length > 1 ? 'line' : 'column';
                    }

                    if (!gm.labelPosition) {
                        gm.labelPosition = ChartsStaticData.LABEL_POSITIONS.BOTTOM.value;
                    }

                    if (gm.showDisplayLabel === undefined) {
                        gm.showDisplayLabel = true;
                    }

                    if (gm.showValue === undefined) {
                        gm.showValue = true;
                    }

                    if (!gm.colorRules) {
                        gm.colorRules = [];
                    }
                    if (!gm.kpiTextAlign) {
                        gm.kpiTextAlign = 'CENTER';
                    }
                    if (!gm.labelTextFormatting) {
                        gm.labelTextFormatting = { fontColor: '#333333', fontSize: 15 };
                    }
                    if (gm.labelFontSize) {
                        //ensuring retrocompability - labelFontSize is legacy
                        gm.labelTextFormatting.fontSize = gm.labelFontSize;
                        delete gm.labelFontSize;
                    }
                    if (!gm.kpiValueFontSizeMode) {
                        gm.kpiValueFontSizeMode = ChartsStaticData.availableFontSizeModes.RESPONSIVE.value;
                    }
                    if (!gm.kpiValueFontSize) {
                        gm.kpiValueFontSize = 32;
                    }
                    if (!gm.responsiveTextAreaFill) {
                        gm.responsiveTextAreaFill = 100;
                    }
                    if (resetAllColumns && columnFound) {
                        gm.displayType = 'line';
                    }
                    if (gm.displayType === 'column') {
                        columnFound = true;
                    }
                    if (indexChanged == index && gm.displayType === 'column') {
                        for (let oi = 0; oi < array.length; oi++) {
                            if (oi === indexChanged) {
                                continue;
                            }
                            if (array[oi].displayType === 'column') {
                                array[oi].displayType = 'line';
                            }
                        }
                    }
                });
                /*
                 * For std-aggregated charts that only support 1 measure when there
                 * are two dimensions, move additional measures to tooltip
                 */
                // eslint-disable-next-line no-undef
                if ((has(chartDef.genericDimension0) || has(chartDef.genericHierarchyDimension)) && has(chartDef.genericDimension1) && chartDef.type !== CHART_TYPES.MULTI_COLUMNS_LINES) {
                    chartDef.genericMeasures.splice(1, chartDef.genericMeasures.length - 1).forEach(function(x) {
                        chartDef.tooltipMeasures.push(x);
                    });
                }

                /* Force column display of all measures (not line) if chart type is grouped_columns */
                if (chartDef.type === CHART_TYPES.GROUPED_COLUMNS) {
                    chartDef.genericMeasures.forEach(function(m) {
                        m.displayType = 'column';
                    });
                }

                /* Force Auto mode for all axes if scatter equalScales mode is enabled */
                if (chartDef.type === CHART_TYPES.SCATTER && chartDef.scatterOptions && chartDef.scatterOptions.equalScales) {
                    chartDef.xAxisFormatting.customExtent = { ...ChartsStaticData.DEFAULT_CUSTOM_EXTENT };
                    ChartAxesUtils.setYAxisCustomExtent(chartDef.yAxesFormatting, { ...ChartsStaticData.DEFAULT_CUSTOM_EXTENT });
                    chartDef.xAxisFormatting.customExtent.editMode = ChartsStaticData.AUTO_EXTENT_MODE;
                }

                /* Map fixup */
                if (!chartDef.mapOptions) {
                    chartDef.mapOptions = {
                        tilesLayer: 'cartodb-positron'
                    };
                }
                if (!chartDef.mapGridOptions) {
                    chartDef.mapGridOptions = {};
                }
                if (_.isNil(chartDef.mapGridOptions.gridLonDeg)) {
                    chartDef.mapGridOptions.gridLonDeg = 0.5;
                }
                if (_.isNil(chartDef.mapGridOptions.gridLatDeg)) {
                    chartDef.mapGridOptions.gridLatDeg = 0.5;
                }

                // Gauge fixup (reset default min and max)
                if (
                    changedDefinition &&
                    (
                        changedDefinition.name === 'genericMeasures' &&
                            changedDefinition.nv[0] &&
                            changedDefinition.ov[0] &&
                            (
                                changedDefinition.nv[0].column !== changedDefinition.ov[0].column ||
                                changedDefinition.nv[0].function !== changedDefinition.ov[0].function
                            )
                        || changedDefinition.name === 'type'
                    )
                ) {
                    chartDef.gaugeOptions = {
                        ...chartDef.gaugeOptions,
                        min: { ...DEFAULT_GAUGE_MIN_MAX },
                        max: { ...DEFAULT_GAUGE_MIN_MAX }
                    };
                }

                // Treemap fixup
                if (chartDef.type === CHART_TYPES.TREEMAP
                    && chartDef.valuesInChartDisplayOptions
                    && !_.isNil(chartDef.valuesInChartDisplayOptions.displayValues)
                    && !!chartDef.genericMeasures
                    && chartDef.genericMeasures.length > 0
                    && chartDef.genericMeasures[0].valuesInChartDisplayOptions
                    && !_.isNil(chartDef.genericMeasures[0].valuesInChartDisplayOptions.displayValues)
                ) {
                    // for treemap, ensure that the global "values in chart" option is always enabled, and if it's explicitely set to false, set it to false on the measure
                    if (chartDef.valuesInChartDisplayOptions.displayValues === false) {
                        chartDef.genericMeasures[0].valuesInChartDisplayOptions.displayValues = false;
                    }
                    chartDef.valuesInChartDisplayOptions.displayValues = true;
                }

                // Pivot table fixup (disable hiding row headers when more than one row dimension)
                if (chartDef.type === CHART_TYPES.PIVOT_TABLE && changedDefinition &&
                    changedDefinition.name === 'yDimension' &&
                    chartDef.yDimension.length > 1
                ) {
                    chartDef.pivotTableOptions.tableFormatting.showRowHeaders = true;
                }

                //purging multiple y axes formatting from scattersMP
                if (chartDef.type !== CHART_TYPES.SCATTER_MULTIPLE_PAIRS) {
                    chartDef.yAxesFormatting = chartDef.yAxesFormatting.filter(v => [ChartsStaticData.LEFT_AXIS_ID, ChartsStaticData.RIGHT_AXIS_ID].includes(v.id));
                }

                // fixup additional measures for stacked columns
                fixupStackedColumnsOptions(chartDef, theme);

                Logger.info('ChartSpec fixup done', chartDef);
            },

            fixupChart: function(chartDef, theme) {
                //  Echarts beta test
                chartDef.hasEchart = ChartFeatures.hasEChartsDefinition(chartDef.type); //  Transient, we don't need to register it in ChartDef, but it is used for dashboard/insights/editor
                chartDef.hasD3 = ChartFeatures.hasD3Definition(chartDef.type); // Transient, we don't need to register it in ChartDef, but it is used for dashboard/insights/editor

                if (ChartFeatures.hasEChartsDefinition(chartDef.type) && chartDef.displayWithECharts === undefined && $rootScope.featureFlagEnabled('echarts')) {
                    chartDef.displayWithECharts = $rootScope.featureFlagEnabled('echarts');
                }

                if (chartDef.displayWithEChartsByDefault === null || chartDef.displayWithEChartsByDefault === undefined) {
                    chartDef.displayWithEChartsByDefault = true;
                }

                // "Auto-migration"

                if (!chartDef.id) {
                    chartDef.id = window.crypto.getRandomValues(new Uint32Array(1))[0].toString(16);
                }

                if (!chartDef.hexbinRadius) {
                    chartDef.hexbinRadius = 20;
                }
                if (!chartDef.hexbinRadiusMode) {
                    chartDef.hexbinRadiusMode = 'NUM_HEXAGONS';
                }
                if (!chartDef.hexbinNumber) {
                    chartDef.hexbinNumber = 20;
                }
                if (!chartDef.yAxisMode) {
                    chartDef.yAxisMode = 'NORMAL';
                }
                if (!chartDef.xAxisMode) {
                    chartDef.xAxisMode = 'NORMAL';
                }
                if (!chartDef.computeMode) {
                    chartDef.computeMode = 'NONE';
                }
                if (chartDef.smoothing === undefined) {
                    chartDef.smoothing = true;
                }
                if (!chartDef.strokeWidth) {
                    chartDef.strokeWidth = ChartsStaticData.DEFAULT_STROKE_WIDTH;
                }
                if (!chartDef.fillOpacity) {
                    chartDef.fillOpacity = ChartsStaticData.DEFAULT_FILL_OPACITY;
                }
                if (!chartDef.chartHeight) {
                    chartDef.chartHeight = 200;
                }
                if (chartDef.showLegend === undefined) {
                    chartDef.showLegend = true;
                }
                if (_.isNil(chartDef.legendFormatting)) {
                    chartDef.legendFormatting = {
                        ...DEFAULT_FONT_FORMATTING
                    };
                    if (theme) {
                        DSSVisualizationThemeUtils.applyToLegend(theme, chartDef.legendFormatting);
                    }
                }
                if (chartDef.colorPaletteType === undefined) {
                    chartDef.colorPaletteType = 'LINEAR';
                }
                if (chartDef.colorMode === undefined) {
                    chartDef.colorMode = 'UNIQUE_SCALE';
                }

                fixupAxes(chartDef, theme);
                fixupReferenceLines(chartDef);
                fixupGaugeTargets(chartDef);

                fixupDynamicMeasure(chartDef.gaugeOptions.range);
                fixupDynamicMeasure(chartDef.gaugeOptions.min);
                fixupDynamicMeasure(chartDef.gaugeOptions.max);
                chartDef.gaugeOptions.targets?.map(dynMeasure => fixupDynamicMeasure(dynMeasure));
                chartDef.referenceLines?.map(dynMeasure => fixupDynamicMeasure(dynMeasure));

                if (chartDef.singleXAxis === undefined) {
                    chartDef.singleXAxis = true;
                }
                if (!chartDef.animationFrameDuration) {
                    chartDef.animationFrameDuration = 3000;
                }
                if (!chartDef.legendPlacement) {
                    chartDef.legendPlacement = 'OUTER_RIGHT';
                }

                if (!chartDef.gridlinesOptions) {
                    chartDef.gridlinesOptions = {
                        $default: true,
                        vertical: {
                            lineFormatting: {
                                type: 'FILLED',
                                size: '1',
                                color: '#d9d9d9'
                            }
                        },
                        horizontal: {
                            displayAxis: {},
                            lineFormatting: {
                                type: 'FILLED',
                                size: '1',
                                color: '#d9d9d9'
                            }
                        }
                    };
                }

                // Since DSS 11, the colored pivot table has been merged with pivot table
                if (chartDef.type === CHART_TYPES.PIVOT_TABLE && chartDef.variant === CHART_VARIANTS.colored) {
                    chartDef.variant = undefined;
                    if (chartDef.colorMeasure.length === 0 && chartDef.genericMeasures.length > 0) {
                        chartDef.colorMeasure.push(chartDef.genericMeasures[0]);
                    }
                }

                if (_.isNil(chartDef.valuesInChartDisplayOptions)) {
                    chartDef.valuesInChartDisplayOptions = {};
                }

                if (_.isNil(chartDef.valuesInChartDisplayOptions.displayValues)) {
                    chartDef.valuesInChartDisplayOptions.displayValues = false;
                }

                if (_.isNil(chartDef.valuesInChartDisplayOptions.displayPieLabelsOrValues)) {
                    chartDef.valuesInChartDisplayOptions.displayPieLabelsOrValues = true;
                }

                if (_.isNil(chartDef.valuesInChartDisplayOptions.displayMode)) {
                    chartDef.valuesInChartDisplayOptions.displayMode = VALUES_DISPLAY_MODES.LABELS;
                }

                if (_.isNil(chartDef.valuesInChartDisplayOptions.overlappingStrategy)) {
                    chartDef.valuesInChartDisplayOptions.overlappingStrategy = ValuesInChartOverlappingStrategy.AUTO;
                }

                if (_.isNil(chartDef.valuesInChartDisplayOptions.textFormatting)) {
                    chartDef.valuesInChartDisplayOptions.textFormatting = DEFAULT_VALUES_DISPLAY_IN_CHART_TEXT_FORMATTING;
                }

                fixupStackedColumnsOptions(chartDef, theme);
                if (!_.isNil(chartDef.valuesInChartDisplayOptions.placementMode) && !_.isNil(chartDef.genericMeasures) && chartDef.genericMeasures.length > 0 && chartDef.valuesInChartDisplayOptions.placementMode !== ValuesInChartPlacementMode.AUTO) {
                    // the default value has been changed, we have to copy it on measures
                    chartDef.genericMeasures.forEach(m => {
                        if (!_.isNil(m.valuesInChartDisplayOptions)) {
                            m.valuesInChartDisplayOptions.placementMode = chartDef.valuesInChartDisplayOptions.placementMode;
                        }
                    });
                    chartDef.valuesInChartDisplayOptions.placementMode = null;
                }

                if (!_.isNil(chartDef.valuesInChartDisplayOptions.spacing) && !_.isNil(chartDef.genericMeasures) && chartDef.genericMeasures.length > 0) {
                    chartDef.genericMeasures.forEach(m => {
                        if (!_.isNil(m.valuesInChartDisplayOptions)) {
                            m.valuesInChartDisplayOptions.spacing = chartDef.valuesInChartDisplayOptions.spacing;
                        }
                    });

                    if (ChartFeatures.canDisplayTotalValues(chartDef)) {
                        chartDef.stackedColumnsOptions.totalsInChartDisplayOptions.spacing = chartDef.valuesInChartDisplayOptions.spacing;
                    }
                    chartDef.valuesInChartDisplayOptions.spacing = null;
                }


                if (_.isNil(chartDef.pivotTableOptions) || _.isEmpty(chartDef.pivotTableOptions)) {
                    chartDef.pivotTableOptions = {
                        measureDisplayMode: ChartsStaticData.pivotTableMeasureDisplayMode.ROWS,
                        displayEmptyValues: false,
                        showSidebar: true,
                        areRowsExpandedByDefault: true,
                        areColumnExpandedByDefault: true,
                        rowIdByCustomExpandedStatus: {},
                        columnIdByCustomExpandedStatus: {},
                        columnIdByCustomWidth: {},
                        displayTotals: angular.copy(ChartsStaticData.defaultPivotDisplayTotals)
                    };
                }

                if (_.isNil(chartDef.pivotTableOptions.measureDisplayMode)) {
                    chartDef.pivotTableOptions.measureDisplayMode = ChartsStaticData.pivotTableMeasureDisplayMode.ROWS;
                }

                if (_.isNil(chartDef.pivotTableOptions.displayTotals)) {
                    chartDef.pivotTableOptions.displayTotals = angular.copy(ChartsStaticData.defaultPivotDisplayTotals);
                }

                if (_.isNil(chartDef.pivotTableOptions.showSidebar)) {
                    chartDef.pivotTableOptions.showSidebar = true;
                }

                if (_.isNil(chartDef.pivotTableOptions.tableFormatting)) {
                    chartDef.pivotTableOptions.tableFormatting = {
                        showRowHeaders: true,
                        showRowMainHeaders: true,
                        rowMainHeaders: { ...DEFAULT_TABLE_HEADER_FONT_FORMATTING },
                        rowSubheaders: { ...DEFAULT_TABLE_HEADER_FONT_FORMATTING },
                        freezeRowHeaders: true,

                        showColumnHeaders: true,
                        showColumnMainHeaders: true,
                        columnMainHeaders: { ...DEFAULT_TABLE_HEADER_FONT_FORMATTING },
                        columnSubheaders: { ...DEFAULT_TABLE_HEADER_FONT_FORMATTING },

                        values: { ...DEFAULT_FONT_FORMATTING },

                        // Obsolete since 13.4 but keeping for retrocompatibility
                        rowHeaders: { ...DEFAULT_TABLE_HEADER_FONT_FORMATTING },
                        columnHeaders: { ...DEFAULT_TABLE_HEADER_FONT_FORMATTING }
                    };
                    if (theme) {
                        DSSVisualizationThemeUtils.applyToTableFormatting(theme, chartDef.pivotTableOptions.tableFormatting);
                    }
                } else {
                    fixupPivotTableFormatting(chartDef.pivotTableOptions.tableFormatting, theme);
                }

                if (_.isNil(chartDef.pivotTableOptions.tableFormatting.showRowHeaders)) {
                    chartDef.pivotTableOptions.tableFormatting.showRowHeaders = true;
                }

                if (_.isNil(chartDef.pivotTableOptions.tableFormatting.showRowMainHeaders)) {
                    chartDef.pivotTableOptions.tableFormatting.showRowMainHeaders = true;
                }

                if (_.isNil(chartDef.pivotTableOptions.tableFormatting.rowMainHeaders)) {
                    chartDef.pivotTableOptions.tableFormatting.rowMainHeaders = { ...DEFAULT_TABLE_HEADER_FONT_FORMATTING, fontColor: theme ? theme.generalFormatting.fontColor : DEFAULT_TABLE_HEADER_FONT_FORMATTING.fontColor };
                }

                if (_.isNil(chartDef.pivotTableOptions.tableFormatting.rowSubheaders)) {
                    chartDef.pivotTableOptions.tableFormatting.rowSubheaders = { ...DEFAULT_TABLE_HEADER_FONT_FORMATTING, fontColor: theme ? theme.generalFormatting.fontColor : DEFAULT_TABLE_HEADER_FONT_FORMATTING.fontColor };
                }

                if (_.isNil(chartDef.pivotTableOptions.tableFormatting.freezeRowHeaders)) {
                    chartDef.pivotTableOptions.tableFormatting.freezeRowHeaders = true;
                }

                if (_.isNil(chartDef.pivotTableOptions.tableFormatting.showColumnHeaders)) {
                    chartDef.pivotTableOptions.tableFormatting.showColumnHeaders = true;
                }

                if (_.isNil(chartDef.pivotTableOptions.tableFormatting.showColumnMainHeaders)) {
                    chartDef.pivotTableOptions.tableFormatting.showColumnMainHeaders = true;
                }

                if (_.isNil(chartDef.pivotTableOptions.tableFormatting.columnMainHeaders)) {
                    chartDef.pivotTableOptions.tableFormatting.columnMainHeaders = { ...DEFAULT_TABLE_HEADER_FONT_FORMATTING, fontColor: theme ? theme.generalFormatting.fontColor : DEFAULT_TABLE_HEADER_FONT_FORMATTING.fontColor };
                }

                if (_.isNil(chartDef.pivotTableOptions.tableFormatting.columnSubheaders)) {
                    chartDef.pivotTableOptions.tableFormatting.columnSubheaders = { ...DEFAULT_TABLE_HEADER_FONT_FORMATTING, fontColor: theme ? theme.generalFormatting.fontColor : DEFAULT_TABLE_HEADER_FONT_FORMATTING.fontColor };
                }

                if (_.isNil(chartDef.pivotTableOptions.tableFormatting.values)) {
                    chartDef.pivotTableOptions.tableFormatting.values = { ...DEFAULT_FONT_FORMATTING, fontColor: theme ? theme.generalFormatting.fontColor : DEFAULT_FONT_FORMATTING.fontColor };
                }

                if (!chartDef.genericDimension0) {
                    chartDef.genericDimension0 = [];
                }
                if (!chartDef.genericDimension1) {
                    chartDef.genericDimension1 = [];
                }
                if (!chartDef.facetDimension) {
                    chartDef.facetDimension = [];
                }
                if (!chartDef.animationDimension) {
                    chartDef.animationDimension = [];
                }
                if (!chartDef.genericMeasures) {
                    chartDef.genericMeasures = [];
                }
                if (!chartDef.xDimension) {
                    chartDef.xDimension = [];
                }
                if (!chartDef.yDimension) {
                    chartDef.yDimension = [];
                }
                if (!chartDef.groupDimension) {
                    chartDef.groupDimension = [];
                }
                if (!chartDef.uaXDimension) {
                    chartDef.uaXDimension = [];
                }
                if (!chartDef.uaYDimension) {
                    chartDef.uaYDimension = [];
                }
                if (!chartDef.uaDimensionPair) {
                    chartDef.uaDimensionPair = [];
                }
                if (!chartDef.genericHierarchyDimension) {
                    chartDef.genericHierarchyDimension = [];
                }
                if (!chartDef.xHierarchyDimension) {
                    chartDef.xHierarchyDimension = [];
                }
                if (!chartDef.yHierarchyDimension) {
                    chartDef.yHierarchyDimension = [];
                }
                if (!chartDef.groupHierarchyDimension) {
                    chartDef.groupHierarchyDimension = [];
                }
                if (!chartDef.boxplotBreakdownHierarchyDimension) {
                    chartDef.boxplotBreakdownHierarchyDimension = [];
                }
                if (!chartDef.xMeasure) {
                    chartDef.xMeasure = [];
                }
                if (!chartDef.yMeasure) {
                    chartDef.yMeasure = [];
                }
                if (!chartDef.sizeMeasure) {
                    chartDef.sizeMeasure = [];
                }
                if (!chartDef.colorMeasure) {
                    chartDef.colorMeasure = [];
                }
                if (!chartDef.tooltipMeasures) {
                    chartDef.tooltipMeasures = [];
                }
                if (!chartDef.uaSize) {
                    chartDef.uaSize = [];
                }
                if (!chartDef.uaColor) {
                    chartDef.uaColor = [];
                }
                if (!chartDef.uaShape) {
                    chartDef.uaShape = [];
                }
                if (!chartDef.uaTooltip) {
                    chartDef.uaTooltip = [];
                }
                if (!chartDef.boxplotBreakdownDim) {
                    chartDef.boxplotBreakdownDim = [];
                }
                if (!chartDef.boxplotValue) {
                    chartDef.boxplotValue = [];
                }
                if (!chartDef.geometry) {
                    chartDef.geometry = [];
                }

                if (!chartDef.colorOptions) {
                    chartDef.colorOptions = ChartsStaticData.DEFAULT_COLOR_OPTIONS;
                }

                if (!chartDef.colorOptions.colorPalette) {
                    chartDef.colorOptions.colorPalette = theme.palettes.discrete ?? ChartsStaticData.DEFAULT_DISCRETE_PALETTE;
                }

                if (!chartDef.colorOptions.customPalette) {
                    chartDef.colorOptions.customPalette = ChartColorUtils.getDefaultCustomPalette();
                }

                if (!chartDef.colorOptions.ccScaleMode) {
                    chartDef.colorOptions.ccScaleMode = 'NORMAL';
                }
                if (!chartDef.colorOptions.paletteType) {
                    chartDef.colorOptions.paletteType = 'CONTINUOUS';
                }
                if (!chartDef.colorOptions.quantizationMode) {
                    chartDef.colorOptions.quantizationMode = 'NONE';
                }
                if (!chartDef.colorOptions.numQuantizeSteps) {
                    chartDef.colorOptions.numQuantizeSteps = 5;
                }
                if (!chartDef.colorOptions.paletteMiddleValue) {
                    chartDef.colorOptions.paletteMiddleValue = 0;
                }
                if (chartDef.colorOptions.paletteMiddleValue <= 0 && chartDef.colorOptions.paletteType === 'DIVERGING' && chartDef.colorOptions.ccScaleMode === 'LOG') {
                    chartDef.colorOptions.paletteMiddleValue = 1;
                }
                if (!chartDef.colorOptions.customColors) {
                    chartDef.colorOptions.customColors = {};
                }

                if (!chartDef.geoLayers) {
                    chartDef.geoLayers = [{ 'geometry': angular.copy(chartDef.geometry), 'uaColor': angular.copy(chartDef.uaColor), colorOptions: angular.copy(chartDef.colorOptions) }];
                }

                if (!chartDef.bubblesOptions) {
                    chartDef.bubblesOptions = {
                        defaultRadius: 5,
                        singleShape: 'FILLED_CIRCLE'
                    };
                }
                if (!chartDef.pieOptions) {
                    chartDef.pieOptions = {
                        donutHoleSize: 54
                    };
                }

                const defaultConnectPointsOpts = {
                    enabled: false,
                    lineFormatting: {
                        size: 1
                    }
                };

                if (!chartDef.scatterOptions) {
                    chartDef.scatterOptions = {
                        regression: {
                            show: false,
                            type: RegressionTypes.LINEAR,
                            lineFormatting: {
                                size: 1,
                                color: '#000'
                            },
                            labelPosition: 'INSIDE_END',
                            displayFormula: false,
                            textFormatting: {
                                ...DEFAULT_FONT_FORMATTING,
                                backgroundColor: '#D9D9D9BF'
                            }
                        }
                    };
                    if (theme) {
                        DSSVisualizationThemeUtils.applyToScatterOptions(theme, chartDef.scatterOptions);
                    }
                }

                if (!chartDef.scatterOptions.connectPoints) {
                    chartDef.scatterOptions.connectPoints = angular.copy(defaultConnectPointsOpts);
                }

                if (!chartDef.sankeyOptions) {
                    chartDef.sankeyOptions = {
                        linkColorVariant: 'GRADIENT',
                        curveness: 0.5
                    };
                }

                if (!chartDef.sankeyOptions.nodeLabelFormatting) {
                    chartDef.sankeyOptions = {
                        ...chartDef.sankeyOptions,
                        nodeLabelFormatting: {
                            ...DEFAULT_AUTO_FONT_FORMATTING,
                            hasBackground: false,
                            backgroundColor: '#D9D9D9BF'
                        }
                    };
                    if (theme) {
                        DSSVisualizationThemeUtils.applyToSankeyNodeLabels(theme, chartDef.sankeyOptions.nodeLabelFormatting);
                    }
                }

                if (!chartDef.scatterMPOptions) {
                    chartDef.scatterMPOptions = {
                        pairColorOptions: {
                            transparency: 0.75,
                            colorPalette: theme && theme.palettes ? theme.palettes.discrete : DKU_PALETTE_NAMES.THEME,
                            customPalette: ChartColorUtils.getDefaultCustomPalette(),
                            customColors: {}
                        }
                    };
                }

                if (!chartDef.scatterMPOptions.connectPoints) {
                    chartDef.scatterMPOptions.connectPoints = angular.copy(defaultConnectPointsOpts);
                }

                if (!chartDef.radarOptions) {
                    chartDef.radarOptions = {
                        filled: false,
                        polygonsSource: PolygonSources.MEASURES,
                        lineStyle: {
                            width: 2,
                            type: LineStyleTypes.SOLID
                        }
                    };
                }

                if (!chartDef.waterfallOptions) {
                    chartDef.waterfallOptions = {
                        totalBarDisplayMode: TotalBarDisplayModes.RIGHT,
                        totalBarLabel: ''
                    };
                }

                if (!chartDef.gaugeOptions) {
                    chartDef.gaugeOptions = {
                        min: null,
                        max: null,
                        displayPointer: false
                    };
                }

                if (!chartDef.scatterZoomOptions) {
                    chartDef.scatterZoomOptions = {
                        enabled: true,
                        persisted: true,
                        scale: [1, 1],
                        translate: [0, 0]
                    };
                }

                if (!chartDef.linesZoomOptions) {
                    chartDef.linesZoomOptions = {
                        enabled: true,
                        persisted: true,
                        displayBrush: true
                    };
                }

                if (!chartDef.drilldownOptions) {
                    chartDef.drilldownOptions = {
                        displayBreadcrumb: true
                    };
                }

                if (chartDef.type === CHART_TYPES.WEBAPP) {
                    chartDef.webAppConfig = chartDef.webAppConfig || {};
                    chartDef.$loadedDesc = WebAppsService.getWebAppLoadedDesc(chartDef.webAppType) || {};
                    chartDef.$pluginDesc = WebAppsService.getOwnerPluginDesc(chartDef.webAppType);
                    chartDef.$pluginChartDesc = chartDef.$loadedDesc.desc.chart;
                    PluginConfigUtils.setDefaultValues(chartDef.$pluginChartDesc.leftBarParams, chartDef.webAppConfig);
                    PluginConfigUtils.setDefaultValues(chartDef.$pluginChartDesc.topBarParams, chartDef.webAppConfig);
                } else {
                    chartDef.$loadedDesc = null;
                    chartDef.$pluginDesc = null;
                    chartDef.$pluginChartDesc = null;
                }


                // fixup display values options on measures
                chartDef.genericMeasures.forEach(measure => {
                    if (!measure.valuesInChartDisplayOptions) {
                        measure.valuesInChartDisplayOptions = {
                            displayValues: true,
                            textFormatting: chartDef.valuesInChartDisplayOptions.textFormatting,
                            additionalMeasures: [],
                            addDetails: false
                        };
                    }
                    if (!measure.valuesInChartDisplayOptions.additionalMeasures) {
                        measure.valuesInChartDisplayOptions.additionalMeasures = [];
                    }
                    measure.valuesInChartDisplayOptions.additionalMeasures.forEach(autocompleteGenericMeasure.bind(this, { contextualMenuMeasureType: 'valuesInChart', chartDef, defaultTextFormatting: measure.valuesInChartDisplayOptions.textFormatting, theme }));
                });

                const formattableMeasuresAndDimensions = [
                    'genericMeasures',
                    'colorMeasure',
                    'uaColor',
                    'uaSize',
                    'uaShape',
                    'boxplotValue',
                    'uaTooltip',
                    'tooltipMeasures',
                    'genericDimension0',
                    'genericDimension1',
                    'facetDimension',
                    'animationDimension',
                    'xDimension',
                    'yDimension',
                    'uaXDimension',
                    'uaYDimension',
                    'groupDimension'
                ];

                const getFormattableMeasureOrDimensionDefaultsByName = (measureOrDimensionName) => {
                    if (measureOrDimensionName === 'genericMeasures') {
                        return {
                            labelPosition: ChartsStaticData.LABEL_POSITIONS.BOTTOM.value,
                            showDisplayLabel: true
                        };
                    }
                    return {};
                };

                for (const measureOrDimension of formattableMeasuresAndDimensions) {
                    if (chartDef[measureOrDimension]) {
                        chartDef[measureOrDimension] = chartDef[measureOrDimension].map(measureOrDimensionDef => ({
                            ...ChartsStaticData.DEFAULT_NUMBER_FORMATTING_OPTIONS,
                            hideTrailingZeros: getDefaultTrailingZeros(measureOrDimensionDef),
                            ...getFormattableMeasureOrDimensionDefaultsByName(measureOrDimension),
                            ...measureOrDimensionDef
                        }));
                    }
                }
                $rootScope.$emit('chart-fixed-up');
                return chartDef;
            },

            defaultNewChart: function(themeToBeUsed) {
                const theme = DSSVisualizationThemeUtils.getThemeOrDefault(themeToBeUsed ? themeToBeUsed : $rootScope.appConfig.selectedDSSVisualizationTheme);
                // Correctly initialize the new chart so that the "blank" chart that is saved is correct.
                const newChart = this.fixupChart({
                    name: 'Chart',
                    type: CHART_TYPES.GROUPED_COLUMNS,
                    variant: 'normal',
                    showLegend: true
                }, theme);
                // Even if the theme is passed to fixup, some structures can be already defined so they need to be themed.
                DSSVisualizationThemeUtils.applyToChart({ chart: newChart, theme });
                return newChart;
            },

            newEmptyGeoPlaceholder: function(index) {
                return { geometry: [], uaColor: [], colorOptions: newColorOptions(index) };
            },

            rejectMeasureOrDimension: reject,
            acceptMeasureOrDimension: accept
        };
        return svc;
    };

})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.charts');

    /**
     * ChartTypeChangeUtils service
     * Provides utils methods to compute chart type changes by retrieving information on given data/measures/dimensions.
     */
    app.factory('ChartTypeChangeUtils', function(CHART_TYPES, ChartsStaticData) {
        const service = {
            isUsableAsUA: (measure) => {
                return measure.column != null && measure.type !== 'CUSTOM';
            },

            isUsableAsMeasure: (ua) => {
                return ua.type === 'NUMERICAL';
            },

            measureToUA: (measure) => {
                return {
                    column: measure.column,
                    type: measure.type,
                    chartDefAttributeName: measure.chartDefAttributeName
                };
            },

            uaToMeasure: (ua) => {
                return {
                    column: ua.column,
                    type: ua.type,
                    function: 'AVG',
                    chartDefAttributeName: ua.chartDefAttributeName
                };
            },

            uaDimToDim: (ua) => {
                return {
                    column: ua.column,
                    type: ua.type,
                    chartDefAttributeName: ua.chartDefAttributeName
                };
            },

            uaPairToDim: (uaPair) => {
                const ret = [];
                if (uaPair.uaXDimension.length) {
                    ret.push({
                        column: uaPair.uaXDimension[0].column,
                        type: uaPair.uaXDimension[0].type,
                        chartDefAttributeName: 'uaXDimension'
                    });
                }
                if (uaPair.uaYDimension.length) {
                    ret.push({
                        column: uaPair.uaYDimension[0].column,
                        type: uaPair.uaYDimension[0].type,
                        chartDefAttributeName: 'uaYDimension'
                    });
                }
                return ret;
            },

            /**
             * Return an array of all "keys" objects and empty them in "object"
             * @param {Record<string, unknown>} object
             * @param {string[]} keys
             * @returns {unknown[]}
             */
            takeAllAndEmpty: (object, keys) => {
                if (!object) {
                    return [];
                }
                let response = [];
                for (const key of keys) {
                    if (!object[key]) {
                        continue;
                    }
                    // Addition of the chartDefAttributeName in each object allowing to retrieve these data more efficiently after (
                    response = [
                        ...response,
                        ...object[key].map(v => ({ ...v, chartDefAttributeName: key }))
                    ];
                    object[key] = [];
                }

                return response;
            },

            takeAllMeasures: (chartDef) => {
                return service.takeAllAndEmpty(chartDef, ChartsStaticData.MEASURES_PROPERTIES);
            },

            takeAllUA: (chartDef) => {
                return service.takeAllAndEmpty(chartDef, [
                    'uaSize',
                    'uaColor',
                    'uaTooltip',
                    'boxplotValue'
                ]);
            },

            takeAllMeasuresWithUA: (chartDef) => {
                return service.takeAllMeasures(chartDef)
                    .concat(service.takeAllUA(chartDef).filter(service.isUsableAsMeasure).map(service.uaToMeasure));
            },

            takeAllUAWithMeasures: (chartDef) => {
                return service.takeAllUA(chartDef)
                    .concat(service.takeAllMeasures(chartDef).filter(service.isUsableAsUA).map(service.measureToUA));
            },

            /**
             * Find and "steal" all existing dimensions from the chart.
             */
            takeAllExistingDimensions:(chartDef) => {
                // Order of props are important, else the logic after will reverse
                const dimensions = service.takeAllAndEmpty(chartDef, [...ChartsStaticData.PRIMARY_DIMENSIONS_PROPERTIES, ChartsStaticData.SECONDARY_DIMENSIONS_PROPERTIES]);

                return [
                    ...dimensions,
                    ...service.takeAllAndEmpty(chartDef, [
                        'uaXDimension',
                        'uaYDimension'
                    ]).map(service.uaDimToDim),
                    ...service.takeAllAndEmpty(chartDef, [
                        'uaDimensionPair'
                    ]).map(service.uaPairToDim).flat()
                ];
            },

            /**
             * Find and "steal" all existing primary dimensions from the chart.
             */
            takeAllExistingPrimaryDimensions:(chartDef) => {
                // Order of props are important, else the logic after will reverse
                const dimensions = service.takeAllAndEmpty(chartDef, ChartsStaticData.PRIMARY_DIMENSIONS_PROPERTIES);

                return [
                    ...dimensions,
                    ...service.takeAllAndEmpty(chartDef, [
                        'uaXDimension',
                        'uaYDimension'
                    ]).map(service.uaDimToDim),
                    ...[].concat(...service.takeAllAndEmpty(chartDef, [
                        'uaDimensionPair'
                    ]).map(service.uaPairToDim))
                ];
            },

            /**
             * Find and "steal" all existing secondary dimensions from the chart.
             */
            takeAllExistingSecondaryDimensions: (chartDef) => {
                // Order of props are important, else the logic after will reverse
                return service.takeAllAndEmpty(chartDef, ChartsStaticData.SECONDARY_DIMENSIONS_PROPERTIES);
            },

            setDimensionsOrHierarachies: (chartDef, propertyPairs, existingPrimaryDimensions, existingSecondaryDimensions, canUseHierarchies = true, canSetMultiple = false) => {
                const existingHierarchies = service.takeAllExistingHierarchies(chartDef);

                for (const pair of propertyPairs) {
                    const [dimensionProp, hierarchyProp] = pair;

                    // first we try to use a primary dimension
                    const primaryDimToUse = canSetMultiple ? [...existingPrimaryDimensions] : existingPrimaryDimensions.splice(0, 1);
                    if (primaryDimToUse.length) {
                        chartDef[dimensionProp] = primaryDimToUse;
                        continue;
                    }

                    // if it wasn't set, we try to set the hierarchy dimension
                    if (canUseHierarchies) {
                        const hierarchyToUse = canSetMultiple ? [...existingHierarchies] : existingHierarchies.splice(0, 1);
                        if (hierarchyToUse.length) {
                            chartDef[hierarchyProp] = hierarchyToUse;
                            continue;
                        }
                    }

                    // if neither primary dimension not hierarchy dimension were set, try to use secondary dimension (animation/subchart)
                    const secondaryDimToUse = canSetMultiple ? [...existingSecondaryDimensions] : existingSecondaryDimensions.splice(0, 1);
                    if (secondaryDimToUse.length) {
                        chartDef[dimensionProp] = secondaryDimToUse;
                    }
                };

            },

            /**
             * Find and "steal" all existing hierarchies from the chart.
             */
            takeAllExistingHierarchies:(chartDef) => {
                return service.takeAllAndEmpty(chartDef, ChartsStaticData.HIERARCHY_DIMENSIONS_PROPERTIES);
            },

            takeAllInX: (chartDef) => {
                switch (chartDef.type) {
                    case CHART_TYPES.STACKED_BARS:
                        return [chartDef.genericMeasures, chartDef.xMeasure, chartDef.uaXDimension, chartDef.xDimension];
                    case CHART_TYPES.SCATTER_MULTIPLE_PAIRS:
                        return chartDef.uaDimensionPair.reduce((acc, pair) => {
                            if (pair.uaXDimension && pair.uaXDimension.length) {
                                acc.push(pair.uaXDimension);
                            }
                            return acc;
                        }, []);
                    default:
                        return [chartDef.genericDimension0, chartDef.xMeasure, chartDef.uaXDimension, chartDef.xDimension];
                }
            },

            takeAllInY: (chartDef) => {
                switch(chartDef.type) {
                    case CHART_TYPES.BOXPLOTS:
                        return [chartDef.boxplotValue];
                    case CHART_TYPES.STACKED_BARS:
                        return [chartDef.genericDimension0, chartDef.yMeasure, chartDef.uaYDimension, chartDef.yDimension];
                    case CHART_TYPES.SCATTER_MULTIPLE_PAIRS:
                        return chartDef.uaDimensionPair.reduce((acc, pair) => {
                            if (pair.uaYDimension && pair.uaYDimension.length) {
                                acc.push(pair.uaYDimension);
                            }
                            return acc;
                        }, []);
                    default:
                        return [chartDef.genericMeasures, chartDef.yMeasure, chartDef.uaYDimension, chartDef.yDimension];
                }
            }
        };

        return service;
    });
})();

;
(function() {
    'use strict';

    /**
     * List of available chart types used in the chart picker ordered by usage data from analytics
     */

    const app = angular.module('dataiku.charts');

    app.factory('ChartsAvailableTypes', function(CHART_TYPES, CHART_VARIANTS, PluginsService, translate, $rootScope) {

        const svc = {};

        const ENGLISH_KEYWORDS = {
            ADMINISTRATIVE: 'administrative',
            AGGREGATION: 'aggregation',
            ASSOCIATIONS: 'associations',
            BARS: 'bars',
            BINNED: 'binned',
            CHANGE: 'change',
            COMPARISON: 'comparison',
            CURVES: 'curves',
            DISTRIBUTION: 'distribution',
            DONUTS: 'donuts',
            DOUGHNUT: 'doughnut',
            FLOW: 'flow',
            GEOGRAPHICAL: 'geographical',
            GROUP: 'group',
            HIERARCHY: 'hierarchy',
            HISTOGRAM: 'histogram',
            LEARNING: 'learning',
            LINES: 'lines',
            LOCATION: 'location',
            MACHINE: 'machine',
            MAGNITUDE: 'magnitude',
            MAPS: 'maps',
            METRIC: 'metric',
            MODELS: 'models',
            PART: 'part',
            PATTERNS: 'patterns',
            PERCENTAGE: 'percentage',
            PIES: 'pies',
            PROPORTION: 'proportion',
            RANGE: 'range',
            RELATIONSHIPS: 'relationships',
            SCATTERS: 'scatters',
            SPEEDOMETER: 'speedometer',
            SPIDER: 'spider',
            SUMMARY: 'summary',
            TABLES: 'tables',
            TIME: 'time',
            WEB: 'web'
        };

        /** @type {Record<keyof typeof ENGLISH_KEYWORDS, string[]>} */
        const KEYWORDS = Object.entries(ENGLISH_KEYWORDS).reduce((acc, [key, value]) => {
            const keywords = [value];
            const translated = translate(`CHARTS.TYPE_PICKER.KEYWORD.${key}`, value);
            // Avoid duplicating the keywords if DSS is in English.
            if (translated !== value) {
                keywords.push(translated);
            }
            acc[key] = keywords;
            return acc;
        }, {});

        const ENGLISH_DISPLAY_NAMES = {
            VERTICAL_BARS: 'Vertical bars',
            WATERFALL: 'Waterfall',
            VERTICAL_STACKED_BARS: 'Vertical stacked bars',
            VERTICAL_STACKED_BARS_100: 'Vertical stacked_bars 100%',
            HORIZONTAL_STACKED_BARS: 'Horizontal stacked bars',
            HORIZONTAL_STACKED_BARS_100: 'Horizontal stacked bars 100%',
            LINES: 'Lines',
            STACKED_AREAS: 'Stacked_areas',
            STACKED_AREAS_100: 'Stacked areas 100%',
            PIE: 'Pie',
            DONUT: 'Donut',
            MIX: 'Mix',
            PIVOT_TABLE: 'Pivot table',
            KPIS: 'KPIs',
            RADAR: 'Radar',
            GAUGE: 'Gauge',
            SANKEY: 'Sankey',
            TREEMAP: 'Treemap',
            SCATTER_PLOT: 'Scatter plot',
            SCATTER_MULTI_PAIR: 'Scatter multi-pair',
            GEOMETRY_MAP: 'Geometry map',
            GRID_MAP: 'Grid map',
            SCATTER_MAP: 'Scatter map',
            ADMINISTRATIVE_MAP_FILLED: 'Administrative map (filled)',
            ADMINISTRATIVE_MAP_BUBBLES: 'Administrative map (bubbles)',
            DENSITY_MAP: 'Density map',
            GROUPED_BUBBLES: 'Grouped bubbles',
            BINNED_BUBBLES: 'Binned bubbles',
            BINNED_HEXAGONS: 'Binned hexagons',
            BINNED_RECTANGLES: 'Binned rectangles',
            BOXPLOT: 'Boxplot',
            '2D_DISTRIBUTION': '2D distribution',
            LIFT_CHART: 'Lift chart'
        };

        /** @type {Record<keyof typeof ENGLISH_DISPLAY_NAMES, string[]>} */
        const ENGLISH_DISPLAY_NAME_KEYWORDS = Object.entries(ENGLISH_DISPLAY_NAMES).reduce((acc, [key, value]) => {
            acc[key] = splitToKeywords(value);
            return acc;
        }, {});

        function splitToKeywords(phrase) {
            return phrase.trim().toLowerCase().split(/[, ]+/);
        }

        svc.getAvailableChartTypes = function() {
            // Keep in sync with dip/pivot/frontend/model/ChartDef.java::getChartDisplayName()
            return [
                {
                    id: 'qa_charts_histogram-chart-type',
                    type: CHART_TYPES.GROUPED_COLUMNS,
                    variant: CHART_VARIANTS.normal,
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.VERTICAL_BARS', ENGLISH_DISPLAY_NAMES.VERTICAL_BARS),
                    get keywords() {
                        return [...KEYWORDS.MAGNITUDE, ...KEYWORDS.COMPARISON, ...KEYWORDS.PATTERNS, ...KEYWORDS.PROPORTION, ...KEYWORDS.HISTOGRAM, ...KEYWORDS.BARS, ...ENGLISH_DISPLAY_NAME_KEYWORDS.VERTICAL_BARS, ...splitToKeywords(this.displayName)];
                    },
                    similar: ['qa_charts_stacked-chart-type', 'qa_charts_stacked-100-chart-type', 'qa_charts_bars-chart-type', 'qa_charts_bars-100-chart-type', 'qa_charts_waterfall-chart-type', 'qa_charts_pivot-chart-type', 'qa_charts_treemap-chart-type'],
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.VERTICAL_BARS', 'Compares values across grouped items or within a subgroup vertically')
                },
                $rootScope?.featureFlagEnabled?.('unaggregatedMeasures') && {
                    id: 'qa_charts_waterfall-chart-type',
                    type: CHART_TYPES.GROUPED_COLUMNS,
                    variant: CHART_VARIANTS.waterfall,
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.WATERFALL', 'Waterfall chart'),
                    get keywords() {
                        return [...KEYWORDS.MAGNITUDE, ...KEYWORDS.COMPARISON, ...KEYWORDS.PATTERNS, ...KEYWORDS.HISTOGRAM, ...KEYWORDS.BARS, ...ENGLISH_DISPLAY_NAME_KEYWORDS.WATERFALL, ...splitToKeywords(this.displayName)];
                    },
                    similar: ['qa_charts_histogram-chart-type', 'qa_charts_stacked-chart-type', 'qa_charts_stacked-100-chart-type', 'qa_charts_bars-chart-type', 'qa_charts_bars-100-chart-type', 'qa_charts_pivot-chart-type'],
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.WATERFALL', 'Shows how negative and positive changes affect a starting value, leading to the final value')
                },
                {
                    id: 'qa_charts_stacked-chart-type',
                    type: CHART_TYPES.STACKED_COLUMNS,
                    variant: CHART_VARIANTS.normal,
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.VERTICAL_STACKED_BARS', ENGLISH_DISPLAY_NAMES.VERTICAL_STACKED_BARS),
                    get keywords() {
                        return [...KEYWORDS.MAGNITUDE, ...KEYWORDS.COMPARISON, ...KEYWORDS.PATTERNS, ...KEYWORDS.PROPORTION, ...KEYWORDS.BARS, ...ENGLISH_DISPLAY_NAME_KEYWORDS.VERTICAL_STACKED_BARS];
                    },
                    similar: ['qa_charts_histogram-chart-type', 'qa_charts_stacked-100-chart-type', 'qa_charts_bars-chart-type', 'qa_charts_bars-100-chart-type', 'qa_charts_pivot-chart-type', 'qa_charts_treemap-chart-type'],
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.VERTICAL_STACKED_BARS', 'Shows part-to-whole relationship within a group vertically')
                },
                {
                    id: 'qa_charts_stacked-100-chart-type',
                    type: CHART_TYPES.STACKED_COLUMNS,
                    variant: CHART_VARIANTS.stacked100,
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.VERTICAL_STACKED_BARS_100', ENGLISH_DISPLAY_NAMES.VERTICAL_STACKED_BARS_100),
                    get keywords() {
                        return [...KEYWORDS.MAGNITUDE, ...KEYWORDS.COMPARISON, ...KEYWORDS.PATTERNS, ...KEYWORDS.PROPORTION, ...KEYWORDS.PERCENTAGE, ...KEYWORDS.BARS, ...ENGLISH_DISPLAY_NAME_KEYWORDS.VERTICAL_STACKED_BARS_100, ...splitToKeywords(this.displayName)];
                    },
                    similar: ['qa_charts_histogram-chart-type', 'qa_charts_stacked-chart-type', 'qa_charts_bars-chart-type', 'qa_charts_bars-100-chart-type', 'qa_charts_pivot-chart-type', 'qa_charts_treemap-chart-type'],
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.VERTICAL_STACKED_BARS_100', 'Highlights the relative difference among values in each group - each bar is always 100%')
                },
                {
                    id: 'qa_charts_bars-chart-type',
                    type: CHART_TYPES.STACKED_BARS,
                    variant: CHART_VARIANTS.normal,
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.HORIZONTAL_STACKED_BARS', ENGLISH_DISPLAY_NAMES.HORIZONTAL_STACKED_BARS),
                    get keywords() {
                        return [...KEYWORDS.MAGNITUDE, ...KEYWORDS.COMPARISON, ...KEYWORDS.PATTERNS, ...KEYWORDS.PROPORTION, ...KEYWORDS.BARS, ...ENGLISH_DISPLAY_NAME_KEYWORDS.HORIZONTAL_STACKED_BARS, ...splitToKeywords(this.displayName)];
                    },
                    similar: ['qa_charts_histogram-chart-type', 'qa_charts_stacked-chart-type', 'qa_charts_stacked-100-chart-type', 'qa_charts_bars-100-chart-type', 'qa_charts_pivot-chart-type', 'qa_charts_treemap-chart-type'],
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.HORIZONTAL_STACKED_BARS', 'Compares values across grouped items or within a subgroup horizontally')
                },
                {
                    id: 'qa_charts_bars-100-chart-type',
                    type: CHART_TYPES.STACKED_BARS,
                    variant: CHART_VARIANTS.stacked100,
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.HORIZONTAL_STACKED_BARS_100', ENGLISH_DISPLAY_NAMES.HORIZONTAL_STACKED_BARS_100),
                    get keywords() {
                        return [...KEYWORDS.MAGNITUDE, ...KEYWORDS.COMPARISON, ...KEYWORDS.PATTERNS, ...KEYWORDS.PROPORTION, ...KEYWORDS.PERCENTAGE, ...KEYWORDS.BARS, ...ENGLISH_DISPLAY_NAME_KEYWORDS.HORIZONTAL_STACKED_BARS_100, ...splitToKeywords(this.displayName)];
                    },
                    similar: ['qa_charts_histogram-chart-type', 'qa_charts_stacked-chart-type', 'qa_charts_stacked-100-chart-type', 'qa_charts_bars-chart-type', 'qa_charts_pivot-chart-type', 'qa_charts_treemap-chart-type'],
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.HORIZONTAL_STACKED_BARS_100', 'Highlights the relative difference among values in each group - each bar is always 100%')
                },
                {
                    id: 'qa_charts_lines-chart-type',
                    type: CHART_TYPES.LINES,
                    variant: CHART_VARIANTS.normal,
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.LINES', ENGLISH_DISPLAY_NAMES.LINES),
                    get keywords() {
                        return [...KEYWORDS.CHANGE, ...KEYWORDS.TIME, ...KEYWORDS.PATTERNS, ...KEYWORDS.LINES, ...KEYWORDS.CURVES, ...ENGLISH_DISPLAY_NAME_KEYWORDS.LINES, ...splitToKeywords(this.displayName)];
                    },
                    similar: ['qa_charts_mix-chart-type', 'qa_charts_stacked-area-chart-type', 'qa_charts_stacked-area-100-chart-type'],
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.LINES', 'Display quantitative values over a continuous interval or time period')
                },
                {
                    id: 'qa_charts_stacked-area-chart-type',
                    type: CHART_TYPES.STACKED_AREA,
                    variant: CHART_VARIANTS.normal,
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.STACKED_AREAS', ENGLISH_DISPLAY_NAMES.STACKED_AREAS),
                    get keywords() {
                        return [...KEYWORDS.CHANGE, ...KEYWORDS.TIME, ...KEYWORDS.PATTERNS, ...KEYWORDS.MAGNITUDE, ...KEYWORDS.COMPARISON, ...KEYWORDS.PROPORTION, ...KEYWORDS.PERCENTAGE, ...KEYWORDS.LINES, ...KEYWORDS.CURVES, ...ENGLISH_DISPLAY_NAME_KEYWORDS.STACKED_AREAS, ...splitToKeywords(this.displayName)];
                    },
                    similar: ['qa_charts_stacked-area-100-chart-type', 'qa_charts_lines-chart-type', 'qa_charts_mix-chart-type'],
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.STACKED_AREAS', 'Shows how values evolve over a continuous interval or time period')
                },
                {
                    id: 'qa_charts_stacked-area-100-chart-type',
                    type: CHART_TYPES.STACKED_AREA,
                    variant: CHART_VARIANTS.stacked100,
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.STACKED_AREAS_100', ENGLISH_DISPLAY_NAMES.STACKED_AREAS_100),
                    get keywords() {
                        return [...KEYWORDS.CHANGE, ...KEYWORDS.TIME, ...KEYWORDS.PATTERNS, ...KEYWORDS.MAGNITUDE, ...KEYWORDS.COMPARISON, ...KEYWORDS.PROPORTION, ...KEYWORDS.PERCENTAGE, ...KEYWORDS.LINES, ...KEYWORDS.CURVES, ...ENGLISH_DISPLAY_NAME_KEYWORDS.STACKED_AREAS_100, ...splitToKeywords(this.displayName)];
                    },
                    similar: ['qa_charts_stacked-area-chart-type', 'qa_charts_lines-chart-type', 'qa_charts_mix-chart-type'],
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.STACKED_AREAS_100', 'Highlights the relative difference among values evolving over a continuous time period')
                },
                {
                    id: 'qa_charts_pie-chart-type',
                    type: CHART_TYPES.PIE,
                    variant: CHART_VARIANTS.normal,
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.PIE', ENGLISH_DISPLAY_NAMES.PIE),
                    get keywords() {
                        return [...KEYWORDS.PART, ...KEYWORDS.PROPORTION, ...KEYWORDS.PIES, ...KEYWORDS.DONUTS, ...ENGLISH_DISPLAY_NAME_KEYWORDS.PIE, ...splitToKeywords(this.displayName)];
                    },
                    similar: ['qa_charts_donut-chart-type'],
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.PIE', 'Shows a quick part-to-whole comparison')
                },
                {
                    id: 'qa_charts_donut-chart-type',
                    type: CHART_TYPES.PIE,
                    variant: CHART_VARIANTS.donut,
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.DONUT', ENGLISH_DISPLAY_NAMES.DONUT),
                    get keywords() {
                        return [...KEYWORDS.PART, ...KEYWORDS.PROPORTION, ...KEYWORDS.PIES, ...KEYWORDS.DONUTS, ...KEYWORDS.DOUGHNUT, ...ENGLISH_DISPLAY_NAME_KEYWORDS.DONUT, ...splitToKeywords(this.displayName)];
                    },
                    similar: ['qa_charts_pie-chart-type'],
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.DONUT', 'Shows a quick part-to-whole comparison with a hole inside')
                },
                {
                    id: 'qa_charts_mix-chart-type',
                    type: CHART_TYPES.MULTI_COLUMNS_LINES,
                    variant: CHART_VARIANTS.normal,
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.MIX', ENGLISH_DISPLAY_NAMES.MIX),
                    get keywords() {
                        return [...KEYWORDS.CHANGE, ...KEYWORDS.TIME, ...KEYWORDS.MAGNITUDE, ...KEYWORDS.COMPARISON, ...KEYWORDS.PATTERNS, ...KEYWORDS.PROPORTION, ...KEYWORDS.PERCENTAGE, ...KEYWORDS.BARS, ...KEYWORDS.LINES, ...KEYWORDS.CURVES, ...ENGLISH_DISPLAY_NAME_KEYWORDS.MIX, ...splitToKeywords(this.displayName)];
                    },
                    similar: ['qa_charts_lines-chart-type', 'qa_charts_stacked-area-chart-type', 'qa_charts_stacked-area-100-chart-type'],
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.MIX', 'Shows a direct comparison between two sets of values with separate axes')
                },
                {
                    id: 'qa_charts_pivot-chart-type',
                    type: CHART_TYPES.PIVOT_TABLE,
                    variant: CHART_VARIANTS.normal,
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.PIVOT_TABLE', ENGLISH_DISPLAY_NAMES.PIVOT_TABLE),
                    get keywords() {
                        return [...KEYWORDS.SUMMARY, ...KEYWORDS.GROUP, ...KEYWORDS.AGGREGATION, ...KEYWORDS.HIERARCHY, ...KEYWORDS.TABLES, ...ENGLISH_DISPLAY_NAME_KEYWORDS.PIVOT_TABLE, ...splitToKeywords(this.displayName)];
                    },
                    similar: ['qa_charts_histogram-chart-type', 'qa_charts_stacked-chart-type', 'qa_charts_stacked-100-chart-type', 'qa_charts_bars-chart-type', 'qa_charts_bars-100-chart-type', 'qa_charts_treemap-chart-type'],
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.PIVOT_TABLE', 'Displays aggregated values in a table grouped across several dimensions')
                },
                {
                    id: 'qa_charts_kpi-chart-type',
                    type: CHART_TYPES.KPI,
                    variant: CHART_VARIANTS.normal,
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.KPI', ENGLISH_DISPLAY_NAMES.KPIS),
                    get keywords() {
                        return [...KEYWORDS.METRIC, ...ENGLISH_DISPLAY_NAME_KEYWORDS.KPIS, ...splitToKeywords(this.displayName)];
                    },
                    similar: ['qa_charts_gauge-chart-type'],
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.KPI', 'Displays a simple value visualization for single or multiple aggregations')
                },
                {
                    id: 'qa_charts_radar-chart-type',
                    type: CHART_TYPES.RADAR,
                    variant: CHART_VARIANTS.normal,
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.RADAR', ENGLISH_DISPLAY_NAMES.RADAR),
                    get keywords() {
                        return [...KEYWORDS.WEB, ...KEYWORDS.SPIDER, ...ENGLISH_DISPLAY_NAME_KEYWORDS.RADAR, ...splitToKeywords(this.displayName)];
                    },
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.RADAR', 'Displays data in a radial axis, where the radii represents the performance of values')
                },
                {
                    id: 'qa_charts_gauge-chart-type',
                    type: CHART_TYPES.GAUGE,
                    variant: CHART_VARIANTS.normal,
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.GAUGE', ENGLISH_DISPLAY_NAMES.GAUGE),
                    get keywords() {
                        return [...KEYWORDS.SPEEDOMETER, ...ENGLISH_DISPLAY_NAME_KEYWORDS.GAUGE, ...splitToKeywords(this.displayName)];
                    },
                    similar: ['qa_charts_kpi-chart-type'],
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.GAUGE', 'Displays how a value performs in a semi-circular scale')
                },
                {
                    id: 'qa_charts_sankey-chart-type',
                    type: CHART_TYPES.SANKEY,
                    variant: CHART_VARIANTS.normal,
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.SANKEY', ENGLISH_DISPLAY_NAMES.SANKEY),
                    get keywords() {
                        return [...KEYWORDS.FLOW, ...ENGLISH_DISPLAY_NAME_KEYWORDS.SANKEY, ...splitToKeywords(this.displayName)];
                    },
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.SANKEY', 'Displays a weighted flow of resources')
                },
                {
                    id: 'qa_charts_treemap-chart-type',
                    type: CHART_TYPES.TREEMAP,
                    variant: CHART_VARIANTS.normal,
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.TREEMAP', ENGLISH_DISPLAY_NAMES.TREEMAP),
                    get keywords() {
                        return [...KEYWORDS.HIERARCHY, ...KEYWORDS.COMPARISON, ...KEYWORDS.PROPORTION, ...KEYWORDS.PART, ...ENGLISH_DISPLAY_NAME_KEYWORDS.TREEMAP, ...splitToKeywords(this.displayName)];
                    },
                    similar: ['qa_charts_pivot-chart-type', 'qa_charts_histogram-chart-type', 'qa_charts_stacked-chart-type', 'qa_charts_stacked-100-chart-type', 'qa_charts_bars-chart-type', 'qa_charts_bars-100-chart-type'],
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.TREEMAP', 'Displays hierarchical data as nested rectangles with proportional sizes')
                },
                {
                    id: 'qa_charts_scatter-plot-chart-type',
                    type: CHART_TYPES.SCATTER,
                    variant: CHART_VARIANTS.normal,
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.SCATTER_PLOT', ENGLISH_DISPLAY_NAMES.SCATTER_PLOT),
                    get keywords() {
                        return [...KEYWORDS.ASSOCIATIONS, ...KEYWORDS.PATTERNS, ...KEYWORDS.RELATIONSHIPS, ...KEYWORDS.SCATTERS, ...ENGLISH_DISPLAY_NAME_KEYWORDS.SCATTER_PLOT, ...splitToKeywords(this.displayName)];
                    },
                    similar: ['qa_charts_scatter-plot-multiple-pairs-chart-type', 'qa_charts_bubble-chart-type', 'qa_charts_rectangle-chart-type', 'qa_charts_hexagon-chart-type', 'qa_charts_grouped-bubbles-chart-type'],
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.SCATTER_PLOT', 'Shows the relationships between numerical variables drawn on the axes')
                },
                {
                    id: 'qa_charts_scatter-plot-multiple-pairs-chart-type',
                    type: CHART_TYPES.SCATTER_MULTIPLE_PAIRS,
                    variant: CHART_VARIANTS.normal,
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.SCATTER_MULTI_PAIR', ENGLISH_DISPLAY_NAMES.SCATTER_MULTI_PAIR),
                    get keywords() {
                        return [...KEYWORDS.ASSOCIATIONS, ...KEYWORDS.PATTERNS, ...KEYWORDS.RELATIONSHIPS, ...KEYWORDS.SCATTERS, ...ENGLISH_DISPLAY_NAME_KEYWORDS.SCATTER_MULTI_PAIR, ...splitToKeywords(this.displayName)];
                    },
                    similar: ['qa_charts_scatter-plot-chart-type', 'qa_charts_bubble-chart-type', 'qa_charts_rectangle-chart-type', 'qa_charts_hexagon-chart-type', 'qa_charts_grouped-bubbles-chart-type'],
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.SCATTER_MULTI_PAIR', 'Shows the relationships between numerical variables drawn on the axes (with multiple pairs)')
                },
                {
                    id: 'qa_charts_geo-map-chart-type',
                    type: CHART_TYPES.GEOMETRY_MAP,
                    variant: CHART_VARIANTS.normal,
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.GEOMETRY_MAP', ENGLISH_DISPLAY_NAMES.GEOMETRY_MAP),
                    get keywords() {
                        return [...KEYWORDS.GEOGRAPHICAL, ...KEYWORDS.LOCATION, ...KEYWORDS.SCATTERS, ...KEYWORDS.MAPS, ...ENGLISH_DISPLAY_NAME_KEYWORDS.GEOMETRY_MAP, ...splitToKeywords(this.displayName)];
                    },
                    similar: ['qa_charts_scatter-map-chart-type', 'qa_charts_bubble-map-chart-type', 'qa_charts_filled-map-chart-type', 'qa_charts_grid-map-chart-type', 'qa_charts_density-map-chart-type'],
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.GEOMETRY_MAP', 'Shows geometries as areas on a map')
                },
                {
                    id: 'qa_charts_grid-map-chart-type',
                    type: CHART_TYPES.GRID_MAP,
                    variant: CHART_VARIANTS.normal,
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.GRID_MAP', ENGLISH_DISPLAY_NAMES.GRID_MAP),
                    get keywords() {
                        return [...KEYWORDS.GEOGRAPHICAL, ...KEYWORDS.LOCATION, ...KEYWORDS.BINNED, ...KEYWORDS.MAPS, ...ENGLISH_DISPLAY_NAME_KEYWORDS.GRID_MAP, ...splitToKeywords(this.displayName)];
                    },
                    similar: ['qa_charts_scatter-map-chart-type', 'qa_charts_bubble-map-chart-type', 'qa_charts_filled-map-chart-type', 'qa_charts_geo-map-chart-type', 'qa_charts_density-map-chart-type'],
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.GRID_MAP', 'Shows geographical data at administrative level on a map as a grid')
                },
                {
                    id: 'qa_charts_scatter-map-chart-type',
                    type: CHART_TYPES.SCATTER_MAP,
                    variant: CHART_VARIANTS.normal,
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.SCATTER_MAP', ENGLISH_DISPLAY_NAMES.SCATTER_MAP),
                    get keywords() {
                        return [...KEYWORDS.GEOGRAPHICAL, ...KEYWORDS.LOCATION, ...KEYWORDS.SCATTERS, ...KEYWORDS.MAPS, ...ENGLISH_DISPLAY_NAME_KEYWORDS.SCATTER_MAP, ...splitToKeywords(this.displayName)];
                    },
                    similar: ['qa_charts_bubble-map-chart-type', 'qa_charts_filled-map-chart-type', 'qa_charts_geo-map-chart-type', 'qa_charts_grid-map-chart-type', 'qa_charts_density-map-chart-type'],
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.SCATTER_MAP', 'Shows geographical data as points on a map')
                },
                {
                    id: 'qa_charts_filled-map-chart-type',
                    type: CHART_TYPES.ADMINISTRATIVE_MAP,
                    variant: CHART_VARIANTS.filledMap,
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.ADMINISTRATIVE_MAP_FILLED', ENGLISH_DISPLAY_NAMES.ADMINISTRATIVE_MAP_FILLED),
                    get keywords() {
                        return [...KEYWORDS.GEOGRAPHICAL, ...KEYWORDS.LOCATION, ...KEYWORDS.ADMINISTRATIVE, ...KEYWORDS.MAPS, ...ENGLISH_DISPLAY_NAME_KEYWORDS.ADMINISTRATIVE_MAP_FILLED, ...splitToKeywords(this.displayName)];
                    },
                    similar: ['qa_charts_scatter-map-chart-type', 'qa_charts_bubble-map-chart-type', 'qa_charts_geo-map-chart-type', 'qa_charts_grid-map-chart-type', 'qa_charts_density-map-chart-type'],
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.ADMINISTRATIVE_MAP_FILLED', 'Draws the boundary of geographic data at administrative level as a filled shaped)'),
                    requiredPluginId: 'geoadmin',
                    isRequiredPluginInstalled: PluginsService.isPluginLoaded('geoadmin')
                },
                {
                    id: 'qa_charts_bubble-map-chart-type',
                    type: CHART_TYPES.ADMINISTRATIVE_MAP,
                    variant: CHART_VARIANTS.normal,
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.ADMINISTRATIVE_MAP_BUBBLES', ENGLISH_DISPLAY_NAMES.ADMINISTRATIVE_MAP_BUBBLES),
                    get keywords() {
                        return [...KEYWORDS.GEOGRAPHICAL, ...KEYWORDS.LOCATION, ...KEYWORDS.ADMINISTRATIVE, ...KEYWORDS.MAPS, ...ENGLISH_DISPLAY_NAME_KEYWORDS.ADMINISTRATIVE_MAP_BUBBLES, ...splitToKeywords(this.displayName)];
                    },
                    similar: ['qa_charts_scatter-map-chart-type', 'qa_charts_filled-map-chart-type', 'qa_charts_geo-map-chart-type', 'qa_charts_grid-map-chart-type', 'qa_charts_density-map-chart-type'],
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.ADMINISTRATIVE_MAP_BUBBLES', 'Shows geographical data at administrative level on the map as a circle'),
                    requiredPluginId: 'geoadmin',
                    isRequiredPluginInstalled: PluginsService.isPluginLoaded('geoadmin')
                },
                {
                    id: 'qa_charts_density-map-chart-type',
                    type: CHART_TYPES.DENSITY_HEAT_MAP,
                    variant: CHART_VARIANTS.normal,
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.DENSITY_MAP', ENGLISH_DISPLAY_NAMES.DENSITY_MAP),
                    get keywords() {
                        return [...KEYWORDS.GEOGRAPHICAL, ...KEYWORDS.LOCATION, ...KEYWORDS.SCATTERS, ...KEYWORDS.MAPS, ...ENGLISH_DISPLAY_NAME_KEYWORDS.DENSITY_MAP, ...splitToKeywords(this.displayName)];
                    },
                    similar: ['qa_charts_scatter-map-chart-type', 'qa_charts_bubble-map-chart-type', 'qa_charts_filled-map-chart-type', 'qa_charts_geo-map-chart-type', 'qa_charts_grid-map-chart-type'],
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.DENSITY_MAP', 'Shows where points are concentrated on a map by their spatial proximity')
                },
                {
                    id: 'qa_charts_grouped-bubbles-chart-type',
                    type: CHART_TYPES.GROUPED_XY,
                    variant: CHART_VARIANTS.normal,
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.GROUPED_BUBBLES', ENGLISH_DISPLAY_NAMES.GROUPED_BUBBLES),
                    get keywords() {
                        return [...KEYWORDS.ASSOCIATIONS, ...KEYWORDS.PATTERNS, ...KEYWORDS.RELATIONSHIPS, ...KEYWORDS.SCATTERS, ...ENGLISH_DISPLAY_NAME_KEYWORDS.GROUPED_BUBBLES, ...splitToKeywords(this.displayName)];
                    },
                    similar: ['qa_charts_scatter-plot-chart-type', 'qa_charts_scatter-plot-multiple-pairs-chart-type', 'qa_charts_bubble-chart-type', 'qa_charts_rectangle-chart-type', 'qa_charts_hexagon-chart-type'],
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.GROUPED_BUBBLES', 'Shows where points are concentrated on a map by their spatial proximity')
                },
                {
                    id: 'qa_charts_bubble-chart-type',
                    type: CHART_TYPES.BINNED_XY,
                    variant: CHART_VARIANTS.normal,
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.BINNED_BUBBLES', ENGLISH_DISPLAY_NAMES.BINNED_BUBBLES),
                    get keywords() {
                        return [...KEYWORDS.ASSOCIATIONS, ...KEYWORDS.PATTERNS, ...KEYWORDS.RELATIONSHIPS, ...KEYWORDS.SCATTERS, ...KEYWORDS.BINNED, ...ENGLISH_DISPLAY_NAME_KEYWORDS.BINNED_BUBBLES, ...splitToKeywords(this.displayName)];
                    },
                    similar: ['qa_charts_scatter-plot-chart-type', 'qa_charts_scatter-plot-multiple-pairs-chart-type', 'qa_charts_rectangle-chart-type', 'qa_charts_hexagon-chart-type', 'qa_charts_grouped-bubbles-chart-type'],
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.BINNED_BUBBLES', 'Shows the relationships between categorical or binned numerical variables on the axes')
                },
                {
                    id: 'qa_charts_hexagon-chart-type',
                    type: CHART_TYPES.BINNED_XY,
                    variant: CHART_VARIANTS.binnedXYHexagon,
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.BINNED_HEXAGONS', ENGLISH_DISPLAY_NAMES.BINNED_HEXAGONS),
                    get keywords() {
                        return [...KEYWORDS.ASSOCIATIONS, ...KEYWORDS.PATTERNS, ...KEYWORDS.RELATIONSHIPS, ...KEYWORDS.SCATTERS, ...KEYWORDS.BINNED, ...ENGLISH_DISPLAY_NAME_KEYWORDS.BINNED_HEXAGONS, ...splitToKeywords(this.displayName)];
                    },
                    similar: ['qa_charts_scatter-plot-chart-type', 'qa_charts_scatter-plot-multiple-pairs-chart-type', 'qa_charts_bubble-chart-type', 'qa_charts_rectangle-chart-type', 'qa_charts_grouped-bubbles-chart-type'],
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.BINNED_HEXAGONS', 'Shows a regular array of hexagons to show density for large datasets')
                },
                {
                    id: 'qa_charts_rectangle-chart-type',
                    type: CHART_TYPES.BINNED_XY,
                    variant: CHART_VARIANTS.binnedXYRectangle,
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.BINNED_RECTANGLES', ENGLISH_DISPLAY_NAMES.BINNED_RECTANGLES),
                    get keywords() {
                        return [...KEYWORDS.ASSOCIATIONS, ...KEYWORDS.PATTERNS, ...KEYWORDS.RELATIONSHIPS, ...KEYWORDS.SCATTERS, ...KEYWORDS.BINNED, ...ENGLISH_DISPLAY_NAME_KEYWORDS.BINNED_RECTANGLES, ...splitToKeywords(this.displayName)];
                    },
                    similar: ['qa_charts_scatter-plot-chart-type', 'qa_charts_scatter-plot-multiple-pairs-chart-type', 'qa_charts_bubble-chart-type', 'qa_charts_hexagon-chart-type', 'qa_charts_grouped-bubbles-chart-type'],
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.BINNED_RECTANGLES', 'Shows the relationships between categorical or binned numerical variables')
                },
                {
                    id: 'qa_charts_boxplot-chart-type',
                    type: CHART_TYPES.BOXPLOTS,
                    variant: CHART_VARIANTS.normal,
                    get keywords() {
                        return [...KEYWORDS.DISTRIBUTION, ...KEYWORDS.RANGE, ...ENGLISH_DISPLAY_NAME_KEYWORDS.BOXPLOT, ...splitToKeywords(this.displayName)];
                    },
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.BOXPLOT', ENGLISH_DISPLAY_NAMES.BOXPLOT),
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.BOXPLOT', 'Displays the distribution of data showing minimum, Q1, median, Q3 and maximum')
                },
                {
                    id: 'qa_charts_density-2d-chart-type',
                    type: CHART_TYPES.DENSITY_2D,
                    variant: CHART_VARIANTS.normal,
                    get keywords() {
                        return [...KEYWORDS.DISTRIBUTION, ...ENGLISH_DISPLAY_NAME_KEYWORDS['2D_DISTRIBUTION'], ...splitToKeywords(this.displayName)];
                    },
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.2D_DISTRIBUTION', ENGLISH_DISPLAY_NAMES['2D_DISTRIBUTION']),
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.2D_DISTRIBUTION', 'Shows the bivariate distribution of two fields')
                },
                {
                    id: 'qa_charts_lift-chart-type',
                    type: CHART_TYPES.LIFT,
                    variant: CHART_VARIANTS.normal,
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.LIFT', ENGLISH_DISPLAY_NAMES.LIFT_CHART),
                    get keywords() {
                        return [...KEYWORDS.MACHINE, ...KEYWORDS.LEARNING, ...KEYWORDS.MODELS, ...ENGLISH_DISPLAY_NAME_KEYWORDS.LIFT_CHART, ...splitToKeywords(this.displayName)];
                    },
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.LIFT', 'Measures the effectiveness of models between results with and without a model')
                }
            ].filter(chartType => !!chartType);
        };

        return svc;
    });
})();

;


(function() {
    'use strict';

    angular.module('dataiku.charts')
        .service('ChartColorScales', ChartColorScales);

    /**
     * Colors scales creation logic
     * (!) This service previously was in static/dataiku/js/simple_report/common/colors.js
     */
    function ChartColorScales(ChartUADimension, ChartDataUtils, StringNormalizer, ColorUtils, ChartColorUtils, ChartsStaticData, DKU_PALETTE_NAMES, CHART_AXIS_TYPES) {

        const svc = {

            /**
             * Create a color scale
             * @param {ChartColorContext} colorContext (typed in chart-color-context.interface.ts)
             * @return {*}
             */
            createColorScale: function(colorContext) {
                if (!colorContext.colorSpec) {
                    return null;
                }

                let colorScale;
                switch (colorContext.colorSpec.type) {
                    case CHART_AXIS_TYPES.DIMENSION:
                        colorScale = svc.discreteColorScale(colorContext.colorOptions, colorContext.defaultLegendDimension, colorContext.chartData, colorContext.colorSpec.withRgba,
                            ChartColorUtils.getColorMeaningInfo(colorContext.colorSpec.dimension, colorContext.chartHandler), colorContext.chartData.getAxisLabels(colorContext.colorSpec.name), colorContext.theme, colorContext.uaColorIndex, colorContext.ignoreLabels);
                        if (colorScale.domain) {
                            if (colorContext.chartData.getAxisIdx(colorContext.colorSpec.name) != undefined) {
                                colorScale.domain(colorContext.chartData.getAxisLabels(colorContext.colorSpec.name).map(function(d, i) {
                                    return i;
                                }));
                            } else {
                                colorScale.domain(colorContext.defaultLegendDimension.map(function(d, i) {
                                    return i;
                                }));
                            }
                        }
                        break;
                    case CHART_AXIS_TYPES.MEASURE:
                        if (!colorContext.colorSpec.extent) {
                            if (colorContext.colorSpec.measureIdx === undefined || colorContext.colorSpec.measureIdx < 0) {
                                return null;
                            }
                            colorContext.colorSpec.extent = ChartDataUtils.getMeasureExtent(colorContext.chartData, colorContext.colorSpec.measureIdx, true, colorContext.colorSpec.binsToInclude, colorContext.colorSpec.colorAggrFn);
                        }

                        if (!colorContext.colorSpec.values) {
                            if (colorContext.colorSpec.measureIdx === undefined || colorContext.colorSpec.measureIdx < 0) {
                                return null;
                            }
                            colorContext.colorSpec.values = ChartDataUtils.getMeasureValues(colorContext.chartData.data, colorContext.colorSpec.measureIdx, colorContext.colorSpec.binsToInclude);
                        }

                        colorScale = svc.continuousColorScale(colorContext.colorOptions, colorContext.colorSpec.extent[0], colorContext.colorSpec.extent[1], colorContext.colorSpec.values, !colorContext.colorSpec.withRgba, colorContext.theme, colorContext.colorSpec.binsToInclude);
                        break;
                    case CHART_AXIS_TYPES.UNAGGREGATED:
                        if (!colorContext.colorSpec.dimension) {
                            return null;
                        }
                        var extent = ChartDataUtils.getUnaggregatedAxisExtent(colorContext.colorSpec.dimension, colorContext.colorSpec.data, colorContext.chartData.data.afterFilterRecords);
                        if (ChartUADimension.isTrueNumerical(colorContext.colorSpec.dimension) || ChartUADimension.isDateRange(colorContext.colorSpec.dimension)) {
                            colorScale = svc.continuousColorScale(colorContext.colorOptions, extent.min, extent.max, extent.values, !colorContext.colorSpec.withRgba, colorContext.theme);
                            colorScale.isContinuous = true;
                        } else {
                            colorScale = svc.discreteColorScale(colorContext.colorOptions, colorContext.defaultLegendDimension, colorContext.chartData, colorContext.colorSpec.withRgba,
                                ChartColorUtils.getColorMeaningInfo(colorContext.colorSpec.dimension, colorContext.chartHandler), colorContext.colorSpec.data.str.sortedMapping, colorContext.theme, colorContext.uaColorIndex);
                            if (colorScale.domain) {
                                colorScale.domain(extent.values.map((v, i) => i));
                            }
                        }
                        break;
                    case 'CUSTOM': {
                        const measures = colorContext.colorSpec.buildGenericMeasures(colorContext.defaultLegendDimension, (colorContext.chartData.getAxisLabels(colorContext.colorSpec.name) || []));
                        colorScale = svc.discreteColorScale(colorContext.colorOptions, measures, colorContext.chartData, colorContext.colorSpec.withRgba,
                            ChartColorUtils.getColorMeaningInfo(colorContext.colorSpec.dimension, colorContext.chartHandler), colorContext.chartData.getAxisLabels(colorContext.colorSpec.name), colorContext.theme, colorContext.uaColorIndex, colorContext.ignoreLabels, colorContext.colorSpec.ignoreColorDim);
                        if (colorScale.domain) {
                            colorScale.domain(measures.map(function(d, i) {
                                return i;
                            }));
                            colorScale.customHandle = colorContext.colorSpec.handle;
                        }
                        break;
                    }
                    default:
                        throw new Error('Unknown scale type: ' + colorContext.colorSpec.type);
                }

                if (colorScale) {
                    colorScale.type = colorContext.colorSpec.type;
                }

                return colorScale;
            },


            /**
             * Create a continuous color scale
             * @param {ChartDef.java} colorOptions
             * @param {number} domainMin
             * @param {number} domainMax
             * @param {array} domainValues: values in the domain (not uniques, this is used to compute quantiles)
             * @param {boolean} noRgba: do not include the opacity setting in the color scale
             * @param {DSSVisualizationTheme} theme
             * @return {*}
             */
            continuousColorScale: function(colorOptions, domainMin, domainMax, domainValues, noRgba, theme, binsToInclude) {

                const isDiverging = colorOptions.paletteType === 'DIVERGING';

                let p = ChartColorUtils.getContinuousPalette(colorOptions.colorPalette, colorOptions.customPalette, isDiverging, theme);

                if (!p) {
                    const defaultDivergingPalette = theme?.palettes.diverging ?? ChartsStaticData.DEFAULT_DIVERGING_PALETTE;
                    const defaultContinuousPalette = theme?.palettes.continuous ?? ChartsStaticData.DEFAULT_CONTINUOUS_PALETTE;
                    const defaultPaletteId = isDiverging ? defaultDivergingPalette : defaultContinuousPalette;
                    colorOptions.colorPalette = defaultPaletteId;
                    p = ChartColorUtils.getContinuousPalette(colorOptions.colorPalette, colorOptions.customPalette, isDiverging, theme);
                }

                // Custom interpolation function to take care of transparency
                function d3_interpolateRgbRound(a, b) {
                    const transparency = !isNaN(colorOptions.transparency) ? colorOptions.transparency : 1;
                    a = d3.rgb(a);
                    b = d3.rgb(b);
                    const ar = a.r,
                        ag = a.g,
                        ab = a.b,
                        br = b.r - ar,
                        bg = b.g - ag,
                        bb = b.b - ab;
                    return function(t) {
                        const tr = Math.round(ar + br * t);
                        const tg = Math.round(ag + bg * t);
                        const tb = Math.round(ab + bb * t);
                        if (!noRgba) {
                            return ['rgba(', tr, ',', tg, ',', tb, ',', transparency, ')'].join('');
                        } else {
                            return ['rgb(', tr, ',', tg, ',', tb, ')'].join('');
                        }
                    };
                }

                if (p.d3Scale) {
                    return p.d3Scale;
                }

                let innerScale;

                if (colorOptions.quantizationMode !== 'QUANTILES') {
                    /*
                     * We use an innerScale to implement the scale computation mode (linear, log, square, sqrt),
                     * that maps the values to a [0, 1] range that will be the input of the actual color scale
                     */

                    if (colorOptions.ccScaleMode == 'LOG') {
                        innerScale = d3.scale.log();
                        domainMin++;
                        domainMax++;
                        innerScale.mode = 'LOG';
                    } else if (colorOptions.ccScaleMode == 'SQRT') {
                        innerScale = d3.scale.sqrt();
                        innerScale.mode = 'SQRT';
                    } else if (colorOptions.ccScaleMode == 'SQUARE') {
                        innerScale = d3.scale.pow().exponent(2);
                        innerScale.mode = 'SQUARE';
                    } else {
                        innerScale = d3.scale.linear();
                        innerScale.mode = 'LINEAR';
                    }
                } else {
                    // No compute mode for quantiles quantization
                    innerScale = d3.scale.linear();
                    innerScale.mode = 'LINEAR';
                }

                switch (colorOptions.paletteType) {
                    case 'DIVERGING':
                        var mid = colorOptions.paletteMiddleValue || 0;
                        if (Math.abs(domainMax - mid) > Math.abs(domainMin - mid)) {
                            innerScale.domain([mid, domainMax]).range([0.5, 1]);
                        } else {
                            innerScale.domain([domainMin, mid]).range([0, 0.5]);
                        }
                        break;
                    case 'CONTINUOUS':
                    default:
                        if (p.fixedValues) {
                            const domain = [],
                                range = [];
                            p.values.forEach(function(value, i) {
                                if (i > p.colors.length - 1) {
                                    return;
                                }
                                if (value == null) {
                                    if (i == 0) {
                                        domain.push(domainMin);
                                        range.push(0);
                                    } else if (i == p.colors.length - 1) {
                                        domain.push(domainMax);
                                        range.push(1);
                                    }
                                } else {
                                    domain.push(value);
                                    range.push(i / (p.colors.length - 1));
                                }
                            });
                            innerScale.domain(domain).range(range);
                        } else {
                            innerScale.domain([domainMin, domainMax]).range([0, 1]);
                        }
                        break;
                }

                let outerScale;

                switch (colorOptions.quantizationMode) {
                    case 'LINEAR':
                    case 'QUANTILES':
                        // Find step colors
                        var numSteps = colorOptions.numQuantizeSteps;
                        var colors = p[numSteps] || p.colors; // Palettes can define special colors for a given number of steps (i.e. colorbrewer palettes)
                        var numColors = colors.length;

                        var linearScale = d3.scale.linear()
                            .domain(Array(numColors).fill().map(function(d, i) {
                                return i / (numColors - 1);
                            }))
                            .range(colors)
                            .interpolate(d3_interpolateRgbRound);
                        var steps = Array(numSteps).fill().map(function(d, i) {
                            return linearScale(i / (numSteps - 1));
                        });

                        if (colorOptions.quantizationMode === 'LINEAR') {
                            outerScale = d3.scale.quantize().domain([0, 1]).range(steps);
                        } else {
                            outerScale = d3.scale.quantile().domain(domainValues.map(innerScale)).range(steps);
                        }
                        break;

                    case 'NONE':
                    default:
                        outerScale = d3.scale.linear()
                            .domain(Array(p.colors.length).fill().map(function(d, i) {
                                return i / (p.colors.length - 1);
                            }))
                            .range(p.colors)
                            .interpolate(d3_interpolateRgbRound);
                        break;

                }

                const ret = function(d, bin) {
                    if (binsToInclude && !binsToInclude.has(bin)) {
                        return undefined;
                    }
                    return outerScale(innerScale(d));
                };

                ret.outerScale = outerScale;
                ret.innerScale = innerScale;
                ret.quantizationMode = colorOptions.quantizationMode;
                ret.diverging = colorOptions.paletteType === 'DIVERGING';

                return ret;
            },

            /**
             * Create a discrete color scale
             * @param colorOptions
             * @param defaultLegendDimension uaDimensionPair for scatters MP, genericMeasures for other charts
             * @param {ChartTensorDataWrapper} chartData
             * @param {boolean} withRgba
             * @param meaningInfo
             * @param colorLabels
             * @param {DSSVisualizationTheme} theme, theme that is applied to the chart
             * @param {number} uaColorIndex (Optional), index of the details column to color a chart
             * @param {Set} ignoreLabels (Optional), labels that should not be included in the colorScale
             * @return {*}
             */
            discreteColorScale: function(colorOptions, defaultLegendDimension, chartData, withRgba, meaningInfo, colorLabels, theme, uaColorIndex, ignoreLabels, ignoreColorDim) {
                if (!colorOptions.colorPalette) {
                    colorOptions.colorPalette = theme?.palettes?.discrete ?? ChartsStaticData.DEFAULT_DISCRETE_PALETTE;
                }
                if (colorOptions.colorPalette === DKU_PALETTE_NAMES.MEANING) {
                    return svc.meaningColorScale(colorOptions, withRgba, meaningInfo, colorLabels);
                }

                let p = ChartColorUtils.getDiscretePalette(colorOptions.colorPalette, colorOptions.customPalette, theme);

                if (!p) {
                    colorOptions.colorPalette = theme?.palettes?.discrete ?? ChartsStaticData.DEFAULT_DISCRETE_PALETTE;
                    p = ChartColorUtils.getDiscretePalette(colorOptions.colorPalette, colorOptions.customPalette, theme);
                }
                if (p.d3Scale) {
                    return p.d3Scale;
                } else {
                    let colorScale = d3.scale.ordinal().range(p.colors);
                    if (colorOptions.customColors && Object.keys(colorOptions.customColors).length) {
                        colorScale = svc.customColorScale(colorOptions, defaultLegendDimension, chartData, colorScale, uaColorIndex, ignoreLabels, ignoreColorDim);
                    }
                    return withRgba ? convertToRgbaColorScale(colorScale, colorOptions.transparency) : colorScale;
                }
            },

            meaningColorScale: function(colorOptions, withRgba, meaningInfo, colorLabels) {
                const normalizer = StringNormalizer.get(meaningInfo.normalizationMode);
                const ret = function(idx) {
                    // TODO fixed fallback color? defined in the chart? in the meaning?
                    if (withRgba) {
                        return ColorUtils.toRgba(meaningInfo.colorMap[normalizer(colorLabels[idx].label)] || 'grey', colorOptions.transparency);
                    } else {
                        return meaningInfo.colorMap[normalizer(colorLabels[idx].label)] || 'grey';
                    }
                };
                ret.domain = function() {
                    return Array.from(Array(colorLabels.length).keys());
                };
                return ret;
            },

            customColorScale: function(colorOptions, dimension, chartData, colorScale, uaColorIndex, ignoreLabels, ignoreColorDim) {
                const colorScaleProxy = index => {
                    const customColors = colorOptions.customColors;
                    const id = ChartColorUtils.getColorId(dimension, chartData, index, uaColorIndex, ignoreLabels, ignoreColorDim && ignoreColorDim(index, dimension));
                    if (id in customColors) {
                        return customColors[id];
                    }
                    return colorScale(index);
                };
                colorScaleProxy.range = colorScale.range;
                colorScaleProxy.domain = colorScale.domain;
                return colorScaleProxy;
            },
            getColor: function(uaColor, color, i, colorScale, colorCache) {
                let cacheKey, rgb;
                if (ChartUADimension.isTrueNumerical(uaColor[0])) {
                    cacheKey = color.num.data[i];
                } else if (ChartUADimension.isDateRange(uaColor[0])) {
                    cacheKey = color.ts.data[i];
                } else {
                    cacheKey = color.str.data[i];
                }

                if (colorCache[cacheKey]) {
                    rgb = colorCache[cacheKey];
                } else {
                    rgb = colorScale(cacheKey);
                    colorCache[cacheKey] = rgb;
                }
                return rgb;
            }
        };

        function convertToRgbaColorScale(colorScale, transparency) {
            const rgbaColorScale = index => {
                const color = colorScale(index);
                return ColorUtils.toRgba(color, transparency);
            };

            rgbaColorScale.range = colorScale.range;
            rgbaColorScale.domain = colorScale.domain;

            return rgbaColorScale;
        }

        /**
         * Create samples for colorpalettes
         */
        function createSamples() {
            $.each(window.dkuColorPalettes.continuous, function(idx, p) {
                if (!p.sample) {
                    const scale = svc.continuousColorScale({ colorPalette: p.id, transparency: 1 }, 0, 100);
                    p.sample = $.map([0, 20, 40, 60, 80, 100], scale);
                    p.sample2 = $.map([0, 10, 20, 30, 40, 50, 60, 70, 80, 90, 100], scale);
                }
            });
            $.each(window.dkuColorPalettes.discrete, function(idx, p) {
                if (!p.sample) {
                    const scale = svc.discreteColorScale({ colorPalette: p.id, transparency: 1 });
                    p.sample = $.map([0, 1, 2, 3, 4], scale);
                }
            });
        }

        createSamples();
        return svc;
    }
})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.charts');

    // (!) This service previously was in static/dataiku/js/simple_report/config_ui.js
    app.factory('ChartColorSelection', function(ChartCustomColors, ChartColorUtils, CHART_TYPES) {
        const customColorsOptions = {
            id: null,
            colorOptions$: null,
            listeners: 0
        };

        const geoLayerCustomColorsOptions = {};

        //TODO: outdated - not needed
        function getGeomMapUaColor(chartDef, geoLayer) {
            if (chartDef.geoLayers.length < 3 && geoLayer === undefined) {
                // if the user selects the color from the left section, retrieve the ua color of the first geo
                return chartDef.geoLayers[0].uaColor;
            } else {
                return geoLayer.uaColor;
            }
        }

        return {
            getOptions: function(chartDef, geoLayer) {
                if (chartDef.type !== CHART_TYPES.GEOMETRY_MAP) {
                    return chartDef.colorOptions;
                } else if (chartDef.geoLayers.length < 3 && geoLayer === undefined) {
                    // if the user selects the color from the left section, retrieve the color options of the first geo
                    return chartDef.geoLayers[0].colorOptions;
                } else {
                    return geoLayer.colorOptions;
                }
            },

            setOptions: function(colorOptions, chartDef, geoLayer) {
                if (chartDef.type !== CHART_TYPES.GEOMETRY_MAP) {
                    chartDef.colorOptions = colorOptions;
                } else if (chartDef.geoLayers.length < 3 && geoLayer === undefined) {
                    // if the user selects the color from the left section, retrieve the color options of the first geo
                    chartDef.geoLayers[0].colorOptions = colorOptions;
                } else {
                    geoLayer.colorOptions = colorOptions;
                }
            },

            getUaColor: function(chartDef, geoLayer) {
                if (chartDef.type !== CHART_TYPES.GEOMETRY_MAP) {
                    return chartDef.uaColor;
                } else {
                    return getGeomMapUaColor(chartDef, geoLayer);
                }
            },

            getColorMeasure: function(chartDef) {
                return chartDef.colorMeasure;
            },

            getColorDimensionOrMeasure: function(chartDef, geoLayer) {
                if (chartDef.type !== CHART_TYPES.GEOMETRY_MAP) {
                    return ChartColorUtils.getColorDimensionOrMeasure(chartDef);
                } else {
                    return getGeomMapUaColor(chartDef, geoLayer)[0];
                }
            },

            getOrCreateCustomColorsOptions(geoLayerId) {
                if (geoLayerId) {
                    if (_.isNil(geoLayerCustomColorsOptions[geoLayerId])) {
                        const { colorOptions$ } = ChartCustomColors.createColorOptions();
                        geoLayerCustomColorsOptions[geoLayerId] = colorOptions$;
                    }

                    return { id: geoLayerId, colorOptions$: geoLayerCustomColorsOptions[geoLayerId] };
                } else {
                    if (_.isNil(customColorsOptions.id)) {
                        const { id, colorOptions$ } = ChartCustomColors.createColorOptions();
                        customColorsOptions.id = id;
                        customColorsOptions.colorOptions$ = colorOptions$;
                    }

                    customColorsOptions.listeners++;

                    return customColorsOptions;
                }
            },

            removeCustomColorsOptions(ids) {
                (ids || []).forEach(id => {
                    if (id.startsWith('geo')) {
                        delete customColorsOptions[id]; // this is prolly wrong, it should be geoLayerCustomColorsOptions[id]
                        ChartCustomColors.removeColorOptions([id]);
                    } else if (customColorsOptions.id === id) {
                        customColorsOptions.listeners--;
                        if (customColorsOptions.listeners === 0) {
                            customColorsOptions.id = null;
                            customColorsOptions.colorOptions$ = null;
                            ChartCustomColors.removeColorOptions([id]);
                        }
                    } else {
                        throw new Error('Ids not found');
                    }
                });
            }
        };
    });
})();

;
(function() {
    'use strict';

    angular.module('dataiku.charts')
        .service('ChartContextualMenu', ChartContextualMenu);

    function ChartContextualMenu($rootScope, $stateParams, ChartFilters, ContextualMenu, AnimatedChartsUtils, DashboardFilters, DashboardUtils, ChartDrilldown, ChartFeatures, ChartHierarchyDimension, ChartDimension, ChartDrilldownType, translate, DRILL_UP_SOURCE, DRILL_DOWN_SOURCE) {

        const svc = {
            create: function(chartData, chartDef, chartStore, animation = {}) {
                const isInDashboard = DashboardUtils.isInDashboard();
                const contextualMenu = {
                    open: function({ event, filteringOptions = { filterableElements: [] }, hierarchyDrillableElements = [], nativeDrillableElements = [], customActions = [], type = 'measure' }) {
                        let canFilter, canDrillHierarchy, canDrillNative, menu;

                        const newScope = $rootScope.$new();
                        newScope.customActions = customActions;
                        newScope.ChartDrilldownType = ChartDrilldownType;

                        switch (type) {
                            case 'measure':
                                canFilter = (!isInDashboard || DashboardFilters.canCrossFilter($stateParams.pageId)) && filteringOptions.filterableElements.length > 0;
                                canDrillHierarchy = ChartFeatures.canDrillHierarchy(chartDef.type) && !!hierarchyDrillableElements.length && hierarchyDrillableElements.some(el => el.drilldown || el.drillup);
                                canDrillNative = !isInDashboard && !!nativeDrillableElements.length && nativeDrillableElements.some(el => el.elements.length);

                                if (canFilter || canDrillHierarchy || canDrillNative || customActions.length) {
                                    menu = new ContextualMenu({
                                        contextual: false,
                                        template: '/templates/simple_report/contextual-menu/chart-contextual-menu.html'
                                    });
                                    const data = {
                                        canFilter,
                                        canHierarchyDrillDown: canDrillHierarchy && hierarchyDrillableElements.some(el => el.drilldown),
                                        canHierarchyDrillUp: canDrillHierarchy && hierarchyDrillableElements.some(el => el.drillup),
                                        disableMultiDimensionalFiltering: filteringOptions.disableMultiDimensionalFiltering,
                                        filterableElements: filteringOptions.filterableElements,
                                        hierarchyDrillupElements: hierarchyDrillableElements.filter(el => el.drillup),
                                        hierarchyDrilldownElements: hierarchyDrillableElements.filter(el => el.drilldown),
                                        nativeDrillableElements,
                                        canDrillNative,
                                        drillDownHierarchy: contextualMenu.drillDownHierarchy,
                                        drillUpHierarchy: contextualMenu.drillUpHierarchy,
                                        drillDownNative: contextualMenu.drillDownNative,
                                        includeOnly: contextualMenu.includeOnly,
                                        exclude: contextualMenu.exclude,
                                        getHierarchyName: contextualMenu.getHierarchyName
                                    };
                                    newScope.data = angular.copy(data);
                                }

                                break;

                            case 'pivotTableHeader':
                                menu = new ContextualMenu({
                                    contextual: false,
                                    template: '/templates/simple_report/contextual-menu/pivot-table-header-contextual-menu.html'
                                });
                                break;
                            case 'background':
                                menu = new ContextualMenu({
                                    contextual: false,
                                    template: '/templates/simple_report/contextual-menu/chart-contextual-menu.html'
                                });
                                break;
                        }
                        if (!menu) {
                            return;
                        }
                        menu.scope = newScope;
                        menu.openAtXY(event.pageX, event.pageY);
                        menu.$menu.addClass('chart-contextual-menu');
                    },
                    includeOnly: (filterableElements) => {
                        if (isInDashboard) {
                            const filters = filterableElements.map((filterableElement) => {
                                const { dimension, axisElt, value } = filterableElement;
                                return ChartFilters.createFilter({
                                    column: dimension.column,
                                    columnType: dimension.type,
                                    isAGlobalFilter: true,
                                    excludeOtherValues: true,
                                    label: `ONLY ${dimension.column}: ${value}`,
                                    useMinimalUi: true,
                                    includeEmptyValues: false
                                }, {
                                    dimension,
                                    axisElt
                                });
                            });
                            $rootScope.$emit('crossFiltersAdded', {
                                filters,
                                wt1Args: {
                                    from: 'chart',
                                    action: 'contextual_menu',
                                    chartType: chartDef.type,
                                    filterType: 'include'
                                }
                            });
                        } else {
                            let filters = chartDef.filters;
                            filterableElements.forEach((filterableElement) => {
                                const { dimension, axisElt } = filterableElement;
                                const { oldFilter, newFilter } = ChartFilters.getIncludeOnly1DFilter(dimension, axisElt, filters);
                                if (!ChartFilters.areFiltersEqual(oldFilter, newFilter)) {
                                    filters = ChartFilters.updateOrAddFilter(filters, newFilter, oldFilter);
                                }
                            });
                            chartDef.filters = filters;
                        }
                    },
                    exclude: (filterableElements) => {
                        const filter = ChartFilters.getExcludeNDFilter(filterableElements);
                        if (isInDashboard) {
                            filter.isAGlobalFilter = true;
                            $rootScope.$emit('crossFiltersAdded', {
                                filters: [filter],
                                wt1Args: {
                                    from: 'chart',
                                    action: 'contextual_menu',
                                    chartType: chartDef.type,
                                    filterType: 'exclude'
                                }
                            });
                        } else if (!chartDef.filters.some(f => ChartFilters.areFiltersEqual(f, filter))){
                            chartDef.filters = ChartFilters.updateOrAddFilter(chartDef.filters, filter);
                        }
                    },
                    drillDownHierarchy: (drillableElement) => {
                        ChartDrilldown.handleDrillDown(chartDef, drillableElement.dimension, DRILL_DOWN_SOURCE.CONTEXTUAL_MENU, drillableElement.axisElt);
                    },
                    drillUpHierarchy: (drillableElement) => {
                        ChartDrilldown.handleDrillUp(chartDef, drillableElement.dimension, DRILL_UP_SOURCE.CONTEXTUAL_MENU);
                    },
                    drillDownNative: (drillableElement) => {
                        const { updatedFilters, newDateRange } = ChartDrilldown.getUpdateForNativeDrillDown(chartDef, drillableElement.axisElt, drillableElement.dimension);
                        if (newDateRange) {
                            _.set(chartDef, `${drillableElement.propertyName}.dateParams.mode`, newDateRange);
                        }
                        chartDef.filters = updatedFilters;
                    },
                    showForCoords: (coords, event, customActions, type) => {
                        const finalCoords = {
                            ...coords,
                            animation: AnimatedChartsUtils.getAnimationCoord(animation)
                        };
                        const filteringOptions = ChartFilters.getFilterableOptions(chartStore, chartData, finalCoords, chartDef);
                        const hierarchyDrillableElements = ChartDrilldown.getDrillableElements(chartStore, chartDef, chartData, finalCoords);
                        const binDrillableElements = filteringOptions.filterableElements.filter(el => ChartDimension.isDrillableForElement(chartDef, el.dimension, el.axisElt) && ChartDimension.isGroupedNumerical(el.dimension));
                        const dateDrillableElements = filteringOptions.filterableElements.filter(el => ChartDimension.isDrillableForElement(chartDef, el.dimension, el.axisElt) && !ChartDimension.isGroupedNumerical(el.dimension));
                        const nativeDrillableElements = [
                            {
                                drilldownType: ChartDrilldownType.BIN,
                                actionLabel: translate('CHARTS.CONTEXTUAL_MENU.DRILL_DOWN.BIN', 'Drill down on bin'),
                                elements: binDrillableElements
                            },
                            {
                                drilldownType: ChartDrilldownType.DATE,
                                actionLabel: translate('CHARTS.CONTEXTUAL_MENU.DRILL_DOWN.DATE', 'Drill down on date'),
                                elements: dateDrillableElements
                            }
                        ];

                        contextualMenu.open({ event, filteringOptions, hierarchyDrillableElements, nativeDrillableElements, customActions, type });
                    },
                    addContextualMenuHandler: (domElement, coords, customActions, type) => {
                        d3.select(domElement)
                            .on('contextmenu.contextualmenu', () => {
                                d3.event.preventDefault();
                                contextualMenu.showForCoords(coords, d3.event, customActions, type);
                            });
                    },
                    removeContextualMenuHandler: (el) => {
                        d3.select(el)
                            .on('contextmenu', null);
                    },
                    addChartContextualMenuHandler: (chart, getTargetElementProperties, getCustomActions) => {
                        d3.select(chart).on('contextmenu.contextualmenu', function() {
                            const target = d3.event.target;
                            const properties = getTargetElementProperties && getTargetElementProperties(target);
                            const customActions = getCustomActions ? getCustomActions(properties) : [];
                            if (!_.isNil(properties)) {
                                d3.event.preventDefault();
                                contextualMenu.showForCoords(properties.coords, d3.event, customActions);
                            } else if (ChartFeatures.canDrillHierarchy(chartDef.type) && customActions.length && target.classList.contains('chart-contextual-menu-anchor')) {
                                d3.event.preventDefault();
                                contextualMenu.open({ event: d3.event, customActions, type: 'background' });
                            }
                        });
                    },
                    removeChartContextualMenuHandler: (chart) => {
                        d3.select(chart).on('contextmenu', null);
                    },
                    getAnimationContext: function() {
                        return AnimatedChartsUtils.getAnimationContext(chartDef.animationDimension[0], animation);
                    },
                    getHierarchyName: function(dimension) {
                        return ChartHierarchyDimension.getDimensionHierarchyName(chartDef, dimension);
                    }
                };
                return contextualMenu;
            }
        };
        return svc;
    }
})();

;
(function() {
    'use strict';

    angular.module('dataiku.charts').service('D3ChartAxes', D3ChartAxes);

    const SCALE_TYPES = {
        ORDINAL: 'ORDINAL',
        LINEAR: 'LINEAR',
        LOG: 'LOG',
        TIME: 'TIME'
    };

    /**
     * A set of helpers to create and enhance d3 svg axes
     */
    function D3ChartAxes(d3Utils, CHART_AXIS_TYPES, CHART_TYPES, CHART_MODES, ChartDimension, ChartUADimension, ChartAxesUtils, AxisTicksConfigMode, ChartFeatures, ChartFormatting, ChartsStaticData, AxisTicksConfiguration, ChartYAxisPosition, ChartFormattingPaneSections, GridlinesAxisType, ChartDataUtils, ChartLabels, ChartMeasure, CHART_VARIANTS, ValuesInChartPlacementMode) {

        const addFormatterToBinnedAxis = (axis, tickExtents, formattingOptions) => {
            axis.tickFormat(ChartFormatting.getForBinnedAxis(tickExtents, formattingOptions));
        };

        const addFormatterToCustomBinnedAxis = (axis, tickExtents, formattingOptions) => {
            axis.tickFormat(ChartFormatting.getForCustomBinnedAxis(tickExtents, formattingOptions));
        };

        const addFormatterToOrdinalAxis = (axis, tickExtents, formattingOptions) => {
            axis.tickFormat((_d, i) => {
                const value = tickExtents[i];
                return ChartFormatting.getForOrdinalAxis(value, formattingOptions);
            });
        };

        const addFormatterToPercentageAxis = (axis, formattingOptions = {}) => {
            const scale = (axis.scale() instanceof Function) ? axis.scale() : axis.scale;
            const minValue = (Math.min(...scale.domain()) || 0) * 100;
            const maxValue = (Math.max(...scale.domain()) || 0) * 100;
            const numValues = axis.tickValues() ? axis.tickValues().length : axis.ticks()[0];
            axis.tickFormat(ChartFormatting.getForAxis(minValue, maxValue, numValues, formattingOptions, true));
        };

        // filtering out the tick that overlaps with the axis (if present)
        const getTicksForGridlines = (axisG, axisToCheck, chartG) => {
            const ticks = axisG.selectAll('g.tick')[0];
            return ticks && ticks.filter(tick => {
                const tickTranslate = tick.getAttribute('transform');
                const tickTranslateValue = tickTranslate.match(/translate\(([^,]+),([^,]+)\)/);
                if (axisToCheck === 'x') {
                    const axisG = chartG.select('g.x.axis')[0][0];
                    const axisTranslate = axisG && axisG.getAttribute('transform');
                    //no transform if we have all negative values
                    const axisTranslateValue = axisTranslate ? axisTranslate.match(/translate\(([^,]+),([^,]+)\)/) : '0';
                    return tickTranslateValue[2] !== axisTranslateValue[2];
                } else if (axisToCheck === 'y') {
                    //the y axis always has x translate of 0
                    return tickTranslateValue[1] !== '0';
                }
            });
        };

        const adjustAxisPadding = function(axis, extent, axisSize, valuesFontSize, addPaddingToStart, addPaddingToEnd) {
            if (!addPaddingToStart && !addPaddingToEnd) {
                return;
            }

            const diff = extent[1] - extent[0];
            const domainPadding = valuesFontSize / ((axisSize - valuesFontSize) / diff);

            const paddedDomain = [
                extent[0] - (addPaddingToStart ? domainPadding : 0),
                extent[1] + (addPaddingToEnd ? domainPadding : 0)
            ];

            axis.scale().domain(paddedDomain);
        };

        const computeTextHeightFromFontSize = (svgs) => (fontSize) => {
            let result = fontSize;
            const textElement = d3.select(svgs.get(0))
                .append('text')
                .attr('class', 'measureInChartTempText')
                .attr('opacity', 0)
                .attr('font-size', `${fontSize}px`)
                .text('0');
            if (textElement) {
                const textHeight = textElement.node()?.getBBox().height;
                textElement.remove();
                if (textHeight) {
                    result = textHeight;
                }
            }
            return result;
        };

        const svc = {

            scaleTypes: Object.freeze(SCALE_TYPES),

            /**
             * Create a svg axis for the given axisSpec
             * @param {ChartTensorDataWrapper}     chartData
             * @param {AxisSpec}                   axisSpec
             * @param {boolean}                    isPercentScale
             * @param {boolean }                   isLogScale
             * @param {boolean}                    includeZero          force inclusion of zero in domain
             * @param {NumberFormattingOptions}    formattingOptions
             * @param {ChartDef.java}              chartDef
             * @returns {d3 axis}
             */
            createAxis: function(chartData, axisSpec, isPercentScale, isLogScale, includeZero, formattingOptions, chartDef, ignoreLabels, otherAxesIndexes = []) {
                if (!axisSpec || !formattingOptions) {
                    return null;
                }

                let axis;
                switch (axisSpec.type) {
                    case CHART_AXIS_TYPES.DIMENSION:
                    case CHART_AXIS_TYPES.UNAGGREGATED:
                        axis = svc.createDimensionAxis(chartData, axisSpec, isLogScale, includeZero, formattingOptions, chartDef, ignoreLabels);
                        break;
                    case CHART_AXIS_TYPES.MEASURE:
                        axis = svc.createMeasureAxis(chartData, axisSpec, isPercentScale, isLogScale, includeZero, formattingOptions, otherAxesIndexes);
                        break;
                    default:
                        throw new Error('Unknown axis type: ' + axisSpec.type);
                }

                if (axis) {
                    axis.type = axisSpec.type;
                    axis.position = axisSpec.position;
                    axis.id = axisSpec.id;
                }

                return axis;
            },


            /**
             * Create a svg axis for a UNAGGREGATED / DIMENSION column
             * @param {ChartTensorDataWrapper}     chartData
             * @param {AxisSpec}                   axisSpec
             * @param {Boolean}                    isLogScale
             * @param {Boolean}                    includeZero
             * @param {NumberFormattingOptions}    formattingOptions
             * @param {ChartDef.java}              chartDef
             * @returns {d3 axis}
             */
            createDimensionAxis: function(chartData, axisSpec, isLogScale, includeZero, formattingOptions, chartDef, ignoreLabels) {
                const finalFormattingOptions = { ...formattingOptions, ...chartData?.getSpecialValuesLabelsOverrides?.() };
                const isNumerical = ChartAxesUtils.isNumerical(axisSpec);
                let extent = ChartAxesUtils.getDimensionExtent(chartData, axisSpec, ignoreLabels);

                if (!isLogScale && !ChartAxesUtils.isManualMode(axisSpec.customExtent)) {
                    extent = ChartAxesUtils.fixUnbinnedNumericalExtent(axisSpec, extent);
                }

                if (includeZero && isNumerical) {
                    extent = ChartAxesUtils.includeZero(extent);
                }

                const linearScale = d3.scale.linear().domain([extent.min, extent.max]);
                const ordinalScale = d3.scale.ordinal().domain(extent.values.map(function(d, i) {
                    return i;
                }));
                let timeScale;
                let logScale;
                let axis;

                if (isLogScale && isNumerical) {
                    extent = ChartAxesUtils.fixNumericalLogScaleExtent(axisSpec, extent, includeZero);
                    logScale = d3.scale.log().domain([extent.min, extent.max]);
                }

                const createNumericAxis = () => {
                    let axis;
                    if (logScale) {
                        axis = d3.svg.axis().scale(logScale);
                        axis.scaleType = SCALE_TYPES.LOG;
                        svc.setLogAxisTicks(axis, extent.min, extent.max);
                    } else {
                        axis = d3.svg.axis().scale(linearScale);
                        axis.scaleType = SCALE_TYPES.LINEAR;
                    }
                    svc.addNumberFormatterToAxis(axis, finalFormattingOptions);
                    return axis;
                };

                const createOrdinalAxis = (isGroupedNumerical = false, isCustomBinning = false) => {
                    const axis = d3.svg.axis().scale(ordinalScale);
                    if (isCustomBinning) {
                        addFormatterToCustomBinnedAxis(axis, extent.values, finalFormattingOptions);
                    } else if (isGroupedNumerical) {
                        addFormatterToBinnedAxis(axis, extent.values, finalFormattingOptions);
                    } else {
                        addFormatterToOrdinalAxis(axis, extent.values, finalFormattingOptions);
                    }
                    axis.scaleType = SCALE_TYPES.ORDINAL;
                    return axis;
                };

                const createTimeAxis = () => {
                    timeScale = d3.time.scale.utc().domain([extent.min, extent.max]);
                    const timeAxis = d3.svg.axis().scale(timeScale);
                    timeAxis.scaleType = SCALE_TYPES.TIME;
                    return timeAxis;
                };

                if (chartDef.type === CHART_TYPES.BOXPLOTS) {
                    axis = createOrdinalAxis(ChartDimension.isGroupedNumerical(axisSpec.dimension));
                } else {
                    // Choose which scale to display on the axis
                    if (ChartDimension.isTimeline(axisSpec.dimension) || (axisSpec.type === CHART_AXIS_TYPES.UNAGGREGATED && ChartUADimension.isDate(axisSpec.dimension))) {
                        // In case we have one value, don't create timescale but use the linearScale else we start at the beginning and not on the middle of the axis
                        axis = (extent.min !== extent.max ? createTimeAxis() : createNumericAxis()).tickFormat((d) => ChartFormatting.getForDate()(d));
                    } else if (ChartDimension.isUngroupedNumerical(axisSpec.dimension)) {
                        axis = createNumericAxis();
                    } else if (ChartDimension.isGroupedNumerical(axisSpec.dimension) || (axisSpec.type === CHART_AXIS_TYPES.UNAGGREGATED && ChartUADimension.isTrueNumerical(axisSpec.dimension))) {
                        if (ChartDimension.hasOneTickPerBin(axisSpec.dimension) && ChartFeatures.chartSupportOneTickPerBin(chartDef)) {
                            axis = createOrdinalAxis(true);
                        } else {
                            axis = createNumericAxis();
                        }
                    } else {
                        axis = createOrdinalAxis();
                    }
                }

                axis.ordinalScale = ordinalScale;
                axis.linearScale = linearScale;

                axis.setScaleRange = function(range) {
                    (logScale || timeScale || linearScale).range(range);

                    const numColumns = ordinalScale.domain().length;
                    const step = range[range.length - 1] / numColumns;
                    const padding = svc.getColumnPadding(numColumns);
                    const axisPadding = axisSpec.padding ?? 0.5;

                    switch (axisSpec.mode) {
                        case CHART_MODES.POINTS:
                            if (step >= 1) {
                                ordinalScale.rangeRoundPoints(range, axisPadding, 0);
                            } else {
                                // 'rangeRoundPoints` can't be used when band widths are not large enough to prevent rounding to zero
                                ordinalScale.rangePoints(range, axisPadding, 0);
                            }
                            break;
                        case CHART_MODES.COLUMNS: {
                            if (step >= 1) {
                                ordinalScale.rangeRoundBands(range, padding, padding / 2);
                            } else {
                                // 'rangeRoundBands` can't be used when band widths are not large enough to prevent rounding to zero
                                ordinalScale.rangeBands(range, padding, padding / 2);
                            }
                            break;
                        }
                        default:
                            throw new Error('Unknown scale type: ' + axisSpec.mode);
                    }
                    return axis;
                };

                svc.fixUpAxis(axis);

                axis.dimension = axisSpec.dimension;

                return axis;
            },


            /**
             * Returns the padding between columns based on the number of columns
             * @param numColumns
             * @returns {number}
             */
            getColumnPadding: function(numColumns) {
                if (numColumns > 20) {
                    return 0.1;
                } else {
                    return 0.45 - numColumns / 20 * 0.35;
                }
            },

            /**
             * Create a svg axis for a MEASURE axisSpec
             *
             * @param {ChartTensorDataWrapper}                                                                                               chartData
             * @param {AxisSpec}                                                                                                             axisSpec
             * @param {boolean}                                                                                                              isPercentScale
             * @param {boolean}                                                                                                              isLogScale
             * @param {boolean}                                                                                                              includeZero
             * @param {{multiplier: keyof ChartsStaticData.availableMultipliers, decimalPlaces: number, prefix: string, suffix: string }}    formattingOptions
             * @returns {d3 axis}
             */
            createMeasureAxis: function(chartData, axisSpec, isPercentScale, isLogScale, includeZero, formattingOptions, otherAxesIndexes = []) {
                axisSpec.extent = ChartAxesUtils.getMeasureExtent(chartData, axisSpec, isLogScale, includeZero, true, otherAxesIndexes);

                if (axisSpec.extent === null) {
                    return null;
                }

                const scale = isLogScale ? d3.scale.log() : d3.scale.linear(),
                    axis = d3.svg.axis().scale(scale).orient('left');

                scale.domain(axisSpec.extent);

                if (isPercentScale || axisSpec.isPercentScale) {
                    addFormatterToPercentageAxis(axis, formattingOptions);
                } else {
                    svc.addNumberFormatterToAxis(axis, formattingOptions);
                }

                if (isLogScale) {
                    svc.setLogAxisTicks(axis, axisSpec.extent[0], axisSpec.extent[1]);
                }

                axis.setScaleRange = function(range) {
                    scale.range(range);
                };

                svc.fixUpAxis(axis);

                axis.measure = axisSpec.measure;
                return axis;
            },

            /**
             * Add hand-picked axis ticks for log scales
             * @param {d3 axis} axis
             * @param {number} minVal: the minimum value of the axis
             * @param {number} maxVal: the maximum value of the axis
             */
            setLogAxisTicks: function(axis, minVal, maxVal) {
                const maxValLog = Math.floor(log10(maxVal));
                const minValLog = Math.ceil(log10(minVal));
                const arr = [];
                for (let i = minValLog; i <= maxValLog; i++) {
                    arr.push(Math.pow(10, i));
                }
                axis.tickValues(arr);
            },

            /**
             * Algorithm finding overlapping and having an identical label ticks. Guaranteeing to keep the first & last tick of the sorted tick list
             * The first pass identifies the overlapping ticks from left to right, excluding the last tick.
             * The second and last pass removes the ticks overlapping the last tick.
             * @param {Array} Array of sorted ticks
             */
            sanitizeTicksDisplay: function(sortedTicks, forceLastTickDisplay = false, onlyRemoveLabel = false) {
                // Remove overlapping labels from left to right (most left and right values excluded)
                let left = 0;
                let right = 1;
                const overlappingTicks = [];
                const length = forceLastTickDisplay ? sortedTicks.length : sortedTicks.length - 1;

                while (right < length) {
                    if (sortedTicks[left].textContent === sortedTicks[right].textContent
                        || areBboxOverlapping(sortedTicks[left].getBoundingClientRect(), sortedTicks[right].getBoundingClientRect())) {
                        overlappingTicks.push(sortedTicks[right]);
                        right++;
                    } else {
                        left = right++;
                    }
                }

                // Check rightmost label overlapping with its left neighbors
                while (!forceLastTickDisplay && left >= 0 && areBboxOverlapping(sortedTicks[left].getBoundingClientRect(), sortedTicks[sortedTicks.length - 1].getBoundingClientRect())) {
                    overlappingTicks.push(sortedTicks[left--]);
                }

                overlappingTicks.forEach(tick => {
                    if (onlyRemoveLabel) {
                        const labelDom = tick.querySelector('text');
                        labelDom && labelDom.remove();
                    } else {
                        tick.remove();
                    }
                });

                return overlappingTicks;
            },

            /**
             * Extra treatment on d3 axis to handle some edge cases
             * @param {d3.axis} axis
             */
            fixUpAxis: function(axis) {
                const scale = axis.scale();

                // d3 axis and scales don't really behave well on empty domains
                if (scale.domain().length == 2 && scale.domain()[0] == scale.domain()[1]) {
                    axis.tickValues([scale.domain()[0]]);
                    scale.domain([scale.domain()[0] - 1, scale.domain()[0] + 1]);
                }
            },

            /**
             * Adjust the bottom margin to make room for the x-axis, and the angle of the tick labels if they need to rotate
             * @param {{top: number, bottom: number, right: number, left: number}} margins
             * @param {jQuery selection} $svg
             * @param {d3 axis} xAxis
             * @param {number} forceRotation
             * @returns {{top: number, bottom: number, right: number, left: number}} the updated margins object
             */
            adjustBottomMargin: function(axisTicksFormatting, margins, $svg, xAxis, forceRotation = 0) {
                const chartHeight = $svg.height(),
                    chartWidth = $svg.width(),
                    svg = d3.select($svg.get(0));

                let labels, usedBand;

                if (xAxis.type === CHART_AXIS_TYPES.MEASURE || xAxis.scaleType === SCALE_TYPES.LINEAR || xAxis.scaleType === SCALE_TYPES.TIME) {
                    const ticks = xAxis.tickValues() || xAxis.scale().ticks();
                    labels = xAxis.tickFormat() ? ticks.map(xAxis.tickFormat()) : ticks;
                    usedBand = (chartWidth - margins.left - margins.right) / (labels.length + 1);
                } else {
                    const ticks = xAxis.tickValues() || xAxis.ordinalScale.domain();
                    labels = xAxis.tickFormat() ? ticks.map(xAxis.tickFormat()) : ticks;
                    if (xAxis.ordinalScale.rangeBand() > 0) {
                        usedBand = xAxis.ordinalScale.rangeBand();
                    } else {
                        usedBand = (chartWidth - margins.left - margins.right) / (labels.length + 1);
                    }
                }

                if (labels.length == 0) { // Nothing to do ...
                    return margins;
                }

                svg.selectAll('.tempText').data(labels)
                    .enter()
                    .append('text').attr('class', 'tempText')
                    .text(function(d) {
                        return ChartFormatting.getForOrdinalAxis(d);
                    });


                const maxLabelWidth = d3.max(svg.selectAll('.tempText')[0].map(function(itm) {
                    return itm.getBoundingClientRect().width;
                }));

                const labelHeight = axisTicksFormatting.fontSize;
                const hasLongLabels = !svg.selectAll('.tempText').filter(function() {
                    return this.getBoundingClientRect().width > usedBand;
                }).empty();

                svg.selectAll('.tempText').remove();

                if (forceRotation) {
                    xAxis.labelAngle = forceRotation;
                } else {
                    xAxis.labelAngle = hasLongLabels ? Math.atan((labelHeight * 2) / usedBand) : 0;
                }

                if (xAxis.labelAngle > Math.PI / 3) {
                    xAxis.labelAngle = Math.PI / 2;
                }

                // Prevent the xAxis from taking more than a quarter of the height of the chart
                margins.bottom = Math.min(chartHeight / 4, margins.bottom + Math.sin(xAxis.labelAngle) * maxLabelWidth + Math.cos(xAxis.labelAngle) * labelHeight);

                return margins;
            },


            /**
             * Get the chart's horizontal margins leaving room for the y-axis labels
             * @param {{top: number, bottom: number, right: number, left: number}} margins
             * @param {jQuery selection} $svg
             * @param {ChartDef.java} chartDef
             * @param {array of d3 axis} yAxes
             * @returns {{top: number, bottom: number, right: number, left: number}} the margins object
             */
            getHorizontalMarginsAndAxesWidth: function(margins, $svg, chartDef, yAxes) {
                const axesWidth = [];
                margins[ChartYAxisPosition.LEFT] = ChartsStaticData.CHART_BASE_MARGIN;
                margins[ChartYAxisPosition.RIGHT] = ChartsStaticData.CHART_BASE_MARGIN;
                yAxes.forEach(axis => {
                    const tempMargins = {
                        [ChartYAxisPosition.LEFT]: 0,
                        [ChartYAxisPosition.RIGHT]: 0
                    };
                    const side = axis.position;
                    const axisFormatting = ChartAxesUtils.getFormattingForYAxis(chartDef.yAxesFormatting, axis.id);
                    if (axis && axisFormatting.displayAxis) {
                        let labels;
                        if (axis.type === CHART_AXIS_TYPES.MEASURE || axis.scaleType === SCALE_TYPES.LINEAR || axis.scaleType === SCALE_TYPES.TIME) {
                            labels = axis.scale().ticks();
                        } else {
                            labels = axis.ordinalScale.domain();
                        }

                        // LOG scale or Number of ticks feature set.
                        const manualTicks = axis.tickValues();
                        if (manualTicks) {
                            labels = manualTicks;
                        }

                        labels = axis.tickFormat() ? labels.map(axis.tickFormat()) : labels;
                        const svg = d3.select($svg.get(0));

                        svg.selectAll('.tempText').data(labels).enter()
                            .append('text').attr('class', 'tempText')
                            .attr('font-size', `${axisFormatting.axisValuesFormatting.axisTicksFormatting.fontSize}px`)
                            .text(function(d) {
                                return d;
                            });

                        const maxLabelWidth = d3.max(svg.selectAll('.tempText')[0].map(function(itm) {
                            return itm.getBoundingClientRect().width;
                        })) || 0;

                        tempMargins[side] += maxLabelWidth + ChartsStaticData.AXIS_WIDTH;
                        svg.selectAll('.tempText').remove();
                    }

                    const canHaveAxisTitle = ((!_.isNil(axisFormatting.axisTitle) && axisFormatting.axisTitle.length > 0)
                    || (axis && (axis.dimension !== undefined || axis.measure !== undefined))
                    || ChartAxesUtils.getMeasuresDisplayedOnAxis(axis.position, chartDef.genericMeasures).length >= 1);

                    if (canHaveAxisTitle && axisFormatting.showAxisTitle) {
                        tempMargins[side] += ChartsStaticData.AXIS_MARGIN + axisFormatting.axisTitleFormatting.fontSize;
                    }

                    margins.left += tempMargins.left;
                    margins.right += tempMargins.right;
                    axesWidth.push({ id: axis.id, width: tempMargins[side] });
                });

                const leftYAxesNb = yAxes.filter(axis => ChartAxesUtils.isLeftYAxis(axis)).length;
                if (leftYAxesNb > 1) {
                    // Adding 16px spacing between each y axis
                    margins.left += (leftYAxesNb - 1) * ChartsStaticData.AXIS_Y_SPACING;
                }

                return { updatedMargins: margins, axesWidth };
            },

            getMarginsAndAxesWidth: function(chartHandler, chartDef, $svg, xAxis, yAxes) {
                let margins = { top: ChartsStaticData.CHART_BASE_MARGIN, bottom: ChartsStaticData.CHART_BASE_MARGIN };
                let yAxesWidth;
                if (!chartHandler.noXAxis && ChartFeatures.canDisplayAxes(chartDef.type)) {
                    if (chartDef.xAxisFormatting.displayAxis) {
                        margins.bottom += ChartsStaticData.AXIS_WIDTH;
                    }

                    const canHaveAxisTitle = ChartFeatures.canShowXAxisTitle(chartDef) && ((!_.isNil(chartDef.xAxisFormatting.axisTitle) && chartDef.xAxisFormatting.axisTitle.length > 0)
                        || (xAxis && (xAxis.dimension !== undefined || xAxis.measure !== undefined))
                        || ChartDimension.getGenericDimensions(chartDef).length === 1);

                    if (canHaveAxisTitle && chartDef.xAxisFormatting.showAxisTitle) {
                        margins.bottom += ChartsStaticData.AXIS_MARGIN + chartDef.xAxisFormatting.axisTitleFormatting.fontSize;
                    }
                }

                if (chartHandler.noYAxis) {
                    margins.left = ChartsStaticData.CHART_BASE_MARGIN;
                    margins.right = ChartsStaticData.CHART_BASE_MARGIN;
                } else {
                    const { updatedMargins, axesWidth } = svc.getHorizontalMarginsAndAxesWidth(margins, $svg, chartDef, yAxes);
                    margins = updatedMargins;
                    yAxesWidth = axesWidth;
                }

                return { margins, yAxesWidth };
            },

            /**
             * - Align 0 on both axis if there is 2 axes with negatives and positives
             * - Set the correct number of ticks to 1 or both axes if ticksConfig.number is set.
             */
            alignAndSetYTickValues: function(axes, axisOptions, yAxesFormatting) {
                const axesMins = [];
                const availableAxesMins = [];
                const axesMaxes = [];
                const availableAxesMaxes = [];
                const axesRes = [];

                const isAxisCompatible = (axis) => {
                    const isAxisLogScale = (yAxesFormatting.find(v => v.id === axis.id) || {}).isLogScale;
                    const [axisMin] = (axis && svc.getCurrentAxisExtent(axis)) || [];
                    const isZeroInRange = axisMin <= 0;
                    return (!axis.dimension || ChartAxesUtils.isNumerical({ dimension: axis.dimension })) && !isAxisLogScale && isZeroInRange;
                };

                const availableAxes = axes.filter(axis => isAxisCompatible(axis));
                axes.forEach(axis => {
                    const [axisMin, axisMax] = (axis && svc.getCurrentAxisExtent(axis)) || [];
                    axesMins.push(axisMin);
                    axesMaxes.push(axisMax);
                    if (isAxisCompatible(axis)) {
                        availableAxesMins.push(axisMin);
                        availableAxesMaxes.push(axisMax);
                    }
                    axesRes.push({ min: axisMin, max: axisMax });
                });


                if (availableAxes.length > 1 && availableAxesMins.some(v => v < 0) && availableAxesMaxes.some(v => v > 0)) {
                    const minFactors = [];
                    const maxFactors = [];
                    const axesAbsolutes = [];
                    availableAxes.forEach(axis => {
                        const index = axes.indexOf(axis);
                        const axisAbsolute = (Math.max(Math.abs(axesMins[index]), Math.abs(axesMaxes[index])));
                        axesAbsolutes.push(axisAbsolute);
                        minFactors.push(Math.abs(Math.min(0, axesMins[index]) / axisAbsolute));
                        maxFactors.push(Math.abs(Math.max(0, axesMaxes[index]) / axisAbsolute));

                    });


                    const maxAxisFactor = Math.max(...maxFactors);
                    const minAxisFactor = Math.max(...minFactors);

                    availableAxes.forEach((axis, i) => {
                        const index = axes.indexOf(axis);
                        axesRes[index].min = -axesAbsolutes[i] * minAxisFactor;
                        axesRes[index].max = axesAbsolutes[i] * maxAxisFactor;
                        axis.scale().domain([axesRes[index].min, axesRes[index].max]);

                    });
                }

                axes.forEach((axis, i) => {
                    const axisFormatting = yAxesFormatting.find(v => v.id === axis.id);
                    // only align axes and return if there is ticks configuration number set
                    if (_.isNil(axisFormatting.ticksConfig.number)) {
                        axis.tickValues(axisOptions[i].tickValues);
                        axis.tickFormat(axisOptions[i].tickFormat);
                    } else {
                        svc.setNumberOfTicks(axis, axesRes[i], axisFormatting.ticksConfig, axisFormatting.numberFormatting);
                    }
                });

            },

            /**
             * Set the correct number of ticks to axis if ticksConfig.number is set.
             */
            alignAndSetXTickValues: function(xAxis, axisOptions, xAxisFormatting) {
                const [axisMin, axisMax] = (xAxis && svc.getCurrentAxisExtent(xAxis)) || [];
                const axisRes = { min: axisMin, max: axisMax };

                // only align axes and return if there is ticks configuration number set
                if (xAxisFormatting.ticksConfig.number == null) {
                    xAxis.tickValues(axisOptions.tickValues);
                    xAxis.tickFormat(axisOptions.tickFormat);
                    return;
                }

                svc.setNumberOfTicks(xAxis, axisRes, xAxisFormatting.ticksConfig, xAxisFormatting.numberFormatting);
            },

            setNumberOfTicks: (axis, resAxis, ticksConfig, numberFormatting) => {
                const maxTicksDisplayed = 500;
                if (svc.canSetNumberOfTicks(axis)) {
                    const axisTicks = Math.min(maxTicksDisplayed, ticksConfig.mode === AxisTicksConfigMode.NUMBER ? Math.floor(ticksConfig.number + 1) : (resAxis.max - resAxis.min) / ticksConfig.number);
                    resAxis.interval = (resAxis.max - resAxis.min) / axisTicks;
                    const values = Array.from({ length: axisTicks + 1 }, (_, i) => resAxis.min + i * resAxis.interval);
                    axis.tickValues(values);
                    axis.tickFormat(ChartFormatting.getForAxis(resAxis.min, resAxis.max, values, numberFormatting));
                }
            },

            /*
             * Interval division might produce arbitrary-precision arithmetic issues (IEEE 754)
             * formatter should round axis origin to '0' (as it might be a value like 4e-14)
             */
            canSetNumberOfTicks: (axis) => {
                if (!axis) {
                    return false;
                } else if (axis.scaleType) {
                    return [SCALE_TYPES.LINEAR, SCALE_TYPES.LOG].includes(axis.scaleType);
                } else {
                    return axis.type === CHART_AXIS_TYPES.MEASURE;
                }
            },

            /**
             * Adjust the domain of two given axis so that they have the same scales
             * @param chartDef
             * @param {d3 scale} xScale
             * @param {d3 scale} yScale
             */
            equalizeScales: function(chartDef, xScale, yScale) {

                const compatibleAxes = ChartUADimension.areAllNumericalOrDate(chartDef);
                if (compatibleAxes) {

                    const axisMin = Math.min(xScale.domain()[0], yScale.domain()[0]);
                    const axisMax = Math.max(xScale.domain()[1], yScale.domain()[1]);

                    const extent = function(v) {
                        return Math.abs(v[1] - v[0]);
                    };

                    // Add a gap to Y axis so that it reaches origin. X axis is always correct because it starts from it.
                    const diff = Math.abs(extent(yScale.range()) - extent(xScale.range()));
                    if (extent(yScale.range()) > extent(xScale.range())) {
                        yScale.range([xScale.range()[1] + diff, xScale.range()[0] + diff]);
                    } else {
                        xScale.range([yScale.range()[1], yScale.range()[0]]);
                    }

                    xScale.domain([axisMin, axisMax]);
                    yScale.domain([axisMin, axisMax]);
                }

                chartDef.compatibleAxes = compatibleAxes;
            },

            clearAxes: function(g) {
                g.selectAll('.x.axis').remove();
                g.selectAll('.y.axis').remove();
            },

            drawGridlines: function(axisG, axisName, formattingOptions, position, chartG) {
                const isYAxis = axisName === 'y';
                const ticks = getTicksForGridlines(axisG, isYAxis ? 'x' : 'y', chartG);
                svc.appendGridlines(d3.selectAll(ticks), position, formattingOptions, isYAxis ? 'hline' : 'vline');
            },

            appendGridlines(el, position, formattingOptions, className) {
                const { x1, x2, y1, y2 } = position;
                const isDashed = formattingOptions.type === 'DASHED';
                el.append('line')
                    .attr('class', className)
                    .attr('stroke', formattingOptions.color)
                    .attr('stroke-width', formattingOptions.size)
                    .attr('stroke-dasharray', isDashed ? 12 : 0)
                    .attr('x1', x1)
                    .attr('x2', x2)
                    .attr('y1', y1)
                    .attr('y2', y2);
            },

            shouldDrawGridlinesForAxis: (chartType, axis, displayAxis) => {
                if (!ChartFeatures.canSelectGridlinesAxis(chartType)) {
                    return true;
                }
                switch (displayAxis.type) {
                    case GridlinesAxisType.CUSTOM_Y_AXIS:
                        return axis.id === displayAxis.axisId;
                    case GridlinesAxisType.LEFT_Y_AXIS:
                        return axis.position === ChartYAxisPosition.LEFT;
                    case GridlinesAxisType.RIGHT_Y_AXIS:
                        return axis.position === ChartYAxisPosition.RIGHT;
                    default:
                        return true;
                }
            },

            /**
             * Draw the given axes for all the given svg(s)
             * @param {d3DrawContext}: an object, with the following properties:
             *      - $svgs {jQuery} $svgs,
             *      - chartDef {ChartDef.java}
             *      - chartHandler {$scope}
             *      - ySpecs {AxisSpec}: y axis specs
             *      - xAxis {d3 axis} (nullable)
             *      - yAxes {d3 axis} (nullable)
             *      - axisOptions {AxisOptions}: (nullable)
             *      - handleZoom {boolean}: (nullable) Applied for lines only if it is zoomable
             *      - yAxesColors {[specID: string]: string}: colors of the yAxes (black for all the charts except scattersMP)
             */
            drawAxes: function(d3DrawContext) {
                if (!d3DrawContext) {
                    return;
                }
                const { $svgs, chartDef, chartHandler, ySpecs, xAxis, yAxes, axisOptions, handleZoom, yAxesColors } = d3DrawContext;
                // Align double axes and set tick values
                if (ChartFeatures.canDisplayAxes(chartDef.type)) {
                    const xOptions = axisOptions.x;
                    const yOptions = [...axisOptions.y];

                    let xTicksConfig = chartDef.xAxisFormatting.ticksConfig;
                    let yAxesFormatting = chartDef.yAxesFormatting.map(v => {
                        return { id: v.id, ticksConfig: v.ticksConfig, numberFormatting: v.axisValuesFormatting.numberFormatting, isLogScale: v.isLogScale };
                    });

                    if (ChartFeatures.isScatterZoomed(chartDef.type, chartDef.$zoomControlInstanceId)) {
                        xTicksConfig = { ...xTicksConfig, number: null };
                        yAxesFormatting = yAxesFormatting.map(v => {
                            return { ...v, ticksConfig: { ...v.ticksConfig, number: null } };
                        });
                    }

                    const yAxesOneTickPerBin = Object.keys(ySpecs).some(key => ChartFeatures.hasOneTickPerBinSelected(chartDef && chartDef.$chartStoreId, key));

                    !handleZoom && !ChartFeatures.hasOneTickPerBinSelected(chartDef && chartDef.$chartStoreId, 'x') && svc.alignAndSetXTickValues(xAxis, xOptions, { ticksConfig: xTicksConfig, numberFormatting: chartDef.xAxisFormatting.axisValuesFormatting.numberFormatting });
                    !yAxesOneTickPerBin && svc.alignAndSetYTickValues(yAxes, yOptions, yAxesFormatting);
                }

                // Initial margins
                const $svg = $svgs.eq(0),
                    width = $svg.width(),
                    height = $svg.height();
                let { margins, yAxesWidth } = svc.getMarginsAndAxesWidth(chartHandler, chartDef, $svg, xAxis, yAxes);

                // Update x scale based on horizontal margins
                const vizWidth = width - margins.left - margins.right;
                if (xAxis) {
                    xAxis.setScaleRange([0, vizWidth]);
                }

                // Adjust bottom margin for x axis
                if (xAxis && !chartHandler.noXAxis && chartDef.xAxisFormatting.displayAxis) {
                    margins = svc.adjustBottomMargin(chartDef.xAxisFormatting.axisValuesFormatting.axisTicksFormatting, margins, $svg, xAxis, chartHandler.forceRotation);
                }

                let allNegative = true;
                let allPositive = true;
                yAxes.forEach(yAxis => {
                    const currentYExtent = svc.getCurrentAxisExtent(yAxis);
                    if (currentYExtent && currentYExtent[1] >= 0) {
                        allNegative = false;
                    }
                    if (currentYExtent && currentYExtent[0] < 0) {
                        allPositive = false;
                    }
                });

                if (xAxis && chartDef.singleXAxis && chartDef.facetDimension.length) {
                    // Override bottom margins when we don't actually need space in the svgs for the axis
                    margins.axisHeight = margins.bottom;
                    margins.bottom = allPositive ? 0 : 0.2 * height;
                    margins.top = allNegative ? 0 : 0.2 * height;
                }

                if (chartDef.xAxisFormatting.isLogScale && ChartFeatures.isUnaggregated(chartDef.type) && !ChartFeatures.isScatterZoomed(chartDef.type, chartDef.$zoomControlInstanceId)) {
                    svc.adjustScatterPlotAxisPadding(xAxis, vizWidth);
                }

                // Update y scales accordingly
                let vizHeight = height - margins.top - margins.bottom;
                yAxes.forEach((axis, i) => {
                    if (i === 0 && ChartAxesUtils.isLeftYAxis(axis)) {
                        if (ySpecs[axis.id].ascendingDown) {
                            axis.setScaleRange([0, vizHeight]);
                        } else {
                            axis.setScaleRange([vizHeight, 0]);
                        }

                        // Enforce minRangeBand (eg) for horizontal bars
                        if (chartDef.facetDimension.length === 0 && axis.type === 'DIMENSION' && ySpecs[axis.id].minRangeBand > 0 && axis.ordinalScale.rangeBand() < ySpecs[axis.id].minRangeBand) {
                            const numLabels = axis.ordinalScale.domain().length;
                            const padding = svc.getColumnPadding(numLabels);
                            const range = d3Utils.getRangeForGivenRangeBand(ySpecs[axis.id].minRangeBand, numLabels, padding, padding / 2);
                            if (ySpecs[axis.id].ascendingDown) {
                                axis.setScaleRange([0, range]);
                            } else {
                                axis.setScaleRange([range, 0]);
                            }
                            vizHeight = range;
                            const svgHeight = vizHeight + margins.top + margins.bottom;
                            $svgs.height(svgHeight);
                        }
                    } else {
                        axis.setScaleRange([vizHeight, 0]);
                    }

                    if (ChartAxesUtils.isYAxisLogScale(chartDef.yAxesFormatting, axis.id) && ChartFeatures.isUnaggregated(chartDef.type) && !ChartFeatures.isScatterZoomed(chartDef.type, chartDef.$zoomControlInstanceId)) {
                        svc.adjustScatterPlotAxisPadding(axis, vizHeight);
                    }

                    if (chartDef.valuesInChartDisplayOptions.displayValues && chartDef.type === CHART_TYPES.GROUPED_COLUMNS) {
                        svc.adjustVerticalBarsYAxisPadding(axis, vizHeight, ChartLabels.getMeasureValueInChartMaxHeight(chartDef, true, false, 2, computeTextHeightFromFontSize($svgs)), true, chartDef);
                    }

                    if (chartDef.valuesInChartDisplayOptions.displayValues && ChartFeatures.canDisplayTotalValues(chartDef) && !chartDef.genericMeasures.some(m => ChartMeasure.isRealUnaggregatedMeasure(m))) {
                        svc.adjustVerticalBarsYAxisPadding(axis, vizHeight, ChartLabels.getMeasureValueInChartMaxHeight(chartDef, false, true, 5, computeTextHeightFromFontSize($svgs)), true);
                    }

                    if (chartDef.valuesInChartDisplayOptions.displayValues && chartDef.type === CHART_TYPES.LINES) {
                        svc.adjustLinesYAxisPadding(axis, vizHeight, ChartLabels.getMeasureValueInChartMaxHeight(chartDef, true, false, 12, computeTextHeightFromFontSize($svgs)));
                    }

                    if (chartDef.valuesInChartDisplayOptions.displayValues && chartDef.type === CHART_TYPES.MULTI_COLUMNS_LINES) {
                        svc.adjustVerticalBarsYAxisPadding(axis, vizHeight, ChartLabels.getMeasureValueInChartMaxHeight(chartDef, true, false, 5, computeTextHeightFromFontSize($svgs)), false);
                    }
                });

                // Equalize x and y to same scale if needed
                if (chartDef.type === CHART_TYPES.SCATTER && chartDef.scatterOptions && chartDef.scatterOptions.equalScales) {
                    svc.equalizeScales(chartDef, xAxis.scale(), yAxes[0].scale());
                }

                const updatedMargins = _.cloneDeep(margins);

                let $xAxisSvgs = $svgs;

                // If chart is facetted and 'singleXAxis' is enabled, we put the axis in a separate svg, fixed at the bottom of the screen
                if (xAxis && chartDef.singleXAxis && chartDef.facetDimension.length) {
                    $xAxisSvgs = $('<svg class="x-axis-svg" xmlns="http://www.w3.org/2000/svg">');
                    $('<div class="x-axis noflex">').css('height', updatedMargins.axisHeight).append($xAxisSvgs).appendTo($svgs.eq(0).closest('.mainzone'));
                    d3.select($xAxisSvgs.get(0)).append('g').attr('class', 'chart').attr('transform', 'translate(' + ($svgs.closest('.chart').find('h2').outerWidth() + updatedMargins.left) + ', -1)');
                }

                // Create a g.chart in every svg
                $svgs.each(function() {
                    const svg = d3.select(this);
                    const g = svg.select('g.chart');
                    //  If g.chart already exists, we don't need to recreate it (useful for zoom)
                    if (!g[0][0]) {
                        svg.append('g').attr('class', 'chart');
                    }
                });

                if (!chartHandler.noXAxis && xAxis) {
                    xAxis.orient('bottom');

                    $xAxisSvgs.each(function() {
                        const g = d3.select(this).select('g');

                        const xAxisG = g.append('g')
                            .attr('class', 'x axis qa_charts_x-axis-column-label-text');

                        if (!chartDef.singleXAxis || !chartDef.facetDimension.length) {
                            if (!allNegative) {
                                xAxisG.attr('transform', 'translate(0,' + vizHeight + ')');
                                xAxis.orient('bottom');
                            } else {
                                xAxis.orient('top');
                                const bottomMargin = updatedMargins.bottom;
                                updatedMargins.bottom = updatedMargins.top;
                                updatedMargins.top = bottomMargin;
                            }
                        }

                        if (chartDef.xAxisFormatting.displayAxis) {
                            xAxisG.call(xAxis);

                            const axisScale = yAxes[0].scale()(0);
                            if (!allNegative && !allPositive && xAxis.type !== CHART_AXIS_TYPES.UNAGGREGATED && axisScale) {
                                xAxisG.select('path.domain').attr('transform', 'translate(0,' + (axisScale - vizHeight) + ')');
                            }

                            if (xAxis.labelAngle) {
                                if (!allNegative) {
                                    xAxisG.selectAll('text')
                                        .attr('transform', (xAxis.labelAngle == Math.PI / 2 ? 'translate(-13, 9)' : 'translate(-10, 0)') + ' rotate(' + xAxis.labelAngle * -180 / Math.PI + ', 0, 0)')
                                        .style('text-anchor', 'end');
                                } else {
                                    xAxisG.selectAll('text')
                                        .attr('transform', 'translate(10, 0) rotate(' + xAxis.labelAngle * -180 / Math.PI + ', 0, 0)')
                                        .style('text-anchor', 'start');
                                }
                            }

                            if (xAxisG) {
                                const ticksFormatting = chartDef.xAxisFormatting.axisValuesFormatting.axisTicksFormatting;
                                ticksFormatting && xAxisG.selectAll('.tick text')
                                    .attr('fill', ticksFormatting.fontColor)
                                    .attr('font-size', `${ticksFormatting.fontSize}px`);

                                // sanitize ticks if a specific configuration is set
                                if (ChartFeatures.shouldRemoveOverlappingTicks(chartDef.xAxisFormatting, chartDef, 'x')) {
                                    xAxisG.selectAll('.tick').forEach(ticks => {
                                        ticks.sort((a, b) => a.__data__ - b.__data__);
                                        const overlappingTicks = svc.sanitizeTicksDisplay(ticks, true, true);
                                        AxisTicksConfiguration.setXTicksAreOverlapping(overlappingTicks.length > 0);
                                    });
                                } else {
                                    AxisTicksConfiguration.setXTicksAreOverlapping(false);
                                }
                            }

                            if (!xAxis.labelAngle) {
                                // The last tick might be at the very end of the axis and overflow the SVG
                                const lastTick = xAxisG.select('.tick:last-of-type');
                                const domainWidth = xAxisG.select('.domain').node()?.getBBox()?.width || 0;
                                const lastTickTransform = lastTick.attr('transform');
                                const lastTickText = lastTick.select('text');

                                if (lastTickTransform) {
                                    const translateMatch = lastTickTransform.match(/translate\(([^,]+),([^)]+)\)/);
                                    const lastTickX = translateMatch ? parseFloat(translateMatch[1]) : 0;
                                    const lastTextWidth = lastTickText.node()?.getBBox()?.width || 0;
                                    if (lastTickX + lastTextWidth / 2 > domainWidth + updatedMargins.right) {
                                        lastTickText.attr('x', -(lastTickX + (lastTextWidth / 2) - domainWidth));
                                    }
                                }
                            }
                        }

                        svc.addTitleToXAxis(xAxisG, xAxis, chartDef, chartHandler, updatedMargins, vizWidth);
                    });
                }

                $svgs.each(function() {
                    const g = d3.select(this).select('g');

                    g.attr('transform', 'translate(' + updatedMargins.left + ',' + updatedMargins.top + ')');

                    if (xAxis && chartDef.gridlinesOptions.vertical.show && ChartFeatures.canHaveVerticalGridlines(chartDef) && !chartHandler.noXAxis) {
                        const gridlinesPosition = { x1: 0, x2: 0, y1: 0, y2: allNegative ? vizHeight : -vizHeight };
                        svc.drawGridlines(g.select('g.x.axis'), 'x', chartDef.gridlinesOptions.vertical.lineFormatting, gridlinesPosition, g);
                    }

                    if (!chartHandler.noYAxis) {
                        yAxes.forEach((axis, index) => {
                            const axisColor = yAxesColors[axis.id] || '#000';
                            const axisFormatting = ChartAxesUtils.getFormattingForYAxis(chartDef.yAxesFormatting, axis.id);

                            axis.orient(axis.position);

                            if (vizHeight < 300) {
                                axis.ticks(svc.optimizeTicksNumber(vizHeight));
                            }

                            const yAxisG = g.append('g').attr('class', `y y${index + 1} axis`);
                            let remainingAxesWidth = 0;
                            if (ChartAxesUtils.isRightYAxis(axis)) {
                                yAxisG.attr('transform', 'translate(' + vizWidth + ',0)');
                            } else {
                                remainingAxesWidth = yAxesWidth.slice(index + 1).reduce((acc, val) => {
                                    const isLeftAxis = ChartAxesUtils.isLeftYAxis(yAxes.find(axis => axis.id === val.id));
                                    if (isLeftAxis && val.width > 0) {
                                        acc += val.width + ChartsStaticData.AXIS_Y_SPACING;
                                    }
                                    return acc;
                                }, 0);
                                yAxisG.attr('transform', `translate(${-remainingAxesWidth},0)`);
                            }
                            if (axisFormatting.displayAxis) {
                                yAxisG.call(axis);

                                if (chartDef.gridlinesOptions.horizontal.show && svc.shouldDrawGridlinesForAxis(chartDef.type, axis, chartDef.gridlinesOptions.horizontal.displayAxis)) {
                                    const gridlinesPosition = { x1: remainingAxesWidth, x2: ChartAxesUtils.isRightYAxis(axis) ? -vizWidth : vizWidth, y1: 0, y2: 0 };
                                    svc.drawGridlines(yAxisG, 'y', chartDef.gridlinesOptions.horizontal.lineFormatting, gridlinesPosition, g);
                                }


                                if (xAxis && chartDef.singleXAxis && chartDef.facetDimension.length && !index) {
                                    yAxisG.select('.domain').attr('d', 'M0,-100V1000'); // TODO @charts dirty, use vizHeight+margins.top+margins.bottom instead of 1000?
                                    if (allPositive && (axis.type === CHART_AXIS_TYPES.MEASURE || axis.scaleType === SCALE_TYPES.LINEAR)) {
                                        yAxisG.select('.tick').remove(); // remove tick for '0'
                                    }
                                }

                                const axisG = g.select(`.y${index + 1}`);
                                const ticksFormatting = axisFormatting.axisValuesFormatting.axisTicksFormatting;

                                axisG.selectAll('.tick text')
                                    .attr('fill', ChartFeatures.canSetAxisTextColor(chartDef.type) ? ticksFormatting.fontColor : axisColor)
                                    .attr('font-size', `${ticksFormatting.fontSize}px`);
                            }

                            const axisG = g.select(`.y${index + 1}`);
                            svc.addTitleToYAxis(axisG, axis, chartDef, chartHandler, margins, vizHeight, yAxesWidth[index].width, axisColor);


                            // sanitize ticks if a specific configuration is set
                            if (ChartFeatures.shouldRemoveOverlappingTicks(axisFormatting, chartDef, axisFormatting.id)) {
                                g.selectAll(`g.y${index + 1}.axis`).selectAll('g.tick').forEach(ticks => {
                                    ticks.sort((a, b) => a.__data__ - b.__data__);
                                    const overlappingTicks = svc.sanitizeTicksDisplay(ticks, true, true);
                                    AxisTicksConfiguration.setYTicksAreOverlapping(overlappingTicks.length > 0);
                                });
                            } else {
                                AxisTicksConfiguration.setYTicksAreOverlapping(false);
                            }
                            const domain = yAxisG.selectAll('.domain, .tick');
                            domain.style('stroke', axisColor);
                        });
                    }
                });

                return { margins, updatedMargins, vizWidth, vizHeight };
            },


            /**
             * Crop the title if it exceeds the length limit
             * @param {d3 selection}labelElement : d3 selection of the text element containing the title
             * @param {string} title: title to display
             * @param {number} maxTextWidth: maximum width for the graphical text element
             */
            formatAxisTitle: function(labelElement, title, maxTextWidth) {
                if (maxTextWidth > 0) {
                    let textLength = labelElement.node().getComputedTextLength();
                    let displayedText = title;
                    while (textLength > 0 && displayedText && displayedText.length && textLength > maxTextWidth) {
                        // crop the title text
                        displayedText = displayedText.substring(0, displayedText.length - 1);
                        labelElement.text(`${displayedText}...`);
                        textLength = labelElement.node().getComputedTextLength();
                    }
                }
            },

            /**
             * When adding a title to an axis corresponding to a DATE dimension, we might need to add the date as a suffix to the title
             * @param {string} title: title to display
             * @param {d3 axis}axis : d3 axis for which we want to display the title
             */
            addDateToTitle: function(title, axis) {
                let finalTitle = title;
                if (title && axis.dimension && ChartUADimension.isDate(axis.dimension) && axis.scaleType === SCALE_TYPES.TIME) {
                // specific case for a date dimension, if the current domain is contained within a single day, we add this date at the end of the title
                    const extent = svc.getCurrentAxisExtent(axis);
                    const computedDateDisplayUnit = ChartDataUtils.computeDateDisplayUnit(extent[0], extent[1]);
                    if (computedDateDisplayUnit.formattedMainDate) {
                        finalTitle += ` (${computedDateDisplayUnit.formattedMainDate})`;
                    }
                }

                return finalTitle;
            },

            /**
             * Add the <text> title to the X axis' <g>
             * @param {SVG:g} axisG
             * @param {d3 axis} xAxis
             * @param {ChartDef.java} chartDef
             * @param {$scope} chartHandler
             * @param {Object {top: top, left: left, right: right, bottom: bottom}} margins
             * @param {number} chartWidth
             */
            addTitleToXAxis: function(axisG, xAxis, chartDef, chartHandler, margins, chartWidth) {
                let titleText = ChartAxesUtils.getXAxisTitle(xAxis, chartDef);
                titleText = svc.addDateToTitle(titleText, xAxis);

                const titleStyle = chartDef.xAxisFormatting.axisTitleFormatting;
                const axisTitleMaxWidth = chartWidth - margins.right;


                if (titleText && chartDef.xAxisFormatting.showAxisTitle) {
                    const rect = axisG.append('rect');

                    const labelElement = axisG.append('text')
                        .attr('x', chartWidth / 2)
                        .attr('y', xAxis.orient() == 'bottom' ? margins.bottom - titleStyle.fontSize - ChartsStaticData.AXIS_MARGIN : titleStyle.fontSize + ChartsStaticData.AXIS_MARGIN - margins.top)
                        .attr('text-anchor', 'middle')
                        .attr('dominant-baseline', 'middle')
                        .attr('fill', titleStyle.fontColor)
                        .attr('class', 'axis-title x qa_charts_x-axis-title')
                        .style('font-size', `${titleStyle.fontSize}px`)
                        .text(titleText);

                    this.formatAxisTitle(labelElement, titleText, axisTitleMaxWidth);

                    if (!chartHandler.noClickableAxisLabels) {
                        labelElement.attr('no-global-contextual-menu-close', true)
                            .on('click', function() {
                                chartHandler.openSection(ChartFormattingPaneSections.X_AXIS);
                            });
                        chartHandler.$on('$destroy', () => {
                            labelElement.on('click', null);
                        });
                    }

                    const bbox = labelElement.node().getBoundingClientRect();

                    rect.attr('x', chartWidth / 2 - bbox.width / 2)
                        .attr('y', xAxis.orient() == 'bottom' ? margins.bottom - 15 - bbox.height / 2 : 15 - margins.top - bbox.height / 2)
                        .attr('width', bbox.width)
                        .attr('height', bbox.height)
                        .attr('fill', 'none')
                        .attr('class', 'chart-wrapper__x-axis-title')
                        .attr('stroke', 'none');
                }
            },

            /**
             * @param {d3 axis} axis
             * @returns {[Float, Float]} [extentMin, extentMax]
             */
            getCurrentAxisExtent(axis) {
                const extent = axis.scale().domain();
                return [Math.min(...extent), Math.max(...extent)];
            },


            /**
             * Add the <text> title to the Y axis' <g>
             * @param {SVG:g} axisG
             * @param {d3 axis} yAxis
             * @param {ChartDef.java} chartDef
             * @param {$scope} chartHandler
             * @param {{top: number, bottom: number, left: number, right: number}} margins
             * @param {number} chartHeight
             * @param {number} axisWidth
             * @param {number} axisColor black for all charts apart from scatters MP
             */
            addTitleToYAxis: function(axisG, yAxis, chartDef, chartHandler, margins, chartHeight, axisWidth, axisColor) {
                const axisFormatting = ChartAxesUtils.getFormattingForYAxis(chartDef.yAxesFormatting, yAxis.id);
                const titleStyle = axisFormatting.axisTitleFormatting;
                const xPosition = ChartAxesUtils.isLeftYAxis(yAxis) ? -axisWidth : axisWidth;
                const axisTitleMaxWidth = chartHeight - margins.top;
                let titleText = ChartAxesUtils.getYAxisTitle(yAxis, chartDef, axisFormatting);
                titleText = svc.addDateToTitle(titleText, yAxis);

                if (titleText && axisFormatting.showAxisTitle) {
                    const labelElement = axisG.append('text')
                        .attr('x', xPosition)
                        .attr('y', (chartHeight - margins.top) / 2)
                        .attr('text-anchor', 'middle')
                        .attr('dominant-baseline', ChartAxesUtils.isLeftYAxis(yAxis) ? 'middle' : 'text-top')
                        .attr('class', 'axis-title y qa_charts_y-axis-title')
                        .attr('fill', ChartFeatures.canSetAxisTextColor(chartDef.type) ? titleStyle.fontColor : axisColor)
                        .style('font-size', `${titleStyle.fontSize}px`)
                        .attr('transform', 'rotate(-90, ' + xPosition + ',' + (chartHeight - margins.top) / 2 + ')')
                        .text(titleText);

                    this.formatAxisTitle(labelElement, titleText, axisTitleMaxWidth);

                    if (!chartHandler.noClickableAxisLabels) {
                        labelElement.attr('no-global-contextual-menu-close', true)
                            .on('click', function() {
                                chartHandler.openSection(ChartFormattingPaneSections.Y_AXIS);
                            });
                        chartHandler.$on('$destroy', () => {
                            labelElement.on('click', null);
                        });
                    }
                }
            },

            /**
             * Add padding to a Scatter Plot axis, because it cannot be computed at axis creation because we need the axis height to compute it (to handle the log scale).
             * Mutates the given axis.
             * @param {Record<string, unknown>} axis a d3 axis object.
             * @param {number} axisHeight the height of the axis in pixels.
             * @param {number} [paddingPct=0.05] - a value between 0 and 1 corresponding to the percentage of padding that should be applied.
             * @returns {Record<string, unknown>} the updated d3 axis object.
             */
            adjustScatterPlotAxisPadding: (axis, axisHeight, paddingPct = 0.05) => {
                const extent = axis.originalDomain || svc.getCurrentAxisExtent(axis);
                const logMin = Math.log(extent[0]);
                const logMax = Math.log(extent[1]);
                const logDifference = logMax - logMin;
                const paddingPx = axisHeight * paddingPct;
                const axisHeightMinusPadding = axisHeight - (paddingPx * 2);
                const logPadding = logDifference / axisHeightMinusPadding * paddingPx;

                const paddedDomain = [
                    Math.exp(logMin - logPadding),
                    Math.exp(logMax + logPadding)
                ];

                axis.originalDomain = axis.originalDomain || [...axis.scale().domain()];
                axis.scale().domain(paddedDomain);

                return axis;
            },

            /**
             * Add padding to a Vertical bars y axis, because it cannot be computed at axis creation because we need the axis height to compute it (to add space between the bar and the x axis to display values).
             * Mutates the given axis.
             * @param {Record<string, unknown>} axis a d3 axis object.
             * @param {number} axisHeight the height of the axis in pixels.
             * @param {number} valuesFontSize - the font size of the values displayed in chart
             * @returns {Record<string, unknown>} the updated d3 axis object.
             */
            adjustVerticalBarsYAxisPadding: (axis, axisHeight, valuesFontSize, addNegativeExtent, chartDef) => {
                const extent = svc.getCurrentAxisExtent(axis);
                if (chartDef?.variant === CHART_VARIANTS.waterfall && chartDef.genericMeasures.length > 0) {
                    const valuePlacement = chartDef.genericMeasures[0].valuesInChartDisplayOptions?.placementMode;
                    switch (valuePlacement) {
                        case ValuesInChartPlacementMode.BELOW:
                            adjustAxisPadding(axis, extent, axisHeight, valuesFontSize, true, false);
                            break;
                        case ValuesInChartPlacementMode.ABOVE:
                            adjustAxisPadding(axis, extent, axisHeight, valuesFontSize, false, true);
                            break;
                        case ValuesInChartPlacementMode.AUTO:
                            adjustAxisPadding(axis, extent, axisHeight, valuesFontSize, extent[0] < 0, extent[1] > 0);
                            break;

                        default:
                            break;
                    }

                } else {
                    adjustAxisPadding(axis, extent, axisHeight, valuesFontSize, extent[0] < 0 && addNegativeExtent, extent[1] > 0);
                }
                return axis;
            },

            /**
             * Add padding to a Line chart y axis, because it cannot be computed at axis creation because we need the axis height to compute it (to add space between the bar and the x axis to display values).
             * Mutates the given axis.
             * @param {Record<string, unknown>} axis a d3 axis object.
             * @param {number} axisHeight the height of the axis in pixels.
             * @param {number} valuesFontSize - the font size of the values displayed in chart
             * @returns {Record<string, unknown>} the updated d3 axis object.
             */
            adjustLinesYAxisPadding: (axis, axisHeight, valuesFontSize) => {
                const extent = svc.getCurrentAxisExtent(axis);
                adjustAxisPadding(axis, extent, axisHeight, valuesFontSize, true, true);

                return axis;
            },

            /**
             * Computes the optimum number of ticks to display on an axis
             * @param {number} axisSize (in pixel)
             * @returns {number} the optimum ticks number
             */
            optimizeTicksNumber: function(axisSize) {
                return Math.floor(Math.max(axisSize / 30, 2));
            },

            /**
             * @param { d3.axis }                   axis
             * @param { FormattingOptions }         [formattingOptions = {}]     The number formatting options. Can be omitted to use the default formatting.
             */
            addNumberFormatterToAxis(axis, formattingOptions = {}) {
                const scale = (axis.scale() instanceof Function) ? axis.scale() : axis.scale;
                const minValue = Math.min(...scale.domain());
                const maxValue = Math.max(...scale.domain());
                const numValues = axis.tickValues() ? axis.tickValues().length : axis.ticks()[0];
                axis.tickFormat(ChartFormatting.getForAxis(minValue, maxValue, numValues, formattingOptions));
            }
        };

        return svc;
    }
})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.charts');

    /**
     * (!) This service previously was in static/dataiku/js/simple_report/chart_view_commons.js
     */
    app.factory('d3Utils', function() {
        return {
            // d3 < v4.0 doesnt have a ordinalScale.rangeStep() function (equivalent of rangeBand() when is the range is set with rangePoints)
            getOrdinalScaleRangeStep: function(ordinalScale) {
                if (ordinalScale.range().length < 2) {
                    return 100;
                }
                return Math.abs(ordinalScale.range()[1] - ordinalScale.range()[0]);
            },

            // Call a function once all transitions of a selection have ended
            endAll: function(transition, globalCallback, perItemCallback) {
                if (transition.size() === 0 && globalCallback) {
                    globalCallback();
                }
                let n = 0;
                transition
                    .each(function() {
                        ++n;
                    })
                    .each('end', function() {
                        if (perItemCallback) {
                            perItemCallback.apply(this, arguments);
                        }
                        if (!--n) {
                            globalCallback.apply(this, arguments);
                        }
                    });
            },

            // Computes the range length for an ordinal scale to have the given rangeBand, given domain length, padding & outerPadding
            getRangeForGivenRangeBand: function(rangeBand, domainLength, padding, outerPadding) {
                const step = rangeBand / (1 - padding);
                return rangeBand * domainLength + step * padding * (domainLength - 1) + 2 * step * outerPadding;
            }
        }
    });
})();

;
(function() {
    'use strict';

    angular.module('dataiku.charts')
        .service('ChartDataUtils', ChartDataUtils);

    /**
     * (!) This service previously was in static/dataiku/js/simple_report/common/data.js
     */
    function ChartDataUtils(ChartDimension, ChartMeasure, ChartUADimension, Fn, $filter, CHART_DATES_LABELS, ChartLabels, CHART_TYPES, ChartsStaticData, translate, CHART_VARIANTS) {
        /**
         * Returns true if:
         * <ul>
         *     <li>no timestamp range is defined</li>
         *     <li>element index is not one of the 'other' bin</li>
         *     <li>element index corresponds to a bins which timestamp is in the specified range</li>
         * </ul>
         */
        function isElementInTimestampRange(elementIndex, axisLabelElements, timestampRange) {
            if (!timestampRange) {
                return true;
            }
            const labelElementIndex = getLabelIndexForTensorIndex(elementIndex, axisLabelElements.length);
            const isOthersCategoryIndex = labelElementIndex === undefined;
            if (isOthersCategoryIndex) {
                return false;
            }
            const axisLabelElementTimestamp = axisLabelElements[labelElementIndex].tsValue;
            const lowestRangeBound = timestampRange[0];
            const highestRangeBound = timestampRange[1];
            return axisLabelElementTimestamp >= lowestRangeBound && axisLabelElementTimestamp <= highestRangeBound;
        }

        /**
         * Returns the index of the axis label element corresponding to the tensor index or undefined
         * if index is one of the 'other" bin.
         */
        function getLabelIndexForTensorIndex(tensorElementIndex, numberOfAxisLabelElements) {
            const numberOfElementsInFacet = numberOfAxisLabelElements + 1; // because of 'other' bin;
            const labelElementIndex = tensorElementIndex % numberOfElementsInFacet;
            const isOthersCategoryElementIndex = labelElementIndex === numberOfAxisLabelElements;
            if (isOthersCategoryElementIndex) {
                return undefined;
            }
            return labelElementIndex;
        }

        /**
         * Filters the tensor to keep only:
         * <ul>
         *     <li>non empty bins (i.e. with a count  > 0) when includeEmptyBins is set as false</li>
         *     <li>if timestampRange is specified, the bins which corresponding timestamp is in the range</li>
         * </ul>
         * @return {Array} the filtered tensor
         */
        function filterTensorOnTimestampRange(tensor, axisLabelElements, counts, timestampRange, includeEmptyBins) {
            return tensor.filter((value, index) => {
                const isEmptyBin = counts[index] === 0;
                return isElementInTimestampRange(index, axisLabelElements, timestampRange) && (!isEmptyBin || includeEmptyBins);
            });
        }

        function buildDefaultExtent() {
            return {
                extent: [Infinity, -Infinity],
                onlyPercent: true
            };
        }

        /**
         * Returns the corresponding date display settings for the specified interval:
         * <ul>
         *     <li>for MILLISECONDS if the range is within the same second</li>
         *     <li>for SECONDS AND MILLISECONDS if the range is lower than 2 seconds</li>
         *     <li>for SECONDS if the range is within the same minute</li>
         *     <li>for MINUTES AND SECONDS if the range is lower than a 2 minutes</li>
         *     <li>for MINUTES if the range is within the same day</li>
         *     <li>for DAYS AND MINUTES if the range is lower than 24 hours</li>
         *     <li>else the default display</li>
         * </ul>
         * @param minTimestamp The lower bound of the interval in milliseconds
         * @param maxTimestamp The upper bound of the interval in milliseconds.
         * @return {{dateFilterOption: string, dateFormat: string, mainDateFormat: string, formatDateFn: function(number, string)}}
         * <ul>
         *     <li>
         *         <b>mainDateFormat</b> format for the main identical part of the interval.
         *         <b>undefined</b> if the interval is not on the same day.
         *     </li>
         *     <li><b>dateFormat</b> format to use for the dates in the interval.</li>
         *     <li><b>dateFilterOption</b> date filter option to be used in an AngularJS filter.</li>
         *     <li><b>formatDateFn</b> function to format a timestamp in milliseconds according to the specified format.</li>
         * </ul>
         */
        function getDateDisplayUnit(minTimestamp, maxTimestamp) {
            if (minTimestamp === undefined || maxTimestamp === undefined) {
                return CHART_DATES_LABELS.DATE_DISPLAY_UNIT_DEFAULT;
            }
            const minDate = new Date(minTimestamp);
            const maxDate = new Date(maxTimestamp);
            const minDateWrapper = moment(minDate).utc();
            const maxDateWrapper = moment(maxDate).utc();

            const isDomainLessThan24Hours = maxDateWrapper.diff(minDateWrapper, 'hours', true) <= 24;
            if (!isDomainLessThan24Hours) {
                return CHART_DATES_LABELS.DATE_DISPLAY_UNIT_DEFAULT;
            }

            const isDomainInSameDay = minDateWrapper.year() === maxDateWrapper.year() && minDateWrapper.dayOfYear() === maxDateWrapper.dayOfYear();
            if (!isDomainInSameDay) {
                return CHART_DATES_LABELS.DATE_DISPLAY_UNIT_DAY_AND_MINUTES;
            }

            const isDomainLessThan2Minutes = maxDateWrapper.diff(minDateWrapper, 'minutes', true) <= 2;
            if (!isDomainLessThan2Minutes) {
                return CHART_DATES_LABELS.DATE_DISPLAY_UNIT_MINUTES;
            }

            const isDomainLessThan2Seconds = maxDateWrapper.diff(minDateWrapper, 'seconds', true) <= 2;
            if (!isDomainLessThan2Seconds) {
                return CHART_DATES_LABELS.DATE_DISPLAY_UNIT_SECONDS;
            }

            return CHART_DATES_LABELS.DATE_DISPLAY_UNIT_MILLISECONDS;
        }
        /**
         * Returns a label to be used to display records count in the UI.
         * @param   {Number}    beforeFiltersCount
         * @param   {Number}    afterFiltersCount
         * @param   {Map}       visibleCountMap
         * @param   {ChartType.java} chartType
         * @return  {String}    Human-readable label
         */
        function getLabelForRecordsCount(beforeFiltersCount, afterFiltersCount, visibleCountMap, chartType) {
            let actualCount = afterFiltersCount;

            if (afterFiltersCount === beforeFiltersCount && afterFiltersCount === 0) {
                return ChartLabels.getNoRecordLabel();
            }

            if ([CHART_TYPES.SCATTER_MULTIPLE_PAIRS, CHART_TYPES.SCATTER].includes(chartType)) {
                actualCount = visibleCountMap && visibleCountMap.size && Math.max(...visibleCountMap.values()) || 0;
            }

            if (actualCount < beforeFiltersCount) {
                return `${actualCount} / ${beforeFiltersCount} ${translate('CHARTS.HEADER.RECORDS', 'records')}`;
            } else {
                return `${beforeFiltersCount} ${beforeFiltersCount !== 1 ? translate('CHARTS.HEADER.RECORDS', 'records') : translate('CHARTS.HEADER.RECORD', 'record')}`;
            }
        }

        const cartesian = (...a) => a.reduce((a, b) => a.flatMap(d => b.map(e => [d, e].flat())));

        const longSmartNumber = $filter('longSmartNumber');

        const svc = {
            /**
             * Returns the min & max values across all dimensions & all measures for the two display axes
             * @param {ChartDef.java} chartDef
             * @param {ChartTensorDataWrapper} chartData
             * @param {string} mainAxisName - id of the main axis
             * @param {Array} timestampRange min and max used to filter the data based on their timestamp
             * @param {Boolean} includeEmptyBins - whether or not to include empty bins
             * @return {Object} { y1: { extent: [Number, Number], onlyPercent: Boolean }, y2: { extent: [Number, Number], onlyPercent: Boolean }, recordsCount: Number, pointsCount: Number }
             */
            getMeasureExtents: function(chartDef, chartData, mainAxisName, timestampRange, includeEmptyBins) {
                const result = {
                    [ChartsStaticData.LEFT_AXIS_ID]: buildDefaultExtent(),
                    [ChartsStaticData.RIGHT_AXIS_ID]: buildDefaultExtent(),
                    recordsCount: 0,
                    pointsCount: 0
                };

                const countsTensor = chartData.data.counts.tensor;
                const mainAxisLabels = chartData.getAxisLabels(mainAxisName);
                chartDef.genericMeasures.forEach(function(measure, measureIndex) {
                    let measureExtent = [];
                    const measureData = chartData.data.aggregations[measureIndex];
                    if (ChartMeasure.isRealUnaggregatedMeasure(measure)) {
                        const filterTensor = (t) => filterTensorOnTimestampRange(t, mainAxisLabels, countsTensor, timestampRange, includeEmptyBins);
                        measureExtent = ChartMeasure.getUaMeasureExtent(measureData, filterTensor);
                    } else {
                        const filteredTensor = filterTensorOnTimestampRange(measureData.tensor, mainAxisLabels, countsTensor, timestampRange, includeEmptyBins);
                        measureExtent = d3.extent(
                            chartDef.variant === CHART_VARIANTS.waterfall
                                ? filteredTensor.reduce((acc, value) => {
                                    const cumulativeValue = acc[acc.length - 1] || 0;
                                    acc.push(cumulativeValue + value);
                                    return acc;
                                }, [])
                                : filteredTensor
                        );
                    }
                    const axis = measure.displayAxis === 'axis1' ? ChartsStaticData.LEFT_AXIS_ID : ChartsStaticData.RIGHT_AXIS_ID;
                    result[axis].onlyPercent = result[axis].onlyPercent && ChartDimension.isPercentScale([measure]);
                    result[axis].extent[0] = _.isFinite(measureExtent[0]) ? Math.min(measureExtent[0], result[axis].extent[0]) : result[axis].extent[0];
                    result[axis].extent[1] = _.isFinite(measureExtent[1]) ? Math.max(measureExtent[1], result[axis].extent[1]) : result[axis].extent[1];
                });

                const countsTensorInRange = filterTensorOnTimestampRange(countsTensor, mainAxisLabels, countsTensor, timestampRange, includeEmptyBins);
                result.recordsCount = countsTensorInRange.reduce((currentCount, countInBin) => currentCount + countInBin, 0);
                result.pointsCount = countsTensorInRange.length;
                return result;
            },

            getValuesForLines: function(binIndexes, chartData, ignoreLabels) {
                // For lines we only want the total bin for the color dimension and all the axis bin (total excluded)
                const valuesForLines = binIndexes.reduce((bins, binValue, binIndex) => {
                    bins.push(chartData.data.axisLabels[binIndex].reduce((acc, cV, cI) => {
                        if (ignoreLabels.has(cV.label)) {
                            return acc;
                        }
                        if (binValue !== undefined) {
                            acc.push(binValue);
                        } else {
                            acc.push(cI);
                        }
                        return acc;
                    }, []));
                    return bins;
                }, []);
                return cartesian(...valuesForLines);
            },

            getValuesForColumns: function(binIndexes, chartData, ignoreLabels) {
                // For columns we don't want the total bin for color and all the axis bin (total exclude)
                const valuesForColumns = binIndexes.reduce((bins, binValue, binIndex) => {
                    bins.push(chartData.data.axisLabels[binIndex].reduce((acc, cV, cI) => {
                        if (ignoreLabels.has(cV.label)) {
                            return acc;
                        }
                        acc.push(cI);
                        return acc;
                    }, []));
                    return bins;
                }, []);

                return cartesian(...valuesForColumns);
            },

            getMeasureExtentsForMixColored: function(chartDef, chartData, includeEmptyBins, ignoreLabels = new Set(), binIndexes = [], ignoreForMeasure) {
                const result = {
                    [ChartsStaticData.LEFT_AXIS_ID]: buildDefaultExtent(),
                    [ChartsStaticData.RIGHT_AXIS_ID]: buildDefaultExtent(),
                    recordsCount: 0,
                    pointsCount: 0
                };

                const mainAxisLabel = chartData.data.axisLabels[0];
                const countsTensor = chartData.data.counts.tensor;
                // Create bins to visit by forcing dimensions to certain values, because total bin must be taken or not
                const valuesToCheckForLines = this.getValuesForLines(binIndexes, chartData, ignoreLabels);
                const valuesToCheckForColumns = this.getValuesForColumns(binIndexes, chartData, ignoreLabels);
                // We now have all the values we want we create the extent, mix can't have timestamp yet so we don't care of them
                chartDef.genericMeasures.forEach((measure, measureIndex) => {
                    const tensorValues = ignoreForMeasure && ignoreForMeasure(measure) ? this.retrieveAggrData(chartData, valuesToCheckForColumns, measureIndex) : this.retrieveAggrData(chartData, valuesToCheckForLines, measureIndex);
                    let measureExtent = [];
                    if (ChartMeasure.isRealUnaggregatedMeasure(measure)) {
                        measureExtent = ChartMeasure.getUaMeasureExtent(this.retrieveUnaggrExtents(chartData, ignoreForMeasure && ignoreForMeasure(measure) ? valuesToCheckForColumns : valuesToCheckForLines, measureIndex));
                    } else {
                        measureExtent = d3.extent(tensorValues);
                    }


                    const axis = measure.displayAxis === 'axis1' ? ChartsStaticData.LEFT_AXIS_ID : ChartsStaticData.RIGHT_AXIS_ID;
                    result[axis].onlyPercent = result[axis].onlyPercent && ChartDimension.isPercentScale([measure]);
                    result[axis].extent[0] = Math.min(measureExtent[0], result[axis].extent[0]);
                    result[axis].extent[1] = Math.max(measureExtent[1], result[axis].extent[1]);
                });

                const countsTensorInRange = filterTensorOnTimestampRange(countsTensor, mainAxisLabel, countsTensor, undefined, includeEmptyBins);
                result.recordsCount = countsTensorInRange.reduce((currentCount, countInBin) => currentCount + countInBin, 0);
                result.pointsCount = countsTensorInRange.length;
                return result;
            },

            retrieveAggrData: function(chartData, coordsArray, measureIndex) {
                return coordsArray.reduce((acc, coords) => {
                    acc.push(chartData.getPoint(chartData.data.aggregations[measureIndex], Array.isArray(coords) ? coords : [coords]));
                    return acc;
                }, []);
            },

            retrieveUnaggrExtents: function(chartData, coordsArray, measureIndex) {
                return coordsArray.reduce((acc, coords) => {
                    const extents = chartData.getPointExtents(chartData.data.aggregations[measureIndex], Array.isArray(coords) ? coords : [coords]);
                    acc.uaDataNegativeExtent.push(extents[0]);
                    acc.uaDataPositiveExtent.push(extents[1]);
                    return acc;
                }, { uaDataNegativeExtent: [], uaDataPositiveExtent: [] });
            },

            /**
             * Returns the min & max values across all dimensions for the given measure
             * @param {ChartTensorDataWrapper} chartData
             * @param {Number} mIdx - measure index
             * @param {Boolean} ignoreEmptyBins - whether or not to ignore empty bins
             * @param {Set<int> | null} binsToInclude - subtotal tensor indexes to include, the other will be excluded
             * @param {number[]} axesIdx - indexes of other axes
             * @return {Array} extent as [min, max]
             */
            getMeasureExtent: function(chartData, mIdx, ignoreEmptyBins, binsToInclude = null, aggrFn = null, axesIdx = []) {
                const allMeasureIndexes = axesIdx.concat(mIdx);
                const data = chartData.data;

                if (allMeasureIndexes.every(mIdx => !data.aggregations[mIdx])) {
                    return null;
                }

                let accessor = Fn.SELF;
                if (ignoreEmptyBins) {
                    accessor = function(d, i) {
                        // if the current axis has null for a value, but one of the other axes don't, we should still consider that value in the extent
                        const hasNonNullElement = allMeasureIndexes.some(mIdx => chartData.getNonNullCount(i, mIdx) > 0);

                        // maybe check isBinMeaningful for every axis ?

                        if (hasNonNullElement || chartData.isBinMeaningful(aggrFn, i, mIdx)) {
                            return !binsToInclude || binsToInclude.has(i) ? d : null;
                        }
                        return null;
                    };
                }

                return d3.extent(data.aggregations[mIdx].tensor, accessor);
            },


            /**
             * Returns an aggregation tensor where empty & all-null bins are filtered out
             * @param {PivotTableTensorResponse.java} data
             * @param {number} mIdx     index  of the measure whose values we're getting
             * @return {Array}          list of values for non-empty and non-null bins
             */
            getMeasureValues: function(data, mIdx, binsToInclude) {
                if (!data.aggregations[mIdx]) {
                    return null;
                }

                return data.aggregations[mIdx].tensor.filter(function(d, i) {
                    if (data.aggregations[mIdx].nonNullCounts) {
                        return data.aggregations[mIdx].nonNullCounts[i] > 0;
                    } else {
                        return (data.counts.tensor[i] > 0) ? (!binsToInclude || binsToInclude.has(i) ? true : false) : false;
                    }
                });
            },

            /**
             * Returns the min, max, and list of values on the given axis
             * @param {ChartTensorDataWrapper} chartData
             * @param {String} axisName: the name of the axis in chartData
             * @param {DimensionDef.java} dimension
             * @param {{ ignoreLabels?: boolean, initialExtent?: [number, number] }} extraOptions
             * @return {Object} extent as {values: [], min: min, max: max}
             */
            getAxisExtent: function(chartData, axisName, dimension, extraOptions = {}) {
                const ignoreLabels = extraOptions.ignoreLabels || new Set();
                const initialExtent = extraOptions.initialExtent || [Infinity, -Infinity];
                const labels = chartData.getAxisLabels(axisName);
                let min = initialExtent[0];
                let max = initialExtent[1];

                const values = labels.filter(label => !ignoreLabels.has(label.label)).map(function(label) {
                    if (ChartDimension.isTimelineable(dimension)) {
                        min = Math.min(min, label.min);
                        max = Math.max(max, label.max);
                    } else if (ChartDimension.isTrueNumerical(dimension)) {
                        if (ChartDimension.isUngroupedNumerical(dimension) || label.min == null) {
                            min = Math.min(min, label.sortValue);
                            max = Math.max(max, label.sortValue);
                        } else {
                            min = Math.min(min, label.min);
                            max = Math.max(max, label.max);
                            return [label.min, label.max];
                        }
                    }
                    return label.label;
                });

                return { values, min, max };
            },

            /**
             * Returns the min, max, and list of values on the given axis
             * @param {String} uaDimensionType
             * @param {ScatterAxis.java} axisData
             * @param {Number} afterFilterRecords
             * @return {Object} extent as {values: [], min: min, max: max}
             */
            getUnaggregatedAxisExtentByType: function(uaDimensionType, axisData, afterFilterRecords) {
                switch (uaDimensionType) {
                    case 'str':
                        return {
                            values: angular.copy(axisData.str.sortedMapping)
                                .sort(function(a, b) {
                                    return d3.ascending(a.sortOrder, b.sortOrder);
                                })
                                .map(Fn.prop('label'))
                        };
                    case 'num':
                    case 'ts':
                        return {
                            values: axisData[uaDimensionType].data.filter((d, i) => i < afterFilterRecords),
                            min: axisData[uaDimensionType].min,
                            max: axisData[uaDimensionType].max
                        };
                    default:
                        throw new Error('Unhandled dimension type: ' + uaDimensionType);
                }
            },
            /**
             * Returns the min, max, and list of values on the given axis
             * @param {NADimensionDef.java} dimension
             * @param {ScatterAxis.java} axisData
             * @param {Number} afterFilterRecords
             * @return {Object} extent as {values: [], min: min, max: max}
             */
            getUnaggregatedAxisExtent: function(dimension, axisData, afterFilterRecords) {
                const uaDimensionType = ChartUADimension.getUnaggregatedDimensionType(dimension);
                return svc.getUnaggregatedAxisExtentByType(uaDimensionType, axisData, afterFilterRecords);
            },

            /**
             * Returns the min, max, and list of values on many axes
             * @param {dimension: NADimensionDef.java, data: ScatterAxis.java} axis
             * @param {Number} afterFilterRecords
             * @return {Object} extent as {values: [], min: min, max: max}
             */
            getUnaggregatedMultipleAxisExtent: function(axes, afterFilterRecords) {
                const finalExtent = {
                    min: undefined,
                    max: undefined,
                    values: []
                };
                axes.forEach(axis => {
                    const uaDimensionType = ChartUADimension.getUnaggregatedDimensionType(axis.dimension);
                    const extent = svc.getUnaggregatedAxisExtentByType(uaDimensionType, axis.data, afterFilterRecords);
                    if (_.isNil(finalExtent.min) || extent.min < finalExtent.min) {
                        finalExtent.min = extent.min;
                    }
                    if (_.isNil(finalExtent.max) || extent.max > finalExtent.max) {
                        finalExtent.max = extent.max;
                    }
                    finalExtent.values = [...finalExtent.values, ...extent.values];
                });
                return finalExtent;
            },

            /**
             * @param   {PivotResponse.java}    pivotResponse
             * @param   {ChartType.java}        chartType
             * @param   {string | undefined}    clickActionMessage
             * @returns the summary for the current sampling and count of records after filtering.
             */
            getSamplingSummaryMessage: function(pivotResponse, chartType, clickActionMessage, visibleCountMap) {
                const sampleMetadata = pivotResponse.sampleMetadata;

                if (!sampleMetadata) {
                    return;
                }
                let samplingSummaryMessage = `<i class="dku-icon-warning-fill-16 tooltip-icon"></i> ${translate('CHARTS.HEADER.DATA_SAMPLING_NOT_REPRESENTATIVE', 'Data sampling can be misrepresentative')}</br>`;

                if (clickActionMessage) {
                    samplingSummaryMessage += `${clickActionMessage}</br>`;
                }

                if (chartType === CHART_TYPES.SCATTER_MULTIPLE_PAIRS) {
                    return samplingSummaryMessage + svc.getScatterMPRecordsSummary(pivotResponse.axesPairs, visibleCountMap);
                } else if (sampleMetadata.datasetRecordCount > -1) {
                    samplingSummaryMessage += translate('CHARTS.HEADER.DATA_SAMPLING.ROWS_OUT_OF', '<strong>{{beforeFilterRecords}}</strong> rows out of {datasetRecordCount}} {{estimated ? \'(estimated)\' : \'\'',
                        { beforeFilterRecords: longSmartNumber(pivotResponse.beforeFilterRecords), datasetRecordCount: longSmartNumber(sampleMetadata.datasetRecordCount), estimated: sampleMetadata.recordCountIsApproximate });
                } else {
                    samplingSummaryMessage += translate('CHARTS.HEADER.DATA_SAMPLING.ROWS', '<strong>{{beforeFilterRecords}}</strong> rows', { beforeFilterRecords: longSmartNumber(pivotResponse.beforeFilterRecords) });
                }
                return samplingSummaryMessage;
            },

            getScatterMPRecordsSummary(axesPairs, visibleCountMap) {
                return (axesPairs || []).map((axisPair, index) => {
                    const value = visibleCountMap ? visibleCountMap.get(index) || 0 : axisPair.afterFilterRecords;
                    return translate('CHARTS.HEADER.DATA_SAMPLING.SCATTER', '<strong>Pair {{index}}</strong>: {{value}} record{{ value !== 1 ? \'s\' : \'\'}} after filtering', { index: index + 1, value: value });
                }).join('</br>');
            },

            /**
             * @param   {PivotResponse.java}    pivotResponse
             * @param   {ChartType.java}        chartType
             * @param   {string | undefined}    clickActionMessage
             * @returns The sample metadata, the sampling summary message, and the clickable summary message to be shared in the UI if necessary (else null).
             */
            getSampleMetadataAndSummaryMessage: (pivotResponse, chartType, clickActionMessage) => {
                const sampleMetadata = pivotResponse && pivotResponse.sampleMetadata;

                if (sampleMetadata) {
                    return {
                        sampleMetadata,
                        summaryMessage: !sampleMetadata.sampleIsWholeDataset ? svc.getSamplingSummaryMessage(pivotResponse, chartType) : null,
                        clickableSummaryMessage: !sampleMetadata.sampleIsWholeDataset && clickActionMessage ? svc.getSamplingSummaryMessage(pivotResponse, chartType, clickActionMessage) : null
                    };
                }

                return { sampleMetadata: null, summaryMessage: null, clickableSummaryMessage: null };
            },

            getRecordsFinalCountTooltip: function(chartType, afterFilterRecords, axisPairs, visibleCountMap) {
                if (chartType === CHART_TYPES.SCATTER_MULTIPLE_PAIRS) {
                    return svc.getScatterMPRecordsSummary(axisPairs, visibleCountMap);
                } else {
                    return `<strong>${afterFilterRecords}</strong> ${translate('CHARTS.HEADER.RECORDS_REMAIN', 'records remain after applying filtering')}`;
                }
            },

            /**
             * Computes the label that will be displayed on the top right of the chart.
             */
            computeRecordsStatusLabel: function(beforeFilterRecords, afterFilterRecords, computedMainAutomaticBinningModeDescription, visibleCountMap, chartType) {
                const result = [];
                const labelForRecordsCount = getLabelForRecordsCount(beforeFilterRecords, afterFilterRecords, visibleCountMap, chartType);

                if (labelForRecordsCount) {
                    result.push(labelForRecordsCount);
                }

                if (computedMainAutomaticBinningModeDescription && labelForRecordsCount !== ChartLabels.getNoRecordLabel()) {
                    result.push(computedMainAutomaticBinningModeDescription);
                }
                return result.length === 0 ? undefined : result.join(' ');
            },

            computeNoRecordsTopRightLabel: function() {
                return this.computeRecordsStatusLabel(0, 0, undefined);
            },
            /**
             * Computes the display settings for the specified interval.
             * If a main identical part is identified also computed the date to display as <b>formattedMainDate</b>
             */
            computeDateDisplayUnit: function(minTimestamp, maxTimestamp) {
                const dateDisplayUnit = getDateDisplayUnit(minTimestamp, maxTimestamp);
                if (minTimestamp !== undefined && dateDisplayUnit.mainDateFormat !== undefined) {
                    return { ...dateDisplayUnit, formattedMainDate: dateDisplayUnit.formatDateFn(minTimestamp, dateDisplayUnit.mainDateFormat) };
                }
                return dateDisplayUnit;
            },
            isSameDay(minTimestamp, maxTimestamp) {
                if (minTimestamp === undefined || maxTimestamp === undefined) {
                    return false;
                }

                const minDateWrapper = moment.utc(minTimestamp);
                const maxDateWrapper = moment.utc(maxTimestamp);
                return minDateWrapper.year() === maxDateWrapper.year() && minDateWrapper.dayOfYear() === maxDateWrapper.dayOfYear();
            },
            isPivotRequestAborted: (errorData) => errorData && errorData.hasResult && errorData.aborted
        };

        return svc;
    }
})();

;
(function() {
    'use strict';
    /** @typedef {import('../../types').ChartDef} ChartDef */
    /** @typedef {import('../../types').GeneratedSources.PivotTableTensorRPivotTableRequestesponse} PivotTableRequest */

    const app = angular.module('dataiku.charts');

    /**
     * Service responsible for computing pivot requests by chart type compatible with the get-pivot-response api.
     * (!) This file previously was in static/dataiku/js/simple_report/chart_compute_requests.js
     */
    app.service('ChartRequestComputer', function(ChartDimension, LoggerProvider, BinnedXYUtils, _GeometryCommon, ChartColorUtils, ChartStoreFactory, CHART_TYPES, CHART_VARIANTS, ChartUADimension, MultiplotUtils, ChartUsableColumns, ChartCustomMeasures, ColumnAvailability, ChartFeatures, ColorMode, ChartHierarchyDimension) {
        const Logger = LoggerProvider.getLogger('chartrequest');

        /* Helpers */

        const has = function(array) {
            return array && array.length > 0;
        };

        const makeAggregatedAxis = function(dim, chartDef, isMainInteractiveDateAxis = false) {
            dim = angular.copy(dim);
            const axis = {
                column: dim.column,
                type: dim.type,
                sortPrune: {},
                name: dim.name,
                isRBND: dim.isRBND
            };

            if (dim.type == 'NUMERICAL' && dim.numParams.mode != 'TREAT_AS_ALPHANUM') {
                axis.numParams = dim.numParams;
            } else if (dim.type == 'NUMERICAL' && dim.numParams.mode == 'TREAT_AS_ALPHANUM') {
                dim.type = 'ALPHANUM';
                axis.type = dim.type;
            } else if (dim.type === 'DATE' && dim.dateParams.mode != 'TREAT_AS_ALPHANUM') {
                axis.dateParams = ChartDimension.buildDateParamsForAxis(dim, chartDef.type, isMainInteractiveDateAxis);
            } else if (dim.type === 'DATE' && dim.dateParams.mode == 'TREAT_AS_ALPHANUM') {
                dim.type = 'ALPHANUM';
                axis.type = dim.type;
            }

            if (dim.type == 'ALPHANUM' || ChartDimension.isUngroupedNumerical(dim)) {
                axis.sortPrune.maxValues = dim.maxValues;
            }

            axis.sortPrune.sortType = dim.sort.type;
            if (dim.sort.type == 'AGGREGATION') {
                axis.sortPrune.aggregationSortId = dim.sort.measureIdx;
                axis.sortPrune.sortAscending = dim.sort.sortAscending;
            } else if (dim.sort.type == 'COUNT') {
                axis.sortPrune.sortAscending = dim.sort.sortAscending;
            } else {
                axis.sortPrune.sortAscending = true; // NATURAL => ASC !
            }

            axis.sortPrune.generateOthersCategory = dim.generateOthersCategory;
            axis.sortPrune.filters = dim.filters;

            if (!ChartFeatures.supportSorting(dim, chartDef, !isMainInteractiveDateAxis)) {
                axis.sortPrune = { sortType: 'NATURAL', sortAscending: true };
            }

            return axis;
        };

        const measureToAggregation = function(measure, id) {
            const ret = {
                ..._.pick(measure, ['column', 'type', 'isUnaggregated', 'uaComputeMode', 'function', 'computeModeDim', 'customFunction', 'percentile']),
                computeMode: measure.computeMode || 'NORMAL',
                id
            };
            return ret;
        };


        /**
         * Populates the request.aggregations array with all relevant measures used across the chart.
         *
         * Important note:
         * The tensor `counts` returned by the backend depends on the full list of requested aggregations.
         * As a result, we assume that a non-zero count means the bin is visible somewhere in the chart context.
         *
         * This means all aggregations added to `request.aggregations` must be strongly related to each other.
         * If unrelated aggregations are mixed in, the resulting counts may incorrectly suggest visibility
         * and lead to misleading or incorrect visual behavior.
         *
         * Always ensure that any added aggregation is relevant to the chart's current grouping context.
         */
        const addAggregations = function(request, chartDef, colorMeasures) {
            const chartWrapperElts = angular.element(document.querySelector('.chart-wrapper-zone'));
            const chartWrapperHeight = chartWrapperElts?.[0]?.offsetHeight;
            request.aggregations = chartDef.genericMeasures.map(m => {
                const res = measureToAggregation(m);
                if (m.isUnaggregated && chartWrapperHeight) {
                    //for unaggregated measure, we limit the number of values per tensor element to the chart wrapper height (no need to fetech more, as we couldn't display them correctly anyway)
                    res.maxUAValuesPerTensorElement = Math.floor(chartWrapperHeight);
                }
                return res;
            })
                .concat(chartDef.tooltipMeasures.map(measureToAggregation))
                .concat(colorMeasures)
                .concat(ChartFeatures.canDisplayTotalValues(chartDef) && chartDef.stackedColumnsOptions
                    && chartDef.stackedColumnsOptions.totalsInChartDisplayOptions
                    && chartDef.stackedColumnsOptions.totalsInChartDisplayOptions.additionalMeasures
                    ? chartDef.stackedColumnsOptions.totalsInChartDisplayOptions.additionalMeasures.map(measureToAggregation) : [])
                .concat(_.flatten(chartDef.genericMeasures.map(m =>
                    m.valuesInChartDisplayOptions && m.valuesInChartDisplayOptions.additionalMeasures
                        ? m.valuesInChartDisplayOptions.additionalMeasures.map(measureToAggregation)
                        : [])));
            request.count = true;
        };

        const addUa = function(request, ua, id) {
            const ret = {
                id: id,
                column: ua.column,
                type: ua.type
            };

            if (ua.treatAsAlphanum && ua.type == 'NUMERICAL') {
                ret.type = 'ALPHANUM';
            } else if (ua.type == 'DATE') {
                ret.dateMode = ua.dateMode;
            }
            request.columns.push(ret);
        };

        /**
         * Adds filters to the given request object from the chart definition filters.
         *
         * @param {PivotTableRequest}  request
         * @param {ChartDef}           chartDef
         */
        const addFilters = function(request, chartDef) {
            request.filters = angular.copy(chartDef.filters).map((filter) => {
                if (filter.selectedValues) {
                    filter.selectedValues = Object.keys(filter.selectedValues);
                }
                if (filter.excludedValues) {
                    filter.excludedValues = Object.keys(filter.excludedValues);
                }

                return filter;
            });
        };

        const clipLeaflet = function(chartSpecific, request) {
            if (chartSpecific.leafletMap) {
                const bounds = chartSpecific.leafletMap.getBounds();
                request.minLon = bounds.getWest();
                request.maxLon = bounds.getEast();
                request.minLat = bounds.getSouth();
                request.maxLat = bounds.getNorth();
            }
        };

        const getScatterAxis = function(dim) {
            const ret = {
                column: dim.column,
                type: dim.type
            };

            if (dim.treatAsAlphanum && dim.type == 'NUMERICAL') {
                ret.type = 'ALPHANUM';
            }
            if (ret.type == 'ALPHANUM') {
                ret.sortPrune = {
                    sortType: dim.sortBy || 'COUNT'
                };
                if (dim.sortBy == 'COUNT') {
                    ret.sortPrune.sortAscending = false;
                } else {
                    ret.sortPrune.sortAscending = true;
                }
            } else if (ret.type == 'DATE') {
                ret.dateMode = dim.dateMode;
            }
            return ret;
        };

        /* Handling of chart types */

        const computeScatter = function(chartDef, chartSpecific) {
            Logger.info('Compute scatter request for', chartDef);
            const request = {};
            request.type = 'SCATTER_NON_AGGREGATED';

            request.maxRows = !_.isNil(chartDef.scatterOptions.numberOfRecords) ? chartDef.scatterOptions.numberOfRecords : 1000000;
            request.xAxis = getScatterAxis(chartDef.uaXDimension[0]);
            request.yAxis = getScatterAxis(chartDef.uaYDimension[0]);
            request.columns = [];

            if (has(chartDef.uaSize)) {
                addUa(request, chartDef.uaSize[0], 'size');
            }

            if (has(chartDef.uaColor)) {
                addUa(request, chartDef.uaColor[0], 'color');
            }

            if (has(chartDef.uaShape)) {
                chartDef.uaShape[0].type = 'ALPHANUM';
                addUa(request, chartDef.uaShape[0], 'shape');
            }

            chartDef.uaTooltip.forEach(function(ua, idx) {
                addUa(request, ua, 'tooltip_' + idx);
            });

            addFilters(request, chartDef);
            addSideRequests(chartDef, request, [chartDef.uaXDimension[0], chartDef.uaYDimension[0]], chartSpecific, false);
            return request;
        };

        const computeScatterMultiplePairs = function(chartDef, chartSpecific) {
            Logger.info('Compute scatter request for', chartDef);
            const request = {};
            request.type = 'SCATTER_MULTIPLE_PAIRS_NON_AGGREGATED';

            request.maxRows = !_.isNil(chartDef.scatterMPOptions.numberOfRecords) ? chartDef.scatterMPOptions.numberOfRecords : 1000000;
            request.axisPairs = [];
            request.columns = [];

            const displayablePairs = ChartUADimension.getDisplayableDimensionPairs(chartDef.uaDimensionPair);
            const usedDimensions = [];

            displayablePairs.forEach(function(pair) {
                const uaXDimension = ChartUADimension.getPairUaXDimension(displayablePairs, pair);
                const xAxis = getScatterAxis(uaXDimension[0]);
                const yAxis = getScatterAxis(pair.uaYDimension[0]);
                const axisPair = { xAxis, yAxis };

                request.axisPairs.push(axisPair);

                usedDimensions.push(uaXDimension[0]);
                usedDimensions.push(pair.uaYDimension[0]);
            });


            chartDef.uaTooltip.forEach(function(ua, idx) {
                addUa(request, ua, 'tooltip_' + idx);
            });

            addFilters(request, chartDef);
            addSideRequests(chartDef, request, usedDimensions, chartSpecific, false);
            return request;
        };

        const computeAdminMap = function(chartDef, chartSpecific) {
            const request = {};
            const dim0 = chartDef.geometry[0];

            request.type = 'AGGREGATED_GEO_ADMIN';
            request.geoColumn = dim0.column;

            if (!dim0.adminLevel) {
                dim0.adminLevel = 2;
            }
            request.adminLevel = dim0.adminLevel;
            request.filled = chartDef.variant == 'filled_map';

            clipLeaflet(chartSpecific, request);

            request.aggregations = [];
            if (has(chartDef.colorMeasure)) {
                const a = measureToAggregation(chartDef.colorMeasure[0]);
                a.id = 'color';
                request.aggregations.push(a);
            }
            if (has(chartDef.sizeMeasure)) {
                const a = measureToAggregation(chartDef.sizeMeasure[0]);
                a.id = 'size';
                request.aggregations.push(a);
            }

            request.aggregations = request.aggregations.concat(chartDef.tooltipMeasures.map(measureToAggregation));

            // post-prune limit is handled backend-side for admin-map

            request.count = true;

            addFilters(request, chartDef);
            return request;
        };

        const computeGridMap = function(chartDef, chartSpecific) {
            const request = {};
            const dim0 = chartDef.geometry[0];

            request.type = 'AGGREGATED_GEO_GRID';
            request.geoColumn = dim0.column;

            clipLeaflet(chartSpecific, request);

            request.lonRadiusDeg = chartDef.mapGridOptions.gridLonDeg;
            request.latRadiusDeg = chartDef.mapGridOptions.gridLatDeg;

            request.aggregations = [];
            if (has(chartDef.colorMeasure)) {
                const a = measureToAggregation(chartDef.colorMeasure[0]);
                a.id = 'color';
                request.aggregations.push(a);
            }
            if (has(chartDef.sizeMeasure)) {
                const a = measureToAggregation(chartDef.sizeMeasure[0]);
                a.id = 'size';
                request.aggregations.push(a);
            }

            chartDef.tooltipMeasures.forEach(function(measure, idx) {
                const a = measureToAggregation(measure);
                a.id = 'tooltip_' + idx;
                request.aggregations.push(a);
            });

            // There is no post-prune limit for grid-map

            request.count = true;

            addFilters(request, chartDef);
            return request;
        };

        const _handleStandardSubchartsAndAnimation = function(chartDef, request) {
            if (has(chartDef.facetDimension)) {
                request.axes.push(makeAggregatedAxis(chartDef.facetDimension[0], chartDef));

                /* Safety limit: refuse to draw more than 200 subcharts */
                request.postPruneLimits.push({ limit: 200, axesToCheck: [request.axes.length - 1], template: 'Too many subcharts: %d, limit %d' });
            }

            if (has(chartDef.animationDimension)) {
                request.axes.push(makeAggregatedAxis(chartDef.animationDimension[0], chartDef));

                /* Safety limit: refuse to have more than 1000 elements in animation */
                request.postPruneLimits.push({ limit: 1000, axesToCheck: [request.axes.length - 1], template: 'Too many animation steps: %d, limit %d' });
            }
        };

        const getSideRequestsFromMeasures = function(chartDef, measures) {
            const sideRequests = [];
            const withFilters = measures.filter(m => !m.ignoreExistingFilters);
            const withoutFilters = measures.filter(m => m.ignoreExistingFilters);

            withFilters.length && sideRequests.push(computeKpi({ ...chartDef, genericMeasures: withFilters }));
            withoutFilters.length && sideRequests.push(computeKpi({ ...chartDef, genericMeasures: withoutFilters }, false));
            return sideRequests;
        };

        const addRequestsFromSource = function(chartDef, request, source_key, source, usedColumns, customMeasures, allMeasures) {
            const DATASET_COLUMN = 'DatasetColumn';
            const CUSTOM_AGG = 'CustomAggregation';

            if (source && !source.length) {
                source = [source];
            }

            const sourceMeasures = source && source.filter(measure => [DATASET_COLUMN, CUSTOM_AGG].includes(measure.sourceType))
                .map(measure => {
                    const measureMeta = ColumnAvailability.getDynamicMeasureMeta(measure, allMeasures, usedColumns, customMeasures);
                    return { measure, measureMeta };
                })
                .filter(({ measureMeta }) => measureMeta.index !== -1)
                .map(({ measure, measureMeta }) => ({
                    ...measureMeta.measure,
                    function: measure.sourceType === CUSTOM_AGG ? 'CUSTOM' : measure.aggregation,
                    percentile: measure.percentile,
                    ignoreExistingFilters: measure.ignoreExistingFilters
                })) || [];

            if (sourceMeasures.length) {
                request.sideRequests = {
                    ...request.sideRequests,
                    [source_key]: getSideRequestsFromMeasures(chartDef, sourceMeasures)
                };
            }
        };

        const addSideRequests = function(chartDef, request, usedColumns, chartSpecific) {
            const customMeasures = ChartCustomMeasures.getMeasuresLikeCustomMeasures(chartSpecific.datasetProjectKey, chartSpecific.datasetName, chartSpecific.context);
            const allMeasures = ChartUsableColumns.getUsableColumns(chartSpecific.datasetProjectKey, chartSpecific.datasetName, chartSpecific.context);

            const sideRequestsSources = {
                'REFERENCE_LINES': chartDef.referenceLines,
                'GAUGE_MAX': chartDef.gaugeOptions.max,
                'GAUGE_MIN': chartDef.gaugeOptions.min,
                'GAUGE_TARGETS': chartDef.gaugeOptions.targets,
                'GAUGE_RANGES': chartDef.gaugeOptions.ranges
            };

            Object.entries(sideRequestsSources).forEach(([source_key, source]) => addRequestsFromSource(chartDef, request, source_key, source, usedColumns, customMeasures, allMeasures));
        };

        const computeStdAggregated = function(chartDef, chartSpecific, computeSubTotals, chartHeight) {
            const request = { type: 'AGGREGATED_ND', axes: [], postPruneLimits: [] };

            /*
             * Global limit on global number of elements in the tensor.
             * This is a pure safety limit to avoid the risk of allocating too much JS memory to parse the tensor
             * or to draw a too crazy number of SVG elements.
             * There are additional limits on the first axis and on subcharts/animation where relevant
             */
            request.postPruneLimits.push({ limit: 100000 });

            let interactiveDateDimension;
            // Should not happen
            if (has(chartDef.genericDimension0) && has(chartDef.genericHierarchyDimension)) {
                throw new Error('The request cannot include both a hierarchy dimension and a standard dimension');
            }
            if (has(chartDef.genericDimension0) || has(chartDef.genericHierarchyDimension)) {
                const dimension = ChartHierarchyDimension.getCurrentHierarchyDimension(chartDef) || chartDef.genericDimension0[0];
                if (ChartDimension.isCandidateForInteractivity(dimension)) {
                    interactiveDateDimension = dimension;
                }
                request.axes.push(makeAggregatedAxis(dimension, chartDef, true));

                /*
                 * In the vast majority of aggregated charts, the axis 0 is the horizontal axis and non-scrolling.
                 * It does not make sense to have much more elements on it than the potential number of pixels
                 * on the screen.
                 * Exception is stacked_bars, for which axis 0 is vertical and scrolling, so can go above
                 */
                if (chartDef.type != CHART_TYPES.STACKED_BARS) {
                    request.postPruneLimits.push({ axesToCheck: [0], limit: 10000, template: 'Too many elements in axis: %d, limit %d' });
                }
            }

            if (has(chartDef.genericDimension1)) {
                const dimension = chartDef.genericDimension1[0];
                request.axes.push(makeAggregatedAxis(dimension, chartDef, false));
            }

            _handleStandardSubchartsAndAnimation(chartDef, request);

            // Should not happen ?
            if (chartDef.genericMeasures.length === 0) {
                throw new Error('To finish your chart, please select what you want to display and drop it in the \'Show\' section');
            }

            addAggregations(request, chartDef, [], chartHeight);
            addFilters(request, chartDef);

            if (interactiveDateDimension && chartSpecific.zoomUtils && !chartSpecific.zoomUtils.disableZoomFiltering) {
                request.filters.push(ChartDimension.buildZoomRuntimeFilter(interactiveDateDimension, chartSpecific.zoomUtils));
            }

            if (chartSpecific.zoomUtils && chartSpecific.zoomUtils.sequenceId) {
                request.sequenceId = chartSpecific.zoomUtils.sequenceId;
            }
            request.computeSubTotals = computeSubTotals;

            addSideRequests(chartDef, request, chartDef.genericMeasures, chartSpecific);

            return request;
        };

        const computeHierarchical = function(chartDef) {
            const request = { type: 'AGGREGATED_ND', axes: [], postPruneLimits: [] };

            // Should not happen
            if (has(chartDef.xDimension) && has(chartDef.xHierarchyDimension)) {
                throw new Error('The request cannot include both a hierarchy dimension and a standard x dimension');
            }
            if (has(chartDef.yDimension) && has(chartDef.yHierarchyDimension)) {
                throw new Error('The request cannot include both a hierarchy dimension and a standard y dimension');
            }

            if (has(chartDef.xHierarchyDimension)) {
                request.axes.push(makeAggregatedAxis(ChartHierarchyDimension.getCurrentHierarchyDimension(chartDef, 'x'), chartDef));
            }
            if (has(chartDef.xDimension)) {
                request.axes.push(...chartDef.xDimension.map(xDim => makeAggregatedAxis(xDim, chartDef)));
            }

            if (has(chartDef.yHierarchyDimension)) {
                request.axes.push(makeAggregatedAxis(ChartHierarchyDimension.getCurrentHierarchyDimension(chartDef, 'y'), chartDef));
            }
            if (has(chartDef.yDimension)) {
                request.axes.push(...chartDef.yDimension.map(yDim => makeAggregatedAxis(yDim, chartDef)));
            }

            request.aggregations = [];
            let colorAggs = [];
            if ((!ChartFeatures.canHaveConditionalFormatting(chartDef.type) || chartDef.colorMode === ColorMode.UNIQUE_SCALE)
                && has(chartDef.colorMeasure)) {
                colorAggs = measureToAggregation(chartDef.colorMeasure[0], 'color');
            } else if (chartDef.colorMode === ColorMode.COLOR_GROUPS) {
                colorAggs = chartDef.colorGroups
                    .filter(group => has(group.colorMeasure))
                    .map((group, i) => measureToAggregation(group.colorMeasure[0], `color_${i}`))
                ;
            }

            addAggregations(request, chartDef, colorAggs);

            addFilters(request, chartDef);
            request.computeSubTotals = true;

            const { store, id } = ChartStoreFactory.getOrCreate(chartDef.$chartStoreId);
            chartDef.$chartStoreId = id;
            const requestOptions = store.getRequestOptions();

            if (!requestOptions.removePivotTableLimitations) {
                /* Global limit on global number of elements in the post-prune tensor. */
                request.postPruneLimits.push({ limit: 200000 });
            }

            return request;
        };

        const computeBinnedXY = function(chartDef, width, height) {
            const request = { type: 'AGGREGATED_ND', aggregations: [], postPruneLimits: [] };

            /*
             * Global limit on global number of elements in the tensor.
             * This is a pure safety limit to avoid the risk of allocating too much JS memory to parse the tensor
             * or to draw a too crazy number of SVG elements.
             */
            request.postPruneLimits.push({ limit: 100000 });

            // Should not happen
            if (has(chartDef.xDimension) && has(chartDef.xHierarchyDimension)) {
                throw new Error('The request cannot include both a hierarchy dimension and a standard dimension on the X axis');
            }
            if (has(chartDef.yDimension) && has(chartDef.yHierarchyDimension)) {
                throw new Error('The request cannot include both a hierarchy dimension and a standard dimension on the Y axis');
            }

            const xDimension = ChartDimension.getXDimension(chartDef);
            const yDimension = ChartDimension.getYDimension(chartDef);

            request.axes = [makeAggregatedAxis(xDimension, chartDef), request.yAxis = makeAggregatedAxis(yDimension, chartDef)];

            if (has(chartDef.sizeMeasure)) {
                request.aggregations.push(measureToAggregation(chartDef.sizeMeasure[0], 'size'));
            }
            if (has(chartDef.colorMeasure)) {
                request.aggregations.push(measureToAggregation(chartDef.colorMeasure[0], 'color'));
            }

            chartDef.tooltipMeasures.forEach(function(measure) {
                request.aggregations.push(measureToAggregation(measure, 'tooltip'));
            });

            _handleStandardSubchartsAndAnimation(chartDef, request);

            addFilters(request, chartDef);

            if (chartDef.variant == CHART_VARIANTS.binnedXYHexagon) {
                request.hexbin = true;
                const margins = { top: 10, right: 50, bottom: 50, left: 50 };
                const chartWidth = width - margins.left - margins.right;
                const chartHeight = height - margins.top - margins.bottom;
                const radius = BinnedXYUtils.getRadius(chartDef, chartWidth, chartHeight);
                request.hexbinXHexagons = Math.floor(chartWidth / (2 * Math.cos(Math.PI / 6) * radius));
                request.hexbinYHexagons = Math.floor(chartHeight / (1.5 * radius));
                request.$expectedVizWidth = chartWidth;
                request.$expectedVizHeight = chartHeight;
            }

            return request;
        };

        const computeGroupedXY = function(chartDef) {
            // Should not happen
            if (has(chartDef.groupDimension) && has(chartDef.groupHierarchyDimension)) {
                throw new Error('The request cannot include both a hierarchy dimension and a standard group dimension');
            }

            const request = {
                type: 'AGGREGATED_ND',
                axes: [makeAggregatedAxis(ChartDimension.getGroupDimension(chartDef), chartDef)],
                postPruneLimits: []
            };

            /*
             * Global limit on global number of elements in the tensor.
             * This is a pure safety limit to avoid the risk of allocating too much JS memory to parse the tensor
             * or to draw a too crazy number of SVG elements.
             */
            request.postPruneLimits.push({ limit: 100000 });

            _handleStandardSubchartsAndAnimation(chartDef, request);

            request.aggregations = [
                measureToAggregation(chartDef.xMeasure[0]),
                measureToAggregation(chartDef.yMeasure[0])
            ];
            if (has(chartDef.sizeMeasure)) {
                request.aggregations.push(measureToAggregation(chartDef.sizeMeasure[0], 'size'));
            }
            if (has(chartDef.colorMeasure)) {
                request.aggregations.push(measureToAggregation(chartDef.colorMeasure[0], 'color'));
            }

            addFilters(request, chartDef);

            return request;
        };

        const computeLift = function(chartDef) {
            // Should not happen
            if (has(chartDef.groupDimension) && has(chartDef.groupHierarchyDimension)) {
                throw new Error('The request cannot include both a hierarchy dimension and a standard group dimension');
            }

            const request = {
                type: 'AGGREGATED_ND',
                axes: [makeAggregatedAxis(ChartDimension.getGroupDimension(chartDef), chartDef)],
                postPruneLimits: []
            };

            /* For lift, it does not make sense to have more elements overall than horizontal pixels */
            request.postPruneLimits.push({ limit: 10000 });

            _handleStandardSubchartsAndAnimation(chartDef, request);

            request.aggregations = [
                measureToAggregation(chartDef.xMeasure[0]),
                measureToAggregation(chartDef.yMeasure[0])
            ];

            addFilters(request, chartDef);

            return request;
        };

        const computeBoxplots = function(chartDef) {
            const request = {
                type: 'BOXPLOTS',
                axes: []
            };
            // Should not happen
            if (has(chartDef.boxplotBreakdownDim) && has(chartDef.boxplotBreakdownHierarchyDimension)) {
                throw new Error('The request cannot include both a hierarchy dimension and a standard breakdown dimension');
            }
            const breakdownDimension = ChartDimension.getBreakdownDimension(chartDef);

            if (breakdownDimension) {
                request.axes.push(makeAggregatedAxis(breakdownDimension, chartDef));
            }
            if (breakdownDimension && has(chartDef.genericDimension1)) {
                request.axes.push(makeAggregatedAxis(chartDef.genericDimension1[0], chartDef));
            }
            request.column = {
                column: chartDef.boxplotValue[0].column,
                type: chartDef.boxplotValue[0].type
            };
            addFilters(request, chartDef);

            // post-prune limit is handled backend-side for boxplots

            return request;
        };

        const computeDensity2D = function(chartDef) {
            const request = {};
            request.type = 'DENSITY_2D';
            request.xColumn = chartDef.xDimension[0].column;
            request.yColumn = chartDef.yDimension[0].column;
            addFilters(request, chartDef);

            // No post-prune-limit for density-2D

            return request;
        };

        const computeKpi = function(chartDef, withFilters = true) {
            const request = {};
            request.type = 'NO_PIVOT_AGGREGATED';

            const colorAggs = chartDef.colorGroups
                .filter(group => has(group.colorMeasure))
                .map((group, i) => measureToAggregation(group.colorMeasure[0], `color_${i}`))
            ;

            addAggregations(request, chartDef, colorAggs);
            withFilters && addFilters(request, chartDef);

            // No post-prune-limit for density-2D

            return request;
        };

        const computeGauge = function(chartDef, chartSpecific) {
            const request = {};
            request.type = 'NO_PIVOT_AGGREGATED';
            addAggregations(request, chartDef, []);

            addFilters(request, chartDef);
            let rangesAndTargetsList = [];
            if (chartDef.gaugeOptions.targets) {
                rangesAndTargetsList = rangesAndTargetsList.concat(chartDef.gaugeOptions.targets);
            }
            if (chartDef.gaugeOptions.ranges) {
                rangesAndTargetsList = rangesAndTargetsList.concat(chartDef.gaugeOptions.ranges);
            }

            const usedDimensions = [
                chartDef.gaugeOptions.min,
                chartDef.gaugeOptions.max,
                ...rangesAndTargetsList
            ];
            addSideRequests(chartDef, request, usedDimensions, chartSpecific, false);

            return request;
        };

        const computeScatterMap = function(chartDef, chartSpecific) {
            const request = {};

            request.type = 'MAP_SCATTER_NON_AGGREGATED';
            request.maxRows = 100000;

            request.geoColumn = chartDef.geometry[0].column;

            clipLeaflet(chartSpecific, request);

            request.columns = [];
            if (has(chartDef.uaSize)) {
                addUa(request, chartDef.uaSize[0], 'size');
            }
            if (has(chartDef.uaColor)) {
                addUa(request, chartDef.uaColor[0], 'color');
            }
            chartDef.uaTooltip.forEach(function(ua, idx) {
                addUa(request, ua, 'tooltip_' + idx);
            });

            addFilters(request, chartDef);
            return request;
        };

        const computeDensityHeatMap = function(chartDef, chartSpecific) {
            // Temporary solution until definition of a new request.type
            return computeScatterMap(chartDef, chartSpecific);
        };

        const computeGeometryMap = function(chartDef, chartSpecific) {
            const request = {};

            request.type = 'RAW_GEOMETRY';
            request.maxRows = 100000;

            request.geoColumns = [];

            request.columns = [];

            const displayedGeoLayers = _GeometryCommon.getDisplayableLayers(chartDef.geoLayers);

            displayedGeoLayers.forEach(function(geoLayer, index) {
                const geometry = geoLayer.geometry[0];
                const geoColumn = { column: geometry.column };
                if (geometry.aggregationFunction) {
                    geoColumn.aggregationFunction = geometry.aggregationFunction;
                }
                request.geoColumns.push(geoColumn);
                if ((has(geoLayer.uaColor)) && (geoLayer.uaColor[0].column)) {
                    addUa(request, geoLayer.uaColor[0], ChartColorUtils.getPaletteName(index));
                }
            });
            chartDef.uaTooltip.forEach(function(ua, idx) {
                addUa(request, ua, 'tooltip_' + idx);
            });

            addFilters(request, chartDef);
            return request;
        };

        const computeWebapp = function(chartDef, chartSpecific) {
            const request = {};
            request.type = 'WEBAPP';
            request.columns = [];
            addFilters(request, chartDef);
            return request;
        };

        const computeFilters = function(chartDef) {
            const request = {
                type: 'FILTERS',
                columns: []
            };
            addFilters(request, chartDef);
            return request;
        };

        const svc = {
            addFilters,

            /**
             * Computes the corresponding pivot request from the given chart def
             *
             * @param {ChartDef} chartDef
             * @param {number} width
             * @param {number} height
             * @param {Record<string, unknown>} chartSpecific
             *
             * @return {PivotTableRequest.java}
             */
            compute: function(chartDef, width, height, chartSpecific) {
                // If the chart is facetted, the height the charts is different from the original height
                if (chartDef.facetDimension && chartDef.facetDimension.length) {
                    height = chartDef.chartHeight;
                }

                switch (chartDef.type) {

                    case CHART_TYPES.GROUPED_COLUMNS:
                    case CHART_TYPES.STACKED_COLUMNS:
                    case CHART_TYPES.STACKED_BARS:
                    case CHART_TYPES.LINES:
                    case CHART_TYPES.STACKED_AREA:
                    case CHART_TYPES.RADAR:
                    case CHART_TYPES.PIE:
                        return computeStdAggregated(chartDef, chartSpecific, false, height);

                    case CHART_TYPES.MULTI_COLUMNS_LINES: {
                        const computeSubTotals = MultiplotUtils.shouldSubtotalsBeCalculated(chartDef);
                        const modifiedChartDef = MultiplotUtils.shouldTakeColorDimensionInAccount(chartDef) ? chartDef : { ...chartDef, genericDimension1: [] };
                        return computeStdAggregated(modifiedChartDef, chartSpecific, computeSubTotals, height);
                    }
                    case CHART_TYPES.PIVOT_TABLE:
                        if (!has(chartDef.xDimension) && !has(chartDef.xHierarchyDimension) && !has(chartDef.yDimension) && !has(chartDef.yHierarchyDimension)) {
                            return computeKpi(chartDef);
                        }
                        return computeHierarchical(chartDef);
                    case CHART_TYPES.SANKEY:
                    case CHART_TYPES.TREEMAP:
                        return computeHierarchical(chartDef);

                    case CHART_TYPES.BINNED_XY:
                        return computeBinnedXY(chartDef, width, height);

                    case CHART_TYPES.GROUPED_XY:
                        return computeGroupedXY(chartDef);

                    case CHART_TYPES.SCATTER:
                        return computeScatter(chartDef, chartSpecific);

                    case CHART_TYPES.SCATTER_MULTIPLE_PAIRS:
                        return computeScatterMultiplePairs(chartDef, chartSpecific);

                    case CHART_TYPES.ADMINISTRATIVE_MAP:
                        return computeAdminMap(chartDef, chartSpecific);

                    case CHART_TYPES.GRID_MAP:
                        return computeGridMap(chartDef, chartSpecific);

                    case CHART_TYPES.SCATTER_MAP:
                        return computeDensityHeatMap(chartDef, chartSpecific);

                    case CHART_TYPES.DENSITY_HEAT_MAP:
                        return computeScatterMap(chartDef, chartSpecific);

                    case CHART_TYPES.GEOMETRY_MAP:
                        return computeGeometryMap(chartDef, chartSpecific);

                    case CHART_TYPES.BOXPLOTS:
                        return computeBoxplots(chartDef);

                    case CHART_TYPES.DENSITY_2D:
                        return computeDensity2D(chartDef);

                    case CHART_TYPES.GAUGE:
                        return computeGauge(chartDef, chartSpecific);

                    case CHART_TYPES.KPI:
                        return computeKpi(chartDef);

                    case CHART_TYPES.LIFT:
                        var req = computeLift(chartDef);
                        req.diminishingReturns = true;
                        return req;

                    case CHART_TYPES.WEBAPP:
                        return computeWebapp(chartDef, chartSpecific);

                    case CHART_TYPES.FILTERS:
                        return computeFilters(chartDef);

                    default:
                        Logger.error('Unhandled chart type', chartDef);
                        throw new Error('unknown chart type', chartDef);
                }
            }
        };
        return svc;
    });

})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.directives.simple_report');

    /**
     * Service responsible for parsing and errors returned by the backend.
     * (!) This service previously was in static/dataiku/js/simple_report/chart_logic.js
     */
    app.service('ChartSetErrorInScope', function() {
        function buildErrorValidityObject(message, showRevertEngineButton) {
            return {
                message,
                showRevertEngineButton,
                valid: false,
                type: 'COMPUTE_ERROR'
            };
        }

        function buildValidityForKnownError(data, status, headers) {
            const errorDetails = getErrorDetails(data, status, headers);
            if (errorDetails.errorType === 'com.dataiku.dip.pivot.backend.model.SecurityAbortedException') {
                return buildErrorValidityObject('Too many elements to draw. Please adjust chart settings (' + errorDetails.message + ')', false);
            } else if (errorDetails.errorType === 'ApplicativeException') {
                return buildErrorValidityObject(errorDetails.message, false);
            } else if (errorDetails.errorType === 'com.dataiku.dip.exceptions.EngineNotAvailableException') {
                return buildErrorValidityObject(errorDetails.message, true);
            }
            return undefined;
        }

        const svc = {
            buildValidityForKnownError: buildValidityForKnownError,
            defineInScope: function(scope) {
                if ('chartSetErrorInScope' in scope) {
                    return;
                } // already defined in a higher scope
                scope.validity = { valid: true };
                scope.setValidity = function(validity) {
                    scope.validity = validity;
                };
                scope.chartSetErrorInScope = function(data, status, headers) {
                    const validity = buildValidityForKnownError(data, status, headers);
                    if (validity) {
                        scope.validity = validity;
                    } else {
                        setErrorInScope.bind(scope)(data, status, headers);
                    }
                };
            }
        };

        return svc;
    });


})();

// eslint-disable-next-line no-redeclare
function ChartIAE(message) {
    this.message = message;
    this.name = 'ChartIAE';
}

ChartIAE.prototype = new Error;

;
(function() {
    'use strict';

    angular.module('dataiku.charts')
        .factory('_GeometryCommon', GeometryCommon);

    // (!) This service previously was in static/dataiku/js/simple_report/geometry/geometry_plot.js
    function GeometryCommon(ColorUtils, ChartColorScales, UaChartsCommon) {
        const svc = {
            hasUAColor: function(geoLayer) {
                return ((geoLayer.uaColor.length > 0) && (geoLayer.uaColor[0].column));
            },
            makeColorScale: function(geoLayer, data, chartHandler, geometryIndex) {
                const uaColor = geoLayer.uaColor[0];
                const colorTokenName = 'color_' + geometryIndex;
                const chartData = { data: data };
                const colorSpec = {
                    type: 'UNAGGREGATED',
                    dimension: uaColor,
                    data: data.values[colorTokenName],
                    withRgba: true
                };
                /** type: ChartColorContext */
                const colorContext = {
                    chartData,
                    colorOptions: geoLayer.colorOptions,
                    defaultLegendDimension: [],
                    colorSpec,
                    chartHandler,
                    uaColorIndex: geometryIndex,
                    theme: chartHandler.getChartTheme()
                };
                return ChartColorScales.createColorScale(colorContext);
            },
            makeSingleColor: function(geoLayer) {
                return ColorUtils.toRgba(geoLayer.colorOptions.singleColor, geoLayer.colorOptions.transparency);
            },

            makeColor: function(geoLayer, data, i, colorScale, resultingColor, colorCache, geometryIndex) {
                if (svc.hasUAColor(geoLayer)) {
                    const color = data.values['color_' + geometryIndex];
                    return ChartColorScales.getColor(geoLayer.uaColor, color, i, colorScale, colorCache);
                } else {
                    return resultingColor;
                }
            },
            formattedColorVal: function(chartDef, data, i, paletteName, geometryIndex) {
                let uaColor;
                if (typeof geometryIndex === 'undefined') {
                    uaColor = chartDef.uaColor[0];
                } else {
                    uaColor = chartDef.geoLayers[geometryIndex].uaColor[0];
                }
                paletteName = paletteName || 'color';
                return UaChartsCommon.formattedVal(data.values[paletteName], uaColor, i);
            },
            getDisplayableLayers: function(geoLayers) {
                let isLayerDisplayable;
                const displayableGeoLayers = [];
                for (const geoLayer of geoLayers.slice(0, -1)) {
                    isLayerDisplayable = (geoLayer.geometry && geoLayer.geometry.length && geoLayer.geometry[0].column);
                    if (isLayerDisplayable) {
                        displayableGeoLayers.push(geoLayer);
                    }
                }
                return displayableGeoLayers;
            }
        };
        return svc;
    }
})();

;
(function() {
    'use strict';

    angular.module('dataiku.charts')
        .factory('ChartLegendUtils', ChartLegendUtils);

    /**
     * (!) This service previously was in static/dataiku/js/simple_report/common/legends.js
     */
    function ChartLegendUtils(CreateCustomElementFromTemplate, $q, $timeout, ColorUtils, ChartColorUtils, ChartLabels, ChartDimension, ChartDataUtils, ChartFeatures, ChartFormatting, ChartUADimension, CHART_TYPES, CHART_AXIS_TYPES, ChartIconUtils) {
        const that = {
            initLegend: function(chartDef, chartData, legendsWrapper, colorScale, options) {
                if (!colorScale || !ChartFeatures.shouldComputeLegend(chartDef.type)) {
                    legendsWrapper.deleteLegends();
                    return;
                }

                // when using color groups should be called once per colorGroup
                if (chartDef.colorMode === 'COLOR_GROUPS' && ChartFeatures.canHaveConditionalFormatting(chartDef.type)) {
                    const legend = that.createContinuousLegend(ChartColorUtils.getColorDimensionOrMeasure(chartDef), colorScale);
                    legendsWrapper.pushLegend(legend);
                    return; // return because currently charts with color groups don't have legends outside groups with scale mode
                }

                switch (colorScale.type) {
                    case CHART_AXIS_TYPES.DIMENSION:
                        return that.initDimensionLegend(chartDef, chartData, legendsWrapper, colorScale, options);
                    case CHART_AXIS_TYPES.MEASURE:
                        return that.initMeasureLegend(chartDef, legendsWrapper, colorScale);
                    case CHART_AXIS_TYPES.UNAGGREGATED:
                        if (colorScale.isContinuous) {
                            return that.initMeasureLegend(chartDef, legendsWrapper, colorScale);
                        }
                        return;
                    case 'CUSTOM':
                        return that.initCustomLegend(chartDef, chartData, legendsWrapper, colorScale, options);
                    default:
                        throw new Error('Unknown scale type: ' + colorScale.type);
                }
            },

            initDimensionLegend: function(chartDef, chartData, legendsWrapper, colorScale, options = { hideLegend: false, ignoreLabels: new Set() }) {
                const { hideLegend, ignoreLabels } = options;
                const colorDimensionLabels = chartData.getAxisLabels('color');
                const defaultLegendDimension = ChartColorUtils.getDefaultLegendDimension(chartDef, chartData);
                const colorLabels = colorDimensionLabels || defaultLegendDimension;
                const colorDimension = ChartColorUtils.getColorDimensionOrMeasure(chartDef);

                const items = colorLabels.filter(colorLabel => !ignoreLabels || !ignoreLabels.has(colorLabel.label)).map(function(colorOrMeasure, c) {
                    const color = colorScale(c);

                    return {
                        label: {
                            colorId: ChartColorUtils.getColorId(defaultLegendDimension, chartData, c, undefined, ignoreLabels),
                            label: colorOrMeasure.label || chartData.getColorMeasureOrDimensionLabel(colorOrMeasure, chartDef.uaDimensionPair),
                            min: colorOrMeasure.min,
                            max: colorOrMeasure.max,
                            sortValue: colorOrMeasure.sortValue,
                            tsValue: colorOrMeasure.tsValue
                        },
                        color,
                        desaturatedColor: ColorUtils.desaturate(color),
                        rgbaColor: ColorUtils.toRgba(colorScale(c), ChartColorUtils.getTransparency(chartDef)),
                        focused: false,
                        unfocusFn: function() { /* focus/unfocus on mouseover */ },
                        focusFn: function() { /* focus/unfocus on mouseover */ },
                        id: c,
                        elements: d3.select()
                    };
                });


                legendsWrapper.deleteLegends();
                legendsWrapper.pushLegend(
                    {
                        type: 'COLOR_DISCRETE',
                        minValue: chartData.getMinValue('color'),
                        maxValue: chartData.getMaxValue('color'),
                        numValues: chartData.getNumValues('color'),
                        numberFormattingOptions: colorDimension && ChartDimension.getNumberFormattingOptions(colorDimension),
                        items,
                        hideLegend
                    }
                );
            },

            initMeasureLegend: function(chartDef, legendsWrapper, colorScale) {
                legendsWrapper.deleteLegends();
                const legend = that.createContinuousLegend(ChartColorUtils.getColorDimensionOrMeasure(chartDef), colorScale);
                legendsWrapper.pushLegend(legend);
            },

            initCustomLegend: function(chartDef, chartData, legendsWrapper, colorScale, options = { hideLegend: false, ignoreLabels: new Set() }) {
                const { hideLegend, ignoreLabels } = options;
                const colorDimensionLabels = chartData.getAxisLabels('color', ignoreLabels) || [];
                const genericMeasures = chartDef.genericMeasures.filter(colorScale.customHandle);
                const colorLabels = colorDimensionLabels.concat(genericMeasures.filter(colorLabel => !ignoreLabels.has(colorLabel.label)));
                const colorDimension = ChartColorUtils.getColorDimensionOrMeasure(chartDef);
                const items = colorLabels.map(function(colorOrMeasure, c) {
                    const color = colorScale(c);

                    return {
                        label: {
                            colorId: ChartColorUtils.getColorId(colorLabels, chartData, c, undefined, ignoreLabels, colorOrMeasure.isA === 'measure'),
                            label: colorOrMeasure.label || ChartLabels.getLongMeasureLabel(colorOrMeasure),
                            min: colorOrMeasure.min,
                            max: colorOrMeasure.max,
                            sortValue: colorOrMeasure.sortValue,
                            tsValue: colorOrMeasure.tsValue
                        },
                        color,
                        desaturatedColor: ColorUtils.desaturate(color),
                        rgbaColor: ColorUtils.toRgba(colorScale(c), chartDef.colorOptions.transparency),
                        focused: false,
                        unfocusFn: function() { /* focus/unfocus on mouseover */ },
                        focusFn: function() { /* focus/unfocus on mouseover */ },
                        id: c,
                        elements: d3.select()
                    };
                });

                legendsWrapper.deleteLegends();
                legendsWrapper.pushLegend({
                    type: 'COLOR_DISCRETE',
                    minValue: chartData.getMinValue('color'),
                    maxValue: chartData.getMaxValue('color'),
                    numValues: items.length,
                    numberFormattingOptions: colorDimension && ChartDimension.getNumberFormattingOptions(colorDimension),
                    items,
                    hideLegend
                });
            },

            drawLegend: function(chartDef, chartHandler, $container) {
                const deferred = $q.defer();
                const $legendZone = $container.find('.legend-zone');

                if (chartDef.legendPlacement === 'SIDEBAR') {
                    // remove the potential previously drawn legend
                    $legendZone.empty();
                    $container.attr('legend-placement', chartDef.legendPlacement);
                    deferred.resolve();
                    return deferred.promise;
                }

                CreateCustomElementFromTemplate('/templates/simple_report/legend/legend-zone.html', chartHandler, null, function() {
                    $timeout(deferred.resolve);
                }, function($el) {
                    $container.attr('legend-placement', chartDef.legendPlacement);

                    //  Used to avoid flickering when redrawing legends
                    if ($legendZone.length) {
                        //  Hack to set same style if legend zone is the same
                        $el.css({
                            'min-width': $legendZone.outerWidth(),
                            'overflow': 'hidden'
                        });

                        //  Removes style attribute to avoid conflict with current legend zone styles
                        $timeout(() => {
                            $legendZone.replaceWith($el);
                            $el.removeAttr('style');
                        });
                    } else {
                        $el.appendTo($container);
                    }
                });

                return deferred.promise;
            },

            /**
             * Create legends, init + draw legends
             * @param {Element} $container
             * @param {ChartDef.java} chartDef
             * @param {Object} chartData
             * @param {chart_views scope} chartHandler
             * @param {Object} colorScale
             * @param {{ ignoreLabels?: Set, hideLegend?: boolean }} extraOptions
             * @returns A promise
             */
            createLegend: ($container, chartDef, chartData, chartHandler, colorSpec, colorScale, extraOptions) => {
                // We need to draw the legend first, because (if it's set to be outside of the chart), its size will influence the remaining available width & height, that we need to know for the following
                that.initLegend(chartDef, chartData, chartHandler.legendsWrapper, colorScale, extraOptions);

                if (chartDef.type === CHART_TYPES.SCATTER) {
                    that.createScatterLegend(chartDef, chartHandler, chartData, colorScale);
                }

                // Create measure formatter for color legend
                if (chartHandler.legendsWrapper.hasLegends() && chartHandler.legendsWrapper.getLegend(0).type === 'COLOR_CONTINUOUS' && ChartFeatures.canDisplayLegend(chartDef.type)) {
                    // Depending on the chart type, the color dimension or measure is stored at different places in the chartDef.
                    const colorUaOrMeasure = chartDef.type === CHART_TYPES.SCATTER ? chartDef.uaColor[0] : chartDef.colorMeasure[0];
                    if (colorSpec.type === 'UNAGGREGATED') {
                        if (ChartUADimension.isAlphanumLike(colorUaOrMeasure) || ChartUADimension.isDiscreteDate(colorUaOrMeasure)) {
                            // chartHandler.legendsWrapper.getLegend(0).type should not be COLOR_CONTINUOUS ?
                        } else if (ChartUADimension.isDateRange(colorUaOrMeasure)) {
                            chartHandler.legendsWrapper.getLegend(0).formatter = ChartFormatting.getForDate();
                        } else {
                            chartHandler.legendsWrapper.getLegend(0).formatter = ChartFormatting.getForLegend(colorUaOrMeasure, colorScale.innerScale.domain());
                        }
                    } else {
                        const extent = colorSpec.extent || ChartDataUtils.getMeasureExtent(chartData, colorSpec.measureIdx, true);
                        chartHandler.legendsWrapper.getLegend(0).formatter = ChartFormatting.getForLegend(colorUaOrMeasure, extent);
                    }
                }
                return that.drawLegend(chartDef, chartHandler, $container);
            },


            adjustLegendPlacement: function(chartDef, $container, margins) {
                const $legendZone = $container.find('.legend-zone');

                const getEffectiveLeftMargin = function() {
                    if (chartDef.facetDimension.length) {
                        return $('.facet-info').width() + margins.left;
                    } else {
                        return margins.left;
                    }
                };

                const setMaxSize = function() {
                    $legendZone
                        .css('max-height', 'calc(100% - ' + (margins.top + margins.bottom) + 'px)')
                        .css('max-width', '25%')
                        .css('visibility', 'visible');
                };

                switch (chartDef.legendPlacement) {
                    case 'INNER_TOP_LEFT':
                        $legendZone.css('left', getEffectiveLeftMargin()).css('top', margins.top);
                        setMaxSize();
                        break;
                    case 'INNER_TOP_RIGHT':
                        $legendZone.css('right', margins.right).css('top', margins.top);
                        setMaxSize();
                        break;
                    case 'INNER_BOTTOM_LEFT':
                        $legendZone.css('left', getEffectiveLeftMargin()).css('bottom', margins.bottom);
                        setMaxSize();
                        break;
                    case 'INNER_BOTTOM_RIGHT':
                        $legendZone.css('right', margins.right).css('bottom', margins.bottom);
                        setMaxSize();
                        break;
                    default:
                        break;
                }
            },

            createContinuousLegend: function(colorDimension, colorScale) {
                const numberFormattingOptions = colorDimension && ChartDimension.getNumberFormattingOptions(colorDimension);
                colorScale.type = 'MEASURE';
                return {
                    type: 'COLOR_CONTINUOUS',
                    scale: colorScale,
                    numberFormattingOptions
                };
            },

            getSingleColorLegend: function(singleColor, label) {
                return {
                    type: 'COLOR_DISCRETE',
                    label: label,
                    items: [{ color: singleColor }]
                };
            },

            createScatterLegend: function(chartDef, chartHandler, chartData, colorScale) {
                const hasUAColor = chartData.hasUAColor(chartDef);
                const hasUAShape = chartData.hasUAShape(chartDef);
                const shapeScale = chartData.makeShapeScale();
                const hasColorLegend = (hasUAColor && (ChartUADimension.isAlphanumLike(chartDef.uaColor[0]) || ChartUADimension.isDiscreteDate(chartDef.uaColor[0])));

                if (hasUAShape || hasColorLegend) {
                    const legend = {
                        type: 'COLOR_DISCRETE',
                        items: []
                    };

                    if (hasColorLegend) {
                        colorScale.domain().forEach(v => {
                            const item = {
                                label: {
                                    colorId: ChartColorUtils.getColorId(chartDef.genericMeasures, chartData, v),
                                    ...chartData.data.values.color.str.sortedMapping[v]
                                },
                                color: colorScale(v),
                                focused: false
                            };
                            legend.items.push(item);
                        });
                    }

                    if (hasUAShape && hasColorLegend) {
                        legend.items.push({ separator: true });
                    }

                    if (hasUAShape) {
                        Object.values(chartData.data.values.shape.str.sortedMapping).forEach((v, index) => {
                            const shapeType = shapeScale(index).type;
                            const item = {
                                label: v,
                                shape: {
                                    svgSrc: ChartIconUtils.computeScatterLegendIcon(shapeType)
                                },
                                focused: false
                            };
                            legend.items.push(item);
                        });
                    }

                    chartHandler.legendsWrapper.deleteLegends();
                    chartHandler.legendsWrapper.pushLegend(legend);
                } else if (hasUAColor) {
                    // Done in initChart
                } else {
                    chartHandler.legendsWrapper.deleteLegends();
                }
            }
        };

        return that;
    }

})();

;
(function() {
    'use strict';

    angular.module('dataiku.charts')
        .factory('BarChartUtils', BarChartUtils);

    /**
     * (!) This service previously was in static/dataiku/js/simple_report/column-bars/bars.js
     */
    function BarChartUtils(ChartDimension, SVGUtils, ValuesInChartOverlappingStrategy) {

        return {
            getTranslateFunctions: function(axisName, axis, labels, thickness) {
                const dimension = axis.dimension;
                const hasOneTickPerBin = ChartDimension.hasOneTickPerBin(dimension);
                let translate;
                if (!hasOneTickPerBin && ChartDimension.isTimeline(dimension)) {
                    translate = (i) => axis.scale()((labels[i].min + labels[i].max) / 2) - thickness / 2;
                } else if (!hasOneTickPerBin && ChartDimension.isTrueNumerical(dimension)) {
                    translate = (i) => axis.scale()(labels[i].sortValue) - thickness / 2;
                } else {
                    translate = (i) => axis.ordinalScale(i);
                }
                switch (axisName) {
                    case 'x':
                        return {
                            x: (d, i) => translate(i),
                            y: () => 0
                        };
                    case 'y':
                        return {
                            x: () => 0,
                            y: (d, i) => translate(i)
                        };
                    default:
                        return null;
                }
            },

            /** Returns a translation function for the specified axis */
            translate: function(axisName, axis, labels, thickness) {
                if (['x', 'y'].includes(axisName)) {
                    const transformations = this.getTranslateFunctions(axisName, axis, labels, thickness);
                    return (d, i) => `translate(${transformations.x(d, i)}, ${transformations.y(d, i)})`;
                }

                return null;
            },



            drawLabels(drawContext) {
                drawContext.transform = this.translate(drawContext.axisName, drawContext.axis, drawContext.labels, drawContext.thickness);
                // we need these transformations to compute labels bounding boxes in absolute coordinates
                drawContext.transformations = this.getTranslateFunctions(drawContext.axisName, drawContext.axis, drawContext.labels, drawContext.thickness);

                SVGUtils.drawLabels(drawContext, 'bars');
            },



            getExtraData(d, i, d3Context, chartBase, mainLabelDetectionHandler, labelDetectionHandlersByColor, colorScaleIndex, getLabelXPosition, getLabelYPosition, overlappingStrategy, defaultFontSize, onHoverMode) {
                const hideOverlaps = overlappingStrategy === ValuesInChartOverlappingStrategy.AUTO;

                const labelPosition = {
                    x: getLabelXPosition(d, i),
                    y: getLabelYPosition(d)
                };

                const overlappingKey = onHoverMode ? 'isOverlapAlt' : 'isOverlap';
                let overlapStatus = false;
                const getCurrentColorLabelCollisionDetectionHandler = () => {
                    if (!labelDetectionHandlersByColor[colorScaleIndex]) {
                        // we need a separate label collision detection handler for each color scale
                        labelDetectionHandlersByColor[colorScaleIndex] = SVGUtils.initLabelCollisionDetection(chartBase);
                    }
                    return labelDetectionHandlersByColor[colorScaleIndex];
                };
                const labelDetectionHandler = onHoverMode ? getCurrentColorLabelCollisionDetectionHandler() : mainLabelDetectionHandler;

                let translate = { x: 0, y: 0 };
                // retrieve the translations applied to the parent node, to compute bounding box in absolute coordinates
                if (d3Context && d3Context.parentNode) {
                    const parentNode = d3Context.parentNode;
                    const parentData = d3.select(parentNode).datum();
                    if (parentData.translate) {
                        translate = parentData.translate;
                    }
                }

                const labelRectangles = SVGUtils.getRectanglesFromPosition(labelPosition, d, defaultFontSize).map(rectangle => ({
                    ...rectangle,
                    x: rectangle.x + translate.x,
                    y: rectangle.y + translate.y
                }));

                //  even if we display all values, we need to add the bounding box to quadtree, in order to avoid overlapping with lines in mix chart
                if ((!hideOverlaps || !labelDetectionHandler.checkOverlaps(labelRectangles))) {
                    const isDisplayed = !!d.valuesInChartDisplayOptions?.displayValues;
                    if (isDisplayed) {
                        labelDetectionHandler.addBoundingBoxesToQuadTree(labelRectangles);
                    }
                } else {
                    overlapStatus = true;
                }
                d[overlappingKey] = overlapStatus;

                return { labelPosition };
            },

            shouldDisplayBarLabel(binCount, barValue) {
                // for most cases binCount !== 0 is sufficient, but for cumulative/differential compute modes, we might have both binCount === 0 and a barValue !== 0, but we want to display the value in that case
                return binCount !== 0 || barValue !== 0;
            }
        };
    }
})();

;
(function() {
    'use strict';

    angular.module('dataiku.charts')
        .factory('GroupedColumnsChart', GroupedColumnsChart);

    /**
     * (!) This service previously was in static/dataiku/js/simple_report/column-bars/column.js
     */
    function GroupedColumnsChart(ChartManager, ChartDataUtils, ChartDataWrapperFactory, GroupedColumnsDrawer, GroupedColumnsUtils, ReferenceLines, ColumnAvailability, ChartUsableColumns, ChartYAxisPosition, ChartAxesUtils, ChartCustomMeasures, SVGUtils, ChartDimension) {
        return function($container, chartDef, chartHandler, axesDef, data) {
            const chartData = chartDef.variant === 'waterfall' ? ChartDataWrapperFactory.chartWaterfallDataWrapper(data, axesDef, chartDef) : ChartDataWrapperFactory.chartTensorDataWrapper(data, axesDef);
            const dataSpec = chartHandler.getDataSpec();

            const customMeasures = ChartCustomMeasures.getMeasuresLikeCustomMeasures(dataSpec.datasetProjectKey, dataSpec.datasetName, chartHandler.getCurrentChartsContext());
            const allMeasures = ChartUsableColumns.getUsableColumns(dataSpec.datasetProjectKey, dataSpec.datasetName, chartHandler.getCurrentChartsContext()).filter(m => ['NUMERICAL', 'ALPHANUM', 'DATE'].includes(m.type));
            const xSpec = { type: 'DIMENSION', mode: 'COLUMNS', dimension: ChartDimension.getGenericDimension(chartDef), name: 'x', customExtent: chartDef.xAxisFormatting.customExtent };
            ColumnAvailability.updateAvailableColumns(chartDef.genericMeasures, allMeasures, customMeasures);

            const yExtents = ChartDataUtils.getMeasureExtents(chartDef, chartData, 'x'),
                leftYAxisID = ChartAxesUtils.computeYAxisID(ChartYAxisPosition.LEFT),
                rightYAxisID = ChartAxesUtils.computeYAxisID(ChartYAxisPosition.RIGHT),
                isLeftYPercentScale = yExtents[leftYAxisID].onlyPercent,
                isRightYPercentScale = yExtents[rightYAxisID].onlyPercent;

            const displayedReferenceLines = ReferenceLines.getDisplayedReferenceLines(chartDef.referenceLines, xSpec, undefined),
                referenceLinesValues = ReferenceLines.getReferenceLinesValues(displayedReferenceLines, chartData, allMeasures, chartDef.genericMeasures, customMeasures),
                referenceLinesExtents = ReferenceLines.getReferenceLinesExtents(displayedReferenceLines, referenceLinesValues, { [leftYAxisID]: { isPercentScale: isLeftYPercentScale }, [rightYAxisID]: { isPercentScale: isRightYPercentScale } });

            const leftYExtent = ReferenceLines.getExtentWithReferenceLines(yExtents[leftYAxisID].extent, referenceLinesExtents[leftYAxisID]),
                rightYExtent = ReferenceLines.getExtentWithReferenceLines(yExtents[rightYAxisID].extent, referenceLinesExtents[rightYAxisID]);

            ReferenceLines.mutateDimensionSpecForReferenceLine(ChartDataUtils.getAxisExtent(chartData, 'x', xSpec.dimension), referenceLinesExtents.x, xSpec);
            const animationData = GroupedColumnsUtils.prepareData(chartDef, chartData);

            const drawFrame = function(frameIdx, chartBase) {
                ReferenceLines.removeReferenceLines($container[0]);

                chartData.fixAxis('animation', frameIdx);
                animationData.frames[frameIdx].facets.forEach(function(facetData, f) {
                    // init the collision detection
                    const labelCollisionDetectionHandler = SVGUtils.initLabelCollisionDetection(chartBase);
                    const g = d3.select(chartBase.$svgs.eq(f).find('g.chart').get(0));
                    GroupedColumnsDrawer(g, chartDef, chartHandler, chartData.fixAxis('facet', f), chartBase, facetData.groups, f, labelCollisionDetectionHandler);
                });
            };

            const leftYCustomExtent = ChartAxesUtils.getYAxisCustomExtent(chartDef.yAxesFormatting, leftYAxisID);
            const rightYCustomExtent = ChartAxesUtils.getYAxisCustomExtent(chartDef.yAxesFormatting, rightYAxisID);
            const ySpecs = {
                [leftYAxisID]: { id: leftYAxisID, type: 'MEASURE', extent: leftYExtent, isPercentScale: isLeftYPercentScale, customExtent: leftYCustomExtent, position: ChartYAxisPosition.LEFT },
                [rightYAxisID]: { id: rightYAxisID, type: 'MEASURE', extent: rightYExtent, isPercentScale: isRightYPercentScale, customExtent: rightYCustomExtent, position: ChartYAxisPosition.RIGHT }
            };

            const availableAxes = ReferenceLines.getAvailableAxesForReferenceLines(chartDef);

            const axisSpecs = { x: xSpec, ...ySpecs };

            ChartAxesUtils.setNumberOfBinsToDimensions(chartData, chartDef, axisSpecs);

            ReferenceLines.updateAvailableAxisOptions([
                { axis: 'LEFT_Y_AXIS', isDisplayed: availableAxes['LEFT_Y_AXIS'], isNumerical: true, isPercentScale: ySpecs[leftYAxisID].isPercentScale },
                { axis: 'RIGHT_Y_AXIS', isDisplayed: availableAxes['RIGHT_Y_AXIS'], isNumerical: true, isPercentScale: ySpecs[rightYAxisID].isPercentScale },
                { axis: 'X_AXIS', isDisplayed: true, isNumerical: ChartAxesUtils.isNumerical(xSpec) && !ChartDimension.hasOneTickPerBin(xSpec.dimension), isContinuousDate: ChartAxesUtils.isContinuousDate(xSpec) }
            ]);

            ChartManager.initChart(chartDef, chartHandler, chartData, $container, drawFrame,
                axisSpecs,
                { type: 'DIMENSION', name: 'color', dimension: chartDef.genericDimension1[0] });
        };
    }
})();

;
(function() {
    'use strict';

    angular.module('dataiku.charts')
        .factory('GroupedColumnsDrawer', GroupedColumnsDrawer);

    /**
     * (!) This service previously was in static/dataiku/js/simple_report/column-bars/column.js
     */
    function GroupedColumnsDrawer(ChartDimension, Fn, BarChartUtils, SVGUtils, ChartAxesUtils, ReferenceLines, ColumnAvailability, ChartColorUtils, ChartCustomMeasures, ChartUsableColumns, ColorFocusHandler, CHART_TYPES, HierarchicalChartsUtils, ChartDrilldown, ChartHierarchyDimension, ValuesInChartPlacementMode, CHART_VARIANTS) {
        const backgroundXMargin = 4;

        return function(g, chartDef, chartHandler, chartData, chartBase, groupsData, f, labelCollisionDetectionHandler, drawReferenceLines = true) {

            const xDimension = ChartDimension.getGenericDimension(chartDef),
                xLabels = chartData.getAxisLabels('x'),
                xAxis = chartBase.xAxis,
                rightAxis = ChartAxesUtils.getRightYAxis(chartBase.yAxes),
                leftAxis = ChartAxesUtils.getLeftYAxis(chartBase.yAxes),
                getScale = function(d) {
                    return chartDef.genericMeasures[d.measure].displayAxis == 'axis1' ? leftAxis.scale() : rightAxis.scale();
                },
                getRectValue = function(d) {
                    return chartData.aggr(d.measure).get(d);
                };

            const nonEmptyBinsCounts = {};
            // do a first pass to compute nonEmptyBinsIdx, we have to do it here, and not in the "prepareData" method, because it would not have the current animation frame or subchart
            groupsData.forEach(group => {
                group.columns.forEach(c => {
                    const columnIndex = c.x;
                    if (_.isNil(nonEmptyBinsCounts[columnIndex])) {
                        nonEmptyBinsCounts[columnIndex] = 0;
                    }
                    const value = getRectValue(c);

                    let nonEmptyIdx = undefined;
                    if (value !== 0) {
                        nonEmptyIdx = nonEmptyBinsCounts[columnIndex];
                        nonEmptyBinsCounts[columnIndex] += 1;
                    }
                    c.nonEmptyIdx = nonEmptyIdx;
                });
            });
            // second pass to set nbNonEmptyBins
            groupsData.forEach(function(group) {
                group.columns.forEach(c => c.nbNonEmptyBins = nonEmptyBinsCounts[c.x]);
            });

            const nbMaxColumns = Math.max(...groupsData.map(group => group.columns.filter(c => !_.isNil(c.nonEmptyIdx)).length));

            let groupWidth = ChartDimension.isUngroupedNumerical(xDimension) ? 10 : Math.max(1, xAxis.ordinalScale.rangeBand());
            // Adjust barWidth if there is a custom extent, (mainly to prevent bars from overlapping each others when extent is reduced)
            groupWidth = Math.max(groupWidth * ChartAxesUtils.getCustomExtentRatio(chartDef.xAxisFormatting.customExtent), 1);

            let barWidth = groupWidth;
            if (groupsData.length) {
                if (HierarchicalChartsUtils.shouldIgnoreEmptyBinsForColorDimension(chartDef)) {
                    barWidth = nbMaxColumns ? groupWidth / nbMaxColumns : 0;
                } else {
                    barWidth = groupWidth / groupsData[0].columns.length;
                }
            }

            // Wrapper to contain all rectangles (used to apply clip-path)
            let wrapper = g.selectAll('g.group-wrapper');
            if (wrapper.empty()) {
                g.append('g').attr('class', 'group-wrapper');
                wrapper = g.selectAll('g.group-wrapper');
            }
            const groups = wrapper.selectAll('g.group-bars').data(groupsData);
            groups.enter().append('g')
                .attr('class', 'group-bars');
            groups.exit().remove();
            groups.attr('transform', BarChartUtils.translate('x', xAxis, xLabels, groupWidth));

            const positionBarWrappers = function(barWrappers) {
                return barWrappers.attr('transform', function(d, i) {
                    let xTranslate = barWidth * i;
                    if (HierarchicalChartsUtils.shouldIgnoreEmptyBinsForColorDimension(chartDef)) {
                        xTranslate = (_.isNil(d.nonEmptyIdx) || _.isNil(d.nbNonEmptyBins)) ? 0 : (barWidth * d.nonEmptyIdx + groupWidth / 2 - (barWidth * d.nbNonEmptyBins) / 2);
                    }
                    return 'translate(' + xTranslate + ')';
                });
            };

            const positionRects = function(rects) {
                return rects.attr('height', function(d) {
                    const yScale = getScale(d);
                    return Math.max(chartDef.variant === CHART_VARIANTS.waterfall ? 1 : 0, Math.abs(yScale(d.base) - yScale(d.top)));
                }).attr('y', function(d) {
                    const yScale = getScale(d);
                    return yScale(Math.max(d.base, d.top));
                });
            };

            const hasColorDim = ChartColorUtils.getColorDimensionOrMeasure(chartDef) !== undefined;

            const labelCollisionDetectionHandlersByColorScaleIndex = {};

            const barWrappers = groups.selectAll('.bar-wrapper').data(Fn.prop('columns'));
            barWrappers.enter().append('g')
                .attr('class', 'bar-wrapper')
                .call(positionBarWrappers);
            barWrappers.exit().remove();

            const rects = barWrappers.selectAll('rect').data(Fn.prop('data'));
            rects.enter().append('rect')
                .attr('fill', function(d) {
                    return chartBase.colorScale(ChartColorUtils.getColorIndex(d, hasColorDim));
                })
                .attr('opacity', function(d) {
                    return d.transparent ? 0 : chartDef.colorOptions.transparency;
                })
                .each(function(d) {
                    chartBase.tooltips.registerEl(this, angular.extend({}, d, { facet: f }), 'fill', undefined, undefined, hasColorDim);
                    chartBase.contextualMenu.addContextualMenuHandler(this, angular.extend({}, d, { facet: f }));
                });
            rects.exit().remove();
            rects
                .attr('width', barWidth)
                .attr('stroke-width', function(d) {
                    return d.isUnaggregated ? 0.5 : 0;
                })
                .attr('stroke', function(d) {
                    return chartHandler.legendsWrapper.getLegend(0).items[ChartColorUtils.getColorIndex(d, hasColorDim)].desaturatedColor;
                })
                .attr('x', 0)
                .interrupt('updateColumns').transition('updateColumns').ease('easeOutQuad')
                .call(positionRects);

            // mix chart uses the same drawer, hence the condition on the chartType
            if (chartDef.valuesInChartDisplayOptions && chartDef.valuesInChartDisplayOptions.displayValues) {
                const getLabelYPosition = (d) => {
                    if (!d.textsElements.length) {
                        // for animation purpose: place empty labels on the X axis, to have a smoother transition if they re-appear at the next animation frame
                        return getScale(d)(0);
                    }

                    const rectValue = getRectValue(d);
                    const defaultSpacing = chartDef.type === CHART_TYPES.MULTI_COLUMNS_LINES ? 5 : 2;
                    const spacing = d.valuesInChartDisplayOptions.spacing ?? defaultSpacing;

                    const yScale = getScale(d);
                    const scaleValue = yScale(rectValue);
                    if (chartDef.variant === CHART_VARIANTS.waterfall) {
                        const placementMode = d.valuesInChartDisplayOptions.placementMode;
                        // for now only one value in waterfal, so we just use the first data element
                        const data = d.data[0];
                        switch (placementMode) {
                            case ValuesInChartPlacementMode.BELOW:
                                return rectValue < 0 ? (yScale(data.top) + spacing) : (yScale(data.base) + spacing);
                            case ValuesInChartPlacementMode.ABOVE:
                                return rectValue < 0 ? (yScale(data.base) - spacing - d.height) : (yScale(data.top) - spacing - d.height);
                            default:
                                return rectValue < 0 ? (yScale(data.top) + spacing) : (yScale(data.top) - spacing - d.height);
                        }
                    }

                    const endBarY = isNaN(scaleValue) ? 0 : scaleValue - (rectValue >= 0 ? spacing : -spacing);
                    if (rectValue < 0) {
                        // for negative values, the label is below the bar
                        return endBarY;
                    }
                    return endBarY - d.height;
                };

                const getLabelXPosition = (d, i) => {
                    if (HierarchicalChartsUtils.shouldIgnoreEmptyBinsForColorDimension(chartDef)) {
                        // nonEmptyIdx undefined means that the bin is empty
                        const xPos = (_.isNil(d.nonEmptyIdx) || _.isNil(d.nbNonEmptyBins)) ? 0 : (barWidth * (d.nonEmptyIdx + 0.5) + groupWidth / 2 - (barWidth * d.nbNonEmptyBins) / 2);
                        return xPos;
                    }
                    return barWidth * (i + 0.5);
                };

                const getBackgroundXPosition = (d, i) => {
                    if (HierarchicalChartsUtils.shouldIgnoreEmptyBinsForColorDimension(chartDef)) {
                        // nonEmptyIdx undefined means that the bin is empty
                        const xPos = (_.isNil(d.nonEmptyIdx) || _.isNil(d.nbNonEmptyBins)) ? 0 : (barWidth * (d.nonEmptyIdx + 0.5) + groupWidth / 2 - (barWidth * d.nbNonEmptyBins) / 2) - d.width / 2 - backgroundXMargin;
                        return xPos;
                    }
                    return barWidth * (i + 0.5) - d.width / 2 - backgroundXMargin;
                };

                const getBackgroundYPosition = (d) => {
                    const y = getLabelYPosition(d);
                    return SVGUtils.getSubTextYPosition(d, y);
                };

                const getLabelText = (d) => {
                    const value = chartData.aggr(d.aggregationIndex).get(d);
                    return BarChartUtils.shouldDisplayBarLabel(chartData.getCount(d), getRectValue(d)) ? chartBase.measureFormatters[d.aggregationIndex](value) : '';
                };

                const labelsDrawContext = {
                    node: wrapper,
                    data: groupsData.map(d => d.columns),
                    axisName: 'x',
                    axis: xAxis,
                    labels: xLabels,
                    thickness: groupWidth,
                    opacity: chartDef.colorOptions.transparency,
                    overlappingStrategy: chartDef.valuesInChartDisplayOptions.overlappingStrategy,
                    colorScale: chartBase.colorScale,
                    getExtraData: function(d, i, n, onHoverMode) {
                        const extraData = BarChartUtils.getExtraData(d, i, this, chartBase, labelCollisionDetectionHandler, labelCollisionDetectionHandlersByColorScaleIndex, ChartColorUtils.getColorIndex(d, hasColorDim), getLabelXPosition, getLabelYPosition, chartDef.valuesInChartDisplayOptions.overlappingStrategy, chartDef.valuesInChartDisplayOptions.textFormatting.fontSize, onHoverMode);

                        return { labelPosition: extraData.labelPosition };
                    },
                    getLabelXPosition: (d) => {
                        return d.labelPosition.x;
                    },
                    getLabelYPosition: (d) => {
                        return d.labelPosition.y;
                    },
                    getLabelText,
                    getBackgroundXPosition,
                    getBackgroundYPosition,
                    backgroundXMargin,
                    theme: chartHandler.getChartTheme()
                };

                BarChartUtils.drawLabels(labelsDrawContext);
            }

            // Clip paths to prevent bars from overlapping axis when user chose a custom range which results in the bar being cropped (out of visible range)
            SVGUtils.clipPaths(chartBase, g, wrapper);

            if (drawReferenceLines) {
                const isPercentScaleOnLeftYAxis = chartDef.genericMeasures.some(measure => measure.displayAxis === 'axis1' && ChartDimension.isPercentScale([measure]));
                const isPercentScaleOnRightYAxis = chartDef.genericMeasures.some(measure => measure.displayAxis !== 'axis1' && ChartDimension.isPercentScale([measure]));

                const d3RightYAxis = ChartAxesUtils.getRightYAxis(chartBase.yAxes);
                const d3LeftYAxis = ChartAxesUtils.getLeftYAxis(chartBase.yAxes);
                const leftYFormattingOptions = d3LeftYAxis && ChartAxesUtils.getYAxisNumberFormatting(chartDef.yAxesFormatting, d3LeftYAxis.id);
                const rightYFormattingOptions = d3RightYAxis && ChartAxesUtils.getYAxisNumberFormatting(chartDef.yAxesFormatting, d3RightYAxis.id);

                const refLinesXAxis = { ...chartBase.xAxis, isPercentScale: false, formattingOptions: chartDef.xAxisFormatting.axisValuesFormatting.numberFormatting };
                const refLinesYAxes = [];

                if (d3LeftYAxis) {
                    refLinesYAxes.push({ ...d3LeftYAxis, isPercentScale: isPercentScaleOnLeftYAxis, formattingOptions: leftYFormattingOptions });
                }
                if (d3RightYAxis) {
                    refLinesYAxes.push({ ...d3RightYAxis, isPercentScale: isPercentScaleOnRightYAxis, formattingOptions: rightYFormattingOptions });
                }

                const dataSpec = chartHandler.getDataSpec();
                const customMeasures = ChartCustomMeasures.getMeasuresLikeCustomMeasures(dataSpec.datasetProjectKey, dataSpec.datasetName, chartHandler.getCurrentChartsContext());
                const allMeasures = ChartUsableColumns.getUsableColumns(dataSpec.datasetProjectKey, dataSpec.datasetName, chartHandler.getCurrentChartsContext()).filter(m => ['NUMERICAL', 'ALPHANUM', 'DATE'].includes(m.type));
                ColumnAvailability.updateAvailableColumns(chartDef.genericMeasures, allMeasures, customMeasures);

                const displayedReferenceLines = ReferenceLines.getDisplayedReferenceLines(chartDef.referenceLines, chartDef.$axisSpecs && chartDef.$axisSpecs.x, undefined),
                    referenceLinesValues = ReferenceLines.getReferenceLinesValues(displayedReferenceLines, chartData, allMeasures, chartDef.genericMeasures, customMeasures, chartDef.variant !== 'waterfall');

                ReferenceLines.drawReferenceLines(
                    wrapper,
                    chartBase.vizWidth,
                    chartBase.vizHeight,
                    refLinesXAxis,
                    refLinesYAxes,
                    displayedReferenceLines,
                    referenceLinesValues
                );
            }

            if (ChartHierarchyDimension.getCurrentHierarchyLevel(chartDef) > 0) {
                const anchor = SVGUtils.addPlotAreaContextMenuAnchor(g, chartBase.vizWidth, chartBase.vizHeight);
                chartBase.contextualMenu.addChartContextualMenuHandler(anchor.node(), undefined, () => ChartDrilldown.getDrillupActions(chartDef));
            }

            const colorFocusHandler = ColorFocusHandler.create(chartDef, chartHandler, wrapper, d3, g);
            ColorFocusHandler.appendFocusUnfocusMecanismToLegend(chartHandler.legendsWrapper.getLegend(0), colorFocusHandler);
        };
    };
})();

;
(function() {
    'use strict';

    angular.module('dataiku.charts')
        .factory('GroupedColumnsUtils', GroupedColumnsUtils);

    /**
     * (!) This service previously was in static/dataiku/js/simple_report/column-bars/column.js
     */
    function GroupedColumnsUtils(SVGUtils, ChartMeasure, ChartAxesUtils, ChartsStaticData, CHART_VARIANTS, UnaggregatedMeasureComputeModes) {
        const computeUAValueData = function(genericData, chartData, measure, measureIndex, hasLogScale) {
            const getBarBase = (total) => {
                if (total === 0 && hasLogScale) {
                    return 1;
                }
                return total;
            };
            const result = {
                ...genericData,
                isUnaggregated: true,
                uaComputeMode: measure.uaComputeMode
            };
            const uaValues = chartData.aggr(measureIndex).get();
            if (measure.uaComputeMode === UnaggregatedMeasureComputeModes.STACK) {
                let positiveTotal = 0;
                let negativeTotal = 0;
                result.data = uaValues.map(function(uaValue, idx) {
                    let base = uaValue >= 0 ? positiveTotal : negativeTotal;
                    let top = base + uaValue;

                    if (uaValue >= 0) {
                        base = getBarBase(base);
                        top = getBarBase(top);
                        positiveTotal = top;
                    } else {
                        negativeTotal = top;
                    }
                    return {
                        isUnaggregated: true,
                        isStacked: true,
                        ...genericData,
                        base,
                        top,
                        uaValueIdx: idx
                    };
                });
            } else {
                // overlay mode
                const positiveData = [];
                const negativeData = [];
                uaValues.forEach(function(uaValue, idx) {
                    const commonData = {
                        isUnaggregated: true,
                        base: hasLogScale ? 1 : 0,
                        top: uaValue,
                        ...genericData,
                        uaValueIdx: idx
                    };
                    if (uaValue >= 0) {
                        positiveData.push(commonData);
                    } else {
                        negativeData.push(commonData);
                    }
                });
                result.data = [
                    /*
                     * in Overlay mode, we display one bar per ua value, to handle tooltip and hover effect
                     * but we color only the biggest bar for positive values and for negative values, other bars are transparent by default
                     */
                    ...positiveData.map(function(data, idx) {
                        return {
                            ...data,
                            transparent: idx > 0
                        };
                    }),
                    ...negativeData.reverse().map(function(data, idx) {
                        return {
                            ...data,
                            transparent: idx > 0
                        };
                    })
                ];
            }

            return result;
        };


        return {
            prepareData: function(chartDef, chartData, measureFilter, ignoreLabels = new Set()) {
                const xLabels = chartData.getAxisLabels('x');
                const colorLabels = chartData.getAxisLabels('color') || [null];
                const facetLabels = chartData.getAxisLabels('facet') || [null];
                const animationLabels = chartData.getAxisLabels('animation') || [null];
                const hasLogScale = chartDef.xAxisFormatting.isLogScale || ChartAxesUtils.isYAxisLogScale(chartDef.yAxesFormatting, ChartsStaticData.LEFT_AXIS_ID);
                const getBarBase = (total) => {
                    if (total === 0 && hasLogScale) {
                        return 1;
                    }
                    return total;
                };

                const animationData = { frames: [], maxTotal: 0, minTotal: 0 };
                animationLabels.forEach(function(animationLabel, a) {
                    chartData.fixAxis('animation', a);

                    const frameData = { facets: [], maxTotal: 0, minTotal: 0 };
                    facetLabels.forEach(function(facetLabel, f) {
                        chartData.fixAxis('facet', f);

                        const facetData = { groups: [], maxTotal: 0, minTotal: 0 };
                        let runningTotal = 0; // only usefull for waterfall
                        xLabels.forEach(function(xLabel, x) {
                            chartData.fixAxis('x', x);
                            const columns = [];
                            if (xLabel && ignoreLabels.has(xLabel.label)) {
                                return;
                            }
                            const isTotalBar = chartData.isTotalBar(x);
                            if (isTotalBar) {
                                // for total bar, reset the total to start from the x axis
                                runningTotal = 0;
                            }

                            let cIndex = 0;
                            colorLabels.forEach(function(colorLabel, c) {
                                chartData.fixAxis('color', c);
                                if (colorLabel && ignoreLabels.has(colorLabel.label)) {
                                    return;
                                }
                                chartDef.genericMeasures.forEach(function(measure, measureIndex) {
                                    if (measureFilter && !measureFilter(measure)) {
                                        return;
                                    }
                                    const measureData = {
                                        color: chartData.getColorOfValue(c, x),
                                        measure: measureIndex,
                                        x: x,
                                        colorScaleIndex: chartData.getColorOfValue(cIndex, x)
                                    };

                                    let subTextElementsData = [];
                                    if (chartData.isBinMeaningful(measure.function, null, measureIndex)) {
                                        if (ChartMeasure.isRealUnaggregatedMeasure(measure)) {
                                            // in that case the value is a double array instead of a single value
                                            Object.assign(measureData, computeUAValueData(measureData, chartData, measure, measureIndex, hasLogScale));
                                        } else {
                                            subTextElementsData = SVGUtils.getLabelSubTexts(chartDef, measureIndex, measure.valuesInChartDisplayOptions, false).map(function(subTextElementData) {
                                                return {
                                                    ...subTextElementData,
                                                    // add measure data at sub text level also, to retrieve it easily
                                                    ...measureData
                                                };
                                            });
                                            const value = chartData.aggr(measureIndex).get();
                                            measureData.data = [{ ...measureData, base: getBarBase(runningTotal), top: getBarBase(runningTotal + value) }];
                                            if (chartDef.variant === CHART_VARIANTS.waterfall && !isTotalBar) {
                                                runningTotal += value;
                                            }
                                        }
                                    } else {
                                        // we don't want to display a bar for non meaningful values
                                        measureData.data = [{ ...measureData, base: getBarBase(runningTotal), top: getBarBase(runningTotal) }];
                                    }

                                    columns.push({
                                        ...measureData,
                                        textsElements: subTextElementsData,
                                        valuesInChartDisplayOptions: measure.valuesInChartDisplayOptions
                                    });
                                });
                                cIndex++;
                            });
                            facetData.groups.push({ x: x, columns: columns });
                        });
                        frameData.facets.push(facetData);
                    });
                    animationData.frames.push(frameData);
                });

                return animationData;
            }
        };
    }
})();

;
(function() {
    'use strict';

    angular.module('dataiku.charts')
        .factory('StackedBarsChart', StackedBarsChart);

    /**
     * (!) This service previously was in static/dataiku/js/simple_report/column-bars/bars.js
     */
    function StackedBarsChart(ChartManager, ChartDimension, ChartDataWrapperFactory, Fn, StackedChartUtils, BarChartUtils, SVGUtils, ChartAxesUtils, ReferenceLines, ColumnAvailability, ChartYAxisPosition, ChartCustomMeasures, ChartUsableColumns, ColorFocusHandler, ChartDataUtils, ChartDrilldown, ChartHierarchyDimension) {
        return function($container, chartDef, chartHandler, axesDef, data) {
            const backgroundXMargin = 4;

            const chartData = ChartDataWrapperFactory.chartTensorDataWrapper(data, axesDef);
            const dataSpec = chartHandler.getDataSpec();

            const isPercentScale = ChartDimension.isPercentScale(chartDef.genericMeasures) || chartDef.variant == 'stacked_100';

            const yAxisID = ChartAxesUtils.computeYAxisID(ChartYAxisPosition.LEFT);
            const yDimension = ChartDimension.getGenericDimension(chartDef);
            const ySpec = { id: yAxisID, type: 'DIMENSION', mode: 'COLUMNS', dimension: yDimension, minRangeBand: 18, ascendingDown: true, customExtent: ChartAxesUtils.getYAxisCustomExtent(chartDef.yAxesFormatting, yAxisID), position: ChartYAxisPosition.LEFT, name: 'y' };

            const customMeasures = ChartCustomMeasures.getMeasuresLikeCustomMeasures(dataSpec.datasetProjectKey, dataSpec.datasetName, chartHandler.getCurrentChartsContext());
            const allMeasures = ChartUsableColumns.getUsableColumns(dataSpec.datasetProjectKey, dataSpec.datasetName, chartHandler.getCurrentChartsContext()).filter(m => ['NUMERICAL', 'ALPHANUM', 'DATE'].includes(m.type));
            ColumnAvailability.updateAvailableColumns(chartDef.genericMeasures, allMeasures, customMeasures);

            const displayedReferenceLines = ReferenceLines.getDisplayedReferenceLines(chartDef.referenceLines, undefined, { [yAxisID]: ySpec }),
                referenceLinesValues = ReferenceLines.getReferenceLinesValues(displayedReferenceLines, chartData, allMeasures, chartDef.genericMeasures, customMeasures),
                referenceLinesExtents = ReferenceLines.getReferenceLinesExtents(displayedReferenceLines, referenceLinesValues, { x: { isPercentScale }, [yAxisID]: { isPercentScale: false } });

            ReferenceLines.mutateDimensionSpecForReferenceLine(ChartDataUtils.getAxisExtent(chartData, 'y', ySpec.dimension), referenceLinesExtents[yAxisID], ySpec);

            const animationData = StackedChartUtils.prepareData(chartDef, chartData, 'y'),
                yLabels = chartData.getAxisLabels('y'),
                xExtent = ReferenceLines.getExtentWithReferenceLines([animationData.minTotal, animationData.maxTotal], referenceLinesExtents.x);

            const drawFrame = function(frameIdx, chartBase) {
                ReferenceLines.removeReferenceLines($container[0]);

                animationData.frames[frameIdx].facets.forEach(function(facetData, f) {
                    const g = d3.select(chartBase.$svgs.eq(f).find('g.chart').get(0));
                    StackedBarsChartDrawer(g, facetData, chartBase, f);
                });
            };

            // Hack: by design bar charts start at 0, but if log scale is checked start at 1 (to make it possible)
            if (chartDef.xAxisFormatting.isLogScale) {
                xExtent[0] = 1;
            }

            const xSpec = { type: 'MEASURE', extent: xExtent, isPercentScale: isPercentScale, measure: chartDef.genericMeasures, customExtent: chartDef.xAxisFormatting.customExtent };
            const axisSpecs = {
                x: xSpec,
                [yAxisID]: ySpec
            };

            ChartAxesUtils.setNumberOfBinsToDimensions(chartData, chartDef, axisSpecs);
            ReferenceLines.updateAvailableAxisOptions([
                { axis: 'X_AXIS', isDisplayed: ['X_AXIS'], isNumerical: true, isPercentScale },
                { axis: 'LEFT_Y_AXIS', isDisplayed: ['LEFT_Y_AXIS'], isNumerical: ChartAxesUtils.isNumerical(ySpec) && !ChartDimension.hasOneTickPerBin(ySpec.dimension), isPercentScale: false, isContinuousDate: ChartAxesUtils.isContinuousDate(ySpec) }
            ]);

            ChartManager.initChart(chartDef, chartHandler, chartData, $container, drawFrame,
                axisSpecs,
                { type: 'DIMENSION', name: 'color', dimension: chartDef.genericDimension1[0] });

            function StackedBarsChartDrawer(g, stacksData, chartBase, f) {
                const percentFormatter = d3.format('.0%');

                let barHeight = ChartDimension.isUngroupedNumerical(yDimension) ? 10 : Math.max(1, chartBase.yAxes[0].ordinalScale.rangeBand());
                // Adjust barHeight if there is a custom extent, (mainly to prevent bars from overlapping each others when extent is reduced)
                const yAxisCustomExtent = ChartAxesUtils.getYAxisCustomExtent(chartDef.yAxesFormatting, chartBase.yAxes[0].id);
                barHeight = Math.max(barHeight * ChartAxesUtils.getCustomExtentRatio(yAxisCustomExtent), 1);

                const stacks = g.selectAll('.stack-bars').data(stacksData.stacks);
                stacks.enter().append('g').attr('class', 'stack-bars');
                stacks.exit().remove();
                stacks.attr('transform', BarChartUtils.translate('y', chartBase.yAxes[0], yLabels, barHeight));

                /*
                 * Display total, not enabled for now TODO:
                 * if (chartDef.valuesInChartDisplayOptions.displayValues && chartDef.variant !== 'stacked_100') {
                 *  var totals = stacks.selectAll('text.total').data(function(d) { return [d]; });
                 *  totals.enter().append('text')
                 *      .attr('class', 'total')
                 *      .attr('y', barHeight/2)
                 *      .attr('text-anchor', 'middle')
                 *      .attr('dominant-baseline', 'middle')
                 *      .attr('font-weight', 500);
                 *  totals
                 *      .text(function(d) { return d.count > 0 ? chartBase.measureFormatters[0](d.total) : ''; })
                 *      .each(function(d) {
                 *          var bbox = this.getBoundingClientRect();
                 *          if (bbox.height > barHeight) {
                 *              d3.select(this).attr('visibility', 'hidden');
                 *          } else {
                 *              d3.select(this).attr('visibility', null);
                 *          }
                 *      })
                 *      .transition()
                 *      .attr('x', function(d) {
                 *          var bbox = this.getBoundingClientRect();
                 *          return chartBase.xAxis.scale()(d.total) + bbox.width/2 + 5;
                 *      });
                 * }
                 */

                const rects = stacks.selectAll('rect').data(Fn.prop('data'));
                const getFinalColorScaleIndex = (d) => d.color + d.measure;
                rects.enter().append('rect')
                    .attr('height', barHeight)
                    .attr('y', 0)
                    .attr('fill', function(d) {
                        return chartBase.colorScale(getFinalColorScaleIndex(d));
                    })
                    .attr('opacity', chartDef.colorOptions.transparency)
                    .each(function(d) {
                        chartBase.tooltips.registerEl(this, { measure: d.measure, y: d.y, color: d.color, facet: f }, 'fill');
                        chartBase.contextualMenu.addContextualMenuHandler(this, { y: d.y, color: d.color, facet: f });
                    });

                if (chartDef.valuesInChartDisplayOptions && chartDef.valuesInChartDisplayOptions.displayValues) {
                    const xScale = chartBase.xAxis.scale();

                    const getLabelYPosition = (d) => {
                        return (barHeight / 2) - (d.height / 2);
                    };

                    const getLabelXPosition = (d, i) => {
                        return xScale((d.top + d.base) / 2);
                    };


                    // init the collision detection
                    const globalLabelCollisionDetectionHandler = SVGUtils.initLabelCollisionDetection(chartBase);

                    const labelCollisionDetectionHandlersByColorScaleIndex = {};
                    const labelsDrawContext = {
                        node: g,
                        data: stacksData.stacks.map(stack => stack.data),
                        axisName: 'y',
                        axis: chartBase.yAxes[0],
                        labels: yLabels,
                        thickness: barHeight,
                        opacity: chartDef.colorOptions.transparency,
                        overlappingStrategy: chartDef.valuesInChartDisplayOptions.overlappingStrategy,
                        colorScale: chartBase.colorScale,
                        getLabelXPosition,
                        getLabelYPosition,
                        getLabelText: (d) => {
                            if (BarChartUtils.shouldDisplayBarLabel(d.count, d.value)) {
                                if (chartDef.variant === 'stacked_100') {
                                    return percentFormatter(d.value);
                                } else {
                                    return chartBase.measureFormatters[d.aggregationIndex](chartData.aggr(d.aggregationIndex).get(d));
                                }
                            } else {
                                return '';
                            }
                        },
                        getExtraData: function(d, i, n, onHoverMode) {
                            const extraData = BarChartUtils.getExtraData(d, i, this, chartBase, globalLabelCollisionDetectionHandler, labelCollisionDetectionHandlersByColorScaleIndex, getFinalColorScaleIndex, getLabelXPosition, getLabelYPosition, chartDef.valuesInChartDisplayOptions.overlappingStrategy, chartDef.valuesInChartDisplayOptions.textFormatting.fontSize, onHoverMode);

                            return { labelPosition: extraData.labelPosition };
                        },
                        getBackgroundXPosition: (d) => {
                            return getLabelXPosition(d) - d.width / 2 - backgroundXMargin;
                        },
                        getBackgroundYPosition: (d) => {
                            const y = getLabelYPosition(d);
                            return SVGUtils.getSubTextYPosition(d, y);
                        },
                        backgroundXMargin,
                        hasBackground: true,
                        theme: chartHandler.getChartTheme()
                    };

                    BarChartUtils.drawLabels(labelsDrawContext);
                }

                rects.interrupt('updateBars').transition('updateBars')
                    .attr('width', function(d) {
                        return Math.abs(chartBase.xAxis.scale()(d.top) - chartBase.xAxis.scale()(d.base));
                    })
                    .attr('x', function(d) {
                        return chartBase.xAxis.scale()(Math.min(d.base, d.top));
                    });

                // Clip paths to prevent bars from overlapping axis when user chose a custom range which results in the bar being cropped (out of visible range)
                ChartAxesUtils.isCroppedChart(chartDef) && SVGUtils.clipPaths(chartBase, g, stacks);

                const xAxis = chartBase.xAxis ? { ...chartBase.xAxis, isPercentScale, formattingOptions: chartDef.xAxisFormatting.axisValuesFormatting.numberFormatting } : null;
                const yAxis = chartBase.yAxes[0] ? { ...chartBase.yAxes[0], formattingOptions: chartDef.yAxesFormatting[0] && chartDef.yAxesFormatting[0].axisValuesFormatting.numberFormatting } : null;

                ReferenceLines.drawReferenceLines(
                    g,
                    chartBase.vizWidth,
                    chartBase.vizHeight,
                    xAxis,
                    [yAxis],
                    displayedReferenceLines,
                    referenceLinesValues
                );

                if (ChartHierarchyDimension.getCurrentHierarchyLevel(chartDef) > 0) {
                    const anchor = SVGUtils.addPlotAreaContextMenuAnchor(g, chartBase.vizWidth, chartBase.vizHeight);
                    chartBase.contextualMenu.addChartContextualMenuHandler(anchor.node(), undefined, () => ChartDrilldown.getDrillupActions(chartDef));
                }

                const colorFocusHandler = ColorFocusHandler.create(chartDef, chartHandler, stacks, d3, g);
                ColorFocusHandler.appendFocusUnfocusMecanismToLegend(chartHandler.legendsWrapper.getLegend(0), colorFocusHandler);
            }
        };
    }
})();

;
(function() {
    'use strict';

    angular.module('dataiku.charts')
        .factory('StackedChartUtils', StackedChartUtils);

    /**
     * (!) This service previously was in static/dataiku/js/simple_report/column-bars/stacked-columns.js
     */
    function StackedChartUtils(ChartAxesUtils, ChartsStaticData, SVGUtils, ChartMeasure, UnaggregatedMeasureComputeModes, CHART_VARIANTS) {
        const computeBaseAndTop = (value, positiveTotal, negativeTotal) => {
            const base = value >= 0 ? positiveTotal : negativeTotal;
            const top = base + value;

            return {
                base,
                top
            };
        };

        const computeUAValueData = function(genericData, chartData, measure, measureIndex, positiveTotal, negativeTotal) {
            const result = {
                ...genericData,
                isUnaggregated: true,
                uaComputeMode: measure.uaComputeMode
            };

            let newPositiveTotal = positiveTotal;
            let newNegativeTotal = negativeTotal;
            const uaValues = chartData.aggr(measureIndex).get();
            if (measure.uaComputeMode === UnaggregatedMeasureComputeModes.STACK) {
                result.data = uaValues.map(function(uaValue, idx) {
                    const { base, top } = computeBaseAndTop(uaValue, newPositiveTotal, newNegativeTotal);
                    if (uaValue >= 0) {
                        newPositiveTotal = top;
                    } else {
                        newNegativeTotal = top;
                    }
                    return {
                        isUnaggregated: true,
                        isStacked: true,
                        ...genericData,
                        base,
                        top,
                        uaValueIdx: idx
                    };
                });
            } else {
                // overlay mode
                const positiveData = [];
                const negativeData = [];
                uaValues.forEach(function(uaValue, idx) {
                    const { base, top } = computeBaseAndTop(uaValue, positiveTotal, negativeTotal);
                    const commonData = {
                        isUnaggregated: true,
                        base,
                        top,
                        ...genericData,
                        uaValueIdx: idx
                    };
                    if (uaValue >= 0) {
                        positiveData.push(commonData);
                        newPositiveTotal = Math.max(newPositiveTotal, commonData.top);
                    } else {
                        negativeData.push(commonData);
                        newNegativeTotal = Math.min(newNegativeTotal, commonData.top);
                    }
                });
                result.data = [
                    /*
                     * in Overlay mode, we display one bar per ua value, to handle tooltip and hover effect
                     * but we color only the biggest bar for positive values and for negative values, other bars are transparent by default
                     */
                    ...positiveData.map(function(data, idx) {
                        return {
                            ...data,
                            transparent: idx > 0
                        };
                    }),
                    ...negativeData.reverse().map(function(data, idx) {
                        return {
                            ...data,
                            transparent: idx > 0
                        };
                    })
                ];
            }

            // return also the new value for totals
            return { ...result, positiveTotal: newPositiveTotal, negativeTotal: newNegativeTotal };
        };


        return {
            prepareData: function(chartDef, chartData, axis = 'x') {
                const colorLabels = chartData.getAxisLabels('color') || [null],
                    facetLabels = chartData.getAxisLabels('facet') || [null],
                    animationLabels = chartData.getAxisLabels('animation') || [null],
                    mainAxisLabels = chartData.getAxisLabels(axis),
                    // only one axis at a time can display in logscale so that logic is enough
                    hasLogScale = chartDef.xAxisFormatting.isLogScale || ChartAxesUtils.isYAxisLogScale(chartDef.yAxesFormatting, ChartsStaticData.LEFT_AXIS_ID);

                const getBarExtremity = (total) => {
                    if (total === 0 && hasLogScale) {
                        return 1;
                    }
                    return total;
                };
                const animationData = { frames: [], maxTotal: 0, minTotal: 0 };
                animationLabels.forEach(function(animationLabel, a) {
                    chartData.fixAxis('animation', a);

                    const frameData = { facets: [], maxTotal: 0, minTotal: 0 };
                    facetLabels.forEach(function(facetLabel, f) {
                        chartData.fixAxis('facet', f);

                        const facetData = { stacks: [], maxTotal: 0, minTotal: 0 };
                        mainAxisLabels.forEach(function(_, val) {
                            chartData.fixAxis(axis, val);

                            let positiveTotal = 0;
                            let negativeTotal = 0;

                            let count = 0;
                            const stackData = [];

                            const totalSubTextElementsData = SVGUtils.getLabelSubTexts(chartDef, undefined, chartDef.stackedColumnsOptions ? chartDef.stackedColumnsOptions.totalsInChartDisplayOptions : chartDef.valuesInChartDisplayOptions, true).map(function(subTextElementData) {
                                const res = {
                                    ...subTextElementData,
                                    aggregationTotal: positiveTotal + negativeTotal
                                };
                                res[axis] = val;
                                return res;
                            });

                            colorLabels.forEach(function(colorLabel, c) {
                                chartData.fixAxis('color', c);
                                totalSubTextElementsData.forEach(function(subTextElementData) {
                                    if (subTextElementData.aggregationIndex !== undefined) {
                                        // compute total for custom aggregations
                                        const d = chartData.aggr(subTextElementData.aggregationIndex).get();
                                        subTextElementData.aggregationTotal += d;
                                    }
                                });

                                chartDef.genericMeasures.forEach(function(measure, m) {
                                    const d = chartData.aggr(m).get();

                                    if (chartDef.variant == CHART_VARIANTS.stacked100 && d < 0) {
                                        throw new ChartIAE('Cannot represent negative values on a 100% Stacked chart. Please use another chart.');
                                    }

                                    let subTextElementsData = [];
                                    let data = [];
                                    const commonData = {
                                        color: c,
                                        measure: m,
                                        facet: f,
                                        animation: a,
                                        count: chartData.getNonNullCount({}, m),
                                        value: d,
                                        [axis]: val
                                    };
                                    if (ChartMeasure.isRealUnaggregatedMeasure(measure)) {
                                        // in that case the value is a double array instead of a single value
                                        if (chartDef.variant == CHART_VARIANTS.stacked100) {
                                            throw new ChartIAE('Cannot represent unaggregated values on a 100% Stacked chart. Please use another chart.');
                                        }

                                        const {
                                            data: uaData,
                                            positiveTotal: newPositiveTotal,
                                            negativeTotal: newNegativeTotal
                                        } = computeUAValueData(commonData, chartData, measure, m, positiveTotal, negativeTotal);
                                        data = uaData;
                                        positiveTotal = newPositiveTotal;
                                        negativeTotal = newNegativeTotal;
                                    } else {
                                        commonData.base = (d >= 0) ? positiveTotal : negativeTotal;
                                        commonData.top = commonData.base + d;

                                        if (d >= 0) {
                                            commonData.base = getBarExtremity(commonData.base);
                                            commonData.top = getBarExtremity(commonData.top);
                                            positiveTotal = commonData.top;
                                        } else {
                                            negativeTotal = commonData.top;
                                        }

                                        subTextElementsData = SVGUtils.getLabelSubTexts(chartDef, m, measure.valuesInChartDisplayOptions, false).map(function(subTextElementData) {
                                            return {
                                                ...subTextElementData,
                                                // add measure data at sub text level also, to retrieve it easily
                                                ...commonData
                                            };
                                        });

                                        data = [commonData];
                                    }

                                    const point = {
                                        ...commonData,
                                        data,
                                        textsElements: subTextElementsData,
                                        valuesInChartDisplayOptions: measure.valuesInChartDisplayOptions
                                    };
                                    stackData.push(point);

                                    count += chartData.getNonNullCount({}, m);
                                });
                            });

                            if (chartDef.variant == CHART_VARIANTS.stacked100 && positiveTotal > 0) {
                                // Do a second pass and divide by total
                                let totalPercent = 0;
                                stackData.forEach(function(point, p) {
                                    const update = {
                                        value: point.value / positiveTotal,
                                        base: totalPercent,
                                        top: point.value / positiveTotal + totalPercent
                                    };
                                    Object.assign(point, update);
                                    // update data on sub texts as well
                                    point.textsElements.forEach(function(subTextElementData) {
                                        Object.assign(subTextElementData, update);
                                    });
                                    point.data.forEach(function(dataElement) {
                                        Object.assign(dataElement, update);
                                    });

                                    totalPercent += point.value;
                                });

                                positiveTotal = 1;
                            }

                            const totalData = { data: stackData, total: positiveTotal + negativeTotal, positiveTotal, negativeTotal, count,
                                spacing: chartDef.stackedColumnsOptions?.totalsInChartDisplayOptions?.spacing
                            };
                            facetData.stacks.push({ ...totalData, textsElements: totalSubTextElementsData.map(function(subTextElementData) {
                                return { ...subTextElementData, ...totalData };
                            }) });
                            facetData.maxTotal = Math.max(facetData.maxTotal, positiveTotal);
                            facetData.minTotal = Math.min(facetData.minTotal, negativeTotal);
                        });

                        frameData.maxTotal = Math.max(frameData.maxTotal, facetData.maxTotal);
                        frameData.minTotal = Math.min(frameData.minTotal, facetData.minTotal);
                        frameData.facets.push(facetData);
                    });

                    animationData.maxTotal = Math.max(animationData.maxTotal, frameData.maxTotal);
                    animationData.minTotal = Math.min(animationData.minTotal, frameData.minTotal);
                    animationData.frames.push(frameData);
                });

                return animationData;
            }
        };
    }

})();

;
(function() {
    'use strict';

    angular.module('dataiku.charts')
        .factory('StackedColumnsChart', StackedColumnsChart);

    /**
     * (!) This service previously was in static/dataiku/js/simple_report/column-bars/stacked-columns.js
     */
    function StackedColumnsChart(ChartManager, ChartDimension, ChartFeatures, ChartDataWrapperFactory, Fn, StackedChartUtils, BarChartUtils, SVGUtils, ChartAxesUtils, ReferenceLines, ColumnAvailability, ChartYAxisPosition, ChartCustomMeasures, ChartUsableColumns, ColorFocusHandler, ValuesInChartOverlappingStrategy, ChartDataUtils, ChartHierarchyDimension, ChartDrilldown, ChartColorUtils, ChartMeasure, CHART_VARIANTS) {
        return function($container, chartDef, chartHandler, axesDef, data) {
            const backgroundXMargin = 4;

            const chartData = ChartDataWrapperFactory.chartTensorDataWrapper(data, axesDef);
            const dataSpec = chartHandler.getDataSpec();

            const yAxisID = ChartAxesUtils.computeYAxisID(ChartYAxisPosition.LEFT);
            const isPercentScale = ChartDimension.isPercentScale(chartDef.genericMeasures) || chartDef.variant == 'stacked_100';

            const customMeasures = ChartCustomMeasures.getMeasuresLikeCustomMeasures(dataSpec.datasetProjectKey, dataSpec.datasetName, chartHandler.getCurrentChartsContext());
            const allMeasures = ChartUsableColumns.getUsableColumns(dataSpec.datasetProjectKey, dataSpec.datasetName, chartHandler.getCurrentChartsContext()).filter(m => ['NUMERICAL', 'ALPHANUM', 'DATE'].includes(m.type));
            const xDimension = ChartDimension.getGenericDimension(chartDef);

            const xSpec = { type: 'DIMENSION', mode: 'COLUMNS', dimension: xDimension, name: 'x', customExtent: chartDef.xAxisFormatting.customExtent };
            ColumnAvailability.updateAvailableColumns(chartDef.genericMeasures, allMeasures, customMeasures);

            const displayedReferenceLines = ReferenceLines.getDisplayedReferenceLines(chartDef.referenceLines, xSpec, undefined),
                referenceLinesValues = ReferenceLines.getReferenceLinesValues(displayedReferenceLines, chartData, allMeasures, chartDef.genericMeasures, customMeasures),
                referenceLinesExtents = ReferenceLines.getReferenceLinesExtents(displayedReferenceLines, referenceLinesValues, { [yAxisID]: { isPercentScale } });

            ReferenceLines.mutateDimensionSpecForReferenceLine(ChartDataUtils.getAxisExtent(chartData, 'x', xSpec.dimension), referenceLinesExtents.x, xSpec);

            const xLabels = chartData.getAxisLabels('x'),
                animationData = StackedChartUtils.prepareData(chartDef, chartData),
                yExtent = ReferenceLines.getExtentWithReferenceLines([animationData.minTotal, animationData.maxTotal], referenceLinesExtents[yAxisID]);

            const drawFrame = function(frameIdx, chartBase) {
                ReferenceLines.removeReferenceLines($container[0]);

                animationData.frames[frameIdx].facets.forEach(function(facetData, f) {
                    const g = d3.select(chartBase.$svgs.eq(f).find('g.chart').get(0));
                    StackedColumnChartDrawer(g, facetData, chartBase);
                });
            };

            // Hack: by design stacked columns start at 0, but if log scale is checked start at 1 (to make it possible)
            if (ChartAxesUtils.isYAxisLogScale(chartDef.yAxesFormatting)) {
                yExtent[0] = 1;
            }

            const ySpecs = { [yAxisID]: { id: yAxisID, type: 'MEASURE', extent: yExtent, isPercentScale, customExtent: ChartAxesUtils.getYAxisCustomExtent(chartDef.yAxesFormatting, yAxisID), position: ChartYAxisPosition.LEFT } };
            const axisSpecs = { x: xSpec, ...ySpecs };

            ChartAxesUtils.setNumberOfBinsToDimensions(chartData, chartDef, axisSpecs);
            ReferenceLines.updateAvailableAxisOptions([
                { axis: 'LEFT_Y_AXIS', isDisplayed: true, isNumerical: true, isPercentScale },
                { axis: 'X_AXIS', isDisplayed: true, isNumerical: ChartAxesUtils.isNumerical(xSpec) && !ChartDimension.hasOneTickPerBin(xSpec.dimension), isContinuousDate: ChartAxesUtils.isContinuousDate(xSpec) }
            ]);

            ChartManager.initChart(chartDef, chartHandler, chartData, $container, drawFrame,
                axisSpecs,
                { type: 'DIMENSION', name: 'color', dimension: chartDef.genericDimension1[0] });

            function StackedColumnChartDrawer(g, stacksData, chartBase) {

                const xAxis = chartBase.xAxis,
                    yAxis = chartBase.yAxes[0],
                    yScale = yAxis.scale(),
                    percentFormatter = d3.format('.0%');

                let barWidth = ChartDimension.isUngroupedNumerical(xDimension) ? 10 : Math.max(1, xAxis.ordinalScale.rangeBand());
                // Adjust barWidth if there is a custom extent, (mainly to prevent bars from overlapping each others when extent is reduced)
                barWidth = Math.max(barWidth * ChartAxesUtils.getCustomExtentRatio(chartDef.xAxisFormatting.customExtent), 1);

                // Wrapper to contain all rectangles (used to apply clip-path)
                let wrapper = g.selectAll('g.stack-wrapper');
                if (wrapper.empty()) {
                    g.append('g').attr('class', 'stack-wrapper');
                    wrapper = g.selectAll('g.stack-wrapper');
                }

                const stacks = wrapper.selectAll('.stack-bars').data(stacksData.stacks);
                stacks.enter().append('g').attr('class', 'stack-bars');
                stacks.exit().remove();
                stacks.attr('transform', BarChartUtils.translate('x', xAxis, xLabels, barWidth));

                const rects = stacks.selectAll('rect').data(d => _.flatten(d.data.map(e => e.data)));
                const hasColorDim = ChartColorUtils.getColorDimensionOrMeasure(chartDef) !== undefined;
                const getFinalColorScaleIndex = (d) => ChartColorUtils.getColorIndex(d, hasColorDim);

                rects.enter().append('rect')
                    .attr('width', barWidth)
                    .attr('x', 0)
                    .attr('fill', function(d, i) {
                        return chartBase.colorScale(getFinalColorScaleIndex(d, i));
                    })
                    .attr('opacity', function(d) {
                        return d.transparent ? 0 : chartDef.colorOptions.transparency;
                    })
                    .each(function(d) {
                        chartBase.tooltips.registerEl(this, _.pick(d, ['measure', 'x', 'color', 'facet', 'isUnaggregated', 'uaValueIdx']), 'fill');
                        chartBase.contextualMenu.addContextualMenuHandler(this, _.pick(d, ['x', 'color', 'facet', 'isUnaggregated', 'uaValueIdx']), 'fill');
                    });
                rects.exit().remove();

                if (chartDef.valuesInChartDisplayOptions && chartDef.valuesInChartDisplayOptions.displayValues) {

                    // init the collision detection
                    const labelCollisionDetectionHandler = SVGUtils.initLabelCollisionDetection(chartBase);

                    const getLabelXPosition = () => {
                        return barWidth / 2;
                    };

                    const commonDrawContext = {
                        node: wrapper,
                        axisName: 'x',
                        axis: xAxis,
                        labels: xLabels,
                        thickness: barWidth,
                        opacity: chartDef.colorOptions.transparency,
                        colorScale: chartBase.colorScale,
                        getLabelXPosition,
                        getBackgroundXPosition: (d, i) => {
                            return getLabelXPosition(d, i) - d.width / 2 - backgroundXMargin;
                        },
                        backgroundXMargin
                    };

                    const drawValuesLabels = () => {
                        const getLabelYPosition = function(d) {
                            return yScale((d.top + d.base) / 2) - (d.height / 2);
                        };

                        const getLabelText = (d) => {
                            if (BarChartUtils.shouldDisplayBarLabel(d.count, d.value)) {
                                if (chartDef.variant === CHART_VARIANTS.stacked100) {
                                    return percentFormatter(d.value);
                                } else {
                                    return chartBase.measureFormatters[d.aggregationIndex](chartData.aggr(d.aggregationIndex).get(d));
                                }
                            } else {
                                return '';
                            }
                        };

                        const overlappingStrategyForValues = [
                            ValuesInChartOverlappingStrategy.AUTO_WITH_VALUES,
                            ValuesInChartOverlappingStrategy.AUTO
                        ].includes(chartDef.valuesInChartDisplayOptions.overlappingStrategy)
                            ? ValuesInChartOverlappingStrategy.AUTO : ValuesInChartOverlappingStrategy.ALL;
                        const labelCollisionDetectionHandlersByColorScaleIndex = {};

                        const labelsDrawContext = {
                            ...commonDrawContext,
                            data: stacksData.stacks.map(stack => stack.data),
                            getLabelYPosition,
                            getExtraData: function(d, i, n, onHoverMode) {
                                const extraData = BarChartUtils.getExtraData(d, i, this, chartBase, labelCollisionDetectionHandler, labelCollisionDetectionHandlersByColorScaleIndex, getFinalColorScaleIndex, getLabelXPosition, getLabelYPosition, overlappingStrategyForValues, chartDef.valuesInChartDisplayOptions.textFormatting.fontSize, onHoverMode);
                                return { labelPosition: extraData.labelPosition };
                            },
                            getLabelText,
                            getBackgroundYPosition: (d) => {
                                const y = getLabelYPosition(d);
                                return SVGUtils.getSubTextYPosition(d, y);
                            },
                            hasBackground: true,
                            overlappingStrategy: overlappingStrategyForValues,
                            theme: chartHandler.getChartTheme()
                        };

                        BarChartUtils.drawLabels(labelsDrawContext);
                    };

                    const drawTotalsLabels = () => {
                        if (ChartFeatures.shouldDisplayTotalValues(chartDef) && !chartDef.genericMeasures.some(m => ChartMeasure.isRealUnaggregatedMeasure(m))) {

                            // specifc overrides for totals
                            const getLabelText = (d) => {
                                // aggregationIndex is not defined if no custom value
                                const text = d.aggregationIndex === undefined
                                    ? chartBase.measureFormatters[0](d.total)
                                    : chartBase.measureFormatters[d.aggregationIndex](d.aggregationTotal);
                                return BarChartUtils.shouldDisplayBarLabel(d.count, d.total) ? text : '';
                            };

                            const getLabelYPosition = (d) => {
                                const spacing = d.spacing || 5;
                                if (d.total < 0) {
                                    // for negative totals, the label is below the bar
                                    return yScale(d.negativeTotal) + spacing;
                                } else {
                                    return yScale(d.positiveTotal) - spacing - d.height;
                                }
                            };

                            const overlappingStrategyForTotals = [
                                ValuesInChartOverlappingStrategy.AUTO_WITH_TOTALS,
                                ValuesInChartOverlappingStrategy.AUTO
                            ].includes(chartDef.valuesInChartDisplayOptions.overlappingStrategy)
                                ? ValuesInChartOverlappingStrategy.AUTO : ValuesInChartOverlappingStrategy.ALL;
                            const labelCollisionDetectionHandlersByColorScaleIndex = {};
                            const totalLabelsDrawContext = {
                                ...commonDrawContext,
                                data: stacksData.stacks,
                                getLabelYPosition,
                                getExtraData: function(d, i, n, onHoverMode) {
                                    const extraData = BarChartUtils.getExtraData(d, i, this, chartBase, labelCollisionDetectionHandler, labelCollisionDetectionHandlersByColorScaleIndex, getFinalColorScaleIndex, getLabelXPosition, getLabelYPosition, overlappingStrategyForTotals, chartDef.valuesInChartDisplayOptions.textFormatting.fontSize, onHoverMode);
                                    return { labelPosition: extraData.labelPosition };
                                },
                                getBackgroundYPosition: (d) => {
                                    const y = getLabelYPosition(d);
                                    return SVGUtils.getSubTextYPosition(d, y);
                                },
                                getLabelText,
                                hasBackground: false,
                                isTotals: true,
                                overlappingStrategy: overlappingStrategyForTotals,
                                theme: chartHandler.getChartTheme()
                            };
                            BarChartUtils.drawLabels(totalLabelsDrawContext);
                        }
                    };

                    if (chartDef.valuesInChartDisplayOptions.overlappingStrategy === ValuesInChartOverlappingStrategy.AUTO_WITH_VALUES) {
                        // in that case we want to display all totals no matter what, so we display them first, to make them priority over values
                        drawTotalsLabels();
                        drawValuesLabels();
                    } else {
                        drawValuesLabels();
                        drawTotalsLabels();
                    }
                }

                rects
                    .attr('stroke-width', function(d) {
                        return d.isUnaggregated ? 0.5 : 0;
                    })
                    .attr('stroke', function(d) {
                        return chartHandler.legendsWrapper.getLegend(0).items[getFinalColorScaleIndex(d)].desaturatedColor;
                    })
                    .interrupt('updateColumns').transition('updateColumns')
                    .attr('height', function(d) {
                        return Math.max(0, Math.abs(yScale(d.base) - yScale(d.top)));
                    }) // height cannot be negative
                    .attr('y', function(d) {
                        return yScale(Math.max(d.base, d.top));
                    });

                // Clip paths to prevent bars from overlapping axis when user chose a custom range which results in the bar being cropped (out of visible range)
                ChartAxesUtils.isCroppedChart(chartDef) && SVGUtils.clipPaths(chartBase, g, wrapper);

                ReferenceLines.drawReferenceLines(
                    wrapper,
                    chartBase.vizWidth,
                    chartBase.vizHeight,
                    xAxis,
                    yAxis ? [{ ...yAxis, isPercentScale, formattingOptions: ChartAxesUtils.getYAxisNumberFormatting(chartDef.yAxesFormatting, yAxis.id) }] : [],
                    displayedReferenceLines,
                    referenceLinesValues
                );

                if (ChartHierarchyDimension.getCurrentHierarchyLevel(chartDef) > 0) {
                    const anchor = SVGUtils.addPlotAreaContextMenuAnchor(g, chartBase.vizWidth, chartBase.vizHeight);
                    chartBase.contextualMenu.addChartContextualMenuHandler(anchor.node(), undefined, () => ChartDrilldown.getDrillupActions(chartDef));
                }

                const colorFocusHandler = ColorFocusHandler.create(chartDef, chartHandler, wrapper, d3, g);
                ColorFocusHandler.appendFocusUnfocusMecanismToLegend(chartHandler.legendsWrapper.getLegend(0), colorFocusHandler);
            }
        };
    }

})();

;
(function() {
    'use strict';

    angular.module('dataiku.charts')
        .factory('LinesBrushDrawer', LinesBrushDrawer);

    // (!) This service previously was in static/dataiku/js/simple_report/curves/lines.js
    function LinesBrushDrawer(LinesUtils, ChartAxesUtils, ChartYAxisPosition, ChartDimension, ReferenceLines) {
        return function(g, chartDef, chartData, chartBase, linesData, facetIndex, brushAxes, referenceLinesValues = undefined) {

            const xDimension = ChartDimension.getGenericDimension(chartDef);
            const emptyBinsMode = xDimension.numParams.emptyBinsMode;
            const xLabels = chartData.getAxisLabels('x');
            const leftYAxisID = ChartAxesUtils.computeYAxisID(ChartYAxisPosition.LEFT);
            const rightYAxisID = ChartAxesUtils.computeYAxisID(ChartYAxisPosition.RIGHT);
            const xAxis = brushAxes.x;
            const leftYAxis = brushAxes[leftYAxisID];
            const rightYAxis = brushAxes[rightYAxisID];

            const wrappers = LinesUtils.drawWrappers(chartDef, chartBase, linesData, g, false, false, 'brush-wrapper');

            LinesUtils.drawPoints(chartDef, chartBase, chartData, facetIndex, wrappers, xAxis, xLabels, leftYAxis, rightYAxis, xDimension, emptyBinsMode, 1);

            const [, lineGs, lineDashGs] = LinesUtils.configureLines(chartDef, chartData, facetIndex, wrappers, undefined, xAxis, leftYAxis, rightYAxis, xDimension, xLabels, emptyBinsMode);

            LinesUtils.drawPaths(chartDef, chartBase, chartData, facetIndex, lineGs, xDimension, xLabels, xAxis, leftYAxis, rightYAxis, emptyBinsMode, false, false, 1, lineDashGs);

            if (referenceLinesValues) {
                const yAxes = [];
                if (leftYAxis) {
                    const isPercentScaleOnYAxis = chartDef.genericMeasures.some(measure => measure.displayAxis === 'axis1' && ChartDimension.isPercentScale([measure]));
                    const leftYFormattingOptions = ChartAxesUtils.getYAxisNumberFormatting(chartDef.yAxesFormatting, leftYAxis.id);
                    yAxes.push({ ...leftYAxis, isPercentScale: isPercentScaleOnYAxis, formattingOptions: leftYFormattingOptions });
                }
                if (rightYAxis) {
                    const isPercentScaleOnY2Axis = chartDef.genericMeasures.some(measure => measure.displayAxis !== 'axis1' && ChartDimension.isPercentScale([measure]));
                    const rightYFormattingOptions = ChartAxesUtils.getYAxisNumberFormatting(chartDef.yAxesFormatting, rightYAxis.id);
                    yAxes.push({ ...rightYAxis, isPercentScale: isPercentScaleOnY2Axis, formattingOptions: rightYFormattingOptions });
                }

                const displayedReferenceLines = ReferenceLines.getDisplayedReferenceLines(chartDef.referenceLines, chartDef.$axisSpecs && chartDef.$axisSpecs.x, chartDef.$axisSpecs && chartDef.$axisSpecs.y);
                ReferenceLines.drawReferenceLines(
                    d3.select(wrappers[0][wrappers[0].length-1]),
                    chartBase.vizWidth,
                    chartBase.vizHeight,
                    xAxis,
                    yAxes,
                    displayedReferenceLines,
                    referenceLinesValues,
                    null,
                    true // in brush
                );
            }
        };
    }

})();

;
(function() {
    'use strict';

    angular.module('dataiku.charts')
        .factory('LinesChart', LinesChart);

    // (!) This service previously was in static/dataiku/js/simple_report/curves/lines.js
    function LinesChart(ChartManager, ChartDataWrapperFactory, LinesDrawer, LinesUtils, ChartDataUtils, MonoFuture, ChartDimension, ReferenceLines, ColumnAvailability, ChartYAxisPosition, ChartAxesUtils, ChartCustomMeasures, ChartUsableColumns, ChartZoomControlAdapter, CHART_ZOOM_CONTROL_TYPES, LinesBrushDrawer, ChartStoreFactory) {
        return function($container, chartDef, chartHandler, axesDef, data, pivotRequest, uiDisplayState, chartActivityIndicator) {
            recomputeAndRedraw($container, chartDef, chartHandler, axesDef, data, pivotRequest, null, uiDisplayState, chartActivityIndicator);
        };

        function recomputeAndRedraw($container, chartDef, chartHandler, axesDef, data, pivotRequest, displayInterval, uiDisplayState, chartActivityIndicator) {

            const initialChartData = ChartDataWrapperFactory.chartTensorDataWrapper(data, axesDef);
            const pivotRequestCallback = MonoFuture().wrap(pivotRequest);

            const dataSpec = chartHandler.getDataSpec();

            const xDimension = ChartDimension.getGenericDimension(chartDef);
            const includeEmptyBins = LinesUtils.hasEmptyBinsToIncludeAsZero(xDimension, initialChartData);
            const customMeasures = ChartCustomMeasures.getMeasuresLikeCustomMeasures(dataSpec.datasetProjectKey, dataSpec.datasetName, chartHandler.getCurrentChartsContext());
            const allMeasures = ChartUsableColumns.getUsableColumns(dataSpec.datasetProjectKey, dataSpec.datasetName, chartHandler.getCurrentChartsContext()).filter(m => ['NUMERICAL', 'ALPHANUM', 'DATE'].includes(m.type));
            ColumnAvailability.updateAvailableColumns(chartDef.genericMeasures, allMeasures, customMeasures);
            const xSpec = { type: 'DIMENSION', mode: 'POINTS', dimension: xDimension, name: 'x', customExtent: chartDef.xAxisFormatting.customExtent };

            const facetLabels = initialChartData.getAxisLabels('facet') || [null], // We'll through the next loop only once if the chart is not facetted
                yExtents = ChartDataUtils.getMeasureExtents(chartDef, initialChartData, 'x', undefined, includeEmptyBins),
                leftYAxisID = ChartAxesUtils.computeYAxisID(ChartYAxisPosition.LEFT),
                rightYAxisID = ChartAxesUtils.computeYAxisID(ChartYAxisPosition.RIGHT),
                isLeftYPercentScale = yExtents[leftYAxisID].onlyPercent,
                isRightYPercentScale = yExtents[rightYAxisID].onlyPercent;

            const displayedReferenceLines = ReferenceLines.getDisplayedReferenceLines(chartDef.referenceLines, xSpec, undefined),
                referenceLinesValues = ReferenceLines.getReferenceLinesValues(displayedReferenceLines, initialChartData, allMeasures, chartDef.genericMeasures, customMeasures),
                referenceLinesExtents = ReferenceLines.getReferenceLinesExtents(displayedReferenceLines, referenceLinesValues, { [leftYAxisID]: { isPercentScale: isLeftYPercentScale }, [rightYAxisID]: { isPercentScale: isRightYPercentScale } });

            const leftYExtent = ReferenceLines.getExtentWithReferenceLines(yExtents[leftYAxisID].extent, referenceLinesExtents[leftYAxisID]),
                rightYExtent = ReferenceLines.getExtentWithReferenceLines(yExtents[rightYAxisID].extent, referenceLinesExtents[rightYAxisID]);

            ReferenceLines.mutateDimensionSpecForReferenceLine(ChartDataUtils.getAxisExtent(initialChartData, 'x', xSpec.dimension), referenceLinesExtents.x, xSpec);
            const linesData = LinesUtils.prepareData(chartDef, initialChartData);
            const isInteractive = ChartDimension.isInteractiveChart(chartDef);
            let zoomContext, initLinesZoom;

            const drawFrame = function(frameIdx, chartBase, redraw, chartData = initialChartData) {
                ReferenceLines.removeReferenceLines($container[0]);

                chartData.fixAxis('animation', frameIdx);

                if (isInteractive && !_.isNil(chartDef.$zoomControlInstanceId)) {
                    const zoomControlInstance = ChartZoomControlAdapter.get(chartDef.$zoomControlInstanceId);
                    zoomControlInstance.setZoomUtils({ ...zoomControlInstance.getZoomUtils(), frameIndex: frameIdx });
                }

                facetLabels.forEach(function(facetLabel, facetIndex) {
                    const g = d3.select(chartBase.$svgs.eq(facetIndex).find('g.chart').get(0));
                    LinesDrawer(g, chartDef, chartHandler, chartData.fixAxis('facet', facetIndex), chartBase, linesData, facetIndex, redraw, isInteractive);
                });
            };

            const cleanFrame = function(chartBase) {
                facetLabels.forEach(function(facetLabel, facetIndex) {
                    const g = d3.select(chartBase.$svgs.eq(facetIndex).find('g.chart').get(0));
                    LinesUtils.cleanChart(g, chartBase);
                });
            };

            const drawBrush = function(chartBase, g, brushAxes) {
                const isAnimated = chartBase.chartData.axesDef.animation !== undefined;
                const hasSubcharts = facetLabels && facetLabels.length > 1;

                if (isAnimated || hasSubcharts) {
                    return;
                }

                LinesBrushDrawer(g, chartDef, initialChartData, chartBase, linesData, 0, brushAxes, referenceLinesValues);
            };

            if (isInteractive) {
                zoomContext = {
                    d3,
                    chartContainer: $container,
                    chartDef,
                    chartHandler,
                    drawFrame,
                    drawBrush,
                    axesDef,
                    chartActivityIndicator,
                    cleanFrame,
                    pivotRequestCallback,
                    linesChartCallback: recomputeAndRedraw,
                    uiDisplayState
                };

                initLinesZoom = (zoomContext) => {
                    let previousZoomUtils = {};
                    if (!_.isNil(chartDef.$zoomControlInstanceId)) {
                        const zoomControlInstance = ChartZoomControlAdapter.get(chartDef.$zoomControlInstanceId);
                        previousZoomUtils = zoomControlInstance.getZoomUtils();
                    }

                    zoomContext = { ...zoomContext, zoomUtils: previousZoomUtils };

                    const id = ChartZoomControlAdapter.create(CHART_ZOOM_CONTROL_TYPES.LINES, zoomContext);
                    ChartZoomControlAdapter.init(CHART_ZOOM_CONTROL_TYPES.LINES, id);

                    return id;
                };
                chartHandler.forceRotation = 0.5;
            } else {
                chartHandler.forceRotation = undefined;
            }

            const leftYCustomExtent = ChartAxesUtils.getYAxisCustomExtent(chartDef.yAxesFormatting, leftYAxisID);
            const rightYCustomExtent = ChartAxesUtils.getYAxisCustomExtent(chartDef.yAxesFormatting, rightYAxisID);

            const ySpecs = {
                [leftYAxisID]: { id: leftYAxisID, type: 'MEASURE', extent: leftYExtent, isPercentScale: isLeftYPercentScale, customExtent: leftYCustomExtent, position: ChartYAxisPosition.LEFT },
                [rightYAxisID]: { id: rightYAxisID, type: 'MEASURE', extent: rightYExtent, isPercentScale: isRightYPercentScale, customExtent: rightYCustomExtent, position: ChartYAxisPosition.RIGHT }
            };

            if (displayInterval && _.isFinite(displayInterval[0]) && _.isFinite(displayInterval[1])) {
                xSpec.initialInterval = { min: displayInterval[0], max: displayInterval[1] };
            }

            if (!isInteractive) {
                // Use x custom user extent only when there is no zoom, otherwise let zoom manage x axis range
                xSpec.customExtent = chartDef.xAxisFormatting.customExtent;
            }

            const availableAxes = ReferenceLines.getAvailableAxesForReferenceLines(chartDef);
            const axisSpecs = { x: xSpec, ...ySpecs };

            ChartAxesUtils.setNumberOfBinsToDimensions(initialChartData, chartDef, axisSpecs);
            ReferenceLines.updateAvailableAxisOptions([
                { axis: 'LEFT_Y_AXIS', isDisplayed: availableAxes['LEFT_Y_AXIS'], isNumerical: true, isPercentScale: ySpecs[leftYAxisID].isPercentScale },
                { axis: 'RIGHT_Y_AXIS', isDisplayed: availableAxes['RIGHT_Y_AXIS'], isNumerical: true, isPercentScale: ySpecs[rightYAxisID].isPercentScale },
                { axis: 'X_AXIS', isDisplayed: true, isNumerical: ChartAxesUtils.isNumerical(xSpec) && !ChartDimension.hasOneTickPerBin(xSpec.dimension), isContinuousDate: ChartAxesUtils.isContinuousDate(xSpec) }
            ]);

            ChartManager.initChart(chartDef, chartHandler, initialChartData, $container, drawFrame,
                axisSpecs,
                { type: 'DIMENSION', name: 'color', dimension: chartDef.genericDimension1[0] },
                zoomContext, initLinesZoom);
        };
    }
})();

;
(function() {
    'use strict';

    angular.module('dataiku.charts')
        .factory('LinesDrawer', LinesDrawer);

    // (!) This service previously was in static/dataiku/js/simple_report/curves/lines.js
    function LinesDrawer(Fn, LinesUtils, ChartDimension, ChartAxesUtils, ReferenceLines, ColumnAvailability, ChartCustomMeasures, ChartUsableColumns) {

        return function(g, chartDef, chartHandler, chartData, chartBase, linesData, facetIndex, redraw, isInteractive, iLabelsBoundingBoxes, getRefLineValuesOverride, refLinesContainer) {

            const xDimension = ChartDimension.getGenericDimension(chartDef);
            const emptyBinsMode = xDimension.numParams.emptyBinsMode;
            const xLabels = chartData.getAxisLabels('x');
            const rightYAxis = ChartAxesUtils.getRightYAxis(chartBase.yAxes);
            const leftYAxis = ChartAxesUtils.getLeftYAxis(chartBase.yAxes);
            let xAxis = chartBase.xAxis;

            chartBase.DOMUtils = chartBase.DOMUtils || {};
            chartBase.DOMUtils[facetIndex] = chartBase.DOMUtils[facetIndex] || {};

            const wrappers = LinesUtils.drawWrappers(chartDef, chartBase, linesData, g, isInteractive, redraw, 'wrapper');

            // During interaction, prevent re-drawing the points and remove them from the DOM for performances.
            if (!redraw) {
                chartBase.DOMUtils[facetIndex].points = LinesUtils.drawPoints(chartDef, chartBase, chartData, facetIndex, wrappers, xAxis, xLabels, leftYAxis, rightYAxis, xDimension, emptyBinsMode);
                LinesUtils.addTooltipAndHighlightAndContextualMenuHandlers(chartDef, chartHandler, chartBase, facetIndex, g, wrappers);
                chartBase.DOMUtils[facetIndex].labels = LinesUtils.drawLabels(chartDef, chartBase, chartData, chartHandler.getChartTheme(), facetIndex, wrappers, xAxis, xLabels, leftYAxis, rightYAxis, xDimension, emptyBinsMode, iLabelsBoundingBoxes);
            } else if (isInteractive && !chartBase.DOMUtils[facetIndex].pointsHaveBeenRemoved) {
                if(chartBase.DOMUtils[facetIndex].points){
                    chartBase.DOMUtils[facetIndex].points.remove();
                }
                if(chartBase.DOMUtils[facetIndex].labels){
                    chartBase.DOMUtils[facetIndex].labels.remove();
                }
                chartBase.DOMUtils[facetIndex].pointsHaveBeenRemoved = true;
            }

            const [lineGenerator, lineGs, lineDashGs] = LinesUtils.configureLines(chartDef, chartData, facetIndex, wrappers, chartBase.DOMUtils[facetIndex].lineGenerator, xAxis, leftYAxis, rightYAxis, xDimension, xLabels, emptyBinsMode);

            chartBase.DOMUtils[facetIndex].lineGenerator = lineGenerator;

            // Add thicker, invisible lines to catch mouseover event
            [lineGs, lineDashGs].forEach(lineGs => {
                let hiddenLines = lineGs.selectAll('path.masked');

                if (!redraw) {
                    hiddenLines = hiddenLines.data(function(d) {
                        return [d];
                    });
                    hiddenLines.enter()
                        .insert('path')
                        .attr('class', 'line masked')
                        .attr('fill', 'none')
                        .attr('stroke-width', '10')
                        .attr('stroke', 'transparent');
                    hiddenLines.exit().remove();
                }

                hiddenLines.attr('d', Fn.SELF);
            });

            LinesUtils.drawPaths(chartDef, chartBase, chartData, facetIndex, lineGs, xDimension, xLabels, xAxis, leftYAxis, rightYAxis, emptyBinsMode, redraw, !isInteractive, chartDef.strokeWidth, lineDashGs);

            const isPercentScaleOnYAxis = chartDef.genericMeasures.some(measure => measure.displayAxis === 'axis1' && ChartDimension.isPercentScale([measure]));
            const isPercentScaleOnY2Axis = chartDef.genericMeasures.some(measure => measure.displayAxis !== 'axis1' && ChartDimension.isPercentScale([measure]));
            const leftYFormattingOptions = leftYAxis && ChartAxesUtils.getYAxisNumberFormatting(chartDef.yAxesFormatting, leftYAxis.id);
            const rightYFormattingOptions = rightYAxis && ChartAxesUtils.getYAxisNumberFormatting(chartDef.yAxesFormatting, rightYAxis.id);

            const dataSpec = chartHandler.getDataSpec();
            const customMeasures = ChartCustomMeasures.getMeasuresLikeCustomMeasures(dataSpec.datasetProjectKey, dataSpec.datasetName, chartHandler.getCurrentChartsContext());
            const allMeasures = ChartUsableColumns.getUsableColumns(dataSpec.datasetProjectKey, dataSpec.datasetName, chartHandler.getCurrentChartsContext()).filter(m => ['NUMERICAL', 'ALPHANUM', 'DATE'].includes(m.type));
            ColumnAvailability.updateAvailableColumns(chartDef.genericMeasures, allMeasures, customMeasures);

            const displayedReferenceLines = ReferenceLines.getDisplayedReferenceLines(chartDef.referenceLines, xAxis, undefined),
                referenceLinesValues = ReferenceLines.getReferenceLinesValues(displayedReferenceLines, chartData, allMeasures, chartDef.genericMeasures, customMeasures, true, getRefLineValuesOverride);

            const yAxes = [];
            if (leftYAxis) {
                yAxes.push({ ...leftYAxis, isPercentScale: isPercentScaleOnYAxis, formattingOptions: leftYFormattingOptions });
            }
            if (rightYAxis) {
                yAxes.push({ ...rightYAxis, isPercentScale: isPercentScaleOnY2Axis, formattingOptions: rightYFormattingOptions });
            }

            xAxis = { ...xAxis, formattingOptions: xAxis && chartDef.xAxisFormatting };

            ReferenceLines.drawReferenceLines(
                // For a mix chart, if we have lines, we will draw ref lines in the last line wrapper, if we have none - in the bar wrapper.
                wrappers[0].length ? d3.select(wrappers[0][wrappers[0].length-1]) : refLinesContainer,
                chartBase.vizWidth,
                chartBase.vizHeight,
                xAxis,
                yAxes,
                displayedReferenceLines,
                referenceLinesValues
            );

            if (!redraw) {
                const isInteractive = ChartDimension.isInteractiveChart(chartDef);
                const isCroppedChart = ChartAxesUtils.isCroppedChart(chartDef);
                /*
                 * Clip paths to prevent lines from overlapping axis:
                 * - during offline zoom
                 * - or when user chose a custom range which results in the line being cropped (out of visible range)
                 */
                (isInteractive || isCroppedChart) && LinesUtils.clipPaths(chartBase, chartDef, g, wrappers);
            }
        };
    }

})();

;
(function() {
    'use strict';

    angular.module('dataiku.charts')
        .factory('LinesUtils', LinesUtils);

    const CLIP_PATH_ID = 'lines-chart-clip-path';


    // (!) This service previously was in static/dataiku/js/simple_report/curves/lines.js
    function LinesUtils(ChartDimension, Fn, ChartAxesUtils, CHART_TYPES, ColorFocusHandler, SVGUtils, ValuesInChartOverlappingStrategy, ValuesInChartPlacementMode, ChartDrilldown, ChartHierarchyDimension, ChartColorUtils) {


        const svc = {

            drawPaths: function(chartDef, chartBase, chartData, facetIndex, lineGs, xDimension, xLabels, xAxis, leftYAxis, rightYAxis, emptyBinsMode, redraw, transition, strokeWidth, lineDashGs) {

                const paths = lineGs.selectAll('path.visible').data(function(d) {
                    return [d];
                });

                paths.enter()
                    .insert('path')
                    .attr('class', 'line visible')
                    .attr('fill', 'none')
                    .attr('stroke-width', strokeWidth)
                    .attr('opacity', chartDef.colorOptions.transparency);

                paths.exit().remove();

                const dashPaths = lineDashGs.selectAll('path.visible').data(function(d) {
                    return [d];
                });

                dashPaths.enter()
                    .insert('path')
                    .attr('class', 'line visible')
                    .attr('fill', 'none')
                    .attr('stroke-dasharray', 12)
                    .attr('stroke-width', strokeWidth);
                dashPaths.exit().remove();
                dashPaths.attr('d', Fn.SELF);

                if (!transition) {
                    paths.attr('d', Fn.SELF)
                        .each(function() {
                            const path = d3.select(this);
                            const wrapper = d3.select(this.parentNode.parentNode);
                            svc.drawPath(path, wrapper, emptyBinsMode, redraw, chartBase, svc.xCoord, svc.yCoord, chartData, chartDef, xDimension, xLabels, xAxis, leftYAxis, rightYAxis);
                        });
                } else {
                    paths.interrupt('updateLines').transition('updateLines').attr('d', Fn.SELF)
                        .each('end', function() {
                            const path = d3.select(this);
                            const wrapper = d3.select(this.parentNode.parentNode);
                            svc.drawPath(path, wrapper, emptyBinsMode, redraw, chartBase, svc.xCoord, svc.yCoord, chartData, chartDef, xDimension, xLabels, xAxis, leftYAxis, rightYAxis);
                        });
                }
            },

            /**
             * - Build a d3 line generator if none provided.
             * - In the wrappers, creates <g>s with class "line".
             * - Bind to them the data computed by the line generator for the given points data.
             */
            configureLines: function(chartDef, chartData, facetIndex, wrappers, lineGenerator, xAxis, leftYAxis, rightYAxis, xDimension, xLabels, emptyBinsMode) {

                if (!lineGenerator) {
                    lineGenerator = d3.svg.line()
                        .x(d => svc.xCoord(xDimension, xLabels, xAxis)(d))
                        .y(d => svc.yCoord(d, chartDef, chartData, leftYAxis, rightYAxis))
                        // in DASHED mode, the dashed lines are drawn separately => we must remove missing values from the main line
                        .defined(x => emptyBinsMode === 'ZEROS' || svc.nonZeroCountFilter(x, facetIndex, chartData));
                    // If smoothing, change the interpolation mode (the process of adding new points between existing ones) to cubic interpolation that preserves monotonicity in y.
                    if (chartDef.smoothing) {
                        lineGenerator.interpolate('monotone');
                    }
                }

                const lineGs = wrappers.selectAll('g.line').data(function(d) {
                    d.filteredPointsData = d.pointsData.filter(d => svc.nonZeroCountFilter(d, facetIndex, chartData));
                    const data = (emptyBinsMode === 'ZEROS' || emptyBinsMode == 'DASHED') ? d.pointsData : d.filteredPointsData;
                    return [lineGenerator(data)];
                });

                const lineDashGs = wrappers.selectAll('g.dashedline').data(function(d) {
                    if (emptyBinsMode === 'DASHED') {
                        // null is added after every segment in order to make them disconnected (using defined() below)
                        const data = svc.getEmptySegments(d.pointsData).flatMap(s => [s[0], s[1], null]);
                        const segmentGenerator = d3.svg.line()
                            .x(d => svc.xCoord(xDimension, xLabels, xAxis)(d))
                            .y(d => svc.yCoord(d, chartDef, chartData, leftYAxis, rightYAxis))
                            .defined(d => d != null);
                        return [segmentGenerator(data)];
                    }
                    return [];
                });

                lineGs.enter().insert('g', ':first-child').attr('class', 'line');
                lineGs.exit().remove();

                lineDashGs.enter().insert('g', ':first-child').attr('class', 'dashedline');
                lineDashGs.exit().remove();

                return [lineGenerator, lineGs, lineDashGs];
            },

            /**
             * - In the given line wrappers, create <circle> with class "point", a given radius for each points of the lines.
             * - These points will have a color defined by the color scale and an attached tooltip if requested.
             */
            drawPoints: function(chartDef, chartBase, chartData, facetIndex, wrappers, xAxis, xLabels, leftYAxis, rightYAxis, xDimension, emptyBinsMode, pointsRadius) {

                let points = wrappers.selectAll('circle.point');

                points = points.data(function(d) {
                    return (emptyBinsMode === 'ZEROS') ? d.pointsData : (d.filteredPointsData = d.pointsData.filter(d => svc.nonZeroCountFilter(d, facetIndex, chartData)));
                }, Fn.prop('x'));

                points.enter().append('circle')
                    .attr('class', 'point point--masked')
                    .attr('r', function(d) {
                        return pointsRadius ||
                        (chartDef.valuesInChartDisplayOptions
                            && chartDef.valuesInChartDisplayOptions.displayValues
                            && d.valuesInChartDisplayOptions.displayValues
                            ? Math.round(chartDef.strokeWidth / 2) + 1 : 5);
                    })
                    .attr('fill', function(d) {
                        return chartBase.colorScale(d.colorScaleIndex);
                    })
                    .attr('opacity', function(d) {
                        return chartDef.valuesInChartDisplayOptions && chartDef.valuesInChartDisplayOptions.displayValues && d.valuesInChartDisplayOptions.displayValues ? chartDef.colorOptions.transparency : 0;
                    });

                // Remove potential duplicates
                points.exit().remove();

                points
                    .interrupt('movePoints')
                    .transition('movePoints')
                    .duration(200)
                    .ease('easeOutQuad')
                    .attr('cx', d => svc.xCoord(xDimension, xLabels, xAxis)(d))
                    .attr('cy', d => svc.yCoord(d, chartDef, chartData, leftYAxis, rightYAxis));

                // Remove points that are not linked to others through lines.
                wrappers.selectAll('circle.lonely').remove();

                return points;
            },

            /**
             * - Draw the labels for all displayed lines
             */
            drawLabels: function(chartDef, chartBase, chartData, theme, facetIndex, wrappers, xAxis, xLabels, leftYAxis, rightYAxis, xDimension, emptyBinsMode, labelCollisionDetectionHandler) {
                // to store all nodes across the different lines
                const selectionNodes = [];

                if (chartDef.valuesInChartDisplayOptions && chartDef.valuesInChartDisplayOptions.displayValues) {
                    const backgroundXMargin = 3;

                    // init the collision detection
                    const mainLabelCollisionDetectionHandler = labelCollisionDetectionHandler || SVGUtils.initLabelCollisionDetection(chartBase);
                    const labelCollisionDetectionHandlersByColorScaleIndex = {};
                    const getCurrentColorLabelCollisionDetectionHandler = (d) => {
                        const colorScaleIndex = d.colorScaleIndex;
                        if (!labelCollisionDetectionHandlersByColorScaleIndex[colorScaleIndex]) {
                            // we need a separate label collision detection handler for each color scale
                            labelCollisionDetectionHandlersByColorScaleIndex[colorScaleIndex] = SVGUtils.initLabelCollisionDetection(chartBase);
                        }
                        return labelCollisionDetectionHandlersByColorScaleIndex[colorScaleIndex];
                    };



                    const getPointPosition = d => ({
                        x: d ? svc.xCoord(xDimension, xLabels, xAxis)(d) : null,
                        y: d ? svc.yCoord(d, chartDef, chartData, leftYAxis, rightYAxis) : null
                    });

                    const isEqualPosition = (point1, point2) => {
                        return point1.x === point2.x && point1.y === point2.y;
                    };

                    const checkBoundaries = (position, textElement) => {
                        const textElementWidth = textElement.width;
                        const rectangleHeight = textElement.height;
                        // make sure we are within the chart boundaries
                        const newPosition = {
                            x: position.x,
                            y: position.y
                        };
                        const minVizX = 0;
                        const maxVizX = chartBase.vizWidth;
                        const minVizY = -chartBase.margins.top;
                        const maxVizY = chartBase.vizHeight;
                        let isOutside = false;

                        if (position.x - textElementWidth / 2 < minVizX) {
                            newPosition.x += (minVizX - (position.x - textElementWidth / 2));
                            isOutside = true;
                        } else if (position.x + textElementWidth / 2 > maxVizX) {
                            newPosition.x -= (position.x + textElementWidth / 2 - maxVizX);
                            isOutside = true;
                        }

                        // keep in mind that the position coordinates correspond to the top center of the text element
                        if (position.y < minVizY) {
                            newPosition.y += (minVizY - position.y);
                            isOutside = true;
                        } else if (position.y + rectangleHeight > maxVizY) {
                            newPosition.y -= (position.y + rectangleHeight - maxVizY);
                            isOutside = true;
                        }

                        return {
                            newPosition,
                            isOutside
                        };
                    };


                    const findFinalPosition = (possiblePositions, textElement, onHoverMode, adjustForBoundaries) => {

                        let finalPosition;
                        let finalRectangles;
                        let isOverlap = false;
                        let isOverlapAlt = false;
                        const fittedPositions = [];
                        const labelDetectionHandler = onHoverMode ? getCurrentColorLabelCollisionDetectionHandler(textElement) : mainLabelCollisionDetectionHandler;
                        for (let idx = 0; idx < possiblePositions.length; idx += 1) {
                            const position = possiblePositions[idx];
                            if (adjustForBoundaries) {
                                const boundariesCheck = checkBoundaries(position, textElement);
                                if (boundariesCheck.isOutside) {
                                    fittedPositions.push(boundariesCheck.newPosition);
                                    continue;
                                }
                            }

                            const rectangles = SVGUtils.getRectanglesFromPosition(position, textElement, chartDef.valuesInChartDisplayOptions.textFormatting.fontSize);
                            if (!labelDetectionHandler.checkOverlaps(rectangles)) {
                                finalPosition = position;
                                finalRectangles = rectangles;
                                break;
                            }
                        }

                        if (!finalPosition) {
                            // no valid position found, maybe some of them crossed the chart boundaries, in that case we check the fitted positions
                            for (let idx = 0; idx < fittedPositions.length; idx += 1) {
                                const position = fittedPositions[idx];
                                const rectangles = SVGUtils.getRectanglesFromPosition(position, textElement, chartDef.valuesInChartDisplayOptions.textFormatting.fontSize);
                                if (!labelDetectionHandler.checkOverlaps(rectangles)) {
                                    finalPosition = position;
                                    finalRectangles = rectangles;
                                    break;
                                }
                            }

                            if (!finalPosition) {
                                // still no valid position, just take the first one
                                finalPosition = fittedPositions[0] || possiblePositions[0];
                                finalRectangles = SVGUtils.getRectanglesFromPosition(finalPosition, textElement, chartDef.valuesInChartDisplayOptions.textFormatting.fontSize);
                                if (onHoverMode) {
                                    isOverlapAlt = true;
                                } else {
                                    isOverlap = true;
                                }
                            }
                        }

                        return {
                            labelPosition: finalPosition,
                            labelRectangles: finalRectangles,
                            isOverlap,
                            isOverlapAlt
                        };
                    };

                    const getLabelCoordinates = (previousPoint, currentPoint, nextPoint, currentElement, onHoverMode) => {
                        if (!currentPoint) {
                            return null;
                        }
                        // one thing to keep in mind, the position of the text elem will be at the top center of the text element

                        const defaultSpacing = chartDef.type === CHART_TYPES.MULTI_COLUMNS_LINES ? 5 : 12;
                        const distanceFromDataPoint = currentElement.valuesInChartDisplayOptions.spacing ?? defaultSpacing;
                        const placementMode = currentElement.valuesInChartDisplayOptions.placementMode;
                        const adjustBoundaries = !placementMode || placementMode === ValuesInChartPlacementMode.AUTO;
                        const overlappingKey = onHoverMode ? 'isOverlapAlt' : 'isOverlap';
                        const labelDetectionHandler = onHoverMode ? getCurrentColorLabelCollisionDetectionHandler(currentElement) : mainLabelCollisionDetectionHandler;

                        const computeCoordinatesFromAngle = (angle, unitVector) => {
                            const tmpRotatedVector = {
                                x: unitVector.x * Math.cos(angle) + unitVector.y * Math.sin(angle),
                                y: -unitVector.x * Math.sin(angle) + unitVector.y * Math.cos(angle)
                            };

                            // compute distance between rectangle center and rectangle boundaries following a particular angle
                            const angleFromXAxis = Math.atan2(tmpRotatedVector.y, tmpRotatedVector.x);
                            const counterclockwiseAngle = Math.PI - angleFromXAxis;
                            const boundaryDistance = Math.min(
                                ((currentElement.width / 2) / Math.abs(Math.cos(counterclockwiseAngle))),
                                ((currentElement.height / 2) / Math.abs(Math.sin(counterclockwiseAngle)))
                            );
                            const distanceToBboxCenter = distanceFromDataPoint + boundaryDistance;

                            const labelCoordinates = {
                                x: currentPoint.x + distanceToBboxCenter * tmpRotatedVector.x,
                                // the y position of the label is at the top center of the text element
                                y: currentPoint.y + distanceToBboxCenter * tmpRotatedVector.y - currentElement.height / 2
                            };

                            return labelCoordinates;
                        };

                        let belowLinePosition = {
                            x: currentPoint.x,
                            y: currentPoint.y + distanceFromDataPoint
                        };
                        let aboveLinePosition = {
                            x: currentPoint.x,
                            y: currentPoint.y - (distanceFromDataPoint + currentElement.height)
                        };

                        let possiblePositions = [belowLinePosition, aboveLinePosition];

                        if (previousPoint && nextPoint) {
                            // standard case, with one data point before and one after

                            // we compute the angle made by the 3 consecutive points
                            const dotProduct = (previousPoint.x - currentPoint.x) * (nextPoint.x - currentPoint.x) + (previousPoint.y - currentPoint.y) * (nextPoint.y - currentPoint.y);
                            const magnitude1 = Math.sqrt(Math.pow(previousPoint.x - currentPoint.x, 2) + Math.pow(previousPoint.y - currentPoint.y, 2));
                            const magnitude2 = Math.sqrt(Math.pow(nextPoint.x - currentPoint.x, 2) + Math.pow(nextPoint.y - currentPoint.y, 2));
                            const angleCosinus = dotProduct / (magnitude1 * magnitude2);

                            // angle is always between 0 and 180 degrees
                            const curveAngle = Math.acos(Math.max(-1, Math.min(1, angleCosinus)));

                            // we have to compute the cross product to get the orientation (clockwise or not), crossProduct < 0 => clockwise orientation
                            const crossProduct = (previousPoint.x - currentPoint.x) * (nextPoint.y - currentPoint.y) - (previousPoint.y - currentPoint.y) * (nextPoint.x - currentPoint.x);

                            // compute the start unit vector
                            const unitVector = {
                                x: (previousPoint.x - currentPoint.x) / magnitude1,
                                y: (previousPoint.y - currentPoint.y) / magnitude1
                            };

                            // preferred position, at the middle of the widest angle formed by the 3 points
                            const rotationAngle = ((2 * Math.PI - curveAngle) / 2) * (crossProduct > 0 ? 1 : -1);
                            // first alternative, at the middle of the smallest angle formed by the 3 points
                            const alternativeAngle1 = (curveAngle / 2) * (crossProduct > 0 ? -1 : 1);
                            // second alternative, at the middle of the 2 preferred positions
                            const alternativeAngle2 = (rotationAngle + alternativeAngle1) / 2;
                            // third alternative, at the opposite of the second alternative
                            const alternativeAngle3 = alternativeAngle2 + Math.PI;

                            const preferredPosition = computeCoordinatesFromAngle(rotationAngle, unitVector);
                            const alternativePosition1 = computeCoordinatesFromAngle(alternativeAngle1, unitVector);
                            const alternativePosition2 = computeCoordinatesFromAngle(alternativeAngle2, unitVector);
                            const alternativePosition3 = computeCoordinatesFromAngle(alternativeAngle3, unitVector);

                            possiblePositions = [preferredPosition, alternativePosition1, alternativePosition2, alternativePosition3];

                            aboveLinePosition = crossProduct <= 0 ? preferredPosition : alternativePosition1;
                            belowLinePosition = crossProduct <= 0 ? alternativePosition1 : preferredPosition;
                        } else if (nextPoint || previousPoint) {
                            let unitVector;
                            if (nextPoint) {
                                // first data point
                                const magnitude = Math.sqrt(Math.pow(nextPoint.x - currentPoint.x, 2) + Math.pow(nextPoint.y - currentPoint.y, 2));

                                // compute the start unit vector
                                unitVector = {
                                    x: (currentPoint.x - nextPoint.x) / magnitude,
                                    y: (currentPoint.y - nextPoint.y) / magnitude
                                };
                            } else {
                                // last data point
                                const magnitude = Math.sqrt(Math.pow(currentPoint.x - previousPoint.x, 2) + Math.pow(currentPoint.y - previousPoint.y, 2));

                                // compute the start unit vector
                                unitVector = {
                                    x: (previousPoint.x - currentPoint.x) / magnitude,
                                    y: (previousPoint.y - currentPoint.y) / magnitude
                                };
                            }

                            const rotationAngle = Math.PI / 2;
                            const alternativeAngle1 = -(Math.PI / 2);
                            // second alternative, at the middle of the 2 preferred positions
                            const alternativeAngle2 = (rotationAngle + alternativeAngle1) / 2;
                            // third alternative, at the opposite of the second alternative
                            const alternativeAngle3 = alternativeAngle2 + Math.PI;

                            belowLinePosition = computeCoordinatesFromAngle(rotationAngle, unitVector);
                            aboveLinePosition = computeCoordinatesFromAngle(alternativeAngle1, unitVector);
                            const alternativePosition2 = computeCoordinatesFromAngle(alternativeAngle2, unitVector);
                            const alternativePosition3 = computeCoordinatesFromAngle(alternativeAngle3, unitVector);

                            possiblePositions = [belowLinePosition, aboveLinePosition, alternativePosition2, alternativePosition3];
                        }

                        if (aboveLinePosition && (placementMode === ValuesInChartPlacementMode.ABOVE)) {
                            possiblePositions = [aboveLinePosition];
                        } else if (belowLinePosition && (placementMode === ValuesInChartPlacementMode.BELOW)) {
                            possiblePositions = [belowLinePosition];
                        }
                        const result = findFinalPosition(possiblePositions, currentElement, onHoverMode, adjustBoundaries);

                        // add the selected position to the for next detections
                        const hideOverlaps = chartDef.valuesInChartDisplayOptions.overlappingStrategy === ValuesInChartOverlappingStrategy.AUTO;
                        const isDisplayed = !!currentElement.valuesInChartDisplayOptions?.displayValues;
                        if ((!hideOverlaps || !result[overlappingKey]) && isDisplayed && result.labelRectangles) {
                            labelDetectionHandler.addBoundingBoxesToQuadTree(result.labelRectangles);
                        }

                        return result;
                    };

                    wrappers.each(function() {
                        const wrapper = d3.select(this);
                        const datum = wrapper.datum();
                        const data = (emptyBinsMode === 'ZEROS') ? datum.pointsData : (datum.filteredPointsData = datum.pointsData.filter(d => svc.nonZeroCountFilter(d, facetIndex, chartData)));
                        const finalData = data.filter(d => d.valuesInChartDisplayOptions && d.valuesInChartDisplayOptions.displayValues && d.valuesInChartDisplayOptions.textFormatting);

                        const getExtraData = (d, idx, onHoverMode) => {
                            const currentPosition = getPointPosition(d);

                            // check if the point is inside charts limits
                            const minVizX = 0;
                            const maxVizX = chartBase.vizWidth;
                            const minVizY = -chartBase.margins.top;
                            const maxVizY = chartBase.vizHeight;

                            if (currentPosition.x < minVizX || currentPosition.x > maxVizX || currentPosition.y < minVizY || currentPosition.y > maxVizY) {
                                return { isInvalid: true };
                            }

                            let previousPosition = null;
                            let nextPosition = null;
                            // find the first predecessor with a different position
                            let prevIndex = idx - 1;
                            while (prevIndex >= 0 && !previousPosition) {
                                const position = getPointPosition(finalData[prevIndex]);
                                if (!isEqualPosition(position, currentPosition)) {
                                    previousPosition = position;
                                }
                                prevIndex -= 1;
                            }
                            // find the first successor with a different position
                            let nextIndex = idx + 1;
                            while (nextIndex < finalData.length && !nextPosition) {
                                const position = getPointPosition(finalData[nextIndex]);
                                if (!isEqualPosition(position, currentPosition)) {
                                    nextPosition = position;
                                }
                                nextIndex += 1;
                            }

                            return getLabelCoordinates(previousPosition, currentPosition, nextPosition, d, onHoverMode);
                        };

                        const labelsDrawContext = {
                            node: wrapper,
                            data: finalData,
                            opacity: chartDef.colorOptions.transparency,
                            textFormatting: chartDef.valuesInChartDisplayOptions.textFormatting,
                            overlappingStrategy: chartDef.valuesInChartDisplayOptions.overlappingStrategy,
                            colorScale: chartBase.colorScale,
                            // to store computed data reused by other function (getLabelXPosition, getBackgroundXPosition...)
                            getExtraData: (d, _, idx, onHoverMode) => {
                                return getExtraData(d, idx, onHoverMode);
                            },
                            getLabelXPosition: (d) => {
                                return d.isInvalid ? 0 : d.labelPosition.x;
                            },
                            getLabelYPosition: (d) => {
                                return d.isInvalid ? 0 : d.labelPosition.y;
                            },
                            getLabelText: (d) => {
                                return chartBase.measureFormatters[d.aggregationIndex](chartData.aggr(d.aggregationIndex).get(d));
                            },
                            getBackgroundXPosition: (d) => {
                                if (d.isInvalid) {
                                    return 0;
                                }
                                const { x } = d.labelPosition;
                                return x - d.width / 2 - backgroundXMargin;
                            },
                            getBackgroundYPosition: (d) => {
                                if (d.isInvalid) {
                                    return 0;
                                }
                                const { y } = d.labelPosition;
                                return SVGUtils.getSubTextYPosition(d, y);
                            },
                            backgroundXMargin,
                            theme
                        };

                        const hasColorDim = ChartColorUtils.getColorDimensionOrMeasure(chartDef) !== undefined;
                        const labelsSelection = SVGUtils.drawLabels(labelsDrawContext, 'lines', hasColorDim);
                        selectionNodes.push(...(labelsSelection?.[0] ?? []));
                    });
                }

                // return the selection to be able to remove them during interaction
                return d3.selectAll(selectionNodes);
            },

            /**
             * - Creates a <g> (group) element with class "wrapper" for each line to be drawn.
             * - Joins the lines data with these wrappers.
             * - Strokes them according to the chart's color scale, set the opacity as per the options, and attach tooltips if requested.
             * - We need to add a key selector (id) to ensures consistent binding between lines data to lines DOM while zooming.
             */
            drawWrappers: function(chartDef, chartBase, linesData, g, isInteractive, redraw, className) {
                let wrappers = g.selectAll('g.' + className);

                if (!redraw) {
                    wrappers = wrappers.data(linesData, d => d.id);
                    wrappers.enter().append('g').attr('class', className)
                        .attr('stroke', function(d) {
                            return chartBase.colorScale(d.colorScaleIndex);
                        });

                    // Remove the exiting selection ie existing DOM elements for which no new data has been found to prevent duplicates.
                    wrappers.exit().remove();
                }

                return wrappers;
            },

            computeTooltipCoords: (datum, facetIndex) => ({ measure: datum.measure, x: datum.xCoord, color: datum.color, facet: facetIndex, colorScaleIndex: datum.colorScaleIndex }),

            getTooltipTargetElementProperties: (domElement, facetIndex) => {
                if (domElement.classList.contains('point')) {
                    const element = d3.select(domElement);
                    element.attr('tooltip-el', true);
                    const datum = d3.select(domElement).datum();
                    if (datum == null) {
                        return null;
                    }
                    return {
                        coords: svc.computeTooltipCoords(datum, facetIndex),
                        measure: datum.measure != null ? datum.measure : 0,
                        showTooltip: true
                    };
                } else if (domElement.classList.contains('line')) {
                    // When pointing a line, we need to access its grand-parent element, which is the line wrapper containing its coordinates.
                    const datum = d3.select(domElement.parentNode.parentNode).datum();
                    if (datum == null) {
                        return null;
                    }
                    return {
                        coords: svc.computeTooltipCoords(datum, facetIndex),
                        measure: datum.measure != null ? datum.measure : 0,
                        showTooltip: false
                    };
                }
                return null;
            },

            computeContextualMenuCoords: (datum, facetIndex) => ({ x: datum.xCoord, color: datum.color, facet: facetIndex }),

            getContextualMenuTargetElementProperties: (domElement, facetIndex) => {
                if (domElement.classList.contains('point')) {
                    const element = d3.select(domElement);
                    element.attr('tooltip-el', true);
                    const datum = d3.select(domElement).datum();
                    return { coords: svc.computeContextualMenuCoords(datum, facetIndex) };
                }
                return null;
            },

            addTooltipAndHighlightAndContextualMenuHandlers: function(chartDef, chartHandler, chartBase, facetIndex, g, wrappers) {
                const colorFocusHandler = ColorFocusHandler.create(chartDef, chartHandler, wrappers, d3, g);

                ColorFocusHandler.appendFocusUnfocusMecanismToLegend(chartHandler.legendsWrapper.getLegend(0), colorFocusHandler);

                const tooltipOptions = { displayColorTooltip: chartDef.type !== CHART_TYPES.MULTI_COLUMNS_LINES };
                chartBase.tooltips.addChartTooltipAndHighlightHandlers(g.node(), (domElement) => svc.getTooltipTargetElementProperties(domElement, facetIndex), colorFocusHandler, tooltipOptions);
                const getTargetElementProperties = (domElement) => svc.getContextualMenuTargetElementProperties(domElement, facetIndex);
                let getCustomActions;
                if (ChartHierarchyDimension.getCurrentHierarchyLevel(chartDef) > 0) {
                    const anchor = SVGUtils.addPlotAreaContextMenuAnchor(g, chartBase.vizWidth, chartBase.vizHeight);
                    getCustomActions = (properties) => properties ? [] : ChartDrilldown.getDrillupActions(chartDef);
                    // we need to attach contextual menu to 2 elements in order to have it on background click, since svg (g) has no background
                    chartBase.contextualMenu.addChartContextualMenuHandler(anchor.node(), getTargetElementProperties, getCustomActions);
                }
                chartBase.contextualMenu.addChartContextualMenuHandler(g.node(), getTargetElementProperties, getCustomActions);

            },

            drawPath: function(path, wrapper, emptyBinsMode, redraw, chartBase, xCoord, yCoord, chartData, chartDef, xDimension, xLabels, xAxis, leftYAxis, rightYAxis) {

                const lineData = wrapper.data()[0];

                // Data points that are not part of a line segment and need to be drawn explicitly
                let lonelyPoints = [];
                if (lineData.filteredPointsData.length === 1) {
                    lonelyPoints = [lineData.filteredPointsData[0]];
                }

                if (emptyBinsMode === 'DASHED' && !redraw) {
                    const emptySegments = svc.getEmptySegments(lineData.pointsData);

                    if (lineData.filteredPointsData.length > 1) {
                        emptySegments.forEach(function(seg, i) {
                            if (i === 0) {
                                if (seg[0].$idx === 0) {
                                    lonelyPoints.push(seg[0]);
                                }
                            } else if (i === emptySegments.length - 1 && seg[1].$idx === lineData.filteredPointsData[lineData.filteredPointsData.length - 1].$idx) {
                                lonelyPoints.push(seg[1]);
                            }
                            if (emptySegments[i + 1] && emptySegments[i][1] === emptySegments[i + 1][0]) {
                                lonelyPoints.push(emptySegments[i][1]);
                            }
                        });
                    }
                }

                const lonelyCircles = wrapper.selectAll('circle.lonely')
                    .data(lonelyPoints, Fn.prop('x'));

                lonelyCircles.remove();

                lonelyCircles.enter().append('circle')
                    .attr('opacity', chartDef.colorOptions.transparency)
                    .attr('class', 'lonely')
                    .attr('fill', function(d) {
                        return chartBase.colorScale(d.colorScaleIndex);
                    })
                    .style('pointer-events', 'none');

                if (emptyBinsMode === 'DASHED') {
                    lonelyCircles.attr('r', 2.5)
                        .attr('cy', chartBase.yAxes[0].scale()(0));
                } else {
                    // If not in dashed mode, lonely circles are lonely normal points
                    lonelyCircles
                        .attr('r', 4)
                        .attr('opacity', chartDef.colorOptions.transparency);
                }

                lonelyCircles.exit().remove();
                lonelyCircles
                    .attr('cx', d => xCoord(xDimension, xLabels, xAxis)(d))
                    .attr('cy', d => yCoord(d, chartDef, chartData, leftYAxis, rightYAxis));
            },

            // Prevent chart to overlap axes
            clipPaths: function(chartBase, chartDef, g, wrappers) {
                const displayedValuesFontSizes = chartDef.genericMeasures.filter(m => m.valuesInChartDisplayOptions && m.valuesInChartDisplayOptions.textFormatting).map(m => m.valuesInChartDisplayOptions.textFormatting.fontSize);
                const extraPadding = displayedValuesFontSizes.length ? Math.max(...displayedValuesFontSizes) + 20 : 10;
                const defs = g.append('defs');
                const clipPathUniqueId = CLIP_PATH_ID + generateUniqueId();

                // Add a bit of margin to handle smoothing mode.
                defs.append('clipPath')
                    .attr('id', clipPathUniqueId)
                    .append('rect')
                    .attr('width', chartBase.vizWidth)
                    .attr('y', -extraPadding)
                    .attr('height', chartBase.vizHeight + extraPadding);

                wrappers.attr('clip-path', 'url(#' + clipPathUniqueId + ')');
                wrappers.style('-webkit-clip-path', 'url(#' + clipPathUniqueId + ')');
            },

            nonZeroCountFilter: function(d, facetIndex, chartData) {
                d.$filtered = chartData.getNonNullCount({ x: d.xCoord, color: d.color, facet: facetIndex }, d.measure) === 0;
                return !d.$filtered;
            },

            xCoord: function(xDimension, xLabels, xAxis) {
                return svc.getXCoord(xDimension, xAxis, xAxis.ordinalScale, xLabels);
            },

            yCoord: function(d, chartDef, chartData, leftYAxis, rightYAxis) {
                let val = chartData.aggr(d.measure).get({ x: d.xCoord, color: d.color });
                const displayAxis = chartDef.genericMeasures[d.measure].displayAxis === 'axis1' ? leftYAxis : rightYAxis;

                if (ChartAxesUtils.isYAxisLogScale(chartDef.yAxesFormatting, displayAxis.id) && val == 0) {
                    val = 1;
                }
                return displayAxis.scale()(val);
            },

            cleanChart: function(g, chartBase) {
                const wrappers = $('g.wrapper');
                const bgAnchor = g.select('.chart-contextual-menu-anchor');
                chartBase.tooltips.removeChartTooltipHandlers(g.node());
                chartBase.contextualMenu.removeChartContextualMenuHandler(g.node());
                !bgAnchor.empty() && chartBase.contextualMenu.removeChartContextualMenuHandler(bgAnchor);
                wrappers.off();
            },

            prepareData: function(chartDef, chartData, measureFilter, ignoreLabels = new Set(), useColorDimension = true) {
                const xLabels = chartData.getAxisLabels('x');
                const axisColorLabels = chartData.getAxisLabels('color');
                const colorLabelsLength = (axisColorLabels || [null]).length;
                const colorLabels = useColorDimension && axisColorLabels || [axisColorLabels && axisColorLabels.length ? chartData.getSubtotalLabelIndex('color') : 0];
                const linesData = [];

                colorLabels.forEach(function(colorLabel, colorIndex) {
                    let mIndex = 0;
                    chartDef.genericMeasures.forEach(function(measure, measureIndex) {
                        if (measureFilter && !measureFilter(measure)) {
                            return;
                        }
                        const newColorIndex = useColorDimension ? colorIndex : colorLabel;
                        const colorScaleIndex = (useColorDimension ? newColorIndex : (colorLabelsLength - 1)) + (axisColorLabels ? mIndex : measureIndex);
                        linesData.push({
                            id: _.uniqueId('line_'),
                            color: newColorIndex,
                            measure: measureIndex,
                            colorScaleIndex,
                            valuesInChartDisplayOptions: measure.valuesInChartDisplayOptions,
                            pointsData: xLabels.reduce(function(result, xLabel, xIndex) {
                                if (ignoreLabels.has(xLabel.label)) {
                                    return result;
                                }

                                const measureData = {
                                    x: result.length,
                                    color: newColorIndex,
                                    colorScaleIndex,
                                    measure: measureIndex,
                                    filtered: true,
                                    xCoord: xIndex
                                };

                                const subTextElementsData = SVGUtils.getLabelSubTexts(chartDef, measureIndex, measure.valuesInChartDisplayOptions, false).map(function(subTextElementData) {
                                    return {
                                        ...subTextElementData,
                                        // add measure data at sub text level also, to retrieve it easily
                                        ...measureData
                                    };
                                });

                                // xCoord is used to get the data in the tensor and x is used for colors
                                result.push({
                                    ...measureData,
                                    textsElements: subTextElementsData,
                                    // add the labels display options to the data point to retrieve it easily
                                    valuesInChartDisplayOptions: measure.valuesInChartDisplayOptions
                                });
                                return result;
                            }, [])
                        });
                        mIndex++;
                    });
                });

                return linesData;
            },

            // Returns the right accessor for the x-coordinate of a label
            getXCoord: function(dimension, xAxis, ordinalXScale, labels) {
                if (ChartDimension.isTimeline(dimension)) {
                    return function(d) {
                        return xAxis.scale()(labels[d.x].tsValue);
                    };
                } else if ((ChartDimension.isGroupedNumerical(dimension) && !ChartDimension.hasOneTickPerBin(dimension)) || ChartDimension.isUngroupedNumerical(dimension)) {
                    return function(d) {
                        return xAxis.scale()(labels[d.x].sortValue);
                    };
                } else {
                    return function(d) {
                        return ordinalXScale(d.x) + (ordinalXScale.rangeBand() / 2);
                    };
                }
            },

            getEmptySegments: function(labels) {
                const emptySegments = [];
                let segment = [];
                let inSegment = false;
                let inLine = false;
                labels.forEach(function(label, i) {
                    label.$idx = i;
                    if (inLine && label.$filtered) {
                        inSegment = true;
                    } else {
                        inLine = true;
                        if (inSegment) {
                            segment[1] = label;
                            emptySegments.push(segment);
                            segment = [label];
                        } else {
                            segment = [label];
                        }
                        inSegment = false;
                    }
                });
                return emptySegments;
            },

            /**
             * Returns true when we must include empty bins to compute measure extent
             * @param {DimensionDef} dimension
             * @return {boolean} true when we must include empty bins to compute measure extent
             */
            hasEmptyBinsToIncludeAsZero: function(dimension, chartData) {
                // In case of a line/mix chart we must look at the emptyBinsMode option
                return dimension.numParams
                && dimension.numParams.emptyBinsMode === 'ZEROS'
                && chartData.hasEmptyBins;
            }
        };
        return svc;
    }
})();

;
(function() {
    'use strict';

    angular.module('dataiku.charts')
        .factory('LinesZoomHelper', LinesZoomHelper);

    const CLIP_PATH_ID = 'lines-chart-clip-path';

    // (!) This service previously was in static/dataiku/js/simple_report/curves/lines.js
    function LinesZoomHelper(ChartDataUtils, D3ChartAxes, ChartAxesUtils, LinesUtils, ReferenceLines, ChartYAxisPosition, ChartUsableColumns, ChartCustomMeasures, ChartZoomLoader,
        ChartSetErrorInScope, Logger, ChartDimension, ChartActivityIndicator, ChartsStaticData) {

        const svc = {

            /*
             * The greyed out areas that represent missing data in the current aggregation level.
             * They appear when zooming out and panning.
             */
            createMissingDataArea(g) {
                const clipPathUniqueId = CLIP_PATH_ID + generateUniqueId();
                const area = g.append('rect')
                    .attr('opacity', '0.6')
                    .attr('class', 'missing-data-area')
                    .attr('x', '0')
                    .attr('width', '0')
                    .attr('y', '0')
                    .attr('height', '0')
                    .attr('clip-path', 'url(#' + clipPathUniqueId + ')')
                    .style('-webkit-clip-path', 'url(#' + clipPathUniqueId + ')')
                    .style('pointer-events', 'none');

                return area;
            },

            /*
             * Updates the greyed out areas upon zooming out and panning.
             */
            updateMissingDataAreas(chartBase, zoomUtils) {
                if (zoomUtils.displayIntervalBeforeInteraction && zoomUtils.displayIntervalBeforeInteraction !== zoomUtils.dataInterval) {
                    const dataIntervalMin = zoomUtils.dataInterval[0];
                    const dataIntervalMax = zoomUtils.dataInterval[1];
                    const displayIntervalBeforeInteractionMin = zoomUtils.displayIntervalBeforeInteraction[0];
                    const displayIntervalBeforeInteractionMax = zoomUtils.displayIntervalBeforeInteraction[1];

                    if (displayIntervalBeforeInteractionMin > dataIntervalMin) {
                        const scaledDataIntervalMin = chartBase.xAxis.scale()(dataIntervalMin);
                        const scaledIntervalBeforeInteractionMin = chartBase.xAxis.scale()(displayIntervalBeforeInteractionMin);

                        zoomUtils.leftMissingDataAreas.forEach(area => {
                            area.attr('x', scaledDataIntervalMin)
                                .attr('width', scaledIntervalBeforeInteractionMin - scaledDataIntervalMin)
                                .attr('height', chartBase.vizHeight);
                        });
                    }

                    if (displayIntervalBeforeInteractionMax < dataIntervalMax) {
                        const scaledDataIntervalMax = chartBase.xAxis.scale()(dataIntervalMax);
                        const scaledIntervalBeforeInteractionMax = chartBase.xAxis.scale()(displayIntervalBeforeInteractionMax);

                        zoomUtils.rightMissingDataAreas.forEach(area => {
                            area.attr('x', scaledIntervalBeforeInteractionMax)
                                .attr('width', scaledDataIntervalMax - scaledIntervalBeforeInteractionMax)
                                .attr('height', chartBase.vizHeight);
                        });
                    }
                }
            },

            /*
             * Resets the greyed out areas upon new pivot request.
             */
            cleanMissingDataAreas(zoomUtils) {
                zoomUtils.leftMissingDataAreas.forEach(area => {
                    area.attr('x', '0')
                        .attr('y', '0')
                        .attr('width', '0')
                        .attr('height', '0');
                });

                zoomUtils.rightMissingDataAreas.forEach(area => {
                    area.attr('x', '0')
                        .attr('y', '0')
                        .attr('width', '0')
                        .attr('height', '0');
                });
            },

            cleanZoomListeners(zoom) {
                zoom && zoom.on('zoomstart', null);
                zoom && zoom.on('zoom', null);
                zoom && zoom.on('zoomend', null);
                zoom = null;
            },

            updateTicksFormat(chartDef, container, xAxis) {
                const xExtent = D3ChartAxes.getCurrentAxisExtent(xAxis);
                const xMin = xExtent[0];
                const xMax = xExtent[1];

                if (!isFinite(xMin) || !isFinite(xMax)) {
                    return;
                }

                const computedDateDisplayUnit = ChartDataUtils.computeDateDisplayUnit(xMin, xMax);

                xAxis.tickFormat(date => {
                    return computedDateDisplayUnit.formatDateFn(date, computedDateDisplayUnit.dateFormat);
                });
            },

            /**
             * Wrapper for ChartDataUtils.getMeasureExtents().
             *
             * @param {ChartDef.java}   chartDef    - The chart definition.
             * @param {Object}          chartBase   - Everything that the chart might need.
             * @param {Array}           interval    - The x min and max values to use as filter when computing the extents.
             */
            getYExtentsForInterval(chartBase, chartDef, chartHandler, interval) {
                const xMin = interval[0];
                const xMax = interval[1];
                const chartData = chartBase.chartData;
                const results = {};
                const includeEmptyBins = LinesUtils.hasEmptyBinsToIncludeAsZero(ChartDimension.getGenericDimension(chartDef), chartData);
                const yExtents = ChartDataUtils.getMeasureExtents(chartDef, chartData, 'x', [xMin, xMax], includeEmptyBins);
                results.recordsCount = yExtents.recordsCount;
                results.pointsCount = yExtents.pointsCount;
                const leftAxis = chartBase.yAxes.filter(axis => axis.position === ChartYAxisPosition.LEFT),
                    rightAxis = chartBase.yAxes.filter(axis => axis.position === ChartYAxisPosition.RIGHT),
                    leftYAxisID = ChartAxesUtils.computeYAxisID(ChartYAxisPosition.LEFT),
                    rightYAxisID = ChartAxesUtils.computeYAxisID(ChartYAxisPosition.RIGHT);

                const dataSpec = chartHandler.getDataSpec();
                const customMeasures = ChartCustomMeasures.getMeasuresLikeCustomMeasures(dataSpec.datasetProjectKey, dataSpec.datasetName, chartHandler.getCurrentChartsContext());
                const allMeasures = ChartUsableColumns.getUsableColumns(dataSpec.datasetProjectKey, dataSpec.datasetName, dataSpec.context).filter(m => ['NUMERICAL', 'ALPHANUM', 'DATE'].includes(m.type));

                const displayedReferenceLines = ReferenceLines.getDisplayedReferenceLines(chartDef.referenceLines, chartBase.xAxis, undefined),
                    referenceLinesValues = ReferenceLines.getReferenceLinesValues(displayedReferenceLines, chartData, allMeasures, chartDef.genericMeasures, customMeasures),
                    referenceLinesExtents = ReferenceLines.getReferenceLinesExtents(displayedReferenceLines, referenceLinesValues, { [leftYAxisID]: { isPercentScale: yExtents[leftYAxisID].onlyPercent }, [rightYAxisID]: { isPercentScale: yExtents[rightYAxisID].onlyPercent } });

                if (leftAxis && leftAxis.length) {
                    results[leftYAxisID] = ReferenceLines.getExtentWithReferenceLines(yExtents[leftYAxisID].extent, referenceLinesExtents[leftYAxisID]);
                }

                if (rightAxis && rightAxis.length) {
                    results[rightYAxisID] = ReferenceLines.getExtentWithReferenceLines(yExtents[rightYAxisID].extent, referenceLinesExtents[rightYAxisID]);
                }

                return results;
            },

            /**
             * Check if extents of left and right axis are valid ie have:
             *  * Finite numbers
             *  * Different mix and max values
             *
             * @param {Array} leftYExtent        - Min and max interval for left Y axis.
             * @param {Array} rightYExtent       - Min and max interval for right Y axis.
             *
             * @returns {Boolean} True if both y extents are valid.
             */
            hasValidYExtents(leftYExtent, rightYExtent) {
                const isLeftYExtentFinite = leftYExtent && isFinite(leftYExtent[0]) && isFinite(leftYExtent[1]);
                const isRightYExtentFinite = rightYExtent && isFinite(rightYExtent[0]) && isFinite(rightYExtent[1]);
                const isLeftYExtentValid = !leftYExtent || (isLeftYExtentFinite && leftYExtent[0] !== leftYExtent[1]);
                const isRightY2ExtentValid = !rightYExtent || (isRightYExtentFinite && rightYExtent[0] !== rightYExtent[1]);

                return isLeftYExtentValid && isRightY2ExtentValid;
            },

            // Simply updates any useful zoom info for incoming zoom actions.
            updateZoomUtils(zoomUtils, inspectedZoom) {

                zoomUtils.displayInterval = inspectedZoom.displayInterval;

                const isDisplayIntervalValid = inspectedZoom.yExtents.pointsCount > 1
                    && svc.hasValidYExtents(inspectedZoom.yExtents[ChartAxesUtils.computeYAxisID(ChartYAxisPosition.LEFT)], inspectedZoom.yExtents[ChartAxesUtils.computeYAxisID(ChartYAxisPosition.RIGHT)])
                    && inspectedZoom.displayInterval[0] !== inspectedZoom.displayInterval[1];

                if (isDisplayIntervalValid) {
                    zoomUtils.lastValidDisplayInterval = inspectedZoom.displayInterval;
                }

                zoomUtils = { ...zoomUtils, ...inspectedZoom.yExtents };
                zoomUtils.disableZoomFiltering = inspectedZoom.disableZoomFiltering;
                zoomUtils.previousZoomEvent = d3.event;

                return zoomUtils;
            },

            redrawChart(chartBase, chartDef, zoomUtils, containerSelection, xAxis, drawFrame) {
                const leftYAxis = ChartAxesUtils.getLeftYAxis(chartBase.yAxes);
                const rightYAxis = ChartAxesUtils.getRightYAxis(chartBase.yAxes);
                svc.updateAxes(containerSelection, chartDef, xAxis, leftYAxis, rightYAxis, zoomUtils, chartBase.vizHeight, chartBase.vizWidth);

                drawFrame(zoomUtils.frameIndex, chartBase, true);
                svc.updateMissingDataAreas(chartBase, zoomUtils);
            },

            buildChartInteractionErrorMessage(data, status, headers) {
                const knownError = ChartSetErrorInScope.buildValidityForKnownError(data, status, headers);
                if (knownError !== undefined) {
                    return knownError.message;
                } else if (data.message) {
                    return data.message;
                }
                return 'An unknown error occurred while interacting with the chart.';
            },

            /**
             * Asks for new data inside the given display interval and create a new one accordingly.
             *
             * @param {Function}                pivotRequestCallback    - The function that executes the pivot request
             * @param {Function}                linesChartCallback      - The function that recomputes and redraws the line chart
             * @param {ChartDef.java}           chartDef                - The chart definition.
             * @param {Object}                  chartBase               - Everything that the chart might need.
             * @param {LinesZoomUtils}          zoomUtils               - All the data related to zoom
             * @param {Function}                cleanFrame              - The callback that will remove the chart from DOM.
             * @param {Object}                  uiDisplayState          - Everything the UI might need.
             * @param {Object}                  chartActivityIndicator  - Activity indicator displayed in chart
             * @param {Object}                  request                 - Request to be executed
             * @param {Object}                  chartContainer
             */
            computePivotRequest(pivotRequestCallback, linesChartCallback, chartDef, chartBase, zoomUtils, chartHandler, cleanFrame, uiDisplayState, chartActivityIndicator, request, chartContainer) {

                pivotRequestCallback(request, false, true).success(function(data) {
                    if (data.result.pivotResponse.axisLabels[0] && data.result.pivotResponse.axisLabels[0].length === 1) {
                        Logger.info('Not enough data in the result: chart won\'t be refreshed.');
                        svc.cleanOfflineFeedbacks(zoomUtils);
                        return;
                    }

                    const responseSequenceId = data.result.pivotResponse.sequenceId;

                    if (responseSequenceId === zoomUtils.sequenceId) {
                        Logger.info('Sequence ids match (' + responseSequenceId + '). Deactivate offline zoom and refresh the chart.');
                        svc.cleanAll(chartBase, cleanFrame, zoomUtils);
                        zoomUtils.preventThumbnailUpdate = true;
                        zoomUtils.offlineZoomDisabled = true;
                        linesChartCallback(chartContainer, chartDef, chartHandler, zoomUtils.axesDef, data.result.pivotResponse, pivotRequestCallback, zoomUtils.displayInterval, uiDisplayState, chartActivityIndicator);
                        uiDisplayState.chartTopRightLabel = ChartDataUtils.computeRecordsStatusLabel(
                            data.result.pivotResponse.beforeFilterRecords,
                            data.result.pivotResponse.afterFilterRecords,
                            ChartDimension.getComputedMainAutomaticBinningModeLabel(data.result.pivotResponse, chartDef),
                            chartDef.type
                        );
                        uiDisplayState.samplingSummaryMessage = ChartDataUtils.getSamplingSummaryMessage(data.result.pivotResponse, chartDef.type);
                        uiDisplayState.chartRecordsFinalCountTooltip = ChartDataUtils.getRecordsFinalCountTooltip(
                            chartDef.type,
                            data.result.pivotResponse.afterFilterRecords,
                            [],
                            undefined
                        );
                    } else {
                        Logger.info('Sequence ids do not match (' + responseSequenceId + ', ' + zoomUtils.sequenceId + '): chart won\'t be refreshed.');
                    }
                }).error(function(data, status, headers) {
                    Logger.info('An error occurred during zoom pivot request');
                    ChartActivityIndicator.displayBackendError(
                        chartActivityIndicator,
                        svc.buildChartInteractionErrorMessage(data, status, headers)
                    );
                    uiDisplayState.chartTopRightLabel = ChartDataUtils.computeNoRecordsTopRightLabel();
                    uiDisplayState.chartRecordsFinalCountTooltip = ChartDataUtils.getRecordsFinalCountTooltip(chartDef.type, 0);
                    uiDisplayState.samplingSummaryMessage = data.result && ChartDataUtils.getSamplingSummaryMessage(data.result.pivotResponse, chartDef.type);
                    svc.cleanOfflineFeedbacks(zoomUtils);
                });
            },

            // Get an updated y axis domain upon zooming
            getUpdatedYDomain(chartDef, yAxis, yExtent) {
                if (!yExtent) {
                    return;
                }
                let yMin = yExtent[0];
                let yMax = yExtent[1];

                const { customExtent, includeZero } = ChartAxesUtils.getFormattingForYAxis(chartDef.yAxesFormatting, yAxis.id);
                const shouldIncludeZero = customExtent.editMode === ChartsStaticData.AUTO_EXTENT_MODE && includeZero;
                if (shouldIncludeZero) {
                    if (yMin > 0) {
                        yMin = 0;
                    } else if (yMax < 0) {
                        yMax = 0;
                    }
                } else if (customExtent.editMode === ChartsStaticData.MANUAL_EXTENT_MODE) {
                // Temp fix to enable custom extent => then custom extent should only be applied when no zoom is on (only at initial state)
                    const customAxisExtent = ChartAxesUtils.getManualExtent(customExtent);
                    yMin = customAxisExtent[0];
                    yMax = customAxisExtent[1];
                }

                return [yMin, yMax];
            },

            /**
             * Clean offline feedbacks, the chart, and every listeners previously attached for interactivity.
             *
             * @param {Object}      chartBase       - Everything that the chart might need.
             * @param {Function}    cleanFrame      - Function that removes the chart.
             */
            cleanAll(chartBase, cleanFrame, zoomUtils) {
                svc.cleanOfflineFeedbacks(zoomUtils);
                svc.cleanZoomListeners(zoomUtils.zoom);
                cleanFrame(chartBase);
            },

            /**
             * Clean every visual feedbacks the user may have when interacting:
             *  - Grey areas for missing data
             *  - Grey text color for top-right aggregations info.
             *
             * @param {Object}  zoomUtils       - Data related to zoom
             */
            cleanOfflineFeedbacks(zoomUtils) {
                ChartZoomLoader.displayLoader(false, zoomUtils.instanceId);
                svc.cleanMissingDataAreas(zoomUtils);
            },

            computeBrushDimensions(chartBase) {
                return {
                    paddingLeft: chartBase.margins.left,
                    width: chartBase.vizWidth
                };
            },

            onBrushInit(chartBase, chartDef, drawBrush, zoomUtils) {
                return function(brushContentG, brushContentHeight, brushContentWidth) {

                    const xAxisLogScale = (chartBase.xSpec && chartBase.xSpec.type == 'MEASURE' && chartDef.xAxisFormatting.isLogScale);

                    const xAxis = D3ChartAxes.createAxis(chartBase.chartData, chartBase.xSpec, chartBase.isPercentChart, xAxisLogScale, undefined, chartDef.xAxisFormatting.axisValuesFormatting.numberFormatting, chartDef);
                    const yAxes = [];
                    for (const key in chartBase.ySpecs) {
                        const yAxisLogScale = ChartAxesUtils.isYAxisLogScale(chartDef.yAxesFormatting, key);
                        const yAxisFormatting = ChartAxesUtils.getFormattingForYAxis(chartDef.yAxesFormatting, key);
                        const axis = D3ChartAxes.createAxis(chartBase.chartData, chartBase.ySpecs[key], chartBase.isPercentChart, yAxisLogScale, yAxisFormatting.includeZero, yAxisFormatting.axisValuesFormatting.numberFormatting, chartDef);
                        axis && yAxes.push(axis);
                    };

                    xAxis.setScaleRange([0, brushContentWidth]);

                    const leftAxis = ChartAxesUtils.getLeftYAxis(yAxes);
                    const rightAxis = ChartAxesUtils.getRightYAxis(yAxes);

                    if (leftAxis) {
                        if (chartBase.ySpecs[ChartsStaticData.LEFT_AXIS_ID].ascendingDown) {
                            leftAxis.setScaleRange([0, brushContentHeight]);
                        } else {
                            leftAxis.setScaleRange([brushContentHeight, 0]);
                        }
                    }

                    if (rightAxis) {
                        rightAxis.setScaleRange([brushContentHeight, 0]);
                    }

                    const brushAxes = {
                        x: xAxis,
                        [ChartAxesUtils.computeYAxisID(ChartYAxisPosition.LEFT)]: leftAxis,
                        [ChartAxesUtils.computeYAxisID(ChartYAxisPosition.RIGHT)]: rightAxis
                    };

                    drawBrush(chartBase, brushContentG, brushAxes);
                    zoomUtils.hasBrushBeenDrawn = true;
                };
            },

            updateAxes(containerSelection, chartDef, xAxis, leftYAxis, rightYAxis, zoomUtils, vizHeight, vizWidth) {
                svc.updateXAxis(containerSelection, chartDef, xAxis, vizHeight, [leftYAxis, rightYAxis]);
                svc.updateYAxes(containerSelection, chartDef, leftYAxis, rightYAxis, zoomUtils, vizWidth);
            },

            updateXAxis(containerSelection, chartDef, xAxis, vizHeight, yAxes) {
                if (!chartDef.xAxisFormatting.displayAxis) {
                    return;
                }

                const allNegative = !yAxes.filter(yAxis => yAxis).some(yAxis => {
                    const currentYExtent = D3ChartAxes.getCurrentAxisExtent(yAxis);
                    return currentYExtent && currentYExtent[1] >= 0;
                });
                svc.updateTicksFormat(chartDef, containerSelection.node(), xAxis);
                const xAxisSelection = [ ...containerSelection.selectAll('.chart-svg .x.axis')][0];
                xAxisSelection.forEach(xG => {
                    xG = d3.select(xG);
                    xG.call(xAxis);
                    xG.selectAll('.tick text')
                        .attr('transform', function() {
                            const labelAngle = 0.5;
                            const translateValue = '-33, 15';
                            const rotateValue = labelAngle * -180 / Math.PI;

                            return `translate(${translateValue}), rotate(${rotateValue}, 0, 0)`;
                        });
                    const gridlinesPosition = { x1: 0, x2: 0, y1: 0, y2: allNegative ? vizHeight : -vizHeight };
                    if (chartDef.gridlinesOptions.vertical.show) {
                        D3ChartAxes.drawGridlines(xG, 'x', chartDef.gridlinesOptions.vertical.lineFormatting, gridlinesPosition, containerSelection);
                    }
                });
            },

            updateYAxes(containerSelection, chartDef, leftAxis, rightAxis, zoomUtils, vizWidth) {
                if (leftAxis) {
                    svc.updateYAxis(containerSelection, chartDef, leftAxis, zoomUtils, '.y1.axis', vizWidth);
                }
                if (rightAxis) {
                    svc.updateYAxis(containerSelection, chartDef, rightAxis, zoomUtils, '.y2.axis', vizWidth);
                }
            },

            updateYAxis(containerSelection, chartDef, yAxis, zoomUtils, selector, vizWidth) {
                const axisFormatting = ChartAxesUtils.getFormattingForYAxis(chartDef.yAxesFormatting, yAxis);
                if (axisFormatting && !axisFormatting.displayAxis) {
                    return;
                }

                const [yMin, yMax] = svc.getUpdatedYDomain(chartDef, yAxis, zoomUtils[yAxis.id]);
                yAxis.scale().domain([yMin, yMax]);

                [...containerSelection.selectAll(selector)[0]].forEach(yG => {
                    d3.select(yG).call(yAxis);
                    if (chartDef.gridlinesOptions.horizontal.show && D3ChartAxes.shouldDrawGridlinesForAxis(chartDef.type, yAxis, chartDef.gridlinesOptions.horizontal.displayAxis)) {
                        const gridlinesPosition = { x1: 0, x2: ChartAxesUtils.isRightYAxis(yAxis) ? -vizWidth : vizWidth, y1: 0, y2: 0 };
                        D3ChartAxes.drawGridlines(d3.select(yG), 'y', chartDef.gridlinesOptions.horizontal.lineFormatting, gridlinesPosition, containerSelection);
                    }
                });
            }
        };
        return svc;
    }

})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.charts');

    /**
     * (!) This service previously was in static/dataiku/js/simple_report/column-bars/multiplot.js
     */
    app.factory('MultiplotChart', function(ChartManager, ChartDataWrapperFactory, GroupedColumnsDrawer, GroupedColumnsUtils, LinesDrawer, LinesUtils, ChartDataUtils, ReferenceLines, ChartUsableColumns, ChartYAxisPosition, ChartAxesUtils, CHART_LABELS, MultiplotUtils, ChartCustomMeasures, SVGUtils, ChartDimension) {
        return function($container, chartDef, chartHandler, axesDef, data) {

            const chartData = ChartDataWrapperFactory.chartTensorDataWrapper(data, axesDef);

            const ignoreLabels = new Set([CHART_LABELS.SUBTOTAL_BIN_LABEL]);
            const genericDimension = ChartDimension.getGenericDimension(chartDef);
            const includeEmptyBins = LinesUtils.hasEmptyBinsToIncludeAsZero(genericDimension, chartData);
            const subtotals = Array.from({ length: chartData.numAxes });
            const isDisplayedAsColumn = MultiplotUtils.isColumn;
            const hasSubtotals = MultiplotUtils.shouldSubtotalsBeCalculated(chartDef);
            let getRefLineValuesOverride;

            if (hasSubtotals) {
                subtotals[chartData.axesDef.color] = chartData.subtotalCoords[chartData.axesDef.color];
                getRefLineValuesOverride = (measure, measureIdx) => {
                    const dataIndices = MultiplotUtils.isColumn(measure) ?
                        ChartDataUtils.getValuesForColumns(subtotals, chartData, ignoreLabels) :
                        ChartDataUtils.getValuesForLines(subtotals, chartData, ignoreLabels);
                    return ChartDataUtils.retrieveAggrData(chartData, dataIndices, measureIdx);
                };
            }

            const leftYAxisID = ChartAxesUtils.computeYAxisID(ChartYAxisPosition.LEFT);
            const rightYAxisID = ChartAxesUtils.computeYAxisID(ChartYAxisPosition.RIGHT);
            const facetLabels = chartData.getAxisLabels('facet', ignoreLabels) || [{ $tensorIndex: 0 }]; // We'll through the next loop only once if the chart is not facetted


            const dataSpec = chartHandler.getDataSpec();
            const xSpec = { type: 'DIMENSION', mode: 'COLUMNS', dimension: genericDimension, name: 'x', customExtent: chartDef.xAxisFormatting.customExtent };
            const customMeasures = ChartCustomMeasures.getMeasuresLikeCustomMeasures(dataSpec.datasetProjectKey, dataSpec.datasetName, chartHandler.getCurrentChartsContext());
            const allMeasures = ChartUsableColumns.getUsableColumns(dataSpec.datasetProjectKey, dataSpec.datasetName, chartHandler.getCurrentChartsContext()).filter(m => ['NUMERICAL', 'ALPHANUM', 'DATE'].includes(m.type));

            /*
             *  If we are not with a color dimension, use the old system it should be faster (just a loop on the tensors)
             */
            const yExtents = hasSubtotals ? ChartDataUtils.getMeasureExtentsForMixColored(chartDef, chartData, includeEmptyBins, ignoreLabels, subtotals, isDisplayedAsColumn) : ChartDataUtils.getMeasureExtents(chartDef, chartData, 'x', undefined, includeEmptyBins);

            const displayedReferenceLines = ReferenceLines.getDisplayedReferenceLines(chartDef.referenceLines, xSpec, undefined),
                referenceLinesValues = ReferenceLines.getReferenceLinesValues(displayedReferenceLines, chartData, allMeasures, chartDef.genericMeasures, customMeasures, true, getRefLineValuesOverride),
                referenceLinesExtents = ReferenceLines.getReferenceLinesExtents(displayedReferenceLines, referenceLinesValues, { [leftYAxisID]: { isPercentScale: yExtents[leftYAxisID].onlyPercent }, [rightYAxisID]: { isPercentScale: yExtents[rightYAxisID].onlyPercent } });

            const leftYExtent = ReferenceLines.getExtentWithReferenceLines(yExtents[leftYAxisID].extent, referenceLinesExtents[leftYAxisID]),
                rightYExtent = ReferenceLines.getExtentWithReferenceLines(yExtents[rightYAxisID].extent, referenceLinesExtents[rightYAxisID]);

            ReferenceLines.mutateDimensionSpecForReferenceLine(ChartDataUtils.getAxisExtent(chartData, 'x', xSpec.dimension), referenceLinesExtents.x, xSpec);

            const animationData = GroupedColumnsUtils.prepareData(chartDef, chartData, isDisplayedAsColumn, ignoreLabels),
                linesData = LinesUtils.prepareData(chartDef, chartData, measure => !isDisplayedAsColumn(measure), ignoreLabels, false);
            const hasColumns = chartDef.genericMeasures.some(isDisplayedAsColumn);

            const drawFrame = (frameIdx, chartBase) => {
                chartData.fixAxis('animation', frameIdx);
                facetLabels.forEach((facetLabel, f) => {
                    const g = d3.select(chartBase.$svgs.eq(f).find('g.chart').get(0));
                    // initialize labels collision detection, common to columns and lines
                    const labelCollisionDetectionHandler = SVGUtils.initLabelCollisionDetection(chartBase);
                    //  Reference lines will be drawn with both drawers, deactivating it for columns, lines drawer will do the job.
                    const columnsData = animationData.frames[frameIdx].facets[facetLabel.$tensorIndex].groups;
                    GroupedColumnsDrawer(g, chartDef, chartHandler, chartData.fixAxis('facet', facetLabel.$tensorIndex), chartBase, columnsData, facetLabel.$tensorIndex, labelCollisionDetectionHandler, false);
                    // priority to the columns labels, which are diplayed first
                    LinesDrawer(g, chartDef, chartHandler, chartData.fixAxis('facet', facetLabel.$tensorIndex), chartBase, linesData, facetLabel.$tensorIndex, false, false, labelCollisionDetectionHandler, getRefLineValuesOverride, hasColumns ? g.select('.group-wrapper') : null);
                });
            };

            const leftYCustomExtent = ChartAxesUtils.getYAxisCustomExtent(chartDef.yAxesFormatting, leftYAxisID);
            const rightYCustomExtent = ChartAxesUtils.getYAxisCustomExtent(chartDef.yAxesFormatting, rightYAxisID);

            const ySpecs = {
                [leftYAxisID]: { id: leftYAxisID, type: 'MEASURE', extent: leftYExtent, customExtent: leftYCustomExtent, position: ChartYAxisPosition.LEFT },
                [rightYAxisID]: { id: rightYAxisID, type: 'MEASURE', extent: rightYExtent, customExtent: rightYCustomExtent, position: ChartYAxisPosition.RIGHT }
            };
            const availableAxes = ReferenceLines.getAvailableAxesForReferenceLines(chartDef);
            const colorSpec = hasSubtotals ? {
                type: 'CUSTOM',
                name: 'color',
                handle: (measure) => !isDisplayedAsColumn(measure),
                buildGenericMeasures: (measures, colorLabels) => {
                    const trimmeredColorlabels = colorLabels.filter(label => !ignoreLabels.has(label.label));
                    const linesMeasures = measures.filter(measure => !isDisplayedAsColumn(measure));
                    return [...trimmeredColorlabels, ...linesMeasures];
                },
                ignoreColorDim: (index, measuresOrColors) => measuresOrColors[index].isA === 'measure'
            } : {
                type: 'DIMENSION',
                name: 'color'
            };

            const axisSpecs = { x: xSpec, ...ySpecs };
            ChartAxesUtils.setNumberOfBinsToDimensions(chartData, chartDef, axisSpecs);
            ReferenceLines.updateAvailableAxisOptions([
                { axis: 'LEFT_Y_AXIS', isDisplayed: availableAxes['LEFT_Y_AXIS'], isNumerical: true, isPercentScale: ySpecs[leftYAxisID].isPercentScale },
                { axis: 'RIGHT_Y_AXIS', isDisplayed: availableAxes['RIGHT_Y_AXIS'], isNumerical: true, isPercentScale: ySpecs[rightYAxisID].isPercentScale },
                { axis: 'X_AXIS', isDisplayed: true, isNumerical: ChartAxesUtils.isNumerical(xSpec) && !ChartDimension.hasOneTickPerBin(xSpec.dimension), isContinuousDate: ChartAxesUtils.isContinuousDate(xSpec) }
            ]);
            ChartManager.initChart(chartDef, chartHandler, chartData, $container, drawFrame,
                axisSpecs,
                colorSpec, undefined, undefined, ignoreLabels);
        };
    });
})();

;
(function() {
    'use strict';

    // (!) This service previously was in static/dataiku/js/simple_report/curves/stacked_area.js
    angular.module('dataiku.charts')
        .factory('StackedAreaChart', function(ChartManager, ChartDimension, ChartDataWrapperFactory, SVGUtils, StackedChartUtils, LinesUtils, ColorUtils, ChartAxesUtils, ChartYAxisPosition, ChartHierarchyDimension, ChartDrilldown) {
            return function($container, chartDef, chartHandler, axesDef, data) {
                let currentAnimationFrame = 0;

                const chartData = ChartDataWrapperFactory.chartTensorDataWrapper(data, axesDef),
                    xDimension = ChartDimension.getGenericDimension(chartDef),
                    xLabels = chartData.getAxisLabels('x'),
                    animationData = StackedChartUtils.prepareData(chartDef, chartData),
                    yExtent = [0, animationData.maxTotal];

                const drawFrame = function(frameIdx, chartBase) {
                    animationData.frames[frameIdx].facets.forEach(function(facetData, f) {
                        const g = d3.select(chartBase.$svgs.eq(f).find('g.chart').get(0));
                        StackedAreaChartDrawer(g, facetData, chartBase, f);
                    });

                    currentAnimationFrame = frameIdx;
                };

                // Hack: by design stacked area start at 0, but if log scale is checked start at 1 (to make it possible)
                if (ChartAxesUtils.isYAxisLogScale(chartDef.yAxesFormatting)) {
                    yExtent[0] = 1;
                }

                const isPercentScale = ChartDimension.isPercentScale(chartDef.genericMeasures) || chartDef.variant == 'stacked_100';
                const yAxisID = ChartAxesUtils.computeYAxisID(ChartYAxisPosition.LEFT);

                ChartManager.initChart(chartDef, chartHandler, chartData, $container, drawFrame,
                    {
                        x: { type: 'DIMENSION', mode: 'POINTS', dimension: xDimension, name: 'x', customExtent: chartDef.xAxisFormatting.customExtent },
                        [yAxisID]: { id: yAxisID, type: 'MEASURE', extent: yExtent, isPercentScale: isPercentScale, customExtent: ChartAxesUtils.getYAxisCustomExtent(chartDef.yAxesFormatting, yAxisID), position: ChartYAxisPosition.LEFT }
                    },
                    { type: 'DIMENSION', name: 'color', dimension: chartDef.genericDimension1[0] });

                function StackedAreaChartDrawer(g, stacksData, chartBase, f) {

                    const yAxis = chartBase.yAxes[0],
                        yScale = yAxis.scale(),
                        xCoord = LinesUtils.getXCoord(xDimension, chartBase.xAxis, chartBase.xAxis.ordinalScale, xLabels, 'x'),
                        yCoord = function(d) {
                            return yScale(d.top);
                        };


                    const wrappers = g.selectAll('g.wrapper').data(stacksData.stacks[0].data, function(d, i) {
                        return d.color + '-' + d.measure;
                    });
                    wrappers.enter().append('g').attr('class', 'wrapper')
                        .attr('fill', function(d, i) {
                            return chartBase.colorScale(i);
                        })
                        .attr('opacity', chartDef.colorOptions.transparency)
                        .each(function(d) {
                            chartBase.tooltips.registerEl(this, { measure: d.measure, x: d.x, color: d.color, animation: currentAnimationFrame, facet: f }, 'fill', true);
                        });
                    wrappers.exit().remove();


                    const points = wrappers.selectAll('circle.point').data(function(d, i) {
                        return stacksData.stacks.map(function(stack) {
                            return stack.data[i];
                        });
                    }, function(d) {
                        return d.x + '-' + d.measure + '-' + d.color;
                    });
                    points.enter().append('circle')
                        .attr('class', 'point')
                        .attr('r', 5)
                        .attr('fill', function(d) {
                            return ColorUtils.darken(chartBase.colorScale(d.color + d.measure));
                        })
                        .attr('opacity', 0)
                        .each(function(d) {
                            chartBase.tooltips.registerEl(this, { measure: d.measure, x: d.x, color: d.color, animation: currentAnimationFrame, facet: f }, 'fill');
                            chartBase.contextualMenu.addContextualMenuHandler(this, { x: d.x, color: d.color, animation: currentAnimationFrame, facet: f });
                        });

                    points.exit().remove();
                    points.interrupt('movePoints').transition('movePoints')
                        .attr('cx', xCoord)
                        .attr('cy', yCoord);

                    const area = d3.svg.area()
                        .x(xCoord)
                        .y0(function(d) {
                            return yScale(d.base);
                        })
                        .y1(function(d) {
                            return yScale(d.top);
                        });

                    if (chartDef.smoothing) {
                        area.interpolate('monotone');
                    }

                    const path = wrappers.selectAll('path.area').data(function(d, i) {
                        return [stacksData.stacks.map(function(stack) {
                            return stack.data[i];
                        })];
                    });
                    path.enter().insert('path', ':first-child').attr('class', 'area');
                    path.exit().remove();
                    path.interrupt('updateArea').transition('updateArea').attr('d', area);


                    wrappers.on('mouseover.area', function(d) {
                        this.parentNode.insertBefore(this, $(this.parentNode).find('g.legend')[0]);
                        d3.select(this).selectAll('.wrapper circle').interrupt('fade').transition('fade').attr('opacity', 1);
                    }).on('mouseout.area', function(d) {
                        d3.select(this).selectAll('.wrapper circle').interrupt('fade').transition('fade').attr('opacity', 0);
                    });


                    if (ChartHierarchyDimension.getCurrentHierarchyLevel(chartDef) > 0) {
                        const anchor = SVGUtils.addPlotAreaContextMenuAnchor(g, chartBase.vizWidth, chartBase.vizHeight);
                        chartBase.contextualMenu.addChartContextualMenuHandler(anchor.node(), undefined, () => ChartDrilldown.getDrillupActions(chartDef));
                    }

                    // Clip paths to prevent area from overlapping axis when user chose a custom range which results in the bar being cropped (out of visible range)
                    ChartAxesUtils.isCroppedChart(chartDef) && SVGUtils.clipPaths(chartBase, g, wrappers);
                }

            };
        });
})();

;
(function() {
    'use strict';

    /**
     * Service responsible for listing existing charts based on a pivot response.
     * (!) This service previously was in static/dataiku/js/simple_report/chart_views.js
     */
    angular.module('dataiku.charts')
        .factory('DKUPivotCharts', function(
            GroupedColumnsChart,
            StackedColumnsChart,
            StackedAreaChart,
            LinesChart,
            MultiplotChart,
            StackedBarsChart,
            ScatterPlotChart,
            ScatterPlotMultiplePairsChart,
            BinnedXYChart,
            GroupedXYChart,
            LiftChart,
            AdministrativeMapChart,
            ScatterMapChart,
            DensityHeatMapChart,
            GridMapChart,
            BoxplotsChart,
            KpiChart,
            PivotTableChart,
            Density2DChart,
            PieChart,
            GeometryMapChart,
            WebappChart,
            //  ECharts
            GaugeEChartDef,
            PieEChartDef,
            RadarEChartDef,
            SankeyEChartDef,
            TreemapEChartDef
        ) {
            return {
                GroupedColumnsChart: GroupedColumnsChart,
                StackedColumnsChart: StackedColumnsChart,
                StackedAreaChart: StackedAreaChart,
                LinesChart: LinesChart,
                MultiplotChart: MultiplotChart,
                StackedBarsChart: StackedBarsChart,
                PivotTableChart: PivotTableChart,
                ScatterPlotChart: ScatterPlotChart,
                ScatterPlotMultiplePairsChart: ScatterPlotMultiplePairsChart,
                BinnedXYChart: BinnedXYChart,
                GroupedXYChart: GroupedXYChart,
                LiftChart: LiftChart,
                AdministrativeMapChart: AdministrativeMapChart,
                ScatterMapChart: ScatterMapChart,
                DensityHeatMapChart: DensityHeatMapChart,
                GridMapChart: GridMapChart,
                BoxplotsChart: BoxplotsChart,
                Density2DChart: Density2DChart,
                KpiChart: KpiChart,
                PieEChartDef: PieEChartDef,
                GeometryMapChart: GeometryMapChart,
                WebappChart: WebappChart,
                //  ECharts
                GaugeEChartDef: GaugeEChartDef,
                PieChart: PieChart,
                RadarEChartDef: RadarEChartDef,
                SankeyEChartDef: SankeyEChartDef,
                TreemapEChartDef: TreemapEChartDef
            };
        });
})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.charts');

    // (!) This service previously was in static/dataiku/js/simple_report/maps/administrative.js

    app.factory('AdministrativeMapChart', function($timeout, CHART_VARIANTS, ChartFormatting, ChartFeatures, _MapCharts, ChartLegendUtils, ColorUtils, ChartColorScales, ChartLabels) {
        return function($container, chartDef, data, chartHandler) {
            const geo = JSON.parse(data.geoJson);

            const aggrVals = function(aggregId) {
                let feat, featIdx;
                const arr = [];
                for (featIdx in geo.features) {
                    feat = geo.features[featIdx];
                    if (!_.isNil(feat.properties[aggregId])) {
                        arr.push(feat.properties[aggregId]);
                    }
                }
                return arr;
            };

            const aggrBounds = function(aggregId) {
                const arr = aggrVals(aggregId);
                return [d3.min(arr), d3.max(arr)];
            };

            let colorScale, singleColor;
            if (chartDef.colorMeasure.length) {
                colorScale = ChartColorScales.continuousColorScale(chartDef.colorOptions, aggrBounds('color')[0], aggrBounds('color')[1], aggrVals('color'), false, chartHandler.getChartTheme());
                colorScale.type = 'MEASURE';
            } else {
                singleColor = ColorUtils.toRgba(chartDef.colorOptions.singleColor, chartDef.colorOptions.transparency);
            }

            ChartLegendUtils.initLegend(chartDef, null, chartHandler.legendsWrapper, colorScale);

            if (colorScale) {
                chartHandler.legendsWrapper.getLegend(0).formatter = ChartFormatting.getForLegend(chartDef.colorMeasure[0], colorScale.innerScale.domain());;
            }

            ChartLegendUtils.drawLegend(chartDef, chartHandler, $container).then(function() {
                _MapCharts.adjustLegendPlacement(chartDef, $container);

                $timeout(() => {
                    const elt = _MapCharts.getOrCreateMapContainer($container);
                    const map = _MapCharts.createMapIfNeeded(elt, chartHandler.chartSpecific, chartDef);
                    if (elt.data('leaflet-data-layer')) {
                        map.removeLayer(elt.data('leaflet-data-layer'));
                    }
                    _MapCharts.repositionMap(map, elt, data);

                    const sizeScale = d3.scale.sqrt().range([chartDef.bubblesOptions.defaultRadius, chartDef.bubblesOptions.defaultRadius*5]).domain(aggrBounds('size'));

                    function onEachFeature(feature, layer) {
                        if (!ChartFeatures.shouldDisplayTooltips(chartHandler.noTooltips, chartDef.tooltipOptions)) {
                            return;
                        }

                        let html = '<h4>' + feature.properties.label+'</h4>';

                        if (!_.isNil(feature.properties.color)) {
                            html += sanitize(ChartLabels.getLongMeasureLabel(chartDef.colorMeasure[0]));
                            html += ': <strong>';
                            html += sanitize(ChartFormatting.getForIsolatedNumber(chartDef.colorMeasure[0])(feature.properties.color)) + '</strong><br />';

                        }
                        if (!_.isNil(feature.properties.size)) {
                            html += sanitize(ChartLabels.getLongMeasureLabel(chartDef.sizeMeasure[0]));
                            html += ': <strong>';
                            html += sanitize(ChartFormatting.getForIsolatedNumber(chartDef.sizeMeasure[0])(feature.properties.size)) + '</strong><br />';
                        }
                        if (!_.isNil(feature.properties.count)) {
                            html += 'Value count' + ': <strong>';
                            html += ChartFormatting.getForIsolatedNumber()(feature.properties.count) + '</strong><br />';
                        }

                        if (chartDef.tooltipMeasures.length > 0) {
                            html += '<hr/>';
                        }
                        chartDef.tooltipMeasures.forEach(function(measure, j) {
                            const tooltipFormatter = ChartFormatting.getForIsolatedNumber(measure);
                            html += sanitize(ChartLabels.getLongMeasureLabel(measure)) + ': <strong>' + sanitize(tooltipFormatter(feature.properties[j])) + '</strong><br/>';
                        });

                        //Classname is a locator for the integration tests - Leaflet API doesn't allow us to add an id
                        layer.bindPopup(html, { className: 'qa-chart-tooltip' });
                    }
                    let layer;

                    if (chartDef.variant == CHART_VARIANTS.filledMap) {
                        chartDef.sizeMeasure = [];
                        const myStyle = function(feature) {
                            return {
                                'color': singleColor || colorScale(feature.properties['color']),
                                'fillColor': singleColor || colorScale(feature.properties['color']),
                                'fillOpacity' : chartDef.fillOpacity,
                                'weight': 1,
                                'opacity': 1
                            };
                        };
                        layer = L.geoJson(geo.features, {
                            style: myStyle,
                            onEachFeature : onEachFeature
                        });
                        map.addLayer(layer);
                    } else {
                        layer = L.geoJson(geo.features, {
                            pointToLayer : function(feature, latlng) {
                                const size = feature.properties['size'] != null ? sizeScale(feature.properties['size']) : chartDef.bubblesOptions.defaultRadius;
                                const color = singleColor || (feature.properties['color'] != null ? colorScale(feature.properties['color']) : '#666');

                                return L.circleMarker(latlng, {
                                    radius : size,
                                    'color': color,
                                    'fillColor': color,
                                    'opacity' : 0.85,
                                    'fillOpacity' : 0.85
                                });
                            },
                            onEachFeature : onEachFeature
                        });
                        map.addLayer(layer);
                    }

                    elt.data('leaflet-data-layer', layer);
                });
            }).finally(function() {
                /*
                 * Signal to the callee handler that the chart has been loaded.
                 * Dashboards use it to determine when all insights are completely loaded.
                 */
                if (typeof(chartHandler.loadedCallback) === 'function') {
                    $timeout(() => {
                        chartHandler.loadedCallback();
                    });
                }
            });
        };
    });
})();

;
/* jshint loopfunc: true*/
(function() {
    'use strict';

    const app = angular.module('dataiku.charts');

    // (!) This service previously was in static/dataiku/js/simple_report/maps/scatter.js
    app.factory('DensityHeatMapChart', function(ChartFormatting, ChartUADimension, _MapCharts, ChartLabels, UaChartsCommon, ChartLegendUtils, ChartColorScales, DKU_PALETTE_NAMES) {
        return function($container, chartDef, data, chartHandler) {

            const elt = _MapCharts.getOrCreateMapContainer($container);
            // Handle the scatter map diverging color palette after transition to density map
            chartHandler.legendsWrapper.deleteLegends();

            chartDef.colorOptions.paletteType = 'CONTINUOUS';

            ChartLegendUtils.drawLegend(chartDef, chartHandler, $container, ChartColorScales).then(function() {
                _MapCharts.adjustLegendPlacement(chartDef, $container);

                // Create leaflet layer
                const layerGroup = L.layerGroup();

                // Get map
                const map = _MapCharts.createMapIfNeeded(elt, chartHandler.chartSpecific, chartDef);
                _MapCharts.repositionMap(map, elt, data);

                // Remove the existing layer to avoid multiple layers
                if (elt.data('leaflet-data-layer')) {
                    map.removeLayer(elt.data('leaflet-data-layer'));
                }


                // Remove the existing heatmap is there is one
                let existingHeatMapLayer;
                map.eachLayer(function(layer) {
                    if (layer.options && layer.options.id) {
                        if (layer.options.id === 'heatmap') {
                            existingHeatMapLayer = layer;
                        }
                    }
                });
                if (existingHeatMapLayer) {
                    map.removeLayer(existingHeatMapLayer);
                }

                // Get the gradient for leaflet heatmap
                const colorOptions = {
                    colorPalette: chartDef.colorOptions.colorPalette,
                    transparency: 1,
                    customPalette: chartDef.colorOptions.customPalette
                };
                let scale = ChartColorScales.continuousColorScale(colorOptions, 0, 1, undefined, undefined, chartHandler.getChartTheme());
                if (colorOptions.colorPalette === DKU_PALETTE_NAMES.CUSTOM && colorOptions.customPalette.colors.length === 0) {
                    scale = (bin) => 'rgba(0, 0, 0, 1)'; // avoid errors when colorOptions.customPalette.colors is empty
                }
                const gradient = {};
                for (let i = 0; i <= 9; i++) {
                    gradient[i / 10] = scale(i / 10);
                }

                const uaLFn = ChartLabels.getUaLabel;
                const hasUAColor = false;

                // Intermediate operation for the scale computation in the scatter plot
                const makeIntensityScale = function(chartDef, data) {
                    if (ChartUADimension.isTrueNumerical(chartDef.uaSize[0])) {
                        return d3.scale.sqrt().range([0, 1])
                            .domain([data.values.size.num.min, data.values.size.num.max]);
                    } else if (ChartUADimension.isDateRange(chartDef.uaSize[0])) {
                        return d3.scale.sqrt().range([0, 1])
                            .domain([data.values.size.ts.min, data.values.size.ts.max]);
                    } else {
                        throw new ChartIAE('Cannot use ALPHANUM as size scale');
                    }
                };

                // If a column is given as a size in the front bar, create the helper function to get the right weight
                const hasUASize = UaChartsCommon.hasUASize(chartDef);
                let intensityScale;
                let getScaleWeight;
                if (hasUASize) {
                    intensityScale = makeIntensityScale(chartDef, data);
                    getScaleWeight = function(chartDef, data, i, sizeScale) {
                        if (chartDef.uaSize.length) {
                            let sizeValue;
                            if (ChartUADimension.isTrueNumerical(chartDef.uaSize[0])) {
                                sizeValue = data.values.size.num.data[i];
                            } else if (ChartUADimension.isDateRange(chartDef.uaSize[0])) {
                                sizeValue = data.values.size.ts.data[i];
                            }
                            return sizeScale(sizeValue);
                        } else {
                            return 1;
                        }
                    };
                }

                // Tuning values for the visual parameters
                const radiusRangeMultiplier = 40;

                // Create the core data that will be displayed by Leaflet.heat
                const geopoints = [];
                data.xAxis.num.data.forEach(function(x, i) {
                    const y = data.yAxis.num.data[i];
                    let r;
                    if (hasUASize) {
                        r = getScaleWeight(chartDef, data, i, intensityScale);
                    } else {
                        r = 1;
                    }
                    geopoints.push([y, x, r * chartDef.colorOptions.heatDensityMapIntensity]);
                });

                // Create the heatmap and add it as a layer
                if (chartDef.colorOptions.heatDensityMapRadius !== 0) {
                    const heatMapLayer = L.heatLayer(geopoints, { radius: chartDef.colorOptions.heatDensityMapRadius * radiusRangeMultiplier, id: 'heatmap', gradient: gradient });
                    heatMapLayer.addTo(map);
                }

                // Options of the Leaflet CircleMarker
                const options = {
                    stroke: false,
                    color: 'rgb(0,0,0)',
                    opacity: 1,
                    fill: false,
                    fillColor: 'rgb(255,0,0)',
                    fillOpacity: 1,
                    radius: 5
                };

                // Create tooltip
                data.xAxis.num.data.forEach(function(x, i) {

                    const y = data.yAxis.num.data[i];

                    const pointLayer = L.circleMarker([y, x], options);

                    let html = '';
                    html += 'Lon: <strong>' + ChartFormatting.getForIsolatedNumber()(x) + '</strong><br />';
                    html += 'Lat: <strong>' + ChartFormatting.getForIsolatedNumber()(y) + '</strong><br />';

                    if (hasUASize && (!hasUAColor || (chartDef.uaSize[0].column !== chartDef.uaColor[0].column || chartDef.uaColor[0].dateMode !== 'RANGE'))) {
                        html += uaLFn(chartDef.uaSize[0]) + ': <strong>' +
                        UaChartsCommon.formattedSizeVal(chartDef, data, i) + '</strong><br />';
                    }
                    if (chartDef.uaTooltip.length > 0) {
                        html += '<hr/>';
                    }
                    chartDef.uaTooltip.forEach(function(ua, j) {
                        html += uaLFn(ua) + ': <strong>' + UaChartsCommon.formattedVal(data.values['tooltip_' + j], ua, i) + '</strong><br/>';
                    });

                    //Classname is a locator for the integration tests - Leaflet API doesn't allow us to add an id
                    pointLayer.bindPopup(html, { className: 'qa-chart-tooltip' });
                    pointLayer.on('mouseover', function(e) {
                        this.setStyle({
                            stroke: true,
                            fill: true
                        });
                        this.openPopup();
                    });
                    pointLayer.on('mouseout', function(e) {
                        this.setStyle({
                            stroke: false,
                            fill: false
                        });
                        this.closePopup();
                    });
                    layerGroup.addLayer(pointLayer);
                });

                // Add layer to map
                layerGroup.addTo(map);
                elt.data('leaflet-data-layer', layerGroup);

            }).finally(function() {
                /*
                 * Signal to the callee handler that the chart has been loaded.
                 * Dashboards use it to determine when all insights are completely loaded.
                 */
                if (typeof (chartHandler.loadedCallback) === 'function') {
                    chartHandler.loadedCallback();
                }
            });
        };
    });

})();

;
/* jshint loopfunc: true*/
(function(){
    'use strict';

    const app = angular.module('dataiku.charts');

    // (!) This service previously was in static/dataiku/js/simple_report/maps/geometry.js
    app.factory('GeometryMapChart', function(ChartFormatting, _MapCharts, ChartLegendUtils, ChartFeatures, ColorUtils, ChartColorUtils, UaChartsCommon, _GeometryCommon, ChartLabels, ChartUADimension, $timeout, BuiltinMapBackgrounds) {
        return function($container, chartDef, data, chartHandler) {

            const elt = _MapCharts.getOrCreateMapContainer($container);
            const geoJsons = data.geoJsons.map(unparsedGeo => JSON.parse(unparsedGeo));
            const layers = {};
            const colorCaches = {};
            const colorScales = {};
            const singleColors = {};

            chartHandler.legendsWrapper.deleteLegends();
            const displayableGeoLayers = _GeometryCommon.getDisplayableLayers(chartDef.geoLayers);

            for (let geometryIndex = 0; geometryIndex < displayableGeoLayers.length; geometryIndex++) {
                const geoLayer = displayableGeoLayers[geometryIndex];
                const geometry = geoLayer.geometry[0];
                colorCaches[geometryIndex] = {};
                // Build color scale
                const hasUAColor = _GeometryCommon.hasUAColor(geoLayer);
                if (hasUAColor) {
                    colorScales[geometryIndex] = _GeometryCommon.makeColorScale(geoLayer, data, chartHandler, geometryIndex);
                } else {
                    singleColors[geometryIndex] = _GeometryCommon.makeSingleColor(geoLayer);
                }

                // Build legend. Can we make some of this common with other UA color scales ?
                const isLegendDiscrete = (ChartUADimension.isAlphanumLike(geoLayer.uaColor[0]) || ChartUADimension.isDiscreteDate(geoLayer.uaColor[0]));
                if (hasUAColor && isLegendDiscrete) {
                    const baseFadeColor = BuiltinMapBackgrounds.getBackgrounds().find(b => b.id === chartDef.mapOptions.tilesLayer).fadeColor || '#333';
                    const fadeColor = ColorUtils.toRgba(baseFadeColor, .5 * geoLayer.colorOptions.transparency);
                    const colorTokenName = ChartColorUtils.getPaletteName(geometryIndex);

                    const legend = {
                        type: 'COLOR_DISCRETE',
                        label: geometry.column,
                        items: data.values[colorTokenName].str.sortedMapping.map(function(_, rowIndex) {
                            const color = colorScales[geometryIndex](rowIndex);
                            return {
                                label: {
                                    ...data.values[colorTokenName].str.sortedMapping[rowIndex],
                                    colorId: ChartColorUtils.getColorId(geoLayer.genericMeasures, { data }, rowIndex, geometryIndex)
                                },
                                color,
                                focusFn: function(){
                                    layers[geometryIndex].setStyle(function(feature) {
                                        if (data.values[colorTokenName].str.data[feature.properties.idx] === rowIndex) {
                                            return {
                                                color,
                                                opacity: 1,
                                                weight: chartDef.strokeWidth
                                            };
                                        } else {
                                            return {
                                                color: fadeColor,
                                                opacity: 1,
                                                weight: chartDef.strokeWidth };
                                        }
                                    });
                                },
                                unfocusFn : function() {
                                    layers[geometryIndex].setStyle(function(feature) {
                                        const c = singleColors[geometryIndex] || _GeometryCommon.makeColor(geoLayer, data, feature.properties.idx, colorScales[geometryIndex], singleColors[geometryIndex], colorCaches[geometryIndex], geometryIndex);
                                        return {
                                            color: c,
                                            opacity: 1,
                                            weight: chartDef.strokeWidth
                                        };
                                    });
                                },
                                focused : false
                            };
                        })
                    };
                    chartHandler.legendsWrapper.pushLegend(legend);

                } else if (colorScales[geometryIndex]) {
                    const legend = ChartLegendUtils.createContinuousLegend(geoLayer.uaColor[0], colorScales[geometryIndex]);
                    legend.label = geometry.column;
                    if (ChartUADimension.isDateRange(geoLayer.uaColor[0])) {
                        legend.formatter = ChartFormatting.getForDate();
                    } else {
                        legend.formatter = ChartFormatting.getForLegend(geoLayer.uaColor[0], colorScales[geometryIndex].innerScale.domain());
                    }
                    chartHandler.legendsWrapper.pushLegend(legend);
                } else {
                    if (chartDef.geoLayers.length < 3) {
                        chartHandler.legendsWrapper.deleteLegends();
                    } else {
                        const legend = ChartLegendUtils.getSingleColorLegend(singleColors[geometryIndex], geometry.column);
                        chartHandler.legendsWrapper.pushLegend(legend);
                    }
                }
            }

            ChartLegendUtils.drawLegend(chartDef, chartHandler, $container).then(function() {

                _MapCharts.adjustLegendPlacement(chartDef, $container);

                const map = _MapCharts.createMapIfNeeded(elt, chartHandler.chartSpecific, chartDef);
                if (elt.data('leaflet-data-layer')) {
                    map.removeLayer(elt.data('leaflet-data-layer'));
                }
                _MapCharts.repositionMap(map, elt, data);

                function onEachFeatureClosure(geometryIndex, geoLayer){
                    return function onEachFeature(feature, layer) {
                        if (!ChartFeatures.shouldDisplayTooltips(chartHandler.noTooltips, chartDef.tooltipOptions)) {
                            return;
                        }

                        let html = '';

                        if (_GeometryCommon.hasUAColor(geoLayer)) {
                            const colorTokenName = ChartColorUtils.getPaletteName(geometryIndex);
                            if (colorTokenName in data.values){
                                html += sanitize(ChartLabels.getUaLabel(geoLayer.uaColor[0])) + ': <strong>' + sanitize(_GeometryCommon.formattedColorVal(chartDef, data, feature.properties.idx, colorTokenName, geometryIndex)) +'</strong><br />';
                            }
                        }

                        chartDef.uaTooltip.forEach(function(ua, j) {
                            html += sanitize(ChartLabels.getUaLabel(ua)) + ': <strong>' + sanitize(UaChartsCommon.formattedVal(data.values['tooltip_' + j], ua, feature.properties.idx)) + '</strong><br/>';
                        });

                        if (html.length) {
                            //Classname is a locator for the integration tests - Leaflet API doesn't allow us to add an id
                            layer.bindPopup(html, { className: 'qa-chart-tooltip' });
                        }
                    };
                }

                function getLayer(geoJsons, geoLayer, geometryIndex){
                    const layerSingleColor = _GeometryCommon.makeSingleColor(geoLayer);
                    return L.geoJson(geoJsons.features, {
                        style: function(feature) {
                            const c = _GeometryCommon.makeColor(
                                geoLayer,
                                data,
                                feature.properties.idx,
                                colorScales[geometryIndex],
                                layerSingleColor,
                                colorCaches[geometryIndex],
                                geometryIndex
                            );
                            return {
                                color: c,
                                opacity: 1,
                                weight: chartDef.strokeWidth,
                                fillOpacity: chartDef.fillOpacity
                            };
                        },
                        onEachFeature : onEachFeatureClosure(geometryIndex, geoLayer),
                        pointToLayer: function(feature, latlng) {
                            const c = _GeometryCommon.makeColor(
                                geoLayer,
                                data,
                                feature.properties.idx,
                                colorScales[geometryIndex],
                                layerSingleColor,
                                colorCaches[geometryIndex],
                                geometryIndex
                            );

                            const geojsonMarkerOptions = {
                                radius: 5,
                                fillColor: c,
                                color: c,
                                weight: chartDef.strokeWidth,
                                opacity: 1,
                                fillOpacity: 1
                            };
                            return L.circleMarker(latlng, geojsonMarkerOptions);
                        }
                    });
                }

                function plot(geometryIndex){
                    map.addLayer(layers[geometryIndex]);
                    if (map.$justCreated) {
                        $timeout(function() {
                            map.fitBounds(layers[geometryIndex]);
                        });
                    }
                    elt.data('leaflet-data-layer', layers[geometryIndex]);
                }

                // delete all the current geometry layers
                for (const displayedGeo of chartHandler.displayedGeometries){
                    map.removeLayer(displayedGeo.geoJsonLayer);
                }
                const newGeometries = [];
                for (let geometryIndex = 0; geometryIndex < displayableGeoLayers.length; geometryIndex++) {
                    const newGeoLayer = displayableGeoLayers[geometryIndex];
                    layers[geometryIndex] = getLayer(
                        geoJsons[geometryIndex],
                        newGeoLayer,
                        geometryIndex
                    );
                    plot(geometryIndex);
                    newGeometries[geometryIndex] = { geometry: angular.copy(newGeoLayer[geometryIndex]), geoJsonLayer: layers[geometryIndex] };
                }
                chartHandler.displayedGeometries= newGeometries;

            }).finally(function(){
                /*
                 * Signal to the callee handler that the chart has been loaded.
                 * Dashboards use it to determine when all insights are completely loaded.
                 */
                if (typeof(chartHandler.loadedCallback) === 'function') {
                    chartHandler.loadedCallback();
                }
            });

        };
    });

})();

;
/* jshint loopfunc: true*/
(function() {
    'use strict';

    const app = angular.module('dataiku.charts');

    // (!) This service previously was in static/dataiku/js/simple_report/maps/scatter.js
    app.factory('GridMapChart', function(ChartFormatting, ChartFeatures, _MapCharts, ChartLabels, UaChartsCommon, ChartLegendUtils, ChartColorScales) {
        return function($container, chartDef, data, chartHandler) {

            const elt = _MapCharts.getOrCreateMapContainer($container);
            const map = _MapCharts.createMapIfNeeded(elt, chartHandler.chartSpecific, chartDef);
            if (elt.data('leaflet-data-layer')) {
                map.removeLayer(elt.data('leaflet-data-layer'));
            }
            _MapCharts.repositionMap(map, elt, data);

            // Build color scale
            const hasColor = chartDef.colorMeasure.length;
            let colorScale, resultingColor;
            if (hasColor) {
                colorScale = ChartColorScales.continuousColorScale(chartDef.colorOptions, data.aggregations.color.min, data.aggregations.color.max, data.aggregations.color.tensor, undefined, chartHandler.getChartTheme());
                colorScale.type = 'MEASURE';
            } else {
                resultingColor = UaChartsCommon.makeSingleColor(chartDef);
            }

            ChartLegendUtils.initLegend(chartDef, null, chartHandler.legendsWrapper, colorScale);

            if (colorScale) {
                chartHandler.legendsWrapper.getLegend(0).formatter = ChartFormatting.getForLegend(chartDef.colorMeasure[0], colorScale.innerScale.domain());
            }

            ChartLegendUtils.drawLegend(chartDef, chartHandler, $container).then(function() {
                _MapCharts.adjustLegendPlacement(chartDef, $container);

                const layerGroup = L.layerGroup();

                data.cellMinLat.forEach(function(x, i) {
                    const minLat = data.cellMinLat[i];
                    const minLon = data.cellMinLon[i];
                    const maxLat = minLat + data.gridLatDeg;
                    const maxLon = minLon + data.gridLonDeg;

                    const c = hasColor ? colorScale(data.aggregations.color.tensor[i]) : resultingColor;

                    const rect = L.rectangle([[minLat, minLon], [maxLat, maxLon]], {
                        stroke: false,
                        fill: true,
                        fillColor: c,
                        fillOpacity: 1
                    });

                    if (ChartFeatures.shouldDisplayTooltips(chartHandler.noTooltips, chartDef.tooltipOptions)) {
                        let html = '';

                        html += 'Lon: <strong>' + ChartFormatting.getForIsolatedNumber()(minLon + (maxLon - minLon) / 2) + '</strong><br />';
                        html += 'Lat: <strong>' + ChartFormatting.getForIsolatedNumber()(minLat + (maxLat - minLat) / 2) + '</strong><br />';

                        if (hasColor) {
                            const colorFormatter = ChartFormatting.getForIsolatedNumber(chartDef.colorMeasure[0]);
                            html += ChartLabels.getShortMeasureLabel(chartDef.colorMeasure[0])
                                + ': <strong>' + sanitize(colorFormatter(data.aggregations.color.tensor[i])) + '</strong>';
                        }

                        if (chartDef.tooltipMeasures.length > 0) {
                            html += '<hr/>';
                        }

                        chartDef.tooltipMeasures.forEach(function(measure, j) {
                            const tooltipFormatter = ChartFormatting.getForIsolatedNumber(measure);
                            html += `${ChartLabels.getLongMeasureLabel(measure)}: <strong>${sanitize(tooltipFormatter(data.aggregations['tooltip_' + j].tensor[i]))}</strong><br/>`;
                        });

                        //Classname is a locator for the integration tests - Leaflet API doesn't allow us to add an id
                        rect.bindPopup(html, { className: 'qa-chart-tooltip' });
                    }

                    layerGroup.addLayer(rect);
                });
                layerGroup.addTo(map);

                elt.data('leaflet-data-layer', layerGroup);
            }).finally(function() {
                /*
                 * Signal to the callee handler that the chart has been loaded.
                 * Dashboards use it to determine when all insights are completely loaded.
                 */
                if (typeof (chartHandler.loadedCallback) === 'function') {
                    chartHandler.loadedCallback();
                }
            });
        };
    });

})();

;
/* jshint loopfunc: true*/
(function() {
    'use strict';

    const app = angular.module('dataiku.charts');

    // (!) This service previously was in static/dataiku/js/simple_report/maps/scatter.js
    app.factory('ScatterMapChart', function(ChartFormatting, ChartFeatures, ChartUADimension, _MapCharts, ChartLabels, UaChartsCommon, ChartLegendUtils, ColorUtils, ChartColorUtils, BuiltinMapBackgrounds) {
        return function($container, chartDef, data, chartHandler) {

            const elt = _MapCharts.getOrCreateMapContainer($container);

            let layerGroup, colorScale;

            // Build color scale
            const hasUAColor = UaChartsCommon.hasUAColor(chartDef);
            if (hasUAColor) {
                colorScale = UaChartsCommon.makeColorScale(chartDef, data, chartHandler);
            } else {
                var resultingColor = UaChartsCommon.makeSingleColor(chartDef);
            }

            // Build legend
            if (hasUAColor && (ChartUADimension.isAlphanumLike(chartDef.uaColor[0]) || ChartUADimension.isDiscreteDate(chartDef.uaColor[0]))) {
                const baseFadeColor = BuiltinMapBackgrounds.getBackgrounds().find(b => b.id === chartDef.mapOptions.tilesLayer).fadeColor || '#333';
                const fadeColor = ColorUtils.toRgba(baseFadeColor, .5 * chartDef.colorOptions.transparency);

                const legend = {
                    type: 'COLOR_DISCRETE',
                    items: data.values.color.str.sortedMapping.map(function(_, v) {
                        return {
                            label: {
                                ...data.values.color.str.sortedMapping[v],
                                colorId: ChartColorUtils.getColorId(chartDef.genericMeasures, { data }, v)
                            },
                            color: colorScale(v),
                            focusFn: function() {
                                layerGroup.getLayers().forEach(function(layer) {
                                    const opts = layer.options;
                                    if (!opts.actualFillColor) {
                                        opts.actualFillColor = opts.fillColor;
                                    }

                                    if (opts.colorIdx !== v) {
                                        opts.fillColor = fadeColor;
                                    } else {
                                        opts.fillColor = opts.actualFillColor;
                                    }

                                    layer.setStyle(opts);
                                });
                            },
                            unfocusFn: function() {
                                layerGroup.getLayers().forEach(function(layer) {
                                    const opts = layer.options;
                                    opts.fillColor = opts.actualFillColor;
                                    layer.setStyle(opts);
                                });
                            },
                            focused: false
                        };
                    })
                };

                chartHandler.legendsWrapper.deleteLegends();
                chartHandler.legendsWrapper.pushLegend(legend);
            } else {
                if (colorScale) {
                    colorScale.type = 'MEASURE';
                }
                ChartLegendUtils.initLegend(chartDef, null, chartHandler.legendsWrapper, colorScale);
                if (colorScale) {
                    if (ChartUADimension.isDateRange(chartDef.uaColor[0])) {
                        chartHandler.legendsWrapper.getLegend(0).formatter = ChartFormatting.getForDate();
                    } else {
                        chartHandler.legendsWrapper.getLegend(0).formatter = ChartFormatting.getForLegend(chartDef.uaColor[0], colorScale.innerScale.domain());
                    }
                }
            }

            // Draw legend, then map
            ChartLegendUtils.drawLegend(chartDef, chartHandler, $container).then(function() {
                _MapCharts.adjustLegendPlacement(chartDef, $container);

                const map = _MapCharts.createMapIfNeeded(elt, chartHandler.chartSpecific, chartDef);
                if (elt.data('leaflet-data-layer')) {
                    map.removeLayer(elt.data('leaflet-data-layer'));
                }
                _MapCharts.repositionMap(map, elt, data);

                const hasUASize = UaChartsCommon.hasUASize(chartDef);
                if (hasUASize) {
                    var sizeScale = UaChartsCommon.makeSizeScale(chartDef.bubblesOptions.defaultRadius, chartDef.uaSize[0], data, 1);
                }
                const uaLFn = ChartLabels.getUaLabel;

              