(function() {
'use strict';

const app = angular.module('dataiku.apideployer');

app.constant('ENDPOINT_TYPES_COMPATIBLE_WITH_DEPLOY_ANYWHERE', [
    "STD_PREDICTION",
    "STD_CLUSTERING",
    "STD_CAUSAL_PREDICTION",
    "CUSTOM_PREDICTION",
    "PY_FUNCTION"
]);

app.service("APIDeployerDeploymentUtils", function(DeployerUtils, ENDPOINT_TYPES_COMPATIBLE_WITH_DEPLOY_ANYWHERE) {
    var svc = {};

    svc.infraSupportOpenAPIURI = function(infraType) {
        return (infraType === "STATIC")
    }

    svc.computeSimpleOpenAPIURLs = function(lightStatus) {
        const deplBI = lightStatus.deploymentBasicInfo;
        const infraBI = lightStatus.infraBasicInfo;
        const serviceId = deplBI.deployedServiceId || deplBI.publishedServiceId;

        if (svc.infraSupportOpenAPIURI(lightStatus.infraBasicInfo.type)) {
            return "/" + serviceId + "/swagger";
        } else {
            return "";
        }
    }

    svc.getDisplayedUrl = function(apiNodeBasicInfo) {
        return apiNodeBasicInfo.externalUrl || apiNodeBasicInfo.url;
    }

    svc.computeEndpointURLs = function(lightStatus, heavyStatus, endpoint) {
        function addEndpointSpecificSuffix(epURL) {
            switch (endpoint.type) {
                case "STD_PREDICTION":
                case "CUSTOM_PREDICTION":
                case "CUSTOM_R_PREDICTION":
                case "STD_CLUSTERING":
                    return epURL + "/predict";
                case "STD_CAUSAL_PREDICTION":
                    return epURL + "/predict-effect";
                case "STD_FORECAST":
                    return epURL + "/forecast";
                case "PY_FUNCTION":
                case "R_FUNCTION":
                    return epURL + "/run";
                case "DATASETS_LOOKUP":
                    return epURL + "/lookup";
                case "SQL_QUERY":
                    return epURL + "/query";
            }
        }

        function removeURLTrailingSlashes(epURL) {
            let ret = epURL;
            while (ret.length > 0 && ret[ret.length - 1] == '/') {
                ret = ret.slice(0, ret.length - 1);
            }
            return ret;
        }

        const deplBI = lightStatus.deploymentBasicInfo;
        const infraBI = lightStatus.infraBasicInfo;
        const serviceId = deplBI.deployedServiceId || deplBI.publishedServiceId;
        const endpointId = endpoint.id;

        if (infraBI.type == "STATIC") {
            return infraBI.apiNodeUrls
                .map(svc.getDisplayedUrl)
                .map(removeURLTrailingSlashes)
                .map(apiNode => `${apiNode}/public/api/v1/${serviceId}/${endpointId}`)
                .map(addEndpointSpecificSuffix);
        } else if (infraBI.type == "SNOWPARK") {
            if (deplBI.deploymentInfo.snowparkEndpointUrl) {
                const endpointHost = deplBI.deploymentInfo.snowparkEndpointUrl;
                const suffix = "https://" + removeURLTrailingSlashes(endpointHost) + "/public/api/v1/" + serviceId + "/"  + endpointId;
                return [addEndpointSpecificSuffix(suffix)];
            } else {
                return [];
            }
        } else {
            if (heavyStatus.publicURL) {
                let publicApiPath = heavyStatus.shouldAddPublicApiPath ? "/public/api/v1/" : "/";
                const suffix = removeURLTrailingSlashes(heavyStatus.publicURL) + publicApiPath + serviceId + "/"  + endpointId;
                return [addEndpointSpecificSuffix(suffix)];
            } else {
                return [];
            }
        }
    };

    svc.buildZeroDatapoints = function(timeRange) {
        const now = Math.round(new Date().getTime() / 1000);
        let delta;
        switch (timeRange) {
            case "ONE_DAY":
                delta = 24 * 3600;
                break;
            case "ONE_HOUR":
                delta = 1 * 3600;
                break;
            case "SIX_HOURS":
            default:
                delta = 6 * 3600;
                break;
        }
        const ret = [];
        for (let i = 71; i >= 0; i--) {
            ret.push([null, Math.round(now-delta*i/71)]);
        }
        return ret;
    };

    /**
     * Helper method to map SM versions from a service lightStatus
     * @param {*} lightStatus
     * @returns A map of bundled SM versions per package and SM id as key
     */
    svc.getSMVersionsPerPackageAndId = function(lightStatus) {
        const ret = {};

        (lightStatus.packages || []).forEach(function(pkg) {
            ret[pkg.id] = (pkg.stdModels || []).reduce((obj, SMVersion) => ({
                ...obj,
                [SMVersion.id]: SMVersion
            }), {});
        });

        return ret;
    };

    /**
     * Helper method to map one to one the package version with the SM versions corresponding to the endpointid provided.
     * @param {{}} lightStatus
     * @param {String} endpointId
     * @param {[String]} packagesFilter (Optional) Array of package ids to filter the result
     * @returns A map of bundled SM versions per package and SM id as key
     */
    svc.getSMVersionPerPackageForEndpoint = function(lightStatus, endpointId, packagesFilter) {
        const ret = {};
        const packages = lightStatus.packages || [];
        const filter = (packagesFilter || []).reduce((obj, pkgId) => ({...obj, [pkgId]: {}}), {});

        packages.forEach(function(pkg) {
            if(packagesFilter && !(pkg.id in filter)) {
                return;
            }
            const packageEndpoint = (pkg.endpoints || []).find((endpoint) => endpointId === endpoint.id)
            ret[pkg.id] = packageEndpoint ? (pkg.stdModels || []).find((model) => model.id === packageEndpoint.modelId) : {};
        });

        return ret;
    };

    /**
     * Helper method to map one to one the package version with the list of endpoints.
     * @param {{}}} lightStatus
     * @returns A map of endpoints per package with package id as key
     */
    svc.getEndpointsPerPackage = function(lightStatus) {
        const ret = {};
        const packages = lightStatus.packages || [];

        packages.forEach(function(pkg) {
            ret[pkg.id] = pkg.endpoints || [];
        });

        return ret;
    };

    svc.isEndpointCompatibleWithDeployAnywhere = function(endpointType) {
        return ENDPOINT_TYPES_COMPATIBLE_WITH_DEPLOY_ANYWHERE.includes(endpointType);
    };

    svc.isEndpointCompatibleWithDeployAnywhereDatabricks = function(endpointType) {
        return ["STD_PREDICTION"].includes(endpointType);
    };

    /**
     * Helper method that groups the endpoint types in more generic groups.
     * @param {*} endpoint
     * @returns String containing the name of the generic group.
     */
    svc.getGenericEndpointType = function(endpoint) {
        switch(endpoint.type) {
            case "STD_PREDICTION":
            case "CUSTOM_PREDICTION":
            case "CUSTOM_R_PREDICTION":
            case "STD_CLUSTERING":
                return "predict";
            case "STD_FORECAST":
                return "forecast"
            case "STD_CAUSAL_PREDICTION":
                return "causal";
            case "R_FUNCTION":
            case "PY_FUNCTION":
                return "function";
            case "SQL_QUERY":
                return "query";
            case "DATASETS_LOOKUP":
                return "lookup";
        }
    };

    /**
     * Helper method to map origin info per package in the deployment info
     * @param {*} lightStatus
     * @returns
     */
    svc.getOriginInfoPerPackage = function(lightStatus) {
        const ret = {};

        (lightStatus.packages || []).forEach(function(pkg) {
            if(pkg.designNodeInfo) {
                ret[pkg.id] = DeployerUtils.getOriginInfo(pkg.designNodeInfo);
                if(pkg.type == "API_SERVICE" && ret[pkg.id].url) {
                    ret[pkg.id].apiDesignerUrl = ret[pkg.id].url + 'api-designer/' + lightStatus.serviceBasicInfo.id +'/summary/';
                }
            }
        });


        return ret;
    };

    svc.sortedInfrasByStages = function (infraBasicInfoList, stages) {
        const stagesIds = stages.map((x) => x.id);
        function compareInfra(a, b) {
            const aIndex = stagesIds.indexOf(a.stage);
            const bIndex = stagesIds.indexOf(b.stage);
            if (aIndex - bIndex != 0) {
                return aIndex - bIndex;
            } else {
                return a.id.localeCompare(b.id);
            }
        }
        return infraBasicInfoList.toSorted(compareInfra);
    };

    return svc;
})

app.service('APIDeployerDeploymentService', function($rootScope, CreateModalFromTemplate) {
    this.startCreateDeployment = function(preselectedService, preselectedVersion) {
        return CreateModalFromTemplate("/templates/api-deployer/new-deployment-modal.html", $rootScope, null, function(modalScope) {
            modalScope.newDepl.publishedServiceId = preselectedService;
            modalScope.newDepl.versionId = preselectedVersion;
            modalScope.disabledServiceAndVersionInputs = !!(preselectedService && preselectedVersion);
        });
    };

    this.openGovernanceStatusDeploymentId = function(deploymentId, infraId, versionId) {
        return CreateModalFromTemplate("/templates/deployer/governance-modal.html", $rootScope, "GovernanceAbstractDeployerModalController", function(modalScope) {
            modalScope.deploymentId = deploymentId;
            modalScope.infraId = infraId;
            modalScope.packageId = versionId;
            modalScope.refreshGovernanceStatuses('apiDeployer');
        });
    };

    this.openGovernanceStatusDeployment = function(deployment) {
        return CreateModalFromTemplate("/templates/deployer/governance-modal.html", $rootScope, "GovernanceAbstractDeployerModalController", function(modalScope) {
            modalScope.deployment = deployment;
            modalScope.refreshGovernanceStatuses('apiDeployer');
        });
    };

    this.openGovernanceStatusNewDeployment = function(publishedServiceId, infraId, versionId) {
        return CreateModalFromTemplate("/templates/deployer/governance-modal.html", $rootScope, "GovernanceAbstractDeployerModalController", function(modalScope) {
            modalScope.publishedItemId = publishedServiceId;
            modalScope.infraId = infraId;
            modalScope.packageId = versionId;
            modalScope.refreshGovernanceStatuses('apiDeployer');
        });
    };
});

app.factory('APIDeploymentInfraHelper', function() {
    return {
        isStaticInfra: function(type) {
            return type === "STATIC";
        },
        isK8SInfra: function(type) {
            return type === "K8S";
        },
        isFullyManagedInfra: function(type) {
            return ['SAGEMAKER', 'SNOWPARK', 'AZURE_ML', 'DATABRICKS', 'VERTEX_AI'].includes(type);
        }
    }
});

app.constant('STATIC_REFRESH_MODES', {
    LIGHT: 'LIGHT',
    FULL: 'FULL'
});

app.factory('ApiDeploymentSyncHelper', function() {
    return {
        isSuccessful: function(result) {
            return ["INFO", "SUCCESS"].includes(result.maxSeverity);
        },
        isSuccessfulWithWarning: function(result) {
            return result.maxSeverity === "WARNING";
        },
        prepareFailed: function(result) {
            return result.withPrepare && (!result.prepareReport || result.prepareReport.maxSeverity === "ERROR");
        },
        isError: function(result) {
            //return !result.syncReport || result.hookFailure || result.syncReport.nbNOKNodes > 0; // syncReport.nbNOKNodes > 0 will never be true for docker based deployments
            return result.maxSeverity === 'ERROR';
        },
    };
});

app.factory('StaticApiDeploymentSyncHelper', function(CreateModalFromTemplate, STATIC_REFRESH_MODES) {
    return {
        askMode: function($scope) {
            return CreateModalFromTemplate("/templates/api-deployer/ask-mode-modal.html", $scope, null, function(modalScope) {
                modalScope.lightMode = function() {
                    $scope.uiState.refreshMode = STATIC_REFRESH_MODES.LIGHT;
                    $scope.deployStatic();
                    modalScope.dismiss();
                }
                modalScope.fullMode = function() {
                    $scope.uiState.refreshMode = STATIC_REFRESH_MODES.FULL;
                    $scope.deployStatic();
                    modalScope.dismiss();
                }
            })
        },
    };
});

app.service("DeploymentStatusEndpointSampleCodeGenerator", function(Assert, APIDeployerDeploymentUtils){
    var svc = {};

    svc.getFirstBaseURI = function(lightStatus, heavyStatus) {
        const infraBI = lightStatus.infraBasicInfo;
        if (infraBI.type == "STATIC") {
            return APIDeployerDeploymentUtils.getDisplayedUrl(infraBI.apiNodeUrls[0]);
        } else {
            return heavyStatus.publicURL;
        }
    };

    svc.getJSONExplanationsParams = function(endpoint) {
        const tq = (endpoint.testQueries || []).find(tq => tq.q && tq.q.explanations);
        return tq ? tq.q.explanations : undefined;
    }

    svc.getJSONData = function(endpoint) {
        let jsonData = null;
        switch (endpoint.type) {
        case "STD_PREDICTION":
        case "STD_CAUSAL_PREDICTION":
        case "CUSTOM_PREDICTION":
        case "CUSTOM_R_PREDICTION":
        case "STD_CLUSTERING": {
            if (endpoint.testQueries && endpoint.testQueries.length) {
                const tq = endpoint.testQueries[0];
                if (tq.q && "features" in tq.q) {
                    jsonData = tq.q.features;
                }
            }
            if (!jsonData) {
                jsonData = {
                    categorical_feature1 : "value1",
                    numerical_feature1: 42,
                    categorical_feature2: "value2"
                }
            }
            break;
        }
        case "STD_FORECAST": {
            if (endpoint.testQueries && endpoint.testQueries.length) {
                const tq = endpoint.testQueries[0];
                if (tq.q && "items" in tq.q) {
                    jsonData = tq.q.items;
                }
            }
            if (!jsonData) {
                jsonData = [{
                    timeFeature: "2013-01-01T00:00:00.000Z",
                    featureToForecast: 1.0,
                    otherNumericalFeature: 10.0,
                    otherCategoricalFeature: "value"
                }];
            }
            break;
        }
        case "PY_FUNCTION":
        case "R_FUNCTION":
        case "SQL_QUERY":
            if (endpoint.testQueries && endpoint.testQueries.length) {
                const tq = endpoint.testQueries[0];
                if (tq.q) {
                    jsonData = tq.q;
                }
            }
            if (!jsonData) {
                jsonData = {
                    param1 : "value1",
                    param2: 42
                };
            }
            break;
        case "DATASETS_LOOKUP":
            if (endpoint.testQueries && endpoint.testQueries.length) {
                const tq = endpoint.testQueries[0];
                if (tq.q && "data" in tq.q) {
                    jsonData = tq.q.data;
                }
            }
            if (!jsonData) {
                jsonData = {
                    string_key1 : "keyvalue1",
                    numerical_key2: 42
                };
            }
            break;
        }
        return jsonData;
    };

    svc.getVertexAIJSONData = function (endpoint) {
        // Vertex testQueries have been modified by AbstractFullyManagedDeploymentHeavyStatus#updateTestQuery()
        // There is no more 'features' field and the query is encapsulated in a "instances" key.
        if (endpoint.testQueries && endpoint.testQueries.length) {
            return endpoint.testQueries[0].q;
        } else {
            return {
                "instances": [{
                    categorical_feature1: "value1",
                    numerical_feature1: 42,
                    categorical_feature2: "value2"
                }]
            };
        }
    }

    svc.getSnowparkJSONData = function (endpoint) {
        if (endpoint.testQueries && endpoint.testQueries.length) {
            return endpoint.testQueries[0].q;
        } else {
            return {
                "features": {
                    categorical_feature1: "value1",
                    numerical_feature1: 42,
                    categorical_feature2: "value2"
                }
            };
        }
    }

    svc.getAzureMLOrSageMakerJSONData = function (endpoint) {
        // Deploy Anywhere testQueries have been modified by AbstractFullyManagedDeploymentHeavyStatus#updateTestQuery()
        if (endpoint.testQueries && endpoint.testQueries.length) {
            return endpoint.testQueries[0].q;
        } else {
            const featuresData = {
                categorical_feature1 : "value1",
                numerical_feature1: 42,
                categorical_feature2: "value2"
            };
            switch (endpoint.type) {
                case "STD_PREDICTION":
                case "STD_CLUSTERING":
                case "STD_CAUSAL_PREDICTION":
                case "CUSTOM_PREDICTION": {
                    return {
                        "items": [{ "features": featuresData }]
                    };
                }
                case "PY_FUNCTION": {
                    return featuresData;
                }
                case "CUSTOM_R_PREDICTION":
                case "STD_FORECAST":
                case "R_FUNCTION":
                case "SQL_QUERY":
                case "DATASETS_LOOKUP": {
                    // TODO: Still not supported in Vertex AI deployments
                    return featuresData;
                }
            }
        }
    }

    svc.getDatabricksJSONData = function(endpoint) {
        if (endpoint.testQueries && endpoint.testQueries.length) {
            return endpoint.testQueries[0].q;
        } else {
            return {
                dataframe_split: {
                    index: [0],
                    columns: ["BooleanFeature1", "CategoricalFeature", "BooleanFeature2", "NumericalFeature", "NullableFeature", "BooleanFeature3"],
                    data: [
                        [true, "value", true, 42, null, false]
                    ]
                }
            };
        }
    };

    svc.getAPIKeyToUse = function(ls) {
        if (!ls.publicAccess && ls.apiKeys.length) {
            return ls.apiKeys[0].key;
        } else {
            return null;
        }
    };

    svc.R_HANDLE_ERROR = '# Handle error if any\n\
if (response$status_code != 200) { \n\
    error_head <- paste0("Query failed (HTTP code ", response$status_code, "): "); \n\
    resp_content_type <- headers(response)$`content-type` \n\
    resp_content <- content(response) \n\
    if (resp_content_type == "application/json") { \n\
        error <- paste0(error_head, resp_content$message); \n\
    } else { \n\
        error <- paste0(error_head, resp_content); \n\
    }\n\
    stop(error); \n\
};\n\n';

    svc.JAVA_HEADER = 'import java.io.IOException;\n\
import java.nio.charset.StandardCharsets;\n\
\n\
import com.dataiku.dss.shadelib.org.apache.commons.io.IOUtils;\n\
import org.apache.http.HttpResponse;\n\
import org.apache.http.client.methods.HttpPost;\n\
import org.apache.http.entity.StringEntity;\n\
import org.apache.http.impl.client.CloseableHttpClient;\n\
import org.apache.http.impl.client.HttpClients;\n\
import org.json.JSONObject;\n\
import org.json.JSONTokener;\n\
\n\
public class DataikuTest {\n\
    public static void main(String[] args) throws Exception{\n\
        try (CloseableHttpClient httpClient = HttpClients.createDefault()) { \n\
\n';

    svc.JAVA_HANDLE_ERROR = '\
            if (resp.getStatusLine().getStatusCode() != 200) {\n\
                if (resp.getEntity().getContentType().getValue().contains("application/json")){\n\
                    JSONObject error = new JSONObject(new JSONTokener(resp.getEntity().getContent()));\n\
            \n\
                    if (error.has("detailedMessage")) {\n\
                        throw new IOException("API query failed with HTTP code " + resp.getStatusLine().getStatusCode() +" and message: " + error.getString("detailedMessage"));\n\
                    } else {\n\
                        throw new IOException("API query failed with HTTP code " + resp.getStatusLine().getStatusCode() +" and body: " + error.toString());\n\
                    }\n\
                } else {\n\
                    String s = IOUtils.toString(resp.getEntity().getContent(), StandardCharsets.UTF_8);\n\
                    throw new IOException("API query failed with HTTP code " + resp.getStatusLine().getStatusCode() +" and body: " + s);\n\
                }\n\
            }\n\
\n';

    svc.JAVA_SET_BODY_AND_RUN = '\
            StringEntity bodyEntity = new StringEntity(body.toString());\n\
            request.setEntity(bodyEntity);\n\
            request.addHeader("content-type", "application/json");\n\
    \n\
            /* Execute the query */\n\
            HttpResponse resp = httpClient.execute(request);\n\
\n';

    svc.JAVA_FOOTER = '\
        }\n\
    }\n\
}';
    function jsonToRList(jsonData) {
        let res = "list(\n";
        let i = 0;
        for (let key of Object.keys(jsonData)) {
            if (i > 0) res +=",\n";
            i++;
            let value;
            if (typeof jsonData[key] === "boolean") {
                value = jsonData[key] ? "TRUE" : "FALSE";
            } else {
                value = JSON.stringify(jsonData[key]);
            }
            res +="    " + key + " = " + value;

        }
        res += ")\n\n";
        return res
    }

    function jsonDataToPython(jsonData) {
        const prettyJSON = JSON.stringify(jsonData, undefined, 4)
            .replace(/([[\s])(true)([\s\],])/g, "$1True$3")
            .replace(/([[\s])(false)([\s\],])/g, "$1False$3")
            .replace(/([[\s])(null)([\s\],])/g, "$1None$3");
        let jsonAsPythonLines = prettyJSON.split("\n");
        if (jsonAsPythonLines.length > 2) {
            jsonAsPythonLines = jsonAsPythonLines.slice(1, jsonAsPythonLines.length - 1);
        }
        return jsonAsPythonLines;
    }

    svc.generateSampleCode = function(ls, serviceURLs, endpoint, language, hs) {
        Assert.trueish(serviceURLs.length > 0);
        let code;
        if (language == "CURL") {
            const jsonData = svc.getJSONData(endpoint);
            let prettyJSON = JSON.stringify(jsonData, undefined, 4);
            let jsonAsCURLLines = prettyJSON.split("\n");
            if (jsonAsCURLLines.length > 2) {
                jsonAsCURLLines = jsonAsCURLLines.slice(1, jsonAsCURLLines.length - 1);
            }

            code = "curl -X POST";

           switch (ls.authMethod) {
              case 'API_KEYS':
                code += ` --header 'Authorization: Bearer ${ls.apiKeys[0].key}'`
                break;
              case 'OAUTH2':
                code += " --header 'Authorization: Bearer YOUR_ACCESS_TOKEN'"
                break;
              case 'PUBLIC':
              default:
            }
            code += " \\\n";
            code += "  " + serviceURLs[0] + " \\\n";

            switch (endpoint.type) {
            case "STD_PREDICTION":
            case "STD_CAUSAL_PREDICTION":
            case "CUSTOM_PREDICTION":
            case "CUSTOM_R_PREDICTION":
            case "STD_CLUSTERING": {
                code += "  --data '{ \"features\" : {\n";
                jsonAsCURLLines.forEach(line => code += line + "\n");
                code += "  }";
                const explanationsParamsJSON = svc.getJSONExplanationsParams(endpoint);
                if (explanationsParamsJSON) {
                    code += ', "explanations": ' + JSON.stringify(explanationsParamsJSON, undefined, 4);
                }
                code += "}'";
                return {
                    mode: "shell",
                    code:code
                };
            }
            case "STD_FORECAST": {
                code += "  --data '{ \"items\" : [\n";
                jsonAsCURLLines.forEach(line => code += line + "\n");
                code += "  ]";
                code += "}'";
                return {
                    mode: "shell",
                    code:code
                };
            }
            case "PY_FUNCTION":
            case "R_FUNCTION":
            case "SQL_QUERY": {
                let entries = Object.entries(jsonData)
                        .map(([key, value]) => `    "${key}": ${JSON.stringify(value)}`)
                        .join(',\n');
                code += ` --data '{\n${entries}\n}'`;

                return {
                    mode: "shell",
                    code:code
                };
            }
            case "DATASETS_LOOKUP":
                code += "  --data '{ \"data\" : {\n";
                jsonAsCURLLines.forEach(line => code += line + "\n");
                code += "  }}'";
                return {
                    mode: "shell",
                    code:code
                };
            }
        } else if (language == "PYTHON") {
            const jsonData = svc.getJSONData(endpoint);
            const baseURI = svc.getFirstBaseURI(ls, hs);
            const ret = {
                mode: "python"
            };
            code = "";
            code += "import dataikuapi\n\n";
            code += "client = dataikuapi.APINodeClient(\"" + baseURI + "\", \""
                + (ls.deploymentBasicInfo.deployedServiceId || ls.deploymentBasicInfo.publishedServiceId) + "\"";

            switch (ls.authMethod) {
              case 'API_KEYS':
                code += `, api_key="${ls.apiKeys[0].key}"`;
                break;
              case 'OAUTH2':
                code += `, bearer_token="YOUR_ACCESS_TOKEN"`;
                break;
              case 'PUBLIC':
              default:
            }
            code += ")\n\n";

            ret.instructions = "You need to install the ``dataiku-api-client`` Python package for this to work.\n\n";
            ret.instructions += "If you are querying from within DSS, this package is already installed. Else, follow ";
            ret.instructions += "[these instructions](https://doc.dataiku.com/dss/latest/publicapi/client-python/)";

            let jsonAsPythonLines = jsonDataToPython(jsonData);

            switch (endpoint.type) {
            case "STD_PREDICTION":
            case "STD_CAUSAL_PREDICTION":
            case "CUSTOM_PREDICTION":
            case "CUSTOM_R_PREDICTION":
            case "STD_CLUSTERING": {
                code += "record_to_predict = {\n";
                jsonAsPythonLines.forEach(line => code += line + "\n");
                code += "}\n";

                code += `prediction = client.predict_record("${endpoint.id}", record_to_predict`;
                const explanationsParamsJSON = svc.getJSONExplanationsParams(endpoint);
                if (explanationsParamsJSON) {
                    if (explanationsParamsJSON.enabled) {
                        code += ", with_explanations=True";
                    } else if(explanationsParamsJSON.enabled === false) {
                        code += ", with_explanations=False";
                    }
                    if (explanationsParamsJSON.nExplanations) {
                        code += ", n_explanations="+explanationsParamsJSON.nExplanations;
                    }
                    if (explanationsParamsJSON.method) {
                        code += `, explanation_method="${explanationsParamsJSON.method}"`;
                    }
                    if (explanationsParamsJSON.nMonteCarloSteps) {
                        code += ", n_explanations_mc_steps="+explanationsParamsJSON.nMonteCarloSteps;
                    }
                }

                code += ")\n";

                code += "print(prediction[\"result\"])\n";

                ret.code = code;
                return ret;
            }
            case "STD_FORECAST": {
                code += "records_to_forecast = [\n";
                jsonAsPythonLines.forEach(line => code += line + "\n");
                code += "]\n";

                code += `forecast = client.forecast("${endpoint.id}", records_to_forecast)\n`;

                code += "print(forecast[\"results\"])\n";
                ret.code = code;
                return ret;
            }
            case "PY_FUNCTION":
            case "R_FUNCTION": {
                code += "result = client.run_function(\"" + endpoint.id + "\"";
                for (let key of Object.keys(jsonData)) {
                    code +=",\n";
                    code +="        " + key + " = " + JSON.stringify(jsonData[key]);
                }
                code += ")\n";
                code += "print(\"Function result: %s\" % result.get(\"response\"))\n";
                ret.code = code;
                return ret;
            }
            case "DATASETS_LOOKUP": {
                code += "lookup_keys = {\n";
                jsonAsPythonLines.forEach(line => code += line + "\n");
                code += "}\n";

                code += "lookup_result = client.lookup_record(\"" + endpoint.id + "\", lookup_keys)\n";
                code += "print(lookup_result[\"data\"])\n";
                ret.code = code;
                return ret;
            }
            case "SQL_QUERY": {
                code += "query_parameters = {\n";
                jsonAsPythonLines.forEach(line => code += line + "\n");
                code += "}\n";

                code += "query_result = client.sql_query(\"" + endpoint.id + "\", query_parameters)\n";
                code += "print(query_result)\n";
                ret.code = code;
                return ret;
            }
            }
        } else if (language == "R") {
            const jsonData = svc.getJSONData(endpoint);
            const baseURI = svc.getFirstBaseURI(ls, hs);
            const ret = {
                mode: "r"
            };
            code = "library(httr)\nlibrary(jsonlite)\n\n";

            ret.instructions = "You need to have the ``httr`` R package installed.\n";
            ret.instructions += "If you are querying from within DSS, this package is already installed.";
            switch (endpoint.type) {
            case "STD_PREDICTION":
            case "STD_CAUSAL_PREDICTION":
            case "CUSTOM_PREDICTION":
            case "CUSTOM_R_PREDICTION":
            case "STD_CLUSTERING": {
                const explanationsParamsJSON = this.getJSONExplanationsParams(endpoint);
                if (explanationsParamsJSON) {
                    code += "explanations_params = "+ jsonToRList(explanationsParamsJSON);
                }
                code += "record_to_predict = "+ jsonToRList(jsonData);
                code += "response <- POST(\"" + serviceURLs[0] +"\",\n";

                switch (ls.authMethod) {
                  case 'API_KEYS':
                    code += "                    authenticate(\""  + ls.apiKeys[0].key + "\", \"\"),\n";
                    break;
                  case 'OAUTH2':
                    code += "                    add_headers(Authorization = \"Bearer YOUR_ACCESS_TOKEN\"),\n";
                    break;
                  case 'PUBLIC':
                  default:
                }

                if (explanationsParamsJSON) {
                    code += "                    body = toJSON(list(features=record_to_predict, explanations=explanations_params), auto_unbox=TRUE))\n\n";
                } else {
                    code += "                    body = toJSON(list(features=record_to_predict), auto_unbox=TRUE))\n\n";
                }

                code += svc.R_HANDLE_ERROR;

                code += "result <- content(response)\n";
                code += "print(paste0(\"Prediction: \", toJSON(result$result)))\n";
                ret.code = code;
                return ret;
            }
            case "STD_FORECAST": {
                code += "records_to_predict = "+ jsonToRList(jsonData);
                code += "response <- POST(\"" + serviceURLs[0] +"\",\n";
                switch (ls.authMethod) {
                  case 'API_KEYS':
                    code += "                    authenticate(\""  + ls.apiKeys[0].key + "\", \"\"),\n";
                    break;
                  case 'OAUTH2':
                    code += "                    add_headers(Authorization = \"Bearer YOUR_ACCESS_TOKEN\"),\n";
                    break;
                  case 'PUBLIC':
                  default:
                }
                code += "                    body = toJSON(list(items=records_to_predict), auto_unbox=TRUE))\n\n";

                code += svc.R_HANDLE_ERROR;

                code += "result <- content(response)\n";
                code += "print(paste0(\"Prediction: \", toJSON(result$result)))\n";
                ret.code = code;
                return ret;
            }
            case "PY_FUNCTION":
            case "R_FUNCTION": {
                let entries = Object.entries(jsonData)
                            .map(([key, value]) => `    "${key}": ${JSON.stringify(value)}`)
                            .join(',\n');
                code += `body = '{\n${entries}\n}'\n\n`;

                code += "response <- POST(\"" + serviceURLs[0] +"\",\n";
                switch (ls.authMethod) {
                  case 'API_KEYS':
                    code += "                    authenticate(\""  + ls.apiKeys[0].key + "\", \"\"),\n";
                    break;
                  case 'OAUTH2':
                    code += "                    add_headers(Authorization = \"Bearer YOUR_ACCESS_TOKEN\"),\n";
                    break;
                  case 'PUBLIC':
                  default:
                }

                code += "                    body = body)\n\n";

                code += svc.R_HANDLE_ERROR;

                code += "result <- content(response)\n";
                code += "print(paste0(\"Function result: \", result$response))\n";
                ret.code = code;
                return ret;
            }
            case "DATASETS_LOOKUP": {
                code += "lookup_keys = list(\n";
                let i = 0;
                for (let key of Object.keys(jsonData)) {
                    if (i > 0) code +=",\n";
                    i++;
                    code +="    " + key + " = " + JSON.stringify(jsonData[key]);
                }
                code += ")\n\n";

                code += "response <- POST(\"" + serviceURLs[0] +"\",\n";

                switch (ls.authMethod) {
                  case 'API_KEYS':
                    code += "                    authenticate(\""  + ls.apiKeys[0].key + "\", \"\"),\n";
                    break;
                  case 'OAUTH2':
                    code += "                    add_headers(Authorization = \"Bearer YOUR_ACCESS_TOKEN\"),\n";
                    break;
                  case 'PUBLIC':
                  default:
                }
                code += "                    body = toJSON(list(data=lookup_keys), auto_unbox=TRUE))\n\n";

                code += svc.R_HANDLE_ERROR;

                code += "result <- content(response)\n";
                code += "print(paste0(\"Looked up: \", toJSON(result$data)))\n";
                ret.code = code;
                return ret;
            }
            case "SQL_QUERY": {
                code += "body = '{\n";
                let i = 0;
                for (let key of Object.keys(jsonData)) {
                    if (i > 0) code +=",\n";
                    i++;
                    code +="    '" + key + "' : '" + JSON.stringify(jsonData[key]) + "'";
                }
                code += "}'\n\n";

                code += "response <- POST(\"" + serviceURLs[0] +"\",\n";
                switch (ls.authMethod) {
                  case 'API_KEYS':
                    code += "                    authenticate(\""  + ls.apiKeys[0].key + "\", \"\"),\n";
                    break;
                  case 'OAUTH2':
                    code += "                    add_headers(Authorization = \"Bearer YOUR_ACCESS_TOKEN\"),\n";
                    break;
                  case 'PUBLIC':
                  default:
                }

                code += "                    body = body)\n\n";

                code += svc.R_HANDLE_ERROR;

                code += "result <- content(response)\n";
                code += "print(paste0(\"Query results: \", result))\n";
                ret.code = code;
                return ret;
            }
            }
        } else if (language == "JAVA") {
            const jsonData = svc.getJSONData(endpoint);
            const baseURI = svc.getFirstBaseURI(ls, hs);
            const ret = {
                mode: "javascript" // Acceptable substitute for Java
            };
            code = svc.JAVA_HEADER;

            code += "            HttpPost request = new HttpPost(\"" + serviceURLs[0] + "\");\n\n";

            switch (ls.authMethod) {
              case 'API_KEYS':
                code += "            /* Handle authentication */\n";
                code += "            String headerValue = \"" + ls.apiKeys[0].key + ":\";\n";
                code += '            String encodedHeaderValue = java.util.Base64.getEncoder().encodeToString(headerValue.getBytes(StandardCharsets.UTF_8));\n';
                code += '            request.addHeader("Authorization", "Basic "  + encodedHeaderValue);\n';
                break;
              case 'OAUTH2':
                code += "            /* Handle authentication */\n";
                code += "            /* Retrieve an access token to the authorization service */\n";
                code += "            String accessToken = \"YOUR_ACCESS_TOKEN\";\n";
                code += '            request.addHeader("Authorization", "Bearer " + accessToken);\n';
                break;
              case 'PUBLIC':
              default:
            }

            function jsonDataToJSONObject(objName, json) {
                code += "            JSONObject " + objName + " = new JSONObject();\n";
                for (let key of Object.keys(json)) {
                    code +="            "+ objName + ".put(\"" + key + "\", " + JSON.stringify(json[key]) + ");\n";
                }
                code += "\n";
            }

            ret.instructions = "You need to have the ``httpclient`` 4.3 or higher and ``org.json`` packages installed.\n";
            code += "            /* Build the JSON POST Body */\n";

            switch (endpoint.type) {
            case "STD_PREDICTION":
            case "STD_CAUSAL_PREDICTION":
            case "CUSTOM_PREDICTION":
            case "CUSTOM_R_PREDICTION":
            case "STD_CLUSTERING": {
                jsonDataToJSONObject("recordToPredict", jsonData);
                const explanationsParamsJSON = svc.getJSONExplanationsParams(endpoint)
                if (explanationsParamsJSON) {
                    jsonDataToJSONObject("explanationsParams", explanationsParamsJSON)
                }
                code += '            JSONObject body = new JSONObject();\n';
                code += '            body.put("features", recordToPredict);\n';
                if (explanationsParamsJSON) {
                    code += '            body.put("explanations", explanationsParams);\n';
                }
                code += svc.JAVA_SET_BODY_AND_RUN;
                code += svc.JAVA_HANDLE_ERROR;
                code += '            /* Parse result */\n';
                code += '            JSONObject result = new JSONObject(new JSONTokener(resp.getEntity().getContent()));\n';
                code += '            System.out.println("Model returned:\\n" + result + "\\n");\n';
                code += svc.JAVA_FOOTER;
                ret.code = code;
                return ret;
            }
            case "STD_FORECAST": {
                code += "            JSONArray " + "recordsToPredict" + " = new JSONArray();\n";
                for (const item of jsonData) {
                    code +="            "+ "recordsToPredict" + ".add(" + JSON.stringify(item) + ");\n";
                }
                code += "\n";
                code += '            JSONObject body = new JSONObject();\n';
                code += '            body.put("items", recordsToPredict);\n';
                code += svc.JAVA_SET_BODY_AND_RUN;
                code += svc.JAVA_HANDLE_ERROR;
                code += '            /* Parse result */\n';
                code += '            JSONObject result = new JSONObject(new JSONTokener(resp.getEntity().getContent()));\n';
                code += '            System.out.println("Model returned:\\n" + result + "\\n");\n';
                code += svc.JAVA_FOOTER;
                ret.code = code;
                return ret;
            }
            case "PY_FUNCTION":
            case "R_FUNCTION":
                jsonDataToJSONObject("body", jsonData);
                code += svc.JAVA_SET_BODY_AND_RUN;
                code += svc.JAVA_HANDLE_ERROR;
                code += '            /* Parse result */\n';
                code += '            JSONObject result = new JSONObject(new JSONTokener(resp.getEntity().getContent()));\n';
                code += '            System.out.println("Function returned:\\n" + result + "\\n");\n';
                code += svc.JAVA_FOOTER;
                ret.code = code;
                return ret;
            case "DATASETS_LOOKUP": {
                jsonDataToJSONObject("lookupKeys", jsonData);
                code += '            JSONObject body = new JSONObject();\n';
                code += '            body.put("data", lookupKeys);\n';
                code += svc.JAVA_SET_BODY_AND_RUN;
                code += svc.JAVA_HANDLE_ERROR;
                code += '            /* Parse result */\n';
                code += '            JSONObject result = new JSONObject(new JSONTokener(resp.getEntity().getContent()));\n';
                code += '            System.out.println("Lookup returned:\\n" + result + "\\n");\n';
                code += svc.JAVA_FOOTER;
                ret.code = code;
                return ret;
            }
            case "SQL_QUERY":
                jsonDataToJSONObject("body", jsonData);
                code += svc.JAVA_SET_BODY_AND_RUN;
                code += svc.JAVA_HANDLE_ERROR;
                code += '            /* Parse result */\n';
                code += '            JSONObject result = new JSONObject(new JSONTokener(resp.getEntity().getContent()));\n';
                code += '            System.out.println("Query returned:\\n" + result + "\\n");\n';
                code += svc.JAVA_FOOTER;
                ret.code = code;
                return ret;
            }
        }
        return code;
    };

    svc.generateDatabricksSampleCode = function(language, endpoint, dbxHostBaseUrl, dbxEndpointName) {
        const endpointName = dbxEndpointName || "<insert Databricks serving endpoint name>";
        const hostBaseUrl = dbxHostBaseUrl || "<insert Databricks host base URL>";

        if (language === "PYTHON") {
            switch (endpoint.type) {
                case "STD_PREDICTION": {
                    break;
                }
                case "STD_CLUSTERING":
                case "STD_CAUSAL_PREDICTION":
                case "CUSTOM_PREDICTION":
                case "PY_FUNCTION":
                case "CUSTOM_R_PREDICTION":
                case "STD_FORECAST":
                case "R_FUNCTION":
                case "SQL_QUERY":
                case "DATASETS_LOOKUP": {
                    return {
                        code: "unsupported"
                    };
                }
            }

            const jsonData = svc.getDatabricksJSONData(endpoint);
            const queryContent = jsonDataToPython(jsonData).join("\n");

            const instructions = [
                `You need to install the <a href="https://pypi.org/project/requests/" target="_blank">requests</a> Python package for this snippet to work.`,
                "",
                "You can find more information about Databricks REST API for invoking serving endpoints " +
                `<a href="https://docs.databricks.com/api/workspace/servingendpoints/query" target="_blank">here</a>.`
            ].join("\n");

            const code = [
                `import requests`,
                ``,
                `token = "<insert Databricks token>"`,
                `endpoint_name = "${endpointName}"`,
                `host_base_url = "${hostBaseUrl}"`,
                `url = f"https://{host_base_url}/serving-endpoints/{endpoint_name}/invocations"`,
                ``,
                `query = {`,
                queryContent,
                `}`,
                ``,
                `response = requests.post(`,
                `    url,`,
                `    auth=('token', token),`,
                `    json=query`,
                `)`,
                ``,
                `print(response.json())`,
                ``
            ].join("\n");

            return {
                mode: "python",
                instructions: instructions,
                code: code
            };
        }
    };

    svc.generateSageMakerSampleCode = function(language, endpoint, awsRegion, sageMakerEndpointName) {
        const endpointName = sageMakerEndpointName || "<insert endpoint name here>";
        let code;
        if (language === "PYTHON") {
            const jsonData = svc.getAzureMLOrSageMakerJSONData(endpoint);
            const ret = {
                mode: "python"
            };

            ret.instructions = "You need to install the [boto3](https://pypi.org/project/boto3/) Python package for this to work.\n\n"
            ret.instructions += "You can find more information about ``invoke_endpoint`` [here](https://boto3.amazonaws.com/v1/documentation/api/latest/reference/services/sagemaker-runtime/client/invoke_endpoint.html)."

            code = "";
            code += "import boto3\n";
            code += "import json\n\n";
            code += "endpointName = \"" + endpointName + "\"\n\n";

            let body = jsonData;

            let responseKey = "";

            switch (endpoint.type) {
                case "STD_PREDICTION":
                case "STD_CLUSTERING":
                case "STD_CAUSAL_PREDICTION":
                case "CUSTOM_PREDICTION": {
                    const explanationsParamsJSON = svc.getJSONExplanationsParams(endpoint);
                    if (explanationsParamsJSON) {
                        body.explanations = explanationsParamsJSON;
                    }

                    responseKey = "results";
                    break;
                }
                case "PY_FUNCTION": {
                    responseKey = "response";
                    break;
                }
                case "CUSTOM_R_PREDICTION":
                case "STD_FORECAST":
                case "R_FUNCTION":
                case "SQL_QUERY":
                case "DATASETS_LOOKUP": {
                    // TODO: Still not supported in SageMaker deployments
                    ret.code = "Unsupported"
                    return ret;
                }
            }

            let jsonAsPythonLines = jsonDataToPython(body);

            code += "query = {\n";
            jsonAsPythonLines.forEach(line => code += line + "\n");
            code += "}\n\n";

            code += "runtime = boto3.Session().client('sagemaker-runtime'"
            if (awsRegion) {
                code += ", region_name='" + awsRegion;
            }
            code += "')\n\n";

            code += "# Send JSON query via InvokeEndpoint API\n";
            code += "response = runtime.invoke_endpoint(\n";
            code += "    EndpointName=endpointName,\n";
            code += "    ContentType='application/json',\n";
            code += "    Accept='application/json',\n";
            code += "    Body=json.dumps(query)\n";
            code += ")\n\n";
            code += "# Unpack response\n";
            code += "results = json.loads(response['Body'].read().decode())\n\n";
            code += "print(results[\"" + responseKey + "\"])\n";

            ret.code = code;

            return ret;
        }
    }

    svc.generateSnowparkSampleCode = function(language, endpoint, udfName, serviceURL) {
        const functionName = udfName || "<insert function name here>";
        const endpointURL = serviceURL || "<insert endpoint URL here>";
        let code;
        if (language === "SQL") {
            const jsonData = svc.getSnowparkJSONData(endpoint);
            const ret = {
                mode: "sql"
            };

            ret.instructions = "You need to install the [SnowSQL](https://docs.snowflake.com/en/user-guide/snowsql) CLI client for this to work.\n\n";

            let body = jsonData;

            code = "";
            code += "SELECT \"" + functionName + "\"('";
            code += JSON.stringify(body);
            code += "');\n";

            ret.code = code;

            return ret;
        } else if (language === "PYTHON") {
            const jsonData = svc.getSnowparkJSONData(endpoint);
            const ret = {
                mode: "python"
            };

            ret.instructions = "You need to install the [Snowflake Connector](https://docs.snowflake.com/developer-guide/python-connector/python-connector/) Python package for this to work. This will only work if the Snowpark endpoint is public.";

            code = "";
            code += "import snowflake.connector\n";
            code += "import requests\n\n";

            code += "url = \"" + endpointURL + "\"\n\n";

            code += "# Obtain a session token.\n";
            code += "ctx = snowflake.connector.connect()  # See https://docs.snowflake.com/en/developer-guide/python-connector/python-connector-connect for more options to connect to Snowflake\n";
            code += "token_data = ctx._rest._token_request('ISSUE')\n";
            code += "token_extract = token_data['data']['sessionToken']\n\n";

            const body = jsonData;

            let jsonAsPythonLines = jsonDataToPython(body);

            code += "# Perform the request.\n";
            code += "headers = {'Authorization': f'Snowflake Token=\\\"{token_extract}\\\"'}\n";
            code += "query = {\n";
            jsonAsPythonLines.forEach(line => code += line + "\n");
            code += "}\n";

            code += "response = requests.post(url, json=query, headers=headers)\n";
            code += "print(response.json())\n";

            ret.code = code;

            return ret;
        }
    }

    svc.generateAzureMLSampleCode = function(language, endpoint, azSubscription, azResourceGroup, azWorkspace, azureMLEndpointName) {
        const endpointName = azureMLEndpointName || "<insert endpoint name here>";
        const subscriptionId = azSubscription || "<insert subscription id here>";
        const resourceGroup = azResourceGroup || "<insert resource group name here>";
        const workspace = azWorkspace || "<insert workspace name here>";
        let code;
        if (language === "PYTHON") {
            const jsonData = svc.getAzureMLOrSageMakerJSONData(endpoint);
            const ret = {
                mode: "python"
            };

            ret.instructions = "You need to install the [azure-ai-ml](https://pypi.org/project/azure-ai-ml/) and [azure-identity](https://pypi.org/project/azure-identity/) Python packages for this to work.\n\n"

            code = "";
            code += "import json\n";
            code += "import tempfile\n\n";
            code += "from azure.ai.ml import MLClient\n";
            code += "from azure.identity import DefaultAzureCredential\n\n";
            code += "endpoint_name = \"" + endpointName + "\"\n\n";
            code += "subscription_id = \"" + subscriptionId + "\"\n";
            code += "resource_group = \"" + resourceGroup + "\"\n";
            code += "workspace = \"" + workspace + "\"\n\n";

            let body = jsonData;

            let responseKey = "";

            switch (endpoint.type) {
                case "STD_PREDICTION":
                case "STD_CLUSTERING":
                case "STD_CAUSAL_PREDICTION":
                case "CUSTOM_PREDICTION": {
                    const explanationsParamsJSON = svc.getJSONExplanationsParams(endpoint);
                    if (explanationsParamsJSON) {
                        body.explanations = explanationsParamsJSON;
                    }

                    responseKey = "results";
                    break;
                }
                case "PY_FUNCTION": {
                    responseKey = "response";
                    break;
                }
                case "CUSTOM_R_PREDICTION":
                case "STD_FORECAST":
                case "R_FUNCTION":
                case "SQL_QUERY":
                case "DATASETS_LOOKUP": {
                    // TODO: Still not supported in SageMaker deployments
                    ret.code = "Unsupported"
                    return ret;
                }
            }

            let jsonAsPythonLines = jsonDataToPython(body);

            code += "query = {\n";
            jsonAsPythonLines.forEach(line => code += line + "\n");
            code += "}\n\n";

            code += "ml_client = MLClient(DefaultAzureCredential(), subscription_id, resource_group, workspace)\n\n"
            code += "with tempfile.NamedTemporaryFile(mode='w') as tmp:\n";
            code += "    json.dump(query, tmp)\n";
            code += "    tmp.flush()\n\n";
            code += "    response = ml_client.online_endpoints.invoke(endpoint_name=endpoint_name, request_file=tmp.name)\n";
            code += "    results = json.loads(response)\n\n";
            code += "    print(results[\"" + responseKey + "\"])\n";

            ret.code = code;

            return ret;
        }
    }

    svc.generateVertexAISampleCode = function(language, endpoint, gcpRegion, gcpProject, vertexAIEndpointId) {
        const endpointId = vertexAIEndpointId || "<insert endpoint id here>";
        const region = gcpRegion || "<insert region here>";
        const project = gcpProject || "<insert project here>";
        let code;
        if (language === "PYTHON") {
            const jsonData = svc.getVertexAIJSONData(endpoint);
            const ret = {
                mode: "python"
            };

            ret.instructions = "You need to install the [google-cloud-aiplatform](https://pypi.org/project/google-cloud-aiplatform/) Python package for this to work.\n\n"

            code = "";
            code += "from google.cloud import aiplatform\n\n";
            code += "endpoint_id = \"" + endpointId + "\"\n";
            code += "location = \"" + region + "\"\n";
            code += "project = \"" + project + "\"\n\n";

            const body = jsonData;

            switch (endpoint.type) {
                case "STD_PREDICTION":
                case "STD_CLUSTERING":
                case "STD_CAUSAL_PREDICTION":
                case "CUSTOM_PREDICTION":
                case "PY_FUNCTION": {
                    break;
                }
                case "CUSTOM_R_PREDICTION":
                case "STD_FORECAST":
                case "R_FUNCTION":
                case "SQL_QUERY":
                case "DATASETS_LOOKUP": {
                    // TODO: Still not supported in Vertex AI deployments
                    ret.code = "Unsupported";
                    return ret;
                }
            }

            let jsonAsPythonLines = jsonDataToPython(body);

            code += "query = {\n";
            jsonAsPythonLines.forEach(line => code += line + "\n");
            code += "}\n";
            code += "instances = query[\"instances\"]\n\n";

            code += "aiplatform.init(project=project, location=location)\n";
            code += "endpoint = aiplatform.Endpoint(endpoint_id)\n\n";
            code += "predictions = endpoint.predict(instances=instances)\n";
            code += "print(predictions)\n";

            ret.code = code;

            return ret;
        }
    }

    return svc;
})

app.constant('GENERATION_MAPPING_STRATEGIES', [
    {id: 'RANDOM', name: 'Random'},
    {id: 'HASH_BASED', 'name': 'Hash based'}
]);


app.constant('GENERATION_MAPPING_MODES', [
    {id: 'SINGLE_GENERATION', name: 'Single generation'},
    {id: 'MULTI_GENERATION', 'name': 'Multiple generations'}
]);

app.filter("deploymentToGenerationList", function(DeployerUtils) {
    return function(deploymentBasicInfo) {
        if (!deploymentBasicInfo) {
            return;
        }
        return DeployerUtils.getParticipatingVersions(deploymentBasicInfo).join(', ');
    };
});


app.factory('ApiDeploymentProgress', function(APIDeploymentInfraHelper, ApiDeploymentSyncHelper) {
    const svc = {
        DEPLOY_STATUS: {
            SUCCEEDED: 'SUCCEEDED',
            SUCCEEDED_WITH_WARNING: 'SUCCEEDED_WITH_WARNING',
            HOOK_FAILED: 'HOOK_FAILED',
            FAILED: 'DEPLOY_FAILED',
            IN_PROGRESS: 'IN_PROGRESS',
            PREPARE_FAILED: 'PREPARE_FAILED',
        },
        initDeployment: function() {
            return {
                status: svc.DEPLOY_STATUS.IN_PROGRESS,
                stateLabels: null,
                futureResponse: null,
                prepareResult: null,
                deployResult: null,
                deploymentHookExecutionStatus: null,
                failureLog: null,
                messages: null,
            };
        },
        staticDeploymentPrepareFailed: function(deployment, deploymentType) {
            return APIDeploymentInfraHelper.isStaticInfra(deploymentType) && deployment.status === svc.DEPLOY_STATUS.PREPARE_FAILED;
        },
        staticDeploymentSyncFailed: function(deployment, deploymentType) {
            return APIDeploymentInfraHelper.isStaticInfra(deploymentType) && svc.deploymentFailed(deployment);
        },
        deploymentFailed: function(deployment) {
            return deployment.status === svc.DEPLOY_STATUS.FAILED || deployment.status === svc.DEPLOY_STATUS.HOOK_FAILED;
        },
        updateDeploymentStatus: function(report, deployment) {
            deployment.deploymentHookExecutionStatus = report.deploymentHookExecutionStatus;
            deployment.messages = report.messages;
            if (ApiDeploymentSyncHelper.isSuccessful(report)) {
                deployment.deployResult = report.syncReport;
                deployment.status = svc.DEPLOY_STATUS.SUCCEEDED;
            } else if (ApiDeploymentSyncHelper.isSuccessfulWithWarning(report)) {
                deployment.deployResult = report.syncReport;
                deployment.status = svc.DEPLOY_STATUS.SUCCEEDED_WITH_WARNING;
            } else if (ApiDeploymentSyncHelper.prepareFailed(report)) { // for now, this can be true only for static deployments
                deployment.prepareResult = report.prepareReport;
                deployment.status = svc.DEPLOY_STATUS.PREPARE_FAILED;
            } else if (ApiDeploymentSyncHelper.isError(report)) {
                deployment.deployResult = report.syncReport;
                if (report.deploymentHookExecutionStatus?.preHookFailed) {
                    deployment.status = svc.DEPLOY_STATUS.HOOK_FAILED;
                } else {
                    deployment.status = svc.DEPLOY_STATUS.FAILED;
                }
            }
        }
    }
    return svc;
});


app.component('apiDeploymentModal', {
    bindings: {
        modalControl: '<',
        deploymentId: '<',
        jobId: '<',
        lastDeploymentAction: '<',
        insufficientPermissionsMessage: '<',
        deploymentType: '<',
        peekingUpdateStarted: '<',
        peekingUpdateEnded: '<',
        deploy: '<',
    },
    templateUrl: '/templates/api-deployer/deploy-modal.html',
    controller: function($scope, $interval, ApiDeploymentProgress, ApiDeploymentSyncHelper, DataikuAPI, ProgressStackMessageBuilder) {
        const $ctrl = this;

        $ctrl.modalTitle = 'Deploying';
        $ctrl.deployment = ApiDeploymentProgress.initDeployment();
        $ctrl.percentage = 0;
        $ctrl.deploymentHookExecutionStatus = null;
        $ctrl.peekDeploymentActionRepeat = null;

        $ctrl.$onInit = function() {
            if ($ctrl.jobId || $ctrl.lastDeploymentAction && $ctrl.lastDeploymentAction.inProgress) {
                $ctrl.peekingUpdateStarted();
                $ctrl.peekDeploymentActionRepeat = $interval(peekDeploymentAction, 1000);
                peekDeploymentAction();
            } else if ($ctrl.lastDeploymentAction) {
                onComplete($ctrl.lastDeploymentAction?.report);
            }
        }

        $ctrl.$onDestroy = function() {
            cancelPeekDeploymentAction();
        }

        $ctrl.abort = function() {
            if ($ctrl.deploymentId) {
                DataikuAPI.apideployer.deployments.abortDeploymentAction($ctrl.deploymentId).error(setErrorInScope.bind($scope));
            }
        }

        $ctrl.close = function() {
            $ctrl.modalControl.dismiss();
        }

        $ctrl.deployAnyway = function() {
            $ctrl.deploy(false);
            $ctrl.close();
        }

        $ctrl.retryDeploy = function() {
            $ctrl.deploy();
            $ctrl.close();
        }

        $ctrl.closeButtonLabel = function() {
            switch ($ctrl.deployment.status) {
                case ApiDeploymentProgress.DEPLOY_STATUS.PREPARE_FAILED:
                    return "Abort";
                case ApiDeploymentProgress.DEPLOY_STATUS.FAILED:
                case ApiDeploymentProgress.DEPLOY_STATUS.INTERRUPTED:
                case ApiDeploymentProgress.DEPLOY_STATUS.HOOK_FAILED:
                    return "Cancel";
                case ApiDeploymentProgress.DEPLOY_STATUS.SUCCEEDED_WITH_WARNING:
                case ApiDeploymentProgress.DEPLOY_STATUS.SUCCEEDED:
                case ApiDeploymentProgress.DEPLOY_STATUS.IN_PROGRESS:
                    return "Close";
            }
        }

        $ctrl.staticDeploymentPrepareFailed = function() {
            return ApiDeploymentProgress.staticDeploymentPrepareFailed($ctrl.deployment, $ctrl.deploymentType);
        }

        $ctrl.staticDeploymentSyncFailed = function() {
            return ApiDeploymentProgress.staticDeploymentSyncFailed($ctrl.deployment, $ctrl.deploymentType);
        }

        $ctrl.deploymentFailed = function() {
            return ApiDeploymentProgress.deploymentFailed($ctrl.deployment);
        }

        $ctrl.deploymentInProgress = function() {
            return $ctrl.deployment.status === ApiDeploymentProgress.DEPLOY_STATUS.IN_PROGRESS;
        }

        $ctrl.deploymentSucceeded = function() {
            return $ctrl.deployment.status === ApiDeploymentProgress.DEPLOY_STATUS.SUCCEEDED_WITH_WARNING || $ctrl.deployment.status === ApiDeploymentProgress.DEPLOY_STATUS.SUCCEEDED;
        }

        function peekDeploymentAction() {
            DataikuAPI.apideployer.deployments.peekDeploymentActionProgress($ctrl.jobId, $ctrl.deploymentId)
                .then(response => {
                    if (response?.data.hasResult || response?.data.unknown) {
                        cancelPeekDeploymentAction();
                        onComplete(response.data.result);
                    } else {
                        onUpdate(response);
                    }
                })
                .catch((error) => {
                    cancelPeekDeploymentAction();
                    onError(error);
                });
        }

        function updateModalTitle(report) {
            if (ApiDeploymentSyncHelper.isSuccessful(report)) {
                $ctrl.modalTitle = "Activation of the new version(s) succeeded";
            } else if (ApiDeploymentSyncHelper.isSuccessfulWithWarning(report)) {
                $ctrl.modalTitle = "Activation of the new version(s) succeeded with warning(s)";
            } else if (ApiDeploymentSyncHelper.prepareFailed(report)) { // for now, this can be true only for static deployments
                if (report.interrupted) {
                    $ctrl.modalTitle = "Pre-sync of packages interrupted";
                } else {
                    $ctrl.modalTitle = "Pre-sync of packages failed";
                }
            } else if (ApiDeploymentSyncHelper.isError(report)) {
                if (report.deploymentHookExecutionStatus?.preHookFailed) {
                    $ctrl.modalTitle = report.interrupted ? "Deployment interrupted" : "A deployment hook failed";
                } else {
                    $ctrl.modalTitle = report.interrupted ? "Deployment interrupted" : "Deployment activation failed";
                }
            }
        }

        function cancelPeekDeploymentAction() {
            if ($ctrl.peekDeploymentActionRepeat) {
                $ctrl.peekingUpdateEnded();
                $interval.cancel($ctrl.peekDeploymentActionRepeat);
            }
            $ctrl.peekDeploymentActionRepeat = null;
        }

        function onUpdate(response) {
            $ctrl.deployment.futureResponse = response.data;
            if (response.data) {
                $ctrl.percentage = ProgressStackMessageBuilder.getPercentage(response.data.progress);
                $ctrl.deployment.stateLabels = ProgressStackMessageBuilder.build(response.data.progress, true);
                if (response.data.progress) {
                    const report = response.data.progress.report;
                    $ctrl.deployment.deploymentHookExecutionStatus = report ? report.deploymentHookExecutionStatus : null;
                }
            }
        }

        function onComplete(report) {
            $ctrl.deployment.futureResponse = null;
            if (report) {
                ApiDeploymentProgress.updateDeploymentStatus(report, $ctrl.deployment);
                updateModalTitle(report);
            } else {
                // we may no longer have the result of the last deployment action
                $ctrl.close();
            }
        }

        function onError(error) {
            $ctrl.deployment.futureResponse = null;
            $ctrl.deployment.failureLog = error.logTail;
            setErrorInScope.bind($scope)(error);
        }
    }
});

app.component('apiDeploymentDeletionDeleteServiceModal', {
    bindings: {
        modalControl: '<'
    },
    templateUrl: '/templates/api-deployer/deployment-deletion-delete-service-modal.html',
    controller: function() {
        const $ctrl = this;
        $ctrl.disableDeployment = true;

        $ctrl.confirm = function() {
            $ctrl.modalControl.resolve($ctrl.disableDeployment);
        };

        $ctrl.cancel = function() {
            $ctrl.modalControl.dismiss();
        };
    }
});

})();
