(function() {
    'use strict';

    var app = angular.module('dataiku.flow.project');

    /**
     * Preview function in main flow
     */
    app.directive('flowPreview', function($rootScope, $timeout, $stateParams, translate, FlowGraphSelection, Logger) {
        return {
            restrict: 'AE',
            scope: true,
            templateUrl: '/templates/flow-editor/flow-preview.html',
            controller: function($scope, WT1) {
                $scope.previewContextProjectKey = $stateParams.projectKey;
                updatePreview(); // initialize the preview when it got added to the DOM
                const deregisterFlowSelectionUpdated = $rootScope.$on('flowSelectionUpdated', () => {
                    const selectedNodes = FlowGraphSelection.getSelectedNodes();
                    if (selectedNodes && selectedNodes.length === 1 && selectedNodes[0]
                        && $scope.previewProjectKey && $scope.previewDatasetName
                        && selectedNodes[0].projectKey === $scope.previewProjectKey && selectedNodes[0].name === $scope.previewDatasetName) {
                        return; // Selection has not actually changed. Do nothing.
                    }
                    // Disable the preview before triggering the preview update to clear the view
                    // before the loading spinner
                    $scope.showPreviewTable = false;
                    // async execution allowing the view to refresh before updating the preview
                    // Use $timeout because setTimeout happens outside of Angular
                    // Using $timeout it will handle $apply for you to kick off a $digest cycle to refresh the view based on dirtiness check
                    $timeout(updatePreview);
                }); // subsequent changes
                $scope.$on('$destroy', deregisterFlowSelectionUpdated);

                // When the toggle is pressed then the flag gets updated. Watching `hasPreview` is the same as watching
                // the toggle button click event.
                // We need to update the preview whenever the toggle button is clicked and it opens up the preview pane.
                $scope.$watch('hasPreview', updatePreview);
                $scope.reloadPreview = () => {
                    $scope.$broadcast('refresh-preview-table-without-cache');
                }

                function updatePreview() {
                    // Don't update the preview when the preview pane is not active. It will affect the performance.
                    if (!$scope.hasPreview) return;
                    const selectedNodes = FlowGraphSelection.getSelectedNodes();
                    let problem, solution;
                    if (selectedNodes.length === 0) {
                        problem = translate("PROJECT.FLOW.PREVIEW.NO_DATESET_SELECTED", "No dataset selected");
                        solution = translate("PROJECT.FLOW.PREVIEW.CLICK_DATASET_TO_PREVIEW", "Click a dataset to preview its content");
                    } else if (selectedNodes.length > 1) {
                        problem = translate("PROJECT.FLOW.PREVIEW.MULTIPLE_ITEMS_SELECTED", "Multiple items selected");
                        solution = translate("PROJECT.FLOW.PREVIEW.CLICK_SINGLE_DATASET_TO_PREVIEW", "Click a single dataset to preview its content");
                    } else if (selectedNodes[0].nodeType && !selectedNodes[0].nodeType.includes('DATASET')) {
                        problem = translate("PROJECT.FLOW.PREVIEW.NOT_DATASET_SELECTED", "Selected item is not a dataset");
                        solution = translate("PROJECT.FLOW.PREVIEW.CLICK_DATASET_TO_PREVIEW", "Click a dataset to preview its content");
                    } else if (selectedNodes[0].datasetType === 'hiveserver2') {
                        problem = translate("PROJECT.FLOW.PREVIEW.HIVE_NOT_SUPPORTED", "Preview is not supported for Hive tables");
                        solution = "";
                    } else if (selectedNodes[0].neverBuilt) {
                        problem = translate("PROJECT.FLOW.PREVIEW.DATASET_NOT_BUILT", "Selected dataset is not built");
                        solution = translate("PROJECT.FLOW.PREVIEW.CLICK_BUILT_DATASET_TO_PREVIEW", "Click a built dataset to preview its content");
                    }

                    if (!problem && !solution) {
                        // All good, trigger the actual load of the preview data
                        // This will trigger ctrl.$onChanges of datasetPreviewTable since it updates the props of the component
                        $scope.previewProjectKey = selectedNodes[0].projectKey;
                        $scope.previewDatasetName = selectedNodes[0].name;
                        // Show the preview table now that everything is OK.
                        $scope.showPreviewTable = true;
                        $scope.previewProblem = null;
                        $scope.previewSolution = null;
                    
                        try {
                            // A quick check to make sure that the WT1 event won't break the app
                            if (selectedNodes[0]) {
                                WT1.event('flow-preview-update',
                                    {
                                        nodeType: selectedNodes[0].nodeType,
                                        datasetType: selectedNodes[0].datasetType,
                                    }
                                );
                            }
                        } catch (e) {
                            Logger.error('Failed to report flow dataset preview event', e);
                        }
                    } else {
                        $scope.previewProjectKey = null;
                        $scope.previewDatasetName = null;
                        $scope.previewProblem = problem;
                        $scope.previewSolution = solution;
                    }
                }
            }
        };
    });
})();