(function() {
'use strict';

const app = angular.module('dataiku.recipes');

app.service("AgentEvaluationMetrics", function() {
    const svc = {};

    svc.metrics = [
        {
            label: "Average number of tool calls",
            name: "averageToolExecutionsPerRow",
            shortDescription: "The number of tool calls made by the Agent.",
            labelRowByRow: "Average number of tool calls",
            nameRowByRow: "totalToolExecutions",
            globalDescription: "The number of tool calls made by the Agent. Average over all rows.",
            selectable: false,
            excludedFromSelectAll: false,
        },
        {
            label: "Average number of failed tool calls",
            name: "averageFailedToolExecutionsPerRow",
            shortDescription: "The number of tool calls made by the Agent that failed on execution.",
            labelRowByRow: "Average number of failed tool calls",
            nameRowByRow: "totalFailedToolExecutions",
            globalDescription: "The number of tool calls made by the Agent that failed on execution. Average over all rows.",
            selectable: false,
            excludedFromSelectAll: false,
        },
        {
            label: "Average tool call execution time",
            name: "averageToolExecutionTimeSecondsPerRow",
            shortDescription: "Time spent executing all the tools, in seconds.",
            labelRowByRow: "Average tool call execution time",
            nameRowByRow: "totalToolExecutionTimeSeconds",
            globalDescription: "Time spent executing all the tools, in ms. Average over all rows.",
            selectable: false,
            excludedFromSelectAll: false,
        },
        {
            label: "Total execution time",
            name: "p95TotalAgentCallExecutionTimeSecondsPerRow",
            shortDescription: "Total time taken by the agent call, in seconds",
            labelRowByRow: "Total agent call execution time",
            nameRowByRow: "totalAgentCallExecutionTimeSeconds",
            globalDescription: "Total time taken by the agent execution, in seconds. P95 across all rows.",
            selectable: false,
            excludedFromSelectAll: false,
        },
        {
            label: "Sample row count",
            name: "sampleRowCount",
            shortDescription: "Total number of rows in the sample. May differ from the number of rows used for metric evaluation as some metrics require specific fields that might be missing in certain rows.",
            labelRowByRow: "Sample row count",
            nameRowByRow: "sampleRowCount",
            globalDescription: "Total number of rows in the sample. May differ from the number of rows used for metric evaluation as some metrics require specific fields that might be missing in certain rows.",
            selectable: false,
            excludedFromSelectAll: false,
        },
        {
            label: "Exact match",
            name: "toolCallExactMatch",
            shortDescription: "Whether the agent executes exactly as expected: Check whether all of the agent’s tool calls happen in the exact same order as the reference. Returns 0 if false, 1 if true.",
            labelRowByRow: "Median of Tool Call Exact match",
            nameRowByRow: "toolCallExactMatch",
            globalDescription: "Whether the agent executes exactly as expected: Check whether all of the agent’s tool calls happen in the exact same order as the reference. Returns 0 if false, 1 if true. Median over all rows.",
            selectable: true,
            excludedFromSelectAll: false,
        },
        {
            label: "Partial match",
            name: "toolCallPartialMatch",
            shortDescription: "How well the agent executes calls in order: Proportion of ordered tool calls that are shared between the agent’s response and the reference. Returns a float between 0 (no match) and 1 (exact match).",
            labelRowByRow: "Median of Tool Call Partial match",
            nameRowByRow: "toolCallPartialMatch",
            globalDescription: "How well the agent executes calls in order: Proportion of ordered tool calls that are shared between the agent’s response and the reference. Returns a float between 0 (no match) and 1 (exact match). Median over all rows.",
            selectable: true,
            excludedFromSelectAll: false,
        },
        {
            label: "Precision / Recall / F1",
            name: "toolCallPrecisionRecallF1",
            shortDescription: "Evaluates the agent’s tool call performance using precision, recall, and F1 Score. Two separate calls to the same tool are considered distinct.",
            nameRowByRow: "toolCallPrecisionRecallF1",
            globalDescription: "Evaluates the agent’s tool call performance using precision, recall, and F1 Score. Two separate calls to the same tool are considered distinct. Median over all rows.",
            selectable: true,
            excludedFromSelectAll: false,
        },
        {
            label: "Agent Goal Accuracy with Reference",
            name: "agentGoalAccuracyWithReference",
            shortDescription: "Evaluate the performance of the LLM in identifying and achieving the goals of the user. Output 1 if the AI has achieved the goal and 0 if the AI has not achieved the goal. Uses Ground truth.",
            globalDescription: "Evaluate the performance of the LLM in identifying and achieving the goals of the user. Output 1 if the AI has achieved the goal and 0 if the AI has not achieved the goal. Average over all rows.",
            labelRowByRow: "Agent Goal Accuracy with Reference",
            nameRowByRow: "agentGoalAccuracyWithReference",
            selectable: true,
            excludedFromSelectAll: false,
            pairedWith: "agentGoalAccuracyWithoutReference"
        },
        {
            label: "Agent Goal Accuracy without Reference",
            name: "agentGoalAccuracyWithoutReference",
            shortDescription: "Evaluate the performance of the LLM in identifying and achieving the goals of the user. Output 1 if the AI has achieved the goal and 0 if the AI has not achieved the goal.",
            globalDescription: "Evaluate the performance of the LLM in identifying and achieving the goals of the user. Output 1 if the AI has achieved the goal and 0 if the AI has not achieved the goal. Average over all rows.",
            labelRowByRow: "Agent Goal Accuracy without Reference",
            nameRowByRow: "agentGoalAccuracyWithoutReference",
            selectable: true,
            excludedFromSelectAll: true,
            pairedWith: "agentGoalAccuracyWithReference"
        },
        {
            label: "Answer correctness",
            labelRowByRow: "Answer correctness",
            name: "answerCorrectness",
            nameRowByRow: "answerCorrectness",
            shortDescription: "Measures the accuracy of the answer compared to the ground truth. This metric is computed using the ground truth and the answer. It requires an embedding LLM and a completion LLM.",
            globalDescription: "Measures the accuracy of the answer compared to the ground truth. This metric is computed using the ground truth and the answer. It requires an embedding LLM and a completion LLM.",
            selectable: true
        },
        {
            label: "Answer similarity",
            labelRowByRow: "Answer similarity",
            name: "answerSimilarity",
            nameRowByRow: "answerSimilarity",
            shortDescription: "Assesses the semantic resemblance between the generated answer and the ground truth. The metric is computed using the ground truth and the answer. It requires an embedding LLM and a completion LLM.",
            globalDescription: "Assesses the semantic resemblance between the generated answer and the ground truth. The metric is computed using the ground truth and the answer. It requires an embedding LLM and a completion LLM.",
            selectable: true,
            excludedFromSelectAll: false,
        },
        {
            label: "BERT Score",
            labelRowByRow: "BERT Score",
            name: "bertScore",
            nameRowByRow: "bertScore",
            shortDescription: "Computes the similarity of answer and ground truth(s) using their tokens' embeddings. The metric is computed using the answer and the ground truth.",
            globalDescription: "Computes the similarity of answer and ground truth(s) using their tokens' embeddings. The metric is computed using the answer and the ground truth.",
            selectable: true,
            excludedFromSelectAll: false,
        },

    ];

    svc.nameByMetricCode =
    {
        // Tool execution statistics, global metrics
        AVERAGE_TOOL_EXECUTIONS_PER_ROW: 'averageToolExecutionsPerRow',
        AVERAGE_FAILED_TOOL_EXECUTIONS_PER_ROW: 'averageFailedToolExecutionsPerRow',
        AVERAGE_TOOL_EXECUTION_TIME_SECONDS_PER_ROW: 'averageToolExecutionTimeSecondsPerRow',
        P95_TOTAL_AGENT_CALL_EXECUTION_TIME_SECONDS_PER_ROW: 'p95TotalAgentCallExecutionTimeSecondsPerRow',
        // Tool execution statistics, row-by-row metrics
        TOTAL_TOOL_EXECUTIONS: 'totalToolExecutions',
        TOTAL_FAILED_TOOL_EXECUTIONS: 'totalFailedToolExecutions',
        TOTAL_TOOL_EXECUTION_TIME_SECONDS: 'totalToolExecutionTimeSeconds',
        TOTAL_AGENT_CALL_EXECUTION_TIME_SECONDS: 'totalAgentCallExecutionTimeSeconds',
        SAMPLE_ROW_COUNT: 'sampleRowCount',
        // Tool call metrics
        TOOL_CALL_EXACT_MATCH: 'toolCallExactMatch',
        TOOL_CALL_PARTIAL_MATCH: 'toolCallPartialMatch',
        TOOL_CALL_PRECISION_RECALL_F1: 'toolCallPrecisionRecallF1',
        TOOL_CALL_PRECISION: 'toolCallPrecisionRecallF1',
        TOOL_CALL_RECALL: 'toolCallPrecisionRecallF1',
        TOOL_CALL_F1: 'toolCallPrecisionRecallF1',
        // Agent goal accuracy metrics
        AGENT_GOAL_ACCURACY_WITH_REFERENCE: 'agentGoalAccuracyWithReference',
        AGENT_GOAL_ACCURACY_WITHOUT_REFERENCE: 'agentGoalAccuracyWithoutReference',
        // Ragas
        ANSWER_CORRECTNESS: "answerCorrectness",
        ANSWER_SIMILARITY: "answerSimilarity",
        // BERTScore
        BERT_SCORE: "bertScore",
        BERT_SCORE_PRECISION: "bertScore",
        BERT_SCORE_RECALL: "bertScore",
        BERT_SCORE_F1: "bertScore",
    }

    // the final possible names of the metrics after computation. Exclude codes that are "input-only"
    svc.computableMetrics = Object.keys(svc.nameByMetricCode).filter(m => ! ["TOOL_CALL_PRECISION_RECALL_F1", "BERT_SCORE"].includes(m));

    svc.fieldsNeededPerMetric = {
        toolCallExactMatch: ["referenceToolCalls", "actualToolCalls"],
        toolCallPartialMatch: ["referenceToolCalls", "actualToolCalls"],
        toolCallPrecisionRecallF1: ["referenceToolCalls", "actualToolCalls"],
        agentGoalAccuracyWithReference: ["input", "output", "groundTruth", "actualToolCalls", "completionLLM", "embeddingLLM"],
        agentGoalAccuracyWithoutReference: ["input", "output", "actualToolCalls", "completionLLM", "embeddingLLM"],
        answerCorrectness: ["input", "output", "groundTruth", "embeddingLLM", "completionLLM"],
        answerSimilarity: ["input", "output", "groundTruth", "embeddingLLM", "completionLLM"],
        bertScore: ["output", "groundTruth"],
    }

    svc.metricGroups = [
        { name: 'toolcallMetricsInOrder', selectAll: true, label: "In order tool call metrics", metrics: [svc.nameByMetricCode.TOOL_CALL_EXACT_MATCH, svc.nameByMetricCode.TOOL_CALL_PARTIAL_MATCH] },
        { name: 'toolcallMetricsOutOfOrder', selectAll: true, label: "Out of order tool call metrics", metrics: [svc.nameByMetricCode.TOOL_CALL_PRECISION_RECALL_F1] },
        { name: 'llmAsAJudge', selectAll: true, label: "LLM-as-a-judge metrics", metrics: [svc.nameByMetricCode.AGENT_GOAL_ACCURACY_WITH_REFERENCE, svc.nameByMetricCode.AGENT_GOAL_ACCURACY_WITHOUT_REFERENCE, svc.nameByMetricCode.ANSWER_CORRECTNESS, svc.nameByMetricCode.ANSWER_SIMILARITY] },
        { name: 'deterministicLlm', selectAll: true, label: "Deterministic LLM metrics", metrics: [svc.nameByMetricCode.BERT_SCORE] },
    ];

    svc.isInGroup = function(metric, groupName) {
        // Check if a group by that name exists
        const group = this.metricGroups.find(g => g.name === groupName);
        if (!group) return false;
        // Check if that metric is in that group
        return group.metrics.includes(metric.name);
    };

    svc.getPotentialWarningForMetricComputation = function(metric, packagesWarning, hasInput, hasOutput, hasGroundTruth, hasActualToolCalls, hasReferenceToolCalls, hasEmbeddingLLM, hasCompletionLLM) {
        if (!Object.keys(svc.fieldsNeededPerMetric).includes(metric)) {
            return "";
        }

        if (svc.fieldsNeededPerMetric[metric].includes("input") && !hasInput) {
            return "Requires an Input column";
        }
        if (svc.fieldsNeededPerMetric[metric].includes("output") && !hasOutput) {
            return "Requires an Output column";
        }
        if (svc.fieldsNeededPerMetric[metric].includes("groundTruth") && !hasGroundTruth) {
            return "Requires a Ground truth column";
        }
        if (svc.fieldsNeededPerMetric[metric].includes("referenceToolCalls") && !hasReferenceToolCalls) {
            return "Requires a Reference tool call column";
        }
        if (svc.fieldsNeededPerMetric[metric].includes("actualToolCalls") && !hasActualToolCalls) {
            return "Requires an Actual tool call column";
        }
        if (svc.fieldsNeededPerMetric[metric].includes("embeddingLLM") && !hasEmbeddingLLM) {
            return "Requires an Embedding LLM";
        }
        if (svc.fieldsNeededPerMetric[metric].includes("completionLLM") && !hasCompletionLLM) {
            return "Requires a Completion LLM";
        }
        return "";
    }

    svc.getMetricByCode = function(metricCode) {
        const name = svc.nameByMetricCode[metricCode];
        if (!name) {
            return undefined;
        }
        return svc.metrics.find(m => m.name === name);
    }

    svc.agentModelTaskTypes = [
        {
            type: "AGENT",
            fullName: "Agent",
            shortName: "AGENT"
        }
    ];

    svc.inputFormats = [
        {
            type: "PROMPT_RECIPE",
            fullName: "Prompt Recipe",
            shortName: "prompt recipe"
        },
        {
            type: "CUSTOM",
            fullName: "Custom",
            shortName: "custom"
        },
    ];


    return svc;
});

app.controller("AgentEvaluationRecipeCreationController", function($scope, $controller) {
    $controller("GenAIBaseEvaluationRecipeCreationController", { $scope: $scope, evaluatedType: 'agent' });
});

app.controller("AgentEvaluationRecipeEditorController", function($scope, $controller, AgentEvaluationMetrics, WT1, DataikuAPI, ActiveProjectKey, FutureProgressModal, StringUtils, CodeMirrorSettingService) {
    $controller("GenAIBaseEvaluationRecipeEditorController", { $scope: $scope, evaluatedType: 'agent' });

    $scope.inputFormats = AgentEvaluationMetrics.inputFormats;

    // Not mutualised for now
    $scope.saveColumnNames = function() {
        if ($scope.desc.inputColumnName !== $scope.NO_COLUMN_SELECTED) {
            $scope.savedColumnNames.input = $scope.desc.inputColumnName;
        }
        if ($scope.desc.outputColumnName !== $scope.NO_COLUMN_SELECTED) {
            $scope.savedColumnNames.output = $scope.desc.outputColumnName;
        }
        if ($scope.desc.actualToolCallsColumnName !== $scope.NO_COLUMN_SELECTED) {
            $scope.savedColumnNames.actualToolCalls = $scope.desc.actualToolCallsColumnName;
        }
    }
    $scope.restoreColumnNames = function() {
        if ($scope.savedColumnNames.input !== null) {
            $scope.desc.inputColumnName = $scope.savedColumnNames.input
            $scope.savedColumnNames.input = null;
        }
        if ($scope.savedColumnNames.output !== null) {
            $scope.desc.outputColumnName = $scope.savedColumnNames.output
            $scope.savedColumnNames.output = null;
        }
        if ($scope.savedColumnNames.actualToolCalls !== null) {
            $scope.desc.actualToolCallsColumnName = $scope.savedColumnNames.actualToolCalls
            $scope.savedColumnNames.actualToolCalls = null;
        }
    }

    $scope.savedColumnNames = {
        input: null,
        output: null,
        actualToolCalls: null
    }

    $scope.$watch("desc.inputFormat", (nv, ov) => {
        if (nv === "PROMPT_RECIPE") {
            if (ov !== "DATAIKU_ANSWERS") {
                $scope.saveColumnNames();
            }
            $scope.desc.inputColumnName = "llm_raw_query";
            $scope.desc.outputColumnName = "llm_raw_response";
            $scope.desc.actualToolCallsColumnName = "llm_raw_response";
        } else {
            $scope.restoreColumnNames();
        }

        $scope.updateMissingColumnsWarning(nv);
    });

    $scope.onInit = function() {
        $scope.filteredMetrics = getFilteredMetrics();
        $scope.metricGroups.forEach(group => $scope.updateGroupSelectAll(group));
    }

    $scope.filteredMetrics = [];

    $scope.fieldsNeeded = AgentEvaluationMetrics.fieldsNeededPerMetric;

    $scope.metricGroups = AgentEvaluationMetrics.metricGroups;

    $scope.filterByGroup = function(groupName) {
        return function(metric) {
            return AgentEvaluationMetrics.isInGroup(metric, groupName);
        };
    };

    $scope.toggleAllMetrics = function(group) {
        $scope.filteredMetrics.filter(m => AgentEvaluationMetrics.isInGroup(m, group.name)).forEach(m =>
            m.$selected = group.selectAll && !m.excludedFromSelectAll
        )
        $scope.updateFilteredMetricsSelection();
    };

    $scope.updateGroupSelectAll = function(group) {
        const groupSelectAllableMetrics = $scope.filteredMetrics.filter(m => AgentEvaluationMetrics.isInGroup(m, group.name) && !m.excludedFromSelectAll);
        const metricSelectAllableCount = groupSelectAllableMetrics.length;
        const selectedMetricCount = groupSelectAllableMetrics.filter(m => m.$selected).length;
        group.selectAll = (metricSelectAllableCount === selectedMetricCount);
    };

    $scope.updateAgentGoalMetrics = function(metric) {
        if (metric.$selected) {
            if (metric.name === "agentGoalAccuracyWithReference") {
                $scope.filteredMetrics.filter(m => m.name === "agentGoalAccuracyWithoutReference").forEach(m => m.$selected = false);
            } else if (metric.name === "agentGoalAccuracyWithoutReference") {
                $scope.filteredMetrics.filter(m => m.name === "agentGoalAccuracyWithReference").forEach(m => m.$selected = false);
            }
        }
    };

    $scope.$watch("desc.inputFormat", () => {
        if ($scope.desc.inputFormat) {
            $scope.filteredMetrics = getFilteredMetrics();
        }
    })

    function getFilteredMetrics()  {
        return AgentEvaluationMetrics.metrics
            .filter(possibleMetric => possibleMetric.selectable)
            .map(possibleMetric => {
                if ($scope.desc.metrics.includes(possibleMetric.name)) {
                    return ({...possibleMetric, $selected: true})
                }
                return ({...possibleMetric, $selected: false})
            });
    }

    $scope.updateFilteredMetricsSelection = function() {
        $scope.desc.metrics = $scope.filteredMetrics.filter(m => m.$selected).map(m => m.name);
    };

    $scope.getPotentialWarningForMetricComputation = function(isSelected, metric) {
        if (!isSelected) {
            return "";
        }
        const hasInput = !!$scope.desc.inputColumnName && $scope.desc.inputColumnName !== $scope.NO_COLUMN_SELECTED;
        const hasOutput = !!$scope.desc.outputColumnName && $scope.desc.outputColumnName !== $scope.NO_COLUMN_SELECTED;
        const hasGroundTruth = !!$scope.desc.groundTruthColumnName && $scope.desc.groundTruthColumnName !== $scope.NO_COLUMN_SELECTED;
        const hasActualToolCalls = !!$scope.desc.actualToolCallsColumnName && $scope.desc.actualToolCallsColumnName !== $scope.NO_COLUMN_SELECTED;
        const hasReferenceToolCalls = !!$scope.desc.referenceToolCallsColumnName && $scope.desc.referenceToolCallsColumnName !== $scope.NO_COLUMN_SELECTED;
        const hasEmbeddingLLM = !!$scope.desc.embeddingLLMId && $scope.desc.embeddingLLMId !== $scope.NO_LLM_SELECTED.id;
        const hasCompletionLLM = !!$scope.desc.completionLLMId && $scope.desc.completionLLMId !== $scope.NO_LLM_SELECTED.id;
        return AgentEvaluationMetrics.getPotentialWarningForMetricComputation(metric, $scope.codeEnvWarning.reason, hasInput, hasOutput, hasGroundTruth, hasActualToolCalls, hasReferenceToolCalls, hasEmbeddingLLM, hasCompletionLLM);
    }

    const customMetricDefaultCode = `def evaluate(input_df, recipe_params, interpreted_columns, **kwargs):
    """
    Custom score function.
    Must return a float representing the metric value over the whole sample, and 
    optionally an array of shape (nb_records,) with the row-by-row values
    - input_df is the Input dataset of the Agent evaluation recipe, as a Pandas DataFrame
    - recipe_params is an object with the Agent evaluation recipe parameters:
        - input_column_name: Name of the Input column as a string
        - output_column_name (may be None): Name of the Output column as a string
        - ground_truth_column_name (may be None): Name of the Ground truth column as 
          a string
        - reference_tool_calls_column_name (may be None): Name of the Reference tool 
          calls column as a string
        - actual_tool_calls_column_name (may be None): Name of the Actual tool calls 
          column as a string
        - embedding_llm (may be None): A DKUEmbeddings object, to be used to query 
          the LLM mesh to embed the inputs
        - completion_llm (may be None): A DKULLM object, to be used to query the 
          LLM mesh with an eventual prompt
    - interpreted_columns are the input columns as they were used by the recipe,
      including some eventual pre-treatment, notably for the prompt recipe and dataiku
      answers cases. Formatted as Pandas Dataseries.
      If no pre-treatment happened, their content is the same as in the input_df
        - input : for Prompt recipe, this is a concatenation of all the prompt messages
        - output : for Prompt recipe, only the text, extracted from the json
        - ground_truth 
        - reference_tool_calls
        - actual_tool_calls
    """
    return 0.5, [optionally, one, value, per, row]`

    $scope.getNewMetricTemplate = function() {
        if (!$scope.desc.customMetrics) {$scope.desc.customMetrics = [];};
        const name = StringUtils.transmogrify("Custom Metric #" + ($scope.desc.customMetrics.length + 1).toString(),
            $scope.desc.customMetrics.map(a => a.name),
            function(i){return "Custom Metric #" + (i+1).toString() }
        );

        const template = {
            name,
            metricCode: customMetricDefaultCode,
            description: "",
            greaterIsBetter: true,
            minValue: 0,
            maxValue: 1,
            type: 'LLM',
            $foldableOpen: true
        };
        return template;
    }

    $scope.snippetCategory = 'py-agent-custom-metric';

    $scope.addNewCustomMetric = function() {
        if (!$scope.desc.customMetrics) {$scope.desc.customMetrics = [];};
        $scope.desc.customMetrics.push($scope.getNewMetricTemplate());
    }

    $scope.testCustomMetricResults = {}
    $scope.testCustomMetric = function(customMetricIndex) {
        const mainInput = $scope.recipe.inputs.main.items[0];
        const serialisedInputDataset = !!mainInput ? $scope.computablesMap[mainInput.ref] : undefined;
        return DataikuAPI.flow.recipes.testCustomMetric(ActiveProjectKey.get(), $scope.recipe, $scope.desc, customMetricIndex, serialisedInputDataset)
            .then(({data}) => {
                return FutureProgressModal.show($scope, data, "Testing");
            })
            .then((customMetricResult) => {
                $scope.testCustomMetricResults[customMetricIndex] = customMetricResult;
            })
            .catch(setErrorInScope.bind($scope));
    }

    $scope.fireCustomMetricAddedWT1Event = function() {
        WT1.event("clicked-item", {"item-id": 'agent-evaluation-add-custom-metric'});
    };

    $scope.fireCustomMetricRemovedWT1Event = function() {
        WT1.event("clicked-item", {"item-id": 'agent-evaluation-remove-custom-metric'});
    };

    $scope.toggleFoldable = function(index) {
        if($scope.desc.customMetrics[index]){
            $scope.desc.customMetrics[index].$foldableOpen = !$scope.desc.customMetrics[index].$foldableOpen
        }
    }

    const customTraitDefaultCode = `# Add assertions to check against the agent result
# RESULT: 
#   - 1 if ALL of the assertions are valid 
#   - 0 if any of them is invalid
# USE: 
#   Assertions should be separated by an empty line.
#   Assertions should be in natural language, using the following fields:
#     - Input: The initial query sent to the agent, e.g., the user's question
#     - Output: The final answer from the agent, as viewed by the user
#     - Ground truth: The expected answer, 
#         from the Ground Truth column defined in the "Input dataset" section
#     - Actual tool calls: The list of tools that were called by the agent
#     - Reference tool calls: The list of tools expected to be called by the agent, 
#         from the Reference tool call column defined in the "Input dataset" section
# EXAMPLE:
The Output uses a polite and professional tone

The Actual tool calls contains a call to mytool,
followed by a call to mytool2,
not followed by any call`

    $scope.getNewCustomTraitTemplate = function() {
        if (!$scope.desc.customTraits) {$scope.desc.customTraits = [];}
        const name = StringUtils.transmogrify("Custom Trait #" + ($scope.desc.customTraits.length + 1).toString(),
            $scope.desc.customTraits.map(a => a.name),
            function(i){return "Custom Trait #" + (i+1).toString() }
        );

        const template = {
            name,
            prompt: customTraitDefaultCode,
            description: "",
            type: 'TRAIT',
            $foldableOpen: true
        };
        return template;
    }

    $scope.traitSnippetCategory = 'sh-custom-trait';


    $scope.addNewCustomTrait = function() {
        if (!$scope.desc.customTraits) {$scope.desc.customTraits = [];}
        $scope.desc.customTraits.push($scope.getNewCustomTraitTemplate());
    }

    $scope.toggleTraitFoldable = function(metric) {
        if(metric){
            metric.$foldableOpen = !metric.$foldableOpen
        }
    }

    $scope.traitEditorOptions = CodeMirrorSettingService.get('text/x-sh');
    $scope.traitEditorOptions.lineNumbers = false;
    $scope.traitEditorOptions.foldGutter = false;
    $scope.traitEditorOptions.gutters = [];
    $scope.traitEditorOptions.lineWrapping = true;

    $scope.fireCustomTraitAddedWT1Event = function() {
        WT1.event("clicked-item", {"item-id": 'agent-evaluation-add-custom-trait'});
    };

    $scope.fireCustomTraitRemovedWT1Event = function() {
        WT1.event("clicked-item", {"item-id": 'agent-evaluation-remove-custom-trait'});
    };

});

}());
