(function() {
    'use strict';

    angular.module('dataiku.charts')
        .service('ChartDataUtils', ChartDataUtils);

    /**
     * (!) This service previously was in static/dataiku/js/simple_report/common/data.js
     */
    function ChartDataUtils(ChartDimension, ChartMeasure, ChartUADimension, Fn, $filter, CHART_DATES_LABELS, ChartLabels, CHART_TYPES, ChartsStaticData, translate, CHART_VARIANTS) {
        /**
         * Returns true if:
         * <ul>
         *     <li>no timestamp range is defined</li>
         *     <li>element index is not one of the 'other' bin</li>
         *     <li>element index corresponds to a bins which timestamp is in the specified range</li>
         * </ul>
         */
        function isElementInTimestampRange(elementIndex, axisLabelElements, timestampRange) {
            if (!timestampRange) {
                return true;
            }
            const labelElementIndex = getLabelIndexForTensorIndex(elementIndex, axisLabelElements.length);
            const isOthersCategoryIndex = labelElementIndex === undefined;
            if (isOthersCategoryIndex) {
                return false;
            }
            const axisLabelElementTimestamp = axisLabelElements[labelElementIndex].tsValue;
            const lowestRangeBound = timestampRange[0];
            const highestRangeBound = timestampRange[1];
            return axisLabelElementTimestamp >= lowestRangeBound && axisLabelElementTimestamp <= highestRangeBound;
        }

        /**
         * Returns the index of the axis label element corresponding to the tensor index or undefined
         * if index is one of the 'other" bin.
         */
        function getLabelIndexForTensorIndex(tensorElementIndex, numberOfAxisLabelElements) {
            const numberOfElementsInFacet = numberOfAxisLabelElements + 1; // because of 'other' bin;
            const labelElementIndex = tensorElementIndex % numberOfElementsInFacet;
            const isOthersCategoryElementIndex = labelElementIndex === numberOfAxisLabelElements;
            if (isOthersCategoryElementIndex) {
                return undefined;
            }
            return labelElementIndex;
        }

        /**
         * Filters the tensor to keep only:
         * <ul>
         *     <li>non empty bins (i.e. with a count  > 0) when includeEmptyBins is set as false</li>
         *     <li>if timestampRange is specified, the bins which corresponding timestamp is in the range</li>
         * </ul>
         * @return {Array} the filtered tensor
         */
        function filterTensorOnTimestampRange(tensor, axisLabelElements, counts, timestampRange, includeEmptyBins) {
            return tensor.filter((value, index) => {
                const isEmptyBin = counts[index] === 0;
                return isElementInTimestampRange(index, axisLabelElements, timestampRange) && (!isEmptyBin || includeEmptyBins);
            });
        }

        function buildDefaultExtent() {
            return {
                extent: [Infinity, -Infinity],
                onlyPercent: true
            };
        }

        /**
         * Returns the corresponding date display settings for the specified interval:
         * <ul>
         *     <li>for MILLISECONDS if the range is within the same second</li>
         *     <li>for SECONDS AND MILLISECONDS if the range is lower than 2 seconds</li>
         *     <li>for SECONDS if the range is within the same minute</li>
         *     <li>for MINUTES AND SECONDS if the range is lower than a 2 minutes</li>
         *     <li>for MINUTES if the range is within the same day</li>
         *     <li>for DAYS AND MINUTES if the range is lower than 24 hours</li>
         *     <li>else the default display</li>
         * </ul>
         * @param minTimestamp The lower bound of the interval in milliseconds
         * @param maxTimestamp The upper bound of the interval in milliseconds.
         * @return {{dateFilterOption: string, dateFormat: string, mainDateFormat: string, formatDateFn: function(number, string)}}
         * <ul>
         *     <li>
         *         <b>mainDateFormat</b> format for the main identical part of the interval.
         *         <b>undefined</b> if the interval is not on the same day.
         *     </li>
         *     <li><b>dateFormat</b> format to use for the dates in the interval.</li>
         *     <li><b>dateFilterOption</b> date filter option to be used in an AngularJS filter.</li>
         *     <li><b>formatDateFn</b> function to format a timestamp in milliseconds according to the specified format.</li>
         * </ul>
         */
        function getDateDisplayUnit(minTimestamp, maxTimestamp) {
            if (minTimestamp === undefined || maxTimestamp === undefined) {
                return CHART_DATES_LABELS.DATE_DISPLAY_UNIT_DEFAULT;
            }
            const minDate = new Date(minTimestamp);
            const maxDate = new Date(maxTimestamp);
            const minDateWrapper = moment(minDate).utc();
            const maxDateWrapper = moment(maxDate).utc();

            const isDomainLessThan24Hours = maxDateWrapper.diff(minDateWrapper, 'hours', true) <= 24;
            if (!isDomainLessThan24Hours) {
                return CHART_DATES_LABELS.DATE_DISPLAY_UNIT_DEFAULT;
            }

            const isDomainInSameDay = minDateWrapper.year() === maxDateWrapper.year() && minDateWrapper.dayOfYear() === maxDateWrapper.dayOfYear();
            if (!isDomainInSameDay) {
                return CHART_DATES_LABELS.DATE_DISPLAY_UNIT_DAY_AND_MINUTES;
            }

            const isDomainLessThan2Minutes = maxDateWrapper.diff(minDateWrapper, 'minutes', true) <= 2;
            if (!isDomainLessThan2Minutes) {
                return CHART_DATES_LABELS.DATE_DISPLAY_UNIT_MINUTES;
            }

            const isDomainLessThan2Seconds = maxDateWrapper.diff(minDateWrapper, 'seconds', true) <= 2;
            if (!isDomainLessThan2Seconds) {
                return CHART_DATES_LABELS.DATE_DISPLAY_UNIT_SECONDS;
            }

            return CHART_DATES_LABELS.DATE_DISPLAY_UNIT_MILLISECONDS;
        }
        /**
         * Returns a label to be used to display records count in the UI.
         * @param   {Number}    beforeFiltersCount
         * @param   {Number}    afterFiltersCount
         * @param   {Map}       visibleCountMap
         * @param   {ChartType.java} chartType
         * @return  {String}    Human-readable label
         */
        function getLabelForRecordsCount(beforeFiltersCount, afterFiltersCount, visibleCountMap, chartType) {
            let actualCount = afterFiltersCount;

            if (afterFiltersCount === beforeFiltersCount && afterFiltersCount === 0) {
                return ChartLabels.getNoRecordLabel();
            }

            if ([CHART_TYPES.SCATTER_MULTIPLE_PAIRS, CHART_TYPES.SCATTER].includes(chartType)) {
                actualCount = visibleCountMap && visibleCountMap.size && Math.max(...visibleCountMap.values()) || 0;
            }

            if (actualCount < beforeFiltersCount) {
                return `${actualCount} / ${beforeFiltersCount} ${translate('CHARTS.HEADER.RECORDS', 'records')}`;
            } else {
                return `${beforeFiltersCount} ${beforeFiltersCount !== 1 ? translate('CHARTS.HEADER.RECORDS', 'records') : translate('CHARTS.HEADER.RECORD', 'record')}`;
            }
        }

        const cartesian = (...a) => a.reduce((a, b) => a.flatMap(d => b.map(e => [d, e].flat())));

        const longSmartNumber = $filter('longSmartNumber');

        const svc = {
            /**
             * Returns the min & max values across all dimensions & all measures for the two display axes
             * @param {ChartDef.java} chartDef
             * @param {ChartTensorDataWrapper} chartData
             * @param {string} mainAxisName - id of the main axis
             * @param {Array} timestampRange min and max used to filter the data based on their timestamp
             * @param {Boolean} includeEmptyBins - whether or not to include empty bins
             * @return {Object} { y1: { extent: [Number, Number], onlyPercent: Boolean }, y2: { extent: [Number, Number], onlyPercent: Boolean }, recordsCount: Number, pointsCount: Number }
             */
            getMeasureExtents: function(chartDef, chartData, mainAxisName, timestampRange, includeEmptyBins) {
                const result = {
                    [ChartsStaticData.LEFT_AXIS_ID]: buildDefaultExtent(),
                    [ChartsStaticData.RIGHT_AXIS_ID]: buildDefaultExtent(),
                    recordsCount: 0,
                    pointsCount: 0
                };

                const countsTensor = chartData.data.counts.tensor;
                const mainAxisLabels = chartData.getAxisLabels(mainAxisName);
                chartDef.genericMeasures.forEach(function(measure, measureIndex) {
                    let measureExtent = [];
                    const measureData = chartData.data.aggregations[measureIndex];
                    if (ChartMeasure.isRealUnaggregatedMeasure(measure)) {
                        const filterTensor = (t) => filterTensorOnTimestampRange(t, mainAxisLabels, countsTensor, timestampRange, includeEmptyBins);
                        measureExtent = ChartMeasure.getUaMeasureExtent(measureData, filterTensor);
                    } else {
                        const filteredTensor = filterTensorOnTimestampRange(measureData.tensor, mainAxisLabels, countsTensor, timestampRange, includeEmptyBins);
                        measureExtent = d3.extent(
                            chartDef.variant === CHART_VARIANTS.waterfall
                                ? filteredTensor.reduce((acc, value) => {
                                    const cumulativeValue = acc[acc.length - 1] || 0;
                                    acc.push(cumulativeValue + value);
                                    return acc;
                                }, [])
                                : filteredTensor
                        );
                    }
                    const axis = measure.displayAxis === 'axis1' ? ChartsStaticData.LEFT_AXIS_ID : ChartsStaticData.RIGHT_AXIS_ID;
                    result[axis].onlyPercent = result[axis].onlyPercent && ChartDimension.isPercentScale([measure]);
                    result[axis].extent[0] = _.isFinite(measureExtent[0]) ? Math.min(measureExtent[0], result[axis].extent[0]) : result[axis].extent[0];
                    result[axis].extent[1] = _.isFinite(measureExtent[1]) ? Math.max(measureExtent[1], result[axis].extent[1]) : result[axis].extent[1];
                });

                const countsTensorInRange = filterTensorOnTimestampRange(countsTensor, mainAxisLabels, countsTensor, timestampRange, includeEmptyBins);
                result.recordsCount = countsTensorInRange.reduce((currentCount, countInBin) => currentCount + countInBin, 0);
                result.pointsCount = countsTensorInRange.length;
                return result;
            },

            getValuesForLines: function(binIndexes, chartData, ignoreLabels) {
                // For lines we only want the total bin for the color dimension and all the axis bin (total excluded)
                const valuesForLines = binIndexes.reduce((bins, binValue, binIndex) => {
                    bins.push(chartData.data.axisLabels[binIndex].reduce((acc, cV, cI) => {
                        if (ignoreLabels.has(cV.label)) {
                            return acc;
                        }
                        if (binValue !== undefined) {
                            acc.push(binValue);
                        } else {
                            acc.push(cI);
                        }
                        return acc;
                    }, []));
                    return bins;
                }, []);
                return cartesian(...valuesForLines);
            },

            getValuesForColumns: function(binIndexes, chartData, ignoreLabels) {
                // For columns we don't want the total bin for color and all the axis bin (total exclude)
                const valuesForColumns = binIndexes.reduce((bins, binValue, binIndex) => {
                    bins.push(chartData.data.axisLabels[binIndex].reduce((acc, cV, cI) => {
                        if (ignoreLabels.has(cV.label)) {
                            return acc;
                        }
                        acc.push(cI);
                        return acc;
                    }, []));
                    return bins;
                }, []);

                return cartesian(...valuesForColumns);
            },

            getMeasureExtentsForMixColored: function(chartDef, chartData, includeEmptyBins, ignoreLabels = new Set(), binIndexes = [], ignoreForMeasure) {
                const result = {
                    [ChartsStaticData.LEFT_AXIS_ID]: buildDefaultExtent(),
                    [ChartsStaticData.RIGHT_AXIS_ID]: buildDefaultExtent(),
                    recordsCount: 0,
                    pointsCount: 0
                };

                const mainAxisLabel = chartData.data.axisLabels[0];
                const countsTensor = chartData.data.counts.tensor;
                // Create bins to visit by forcing dimensions to certain values, because total bin must be taken or not
                const valuesToCheckForLines = this.getValuesForLines(binIndexes, chartData, ignoreLabels);
                const valuesToCheckForColumns = this.getValuesForColumns(binIndexes, chartData, ignoreLabels);
                // We now have all the values we want we create the extent, mix can't have timestamp yet so we don't care of them
                chartDef.genericMeasures.forEach((measure, measureIndex) => {
                    const tensorValues = ignoreForMeasure && ignoreForMeasure(measure) ? this.retrieveAggrData(chartData, valuesToCheckForColumns, measureIndex) : this.retrieveAggrData(chartData, valuesToCheckForLines, measureIndex);
                    let measureExtent = [];
                    if (ChartMeasure.isRealUnaggregatedMeasure(measure)) {
                        measureExtent = ChartMeasure.getUaMeasureExtent(this.retrieveUnaggrExtents(chartData, ignoreForMeasure && ignoreForMeasure(measure) ? valuesToCheckForColumns : valuesToCheckForLines, measureIndex));
                    } else {
                        measureExtent = d3.extent(tensorValues);
                    }


                    const axis = measure.displayAxis === 'axis1' ? ChartsStaticData.LEFT_AXIS_ID : ChartsStaticData.RIGHT_AXIS_ID;
                    result[axis].onlyPercent = result[axis].onlyPercent && ChartDimension.isPercentScale([measure]);
                    result[axis].extent[0] = Math.min(measureExtent[0], result[axis].extent[0]);
                    result[axis].extent[1] = Math.max(measureExtent[1], result[axis].extent[1]);
                });

                const countsTensorInRange = filterTensorOnTimestampRange(countsTensor, mainAxisLabel, countsTensor, undefined, includeEmptyBins);
                result.recordsCount = countsTensorInRange.reduce((currentCount, countInBin) => currentCount + countInBin, 0);
                result.pointsCount = countsTensorInRange.length;
                return result;
            },

            retrieveAggrData: function(chartData, coordsArray, measureIndex) {
                return coordsArray.reduce((acc, coords) => {
                    acc.push(chartData.getPoint(chartData.data.aggregations[measureIndex], Array.isArray(coords) ? coords : [coords]));
                    return acc;
                }, []);
            },

            retrieveUnaggrExtents: function(chartData, coordsArray, measureIndex) {
                return coordsArray.reduce((acc, coords) => {
                    const extents = chartData.getPointExtents(chartData.data.aggregations[measureIndex], Array.isArray(coords) ? coords : [coords]);
                    acc.uaDataNegativeExtent.push(extents[0]);
                    acc.uaDataPositiveExtent.push(extents[1]);
                    return acc;
                }, { uaDataNegativeExtent: [], uaDataPositiveExtent: [] });
            },

            /**
             * Returns the min & max values across all dimensions for the given measure
             * @param {ChartTensorDataWrapper} chartData
             * @param {Number} mIdx - measure index
             * @param {Boolean} ignoreEmptyBins - whether or not to ignore empty bins
             * @param {Set<int> | null} binsToInclude - subtotal tensor indexes to include, the other will be excluded
             * @param {number[]} axesIdx - indexes of other axes
             * @return {Array} extent as [min, max]
             */
            getMeasureExtent: function(chartData, mIdx, ignoreEmptyBins, binsToInclude = null, aggrFn = null, axesIdx = []) {
                const allMeasureIndexes = axesIdx.concat(mIdx);
                const data = chartData.data;

                if (allMeasureIndexes.every(mIdx => !data.aggregations[mIdx])) {
                    return null;
                }

                let accessor = Fn.SELF;
                if (ignoreEmptyBins) {
                    accessor = function(d, i) {
                        // if the current axis has null for a value, but one of the other axes don't, we should still consider that value in the extent
                        const hasNonNullElement = allMeasureIndexes.some(mIdx => chartData.getNonNullCount(i, mIdx) > 0);

                        // maybe check isBinMeaningful for every axis ?

                        if (hasNonNullElement || chartData.isBinMeaningful(aggrFn, i, mIdx)) {
                            return !binsToInclude || binsToInclude.has(i) ? d : null;
                        }
                        return null;
                    };
                }

                return d3.extent(data.aggregations[mIdx].tensor, accessor);
            },


            /**
             * Returns an aggregation tensor where empty & all-null bins are filtered out
             * @param {PivotTableTensorResponse.java} data
             * @param {number} mIdx     index  of the measure whose values we're getting
             * @return {Array}          list of values for non-empty and non-null bins
             */
            getMeasureValues: function(data, mIdx, binsToInclude) {
                if (!data.aggregations[mIdx]) {
                    return null;
                }

                return data.aggregations[mIdx].tensor.filter(function(d, i) {
                    if (data.aggregations[mIdx].nonNullCounts) {
                        return data.aggregations[mIdx].nonNullCounts[i] > 0;
                    } else {
                        return (data.counts.tensor[i] > 0) ? (!binsToInclude || binsToInclude.has(i) ? true : false) : false;
                    }
                });
            },

            /**
             * Returns the min, max, and list of values on the given axis
             * @param {ChartTensorDataWrapper} chartData
             * @param {String} axisName: the name of the axis in chartData
             * @param {DimensionDef.java} dimension
             * @param {{ ignoreLabels?: boolean, initialExtent?: [number, number] }} extraOptions
             * @return {Object} extent as {values: [], min: min, max: max}
             */
            getAxisExtent: function(chartData, axisName, dimension, extraOptions = {}) {
                const ignoreLabels = extraOptions.ignoreLabels || new Set();
                const initialExtent = extraOptions.initialExtent || [Infinity, -Infinity];
                const labels = chartData.getAxisLabels(axisName);
                let min = initialExtent[0];
                let max = initialExtent[1];

                const values = labels.filter(label => !ignoreLabels.has(label.label)).map(function(label) {
                    if (ChartDimension.isTimelineable(dimension)) {
                        min = Math.min(min, label.min);
                        max = Math.max(max, label.max);
                    } else if (ChartDimension.isTrueNumerical(dimension)) {
                        if (ChartDimension.isUngroupedNumerical(dimension) || label.min == null) {
                            min = Math.min(min, label.sortValue);
                            max = Math.max(max, label.sortValue);
                        } else {
                            min = Math.min(min, label.min);
                            max = Math.max(max, label.max);
                            return [label.min, label.max];
                        }
                    }
                    return label.label;
                });

                return { values, min, max };
            },

            /**
             * Returns the min, max, and list of values on the given axis
             * @param {String} uaDimensionType
             * @param {ScatterAxis.java} axisData
             * @param {Number} afterFilterRecords
             * @return {Object} extent as {values: [], min: min, max: max}
             */
            getUnaggregatedAxisExtentByType: function(uaDimensionType, axisData, afterFilterRecords) {
                switch (uaDimensionType) {
                    case 'str':
                        return {
                            values: angular.copy(axisData.str.sortedMapping)
                                .sort(function(a, b) {
                                    return d3.ascending(a.sortOrder, b.sortOrder);
                                })
                                .map(Fn.prop('label'))
                        };
                    case 'num':
                    case 'ts':
                        return {
                            values: axisData[uaDimensionType].data.filter((d, i) => i < afterFilterRecords),
                            min: axisData[uaDimensionType].min,
                            max: axisData[uaDimensionType].max
                        };
                    default:
                        throw new Error('Unhandled dimension type: ' + uaDimensionType);
                }
            },
            /**
             * Returns the min, max, and list of values on the given axis
             * @param {NADimensionDef.java} dimension
             * @param {ScatterAxis.java} axisData
             * @param {Number} afterFilterRecords
             * @return {Object} extent as {values: [], min: min, max: max}
             */
            getUnaggregatedAxisExtent: function(dimension, axisData, afterFilterRecords) {
                const uaDimensionType = ChartUADimension.getUnaggregatedDimensionType(dimension);
                return svc.getUnaggregatedAxisExtentByType(uaDimensionType, axisData, afterFilterRecords);
            },

            /**
             * Returns the min, max, and list of values on many axes
             * @param {dimension: NADimensionDef.java, data: ScatterAxis.java} axis
             * @param {Number} afterFilterRecords
             * @return {Object} extent as {values: [], min: min, max: max}
             */
            getUnaggregatedMultipleAxisExtent: function(axes, afterFilterRecords) {
                const finalExtent = {
                    min: undefined,
                    max: undefined,
                    values: []
                };
                axes.forEach(axis => {
                    const uaDimensionType = ChartUADimension.getUnaggregatedDimensionType(axis.dimension);
                    const extent = svc.getUnaggregatedAxisExtentByType(uaDimensionType, axis.data, afterFilterRecords);
                    if (_.isNil(finalExtent.min) || extent.min < finalExtent.min) {
                        finalExtent.min = extent.min;
                    }
                    if (_.isNil(finalExtent.max) || extent.max > finalExtent.max) {
                        finalExtent.max = extent.max;
                    }
                    finalExtent.values = [...finalExtent.values, ...extent.values];
                });
                return finalExtent;
            },

            /**
             * @param   {PivotResponse.java}    pivotResponse
             * @param   {ChartType.java}        chartType
             * @param   {string | undefined}    clickActionMessage
             * @returns the summary for the current sampling and count of records after filtering.
             */
            getSamplingSummaryMessage: function(pivotResponse, chartType, clickActionMessage, visibleCountMap) {
                const sampleMetadata = pivotResponse.sampleMetadata;

                if (!sampleMetadata) {
                    return;
                }
                let samplingSummaryMessage = `<i class="dku-icon-warning-fill-16 tooltip-icon"></i> ${translate('CHARTS.HEADER.DATA_SAMPLING_NOT_REPRESENTATIVE', 'Data sampling can be misrepresentative')}</br>`;

                if (clickActionMessage) {
                    samplingSummaryMessage += `${clickActionMessage}</br>`;
                }

                if (chartType === CHART_TYPES.SCATTER_MULTIPLE_PAIRS) {
                    return samplingSummaryMessage + svc.getScatterMPRecordsSummary(pivotResponse.axesPairs, visibleCountMap);
                } else if (sampleMetadata.datasetRecordCount > -1) {
                    samplingSummaryMessage += translate('CHARTS.HEADER.DATA_SAMPLING.ROWS_OUT_OF', '<strong>{{beforeFilterRecords}}</strong> rows out of {datasetRecordCount}} {{estimated ? \'(estimated)\' : \'\'',
                        { beforeFilterRecords: longSmartNumber(pivotResponse.beforeFilterRecords), datasetRecordCount: longSmartNumber(sampleMetadata.datasetRecordCount), estimated: sampleMetadata.recordCountIsApproximate });
                } else {
                    samplingSummaryMessage += translate('CHARTS.HEADER.DATA_SAMPLING.ROWS', '<strong>{{beforeFilterRecords}}</strong> rows', { beforeFilterRecords: longSmartNumber(pivotResponse.beforeFilterRecords) });
                }
                return samplingSummaryMessage;
            },

            getScatterMPRecordsSummary(axesPairs, visibleCountMap) {
                return (axesPairs || []).map((axisPair, index) => {
                    const value = visibleCountMap ? visibleCountMap.get(index) || 0 : axisPair.afterFilterRecords;
                    return translate('CHARTS.HEADER.DATA_SAMPLING.SCATTER', '<strong>Pair {{index}}</strong>: {{value}} record{{ value !== 1 ? \'s\' : \'\'}} after filtering', { index: index + 1, value: value });
                }).join('</br>');
            },

            /**
             * @param   {PivotResponse.java}    pivotResponse
             * @param   {ChartType.java}        chartType
             * @param   {string | undefined}    clickActionMessage
             * @returns The sample metadata, the sampling summary message, and the clickable summary message to be shared in the UI if necessary (else null).
             */
            getSampleMetadataAndSummaryMessage: (pivotResponse, chartType, clickActionMessage) => {
                const sampleMetadata = pivotResponse && pivotResponse.sampleMetadata;

                if (sampleMetadata) {
                    return {
                        sampleMetadata,
                        summaryMessage: !sampleMetadata.sampleIsWholeDataset ? svc.getSamplingSummaryMessage(pivotResponse, chartType) : null,
                        clickableSummaryMessage: !sampleMetadata.sampleIsWholeDataset && clickActionMessage ? svc.getSamplingSummaryMessage(pivotResponse, chartType, clickActionMessage) : null
                    };
                }

                return { sampleMetadata: null, summaryMessage: null, clickableSummaryMessage: null };
            },

            getRecordsFinalCountTooltip: function(chartType, afterFilterRecords, axisPairs, visibleCountMap) {
                if (chartType === CHART_TYPES.SCATTER_MULTIPLE_PAIRS) {
                    return svc.getScatterMPRecordsSummary(axisPairs, visibleCountMap);
                } else {
                    return `<strong>${afterFilterRecords}</strong> ${translate('CHARTS.HEADER.RECORDS_REMAIN', 'records remain after applying filtering')}`;
                }
            },

            /**
             * Computes the label that will be displayed on the top right of the chart.
             */
            computeRecordsStatusLabel: function(beforeFilterRecords, afterFilterRecords, computedMainAutomaticBinningModeDescription, visibleCountMap, chartType) {
                const result = [];
                const labelForRecordsCount = getLabelForRecordsCount(beforeFilterRecords, afterFilterRecords, visibleCountMap, chartType);

                if (labelForRecordsCount) {
                    result.push(labelForRecordsCount);
                }

                if (computedMainAutomaticBinningModeDescription && labelForRecordsCount !== ChartLabels.getNoRecordLabel()) {
                    result.push(computedMainAutomaticBinningModeDescription);
                }
                return result.length === 0 ? undefined : result.join(' ');
            },

            computeNoRecordsTopRightLabel: function() {
                return this.computeRecordsStatusLabel(0, 0, undefined);
            },
            /**
             * Computes the display settings for the specified interval.
             * If a main identical part is identified also computed the date to display as <b>formattedMainDate</b>
             */
            computeDateDisplayUnit: function(minTimestamp, maxTimestamp) {
                const dateDisplayUnit = getDateDisplayUnit(minTimestamp, maxTimestamp);
                if (minTimestamp !== undefined && dateDisplayUnit.mainDateFormat !== undefined) {
                    return { ...dateDisplayUnit, formattedMainDate: dateDisplayUnit.formatDateFn(minTimestamp, dateDisplayUnit.mainDateFormat) };
                }
                return dateDisplayUnit;
            },
            isSameDay(minTimestamp, maxTimestamp) {
                if (minTimestamp === undefined || maxTimestamp === undefined) {
                    return false;
                }

                const minDateWrapper = moment.utc(minTimestamp);
                const maxDateWrapper = moment.utc(maxTimestamp);
                return minDateWrapper.year() === maxDateWrapper.year() && minDateWrapper.dayOfYear() === maxDateWrapper.dayOfYear();
            },
            isPivotRequestAborted: (errorData) => errorData && errorData.hasResult && errorData.aborted
        };

        return svc;
    }
})();
