import base64
import enum
import logging
import os
from abc import ABC, abstractmethod
from functools import lru_cache
from typing import Optional

from requests import utils

class ImageRetrieval(enum.Enum):
    IMAGE_REF = "IMAGE_REF",
    IMAGE_INLINE = "IMAGE_INLINE"


class MultipartContent(ABC):
    class PartType(enum.Enum):
        TEXT = "TEXT"
        IMAGE_INLINE = "IMAGE_INLINE"
        IMAGE_REF = "IMAGE_REF"
        CAPTIONED_IMAGE_INLINE = "CAPTIONED_IMAGE_INLINE"
        CAPTIONED_IMAGE_REF = "CAPTIONED_IMAGE_REF"

    def __init__(self, index: Optional[int]):
        # index stays optional without default since it is always expected but can be passed as None. See `MetadataHandler.get_multipart_content`
        self.index = index
        self.type: Optional[str] = None

    @abstractmethod
    def to_text(self) -> str:
        pass


class TextPart(MultipartContent):

    def __init__(self, index: Optional[int], text: str):
        super().__init__(index)
        self.type = self.PartType.TEXT.value
        self.text = text

    def to_text(self) -> str:
        return self.text


class ImagePart(MultipartContent, ABC):

    def __init__(self, index: Optional[int]):
        super().__init__(index)


class InlineImagePart(ImagePart):

    def __init__(self, index: Optional[int], image_bytes: bytes, mime_type: Optional[str]):
        super().__init__(index)
        self.type = self.PartType.IMAGE_INLINE.value
        self.inline_image = base64.b64encode(image_bytes).decode("utf8")
        self.image_mime_type = mime_type

    def to_text(self) -> str:
        return self.inline_image


class ImageRefPart(ImagePart):

    def __init__(self, index: Optional[int], full_folder_id: str, path: str):
        super().__init__(index)
        self.type = self.PartType.IMAGE_REF.value
        self.full_folder_id = full_folder_id
        self.path = path

    def to_text(self) -> str:
        return self.full_folder_id + self.path

class CaptionedImagePart(ImagePart, ABC):

    def __init__(self, caption: str, index: Optional[int]):
        super().__init__(index)
        self.caption = caption

class CaptionedImageRefPart(CaptionedImagePart):
    def __init__(self, caption: str, index: Optional[int], full_folder_id: str, path: str):
        super().__init__(caption, index)
        self.type = self.PartType.CAPTIONED_IMAGE_REF.value
        self.full_folder_id = full_folder_id
        self.path = path

    def to_text(self) -> str:
        return self.caption + " - " + self.full_folder_id + self.path


class InlineCaptionedImagePart(CaptionedImagePart):
    def __init__(self, caption: str, index: Optional[int], image_bytes: bytes, mime_type: Optional[str]):
        super().__init__(caption, index)
        self.type = self.PartType.CAPTIONED_IMAGE_INLINE.value
        self.inline_image = base64.b64encode(image_bytes).decode("utf8")
        self.image_mime_type = mime_type

    def to_text(self) -> str:
        return self.caption + " - " + self.inline_image


@lru_cache(maxsize=20)
def get_image_content(image_path: str, full_folder_id: str):
    from dataiku.core import intercom

    project_key, lookup = full_folder_id.split(".", 1)
    download_response = intercom.backend_api_get_call(
        "managed-folders/download-path?projectKey=" + project_key + "&lookup=" + lookup + "&path=" + utils.quote(image_path), None)

    if download_response.status_code == 200:
        mime_type = "image/" + os.path.splitext(image_path)[1][1:]
        return download_response.content, mime_type
    else:
        logging.warning("Error when retrieving file {image_path} in folder {folder_id} : {err_msg} - skipping"
                        .format(image_path=image_path,
                                folder_id=full_folder_id,
                                err_msg=download_response.content),
                        )
        raise ValueError
