########### RemoteRunEnvDef manipulation

dku_remoterun__fetch_remote_run_env_def <- function(noFail=TRUE) {
  x = NULL
  file_name <- "remote-run-env-def.json"
  home_env_file_path <- paste0(Sys.getenv("HOME"), "/", file_name)
  if (file.exists(file_name)) {
      library("RJSONIO");
      json <- readChar(file_name, file.info(file_name)$size)
      x = RJSONIO::fromJSON(json)
    } else if (Sys.getenv("DKU_CONTAINER_EXEC") == "1" && file.exists(home_env_file_path)) {
      library("RJSONIO");
      json <- readChar(home_env_file_path, file.info(home_env_file_path)$size)
      x = RJSONIO::fromJSON(json)
    } else {
      x = list(env=list())
      for (name in names(Sys.getenv())) {
        x$env[name] = Sys.getenv(name)
      }
      x$r = list()
  }
  assign("dku_remoterun__cached_env_def", x, envir = .GlobalEnv)
}
dku_remoterun__get_remote_run_env_def <- function(noFail=TRUE) {
    if (!exists("dku_remoterun__cached_env_def", envir = .GlobalEnv)) {
      dku_remoterun__fetch_remote_run_env_def(noFail)
  }
  return(get("dku_remoterun__cached_env_def", envir = .GlobalEnv))
}

dku_remoterun__get_dku_env_vars <- function(noFail=TRUE) {
    envDef <- dku_remoterun__get_remote_run_env_def(noFail)
    if ("env" %in% names(envDef)) {
      return(envDef$env)
      } else {
        return(list())
    }
}

dku_remoterun__get_env_var <- function(k, unset=NULL, noFail=TRUE) {
  dkuVars <- dku_remoterun__get_dku_env_vars(noFail)
  if (k %in% names(dkuVars)) {
    return(dkuVars[[k]])
    } else if (k %in% names(Sys.getenv())) {
      return(Sys.getenv(k))
      } else if (!missing(unset)) {
        return(unset)
        } else {
        return("") # like Sys.getenv()
    }
}

########### Interaction with ~/.dataiku/config.json for persistence of settings


dku_configfile__read <- function() {
    cfgFile <- "~/.dataiku/config.json"
    if(file.exists(cfgFile)){
        json <- readChar(cfgFile, file.info(cfgFile)$size)
        return(RJSONIO::fromJSON(json))
    } else {
        print("No config file")
        print(cfgFile)
        ret <- list()
        ret$dss_instances <- list()
        return(ret)
    }
}

dku_configfile__write <- function(cfg) {
    cfgFile <- "~/.dataiku/config.json"
    val <- RJSONIO::toJSON(cfg, pretty=TRUE)
    if (is.null(val) || length(val) == 0) {
        print(paste("Empty value, don't write in", fileName))
    }
    if (!dir.exists("~/.dataiku")) {
        dir.create("~/.dataiku", recursive=TRUE)
    }
    writeLines(val, cfgFile)
}

dku_configfile__get_instance <- function(instance_name=NULL) {
    config <- dku_configfile__read()

    if (is.null(instance_name)) {
        instance_name <- config$default_instance
    }
    if (is.null(instance_name)) {
        stop("Select a DSS instance to connect to")
    }
    if (!(instance_name %in% names(config$dss_instances))) {
        stop(paste("DSS instance is not in config", instance_name))
    }
    config$dss_instances[[instance_name]]
}


########### Networking and Intercom

dku_intercom__clear_location_info <- function(){
    remove("dku_intercom__cached_location_data", envir = .GlobalEnv)
}

dku_intercom__get_location_info <- function() {
    if (exists("dku_intercom__cached_location_data", envir = .GlobalEnv)) {
        return(get("dku_intercom__cached_location_data", envir = .GlobalEnv))
    }

    api_ticket <- dku_remoterun__get_env_var("DKU_API_TICKET", unset=NA)

    if (!is.na(api_ticket)) {
        # We have an API ticket so we are in DSS

        x = list(auth_mode= "TICKET", api_ticket=api_ticket)

        x$backend_url = paste0(dku_remoterun__get_env_var("DKU_BACKEND_PROTOCOL", unset="http"),
              "://",
              dku_remoterun__get_env_var("DKU_BACKEND_HOST", unset="127.0.0.1"),
              ":",
              dku_remoterun__get_env_var("DKU_BACKEND_PORT"))

        if (Sys.getenv("DKU_SERVER_KIND", unset="BACKEND") == "BACKEND") {
            x$has_a_jek = FALSE

        } else {
            x$has_a_jek = TRUE
            x$jek_url = paste0(dku_remoterun__get_env_var("DKU_BACKEND_PROTOCOL", unset="http"),
              "://",
              dku_remoterun__get_env_var("DKU_SERVER_HOST", unset="127.0.0.1"),
              ":",
              dku_remoterun__get_env_var("DKU_SERVER_PORT"))
        }

        if (dku_remoterun__get_env_var("DKU_BACKEND_PROTOCOL", unset="http") == "https") {
            #print("Use HTTPS with custom cert")
            tmpFilePath <- tempfile(pattern='r-ssl-cert-')
            cert <- dku_remoterun__get_env_var("DKU_SERVER_CERT", unset="")

            if (startsWith(cert, "b64:")) {
                cert <- rawToChar(jsonlite::base64_dec(substr(cert, 5, 10000000)))
            }

            write(cert, tmpFilePath)

            x$custom_cert = TRUE
            x$curlopt_cainfo = tmpFilePath
        } else {
            x$custom_cert = FALSE
        }

        assign("dku_intercom__cached_location_data", x, envir = .GlobalEnv)
    } else {
        # No API ticket so we are running outside of DSS, start the dance to find remote DSS authentication
        # info
        # In that order:
        #   - dkuSetRemoteDSS (has been handled at the top of this method)
        #   - Environment variables DKU_DSS_URL, DKU_API_KEY, DKU_NO_CHECK_CERTIFICATE
        #   - ~/.dataiku/config.json (with optional DKU_DEFAULT_INSTANCE environment variable to set the default instance)
        if (!is.na(Sys.getenv("DKU_DSS_URL", unset=NA))) {
            noCheckCertificateEnvVar = Sys.getenv("DKU_NO_CHECK_CERTIFICATE", unset="false")
            ncc = !is.na(noCheckCertificateEnvVar) & noCheckCertificateEnvVar != "" & tolower(noCheckCertificateEnvVar) != "false"
            dkuSetRemoteDSS(Sys.getenv("DKU_DSS_URL"), Sys.getenv("DKU_API_KEY"), ncc)
        } else {
            defaultInstanceEnvVar <- Sys.getenv("DKU_DEFAULT_INSTANCE", unset=NA)
            if (is.na(defaultInstanceEnvVar)) {
                instance <- dku_configfile__get_instance()
            } else {
                instance <- dku_configfile__get_instance(defaultInstanceEnvVar)
            }
            ncc = ("no_check_certificate" %in% names(instance) && instance$no_check_certificate)
            dkuSetRemoteDSS(instance["url"], instance["api_key"], no_check_certificate=ncc)
        }
    }
    return(get("dku_intercom__cached_location_data", envir = .GlobalEnv))
}

dkuImpersonateShinyCalls <- function(request, block) {
    Sys.setenv(DKU_IMPERSONATE_CALLS="TRUE")
    assign("dku_shiny_headers", request, envir = .GlobalEnv)
    tryCatch(block, finally={
        Sys.unsetenv("DKU_IMPERSONATE_CALLS")
        assign("dku_shiny_headers", NULL, envir = .GlobalEnv)
    })
}

dku_intercom__get_auth_headers_base <- function() {
    location_data <- dku_intercom__get_location_info()
    call_origin <- dku_remoterun__get_env_var("DKU_CALL_ORIGIN")

    header_list = c()
    if (location_data$auth_mode == "TICKET") {
        header_list["X-DKU-APITicket"] = location_data$api_ticket
        header_list["X-DKU-CallOrigin"] = call_origin
      } else {
        library(base64enc)
        b64 <- base64encode(charToRaw(paste0(location_data$api_key, ":", sep="")))
        header_list["Authorization"] = paste0("Basic ", b64, sep="")
    }
    return(header_list)
}

dku_intercom__get_httr_config <- function() {
    location_data <- dku_intercom__get_location_info()

    # R added a timeout of 10 mins for IDLE connections https://github.com/jeroen/curl/commit/f3e958fe973a1e460310db802e3266e45a8cd6a5 in curl
    # This fails on long jobs with DSS, set it back with LOW_SPEED_TIME=0
    if (location_data$custom_cert) {
        #print("SET A CAINFO")
        #print(location_data$curlopt_cainfo)
        return(config(cainfo = location_data$curlopt_cainfo, LOW_SPEED_TIME=0))
    } else{
        return(config(LOW_SPEED_TIME=0))
    }
}

dku_intercom__get_auth_headers <- function() {
    header_list = dku_intercom__get_auth_headers_base()
    
    if ("DKU_IMPERSONATE_CALLS" %in% names(Sys.getenv()) && tolower(Sys.getenv("DKU_IMPERSONATE_CALLS")) == "true") {
        # in a Shiny webapp, the user should offer the headers via dku_intercom__set_shiny_headers
        if (exists("dku_shiny_headers", envir = .GlobalEnv)) {
            shiny_headers <- get("dku_shiny_headers", envir = .GlobalEnv)
            tryCatch({
                # keep only the cookie
                cookie <- shiny_headers$HTTP_COOKIE
                # check cache
                if (exists("dku_intercom__cached_cookie_to_ticket", envir = .GlobalEnv)) {
                    cookie_to_ticket <- get("dku_intercom__cached_cookie_to_ticket", envir = .GlobalEnv)
                } else {
                    cookie_to_ticket <- list()
                }
                if (cookie %in% names(cookie_to_ticket)) {
                    header_list["X-DKU-APITicket"] = cookie_to_ticket[[cookie]]
                } else {
                    clean_headers <- list()
                    clean_headers$Cookie <- cookie
                    # ask the backend for the user
                    ticket <- dkuGetTicketFromBrowserHeaders(clean_headers)
                    header_list["X-DKU-APITicket"] = ticket
                    cookie_to_ticket[cookie] = ticket
                    assign("dku_intercom__cached_cookie_to_ticket", cookie_to_ticket, envir = .GlobalEnv)
                }
            }, error=function(e) {
                print(e)
            });
        }
    }
    
    return(add_headers(.headers = header_list))
}

# Backwards compat with some people who may have been using that
dku__get_auth_headers <-function(){
    dku_intercom__get_auth_headers()
}

dku_intercom__get_backend_url <- function(path) {
  prefix = "/dip/api/tintercom"
  location_data <- dku_intercom__get_location_info()
  return(paste0(location_data$backend_url, prefix, path))
}

dku_intercom__get_jek_url <- function(path) {
    prefix = "/kernel/tintercom"
    location_data <- dku_intercom__get_location_info()
    return(paste0(location_data$jek_url, prefix, path))
}

dku_intercom__get_jek_or_backend_url <- function(path) {
    location_data <- dku_intercom__get_location_info()
    if (location_data$has_a_jek) {
        return(dku_intercom__get_jek_url(path))
    } else {
        return(dku_intercom__get_backend_url(path))
    }
}

dku__check_api_error <- function(resp, message) {
    if (resp$status_code != 200) {
        error_head <- paste0(message, " (HTTP code ", resp$status_code, "): ");
        resp_content_type <- headers(resp)$`content-type`
        if (resp_content_type == "application/json"){
            resp_content = content(resp)
            if ("detailedMessage" %in% names(resp_content)) {
                error <- paste0(error_head, resp_content$detailedMessage);
            } else {
                error <- paste0(error_head, resp_content$message);
            }
        } else {
            error <- paste0(error_head, content(resp));
        }
        stop(error);
    }
}

dku__check_curl_api_error <- function(handle, con, message) {
    data <- curl::handle_data(handle)
    if (data$status_code != 200) {
      print(paste("Handling HTTP error, code", data$status_code))
      error_head <- paste0(message, " (HTTP code ", data$status_code, "): ");

      # print("Handle data")
      # print(data)

      parsed_headers <- curl:::parse_headers_list(data$headers)
      # print("parsed headers")
      # print(parsed_headers)

      resp_content_type <- parsed_headers$`content-type`
      open(con)
      raw_content <- paste(readLines(con), sep="\n")
      close(con)
      
      # print("raw_content")
      # print(raw_content)
        
      if (resp_content_type == "application/json"){
        parsed_content <- jsonlite::fromJSON(raw_content)
        if ("detailedMessage" %in% names(parsed_content)) {
            error <- paste0(error_head, parsed_content$detailedMessage);
        } else {
            error <- paste0(error_head, parsed_content$message);
        }
        } else {
          error <- paste0(error_head, raw_content);
      }
      stop(error);
  }
}

######### Shiny webapp instrumentation

decay_rate <- 0.138629436111989 # halved every 5s
dku__calc_decay <- function(duration) {
    return(exp(-decay_rate * duration))
}

dku__apply_decay <- function(values, duration, now) {
    decay <- dku__calc_decay(duration)
    decayed <- list(usage=NULL, rate=NULL, users=list())
    if (!is.element("usage", names(values)) || is.null(values$usage)) {
        decayed$usage <- 0
    } else {
        decayed$usage <- values$usage * decay
    }
    if (!is.element("rate", names(values)) || is.null(values$rate)) {
        decayed$rate <- 0
    } else {
        decayed$rate <- values$rate * decay
    }
    if (!is.element("users", names(values)) || is.null(values$users)) {
        decayed$users <- list()
    } else {
        users <- list()
        for (user in names(values$users)) {
            if (now - values$users[[user]] <= 300) {# expire after 5min
                users[user] <- values$users[[user]]
            }
        }
        decayed$users <- users
    }
    return(decayed)
}

dku__write_out_metrics <- function(values) {
    metrics_folder <- "./.metrics"
    metrics_file_path <- paste0(metrics_folder, "/calls-", Sys.getpid())
    if (!dir.exists(metrics_folder)) {
        dir.create(metrics_folder)
    }
    lock <- filelock::lock(metrics_file_path)
    tryCatch({
        val <- RJSONIO::toJSON(values, pretty=TRUE, digits = 14) # digits is important otherwise too much precision is lost
        writeLines(val, metrics_file_path)
    }, finally = {
        filelock::unlock(lock)
    })
}

dkuReportCall <- function(started, session_id) {
    if (!exists("dku_shiny_metrics", envir = .GlobalEnv)) {
        assign("dku_shiny_metrics", list(), envir = .GlobalEnv)
    }
    values <- get("dku_shiny_metrics", envir = .GlobalEnv)
    
    now <- as.numeric(Sys.time())
    if (!is.element("last", names(values)) || is.null(values$last)) {
        previous <- 0
    } else {
        previous <- values$last 
    }
    
    increased <- dku__apply_decay(values, now - previous, now)
    increased$rate <- increased$rate + 1
    increased$usage <- increased$usage + (1 - exp(-decay_rate * (now-started))) / decay_rate
    increased$last <- now
    if (!is.null(session_id)) {
        increased$users[session_id] <- started
    }
    assign("dku_shiny_metrics", increased, envir = .GlobalEnv)
    dku__write_out_metrics(increased)
}


########### Misc

dku__resolve_smart_name <- function(smartName) {
    if(is.null(smartName)) {
        stop("Invalid dataset name: NULL")
    }
    defaultProject <- dku_remoterun__get_env_var("DKU_CURRENT_PROJECT_KEY")
    splitted = strsplit(smartName,"\\.")
    if (lapply(splitted,length)==2) {
        return(smartName)
    } else {
        if(defaultProject=="" || is.null(defaultProject)) {
            stop(paste("Default project is undefined. Unable to resolve dataset name:",smartName))
        }
        return(paste0(defaultProject,".",smartName))
    }
}

dku__ref_to_name <- function(fullName) {
    return (unlist(strsplit(fullName,"\\."))[2])
}

dku__ref_to_pkey <- function(fullName) {
    return (unlist(strsplit(fullName,"\\."))[1])
}

dku__convert_logical <- function(x){
    if(is.character(x)){
        if(all(sample(x, 20, replace=T) %in% c("true", "false", "", NA))){
            if(all(x %in% c("true", "false", NA))){
                x == "true"
            } else if(all(x %in% c("true", "false", ""))){
                y = x == "true"
                y[x==""] = NA
                y
            } else x
        } else x
    } else x
}

dku__convert_iso8601date <- function(x) {
    # as.Date() sux and loses the time information
    return (as.POSIXlt(x, format="%Y-%m-%dT%H:%M:%OSZ", tz="UTC"))
}

dku__convert_type <- function(x) {
    conv <- function(t) {
        switch(t, tinyint="integer",smallint="integer",int="integer",bigint="integer",float="numeric",double="numeric",boolean="logical",date="character","character")
    }
    return (conv(x))
}

dku__convert_schema_to_colClasses <- function(schema, columns) {
    return (sapply(schema$columns, function(c) dku__convert_type(c$type)))
}

dku__fixup_dates_for_schema <- function(df, schema) {
    options(digits.secs = 3) # because good luck having '%OS3' to work, despite being advertised on R doc as the solution
    for (c in schema$columns) {
        if (c$type == "date") {
            df[[c$name]] <- dku__convert_iso8601date(df[,c$name])
        }
    }
    return (df)
}

dkuSourceLibR <- function(f, local = FALSE, verbose = getOption("verbose"), chdir = FALSE, encoding = getOption("encoding")) {
	paths <- NA
	print("SEARCHING FOR ")
	print(f)
	print("IN")
	print(Sys.getenv("DKU_SOURCE_LIB_R_PATH"))
    if (!is.na(dku_remoterun__get_env_var("DKU_SOURCE_LIB_R_PATH", unset=NA))) {
        paths <- strsplit(dku_remoterun__get_env_var("DKU_SOURCE_LIB_R_PATH"), ":");
        for (path in paths[[1]]) {
            if (length(path) == 0 || nchar(path) == 0) {
                next
            }
            file_path <- file.path(path, f)
            if (file.exists(file_path)) {
                print(paste("Loading source file", file_path))
                source(file_path, local=local, verbose=verbose, chdir=chdir, encoding=encoding)
                return()
            }
        }
    }
    # Not found in project libs, search in global lib/R (and also cwd because why not)
    libr <- file.path(dku_remoterun__get_env_var("DIP_HOME"), "lib", "R")
    if (file.exists(file.path(libr, f))) {
        print(paste("Loading source file", file.path(libr, f)))
        source(file.path(libr, f), local=local, verbose=verbose, chdir=chdir, encoding=encoding);
    } else if (file.exists(f)) {
        print(paste("Loading source file", f))
        source(f, local=local, verbose=verbose, chdir=chdir, encoding=encoding);
    } else {
        stop(paste("Sourced R file not found: ", f, " searched in ", paths, "and lib/R"))
    }
}

getDkuFlowSpec <- function() {
    envDef <- dku_remoterun__get_remote_run_env_def(noFail)
    if ("flowSpec" %in% names(envDef)) {
        return(envDef$flowSpec)
    } else {
        rSpec <- dku_remoterun__get_env_var("DKUFLOW_SPEC")

        # the DKUFLOW_SPEC should not arrive here anymore, unless you start R from a shell recipe 
        # and even then the remote run env def will have the flowSpec field
        # Notebook mode
        if (is.null(rSpec) || rSpec == "") {
            return(NULL)
        } else {
            # Flow mode
            return(RJSONIO::fromJSON(rSpec))
        }
    }
}
