#' Displays a googleVis chart in the Jupyter notebook.
#
#' @param gvis: the googleVis chart.
#' @param height: Optional height for the figure
#' @return Nothing
#'
#' @details
#'
#' See https://doc.dataiku.com/dss/latest/R/googlevis.html for more information
#'
dkuDisplayGooglevis <- function(gvis, height=400) {
    IRdisplay::display_html(repr::repr_text(gvis))
    IRdisplay::display_html(paste0("<script>$('.box-flex1').css('height', '", height, "px')</script>"))
}

#' Installs required frontend-side dependencies for displaying
#' ggvis charts in the Jupyter notebook
#' @details
#'
#' See https://doc.dataiku.com/dss/latest/R/ggvis.html for more information
#'
dkuInstallGgvisDependenciesOnce <- function(){
    gg <- mtcars %>% ggvis(~wt, ~mpg)
    dir <- paste0(dku_remoterun__get_env_var("DIP_HOME"), "/local/static/rcharts-support/ggvis/")
    dir.create(dir, recursive=TRUE)
    
    oldwd <- setwd(dir)
    on.exit(setwd(oldwd), add = TRUE)

    rendered <- htmltools:::renderTags(gg %>% view_static())

    lapply(rendered$dependencies, function(dep) {
        htmltools:::copyDependencyToDir(dep, "lib", FALSE)
    })
}

dkuGgvisToHTML <- function(gg) {
    dir <- "/local/static/rcharts-support/ggvis/lib/"

    rendered <- htmltools:::renderTags(gg %>% view_static())
    deps <- lapply(rendered$dependencies, function(dep) {
        dep$src$file <- paste0(dir, dep$name, "-", dep$version)
        dep                                     
    })

    paste(c("<!DOCTYPE html>",
            "<html>",
            "<head>",
            "<meta charset=\"utf-8\"/>",
            htmltools:::renderDependencies(deps, c("href", "file")),
            rendered$head,
            "</head><body>",
            rendered$html,
            "</body>",
            "</html>"), collapse="")
}

#' Displays a ggvis chart in the Jupyter notebook.
#
#' @param gvis: the ggvis chart.
#' @param height: Optional height for the figure
#' @return Nothing
#'
#' See https://doc.dataiku.com/dss/latest/R/ggvis.html for more information
#'
dkuDisplayGgvis <- function(gg, height=400) {
    IRdisplay::display_html(dkuGgvisToHTML(gg))
    IRdisplay::display_html(paste0("<script>$('.box-flex1').css('height', '", height, "px')</script>"))
}

#' Displays a dygraph chart in the Jupyter notebook.
#
#' @param dygraph: the dygraph chart.
#' @param height: Optional height for the figure
#' @return Nothing
#'
#' See https://doc.dataiku.com/dss/latest/R/dygraphs.html for more information
#'
dkuDisplayDygraph <- function(dygraph, height=400) {
    IRdisplay::display_html(repr::repr_html(dygraph))
    IRdisplay::display_html(paste0("<script>$('.box-flex1').css('height', '", height, "px')</script>"))   
}