/**
 * This function appends a hash to DSS asset URLs (myfile.json -> myfile.hashed.XXX.json).
 * The hash is typically generated randomly every time DSS is built. Nginx is configured to recognize
 * this URL pattern and reponds with headers allowing long-term caching, so that the browser does not 
 * ever need to re-check if asset files have changed.
 *
 * This function is used **both at build time and runtime**. Normally it should only be used on path to
 * DSS assets but it has best-effort checks to avoid mangling other URLs if misused.
 *
 * /!\ Keep in sync with rules in nginx_config.py /!\
 */
globalThis._dkuAppendHashToPath = (filePath, hash) => {
    if (!hash || typeof filePath != 'string'
        // Reject non-DSS-asset paths
        || !/^\s*(\.?\/)?([^/]+$|static|ngxdist|bower_components|templates|html_app)($|\/)/.test(filePath)
        // Reject full URLs, already-hashed paths (for idempotence) and paths containing query strings
        || /(?:^\s*https?:|^\s*data:|\.hashed\.[0-9a-zA-Z-_]+\.[^\\.]+\s*$|\?.*)/.test(filePath)
        // Reject unsupported file extensions
        || !/\.(html|css|js|json|svg|jpg|png|ico|woff|woff2)$/.test(filePath)
    ) {
        return filePath;
    }
    const base = filePath.slice(0, filePath.lastIndexOf("."));
    const ext = filePath.slice(filePath.lastIndexOf("."));
    return base + ".hashed." + hash + ext;
};
