(function() {
'use strict';

var app = angular.module('dataiku.admin.codeenvs.design', []);

app.controller("AdminCodeEnvsDesignController", function($scope, $rootScope, TopNav, DataikuAPI, Dialogs, FutureProgressModal, CreateModalFromTemplate, ActivityIndicator, Deprecation, CodeEnvsDesignService) {
    $scope.canCreateCodeEnv = $rootScope.mayCreateCodeEnvs;

    $scope.openDeleteEnvModal = function(codeEnv){
        var newScope = $scope.$new();
        newScope.codeEnv = codeEnv;
        // modal appears when usages are ready
        CreateModalFromTemplate("/templates/admin/code-envs/common/delete-env-modal.html", newScope, "AdminCodeEnvsDesignDeleteController");
    }
    $scope.isExportable = function(codeEnv) {
        return codeEnv && ['PLUGIN_MANAGED', 'PLUGIN_NON_MANAGED', 'DSS_INTERNAL'].indexOf(codeEnv.deploymentMode) < 0;
    };
    $scope.exportEnv = function(envLang, envName) {
        ActivityIndicator.success("Exporting code env ...");
        downloadURL(DataikuAPI.admin.codeenvs.design.getExportURL(envLang, envName));
    };

    $scope.getEnvDiagnostic = function(envLang, envName) {
        ActivityIndicator.success("Generating code env diagnostic ...");
        downloadURL(DataikuAPI.admin.codeenvs.design.getDiagnosticURL(envLang, envName));
    };

    $scope.isPythonDeprecated = function (pythonInterpreter) {
        return Deprecation.isPythonDeprecated(pythonInterpreter);
    };

    $scope.openCRANEditHelp = function() {Dialogs.ackMarkdown($scope, "R packages", CodeEnvsDesignService.CRANEditHelpString);}
    $scope.openRCondaSpecEditHelp = function() {Dialogs.ackMarkdown($scope, "Conda packages", CodeEnvsDesignService.rCondaSpecEditHelpString);}
    $scope.openPipRequirementsEditHelp = function() {Dialogs.ackMarkdown($scope, "Pip requirements", CodeEnvsDesignService.pipRequirementsEditHelpString);}
    $scope.openPyCondaSpecEditHelp = function() {Dialogs.ackMarkdown($scope, "Conda packages", CodeEnvsDesignService.pyCondaSpecEditHelpString);}
});

app.controller("AdminCodeEnvsDesignListController", function($scope, $controller, CodeEnvsDesignService, TopNav, DataikuAPI, Dialogs, CreateModalFromTemplate, $state, RequestCenterService, CodeEnvService) {
    $controller("AdminCodeEnvsDesignController", {$scope:$scope});
    TopNav.setLocation(TopNav.DSS_HOME, "administration");
    $scope.refreshList = function() {
        return DataikuAPI.admin.codeenvs.design.list().success(function(data) {
            for (const codeEnv of data) {
                codeEnv["languageDisplayName"] = codeEnv["pythonInterpreter"] || codeEnv["envLang"]
            }
            $scope.codeEnvs = data;
        }).error(setErrorInScope.bind($scope));
    };

    CodeEnvsDesignService.getUserAccessibleSettings().then(({data}) => {
        $scope.enableCodeEnvResources = data.enableCodeEnvResources;
        $scope.useConda = data.useConda;
    });

    $scope.refreshList();

    $scope.canRequestCodeEnvInstall = $scope.appConfig.codeEnvInstallRequestsEnabled;

    $scope.isAdmin = $scope.appConfig.admin;

    $scope.openNewPythonEnvModal = function(){
        CreateModalFromTemplate("/templates/admin/code-envs/design/new-python-env-modal.html", $scope, "AdminCodeEnvsDesignNewPythonController")
    }

    $scope.openNewREnvModal = function(){
        CreateModalFromTemplate("/templates/admin/code-envs/design/new-R-env-modal.html", $scope, "AdminCodeEnvsDesignNewRController")
    }
    $scope.openImportEnvModal = function(){
        CreateModalFromTemplate("/templates/admin/code-envs/design/import-env-modal.html", $scope, "AdminCodeEnvsDesignImportController")
    }
    $scope.actionAfterDeletion = function() {
        $scope.refreshList();
    };
    $scope.goToEditIfExists = function(envName) {
        const env = $scope.codeEnvs.find(e => e.envName === envName);
        if(env && env.envLang === 'R') {
            $state.go("admin.codeenvs-design.r-edit", { envName });
        } else if(env && env.envLang === 'PYTHON'){
            $state.go("admin.codeenvs-design.python-edit", { envName });
        }
    };

    $scope.openNewEnvRequestModal = function(envLang) {
        if (envLang === "PYTHON") {
            $scope.requestedEnv = CodeEnvsDesignService.getDefaultPythonEnv();
        } else if (envLang === "R") {
            $scope.requestedEnv = CodeEnvsDesignService.getDefaultREnv();
        }

        $scope.requestedEnv.conda = $scope.useConda;

        $scope.refreshList();

        $scope.isCodeEnvNameUnique = function(value){
            if ($scope.codeEnvs) {
               for(let k in $scope.codeEnvs) {
                  if($scope.codeEnvs[k].envName === value) {
                     return false;
                  }
               }
            }
            return true;
        }

        $scope.codeEnvModal = {$modalTab: 'GENERAL'};
        $scope.nextTab = function() {
            if ($scope.codeEnvModal.$modalTab==="GENERAL") {$scope.codeEnvModal.$modalTab="PACKAGES";}
        }

        $scope.openCRANEditHelp = function() {Dialogs.ackMarkdown($scope, "R packages", CodeEnvsDesignService.CRANEditHelpString);}
        $scope.openRCondaSpecEditHelp = function() {Dialogs.ackMarkdown($scope, "Conda packages", CodeEnvsDesignService.rCondaSpecEditHelpString);}
        $scope.openPipRequirementsEditHelp = function() {Dialogs.ackMarkdown($scope, "Pip requirements", CodeEnvsDesignService.pipRequirementsEditHelpString);}
        $scope.openPyCondaSpecEditHelp = function() {Dialogs.ackMarkdown($scope, "Conda packages", CodeEnvsDesignService.pyCondaSpecEditHelpString);}

        $scope.codeEnvResourcesEditorOptions = $scope.codeMirrorSettingService.get("text/x-python");
        $scope.specPackageListEditorOptionsPip = $scope.codeMirrorSettingService.get("text/plain", {onLoad: function(cm){$scope.codeMirrorPip = cm;}});
        $scope.specPackageListEditorOptionsConda = $scope.codeMirrorSettingService.get("text/plain", {onLoad: function(cm){$scope.codeMirrorConda = cm;}});
        $scope.updateModalPackages = function() {
            // Fetch mandatory packages list
            DataikuAPI.codeenvs.getMandatoryPackages(envLang, $scope.requestedEnv).then(({data}) => {
                $scope.requestedEnv.mandatoryPackageList = data;
            });
            //Fetch package presets
            if (envLang == "PYTHON") {
                CodeEnvsDesignService.fetchPackagePresets({desc: $scope.requestedEnv}, $scope)
            }
        }

        $scope.requestedEnv.mandatoryPackageList = $scope.updateModalPackages();
        $scope.requestedEnv.specPackageList = "";
        $scope.requestedEnv.specCondaEnvironment = "";

        $scope.insertAdditionalPackages = (editorType) => {
            const newScope = $scope.$new();

            newScope.packageListTypes = $scope.packageListTypes;
            newScope.packageListValues = $scope.packageListValues;
            newScope.selectedPackages = $scope.packageListTypes[0][0];
            newScope.codeMirrorPip = $scope.codeMirrorPip;
            newScope.codeMirrorConda = $scope.codeMirrorConda;
            const editorSettings = $scope.codeMirrorSettingService.get('text/plain');
            newScope.insertReadOnlyOptions = {
                ...editorSettings,
                readOnly: true,
                lineNumbers: false,
                foldGutter: false,
            };

            CreateModalFromTemplate("/templates/admin/code-envs/design/add-additional-packages-modal.html", newScope, null, (scope) => {
                // We must define this function here (rather than in newScope
                // itself) because scope.dismiss() is not available at this stage.
                // It is added when the actual modal scope is built from newScope.
                scope.insertPackages = () => {
                    const selectedContent = scope.packageListValues[scope.selectedPackages];
                    CodeEnvsDesignService.replacePackageListEditorContent(selectedContent, editorType, newScope.codeMirrorPip, newScope.codeMirrorConda);
                    scope.dismiss();
                };
            });
        };

        if (envLang === "PYTHON") {
            CreateModalFromTemplate("/templates/admin/code-envs/design/new-python-env-request-modal.html", $scope, "AdminCodeEnvsDesignNewPythonController", (modalScope) => {
                   modalScope.requestedEnv = $scope.requestedEnv;

                   modalScope.$watch('[requestedEnv.pythonInterpreter, requestedEnv.installCorePackages, requestedEnv.corePackagesSet, requestedEnv.installJupyterSupport]',function(nv, ov) {
                        $scope.updateModalPackages();
                   })
                   modalScope.cancelRequest = function() {
                       modalScope.$destroy();
                       modalScope.dismiss();
                   }
                   modalScope.sendRequest = function() {
                        DataikuAPI.requests.createCodeEnvRequest("INSTALL_CODE_ENV", $scope.requestedEnv.envName, envLang, JSON.stringify($scope.requestedEnv) ,$scope.requestedEnv.specPackageList, $scope.requestedEnv.specCondaEnvironment, $scope.codeEnvModal.message, 'MANUAL').success(function(data){
                            RequestCenterService.WT1Events.onRequestSent("CODE_ENV", null, $scope.requestedEnv.envName, $scope.codeEnvModal.message, data.id);
                        }).error(setErrorInScope.bind($scope));
                        modalScope.$destroy();
                        modalScope.dismiss();
                   }

                   modalScope.updateDefaultCorePackageSet = function() {
                        DataikuAPI.codeenvs.getDefaultCorePackageSet($scope.requestedEnv.pythonInterpreter).success(function(data){
                            $scope.requestedEnv.corePackagesSet = data;
                        }).error(setErrorInScope.bind($scope));
                   }

                   modalScope.updatePythonInterpreters = function(){
                        if (modalScope.requestedEnv.conda === true) {
                            modalScope.requestPythonInterpreters = CodeEnvService.pythonCondaInterpreters;
                        } else {
                            modalScope.requestPythonInterpreters = CodeEnvService.pythonInterpreters.filter(i => i[0] !== "CUSTOM");
                            CodeEnvService.getPythonInterpreters().then((interpreters) => {modalScope.requestPythonInterpreters = interpreters.filter(i => i[0] !== "CUSTOM")});
                        }
                   }
                   modalScope.updatePythonInterpreters();
               });
        } else if (envLang === "R") {
            CreateModalFromTemplate("/templates/admin/code-envs/design/new-R-env-request-modal.html", $scope, "AdminCodeEnvsDesignNewRController", (modalScope) => {
                modalScope.requestedEnv = $scope.requestedEnv;
                modalScope.$watch('[requestedEnv.installCorePackages, requestedEnv.corePackagesSet, requestedEnv.installJupyterSupport]',function(nv, ov) {
                    $scope.updateModalPackages();
                })
                modalScope.cancelRequest = function() {
                    modalScope.$destroy();
                    modalScope.dismiss();
                }
                modalScope.sendRequest = function() {
                    DataikuAPI.requests.createCodeEnvRequest("INSTALL_CODE_ENV", $scope.requestedEnv.envName, envLang, JSON.stringify($scope.requestedEnv) ,$scope.requestedEnv.specPackageList, $scope.requestedEnv.specCondaEnvironment, $scope.codeEnvModal.message, 'MANUAL').success(function(data){
                        RequestCenterService.WT1Events.onRequestSent("CODE_ENV", null, $scope.requestedEnv.envName, $scope.codeEnvModal.message, data.id);
                    }).error(setErrorInScope.bind($scope));
                    modalScope.$destroy();
                    modalScope.dismiss();
                }
            });
        }
    }
});

app.controller("AdminCodeEnvsDesignDeleteController", function($scope, TopNav, DataikuAPI, Dialogs, FutureProgressModal, $q) {
    $scope.delete = function() {
        var parentScope = $scope.$parent;
        DataikuAPI.admin.codeenvs.design.delete($scope.codeEnv.envLang, $scope.codeEnv.envName).success(function(data){
            $scope.dismiss();
            FutureProgressModal.show(parentScope, data, "Env deletion").then(function(result){
                const infoModalClosed = result
                    ? Dialogs.infoMessagesDisplayOnly(parentScope, "Deletion result", result.messages, result.futureLog)
                    : $q.resolve();
                infoModalClosed.then(() => $scope.actionAfterDeletion());
            });
        }).error(setErrorInScope.bind($scope));

    };
});

app.controller("AdminCodeEnvsDesignChangePythonSuccessfulController", function($scope, $state, CodeEnvService, $stateParams) {
    const parentScope = $scope.$parent.$parent;
    $scope.goToEnv = function(envName) {
         $state.go("admin.codeenvs-design.python-edit", {envName});
    }

    $scope.installPackages = function (envName) {
        CodeEnvService.updateEnv(parentScope, envName, true, false, false, "PYTHON", function() {
            if ($stateParams.envName != envName) {
                $scope.goToEnv(envName);
            } else {
                $scope.uiState.active = 'actual';
            }
        })
    }
});

app.controller("AdminCodeEnvsDesignChangePythonController", function($scope, TopNav, DataikuAPI, Dialogs, FutureProgressModal, $q, $state, CreateModalFromTemplate, CodeEnvService, Deprecation) {
    $scope.inplace = true;
    $scope.supportedDeprecatedPythons = ['PYTHON36', 'PYTHON37', 'PYTHON38'];

    $scope.pythonInterpreters = CodeEnvService.pythonInterpreters;
    CodeEnvService.getPythonInterpreters().then((interpreters) => {$scope.pythonInterpreters = interpreters});
    $scope.insertReadOnlyOptions = {
        ...$scope.codeMirrorSettingService.get('text/plain'),
        readOnly: true,
        lineNumbers: true,
        foldGutter: true,
    };
    $scope.insertReadWriteOptions = {
        ...$scope.codeMirrorSettingService.get('text/plain'),
        lineNumbers: true,
        foldGutter: true,
    };

    const isDeprecatedPythonInHooks = function(hooks, pythonInterpreter) {
        return $scope.supportedDeprecatedPythons.includes(pythonInterpreter) && hooks.some(hook => hook.type === pythonInterpreter + "_SUPPORT");
    };

    const parentScope = $scope.$parent.$parent;
    $scope.$watch("newEnv.pythonInterpreter", function(nv) {
        if (!$scope.nameChangedByUser) {
            $scope.newEnv.envName = $scope.oldEnv.envName + "_" + $scope.newEnv.pythonInterpreter.toLowerCase();
        }

        $scope.newEnv.pythonInterpreterFriendlyName = CodeEnvService.pythonInterpreterFriendlyName($scope.newEnv.pythonInterpreter);
        DataikuAPI.admin.codeenvs.design.updatePythonPackagePresets($scope.newEnv.specPackageList, $scope.newEnv.pythonInterpreter).success(function(data) {
            $scope.newEnv.specPackageList = data;
        }).error(setErrorInScope.bind($scope));
        if (!['PYTHON27', 'PYTHON35', 'PYTHON36', 'PYTHON37', 'CUSTOM'].includes($scope.newEnv.pythonInterpreter) && $scope.newEnv.corePackagesSet == "LEGACY_PANDAS023") {
            $scope.previousCorePackagesSet = $scope.newEnv.corePackagesSet;
            $scope.newEnv.corePackagesSet = "PANDAS13";
        } else if ($scope.newEnv.pythonInterpreter == 'PYTHON27') {
            $scope.previousCorePackagesSet = $scope.newEnv.corePackagesSet;
            $scope.newEnv.corePackagesSet = "LEGACY_PANDAS023";
        }

        $scope.oldEnvContainsDeprecatedPythonCRA = function(newPythonInterpreter) {
            if (!$scope.oldEnv || !$scope.oldEnv.desc ||
                    !$scope.oldEnv.desc.predefinedContainerHooks ||
                    !newPythonInterpreter ||
                    !$scope.supportedDeprecatedPythons.includes($scope.oldEnv.desc.pythonInterpreter)) {
                return false;
            }

            if ($scope.supportedDeprecatedPythons.includes(newPythonInterpreter)) {
                return isDeprecatedPythonInHooks($scope.oldEnv.desc.predefinedContainerHooks, $scope.oldEnv.desc.pythonInterpreter) &&
                    !isDeprecatedPythonInHooks($scope.oldEnv.desc.predefinedContainerHooks, newPythonInterpreter);
            } else if (!$scope.supportedDeprecatedPythons.includes(newPythonInterpreter)) {
                return isDeprecatedPythonInHooks($scope.oldEnv.desc.predefinedContainerHooks, "PYTHON36") ||
                    isDeprecatedPythonInHooks($scope.oldEnv.desc.predefinedContainerHooks, "PYTHON37") ||
                    isDeprecatedPythonInHooks($scope.oldEnv.desc.predefinedContainerHooks, "PYTHON38");
            }
        };

        $scope.newEnvRequiresDeprecatedPythonCRA = function (newPythonInterpreter) {
            if (!$scope.oldEnv || !$scope.oldEnv.desc || !$scope.oldEnv.desc.predefinedContainerHooks || !newPythonInterpreter) {
                return false;
            }

            return !$scope.supportedDeprecatedPythons.includes($scope.oldEnv.desc.pythonInterpreter) &&
                !isDeprecatedPythonInHooks($scope.oldEnv.desc.predefinedContainerHooks, newPythonInterpreter);
        }
    });

    $scope.create = function() {
        DataikuAPI.admin.codeenvs.design.create("PYTHON", $scope.newEnv).success(function(data){
            $scope.dismiss();
            FutureProgressModal.show(parentScope, data, "Env creation", undefined, 'static', false).then(function(result){
                const modalClosed = result
                    ? Dialogs.infoMessagesDisplayOnly(parentScope, "Creation result", result.messages, result.futureLog, undefined, 'static', false)
                    : $q.resolve();
                modalClosed.then(() => {
                    DataikuAPI.admin.codeenvs.design.get("PYTHON", $scope.newEnv.envName).success(function(data) {
                        let packageList = $scope.newEnv.specPackageList;
                        $scope.newEnv = data;
                        $scope.newEnv.specPackageList = packageList;
                        $scope.newEnv.resourcesInitScript = $scope.oldEnv.resourcesInitScript;
                        DataikuAPI.admin.codeenvs.design.save("PYTHON", $scope.newEnv.envName, $scope.newEnv).success(function() {
                            CreateModalFromTemplate("/templates/admin/code-envs/design/change-python-modal-successful.html", parentScope, "AdminCodeEnvsDesignChangePythonSuccessfulController", function (newScope) {
                                newScope.env = $scope.newEnv;
                                newScope.inplace = $scope.inplace;
                                newScope.env.pythonInterpreterFriendlyName = CodeEnvService.pythonInterpreterFriendlyName(newScope.env.desc.pythonInterpreter);
                                newScope.previousCorePackagesSet = $scope.previousCorePackagesSet;
                            })
                        }).error(setErrorInScope.bind($scope));
                    }).error(setErrorInScope.bind($scope));
                });
            });

        }).error(setErrorInScope.bind(parentScope));
    }
    $scope.update = function(){
        $scope.dismiss();

        if ($scope.newEnv.removeContainerRuntimeAdditions && !$scope.supportedDeprecatedPythons.includes($scope.newEnv.pythonInterpreter)) {
            var filteredHooks = $scope.oldEnv.desc.predefinedContainerHooks.filter(hook => hook.type !== "PYTHON36_SUPPORT" && hook.type !== "PYTHON37_SUPPORT" && hook.type !== "PYTHON38_SUPPORT");
            $scope.oldEnv.desc.predefinedContainerHooks = filteredHooks;
        } else if ($scope.newEnv.updateContainerRuntimeAdditions && $scope.supportedDeprecatedPythons.includes($scope.newEnv.pythonInterpreter)) {
            var updatedHooks = $scope.oldEnv.desc.predefinedContainerHooks.filter(hook => hook.type !== ($scope.oldEnv.desc.pythonInterpreter + "_SUPPORT"));
            updatedHooks.push({ type: $scope.newEnv.pythonInterpreter + "_SUPPORT" });
            $scope.oldEnv.desc.predefinedContainerHooks = updatedHooks;
        }

        $scope.oldEnv.desc.pythonInterpreter = $scope.newEnv.pythonInterpreter;
        $scope.oldEnv.desc.customInterpreter = $scope.newEnv.customInterpreter;
        $scope.oldEnv.desc.corePackagesSet = $scope.newEnv.corePackagesSet;
        $scope.oldEnv.specPackageList = $scope.newEnv.specPackageList;
        DataikuAPI.admin.codeenvs.design.save("PYTHON", $scope.oldEnv.envName, $scope.oldEnv).success(function(data) {
            DataikuAPI.admin.codeenvs.design.recreate("PYTHON", $scope.oldEnv.envName).success(function(data) {
                FutureProgressModal.show(parentScope, data, "Change python interpreter", undefined, 'static', false).then(function(result){
                    const modalClosed = result
                        ? Dialogs.infoMessagesDisplayOnly(parentScope, "Migration result", result.messages, result.futureLog, undefined, 'static', false)
                        : $q.resolve();
                    if (!result.messages.error) {
                        modalClosed.then(() => {
                            CreateModalFromTemplate("/templates/admin/code-envs/design/change-python-modal-successful.html", parentScope, "AdminCodeEnvsDesignChangePythonSuccessfulController", function (newScope) {
                                $scope.oldEnv.desc.corePackagesSet = $scope.newEnv.corePackagesSet;
                                newScope.env = $scope.oldEnv;
                                newScope.inplace = $scope.inplace;
                                newScope.env.pythonInterpreterFriendlyName = CodeEnvService.pythonInterpreterFriendlyName(newScope.env.desc.pythonInterpreter);
                                newScope.previousCorePackagesSet = $scope.previousCorePackagesSet;
                            })
                        });
                    }
                });
            CodeEnvService.refreshEnv(parentScope);
            }).error(setErrorInScope.bind($scope));
        }).error(setErrorInScope.bind(parentScope));
    }
});

app.controller("AdminCodeEnvsDesignNewPythonController", function($scope, CodeEnvsDesignService, TopNav, DataikuAPI, Dialogs, FutureProgressModal, $q, Deprecation, CodeEnvService) {

    $scope.newEnv = CodeEnvsDesignService.getDefaultPythonEnvNoCorePackage();

    $scope.deploymentModes = [
        ["DESIGN_MANAGED", "Managed by DSS (recommended)"],
        ["DESIGN_NON_MANAGED", "Non-managed path"],
        ["EXTERNAL_CONDA_NAMED", "Named external Conda env"]
    ]

    CodeEnvsDesignService.getUserAccessibleSettings().then(({data}) => {
        $scope.enableCodeEnvResources = data.enableCodeEnvResources;
    })

    $scope.$watch("newEnv.conda", function(nv) {
        if (nv === true) {
            $scope.pythonInterpreters = CodeEnvService.pythonCondaInterpreters;
        } else if (nv === false) {
            $scope.pythonInterpreters = CodeEnvService.pythonInterpreters;
            CodeEnvService.getPythonInterpreters().then((interpreters) => {$scope.pythonInterpreters = interpreters});
        }
    });
    $scope.$watch("newEnv.deploymentMode", $scope.isPythonDeprecated);
    $scope.create = function(){
        var parentScope = $scope.$parent.$parent;
        DataikuAPI.admin.codeenvs.design.create("PYTHON", $scope.newEnv).success(function(data){
            $scope.dismiss();
            FutureProgressModal.show(parentScope, data, "Env creation", undefined, 'static', false).then(function(result){
                const modalClosed = result
                    ? Dialogs.infoMessagesDisplayOnly(parentScope, "Creation result", result.messages, result.futureLog, undefined, 'static', false)
                    : $q.resolve();

                const refreshed = parentScope.refreshList();

                $q.all([modalClosed, refreshed]).then(() => {
                    parentScope.goToEditIfExists(result && result.envName);
                });
            });
        }).error(setErrorInScope.bind($scope));
    }
    $scope.isPythonDeprecated = function() {
        return $scope.newEnv && ($scope.newEnv.deploymentMode === "DESIGN_MANAGED") && Deprecation.isPythonDeprecated($scope.newEnv.pythonInterpreter);
    }
});

app.controller("AdminCodeEnvsDesignNewRController", function($scope, CodeEnvsDesignService, TopNav, DataikuAPI, Dialogs, FutureProgressModal, $q) {

    $scope.newEnv = CodeEnvsDesignService.getDefaultREnv();

    $scope.deploymentModes = [
        ["DESIGN_MANAGED", "Managed by DSS (recommended)"],
        ["DESIGN_NON_MANAGED", "Non-managed path"],
        ["EXTERNAL_CONDA_NAMED", "Named external Conda env"]
    ]

    CodeEnvsDesignService.getUserAccessibleSettings().then(({data}) => {
        $scope.enableCodeEnvResources = data.enableCodeEnvResources;
    })

    $scope.create = function(){
        var parentScope = $scope.$parent.$parent;
        DataikuAPI.admin.codeenvs.design.create("R", $scope.newEnv).success(function(data){
            $scope.dismiss();
            FutureProgressModal.show(parentScope, data, "Env creation", undefined, 'static', false).then(function(result){
                const modalClosed = result
                    ? Dialogs.infoMessagesDisplayOnly(parentScope, "Creation result", result.messages, result.futureLog, undefined, 'static', false)
                    : $q.resolve();

                const refreshed = parentScope.refreshList();

                $q.all([modalClosed, refreshed]).then(() => {
                    parentScope.goToEditIfExists(result && result.envName);
                });
            });
        }).error(setErrorInScope.bind($scope));
    }
});

app.controller("AdminCodeEnvsDesignCreateFromDraftController", function($scope, $controller, $state, $stateParams, Assert, TopNav, DataikuAPI, FutureProgressModal, CreateModalFromTemplate, Dialogs, Logs, $q, CodeEnvService) {
    $controller("AdminCodeEnvsDesignListController", {$scope:$scope});

    $scope.loadDraftCodeEnv = function() {
        return DataikuAPI.admin.codeenvs.design.getDraft($stateParams.draftId).success(function(data) {
            $scope.draftId = data.config.id;
            $scope.newEnv = data.env;
            $scope.newEnv.envName = data.config.envName;
            $scope.envLang = data.config.envLang;
            $scope.openDraftModal();
        }).error(setErrorInScope.bind($scope));
    };
    $scope.loadDraftCodeEnv();

    $scope.isAdmin = $scope.appConfig.admin;

    $scope.openDraftModal = function(){
        if ($scope.envLang === "PYTHON") {
            CreateModalFromTemplate("/templates/admin/code-envs/design/new-python-env-from-draft-modal.html", $scope, "AdminCodeEnvsDesignNewPythonController", (modalScope) => {
                modalScope.requestedEnv = $scope.newEnv;

                if (modalScope.requestedEnv.conda === true) {
                   modalScope.draftPythonInterpreters = CodeEnvService.pythonCondaInterpreters;
                } else {
                   modalScope.draftPythonInterpreters = CodeEnvService.pythonInterpreters;
                   CodeEnvService.getPythonInterpreters().then((interpreters) => {modalScope.draftPythonInterpreters = interpreters});
                }

                modalScope.createEnvFromDraft = function() {
                    DataikuAPI.admin.codeenvs.design.createFromDraft($scope.draftId, "PYTHON", $scope.newEnv).success(function(data){
                        modalScope.dismiss();
                        handleEnvCreationResult(data);
                    }).error(setErrorInScope.bind($scope));
                };
            });
        } else if ($scope.envLang === "R") {
            CreateModalFromTemplate("/templates/admin/code-envs/design/new-R-env-from-draft-modal.html", $scope, "AdminCodeEnvsDesignNewRController", (modalScope) => {
                modalScope.requestedEnv = $scope.newEnv;
                modalScope.createEnvFromDraft = function() {
                    DataikuAPI.admin.codeenvs.design.createFromDraft($scope.draftId, "R", $scope.newEnv).success(function(data){
                        modalScope.dismiss();
                        handleEnvCreationResult(data);
                    }).error(setErrorInScope.bind($scope));
                }
            });
        }
    }

    function handleEnvCreationResult(data){
        FutureProgressModal.show($scope, data, "Env creation", undefined, 'static', false).then(function(result){
            const modalClosed = result
                ? Dialogs.infoMessagesDisplayOnly($scope, "Creation result", result.messages, result.futureLog, undefined, 'static', false)
                : $q.resolve();
            const refreshed = $scope.refreshList();

            $q.all([modalClosed, refreshed]).then(() => {
                $scope.goToEditIfExists(result && result.envName);
            });
        });
    }
});


app.controller("AdminCodeEnvsDesignImportController", function($scope, $state, $stateParams, Assert, TopNav, DataikuAPI, FutureProgressModal, Dialogs, Logs, $q) {
    $scope.newEnv = {}

    $scope.import = function() {
        Assert.trueish($scope.newEnv.file, "No code env file");

        const parentScope = $scope.$parent.$parent;
        DataikuAPI.admin.codeenvs.design.import($scope.newEnv.file).then(function(data) {
            $scope.dismiss();
            FutureProgressModal.show(parentScope, JSON.parse(data), "Env import", undefined, 'static', false).then(function(result) {
                const modalClosed = result
                    ? Dialogs.infoMessagesDisplayOnly(parentScope, "Creation result", result.messages, result.futureLog, undefined, 'static', false)
                    : $q.resolve();

                const refreshed = parentScope.refreshList();

                $q.all([modalClosed, refreshed]).then(() => {
                    parentScope.goToEditIfExists(result && result.envName);
                });
            });
        }, function(payload) {
            setErrorInScope.bind($scope)(JSON.parse(payload.response), payload.status, function(h) {return payload.getResponseHeader(h)});
        });
    }
});


app.controller("_AdminCodeEnvsDesignEditController", function ($scope, $controller, $state, $stateParams, TopNav, DataikuAPI, ActivityIndicator, CreateModalFromTemplate, FutureProgressModal, Dialogs, Logs, CodeEnvService) {
    $controller("AdminCodeEnvsDesignController", { $scope });
    TopNav.setLocation(TopNav.DSS_HOME, "administration");

    $scope.envName = $stateParams.envName;

    $scope.uiState = {
        performChangesOnSave: true,
        upgradeAllPackages: true,
        updateResources: true,
        forceRebuildEnv: false,
        active : 'info'
    }

    $scope.getSingleVersion = function(codeEnv) {
        return null;
    };

    $scope.actionAfterDeletion = function() {
        $state.go("admin.codeenvs-design.list");
    };

    $scope.canBeUpdated = function() {
        return $scope.codeEnv && $scope.codeEnv.canUpdateCodeEnv && ['DESIGN_MANAGED', 'PLUGIN_MANAGED', 'AUTOMATION_SINGLE', 'AUTOMATION_VERSIONED', 'DSS_INTERNAL'].indexOf($scope.codeEnv.deploymentMode) >= 0;
    };

    $scope.getPackagesThatWereRequiredAndInstalledButAreNotRequiredAnymore = function(packageSystem){
        if (!$scope.codeEnv || !$scope.previousPackagesSetForRemovalWarning[packageSystem]) return [];

        const newSet = CodeEnvService.getPackagesThatAreRequired($scope.codeEnv, packageSystem);

        return [...CodeEnvService.difference($scope.previousPackagesSetForRemovalWarning[packageSystem], newSet)];
    }

    $scope.isPackageRequired = function(packageSystem, packageName) {
        let required = "";
        if (packageSystem == 'pip') {
            required = $scope.codeEnv.specPackageList;
        } else if (packageSystem == 'conda') {
            required = $scope.codeEnv.specCondaEnvironment;
        }
        return CodeEnvService.getPackageNames(required).has(packageName);
    }

    $scope.isPackageInstalled = function(packageSystem, packageName) {
        let installed = "";
        if (packageSystem == 'pip') {
            installed = $scope.codeEnv.actualPackageList;
        } else if (packageSystem == 'conda') {
            installed = $scope.codeEnv.actualCondaEnvironment;
        }
        return CodeEnvService.getPackageNames(installed).has(packageName);
    }

    $scope.isInternalCodeEnv = function() {
        return $scope.codeEnv && ($scope.codeEnv.deploymentMode === 'DSS_INTERNAL');
    }

    $scope.hasOptedOutOfReferenceSpec = function() {
        return $scope.isInternalCodeEnv() && $scope.codeEnv.desc && ($scope.codeEnv.desc.useReferenceSpec === false)
    }

    $scope.useReferenceSpec = function() {
        return $scope.isInternalCodeEnv() && $scope.codeEnv.desc && ($scope.codeEnv.desc.useReferenceSpec === true)
    }

    $scope.specIsDirty = function() {
        if (!$scope.codeEnv) return false;
        var currentSpec = CodeEnvService.makeCurrentSpec($scope.codeEnv);
        return !angular.equals(currentSpec, $scope.previousSpec);
    };
    checkChangesBeforeLeaving($scope, $scope.specIsDirty);

    $scope.previousSpec = {
    }
    $scope.previousPackagesSetForRemovalWarning = {'pip': new Set(), 'conda': new Set()};

    $scope.refreshMandatoryPackageList = function(){
        if(!$scope.codeEnv) return;

        if(!$scope.codeEnv.desc.installCorePackages && !$scope.codeEnv.desc.installJupyterSupport){
            $scope.codeEnv.mandatoryPackageList = "";
            $scope.codeEnv.mandatoryCondaEnvironment = "";
            return;
        }

        DataikuAPI.codeenvs.getMandatoryPackages($scope.envLang, $scope.codeEnv.desc).then(({data}) => {
           if($scope.codeEnv.conda){
              $scope.codeEnv.mandatoryCondaEnvironment = data;
           }else{
              $scope.codeEnv.mandatoryPackageList = data;
           }
        });
    }

    CodeEnvService.refreshEnv($scope).then(() => $scope.uiState.forceRebuildEnv = $scope.isInternalCodeEnv());

    $scope.updateEnv = function() {
        CodeEnvService.updateEnv(
            $scope, 
            $stateParams.envName, 
            $scope.uiState.upgradeAllPackages, 
            $scope.uiState.updateResources, 
            $scope.uiState.forceRebuildEnv, 
            $scope.envLang, 
            null
        );
    };

    $scope.saveAndMaybePerformChanges = function(performChangesOnSave){
        DataikuAPI.admin.codeenvs.design.save($scope.envLang, $stateParams.envName, $scope.codeEnv).success(function(data) {
            CodeEnvService.refreshEnv($scope);
            if (performChangesOnSave) {
                $scope.updateEnv();
            }
        }).error(setErrorInScope.bind($scope));
    }

    $scope.fetchNonManagedEnvDetails = function(){
        DataikuAPI.admin.codeenvs.design.fetchNonManagedEnvDetails($scope.envLang, $stateParams.envName).success(function(data) {
            $scope.nonManagedEnvDetails = data;
        }).error(setErrorInScope.bind($scope));
    }

    $scope.installJupyterSupport = function(){
        DataikuAPI.admin.codeenvs.design.installJupyterSupport($scope.envLang, $stateParams.envName).success(function(data) {
            FutureProgressModal.show($scope, data, "Env update").then(function(result){
                if (result) { // undefined in case of abort
                    Dialogs.infoMessagesDisplayOnly($scope, "Update result", result.messages, result.futureLog);
                }
                CodeEnvService.refreshEnv($scope);
            })
        }).error(setErrorInScope.bind($scope));
    }

    $scope.removeJupyterSupport = function(){
        DataikuAPI.admin.codeenvs.design.removeJupyterSupport($scope.envLang, $stateParams.envName).success(function(data) {
            FutureProgressModal.show($scope, data, "Env update").then(function(result){
                if (result) { // undefined in case of abort
                    Dialogs.infoMessagesDisplayOnly($scope, "Update result", result.messages, result.futureLog);
                }
                CodeEnvService.refreshEnv($scope);
            })
        }).error(setErrorInScope.bind($scope));
    }

    $scope.showResourcesTab = function() {
        return ($scope.appConfig.admin && $scope.isInternalCodeEnv()) || !['EXTERNAL_CONDA_NAMED', 'DSS_INTERNAL'].includes($scope.codeEnv?.deploymentMode);
    };

    $scope.specPackageListEditorOptionsPip = $scope.codeMirrorSettingService.get("text/plain", {onLoad: function(cm){$scope.codeMirrorPip = cm;}});
    $scope.specPackageListEditorOptionsConda = $scope.codeMirrorSettingService.get("text/plain", {onLoad: function(cm){$scope.codeMirrorConda = cm;}});
    $scope.codeEnvResourcesEditorOptions = $scope.codeMirrorSettingService.get("text/x-python");
    $scope.getLog = DataikuAPI.admin.codeenvs.design.getLog;
    $scope.downloadURL = function(envLang, envName, logName) {
        return "/dip/api/code-envs/design/stream-log?envLang=" + envLang + "&envName=" + encodeURIComponent(envName) + "&logName=" + encodeURIComponent(logName);
    }
});

app.controller("AdminCodeEnvsDesignPythonEditController", function($scope, $controller, $stateParams, CodeEnvsDesignService, Dialogs, DataikuAPI, CreateModalFromTemplate, PandasSupport) {
    $scope.envLang = "PYTHON";
    $controller("_AdminCodeEnvsDesignEditController", { $scope });

    // Retrieve a handle on the editor for the spec packages (pip).
    let codeMirrorPip = null;
    $scope.specPackageListEditorOptionsPip = $scope.codeMirrorSettingService.get("text/plain", {
        onLoad: (codeMirror) => {
            codeMirrorPip = codeMirror;
        },
    });

    // Get a handle on the editor for the spec packages (conda).
    let codeMirrorConda = null;
    $scope.specPackageListEditorOptionsConda = $scope.codeMirrorSettingService.get("text/plain", {
        onLoad: (codeMirror) => {
            codeMirrorConda = codeMirror;
        },
    });

    // Fetch the package presets.
    $scope.packageListTypes = [];
    $scope.packageListValues = {};

    $scope.corePackagesVersions = [];
    $scope.$watch("codeEnv", function(nv) {
        if (!nv) return;
        $scope.corePackagesVersions = PandasSupport.listSupportedVersions($scope.codeEnv.desc.pythonInterpreter, $scope.codeEnv.desc.corePackagesSet);
    });

    // The code environment is not yet loaded when this code is executed.
    CodeEnvsDesignService.getUserAccessibleSettings().then(({data}) => {
        $scope.enableCodeEnvResources = data.enableCodeEnvResources;
    })

    $scope.insertAdditionalPackages = (editorType) => {
        const newScope = $scope.$new();

        const editorSettings = $scope.codeMirrorSettingService.get('text/plain');
        newScope.insertReadOnlyOptions = {
            ...editorSettings,
            readOnly: true,
            lineNumbers: false,
            foldGutter: false,
        };

        CreateModalFromTemplate("/templates/admin/code-envs/design/add-additional-packages-modal.html", newScope, null, (scope) => {
            CodeEnvsDesignService.fetchPackagePresets(newScope.codeEnv, newScope).then(() => {
                newScope.selectedPackages = newScope.packageListTypes[0][0];
            });

            // We must define this function here (rather than in newScope
            // itself) because scope.dismiss() is not available at this stage.
            // It is added when the actual modal scope is built from newScope.
            scope.insertPackages = () => {
                const selectedContent = scope.packageListValues[scope.selectedPackages];
                CodeEnvsDesignService.replacePackageListEditorContent(selectedContent, editorType, codeMirrorPip, codeMirrorConda);

                scope.dismiss();
            };
        });
    };

    $scope.openChangePythonInterpreterModal = function(codeEnv){
        CreateModalFromTemplate("/templates/admin/code-envs/design/change-python-modal.html", $scope, "AdminCodeEnvsDesignChangePythonController", function (newScope, newDOMElt) {
            newScope.oldEnv = codeEnv;
            newScope.previousPython = codeEnv.desc.pythonInterpreter;
            newScope.newEnv = angular.copy(codeEnv.desc);
            newScope.newEnv.pythonInterpreter = "PYTHON39";
            newScope.newEnv.envLang = codeEnv.envLang;
            newScope.newEnv.deploymentMode = codeEnv.deploymentMode;
            newScope.newEnv.specPackageList = newScope.oldEnv.specPackageList;
            newScope.newEnv.updateContainerRuntimeAdditions = true;
            newScope.newEnv.removeContainerRuntimeAdditions = true;
            newScope.nameChangedByUser = false;
            newDOMElt.on('keydown', 'input', function(e) {
                newScope.nameChangedByUser = true;
            })
        })
    }

});

app.controller("AdminCodeEnvsDesignREditController", function($scope, $controller) {
    $scope.envLang = "R";
    $controller("_AdminCodeEnvsDesignEditController", { $scope });
});

app.service("CodeEnvsDesignService", (DataikuAPI, $timeout) => {
        const svc = {};

        svc.getDefaultPythonEnv = () => {
            return {
                   envLang: "PYTHON",
                   deploymentMode: "DESIGN_MANAGED",
                   pythonInterpreter: "PYTHON39",
                   conda: false,
                   installCorePackages: true,
                   corePackagesSet : "PANDAS10",
                   installJupyterSupport: true
               };
        }
        svc.getDefaultPythonEnvNoCorePackage = () => {
            return {
                   envLang: "PYTHON",
                   deploymentMode: "DESIGN_MANAGED",
                   pythonInterpreter: "PYTHON39",
                   conda: false,
                   installCorePackages: true,
                   installJupyterSupport: true
               };
        }
        svc.getDefaultREnv = () => {
            return {
                envLang: "R",
                deploymentMode: "DESIGN_MANAGED",
                conda: false,
                installCorePackages: true,
                installJupyterSupport: true
               };

        }

        svc.CRANEditHelpString = "Specify the packages you want:\n\n"+
                                 "* one row per package\n"+
                                 "* each row is a pair of package name and minimal package version (optional)\n"+
                                 "\n"+
                                 "The version is only a minimal version. It is not supported to specify an explicity version.\n\n"+
                                 "Examples of package specifications:\n\n"+
                                 "* RJSONIO,0.13\n"+
                                 "* dplyr,";
        svc.rCondaSpecEditHelpString = "Specify the packages you want:\n\n"+
                                       "* one row per package\n"+
                                       "* each row is a Conda package match specification ( [link](https://conda.io/docs/user-guide/tasks/build-packages/package-spec.html#package-match-specifications) )\n"+
                                       "\n"+
                                       "Examples of package specifications:\n\n"+
                                       "* r-irkernel>=0.7";
        svc.pipRequirementsEditHelpString = "Specify the packages you want:\n\n"+
                                            "* one row per package\n"+
                                            "* each row is a PIP package specification ( [link](https://setuptools.readthedocs.io/en/latest/pkg_resources.html#requirement-objects) )\n"+
                                            "\n"+
                                            "Examples of package specifications:\n\n"+
                                            "* pandas==0.20.3\n"+
                                            "* numpy>=0.19";
        svc.pyCondaSpecEditHelpString = "Specify the packages you want:\n\n"+
                                        "* one row per package\n"+
                                        "* each row is a Conda package match specification ( [link](https://conda.io/docs/user-guide/tasks/build-packages/package-spec.html#package-match-specifications) )\n"+
                                        "\n"+
                                        "Examples of package specifications:\n\n"+
                                        "* pandas=0.20.3\n"+
                                        "* numpy>=0.19";

        svc.fetchPackagePresets = (codeEnv, $scope) => {
            const { pythonInterpreter, conda, installCorePackages, corePackagesSet } = codeEnv.desc;
            let pythonCorePackageSet = corePackagesSet;
            if (installCorePackages === false) {
                pythonCorePackageSet = null;
            }
            return DataikuAPI.admin.codeenvs.design.listPythonPackagePresets(pythonInterpreter, conda, pythonCorePackageSet)
                .success((presets) => {
                    $scope.packageListTypes = []
                    $scope.packageListValues = {}
                    presets.forEach((preset) => {
                        const { id, description, packages } = preset;
                        $scope.packageListTypes.push([id, description]);
                        $scope.packageListValues[id] = ("# "+description+"\n").concat(packages.join("\n"));
                    });
                })
                .error(setErrorInScope.bind($scope));
        }

        svc.replacePackageListEditorContent = (content, editorType, codeMirrorPip, codeMirrorConda) => {
            const codeMirror = (editorType === "pip") ? codeMirrorPip : codeMirrorConda;

            // Move cursor to end of editor content
            const lastLine = codeMirror.lastLine();
            const lastChar = codeMirror.getLine(lastLine).length;
            if (lastLine > 0) {
                codeMirror.replaceRange('\n\n', {line: lastLine, ch: lastChar});
                codeMirror.setCursor(lastLine + 2, 0);
            } else {
                codeMirror.setCursor(lastLine, lastChar);
            }

            // Timeout to make sure of an angular safe apply.
            $timeout(() => {
                codeMirror.replaceSelection(content, "end");
            });
            codeMirror.focus();
        }

        svc.getUserAccessibleSettings = function() {
            return DataikuAPI.codeenvs.getUserAccessibleCodeEnvSettings();
        }
        return svc;
    }
);

}());
