(function() {
    'use strict';

    const app = angular.module('dataiku.dashboards.insights');

    app.component('titleTile', {
        templateUrl: '/templates/dashboards/insights/title/title_tile.html',
        bindings: {
            hook: '<',
            tile: '<',
            page: '<',
            pageIdx: '<',
            tileSpacing: '<',
            dashboardTheme: '<'
        },
        controller: function($timeout, $scope, TileLoadingState, DashboardPageGridFacadeService, PageGridTitleService, translate, $rootScope, DashboardGridElementService, DashboardUtils) {
            const ctrl = this;
            let gridScrollingContainerResizeSubscription = null;

            ctrl.load = function(resolve) {
                ctrl.hook.loadStates[ctrl.tile.$tileId] = TileLoadingState.COMPLETE;
                resolve();
            };

            ctrl.titleTileZoom = 1.0;
            ctrl.pageTitle = (
                (ctrl.page && ctrl.page.title)
                    ? ctrl.page.title
                    : ''
            );

            gridScrollingContainerResizeSubscription = DashboardGridElementService.getPageInnerGridResizedWithThrottle$(10).subscribe(({
                gridId,
                newClientWidth
            }) => {
                if (!gridId.isPageGrid() || gridId.dashboardPageCid !== ctrl.page.cid) {
                    return;
                }
                syncPageGridTitleZoom(newClientWidth);
            });

            ctrl.$onInit = function() {

                ctrl.hook.loadPromises[ctrl.tile.$tileId] = ctrl.load;
                ctrl.hook.reloadPromises[ctrl.tile.$tileId] = ctrl.load;
                ctrl.hook.loadStates[ctrl.tile.$tileId] = TileLoadingState.WAITING;

                $timeout(() => {
                    syncPageGridTitle();
                });
            };

            ctrl.$onDestroy = function() {
                if (gridScrollingContainerResizeSubscription != null) {
                    gridScrollingContainerResizeSubscription.unsubscribe();
                }
            };

            $scope.$watchGroup([
                '$ctrl.page.titleFontSize',
                '$ctrl.page.displayedTitle',
                '$ctrl.page.backgroundColor',
                '$ctrl.tile.box.height',
                '$ctrl.tileSpacing',
                '$ctrl.dashboardTheme.generalFormatting.fontFamily'
            ], () => {
                syncPageGridTitle();
            });

            function syncPageGridTitleZoom(scrollingContainerClientWidth) {
                const titleTileWidthRatio = Math.floor(100 * scrollingContainerClientWidth / DashboardPageGridFacadeService.getPageGridScrollingContainerNormalZoom(ctrl.tileSpacing)) / 100;

                const defaultTitle = translate('DASHBOARD.NAVIGATION_DRAWER.PAGE_INDEX', 'Page ' + (ctrl.pageIdx + 1), { index: ctrl.pageIdx + 1 });
                const pageGridTitleValue = PageGridTitleService.create({
                    page: ctrl.page,
                    defaultTitle,
                    titleWrapperFullWidth: DashboardPageGridFacadeService.getPageGridScrollingContainerNormalZoom(ctrl.tileSpacing),
                    titleFontFamilly: (
                        (ctrl.dashboardTheme && ctrl.dashboardTheme.generalFormatting)
                            ? ctrl.dashboardTheme.generalFormatting.fontFamily
                            : undefined
                    )
                });

                const gridCellSize = scrollingContainerClientWidth / DashboardUtils.getColumnNumber();
                const tileHeight = gridCellSize * ctrl.tile.box.height - ctrl.tileSpacing;
                const titleTileHeightRatio = Math.floor(100 * tileHeight / pageGridTitleValue.titleHeightWithPaddingPx) / 100;

                ctrl.titleTileZoom = Math.min(
                    1,
                    titleTileWidthRatio,
                    titleTileHeightRatio
                );
            }

            function syncPageGridTitle() {
                if (ctrl.page === undefined || ctrl.page === null) {
                    return;
                }
                const defaultTitle = translate('DASHBOARD.NAVIGATION_DRAWER.PAGE_INDEX', 'Page ' + (ctrl.pageIdx + 1), { index: ctrl.pageIdx + 1 });

                ctrl.pageTitle = (
                    ctrl.page.displayedTitle
                        ? ctrl.page.displayedTitle :
                        defaultTitle
                );
                ctrl.pageBackgroundColor = ctrl.page.backgroundColor;
                const pageGridTitleValue = PageGridTitleService.create({
                    page: ctrl.page,
                    defaultTitle,
                    titleWrapperFullWidth: DashboardPageGridFacadeService.getPageGridScrollingContainerNormalZoom(ctrl.tileSpacing),
                    titleFontFamilly: (
                        (ctrl.dashboardTheme && ctrl.dashboardTheme.generalFormatting)
                            ? ctrl.dashboardTheme.generalFormatting.fontFamily
                            : undefined
                    )
                });

                // Since AngularJS doesn’t support the ?. operator, the easiest way to use pageGridTitleValue in the template is to spread its properties.
                ctrl.pageGridTitlePadding = `${pageGridTitleValue.titlePaddingY}px ${pageGridTitleValue.titlePaddingX}px`;
                ctrl.pageGridTitleFontSize = pageGridTitleValue.titleFontSize;
                ctrl.pageGridTitleHeight = pageGridTitleValue.titleHeightWithoutPaddingPx;
                ctrl.pageGridTitleLineHeight = pageGridTitleValue.titleLineHeightPx;
                ctrl.pageGridTitleLineCount = pageGridTitleValue.titleLineCount;
                const initialPageGridTitleHeightWithPadding = ctrl.pageGridTitleHeightWithPadding;
                ctrl.pageGridTitleHeightWithPadding = pageGridTitleValue.titleHeightWithPaddingPx;

                const scrollingContainer = document.querySelector('dashboard-grid');
                syncPageGridTitleZoom(scrollingContainer.clientWidth);

                if (initialPageGridTitleHeightWithPadding !== undefined && initialPageGridTitleHeightWithPadding !== pageGridTitleValue.titleHeightWithPaddingPx) {
                    $rootScope.$broadcast('pageGridTitleHeightUpdatedEvent', { dashboardPageCid: ctrl.page.cid, pageGridTitleHeight:  pageGridTitleValue.titleHeightWithPaddingPx });
                }

            }
        }
    });
})();
