(function(){
    'use strict';

    const app = angular.module('dataiku.directives.widgets');

    app.directive('browsePath', function($timeout, $filter, PathUtils) {
        return {
            restrict: 'E',
            replace: true,
            templateUrl: '/templates/widgets/browse-path/browse-path.html',
            scope: {
                title: '@',
                path: '=',
                browseFn: '=',
                onBrowseDoneFn: '=?',
                canSelectFn: '=',
                canBrowseFn: '=?',
                selectedItems: '=?',
                isMultiSelect: '=?',
                destinationLabel: '@?',
                displayItemFn: '=?',
                cantWriteContentVerb: '@',
                searchable: '@?',
                onRenameFn: '=?',
                onMoveFn: '=?',
                onDeleteFn: '=?',
                itemsAreProjects: '=',
                showSize: '=?',
                showActionsMenu: '=?',
                registerExecuteBrowseFn: '=?'
            },
            link: function(scope, element, attrs) {
                scope.searchable = attrs.searchable ? scope.$eval(scope.searchable) : true;
                scope.displayItemFn = attrs.displayItemFn !== undefined ? scope.displayItemFn : item => item;
                scope.clickOnItem = function(item, event) {
                    if (!item.directory && scope.canSelectFn(item)) {
                        toggleSelect(item)
                    } else if (item.directory) {
                        if (((event.ctrlKey || event.metaKey) || !scope.canBrowseFn(item)) && scope.canSelectFn(item)) {
                            toggleSelect(item)
                            return;
                        }
                        if (scope.canBrowseFn(item)) {
                            scope.executeBrowse(item.fullPath, true);
                            return;
                        }
                    }
                }

                // -- selection
                const itemFullPath = item => item.directory ? PathUtils.makeT(item.fullPath) : item.fullPath;
                let selectedItemsMap = {};

                if (typeof scope.selectedItems === "undefined") {
                    scope.selectedItems = [];
                } else {
                    scope.selectedItems.forEach(item => selectedItemsMap[itemFullPath(item)] = item);
                }

                scope.isSelected = function(item) {
                    return scope.canSelectFn(item) && itemFullPath(item) in selectedItemsMap;
                }

                function toggleSelect(item) {
                    if (scope.isSelected(item)) {
                        delete selectedItemsMap[itemFullPath(item)];
                    } else {
                        if (!scope.isMultiSelect) {
                            selectedItemsMap = {};
                        }
                        selectedItemsMap[itemFullPath(item)] = item;
                    }
                    scope.selectedItems = Object.values(selectedItemsMap);
                }

                // -- filtering

                scope.filter = {query:""};
                function updateFiltered(){
                    if (scope.pathContent == null) {
                        return;
                    }
                    if (!scope.filter.query) {
                        scope.pathContent.filteredChildren = scope.pathContent.children;
                    } else {
                        scope.pathContent.filteredChildren = $filter("filter")(scope.pathContent.children, {name: scope.filter.query});
                    }
                }

                scope.$watch("filter.query", updateFiltered);

                // -- browsing

                scope.browseError = null;
                scope.firstBrowseDone = false;
                if (scope.path == null) scope.path = "/";
                scope.pathContent = [];
                scope.browsing = false;

                scope.executeBrowse = function(newPath, isDirectory = newPath.endsWith('/')){
                    scope.browseError = null;
                    scope.path = newPath.replace(/^(ROOT)?\/+/,'/');
                    if (isDirectory) scope.path = PathUtils.makeT(scope.path);
                    scope.browsing = true;
                    scope.browseFn(scope.path, isDirectory).success(updatePathContent).error(function(data, status, error){
                        scope.browsing = false;
                        scope.browseErrorMsg = getErrorDetails(data, status, error).detailedMessage;
                        scope.browseError = getErrorDetails(data, status, error);
                    });
                };

                function updatePathContent(data) {
                    scope.pathContent = data;
                    scope.pathContent.children.sort(function(a,b) {
                        if (a.directory != b.directory)  {
                            return a.directory ? -1 : 1;
                        }
                        return a.name.localeCompare(b.name);
                    });
                    updateFiltered();
                    scope.browsing = false;
                    // Refocus search after filter
                    scope.filter.query = "";
                    scope.selectedItems = [];
                    selectedItemsMap = {};
                    if (typeof scope.onBrowseDoneFn === "function") {
                        const currentFolder = angular.extend(
                            { canWriteContents: true },
                            scope.pathContent.pathElts[scope.pathContent.pathElts.length - 1],
                            { pathElts: scope.pathContent.pathElts.map(scope.displayItemFn).join('/') }
                        );
                        scope.onBrowseDoneFn(currentFolder);
                    }
                    $timeout(function(){
                        element.find(".search-input").focus();
                    }, 0);
                }

                scope.renameResource = function(item) {
                    if (scope.onRenameFn) {
                        scope.onRenameFn(item).then(data => {
                            if (data) {
                                updatePathContent(data);
                            }
                        }).catch(setErrorInScope.bind(scope));
                    }
                };

                scope.moveResource = function(item) {
                    if (scope.onMoveFn) {
                        scope.onMoveFn(item).then(data => {
                            if (data) {
                                updatePathContent(data);
                            }
                        }).catch(setErrorInScope.bind(scope));
                    }
                };

                scope.deleteResource = function(item) {
                    if (scope.onDeleteFn) {
                        scope.onDeleteFn(item).then(() => {
                            if (scope.pathContent?.children) {
                                scope.pathContent.children = scope.pathContent.children.filter(child => child.fullPath !== item.fullPath);
                                updatePathContent(scope.pathContent);
                            }

                        }).catch(setErrorInScope.bind(scope));
                    }
                };

                if (typeof(scope.canBrowseFn) !== "function") (
                    scope.canBrowseFn = function(item) {
                        return true;
                    }
                )

                if (typeof(scope.registerExecuteBrowseFn) === "function") {
                    scope.registerExecuteBrowseFn(scope.executeBrowse);
                }

                if (typeof scope.showSize === "undefined") {
                    scope.showSize = false;
                }

                scope.$watch('showSize', () => {
                    if (scope.showSize) {
                        scope.executeBrowse(scope.path);
                    }
                });

                // postponing it to next digest so that fatRepeat's container does not have its full height
                // (b/c many other elements in the modal will be hidden using ng-show, leaving space to fatRepeat's container once digest is done)
                $timeout(function() {
                    scope.executeBrowse(scope.path);
                    element.find(".search-input").focus();
                }, 0);
            }
        }
    });

    app.directive('browsePathInput', function(DataikuAPI, $timeout, $filter) {
        return {
            restrict: 'E',
            replace: true,
            templateUrl: '/templates/widgets/browse-path/browse-path-input.html',
            scope: {
                title: '@',
                path: '=',
                browseFn: '=',
                onBrowseDoneFn: '=?',
                canSelectFn: '=',
                canBrowseFn: '=?',
                itemsAreProjects: '=',
                acceptNewPath: '='     //just to inform the user with an help-inline
            },
            link: function (scope, element, attrs) {

            }
        }
    });

})();

