 /* Shaker support functions for processors library */

function StepIAE(message) {
    this.message = message;
    this.name = "StepIAE";
}
StepIAE.prototype = new Error;

(function(){
'use strict';

if (!String.prototype.format) {
    String.prototype.format = function() {
        var formatted = this;
        for(var arg in arguments) {
            formatted = formatted.replace("{" + arg + "}", arguments[arg]);
        }
        return formatted;
    };
}

function truncate(str, len) {
    /**
     * Truncate a string to make sure it takes at most
     * n characters.
     * Whenever possible truncates on special chars.
     *
     * If str is not a string, returns str unchanged.
     */
    if ((typeof str !== "string") || (str.length <= len)) {
        return str;
    }

    // eslint-disable-next-line no-useless-escape
    var cutOn = /[ ,\.,;\-\\\"\n\?\!\|]/g
    var truncated = str.substring(0, len-1);
    var lastSeparatorIdx = regexLastIndexOf(cutOn, truncated);

    // we don't want to cut more too much.
    if (lastSeparatorIdx > len / 2) {
        truncated = str.substring(0, lastSeparatorIdx);
    }
    return truncated + '…';
}

function hasNonEmptyParam(params, key){
    switch (typeof params[key]) {
        case 'undefined': return false;
        case 'string':    return params[key].length > 0;
        default:          return true;
    }
}

function hasAll(params, keys) {
    for (var index in keys) {
        if (!hasNonEmptyParam(params, keys[index])) return false;
    }
    return true;
}

function inColName(value) {
    return "<span class=\"input-column-name\">" + sanitize(value) + "</span>";
}
function outColName(value) {
    return "<span class=\"output-column-name\">" + sanitize(value) + "</span>";
}
function numLiteral(value) {
    return "<span class=\"num-literal\">" + sanitize(value) + "</span>";
}
function anumLiteral(value) {
    if (value == null || value.length === 0) {
        return '<span class="alphanum-literal">\'\'</span>';
    } else {
        return "<span class=\"alphanum-literal\">" + sanitize(value) + "</span>";
    }
}
function actionVerb(value) {
    return "<span class=\"action-verb\">" + sanitize(value) + "</span>";
}
function meaningName(value) {
    return "<span class=\"meaning-label\">" + sanitize(value) + "</span>";
}

// strong param value
// Please use with caution, it breaks the right panel view.
function strongify(value) {
    return "<strong>" + value + "</strong>";
}

// Boolean param value
function toBoolean(value) {
    return value == true || value == "true";
}

function isBlank(x) {
    return x == null|| x.length === 0;
}

function checkDate(d, name) {
    var match = /^(?:[1-9]\d{3}-(?:(?:0[1-9]|1[0-2])-(?:0[1-9]|1\d|2[0-8])|(?:0[13-9]|1[0-2])-(?:29|30)|(?:0[13578]|1[02])-31)|(?:[1-9]\d(?:0[48]|[2468][048]|[13579][26])|(?:[2468][048]|[13579][26])00)-02-29)(T(?:[01]\d|2[0-3])(:[0-5]\d:[0-5]\d(\.\d{3})?)?(?:Z|[+-][01]\d:[0-5]\d)?)?$/
    if (!(match.exec(d))) {
        throw new StepIAE("Invalid " + name);
    }
}

const booleanLiteralsMapping = {
    "true":true,
    "false":false,
    "yes": true,
    "no": false,
    "1": true,
    "0": false,
    "y": true,
    "n": false,
    "t": true,
    "f": false,
    "o": true
};

function isNumber(type) {
    return ["tinyint", "smallint", "int", "bigint", "float", "double"].indexOf(type) > -1;
}

function isBool(value) {
    return Object.keys(booleanLiteralsMapping).indexOf(value.toLowerCase()) > -1;
}

function parseBoolean(value) {
    return booleanLiteralsMapping[value.toLowerCase()] || false;
}

function filterDescEqualityOperator(type, value) {
    if (isNumber(type)) {
        return "== [number]";
    } else if(type === "date" && !isNaN(new Date(Date.parse(value)))) {
        return "== [date]";
    } else if(type === "boolean" && isBool(value)) {
        return parseBoolean(value).toString();
    } else if(type === "array" && "[]" === value) {
        return "empty array";
    }
    return "== [string]";
}

function parseNumber(type, value) {
    switch (type) {
        case "tinyint":
        case "smallint":
        case "int":
        case "bigint":
            return Number.parseInt(value);
        case "float":
        case "double":
            return Number.parseFloat(value);
        default:
            return value;
        }
}

function flagCheckValid(params) {
    if (!params["flagColumn"]) throw new StepIAE("Flag output column not selected");
}

function visualIfParams(input, type, value, operator) {
    let condition = {
        "input": input,
        "operator": operator
    };
    if(value) {
        if(isNumber(type)) {
            condition["num"] = parseNumber(type, value);
        } else if (type === "date") {
            const dateTime = new Date(Date.parse(value));
            if(!isNaN(dateTime)) {
                const dateTimeUTC = moment(dateTime).utc();
                const date = dateTimeUTC.format('YYYY-MM-DD');
                const time = dateTimeUTC.format('HH:mm');
                condition["date"] = date;
                condition["time"] = time;
                condition["unit"] = "minutes";
            } else {
                condition["string"] = value;
            }
        } else if (type === "boolean") {
            //nothing to do here
        } else {
            condition["string"] = value;
        }
    }
    return {
        "visualIfDesc": {
            "elseIfThens": [],
            "elseActions": [],
            "ifThen": {
                "filter": {
                    "uiData": {
                        "mode": "&&",
                        "conditions": [
                            condition
                        ]
                    },
                    "distinct": true,
                    "enabled": true
                },
                "actions": [
                    {
                        "outputColumnName": "",
                        "formula": "",
                        "value": "",
                        "operator": "ASSIGN_VALUE"
                    }
                ]
            }
        }
    };
}

var app = angular.module('dataiku.shaker.library', ['dataiku.filters', 'platypus.utils']);

app.factory("ShakerProcessorsInfo", function($filter, Fn, Assert, $translate, translate, DateUtilsService, ChartFilterUtils) {

    function isValidIfHasColumn(params, columnName, translateID, translateDefaultValue, {throwError}) {
        const isValid = hasNonEmptyParam(params, columnName);
        if (!isValid && throwError) {
            throw new StepIAE(translate(translateID, translateDefaultValue));
        }
        return isValid;
    }

    function defaultIsValidIfHasColumn(params, columnName, {throwError}) {
         return isValidIfHasColumn(params,
                                   columnName,
                                   "SHAKER.PROCESSORS.ERROR.COLUMN_NOT_SPECIFIED", "Column not specified",
                                   {throwError});
    }

    function isCurrencyConverterProcessorValid(params, {throwError}) {
        if (params["isCurrencyFromColumn"] && !hasNonEmptyParam(params, "refCurrencyColumn")) {
            if (throwError) {
                throw new StepIAE(translate('SHAKER.PROCESSOR.CurrencyConverter.FORM.ERROR.MISSING_CURRENT_CODE', "'Currency code' column expected"));
            }
            return false;
        }
        if (params["dateInput"] === "COLUMN" && !hasNonEmptyParam(params, "refDateColumn")) {
            if (throwError) {
                throw new StepIAE(translate('SHAKER.PROCESSOR.CurrencyConverter.FORM.ERROR.MISSING_DATE_COLUMN', "'Date parsed' column expected"));
            }
            return false;
        }
        if (params["dateInput"] === "CUSTOM" && !hasNonEmptyParam(params, "refDateCustom")) {
            if (throwError) {
                throw new StepIAE(translate('SHAKER.PROCESSOR.CurrencyConverter.FORM.ERROR.MISSING_DATE_CUSTOM', "'Custom date' column expected"));
            }
            return false;
        }
        // No need to throw an error here, the UI display is already correct on some basic inputs missing
        return hasAll(params, ["inputColumn", "inputCurrency", "outputCurrency"]);
    }

    function isColumnRenamerValid(params, {throwError}) {
        if (Array.isArray(params["renamings"])) {
            for (const elt of params["renamings"]) {
                if (!hasAll(elt, ["from", "to"])) {
                    if (throwError) {
                        throw new StepIAE(translate('SHAKER.PROCESSOR.ColumnRenamer.ERROR.MISSING_RENAMINGS_SRC', "Missing parameter: Renamings column"));
                    }
                    return false;
                }
            }
        }
        // No need to throw an error here, the UI display is already correct on some basic inputs missing
        return hasNonEmptyParam(params, "renamings");
    }

    function isDateComponentsExtractorValid(params, {throwError}) {
        if ((params["timezone_id"] === "extract_from_column" || params["timezone_id"] === "extract_from_ip") && !hasNonEmptyParam(params, "timezone_src")) {
            if (throwError) {
                throw new StepIAE(translate('SHAKER.PROCESSOR.DateComponentsExtractor.ERROR.MISSING_TZ_SOURCE', "Missing parameter: Source column"));
            }
            return false;
        }
        // No need to throw an error here, the UI display is already correct on some basic inputs missing
        return hasAll(params, ["column", "timezone_id"]);
    }

    function isHolidaysComputerValid(params, {throwError}) {
        if (params["calendar_id"] === "extract_from_column" && !hasNonEmptyParam(params, "calendar_src")) {
            if (throwError) {
                throw new StepIAE(translate('SHAKER.PROCESSOR.HolidaysComputer.ERROR.MISSING_CALENDAR_SRC', "Missing parameter: Country code source column"));
            }
            return false;
        }
        if ((params["timezone_id"] === "extract_from_column" || params["timezone_id"] === "extract_from_ip") && !hasNonEmptyParam(params, "timezone_src")) {
            if (throwError) {
                throw new StepIAE(translate('SHAKER.PROCESSOR.HolidaysComputer.ERROR.MISSING_TIMEZONE_SRC', "Missing parameter: Timezone source column"));
            }
            return false;
        }
        // No need to throw an error here, the UI display is already correct on some basic inputs missing
        return hasAll(params, ["inCol", "calendar_id", "timezone_id"]);
    }

    function appliesToCheckValid(params) {
        if (!params["appliesTo"]) throw new StepIAE(translate('SHAKER.PROCESSORS.ERROR.APPLIES_TO_NOT_SELECTED', "'Applies to' mode not selected"));

        switch (params["appliesTo"]) {
            case "SINGLE_COLUMN":
                if (!params.columns || params.columns.length === 0 || params.columns[0] == null || params.columns[0].length === 0) {
                    throw new StepIAE(translate('SHAKER.PROCESSORS.ERROR.COLUMN_NOT_SPECIFIED', "Column not specified"));
                }
                break;
            case "COLUMNS":
                if(!params.columns || params.columns.length === 0){
                    throw new StepIAE(translate('SHAKER.PROCESSORS.ERROR.COLUMN_NOT_SPECIFIED', "Column not specified"));
                }
                params.columns.forEach(colName => {
                    if (!colName) {throw new StepIAE(translate('SHAKER.PROCESSORS.ERROR.EMPTY_COLUMN_NAME', "Empty column name"));}
                })
                break;
            case "PATTERN":
                if (params.appliesToPattern == null || params.appliesToPattern.length === 0) {
                    throw new StepIAE(translate('SHAKER.PROCESSORS.ERROR.PATTERN_NOT_SPECIFIED', "Pattern not specified"));
                }
                break;
            default:
                break;
        }
    }

    function appliesToDescription(params) {
        switch (params["appliesTo"]) {
            case "SINGLE_COLUMN":
                return inColName(params["columns"][0]);
            case "COLUMNS":
                if (params["columns"].length > 4) {
                    return translate('SHAKER.PROCESSORS.DESCRIPTION.APPLIES_TO.COLUMNS.MANY', "{{count}} columns", {
                        count: anumLiteral(params["columns"].length)
                    });
                } else {
                    return translate('SHAKER.PROCESSORS.DESCRIPTION.APPLIES_TO.COLUMNS.FEW', '{{count > 1 ? "columns" : "column"}} {{columns}}', {
                        count: params["columns"].length,
                        columns: params["columns"].map(inColName).join(", ")
                    });
                }
            case "PATTERN":
                return translate('SHAKER.PROCESSORS.DESCRIPTION.APPLIES_TO.PATTERN', "columns matching /{{pattern}}/", {
                    pattern: anumLiteral(params["appliesToPattern"])
                });
            case "ALL":
                if (params["booleanMode"] === "OR") {
                    return translate('SHAKER.PROCESSORS.DESCRIPTION.APPLIES_TO.AT_LEAST_ONE', strongify("at least") + " one column");
                } else {
                    return translate('SHAKER.PROCESSORS.DESCRIPTION.APPLIES_TO.ALL', strongify("all") + " columns");
                }
        }
    }

    function comaList(items, lastSeparator) {
        if (items.length === 1) {
            return items[0]
        }
        return items.slice(0, items.length - 1).join(', ') + " " + lastSeparator + " " + items[items.length - 1]
    }

    function filterAndFlagSummary(params, conditionFunction, columns) {
        const columnsDesc = columns === undefined ? appliesToDescription(params) : columns;
        const multipleColumns = !(columns !== undefined
            || params["appliesTo"] === "SINGLE_COLUMN"
            || (params["appliesTo"] === "COLUMNS" && params["columns"].length === 1)
            || (params["appliesTo"] === "ALL" && params["booleanMode"] === "OR"));
        switch (params["action"]) {
            case "KEEP_ROW":
                return translate('SHAKER.PROCESSOR.FilterAndFlag.SUMMARY.KEEP_ROW', actionVerb("Keep") + " rows where {{condition}}",
                    { condition: conditionFunction(params, columnsDesc, multipleColumns) });
            case "REMOVE_ROW":
                return translate('SHAKER.PROCESSOR.FilterAndFlag.SUMMARY.REMOVE_ROW', actionVerb("Remove") + " rows where {{condition}}",
                    { condition: conditionFunction(params, columnsDesc, multipleColumns) });
            case "FLAG":
                return translate('SHAKER.PROCESSOR.FilterAndFlag.SUMMARY.FLAG', actionVerb("Flag") + " rows where {{condition}}",
                    { condition: conditionFunction(params, columnsDesc, multipleColumns) });
            case "CLEAR_CELL":
                return translate('SHAKER.PROCESSOR.FilterAndFlag.SUMMARY.CLEAR_CELL', actionVerb("Clear") + " cells in {{columns}} if {{condition}}",
                    { columns: columnsDesc, condition: conditionFunction(params, "value", false) });
            case "DONTCLEAR_CELL":
                return translate('SHAKER.PROCESSOR.FilterAndFlag.SUMMARY.DONTCLEAR_CELL', actionVerb("Keep") + " cells in {{columns}} only if {{condition}}",
                    { columns: columnsDesc, condition: conditionFunction(params, "value", false) });
            default:
                return false;
        }
    }

    function filterAndFlagOnNumericRangeCondition(params, columns) {
        if (params["min"] && params["max"]) {
            return translate("SHAKER.PROCESSOR.FilterAndFlag.SUMMARY.CONDITION.BETWEEN", "{{min}} &le; {{columns}} &le; {{max}}",
                { min: numLiteral(params["min"]), max: numLiteral(params["max"]), columns: columns });
        } else if (params["min"]) {
            return translate("SHAKER.PROCESSOR.FilterAndFlag.SUMMARY.CONDITION.MIN", "{{columns}} &ge; {{min}}",
                { min: numLiteral(params["min"]), columns: columns });
        } else {
            return translate("SHAKER.PROCESSOR.FilterAndFlag.SUMMARY.CONDITION.MAX", "{{columns}} &le; {{max}}",
                { max: numLiteral(params["max"]), columns: columns });
        }
    }

    function filterAndFlagOnValueCondition(params, columns, multipleColumns) {
        const p = {
            literal: anumLiteral(truncate(comaList(params["values"], translate('SHAKER.PROCESSOR.COMMA_LIST_OR', 'or')), 60)),
            columns: columns,
            multipleColumns: multipleColumns
        }
        const mode = params["matchingMode"] || "FULL_STRING";
        const exclude = params["exclude"] || false;
        if (exclude) {
            if (mode === "FULL_STRING") {
                return translate("SHAKER.PROCESSOR.FilterAndFlag.SUMMARY.CONDITION.FULL_STRING.EXCLUDE", "{{columns}} {{multipleColumns?'are':'is'}} not {{literal}}", p);
            } else if (mode === "SUBSTRING") {
                return translate("SHAKER.PROCESSOR.FilterAndFlag.SUMMARY.CONDITION.SUBSTRING.EXCLUDE", "{{columns}} {{multipleColumns?'do':'does'}} not contain {{literal}}", p);
            } else {
                return translate("SHAKER.PROCESSOR.FilterAndFlag.SUMMARY.CONDITION.MATCHES.EXCLUDE", "{{columns}} {{multipleColumns? 'do' : 'does'}} not match {{literal}}", p);
            }
        } else {
            if (mode === "FULL_STRING") {
                return translate("SHAKER.PROCESSOR.FilterAndFlag.SUMMARY.CONDITION.FULL_STRING", "{{columns}} {{multipleColumns?'are':'is'}} {{literal}}", p);
            } else if (mode === "SUBSTRING") {
                return translate("SHAKER.PROCESSOR.FilterAndFlag.SUMMARY.CONDITION.SUBSTRING", "{{columns}} {{multipleColumns?'contain':'contains'}} {{literal}}", p);
            } else {
                return translate("SHAKER.PROCESSOR.FilterAndFlag.SUMMARY.CONDITION.MATCHES", "{{columns}} {{multipleColumns?'match':'matches'}} {{literal}}", p);
            }
        }
    }

    function filterAndFlagImpactVerb(params){
        if (params.action === "REMOVE_ROW" || params.action === "KEEP_ROW") {
            return "deleted";
        } else {
            return "modified";
        }
    }

    function filterAndFlagOnDateSummary(type, params) {
        const datePartLabels = {
            YEAR: translate("GLOBAL.DATE.YEAR", "Year"),
            QUARTER_OF_YEAR: translate("GLOBAL.DATE.QUARTER", "Quarter"),
            WEEK_OF_YEAR: translate("GLOBAL.DATE.WEEK", "Week"),
            MONTH_OF_YEAR: translate("GLOBAL.DATE.MONTH", "Month"),
            DAY_OF_MONTH: translate("GLOBAL.DATE.DAY_OF_MONTH", "Day of month"),
            DAY_OF_WEEK: translate("GLOBAL.DATE.DAY_OF_WEEK", "Day of week"),
            HOUR_OF_DAY: translate("GLOBAL.DATE.HOUR", "Hour"),
            INDIVIDUAL: translate("GLOBAL.DATE_PART.INDIVIDUAL", "Individual dates")
        };
        const relativeDatePartsLabels = {
            YEAR: translate("SHAKER.PROCESSOR.FilterAndFlagOnDate.SUMMARY.RELATIVE_PART.YEAR", "year"),
            QUARTER_OF_YEAR: translate("SHAKER.PROCESSOR.FilterAndFlagOnDate.SUMMARY.RELATIVE_PART.QUARTER_OF_YEAR", "quarter"),
            WEEK_OF_YEAR: translate("SHAKER.PROCESSOR.FilterAndFlagOnDate.SUMMARY.RELATIVE_PART.WEEK_OF_YEAR", "week"),
            MONTH_OF_YEAR: translate("SHAKER.PROCESSOR.FilterAndFlagOnDate.SUMMARY.RELATIVE_PART.MONTH_OF_YEAR", "month"),
            DAY_OF_MONTH: translate("SHAKER.PROCESSOR.FilterAndFlagOnDate.SUMMARY.RELATIVE_PART.DAY_OF_MONTH", "day"),
            HOUR_OF_DAY: translate("SHAKER.PROCESSOR.FilterAndFlagOnDate.SUMMARY.RELATIVE_PART.HOUR_OF_DAY", "hour")
        };

        function relativeDatePartsLabelPlural(part) {
            const translationID = "SHAKER.PROCESSOR.FilterAndFlagOnDate.SUMMARY.RELATIVE_PART." + part;
            return translate(translationID + ".PLURAL", translate(translationID) + "s");
        }

        function dateRangeCondition(params, columns, multipleColumns) {
            if (!params["min"] && params["max"]) {
                return translate('SHAKER.PROCESSOR.FilterAndFlagOnDate.SUMMARY.RANGE.MAX', "{{columns}} is before {{dateMax}}", {
                    columns: columns, multipleColumns: multipleColumns,
                    dateMax: numLiteral(DateUtilsService.convertDateFromTimezone(new Date(params["max"]), params["timezone_id"]).toISOString())
                });
            } else if (params["min"] && !params["max"]) {
                return translate('SHAKER.PROCESSOR.FilterAndFlagOnDate.SUMMARY.RANGE.MIN', "{{columns}} is after {{dateMin}}", {
                    columns: columns, multipleColumns: multipleColumns,
                    dateMin: numLiteral(DateUtilsService.convertDateFromTimezone(new Date(params["min"]), params["timezone_id"]).toISOString())
                });
            } else if (params["min"] && params["max"]) {
                return translate('SHAKER.PROCESSOR.FilterAndFlagOnDate.SUMMARY.RANGE.BETWEEN', "{{columns}} is between {{dateMin}} and {{dateMax}}", {
                    columns: columns, multipleColumns: multipleColumns,
                    dateMin: numLiteral(DateUtilsService.convertDateFromTimezone(new Date(params["min"]), params["timezone_id"]).toISOString()),
                    dateMax: numLiteral(DateUtilsService.convertDateFromTimezone(new Date(params["max"]), params["timezone_id"]).toISOString())
                });
            } else {
                return translate('SHAKER.PROCESSOR.FilterAndFlagOnDate.SUMMARY.MISSING_BOUNDS', "{{columns}} {{multipleColumns?'are':'is'}} not empty", {
                    columns: columns, multipleColumns: multipleColumns
                });
            }
        }
        function dateRelativeCondition(params, columns, multipleColumns) {
            if (params["part"] == null  || params["option"] == null || !ChartFilterUtils.isRelativeDateFilterEffective(params["part"], params["option"])) {
                return translate('SHAKER.PROCESSOR.FilterAndFlagOnDate.SUMMARY.MISSING_BOUNDS', "{{columns}} {{multipleColumns?'are':'is'}} not empty", {
                    columns: columns, multipleColumns: multipleColumns
                });
            }
            const translateIdParts = [];
            const defaultValueParts = [];
            if (params["option"].last) {
                if (params["option"].last > 1) {
                    translateIdParts.push('.LAST.MULTIPLE');
                    defaultValueParts.push('{{columns}} is within the last {{last}} {{parts}}')

                } else {
                    translateIdParts.push('.LAST');
                    defaultValueParts.push('{{columns}} is within the last {{part}}')
                }
            }
            if (params["option"].containsCurrentDatePart) {
                if (params["option"].isUntilNow) {
                    translateIdParts.push('.THIS.TO');
                    defaultValueParts.push('{{columns}} is between the beginning of this {{part}} and now')
                    return translate(`SHAKER.PROCESSOR.FilterAndFlagOnDate.SUMMARY.RELATIVE${translateIdParts.join('')}`, defaultValueParts.join(', '), { 
                        columns, 
                        multipleColumns, 
                        part: relativeDatePartsLabels[params["part"]], 
                        parts: relativeDatePartsLabelPlural(params["part"]),
                        last: params["option"].last,
                        next: params["option"].next
                    });
                } else {
                    translateIdParts.push('.THIS');
                    defaultValueParts.push('{{columns}} is in this {{part}}')
                }
                if (!params["option"].last && !params["option"].next) {
                    if (params["part"] === 'HOUR_OF_DAY') {
                        return translate('SHAKER.PROCESSOR.FilterAndFlagOnDate.SUMMARY.RELATIVE.THIS.HOUR_OF_DAY', "{{columns}} is today within this hour", {
                            columns, 
                            multipleColumns
                        });
                    } else if (params["part"] === 'DAY_OF_MONTH') {
                        return translate('SHAKER.PROCESSOR.FilterAndFlagOnDate.SUMMARY.RELATIVE.THIS.DAY_OF_MONTH', "{{columns}} is today", {
                            columns, 
                            multipleColumns 
                        });
                    } else {
                        return translate('SHAKER.PROCESSOR.FilterAndFlagOnDate.SUMMARY.RELATIVE.THIS.OTHER', "{{columns}} is in this {{part}}", { 
                            columns,
                            multipleColumns, 
                            part: relativeDatePartsLabels[params["part"]]
                        });
                    }
                }
            }
            
            if (params["option"].next) {
                if (params["option"].next > 1) {
                    translateIdParts.push('.NEXT.MULTIPLE');
                    defaultValueParts.push('{{columns}} is within the next {{next}} {{parts}}')
                } else {
                    translateIdParts.push('.NEXT');
                    defaultValueParts.push('{{columns}} is within the next {{part}}')
                }
            }

            return translate(`SHAKER.PROCESSOR.FilterAndFlagOnDate.SUMMARY.RELATIVE${translateIdParts.join('')}`, defaultValueParts.join(', '), { 
                columns, 
                multipleColumns, 
                part: relativeDatePartsLabels[params["part"]], 
                parts: relativeDatePartsLabelPlural(params["part"]),
                last: params["option"].last,
                next: params["option"].next
            });
        }

        function datePartCondition(params, columns, multipleColumns) {
            if (params["part"] && params["values"] && params["values"].length > 0) {
                if (params["part"] === 'INDIVIDUAL') {
                    return translate('SHAKER.PROCESSOR.FilterAndFlagOnDate.SUMMARY.PART.INDIVIDUAL', "{{columns}} with {{part}}", { columns: columns, multipleColumns: multipleColumns, part: datePartLabels[params["part"]] });
                } else {
                    return translate('SHAKER.PROCESSOR.FilterAndFlagOnDate.SUMMARY.PART.MULTIPLE', "{{columns}} with {{part}} in {{values}}", { columns: columns, multipleColumns: multipleColumns, part: datePartLabels[params["part"]], values: params["values"] });
                }
            }
            return translate('SHAKER.PROCESSOR.FilterAndFlagOnDate.SUMMARY.MISSING_BOUNDS', "{{columns}} {{multipleColumns?'are':'is'}} not empty", { columns: columns, multipleColumns: multipleColumns });
        }

        switch (params['filterType']) {
            case 'RANGE':
                return filterAndFlagSummary(params, dateRangeCondition);
            case 'RELATIVE':
                return filterAndFlagSummary(params, dateRelativeCondition);
            case 'PART':
                return filterAndFlagSummary(params, datePartCondition);
        }
    }

    function filterAndFlagOnDateRangeSummary(type, params){
        return filterAndFlagSummary(params, function(params, columns, multipleColumns) {
            const min = params["min"];
            const max = params["max"];
            if (!min && max) {
                return translate('SHAKER.PROCESSOR.FilterAndFlagOnDate.SUMMARY.RANGE.MAX', "{{columns}} {{multipleColumns?'are':'is'}} before {{dateMax}}", {
                    columns: columns,
                    multipleColumns: multipleColumns,
                    dateMax: numLiteral(max)
                });
            } else if (min && !max) {
                return translate('SHAKER.PROCESSOR.FilterAndFlagOnDate.SUMMARY.RANGE.MIN', "{{columns}} {{multipleColumns?'are':'is'}} after {{dateMin}}", {
                    columns: columns,
                    multipleColumns: multipleColumns,
                    dateMin: numLiteral(min)
                });
            } else if (min && max) {
                return translate('SHAKER.PROCESSOR.FilterAndFlagOnDate.SUMMARY.RANGE.BETWEEN', "{{columns}} {{multipleColumns?'are':'is'}} between {{dateMin}} and {{dateMax}}", {
                    columns: columns,
                    multipleColumns: multipleColumns,
                    min: numLiteral(min),
                    dateMax: numLiteral(max)
                });
            } else {
                return translate('SHAKER.PROCESSOR.FilterAndFlagOnDate.SUMMARY.MISSING_BOUNDS', "{{columns}} {{multipleColumns?'are':'is'}} not empty", {
                    columns: columns, multipleColumns: multipleColumns
                });
            }
        });
    }

    function geoDistanceCheckValid(params){
        if (!params["input2"] && params["compareTo"] === "COLUMN") {
            throw new StepIAE(translate('SHAKER.PROCESSOR.GeoDistance.ERROR.MISSING_PARAM_OTHER_COLUMN', "Missing parameter: Other column"));
        }
        if (params["compareTo"] === "GEOPOINT") {
            if (!params["refLatitude"]) throw new StepIAE(translate('SHAKER.PROCESSOR.GeoDistance.ERROR.MISSING_PARAM_LATITUDE', "Missing parameter:  Geopoint latitude"));
            if (!params["refLongitude"]) throw new StepIAE(translate('SHAKER.PROCESSOR.GeoDistance.ERROR.MISSING_PARAM_LONGITUDE', "Missing parameter: Geopoint longitude"));
        }
        if (!params["refGeometry"] && params["compareTo"] === "GEOMETRY") {
            throw new StepIAE(translate('SHAKER.PROCESSOR.GeoDistance.ERROR.MISSING_PARAM_GEOMETRY', "Missing parameter: Geometry"));
        }
    }

    function switchCaseCheckValid(params){
        if (!params["inputColumn"] || params.inputColumn  === null || params.inputColumn.length === 0) {
            throw new StepIAE(translate('SHAKER.PROCESSOR.SwitchCase.ERROR.MISSING_INPUT_COLUMN', "Input column not specified"));
        }
        if (!params["outputColumn"] || params.outputColumn==="" || params.outputColumn===null) {
            throw new StepIAE(translate('SHAKER.PROCESSOR.SwitchCase.ERROR.MISSING_OUTPUT_COLUMN', "Output column not specified"));
        }
    }

    function visualIfRuleCheckValid(params) {
        if (params.visualIfDesc.ifThen.filter.uiData.conditions.length===0) {
            throw new StepIAE(translate("SHAKER.PROCESSOR.VisualIfRule.ERROR.MISSING_IF_CONDITION", "Missing if condition"));
        }
        if (params.visualIfDesc.ifThen.actions.length===0) {
            throw new StepIAE(translate("SHAKER.PROCESSOR.VisualIfRule.ERROR.MISSING_THEN_ACTION", "Missing then action"));
        }

        params.visualIfDesc.elseIfThens.forEach((element) => {
            if (element.filter.uiData.conditions.length===0) {
                throw new StepIAE(translate("SHAKER.PROCESSOR.VisualIfRule.ERROR.MISSING_IF_CONDITION", "Missing if condition"));

            } else if (element.actions.length===0) {
                throw new StepIAE(translate("SHAKER.PROCESSOR.VisualIfRule.ERROR.MISSING_ELSE_IF_ACTION", "Missing else if action"));
            }
            element.actions.forEach((action) => {
                if(!("outputColumnName" in action) || !action.outputColumnName || action.outputColumnName==="") {
                    throw new StepIAE(translate("SHAKER.PROCESSOR.VisualIfRule.ERROR.MISSING_OUTPUT_COLUMN_NAME", "Missing output column name"));
                }
            });
        });

        params.visualIfDesc.ifThen.actions.forEach((action) => {
            if (!("outputColumnName" in action) || !action.outputColumnName || action.outputColumnName==="") {
                throw new StepIAE(translate("SHAKER.PROCESSOR.VisualIfRule.ERROR.MISSING_OUTPUT_COLUMN_NAME", "Missing output column name"));
            }
        });

        params.visualIfDesc.elseActions.forEach((action) => {
            if (!("outputColumnName" in action) || !action.outputColumnName || action.outputColumnName==="") {
                throw new StepIAE(translate("SHAKER.PROCESSOR.VisualIfRule.ERROR.MISSING_OUTPUT_COLUMN_NAME", "Missing output column name"));
            }
        });
    }

    function fmtMeaningLabel(x){
        return meaningName($filter("meaningLabel")(x));
    }

    var svc = {};

    svc.map = {
    "CurrencySplitter": {
        description: function(type, params){
            if (!hasAll(params, ["inCol"])) return null;
            return translate("SHAKER.PROCESSOR.CurrencySplitter.SUMMARY", actionVerb("Split") + " {{column}} between currency and amount", { column: inColName(params.inCol)});
        },
        icon: 'dku-icon-scissors'
    },
    "ColumnSplitter": {
        description: function(type, params){
            if (!hasAll(params, ["inCol", "separator"])) return null;
            return translate("SHAKER.PROCESSOR.ColumnSplitter.SUMMARY", actionVerb("Split") + " {{column}} on {{separator}}", { column: inColName(params.inCol), separator: anumLiteral(params.separator)});
        },
        icon: 'dku-icon-scissors'
    },
    "FindReplace": {
        checkValid: appliesToCheckValid,
        description: function(type, params){
            if (!hasAll(params, ["matching", "normalization"])) return null;
            if (!params.mapping) {
                params.mapping = [];
            }
            const nbReplaces = Object.keys(params.mapping).length;
            if (nbReplaces === 1) {
                return translate("SHAKER.PROCESSOR.FindReplace.SUMMARY.SINGLE", actionVerb("Replace") + " {{key}} by {{value}} in {{columns}}",
                    { key: anumLiteral(params.mapping[0].from), value: anumLiteral(params.mapping[0].to), columns: appliesToDescription(params)}
                );
            } else {
                return translate("SHAKER.PROCESSOR.FindReplace.SUMMARY.MULTIPLE", actionVerb("Replace") + " {{count}} values in {{columns}}",
                    { rawCount: nbReplaces, count: numLiteral(nbReplaces), columns: appliesToDescription(params)});
            }
        },
        icon: 'dku-icon-edit-note',
    },
    "SwitchCase": {
        checkValid: switchCaseCheckValid,
        description: function(type, params) {
            if (!hasAll(params, ["normalization"])) return null;
            if (!params.mapping) {
                params.mapping = [];
            }
            const nbRules = Object.keys(params.mapping).length;
            return translate("SHAKER.PROCESSOR.SwitchCase.SUMMARY", actionVerb("Apply") + " {{count}} {{rawCount === 1 ? 'rule':'rules'}} from {{inCol}} to {{outCol}}",
                { rawCount: nbRules, count: numLiteral(nbRules), inCol: inColName(params.inputColumn), outCol: inColName(params.outputColumn)});
        },
        icon: 'dku-icon-brackets-curly',
    },
    "VisualIfRule" : {
        checkValid: visualIfRuleCheckValid,
        description: function(type, params) {
            if (params.visualIfDesc.ifThen.filter.uiData.conditions.length===0){
                return actionVerb("Create ") +  "if, then, else statements";
            }
            let outputColumnNames = params.visualIfDesc.ifThen.actions.map(action => action.outputColumnName);
            if (params.visualIfDesc.elseIfThens.length !== 0){
                for (let elseIfThen of params.visualIfDesc.elseIfThens) {
                    outputColumnNames = outputColumnNames.concat(elseIfThen.actions.map(action => action.outputColumnName));
                }
            }
            if (params.visualIfDesc.elseActions.length!==0){
                outputColumnNames = outputColumnNames.concat(params.visualIfDesc.elseActions.map(action => action.outputColumnName));
            }
            outputColumnNames = Array.from(new Set(outputColumnNames));
            if (outputColumnNames.length === 1) {
                return translate("SHAKER.PROCESSOR.VisualIfRule.SUMMARY.SINGLE_OUTPUT", actionVerb("Create") + " column {{column}} from if, then, else statements", {
                    column: outColName(outputColumnNames[0])
                });
            } else if (outputColumnNames.length < 5) {
                return translate("SHAKER.PROCESSOR.VisualIfRule.SUMMARY.UP_TO_FIVE_OUTPUTS", actionVerb("Create") + " columns {{columns}} from if, then, else statements", {
                    columns: outColName(outputColumnNames.join(', '))
                });
            } else {
                return translate("SHAKER.PROCESSOR.VisualIfRule.SUMMARY.MULTIPLE_OUTPUTS", actionVerb("Create") + " {{count}} columns from if, then, else statements", {
                    count: numLiteral(outputColumnNames.length)
                });
            }
        },
        icon: 'dku-icon-brackets-curly'
    },
    "MatchCounter": {
        description: function(type, params){
            if (!hasAll(params, ["inCol", "outCol", "pattern", "normalizationMode", "matchingMode"])) return null;
            return translate("SHAKER.PROCESSOR.MatchCounter.SUMMARY", actionVerb("Count") + " number of occurrences of {{pattern}} in {{column}}", {
                pattern: (params.matchMode === 2) ? ("/" + params.pattern + "/") : params.pattern,
                column: inColName(params.inCol)
            });
        },
        icon: 'dku-icon-search',
    },
    "MeaningTranslate": {
        checkValid: appliesToCheckValid,
        description: function(type, params){
            if (!hasAll(params, ["meaningId"])) return null;
            return translate("SHAKER.PROCESSOR.MeaningTranslate.SUMMARY", actionVerb("Replace") + " values in {{columns}} using meaning {{meaning}}", {
                columns: appliesToDescription(params),
                meaning: strongify(params.meaningId)
            });
        },
        icon: 'dku-icon-edit-note',
    },
    "ColumnReorder": {
        checkValid: appliesToCheckValid,
        description: function(type, params){
            const p = {
                columns: appliesToDescription(params),
                referenceColumn: inColName(isBlank(params.referenceColumn) ? translate('SHAKER.PROCESSOR.ColumnReorder.DESCRIPTION.MISSING', "'missing'") : params.referenceColumn)
            }
            switch (params.reorderAction) {
                case "AT_START":
                    return translate("SHAKER.PROCESSOR.ColumnReorder.DESCRIPTION.AT_START", actionVerb("Move") + " {{columns}} at beginning", p);
                case "AT_END":
                    return translate("SHAKER.PROCESSOR.ColumnReorder.DESCRIPTION.AT_END", actionVerb("Move") + " {{columns}} at end", p);
                case "BEFORE_COLUMN":
                    return translate("SHAKER.PROCESSOR.ColumnReorder.DESCRIPTION.BEFORE_COLUMN", actionVerb("Move") + " {{columns}} before {{referenceColumn}}", p);
                case "AFTER_COLUMN":
                    return translate("SHAKER.PROCESSOR.ColumnReorder.DESCRIPTION.AFTER_COLUMN", actionVerb("Move") + " {{columns}} after {{referenceColumn}}", p);
            }
        },
        icon: 'dku-icon-text-align-justified',
    },
    "ColumnRenamer": {
        checkValid: (params) => isColumnRenamerValid(params, {throwError: true}),
        description: function(type, params) {
            if (!isColumnRenamerValid(params, {throwError: false})) {
                return null;
            }
            if (params["renamings"].length === 1) {
                const inCol = params["renamings"][0].from;
                const outCol = params["renamings"][0].to;
                return translate('SHAKER.PROCESSOR.ColumnRenamer.SUMMARY.SINGLE', actionVerb("Rename") + " column '{{inCol}}' to '{{outCol}}'", {
                    inCol: inColName(inCol),
                    outCol: inColName(outCol)
                });
            } else {
                const columnCount = numLiteral(params["renamings"].length);
                return translate('SHAKER.PROCESSOR.ColumnRenamer.SUMMARY.MULTIPLE', actionVerb("Rename") + " {{columnCount}} columns", {
                    columnCount: columnCount
                });
            }
        },
        paramsMatcher: function(query, params) {
            query = query.toLowerCase();
            const { renamings } = params;
            // If the step is empty, it should not match anything
            if (!renamings) return false;
            return renamings.some(({from, to}) => (from && from.toLowerCase().includes(query)) || (to && to.toLowerCase().includes(query)));
        },
        icon: 'dku-icon-edit-note'
    },

    "TextSimplifierProcessor": {
        description: function(type, params){
            if (!hasAll(params, ["inCol"])) return null;
            return translate('SHAKER.PROCESSOR.TextSimplifierProcessor.SUMMARY', '<span class="action-verb">Simplify</span> text in {{column}}', {
                column:inColName(params.inCol)
            });
        },
        icon: 'dku-icon-edit-note',
    },
    "Tokenizer": {
        description: function(type, params){
            if (!hasAll(params, ["inCol"])) return null;
            const inCol = inColName(params["inCol"]);
            if (params.operation === "TO_JSON") {
                return translate('SHAKER.PROCESSOR.Tokenizer.SUMMARY.TO_JSON', actionVerb("Tokenize") + " column {{column}} into JSON", { column: inCol });
            } else if (params.operation === "FOLD") {
                return translate('SHAKER.PROCESSOR.Tokenizer.SUMMARY.FOLD', actionVerb("Tokenize") + " column {{column}} and fold tokens (one per row)", { column: inCol });
            } else if (params.operation === "SPLIT") {
                return translate('SHAKER.PROCESSOR.Tokenizer.SUMMARY.SPLIT', actionVerb("Tokenize") + " column {{column}} and split tokens (one per column)", { column: inCol });
            }
        },
        icon: 'dku-icon-scissors',
    },
    "SplitIntoChunks": {
        description: function(type, params){
            if (!hasAll(params, ["inCol"])) return null;
            const inCol = inColName(params["inCol"]);
            return translate('SHAKER.PROCESSOR.SplitIntoChunks.SUMMARY', actionVerb("Split") + " column {{column}} into chunks", { column: inCol });
        },
        icon: 'dku-icon-scissors-horizontal',
    },
    "NGramExtract": {
        description: function(type, params){
            if (!hasAll(params, ["inCol"])) return null;
            const inCol = inColName(params["inCol"]);
            const op = params["operation"];
            if (op === "TO_JSON") {
                return translate('SHAKER.PROCESSOR.NGramExtract.SUMMARY.TO_JSON', actionVerb("Extract ngrams") + " from column {{column}} into JSON", { column: inCol });
            } else if (op === "FOLD") {
                return translate('SHAKER.PROCESSOR.NGramExtract.SUMMARY.FOLD', actionVerb("Extract ngrams") + " from column {{column}} and fold them (one per row)", { column: inCol });
            } else if (op === "SPLIT") {
                return translate('SHAKER.PROCESSOR.NGramExtract.SUMMARY.SPLIT', actionVerb("Extract ngrams") + " from column {{column}} and split them (one per column)", { column: inCol });
            }
        },
        icon: 'dku-icon-beaker',
    },
    "ExtractNumbers": {
        description: function(type, params){
            if (!hasAll(params, ["input"])) return null;
            const input = inColName(params.input);
            if (toBoolean(params["multipleValues"])) {
                return translate('SHAKER.PROCESSOR.ExtractNumbers.SUMMARY.MULTIPLE', actionVerb("Extract numbers") + " from {{column}}", { column: input });
            } else {
                return translate('SHAKER.PROCESSOR.ExtractNumbers.SUMMARY.SINGLE',  actionVerb("Extract a number") + " from {{column}}", { column: input });
            }
        },
        icon: 'dku-icon-beaker',
    },
    "NumericalFormatConverter": {
        checkValid: appliesToCheckValid,
        description: function(type, params){
            if (!hasAll(params, ["inFormat", "outFormat"])) return null;
            const formats = {
                RAW: translate('SHAKER.PROCESSOR.NumericalFormatConverter.SUMMARY.RAW_FORMAT', "Raw"),
                EN: translate('GLOBAL.LANGUAGE.ENGLISH', "English"),
                FR: translate('GLOBAL.LANGUAGE.FRENCH', "French"),
                IT: translate('GLOBAL.LANGUAGE.ITALIAN', "Italian"),
                GE: translate('GLOBAL.LANGUAGE.GERMAN', "German")
            };
            return translate('SHAKER.PROCESSOR.NumericalFormatConverter.SUMMARY',
                actionVerb("Convert number formats") + " from {{inFormat}} to {{outFormat}} in {{columns}}",
                { inFormat: formats[params['inFormat']], outFormat: formats[params['outFormat']], columns: appliesToDescription(params) });
        },
        icon: 'dku-icon-text-superscript',
    },
    "TypeSetter": {
        description: function(type, params){
            if (!hasAll(params, ["column", "type"])) return null;
            const column = inColName(params["column"]);
            const typeloc = sanitize(params["type"]);
            const typestr = meaningName(typeloc);
            return translate('SHAKER.PROCESSOR.TypeSetter.SUMMARY', "Set meaning of {{column}} to {{meaning}}", { column: column, meaning: typestr });
        },
        icon: 'dku-icon-edit-note',
        postLinkFn : function(scope, element) {
            if (angular.isUndefined(scope.step.params.type)) {
                scope.step.params.type = "Text";
            }
        }
    },
    "StringTransformer" : {
        checkValid : function(params) {
            appliesToCheckValid(params);
            if (!params.mode) {
                throw new StepIAE(translate('SHAKER.PROCESSOR.StringTransformer.ERROR.MODE_NOT_SPECIFIED', "Mode not specified"));
            }
        },
        description : function(type, params) {
            if (!hasAll(params, ["mode"])) return null;
            const mode = sanitize(params["mode"]).toLowerCase();
            return translate('SHAKER.PROCESSOR.StringTransformer.SUMMARY', 'Perform <span class="action-verb">{{mode}}</span> on {{columns}}',
                { mode: mode, columns: appliesToDescription(params) });
        },
        postLinkFn : function(scope, element) {
            if (angular.isUndefined(scope.step.params.mode)) {
                scope.step.params.mode = "TO_LOWER";
            }
        },
        icon: 'dku-icon-edit-note'
    },
    "ColumnsConcat" : {
        description: function(type, params) {
            if (!hasAll(params, ["outputColumn"])) return null;
            return translate('SHAKER.PROCESSOR.ColumnsConcat.SUMMARY', actionVerb("Concatenate") + " columns in {{outCol}}", { outCol: outColName(params.outputColumn) });
        },
        icon: 'dku-icon-arrow-expand-exit'
    },
    "ArrayExtractProcessor" : {
        checkValid: (params) => defaultIsValidIfHasColumn(params, "input", {throwError: true}),
        description: function(type, params) {
            if (!defaultIsValidIfHasColumn(params, "input", {throwError: false})) {
                return null;
            }
            if (params.mode === "INDEX") {
                return translate('SHAKER.PROCESSOR.ArrayExtractProcessor.SUMMARY.INDEX', actionVerb("Extract") + " elt {{index}} from {{column}}",
                    { index: numLiteral(params.index), column: inColName(params.input) });
            } else {
                return translate('SHAKER.PROCESSOR.ArrayExtractProcessor.RANGE', actionVerb("Extract") + " elts {{begin}}-{{end}} from {{column}}",
                    { begin: numLiteral(params.begin), end: numLiteral(params.end), column: inColName(params.input) });
            }
        },
        icon: 'dku-icon-arrow-external-link'
    },
    "ArraySortProcessor" : {
        checkValid: (params) => defaultIsValidIfHasColumn(params, "input", {throwError: true}),
        description: function(type, params) {
            if (!defaultIsValidIfHasColumn(params, "input", {throwError: false})) {
                return null;
            }
            return translate('SHAKER.PROCESSOR.ArraySortProcessor.SUMMARY', actionVerb("Sort") + " array in {{column}}", { column: inColName(params.input) });
        },
        icon: 'dku-icon-sort-ascending'
    },
    "ColumnsSelector": {
        checkValid: appliesToCheckValid,
        description: function(type, params) {
            return toBoolean(params["keep"]) ?
                translate('SHAKER.PROCESSOR.ColumnsSelector.FORM.KEEP', '<span class="action-verb">Keep only</span> {{columns}}', { columns: appliesToDescription(params) }) :
                translate('SHAKER.PROCESSOR.ColumnsSelector.FORM.REMOVE', '<span class="action-verb">Remove</span> {{columns}}', { columns: appliesToDescription(params) });
        },
        icon: 'dku-icon-filter',
    },
    "FilterOnValue": {
        checkValid: appliesToCheckValid,
        description: function(type, params){
            if (!hasAll(params, ["matchingMode", "values"]) || params["values"].length === 0) return null;
            return filterAndFlagSummary(params, filterAndFlagOnValueCondition);
        },
        impactVerb : filterAndFlagImpactVerb,
        icon: 'dku-icon-trash',
    },
    "FlagOnValue": {
        checkValid : function(params) {
            appliesToCheckValid(params);
            flagCheckValid(params);
        },
        description: function(type, params){
            if (!hasAll(params, ["matchingMode", "values", "flagColumn"]) || params["values"].length === 0) return null;
            return filterAndFlagSummary(params, filterAndFlagOnValueCondition);
        },
        impactVerb : filterAndFlagImpactVerb,
        icon: 'dku-icon-flag',
    },
    "FilterOnNumericalRange": {
        checkValid: function(params) {
            appliesToCheckValid(params);
            if (params["min"] === undefined && params["max"] === undefined) {
                throw new StepIAE(translate('SHAKER.PROCESSOR.FilterAndFlagOnNumericalRange.ERROR.BOUNDS_NOT_DEFINED', "Bounds are not defined"));
            }
        },
        description: function(type, params) {
            return filterAndFlagSummary(params, filterAndFlagOnNumericRangeCondition);
        },
        impactVerb: filterAndFlagImpactVerb,
        icon: 'dku-icon-trash',
    },
    "FlagOnNumericalRange": {
        checkValid: function(params) {
            appliesToCheckValid(params)
            flagCheckValid(params);
            if (params["min"] === undefined && params["max"] === undefined) {
                throw new StepIAE(translate('SHAKER.PROCESSOR.FilterAndFlagOnNumericalRange.ERROR.BOUNDS_NOT_DEFINED', "Bounds are not defined"));
            }
        },
        description: function(type, params) {
            return filterAndFlagSummary(params, filterAndFlagOnNumericRangeCondition);
        },
        impactVerb : filterAndFlagImpactVerb,
        icon: 'dku-icon-flag',
    },
    "FilterOnDate": {
        checkValid: appliesToCheckValid,
        description: filterAndFlagOnDateSummary,
        impactVerb: filterAndFlagImpactVerb,
        icon: 'dku-icon-trash',
    },
    "FlagOnDate": {
        checkValid: function(params) {
            appliesToCheckValid(params);
            flagCheckValid(params);
        },
        description: filterAndFlagOnDateSummary,
        impactVerb: filterAndFlagImpactVerb,
        icon: 'dku-icon-flag',
    },
    "FilterOnDateRange": {
        checkValid : function(params) {
            appliesToCheckValid(params);
            if (params["max"]) checkDate(params["max"], "upper-bound date");
            if (params["min"]) checkDate(params["min"], "lower-bound date");
            if (!(params["max"] || params["min"])) {
                throw new StepIAE("Input at least one date bound.");
            }
        },
        description: filterAndFlagOnDateRangeSummary,
        impactVerb : filterAndFlagImpactVerb,
        icon: 'dku-icon-trash',
    },
    "FlagOnDateRange": {
        checkValid : function(params) {
            appliesToCheckValid(params);
            flagCheckValid(params);
            if (!(params["max"] || params["min"])) {
                throw new StepIAE("Input at least one date bound.");
            }
            if (params["max"]) checkDate(params["max"], "upper-bound date");
            if (params["min"]) checkDate(params["min"], "lower-bound date");
        },
        description: filterAndFlagOnDateRangeSummary,
        impactVerb : filterAndFlagImpactVerb,
        icon: 'dku-icon-flag',
    },
    "FilterOnCustomFormula" :{
        checkValid : function(params) {
            if (isBlank(params.expression)) throw new StepIAE(translate('SHAKER.PROCESSOR.FilterAndFlagOnCustomFormula.ERROR.EXPRESSION_NOT_SPECIFIED', "Expression not specified"));
            if (params.action === "CLEAR_CELL" || params.action === "DONTCLEAR_CELL") {
                if (isBlank(params.clearColumn)) {
                    throw new StepIAE(translate('SHAKER.PROCESSOR.FilterAndFlagOnCustomFormula.ERROR.COLUMN_TO_CLEAR_NOT_SPECIFIED', "Column to clear not specified"))
                }
            }
        },
        description : function(type, params) {
            const conditionFunction = function(params) {
                return params.expression.length > 20 ? translate('SHAKER.PROCESSOR.FilterAndFlagOnCustomFormula.SUMMARY.FORMULA_IS_TRUE', "formula is true") : anumLiteral(params.expression)
            };
            return filterAndFlagSummary(params, conditionFunction, params.clearColumn);
        },
        shouldExpandFormula: function (params) {
            return params.expression.length > 20;
        },
        impactVerb : filterAndFlagImpactVerb,
        icon : "dku-icon-trash"
    },
    "FlagOnCustomFormula" : {
        checkValid : function(params) {
            if (isBlank(params.expression)) throw new StepIAE(translate('SHAKER.PROCESSOR.FilterAndFlagOnCustomFormula.ERROR.EXPRESSION_NOT_SPECIFIED', "Expression not specified"));
            if (isBlank(params.flagColumn)) {
                throw new StepIAE(translate('SHAKER.PROCESSOR.FilterAndFlagOnCustomFormula.ERROR.COLUMN_TO_FLAG_NOT_SPECIFIED', "Column to flag not specified"))
            }
        },
        description : function(type, params) {
            const conditionFunction = function(params) {
                return params.expression.length > 20 ? translate('SHAKER.PROCESSOR.FilterAndFlagOnCustomFormula.SUMMARY.FORMULA_IS_TRUE', "formula is true") : anumLiteral(params.expression)
            };
            return filterAndFlagSummary(params, conditionFunction, params.clearColumn);
        },
        shouldExpandFormula: function(params) {
            return params.expression.length > 20;
        },
        impactVerb : filterAndFlagImpactVerb,
        icon : "dku-icon-flag"
    },

    /* Cleansing */
    "FilterOnBadType": {
        checkValid: appliesToCheckValid,
        description: function(type, params){
            if (!hasAll(params, ["type"])) return null;
            const conditionFunction = function(params, columns, multipleColumns) {
                return translate('SHAKER.PROCESSOR.FilterAndFlagOnBadType.SUMMARY.CONDITION', "{{columns}} {{multipleColumns?'are':'is'}} not a valid {{meaning}}",
                    {columns: columns, multipleColumns:multipleColumns, meaning:fmtMeaningLabel(params["type"])});
            };
            return filterAndFlagSummary(params, conditionFunction);
        },
        icon: 'dku-icon-trash',
        impactVerb : filterAndFlagImpactVerb,
        postLinkFn : function(scope, element) {
            if (angular.isUndefined(scope.step.params.type)) {
                scope.step.params.type = "Text";
            }
        }
    },
    "FlagOnBadType": {
        checkValid : function(params){
            appliesToCheckValid(params);
            flagCheckValid(params);
        },
        description: function(type, params){
            if (!hasAll(params, ["type", "flagColumn"])) return null;
            const conditionFunction = function(params, columns, multipleColumns) {
                return translate('SHAKER.PROCESSOR.FilterAndFlagOnBadType.SUMMARY.CONDITION', "{{columns}} {{multipleColumns?'are':'is'}} not a valid {{meaning}}",
                    {columns: columns, multipleColumns:multipleColumns, meaning:fmtMeaningLabel(params["type"])});
            };
            return filterAndFlagSummary(params, conditionFunction);
        },
        icon: 'dku-icon-flag',
        impactVerb : filterAndFlagImpactVerb,
        postLinkFn : function(scope, element) {
            if (angular.isUndefined(scope.step.params.type)) {
                scope.step.params.type = "Text";
            }
        }
    },
    "RemoveRowsOnEmpty": {
        checkValid: appliesToCheckValid,
        description: function(type, params){
            const applyToAllOrPattern = params["appliesTo"] === "ALL" || params["appliesTo"] === "PATTERN";
            const p = { columns: appliesToDescription(params), hasMultipleColumns: (params["appliesTo"] != "SINGLE_COLUMN" && params["columns"] && params["columns"].length > 1) || applyToAllOrPattern };
            if (params["keep"]) {
                return translate('SHAKER.PROCESSOR.RemoveRowsOnEmpty.SUMMARY.KEEP', actionVerb("Keep") + " rows with empty values in {{hasMultipleColumns ? 'at least one of ' : '' }}{{columns}}", p);
            } else {
                return translate('SHAKER.PROCESSOR.RemoveRowsOnEmpty.SUMMARY.REMOVE', actionVerb("Remove") + " rows with empty values in {{hasMultipleColumns ? 'at least one of ' : '' }}{{columns}}", p);
            }
        },
        impactVerb : Fn.cst("deleted"),
        icon: 'dku-icon-trash',
    },
    "FillEmptyWithValue": {
        checkValid: appliesToCheckValid,
        description: function(type, params){
            if (!hasAll(params, ["value"])) return null;
            return translate('SHAKER.PROCESSOR.FillEmptyWithValue.SUMMARY', actionVerb("Fill empty") +" cells of {{columns}} with '{{value}}'", {
                columns: appliesToDescription(params),
                value: anumLiteral(truncate(params["value"], 60))
            });
        },
        impactVerb : Fn.cst("filled"),
        icon: 'dku-icon-circle-fill',
    },
    "FillEmptyWithComputedValue": {
        checkValid: appliesToCheckValid,
        description: function(type, params){
            if (!hasAll(params, ["mode"])) return null;
            return translate('SHAKER.PROCESSOR.FillEmptyWithComputedValue.SUMMARY', actionVerb("Impute missing values") + " of {{columns}} with {{value}}", {
                columns: appliesToDescription(params),
                value: translate('SHAKER.PROCESSOR.FillEmptyWithComputedValue.SUMMARY.' + params["mode"], strongify(params["mode"]).toLowerCase())
            });
        },
        impactVerb : Fn.cst("filled"),
        icon: 'dku-icon-circle-fill',
    },
    "SplitInvalidCells" : {
        description : function(type, params) {
            if (!hasAll(params, ["column"])) return null;
            return translate('SHAKER.PROCESSOR.SplitInvalidCells.SUMMARY', actionVerb("Move invalid") + " cells of {{column}} to {{invalidColumn}}", {
                column: inColName(params["column"]),
                invalidColumn: outColName(params["invalidColumn"])
            });
        },
        impactVerb : Fn.cst("splitted"),
        icon : 'dku-icon-scissors'
    },
    "UpDownFiller" : {
        description : function(type, params) {
            if (!params.columns || params.columns.length < 1) return null;
            return translate('SHAKER.PROCESSOR.UpDownFiller.SUMMARY', actionVerb("Fill empty") +" cells of {{columns}} with {{up?'next':'previous'}} value", {
                columns: inColName(params["columns"]),
                up: toBoolean(params["up"])
            });
        },
        impactVerb : Fn.cst("filled"),
        icon : 'dku-icon-circle-fill'
    },
    "LongTailGrouper" : {
        description : function(type, params) {
            if (!hasAll(params, ["column"])) return null;
            return translate('SHAKER.PROCESSOR.LongTailGrouper.SUMMARY', actionVerb("Merge") + " long-tail values in {{column}}", {
                column: inColName(params["column"]),
            });
        },
        impactVerb : Fn.cst("merged"),
        icon: 'dku-icon-arrow-expand-exit'
    },
    "ComputeNTile" : {
        checkValid: appliesToCheckValid,
        description: function(type, params){
            if (!hasAll(params, ["n"])) return null;
            return translate('SHAKER.PROCESSOR.ComputeNTile.SUMMARY', actionVerb("Compute quantile") +" of {{columns}} with {{binCount}} bins", {
                columns: appliesToDescription(params),
                binCount: strongify(params["n"])
            });
        },
        impactVerb : Fn.cst("filled"),
        icon: 'dku-icon-circle-fill'
    },
    "MergeLongTailValues" : {
        checkValid: appliesToCheckValid,
        description: function(type, params){
            if (!hasAll(params, ["thresholdMode", "replacementValue"])) return null;
            return translate('SHAKER.PROCESSOR.MergeLongTailValues.SUMMARY', actionVerb("Replace long tail") + " of {{columns}} with {{value}}", {
                columns: appliesToDescription(params),
                value: strongify(params["replacementValue"])
            });
        },
        impactVerb : Fn.cst("filled"),
        icon: 'dku-icon-circle-fill'
    },

    /* Types */
    "QueryStringSplitter": {
        description: function(type, params){
            if (!hasAll(params, ["column"])) return null;
            return translate('SHAKER.PROCESSOR.QueryStringSplitter.SUMMARY', actionVerb("Split") + " HTTP Query string in {{column}}", {
                column: inColName(params["column"])
            });
        },
        icon: 'dku-icon-scissors',
    },
    "URLSplitter": {
        description: function(type, params){
            if (!hasAll(params, ["column"])) return null;
            return translate('SHAKER.PROCESSOR.URLSplitter.SUMMARY', actionVerb("Split") + " URL in {{column}}", {
                column: inColName(params["column"])
            });
        },
        icon: 'dku-icon-scissors',
    },
    "EmailSplitter": {
        description: function(type, params){
            if (!hasAll(params, ["column"])) return null;
            return translate('SHAKER.PROCESSOR.EmailSplitter.SUMMARY', actionVerb("Split") + " email in {{column}}", {
                column: inColName(params["column"])
            });
        },
        icon: 'dku-icon-scissors',
    },
    "VisitorIdGenerator" : {
        description : function(type, params) {
            if (!hasAll(params, ["outputColumn"])) return null;
            return translate('SHAKER.PROCESSOR.VisitorIdGenerator.SUMMARY', actionVerb("Generate") +" visitor-id in {{column}}", {
                column: outColName(params["outputColumn"])
            });
        },
        icon : 'dku-icon-beaker'
    },
    "RegexpExtractor": {
        description: function(type, params){
            if (!hasAll(params, ["column","pattern"])) return null;
            return translate("SHAKER.PROCESSOR.RegexpExtractor.SUMMARY", actionVerb("Extract") +" from {{column}} with {{pattern}}", {
                column: inColName(params.column),
                pattern: anumLiteral(params.pattern)
            });
        },
        icon: 'dku-icon-beaker',
    },
    "UserAgentClassifier": {
        description: function(type, params){
            if (!hasAll(params, ["column"])) return null;
            return translate("SHAKER.PROCESSOR.UserAgentClassifier.SUMMARY", actionVerb("Classify") +" User-Agent in {{column}}", {
                column: inColName(params.column)
            });
        },
        icon: 'dku-icon-laptop',
    },
    "UseRowAsHeader": {
        description: function(type, params){
            if (!hasAll(params, ["rowIdx"])) return null;
            return translate("SHAKER.PROCESSOR.UseRowAsHeader.SUMMARY", "Use row {{index}}'s values as column names", {
                index: strongify(sanitize(params["rowIdx"]))
            });
        },
        icon: 'dku-icon-list-numbered',
    },

    /* Basic */
    "JSONPathExtractor" : {
        description: function(type, params) {
            if (!hasAll(params, ["inCol"])) return null;
            return translate("SHAKER.PROCESSOR.JSONPathExtractor.SUMMARY", actionVerb("Extract") + " from {{column}} with JSONPath {{expression}}", {
                column: inColName(params.inCol),
                expression: anumLiteral(truncate(params.expression, 60))
            });
        },
        icon: 'dku-icon-beaker',
    },

    /* Time */
    "DateParser": {
        checkValid: appliesToCheckValid,
        description: function(type, params){
            if (!hasAll(params, ["formats"])) return null;
            return translate('SHAKER.PROCESSOR.DateParser.SUMMARY', actionVerb("Parse date") + " in {{columns}}", {columns: appliesToDescription(params)});
        },
        icon: 'dku-icon-calendar',
    },
    "DateIncrement": {
        checkValid : function(params) {
            if (params["incrementBy"] !== 'STATIC' && params["valueCol"] === undefined) {
                throw new StepIAE("Value column is not defined");
            }
        },
        description: function (type, params) {
            if (!hasAll(params, ["inCol"])) return null;
            function datePartsLabelSingular(part) {
                return translate("SHAKER.PROCESSOR.DateIncrement.PART." + part, part.toLowerCase());
            }
            function datePartsLabelPlural(part) {
                const translationID = "SHAKER.PROCESSOR.DateIncrement.SUMMARY.PART." + part;
                return translate(translationID + ".PLURAL", datePartsLabelSingular(part) + "s");
            }
            function datePartsLabel(part, count) {
                return count > 1 ? datePartsLabelPlural(part) : datePartsLabelSingular(part);
            }
            const p = {
                column: inColName(params["inCol"]),
            };
            if (params["incrementBy"] === 'STATIC') {
                p.increment = params["increment"] > -1;
                p.count = Math.abs(params["increment"]);
                p.part = datePartsLabel(params["datePart"], Math.abs(params["increment"]));
            } else if (params["valueCol"] === undefined) {
                return null;
            } else {
                p.increment = true;
                p.count = inColName(params["valueCol"]);
                p.part = datePartsLabelPlural(params["datePart"]);
            }
            return translate("SHAKER.PROCESSOR.DateIncrement.SUMMARY", "<span class=\"action-verb\">{{increment?'Increment':'Decrement'}} date</span> in {{column}} by {{count}} {{part}}", p);
        },
        icon: "dku-icon-calendar",
    },
    "DateTruncate": {
        description: function (type, params) {
            if (!hasAll(params, ["inCol", "datePart"])) return null;
            return translate("SHAKER.PROCESSOR.DateTruncate.SUMMARY", actionVerb("Truncate") + " {{column}} on {{part}}", {
                column: inColName(params["inCol"]),
                part: anumLiteral(translate('SHAKER.PROCESSOR.DateTruncate.PART.' + params["datePart"], params["datePart"].toLowerCase()))
            });
        },
        icon: "dku-icon-calendar",
    },
    "DateFormatter": {
        description: function(type, params){
            if (!hasAll(params, ["inCol"])) return null;
            return translate("SHAKER.PROCESSOR.DateFormatter.SUMMARY", actionVerb("Format date") + " in {{column}}", {
                column: inColName(params["inCol"])
            });
        },
        icon: 'dku-icon-calendar',
    },

    "DateDifference": {
        checkValid : function(params) {
            if(params.compareTo === "DATE"){
                if(!params.refDate){
                    throw new StepIAE(translate("SHAKER.PROCESSOR.DateDifference.ERROR.MISSING_REF_DATE", "No date specified"));
                } else {
                     checkDate(params.refDate)
                }
            } else if (params.compareTo === "COLUMN") {
                if (!params.input2) {
                    throw new StepIAE(translate("SHAKER.PROCESSOR.DateDifference.ERROR.MISSING_INPUT2", "No column specified"))
                }
            }
             if (params.excludeHolidays === true || params.excludeWeekends === true) {
                 isHolidaysComputerValid(params, {throwError: true});
             }
         },
        description: function(type, params){
            if (!hasAll(params, ["input1", "output", "outputUnit", "compareTo"])) return null;
            const in1 = inColName(params["input1"]);
            const in2 = inColName(params["input2"]);
            if (params.compareTo === "NOW") {
                return translate("SHAKER.PROCESSOR.DateDifference.SUMMARY.NOW", actionVerb("Compute time difference") + " between {{date}} and now", {
                    date: in1
                });
            } else if (params.compareTo === "COLUMN") {
                return translate("SHAKER.PROCESSOR.DateDifference.SUMMARY.BETWEEN", actionVerb("Compute time difference") + " between {{date1}} and {{date2}}", {
                    date1: in1,
                    date2: in2
                });
            } else if (params.compareTo === "DATE") {
                return translate("SHAKER.PROCESSOR.DateDifference.SUMMARY.REFERENCE", actionVerb("Compute time difference") + " between {{date}} and a reference", {
                    date: in1
                });
            }
        },
        icon: 'dku-icon-calendar',
    },
    "UNIXTimestampParser": {
        description : function(type, params) {
            if (!hasAll(params, ["inCol"])) return null;
            return translate("SHAKER.PROCESSOR.UNIXTimestampParser.SUMMARY", actionVerb("Convert UNIX") + " timestamp in {{column}}", {
                column: inColName(params["inCol"])
            });
        },
        icon: 'dku-icon-calendar'
    },
    "JSONFlattener": {
        description : function(type, params) {
            if (!hasAll(params, ["inCol"])) return null;
            return translate("SHAKER.PROCESSOR.JSONFlattener.SUMMARY", actionVerb("Unnest") + " object in {{column}}", {
                column: inColName(params["inCol"])
            });
        },
        icon: 'dku-icon-scissors'
    },
    "DateComponentsExtractor": {
        checkValid: (params) => isDateComponentsExtractorValid(params, {throwError: true}),
        description: function(type, params) {
            if (!isDateComponentsExtractorValid(params, {throwError: false})) {
                return null;
            }
            return translate("SHAKER.PROCESSOR.DateComponentsExtractor.SUMMARY", actionVerb("Extract date") + " components from {{column}}", {
                column: inColName(params["column"])
            });
        },
        icon: 'dku-icon-calendar'
    },
    "HolidaysComputer": {
        checkValid: (params) => isHolidaysComputerValid(params, {throwError: true}),
        description: function(type, params) {
            if (!isHolidaysComputerValid(params, {throwError: false})) {
                return null;
            }
            if (params["calendar_id"] !== "extract_from_column") {
                return translate("SHAKER.PROCESSOR.HolidaysComputer.SUMMARY.CALENDAR_ID", actionVerb("Extract holidays") + " from {{column}} using calendar {{calendar}}", {
                    column: inColName(params["inCol"]),
                    calendar: anumLiteral(params["calendar_id"])
                });
            } else {
                return translate("SHAKER.PROCESSOR.HolidaysComputer.SUMMARY.CALENDAR_SRC", actionVerb("Extract holidays") + " from {{column}} using the country in {{calendar}}", {
                    column: inColName(params["inCol"]),
                    calendar: anumLiteral(params["calendar_src"])
                });
            }
        },
        icon: 'dku-icon-calendar'
    },

    /* Numbers */
    "BinnerProcessor": {
        checkValid: (params) => defaultIsValidIfHasColumn(params, "input", {throwError: true}),
        description: function(type, params) {
            if (!defaultIsValidIfHasColumn(params, "input", {throwError: false})) {
                return null;
            }
            return translate("SHAKER.PROCESSOR.BinnerProcessor.SUMMARY.CALENDAR_SRC", actionVerb("Bin") + " values in {{column}}", {
                column: inColName(params["input"])
            });
        },
        icon: 'dku-icon-scissors',
    },
    "NumericalCombinator": {
        checkValid : function(params) {
            appliesToCheckValid(params);
            const allOps = ["add", "sub", "mul", "div"];
            let hasOp = false;
            for (let opidx in allOps) {
                if (toBoolean(params[allOps[opidx]])) {
                    hasOp = true;
                }
            }
            if (!hasOp) {
                throw new StepIAE(translate("SHAKER.PROCESSOR.NumericalCombinator.ERROR.MISSING_OPERATION", "No selected operation"));
            }
        },
        description: function(type, params){
            const ops = [],
                allOps = ["add", "sub", "mul", "div"],
                opDesc = ["+", "-", "×", "÷"];
            for (let opidx in allOps) {
                if (toBoolean(params[allOps[opidx]])) {
                    ops.push(opDesc[opidx]);
                }
            }
            if (!ops.length) { return null; }
            const p = {
                columns: appliesToDescription(params),
                opsCount: ops.length,
                ops: anumLiteral(ops.join(", ")),
            }
            if (hasNonEmptyParam(params, "prefix")) {
                p.prefix = anumLiteral(params.prefix);
                return translate("SHAKER.PROCESSOR.NumericalCombinator.SUMMARY.PREFIX", actionVerb("Combine values") +" in {{columns}} with {{opsCount>1?'operations':'operation'}} {{ops}} into {{prefix}}*", p);
            } else {
                return translate("SHAKER.PROCESSOR.NumericalCombinator.SUMMARY", actionVerb("Combine values") +" in {{columns}} with {{opsCount>1?'operations':'operation'}} {{ops}}", p);
            }
        },
        icon: 'dku-icon-plus',
    },
    "RoundProcessor": {
        checkValid: appliesToCheckValid,
        description: function(type, params){
            return translate("SHAKER.PROCESSOR.RoundProcessor.SUMMARY", actionVerb("Round") + " values in {{columns}}", { columns: appliesToDescription(params) });
        },
        icon: 'dku-icon-text-superscript',
    },
    "MinMaxProcessor": {
        description: function(type, params){
            if (!hasAll(params, ["columns"])) return null;
            const min = parseFloat(params["lowerBound"]);
            const max = parseFloat(params["upperBound"]);
            const hasMin = !isNaN(min);
            const hasMax = !isNaN(max);
            if ((!hasMin && !hasMax) || min > max) return null;
            const p = {
                clear: params['clear'],
                lowerBound: numLiteral(params["lowerBound"]),
                upperBound: numLiteral(params["upperBound"]),
                column: inColName(params["columns"])
            }
            if (hasMin && hasMax) {
                return translate("SHAKER.PROCESSOR.MinMaxProcessor.SUMMARY.MIN_MAX",
                    "<span class=\"action-verb\">{{clear?'Clear':'Clip'}}</span> values outside [{{lowerBound}},{{upperBound}}] in {{column}}", p);
            } else if (hasMin) {
                return translate("SHAKER.PROCESSOR.MinMaxProcessor.SUMMARY.MIN",
                    "<span class=\"action-verb\">{{clear?'Clear':'Clip'}}</span> values &lt;&nbsp;{{lowerBound}} in {{column}}", p);
            } else {
                return translate("SHAKER.PROCESSOR.MinMaxProcessor.SUMMARY.MAX",
                    "<span class=\"action-verb\">{{clear?'Clear':'Clip'}}</span> values &gt;&nbsp;{{upperBound}} in {{column}}", p);
            }
        },
        icon: 'dku-icon-beaker',
    },
    "MeanProcessor": {
        checkValid: appliesToCheckValid,
        description: function(type, params){
            if (!hasAll(params, ["columns"])) return null;
            return translate("SHAKER.PROCESSOR.MeanProcessor.SUMMARY", actionVerb('Compute') + " mean of {{columns}}", {
                columns: appliesToDescription(params)
            });
        },
        icon: 'dku-icon-text-superscript',
    },
    "CurrencyConverterProcessor": {
        checkValid: (params) => isCurrencyConverterProcessorValid(params, {throwError: true}),
        description: function(type, params) {
            if (!isCurrencyConverterProcessorValid(params, {throwError: false})) {
                return null;
            }
            return translate("SHAKER.PROCESSOR.CurrencyConverterProcessor.SUMMARY", actionVerb("Convert") + " {{column}} to {{currency}}", {
                column: inColName(params["inputColumn"]),
                currency: anumLiteral(params["outputCurrency"])
            });
        },
        icon: 'dku-icon-edit-note'
    },

    /* Geo */
    "GeoIPResolver": {
        description: function(type, params){
            if (!hasAll(params, ["inCol"])) return null;
            return translate("SHAKER.PROCESSOR.GeoIPResolver.SUMMARY", actionVerb("Geo-locate IP") + " in {{column}}", {
                column: inColName(params["inCol"])
            });
        },
        icon: 'dku-icon-globe',
    },
    "GeoPointExtractor": {
        description: function(type, params){
            if (!hasAll(params, ["column"])) return null;
            return translate("SHAKER.PROCESSOR.GeoPointExtractor.SUMMARY", actionVerb("Extract") + " latitude/longitude from {{column}}", {
                column: inColName(params["column"])
            });
        },
        icon: 'dku-icon-globe',
    },
    "GeoPointCreator": {
        description: function(type, params){
            if (!hasAll(params, ["out_column","lat_column","lon_column"])) return null;
            return translate("SHAKER.PROCESSOR.GeoPointCreator.SUMMARY", actionVerb("Create GeoPoint") + " from {{latitude}} & {{longitude}}", {
                latitude: inColName(params["lat_column"]),
                longitude: inColName(params["lon_column"])
            });
        },
        icon: 'dku-icon-globe',
    },

    "NearestNeighbourGeoJoiner" : {
        description : function(type, params) {
            return translate("SHAKER.PROCESSOR.NearestNeighbourGeoJoiner.SUMMARY", actionVerb("Geo-join") + " from dataset {{rightInput}}", {
                rightInput: inColName(params["rightInput"])
            });
        },
        icon : 'dku-icon-globe'
    },
    "Geocoder" : {
        description : function(type, params) {
            if (!hasAll(params, ["inCol","api", "prefixOutCol", "apiKey"])) return null;
            return translate("SHAKER.PROCESSOR.Geocoder.SUMMARY", actionVerb("Geocode") + " from address {{column}}", {
                column: inColName(params["inCol"])
            });
        },
        icon : 'dku-icon-globe'
    },
    "CityLevelReverseGeocoder" : {
        description : function(type, params) {
            if (!hasAll(params, ["inputCol"])) return null;
            return translate("SHAKER.PROCESSOR.CityLevelReverseGeocoder.SUMMARY", actionVerb("Reverse-geocode") + " location {{column}}", {
                column: inColName(params["inputCol"])
            });
        },
         icon : 'dku-icon-globe'
    },
    "ChangeCRSProcessor" : {
        description : function(type, params) {
            if (!hasAll(params, ["geomCol"])) return null;
            return translate("SHAKER.PROCESSOR.ChangeCRSProcessor.SUMMARY", actionVerb("Change CRS") + " in {{column}}", {
                column: inColName(params["geomCol"])
            });
        },
         icon : 'dku-icon-globe'
    },
    "ZipCodeGeocoder" : {
        description : function(type, params) {
            if (!hasAll(params, ["zipCodeCol"])) return null;
            return translate("SHAKER.PROCESSOR.ZipCodeGeocoder.SUMMARY", actionVerb("Geocode zipcode") + " in {{column}}", {
                column: inColName(params["zipCodeCol"])
            });
        },
        icon : 'dku-icon-globe'
    },
    "GeometryInfoExtractor" : {
        description : function(type, params) {
            if (!hasAll(params, ["inputCol"])) return null;
            return translate("SHAKER.PROCESSOR.GeometryInfoExtractor.SUMMARY", actionVerb("Extract geo info") + " from {{column}}", {
                column: inColName(params["inputCol"])
            });
        },
        icon : 'dku-icon-globe'
    },
    "GeoDistanceProcessor" : {
        checkValid: geoDistanceCheckValid,
        description : function(type, params) {
            if (!hasAll(params, ["input1", "output", "outputUnit", "compareTo"])) return null;
            const p = {
                in1: inColName(params["input1"]),
                in2: inColName(params["input2"])
            }
            if (params.compareTo === "COLUMN"){
                return translate("SHAKER.PROCESSOR.GeoDistanceProcessor.SUMMARY", actionVerb("Compute distance") + " between {{in1}} and {{in2}}", p);
            } else if(params.compareTo === "GEOPOINT"  || params.compareTo === "GEOMETRY"){
                return translate("SHAKER.PROCESSOR.GeoDistanceProcessor.SUMMARY.REFERENCE", actionVerb("Compute distance") + " between {{in1}} and a reference", p);
            }
        },
        icon : 'dku-icon-globe'
    },
    "GeoPointBufferProcessor" : {
        description: function(type, params) {
            if (!params["inputColumn"] || params["inputColumn"].length === 0) {
                return translate("SHAKER.PROCESSOR.GeoPointBufferProcessor.SUMMARY.ERROR.MISSING_INPUT_COLUMN","Choose an input geopoint column.");
            }
            switch (params["shapeMode"]) {
                case "RECTANGLE":
                    return translate('SHAKER.PROCESSOR.GeoPointBufferProcessor.SUMMARY.RECTANGLE',
                        actionVerb("Generate a rectangle") + " centered on {{column}} with dimensions {{width}} x {{height}} {{unit==='KILOMETERS'?'km':'mi'}}", {
                            column: inColName(params["inputColumn"]),
                            width: sanitize(params["width"]),
                            height: sanitize(params["height"]),
                            unit: params["unitMode"]
                    });
                case "CIRCLE":
                    return translate('SHAKER.PROCESSOR.GeoPointBufferProcessor.SUMMARY.CIRCLE',
                        actionVerb("Generate a circle") + " centered on {{column}} with a {{radius}} {{unit==='KILOMETERS'?'km':'mi'}} radius", {
                            column: inColName(params["inputColumn"]),
                            radius: sanitize(params["radius"]),
                            unit: params["unitMode"]
                        });
                default:
                    return translate('SHAKER.PROCESSOR.GeoPointBufferProcessor.SUMMARY.POLYGON', actionVerb("Generate a polygon") + " centered on {{column}}", {
                        column: inColName(params["inputColumn"])
                    });
            }
        },
        icon: 'dku-icon-globe'
    },

    /* Open data */
    "EnrichFrenchPostcode" : {
        description : function(type, params) {
            if (!hasAll(params, ["column"])) return null;
            return translate("SHAKER.PROCESSOR.EnrichFrenchPostcode.SUMMARY", actionVerb("Enrich French postcode") + " info from {{column}}", {
                column: inColName(params["column"])
            });
        },
        icon : 'dku-icon-user-group'
    },
    "EnrichFrenchDepartement" : {
        description : function(type, params) {
            if (!hasAll(params, ["column"])) return null;
            return translate("SHAKER.PROCESSOR.EnrichFrenchDepartment.SUMMARY", actionVerb("Enrich French department") + " info from {{column}}", {
                column: inColName(params["column"])
            });
        },
        icon : 'dku-icon-user-group'
    },

    /* Reshaping */
    "Unfold" : {
        description : function(type, params) {
            if (!hasAll(params, ["column"])) return null;
            return translate("SHAKER.PROCESSOR.Unfold.SUMMARY", actionVerb("Create dummy") + " columns from values of {{column}}", {
                column: inColName(params["column"])
            });
        },
        icon : 'dku-icon-level-up'
    },
    "ArrayUnfold" : {
    description : function(type, params) {
        if (!hasAll(params, ["column"])) return null;
        return translate("SHAKER.PROCESSOR.ArrayUnfold.SUMMARY", actionVerb("Unfold") + " columns from values of {{column}}", {
            column: inColName(params["column"])
        });
    },
    icon : 'dku-icon-level-up'
    },
    "SplitUnfold" : {
        description : function(type, params) {
            if (!hasAll(params, ["column", "separator"])) return null;
            return translate("SHAKER.PROCESSOR.SplitUnfold.SUMMARY", actionVerb("Create dummy") + " columns by splitting {{column}} on {{separator}}", {
                column: inColName(params["column"]),
                separator: anumLiteral(params.separator)
            });
        },
        icon : 'dku-icon-level-up'
    },
    "Pivot" : {
        description : function(type, params) {
            if (!hasAll(params, ["indexColumn", "labelsColumn", "valuesColumn"])) return null;
            return translate("SHAKER.PROCESSOR.Pivot.SUMMARY", actionVerb("Pivot") + " around {{indexColumn}}, labels from {{labelsColumn}}, values from {{valuesColumn}}", {
                indexColumn: inColName(params["indexColumn"]),
                labelsColumn: inColName(params["labelsColumn"]),
                valuesColumn: inColName(params["valuesColumn"])
            });
        },
        icon : 'dku-icon-level-up'
    },
    "ZipArrays" : {
        description : function(type, params) {
            if (!hasAll(params, ["inputColumns", "outputColumn"])) return null;
            return translate("SHAKER.PROCESSOR.ZipArrays.SUMMARY", actionVerb("Zip arrays") + " from {{columns}} to arrays of objects in {{outputColumn}}", {
                columns: inColName(params["inputColumns"].join(", ")),
                outputColumn: outColName(params["outputColumn"])
            });
        },
        icon : 'dku-icon-level-up'
    },
    "ConcatArrays" : {
        description : function(type, params) {
            if (!hasAll(params, ["inputColumns", "outputColumn"])) return null;
            return translate("SHAKER.PROCESSOR.ConcatArrays.SUMMARY", actionVerb("Concatenate arrays") + " from {{inputColumns}} in {{outputColumn}}", {
                inputColumns: inColName(params["inputColumns"].join(", ")),
                outputColumn: outColName(params["outputColumn"])
            });
        },
        icon : 'dku-icon-arrow-expand-exit'
    },
    "SplitFold" : {
        description : function(type, params) {
            if (!hasAll(params, ["column", "separator"])) return null;
            return translate("SHAKER.PROCESSOR.SplitFold.SUMMARY", actionVerb("Split") + " values of {{column}} on {{separator}} and " + actionVerb("fold") + " to new rows", {
                column: inColName(params["column"]),
                separator: anumLiteral(params["separator"])
            });
        },
        icon : 'dku-icon-level-down'
    },
    "ArrayFold" : {
        description : function(type, params) {
            if (!hasAll(params, ["column"])) return null;
            return translate("SHAKER.PROCESSOR.ArrayFold.SUMMARY", actionVerb("Fold array values") + " of {{column}} to new rows", {
                column: inColName(params["column"])
            });
        },
        icon : 'dku-icon-level-down'
    },
    "ObjectFoldProcessor" : {
        description : function(type, params) {
            if (!hasAll(params, ["column"])) return null;
            return translate("SHAKER.PROCESSOR.ObjectFoldProcessor.SUMMARY", actionVerb("Fold keys/values") + " of {{column}} to new rows", {
                column: inColName(params["column"])
            });

        },
        icon : 'dku-icon-level-down'
    },
    "MultiColumnFold" : {
         description : function(type, params) {
            if (!hasAll(params, ["foldNameColumn", "foldValueColumn"])) return null;
             return translate("SHAKER.PROCESSOR.MultiColumnFold.SUMMARY", actionVerb("Fold multiple columns") + " into {{foldNameColumn}} / {{foldValueColumn}}", {
                 foldNameColumn: outColName(params["foldNameColumn"]),
                 foldValueColumn: outColName(params["foldValueColumn"])
             });
        },
        icon : 'dku-icon-level-down'
    },
    "RepeatableUnfold" : {
        description : function(type, params) {
            if (!hasAll(params, ["keyColumn", "foldColumn", "foldTrigger", "dataColumn"])) return null;
            return translate("SHAKER.PROCESSOR.RepeatableUnfold.SUMMARY", "Foreach {{keyColumn}}, create columns for values of {{dataColumn}} between each occurrence of '{{foldColumn}}' in {{foldTrigger}}", {
                keyColumn: inColName(params["keyColumn"]),
                dataColumn: inColName(params["dataColumn"]),
                foldColumn: inColName(params["foldColumn"]),
                foldTrigger: anumLiteral(params["foldTrigger"])
            });
        },
        icon : 'dku-icon-level-up'
    },
    "MultiColumnByPrefixFold" : {
        description : function(type, params) {
           if (!hasAll(params, ["columnNamePattern", "columnNameColumn", "columnContentColumn"])) return null;
            return translate("SHAKER.PROCESSOR.MultiColumnByPrefixFold.SUMMARY", actionVerb("Fold columns") + " matching {{columnNamePattern}} into {{columnNameColumn}} / {{columnContentColumn}}", {
                columnNamePattern: anumLiteral(params["columnNamePattern"]),
                columnNameColumn: outColName(params["columnNameColumn"]),
                columnContentColumn: outColName(params["columnContentColumn"])
            });
       },
       icon : 'dku-icon-level-down'
    },
    "NestProcessor" : {
        checkValid : function(params) {
            appliesToCheckValid(params);
            if (params["outputColumn"] === undefined) {
                throw new StepIAE(translate('SHAKER.PROCESSOR.SwitchCase.ERROR.MISSING_OUTPUT_COLUMN', "Output column not specified"));
            }
        },
        description : function(type, params) {
            return translate("SHAKER.PROCESSOR.NestProcessor.SUMMARY", actionVerb("Nest") + " {{columns}} into {{outputColumn}}", {
                columns: appliesToDescription(params),
                outputColumn: outColName(params["outputColumn"])
            });
        },
        icon : 'dku-icon-arrow-expand-exit'
    },

    /* Join */
    "MemoryEquiJoiner": {
        description : function(type, params){
            if (!hasAll(params, ["leftCol", "rightInput", "rightCol"])) return null;
            return translate("SHAKER.PROCESSOR.MemoryEquiJoiner.SUMMARY", actionVerb("Join") + " column {{leftCol}} with column {{rightCol}} of dataset {{rightInput}}", {
                leftCol: inColName(params["leftCol"]),
                rightCol: inColName(params["rightCol"]),
                rightInput: inColName(params["rightInput"])
            });
        },
        icon : 'dku-icon-arrow-expand-exit'
    },
    "MemoryEquiJoinerFuzzy": {
        description : function(type, params){
            if (!hasAll(params, ["leftCol", "rightInput", "rightCol"])) return null;
            return translate("SHAKER.PROCESSOR.MemoryEquiJoinerFuzzy.SUMMARY", actionVerb("Fuzzy-join") + " column {{leftCol}} with column {{rightCol}} of dataset {{rightInput}}", {
                leftCol: inColName(params["leftCol"]),
                rightCol: inColName(params["rightCol"]),
                rightInput: inColName(params["rightInput"])
            });
        },
        icon : 'dku-icon-arrow-expand-exit'
    },

    /* GREL */
    "CreateColumnWithGREL" : {
        description: function(type, params){
            if (!hasAll(params, ["column", "expression"])) return null;
            if (params.expression.length > 30) {
                return translate('SHAKER.PROCESSOR.CreateColumnWithGREL.DESCRIPTION.LONG', actionVerb("Create column") + " {{column}} with formula", {
                    column: outColName(params["column"])
                });
            } else {
                return translate('SHAKER.PROCESSOR.CreateColumnWithGREL.DESCRIPTION.SHORT', actionVerb("Create column") + " {{column}} with formula {{formula}}", {
                    column: outColName(params["column"]),
                    formula: anumLiteral(params.expression)
                });
            }
        },
        shouldExpandFormula: function(params) {
            if (!hasAll(params, ["expression"])) return false;
            return params.expression.length > 30;
        },
        icon : 'dku-icon-beaker'
    },

    /* Custom */
    "PythonUDF" : {
        checkValid: function(params) {
            if (params.mode === "CELL") {
                defaultIsValidIfHasColumn(params, "column", {throwError: true})
            }
        },
        description: function(type, params) {
            if (params.mode === "CELL") {
                if (!defaultIsValidIfHasColumn(params, "column", {throwError: false})) {
                    return null;
                }
                return translate("SHAKER.PROCESSOR.PythonUDF.SUMMARY.CELL", actionVerb("Create column") + "{{column}} with Python code", {
                    column: outColName(params.column)
                });
            } else if (params.mode === "ROW") {
                return translate("SHAKER.PROCESSOR.PythonUDF.SUMMARY.ROW", actionVerb("Modify row") + " with Python code");
            } else {
                return translate("SHAKER.PROCESSOR.PythonUDF.SUMMARY.DATA", actionVerb("Modify data") + " with Python code");
            }
        },
        shouldExpandFormula: function(params) {
            return params.pythonSourceCode;
        },
        icon : 'dku-icon-beaker'
    },
    "GenerateBigData" : {
        description: function(type, params) {
            return translate("SHAKER.PROCESSOR.GenerateBigData.SUMMARY.CELL", actionVerb("Generate Data") + " {{expansionFactor}} times bigger", {
                expansionFactor: numLiteral(params["expansionFactor"])
            });
        },
        icon : 'dku-icon-beaker'

    },
    "ColumnCopier" : {
        description: function(type, params) {
            if (!hasAll(params, ["inputColumn", "outputColumn"])) {
                return null;
            }
            return translate("SHAKER.PROCESSOR.ColumnCopier.SUMMARY.CELL", actionVerb("Copy") + " column {{inputColumn}} to {{outputColumn}}", {
                inputColumn: inColName(params["inputColumn"]),
                outputColumn: outColName(params["outputColumn"])
            });
        },
        icon : 'dku-icon-copy-step'
    },
    "EnrichWithRecordContextProcessor" : {
        description: function() {
            return translate("SHAKER.PROCESSOR.EnrichWithRecordContextProcessor.SUMMARY.CELL", actionVerb("Enrich") + " records with files info");
        },
        icon : 'dku-icon-file'
    },
    "EnrichWithBuildContextProcessor" : {
        description: function() {
            return translate("SHAKER.PROCESSOR.EnrichWithBuildContextProcessor.SUMMARY.CELL", actionVerb("Enrich") + " records with build info");
        },
        icon : 'dku-icon-play-outline'
    },
    "FastPredict" : {
        description: function(type, params) {
            return translate("SHAKER.PROCESSOR.FastPredict.SUMMARY.CELL", actionVerb("Predict missing") + " values on {{targetColumn}}", {
                inputColumn: strongify(sanitize(params["targetColumn"])),
                outputColumn: outColName(params["outputColumn"])
            });
        },
        icon : 'dku-icon-beaker'

    },
    "BooleanNot" : {
        description: function(type, params) {
            return translate("SHAKER.PROCESSOR.BooleanNot.SUMMARY.CELL", actionVerb("Negate") + " boolean {{column}}", {
                column: inColName(params["column"])
            });
        },
        icon : 'dku-icon-beaker'
    },
    "FillColumn" : {
        description: function(type, params) {
            return translate("SHAKER.PROCESSOR.FillColumn.SUMMARY.CELL", actionVerb("Fill") + " column{{hasColumn?(' '+column):''}}{{hasValue?(' with '+value):''}}", {
                hasColumn: !!params["column"],
                hasValue:!!params["value"],
                column: outColName(params["column"]),
                value: anumLiteral(params["value"])
            });
        },
        icon : 'dku-icon-beaker'
    },
    "ColumnPseudonymization": {
        checkValid: appliesToCheckValid,
        description: function(_, params) {
            return translate("SHAKER.PROCESSOR.ColumnPseudonymization.DESCRIPTION", "{{verbStylised}} {{columns}} {{saltColumn ? (' with column {{saltColumnStylised}} for salting') : ''}} {{pepper ? (' with {{pepperStylised}} as pepper') : ''}}", {
                 verbStylised: actionVerb("Pseudonymize"),
                 columns: appliesToDescription(params),
                 saltColumn: params["saltColumn"],
                 saltColumnStylised: inColName(params["saltColumn"]),
                 pepper: params["pepper"],
                 pepperStylised: anumLiteral(params["pepper"]),
             });
        },
        icon: 'dku-icon-edit-note'
    },
    "Coalesce": {
        checkValid: appliesToCheckValid,
        description: function(type, params) {
            if (!hasAll(params, ["outputColumn"])) return null;
            return translate(
                "SHAKER.PROCESSOR.Coalesce.SUMMARY",
                actionVerb("Coalesce") + " returns the first non-null value into column {{outCol}}",
                { outCol: outColName(params.outputColumn) }
            );
        },
        icon: "dku-icon-true-false"
    },
    "MeasureNormalize" : {
        description: function(type, params) {
            return translate("SHAKER.PROCESSOR.MeasureNormalize.SUMMARY.CELL", actionVerb("Normalize") + " measure {{column}} to SI units", {
                column: inColName(params["column"])
            });
        },
        icon : 'dku-icon-beaker'
    },
    "Transpose": {
        description: function (type, params) {
            return translate("SHAKER.PROCESSOR.Transpose.SUMMARY.CELL", actionVerb("Transpose") + " rows into columns around {{column}}", {
                column: inColName(params["column"])
            });
        },
        icon: 'dku-icon-arrow-clockwise'
    },
    "GrokProcessor" : {
        description: function(type, params) {
            return translate("SHAKER.PROCESSOR.GrokProcessor.SUMMARY.CELL", actionVerb("Extract") + " {{column}} with grok.", {
                column: inColName(params["sourceColumn"])
            });
        },
        shouldExpandFormula: function(params) {
            return params.grokPattern;
        },
        icon : 'dku-icon-beaker'
    },
    }

    function jythonEntry(processor, loadedDesc){
        return {
            description : function(type, params) {
                return loadedDesc.desc.meta.label;
            },
            icon : loadedDesc.desc.meta.icon || 'dku-icon-puzzle-piece',
            checkValid : function(params) {
                Object.keys(params.customConfig).forEach(function(k) {
                    let v = params.customConfig[k];
                    let param = loadedDesc.desc.params.find(x => x.name === k);
                    if (param != null && param.mandatory && (v ==null || v.length === 0)) {
                        throw new StepIAE("No value for parameter " + param.name);
                    }
                })
            }
        }
    }

    svc.get = function(processor) {
        if (processor && processor.indexOf("jython-processor") === 0) {
            let loadedDesc = window.dkuAppConfig.customJythonProcessors.find(x => x.elementType === processor);
            Assert.trueish(loadedDesc, 'processor desc not loaded');
            return jythonEntry(processor, loadedDesc);
        } else {
            return svc.map[processor];
        }
    };

    return svc;
});

app.factory("ShakerProcessorsUtils", function($filter, ShakerProcessorsInfo, translate) {

    var svc = {}

    // Wonderfully copy/pasted from script.js, with some additional try/catch to avoid some exceptions
    /* Perform JS validation of the step. Does not set frontError */
    svc.validateStep = function(step, processors) {
        if (step.metaType == "GROUP") {
            if (step.steps != null) {
                for (let i = 0; i < step.steps.length; i++) {
                    var subValidationResult = svc.validateStep(step.steps[i], processors);
                    if (subValidationResult) {
                        return subValidationResult;
                    }
                }
            }
        } else {
            try {
                var processorType = $filter('processorByType')(processors, step.type);
                /* If we have some stepParams, then check using them */
                if (processorType.params) {
                    for (var paramIdx in processorType.params) {
                        var param = processorType.params[paramIdx];
                        var value = step.params[param.name];
                        if (param.mandatory && !param.canBeEmpty && (value == null || value.length === 0)) {
                            // eslint-disable-next-line no-undef
                            return new StepIAE("Missing parameter: " + (param.label || param.name));
                        }
                    }
                }
                /* Then also play the specific validation of each step */
                let processorInfo = ShakerProcessorsInfo.get(step.type);
                if (processorInfo.checkValid){
                    try {
                        processorInfo.checkValid(step.params);
                    } catch (e) {
                        return e;
                    }
                }
            } catch(e) { // happens for a plugin processor if plugin has been uninstalled
                // eslint-disable-next-line no-undef
                return new StepIAE("Unknown processor: " + step.type);
            }
        }
        return null;
    };

    /**
     * The purpose of this method is to manipulate the provided steps to display them in the UI.
     * Requirements to display steps are:
     * - A single array of steps with all standalone steps, group steps and their sub-steps at the same level.
     * - Only enabled steps are displayed.
     * - In case a step has a validation error, it should be displayed with its validation error message.
     * - Group sub-steps must be distinguishable from standalone steps.
     * The method returns a dictionary containing:
     * - an array with the flattened enabled steps with the enriched information for each step
     * - an array with all the provided steps with the enriched information for each step
     */
    svc.getStepsWithSubSteps = function(steps, processors) {
        // compute recursive steps
        let flattenedEnabledSteps = [];
        steps.forEach(step => {
            step["mainStep"] = true; // Normal step => No padding
            step["errorMsg"] = svc.validateStep(step, processors);
            step["showFormula"] = true; // Expand formulas by default
            if(!step.disabled) {
                if (step.metaType === "GROUP") {
                    let enabledSubSteps = [];
                    step.steps.forEach(subStep => {
                        subStep["mainStep"] = false; // Step inside a group => need padding
                        subStep["errorMsg"] = svc.validateStep(subStep, processors);
                        subStep["showFormula"] = true; // Expand formulas by default
                        if(!subStep.disabled) {
                            enabledSubSteps.push(subStep);
                        }
                    });
                    if (enabledSubSteps.length > 0) {
                        step.steps = enabledSubSteps;
                        flattenedEnabledSteps = [...flattenedEnabledSteps, step, ...enabledSubSteps];
                    }
                } else {
                    flattenedEnabledSteps.push(step);
                }
            }
        });
        return { flattenedEnabledSteps, enrichedSteps: steps };
    }

    svc.getGroupName = function(step, steps) {
        if (step.metaType == 'GROUP') {
            if (step.name && step.name.length > 0) {
                return step.name;
            } else {
                const groupIndex = steps.filter(function(s) { return s.metaType === 'GROUP'; })
                                        .indexOf(step);
                return 'GROUP ' + (groupIndex < 0 ? '' : groupIndex + 1);
            }
        } else {
            return "Invalid group";
        }
    }

    svc.getStepDescription = function(processor, type, params) {
        const prefix = (processor && processor.deprecated) ? "<span class=\"deprecation-tag\">"+ translate("SHAKER.PROCESSORS.DESCRIPTION.DEPRECATED", "[DEPRECATED]") +"</span> " : "";
        const e = ShakerProcessorsInfo.get(type);
        if (!e || !e.description) {
            return prefix + type + " " + translate("SHAKER.PROCESSORS.DESCRIPTION.UNKNOWN", "(UNKNOWN)");
        }
        if (e.checkValid) {
            try {
                e.checkValid(params);
            } catch (e) {
                const invalidDecorator = " " + translate("SHAKER.PROCESSORS.DESCRIPTION.INVALID", "(invalid)")
                if (processor) {
                    return processor.enDescription + invalidDecorator;  // no need for prefix since already handled in CachedAPICalls with enDescription
                } else {
                    return type + invalidDecorator; // same as above
                }
            }
        }
        const desc = e.description(type, params);
        if (!desc) {
            const invalidNoDescDecorator = " " + translate("SHAKER.PROCESSORS.DESCRIPTION.INVALID_NO_DESC", "(invalid - no desc)")
            if (processor) {
                return processor.enDescription + invalidNoDescDecorator; // same as above
            } else {
                return type + invalidNoDescDecorator; // same as above
            }
        } else {
            return prefix + desc;
        }
    };

    // Match a query to the step params. A step belongs to a processor
    // Every processor is registered in `ShakerProcessorInfo.svc.map`
    // Define a `paramsMatcher` on the processor that matches the step params with the given query
    svc.matchStepParams = function(step, query) {
        if (!step || !query || !step.params) {
            return false;
        }
        function findMatchInObject(object, query) {
            if (['string', 'bigint', 'number'].includes(typeof object)) {
                return normalizeForSearch(object).includes(query);
            }
            if (typeof object === 'boolean') {
                return object ? query === 'true' : query === 'false';
            }
            if (Array.isArray(object)) {
                return object.some(value => findMatchInObject(value, query));
            }
            if (typeof object === 'object') {
                return Object.values(object).some(value => findMatchInObject(value, query));
            }
            return false;
        }


        const processorInfo = ShakerProcessorsInfo.get(step.type);
        if (processorInfo) {
            if (processorInfo.paramsMatcher) {
                return processorInfo.paramsMatcher(query, step.params);
            }
            return findMatchInObject(step.params, query);
        }
        // If the processor is unknown or the step does not have any params
        // It cannot match with anything
        return false;
    }

    svc.getHtmlStepDescriptionOrError = function(step, processors) {
        try {
            var processor = $filter('processorByType')(processors, step.type);
        } catch(e) { // may happen for a processor from a deleted plugin
            return "<span class=\"text-error\">Unknown processor: " + sanitize(step.type) + "</span>"
        }
        if (step.errorMsg) {
            return "<span class=\"text-error\">" + processor.enDescription + ": " + step.errorMsg.message + "</span>";
        } else {
            return svc.getStepDescription(processor, step.type, step.params);
        }
    }

    svc.getStepIcon = function(type, params, size) {
        const e = ShakerProcessorsInfo.get(type);
        const icon = e && e.icon && (typeof e.icon == 'function' ? e.icon(type, params) : e.icon) || 'dku-icon-warning-fill';

        if(icon.startsWith('dku-icon-')) {
            // new icons require a size
            return `${icon}-${size}`;
        } else {
            // not new icons (from plugins) don't
            return icon
        }
    };

    svc.getStepImpactVerb = function(type, params) {
        const e = ShakerProcessorsInfo.get(type);
        if (e && e.impactVerb) {
            return e.impactVerb(params);
        }
        return translate("SHAKER.PROCESSORS.IMPACT.MODIFIED", "modified");
    };

    return svc;
});

app.factory("ShakerSuggestionsEngine", function(ShakerProcessorsUtils, CreateModalFromDOMElement, Dialogs, PluginsService, $filter, $rootScope, $q, translate) {
var svc = {};

const categoryFilterData = translate('SHAKER.SUGGESTIONS.CATEGORIES.FILTER_DATA', "Filter data");
const categoryNumbers = translate('SHAKER.SUGGESTIONS.CATEGORIES.NUMBERS', "Numbers");
const categoryGeography = translate('SHAKER.SUGGESTIONS.CATEGORIES.GEOGRAPHY', "Geography");
const categoryWeb = translate('SHAKER.SUGGESTIONS.CATEGORIES.WEB', "Web");
const categoryEmail = translate('SHAKER.SUGGESTIONS.CATEGORIES.EMAIL', "Email");
const categoryBoolean = translate('SHAKER.SUGGESTIONS.CATEGORIES.BOOLEAN', "Boolean");
const categoryMeasure = translate('SHAKER.SUGGESTIONS.CATEGORIES.MEASURE', "Measure");
const categoryDate = translate('SHAKER.SUGGESTIONS.CATEGORIES.DATE', "Date");
const categoryTransformations = translate('SHAKER.SUGGESTIONS.CATEGORIES.TRANSFORMATIONS', "Transformations");
const categoryKeepDelete = translate('SHAKER.SUGGESTIONS.CATEGORIES.KEEP_DELETE', "Delete/Keep");
const categoryDataCleansing = translate('SHAKER.SUGGESTIONS.CATEGORIES.DATA_CLEANSING', "Data cleansing");
const categorySmartPatternBuilder = translate('SHAKER.SUGGESTIONS.CATEGORIES.SMART_PATTERN_BUILDER', "Smart Pattern Builder");
const categoryText = translate('SHAKER.SUGGESTIONS.CATEGORIES.TEXT', "Text");

function addCustomSuggestion(map, category, id, func, text, icon) {
    (category in map ?
         map[category] :
         (map[category] = []))
        .push({id: id, text: text, action: func, icon});
}

svc.getCategoryPriority = function (category) {
    // default category is 0, unspecified categories will remain in insertion order
    return {
        [categorySmartPatternBuilder]: -1
    } [category] ?? 0;
}

/* Cell suggestions
 * MUST NOT include column-level suggestions
 */
svc.computeCellSuggestions =  function(column, value, validity) {
    const truncateFilter = $filter("gentleTruncate");

    function addStepBasedSuggestion(map, category, id, type, params, text) {
        addCustomSuggestion(map, category, id, function(shakerScope) {
            window.WT1SVC.event("shaker-use-cell-sugg", {
                "colMeaning" : column.selectedType ? column.selectedType.name : "unk" ,
                "cellValid" : validity,
                "processor" : type,
                "suggText" : text,
            });
            shakerScope.addStepNoPreview(type, params);
            if (type == "FilterOnValue") {
            	shakerScope.mergeLastDeleteRows();
            }
            shakerScope.autoSaveAutoRefresh();
        }, text, ShakerProcessorsUtils.getStepIcon(type, params, '16')
        );
    }

    function addUnconfiguredStepBasedSuggestion(map, category, id, type, params, text) {
        addCustomSuggestion(map, category, id, function(shakerScope) {
            window.WT1SVC.event("shaker-use-cell-sugg", {
                "colMeaning" : column.selectedType ? column.selectedType.name : "unk" ,
                "cellValid" : validity,
                "processor" : type,
                "suggText" : text,
            });
            shakerScope.addUnconfiguredStep(type, params);
            if (type === "FilterOnValue") {
            	shakerScope.mergeLastDeleteRows();
            }
        }, text, ShakerProcessorsUtils.getStepIcon(type, params, '16')
        );
    }


    function visualIfSuggestionTitle(type, columnName, value) {
        const prefix = actionVerb("Create") +  " If Then Else rule based on " +  inColName(truncateFilter(columnName, 20));
        if(!value) {
            return prefix + " " + anumLiteral("is not defined");
        } else {
            if(type === "boolean") {
                return prefix + " is " + anumLiteral(parseBoolean(value));
            } else if(type === "array" && "[]" === value) {
                return prefix + " is " + anumLiteral("an empty array");
            } else {
                const operator = isNumber(type) ? " == " : " equals "
                return prefix + operator + anumLiteral(truncateFilter(value, 20));
            }
        }
    }

    var map = Object();

    if (value) {
        addStepBasedSuggestion(map, categoryFilterData, "keep", "FilterOnValue", {
            'appliesTo': "SINGLE_COLUMN",
            'action': "REMOVE_ROW",
            'columns': [column.name],
            'values': [value],
            'matchingMode' : "FULL_STRING",
            'normalizationMode' : "EXACT"
        }, actionVerb("Remove") + " rows equal to " + anumLiteral(truncateFilter(value, 35)));

        addStepBasedSuggestion(map, categoryFilterData, "remove", "FilterOnValue", {
            'appliesTo': "SINGLE_COLUMN",
            'action': "KEEP_ROW",
            'columns': [column.name],
            'values': [value],
            'matchingMode' : "FULL_STRING",
            'normalizationMode' : "EXACT"
        }, actionVerb("Keep") + " only rows equal to " + anumLiteral(truncateFilter(value, 35)));

        addStepBasedSuggestion(map, categoryFilterData, "clear", "FilterOnValue", {
            'appliesTo': "SINGLE_COLUMN",
            'action': "CLEAR_CELL",
            'columns': [column.name],
            'values': [value],
            'matchingMode' : "FULL_STRING",
            'normalizationMode' : "EXACT"
        }, actionVerb("Clear") + " cells equal to " + anumLiteral(truncateFilter(value, 35) ));
    }

    const type = column.recipeSchemaColumn && column.recipeSchemaColumn.column
            ? column.recipeSchemaColumn.column.type
            : column.datasetSchemaColumn
                ? column.datasetSchemaColumn.type
                : "string";
    const operator = value ? filterDescEqualityOperator(type, value) : "is empty";

    if(value) {
        addUnconfiguredStepBasedSuggestion(map, categoryFilterData, "visual_if", "VisualIfRule",
            visualIfParams(column.name, type, value, operator), visualIfSuggestionTitle(type, column.name, value));
    }


    if (value && validity && validity.indexOf('I') === 0) {
        addStepBasedSuggestion(map, categoryDataCleansing, "c1", "FilterOnBadType", {
            'appliesTo': "SINGLE_COLUMN",
            'action': 'REMOVE_ROW',
            'columns': [column.name],
            'type': column.selectedType.name,
        }, actionVerb("Remove invalid") + " rows for meaning");
        addStepBasedSuggestion(map, categoryDataCleansing, "c1", "FilterOnBadType", {
            'appliesTo': "SINGLE_COLUMN",
            'action': 'CLEAR_CELL',
            'columns': [column.name],
            'type': column.selectedType.name,
        }, actionVerb("Clear invalid") + " cells for meaning");
    }

    if (!value) {
        addStepBasedSuggestion(map, categoryDataCleansing, "c1", "RemoveRowsOnEmpty", {
            'columns': [column.name],
            appliesTo : 'SINGLE_COLUMN',
            'keep' : false
        }, actionVerb("Remove") +" rows where cell is empty");
        addStepBasedSuggestion(map, categoryDataCleansing, "c1", "RemoveRowsOnEmpty", {
            'columns': [column.name],
            appliesTo : 'SINGLE_COLUMN',
            'keep' : true
        }, actionVerb("Keep only") +  " rows where cell is empty");
        addCustomSuggestion(map, categoryDataCleansing, "fill_empty_withval", function(scope) {
            CreateModalFromDOMElement("#fill-empty-with-value-box", scope, "FillEmptyWithValueController",
                function(newScope) { newScope.$apply(function() {
                        newScope.setColumns([column.name]);
                        newScope.isNumericOnly = ["LongMeaning", "DoubleMeaning"]
                            .indexOf(column.selectedType && column.selectedType.name) > -1;
                }); });
        }, actionVerb("Fill") +  " empty rows with...", ShakerProcessorsUtils.getStepIcon("FillEmptyWithValue", {}, '16'));
        addUnconfiguredStepBasedSuggestion(map, categoryDataCleansing, "visual_if", "VisualIfRule",
            visualIfParams(column.name, type, value, operator), visualIfSuggestionTitle(type, column.name, value));
    }
    return map;
};

svc.computeContentSuggestions = function(column, cellValue, value, validity, CreateModalFromTemplate, selectionStartOffset, selectionEndOffset) {
    const truncateFilter = $filter("gentleTruncate");
    function addStepBasedSuggestion(map, category, id, type, params, text) {
        addCustomSuggestion(map, category, id, function(shakerScope) {
            window.WT1SVC.event("shaker-use-content-sugg", {
                "colMeaning" : column.selectedType ? column.selectedType.name : "unk" ,
                "cellValid" : validity,
                "value" : value, "startOff" : selectionStartOffset, "endOff" : selectionEndOffset,
                "processor" : type,
                "suggText" : text,
            });
            shakerScope.addStepAndRefresh(type, params);
        }, text, ShakerProcessorsUtils.getStepIcon(type, params, '16'));
    }

    function addUnconfiguredStepBasedSuggestion(map, category, id, type, params, text) {
        addCustomSuggestion(map, category, id, function(shakerScope) {
            window.WT1SVC.event("shaker-use-content-sugg", {
                "colMeaning" : column.selectedType ? column.selectedType.name : "unk" ,
                "cellValid" : validity,
                "value" : value,
                "startOff" : selectionStartOffset,
                "endOff" : selectionEndOffset,
                "processor" : type,
                "suggText" : text,
            });
            shakerScope.addUnconfiguredStep(type, params);
        }, text, ShakerProcessorsUtils.getStepIcon(type, params, '16')
        );
    }


    var map = Object();
    const type = column.recipeSchemaColumn && column.recipeSchemaColumn.column
            ? column.recipeSchemaColumn.column.type
            : column.datasetSchemaColumn
                ? column.datasetSchemaColumn.type
                : "string";

    addStepBasedSuggestion(map, categoryTransformations, "c1", "ColumnSplitter", {
            'inCol': column.name,
            'separator': value,
            'outColPrefix' : column.name + "_",
            'target' : "COLUMNS"
        }, actionVerb("Split") +  " column on " + anumLiteral(value) + "");
    addStepBasedSuggestion(map, categoryTransformations, "c1", "FindReplace", {
            'appliesTo' :"SINGLE_COLUMN",
            "columns" : [column.name],
            'matching' : "SUBSTRING",
            'normalization' : "EXACT",
            "mapping" : [{"from":value,"to":""}],
        }, actionVerb("Replace") +  " " + anumLiteral(value) + " by ...");

    addCustomSuggestion(map, categorySmartPatternBuilder, "extractor",
        function(scope) {
            var deferred = $q.defer();
            CreateModalFromTemplate("/templates/shaker/regexbuilder-box.html", scope, "RegexBuilderController",
                function(newScope) {
                    newScope.deferred = deferred;
                    newScope.$apply(function() {
                        newScope.columnName = column.name;
                        newScope.firstSentence = cellValue;
                        newScope.addSelection(cellValue, selectionStartOffset, selectionEndOffset);
                        newScope.calledFrom = "highlight-extract_like";
                    });
                    deferred.promise.then(function(newPattern) {
                        var params = {
                            "column" : column.name,
                            "prefix": column.name + '_extracted_',
                            "pattern": newPattern.regex,
                            "extractAllOccurrences": newPattern.hasMultiOccurrences,
                        };
                        scope.addStepAndRefresh("RegexpExtractor", params);
                    });
                },
                "sd-modal");
        },
        actionVerb("Extract") + " text like " + anumLiteral(value) + "...",
        ShakerProcessorsUtils.getStepIcon('RegexpExtractor', {}, '16')
    );

    addCustomSuggestion(map, categorySmartPatternBuilder, "filter",
        function(scope) {
            var deferred = $q.defer();
            CreateModalFromTemplate("/templates/shaker/regexbuilder-box.html", scope, "RegexBuilderController",
                function(newScope) {
                    newScope.deferred = deferred;
                    newScope.$apply(function() {
                        newScope.columnName = column.name;
                        newScope.firstSentence = cellValue;
                        newScope.addSelection(cellValue, selectionStartOffset, selectionEndOffset);
                        newScope.calledFrom = "highlight-remove_like";
                    });
                    deferred.promise.then(function(newPattern) {
                        var params = {
                            'action': "REMOVE_ROW",
                            'appliesTo' : "SINGLE_COLUMN",
                            'columns': [column.name],
                            'values': [newPattern.regex],
                            'matchingMode' : "PATTERN",
                            'normalizationMode' : "EXACT"
                        };
                        scope.addStepAndRefresh("FilterOnValue", params);
                    });
                },
                "sd-modal");
        },
        actionVerb("Remove") + " rows containing text like " + anumLiteral(value) + "...",
        ShakerProcessorsUtils.getStepIcon('FilterOnValue', {}, '16')
    );

    addCustomSuggestion(map, categorySmartPatternBuilder, "keep",
        function(scope) {
            var deferred = $q.defer();
            CreateModalFromTemplate("/templates/shaker/regexbuilder-box.html", scope, "RegexBuilderController",
                function(newScope) {
                    newScope.deferred = deferred;
                    newScope.$apply(function() {
                        newScope.columnName = column.name;
                        newScope.firstSentence = cellValue;
                        newScope.addSelection(cellValue, selectionStartOffset, selectionEndOffset);
                        newScope.calledFrom = "highlight-filter_like";
                    });
                    deferred.promise.then(function(newPattern) {
                        var params = {
                            'action': "KEEP_ROW",
                            'appliesTo' : "SINGLE_COLUMN",
                            'columns': [column.name],
                            'values': [newPattern.regex],
                            'matchingMode' : "PATTERN",
                            'normalizationMode' : "EXACT"
                        };
                        scope.addStepAndRefresh("FilterOnValue", params);
                    });
                },
                "sd-modal");
        },
        actionVerb("Keep") + " rows containing text like " + anumLiteral(value) + "...",
        ShakerProcessorsUtils.getStepIcon("FilterOnValue", {}, '16')
    );

    addCustomSuggestion(map, categorySmartPatternBuilder, "flag",
        function(scope) {
            var deferred = $q.defer();
            CreateModalFromTemplate("/templates/shaker/regexbuilder-box.html", scope, "RegexBuilderController",
                function(newScope) {
                    newScope.deferred = deferred;
                    newScope.$apply(function() {
                        newScope.columnName = column.name;
                        newScope.firstSentence = cellValue;
                        newScope.addSelection(cellValue, selectionStartOffset, selectionEndOffset);
                        newScope.calledFrom = "highlight-flag_like";
                    });
                    deferred.promise.then(function(newPattern) {
                        var params = {
                            "flagColumn" : column.name + "_flagged",
                            'action': "KEEP_ROW",
                            'appliesTo' : "SINGLE_COLUMN",
                            'columns': [column.name],
                            'values': [newPattern.regex],
                            'matchingMode' : "PATTERN",
                            'normalizationMode' : "EXACT"
                        };
                        scope.addStepAndRefresh("FlagOnValue", params);
                    });
                },
                "sd-modal");
        },
        actionVerb("Flag") + " rows containing text like " + anumLiteral(value) + "...",
        ShakerProcessorsUtils.getStepIcon("FlagOnValue", {}, '16')
    );

    addStepBasedSuggestion(map, categoryFilterData, "c1", "FilterOnValue", {
            'action': "REMOVE_ROW",
            'appliesTo' : "SINGLE_COLUMN",
            'columns': [column.name],
            'values': [value],
            'matchingMode' : "SUBSTRING",
            'normalizationMode' : "EXACT"
        }, actionVerb("Remove") +  " rows containing " + anumLiteral(value) + "");

    addStepBasedSuggestion(map, categoryFilterData, "c1", "FilterOnValue", {
            'action': "KEEP_ROW",
            'appliesTo' : "SINGLE_COLUMN",
            'columns': [column.name],
            'values': [value],
            'matchingMode' : "SUBSTRING",
            'normalizationMode' : "EXACT"
        }, actionVerb("Keep only") + " rows containing " + anumLiteral(value) + "");

    if (type === 'string') {
        addUnconfiguredStepBasedSuggestion(map, categoryFilterData, "visual_if", "VisualIfRule",
            visualIfParams(column.name, type, value, 'contains'),
            actionVerb("Create") +  " If Then Else rule based on " +  inColName(truncateFilter(column.name, 20)) + " contains " + anumLiteral(truncateFilter(value, 20))
        );
    }

    return map;

};


/* Column suggestions are handled fully in Javascript based on the data in the column header, to avoid useless
 * calls.
 *
 * Returns an array of two maps of <category name, array[]>
 *   - id (string)
 *   - text (string)
 *   - action (function)
 * The first map contains "important" suggestions, the second one more suggestions.
 * We ensure that there is always at least one non-additional suggestion per column
 */
svc.computeColumnSuggestions = function(column, CreateModalFromDOMElement, CreateModalFromTemplate, forCell, forInvalidCell, appConfig) {
    // UGLY
    var meaningLabelFilter = angular.element("body").injector().get("$filter")("meaningLabel");

    var COLUMN_NAME_PH = '__DKU_COLUMN_NAME__', CNRE = new RegExp(COLUMN_NAME_PH, 'g'),
        TYPE_NAME_PH   = '__DKU_COLUMN_TYPE__', TNRE = new RegExp(TYPE_NAME_PH, 'g');

    function addStepBasedSuggestion(map, category, id, type, params, text, incomplete, oneStep) {
        params = JSON.stringify(params);
        var reportEvent = report("shaker-use-col-sugg", type, text),
            addStepFnName = incomplete ? 'addUnconfiguredStep' : multi ? 'addStepNoPreview' : 'addStep',
            addStepFn = function (c) { this[addStepFnName](type, JSON.parse(params
                    .replace(CNRE, c.name).replace(TNRE, c.selectedType && c.selectedType.name || ''))); };
        addCustomSuggestion(map, category, id, function(shakerScope) {
            reportEvent();
            if (oneStep) {
            	addStepFn.call(shakerScope, columns[0]);
            } else {
            	columns.forEach(addStepFn, shakerScope);
            }
            if (type == "FindReplace") {
            	shakerScope.mergeLastFindReplaces();
            }
            shakerScope.autoSaveAutoRefresh();

        }, text, ShakerProcessorsUtils.getStepIcon(type, params, '16'));
    }

    function addAppliesToSuggestion(map, category, id, type, params, text, incomplete) {
        if (columnNames.length === 1) {
            params["appliesTo"] = "SINGLE_COLUMN";
        } else {
            params["appliesTo"] = "COLUMNS";
        }
        params["columns"] = columnNames;

        addStepBasedSuggestion(map, category, id, type, params, text, incomplete, true);
    }

    var map = {}, moreMap = {}, target,
        columns = [column];
    if (Array.isArray(column)) {
        columns = column;
        column = column.length === 1 ? column = column[0] : { name: COLUMN_NAME_PH };
    }

    var types = columns.reduce(function(ts, c){
            if (c.selectedType && ts.indexOf(c.selectedType.name) === -1) {
                ts.push(c.selectedType.name);
            } return ts; }, []),
        multi = columns.length > 1,
        hasNOK   = columns.some(function(c) { return c.selectedType && c.selectedType.nbNOK   > 0; }),
        hasEmpty = columns.some(function(c) { return c.selectedType && c.selectedType.nbEmpty > 0; }),
        typesAllIn = types.every.bind(types, function(t) { return this.indexOf(t) !== -1; }),
        noTypeIn   = types.every.bind(types, function(t) { return this.indexOf(t) === -1; }),
        typeIs     = function(t) { return types.length === 1 && types[0] === t; },
        typeName = types.length === 1 ? types[0] : TYPE_NAME_PH,
        columnName = column.name,
        columnNames = columns.map(function(c){ return c.name; }),
        eventBase = multi ? {
            colMeaning: column.selectedType ? column.selectedType.name    : "unk" ,
            nbOK:       column.selectedType ? column.selectedType.nbOK    : -1,
            nbNOK:      column.selectedType ? column.selectedType.nbNOK   : -1,
            nbEmpty:    column.selectedType ? column.selectedType.nbEmpty : -1
        } : {
            colMeanings: types.join(','),
            colCount: columns.length,
            hasNOK: hasNOK,
            hasEmpty: hasEmpty
        },
        report = function(evt, proc, text) {
            var ep = { processor: proc, text: text }, i;
            for (i in eventBase) {
                ep[i] = eventBase[i];
            }
            return window.WT1SVC.event.bind(window.WT1SVC, evt, ep);
        };

    /* Step 1 : add type-based main suggestions */
    if (typeIs("IPAddress")) {
        addStepBasedSuggestion(map, categoryGeography, "geoip", "GeoIPResolver", {
            'inCol': columnName,
            'outColPrefix': columnName+'_',
            'extract_country' : true,
            'extract_countrycode' : true,
            'extract_region' : true,
            'extract_city' : true,
            'extract_geopoint' : true
        }, translate('SHAKER.SUGGESTIONS.COLUMN.IPAddress.RESOLVE', actionVerb("Resolve") + " GeoIP"));
    } else if (typeIs("GeoPoint")) {
        addStepBasedSuggestion(map, categoryGeography, "extractlatlon", "GeoPointExtractor", {
            'column': columnName,
            'lat_col': columnName+'_latitude',
            'lon_col' : columnName+'_longitude'
        }, translate('SHAKER.SUGGESTIONS.COLUMN.GeoPoint.EXTRACT_COORDINATES', actionVerb("Extract") + " latitude/longitude"));
        if (PluginsService.isPluginLoaded('geoadmin')) {
            addStepBasedSuggestion(map, categoryGeography, "geoadmin", "CityLevelReverseGeocoder", {
                'inputCol': columnName
            }, translate('SHAKER.SUGGESTIONS.COLUMN.GeoPoint.REVERSE_GEOCODE', actionVerb("Reverse-geocode") + " location"), true);
        } else {
            addCustomSuggestion(map, categoryGeography, "geoadmin", function(shakerScope) {
                Dialogs.ack(shakerScope, "Plugin required",
                    translate("SHAKER.SUGGESTIONS.COLUMN.GeoPoint.REVERSE_GEOCODE.PLUGIN_REQUIRED",
                        "The <a href='https://doc.dataiku.com/dss/latest/preparation/geo_processors.html' target='_blank'>Reverse Geocoding</a> plugin is required for this action.<br><br><a href='/plugins-explore/store/' target='_blank'>Go to plugins</a> to install or request it.",
                        { pluginStoreUrl: "https://doc.dataiku.com/dss/latest/preparation/geo_processors.html", pluginPageUrl: "/plugins-explore/store/" })
                );
            }, translate('SHAKER.SUGGESTIONS.COLUMN.GeoPoint.REVERSE_GEOCODE', actionVerb("Reverse-geocode") + " location"),
            'dku-icon-globe-16');
        }
    } else if (typeIs("QueryString")) {
        addStepBasedSuggestion(map, categoryWeb, "querystring", "QueryStringSplitter", {
            'column': columnName,
        }, translate('SHAKER.SUGGESTIONS.COLUMN.QueryString.SPLIT', actionVerb("Split query string") + " elements"));
    } else if (typeIs("URL")) {
        addStepBasedSuggestion(map, categoryWeb, "urlsplit", "URLSplitter", {
            'column': columnName,
            extractPath:true,
            extractQueryString:true,
            extractPort:true,
            extractAnchor:true,
            extractScheme:true,
            extractHost:true
        }, translate('SHAKER.SUGGESTIONS.COLUMN.URL.SPLIT', actionVerb("Split URL") + " into host, port ..."));
    } else if (typeIs("Email")) {
        addStepBasedSuggestion(map, categoryEmail, "emailsplit", "EmailSplitter", {
            'column': columnName,
        }, translate('SHAKER.SUGGESTIONS.COLUMN.Email.SPLIT', actionVerb("Split email") + " address"));
    } else if (typeIs("Boolean")) {
        addStepBasedSuggestion(map, categoryBoolean, "booleannot", "BooleanNot", {
            'column': columnName
        }, translate('SHAKER.SUGGESTIONS.COLUMN.Boolean.NEGATE', actionVerb("Negate") + " boolean"));
    } else if (typeIs("Measure")) {
         addStepBasedSuggestion(map, categoryMeasure, "measurenormalize", "MeasureNormalize", {
             'column': columnName
         }, translate('SHAKER.SUGGESTIONS.COLUMN.Measure.NORMALIZE', actionVerb("Normalize") + " to SI units"));
    } else if (typeIs("DateSource") && !multi) {
        addCustomSuggestion(map, categoryDate, "date_stuff", function(scope) {
            report("shaker-use-col-sugg", "SmartDate", "Parse date...")();
            var deferred = $q.defer();
            CreateModalFromTemplate("/templates/shaker/smartdate-box.html", scope, "SmartDateController",
                function(newScope) { newScope.$apply(function() {
                        newScope.deferred = deferred;
                        newScope.setColumn(columnName);
                }); }, "sd-modal");
            deferred.promise.then(function([newFormat, newFormatLanguage]) {
                scope.addStepAndRefresh("DateParser", {
                    "appliesTo" : "SINGLE_COLUMN",
                    "columns" : [columnName],
                    "outCol" : columnName + "_parsed", "formats" : [newFormat],
                    "lang" : newFormatLanguage || "en_US", "timezone_id" : "UTC"});
            });
        }, translate('SHAKER.SUGGESTIONS.COLUMN.DateSource.PARSE', actionVerb("Parse") + " date..."), "dku-icon-calendar-16");
    } else if (typeIs("DateSource")) {
        addAppliesToSuggestion(map, categoryDate, null, "DateParser", {
            "lang" : "en_US", "timezone_id" : "UTC"
        },  translate('SHAKER.SUGGESTIONS.COLUMN.DateSource.PARSE', actionVerb("Parse") + " date..."), true);
    }
    else if (typeIs("Date") || typeIs("DatetimeNoTz") || typeIs("DateOnly") ) {
        addStepBasedSuggestion(map, categoryDate, "date_difference", "DateDifference", {
            "input1" : columnName,
            "output" : "since_" + columnName + "_days",
            "compareTo" : "NOW",
            "outputUnit":  "DAYS"
        }, translate('SHAKER.SUGGESTIONS.COLUMN.DateSource.COMPUTE_TIME_SINCE', actionVerb("Compute time") + " since"));
        addStepBasedSuggestion(map, categoryDate, "extract_components", "DateComponentsExtractor", {
                "column" : columnName,
                "timezone_id" : "UTC",
                "outYearColumn" : columnName + "_year",
                "outMonthColumn" : columnName + "_month",
                "outDayColumn" : columnName + "_day"
        }, translate('SHAKER.SUGGESTIONS.COLUMN.DateSource.EXTRACT_DATE_COMPONENTS', actionVerb("Extract") + " date components"));
        addStepBasedSuggestion(map, categoryDate, "filterOnDate", "FilterOnDate", {
                "appliesTo": 'SINGLE_COLUMN',
                "columns" : [columnName],
                "action": 'KEEP_ROW',
                "filterType": 'RANGE',
                "timezone_id": "UTC",
                "part": 'YEAR',
                "option": {
                    containsCurrentDatePart: true,
                    isUntilNow: false,
                    last: 0,
                    next: 0
                },
        }, translate('SHAKER.SUGGESTIONS.COLUMN.DateSource.FILTER_ON_DATE', actionVerb("Filter") + " on date"), true);
    } else if (typeIs("UserAgent")) {
         addStepBasedSuggestion(map, categoryWeb, "useragent", "UserAgentClassifier", {
            'column': columnName,
        }, translate('SHAKER.SUGGESTIONS.COLUMN.UserAgent.CLASSIFY', actionVerb("Classify") + " User-Agent"));
    } else if (typeIs("JSONObjectMeaning")) {
        addStepBasedSuggestion(map, categoryTransformations, null, "JSONFlattener", {
            "inCol" : columnName,
            "maxDepth" : 1,
            'prefixOutputs' : true,
            'nullAsEmpty' : true,
            'separator' : '_'
        }, translate('SHAKER.SUGGESTIONS.COLUMN.JSONObject.UNNEST', actionVerb("Unnest") + " object"));
    } else if (typeIs("FreeText")) {
         addStepBasedSuggestion(map, categoryText, "null", "Tokenizer", {
             'inCol': columnName,
             'operation' : 'TO_JSON',
             'language' : 'english'
         }, translate('SHAKER.SUGGESTIONS.COLUMN.FreeText.TOKENIZE', actionVerb("Split") + " in words (tokenize)"));
         addStepBasedSuggestion(map, categoryText, "null", "SplitIntoChunks", {
             'inCol': columnName,
             'outCol': columnName + '_chunked',
             "separators": [
                 {
                     value: "\\n\\n",
                     isDefault: true,
                     description: "Double new lines",
                     enabled: true
                 },
                 {
                     value: "\\n",
                     isDefault: true,
                     description: "New Lines",
                     enabled: true
                 },
                 {
                     value: " ",
                     isDefault: true,
                     description: "Spaces",
                     enabled: true
                 },
                 {
                     value: "",
                     isDefault: true,
                     description: "Each character",
                     enabled: true
                 }
             ],
             "chunkSize": 4000,
             "chunkOverlap": 200,
             "keepSeparator": true,
             "isRegex": false,
             "stripWhitespace": true,
         }, translate('SHAKER.SUGGESTIONS.COLUMN.FreeText.SPLIT_INTO_CHUNKS', actionVerb("Split") + " into chunks"));
         addStepBasedSuggestion(map, categoryText, "null", "TextSimplifierProcessor", {
             'inCol': columnName,
             'normalize' : true,
             'language' : 'english'
         }, translate('SHAKER.SUGGESTIONS.COLUMN.FreeText.SIMPLIFY', actionVerb("Simplify") + " text (normalize, stem, clear stop words)"));
     } else if (typeIs("BagOfWordsMeaning")) {
        addStepBasedSuggestion(map, categoryText, null, "ArrayFold", {
             "column" : columnName
        }, translate('SHAKER.SUGGESTIONS.COLUMN.Text.FOLD', actionVerb("Fold") + " to one word per line"));
        addStepBasedSuggestion(map, categoryText, null, "ArrayUnfold", {
             "column" : columnName,
             "prefix": columnName + "_",
             'countVal': true,
             'limit': 100,
             'overflowAction': 'ERROR'
        }, translate('SHAKER.SUGGESTIONS.COLUMN.Text.UNFOLD', actionVerb("Unfold") + " to several columns"));
    } else if (typeIs("JSONArrayMeaning")) {
        addStepBasedSuggestion(map, categoryText, null, "ArrayFold", {
             "column" : columnName
        }, translate('SHAKER.SUGGESTIONS.COLUMN.JSONArray.FOLD', actionVerb("Fold") + " to one element per line"));
        addStepBasedSuggestion(map, categoryText, null, "ArrayUnfold", {
             "column" : columnName,
             "prefix": columnName + "_",
             'countVal': true,
             'limit': 100,
             'overflowAction': 'ERROR'
        }, translate('SHAKER.SUGGESTIONS.COLUMN.JSONArray.UNFOLD', actionVerb("Unfold") + " to several columns"));
    } else if (typeIs("DoubleMeaning")) {
        addAppliesToSuggestion(map, categoryNumbers, null, "RoundProcessor", {
            "mode": "ROUND",
            "precision": 0,
            "places": 0
        }, translate('SHAKER.SUGGESTIONS.COLUMN.Double.ROUND', actionVerb("Round") + " to integer"), false);
    } else if (typeIs("FrenchDoubleMeaning")) {
        addAppliesToSuggestion(map, categoryNumbers, null, "NumericalFormatConverter", {
            inFormat: 'FR', outFormat: 'RAW'
        }, translate('SHAKER.SUGGESTIONS.COLUMN.FrenchDouble.CONVERT', actionVerb("Convert") + " French format to regular decimal"), true);
    } else if (typeIs("CurrencyAmountMeaning")) {
        addStepBasedSuggestion(map, categoryTransformations, "c1", "CurrencySplitter", {
            'inCol': column.name,
            'outColCurrencyCode' : column.name + "_currency_code",
            'outColAmount' : column.name + "_amount",
            'pristineAmount': false,
        }, translate('SHAKER.SUGGESTIONS.COLUMN.CurrencyAmount.SPLIT', actionVerb("Split") + " currency and amount"));
    }

    /* Suggestions for bad data */
    if (hasNOK) {
        if (!(forCell && forInvalidCell)) {
            addAppliesToSuggestion(map, categoryDataCleansing, "remove_badtype", "FilterOnBadType", {
                "action": "REMOVE_ROW", "type" : typeName, "booleanMode" : "AND"
            }, types.length === 1
                ? translate('SHAKER.SUGGESTIONS.COLUMN.RemoveBadType.REMOVE.SINGLE', actionVerb("Remove invalid") + " rows for meaning {{meaning}}", { meaning: meaningName(meaningLabelFilter(typeName))})
                : translate('SHAKER.SUGGESTIONS.COLUMN.RemoveBadType.REMOVE.MULTIPLE', actionVerb("Remove invalid") + " rows for meaning")
            );
            addAppliesToSuggestion(map, categoryDataCleansing, "clear_badtype", "FilterOnBadType", {
                "action": "CLEAR_CELL", "type" : typeName
            }, types.length === 1
                ? translate('SHAKER.SUGGESTIONS.COLUMN.RemoveBadType.CLEAR.SINGLE', actionVerb("Clear invalid") + " cells for meaning {{meaning}}", { meaning: meaningName(meaningLabelFilter(typeName))})
                : translate('SHAKER.SUGGESTIONS.COLUMN.RemoveBadType.CLEAR.MULTIPLE', actionVerb("Clear invalid") + " cells for meaning")
            );
        }
        addStepBasedSuggestion(moreMap, categoryDataCleansing, "split_badtype", "SplitInvalidCells", {
            "column" : columnName, "type" : typeName,
            "invalidColumn" : columnName + "_invalid"
        }, types.length === 1
            ? translate('SHAKER.SUGGESTIONS.COLUMN.SplitBadType.SPLIT.SINGLE', "Move invalid cells for meaning {{meaning}} to <em>{{multi ? '&lt;column name&gt;' : columnName}}_invalid</em>", { meaning: meaningName(meaningLabelFilter(typeName)), multi: multi, columnName: columnName })
            : translate('SHAKER.SUGGESTIONS.COLUMN.SplitBadType.SPLIT.MULTIPLE', "Move invalid cells for meaning to <em>{{multi ? '&lt;column name&gt;' : columnName}}_invalid</em>", { multi: multi, columnName: columnName })
        );
    }

    if (types.length === 1 && appConfig.meanings.categories.filter(cat => cat.label === "User-defined")[0].meanings.filter(mapping => mapping.id === typeName && mapping.type === 'VALUES_MAPPING').length) {
        addAppliesToSuggestion(map, categoryTransformations, "translate_meaning", "MeaningTranslate", { "meaningId": typeName },
            translate('SHAKER.SUGGESTIONS.COLUMN.MeaningTranslate.TRANSLATE', actionVerb("Translate") + " using meaning {{meaning}}", { meaning: meaningName(typeName) }), false);
    }

    /* Now, if at this point we have 0 items in the main suggestions,
     * we add some that would normally go to more */

    if (hasEmpty) {
        target = Object.keys(map).length > 0 ? moreMap : map;
        addAppliesToSuggestion(target, categoryDataCleansing, "rm_empty", "RemoveRowsOnEmpty", {"keep": false},
            translate('SHAKER.SUGGESTIONS.COLUMN.RemoveRowsOnEmpty.REMOVE', actionVerb("Remove") + " rows with no value"), false);

        addCustomSuggestion(target, categoryDataCleansing, "fill_empty_withval", function(scope) {
            CreateModalFromDOMElement("#fill-empty-with-value-box", scope, "FillEmptyWithValueController",
                function(newScope) { newScope.$apply(function() {
                        newScope.setColumns(columnNames);
                        newScope.isNumericOnly = ["LongMeaning", "DoubleMeaning"]
                            .indexOf(column.selectedType && column.selectedType.name) > -1;
                }); });
        }, translate('SHAKER.SUGGESTIONS.COLUMN.FillEmptyWithVal.FILL', actionVerb("Fill") + " empty rows with..."),
        ShakerProcessorsUtils.getStepIcon('FillEmptyWithValue', {}, '16'));

        addStepBasedSuggestion(moreMap, categoryDataCleansing, "fill_empty_prev", "UpDownFiller", {
            "columns" : [columnName],
            "up" : false
        }, translate('SHAKER.SUGGESTIONS.COLUMN.FillEmptyWithVal.FILL_WITH_PREVIOUS', actionVerb("Fill") + " empty rows with previous value"), true);
    }

    if (typesAllIn(["LongMeaning", "DoubleMeaning"])) {
        target = Object.keys(map).length > 2 ? moreMap : map;
        if (multi) {
            addCustomSuggestion(target, categoryKeepDelete, "numerical_range_selector", function(scope) {
                report("shaker-use-col-sugg", "FilterOnNumericalRange", "Delete/Keep on number range...")();
                CreateModalFromDOMElement("#value-range-filter-box", scope, "MultiRangeController",
                    function(newScope) { newScope.$apply(function() {
                            newScope.setColumns(columnNames);
                    }); });
            }, translate('SHAKER.SUGGESTIONS.COLUMN.FilterOnNumericalRange.FILTER', actionVerb("Filter") + " with numerical range..."),
            ShakerProcessorsUtils.getStepIcon('FilterOnNumericalRange', {}, '16')
        );
        } else {
            addStepBasedSuggestion(target, categoryKeepDelete, null, "FilterOnNumericalRange", {
                    "appliesTo" : "SINGLE_COLUMN",
                    "columns": [columnName],
                    "min": undefined,
                    "max" : undefined,
                    "action" : "KEEP_ROW",
                }, translate('SHAKER.SUGGESTIONS.COLUMN.FilterOnNumericalRange.FILTER', actionVerb("Filter") + " with numerical range..."), true);
        }
    }

    if (noTypeIn(["LongMeaning", "DoubleMeaning", "Date"])) {
        target = Object.keys(map).length > 0 ? moreMap : map;

        addAppliesToSuggestion(target, categoryTransformations, null, "StringTransformer", {
                "mode" : "TO_LOWER"
        }, translate('SHAKER.SUGGESTIONS.COLUMN.StringTransformer.TO_LOWER', actionVerb("Convert") + " to lowercase"));
        addAppliesToSuggestion(target, categoryTransformations, null, "StringTransformer", {
                "mode" : "TO_UPPER"
        }, translate('SHAKER.SUGGESTIONS.COLUMN.StringTransformer.TO_UPPER', actionVerb("Convert") + " to uppercase"));
        addAppliesToSuggestion(target, categoryTransformations, null, "StringTransformer", {
                "mode": "TO_LOWER"
        }, translate('SHAKER.SUGGESTIONS.COLUMN.StringTransformer.TRANSFORM', actionVerb("Transform") + " string"), true);

    }
    /* At this point, we should always have something in the main map */

    /* Add the rest of type-based tranformations to the more map */

    if (typeIs("Date")) {
        addStepBasedSuggestion(moreMap, categoryDate, "extract_ts", "DateComponentsExtractor", {
                "column" : columnName,
                "timezone_id" : "UTC",
                "outTimestampColumn" : columnName + "_timestamp",
        }, translate('SHAKER.SUGGESTIONS.COLUMN.Date.CONVERT_TO_UNIX', actionVerb("Convert") + " to UNIX timestamp"));
        addStepBasedSuggestion(moreMap, categoryDate, "holidays", "HolidaysComputer", {
            "inCol" : columnName,
            "outColPrefix" : columnName+'_holiday_',
            "flagBankHolidays":true,
            "calendar_id":"FR",
            "timezone_id":"use_preferred_timezone",
            "flagWeekends":true,
            "flagSchoolHolidays":true
        }, translate('SHAKER.SUGGESTIONS.COLUMN.Date.FLAG_HOLIDAYS', actionVerb("Flag") + " holidays"));
        addStepBasedSuggestion(moreMap, categoryDate, "date_format", "DateFormatter", {
            "inCol": columnName,
            "outCol": columnName + "_formatted",
            "lang" : "en_US",
            "timezone_id" : "UTC",
            "format": "yyyy-MM-dd HH:mm:ss"
        }, translate('SHAKER.SUGGESTIONS.COLUMN.Date.REFORMAT', actionVerb("Reformat") + " date"));
    }

    /* Text also gets some text handling, but in more Map unlike FreeText */
    if (typeIs("Text")) {
        addStepBasedSuggestion(moreMap, categoryText, null, "Tokenizer", {
             'inCol': columnName,
             'operation' : 'TO_JSON',
             'language' : 'english'
         }, translate('SHAKER.SUGGESTIONS.COLUMN.Text.TOKENIZE', actionVerb("Split") + " in words (tokenize)"));
         addStepBasedSuggestion(moreMap, categoryText, null, "TextSimplifierProcessor", {
             'inCol': columnName,
             'normalize':true,
             'language' : 'english'
         }, translate('SHAKER.SUGGESTIONS.COLUMN.Text.NORMALIZE', actionVerb("Normalize") + " text"));
    }

    addStepBasedSuggestion(moreMap, categoryTransformations, null, "ColumnCopier", {
        "inputColumn": columnName,
        "outputColumn": columnName+"_copy"
    }, translate('SHAKER.SUGGESTIONS.COLUMN.DUPLICATE', actionVerb("Duplicate") + " column"));

    if (!multi) {
        addAppliesToSuggestion(moreMap, categoryTransformations, null, "FindReplace", {
            "output": "",
            "mapping": [],
            "matching" : "FULL_STRING",
            "normalization" : "EXACT"
        }, translate('SHAKER.SUGGESTIONS.COLUMN.FIND_AND_REPLACE', actionVerb("Find") + " and " + actionVerb("replace") + "..."), true);
        addStepBasedSuggestion(
            ["DoubleMeaning", "LongMeaning"].indexOf(typeName) === -1 ? moreMap : map,
            categoryNumbers, null, "CreateColumnWithGREL",{
                "column" : columnName,
                "expression" : columnName.match(/^[a-z0-9_]+$/i) ? columnName : `val("${columnName}")`
            }, translate('SHAKER.SUGGESTIONS.COLUMN.PROCESS_WITH_FORMULA', actionVerb("Process") + " with formula..."));
    } else {
        addCustomSuggestion(moreMap, categoryTransformations, null, function(shakerScope) {
            report("shaker-use-col-sugg", "ColumnsConcat", "Concatenate columns")();
            shakerScope.addUnconfiguredStep("ColumnsConcat",
                { columns: columnNames, join: ',' });
            shakerScope.autoSave();
        }, translate('SHAKER.SUGGESTIONS.COLUMN.CONCAT_COLUMNS', actionVerb("Concatenate") + " columns"),
        ShakerProcessorsUtils.getStepIcon('ColumnsConcat', {}, '16'));
        addCustomSuggestion(moreMap, categoryTransformations, null, function(shakerScope) {
            report("shaker-use-col-sugg", "NestProcessor", "Nest columns")();
            shakerScope.addUnconfiguredStep("NestProcessor",
                { columns: columnNames, join: ',', appliesTo: 'COLUMNS' });
            shakerScope.autoSave();
        }, translate('SHAKER.SUGGESTIONS.COLUMN.NEST_TO_OBJECT', actionVerb("Nest") + " columns to object"),
        ShakerProcessorsUtils.getStepIcon('FilterOnNumericalRange', {}, '16'));
        addCustomSuggestion(moreMap, categoryTransformations, null, function(shakerScope) {
            report("shaker-use-col-sugg", "MultiColumnFold", "Fold columns")();
            shakerScope.addUnconfiguredStep("MultiColumnFold",
                { columns: columnNames, join: ',' });
            shakerScope.autoSave();
        }, translate('SHAKER.SUGGESTIONS.COLUMN.FOLD_COLUMNS', actionVerb("Fold ") + " columns"),
        ShakerProcessorsUtils.getStepIcon('MultiColumnFold', {}, '16'));
        addCustomSuggestion(moreMap, categoryTransformations, null, function(shakerScope) {
            report("shaker-use-col-sugg", "FindReplace", "Find & replace")();
            shakerScope.addUnconfiguredStep("FindReplace",
                { columns: columnNames, join: ',', appliesTo: 'COLUMNS'});
            shakerScope.autoSave();
        }, translate('SHAKER.SUGGESTIONS.COLUMN.FIND_AND_REPLACE_IN_COLUMNS', actionVerb("Find") + " and " + actionVerb("replace") + " in columns"),
        ShakerProcessorsUtils.getStepIcon('FindReplace', {}, '16'));
    }

    if (Object.keys(map).length === 0) { // shift
        map = moreMap;
        moreMap = {};
    }
    return [map, moreMap, Object.keys(moreMap).length];
};

return svc;
});

})();
