import logging
import json
from dataikuapi.dss.llm import DSSLLMStreamedCompletionFooter
from dataikuapi.dss.llm_tracing import SpanBuilder
from dataiku.llm.python.tools_using_2 import InternalCompletionChunk

from .. import NextBlock, BlockHandler
from ...utils import get_completion_query_safe_for_logging

logger = logging.getLogger("dku.agents.blocks_graph")

class DelegateToOtherAgentBlockHandler(BlockHandler):
    def __init__(self, turn, sequence_context, block_config):
        super().__init__(turn, sequence_context, block_config)

    def process_stream(self, trace: SpanBuilder):
        logger.info("Delegate to other agent block starting with config %s" % self.block_config)

        agent_ref = self.block_config.get("agentRef")
        if not agent_ref:
            raise ValueError("Delegate to other agent block requires an agentRef")

        with trace.subspan("DKU_AGENT_SUBAGENT_CALL") as subagent_trace:
            yield {"chunk": {"type": "event", "eventKind": "AGENT_CALLING_A_FRIEND", "eventData": {}}}

            llm = self.agent.project.get_llm(f"agent:{agent_ref}")
            completion = llm.new_completion()

            # TODO @power-agents-hitl If the friend agent is restarting from a HITL call, then we need to restore the HITL state here.
            #                         See the LLM Mesh query tool: LLMMeshLLMQueryTool.java#L231
            completion.cq["messages"].extend(self.turn.initial_messages)
            completion.cq["messages"].extend(self.sequence_context.generated_messages)

            completion.cq["context"] = self.turn.current_merged_context

            logger.trace("About to run completion: %s" % get_completion_query_safe_for_logging(completion.cq))
            logger.trace("With settings: %s" % completion.settings)

            accumulated_text_output = ""
            for chunk in completion.execute_streamed():
                artifacts = chunk.data.get("artifacts", [])
                sources = chunk.data.get("additionalInformation", {}).get("sources", [])  # TODO @structured-agents Return sources somehow
                tool_validation_requests = chunk.data.get("toolValidationRequests")

                self.sequence_context.sources.extend(sources)

                #yield {"chunk": {"type": "event", "eventKind": "AGENT_FRIEND_GOT_CHUNK", "eventData": {"chunk": json.dumps(chunk.data)}}}

                # TODO @power-agents-hitl If we got tool validation requests from the friend agent, then we need to return them here, and make a memory fragment with the tool validation requests, stashed sources, and text output.
                #                         See the LLM Mesh query tool: LLMMeshLLMQueryTool.java#L374
                if tool_validation_requests:
                    raise Exception("Delegated agent called a tool that requires human validation. Currently not supported by delegate to other agent blocks. Please use a core loop block instead.")

                if isinstance(chunk, DSSLLMStreamedCompletionFooter):
                    subagent_trace.append_trace(chunk.trace)

                    #yield {"chunk": {"type": "event", "eventKind": "AGENT_FRIEND_GOT_FOOTER", "eventData": {"footer": json.dumps(chunk.data)}}}

                    if chunk.data.get("contextUpsert") is not None:
                        # Handle context upsert from delegated agent
                        logger.info("Delegated agent requested context upsert: %s" % chunk.data.get("contextUpsert"))
                        for (key, val) in chunk.data.get("contextUpsert").items():
                            self.turn.context_set(key, val)

                    # TODO: Do something with the rest of the footer?
                else:
                    # DEBUG
                    #if chunk.text is None:
                    #    yield {"chunk": {"type": "event", "eventKind": "AGENT_FRIEND_GOT_NON_TEXT", "eventData": {"data": json.dumps(chunk.data)}}}

                    if chunk.text is not None:
                        if self.block_config.get("streamOutput", False):
                            yield {"chunk": {"text": chunk.text}}
                        accumulated_text_output += chunk.text
                    elif chunk.type == "event":
                        yield chunk
                    elif chunk.type == "content" and "memoryFragment" in chunk.data:
                        # Incorporate child's generated messages into parent's sequence context
                        messages = chunk.data["memoryFragment"].get("messages", [])
                        for msg in messages:
                            self.sequence_context.generated_messages.append(msg)
                    else:
                        logger.warning("Unknown chunk type from delegated agent: %s, sending it nevertheless" % chunk)
                        yield chunk

                if len(artifacts) > 0:
                    yield {"chunk": {"artifacts": artifacts}}

            if self.block_config.get("streamOutput", False) and self.agent.config.get("newLineAfterBlockOutput", True):
                yield {"chunk":  {"text": "\n"}}

            self.sequence_context.last_text_output = accumulated_text_output
            logger.info("Delegated agent response stored in last_text_output")

            output_mode = self.block_config.get("outputMode", None)
            if output_mode == 'NONE':
                pass  # Do not store the output anywhere, just keep it in last_text_output
            elif output_mode == "ADD_TO_MESSAGES":
                if accumulated_text_output:
                    assistant_message = {
                        "role": "assistant",
                        "content": accumulated_text_output
                    }
                    self.sequence_context.generated_messages.append(assistant_message)
                    logger.info("Delegated agent response added to generated messages")
            elif output_mode == "SAVE_TO_STATE":
                output_key = self.block_config["outputStateKey"]
                self.turn.state_set(output_key, accumulated_text_output)
                logger.info("Saved delegated agent output to state key '%s'", output_key)
            elif output_mode == "SAVE_TO_SCRATCHPAD":
                output_key = self.block_config["outputScratchpadKey"]
                self.sequence_context.scratchpad[output_key] = accumulated_text_output
                logger.info("Saved delegated agent output to scratchpad key '%s'", output_key)
            else:
                raise ValueError("Unsupported output mode for Delegate block: %s" % output_mode)

        yield NextBlock(id=self.block_config.get("nextBlock"))