from dataiku.project_standards import (
    ProjectStandardsCheckRunResult,
    ProjectStandardsCheckSpec,
)


def saved_model_md_link(project_key, id, display_name=None):
    """
    Return a formatted string to display md link.
    """
    if not display_name:
        display_name = id
    return "[{}](saved_model:{}.{})".format(display_name, project_key, id)


def markdown_for_agents(project_key, agents_with_missing_guardrails):
    if not agents_with_missing_guardrails:
        return "-"

    return ", ".join(
        [markdown_for_agent(project_key, agent, guardrails) for agent, guardrails in agents_with_missing_guardrails]
    )


def markdown_for_agent(project_key, agent, missing_guardrails):
    agent_md = saved_model_md_link(project_key, agent.id, agent.name)
    guardrails_md = ", ".join(missing_guardrails)
    return "{} ({})".format(agent_md, guardrails_md) if guardrails_md else agent_md


class ProjectStandardsCheck(ProjectStandardsCheckSpec):

    def run(self):
        """
        Check that all agents in the project have enabled all required guardrails.
        - If no required guardrails have been configured in the check -> Not applicable.
        - If the project has no agents -> Not applicable.
        - For each agent:
            - Look at the active version.
            - Read guardrailsPipelineSettings.guardrails.
            - Compare enabled guardrails to the required guardrails from config.
            - Record missing required guardrails in details.
        - If an agent does not have any of the required guardrails -> Critical Failure.
        - If all agents have at least one of the required guardrails but some of the required guardrails are missing -> Failure.
        - If all required guardrails are used on all agents -> Success.
        """

        required_guardrails = set(self.config.get("requiredGuardrails", [])) or set()

        # If no required guardrails are configured treat as not applicable rather than pass
        if not required_guardrails:
            return ProjectStandardsCheckRunResult.not_applicable(
                message="Skipping this check because no required guardrails were configured."
            )

        # List agents
        agents_list = self.project.list_agents()

        # Not applicable if there are no agents in the project
        if not agents_list:
            return ProjectStandardsCheckRunResult.not_applicable(
                message="Skipping this check because the project does not have any agents."
            )

        agents_with_missing_guardrails = []
        agents_in_error = {}

        # Loop through agents and check guardrails for active version
        for agent in agents_list:
            agent_id = agent.id
            try:
                agent_settings = self.project.get_agent(agent_id).get_settings()

                active_version_id = agent_settings.active_version
                active_version = agent_settings.get_version_settings(active_version_id) if active_version_id else None
                raw_active_version = active_version.get_raw() if active_version else {}

                guardrails_pipeline_settings = raw_active_version.get("guardrailsPipelineSettings", {}) or {}
                guardrails = guardrails_pipeline_settings.get("guardrails", []) or []

                # Collect enabled guardrails for this agent
                used_guardrails = {gr.get("type") for gr in guardrails if gr.get("enabled", True)}

                # Determine which required guardrails are missing for this agent
                missing_guardrails = required_guardrails - used_guardrails

                if missing_guardrails:
                    agents_with_missing_guardrails.append((agent, missing_guardrails))
            except Exception as e:
                agents_in_error[agent_id] = str(e)

        details = {}
        details["required_guardrails"] = list(required_guardrails)
        details["missing_guardrails_by_agent"] = {
            agent.id: list(guardrails) for agent, guardrails in agents_with_missing_guardrails
        }
        details["agents_with_missing_guardrails_links"] = markdown_for_agents(
            self.original_project_key, agents_with_missing_guardrails
        )

        if agents_in_error:
            # An error happened when checking one of the agent.
            # Nevertheless, the info of the other agents are still available in the details.
            details["exceptions_by_agent"] = agents_in_error
            return ProjectStandardsCheckRunResult.error(
                message="An error occurred while verifying some agents. See the details for more information.",
                details=details,
            )

        max_count_of_missing_guardrails = max(
            [len(missing_guardrails) for agent, missing_guardrails in agents_with_missing_guardrails], default=0
        )
        if max_count_of_missing_guardrails == len(required_guardrails):
            # No required guardrails used by any agent in the project
            return ProjectStandardsCheckRunResult.failure(
                severity=self.config.get("zeroGuardrailsSeverity"),
                message="At least one of the agent is using none of the required guardrails.",
                details=details,
            )

        if agents_with_missing_guardrails:
            # Some required guardrails are used
            return ProjectStandardsCheckRunResult.failure(
                severity=self.config.get("someMissingGuardrailsSeverity"),
                message="Agents are using some guardrails, but not all required ones.",
                details=details,
            )

        # All agents have all required guardrails
        return ProjectStandardsCheckRunResult.success(message="Agents are using required guardrails.", details=details)
