/*
 * Decompiled with CFR 0.152.
 */
package com.databricks.jdbc.api.impl.converters;

import com.databricks.internal.apache.arrow.vector.TimeStampMicroTZVector;
import com.databricks.internal.apache.arrow.vector.ValueVector;
import com.databricks.internal.apache.arrow.vector.util.Text;
import com.databricks.jdbc.api.impl.AbstractDatabricksGeospatial;
import com.databricks.jdbc.api.impl.ComplexDataTypeParser;
import com.databricks.jdbc.api.impl.DatabricksArray;
import com.databricks.jdbc.api.impl.DatabricksGeography;
import com.databricks.jdbc.api.impl.DatabricksGeometry;
import com.databricks.jdbc.api.impl.DatabricksMap;
import com.databricks.jdbc.api.impl.converters.IntervalConverter;
import com.databricks.jdbc.api.impl.converters.WKTConverter;
import com.databricks.jdbc.exception.DatabricksParsingException;
import com.databricks.jdbc.exception.DatabricksSQLException;
import com.databricks.jdbc.exception.DatabricksValidationException;
import com.databricks.jdbc.log.JdbcLogger;
import com.databricks.jdbc.log.JdbcLoggerFactory;
import com.databricks.jdbc.model.core.ColumnInfo;
import com.databricks.jdbc.model.core.ColumnInfoTypeName;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.sql.Date;
import java.sql.Timestamp;
import java.time.DateTimeException;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.function.Function;

public class ArrowToJavaObjectConverter {
    private static final JdbcLogger LOGGER = JdbcLoggerFactory.getLogger(ArrowToJavaObjectConverter.class);
    private static final List<DateTimeFormatter> DATE_FORMATTERS = Arrays.asList(DateTimeFormatter.ofPattern("yyyy-MM-dd"), DateTimeFormatter.ofPattern("yyyy/MM/dd"), DateTimeFormatter.ofPattern("yyyy.MM.dd"), DateTimeFormatter.ofPattern("yyyyMMdd"), DateTimeFormatter.ofPattern("dd-MM-yyyy"), DateTimeFormatter.ofPattern("dd/MM/yyyy"), DateTimeFormatter.ofPattern("dd.MM.yyyy"), DateTimeFormatter.ofPattern("ddMMyyyy"), DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss"), DateTimeFormatter.ofPattern("yyyy/MM/dd HH:mm:ss"), DateTimeFormatter.ofPattern("yyyy.MM.dd HH:mm:ss"), DateTimeFormatter.ofPattern("yyyyMMdd HH:mm:ss"), DateTimeFormatter.ofPattern("dd-MM-yyyy HH:mm:ss"), DateTimeFormatter.ofPattern("dd/MM/yyyy HH:mm:ss"), DateTimeFormatter.ofPattern("dd.MM.yyyy HH:mm:ss"), DateTimeFormatter.ofPattern("ddMMyyyy HH:mm:ss"), DateTimeFormatter.ISO_LOCAL_DATE_TIME, DateTimeFormatter.ofPattern("yyyy-MM-dd'T'HH:mm:ss.SSS"), DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss.SSS"), DateTimeFormatter.ofPattern("yyyy-MM-dd'T'HH:mm:ss.SSS'Z'"), DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss.SS"), DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss.S"), DateTimeFormatter.RFC_1123_DATE_TIME);

    public static Object convert(ValueVector columnVector, int vectorIndex, ColumnInfoTypeName requiredType, String arrowMetadata, ColumnInfo columnInfo) throws DatabricksSQLException {
        if (columnVector.isNull(vectorIndex)) {
            return null;
        }
        Object object = columnVector.getObject(vectorIndex);
        if (arrowMetadata != null) {
            if (arrowMetadata.startsWith("ARRAY")) {
                requiredType = ColumnInfoTypeName.ARRAY;
            }
            if (arrowMetadata.startsWith("STRUCT")) {
                requiredType = ColumnInfoTypeName.STRUCT;
            }
            if (arrowMetadata.startsWith("MAP")) {
                requiredType = ColumnInfoTypeName.MAP;
            }
            if (arrowMetadata.startsWith("VARIANT")) {
                requiredType = ColumnInfoTypeName.STRING;
            }
            if (arrowMetadata.startsWith("TIMESTAMP")) {
                requiredType = ColumnInfoTypeName.TIMESTAMP;
            }
            if (arrowMetadata.startsWith("GEOMETRY")) {
                requiredType = ColumnInfoTypeName.GEOMETRY;
            }
            if (arrowMetadata.startsWith("GEOGRAPHY")) {
                requiredType = ColumnInfoTypeName.GEOGRAPHY;
            }
        }
        if (object == null) {
            return null;
        }
        switch (requiredType) {
            case BYTE: {
                return ArrowToJavaObjectConverter.convertToNumber(object, Byte::parseByte, Number::byteValue);
            }
            case SHORT: {
                return ArrowToJavaObjectConverter.convertToNumber(object, Short::parseShort, Number::shortValue);
            }
            case INT: {
                return ArrowToJavaObjectConverter.convertToNumber(object, Integer::parseInt, Number::intValue);
            }
            case LONG: {
                return ArrowToJavaObjectConverter.convertToNumber(object, Long::parseLong, Number::longValue);
            }
            case FLOAT: {
                return ArrowToJavaObjectConverter.convertToNumber(object, Float::parseFloat, Number::floatValue);
            }
            case DOUBLE: {
                return ArrowToJavaObjectConverter.convertToNumber(object, Double::parseDouble, Number::doubleValue);
            }
            case DECIMAL: {
                return ArrowToJavaObjectConverter.convertToDecimal(object, columnInfo);
            }
            case BINARY: {
                return ArrowToJavaObjectConverter.convertToByteArray(object);
            }
            case BOOLEAN: {
                return ArrowToJavaObjectConverter.convertToBoolean(object);
            }
            case CHAR: {
                return Character.valueOf(ArrowToJavaObjectConverter.convertToChar(object));
            }
            case STRUCT: {
                return ArrowToJavaObjectConverter.convertToStruct(object, arrowMetadata);
            }
            case ARRAY: {
                return ArrowToJavaObjectConverter.convertToArray(object, arrowMetadata);
            }
            case MAP: {
                return ArrowToJavaObjectConverter.convertToMap(object, arrowMetadata);
            }
            case STRING: {
                return ArrowToJavaObjectConverter.convertToString(object);
            }
            case DATE: {
                return ArrowToJavaObjectConverter.convertToDate(object);
            }
            case TIMESTAMP: {
                Optional<String> timeZone = Optional.empty();
                if (columnVector instanceof TimeStampMicroTZVector) {
                    timeZone = Optional.of(((TimeStampMicroTZVector)columnVector).getTimeZone());
                }
                return ArrowToJavaObjectConverter.convertToTimestamp(object, timeZone);
            }
            case INTERVAL: {
                if (arrowMetadata == null) {
                    String errorMessage = String.format("Failed to read INTERVAL %s with null metadata.", object);
                    LOGGER.error(errorMessage);
                    throw new DatabricksValidationException(errorMessage);
                }
                IntervalConverter ic = new IntervalConverter(arrowMetadata);
                return ic.toLiteral(object);
            }
            case GEOMETRY: 
            case GEOGRAPHY: {
                return ArrowToJavaObjectConverter.convertToGeospatial(object, requiredType);
            }
            case NULL: {
                return null;
            }
        }
        String errorMessage = String.format("Unsupported conversion type %s", new Object[]{requiredType});
        LOGGER.error(errorMessage);
        throw new DatabricksValidationException(errorMessage);
    }

    private static DatabricksMap convertToMap(Object object, String arrowMetadata) throws DatabricksParsingException {
        ComplexDataTypeParser parser = new ComplexDataTypeParser();
        return parser.parseJsonStringToDbMap(object.toString(), arrowMetadata);
    }

    private static DatabricksArray convertToArray(Object object, String arrowMetadata) throws DatabricksParsingException {
        ComplexDataTypeParser parser = new ComplexDataTypeParser();
        return parser.parseJsonStringToDbArray(object.toString(), arrowMetadata);
    }

    private static Object convertToStruct(Object object, String arrowMetadata) throws DatabricksParsingException {
        ComplexDataTypeParser parser = new ComplexDataTypeParser();
        return parser.parseJsonStringToDbStruct(object.toString(), arrowMetadata);
    }

    private static AbstractDatabricksGeospatial convertToGeospatial(Object object, ColumnInfoTypeName type) throws DatabricksSQLException {
        String ewkt = ArrowToJavaObjectConverter.convertToString(object);
        int srid = WKTConverter.extractSRIDFromEWKT(ewkt);
        String cleanWkt = WKTConverter.removeSRIDFromEWKT(ewkt);
        return type == ColumnInfoTypeName.GEOMETRY ? new DatabricksGeometry(cleanWkt, srid) : new DatabricksGeography(cleanWkt, srid);
    }

    private static Object convertToTimestamp(Object object, Optional<String> timeZoneOpt) throws DatabricksSQLException {
        if (object instanceof Text) {
            return ArrowToJavaObjectConverter.convertArrowTextToTimestamp(object.toString());
        }
        if (object instanceof LocalDateTime) {
            return Timestamp.valueOf((LocalDateTime)object);
        }
        long timeMicros = object instanceof Integer ? (long)((Integer)object).intValue() : (Long)object;
        Instant instant = Instant.ofEpochSecond(timeMicros / 1000000L, timeMicros % 1000000L * 1000L);
        ZoneId zoneId = ArrowToJavaObjectConverter.getZoneIdFromTimeZoneOpt(timeZoneOpt);
        LocalDateTime localDateTime = LocalDateTime.ofInstant(instant, zoneId);
        return Timestamp.valueOf(localDateTime);
    }

    static ZoneId getZoneIdFromTimeZoneOpt(Optional<String> timeZoneOpt) {
        if (timeZoneOpt.isPresent()) {
            String tz = timeZoneOpt.get();
            try {
                return ZoneId.of(tz);
            }
            catch (DateTimeException e) {
                if (tz.matches("[+-]\\d+:\\d+")) {
                    boolean isNegative = tz.startsWith("-");
                    String[] parts = tz.substring(1).split(":");
                    int hours = Integer.parseInt(parts[0]);
                    int minutes = Integer.parseInt(parts[1]);
                    return ZoneOffset.ofHoursMinutes(isNegative ? -hours : hours, isNegative ? -minutes : minutes);
                }
                throw e;
            }
        }
        return ZoneId.systemDefault();
    }

    private static Object convertArrowTextToTimestamp(String arrowText) throws DatabricksSQLException {
        LocalDateTime localDateTime = ArrowToJavaObjectConverter.parseDate(arrowText);
        return Timestamp.valueOf(localDateTime);
    }

    private static LocalDateTime parseDate(String text) throws DatabricksSQLException {
        for (DateTimeFormatter formatter : DATE_FORMATTERS) {
            try {
                return LocalDateTime.parse(text, formatter);
            }
            catch (DateTimeParseException dateTimeParseException) {
            }
        }
        String errorMessage = String.format("Unsupported text for date conversion: %s", text);
        LOGGER.error(errorMessage);
        throw new DatabricksValidationException(errorMessage);
    }

    private static Date convertToDate(Object object) throws DatabricksSQLException {
        if (object instanceof Text) {
            LocalDateTime localDateTime = ArrowToJavaObjectConverter.parseDate(object.toString());
            return Date.valueOf(localDateTime.toLocalDate());
        }
        LocalDate localDate = LocalDate.ofEpochDay(((Integer)object).intValue());
        return Date.valueOf(localDate);
    }

    private static char convertToChar(Object object) {
        return object.toString().charAt(0);
    }

    private static String convertToString(Object object) {
        return object.toString();
    }

    private static boolean convertToBoolean(Object object) {
        if (object instanceof Text) {
            return Boolean.parseBoolean(object.toString());
        }
        return (Boolean)object;
    }

    private static byte[] convertToByteArray(Object object) {
        if (object instanceof Text) {
            return object.toString().getBytes();
        }
        return (byte[])object;
    }

    static BigDecimal convertToDecimal(Object object, ColumnInfo columnInfo) throws DatabricksValidationException {
        if (object instanceof Text || object instanceof Number) {
            BigDecimal bigDecimal = new BigDecimal(object.toString());
            Optional<Integer> bigDecimalScale = columnInfo.getTypeScale() != null ? Optional.of(columnInfo.getTypeScale().intValue()) : Optional.empty();
            return bigDecimalScale.map(scale -> bigDecimal.setScale((int)scale, RoundingMode.HALF_UP)).orElse(bigDecimal);
        }
        String errorMessage = String.format("Unsupported object type for decimal conversion: %s", object.getClass());
        LOGGER.error(errorMessage);
        throw new DatabricksValidationException(errorMessage);
    }

    private static <T extends Number> T convertToNumber(Object object, Function<String, T> parseFunc, Function<Number, T> convertFunc) throws DatabricksSQLException {
        if (object instanceof Text) {
            return (T)((Number)parseFunc.apply(object.toString()));
        }
        if (object instanceof Number) {
            return (T)((Number)convertFunc.apply((Number)object));
        }
        String errorMessage = String.format("Unsupported object type for number conversion: %s", object.getClass());
        LOGGER.error(errorMessage);
        throw new DatabricksValidationException(errorMessage);
    }
}

