(function () {
    "use strict";
    var app = angular.module("dataiku.admin.alertMessageBanner", []);
    const ALERT_MESSAGE_BANNER_ID_KEY = "dss.admin.ALERT_MESSAGE_BANNER_ID_KEY";
    app.component("alertMessageBanner", {
        template: `
            <div ng-if="$ctrl.display && !$root.appConfig.unattendedMode" class="alert alert-warning alert-message-banner children-horizontal-spacing-2x" role="alert">
                <div class="alert-message-banner__header children-horizontal-spacing-2x">
                    <i class="dku-icon-warning-fill-16"></i>
                    <blockquote class="alert-message-banner__title" from-markdown="$ctrl.alertBanner.message"></blockquote>
                </div>
                <button class="btn btn--outline btn--secondary" ng-click="$ctrl.dismiss()">Dismiss</button>
            </div>
        `,
        controller: function ($rootScope, LocalStorage, Notification, Logger, DataikuAPI, $scope) {
            this.display = false;
            this.alertBanner = undefined;

            const handleLocalStorageAndDisplay = (alertMessageBanner) => {
                const storedAlertMessageBannerId = LocalStorage.get(ALERT_MESSAGE_BANNER_ID_KEY);
                if (alertMessageBanner && storedAlertMessageBannerId !== alertMessageBanner.id) {
                    LocalStorage.clear(ALERT_MESSAGE_BANNER_ID_KEY);
                    // don't show to the alert banner author
                    if (md5($rootScope.appConfig.login) !== alertMessageBanner.authorId) {
                        this.display = true;
                    }
                } else {
                    this.display = false;
                }
            }

            this.$onInit = function () {
                try {
                    this.alertBanner = $rootScope.appConfig.alertBanner;
                    handleLocalStorageAndDisplay(this.alertBanner);
                } catch (error) {
                    Logger.error("Error handling initialisation of alertMessageBanner.", error)
                    this.display = false;
                }
            };

            const notificationUnsubscribe = Notification.registerEvent("admin-message-banner-changed", (_, message) => {
                this.alertBanner = message.alertBanner;
                handleLocalStorageAndDisplay(message.alertBanner);
            });
            $scope.$on('$destroy', notificationUnsubscribe);

            this.dismiss = function () {
                LocalStorage.set(ALERT_MESSAGE_BANNER_ID_KEY, this.alertBanner.id);
                this.display = false;

                // Ignore errors returned by this call: we're just notifying the backend so that it can add a message in audit log
                DataikuAPI.profile.notifyAlertBannerDismissed(this.alertBanner.id);
            };
        },
    });

    app.component("alertMessageBannerForm", {
        bindings: {
            alertBanner: "=",
        },
        template: `
            <form class="dkuform-horizontal">
                <h2 id="alert-message-banner-form">Alert Banner</h2>
                <div class="control-group">
                    <label for="alert-banner-message" class="control-label">Message</label>
                    <div class="controls">
                        <textarea id="alert-banner-message" style="height: 5rem;" ng-model="$ctrl.workingAlertBannerMessage" ng-change="$ctrl.onMessageChange()" />
                        <span class="help-inline">Text of the alert banner. Supports Markdown.</span>
                    </div>
                    <input type="hidden" ng-model="$ctrl.alertBanner.id">
                    <input type="hidden" ng-model="$ctrl.alertBanner.authorId">
                </div>
            </form>
        `,
        controller: function ($rootScope) {
            let previousContent;
            // To avoid updating parent scope alert banner in an uncontrolled way
            this.workingAlertBannerMessage;

            // because $onInit `alertBanner` may be undefined
            this.$doCheck = () => {
                if (this.alertBanner !== undefined && previousContent === undefined) {
                    previousContent = angular.copy(this.alertBanner);
                    this.workingAlertBannerMessage = angular.copy(this.alertBanner.message);
                }
            };

            /**
             * Triggered on message change or message paste
             */
            this.onMessageChange = function () {
                // we don't want to send an empty alert to users
                if (this.workingAlertBannerMessage === "") {
                    this.alertBanner = undefined;
                } else if (previousContent && previousContent.message !== this.workingAlertBannerMessage || previousContent === undefined) {
                    this.alertBanner = {};
                    this.alertBanner.message = this.workingAlertBannerMessage;
                    this.alertBanner.id = md5(this.workingAlertBannerMessage);
                    this.alertBanner.authorId = md5($rootScope.appConfig.login);
                } else {
                    this.alertBanner = previousContent;
                }

            };
        },
    });
})();
