(function(){
'use strict';

const app = angular.module('dataiku.analysis.mlcore');

app.component("diagnosticsIcon", {
    bindings: { noBorder: "@?"},
    template: `<div class="feature-badge-icon-wrapper feature-badge-icon-wrapper--diagnostics" ng-class="{'feature-badge-icon-wrapper--no-border': $ctrl.noBorder}"><div class="feature-badge-icon"><i class="dku-icon-stethoscope-12"></i></div></div>`
});

app.component("overridesIcon", {
    template: `<div class="feature-badge-icon-wrapper feature-badge-icon-wrapper--overrides"><div class="feature-badge-icon"><i class="icon-dku-override"></i></div></div>`
});

app.component('customMetricFailurePopup', {
    bindings: {
        error: '<',
        fmi: '<'
    },
    templateUrl: 'templates/ml/prediction-model/fragments/custom_metric_failed_button.html',
    controller: function($scope, $state, $stateParams, FullModelLikeIdUtils) {
        $scope.$state = $state;
        const hasLinks = () => this.fmi && !$stateParams.evaluationId && !$stateParams.dashboardId && !$stateParams.insightId;
        $scope.hasDesignLink = () => hasLinks() && FullModelLikeIdUtils.isAnalysis(this.fmi);
        $scope.hasLogLink = () => hasLinks() && (FullModelLikeIdUtils.isAnalysis(this.fmi) || FullModelLikeIdUtils.isSavedModel(this.fmi));
    }
});

app.directive("diagnosticsModal", function() {
    return {
        scope: { diagnostics: '=', textContent: "@", popupContent: "@", iconWithoutBorder: "@?" },
        templateUrl: '/templates/ml/diagnostics-modal.html',
        link: ($scope, element, attrs) => {
            $scope.maxLength = attrs.maxLength || 120; // Number of characters per message
            $scope.maxDiagnostics = 5; // Number of diagnostics to display in the popup

            const firstNDiagnostics = (firstN) => {
                if (!$scope.diagnostics) {
                    return null;
                }

                const diagnostics = {};
                let total = 0;
                for (const diagnostic of $scope.diagnostics) {
                    if (attrs.state && attrs.state !== diagnostic.step) { // Skip if only display for a single state
                        continue;
                    }

                    diagnostics[diagnostic.displayableType] = diagnostics[diagnostic.displayableType] || [];
                    diagnostics[diagnostic.displayableType].push(diagnostic.message);
                    total++;

                    if (total >= firstN) { // Reached max numbers of diagnostics to display
                        return diagnostics;
                    }
                }
                return diagnostics;
            }

            $scope.$watch("diagnostics", () => {
                $scope.filteredDiagnostics = firstNDiagnostics($scope.maxDiagnostics);
            }, true);
        }
    };
});

app.component('modelName', {
    template: `
        <span title="{{ $ctrl.name }}">
            {{ $ctrl.nameStart }} <span class="text-prompt">{{ $ctrl.nameEnd}}</span>
        </span>
    `,
    controller: function() {
        const $ctrl = this;

        $ctrl.$onChanges = function(changesObj) {
            if (changesObj.name && changesObj.name.currentValue) {
                [, $ctrl.nameStart, $ctrl.nameEnd] = changesObj.name.currentValue.match(/^(.+)\s+(\([^()]+\))$/) || [null, $ctrl.name, ''];
            }
        }
    },
    bindings: {
        name: '<'
    }
});


app.controller('EnsembleModalController', function($scope){
    $scope.params = {};
    $scope.getMethod = function(){ return $scope.params.method; };
});


app.controller("MLDiagnosticsSettingsController", function($scope, CachedAPICalls, Logger, $translate) {

    $scope.addNewDiagnosticSettings = function(diagnosticSettings) {
        const settingsTypeSet = new Set();
        diagnosticSettings.settings.forEach(setting => {
            settingsTypeSet.add(setting.type);
        });

        for (const diagnosticType of Object.keys($scope.diagnosticsDefinition)) {
            if (!settingsTypeSet.has(diagnosticType)) {
                settingsTypeSet.add(diagnosticType);
                const newSetting = {"type": diagnosticType, "enabled": true}; // new diagnostic, enable by default
                Logger.log("Adding new missing diagnostic type:" + diagnosticType);
                diagnosticSettings.settings.push(newSetting);
            }
        }
    }

    $scope.isBackendDiagsCompatible = function() {
        return $scope.isMLBackendType("PY_MEMORY") || $scope.isMLBackendType("KERAS") || $scope.isMLBackendType("DEEP_HUB");
    }

    $scope.isDiagnosticEnabled = function(diagnosticSettings, type) {
        const setting = $scope.getDiagnosticSetting(diagnosticSettings, type);
        return setting && setting.enabled;
    }

    $scope.isDiagnosticUnavailable = function(type) {
        return (!$scope.mlTaskDesign.diagnosticsSettings.enabled ||
        (["ML_DIAGNOSTICS_CAUSAL_TREATMENT_CHECKS", "ML_DIAGNOSTICS_CAUSAL_PROPENSITY_CHECKS"].includes(type) && !$scope.mlTaskDesign.modeling.propensityModeling.enabled));
    }

    $scope.getDiagnosticUnavailableMessage = function(type) {
        if (["ML_DIAGNOSTICS_CAUSAL_TREATMENT_CHECKS", "ML_DIAGNOSTICS_CAUSAL_PROPENSITY_CHECKS"].includes(type) && !$scope.mlTaskDesign.modeling.propensityModeling.enabled) {
            return "This diagnostic relies on a propensity model. You can enable it in the Treatment Analysis tab."
        }
        return;
    }

    $scope.getDiagnosticSetting = function(diagnosticSettings, type) {
        // $scope.mlTaskDesign.diagnosticsSettings.settings should have been an object with {type: enabled} but we wanted keys to be sorted
        // $scope.diagnosticsDefinition are actually sorted by default
        for (const setting of diagnosticSettings.settings) {
            if (type === setting.type) {
                return setting;
            }
        }
        return null;
    }
});

app.controller("_MLTaskDesignController", function($scope, $stateParams, DataikuAPI,
    CreateModalFromTemplate, Collections, CodeMirrorSettingService, AlgorithmsSettingsService, WT1) {

    DataikuAPI.analysis.listHeads($stateParams.projectKey).success(function(data) {
        $scope.analyses = data;
    });

    $scope.backendTypeNames = {
        "PY_MEMORY": "Python in-memory",
        "MLLIB": "MLLib",
        "H20": "H20",
        "VERTICA": "Vertica",
        "KERAS": "Keras"
    };

    $scope.displayTypes = {
        CLUSTERING: "Clustering",
        BINARY_CLASSIFICATION: "Classification",
        MULTICLASS: "Classification",
        REGRESSION: "Regression",
        CAUSAL_BINARY_CLASSIFICATION: "Causal Classification",
        CAUSAL_REGRESSION: "Causal Regression",
        TIMESERIES_FORECAST: "Forecasting",
        // Not needed for Deep hub; only used for copy algo settings
    };

    $scope.showAlgorithm = function(algo) {
        // additional condition for causal predictions: algo must support the selected learning method to be displayed in its algo list
        if ($scope.isCausalPrediction !== undefined && $scope.isCausalPrediction() && algo.supportedCausalMethod !== $scope.uiState.selectedCausalMethod) return false;

        return !algo.condition || algo.condition();
    };

    $scope.hasAlgoDisplayGroup = function(displayGroup) {
        return function(item) {
            return $scope.showAlgorithm(item) && item.displayGroups && item.displayGroups.includes(displayGroup);
        };
    };

    $scope.isAlgoWithoutDisplayGroup = function() {
        return function(item) {
            return $scope.showAlgorithm(item) && (!item.displayGroups || item.displayGroups.length === 0);
        };
    };

    $scope.getCustomAlgorithm = function(custom_id) {
        if (custom_id.startsWith('custom_python_')) {
            return $scope.mlTaskDesign.modeling.custom_python[custom_id.slice(14)];
        } else if (custom_id.startsWith('custom_mllib_')) {
            return $scope.mlTaskDesign.modeling.custom_mllib[custom_id.slice(13)];
        }
    };

    $scope.isPluginAlgorithm = function(alg) {
        return alg.algKey.startsWith("CustomPyPredAlgo_");
    }

    $scope.getPluginAlgorithm = function(algKey) {
        return $scope.algorithms["PY_MEMORY"].find(alg => alg.algKey === algKey);
    }

    $scope.getAlgorithmModeling = function(algKey) {
        const alg = Collections.indexByField($scope.algorithms[$scope.mlTaskDesign.backendType], 'algKey')[algKey];
        if (!alg) {
            throw new Error("Algorithm not found: " + algKey);
        } else if (alg.isCustom) {
            return AlgorithmsSettingsService.getCustomAlgorithmSettings($scope.mlTaskDesign, algKey);
        } else if ($scope.isPluginAlgorithm(alg)) {
            return AlgorithmsSettingsService.getPluginAlgorithmSettings($scope.mlTaskDesign, alg.algKey);
        } else {
            return $scope.mlTaskDesign.modeling[alg.hpSpaceName || algKey];
        }
    };

    $scope.hasActiveMLTask = function(){
        return $scope.mlTasksContext && !!$scope.mlTasksContext.activeMLTask;
    };

    $scope.isBayesianSearchWithSkopt = function() {
        if (!$scope.mlTaskDesign
            || !$scope.mlTaskDesign.modeling
            || !$scope.mlTaskDesign.modeling.gridSearchParams) {
            return false;
        }
        const gridSearchParams = $scope.mlTaskDesign.modeling.gridSearchParams;
        return (gridSearchParams.strategy === "BAYESIAN" && gridSearchParams.bayesianOptimizer === "SCIKIT_OPTIMIZE");
    }



    $scope.removeCustomAlgorithm = function(custom_id) {
        var idx;
        if (custom_id.startsWith('custom_python_')) {
            idx = parseInt(custom_id.slice(14));
            $scope.mlTaskDesign.modeling.custom_python.splice(idx, 1);
        } else if (custom_id.startsWith('custom_mllib_')) {
            idx = parseInt(custom_id.slice(13));
            $scope.mlTaskDesign.modeling.custom_mllib.splice(idx, 1);
        }
        $scope.setAlgorithms($scope.mlTaskDesign);
        $scope.setSelectedAlgorithm(AlgorithmsSettingsService.getDefaultAlgorithm(
            $scope.mlTaskDesign,
            $scope.algorithms[$scope.mlTaskDesign.backendType]
        ));
    };

    // TODO: use a dedicated controller for the modal
    $scope.copyFeaturesHandling = function(exportSettings) {
        if ($scope.dirtySettings()) {
            $scope.saveSettings();
        }
        DataikuAPI.projects.listHeads(exportSettings ? 'WRITE_CONF' : null).success(function(projectData) {
            CreateModalFromTemplate("/templates/analysis/mlcommon/settings/copy-settings.html", $scope, null, function(newScope) {
                newScope.title = "Copy features handling " + (exportSettings ? "to" : "from");
                newScope.totem = "icon-" + (exportSettings ? "copy" : "paste");
                newScope.infoMessages = ["Features handling will be copied based on their names"];
                if ($scope.mlTaskDesign.taskType === "PREDICTION") {
                    newScope.infoMessages.push(`Pasting features handling on the ${exportSettings ? "selected" : "current"}
                        model will not change the features handling of its ${$scope.isCausalPrediction() ? "outcome" : "target"}
                        ${$scope.isSampleWeightEnabled() ? " nor of its sample weight" : ""}`
                    );
                }
                newScope.projects = projectData;
                newScope.selectProject = function() {
                    DataikuAPI.analysis.listHeads(newScope.selectedProjectKey).success(function(analysisData) {
                        newScope.analyses = analysisData;
                        newScope.selectedAnalysisId = undefined;
                        newScope.selectedTask = undefined;
                    }).error(setErrorInScope.bind($scope));
                };
                newScope.selectAnalysis = function () {
                    DataikuAPI.analysis.listMLTasks(newScope.selectedProjectKey, newScope.selectedAnalysisId)
                    .success(function(taskData) {
                        newScope.tasks = taskData;
                        newScope.descriptions = [];
                        newScope.tasks.forEach(task => {
                            // task can be selected if it is not the current one
                            task.isNotSelectable = task.mlTaskId === $stateParams.mlTaskId
                                            && newScope.selectedAnalysisId === $stateParams.analysisId
                                            && newScope.selectedProjectKey === $stateParams.projectKey;
                            newScope.descriptions.push($scope.displayTypes[task.predictionType || task.taskType] + " ("
                            + ($scope.backendTypeNames[task.backendType] || $scope.mlTaskDesign.backendType)
                            + ")");
                        });
                        newScope.selectedTask = undefined;
                    }).error(setErrorInScope.bind($scope));
                };
                if (newScope.projects.some(_ => _.projectKey === $stateParams.projectKey)) {
                    newScope.selectedProjectKey = $stateParams.projectKey;
                    newScope.analyses = $scope.analyses;
                    newScope.selectedAnalysisId = $stateParams.analysisId;
                    newScope.selectAnalysis();
                }
                newScope.confirm = function() {
                    const currentIds = [
                       $stateParams.projectKey, $stateParams.analysisId, $stateParams.mlTaskId
                    ];

                    const selectedIds = [
                        newScope.selectedProjectKey, newScope.selectedAnalysisId, newScope.selectedTask.mlTaskId
                    ];

                    const originIds = exportSettings ? currentIds : selectedIds;
                    const destinationIds = exportSettings ? selectedIds : currentIds;

                    DataikuAPI.analysis.mlcommon.copyFeatureSettings(...originIds, ...destinationIds).success(function(data) {
                        if (!exportSettings) {
                            // Keep existing order of features
                            for (let featureName in $scope.mlTaskDesign.preprocessing.per_feature) {
                                Object.assign($scope.mlTaskDesign.preprocessing.per_feature[featureName], data.preprocessing.per_feature[featureName]);
                            }
                        }

                        newScope.dismiss();
                    }).error(setErrorInScope.bind($scope));

                    WT1.event("mltask-copy-features-handling", {
                        export: exportSettings,
                        sameProject: $stateParams.projectKey === newScope.selectedProjectKey,
                        sameAnalysis: $stateParams.analysisId === newScope.selectedAnalysisId,
                        typeDest: newScope.selectedTask.taskType === "CLUSTERING" ? "CLUSTERING" : newScope.selectedTask.predictionType,
                        typeSrc: $scope.mlTaskDesign.taskType === "CLUSTERING" ? "CLUSTERING" : $scope.mlTaskDesign.predictionType
                    });
                };
                newScope.cancel = function() {
                    newScope.dismiss();
                };
            });
        }).error(setErrorInScope.bind($scope));
    };
});


/**
 * Injected into all controllers that display a single ML task.
 * It handles:
 *   - the global nav handle to switch between ML tasks
 *   - setting the top nav
 */
app.controller("_MLTaskBaseController", function($scope, $state, $filter, Collections, DataikuAPI, TopNav, $stateParams,
    $location, CreateModalFromTemplate, Dialogs, ActivityIndicator, Fn, $q, Throttle, MLTasksNavService, $rootScope, $timeout, CAUSAL_META_LEARNERS,
    algorithmsPalette, COLOR_PALETTES, gradientGenerator, DatasetUtils, WT1, PartitionedModelsService, ModelLabelUtils, CustomMetricIDService, MLModelsUIRouterStates,
    AlgorithmsSettingsService, CodeMirrorSettingService, CachedAPICalls) {

    TopNav.setLocation(TopNav.TOP_ANALYSES, TopNav.LEFT_ANALYSES, TopNav.TABS_ANALYSIS, "models");
    TopNav.setItem(TopNav.ITEM_ANALYSIS, $stateParams.analysisId);

    $scope.selection = {
        partialProperty: 'sessionId'
    };
    $scope.sessionInfo = {};
    $scope.hooks = {};
    $scope.uiState = {
        currentMetric: '',
        currentMetricIsCustom: false
    };

    $scope.setAlgorithms = function(mlTaskDesign) {
        $scope.algorithms = angular.copy($scope.base_algorithms);
        const isPythonBackend = mlTaskDesign.backendType === "PY_MEMORY";
        const customAlgos = isPythonBackend ? mlTaskDesign.modeling.custom_python : mlTaskDesign.modeling.custom_mllib;
        $scope.editorOptionsCustom = isPythonBackend ? CodeMirrorSettingService.get("text/x-python") : CodeMirrorSettingService.get("text/x-scala");
        AlgorithmsSettingsService.addCustomAlgorithmsToBaseAlgorithms(
            $scope.algorithms[mlTaskDesign.backendType],
            customAlgos,
            isPythonBackend);
    }

    $scope.setSelectedAlgorithm = function(algorithm) {
        $scope.uiState.algorithm = algorithm;
        $scope.uiState.scrollToMeAlgorithm = $scope.uiState.algorithm;
    };

    $scope.isMLBackendType = function(mlBackendType) {
        if ($scope.mlTasksContext && $scope.mlTasksContext.activeMLTask) {
           return $scope.mlTasksContext.activeMLTask.backendType === mlBackendType;
        }
    };

    $scope.backendIsPythonBased = function() {
        return $scope.isMLBackendType('KERAS') || $scope.isMLBackendType('PY_MEMORY');
    };

    $scope.isPythonClassicalMl = function() {
        const isPythonBackend = $scope.isMLBackendType("PY_MEMORY");
        if (!$scope.mlTaskDesign) return isPythonBackend;
        return isPythonBackend
            && ($scope.mlTaskDesign.taskType === 'CLUSTERING' || $scope.isClassicalPrediction());
    }

    $scope.listMLTasks = function() {
        return DataikuAPI.analysis.listMLTasks($stateParams.projectKey, $stateParams.analysisId).success(function(data){
            $scope.mlTasksContext.type = "mltasks";
            $scope.mlTasksContext.analysisMLTasks = data;
            $scope.mlTasksContext.activeMLTask = null;
            for (var i in data) {
                if (data[i].mlTaskId == $stateParams.mlTaskId) {
                    $scope.mlTasksContext.activeMLTask = data[i];
                    break;
                }
            }
        });
    }

    $scope.createNewMLTask = function() {
        CreateModalFromTemplate("/templates/analysis/new-mltask-modal.html", $scope, "AnalysisNewMLTaskController");
    };

    $scope.renameMLTask = function() {
        DataikuAPI.analysis.mlcommon.getCurrentSettings($stateParams.projectKey, $stateParams.analysisId, $stateParams.mlTaskId)
        .success(function(mlTaskDesign){
            Dialogs.prompt($scope, "Rename modeling task", "Rename modeling task", mlTaskDesign.name).then(function(newName) {
                var fn;
                if (mlTaskDesign.taskType == "PREDICTION") {
                    fn = DataikuAPI.analysis.pml.saveSettings;
                } else if (mlTaskDesign.taskType == "CLUSTERING") {
                    fn = DataikuAPI.analysis.cml.saveSettings;
                } else {
                    throw "Unknown mlTaskDesign Type"
                }
                mlTaskDesign.name = newName;
                fn($stateParams.projectKey, $stateParams.analysisId, mlTaskDesign).success(function(data){
                    $state.go("projects.project.analyses.analysis.ml.list");
                });
            });
        }).error(setErrorInScope.bind($scope));
    };

    $scope.deleteMLTask = function() {
        Dialogs.confirm($scope, "Delete modeling task", "Do you want to delete this modeling task ?").then(function(data){
            DataikuAPI.analysis.mlcommon.deleteMLTask($stateParams.projectKey, $stateParams.analysisId, $stateParams.mlTaskId).success(function(data) {
                $state.go("projects.project.analyses.analysis.ml.list");
            }).error(setErrorInScope.bind($scope));
        });
    };

    $scope.duplicateMLTask = function() {
        const DEFAULT_ANALYSIS = {id: "new", name: "Create a new analysis…"};
        if ($scope.dirtySettings()) {
            $scope.saveSettings();
        }
        DataikuAPI.projects.listHeads('WRITE_CONF').success(function(writableProjects) {
            if (writableProjects.length == 0) {
                Dialogs.error($scope, "No writable project", "You don't have write access to any project, can't duplicate model.");
                return;
            }
            const currentProjectWritable = writableProjects.some(_ => _.projectKey === $stateParams.projectKey);
            CreateModalFromTemplate("/templates/analysis/mlcommon/duplicate-mltask.html", $scope, null, function(newScope) {
                newScope.totem = "icon-machine_learning_" +
                                ($scope.mlTaskDesign.taskType === "CLUSTERING" ? "clustering" : "regression");
                newScope.projects = writableProjects;
                newScope.selectedProject = currentProjectWritable ? $stateParams.projectKey : writableProjects[0].projectKey;

                newScope.$watch('selectedProject', function(project) {
                    if (!project) return;
                    DatasetUtils.listDatasetsUsabilityForAny(project).success(function(datasets) {
                        newScope.availableDatasets = datasets;
                        newScope.selectedDataset = project == $stateParams.projectKey ?
                                $scope.analysisCoreParams.inputDatasetSmartName : undefined;
                    }).error(setErrorInScope.bind(newScope));
                });
                newScope.$watch('selectedDataset', function(dataset) {
                    newScope.analyses = undefined;
                    if (!dataset) return;
                    const selectedDataset = newScope.availableDatasets.find(_ => _.smartName === dataset);

                    // Use selectedProject instead of contextProject because we do not need to check exposed objects
                    DataikuAPI.datasets.get(newScope.selectedProject, selectedDataset.name, newScope.selectedProject)
                        .then(({data}) => newScope.columnNames = data.schema.columns.map(_ => _.name))
                        .then(() => DataikuAPI.analysis.listOnDataset(newScope.selectedProject, dataset))
                        .then(({data}) => {
                            const analyses = data;
                            analyses.unshift(Object.assign({newName: "Analyze " + newScope.selectedDataset}, DEFAULT_ANALYSIS));
                            newScope.analyses = analyses;
                            newScope.selectedAnalysis = analyses[0];
                            if (newScope.selectedProject == $stateParams.projectKey
                                    && newScope.selectedDataset == $scope.analysisCoreParams.inputDatasetSmartName) {
                                newScope.selectedAnalysis = analyses.find(_ => _.id === $scope.analysisCoreParams.id);
                            }
                        })
                        .catch(setErrorInScope.bind(newScope));
                });

                // checks whether the target is in the feature columns of the dataset or not;
                // if not, a dropdown is displayed with the actual features so the user can pick a new target
                // the check is only done if the selected analysis has no step or is new,
                // else the target could have been created by the script => checked in backend
                function checkDatasetContainsTarget() {
                    newScope.features = {};
                    if ($scope.mlTaskDesign.taskType == "CLUSTERING"
                            || ! newScope.selectedAnalysis
                            || newScope.selectedAnalysis.id === $stateParams.analysisId) {
                        return;
                    }
                    if (newScope.selectedAnalysis.id == "new" || newScope.selectedAnalysis.nbSteps === 0) {
                        if (! newScope.columnNames.includes($scope.mlTaskDesign.targetVariable)) {
                            newScope.features.available = newScope.columnNames;
                        }
                    }
                }
                newScope.$watch('selectedAnalysis', checkDatasetContainsTarget);

                function duplicate() {
                    if ($scope.mlTaskDesign.taskType == "PREDICTION") {
                        DataikuAPI.analysis.pml.duplicate(
                            $stateParams.projectKey, $stateParams.analysisId, $stateParams.mlTaskId,
                            newScope.selectedProject, newScope.selectedAnalysis.id, newScope.features.selected
                        ).success(function(result) {
                            if (result.success) {
                                $state.go("projects.project.analyses.analysis.ml.predmltask.list.design", {
                                    projectKey: newScope.selectedProject,
                                    analysisId: newScope.selectedAnalysis.id,
                                    mlTaskId: result.newMlTaskId.id
                                });
                            } else {
                                newScope.features.available = result.possibleTargets;
                            }
                        }).error(setErrorInScope.bind(newScope));
                    } else {
                        DataikuAPI.analysis.cml.duplicate($stateParams.projectKey, $stateParams.analysisId,
                            $stateParams.mlTaskId, newScope.selectedProject, newScope.selectedAnalysis.id
                        ).success(function(data) {
                                $state.go("projects.project.analyses.analysis.ml.clustmltask.list.design", {
                                    projectKey: newScope.selectedProject,
                                    analysisId: newScope.selectedAnalysis.id,
                                    mlTaskId: data.id
                                });
                        }).error(setErrorInScope.bind(newScope));
                    }
                }

                newScope.confirm = function () {
                    if (newScope.selectedAnalysis.id == "new") {
                        DataikuAPI.analysis.create(newScope.selectedProject, newScope.selectedDataset,
                            newScope.selectedAnalysis.newName).success(function (data) {
                                newScope.selectedAnalysis.id = data.id;
                                duplicate();
                        }).error(setErrorInScope.bind(newScope));
                    } else {
                        duplicate();
                    }

                    WT1.event("mltask-duplicate", {
                        sameProject: $stateParams.projectKey == newScope.selectedProject,
                        sameDataset: $scope.analysisCoreParams.inputDatasetSmartName == newScope.selectedDataset,
                        sameAnalysis: $stateParams.analysisId == newScope.selectedAnalysis.id,
                        taskType: $scope.mlTaskDesign.taskType == "CLUSTERING" ? "CLUSTERING" : $scope.mlTaskDesign.predictionType,
                    });
                };
            });
        }).error(setErrorInScope.bind($scope));
    };

    $scope.mlTaskFeatures = function(features, roles) {
        if (!features) return 0;
        return $.map(features, function(v,k){
            v._name = k;
            return v;
        }).filter(f => !roles || roles && roles.includes(f.role));
    };

    $scope.getEnabledModels = function(models) {
        var enabledModels = [];
        for (var name in models) {
            if (name === 'custom_mllib' || name === 'custom_python') {
                const enabledCustomModels = models[name].filter(m => m.enabled);
                enabledModels = enabledModels.concat(enabledCustomModels);
            } else if (name.startsWith("plugin_python")) {
                const enabledPluginModels = Object.values(models[name]).filter(m => m.enabled);
                enabledModels = enabledModels.concat(enabledPluginModels);
            } else if (name === 'propensityModeling') {
                // enabled by default in PredictionModelingParams but should not be considered as a model on its own
                continue;
            } else if (models[name].enabled) {
                enabledModels.push(models[name]);
            }
        }
        return enabledModels;
    };

    $scope.dirtySettings = function() {
        return !angular.equals($scope.savedSettings, dkuDeepCopy($scope.mlTaskDesign, $scope.SettingsService.noDollarKey));
    };

    const getEvaluationMetricId = function(modelingMetrics) {
        if (modelingMetrics.evaluationMetric === "CUSTOM") {
            return CustomMetricIDService.getCustomMetricId(modelingMetrics.customEvaluationMetricName);
        } else {
            return modelingMetrics.evaluationMetric;
        }
    }

    $scope.isSparkBased = function(){
        return $scope.mlTaskDesign.backendType == 'MLLIB' || $scope.mlTaskDesign.backendType == 'H2O';
    };

    $scope.isStandardBuiltinMetric = function(metric) {
        return !$scope.isSparkBased() || metric[0] !== "CUSTOM";
    }

    $scope.hasFeatureReductionIncompatibleWithSparse = function() {
        return ['ICA', 'PCA'].includes($scope?.mlTaskDesign?.preprocessing?.feature_selection_params?.method);
    }

    $scope.setMlTaskDesign = function(mlTaskDesign) {
        $scope.mlTaskDesign = mlTaskDesign;
        $scope.uiState.preprocessingPerFeature = $scope.mlTaskDesign.preprocessing && $scope.mlTaskDesign.preprocessing.per_feature;
        $scope.uiState.evaluationMetricId = getEvaluationMetricId(mlTaskDesign.modeling.metrics);
        $scope.selectCustomAlgo();
        $scope.retrieveCodeEnvsInfo();
        if (mlTaskDesign.taskType === 'PREDICTION') {
            $scope.uiState.targetVariable = mlTaskDesign.targetVariable;
            $scope.uiState.splitMethodDesc = (mlTaskDesign.splitParams && mlTaskDesign.splitParams.ssdSplitMode) === "SORTED" ? "Based on time variable" : "Randomly";

            CachedAPICalls.pmlDiagnosticsDefinition.then(pmlDiagnosticsDefinition => {
                $scope.diagnosticsDefinition = pmlDiagnosticsDefinition(mlTaskDesign.backendType, mlTaskDesign.predictionType);
            });
        } else if (mlTaskDesign.taskType === 'CLUSTERING') {
            CachedAPICalls.cmlDiagnosticsDefinition.then(cmlDiagnosticsDefinition => {
                $scope.diagnosticsDefinition = cmlDiagnosticsDefinition(mlTaskDesign.backendType);
            });
        }
    };

    $scope.retrieveCodeEnvsInfo = function() {
        if ($scope.appConfig.isAutomation) {
            return;
        }
        // On Design node, listing all available envs for this MLtask, along with compatibility to run ML, for
        // further use:
        // * in "Feature handling" to make sure the runtime env is compatible with the selected preprocessing. 
        // * in "Algorithms" tab to make sure the env is compatible with the selected algorithms
        // * in "Hyperparameters" tab to make sure the env is compatible with bayesian search (prediction)
        // * in "Runtime environment" to make sure a proper env is used for selected algos / preprocessings / HP search
        if ($scope.isMLBackendType("PY_MEMORY") || $scope.isMLBackendType("KERAS")) {
            DataikuAPI.codeenvs.listWithVisualMlPackages($stateParams.projectKey).success(function(data) {
                $scope.codeEnvsCompat = data;
            }).error(setErrorInScope.bind($scope));
        }
        // TODO @deepHub list envs with deep hub packages
    };

    $scope.selectCustomAlgo = function() {
        if ($scope.mlTaskDesign.guessPolicy !== 'CUSTOM') return;

        if ($scope.mlTaskDesign.backendType === 'PY_MEMORY' && $scope.mlTaskDesign.modeling.custom_python.length > 0) {
            const expectedAlgKeyPython = 'custom_python_' + ($scope.mlTaskDesign.modeling.custom_python.length - 1);
            $scope.uiState.algorithm = expectedAlgKeyPython;
        }

        if ($scope.mlTaskDesign.backendType === 'MLLIB' && $scope.mlTaskDesign.modeling.custom_mllib.length > 0) {
            const expectedAlgKeyMllib = 'custom_mllib_' + ($scope.mlTaskDesign.modeling.custom_mllib.length - 1);
            $scope.uiState.algorithm = expectedAlgKeyMllib;
        }
    }

    $scope.beforeUpdateSettingsCallback = function(settings) {
        // Do nothing. Will be overriden in PMLTaskBaseController
    };

    $scope.updateSettings = function(settings) {
        $scope.beforeUpdateSettingsCallback(settings);
        $scope.setMlTaskDesign(settings);
        $scope.saveSettings();
    };

    $scope.revertScriptToSession = function (projectKey, analysisId, mlTaskDesignId, sessionId) {
        return DataikuAPI.analysis.mlcommon.revertScriptToSession(projectKey, analysisId, mlTaskDesignId, sessionId)
        .then(function(response) {
            return response.data;
        }, setErrorInScope.bind($scope));
     };

    $scope.revertDesignToSession = function(sessionId) {
        $scope.MLAPI.getSessionTask($stateParams.projectKey, $stateParams.analysisId, $stateParams.mlTaskId, sessionId).success(function(sessionDesign){
            CreateModalFromTemplate("/templates/analysis/mlcommon/dump-session-design-modal.html", $scope, null, function(newScope) {
                newScope.sessionId = sessionId.slice(1);
                newScope.sessionDesign = sessionDesign;
                newScope.algorithms = {};
                newScope.selectAlgorithms = false;
                const algByKey = Collections.indexByField($scope.base_algorithms[$scope.mlTaskDesign.backendType], 'algKey');
                // for some reason XGBoost is referred to by its algKey "xgboost_regression" / "xgboost_classification" in base_algorithms
                // but its key is xgboost in mlTask TODO: set this straight
                algByKey["xgboost"] = { name: "XGBoost" };
                angular.forEach(sessionDesign.modeling, function(v,k) {
                    if (v.enabled && algByKey[k]) {
                        newScope.algorithms[k] = {
                            enabled: true,
                            name: algByKey[k].name
                        };
                    }
                });
                angular.forEach(sessionDesign.modeling.custom_mllib, function(v,k) {
                    if (v.enabled) {
                        newScope.algorithms["custom_mllib_"+k] = {
                            enabled: true,
                            name: "Custom MLLIB algorithm #" + k,
                        };
                    }
                });
                angular.forEach(sessionDesign.modeling.custom_python, function(v,k) {
                    if (v.enabled) {
                        newScope.algorithms["custom_python_"+k] = {
                            enabled: true,
                            name: v.name !== "Custom Python model" ? v.name : "Custom python algorithm #" + k,
                        };
                    }
                });
                newScope.noEnabledAlgorithms = function() {
                    return $.map(newScope.algorithms, function(v,k) {return !v.enabled}).reduce(Fn.AND,true);
                }
                newScope.confirm = function() {
                    $scope.revertScriptToSession(newScope.projectSummary.projectKey, newScope.analysisId, newScope.mlTaskDesign.id, sessionId).then(function(scriptFile) {
                        $scope.analysisCoreParams.script = scriptFile;
                        if (newScope.selectAlgorithms) {
                            angular.forEach(newScope.algorithms, function(v,k) {
                                if (k.startsWith("custom_mllib_")) {
                                    newScope.sessionDesign.modeling.custom_mllib[parseInt(k.slice(13))].enabled = v.enabled;
                                } else if (k.startsWith("custom_python_")) {
                                    newScope.sessionDesign.modeling.custom_python[parseInt(k.slice(14))].enabled = v.enabled;
                                } else {
                                    newScope.sessionDesign.modeling[k].enabled = v.enabled;
                                }
                            });
                        }
                        $scope.updateSettings(newScope.sessionDesign);
                        $state.go('^.^.design');
                        newScope.dismiss();
                    });
                }
            });
        });
    }

    $scope.removeTaskStatus = function() {
        $scope.mlTaskStatus = null;
    }

    $scope.trainDirectly = function() {
        $scope.touchMlTask();
        // Remove the mltaskstatus to prevent the fugitive "no model trained"
        $scope.mlTaskStatus = null;
        $scope.MLAPI.trainStart($stateParams.projectKey, $stateParams.analysisId, $stateParams.mlTaskId, null, null, null, true)
            .success(function(){
                $scope.initialRefreshAndAutoRefresh();
            }).error(setErrorInScope.bind($scope));
    };

    function updateSessionModels(){
        if ($scope.selection && $scope.selection.allObjects && $scope.sessionTask && $scope.sessionTask.sessionId) {
            $scope.selection.sessionModels = $scope.selection.allObjects.filter(function(x){
                return x.sessionId == $scope.sessionTask.sessionId;
            });
        }
    };

    $scope.getSessionName = function(sessionId, sessionType) {
        let name = 'Session ' + $filter('onlyNumbers')(sessionId);

        if (sessionType === 'QUEUED') {
            const queueInfo = $scope.queueMap[`QUEUED-${sessionId}`];

            if (queueInfo && queueInfo.userMeta && queueInfo.userMeta.name) {
                name = `${queueInfo.userMeta.name} (${sessionId})` ;
            }
        } else if($scope.modelSnippets){
            const sessionModel = Object.values($scope.modelSnippets || [])
                    .find(ms => ms.sessionId === sessionId && ms.userMeta && ms.userMeta.sessionName);
            if(sessionModel){
                name = sessionModel.userMeta.sessionName;
            }
        }
        return name;
    };

    $scope.$watch("selection.allObjects", updateSessionModels, true);
    $scope.$watch("sessionTask", updateSessionModels, true);

    $scope.abortTraining = function(){
        CreateModalFromTemplate("/templates/analysis/mlcommon/abort-train-modal.html", $scope, null, function(newScope) {
            newScope.uiState = {
                pauseQueue: false
            };
            newScope.nextSession = $scope.mlTaskStatus.queueStatus === 'RUNNING' && $scope.queuedSessionIds ? $scope.queuedSessionIds[0] : null;
            newScope.confirm = function() {
                var toAbort = $scope.selection.allObjects
                .filter(function(o){ return o.trainInfo.state === 'PENDING' || o.trainInfo.state === 'RUNNING' });
                DataikuAPI.analysis.mlcommon
                    .trainAbort($stateParams.projectKey, $stateParams.analysisId, $stateParams.mlTaskId, newScope.uiState.pauseQueue)
                    .success(function(){
                        ActivityIndicator.success("Abort requested");
                        toAbort.map(function(o){ o.trainInfo.$userRequestedState = "ABORTED" });
                        refreshStatusAndModelSnippets();
                    }).error(setErrorInScope.bind($scope));
                newScope.dismiss();

                WT1.event('mltask-abort', {
                    taskType: $scope.mlTaskDesign.taskType,
                    pauseQueue: newScope.uiState.pauseQueue,
                    isPartialAbort: false
                });
            }
            newScope.finalize = function() {
                var toAbort = $scope.selection.allObjects
                .filter(function(o){return o.trainInfo.state === 'RUNNING' || o.trainInfo.state === 'PENDING'});
                DataikuAPI.analysis.mlcommon.stopGridSearchSession($scope.analysisCoreParams.projectKey,
                $scope.analysisCoreParams.id, $scope.sessionTask.id, $scope.sessionTask.sessionId)
                    .success(function(data){
                        toAbort.map(function(o){ o.trainInfo.$userRequestedState = "FINALIZE" });
                        $scope.refreshStatus();
                    }).error(setErrorInScope.bind($scope));
                newScope.dismiss();
            }
        });
    };

    // init & refresh

    // Refreshing mltaskStatus (general refresh)

    $scope.selectRunningOrFirstSession = function() {
        if ($scope.mlTaskStatus.fullModelIds.length > 0) {
            var sids = $scope.mlTaskStatus.fullModelIds.filter(function(o){ return o.training });
            if (sids.length === 0) { sids = $scope.mlTaskStatus.fullModelIds }
            sids = sids.map(Fn.propStr('fullModelId.sessionId'))
               .map(function(sid){ return parseInt(sid.slice(1)) }).sort(function(a,b){ return b-a });
            $scope.getSessionTaskIfChanged("s" + sids[0], true);
        }
    }

    $scope.refreshStatus = function() {
        return $scope.MLAPI.getTaskStatus($stateParams.projectKey, $stateParams.analysisId, $stateParams.mlTaskId)
            .success(function(data){
                $scope.mlTaskStatus = data;
                $scope.$broadcast("mlTaskStatusRefresh");
            })
            .error(setErrorInScope.bind($scope));
    };

    let recentlyTrainingModelIds = [];
    const refreshStatusAndModelSnippets = function(){
        return $scope.refreshStatus().then(function(){
            // since multiple sessions can run one after the other (queue), make
            // sure we get the updated snippet for the recently finished model
            const modelIdsCurrentlyTraining = $scope.mlTaskStatus.fullModelIds
                .filter(function(o){ return o.training })
                .map(function(o){ return o.id });
            // get modelIds that recently finished
            const noLongerTraining = recentlyTrainingModelIds.filter(id => !modelIdsCurrentlyTraining.includes(id));
            recentlyTrainingModelIds = modelIdsCurrentlyTraining;

            getModelSnippets(modelIdsCurrentlyTraining.concat(noLongerTraining));

            if (!$scope.mlTaskStatus.training && !$scope.mlTaskStatus.guessing) {
                $scope.listQueuedSessions();
                if ($scope.modelSnippets) {
                    // If the snippets have changed with respect to the latest `mlTaskStatus`, refresh remaining models one last time
                    getModelSnippets(Object.values($scope.modelSnippets)
                        .filter(function(o){ return o.trainInfo.state === 'RUNNING' || o.trainInfo.state === 'PENDING' })
                        .map(Fn.prop('fullModelId'))).then(() => {
                            setPossibleCustomMetrics();
                            $scope.$broadcast('snippetUpdate');
                        });;
                }
            }
        });
    }

    // Poll the backend to refresh the mlTaskStatus (in training or guessing)
    // Retrieve snippets that are currently training for this session
    // After guessing (or if already guessed), resolve the promise to continue more init tasks
    $scope.initialRefreshAndAutoRefresh = function() {
        const afterGuessingDeffered = $q.defer(),
            refreshStartDate = new Date(),
            refreshFirstDelay = 1000,
            refreshLastDelay = 15 * 1000,
            refreshGrowLength = 120 * 1000,
            throttle = Throttle().withScope($scope).withDelay(refreshFirstDelay);
        const autoRefresh = throttle.wrap(function() {
            refreshStatusAndModelSnippets().then(() => {

                if (!$scope.mlTaskStatus.guessing) {
                    afterGuessingDeffered.resolve();
                }

                if ($scope.mlTaskStatus.training || $scope.mlTaskStatus.guessing || $scope.mlTaskStatus.queueStatus === 'RUNNING') {
                    // Delay progressively grows from refreshFirstDelay to refreshLastDelay over time (refreshGrowLength)
                    // Scaling of the delay is to the power of 2, until we reach refreshLastDelay
                    const newDelay = refreshFirstDelay + Math.round( (refreshLastDelay-refreshFirstDelay)
                        * Math.min((new Date() - refreshStartDate) / refreshGrowLength)**2, 1);
                    throttle.withDelay(newDelay);
                    autoRefresh();
                }
            });
        });
        $scope.refreshStatus()
        .then($scope.selectRunningOrFirstSession)
        .then(autoRefresh);
        return afterGuessingDeffered.promise;
    };

    $scope.$watchCollection('modelSnippets', () => {
        setAllSnippets();
        $scope.listQueuedSessions();
    });
    $scope.$watchCollection('queuedSessionIds', setAllSnippets);

    function setAllSnippets() {
        if (!$scope.modelSnippets && $scope.queuedSessionIds) return; // ensure model snippets have loaded first

        const modelSnippets = $scope.modelSnippets || {};
        const queuedSnippets = ($scope.queuedSessionIds || [])
            .reduce((obj, sessionId) => ({ 
                ...obj, 
                [`QUEUED-${sessionId}`]: $scope.queueMap[`QUEUED-${sessionId}`] 
            }), {});

        // combine model and queued snippets together (and create fake model snippet for queued item)
        $scope.allModelSnippets = {
            ...modelSnippets,
            ...queuedSnippets
        };
        setPossibleCustomMetrics();
        setMetricScales();
    }

    // Refreshing sessionTasks (task info)
    const setPossibleMetrics = function() {
        $scope.possibleMetrics = $scope.FilteringService.getPossibleMetrics($scope.mlTaskStatus.headSessionTask);
        $scope.allMetrics = [...($scope.possibleMetrics || [])];
        $scope.allMetricsHooks = $scope.allMetrics.map((m) => m[0]);
    }

    const setUICurrentMetric = function(headSessionTask) {
        if (headSessionTask && headSessionTask.modeling && headSessionTask.modeling.metrics) {
            $scope.uiState.currentMetric = getEvaluationMetricId(headSessionTask.modeling.metrics);
        }
    }
    const setPossibleCustomMetrics = function() {
        if ($scope.allModelSnippets) {
            $scope.possibleCustomMetrics = $scope.FilteringService.getPossibleCustomMetrics($scope.allModelSnippets).map(item => {
                return [item.id, item]
            });
            if ($scope.possibleMetrics) {
                $scope.allMetrics = [...$scope.possibleMetrics];
            } else {
                $scope.allMetrics = [];
            }
            $scope.possibleCustomMetrics.forEach(customMetric => {
                $scope.allMetrics.push([customMetric[0], CustomMetricIDService.getCustomMetricName(customMetric[0])]);
            });
            $scope.allMetricsHooks = $scope.allMetrics.map((m) => m[0]);
            // due to queued loading in setAllSnippets(), this is required to ensure all custom metrics appear in table following page refresh
            // otherwise, table headers load before all custom metrics are known about
            $scope.$broadcast('redrawFatTable');
        }
        if (!$scope.uiState.currentMetric) {
            setUICurrentMetric($scope.mlTaskStatus.headSessionTask);
        }
    };

    const sessions = {};
    $scope.getSessionTaskIfChanged = function(newSessionId, dropCache) {
        if (!newSessionId || (!dropCache && $scope.sessionTask && $scope.sessionTask.sessionId === newSessionId)) return;
        if (!dropCache && sessions[newSessionId]) {
            $scope.sessionTask = sessions[newSessionId];
            $scope.setAdditionalSnippetParams && $scope.setAdditionalSnippetParams();
            return;
        }

        $scope.MLAPI.getSessionTask($stateParams.projectKey, $stateParams.analysisId, $stateParams.mlTaskId, newSessionId)
        .success(function(sessionTask) {
            $scope.sessionTask = sessionTask;
            $scope.sessionTask.sessionId = newSessionId;
            $scope.setAdditionalSnippetParams && $scope.setAdditionalSnippetParams();
            setPossibleMetrics();
            setPossibleCustomMetrics();
            if ($scope.uiState.evaluationMetricId === undefined) {
                $scope.uiState.evaluationMetricId = getEvaluationMetricId(sessionTask.modeling.metrics);
            }
            sessions[newSessionId] = $scope.sessionTask;
        })
        .error(setErrorInScope.bind($scope));
    }

    $scope.openDeleteModelAndSessionModal = function(fullModelIds, queuedSessionIds) {
        const hasModels = fullModelIds && fullModelIds.length;
        const hasQueuedSessions = queuedSessionIds && queuedSessionIds.length;
        let dialogTitle = 'Delete ';
        let dialogText = 'Are you sure you want to delete ';

        if (hasModels) {
            dialogTitle += fullModelIds.length + ' model' + (fullModelIds.length > 1 ? 's' : '');
            dialogText += fullModelIds.length > 1 ? 'these models' : 'this model';

            if (hasQueuedSessions) {
                dialogTitle += ' and ';
                dialogText += ' and ';
            }
        }

        if (hasQueuedSessions) {
            dialogTitle += queuedSessionIds.length + ' queued session' + (queuedSessionIds.length > 1 ? 's' : '');
            dialogText += queuedSessionIds.length > 1 ? 'these queued sessions' : 'this queued session';
        }

        dialogText += '?';

        Dialogs.confirm($scope, dialogTitle, dialogText).then(function() {
            if (hasModels) {
                DataikuAPI.ml.deleteModels(fullModelIds).success(function(){
                    fullModelIds.forEach(function(fmi){ delete $scope.modelSnippets[fmi] });
                    if (Object.values($scope.modelSnippets)
                        .filter(function(s){ return s.sessionId === $scope.sessionTask.sessionId }).length === 0) {
                        $scope.refreshStatus().then($scope.selectRunningOrFirstSession);
                    }
                });
            }

            if (hasQueuedSessions) {
                DataikuAPI.analysis.mlcommon.deleteQueuedSessions($stateParams.projectKey, $stateParams.analysisId, $stateParams.mlTaskId, queuedSessionIds)
                    .success(() => {
                        $scope.listQueuedSessions();
                    })
                    .error(setErrorInScope.bind($scope));
                WT1.event("mltask-delete-queued-session", {
                    taskType: $scope.mlTaskDesign.taskType
                });
            }
        });
    };

    // Queues
    $scope.queuedSessionIds = [];
    $scope.queueMap = {};

    $scope.listQueuedSessions = function() {
        DataikuAPI.analysis.mlcommon.listQueuedSessions($stateParams.projectKey, $stateParams.analysisId, $stateParams.mlTaskId)
            .success(function (queuedSessions) {
                $scope.queuedSessionIds = queuedSessions.map(session => session.id);
                queuedSessions.forEach(queuedSession => {
                    const id = queuedSession.id;
                    // use updateNoDeference so that items stay selected on session list modification
                    $scope.queueMap[`QUEUED-${id}`] = Collections.updateNoDereference($scope.queueMap[`QUEUED-${id}`], queuedSessionToSnippet(queuedSession));
                });
            })
            .error(setErrorInScope.bind($scope));
    };

    function queuedSessionToSnippet(session) {
        return {
            sessionId: session.id,
            trainInfo: {
                state: 'QUEUED'
            },
            userMeta: {
                starred: false,
                name: session.metadata.userSessionName,
                description: session.metadata.userSessionDescription
            }
        }
    }

    function pauseQueue() {
        DataikuAPI.analysis.mlcommon.pauseQueue($stateParams.projectKey, $stateParams.analysisId, $stateParams.mlTaskId)
            .error(setErrorInScope.bind($scope))
            .finally(() => {
                $scope.listQueuedSessions();
            });;
    };

    $scope.openPauseQueueDialog = function() {
        Dialogs.confirm($scope, 'Pause queue', 'Are you sure you want to pause the queue? The queue will be paused once the current session has finished training.').then(function() {
            pauseQueue();
        });
    };

    $scope.deleteQueuedSessions = function(sessionIds) {
        $scope.openDeleteModelAndSessionModal(null, sessionIds);
    };

    $scope.listQueuedSessions();

    // Refreshing snippets

    const setContainerUsageMetrics = function () {
        if (!$scope.mlTaskStatus || !$scope.modelSnippets || !$scope.mlTaskDesign || !$scope.mlTaskDesign.modeling || !$scope.mlTaskDesign.modeling.gridSearchParams || !$scope.mlTaskDesign.modeling.gridSearchParams.distributed) {
            return;
        }
        angular.forEach($scope.modelSnippets, function (snippet) {
            snippet.maxKubernetesContainers = $scope.mlTaskDesign.modeling.gridSearchParams.nContainers;
            if (snippet.partitionedModelEnabled) {
                snippet.maxKubernetesContainers *= PartitionedModelsService.getPartitionsSnippetStateSize(snippet, 'RUNNING');
            }
            snippet.containerUsageMetrics = $scope.mlTaskStatus.fullModelIds.filter((_) => _.id === snippet.fullModelId)[0].containerUsageMetrics;
        });
    };

    const setAlgorithmColors = function() {
        if (!($scope.mlTasksContext.activeMLTask.backendType in $scope.base_algorithms)){
            return;
        }
        const algList = $scope.base_algorithms[$scope.mlTasksContext.activeMLTask.backendType].filter(function(o){return !o.condition||o.condition()});

        if ($scope.mlTaskDesign && $scope.mlTaskDesign.taskType === 'CLUSTERING') {
            const actualAlgListKeys = new Set(Object.values($scope.modelSnippets).map((snippet) => `${snippet.algorithm}-${snippet.usesPCA}`));
            const colorByAlg = {};
            angular.forEach(Array.from(actualAlgListKeys).sort(), function(algKey, idx) {
                colorByAlg[algKey] = algorithmsPalette(idx);
            });
            let offsetOrderIdx = 0;
            Object.entries(Object.values($scope.modelSnippets).groupBy(snippet => `${snippet.algorithm}-${snippet.usesPCA}`)).map(([algorithmGroupKey, snippetsByKey], groupIdx) => {
                const algorithmColor = colorByAlg[algorithmGroupKey];
                const snippetsColors = gradientGenerator(algorithmColor, snippetsByKey.length);

                angular.forEach(snippetsByKey.sort((a, b) => a.nbClusters - b.nbClusters), function(snippet, k) {
                    snippet.color = snippetsColors[k];
                    snippet.algorithmOrder = offsetOrderIdx + k;
                });

                offsetOrderIdx += snippetsByKey.length;
            });
        } else {
            const algKeyList = algList.map(alg => alg.algEnumName || alg.algKey);
            let offset = 1;
            angular.forEach($scope.modelSnippets, function(snippet, k){
                let idx = -1;
                if (snippet.algorithm) { // Deep Hub does not have algorithm
                    idx = algKeyList.indexOf(snippet.algorithm.toLowerCase());
                }

                if (idx === -1) {
                    idx = algKeyList.length + offset;
                    offset++;
                }

                if (snippet.metaLearner) { // Causal only
                    idx += CAUSAL_META_LEARNERS.displayNames.indexOf(snippet.metaLearner);
                }

                if (snippet.predictionType === "TIMESERIES_FORECAST" && algList[idx]) { // Time-series only
                    let displayGroups = algList[idx].displayGroups;
                    if (displayGroups.includes("statistical")) {
                        // display on top
                    } else if (displayGroups.includes("deepLearning")) {
                        // "deepLearning" are right after "statistical" in algKeyList, nothing to do
                    } else if (displayGroups.includes("classicalML")) {
                        idx += 20*COLOR_PALETTES.algorithms.length + 4;
                    } else if (displayGroups.includes("baseline")) {
                        idx += 40*COLOR_PALETTES.algorithms.length + 3;
                    } else if (displayGroups.includes("legacy")) {
                        // "legacy" are right after "baseline" in algKeyList, keep the same offset
                        idx += 40*COLOR_PALETTES.algorithms.length + 3;
                    }
                }

                snippet.color = algorithmsPalette(idx);
                snippet.algorithmOrder = idx;
            });
        }
    }

    const setMainMetric = function() {
        if ( !$scope.mlTaskStatus || !$scope.modelSnippets || !$scope.mlTaskDesign || !$scope.mlTaskDesign.modeling || !$scope.uiState.currentMetric ) { return }
        $scope.uiState.currentMetricIsCustom = CustomMetricIDService.checkMetricIsCustom($scope.uiState.currentMetric);
        $scope.FilteringService.setMainMetric(
            Object.values($scope.modelSnippets),
            [],
            $scope.uiState.currentMetricIsCustom
                ? CustomMetricIDService.getCustomMetricName($scope.uiState.currentMetric)
                : $scope.uiState.currentMetric,
            $scope.mlTaskDesign.modeling.metrics.customMetrics,
            $scope.uiState.currentMetricIsCustom
        );
    }

    $scope.libMetric = function(metricName, customMetrics, customEvaluationMetricName="") {
        // If the metricName = CUSTOM, we need to check the customEvaluationMetricName
        return $scope.SettingsService.sort.lowerIsBetter(metricName, customMetrics, customEvaluationMetricName)
    }

    const setMetricScales = function() {
        if ($scope.sessionTask && $scope.sessionTask.modeling) {
            $scope.metricScales = {}
            $scope.possibleMetrics.map(Fn.prop(0)).forEach(function(metric) {
                const metrics = Object.values($scope.modelSnippets).map(Fn.prop(this.metricMap[metric]));
                $scope.metricScales[metric] = calculateMetricScales(metrics, $scope.libMetric(metric, []));
            }, $scope.FilteringService);

            $scope.possibleCustomMetrics.forEach(function(metricItem) {
                var metricId = metricItem[0], metric = metricItem[1];
                var greaterIsBetter = metric.greaterIsBetter;
                var metricName = CustomMetricIDService.getCustomMetricName(metricId);

                var metrics = Object.values($scope.modelSnippets).map(snippet => {
                    if (snippet.customMetricsResults) {
                        var filteredMetrics = snippet.customMetricsResults.filter(customMetricResult => customMetricResult.metric.name === metricName);
                        if (filteredMetrics[0]) {
                            return filteredMetrics[0].value;
                        }
                    }
                });

                $scope.metricScales[metricId] = calculateMetricScales(metrics, !greaterIsBetter)
            }, $scope.FilteringService)
        }
    }

    const calculateMetricScales = function(metrics, lowerIsBetter) {
        var min = d3.min(metrics), max = d3.max(metrics);
        return min === max ? Fn.cst('grey') :
                   d3.scale.linear().range(['red', 'orange', 'green'])
                       .domain([lowerIsBetter ? max : min, (max + min) / 2, lowerIsBetter ? min : max]);
    }

    const getModelSnippets = function(fullModelIds, getAll) { // getAll calls getModelSnippets with an empty list -> long call
        if (!$scope.modelSnippets) { $scope.modelSnippets = {} }
        if (!getAll && fullModelIds.length==0) { return $q.when(null) }
        return $scope.MLAPI.getModelSnippets($stateParams.projectKey, $stateParams.analysisId, $stateParams.mlTaskId, fullModelIds, getAll).then(function(response){
            angular.forEach(response.data, function(model, fmi) {
                $scope.modelSnippets[fmi] = Collections.updateNoDereference($scope.modelSnippets[fmi], model);
            });
            $scope.setAdditionalSnippetParams && $scope.setAdditionalSnippetParams();
            setAlgorithmColors();
            setMainMetric();
            setContainerUsageMetrics();
        }).catch(setErrorInScope.bind($scope));
    }

    // Init the MLTask by:
    // - Retrieving the Analysis
    // - List MLTask
    // - Init the mlTaskStatus and guess if the MLTask is being created
    // - Poll snippets of models being trained until training is over
    // - When guessing is done, then init MLTaskDesign
    // - Retrieve all snippets
    // - The promise is then passed to controllers in other parts of the page to do more init work (only called once)
    //
    // This part replaces several $scope.$watch() that used to do the init.
    // If there is a need to add a $scope.$watch() to init a variable once (in this or another controller), try using this promise (or another one)
    $scope.deferredAfterInitMlTaskDesign = DataikuAPI.analysis.getCore($stateParams.projectKey, $stateParams.analysisId)
    .then(({data}) => {
        $scope.analysisCoreParams = data;
        TopNav.setItem(TopNav.ITEM_ANALYSIS, $stateParams.analysisId, {name:data.name, dataset: data.inputDatasetSmartName});
        TopNav.setPageTitle(data.name + " - Analysis");
    })
    .then($scope.listMLTasks)
    .then($scope.initialRefreshAndAutoRefresh)
    .then($scope.initMlTaskDesign) // guess is done
    .then(() => {
        const compareFMIs = function(b,a) {
            const af = a.fullModelId, bf = b.fullModelId;
            if (af.sessionId !== bf.sessionId) return parseInt(af.sessionId.slice(1)) - parseInt(bf.sessionId.slice(1));
            if (af.preprocessingId !== bf.preprocessingId) return parseInt(af.preprocessingId.slice(2)) - parseInt(bf.preprocessingId.slice(2));
            return parseInt(af.modelId.slice(1)) - parseInt(bf.modelId.slice(1));
        }

        const headIds = $scope.mlTaskStatus.fullModelIds
            .sort(compareFMIs)
            .map(o => o.id).slice(0, 1);
        getModelSnippets([], true); // get all snippets (long call)
        getModelSnippets(headIds); // first quick call to display outline
    })
    .catch(setErrorInScope.bind($scope));

    $scope.$on("$destroy", $scope.clearMLTasksContext);

    MLTasksNavService.setMlTaskIdToGo($stateParams.analysisId, $stateParams.mlTaskId);
    checkChangesBeforeLeaving($scope, $scope.dirtySettings, null, MLModelsUIRouterStates.getAllowedTransitions());

    if ($rootScope.mlTaskJustCreated === true) {
        delete $rootScope.mlTaskJustCreated;
        $scope.mlTaskJustCreated = true;
        $scope.touchMlTask = function() { delete $scope.mlTaskJustCreated; };
    } else {
        $scope.touchMlTask = function(){
            // nothing to touch, not just created
        };
    }

    $scope.prepareGuessPolicies = function(policies) {
        policies.forEach(policy => {
            // Disabled every policy that does not support current backend type
            if (!policy.supported_backends.includes($scope.mlTaskDesign.backendType)) {
                policy.disabled = true;
            }

            // Inject current backend type in custom algorithms policy description
            if (policy.id === 'CUSTOM') {
                policy.description = `Train your own ${ $scope.mlTaskDesign.backendType === 'PY_MEMORY' ? 'Python' : 'Scala' } models.`;
            }
        });
        return policies;
    }
    
    $scope.switchGuessPolicy = function(policy) {

        if (policy.disabled) { return; }

        if ($scope.dirtySettings()) {
            $scope.saveSettings();
        }
        
        CreateModalFromTemplate("/templates/analysis/mlcommon/settings/change-algorithm-presets-modal.html", $scope, null, function(newScope) {
            newScope.taskType = $scope.mlTaskDesign.taskType.toLowerCase();
            newScope.confirm = function() {
                $scope.MLAPI.changeGuessPolicy($stateParams.projectKey, $stateParams.analysisId, $stateParams.mlTaskId, policy.id).then(function(response){
                        $scope.setMlTaskDesign(response.data);
                        $scope.saveSettings();
                        $scope.setAlgorithms($scope.mlTaskDesign);
                        $scope.setSelectedAlgorithm(AlgorithmsSettingsService.getDefaultAlgorithm($scope.mlTaskDesign, $scope.algorithms[$scope.mlTaskDesign.backendType]));
                }, setErrorInScope.bind($scope));
                newScope.dismiss();
            };
            newScope.cancel = function() {
                newScope.dismiss();
            };
        });
    };

    $scope.$watch("selection.selectedObject", function(nv){
        $scope.getSessionTaskIfChanged((nv||{}).sessionId)
    });

    $scope.$watch('uiState.currentMetric', setMainMetric);

    $scope.canSave = function() {
        return ModelLabelUtils.validateLabels($scope.mlTaskDesign);
    }
});


app.controller("_MLTaskResultsController", function($scope, $timeout, $state, $stateParams, ActivityIndicator,
    CreateModalFromTemplate, DataikuAPI, Fn, Dialogs, PartitionedModelsService, FullModelLikeIdUtils, MetricsUtils,
    MLDiagnosticsService, createOrAppendMELikeToModelComparisonModalDirective, createOrAppendModelsToExperimentTrackingModalDirective, CreateModalFromComponent,
    WT1, MLTaskInformationService) {
    angular.extend($scope, PartitionedModelsService);
    angular.extend($scope, MLDiagnosticsService);

    $scope.partiallyAbortTraining = function(fullModelIds){
        var gsModels;
        if (!$scope.isModelOptimizing) {
            gsModels = [];
        } else {
            gsModels = fullModelIds.map(function(o){return $scope.modelSnippets[o]})
                .filter($scope.isModelOptimizing)
                .map(Fn.prop("fullModelId"));
        }
        CreateModalFromTemplate("/templates/analysis/mlcommon/abort-train-modal.html", $scope, null, function(newScope) {
            newScope.gsModels = gsModels;
            // only show next session if this is the last model to train
            newScope.nextSession = $scope.mlTaskStatus.queueStatus === 'RUNNING' && $scope.queuedSessionIds && ($scope.selection.sessionModels || []).filter(function(model) {
                return $scope.isModelRunning(model);
            }).length === 1 ? $scope.queuedSessionIds[0] : null;

            newScope.confirm = function() {
                DataikuAPI.analysis.mlcommon
                    .trainAbortPartial($stateParams.projectKey, $stateParams.analysisId, $stateParams.mlTaskId, fullModelIds, newScope.uiState.pauseQueue)
                    .success(function(){
                        ActivityIndicator.success("Abort requested");
                        fullModelIds.forEach(function(fmi){
                            $scope.modelSnippets[fmi].trainInfo.$userRequestedState = 'ABORTED';
                        });
                    }).error(setErrorInScope.bind($scope));

                    WT1.event('mltask-abort', {
                        taskType: $scope.mlTaskDesign.taskType,
                        pauseQueue: newScope.uiState.pauseQueue,
                        isPartialAbort: true
                    });
                newScope.dismiss();
            }
            newScope.finalize = function() {
                DataikuAPI.analysis.mlcommon.stopGridSearch(gsModels)
                    .success(function(data){
                        gsModels.map(function(fmi){ $scope.modelSnippets[fmi].trainInfo.$userRequestedState = 'FINALIZE' });
                        $scope.refreshStatus();
                    })
                    .error(setErrorInScope.bind($scope));
                newScope.dismiss();
            }
        });
    };
    
    $scope.revertDesignToModel = function(fullModelId, algorithm){
        const idTokens = FullModelLikeIdUtils.parse(fullModelId);
        const sessionId = idTokens.sessionId;
        CreateModalFromTemplate("/templates/analysis/mlcommon/dump-model-design-modal.html", $scope, null, function(newScope) {
            newScope.sessionId = sessionId.slice(1);            
            newScope.canChoose = ("SCIKIT_MODEL" !== algorithm && !$scope.isPartitionedSession(sessionId));
            if (newScope.canChoose) {
                newScope.dumpMode = "OPTIMIZED";
            } else {
                newScope.dumpMode = "INITIAL";
            }
            newScope.confirm = function() {
                $scope.revertScriptToSession(newScope.projectSummary.projectKey, newScope.analysisId, newScope.mlTaskDesign.id, sessionId).then(function(scriptFile) {
                    $scope.analysisCoreParams.script = scriptFile;
                    if (newScope.dumpMode=="OPTIMIZED") {
                        $scope.revertDesignToGridsearchedModel(fullModelId);
                        newScope.dismiss();
                    } else {
                        $scope.revertDesignToPretrainModel(fullModelId);
                        newScope.dismiss();
                    }
                });
            }
        });
    }

    $scope.downloadTrainDiagnosis = function(fullModelId){
        CreateModalFromTemplate("/templates/analysis/mlcommon/download-train-diagnosis-modal.html", $scope, null, function(newScope) {
            newScope.includeTrainingData = false;
            if ($scope.isSparkBased()) {
                newScope.includeTrainingDataDisabledMessage = "Training data can't be downloaded for Spark-based models.";
            } else if (!$scope.projectSummary || !$scope.projectSummary.canExportDatasetsData) {
                newScope.includeTrainingDataDisabledMessage = "You don't have the permission to download training data";
            }
            newScope.confirm = function() {
                ActivityIndicator.success("Preparing train diagnosis ...");
                downloadURL(DataikuAPI.analysis.mlcommon.getTrainDiagnosisURL(fullModelId, newScope.includeTrainingData));
                newScope.dismiss();
            }
        });
    }

    const getPretrainEquivalentMLTask = function(fullModelId, usePostTrain){
        return $scope.MLAPI.getPretrainEquivalentMLTask(fullModelId, usePostTrain)
            .then((response) => {
                return response.data;
            }, setErrorInScope.bind($scope));
    };

    $scope.revertDesignToPretrainModel = function (fullModelId) {
        getPretrainEquivalentMLTask(fullModelId,false).then(function(sessionDesign){
            $scope.updateSettings(sessionDesign);
            $state.go('^.^.design');
        });
    }

    $scope.revertDesignToGridsearchedModel = function (fullModelId) {
        getPretrainEquivalentMLTask(fullModelId, true).then(function(sessionDesign) {
            $scope.updateSettings(sessionDesign);
            $state.go('^.^.design');
        });
    }

    $scope.updateOrderQueryMetric = function(metric) {
        var ss = $scope.selection;
        ss.orderQuery = '-sortMainMetric';
        if ($scope.uiState.currentMetric === metric) {
            ss.orderReversed = !ss.orderReversed;
        } else {
            ss.orderReversed = false;
        }
        $scope.uiState.currentMetric = metric;
        $timeout($scope.updateSorted);
    }

    $scope.forbiddenModelDeletionReason = function(models) {
        if (models.length < 1) {
            return "At least one model must be selected";
        }
        if(models.some(model => model.trainInfo.state === 'RUNNING' || model.trainInfo.state === 'PENDING' || $scope.isSessionRunning(model.sessionId))) {
            return "Cannot delete models belonging to a running session";
        }
        if (!$scope.canWriteProject()){
            return "You don't have write permissions for this project";
        }
    }

    $scope.forbiddenRevertDesignToThisModelReason = function(model) {
        if (model.isEnsembled) {
            return "Cannot revert design to an Ensemble model";
        }
        if (!$scope.canWriteProject()){
            return "You don't have write permissions for this project";
        }
    };

    $scope.isEnsembleModel = function(sessionId) {
        return $scope.selection && $scope.selection.allObjects
            && $scope.selection.allObjects.some(model => model.sessionId === sessionId && model.isEnsembled);
    };

    $scope.comparisonForbiddenReason = function() {
        if($scope.mlTasksContext.activeMLTask.backendType === "DEEP_HUB") {
            return "Computer vision model comparison is not supported";
        }
        if($scope.mlTasksContext.activeMLTask.taskType !== "PREDICTION") {
            return "Only prediction models can be compared";
        }
        if(!$scope.selection || !$scope.selection.selectedObjects || $scope.selection.selectedObjects.length < 1) {
            return "At least one model must be selected";
        }
        if(!$scope.selection.selectedObjects.every(model => model.predictionType === $scope.selection.selectedObjects[0].predictionType)) {
            return "Compared models must have the same prediction type";
        }
        if(!$scope.selection.selectedObjects.every(model => model.trainInfo.state === 'DONE')) {
            return "Only successfully trained models can be compared";
        }
        if($scope.selection.selectedObjects.some(model => model.partitionedModelEnabled)) {
            return "Partitioned models cannot be compared";
        }
        if($scope.selection.selectedObjects.some(model => model.isEnsembled)) {
            return "Ensembled models cannot be compared";
        }
        if (!$scope.canWriteProject()){
            return "You don't have write permissions for this project";
        }
    }

    $scope.compareWithExperimentTrackingForbiddenReason = function() {
        if(!["PY_MEMORY", "KERAS"].includes($scope.mlTasksContext.activeMLTask.backendType) ||
        !(["MULTICLASS", "BINARY_CLASSIFICATION", "REGRESSION"].includes($scope.mlTasksContext.activeMLTask.predictionType) ||
        $scope.mlTasksContext.activeMLTask.taskType === "CLUSTERING")) {
            return "This type of model cannot be compared in Experiment Tracking.";
        }
        if(!$scope.selection || !$scope.selection.selectedObjects || $scope.selection.selectedObjects.length < 1) {
            return "At least one model must be selected";
        }
        if(!$scope.selection.selectedObjects.every(model => model.trainInfo.state === 'DONE')) {
            return "Only successfully trained models can be compared";
        }
        if (!$scope.canWriteProject()){
            return "You don't have write permissions for this project";
        }
    }

    $scope.createModelComparison = function() {
        const nbModels = $scope.selection.selectedObjects.length;
        const predictionType = $scope.selection.selectedObjects[0].predictionType;
        const mlTaskName = $scope.mlTaskDesign.name;

        CreateModalFromComponent(createOrAppendMELikeToModelComparisonModalDirective, {
            fullIds: $scope.selection.selectedObjects.map(model => model.fullModelId),
            modelTaskType: predictionType, // ModelTaskType values encompass PredictionType possible values. So this "cast" will work.
            suggestedMCName: `Compare ${nbModels} models from ${mlTaskName}`,
            projectKey: $stateParams.projectKey,
            trackFrom: 'lab-sessions-list'
        });
    }

    $scope.compareWithExperimentTracking = function() {
        DataikuAPI.experimenttracking.getExperiments($stateParams.projectKey).success(function(data) {
            CreateModalFromComponent(createOrAppendModelsToExperimentTrackingModalDirective, {
                experiments: data,
                fmis : $scope.selection.selectedObjects.map(o =>  o.fullModelId),
                projectKey : $stateParams.projectKey,
                trackFrom: 'lab-sessions-list'
            });
        }).error(setErrorInScope.bind($scope));
    }

    $scope.ensembleCreationForbiddenReason = function() {
        if($scope.mlTasksContext.activeMLTask.backendType === "DEEP_HUB") {
            return "Computer vision models cannot be used to create an ensemble model";
        }
        if($scope.mlTasksContext.activeMLTask.backendType === "KERAS") {
            return "Deep learning models cannot be used to create an ensemble model";
        }
        if($scope.mlTasksContext.activeMLTask.predictionType === "TIMESERIES_FORECAST") {
            return "Time series forecasting models cannot be used to create an ensemble model";
        }
        if(["CAUSAL_REGRESSION", "CAUSAL_BINARY_CLASSIFICATION"].includes($scope.mlTasksContext.activeMLTask.predictionType)) {
            return "Causal prediction models cannot be used to create an ensemble model";
        }
        if($scope.mlTasksContext.activeMLTask.taskType !== "PREDICTION") {
            return "Only prediction models can be used to create an ensemble model";
        }
        if(!$scope.selection || !$scope.selection.selectedObjects || $scope.selection.selectedObjects.length < 2) {
            return "At least two models must be selected";
        }
        if(!$scope.selection.selectedObjects.every(model => model.trainInfo.state == 'DONE')) {
            return "Only successfully trained models can be part of an ensemble model";
        }
        if (!$scope.canWriteProject()){
            return "You don't have write permissions for this project";
        }
    }


    $scope.createEnsemble = function(){
        var fmis = $scope.selection.selectedObjects.map(function(o){ return o.fullModelId; });

        DataikuAPI.analysis.pml.checkCanEnsemble(fmis).success(function(data){
            CreateModalFromTemplate("/templates/analysis/prediction/create-ensemble-modal.html", $scope, "EnsembleModalController", function(newScope){
                newScope.fmis = fmis;
                newScope.params.method = (data.availableMethods && data.availableMethods.length) ? data.availableMethods[0].method : null;
                newScope.cannotEnsembleReason = data.cannotEnsembleReason;
                newScope.availableMethods = data.availableMethods;
                const methodMap = {};
                newScope.availableMethods.forEach(par => {methodMap[par.method] = par.description});
                newScope.getSelectedMethodDescription = function(){
                    return methodMap[newScope.getMethod()];
                };
                newScope.showTiesWarning = function(){
                    if(newScope.getMethod()==='VOTE'){
                        if($scope.mlTaskDesign.predictionType==="BINARY_CLASSIFICATION"){
                            return ((fmis.length % 2)===0); // show warning only if even number of models
                        } else {
                            // for c classes, and m models the condition for no-ties guaranty is that
                            // m > c and
                            // m % 2 ≠ 0, ..., m % c ≠ 0 which is, more often than not, false.
                            return ($scope.mlTaskDesign.predictionType==="MULTICLASS");
                        }
                    } else {
                        return false;
                    }
                };
                newScope.submit = function(){
                    DataikuAPI.analysis.pml.createEnsemble(
                        $stateParams.projectKey, $stateParams.analysisId, $stateParams.mlTaskId, fmis, newScope.getMethod()
                    ).success(function(){
                        $scope.removeTaskStatus();
                        $scope.initialRefreshAndAutoRefresh();
                        newScope.dismiss();
                    }).error(setErrorInScope.bind($scope));
                };
            });
        }).error(setErrorInScope.bind($scope));
    };

    $scope.allStarredModels = function() {
        var selectedObjects = $scope.selection.selectedObjects.filter(function(o){return o.trainInfo.state==='DONE';})
        return selectedObjects.map(Fn.prop("userMeta"))
            .map(Fn.prop("starred"))
            .reduce(function(a,b){return a&&b},true);
    }
    $scope.canStarSelectedModels = function() {
        return $scope.selection.selectedObjects.map(Fn.prop("trainInfo"))
            .map(function(o){return o.state=='DONE';})
            .reduce(function(a,b){return a&&b},true);
    }
    $scope.starSelectedModels = function(star) {
        var selectedObjects = $scope.selection.selectedObjects.filter(function(o){return o.trainInfo.state==='DONE';})
        selectedObjects.map(function(m){
            m.userMeta.starred = star;
        });
    }

    $scope.isModelFinalizing = function(model) {
        let key = $scope.isMLBackendType("KERAS") ? "modelTrainingInfo" : "gridsearchData";
        if (!model||!model[key]) { return false }
        return model[key].isFinalizing && $scope.isModelRunning(model);
    }

    $scope.isModelOptimizing = function(model) {
        // Interrupting model is currently not supported for partitioned models
        if (model.partitionedModelEnabled) {
            return false;
        }
        if ($scope.isMLBackendType("KERAS") || $scope.isMLBackendType("DEEP_HUB")) {
            if (!model||!model.modelTrainingInfo) { return false }
                return !model.modelTrainingInfo.isFinalizing
                    && $scope.anySessionModelHasOptimizationResults()
                    && !$scope.anyModelHasAllEpochsFinished()
                    && $scope.isModelRunning(model);
        } else {
            if (!model) { return false }
            const doingGridSearch = model.gridsearchData && !model.gridsearchData.isFinalizing && model.gridsearchData.gridPoints.length > 0;
            return doingGridSearch && $scope.isModelRunning(model);
        }
    };

    $scope.hasOptimizingModels = function(sessionId) {
        return $scope.selection && $scope.selection.allObjects && $scope.selection.allObjects
            .some(function(model) {
                return model.sessionId === sessionId
                    && $scope.isModelOptimizing(model);
            });
    }

    $scope.stopGridSearch = function(fullModelIds, setUiState = false) {
        Dialogs.confirm($scope, "Suspend optimization for this model",
            "Do you want to suspend the optimization for this model?").then(function() {
                WT1.event("stop-grid-search", {});
                DataikuAPI.analysis.mlcommon.stopGridSearch(fullModelIds)
                    .success(() => {
                        fullModelIds.forEach(fmi => { $scope.modelSnippets[fmi].trainInfo.$userRequestedState = "FINALIZE" });
                        if (setUiState) {
                            $scope.uiState.$userRequestedState = 'FINALIZE';
                        }
                        $scope.refreshStatus();
                    }).error(setErrorInScope.bind($scope));
            });
    };

    $scope.stopGridSearchSession = function(sessionId) {
        const fullModelIds = $scope.selection.allObjects
            .filter(model => model.sessionId === sessionId && $scope.isModelOptimizing(model))
            .map(Fn.prop('fullModelId'));

        $scope.stopGridSearch(fullModelIds, true);
    };

    $scope.isModelOptimizationResumable = function(model) {
        // Resuming optimization is not supported for KERAS & deephub algorithms
        if ($scope.isMLBackendType("KERAS") || $scope.isMLBackendType("DEEP_HUB")) {
            return false;
        }

        // Resuming optimization is not supported for partitioned models
        if (model.partitionedModelEnabled) {
             return false;
        }
        
        const searchProgress = $scope.getModelSearchProgress(model);
        return $scope.isModelDone(model) 
            && ($scope.isModelOptimizationBoundByTimeout(model) || (searchProgress !== undefined && searchProgress < 1));
    };

    $scope.isModelOptimizationBoundByTimeout = function(model) {
        const gsd = model.gridsearchData;
        if (!gsd) {
            return false;
        }
        return (gsd.gridSize === 0 && gsd.timeout > 0);
    };
    
    $scope.getModelSearchProgress = function(model) {
        const gsd = model.gridsearchData;
        if (!gsd) {
            return undefined;
        }

        if (gsd.gridSize !== 0) {
            return model.gridsearchData.gridPoints.length / model.gridsearchData.gridSize;
        } else if (gsd.timeout > 0) { // model search is bound by timeout, need to look at time spent
            return  Math.min(1, (model.trainInfo.hyperparamsSearchTime / 1000) / (gsd.timeout * 60));
        }
        return 0;
    };

    $scope.isModelDone = function(model) {
        if (!model||!model.trainInfo) { return false }
        return model.trainInfo.state === 'DONE';
    }

    $scope.isModelPending = function(model) {
        if (!model||!model.trainInfo) { return false }
        return model.trainInfo.state === 'PENDING';
    }

    $scope.isModelFailedOrAborted = function(model) {
        if (!model||!model.trainInfo) { return false }
        return model.trainInfo.state === 'FAILED' || model.trainInfo.state === 'ABORTED';
    }

    $scope.isMetricFailed = MetricsUtils.isMetricFailed;
    $scope.getSpecificCustomMetricResult = MetricsUtils.getSpecificCustomMetricResult;

    $scope.isModelFailed = function(model) {
        if (!model||!model.trainInfo) { return false }
        return model.trainInfo.state === 'FAILED';
    }

    $scope.isModelRetrainable = function(model) {
        return  !$scope.mlTaskStatus.training
            && $scope.mlTasksContext.activeMLTask.taskType === "PREDICTION"
            && !($scope.isMLBackendType("KERAS") || $scope.isMLBackendType("DEEP_HUB"))
            && ($scope.isModelAborted(model)
                || model.partitionedModelEnabled && $scope.getPartitionsSnippetStateSize(model, "ABORTED") > 0);
    }

    $scope.hasResumableModels = function(sessionId) {
        return $scope.selection && $scope.selection.allObjects
            && $scope.selection.allObjects.some(function(model) {
                return model.sessionId === sessionId
                    && ($scope.isModelOptimizationResumable(model) || $scope.isModelRetrainable(model));
            });
    }

    $scope.isModelAborted = function(model) {
        if (!model||!model.trainInfo) { return false }
        return model.trainInfo.state === 'ABORTED';
    }

    $scope.isModelRunning = function(model) {
        if (!model||!model.trainInfo) { return false }
        return model.trainInfo.state === 'RUNNING' || model.trainInfo.state === 'PENDING';
    }

    $scope.isBestModelScore = function(model, sameSession) {
        if (!model.mainMetric) return false;
        return model.sortMainMetric >= $scope.selection.allObjects
            .filter(o => !sameSession || o.sessionId === model.sessionId)
            .map(Fn.prop('sortMainMetric'))
            .reduce(Fn.MAX,-Number.MAX_VALUE);
    }

    $scope.sessionRunningCount = function(sessionId) {
        if (!$scope.selection||!$scope.selection.allObjects){return false;}
        return $scope.selection.allObjects.filter(function(o){return o.sessionId===sessionId})
            .map($scope.isModelRunning).map(function(o){return o?1:0}).reduce(Fn.SUM,0);
    }

    $scope.isSessionRunning = (sessionId) => MLTaskInformationService.isSessionRunning($scope, sessionId);

    $scope.isPartitionedSession = function(sessionId) {
        if (!$scope.selection || !$scope.selection.allObjects) { return false; }

        return $scope.selection.allObjects
            .filter(o => o.sessionId === sessionId)
            .every(m => m.partitionedModelEnabled);
    };

    $scope.getAggregationExplanation = function(metricName) {
        const hideTestWeightMention = $scope.mlTaskDesign && $scope.mlTaskDesign.predictionType === 'TIMESERIES_FORECAST';
        if ($scope.uiState.currentMetricIsCustom) {
            return PartitionedModelsService.getAggregationExplanation(metricName, metricName, true, hideTestWeightMention);
        } else {
            const displayName = $scope.allMetrics.find(_ => _[0] === metricName)[1];
            return PartitionedModelsService.getAggregationExplanation(metricName, displayName, false, hideTestWeightMention);
        }
    }

    $scope.getSessionStartDate = function(sessionId) {
        var minDate = 0;
        $scope.getSessionModels($scope.selection.allObjects, sessionId).forEach(function(x) {
            if (x && x.trainInfo && x.trainInfo.startTime) {
                minDate = Math.max(minDate, x.trainInfo.startTime)
            }
        });
        if (minDate == 0) {
            return null;
        } else {
            return new Date(minDate);
        }
    }

    $scope.getSessionEndDate = function(sessionId) {
        var maxDate = 0;
        $scope.getSessionModels($scope.selection.allObjects, sessionId).forEach(function(x) {
            if (x && x.trainInfo && x.trainInfo.endTime) {
                maxDate = Math.max(maxDate, x.trainInfo.endTime)
            }
        });
        if (maxDate == 0) {
            return null;
        } else {
            return new Date(maxDate);
        }
    };

    $scope.deleteSession = function(sessionId) {
        const models = $scope.selection.allObjects.filter(function(o){return !$scope.isModelRunning(o) && o.sessionId === sessionId});
         // if session doesn't have fmi, it's queued
        const queuedSessionIds = models.length && models[0].trainInfo && models[0].trainInfo.state === 'QUEUED' ? [sessionId] : [];
        const fullModelIds = !queuedSessionIds.length ? models.map(Fn.prop("fullModelId")) : [];

        $scope.openDeleteModelAndSessionModal(fullModelIds, queuedSessionIds);
    };

    $scope.deleteModel = function(model) {
        if ($scope.isModelRunning(model)) { return; }

        $scope.openDeleteModelAndSessionModal([model.fullModelId]);
    };

    $scope.deleteSelectedModelsAndQueuedSessions = function(includeQueuedSessionIds) {
        const queuedSessionIds = includeQueuedSessionIds ? $scope.selection.selectedObjects.filter(o => o.trainInfo.state === 'QUEUED').map(Fn.prop('sessionId')) : [];
        const fullModelIds = $scope.selection.selectedObjects.filter(o => o.trainInfo.state !== 'RUNNING' && o.trainInfo.state !== 'PENDING' && o.trainInfo.state !== 'QUEUED' ).map(Fn.prop("fullModelId"));

        $scope.openDeleteModelAndSessionModal(fullModelIds, queuedSessionIds);
    };

    $scope.toggleStarred = function(snippetData) {
        snippetData.userMeta.starred = !snippetData.userMeta.starred;
        DataikuAPI.ml.saveModelUserMeta(snippetData.fullModelId, snippetData.userMeta)
                            .error(setErrorInScope.bind($scope.$parent));
        $scope.$emit('refresh-list');
    }

    function getSessions(models, states, excludedSessions = [], orderByRecent = true) {
        return (models || [])
            .filter(m => states.includes(m.trainInfo.state) && !(excludedSessions.length && excludedSessions.includes(m.sessionId)))
            .map(m => m.sessionId)
            .filter((value, index, self) => self.indexOf(value) === index)
            .sort((a, b) => parseInt((orderByRecent ? b : a).slice(1)) - parseInt((orderByRecent ? a : b).slice(1)));
    }

    $scope.getSessionModels = function(models, sessionId) {
        return (models || []).filter(m => m.sessionId === sessionId && !(m.trainInfo && m.trainInfo.state === 'QUEUED'));
    };

    $scope.sessionGroups = [];
    function setSessionGroups(sessions) {
        const training = { type: 'TRAINING', sessions: getSessions(sessions, ['RUNNING', 'PENDING']) };

        $scope.sessionGroups = [
            { type: 'QUEUED', sessions: getSessions(sessions, ['QUEUED'], training.sessions) }, 
            training, 
            { type: 'OTHER', sessions: getSessions(sessions, ['DONE', 'ABORTED', 'FAILED'], training.sessions) }
        ];
    }

    let alreadyScrolled = false;
    $scope.$watch('selection.filteredObjects', (nv) => {
        setSessionGroups(nv);

        $scope.scrollToMe = false;
        // don't continue scrolling to currently training item every time the list changes
        if ($scope.mlTaskStatus.training && !alreadyScrolled) {
            $scope.scrollToMe = true;
            alreadyScrolled = true;
        }
    });

    $scope.$on('snippetUpdate', () => {
        setSessionGroups($scope.selection.filteredObjects);
    });

    $scope.$watch('mlTaskStatus.training', () => {
        if ($scope.mlTaskStatus && !$scope.mlTaskStatus.training) {
            alreadyScrolled = false;
        }
    });

    $scope.$watch("uiState.viewMode", function(nv){
        if (nv==='sessions') {
            $scope.selection.orderQuery = ['-sessionDate','algorithmOrder'];
        } else {
            $timeout(function(){
                $scope.$broadcast('redrawFatTable');
            });
            $scope.selection.orderQuery = [];
        }
    });

    $scope.scrollToModel = (selectedModel) => {
        if (!selectedModel) { return }
        const selectedModelDOM = document.getElementById(selectedModel.fullModelId);
        if (selectedModelDOM) {
            selectedModelDOM.scrollIntoView();
        }
    };

    $scope.selection.filterCategory = {};
    $scope.clearModelsListFilters = function() {
        $scope.clearFilters();
        $scope.selection.filterCategory = {};
    };

    // returns number of models, excluding queued sessions
    $scope.getModelCount = function(models) {
        return models.filter(model => model.trainInfo.state !== 'QUEUED').length;
    };

    $scope.isModel = function(model) {
        return model.trainInfo.state !== 'QUEUED';
    };
});


app.component("algorithmModelFilter", {
    templateUrl: "/templates/ml/advanced-models-filter.html",
    controller: function($scope) {
        const $ctrl = this;

        function getAllAlgorithmTypes() {
            let allAlgorithmTypes = [];
            Object.keys($ctrl.algorithmCategories).forEach((key, index) => {
                allAlgorithmTypes = allAlgorithmTypes.concat($ctrl.algorithmCategories[key])
            });
            return allAlgorithmTypes;
        }

        $ctrl.resetCategories = function() {
            for (let key of $ctrl.extendedAlgorithmCategories) {
                $ctrl.filterCategory[key] = false;
            }
        }

        $ctrl.algorithmsFilter = function (filteredObjects) {
            if ($ctrl.filterCategory) {
                for (let key of Object.keys($ctrl.filterCategory)) {
                    if (key === "Others" && $ctrl.filterCategory["Others"]) {
                        filteredObjects = filteredObjects.filter(x => getAllAlgorithmTypes().includes(x.algorithm));
                    }
                    else if ($ctrl.filterCategory[key]) {
                        filteredObjects = filteredObjects.filter(x => !$ctrl.algorithmCategories[key].includes(x.algorithm));
                    }
                }
            }
            return filteredObjects;
        };

        $ctrl.$onChanges = function(changesObj) {
            $ctrl.filterAlgorithms = false;

            if (changesObj.algorithmCategories && changesObj.algorithmCategories.currentValue) {
                $ctrl.extendedAlgorithmCategories = Object.keys(changesObj.algorithmCategories.currentValue);
                if ($ctrl.algorithmCategories.others) {
                    $ctrl.extendedAlgorithmCategories.push("Others");
                }
            }
            if ($ctrl.extendedAlgorithmCategories) {
                $ctrl.resetCategories();
            }
        }
    },
    bindings: {
        selection: "=",
        algorithmCategories: "<",
        filterCategory: "<"
    }
});

app.service("MLDiagnosticsService", () => {
    return {
        groupByStepAndType: (mlDiagnostics) => {
            if (!mlDiagnostics || !mlDiagnostics.diagnostics) {
                return {};
            }

            const groupedDiagnostics = {};
            for (const diagnostic of mlDiagnostics.diagnostics) {
                groupedDiagnostics[diagnostic.step] = groupedDiagnostics[diagnostic.step] || {};
                const groupedStep = groupedDiagnostics[diagnostic.step];
                groupedStep[diagnostic.displayableType] = groupedStep[diagnostic.displayableType] || [];

                const messages = groupedStep[diagnostic.displayableType];
                messages.push(diagnostic.message);
            }

            return groupedDiagnostics;
        },
        groupByType: (mlDiagnostics) => {
            if (!mlDiagnostics || !mlDiagnostics.diagnostics) {
                return {};
            }

            const groupedDiagnostics = {};
            for (const diagnostic of mlDiagnostics.diagnostics) {
                groupedDiagnostics[diagnostic.type] = groupedDiagnostics[diagnostic.type] || [];
                const messages = groupedDiagnostics[diagnostic.type];
                messages.push(diagnostic.message);
            }

            return groupedDiagnostics;
        },
        hasDiagnostics: (model) => {
            if (!model) {
                return false;
            }

            const _hasDiagnostics = model => ((model.mlDiagnostics && model.mlDiagnostics.diagnostics) || []).length > 0;

            if (!model.partitionedModelEnabled) {
                return _hasDiagnostics(model);
            } else {
                return model.partitions &&  // check warnings in each summary
                    Object.values(model.partitions.summaries).map(s => s.snippet).some(_hasDiagnostics);
            }
        },
        countDiagnostics: model => {
            if (!model) {
                return 0;
            }

            const _countDiagnostics = model => {
                if (!model.mlDiagnostics || !model.mlDiagnostics.diagnostics) {
                    return 0;
                }

                return model.mlDiagnostics.diagnostics.length;
            }

            let total = 0;
            if (!model.partitionedModelEnabled) {
                total = _countDiagnostics(model);
            } else { // check warnings in each summary
                for (const s of Object.values(model.partitions.summaries)) {
                    if (s.state === "FAILED") {
                        continue;
                    }
                    total += _countDiagnostics(s.snippet);
                }
            }

            return total;
        },
        getDiagnosticsTextForPartitions: model => {
            if (!model || !model.partitionedModelEnabled) {
                return null;
            }

            let countWithDiagnostics = 0;
            for (const s of Object.values(model.partitions.summaries)) {
                if (s.state !== "FAILED" && Object.keys((s.snippet.mlDiagnostics && s.snippet.mlDiagnostics.diagnostics) || []).length > 0) {
                    countWithDiagnostics++;
                }
            }

            const totalPartitions = Object.values(model.partitions.summaries).length;
            if (countWithDiagnostics === 1) {
                return `On one partition out of ${totalPartitions}`;
            } else {
                return `On ${countWithDiagnostics} out of ${totalPartitions} partitions`;
            }
        }
    };
});

app.factory("VisualMlCodeEnvCompatibility", function() {
    const service = {
        getCodeEnvCompat,
        isEnvAtLeastTensorflow2_2,
    };

    return service;

    /* 
    * Returns the compatibility info of the selected code env (envSelection) retrieved from the list of compatibility info of all code envs (envCompatList)
    */
    function getCodeEnvCompat(envSelection, envCompatList) {
        if (!envSelection) return;
        let envCompat;
        switch(envSelection.envMode) {
            case "USE_BUILTIN_MODE":
                envCompat = envCompatList.builtinEnvCompat;
                break;
            case "INHERIT": {
                if (!envCompatList.resolvedInheritDefault) { // Project code-env is builtin
                    envCompat = envCompatList.builtinEnvCompat;
                } else {
                    envCompat = envCompatList.envs.find(env => env.envName === envCompatList.resolvedInheritDefault);
                }
                break;
            }
            case "EXPLICIT_ENV":
                envCompat = envCompatList.envs.find(env => env.envName === envSelection.envName);
                break;
        }
        return envCompat;
    };

    /* 
    * Returns whether the current env contains tensorflow >= 2.2.
    * Defaults to false because our code for previous versions is TF2.2+ compatible,
    * whereas the TF2.2+ specific code is not backward compatible. Thus returning false is the safe option.
    */
    function isEnvAtLeastTensorflow2_2(mlTaskDesign, codeEnvsCompat) {
        if (!mlTaskDesign || !codeEnvsCompat) return false;

        const envCompat = getCodeEnvCompat(mlTaskDesign.envSelection, codeEnvsCompat);
        return envCompat && envCompat.keras && envCompat.keras.isAtLeastTensorflow2_2;
    };
})

/**
 * This component assumes that `mltaskDesign` will never change (except from its envSelection or containerSelection) after its creation,
 * which is the case at the moment.
 *
 * If this were to evolve, we would need to implement a slightly more complex logic to check that mltaskDesign changes
 * on something else than envSelection, in order not to refetch the env list everytime one envSelection changes.
 */
app.component("codeEnvSelectionWithMlPackagesForm",  {
    bindings: {
        mltaskDesign: "="
    },
    templateUrl : '/templates/analysis/mlcommon/code-env-selection-with-ml-packages-form.html',
    controller: function($scope, $stateParams, DataikuAPI) {
        const $ctrl = this;

        $ctrl.$onInit = () => {
            $ctrl.envModes = [
                ['USE_BUILTIN_MODE', 'Use DSS builtin env'],
                ['INHERIT', 'Inherit project default'],
                ['EXPLICIT_ENV', 'Select an environment']
            ];
            fetchAndFormatEnvs($ctrl.mltaskDesign);
        }

        let currentEnvSelection = null;
        let currentContainerSelection = null;
        $ctrl.$doCheck = () => {
            const envSelectionChanged = !angular.equals(currentEnvSelection, $ctrl.mltaskDesign.envSelection);
            const containerSelectionChanged = !angular.equals(currentContainerSelection, $ctrl.mltaskDesign.containerSelection);
            if ($ctrl.envsCompatInfo && (envSelectionChanged || containerSelectionChanged)) {
                updateInformation();
                currentEnvSelection = angular.copy($ctrl.mltaskDesign.envSelection);
                currentContainerSelection = angular.copy($ctrl.mltaskDesign.containerSelection);
            }
        }

        function fetchAndFormatEnvs(mltaskDesign) {
            DataikuAPI.codeenvs.listCompatibilityWithMLTask($stateParams.projectKey, mltaskDesign).then(function(response) {
                $ctrl.envsCompatInfo = response.data;
                updateInformation();
                if (response.data.resolvedInheritDefaultCodeEnv == null) {
                    $ctrl.envModes[1][1] = "Inherit project default (DSS builtin env)"
                } else {
                    $ctrl.envModes[1][1] = "Inherit project default (" + response.data.resolvedInheritDefaultCodeEnv + ")";
                }
            }).catch(setErrorInScope.bind($scope));
        }

        function updateInformation() {
            if (!$ctrl.envsCompatInfo || !$ctrl.envsCompatInfo.envs) { // Not ready
                return;
            }
            $ctrl.usedContainerConfName = getUsedContainerConfName();

            $ctrl.envsInfo = $ctrl.envsCompatInfo.envs
                                .map(env => formatCompatInfo(env.envName, $ctrl.usedContainerConfName))
                                // Putting compatible envs first
                                .sort((env1, env2) => Number(env1.hasAnyIncompatibilities) - Number(env2.hasAnyIncompatibilities));

            setDefaultExplicitEnvIfNone($ctrl.envsInfo);
            const usedEnvName = getUsedEnvName();
            $ctrl.usedEnvInfo = formatCompatInfo(usedEnvName, $ctrl.usedContainerConfName);
            $ctrl.envDescriptions = $ctrl.envsInfo.map(envInfo => envInfo.formattedDescription);
            $ctrl.compatibilityDescription = $ctrl.usedEnvInfo.compatibilityDescription;
       }

       function setDefaultExplicitEnvIfNone(envsInfo) {
           let envSelection = $ctrl.mltaskDesign.envSelection;
           if (envSelection.envMode == "EXPLICIT_ENV" && !envSelection.envName) {
                const firstCompatibleEnv = (envsInfo.find(e => !e.hasAnyIncompatibilities) || {}).envName;
                if (firstCompatibleEnv) {
                    envSelection.envName = firstCompatibleEnv;
                }
            }

            // used for gpu selection code env information
            if (envSelection.envMode == "INHERIT" && $ctrl.envsCompatInfo.resolvedInheritDefaultCodeEnv) {
                envSelection.defaultEnvName = $ctrl.envsCompatInfo.resolvedInheritDefaultCodeEnv;
            }
        }

        function formatCompatInfo(envName, containerConfName) {
            let envIncompatibilityReasons;
            let builtForContainerConf;
            let compatibilityDescription;
            if (!envName)  { // builtin env
                envIncompatibilityReasons = $ctrl.envsCompatInfo.builtinIncompatibilityReasons;
                compatibilityDescription = $ctrl.envsCompatInfo.builtinEnvDescription;
                builtForContainerConf = true;
            } else {
                const envInfo = ($ctrl.envsCompatInfo.envs.find(env => env.envName === envName) || {});
                envIncompatibilityReasons = envInfo.incompatibilityReasons;
                compatibilityDescription = envInfo.description;
                builtForContainerConf = containerConfName ? (envInfo.builtForContainerConfs || []).indexOf(containerConfName) > -1 : true;
            }
            const allIncompatibilityReasons = angular.copy(envIncompatibilityReasons || []);
            if (containerConfName && !builtForContainerConf) {
                allIncompatibilityReasons.push("Code-env was not built for containerized execution " + containerConfName);
            }

            let formattedDescription;
            if (hasIncompatibilities(allIncompatibilityReasons)) {
                formattedDescription = getIncompatibleDesc(allIncompatibilityReasons);
            } else {
                formattedDescription = "has required packages for " + compatibilityDescription + ".";
            }

            const hasEnvIncompatibilities = hasIncompatibilities(envIncompatibilityReasons);
            const hasAnyIncompatibilities = hasIncompatibilities(allIncompatibilityReasons);

            return {
                envName,
                builtForContainerConf,
                formattedDescription,
                compatibilityDescription,
                hasEnvIncompatibilities,
                hasAnyIncompatibilities
            }
        }

        function getIncompatibleDesc(reasons) {
            if (reasons.length === 1) {
                return `<span class='text-warning'>${reasons[0]}.</span>`;
            } else {
                return reasons.map(r => `<div class='text-warning'>${r}.</div>`).join('');
            }
        }

        function hasIncompatibilities(reasons) {
            return reasons && reasons.length > 0;
        }

       function getUsedContainerConfName() {
            switch ($ctrl.mltaskDesign.containerSelection.containerMode) {
                case "INHERIT":
                    return $ctrl.envsCompatInfo.resolvedInheritDefaultContainer;
                case "EXPLICIT_CONTAINER":
                    return $ctrl.mltaskDesign.containerSelection.containerConf;
                default: return undefined;
            }
       }

       function getUsedEnvName() {
            switch($ctrl.mltaskDesign.envSelection.envMode) {
                case "USE_BUILTIN_MODE":
                    return undefined;
                case "INHERIT": {
                    return $ctrl.envsCompatInfo.resolvedInheritDefaultCodeEnv;
                }
                case "EXPLICIT_ENV":
                    return $ctrl.mltaskDesign.envSelection.envName;
            }
       }
    }
});

app.directive('modelsTableData', ['computeColumnWidths', '$window', 'CustomMetricIDService', function(computeColumnWidths, $window, CustomMetricIDService) {
    return {
        scope : false,
        restrict : 'A',
        link: function($scope, element) {
            $scope.displayedTableColumns = [];
            $scope.displayedTableRows = [];

            // To correct table width on window resize
            angular.element($window).on('resize', function(){
                $scope.$apply($scope.adjustColumnWidths);
            });

            $scope.$on("$destroy",function (){
                angular.element($window).off("resize"); //remove the handler added earlier
            });

            $scope.adjustColumnWidths = function() {
                let maxWidth = element.prop("clientWidth") - 16; // The -16px is to take into account the fatTable
                                                                 // vertical scrollbar width on firefox

                if ($scope.computedTableWidth < maxWidth) {
                    $scope.displayedColumnsWidths = [];
                    let totalWidth = 0; // Keep track of the total width to correct rounding error
                    $scope.computedColumnWidths.forEach(function(width) {
                       let newWidth = Math.round(width * maxWidth/$scope.computedTableWidth);
                       totalWidth += newWidth;
                       $scope.displayedColumnsWidths.push(newWidth);
                    });
                    if (totalWidth > maxWidth) {
                        // Correct width rounding error (usually difference is 0 or 1), by removing
                        // the difference to last column's width
                        $scope.displayedColumnsWidths.push(
                            $scope.displayedColumnsWidths.splice(-1, 1)[0] - (totalWidth - maxWidth)
                        );
                    }
                } else {
                    $scope.displayedColumnsWidths = $scope.computedColumnWidths;
                }
            }

            var refreshDisplayedColumns = function() {
                $scope.displayedTableColumns = [
                    {isModelSel: true},
                    {isModelName: true},
                    {isModelTrainTime: true},
                    {isModelTrainTimeMetric: true}
                ];

                let header = [
                    {name: "sel", ncharsToShow: 5},
                    {name: "Name", ncharsToShow: 25},
                    {name: "Trained", ncharsToShow: 20},
                    {name: "Train time", ncharsToShow: 6}
                ];

                if ($scope.mlTaskDesign.backendType === 'PY_MEMORY'
                    && $scope.mlTaskDesign.taskType === 'PREDICTION'
                    && ['BINARY_CLASSIFICATION', 'MULTICLASS', 'REGRESSION'].includes($scope.mlTaskDesign.predictionType)) {
                    $scope.displayedTableColumns.push({isSampleWeights: true});
                    header.push({name: "Sample weights", ncharsToShow: 5})
                }

                if ($scope.possibleMetrics) {
                    $scope.possibleMetrics.forEach(function(metric) {
                        $scope.displayedTableColumns.push({isModelMetric: true, metric: metric, isEvaluationMetric: $scope.uiState.evaluationMetricId === metric[0]});
                        header.push({name: "MMMMM MMMMM", ncharsToShow: 10}) // Dummy name of length 11 for all metrics
                    });
                }

                if($scope.possibleCustomMetrics) {
                    $scope.possibleCustomMetrics.forEach(function(metric) {
                        $scope.displayedTableColumns.push({ isCustomMetric: true, metric: metric, isEvaluationMetric: metric[0] === $scope.uiState.evaluationMetricId });
                        header.push({name: "Custom Metric #X", ncharsToShow: 14}); // Dummy name of length 16 for custom metrics to ensure default custom metric name fits without overflow
                    });
                }

                $scope.displayedTableColumns.push({isModelStarred: true});
                header.push({name: "star", ncharsToShow: 1})

                $scope.computedColumnWidths = computeColumnWidths([], header, 30, () => false, {}, {}, true)[0];
                $scope.computedTableWidth =  $scope.computedColumnWidths.reduce((a, b) => a + b);
                $scope.adjustColumnWidths();
            };

            var refreshDisplayedRows = function() {

                // build rows for the fattable
                $scope.displayedTableRows = [];
                if ($scope.selection.filteredObjects) {
                    $scope.selection.filteredObjects.forEach(function(summ) {
                        if (summ.trainInfo.state === 'QUEUED') {
                            return;
                        }

                        var row = [
                            {isModelSel: true, summ: summ},
                            {isModelName: true, summ: summ},
                            {isModelTrainTime: true, summ: summ},
                            {isModelTrainTimeMetric: true, summ: summ}
                        ];

                        if ($scope.mlTaskDesign.backendType === 'PY_MEMORY'
                            && $scope.mlTaskDesign.taskType === 'PREDICTION'
                            && ['BINARY_CLASSIFICATION', 'MULTICLASS', 'REGRESSION'].includes($scope.mlTaskDesign.predictionType)) {
                            row.push({isSampleWeights: true, summ: summ});
                        }
                        if ($scope.possibleMetrics) {
                            $scope.possibleMetrics.forEach(function(metric) {
                                row.push({isModelMetric: true, metric: metric, summ: summ, isEvaluationMetric: metric[0] === summ.evaluationMetric});
                            });
                        }

                        if($scope.possibleCustomMetrics) {
                            $scope.possibleCustomMetrics.forEach(function(metric) {
                                const result = (summ.customMetricsResults || []).filter(
                                    result => result.metric.name === CustomMetricIDService.getCustomMetricName(metric[0]))[0];
                                row.push({
                                    isCustomMetric: true,
                                    metric: metric,
                                    summ: summ, result,
                                    isEvaluationMetric: CustomMetricIDService.getCustomMetricName(metric[0]) === summ.evaluationMetric});
                            });
                        }

                        row.push({isModelStarred: true, summ: summ});
                        $scope.displayedTableRows.push(row);
                    });
                }

            };

            refreshDisplayedColumns();

            $scope.$watchCollection('selection.filteredObjects', function() {
                refreshDisplayedRows();
            });

            var refreshDisplayedColumnsAndRows = function() {
                refreshDisplayedColumns();
                refreshDisplayedRows();
            };

            $scope.$on('redrawFatTable', refreshDisplayedColumnsAndRows);
            $scope.$on('reflow', refreshDisplayedColumnsAndRows);
       }
    };
}])

})();
