(function(){
    'use strict';

    const app = angular.module('dataiku.dashboards');

    const GrantBy = {
        USER: 'USER',
        EMAIL: 'EMAIL'
    };

    const normalizeTextForSearch = function(str) {
        // from the normalizeTextForSearch function in server/src/frontend/src/app/utils/string-utils.ts
        return str ? str.toString().toLowerCase().normalize('NFD').replace(/[\u0300-\u036f]/g, '') : '';
    };

    app.controller('ShareDashboardModalController', function($rootScope, $scope, DataikuAPI, ActivityIndicator) {
        $scope.uiState = {
            personalMessage: ''
        };

        $scope.init = function(dashboard) {
            $scope.canEditPermissions = $rootScope.appConfig.admin || ($scope.projectSummary && $scope.projectSummary.canEditPermissions);
            $scope.shareByEmailEnabled = $rootScope.appConfig.permissionsByEmailEnabled;
            $scope.mailChannelAvailable = $rootScope.appConfig.emailChannelId;
            $scope.dashboard = dashboard;
            $scope.invitations = [];
            $scope.GrantBy = GrantBy;

            $scope.selector = {
                selectedGrantType: $scope.shareByEmailEnabled ? GrantBy.EMAIL : GrantBy.USER,
                users: [],
                email: ''
            };

            $scope.grantTypes = [
                { value: GrantBy.USER, label: 'By user' }
            ];
            if ($scope.shareByEmailEnabled) {
                $scope.grantTypes.unshift({ value: GrantBy.EMAIL, label: 'By email' });
            }

            DataikuAPI.security.listUsers().success(function(data) {
                const currentUserLogin = $rootScope.appConfig.login;
                $scope.allUsers = data
                    .filter(user => user.login !== currentUserLogin)
                    .sort((a, b) => a.displayName.localeCompare(b.displayName));
            });
        };

        $scope.grantAccess = function() {
            switch ($scope.selector.selectedGrantType) {
                case GrantBy.USER:
                    $scope.addUsersToInvitations($scope.selector.users);
                    $scope.selector.users = [];
                    break;
                case GrantBy.EMAIL:
                    if ($scope.shareModalForm.pendingUserEmailInput.$valid) {
                        $scope.addMailToInvitations($scope.selector.email);
                        $scope.selector.email = '';
                    }
                    break;
            }
        };

        $scope.isGrantAccessDisabled = function() {
            switch ($scope.selector.selectedGrantType) {
                case GrantBy.USER:
                    return !$scope.selector.users || $scope.selector.users.length === 0;
                case GrantBy.EMAIL:
                    return $scope.shareModalForm.pendingUserEmailInput.$invalid;
                default:
                    return true;
            }
        };

        $scope.handleEnterKey = function(event) {
            if (event.key === 'Enter') {
                event.preventDefault();
                if (!$scope.isGrantAccessDisabled()) {
                    $scope.grantAccess();
                }
            }
        };

        $scope.searchUser = function(searchTerms, user) {
            const normalizedSearch = normalizeTextForSearch(searchTerms);
            const allTokens = normalizeTextForSearch(`${user.displayName} ${user.login} ${user.email}`);
            return allTokens.includes(normalizedSearch);
        };

        $scope.addUsersToInvitations = (usersToAdd) => {
            usersToAdd.forEach(user => {
                const isAlreadyInvited = $scope.invitations.some(invite => invite.user && invite.user.login === user.login);

                if (!isAlreadyInvited) {
                    $scope.invitations.push({ user: user.login });
                }
            });
        };

        $scope.addMailToInvitations = (pendingEmail) => {
            if (!pendingEmail || !$rootScope.appConfig.permissionsByEmailEnabled) {
                return;
            }
            pendingEmail = pendingEmail.toLowerCase();

            const addExistingUsersIfNotPresent = (logins) => {
                logins.forEach(login => {
                    const matchingPermission = $scope.invitations.find(user => user.user === login);
                    if (!matchingPermission) {
                        $scope.invitations.push({ user: login });
                    };
                });
            };

            const addPendingEmailIfNotPresent = (pendingEmail) => {
                const existingPendingEmail = $scope.invitations.some(user => user != null && user.pendingUserEmail && user.pendingUserEmail.toLowerCase() === pendingEmail);
                if (existingPendingEmail) {
                    ActivityIndicator.warning('An invitation is already pending for this email');
                } else {
                    $scope.invitations.push({
                        pendingUserEmail: pendingEmail });
                }
            };

            /*
             * We only check for existing users if the current user can edit permissions
             * otherwise we just add the pending email directly because only mails are sent not permissions changed
             */
            if ($scope.canEditPermissions) {
                DataikuAPI.security.listUsersMatchingEmail( $scope.dashboard.projectKey, null, pendingEmail).success((matchingUserLogins) => {
                    if (matchingUserLogins.length) {
                        addExistingUsersIfNotPresent(matchingUserLogins);
                    } else {
                        addPendingEmailIfNotPresent(pendingEmail);
                    }
                }).error(setErrorInScope.bind($scope));
            } else {
                addPendingEmailIfNotPresent(pendingEmail);
            }
        };

        $scope.removeInvitation = (index) => {
            $scope.invitations.splice(index, 1);
        };

        $scope.save = function() {
            const invitedEmails = $scope.invitations.map(invitation => invitation.pendingUserEmail).filter(email => email);
            const invitedUsers = $scope.invitations.map(invitation => invitation.user).filter(user => user);
            return DataikuAPI.dashboards.share(
                $scope.dashboard.projectKey,
                $scope.dashboard.id,
                invitedEmails,
                invitedUsers,
                $scope.uiState.personalMessage
            ).success(function() {
                $scope.resolveModal();
            }).error(setErrorInScope.bind($scope));
        };
    });
})();
