(function(){
    'use strict';

    const app = angular.module('dataiku.dashboards.insights');

    app.constant('JupyterInsightHandler', {
        name: 'Notebook',
        desc: 'Code analysis',
        i18nNameKey: 'INSIGHTS.JUPYTER_NOTEBOOK.NAME',
        i18nDescKey: 'INSIGHTS.JUPYTER_NOTEBOOK.DESC',
        icon: 'icon-dku-nav_notebook',
        color: 'notebook',

        getSourceId: function(insight) {
            return insight.params.notebookSmartName;
        },
        sourceType: 'JUPYTER_NOTEBOOK',
        hasEditTab: false,
        defaultTileParams: {
            showCode: false
        },
        defaultTileDimensions: [12, 15]
    });

    app.controller('JupyterInsightCommonController', function($scope, $sce, DataikuAPI, $stateParams, $timeout) {

    	$scope.getLoadingPromise = function() {
        	if ($scope.insight.params.loadLast) {
        		return DataikuAPI.jupyterNotebooks.export.getLast($scope.insight.projectKey, $scope.insight.params.notebookSmartName);
        	} else {
        		return DataikuAPI.jupyterNotebooks.export.get($scope.insight.projectKey, $scope.insight.params.notebookSmartName, $scope.insight.params.exportTimestamp);
        	}
    	};

    	$scope.displayExport = function(html, showCode, pointer) {
    		if (html) {
    			$scope.exportNotFound = false;
            	$timeout(function() {
                    showHTML(html, showCode, pointer);
                }, 0);
            } else {
            	$scope.exportNotFound = true;
            }
    	};

    	function showHTML(html, showCode, pointer) {
            if (showCode != null) {
                // if one day this becomes too annoying one may consider using an html parser and pick up the element with id toggleCode
                html = html
                    // legacy nbconvert<7.7, see https://github.com/jupyter/nbconvert/pull/2021/files#diff-5d5d0e216c4c7e2bdbdce10cc1bac7804d432ee61e1643be8f880cd422d14cd0R260
                    .replace('<a id=\'toggleCode\' onclick=\'toggleCodeVisibility()\'>Show code</a>', '')
                    // new format
                    .replace('<a id="toggleCode" onclick="toggleCodeVisibility()">Show code</a>', '');
                if (showCode) {
                    html = html.replace('<body class="hideCode">', '<body>');
                } else {
                    html = html.replace('<body>', '<body class="hideCode">');
                }
                if (pointer) {
                    html = html.replace('<body', '<body style="cursor: pointer;" onload="_load()"');
                }
            }
            html = html.replace('</body>', '<style>div.output_subarea { max-width: none; } ::-webkit-scrollbar { -webkit-appearance: none; width: 5px; height: 7px; } ::-webkit-scrollbar-thumb { border-radius: 4px; background-color: rgba(0,0,0,.4); box-shadow: 0 0 1px rgba(255,255,255,.5); }</style></body>');
            $scope.jupyterInsightContent = $sce.trustAsHtml(html);
    	};
    });

    app.directive('jupyterInsightTile', function($stateParams, $timeout, DataikuAPI, $controller, DashboardUtils, TileLoadingState, translate){
        return {
            templateUrl: '/templates/dashboards/insights/jupyter/jupyter_tile.html',
            scope: {
                insight: '=',
                tile: '=',
                hook: '='
            },
            link: function($scope, element, attrs){
            	$controller('JupyterInsightCommonController', { $scope: $scope });

                let html;

                $scope.loaded = false;
            	$scope.loading = false;
                $scope.error = null;
                $scope.translate = translate;

                $scope.load = function(resolve, reject) {
                	$scope.loading = true;
                	const loadingPromise = $scope.getLoadingPromise().noSpinner();
                	//any case write in iframe to display html
                	loadingPromise
                        .success(DashboardUtils.setLoaded.bind([$scope, resolve]))
                        .success(function(data) {
                            html = data.html; $scope.displayExport(html, !!$scope.tile.tileParams.showCode, $scope.tile.clickAction != 'DO_NOTHING');

                            if ($scope.tile.clickAction != 'DO_NOTHING') {
                                /*
                                 * The following code aims to handle configurable click behavior on tile
                                 * It is broken for two reasons:
                                 * * the sandboxing prevents accessing the contentWindow: https://app.shortcut.com/dataiku/story/144229/resolve-xss-in-dashboards-related-to-jupyter-nb-r-markdown-reports-static-insight-file
                                 * * the main-click was removed: https://github.com/dataiku/dip/pull/17980/files#diff-b9d7e6d55478e7e6c5f1783bf22f061af476569967d0bde471ed49a7c99a88a8L59
                                 * On click on body, redirect event to main-click link
                                 */
                                $timeout(function() {
                                    element.find('iframe')[0].contentWindow._load = function() {
                                        element.find('iframe').contents().find('body').on('click', function(evt) {
                                            if(evt.originalEvent.preventRecursionMarker === 'norec') {
                                                // Prevent event recursion : do not handle this event if we generated it!
                                                return;
                                            }
                                            const cloneEvent = document.createEvent('MouseEvents');
                                            cloneEvent.preventRecursionMarker = 'norec';
                                            const e = evt.originalEvent;
                                            cloneEvent.initMouseEvent(e.type, e.bubbles, e.cancelable, window, e.detail,
                                                e.screenX, e.screenY, e.clientX, e.clientY, e.ctrlKey, e.altKey, e.shiftKey,
                                                e.metaKey, e.button, e.relatedTarget);
                                            element.closest('.tile-wrapper').find('[main-click]')[0].dispatchEvent(cloneEvent);
                                            e.stopPropagation();
                                        });
                                    };
                                });
                            }
                        })
                        .error(DashboardUtils.setError.bind([$scope, reject]));
                };
                $scope.hook.loadPromises[$scope.tile.$tileId] = $scope.load;
                $scope.hook.reloadPromises[$scope.tile.$tileId] = $scope.load;

                if ($scope.tile.autoLoad) {
                    $scope.hook.loadStates[$scope.tile.$tileId] = TileLoadingState.WAITING;
                }

                $scope.$watch('tile.tileParams.showCode', function(nv) {
                    if (nv == null || !$scope.loaded) {
                        return;
                    }
                    $scope.displayExport(html, !!nv);
                });
            }
        };
    });

    app.directive('jupyterInsightView', function($controller){
        return {
            templateUrl: '/templates/dashboards/insights/jupyter/jupyter_view.html',
            scope: {
                insight: '=',
                tileParams: '='
            },
            link: function($scope, element, attrs) {
                $controller('JupyterInsightCommonController', { $scope: $scope });

                const loadingPromise = $scope.getLoadingPromise();
            	//any case write in iframe to display html
            	loadingPromise.success(function(data) {
            		$scope.displayExport(data.html);
            	}).error(function(data, status, headers, config, statusText) {
                	setErrorInScope.bind($scope)(data, status, headers, config, statusText);
                });

            }
        };
    });

    app.directive('jupyterInsightTileParams', function(){
        return {
            templateUrl: '/templates/dashboards/insights/jupyter/jupyter_tile_params.html',
            scope: {
                tileParams: '='
            },
            link: function($scope, element, attrs){
            }
        };
    });


    app.directive('jupyterInsightCreateForm', function(DataikuAPI, $filter, $stateParams, translate) {
        return {
            templateUrl: '/templates/dashboards/insights/jupyter/jupyter_create_form.html',
            scope: true,
            link: function($scope, element, attrs){

                $scope.hook.defaultName = translate('DASHBOARD.NEW_TILE_MODAL.JUPYTER_NOTEBOOK', 'Jupyter notebook');
                $scope.$watch('hook.sourceObject', function(nv) {
                    if (!nv || !nv.label) {
                        return;
                    }
                    $scope.hook.defaultName = nv.label;
                });

            	$scope.insight.params.loadLast = true;

                $scope.facade = {
            		notebookSmartName : null,
                    availableExports : [],
                    createExport : $scope.canWriteProject()
                };

                $scope.setNotebook = function() {
                    if (!$scope.facade.notebookSmartName) {
                        return;
                    }
                	$scope.insight.params.notebookSmartName = $scope.facade.notebookSmartName;
                	$scope.facade.availableExports = $scope.notebookToExportsMap[$scope.facade.notebookSmartName];
                };

                $scope.$watch('facade.notebookSmartName', $scope.setNotebook);

                $scope.hook.beforeSave = function(resolve, reject) {
                	if ($scope.facade.createExport) {
                		DataikuAPI.jupyterNotebooks.export.create($stateParams.projectKey, $scope.insight.params.notebookSmartName)
                		.success(function(data) {
                			if (!$scope.insight.params.loadLast) {
                				$scope.insight.params.exportTimestamp = data.timestamp;
                			}
                			resolve();
                		})
                		.error(function(data, status, headers, config, statusText){
                        	reject(arguments);
                            });
                	} else {
                		resolve();
                	}
                };

                $scope.checkLoadLastAndTimestampConsistency = function() {
                	if (!$scope.insight.params.loadLast && !$scope.insight.params.exportTimestamp && !$scope.facade.createExport) {
                		$scope.newInsightForm.$setValidity('loadLastAndTimestampConsistency',false);
                	} else {
                		$scope.newInsightForm.$setValidity('loadLastAndTimestampConsistency',true);
                	}
                	return true;
                };

                $scope.formatDate = function(timestamp) {
                	return $filter('date')(timestamp, 'short');
                };

                $scope.resetTimestamp = function() {
                	$scope.insight.params.exportTimestamp = null;
                };

                DataikuAPI.jupyterNotebooks.mapNotebooksToExports($scope.insight.projectKey).success(function(data) {
                    $scope.notebookMap = data.first;
                    $scope.notebookToExportsMap = data.second;
                }).error($scope.hook.setErrorInModaleScope);
            }
        };
    });

    app.directive('jupyterInsightEdit', function($controller, DataikuAPI, $rootScope, Dialogs){
        return {
            templateUrl: '/templates/dashboards/insights/jupyter/jupyter_edit.html',
            scope: {
                insight: '='
            },
            link: function($scope, element, attrs) {
                $controller('JupyterInsightCommonController', { $scope: $scope });

                $scope.canWriteProject = $rootScope.topNav.isProjectAnalystRW;

                DataikuAPI.jupyterNotebooks.export.list($scope.insight.projectKey, $scope.insight.params.notebookSmartName).success(function(data) {
                    $scope.exports = data;
                });

                function refresh() {
                    if (!$scope.insight.params) {
                        return;
                    }
                    if (!$scope.insight.params.loadLast && !$scope.insight.params.exportTimestamp) {
                        $scope.insight.params.exportTimestamp = $scope.exports[0].timestamp;
                    }

                    const loadingPromise = $scope.getLoadingPromise();
                    //any case write in iframe to display html
                    loadingPromise.success(function(data) {
                        $scope.displayExport(data.html);
                    }).error(function(data, status, headers, config, statusText) {
                        setErrorInScope.bind($scope)(data, status, headers, config, statusText);
                    });
                }

                $scope.$watch('insight.params', refresh, true);

                $scope.createNewExport = function() {
                    Dialogs.confirmPositive($scope, 'Export Jupyter notebook',
                        'Create a new export of this Jupyter notebook? Note that it will not rerun the code of this notebook '+
                        'and will use the last saved state. To rerun the notebook, go to the notebook or use a DSS scenario').then(function(){
                        DataikuAPI.jupyterNotebooks.export.create($scope.insight.projectKey, $scope.insight.params.notebookSmartName)
                            .success(function(data) {
                                if (!$scope.insight.params.loadLast) {
                                    $scope.insight.params.exportTimestamp = data.timestamp;
                                }
                                refresh();
                                $scope.exports.unshift(data);
                            }).error(function(data, status, headers, config, statusText) {
                                setErrorInScope.bind($scope)(data, status, headers, config, statusText);
                            });
                    });
                };
            }
        };
    });

})();
