(function () {
  "use strict";

  const app = angular.module("dataiku.savedmodels");

  // templates must be tested in test_code_agents.py::TestCodeAgentsCodeTemplates
  app.constant("AgentCodeTemplates", {
    "simple-llm-mesh-proxy-streamed": {
      title: "Simple LLM Mesh integration with streaming",
      subtitle: "The response is streamed as it gets generated",
      description: "Starter implementation of an agent that performs a single query to an LLM from the LLM Mesh.\n\nThe agent takes a conversation and streams a response.",
      codeSample: `import dataiku
from dataiku.llm.python import BaseLLM
from dataikuapi.dss.llm import DSSLLMStreamedCompletionChunk, DSSLLMStreamedCompletionFooter

OPENAI_CONNECTION_NAME = "REPLACE_BY_YOUR_OPENAI_CONNECTION_NAME"

class MyLLM(BaseLLM):
    def __init__(self):
        pass

    def process_stream(self, query, settings, trace):
        llm = dataiku.api_client().get_default_project().get_llm(f"openai:{OPENAI_CONNECTION_NAME}:gpt-5-mini")
        completion = llm.new_completion().with_message("The user wrote this. What do you think about this question?", "system")
        for message in query["messages"]:
            if message.get("content"):
                completion = completion.with_message(message["content"], message.get("role", "user"))
        completion.settings.update(settings)

        for chunk in completion.execute_streamed():
            if isinstance(chunk, DSSLLMStreamedCompletionChunk):
                yield {"chunk": {"text": chunk.text}}
            elif isinstance(chunk, DSSLLMStreamedCompletionFooter):
                yield {"footer": chunk.data}
    `,
    },
    "simple-llm-mesh-proxy-non-streamed": {
      title: "Simple LLM Mesh integration without streaming",
      subtitle: "The response is returned once fully generated",
      description: "Starter implementation of an agent that performs a single query to an LLM from the LLM Mesh.\n\nThe agent takes a conversation and returns a response.",
      codeSample: `import dataiku
from dataiku.llm.python import BaseLLM
from dataikuapi.dss.llm import DSSLLMStreamedCompletionChunk, DSSLLMStreamedCompletionFooter

OPENAI_CONNECTION_NAME = "REPLACE_BY_YOUR_OPENAI_CONNECTION_NAME"

class MyLLM(BaseLLM):
    def __init__(self):
        pass

    def process(self, query, settings, trace):
        llm = dataiku.api_client().get_default_project().get_llm(f"openai:{OPENAI_CONNECTION_NAME}:gpt-5-mini")
        completion = llm.new_completion().with_message("The user wrote this. What do you think about this question?", "system")
        for message in query["messages"]:
            if message.get("content"):
                completion = completion.with_message(message["content"], message.get("role", "user"))
        completion.settings.update(settings)
        llm_resp = completion.execute()

        resp_text = "I am a starter agent. Let's see what the LLM thinks about your messages"
        resp_text = resp_text + "\\n" + llm_resp.text
        return {"text": resp_text}
    `,
      },
    "async-agent-streamed": {
      title: "Asynchronous agent with streaming",
      subtitle: "The response is streamed as it gets generated",
      description: "Starter implementation of an async agent.\n\nCan be useful to integrate with asynchronous libraries.",
      codeSample: `import asyncio
from dataiku.llm.python import BaseLLM

class MyLLM(BaseLLM):
    def __init__(self):
        pass

    async def aprocess_stream(self, query, settings, trace):
        prompt = query["messages"][-1]["content"]

        await asyncio.sleep(1)
        yield {
            "chunk": {
                "text" : "Hello I am a streamed agent\\n"
            }
        }
        await asyncio.sleep(1)
        yield {
            "chunk": {
                "text" : "Your last message was: %s\\n" % prompt
            }
        }
        await asyncio.sleep(1)
        yield {
            "chunk": {
                "text" : "I should think about how to respond\\n"
            }
        }
    `,
    },
    "async-agent-non-streamed": {
      title: "Asynchronous agent without streaming",
      subtitle: "The response is returned once fully generated",
      description: "Starter implementation of an async agent.\n\nCan be useful to integrate with asynchronous libraries.",
      codeSample: `import asyncio
from dataiku.llm.python import BaseLLM

class MyLLM(BaseLLM):
    def __init__(self):
        pass

    async def aprocess(self, query, settings, trace):
        prompt = query["messages"][-1]["content"]

        resp_text = "I am an async agent, and your last message was: %s. What do you want to do next?" % prompt
        await asyncio.sleep(1)

        return {"text": resp_text}
    `,
      },
    "langchain-basic-tools-non-streamed": {
      title: "Simple tool-calling agent",
      subtitle: "With a custom trace",
      description: "Starter code of a tool-calling agent that leverages functions to enhance its abilities, using the LLM Mesh and basic Langchain.\n\nA custom trace is returned, alongside the agent response.",
      codeSample: `from langchain_core.tools import tool
from langchain_core.messages import HumanMessage, ToolMessage

import dataiku
from dataiku.llm.python import BaseLLM

OPENAI_CONNECTION_NAME = "REPLACE_BY_YOUR_OPENAI_CONNECTION_NAME"

@tool
def add(a: int, b: int) -> int:
    """Adds a and b."""
    return a + b

@tool
def multiply(a: int, b: int) -> int:
    """Multiplies a and b."""
    return a * b

tools = [add, multiply]

# add a visual tool
project = dataiku.api_client().get_default_project()
project_visual_tools = project.list_agent_tools()

def find_tool(name: str) -> object:
    for tool in project_visual_tools:
        if tool["name"] == name:
            return project.get_agent_tool(tool['id'])
    return None

visual_tool = find_tool("Calculator") # use any visual tool by name
if visual_tool:
    tools.append(visual_tool.as_langchain_structured_tool())


class MyLLM(BaseLLM):
    def __init__(self):
        pass

    def process(self, query, settings, trace):
        llm = dataiku.api_client().get_default_project().get_llm(f"openai:{OPENAI_CONNECTION_NAME}:gpt-5-mini").as_langchain_chat_model(completion_settings=settings)
        llm_with_tools = llm.bind_tools(tools)

        messages = [m for m in query["messages"] if m.get("content")]
        iterations = 0
        while True:
            iterations += 1
            
            if iterations < 10:
                with trace.subspan("Invoke LLM with tools") as llm_invoke_span:
                    llm_response = llm_with_tools.invoke(messages)
            else:
                with trace.subspan("Invoke LLM without tools") as llm_invoke_span:
                    llm_response = llm.invoke(messages)

            if len(llm_response.tool_calls) == 0:
                return {"text": llm_response.content}
            
            with llm_invoke_span.subspan("Call the tools") as tools_subspan:
                messages.append(llm_response)
                for tool_call in llm_response.tool_calls:
                    with tools_subspan.subspan("Call a tool") as tool_subspan:
                        tool_subspan.attributes["tool_name"] = tool_call["name"]
                        tool_subspan.attributes["tool_args"] = tool_call["args"]
                        if tool_call["name"] == "add":
                            tool_output = add.invoke(tool_call["args"])
                        elif tool_call["name"] == "multiply":
                            tool_output = multiply.invoke(tool_call["args"])
                        elif visual_tool:
                            tool_output = visual_tool.run(tool_call["args"])
                    messages.append(ToolMessage(tool_call_id =tool_call["id"], content=tool_output))
    `,
    },
    "langchain-complete-streamed": {
      title: 'Complete Langchain agent',
      subtitle: "With streamed implementation and a custom trace",
      description:"Starter implementation of an asynchronous streaming agent that leverages comprehensive tool-calling, using the LLM Mesh and full Langchain capabilities.\n\nIncludes full tracing.",
      codeSample: `import random

try:
    from langchain_classic import hub
    from langchain_classic.agents import AgentExecutor, create_openai_tools_agent
except ModuleNotFoundError:
    from langchain import hub
    from langchain.agents import AgentExecutor, create_openai_tools_agent
from langchain.tools import tool
from langchain_core.callbacks import Callbacks

import dataiku
from dataiku.llm.python import BaseLLM
from dataiku.langchain import LangchainToDKUTracer, dku_span_builder_for_callbacks

OPENAI_CONNECTION_NAME = "REPLACE_BY_YOUR_OPENAI_CONNECTION_NAME"
model = dataiku.api_client().get_default_project().get_llm(f"openai:{OPENAI_CONNECTION_NAME}:gpt-5-mini").as_langchain_chat_model()

@tool
async def where_cat_is_hiding(cat_name: str, callbacks: Callbacks = None) -> str:
    """Where is the cat hiding right now?"""

    with dku_span_builder_for_callbacks(callbacks).subspan("Verifying cat location...") as s:
        return random.choice(["under the bed", "on the shelf"])

@tool
async def get_items(place: str) -> str:
    """Use this tool to look up which items are in the given place."""
    if "bed" in place:  # For 'under the bed'
        return "socks, shoes and dust bunnies"
    if "shelf" in place:  # For 'on the shelf'
        return "books, pencils and pictures"
    else:  # if the agent decides to ask about a different place
        return "cat snacks"

tools = [get_items, where_cat_is_hiding]
agent = create_openai_tools_agent(model.with_config({"tags": ["agent_llm"]}), tools,  hub.pull("hwchase17/openai-tools-agent"))

class MyLLM(BaseLLM):
    def __init__(self):
        pass

    async def aprocess_stream(self, query, settings, trace):
        tracer = LangchainToDKUTracer(dku_trace=trace)
        agent_executor = AgentExecutor(agent=agent, tools=tools)

        async for event in agent_executor.astream_events({"input": query["messages"]}, version="v2", config={"callbacks": [tracer]}):
            kind = event["event"]
            if kind == "on_chat_model_stream":
                content = event["data"]["chunk"].content
                if content:
                    yield {"chunk":  {"text": content}}
            elif kind == "on_tool_start":
                # Event chunks are not part of the answer itself, but can provide progress information
                yield {"chunk":  {"type": "event", "eventKind": "tool_call", "eventData": {"name": event["name"]}}}        
    `,
    },
    "langgraph-router-tools": {
      title: 'LangGraph routing agent',
      subtitle: "With multi-agent logic and expert delegation",
      description: "This agent uses a classifier node as its entry point then routes the task to either a tool-calling loop or to a separate \"expert\" LLM (for complex, general-purpose questions).",
      codeSample: `
import json

from langgraph.graph import StateGraph, END, MessagesState
from langgraph.prebuilt import ToolNode
from langchain_core.tools import tool
from langchain_core.messages import HumanMessage, SystemMessage

import dataiku
from dataiku.llm.python import BaseLLM
from dataiku.langchain import LangchainToDKUTracer

OPENAI_CONNECTION_NAME = "REPLACE_BY_YOUR_OPENAI_CONNECTION_NAME"
ROUTER_LLM_ID = f"openai:{OPENAI_CONNECTION_NAME}:gpt-5-chat-latest"
TOOLS_LLM_ID = f"openai:{OPENAI_CONNECTION_NAME}:gpt-5-mini"
EXPERT_LLM_ID = f"openai:{OPENAI_CONNECTION_NAME}:gpt-5"


@tool
def search_car_database(query: str):
    """
    Searches a database for information about car sales, prices, or models.
    Use this tool for any specific questions about car data.
    """
    print(f"--- Tool Used: search_car_database(query='{query}') ---")
    query = query.lower()
    if "dealer" in query and "chicago" in query:
        return json.dumps(
            {"dealer": "Windy City Auto", "total_sales": 150, "top_model": "Sedan"}
        )
    else:
        return json.dumps(
            {"error": "No specific information found for that query."}
        )


tools = [search_car_database]
tool_node = ToolNode(tools)

router_llm = dataiku.api_client().get_default_project().get_llm(ROUTER_LLM_ID).as_langchain_chat_model()
expert_llm = dataiku.api_client().get_default_project().get_llm(EXPERT_LLM_ID).as_langchain_chat_model()
tools_llm = dataiku.api_client().get_default_project().get_llm(TOOLS_LLM_ID).as_langchain_chat_model()
llm_with_tools = tools_llm.bind_tools(tools)


def call_router_llm(state):
    """
    First node to classify the user's intent.
    Does the query require a tool, or is it a complex question for an expert?
    Returns the name of the next node to call.
    """
    print("--- Classifying query ---")
    messages = state["messages"]
    last_human_message = ""
    for msg in reversed(messages):
        if type(msg) == HumanMessage:
            last_human_message = msg.content or ""
            break

    # Create a router prompt
    router_prompt = [
        SystemMessage(
            content="You are an expert router. Your job is to classify the user's query. "
            f"The user's query is: '{last_human_message}'\\n\\n"
            "Based on this query, decide if it can be answered with a simple tool "
            f"that can search a car database (tool schema: {search_car_database.args_schema.schema()}), "
            "or if it requires a complex, expert-level response (e.g., 'what is the future of electric vehicles?').\\n\\n"
            "Respond with only the single word 'TOOL' or 'EXPERT'."
        ),
    ]
    
    response = router_llm.invoke(router_prompt)
    return {"messages": [response]}


def route_call(state):
    """
    Router function. Checks whether to use an agent with tools or an expert LLM
    """
    print("--- Checking where to route the query ---")
    decision = state["messages"][-1].content.strip().upper()
    print(f"--- Router decision: {decision} ---")
    if "TOOL" in decision:
        return "call_llm_with_tools"
    else:
        return "call_expert_llm"
    
    
def call_expert_llm(state):
    """
    Calls the LLM with an "expert" persona for complex questions.
    """
    print("--- Calling Expert LLM ---")
    messages = state["messages"]
    expert_messages = [
        {"role": "system", "content": "You are a world-class automotive industry expert. "
         "Provide a detailed, insightful, and comprehensive answer to the user's query."}
    ] + messages[:-1]
    
    response = expert_llm.invoke(expert_messages)
    return {"messages": [response]}


def call_llm_with_tools(state):
    """
    Calls the LLM with the current message history.
    The LLM's response will either be a final answer or a tool call.
    """
    print("--- Calling LLM ---")
    messages = state["messages"]
    response = llm_with_tools.invoke(messages[:-1])
    return {"messages": [response]}


def should_call_tool(state):
    """
    Router function. Checks the last message to see if it contains a tool call.
    """
    print("--- Checking for tool call ---")
    last_message = state["messages"][-1]
    if last_message.tool_calls:
        print("--- Decision: Call tool ---")
        return "call_tool"
    else:
        print("--- Decision: End session ---")
        return END


class ToolAgentLLM(BaseLLM):
    def __init__(self):
        graph = StateGraph(MessagesState)
        graph.add_node("router_llm", call_router_llm)
        graph.add_conditional_edges("router_llm", route_call)
        graph.set_entry_point("router_llm")
        graph.add_node("call_llm_with_tools", call_llm_with_tools)
        graph.add_conditional_edges("call_llm_with_tools", should_call_tool)
        graph.add_node("call_tool", tool_node)
        graph.add_node("call_expert_llm", call_expert_llm)
        graph.add_edge("call_expert_llm", END)
        self.graph = graph.compile()

    def process(self, query, settings, trace):
        tracer = LangchainToDKUTracer(dku_trace=trace)
        initial_state = {"messages": query["messages"]}
        result = self.graph.invoke(
            initial_state, 
            config={"callbacks": [tracer]}
        )
        final_message = result["messages"][-1]
        return {"text": final_message.content}
    `,
    },
    "multimodal-agent": {
      title: "Multimodal agent",
      subtitle: "With text & image prompting",
      description:"Starter code of a multimodal agent that takes text and images as input. Can be used for analyzing, understanding, or retrieving information from documents and other data sources.",
      codeSample: `import dataiku
from dataiku.llm.python import BaseLLM

OPENAI_CONNECTION_NAME = "REPLACE_BY_YOUR_OPENAI_CONNECTION_NAME"

class MyLLM(BaseLLM):
    def __init__(self):
        pass
    
    def process(self, query, settings, trace):
        try:
            text, img_b64 = self._extract_text_image_base64(query)
        except ValueError as e:
            return {"text": f"I'm sorry, but my job is to process an image according to your question. Please respect the input format: {e}"}

        llm = dataiku.api_client().get_default_project().get_llm(f"openai:{OPENAI_CONNECTION_NAME}:gpt-5-mini")
        completion = llm.new_completion()

        #1 System Message
        completion.with_message("Please answer the questions with artistic sensitivity", "system")
        
        #2 Create multipart message
        mp_message = completion.new_multipart_message('user')
        # Task description
        mp_message.with_text(text)
        # Image
        mp_message.with_inline_image(img_b64)
        # Add it to the completion request
        mp_message.add()
        
        llm_resp = completion.execute()

        return {"text": llm_resp.text}
    
    def _extract_text_image_base64(self, query):
        messages = query.get('messages')
        if not messages:
            raise ValueError("No messages found in the query.")

        parts = messages[-1].get('parts')
        if not parts:
            raise ValueError("No parts found in the first message.")

        # Initialize placeholders
        img_b64 = None
        text = None

        # Loop through parts to find both IMAGE_INLINE and TEXT
        for part in parts:
            part_type = part.get('type')
            if not img_b64 and part_type == 'IMAGE_INLINE':
                img_b64 = part.get('inlineImage')
            elif not text and part_type == 'TEXT':
                text = part.get('text')

        # Check if both were found
        if not img_b64:
            raise ValueError("Inline image not found in the query parts.")
        if not text:
            raise ValueError("Text not found in the query parts.")

        return text, img_b64

    `,
    quickTestQuery:{
        "messages": [
           {
              "role": "user",
              "parts": [
                 {
                    "type": "IMAGE_INLINE",
                    "inlineImage": "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"
                 },
                 {
                    "type": "TEXT",
                    "text": "Please provide a description of the provided image in 3 parts: color, theme and text."
                 }
              ]
           }
        ],
        'context': {}
     },
     supportsImageInputs: true,     
    },
    "minimal-agent": {
      title: "Minimal agent template",
      subtitle: "To start from scratch",
      description:"Lightweight starter code with just the mandatory class and function to override when creating an agent from the ground up.",
      codeSample: `import dataiku
from dataiku.llm.python import BaseLLM

# Add code here to define tools for instance

class MyLLM(BaseLLM):
    def __init__(self):
        pass

    def process(self, query, settings, trace):
        prompt = query["messages"]

        # Add code here, for LLM completion, tool execution, etc...

        return {"text": "the agent answer"}
    `,
    },
  });

  app.constant("AgentToolCodeTemplates", {
    "empty-template": {
      title: "Empty template",
      description: "The basic structure of the agent tool class, with empty implementations for the required methods",
      codeSample: `import dataiku
from dataiku.llm.agent_tools import BaseAgentTool

class MyAgentTool(BaseAgentTool):
    """An empty interface for a code-based agent tool"""

    def __init__(self):
        pass

    def get_descriptor(self, tool):
        """
        Returns the descriptor of the tool, as a dict containing:
           - description (str)
           - inputSchema (dict, a JSON Schema representation)
        """
        return {
            "description": "",
            "inputSchema" : {
                "$id": "",
                "title": "",
                "type": "object",
                "properties": {},
                "required": []
            }
        }


    def invoke(self, input, trace):
        """
        Invokes the tool.

        The arguments of the tool invocation are in input["input"], a dict
        """
        return {
            "output": "",
            "sources": [],
        }
`,
    },
    "hashing-example": {
      title: "Hashing example",
      description: "A simple example tool that returns the hash of its input",
      codeSample: `import dataiku
from dataiku.llm.agent_tools import BaseAgentTool

class HashingAgentTool(BaseAgentTool):
    def get_descriptor(self, tool):
        return {
            "description": "Hashes a string. Returns the SHA1 hash of the string.",
            "inputSchema" : {
                "$id": "https://example.com/agents/tools/hash/input",
                "title": "Input for the hashing tool",
                "type": "object",
                "properties" : {
                    "payload" : {
                        "type": "string",
                        "description": "The payload to hash"
                    }
                },
                "required": ["payload"]
            }
        }

    def invoke(self, input, trace):
        args = input["input"]
        payload = args["payload"]

        import hashlib
        sha1_hash = hashlib.sha1()
        sha1_hash.update(payload.encode('utf-8'))
        hashed = sha1_hash.hexdigest()

        return {
            "output": hashed,
            "sources":  [{
                "toolCallDescription": "Payload was hashed"
            }]
        }
`,
    },
  });
})();
