(function() {
'use strict';

const app = angular.module('dataiku.services', [ 'dataiku.constants' ]);

// Mapping DOMRect to Object for better usability
function getBoundingClientRect(element) {
    const rect = element.getBoundingClientRect();
    return {
        top: rect.top,
        right: rect.right,
        bottom: rect.bottom,
        left: rect.left,
        width: rect.width,
        height: rect.height,
        x: rect.x,
        y: rect.y
    };
}

app.factory("DKUConstants", function($rootScope) {
    const cst = {
        ARCHIVED_PROJECT_STATUS: "Archived",
        design: {
            alertClasses: {
                SUCCESS: 'alert-success',
                FATAL: 'alert-danger',
                ERROR: 'alert-danger',
                WARNING: 'alert-warning',
                INFO: 'alert-info'
            }
        }
    };
    $rootScope.DKUConstants = cst;

    return cst;
});


app.service('AppConfig', function (Logger) {
    let config;
    this.set = function(cfg) {
        Logger.info('Set appConfig')
        config = cfg;
    };
    this.get = function() {
        return config;
    }
});

app.service('FeatureFlagsService', function($rootScope) {
    this.featureFlagEnabled = function(flagName) {
        return $rootScope.appConfig && $rootScope.appConfig.featureFlags.includes(flagName);
    };

    // put in rootScope for easy use in templates
    $rootScope.featureFlagEnabled = this.featureFlagEnabled;
});

app.service('uiCustomizationService', ['$rootScope', 'DataikuAPI', '$q', function($rootScope, DataikuAPI, $q) {
    const datasetTypeStatus = { HIDE: 'HIDE', SHOW: 'SHOW', NO_CONNECTION: 'NO_CONNECTION'};
    let computeDatasetTypeStatusCache = {}; // this is project-dependant, we cache for each project

    this.datasetTypeStatus = datasetTypeStatus;

    // returns a promise resolving into a function (type) => datasetTypeStatus
    this.getComputeDatasetTypesStatus = (scope, projectKey) => {
        if(! computeDatasetTypeStatusCache[projectKey]) {
            const deferred = $q.defer();
            DataikuAPI.datasets.listCreatableDatasetTypes(projectKey).success((datasetTypes) => {
                const uiCustomization = $rootScope.appConfig.uiCustomization; // just a shortcut

                deferred.resolve((type) => {
                    const typeIsIn = (array) => array.indexOf(type) !== -1;

                    // Hive is always hidden if showTraditionalHadoop is unchecked
                    if(!uiCustomization.showTraditionalHadoop && type === 'hiveserver2') {
                        return datasetTypeStatus.HIDE;
                    }

                    if(uiCustomization.hideDatasetTypes && uiCustomization.hideDatasetTypes.length !== 0) {
                        // we hide datasets that are blackListed except if they are allowed by personal connections
                        const isBlackListed = typeIsIn(uiCustomization.hideDatasetTypes);
                        const fromPersonalConnections = typeIsIn(datasetTypes.fromPersonalConnections);
                        if(isBlackListed && !fromPersonalConnections) {
                            return datasetTypeStatus.HIDE;
                        }
                    }

                    if(! uiCustomization.showDatasetTypesForWhichThereIsNoConnection) {
                        // we hide / disable dataset that have no adequate connection available
                        let hasConnection = typeIsIn(datasetTypes.fromAllConnections) || typeIsIn(datasetTypes.outsideOfConnections);

                        // if user is admin we consider all connections are available since he could create them (except UploadedFiles)
                        if($rootScope.isDSSAdmin() && type !== 'UploadedFiles') {
                            hasConnection = true;
                        }
                        // if user can create personal connections, we consider all connections (except UploadedFiles, Filesystem and hiveserver2) are available since he could create them
                        if($rootScope.appConfig.globalPermissions.mayCreateAuthenticatedConnections
                                && type !== 'UploadedFiles' && type !== 'Filesystem' && type !== 'hiveserver2') {
                            hasConnection = true;
                        }

                        if(!hasConnection) {
                            return datasetTypeStatus.NO_CONNECTION;
                        }

                    }

                    return datasetTypeStatus.SHOW;
                });
            }).error(setErrorInScope.bind(scope))
            .catch(() => {
                // In case of API error, we resolve the promise with a filter that will show every dataset type
                // not ideal, but better than showing nothing
                deferred.resolve(() => datasetTypeStatus.SHOW);
            });
            computeDatasetTypeStatusCache[projectKey] = deferred.promise;
        }
        return computeDatasetTypeStatusCache[projectKey];
    }

    // clear the cache when settings are changed.
    $rootScope.$watch(() => $rootScope.appConfig.uiCustomization, () => computeDatasetTypeStatusCache = {});
}]);

    /**
     * ActiveProjectKey is intended to replace explicit references to $stateParams.projectKey references
     * that litter the current code base.  These references make it impossible to use many directives outside
     * of an opened project.
     * The service get() method will return $stateParams.projectKey whenever possible, but when this is not
     * defined it will return a value previously saved via the set() method.
     */
    app.service('ActiveProjectKey', function($stateParams) {
    let explicitProjectKey;
    this.set = function(projectKey) {
        explicitProjectKey = projectKey;
    };
    this.get = function() {
        let key = $stateParams.projectKey;
        if (typeof(key) === 'undefined') key = explicitProjectKey;
        return key;
    }
});

app.factory("TableChangePropagator", function() {
    var svc = {seqId: 0};
    svc.update = function() {
        svc.seqId++;
    };
    return svc;
});

app.factory("executeWithInstantDigest", function($timeout) {
    return function executeWithInstantDigest(fn, scope) {
        return $timeout(function(){
            scope.$apply(fn);
        });
    }
});




app.factory("textSize", function() {
    // if text is a string returns its length
    // if text is a list of texts returns the maximum width.
    // if container is not undefined, append it to container
    // element will be added and then remove before any repaint.
    return function(texts, tagName, className, container) {
        if (texts.length === undefined) {
            texts = [texts];
        }
        if (className === undefined) {
            className = "";
        }
        className += " text-sizer";
        if (container === undefined) {
            container = $("body");
        }
        else {
            container = $(container);
        }
        var maxWidth = 0;
        var elements = [];
        for (let i = 0; i < texts.length; i++) {
            let text = texts[i];
            let el = document.createElement(tagName);
            el.className = className;
            let $el = $(el);
            $el.text(text);
            container.append($el);
            elements.push($el);
        }
        ;
        for (let i = 0; i < elements.length; i++) {
            let $el = elements[i];
            maxWidth = Math.max(maxWidth, $el.width());
            $el.remove();
        }
        ;
        return maxWidth;
    }
});


app.factory("CollectionFiltering", function(Fn) {
    // service to filter collection with complex filters
    // i.e. magic around $filter('filter')

    // usage : CollectionFiltering.filter([obj,obj,obj], filterQuery, params)

    // filterQuery is an object that's matched against objects from the list
    // JS objects : {} means a logical AND
    // JS arrays : [] means a logical OR
    // string/Regex properties are then matched against each others with a substr logic

    // ** Parameters **
    // * filterQuery.userQuery : special str property that's matched against all properties of the object
    // *        this filter is cumulative (AND behavior) with regard to other fields in filterQuery,
    // *        but has an OR behavior with regard to the object properties (only needs to match a single field to validate)
    // *        property:value strings are extracted from filterQuery.userQuery and added to the filterQuery object
    // * params.userQueryTargets : single (or list of) dotted property paths to restrict the match of userQuery
    // *                            (except specified property:value patterns in the userQuery, which will bypass this restriction)
    // * params.propertyRules : if you want to give the user shortcuts for properties (replace dict)
    // *                        for example, if propertyRules = { 'name': 'smartId' }, a userQuery 'name:toto' would look for toto in the smartId field
    // * params.exactMatch: array of keys for which the search will require an exact match

    // params are enriched with :
    // * params.userQueryResult : a list of Regex+str that where used in the matching (for highlight in list)
    // * params.userQueryErrors : a dict : {str:errMessage} that occurred during parsing

    /**
     * Returns true if the query and obj matches
     *
     * @param query the search term to look for in the object (can be a regex or string)
     * @param obj the object to match
     * @param exactMatch if the string comparison requires an exact match
     *
     * @return {Boolean} true if the search term matches the object
     */
    var matchString = function(query, obj, exactMatch) {
        if (query instanceof RegExp) {
            return query.test(obj);
        } else {
            return !exactMatch ? ('' + obj).toLowerCase().indexOf(query) > -1 : ('' + obj).toLowerCase() === query;
        }
    }

    /**
     * Returns true if the searchTerm matches any allowed property from the specified object
     *
     * @param searchTerm the search term to look for in the object properties
     * @param obj the object on which properties to look
     * @param searchInFields list of allowed fields to search in (absolute paths). if undefined, allows any path
     * @param currentPath current absolute path of the object being tested (used to compare with the allowed paths when matching)
     * @return {Boolean} true if the search term matches any property from the object
     */
    var searchTermMatchesAnyProperty = function(searchTerm, obj, searchInFields, currentPath = '') {
        if (Array.isArray(obj)) {
            // ok if any of the array item matches. array dereference does not 'consume' path
            return obj.some(item => searchTermMatchesAnyProperty(searchTerm, item, searchInFields, currentPath));
        } else if ($.isPlainObject(obj)) {
            // ok if any of the field matches, but if the subpath matches the allowed paths.
            return Object.keys(obj)
                .filter((objKey) => objKey.charAt(0) !== '$' && hasOwnProperty.call(obj, objKey))
                .some((objKey) => {
                    const newPath = currentPath + '.' + objKey;
                    return searchTermMatchesAnyProperty(searchTerm, obj[objKey], searchInFields, newPath);
                });
        } else if (searchTerm instanceof RegExp || ((typeof(searchTerm) === "string" || typeof(searchTerm) === "number") && searchTerm !== "")) {
            return (searchInFields === undefined || searchInFields.some(allowedPath => currentPath.startsWith('.'+allowedPath))) // startsWith because if the match is deep under an allowed path, it's ok
                && matchString(searchTerm, obj);
        } else {
            return true;
        }
    };

    /**
     * Returns true if each word the query matches any property from the specified object (with an AND behaviour)
     * For special syntax (i.e. when you search for tag:tag_name) it must match any value for that specific property
     *
     * @param query an object containing the prepared and clean query of the user
     *        The object usually looks something like :
     *          {
     *             userQuery: '',
     *             tags: [],
     *             interest: {
     *                 starred: '',
     *             },
     *          }
     * @param params the filter params for that query. Contains the properties we want to target, those on which we want an exact match, property shortcuts, etc.
     * @param obj the object we want to search on
     * @param queryArrayIsAND if true and the query is an array, all terms must match a property of obj to return true
     * @param exactMatch true if the two compared strings must be exact match
     *
     * @return {Boolean} true if the query matches any property from the specified object
     */
    var filterFilter = function(query, params, obj, queryArrayIsAND, exactMatch) {
        if (Array.isArray(query)) {
            if (query.length == 0) {
                return true
            }
            if (queryArrayIsAND) {
                return query.every(searchTerm => filterFilter(searchTerm, params, obj, undefined, exactMatch));
            } else {
                return query.some(searchTerm => filterFilter(searchTerm, params, obj, undefined, exactMatch));
            }
        } else if (Array.isArray(obj)) {
            if (obj.length == 0 && query != "") {
                return false;
            }
            return obj.some(value => filterFilter(query, params, value, undefined, exactMatch));
        } else if ($.isPlainObject(obj) || $.isPlainObject(query)) {
            if (!$.isPlainObject(obj)) {
                return false
            }
            if (!$.isPlainObject(query)) {
                return true
            }

            // $requiredSearchTerms must all appear in the object, though not necessarily all in the same field
            if (query.$requiredSearchTerms) {
                const userQueryTargets = params.userQueryTargets
                        ? Array.isArray(params.userQueryTargets) ? params.userQueryTargets : [params.userQueryTargets]
                        : undefined;
                if(!query.$requiredSearchTerms.every(
                    searchTerm => searchTermMatchesAnyProperty(searchTerm, obj, userQueryTargets)
                )) {
                    return false;
                }
            }

            for (var objKey in query) {
                const requiresExactMatch = params.exactMatch && params.exactMatch.includes(objKey);

                if (objKey.charAt(0) !== '$' && objKey !== 'userQuery') {
                    const queryIsNOT = objKey.endsWith("__not");
                    const queryIsAND = objKey.endsWith("__and");
                    if (!queryIsNOT && !queryIsAND
                        && !filterFilter(query[objKey], params, Fn.propStr(objKey)(obj), undefined, requiresExactMatch)) {
                        return false;
                    }
                    if (!queryIsNOT && queryIsAND) {
                        const objValue = Fn.propStr(objKey.substr(0,objKey.length-5))(obj)
                        if (!objValue || !filterFilter(query[objKey], params, objValue, true, requiresExactMatch)) {
                            return false;
                        }
                    }
                    if (queryIsNOT && !queryIsAND) {
                        const objValue = Fn.propStr(objKey.substr(0,objKey.length-5))(obj)
                        if (objValue && filterFilter(query[objKey], params, objValue, undefined, requiresExactMatch)) {
                            return false;
                        }
                    }
                }
            }
            return true;
        } else if (query instanceof RegExp || ((typeof(query) === "string" || typeof(query) === "number") && query !== "")) {
            return matchString(query, obj, exactMatch);
        } else {
            return true;
        }
    };

    var prepareStringKey = function(params, a) {
        if (a.endsWith("/") && a.startsWith("/") && a.length > 2) {
            try {
                return new RegExp(a.substr(1, a.length - 2));
            } catch (err) {
                if (params) {
                    params.userQueryErrors = params.userQueryErrors || {};
                    params.userQueryErrors[a] = err;
                }
                return a;
            }
        } else {
            return a.toLowerCase();
        }
    };

    var cleanFilterQuery = function(obj, params) {
        if ($.isPlainObject(obj)) {
            for (let k in obj) {
                obj[k] = cleanFilterQuery(obj[k], params);
                if (($.isEmptyObject(obj[k]) && !(obj[k] instanceof RegExp) && (typeof obj[k] !== 'boolean')) || obj[k] === "") {
                    delete obj[k]
                }
            }
        } else if ($.isArray(obj)) {
            for (let k = obj.length - 1; k >= 0; k--) {
                obj[k] = cleanFilterQuery(obj[k], params);
                if (($.isEmptyObject(obj[k]) && !(obj[k] instanceof RegExp) && (typeof obj[k] !== 'boolean')) || obj[k] === "") {
                    obj.splice(k, 1)
                }
            }
        } else if (typeof(obj) === "string" || typeof(obj) === "number" || typeof(obj) === "boolean") {
            obj = prepareStringKey(params, ''+obj);
        }
        return obj;
    };


    var translatePropertyName = function(params, propName) {
        if (propName === 'not') {
            return propName
        }
        propName = (params.propertyRules || {})[propName] || propName;
        if (propName.charAt(0) === '-') {
            propName = ((params.propertyRules || {})[propName.substr(1)] || propName.substr(1)) + "__not"
        } else {
            propName = propName + "__and"
        }
        return propName;
    }

    /**
     * Adds the value to the propStr entry (propStr -> value) of obj
     * If it's not an array convert it to an array and add value
     * If value is an array, add each value of array
     *
     * @param obj the object to add the value to
     * @param propStr the property on which to add the value
     * @param value the value to add
     *
     * @return void
     */
    var safePushProp = function(obj, propStr, value) {
        if (!obj || !propStr) {
            return
        }
        var initialValue = Fn.propStrSafe(propStr)(obj);
        if (!$.isArray(initialValue)) {
            if (initialValue) {
                Fn.setPropStr([initialValue], propStr)(obj)
            }
            else {
                Fn.setPropStr([], propStr)(obj)
            }
            initialValue = Fn.propStr(propStr)(obj);
        }
        if ($.isArray(value)) {
            value.forEach(function(o) {
                initialValue.push(o)
            })
        }
        else {
            initialValue.push(value)
        }
    }

    var handleUserQuery = function(filterQuery, params) {
        if (!filterQuery.userQuery) {
            return filterQuery
        }
        var remainingUserQuery = [];
        filterQuery.userQuery.split(" ").filter(function(o) {
            return o !== ""
        }).forEach(function(dottedProp) {
            if (dottedProp.indexOf(':') > -1 && !( dottedProp.startsWith("/") && dottedProp.endsWith("/") )) {
                var fqelem = dottedProp.split(":");
                var propName = translatePropertyName(params, fqelem.shift());
                safePushProp(filterQuery, propName, fqelem.join(":"));
            } else {
                remainingUserQuery.push(dottedProp);
            }
        });
        filterQuery.userQuery = remainingUserQuery.join(" ").trim();

        // prepare the remaining search terms so that we can match them on the object properties
        filterQuery.$requiredSearchTerms = ('' + filterQuery.userQuery).split(" ").filter(a => a != "").map(prepareStringKey.bind(null, null));

        params.userQueryResult = ('' + filterQuery.userQuery).split(" ").filter(a => a).map(prepareStringKey.bind(null, null));
        return filterQuery;
    };

    var handleUserQueryTargets = function(filterQuery, params) {
        if (!params.userQueryTargets || !filterQuery) {
            return filterQuery
        }
        var userQueryTargets = $.isArray(params.userQueryTargets) ? params.userQueryTargets : [params.userQueryTargets];
        // TODO : there should be a proper "summary of the query after the whole processing"

        // Add appropriate filters if user imputed not:something (only applies to userQueryTargets)
        if (filterQuery.not) {
            userQueryTargets.forEach(function(userQueryTarget) {
                safePushProp(filterQuery, userQueryTarget + "__not", filterQuery.not)
            });
            delete filterQuery.not;
        }

        return filterQuery
    };

    var filterWrapper = function(collection, query, params) {
        var filterQuery = angular.copy(query || {});
        if (!params) {
            params = {}
        } else {
            delete params.userQueryErrors;
            delete params.userQueryResult;
        }
        filterQuery = handleUserQuery(filterQuery, params);
        // console.info('HANDLE',JSON.stringify(filterQuery));
        filterQuery = handleUserQueryTargets(filterQuery, params);
        // console.info('TARGETS',JSON.stringify(filterQuery));
        filterQuery = cleanFilterQuery(filterQuery, params);
        // console.info('CLEANED',JSON.stringify(filterQuery));
        return collection.filter(function(o) {
            return filterFilter(filterQuery, params, o);
        });
    };

    return {
        filter: filterWrapper,
    }
});


let algorithmsPaletteColors = [
    "#f07c48", // poppy red (no, it's actually orange)
    "#fdc766", // chicky yellow
    "#7bc9a6", // turquoisy green
    "#4ec5da", // sky blue
    "#548ecb", // sea blue
    "#d848c0", // lilas
    "#41bb15" // some no name green
    //"#97668f", // lilas
    //"#5e2974", // dark purple
];

app.factory("algorithmsPalette", function() {
    var COLORS = algorithmsPaletteColors;
    return function(i) {
        return COLORS[i % COLORS.length];
    }
});


app.factory("categoricalPalette", function() {  // Keep in sync with PaletteFactory/categorical
    var COLORS = [
        "#f06548", // poppy red
        "#fdc766", // chicky yellow
        "#7bc9a6", // turquoisy green
        "#4ec5da", // sky blue
        "#548ecb", // sea blue
        "#97668f", // lilas
        "#5e2974", // dark purple
    ];
    return function(i) {
        return COLORS[i % COLORS.length];
    }
});

app.factory("gradientGenerator", function() {
    function toHexString(n) {
        return Math.round(n).toString(16);
    }
    function toHexColor(c) {
        return `#${toHexString(c.r)}${toHexString(c.g)}${toHexString(c.b)}`;
    }

    return function(hexColor, nbColors) {
        let currentColor = d3.rgb(hexColor);
        const colors = [];
        for (let i = 0; i < nbColors; i++) {
            colors.push(currentColor);
            currentColor = currentColor.brighter(0.05);
        }

        return colors.map(toHexColor);
    }
})

app.factory("divergingPalette", function() {
    /* given a value going from -1 to 1
     returns the color associated by a diverging palette
     going form blue to red. */
    var rgbToHsl = function(s) {
        return d3.rgb(s).hsl();
    };
    var RED_SCALE = ["#fbefef", "#FBDAC8", "#F3A583", "#D75D4D", "#B11F2C"].map(rgbToHsl);
    var BLUE_SCALE = ["#f0f1f1", "#CFE6F1", "#94C5DE", "#4793C3", "#2966AB"].map(rgbToHsl);

    var divergingPalette = function(r) {
        var SCALE = (r > 0) ? RED_SCALE : BLUE_SCALE;
        if (r < 0) {
            r = -r;
        }
        if (r >= 1.) r = 0.99;
        var N = SCALE.length - 1;
        var bucket = r * N | 0;
        var low_color = SCALE[bucket];
        var high_color = SCALE[bucket + 1];
        r = r * N - bucket;
        var h = high_color.h;
        var l = low_color.l * (1 - r) + high_color.l * r;
        var s = low_color.s * (1 - r) + high_color.s * r;
        return d3.hsl(h, s, l);
    };
    return divergingPalette;
});

app.constant("COLOR_PALETTES", {
    highContrast: ['#FFD502', '#02FFFF', '#FF0256', '#02FFD5', '#FF02AB', '#02D5FF', '#2C02FF', '#8102FF', '#022CFF', '#02ABFF', '#FF02D5', '#FF02FF', '#02FF81', '#FF0281', '#F0E704', '#02FFAB', '#5602FF', '#AB02FF', '#CDF600', '#D502FF', '#0256FF', '#FF5602', '#FF8102', '#0281FF', '#FFAB02', '#ABFF02'],
    algorithms: algorithmsPaletteColors
});

app.factory("AnyLoc", function(Assert) {
    // Caution: not in line with backend type AnyLoc
    // On the angular side, to keep in line with server/src/frontend/src/app/utils/loc.ts
    return {
        makeLoc : function(projectKey, localId) {
            return {
                    projectKey: projectKey,
                    localId: localId,
                    fullId: projectKey + "." + localId
            }
        },
        getLocFromSmart: function(contextProjectKey, name) {
            if (name.indexOf(".") >= 0) {
                return {
                    projectKey: name.split(".")[0],
                    localId: name.split(".")[1],
                    fullId: name
                }
            } else {
                return {
                    projectKey: contextProjectKey,
                    localId: name,
                    fullId: contextProjectKey + "." + name
                }
            }
        },
        getLocFromFull: function(fullName) {
            Assert.trueish(fullName.includes('.'), 'no dot in fullname');
            return {
                projectKey: fullName.split(".")[0],
                localId: fullName.split(".")[1],
                fullId: fullName
            };
        },
    }
});

app.factory("AutomationUtils", function() {
    const svc = {
        pythonEnvImportTimeModes: [
            ["INSTALL_IF_MISS", "Create new if not present"],
            ["FAIL_IF_MISS", "Fail if not present"],
            ["DO_NOTHING", "Ignore"]
        ],
        pythonEnvImportTimeModeDescs: [
            "When the bundle declares a dependency on a code env that does not exist on this instance, create a new code env with that name",
            "When the bundle declares a dependency on a code env that does not exist on this instance, stop the preloading",
            "Do not take action if a code env is missing"
        ],
        envImportSpecificationModes: [
            ["SPECIFIED", "User-specified list of packages"],
            ["ACTUAL", "Actual list of packages"]
        ],
        envImportSpecificationModeDescs: [
            "Use the list of packages that the user specified, and the default list of required packages",
            "Use the list of packages built by inspecting the environment"
        ]
    }

    return svc;
});

app.factory("ConnectionUtils", function() {
    const connectionSqlTypes =
        ["hiveserver2", "MySQL", "PostgreSQL", "AlloyDB", "Vertica", "Greenplum", "Redshift", "Teradata", "Oracle", "SQLServer", "Synapse", "FabricWarehouse", "Netezza", "BigQuery", "SAPHANA", "Snowflake", "JDBC", "Athena", "Trino", "Databricks", "DatabricksLakebase", "TreasureData"]; //TODO @datasets remove
    return {
        isIndexable: function(connection) {
            return connection && connection.type && connectionSqlTypes.includes(connection.type);
        }
    }
});

app.factory("DatasetUtils", function(Assert, DataikuAPI, $q, Logger, $rootScope, SmartId, RecipesUtils) {
    const sqlTypes = ["MySQL", "PostgreSQL", "AlloyDB", "Vertica", "Greenplum", "Redshift", "Teradata", "Oracle", "SQLServer", "Synapse", "FabricWarehouse", "Netezza", "BigQuery", "SAPHANA", "Snowflake", "JDBC", "Athena", "Trino", "Denodo", "Databricks", "DatabricksLakebase", "TreasureData"]; //TODO @datasets remove
    const sqlAbleTypes = ["S3", "iceberg"];
    const svc = {
        canUseSQL: function(dataset) {
            if (sqlTypes.indexOf(dataset.type) >= 0 &&
                dataset.params.mode == "table") {
                return true;
            }
            if (sqlAbleTypes.indexOf(dataset.type) >= 0) {
                return true;
            }
            if (dataset.type == "HDFS" || dataset.type == "hiveserver2") {// && $scope.appConfig.impalaEnabled) {
                return true;
            }
            Logger.info("Dataset is not SQL-capable: " + dataset.type);
            return false;
        },

        canUseSparkSQL: function(dataset) {
            if ($rootScope.appConfig.interactiveSparkEngine == "DATABRICKS") {
                return dataset.type == "HDFS";
            } else {
                if (sqlTypes.indexOf(dataset.type) >= 0 && dataset.params.mode == "table") {
                    return false;
                }
                return true;
            }
        },

        hasSizeStatus: function(type) {
            return svc.getKindForConsistency({type: type}) == "files";
        },

        isSQL: function(dataset) {
            return sqlTypes.indexOf(dataset.type) >= 0;
        },

        isSQLTable: function(dataset) {
            return sqlAbleTypes.indexOf(dataset.type) >= 0 ||
                sqlTypes.indexOf(dataset.type) >= 0 && (!dataset.params || dataset.params.mode == "table"); // A bit hackish, when we don't have the params, let's not block functionality
        },

        // Can we run SQL queries on this dataset
        isSQLQueryAble: function(dataset) {
            return sqlAbleTypes.indexOf(dataset.type) >= 0 ||
                sqlTypes.indexOf(dataset.type) >= 0;
        },

        supportsReadOrdering : function(dataset) {
            return svc.isSQLTable(dataset) && dataset.type !== 'iceberg';
        },

        getKindForConsistency : function(dataset) {
            if (sqlTypes.indexOf(dataset.type) >= 0) {
                return "sql";
            } else if (dataset.type == "MongoDB") {
                return "mongodb";
            } else if (dataset.type == "SharePointOnlineList") {
                return "sharepointonlinelist";
            } else if (dataset.type == "DynamoDB") {
                return "dynamodb";
            } else if (dataset.type == "Cassandra") {
                return "cassandra";
            } else if (dataset.type == "iceberg") {
                return "iceberg";
            } else if (dataset.type == "Twitter") {
                return "generic";
            } else if (dataset.type == "ElasticSearch") {
                return "generic";
            } else if (dataset.type == "Kafka") {
                return "generic";
            } else if (dataset.type == "SQS") {
                return "generic";
            } else {
                return "files";
            }
        },

        getLocFromSmart: function(contextProjectKey, name) {
            if (name.indexOf(".") >= 0) {
                return {
                    projectKey: name.split(".")[0],
                    name: name.split(".")[1],
                    fullName: name
                };
            } else {
                return {
                    projectKey: contextProjectKey,
                    name: name,
                    fullName: contextProjectKey + "." + name
                };
            }
        },

        getLocFromFull: function(fullName) {
            Assert.trueish(fullName.includes('.'), 'no dot in fullname');
            return {
                projectKey: fullName.split(".")[0],
                name: fullName.split(".")[1],
                fullName: fullName
            };
        },

        getSchema: function(scope, datasetName) {
            return svc.getSchemaFromComputablesMap(scope.computablesMap,datasetName)
        },

        getSchemaFromComputablesMap: function(computablesMap, datasetName) {
            if (!computablesMap || !computablesMap[datasetName]) {
                return;
            }
            var it = computablesMap[datasetName];
            if (!it || !it.dataset) {
                throw Error('dataset is not in computablesMap');
            }
            return it.dataset.schema;
        },

        makeLoc: function(datasetProjectKey, datasetName) {
            return {
                projectKey: datasetProjectKey,
                name: datasetName,
                fullName: datasetProjectKey + "." + datasetName
            };
        },

        makeSmart: function(loc, contextProjectKey) {
            if (loc.projectKey == contextProjectKey) {
                return loc.name;
            } else {
                return loc.fullName;
            }
        },

        makeHeadSelection: function(lines) {
            return {
                partitionSelectionMethod: "ALL",
                samplingMethod: "HEAD_SEQUENTIAL",
                maxRecords: lines
            };
        },

        updateRecipeComputables: function(scope, recipe, projectKey, contextProjectKey) {
            if (!scope.computablesMap) return Promise.resolve();
            let references = new Set(RecipesUtils.getFlatIOList(recipe).map(role => role.ref));
            references = [...references].filter(ref => (ref in scope.computablesMap) && ((scope.computablesMap[ref].dataset && !scope.computablesMap[ref].dataset.schema) || (scope.computablesMap[ref].streamingEndpoint && !scope.computablesMap[ref].streamingEndpoint.schema)));
            return $q.all(references
                .map(name => {
                    let resolvedSmartId = SmartId.resolve(name, contextProjectKey);
                    if (scope.computablesMap[name].dataset) {
                        return DataikuAPI.datasets.get(resolvedSmartId.projectKey, resolvedSmartId.id, contextProjectKey).success(function(data){
                            scope.computablesMap[name].dataset = data;
                        }).error(setErrorInScope.bind(scope));
                    } else if (scope.computablesMap[name].streamingEndpoint) {
                        return DataikuAPI.streamingEndpoints.get(resolvedSmartId.projectKey, resolvedSmartId.id, contextProjectKey).success(function(data){
                            scope.computablesMap[name].streamingEndpoint = data;
                        }).error(setErrorInScope.bind(scope));
                    }
                })
            );
        },

        updateDatasetInComputablesMap: function(scope, dsName, projectKey, contextProjectKey) {
            let resolvedSmartId = SmartId.resolve(dsName, contextProjectKey);
            return DataikuAPI.datasets.get(resolvedSmartId.projectKey, resolvedSmartId.id, contextProjectKey).success(function(data){
                scope.computablesMap[dsName].dataset = data;
            }).error(setErrorInScope.bind(scope));
        },

        listDatasetsUsabilityForAny: function(contextProjectKey) {
            return DataikuAPI.flow.listUsableComputables(contextProjectKey, {
                datasetsOnly: true
            });
        },

        listFoldersUsabilityForOutput : function(contextProjectKey, recipeType) {
            var d = $q.defer();
            DataikuAPI.flow.listUsableComputables(contextProjectKey, {
                datasetsOnly : false,
                forRecipeType : recipeType,
            }).success(function(data) {
                data.forEach(function(x) {
                    x.usable = x.usableAsInput;
                    x.usableReason = x.inputReason;
                });
                d.resolve(data);
            });
            return d.promise;
        },

        listDatasetsUsabilityForInput: function(contextProjectKey, recipeType) {
            var d = $q.defer();
            DataikuAPI.flow.listUsableComputables(contextProjectKey, {
                datasetsOnly: true,
                forRecipeType: recipeType,
            }).success(function(data) {
                data.forEach(function(x) {
                    x.usable = x.usableAsInput;
                    x.usableReason = x.inputReason;
                });
                d.resolve(data);
            });
            return d.promise;
        },

        /**
         * Returns a promise on an arry of two array, "availableInputDatasets" and "availableOutputDataset"
         * On each, the "usable" and "usableReason" are set
         */
        listDatasetsUsabilityInAndOut: function(contextProjectKey, recipeType, datasetsOnly) {
            var d = $q.defer();
            DataikuAPI.flow.listUsableComputables(contextProjectKey, {
                datasetsOnly: datasetsOnly == null ? true : datasetsOnly,
                forRecipeType: recipeType,
            }).success(function(data) {
                var avlIn = angular.copy(data);
                var avlOut = angular.copy(data);
                avlIn.forEach(function(x) {
                    x.usable = x.usableAsInput;
                    x.usableReason = x.inputReason;
                });
                avlOut.forEach(function(x) {
                    x.usable = x.usableAsOutput;
                    x.usableReason = x.outputReason;
                });
                d.resolve([avlIn, avlOut]);
            });
            return d.promise;
        },

        /**
         * Returns a promise on an arry of two array, "availableInputDatasets" and "availableOutputDataset"
         * On each, the "usable" and "usableReason" are set
         */
        listUsabilityInAndOut: function(contextProjectKey, recipeType) {
            var d = $q.defer();
            DataikuAPI.flow.listUsableComputables(contextProjectKey, {
                forRecipeType: recipeType
            }).success(function(data) {
                var avlIn = angular.copy(data);
                var avlOut = angular.copy(data);
                avlIn.forEach(function(x) {
                    x.usable = x.usableAsInput;
                    x.usableReason = x.inputReason;
                });
                avlOut.forEach(function(x) {
                    x.usable = x.usableAsOutput;
                    x.usableReason = x.outputReason;
                });
                d.resolve([avlIn, avlOut]);
            });
            return d.promise;
        },

        isOnDifferentConnection: function (dataset, primaryDatasetConnection) {
            return primaryDatasetConnection && dataset.dataset.params.connection !== primaryDatasetConnection;
        },

        isOutputDataset: function (dataset, recipe, outputDatasetName) {
            return (dataset.dataset.name === outputDatasetName && recipe.projectKey && dataset.dataset.projectKey === recipe.projectKey);
        },

        INVALID_INPUT_DATASET_REASONS: {
            DIFF_CONN: "Dataset connection is different from primary one",
            IS_RECIPE_OUTPUT: "Dataset cannot be the recipe's output"
        },

        /**
         * Mark the recipe output dataset as unusable.
         * Currently used for join input dataset select options.
         * @param datasets
         * @returns Copy of input datasets with the recipe output dataset marked as unusable.
         */
        setInputDatasetsUsability: function (datasets, recipe, outputDatasetName) {
            const availableInputDatasetsForJoin = [];
            for (const originalDataset of datasets) {
                const dataset = angular.copy(originalDataset);
                if (this.isOutputDataset(dataset, recipe, outputDatasetName)) {
                    dataset.usable = false;
                    dataset.usableReason = this.INVALID_INPUT_DATASET_REASONS.IS_RECIPE_OUTPUT;
                }
                availableInputDatasetsForJoin.push(dataset);
            }
            return availableInputDatasetsForJoin;
        }
    };
    return svc;
});


app.factory("Breadcrumb", function($stateParams, $rootScope) {
    var ret = {}
    $rootScope.masterBreadcrumbData = {}

    ret.setProjectSummary = function(projectSummary) {
        $rootScope.currentProjectSummary = projectSummary;
    }

    ret.setData = function(k, v) {
        $rootScope.masterBreadcrumbData[k] = v;
    }

    ret.projectBreadcrumb = function() {
        return [
            //{ "type" : "home" },
            {"type": "project", "projectKey": $stateParams.projectKey}
        ]
    }
    ret.datasetBreadcrumb = function() {
        return ret.projectBreadcrumb().concat([
            //{"type" : "datasets", projectKey : $stateParams.projectKey},
            {
                "type": "dataset", projectKey: $stateParams.projectKey, id: $stateParams.datasetName,
                displayName: $stateParams.datasetName
            }
        ]);
    }
    ret.recipeBreadcrumb = function() {
        return ret.projectBreadcrumb().concat([
            {"type": "recipes", projectKey: $stateParams.projectKey},
            {
                "type": "recipe", projectKey: $stateParams.projectKey, id: $stateParams.recipeName,
                displayName: $stateParams.recipeName
            }
        ]);
    }
    ret.insightBreadcrumb = function(insightName) {
        return ret.projectBreadcrumb().concat([
            {"type": "insights", "projectKey": $stateParams.projectKey},
            {
                "type": "insight",
                "projectKey": $stateParams.projectKey,
                "id": $stateParams.insightId,
                displayName: insightName
            }
        ]);
    }

    ret.set = function(array) {
        $rootScope.masterBreadcrumb = array;
    }

    ret.setWithProject = function(array) {
        ret.set(ret.projectBreadcrumb().concat(array));
    }
    ret.setWithDataset = function(array) {
        ret.set(ret.datasetBreadcrumb().concat(array));
    }
    ret.setWithInsight = function(insightName, array) {
        ret.set(ret.insightBreadcrumb(insightName).concat(array));
    }
    ret.setWithRecipe = function(array) {
        ret.set(ret.recipeBreadcrumb().concat(array));
    }
    ret.setWith
    return ret;
});


app.service('LocalStorage', ['$window', function($window) {
    return {
        set: function(key, value) {
            if (value !== undefined) {
                $window.localStorage[key] = JSON.stringify(value);
            }
        },
        get: function(key) {
            var ret = $window.localStorage[key];
            if (ret !== undefined) {
                ret = JSON.parse(ret);
            }
            return ret;
        },
        clear: function(key) {
            delete $window.localStorage[key];
        }
    }
}]);


app.service('RemoteResourcesLinksUtils', function() {
    const svc = this;

    svc.getSageMakerResourceLink = function(resource, region, resourceName) {
        return `https://${region}.console.aws.amazon.com/sagemaker/home#/${resource}/${resourceName}`;
    }

    svc.getVertexAIResourceLink = function(resource, project, location, resourceId) {
        return `https://console.cloud.google.com/vertex-ai/locations/${location}/${resource}/${resourceId}?project=${project}`;
    }

    const getWsid = function(workspace, resourceGroup, subscription) {
        return `wsid=/subscriptions/${subscription}/resourcegroups/${resourceGroup}/providers/Microsoft.MachineLearningServices/workspaces/${workspace}`;
    }

    const getTid = function(tenantId) {
        return (tenantId) ? `tid=${tenantId}` : '';
    }

    svc.getAzureMLReference = function(resourceName, resourceVersion) {
        return `azureml:${resourceName}:${resourceVersion}`;
    }

    svc.getAzureMLModelLink = function(resourceInfo) {
        const wsid = getWsid(resourceInfo.azWorkspace, resourceInfo.azResourceGroup, resourceInfo.azSubscription);
        const tid = '&' + getTid(resourceInfo.azTenantId);
        return `https://ml.azure.com/model/${resourceInfo.azModelName}:${resourceInfo.azModelVersion}/details?${wsid}${tid}`;
    }

    svc.getAzureMLEnvironmentLink = function(resourceInfo) {
        const wsid = getWsid(resourceInfo.azWorkspace, resourceInfo.azResourceGroup, resourceInfo.azSubscription);
        const tid = '&' + getTid(resourceInfo.azTenantId);
        return `https://ml.azure.com/environments/${resourceInfo.azEnvironmentName}/version/${resourceInfo.azEnvironmentVersion}?${wsid}${tid}`;
    }

    svc.getAzureMLOnlineEndpointLink = function(resourceInfo) {
        const wsid = getWsid(resourceInfo.azWorkspace, resourceInfo.azResourceGroup, resourceInfo.azSubscription);
        const tid = '&' + getTid(resourceInfo.azTenantId);
        return `https://ml.azure.com/endpoints/realtime/${resourceInfo.azOnlineEndpointName}?${wsid}${tid}`;
    }

    svc.getDatabricksEndpointLink = function(host, endpointName) {
        return `https://${host}/ml/endpoints/${endpointName}`;
    }

    svc.getDatabricksModelLink = function(host, isUnityCatalogUsed, modelName) {
        if (isUnityCatalogUsed) {
            const [catalogName, schemaName, unityCatalogModelName] = modelName.split(".");
            return `https://${host}/explore/data/models/${catalogName}/${schemaName}/${unityCatalogModelName}`;
        }
        return `https://${host}/ml/models/${modelName}`;
    }

    svc.getDatabricksModelVersionLink = function(host, isUnityCatalogUsed, modelName, modelVersion) {
        const modelLink = svc.getDatabricksModelLink(host, isUnityCatalogUsed, modelName);
        if (isUnityCatalogUsed) {
            return `${modelLink}/version/${modelVersion}`;
        }
        return `${modelLink}/versions/${modelVersion}`;
    }

    svc.getDatabricksExperimentLink = function(host, expId) {
        return `https://${host}/ml/experiments/${expId}`;
    }

    svc.getDatabricksExperimentRunLink = function(host, expId, runId) {
        return `https://${host}/ml/experiments/${expId}/runs/${runId}`;
    }
});

app.factory("ContextualMenu", function($compile, $rootScope, $window, $templateRequest) {
        // Class describing a menu.
        // Can be used for both contextual menu and
        // regular menues.
        //
        function Menu(params) {
            /*
             Contextual or not, only one menu can be visible at the same time
             on the screen.
             The contextual menu content does not live in the DOM until it is displayed
             to the user.

             Parameters contains the following options
             - template (required) : template path for the content of the menu.
             - controller (optional) : name of the controller
             - scope  (optional) : if not added, a new scope will be created.
             - contextual (option, true|false, default:true
             in contextual menu mode, all clicks outside of the
             popup is captured.
             - onOpen (optional): called on menu open
             - onClose (optional): called on menu close
             - cssClass: CSS class on the ul
             */
            this.template = params.template;
            if (typeof this.template != "string") {
                throw "Template parameter is required";
            }
            this.cssClass = params.cssClass;
            this.controller = params.controller;
            this.contextual = params.contextual;
            if (this.contextual === undefined) {
                this.contextual = true;
            }
            this.enableClick = params.enableClick;
            if (this.enableClick === undefined) {
                this.enableClick = false;
            }
            this.handleKeyboard = params.handleKeyboard;
            if (this.handleKeyboard === undefined) {
                this.handleKeyboard = true;
            }
            this.scope = params.scope;
            this.tmplPromise = $templateRequest(this.template);
            this.onClose = params.onClose || function() {
                };
            this.onOpen = params.onOpen || function() {
                };
        }

        Menu.prototype.newScope = function() {
            if (this.scope) {
                return this.scope.$new();
            }
            else {
                return $rootScope.$new(true);
            }
        };

        Menu.prototype.globalOnClose = function() {
        };
        Menu.prototype.globalOnOpen = function() {
        };

        // close any popup currently visible on the screen.
        Menu.prototype.closeAny = function(e) {
            // remove and unbind any overlays
            Menu.prototype.$overlay.unbind("click");
            Menu.prototype.$overlay.unbind("contextmenu");
            Menu.prototype.$overlay.remove();

            // remove the document click
            $(document).off(".closeMenu");

            Menu.prototype.$menu.remove();
            Menu.prototype.globalOnClose();
            Menu.prototype.globalOnClose = function() {
            };
            Menu.prototype.globalOnOpen = function() {
            };
            Menu.prototype.$menu.removeClass();
            Menu.prototype.$menu.addClass('dropdown-menu');
            if (e) e.preventDefault();
            return false;
        };

        Menu.prototype.setup = function($menu) {
            var me = this;

            Menu.prototype.globalOnClose = this.onClose;
            Menu.prototype.globalOnOpen = this.onOpen;
            var index = -1;
            var currentMenu = Menu.prototype.$menu;

            if (me.contextual) {
                $menu.before(Menu.prototype.$overlay);
                $(Menu.prototype.$overlay).bind("contextmenu", me.closeAny.bind(me));
                Menu.prototype.$overlay.click(me.closeAny.bind(me));
            } else {
                window.setTimeout(function() {
                    // handle click when menu is open
                    $(document)
                        .on('click.closeMenu', function(evt) {
                            const targetParents = $(evt.target).parents();
                            const isClickedOutsideMenu = targetParents.index(Menu.prototype.$menu) === -1;
                            // Dropdown panels can be appended in the body but clicking in a dropdown panel shouldn't close a contextual menu.
                            const getClassList = (parent) => parent.classList ? Array.from(parent.classList) : [];
                            const isClickedElementADropdownItem = targetParents.toArray().some(parent => getClassList(parent).includes('ng-dropdown-panel'));
                            if (isClickedOutsideMenu && !isClickedElementADropdownItem) {
                                Menu.prototype.closeAny();
                            }
                        });
                }, 0);
            }
            if (!me.enableClick) {
                $menu.on('click.ctxmenu', function(e) {
                    me.closeAny();
                });
            }

            window.setTimeout(function() {
                // makes the links focusable
                Menu.prototype.$menu.find('a').attr('tabindex', -1);
                // focus on the first link
                const items = Menu.prototype.$menu.find('a');
                if (items.length > 0) {
                    const item = items.eq(0);
                    const parentMenus = item.parents('ul');
                    if (parentMenus.length > 0) {
                        currentMenu = parentMenus.eq(0);
                        index = 0;
                        item.focus();
                    }
                }

                var handleKey = function(evt) {
                    if (Menu.prototype.$menu.height()) {
                        if (evt.which === 27) {
                            // esc
                            Menu.prototype.closeAny();
                        }

                        const isTargetInMenu = $(evt.target).parents().index(currentMenu) !== -1; // Only process events when the target is an item of the menu
                        if (isTargetInMenu) {
                            if (evt.which === 40) {
                                // down
                                const items = currentMenu.find('>li>a');
                                if (items.length) {
                                    index = Math.min(items.length - 1, index + 1);
                                    items[index].focus();
                                }
                                evt.preventDefault();
                                evt.stopPropagation();
                            }
                            if (evt.which === 38) {
                                // up
                                index = Math.max(0, index - 1);
                                const items = currentMenu.find('>li>a, >*>li>a');
                                if (items.length) {
                                    items[index].focus();
                                }
                                evt.preventDefault();
                                evt.stopPropagation();
                            }

                            if (evt.which === 37) {
                                // left
                                // Go up one menu
                                if (currentMenu != Menu.prototype.$menu) {
                                    index = currentMenu.parents('ul').eq(0).find('>li>a').index(currentMenu.parent('li.hover').removeClass('hover').find('>a'));
                                    currentMenu = currentMenu.parents('ul').eq(0);
                                    const items = currentMenu.find('>li>a');
                                    if (items.length) {
                                        items[index].focus();
                                    }
                                }
                                evt.preventDefault();
                                evt.stopPropagation();
                            }
                            if (evt.which === 39) {
                                // right
                                // go into submenu
                                const items = currentMenu.find('>li>a');
                                const submenus = items.eq(index).siblings('ul');
                                if (submenus.length) {
                                    items.eq(index).parent().addClass('hover');
                                    currentMenu = submenus.eq(0);
                                    index = 0;
                                    currentMenu.find('a').eq(0).focus();
                                }
                                evt.preventDefault();
                                evt.stopPropagation();
                            }
                            if (evt.which === 13) {
                                // enter
                                currentMenu.find('>li>a').eq(index).trigger('click');
                                Menu.prototype.closeAny();
                            }
                        }
                    }
                };
                if (me.handleKeyboard) {
                    $(document).on('keydown.closeMenu', handleKey); // handle keypress while the menu doesn't have the focus
                    Menu.prototype.$menu.on('keydown', handleKey);
                }
                Menu.prototype.$menu.on('mouseenter', 'a', function(e) {
                    Menu.prototype.$menu.find('.hover').removeClass('hover');
                    currentMenu = $(this).parents('ul').eq(0);
                    const items = currentMenu.find('>li>a')
                    index = items.index($(this));
                    if (index !== -1) {
                        items[index].focus(); // Set focus to current item, making it the target for keyboard events
                    }
                });

            });
        };

        // Fill the shared menu element with menu instance
        // content.
        //
        // Template is compiled against the scope
        // at each call.
        //
        Menu.prototype.fill = function(cb) {
            var me = this;
            this.tmplPromise.then(function(tmplData) {
                if (me.controller !== undefined) {
                    me.$menu.attr("ng-controller", me.controller);
                }
                else {
                    me.$menu.removeAttr("ng-controller");
                }
                me.$menu.html(tmplData);
                Menu.prototype.destroyCurrentScope();
                var newScope = me.newScope();
                $compile(me.$menu)(newScope);
                Menu.prototype.currentScope = newScope;
                if (cb !== undefined) {
                    cb(me.$menu);
                }
                if (me.cssClass) {
                    me.$menu.addClass(me.cssClass);
                }
            });
        };

        Menu.prototype.destroyCurrentScope = function() {
            if (Menu.prototype.currentScope != undefined) {
                Menu.prototype.currentScope.$destroy();
            }
            Menu.prototype.currentScope = undefined;
        };

        Menu.prototype.openAlignedWithElement = function(alignElement, callback, followScroll, exactAlignment) {
            var me = this;
            me.closeAny();
            me.fill(function($menu) {
                // place the element.
                var $body = $("body");
                var $alignElement = $(alignElement);
                var alignElementOffset = $alignElement.offset();
                var scrollOffsetLeft = alignElementOffset.left;
                var scrollOffsetTop = alignElementOffset.top;

                var box = $alignElement.offset();
                box.width = $alignElement.width();
                box.height = $alignElement.outerHeight();
                $body.append($menu);
                var left = Math.max(0, box.left - (exactAlignment ? 0 : 10));

                // we also want to move the dropdown menu to the left
                // to stay on the screen.
                var menuWidth = $menu.width();
                var bodyWidth = $body.width();

                $menu.detach();

                left = Math.min(left, bodyWidth - menuWidth - 10);

                if (bodyWidth - left - menuWidth < 180) {
                    // let's step into bizarro land
                    // where submenues open to the left.
                    $menu.addClass("bizarro");
                }
                else {
                    $menu.removeClass("bizarro");
                }

                var position = {
                    left: left,
                    top: box.top + box.height,
                    bottom: "auto",
                    right: "auto"
                };

                let containerElement = $body;

                if (followScroll) {
                    containerElement = $(alignElement).offsetParent();
                    const alignElementPosition = $alignElement.position();
                    scrollOffsetLeft = alignElementPosition.left - box.left;
                    scrollOffsetTop = alignElementPosition.top - box.top;
                    position.left += scrollOffsetLeft;
                    position.top += scrollOffsetTop;
                }

                $menu.appendTo(containerElement);
                $menu.css(position);

                me.setup($menu);
                if (callback !== undefined) {
                    callback($menu);
                }
                Menu.prototype.globalOnOpen();
            });
        };

        Menu.prototype.openAtXY = function(left, top, callback, dummyLateralPosition, dummyVerticalPosition) {
            var me = this;
            me.closeAny();
            me.fill(function($menu) {
                $("body").append($menu);
                var offset = {};
                if (left < $($window).width() / 2 || dummyLateralPosition) {
                    offset.left = left;
                    offset.right = 'auto';
                } else {
                    offset.left = 'auto';
                    offset.right = $($window).width() - left;
                }
                if (top < $($window).height() / 2 || dummyVerticalPosition) {
                    offset.top = top;
                    offset.bottom = 'auto';
                } else {
                    offset.top = 'auto';
                    offset.bottom = $($window).height() - top;
                }
                $menu.css(offset);
                me.setup($menu);
                if (callback !== undefined) {
                    callback($menu);
                }
                Menu.prototype.globalOnOpen();
            });
        };

        Menu.prototype.openAtEventLoc = function (evt) {
            if (!evt) return;
            this.openAtXY(evt.pageX, evt.pageY);
        }

        // TODO get rid of the id
        Menu.prototype.$menu = $('<ul id="dku-contextual-menu" class="dropdown-menu" style="position:absolute" role="menu">');
        // overlay element that helps capturing any click
        // outside of the menu.
        // Used in ContextualMenu mode.
        Menu.prototype.$overlay = $('<div class="contextualMenuOverlay"></div>');
        // Menu.prototype.$overlay

        return Menu;
    });


app.factory("ActivityIndicatorManager", ["$timeout", function($timeout) {
    function hide(activityIndicator) {
        activityIndicator.hidden = true;
    }

    function getActivityIndicatorType(type) {
        switch (type) {
            case 'waiting':
            case 'info':
                return 'progress';
            case 'success':
                return 'success';
            case 'warning':
                return 'warning';
            case 'error':
                return 'error';
            default:
                throw new Error('Unknown type: ' + type);
        }
    }

    return {
        hide,
        /**
         *
         * @param {ActivityIndicator} activityIndicator
         * @param {'waiting' | 'success' | 'warning' | 'info' | 'error'} type
         * @param {string} text
         * @param {number} time
         * @param {boolean?} faded
         * @param {{ label: string, callback: () => void, icon?: string }} action
         */
        configureActivityIndicator: function (activityIndicator, type, text, time, faded = true, action) {
            activityIndicator.hidden = false;
            activityIndicator.text = text;
            activityIndicator.type = getActivityIndicatorType(type);
            activityIndicator.faded = faded;
            activityIndicator.action = action;
            activityIndicator.hide = () => hide(activityIndicator);
            activityIndicator.closeable = false;

            if (type === 'waiting') {
                activityIndicator.spinner = true;
            } else {
                activityIndicator.spinner = false;
                if (!time) {
                    time = 2000;
                }
                if(activityIndicator.pending) {
                    $timeout.cancel(activityIndicator.pending);
                }
                if (time >= 0) {
                    activityIndicator.pending = $timeout(function () {
                        activityIndicator.pending = undefined;
                        hide(activityIndicator);
                    }, time);
                } else {
                    activityIndicator.closeable = true;
                }
            }
        },
        buildDefaultActivityIndicator: function() {
            return {
                pending: undefined,
                hidden: true,
                text: '',
                type: 'progress',
                spinner: false,
                faded: true
            };
        },
        isDisplayed: function(activityIndicator) {
            return !activityIndicator.hidden;
        }
    };
}]);

app.factory("ActivityIndicator", ["$rootScope", "ActivityIndicatorManager", function($rootScope, ActivityIndicatorManager) {
    $rootScope.activityIndicator = ActivityIndicatorManager.buildDefaultActivityIndicator();
    return {
        isDisplayed: function() {
            return ActivityIndicatorManager.isDisplayed($rootScope.activityIndicator)
        },
        waiting: function(text) {
            ActivityIndicatorManager.configureActivityIndicator($rootScope.activityIndicator, 'waiting', text);
        },
        hide: function() {
            ActivityIndicatorManager.hide($rootScope.activityIndicator);
        },
        success: function(text, time, action) {
            ActivityIndicatorManager.configureActivityIndicator($rootScope.activityIndicator,'success', text, time, true, action);
        },
        warning: function(text, time, action) {
            ActivityIndicatorManager.configureActivityIndicator($rootScope.activityIndicator,'warning', text, time, true, action);
        },
        info: function(text, time, action) {
            ActivityIndicatorManager.configureActivityIndicator($rootScope.activityIndicator,'info', text, time, true, action);
        },
        error: function(text, time, action) {
            ActivityIndicatorManager.configureActivityIndicator($rootScope.activityIndicator,'error', text, time, true, action);
        }
    };
}]);

app.factory("ChartActivityIndicator", ["ActivityIndicatorManager", function(ActivityIndicatorManager) {
    return {
        buildDefaultActivityIndicator: function () {
            return ActivityIndicatorManager.buildDefaultActivityIndicator()
        },
        displayBackendError: function (chartActivityIndicator, errorMessage) {
            ActivityIndicatorManager.configureActivityIndicator(chartActivityIndicator, 'error', errorMessage, 5000, false);
        },
        waiting: function(chartActivityIndicator, text) {
            ActivityIndicatorManager.configureActivityIndicator(chartActivityIndicator, 'waiting', text);
        },
        hide: function(chartActivityIndicator) {
            ActivityIndicatorManager.hide(chartActivityIndicator);
        },
        success: function(chartActivityIndicator, text, time, action) {
            ActivityIndicatorManager.configureActivityIndicator(chartActivityIndicator,'success', text, time, false, action);
        },
        warning: function(chartActivityIndicator, text, time, action) {
            ActivityIndicatorManager.configureActivityIndicator(chartActivityIndicator,'warning', text, time, false, action);
        },
        info: function(chartActivityIndicator, text, time, action) {
            ActivityIndicatorManager.configureActivityIndicator(chartActivityIndicator,'info', text, time, false, action);
        },
        error: function(chartActivityIndicator, text, time, action) {
            ActivityIndicatorManager.configureActivityIndicator(chartActivityIndicator,'error', text, time, false, action);
        }
    };
}]);

app.factory("APIXHRService", ["$rootScope", "$http", "$q", "Logger", "HistoryService", function($rootScope, $http, $q, Logger, HistoryService) {
    $rootScope.httpRequests = [];

    var unloadingState = false;

    $(window).bind("beforeunload", function() {
        unloadingState = true;
    });

    // Return a proxified promise that can be disabled
    function disableOnExit(promise) {

        function isEnabled() {
            return !unloadingState;
        }

        var deferred = $q.defer();

        // $q promises
        promise.then(function(data) {

                if (isEnabled()) {
                    deferred.resolve(data);
                }
            },
            function(data) {
                if (isEnabled()) {
                    deferred.reject(data);
                }
            },
            function(data) {
                if (isEnabled()) {
                    deferred.notify(data);
                }
            });

        // $http specific
        if (promise.success) {
            deferred.promise.success = function(callback) {
                promise.success(function(data, status, headers, config, statusText, xhrStatus) {
                    if (isEnabled()) {
                        callback(data === 'null' ? null : data, status, headers, config, statusText, xhrStatus);
                    }
                });
                return deferred.promise;
            };
        }

        if (promise.error) {
            promise.error(function(data, status, headers) {
                var apiError = getErrorDetails(data, status, headers);
                Logger.error("API error: ", apiError.errorType + ": " + apiError.message);
            })
            deferred.promise.error = function(callback) {
                promise.error(function(data, status, headers, config, statusText, xhrStatus) {
                    if (isEnabled()) {
                        callback(data, status, headers, config, statusText, xhrStatus);
                    }
                });
                return deferred.promise;
            };
        }

        if (promise.noSpinner) {
            deferred.promise.noSpinner = promise.noSpinner;
        }

        return deferred.promise;
    }

    return function(method, url, data, spinnerMode, contentType, timeoutPromise) {
        var headers = {
            'Content-Type': contentType === 'json' ? 'application/json;charset=utf-8' : 'application/x-www-form-urlencoded;charset=utf-8'
        };
        var versionIdHeader = "version-id";

        if ($rootScope.versionId && !isAbsoluteURL(url)) {
            headers[versionIdHeader] = $rootScope.versionId;
        }

        var start = new Date().getTime();
        Logger.debug("[S] " + method + ' ' + url);

        var params = {
            method: method,
            url: url,
            headers: headers,
            transformRequest: function(data) {
                // Transform data based on the content type
                if (contentType === 'json') {
                    return angular.isObject(data) ? JSON.stringify(data) : data;
                } else {
                    return angular.isObject(data) && String(data) !== '[object File]' ? jQuery.param(data) : data;
                }
            },
        };
        if ($rootScope.appConfig) {
            params.xsrfCookieName = $rootScope.appConfig.xsrfCookieName;
        }
        if (method == 'GET') {
            params.params = data;
        } else {
            params.data = data;
        }
        if (timeoutPromise) {
            params.timeout = timeoutPromise;
        }

        var promise = $http(params);
        var disableSpinner = spinnerMode && spinnerMode == "nospinner";

        const logDone = function (result) {
            const end = new Date().getTime();
            const details = [];
            if (result && result.status) {
                details.push("s=" + result.status);
            }
            details.push("f=" + (end - start) + "ms");
            if (result != null && result.headers != null) {
                const backendTime = result.headers("DKU-Call-BackendTime");
                if (backendTime != null) {
                    details.push("b=" + backendTime + "ms");
                }
                const versionId = result.headers(versionIdHeader);
                // The "Version-Id" header is *generally* present but it is not always guaranteed (e.g. 'hideVersionStringsWhenNotLogged')
                // The absence of this header can help diagnose spoofed requests (e.g. blocked by WAF, reverse proxy issues, etc)
                details.push("v=" + (versionId == null ? "?" : versionId));
            }
            Logger.debug("[D] " + method + " " + url + " (" + details.join(", ") + ")");
        };
        promise.then(logDone, logDone);

        if (!disableSpinner) {
            promise.spinnerMode = spinnerMode;
            $rootScope.httpRequests.push(promise);

            var removeRequest = function(result) {
                var idx = $rootScope.httpRequests.indexOf(promise);
                if (idx != -1) $rootScope.httpRequests.splice(idx, 1);
            };

            promise.noSpinner = function() {
                removeRequest();
                safeApply($rootScope);
                return promise;
            };

            promise.spinner = function(enable = true) {
                if (enable === false) {
                    promise.noSpinner();
                }
                return promise;
            };

            promise.then(removeRequest, removeRequest);
        }

        function isAbsoluteURL(url) {
            try {
                new URL(url);
            } catch (e) {
                if (e instanceof TypeError) {
                    return false;
                } else {
                    throw e;
                }
            }
            return true;
        }

        function retrieveVersionId(result) {
            if (result.headers) {
                const versionId = result.headers(versionIdHeader)

                if (versionId && !$rootScope.versionId) {
                    $rootScope.versionId = versionId;
                }
            }
        }

        promise.then(retrieveVersionId, retrieveVersionId);

        function promptRefreshOnVersionMismatch(result) {
            if (result.headers) {
                const promptRefresh = Boolean(result.headers("version-mismatch-refresh-prompt"));
                const backendVersionId = result.headers(versionIdHeader);

                if (promptRefresh && $rootScope.versionId &&
                    $rootScope.versionId !== $rootScope.promptRefreshFromVersionId &&
                    backendVersionId !== $rootScope.promptRefreshFromVersionId) {
                    $rootScope.promptRefresh = promptRefresh;
                    $rootScope.promptRefreshFromVersionId=backendVersionId;
                }
            }
        }

        promise.then(promptRefreshOnVersionMismatch, promptRefreshOnVersionMismatch);


        if (method=="POST") HistoryService.recordItemPost(url, data);

        return app.addSuccessErrorToPromise(promise);
    };
}]);


app.factory("CreateModalFromDOMElement", function(CreateModalFromHTML) {
    return function(selector, scope, controller, afterCompileCallback) {
        return CreateModalFromHTML($(selector).html(), scope, controller, afterCompileCallback, false);
    };
});

app.factory("CreateModalFromTemplate", function(CreateModalFromHTML, $http, $templateRequest, $q) {
    /**
     * Callback for preparing the modal scope
     *
     * @callback afterCompileCallback
     * @param {Object} modalScope the initialized modal scope
     * @returns {void}
     */


    /**
     * Create a modal popup from a template
     *
     * @param {String} location path to the modal HTML template
     * @param {Object} scope The scope variable that can be passed along and used in the controller
     * @param {String} controller The name of the controller that the modal scope should inherit from during initialization
     * @param {afterCompileCallback} afterCompileCallback an anonymous function that will be executed after the controller initialization
     * @param noFocus
     * @param backdrop can be 'true', 'false' or 'static' (check bootstrap documentation) or custom 'confirm' value (in this case a click on the backdrop will be treated as a confirmation on the modal)
     * @param keyboard
     */
    return function(location, scope, controller, afterCompileCallback, noFocus, backdrop, keyboard) {
        var deferred = $q.defer();
        $templateRequest(location).then(function(template) {
            if (angular.isArray(template)) {
                template = template[1];
            } else if (angular.isObject(template)) {
                template = template.data;
            }
            deferred.resolve(CreateModalFromHTML(template, scope, controller, afterCompileCallback, noFocus, backdrop, keyboard));
        });
        return deferred.promise;
    }
});

/**
 * Create a modal from an AngularJS component
 *
 * Usage:
 *
 *  CreateModalFromComponent(injectedComponentDirective, {
 *       paramOne: ...,
 *       paramTwo: ...,
 *  }, ['custom-modal-class-1', 'custom-modal-class-2']);
 *
 *  1) 'paramOne' & 'paramTwo' will be respectively bound to the 'param-one' and 'param-two' inputs of the component
 *  2) 'injectedComponentDirective' is the component's directive
 *     /!\ For a component('myComponent') you need to inject 'myComponentDirective' (AngularJS magically appends a 'Directive' suffix to all components names)
 *  3) the 3rd parameter is an optional array of CSS classes to add to the modal (for example, 'modal-wide')
 */
app.factory("CreateModalFromComponent", function(CreateModalFromHTML, $rootScope) {
    function camelToKebab(input) {
        return input.replace(/([A-Z])/g, function($1){return "-"+$1.toLowerCase();});
    }

    function getComponentTagName(component) {
        return camelToKebab(component[0].name); // Lol
    }

    return function(component, params, customClasses) {
        const newScope = $rootScope.$new(true);
        const tagName = getComponentTagName(component);
        let html = '<div class="modal modal3';
        if (customClasses) {
            for (let customClass of customClasses) {
                html += ` ${customClass}`;
            }
        }
        html += '"><' + tagName + ' ';
        for (let p in params) {
            newScope[p] = params[p];
            html += camelToKebab(p) + "=" + '"' + p + '" ';
        }
        html += 'modal-control="modalControl"'
        html += '></'+tagName+ '><div>';
        const promise = CreateModalFromHTML(html, newScope, null, function($scope, element) {
            newScope.modalControl = {
                resolve: value => $scope.resolveModal(value),
                dismiss: () => $scope.dismiss()
            }
        });
        return promise;
    }
});

app.factory("CreateModalFromHTML", function($timeout, $compile, $q, $rootScope) {
    let activeModals = [];

    $rootScope.$on('dismissModals', function() {
        activeModals.forEach((modalScope)=> unregisterModal(modalScope));
    });

    function registerModal(modalScope) {
        activeModals.unshift(modalScope);
    }

    function unregisterModal(modalScope) {
        activeModals = activeModals.filter((activeModalScope)=> {
            if(modalScope == activeModalScope) {
                modalScope.dismiss();
                return false;
            }
            return true;
        });
    }

    return function(template, scope, controller, afterCompileCallback, noFocus, backdrop, keyboard) {
        var deferred = $q.defer();
        var newDOMElt = $(template);
        if (controller != null) {
            newDOMElt.attr("ng-controller", controller);
        }
        newDOMElt.addClass("ng-cloak");

        const $existingModal = $('div.modal-container');
        let stackedClass = "";
        let waitForTransition = 0;
        if ($existingModal.length>0) {
            $existingModal.addClass('aside').removeClass('restored'); //move aside any existing modal in case of stacking
            waitForTransition = 250;
            stackedClass = "new-stacked"
        }

        var wrapper = $("<div>").addClass("modal-container " + stackedClass).append(newDOMElt);
        $("body").append(wrapper);

        if(scope.fromAngularContext === true) {
            wrapper.addClass("modal-container--from-angular");
        }

        $timeout(function() {
            var newScope = scope.$new();
            $compile(newDOMElt)(newScope);

            var modalScope = angular.element(newDOMElt).scope();

            if (afterCompileCallback) {
                modalScope.$apply(afterCompileCallback(modalScope, newDOMElt));
            }
            newDOMElt.on('hidden', function(e){
                if (e.target == newDOMElt.get(0)) {
                    unregisterModal(modalScope);
                    wrapper.remove();
                    modalScope.$destroy();
                    if (deferred != null) {
                        deferred.reject("modal hidden");
                        deferred = null;
                    }
                }
            });

            var prepareForModalStack = function () {
                $('div.modal-backdrop').addClass('modal-rollup').removeClass('non-see-through'); //mjt in the event of stacking a modal
                $("div.modal-container.new-stacked").addClass("modal-stacked-on-top").removeClass("new-stacked");
            }
            prepareForModalStack();

            if (backdrop) {
                 newDOMElt.attr('data-backdrop', backdrop === 'confirm' ? 'static' : backdrop);
            }

            newDOMElt.modal("show");
            $rootScope.$broadcast("dismissPopovers");

            modalScope.unwindModalStack = function (newDOMElt) {
                $('div.modal-backdrop.modal-rollup').removeClass('modal-rollup').click(modalScope.dismiss);
                $('div.modal-container.aside').removeClass('aside').addClass('restored'); //move aside any existing modal in case of stacking
            };
            modalScope.dismiss = function() {
                newDOMElt.modal("hide");
                if (deferred != null) {
                    deferred.reject("dismissed modal");
                    deferred = null;
                }
            };

            registerModal(modalScope);

            modalScope.resolveModal = function(value) {
                if (deferred != null) {
                    deferred.resolve(value);
                    deferred = null;
                }
                newDOMElt.modal("hide");
            };
            modalScope.$modalScope = modalScope;

            $(newDOMElt).on('hide.bs.modal', function (e) {
                if (modalScope && modalScope.canCloseModal && typeof modalScope.canCloseModal === 'function') {
                    if (!modalScope.canCloseModal()) {
                        e.preventDefault();
                        e.stopImmediatePropagation();
                        return false;
                    }
                }
                modalScope.unwindModalStack(newDOMElt);
            });

            if (backdrop === 'confirm') {
                // a click on the backdrop will be treated as a confirmation
                     $(".modal-backdrop").on('click', function() {
                            modalScope.confirm();
                    });
            }

            modalScope.$on("dismissModal", modalScope.dismiss);

            if (!noFocus) {
                // the first form of the modal, should contain the modal-body in 99% of cases
                var firstForm = newDOMElt.find('form').first();
                // list of focusable elements we want to try, in order of preference
                var focusCandidateFinders = [];
                focusCandidateFinders.push(function() {
                    return firstForm.find('input[type="text"]:not([readonly])').first();
                });
                focusCandidateFinders.push(function() {
                    return firstForm.find('button:submit').first();
                });
                focusCandidateFinders.push(function() {
                    return firstForm.find('button:button').first();
                });
                // if the modal has no form, or footer buttons are not in the form, look in the full modal
                focusCandidateFinders.push(function() {
                    return newDOMElt.find('input[type="text"]:not([readonly])').first();
                });
                focusCandidateFinders.push(function() {
                    return newDOMElt.find('button:submit').first();
                });
                focusCandidateFinders.push(function() {
                    return newDOMElt.find('button:button').first();
                });
                focusCandidateFinders.push(function() {
                    return newDOMElt.find('.close').first();
                });

                var focusCandidate;
                for (var i = 0; i < focusCandidateFinders.length; i++) {
                    var focusCandidateFinder = focusCandidateFinders[i];
                    focusCandidate = focusCandidateFinder().not('.no-modal-autofocus');
                    if (focusCandidate.length > 0) {
                        focusCandidate.focus();
                        // in some cases the element is disabled by a ng-disabled, and the focus behavior becomes a bit erratic
                        // so for safety we focus once more
                        $timeout(function() {
                            focusCandidate.focus();
                        });
                        break;
                    }
                }

                // in case the submit button is dangerous, prevent submit-on-enter
                if (firstForm.length > 0 && focusCandidate.hasClass('btn--danger')) { //NOSONAR: focusCandidate always initialized thanks to jquery first() specs
                    focusCandidate.bind("keydown keypress", function(event) {
                        if (event.which === 13) {
                            event.preventDefault();
                        }
                    });
                }
            }
        }, waitForTransition);

        return deferred.promise;
    };
});


/**
 * Create a custom body-attached DOM element within a new scope.
 * The new scope is fitted with a "dismiss" function, which destroys the DOM
 * element and the scope.
 */
app.factory("CreateCustomElementFromTemplate", ["$timeout", "$compile", "$templateRequest",
    function($timeout, $compile, $templateRequest) {
        return function(location, scope, controller, afterCompileCallback, domInsertionCallback) {
            $templateRequest(location).then(function onSuccess(template) {
                if (angular.isArray(template)) {
                    template = template[1];
                } else if (angular.isObject(template)) {
                    template = template.data;
                }
                var newDOMElt = $(template);
                if (controller != null) {
                    newDOMElt.attr("ng-controller", controller);
                }

                if (domInsertionCallback != null) {
                    domInsertionCallback(newDOMElt);
                } else {
                    $("body").append(newDOMElt);
                }

                /* Now, compile the element, set its scope, call the callback */
                $timeout(function() {
                    var newScope = scope.$new();
                    $compile(newDOMElt)(newScope);
                    var newScope2 = angular.element(newDOMElt).scope();

                    if (afterCompileCallback) {
                        newScope2.$apply(afterCompileCallback(newScope2));
                    }
                    newScope2.$on("dismissModalInternal_", function() {
                        $timeout(function() {
                            newScope2.dismiss();
                        }, 0);
                    });
                    newScope2.dismiss = function() {
                        newDOMElt.remove();
                        newScope2.$destroy();
                    };
                    scope.$on("$destroy", newScope2.dismiss);
                });
            });
        };
    }
]);


/** Keeps a map of promises for static API calls */
app.factory("CachedAPICalls", function(DataikuAPI, Assert, $http, $rootScope, $q, translate, $translate, AssetsUtils) {
    var deferredLogin = $q.defer();
    var whenLoggedIn = deferredLogin.promise;

    let staticData = app.addSuccessErrorToPromise(whenLoggedIn.then(() => DataikuAPI.staticData.getStaticData($translate.proposedLanguage())));

    return {
        notifyLoggedIn: () => {
            deferredLogin.resolve();
        },
        processorsLibrary: app.addSuccessErrorToPromise(whenLoggedIn.then(() => DataikuAPI.shakers.getProcessorsLibrary($translate.proposedLanguage()).success(function(processors) {
            processors.deprecatedTypes = [];

            // Inject the doc link at this point so it is only done once.
            processors.processors.forEach(function(p) {
                if (p.docPage) {
                    p.help += "\n\n" + translate("SHAKER.PROCESSORS.HELP.MORE_INFO",
                        "For more info, <a target=\"_blank\" href=\"{{docPageUrl}}\">please see the processor's reference</a>",
                        { docPageUrl: $rootScope.versionDocRoot + "preparation/processors/" + p.docPage + ".html"})
                        + "\n";
                }

                if (p.deprecated) {
                    processors.deprecatedTypes.push(p.type);

                    var deprecationMsg = "### " + translate("SHAKER.PROCESSORS.HELP.DEPRECATED", "This processor is deprecated.");
                    if (p.replacementDocLink) {
                        var replacementLink = $rootScope.versionDocRoot + p.replacementDocLink + ".html";
                        var replacementOptions = p.replacementName ? (p.replacementName + " options") : "options";
                        deprecationMsg += ` Please see <a target="_blank" href="${replacementLink}">our documentation</a> for our newer ${replacementOptions}.`;
                    }
                    p.help = deprecationMsg + "\n\n---\n\n" + p.help;
                    p.enDescription = "<span class=\"deprecation-tag\">"+ translate("SHAKER.PROCESSORS.DESCRIPTION.DEPRECATED", "[DEPRECATED]") + " </span>" + p.enDescription;
                }
            });

            return processors
        }))),
        staticData: staticData,
        customFormulasFunctions: staticData.then(resp => resp.data.customFormulasFunctions),
        udafCustomFormulasFunctions: staticData.then(resp => resp.data.udafCustomFormulasFunctions),
        customFormulasReference: staticData.then(resp => resp.data.customFormulasReference),
        udafCustomFormulasReference: staticData.then(resp => resp.data.udafCustomFormulasReference),
        datasetCommonCharsets: staticData.then(resp => resp.data.datasetCommonCharsets),
        datasetFormatTypes: staticData.then(resp => resp.data.datasetFormatTypes),
        datasetTypes: staticData.then(resp => resp.data.datasetTypes),
        pmlGuessPolicies: staticData.then(resp => resp.data.pmlGuessPolicies),
        cmlGuessPolicies: staticData.then(resp => resp.data.cmlGuessPolicies),
        timezonesList: staticData.then(resp => resp.data.timezonesList),
        timezonesShortList: staticData.then(resp => resp.data.timezonesShortList),
        webAppTypes: staticData.then(resp => resp.data.webAppTypes),
        mlCommonDiagnosticsDefinition: staticData.then(resp => resp.data.mlCommonDiagnosticsDefinition),
        pmlDiagnosticsDefinition: staticData.then(resp => function(backendType, predictionType) {
            const backendTypeNullable = backendType === null ? undefined : backendType;
            const matchingItem = resp.data.pmlDiagnosticsDefinition.find((item) => item.backendType === backendTypeNullable && item.predictionType === predictionType);
            return !matchingItem ? {} : matchingItem.definitions;
        }),
        cmlDiagnosticsDefinition: staticData.then(resp => function(backendType) {
            const backendTypeNullable = backendType === null ? undefined : backendType;
            const matchingItem = resp.data.cmlDiagnosticsDefinition.find((item) => item.backendType === backendTypeNullable);
            return !matchingItem ? {} : matchingItem.definitions;
        }),
        flowIcons: $http.get(AssetsUtils.appendAssetsHash("/static/dataiku/flow-iconset.json")),
        emojisTable: $http.get(AssetsUtils.appendAssetsHash("/static/third/emoji.json")).then(function(response) {
            Assert.trueish(response.data, 'No emoji returned');
            Assert.trueish(angular.isArray(response.data), 'Emojis were not returned as an array');
            const emojisTable = {};
            response.data.forEach(function(x) {
                emojisTable[x['sn']] = x['code'].split('-').map(x => '&#x' + x + ';').join('');
            });
            return emojisTable;
        })
    };
});


app.service('ComputablesService', function(CreateModalFromTemplate, DataikuAPI, TaggableObjectsUtils, Dialogs) {
    this.clear = function(scope, taggableItems) {
        return CreateModalFromTemplate("/templates/taggable-objects/clear-data-modal.html", scope, null, function(modalScope) {
            modalScope.taggableItems = taggableItems;
            modalScope.itemsType = TaggableObjectsUtils.getCommonType(taggableItems, it => it.type);

            modalScope.confirm = function() {
                DataikuAPI.taggableObjects.clear(taggableItems).success(function(data) {
                    if (data.anyMessage && !data.success) {
                        modalScope.dismiss();
                        Dialogs.infoMessagesDisplayOnly(scope, "Clear result", data);
                    } else {
                        modalScope.resolveModal(data);
                    }
                }).error(setErrorInScope.bind(scope));
            }
        });
    };
});

app.service('ManagedFoldersService', function($rootScope, $q, DataikuAPI, Logger, Notification, ComputablesService, FutureProgressModal, CreateModalFromTemplate, FlowGraph) {
    const svc = this;

    svc.clear = function(scope, projectKey, managedFolderId, managedFolderName) {
        return ComputablesService.clear(scope, [{
            type: 'MANAGED_FOLDER',
            projectKey: projectKey,
            id: managedFolderId,
            displayName: managedFolderName
        }]);
    };
});

app.service('DatasetsService', function($rootScope, $q, DataikuAPI, Logger, Notification, ComputablesService, FutureProgressModal, CreateModalFromTemplate, FlowGraph) {
    const svc = this;

    const listsWithAccessiblePerProject = {};

    svc.listWithAccessible = function(projectKey) {
        if (listsWithAccessiblePerProject[projectKey] == null) {
            listsWithAccessiblePerProject[projectKey] = DataikuAPI.datasets.listWithAccessible(projectKey);
        }
        return listsWithAccessiblePerProject[projectKey];
    };

    svc.clear = function(scope, projectKey, datasetName) {
        return ComputablesService.clear(scope, [{
            type: 'DATASET',
            projectKey: projectKey,
            id: datasetName,
            displayName: datasetName
        }]);
    };

    svc.refreshSummaries = function (scope, selectedItems, computeRecords = true, forceRecompute = false, displayErrorsInFlow = false) {
        const deferred = $q.defer();
        DataikuAPI.datasets.refreshSummaries(selectedItems, computeRecords, forceRecompute).success(function (data) {
            FutureProgressModal.show(scope, data, "Refresh datasets status")
                .then(data => deferred.resolve(data), data => deferred.reject(data));
        }).error(displayErrorsInFlow ? FlowGraph.setError() : setErrorInScope.bind(scope));
        return deferred.promise;
    };

    svc.setVirtualizable = function(scope, selectedItems, virtualizable) {
        const datasets = selectedItems.filter(it => it.type == 'DATASET');
        return DataikuAPI.datasets.setVirtualizable(datasets, !!virtualizable)
            .error(setErrorInScope.bind(scope));
    };

    svc.startSetAutoCountOfRecords = function(selectedItems) {
        return CreateModalFromTemplate("/templates/datasets/set-auto-count-of-records-modal.html", $rootScope, null, function(modalScope) {
            modalScope.autoCountOfRecords = false;

            modalScope.ok = function(vitualizable) {
                svc.setAutoCountOfRecords(selectedItems, modalScope.autoCountOfRecords)
                    .then(modalScope.resolveModal, setErrorInScope.bind(modalScope));
            };
        });
    };

    svc.setAutoCountOfRecords = function(selectedItems, autoCountOfRecords) {
        return DataikuAPI.datasets.setAutoCountOfRecords(selectedItems, autoCountOfRecords);
    };

    Notification.registerEvent("datasets-list-changed", function(evt, message) {
        Logger.info("Datasets list changed, updating");
        delete listsWithAccessiblePerProject[message.projectKey]; // just invalidate
    });
});

/** Cached access to datasets information */
app.factory("DatasetInfoCache", function($stateParams, DataikuAPI, $q, Notification, Logger) {
    // Cache for results of datasets/get
    var simpleCache = {}
    Notification.registerEvent("websocket-status-changed", function() {
        Logger.info("Websocket status change, dropping dataset cache");
        simpleCache = {};
    });
    Notification.registerEvent("datasets-list-changed", function(evt, message) {
        Logger.info("Datasets list changed, dropping cache for ", message.projectKey);
        delete simpleCache[message.projectKey];
    });
    var svc = {
        getSimple: function(projectKey, name) {
            var projectCache = simpleCache[projectKey];
            if (projectCache != null) {
                var data = projectCache[name];
                if (data != null) {
                    Logger.info("Cache hit: " + projectKey + "." + name);
                    return $q.when(data);
                }
            } else {
                simpleCache[projectKey] = {};
            }
            Logger.info("Cache miss: " + projectKey + "." + name);
            return DataikuAPI.datasets.get(projectKey, name, $stateParams.projectKey).then(function(data) {
                simpleCache[projectKey][name] = data.data;
                return data.data;
            });
        }
    }
    return svc;
});

// Queue
//
// var lockable = Queue();
// lockable.exec(function() { alert('A'); }); // Executed right now
// var unlock = lockable.lock(); // Lock the object
// lockable.exec(function() { alert('B'); }); // Not executed
// lockable.exec(function() { alert('C'); }); // Not executed
// unlock(); // Execute alert('A') & alert('B');
//
// The queue can be tied to a promise
// lockable.lockOnPromise(DataikuAPI.xxx(yyy,zzz).success(function() {
//     ...
// }));
//
app.factory("Queue", function() {

    return function() {

        var semaphore = 0;
        var queue = [];
        var destroyed = false;
        var scopeUnregisterer = undefined;
        var inside = false;

        var processQueue = function() {
            while (!destroyed && semaphore == 0 && queue.length > 0) {
                if (!inside) {
                    try {
                        inside = true;
                        queue.splice(0, 1)[0]();
                    } finally {
                        inside = false;
                    }
                } else {
                    break;
                }
            }
        };

        var exec = function(fn) {
            if (fn && !destroyed) {
                queue.push(fn);
                processQueue();
            }
        };

        var destroy = function() {
            destroyed = true;
            queue = [];
        };

        var wrap = function(func) {
            return function() {
                var args = arguments;
                exec(function() {
                    if (func) {
                        func.apply(null, args);
                    }
                });
            };
        };

        var lock = function() {
            if (destroyed) {
                return;
            }
            semaphore++;
            var unlocked = false;
            return function() {
                if (!unlocked) {
                    semaphore--;
                    unlocked = true;
                    processQueue();
                }
            };
        };

        var ret = {

            withScope: function(scope) {
                if (scopeUnregisterer) {
                    scopeUnregisterer();
                }
                if (scope) {
                    scopeUnregisterer = scope.$on('$destroy', destroy);
                }
                return ret;
            },
            locked: function() {
                return destroyed || semaphore > 0;
            },
            exec: function(fn) {
                exec(fn);
            },
            wrap: function(fn) {
                return wrap(fn);
            },
            lockOnPromise: function(promise) {
                if (promise && promise['finally']) {
                    var unlocker = lock();
                    promise['finally'](function() {
                        unlocker();
                    });
                }
                return promise;
            },
            lock: function() {
                return lock();
            }
        };
        return ret;
    };

});

function enrichFuturePromise(promise) {
    promise.success = function(fn) {
        promise.then(function(data) {
            fn(data.data, data.status, data.headers);
        });
        return promise;
    };

    promise.error = function(fn) {
        promise.then(null, function(data) {
            fn(data.data, data.status, data.headers);
        });
        return promise;
    };

    promise.update = function(fn) {
        promise.then(null, null,function(data) {
            fn(data.data, data.status, data.headers);
        });
        return promise;
    };
    return promise;
}

// MonoFuture
//
// - Wait for future result
// - Abort the previous future as soon as a new one is started
//
// Usage :
//
// var monoFuture = MonoFuture(scope); // If a scope is passed, life time of monofuture = life time of the scope
// monoFuture.exec(DataikuAPI.xxx.getYYYFuture(zzz)).success(function(data) {
// ...
// }).update(function(data) {
// ...
// }).error(...);
//
//  OR by wrapping the DataikuAPI function directly
//
//  var apiCall = MonoFuture().wrap(DataikuAPI.xxx.getYYYFuture);
//  apiCall(zzz).success(...);
//
//
//

app.factory("MonoFuture", ["$q", "DataikuAPI", "Throttle", function($q, DataikuAPI, Throttle) {

    return function(scope, spinner=true, delay=1000) {

        var promises = [];
        var destroyed = false;

        // Refresh the state of the last promise
        var refreshFutureState = Throttle().withDelay(delay).wrap(function() {

            updateInternalState();

            if (promises.length > 0) {
                var last = promises[promises.length - 1];
                if (!last.hasResult && !last.waiting) {
                    last.waiting = true;
                    DataikuAPI.futures.getUpdate(last.id).success(function(data, status, headers) {

                        last.waiting = false;
                        last.result = {data: data, status: status, headers: headers};
                        if (data.hasResult) {
                            last.hasResult = true;
                        } else {
                            if (!last.aborted) {
                                last.deferred.notify(last.result);
                            }
                            refreshFutureState();
                        }

                        updateInternalState();

                    }).error(function(data, status, headers) {

                        last.failed = true;
                        last.result = {data: data, status: status, headers: headers};
                        last.hasResult = true;
                        last.waiting = false;

                        updateInternalState();
                    });
                }
            }
        });

        // Update current state
        var updateInternalState = function() {
            // Abort all abortable futures & remove them
            var loop = false;
            do {
                loop = false;
                for (var i = 0; i < promises.length; i++) {
                    var isLast = i == (promises.length - 1);
                    var promise = promises[i];
                    promise.aborted |= !isLast;
                    if (promise.aborted && !promise.waiting) {

                        // We have the future id, and not the result meaning that the future is running
                        if (promise.id && !promise.hasResult) {
                            // Abort me
                            DataikuAPI.futures.abort(promise.id, spinner);
                        }

                        promises.splice(i, 1);
                        loop = true;
                        break;
                    }
                }
            } while (loop);

            // Check last one : finished?
            if (promises.length > 0) {
                var last = promises[promises.length - 1];
                if (last.hasResult && !last.aborted) {
                    promises.splice(promises.length - 1, 1);
                    if (last.failed) {
                        last.deferred.reject(last.result);
                    } else if (last.result && last.result.data && last.result.data.aborted) {
                        // The future has been aborted by someone
                        last.deferred.reject(last.result);
                    } else {
                        last.deferred.resolve(last.result);
                    }
                }
            }
        };

        var fakePromise = function() {
            var promise = $q.defer().promise;
            promise.success = function() {
                return promise;
            };
            promise.error = function() {
                return promise;
            };
            promise.update = function() {
                return promise;
            };
            return promise;
        };

        var exec = function(apiPromise) {
            if (destroyed || !apiPromise || !apiPromise.success) {
                return fakePromise();
            }
            var deferred = $q.defer();
            var promise = {
                id: null,
                hasResult: false,
                result: undefined,
                deferred: deferred,
                aborted: false,
                failed: false,
                waiting: true,
                noSpinner: apiPromise.noSpinner
            };
            promises.push(promise);
            updateInternalState();
            apiPromise.success(function(data, status, headers) {
                promise.waiting = false;
                promise.result = {data, status, headers};
                if (data) {
                    promise.id = data.jobId;
                }
                if (data.hasResult) {
                    promise.hasResult = true;
                } else {
                    refreshFutureState();
                }
                updateInternalState();
            }).error(function(data, status, headers) {
                promise.failed = true;
                promise.result = {data, status, headers};
                if (data) {
                    promise.id = data.jobId;
                }
                promise.hasResult = true;
                promise.waiting = false;
                updateInternalState();
            });

            enrichFuturePromise(deferred.promise);

            deferred.promise.noSpinner = function() {
                promises.forEach(function(p) {
                    if (p.noSpinner) p.noSpinner();
                });
                return deferred.promise;
            };

            return deferred.promise;
        };

        var abort = function() {
            if (promises.length > 0) {
                promises[promises.length - 1].aborted = true;
                updateInternalState();
            }
        };

        var destroy = function() {
            abort();
            destroyed = true;
        };

        const active = function() {
            return promises.length > 0;
        };

        if (scope && scope.$on) {
            scope.$on('$destroy', destroy);
        }

        return {

            exec: exec,
            wrap: function(func) {
                return function() {
                    return exec(func.apply(func, arguments));
                };
            },
            abort: abort,
            destroy: destroy,
            active: active,
            destroyed: () => destroyed
        };
    };
}]);


/* MonoFuturePool
    Wrapper around MonoFuture that allows a bunch of related monofutures to run sequencially, never running more than one at a time
    API is mostly the same as for the MonoFuture, except: exec makes no sense for the pool you have to use wrap, and the returned promise doesn't have the noSpinner method.

   Example: imagine you have this kind of stuff in a component:

    const wrapper = MonoFuture(scope).wrap(DataikuApi.blah.blah);
    $scope.$watch('someApiParams', (nv) => {
        $scope.something = wrapper(nv)
    });

    And there are 20 instances of this component => you may have many future running in parallel, one for each component

    Using:
    // in some shared place
    const pooledMonoFuture = MonoFuturePool(scope);

    ...

    //in each component
    const wrapper = pooledMonoFuture.createMonoFuture(scope, false).wrap(DataikuApi.blah.blah);
    $scope.$watch('someApiParams', (nv) => {
        $scope.something = wrapper(nv)
    });

    Future calls from each component will be queued and only one will run at once.
    If any of the components triggers a new call (someApiParams changed) while a future is running, it will be put on hold to be executed once the running query is done. If any of the component triggers a new call while it already has a call on hold, the previous one is dropped and replaced by the new one.

    Timeline example:
    - Component 1 triggers a future with call A1 => api call is done & progress is tracked
    - Component 2 triggers a future with call A2 => put on hold
    - Component 3 triggers a future with call A3 => put on hold
    - Component 3 triggers a future with call B3 => put on hold, A3 is dropped
    - API call A1 is done => API call A2 is triggered and and progress is tracked
    - API call A2 is done => API call B3 is triggered and and progress is tracked

    key points:
    - only one future is in progress at any time
    - A3 has been dropped before even started because it has been discarded by B3
    - The results obtained by each component is no different than what they would have got independantly, except for the longer delay.

    When the pool is created with a scope, all child MonoFutures are destroyed when this scope is destroyed
    When a child MonoFuture is created with a scope, it's destroyed when this scope is destroyed (and releases the lock for the others when applicable)
    It is possible to use both scope-based auto-destroy at the same time (if the Pool is bound to a component but each MonoFuture to one of many chilren for example)
*/

app.factory("MonoFuturePool", ["$q", "$timeout", "MonoFuture", "Queue", function($q, $timeout, MonoFuture, Queue) {
    return function MonoFuturePool(poolScope) {
        const allMonoFutures = new Set();
        const lockable = Queue().withScope(poolScope);
        poolScope?.$on?.('$destroy', destroy);

        return {
            createMonoFuture,
            destroy,
        };

        function createMonoFuture(monoFutureScope, spinner, delay) {
            const monoFuture = MonoFuture(undefined, spinner, delay); // we don't give the scope to the base monofuture because we need to manage destroy here to release the lock
            let waitingTask // the next scheduled task for this MonoFuture. When the monofuture is wrapping an API call, running waitingTask is what triggers the API call.
            let releaseThisMonoFutureLockIfAny = () => {}; // method to unlock the current query lock - only works when current monoFuture holds the lock, otherwise it's a noop

            const pooledMonoFuture = {
                wrap: function(func) {
                    return function(...args) {
                        const pendingMonoFuturePromiseDeferred = $q.defer(); // simple wrapper on the MonoFuture.exec output to handle the fact we don't have it synchronously, but we need to return it

                        if (waitingTask) {
                            // there is already a waiting task for this MonoFuture => we don't need to schedule anything, we just override the waitingTask & it will be executed in place of the previously scheduled one.
                            waitingTask = () => func.apply(func, args);
                        } else {
                            // otherwise, we schedule the task for when the lock is released. If the lock is free, lockable.exec will run synchronously
                            waitingTask = () => func.apply(func, args);
                            lockable.exec(() => {
                                releaseThisMonoFutureLockIfAny = lockable.lock();

                                // at this point, waitingTask may have been overridden by a later call, or the monoFuture may have be destroyed
                                // we run the task if the monoFuture has not be destroyed / the task has not been discarded in-between
                                const apiPromise = monoFuture.destroyed() ? undefined : waitingTask?.(); // let's not even make the call if destroyed
                                waitingTask = undefined;

                                if (!apiPromise || !apiPromise.success) {
                                    // cases when there isn't anything to query anymore: monoFuture is destroyed, task cancelled, wrapped method didn't return a promise
                                    // nothing is pending aymore => release the lock
                                    releaseThisMonoFutureLockIfAny();
                                } else {
                                    // normal case: track progress using MonoFuture then release lock
                                    const run = monoFuture.exec(apiPromise).finally(() => {
                                        releaseThisMonoFutureLockIfAny();
                                    });
                                    pendingMonoFuturePromiseDeferred.resolve(run); // forwards resolve, notify & rejects from the inner MonoFuture to the outer pooled one.
                                }
                            });
                        }

                        return enrichFuturePromise(pendingMonoFuturePromiseDeferred.promise);
                    };
                },
                abort() {
                    waitingTask = undefined; // before release otherwise a waiting task could be triggered
                    releaseThisMonoFutureLockIfAny();
                    monoFuture.abort();
                },
                destroy() {
                    waitingTask = undefined;
                    // when many monoFuture instances from the same pool are destroyed at once, we could have each destroy release the lock let the next query start, just for it to be destroyed after
                    // this micro-delay allows the next task to only be executed after the end of a synchronous bulk delete.
                    $timeout(releaseThisMonoFutureLockIfAny, 0);
                    allMonoFutures.delete(pooledMonoFuture);
                    monoFuture.destroy();
                },
                active() {
                    return monoFuture.active() || waitingTask !== undefined;
                },
            }

            if (monoFutureScope && monoFutureScope.$on) {
                monoFutureScope.$on('$destroy', () => pooledMonoFuture.destroy());
            }

            allMonoFutures.add(pooledMonoFuture);
            return pooledMonoFuture;
        }

        function destroy() {
            [...allMonoFutures].forEach(mf => mf.destroy());
        }
    }

}]);


// Provide a way to take/release the spinner
//
// Usage:
//
//  var spinner = SpinnerService();   // The spinner is permanently released automatically when scope is destroyed
//  spinner.acquire();
//  /* spinning... */
//  spinner.release(); // It's safe to release it multiple time
//
//
//  The spinner can be tied to a promise like that:
//  SpinnerService.lockOnPromise(promise);
app.factory("SpinnerService", ["$rootScope", function($rootScope) {
    let fakeReq = {}; // Doesn't matter what this object is, it's never used anyway...
    let actives = 0; // Number of active spinners
    let scopeUnregisterer = null;

    // TODO : implement this properly
    // (currently it's  is a little hack around $rootScope.httpRequests)
    function update() {
        // Reset
        let idx = $rootScope.httpRequests.indexOf(fakeReq);
        if (idx != -1) {
            $rootScope.httpRequests.splice(idx, 1);
        }
        // Activate
        if (actives > 0) {
            $rootScope.httpRequests.push(fakeReq);
        }
    }

    function fnr() {
        let acquired = false;
        let destroyed = false;
        function acquire() {
            if (destroyed) {
                return;
            }
            if (!acquired) {
                acquired = true;
                actives++;
                update();
            }
        }
        function release() {
            if (acquired) {
                acquired = false;
                actives--;
                update();
            }
        }
        function destroy() {
            destroyed = true;
            release();
        }

        const ret = {
            acquire: acquire,
            release: release,
            destroy: destroy,
            withScope: function(scope) {
                if (scopeUnregisterer) {
                    scopeUnregisterer();
                }
                if (scope) {
                    scopeUnregisterer = scope.$on('$destroy', destroy);
                }
                return ret;
            }
        };
        return ret;
    };

    fnr.lockOnPromise = function(promise) {
        if (promise && promise['finally']) {
            var lock = fnr();
            lock.acquire();
            promise['finally'](function() {
                lock.release();
            });
        }
    }
    return fnr;
}]);


// Ability to merge watch calls into one (old value is kept, new value is updated)
function wrapWatchHelper(ctrl, fn) {
    var isSet = false;
    var oldVal = undefined;
    var newVal = undefined;

    var trueExec = ctrl.wrap(function() {
        if (!angular.equals(newVal, oldVal)) {
            isSet = false;
            fn(newVal, oldVal);
        }
    });

    return function(nv, ov) {
        if (isSet) {
            newVal = angular.copy(nv);
        } else {
            isSet = true;
            oldVal = angular.copy(ov);
            newVal = angular.copy(nv);
        }
        trueExec();
    };
};

// Debounce
// API is similar to Throttle but it implements a behavior similar to the "onSmartChange" directive as a service
//
// var fn = Debounce
//    .withDelay(500,1000)   // initial delay = 500ms, then delay = 1000ms
//    .withSpinner(true)
//    .wrap(function() {
//        console.log('hello');
//     });
//
// Example 1:
//
// fn(); // Will be executed in 500ms
//
// Example 2 :
// fn();  // Will be dropped
// [sleep less than 100ms]
// fn();  // Will be executed in 1000s
//
//
app.factory("Debounce", ["SpinnerService", "$rootScope", function(SpinnerService, $rootScope) {

    return function(scope) {

        var initialDelay = 0;
        var delay = 0;
        var destroyed = false;
        var spinner = SpinnerService();
        var enableSpinner = false;
        var stdTimer = null;
        var initTimer = null;
        var scopeUnregisterer = null;

        var exec = function(func) {

            if (destroyed) {
                return;
            }

            var wrapped = function debounceExecWrapped() {
                var toBeExec;
                if (func) {
                    spinner.release();
                    toBeExec = func;
                    func = null;
                }
                if (toBeExec) {
                    $rootScope.$apply(function debounceExec() {
                        toBeExec();
                    });
                } else {
                    // Important because the activity state may have changed!
                    $rootScope.$digest();
                }
            };

            var isFirst = true;
            if (stdTimer) {
                clearTimeout(stdTimer.key);
                stdTimer = null;
                isFirst = false;
            }

            if (initTimer) {
                clearTimeout(initTimer.key);
                initTimer = null;
                isFirst = false;
            }

            if (enableSpinner) {
                spinner.acquire();
            }

            if (isFirst) {

                initTimer = {
                    key: setTimeout(function() {
                        initTimer = null;
                        wrapped();
                    }, initialDelay)
                };

                stdTimer = {
                    key: setTimeout(function() {
                        stdTimer = null;
                        wrapped();
                    }, delay)
                };

            } else {

                stdTimer = {
                    key: setTimeout(function() {
                        stdTimer = null;
                        wrapped();
                    }, delay)
                };
            }
        }

        var abort = function() {
            spinner.release();
            if (initTimer) {
                clearTimeout(initTimer.key);
                initTimer = null;
            }
            if (stdTimer) {
                clearTimeout(stdTimer.key);
                stdTimer = null;
            }
        };

        var wrap = function debounceWrap(func) {
            return function debounceWrapped() {
                var args = arguments;
                var argsArray = Array.prototype.slice.call(arguments);

                if (argsArray.indexOf("SkipDebounceAndRunImmediately") !== -1) {
                    // eslint-disable-next-line no-console
                    console.debug("Skipping debounce, running function immediately");
                    argsArray = argsArray.filter(arg => arg != "SkipDebounceAndRunImmediately");
                    func.apply(null, argsArray);
                } else {
                    exec(function debounceWrappedCB() {
                        func.apply(null, args);
                    });
                }
            };
        };

        var destroy = function() {
            // eslint-disable-next-line no-console
            console.info("Destroy debounce on scope", scope); // NOSONAR
            destroyed = true;
            abort();
        };

        var ret = {
            exec: function(func) {
                exec(func);
            },
            wrap: function(func) {
                return wrap(func);
            },
            wrapWatch: function(fn) {
                return wrapWatchHelper(ret, fn);
            },
            active: function() {
                return !!(initTimer || stdTimer);
            },
            abort: function() {
                abort();
            },
            destroy: function() {
                destroy();
            },
            withDelay: function(newInitialDelay, newDelay) {
                delay = newDelay;
                initialDelay = newInitialDelay;
                return ret;
            },
            withSpinner: function(enabled) {
                enableSpinner = enabled;
                return ret;
            },
            withScope: function(scope) {
                if (scopeUnregisterer) {
                    scopeUnregisterer();
                }
                if (scope) {
                    scopeUnregisterer = scope.$on('$destroy', destroy);
                }
                spinner.withScope(scope);
                return ret;
            }
        };
        return ret;

    };
}]);


// Limit the maximum update frequency to 1/delay
// Some calls will be dropped in order to limit update frequency
//
// Usage:
//
// var throttle = Throttle().withScope(scope).withDelay(1000);  // If a scope is passed, life time of monofuture = life time of the scope
//
// throttle.exec(func1); // Executed now (= after the current event loop)
// throttle.exec(func2); // Dropped (because of func3)
// throttle.exec(func3); // Dropped (because of func4)
// throttle.exec(func4); // Executed 1 second later
//
// It's also possible to permanently wrap a function :
// myFunc = Throttle().wrap(function() {
//    ...
// });
// myFunc(); // executed (= after the current event loop)
// myFunc(); // dropped
// myFunc(); // delayed
//
app.factory("Throttle", ["$timeout", function($timeout) {

    return function() {

        var delay = 0;
        var currentlyWaitingOn = null;
        var storedFunc = null;
        var destroyed = false;
        var scopeUnregisterer = null;

        var waitCallback = function() {
            var toBeExec;
            if (storedFunc) {
                toBeExec = storedFunc;
                storedFunc = null;
                currentlyWaitingOn = $timeout(waitCallback, delay);
            } else {
                currentlyWaitingOn = null;
            }
            // Re-entrant safe
            if (toBeExec) {
                toBeExec();
            }
        }
        var exec = function(func) {
            if (destroyed) {
                return;
            }
            if (!func) {
                func = function() {
                };
            }
            if (currentlyWaitingOn) {
                storedFunc = func;
                // It will be called later :)
            } else {
                // Execute now
                // ... and setup a timeout to drop further calls for 'delay' ms
                $timeout(func);
                storedFunc = null;
                currentlyWaitingOn = $timeout(waitCallback, delay);
            }
        };

        var wrap = function(func) {
            return function() {
                var args = arguments;
                exec(function() {
                    func.apply(null, args);
                });
            };
        };

        var abort = function() {
            if (currentlyWaitingOn) {
                $timeout.cancel(currentlyWaitingOn);
                currentlyWaitingOn = null;
            }
            storedFunc = null;
        };

        var destroy = function() {
            abort();
            destroyed = true;
        };


        var ret = {
            exec: function(func) {
                exec(func);
            },
            wrap: function(func) {
                return wrap(func);
            },
            wrapWatch: function(fn) {
                return wrapWatchHelper(ret, fn);
            },
            active: function() {
                return !!storedFunc;
            },
            abort: function() {
                abort();
            },
            destroy: function() {
                destroy();
            },
            withDelay: function(newDelay) {
                delay = newDelay;
                return ret;
            },
            withScope: function(scope) {
                if (scopeUnregisterer) {
                    scopeUnregisterer();
                }
                if (scope) {
                    scopeUnregisterer = scope.$on('$destroy', destroy);
                }
                return ret;
            }
        };
        return ret;
    };

}]);


app.factory("DKUtils", function($rootScope, $state, $stateParams, $timeout, Logger) {
    return {
        /* Reflows at current digest */
        reflowNow: function() {
            $rootScope.$broadcast("reflow");
        },

        /* Reflows at next digest */
        reflowNext: function() {
            $timeout(function() {
                $rootScope.$broadcast("reflow")
            }, 0);
        },
        /* It's probably bad if you need this */
        reflowLater: function() {
            $timeout(function() {
                Logger.info("delayed reflow");
                $rootScope.$broadcast("reflow")
            }, 400);
        },
        /* Reload current state. Works around broken $state.reload() */
        reloadState: function() {
            return $state.transitionTo($state.current,
                angular.copy($stateParams),
                {reload: true, inherit: true, notify: true});
        }
    }
});


app.factory("PluginsService", function($rootScope, DataikuAPI, WT1, StateUtils, $q, DataikuCloudService) {
    const namingConvention = '^[a-z][a-z0-9-]*$';

    const isUniqueInPlugin = function(newComponentId, pluginId, pluginComponentsOfSameType) {
        return !pluginComponentsOfSameType.some(existingComponent => existingComponent.id == newComponentId
                               || `${pluginId}_${existingComponent.id}` === newComponentId
                               || existingComponent.id === `${pluginId}_${newComponentId}`
                               || `${pluginId}_${existingComponent.id}` === `${pluginId}_${newComponentId}`);
    }

    const validateComponentId = function(newComponentId, pluginId, pluginComponentsOfSameType) {
        if (!newComponentId) return [];
        const warnings = [];
        if (!new RegExp(namingConvention).test(newComponentId)) {
            warnings.push("Component ID is invalid. It should start with a letter and continue with letters, numbers or hyphens.");
        }
        if (newComponentId.startsWith(pluginId)) {
            warnings.push("Component ID starts with plugin ID.");
        }
        if (!isUniqueInPlugin(newComponentId, pluginId, pluginComponentsOfSameType)) {
            warnings.push("Component ID is not unique.");
        }
        return warnings;
    }

    const isValidComponentId = function(newComponentId, pluginId, pluginComponentsOfSameType) {
        if (!newComponentId) return false;
        return validateComponentId(newComponentId, pluginId, pluginComponentsOfSameType).length == 0;
    };

    const getLoadedDescForDataset = function(svc, datasetType) {
        return svc.getDatasetLoadedDesc(datasetType) ||
            svc.getFSProviderLoadedDesc(datasetType) ||
            svc.getSampleLoadedDesc(datasetType);
    };

    var svc = {
        namingConvention: namingConvention,
        transformToDevPlugin: function(modalScope, convertAPIFunc, getAPICallParams, eventWT1Name, componentType, originalType) {
            DataikuAPI.plugindev.list().success(function(data) {
                modalScope.devPlugins = data;
            }).error(setErrorInScope.bind(modalScope));

            modalScope.convert = {
                mode: 'NEW',
                pattern: namingConvention
            };

            const noErrorMessage = {"display": false, "message": ""};
            const mustNotStartWithPluginId = {"display": true, "message": "New plugin recipe id should not start with the plugin id."};
            const mustBeUnique = {"display": true, "message": "New plugin recipe id must be unique across the recipe components of this plugin."};
            modalScope.matchesTargetFolder = function(pluginId) {
                return modalScope.convert.targetFolder.startsWith(pluginId);
            }
            modalScope.matchesOtherComponent = function(pluginId) {
                const plugin = modalScope.devPlugins.find(_ => _.desc.id === pluginId);
                return !isUniqueInPlugin(modalScope.convert.targetFolder, pluginId, plugin.content[componentType])
            }

            modalScope.modalWarning = function() {
            /**
            we add a special warning to explicitly say what the problem is, but only when it may be hard to understand
            **/
                if (modalScope.convert.mode === 'EXISTING') {
                    if (modalScope.convert.targetFolder && modalScope.matchesTargetFolder(modalScope.convert.targetPluginId)) {
                        return mustNotStartWithPluginId;
                    }
                    if (modalScope.convert.targetPluginId && modalScope.matchesOtherComponent(modalScope.convert.targetPluginId)) {
                        return mustBeUnique;
                    }
                } else {
                    if (modalScope.convert.targetFolder && modalScope.matchesTargetFolder(modalScope.convert.newPluginId)) {
                        return mustNotStartWithPluginId;
                    }
                }
                return noErrorMessage;
            };

            modalScope.isIdValid = function() {
                if (modalScope.convert.mode === 'EXISTING') {
                    if (!modalScope.convert.targetPluginId) return false;
                    const plugin = modalScope.devPlugins.find(_ => _.desc.id === modalScope.convert.targetPluginId);
                    return isValidComponentId(modalScope.convert.targetFolder,
                                                            modalScope.convert.targetPluginId,
                                                            plugin.content[componentType]);
                }
                if (!modalScope.convert.newPluginId) return false;
                return isValidComponentId(modalScope.convert.targetFolder,
                                                        modalScope.convert.newPluginId,
                                                        []);
            };

            modalScope.go = function() {
                resetErrorInScope(modalScope);
                convertAPIFunc(...getAPICallParams(modalScope)).success(function(data) {
                    WT1.event(eventWT1Name, {original: originalType});
                    modalScope.reloadPluginConfiguration();
                    StateUtils.go.pluginEditor(data.pluginId, data.relativePathToOpen);
                }).error(setErrorInScope.bind(modalScope));
            };
        },
        validateComponentId: validateComponentId,
        isValidComponentId: isValidComponentId,
        getPluginDesc(pluginId) {
            return $rootScope.appConfig.loadedPlugins.find(x => x.id == pluginId);
        },
        isPluginLoaded: function(pluginId) {
            var i, plugin;
            for (i = 0; i < $rootScope.appConfig.loadedPlugins.length; i++) {
                plugin = $rootScope.appConfig.loadedPlugins[i];
                if (plugin.id == pluginId) return true;
            }
            return false;
        },
        getSampleLoadedDesc : function(datasetType) {
            return $rootScope.appConfig.customSampleDatasets.find(x => x.datasetType === datasetType);
        },
        getDatasetLoadedDesc : function(datasetType) {
            return $rootScope.appConfig.customDatasets.find(x => x.datasetType == datasetType);
        },
        getFSProviderLoadedDesc : function(fsProviderType) {
            return $rootScope.appConfig.customFSProviders.find(x => x.fsProviderType == fsProviderType);
        },
        getRecipeLoadedDesc : function(recipeType) {
            return $rootScope.appConfig.customCodeRecipes.find(x => x.recipeType == recipeType);
        },
        getOwnerPluginDesc: function(loadedDesc) {
            if (loadedDesc != null) {
                return $rootScope.appConfig.loadedPlugins.find(x => x.id == loadedDesc.ownerPluginId);
            } else {
                return null; // plugin most likely removed
            }
        },
        getRecipeIcon: function(recipeType) {
            var loadedDesc = svc.getRecipeLoadedDesc(recipeType);
            if (loadedDesc && loadedDesc.desc && loadedDesc.desc.meta && loadedDesc.desc.meta.icon) {
                return loadedDesc.desc.meta.icon;
            } else {
                var pluginDesc = svc.getOwnerPluginDesc(loadedDesc);
                if (pluginDesc) {
                    return pluginDesc.icon || "icon-visual_prep_sync_recipe";
                } else {
                    return "icon-visual_prep_sync_recipe"; // plugin has been removed
                }
            }
        },
        getDatasetIcon: function(datasetType) {
            const loadedDesc = getLoadedDescForDataset(svc, datasetType);
            const defaultSampleDatasetIcon = "fas fa-flask";

            if (loadedDesc == null) {
                return datasetType.startsWith("Sample_") ? defaultSampleDatasetIcon : "icon-question-sign";
            }

            return loadedDesc.desc?.meta?.icon ||
                svc.getOwnerPluginDesc(loadedDesc)?.icon ||
                (datasetType.startsWith("Sample_") ? defaultSampleDatasetIcon : "icon-puzzle-piece");
        },
        getDatasetLabel : function(datasetType) {
            return getLoadedDescForDataset(svc, datasetType)?.desc?.meta?.label || datasetType;
        },
        checkInstalledStatus: function(pluginId, forceFetch) {
            return $q((resolve, reject) => {
                DataikuAPI.plugins.list(forceFetch)
                .success(function(data) {
                    const isInstalled = data.plugins.some(plugin => plugin.id === pluginId && plugin.installed);
                    resolve(isInstalled);
                })
                .error(reject)
            });
        },
        getPluginVersion: function(pluginId, forceFetch) {
            return $q((resolve, reject) => {
                DataikuAPI.plugins.list(forceFetch)
                .success(function(data) {
                    const plugin = data.plugins.find(plugin => plugin.id === pluginId);
                    if (plugin && plugin.installedDesc && plugin.installedDesc.desc) {
                        resolve(plugin.installedDesc.desc.version);
                    } else {
                        reject(undefined);
                    }
                })
                .error(reject)
            });
        },
        /**
         * Get the link to the plugin details store listing of the plugin with the specified ID
         * @param {String} pluginId
         * @param {Boolean} isPluginInstalled
         * @param {Boolean} isAppDeploymentCloud
         * @returns {String} If DSS cloud is detected the listing will be the public wordpress info page. Otherwise the local store listing will be returned\
         */
        getPluginDetailsLink: function(pluginId, isPluginInstalled, isAppDeploymentCloud) {
            if (isAppDeploymentCloud) return `https://www.dataiku.com/product/plugins/${pluginId}/`;
            if (isPluginInstalled) return `plugins/${pluginId}/summary/`;
            return `/plugins-explore/store/${pluginId}`;
        },
        /**
         * Get the link to the plugin store depending on how DSS is deployed (Cloud and On Prem.)
         */
        getPluginStoreLink: function(cloudInfo = {}, pluginName = '') {
            if (cloudInfo.isDataikuCloud && cloudInfo.isSpaceAdmin) {
                return DataikuCloudService.getLaunchpadUrl() + '/plugins/';
            }

            return (cloudInfo.isDataikuCloud ? 'https://www.dataiku.com/product/plugins/' : StateUtils.href.pluginStore()) + pluginName;
        }
    };
    return svc;
});


app.factory("PluginConfigUtils", function() {
    return {
        setDefaultValues: function(params, customConfig) {
            if (!customConfig) {
                return;
            }
            params.forEach(function(param) {
                if (customConfig[param.name] === undefined) {
                    if (param.defaultValue) {
                        // the type is not checked, so if the default is not of the right type, strange things can happen
                        customConfig[param.name] = param.defaultValue;
                    } else if (param.type == 'BOOLEAN') {
                        customConfig[param.name] = false;
                    } else if (param.type == 'INT' || param.type == 'DOUBLE') {
                        customConfig[param.name] = 0;
                    } else if (param.type == 'MAP') {
                        customConfig[param.name] = {};
                    } else if (param.type == 'KEY_VALUE_LIST') {
                        customConfig[param.name] = [];
                    } else if (param.type == 'ARRAY') {
                        customConfig[param.name] = [];
                    } else if (param.type == 'OBJECT_LIST') {
                        customConfig[param.name] = [];
                    }
                }
            });
        },
        shouldComponentBeVisible: function (loadedPlugins, selectedComponentId, componentIdGetter = (c) => c.id) {
            return (component) => {
                const ownerPlugin = loadedPlugins.find(p => p.id === component.ownerPluginId);

                const isHiddenByPlugin = ownerPlugin && ownerPlugin.hideComponents;
                const isSelected = selectedComponentId && selectedComponentId === componentIdGetter(component);

                return !isHiddenByPlugin || isSelected;
            };
        }
    }
});

app.factory("FutureProgressUtils", function() {
    var svc = {
        getTotalProgressPercentage: function(progress) {
            var percentage = 0;
            var fractionOf = 100;
            if (progress && progress.states) {
                angular.forEach(progress.states, function(state) {
                    if (state.target > 0) {
                        fractionOf = fractionOf / (state.target + 1);
                        percentage += fractionOf * state.cur;
                    }
                });
            }
            return percentage;
        }
    }
    return svc;
})

// Store large things, and give them a unique ID. It's backed by a very small LRU cache, and its purpose is
// to overcome the limitations of $stateParams when trying to pass big data in the URL
app.factory("BigDataService", ['$cacheFactory', function($cacheFactory) {
    var cache = $cacheFactory('BigDataServiceCache', {
        number: 20
    });
    return {
        store: function(bigdata) {
            var id = generateRandomId(10);
            cache.put(id, {data: bigdata});
            return id;
        },
        fetch: function(id) {
            var val = cache.get(id);
            if (!val) {
                return undefined;
            }
            return val.data;
        }
    };
}]);


app.factory("SQLExplorationService", ['$rootScope', '$cacheFactory', '$q', '$stateParams', 'Logger', 'MonoFuture', 'DataikuAPI',
            function($rootScope, $cacheFactory, $q, $stateParams, Logger, MonoFuture, DataikuAPI) {

    var cache = $cacheFactory('SQLExplorationServiceCache', {
        number: 50
    });

    function _cachedDataFetcher(promiseFunction, loggedName) {
       let runningPromise = {};
       return function(cacheKey, args) {
           if (cache.get(cacheKey)) {
               Logger.info("Loaded " + loggedName + " from cache");
               return $q.when(cache.get(cacheKey));
           }
           if (runningPromise[cacheKey]) {
               Logger.info("Loading " + loggedName + " leveraging already initiated promise");
               return runningPromise[cacheKey].promise;
           }
           const deferred = $q.defer();
           runningPromise[cacheKey] = deferred;

           Logger.info("Loading " + loggedName + " from backend");
           promiseFunction(...args)
               .success(function(data) {
                   cache.put(cacheKey, data);
                   deferred.resolve(data);
               })
               .error((data, status, headers) => {
                   setErrorInScope.bind($rootScope)(data, status, headers)
                   deferred.reject();
               })
               .finally(function() {
                   runningPromise[cacheKey] = null;
               });
           return deferred.promise;
       }
    }
    const listFieldDataFetcher = _cachedDataFetcher(DataikuAPI.connections.listSQLFields, "fields");
    const listTablesFromProjectDataFetcher = _cachedDataFetcher(DataikuAPI.connections.listSQLTablesFromProject, "project tables");
    const listTablesDataFetcher = _cachedDataFetcher(MonoFuture().wrap(DataikuAPI.connections.listSQLTables), "tables");

    var sep = '^~\n$* \\#';

    function connectionKey(connectionName) {
        return 'CNX' + sep + connectionName + sep;
    }

    function tableKey(projectKey, connection, schema, table) {
        if (!schema) {
            schema = '';
        }
        return 'TBL' + sep + schema + sep + table + sep + connection + sep + projectKey;
    }

    function listFieldsForTable(connection, table, projectKey) {
        const cacheKey = tableKey(projectKey, connection, table.schema, table.table);
        return listFieldDataFetcher(cacheKey, [connection, [table], projectKey]);
    }

    return {
        getCacheId: function(connectionName, projectKey, catalog, schema) {
            let id = connectionKey(connectionName) + '__all__' + projectKey;
            if (catalog) {
                id += '__catalog__' + catalog
            }
            if (schema) {
                id += '__schema__' + schema
            }
            return id;
        },
        getCacheIdFromProject: function(connectionName, projectKey) {
            return connectionKey(connectionName) + '__fromProject__' + projectKey;
        },
        isTableInCache: function(id) {
            return cache.get(id);
        },
        listTables: function(connectionName, projectKey, catalog, schema) {
            Logger.info(`Listing tables connection=${connectionName}, projectKey=${projectKey}, catalog=${catalog}, schema=${schema}`);
            const cacheKey = this.getCacheId(connectionName, projectKey, catalog, schema);
            return listTablesDataFetcher(cacheKey, [connectionName, projectKey, catalog, schema]).then(function(future) {
                return future.result;
            });
        },
        listTablesFromProject: function(connectionName, projectKey) {
            const cacheKey = this.getCacheIdFromProject(connectionName, projectKey);
            return listTablesFromProjectDataFetcher(cacheKey, [connectionName, projectKey]);
        },
        clearCache: function() {
            cache.removeAll();
        },
        listFields: function(connectionName, tables) {
            var promises = [];
            for (var i = 0; i < tables.length; i++) {
                promises.push(listFieldsForTable(connectionName, tables[i], $stateParams.projectKey));
            }
            var deferred = $q.defer();
            $q.all(promises).then(function(results) {
                var out = [];
                for (var i in results) {
                    out = out.concat(results[i]);
                }
                deferred.resolve(angular.copy(out));
            });
            return deferred.promise;
        }
    };
}]);


app.factory("SmartId", function($stateParams) {
    return {
        /**
         * Creates a smartId based on a given project key and contextProjectKey. If those two match, only the id is returned, otherwise the projectKey is also added as a prefix.
         * @param {string} id the object id
         * @param {string} projectKey the object project key
         * @param {string} contextProjectKey (optional) the project used as reference for the smartId. if omitted, defaults to $stateParams.projectKey
         * @returns {string} the smartId
         */
        create: function(id, projectKey, contextProjectKey) {
            contextProjectKey = contextProjectKey || $stateParams.projectKey;
            if (projectKey == contextProjectKey) {
                return id;
            } else {
                return projectKey + "." + id;
            }
        },

        /**
         * Resolves a smart or full id relative to a given project key.
         * @param {string} smartOrFullId the smartId to resolve. Can also be a fullId (eg no need to make sure the projectKey is removed if it's a local object)
         * @param {string} contextProjectKey (optional) the reference projectKey used to resolve. If omitted, defaults to $stateParams.projectKey
         * @returns {{projectKey: string, id: string}} the resolved loc
         */
        resolve: function(smartOrFullId, contextProjectKey) {
            if (contextProjectKey === undefined) {
                contextProjectKey = $stateParams.projectKey
            }
            if (smartOrFullId && smartOrFullId.indexOf(".") > 0) {
                var chunks = smartOrFullId.split(".");
                return {projectKey: chunks[0], id: chunks[1]}
            } else {
                return {projectKey: contextProjectKey, id: smartOrFullId};
            }
        },

        fromRef: function(smartObjectRef, contextProject) {
            if (contextProject === undefined) {
                contextProject = $stateParams.projectKey
            }
            if (smartObjectRef.objectType == 'PROJECT') {
                return smartObjectRef.objectId;
            }
            if (!smartObjectRef.projectKey || !smartObjectRef.projectKey.length || smartObjectRef.projectKey == contextProject) {
                return smartObjectRef.objectId;
            } else {
                return smartObjectRef.projectKey + "." + smartObjectRef.objectId;
            }
        },

        fromTor: function(tor, contextProject) {
            if (contextProject === undefined) {
                contextProject = $stateParams.projectKey
            }
            if (tor.taggableType == 'PROJECT') {
                return tor.id;
            }
            if (!tor.projectKey || !tor.projectKey.length || tor.projectKey == contextProject) {
                return tor.id;
            } else {
                return tor.projectKey + "." + tor.id;
            }
        }
    }
});


app.service("FeatureNameUtils", function($filter, Fn) {

    /*
     Returns an object with :
     .elements, an array of strings representing
     consecutive elements of processed the feature name.
     .isCode, an array of booleans with same length indicating whether the corresponding
     element should be treated as "code"
     .value, an optional value for the feature. Defaults to null.
     .operator, the operator describing the operation when there is a value. Defaults to null.
     .no_operator, the inverse operator. Defaults to null.
     */
    var getAsElements = function(input, asHtml) {
        if (asHtml === undefined) {
            asHtml = false;
        }
        if (input == null) {
            input = "";
        }
        // Formatting
        var esc = asHtml ? $filter('escapeHtml') : Fn.SELF;
        var code = [];
        var els = [];
        var value = null, rawValue = null;
        var operator = null;
        var no_operator = null;
        var type = null;

        var addCode = function(c) {
            code.push(true);
            els.push(esc(c));
        };

        var addText = function(c) {
            code.push(false);
            els.push(esc(c));
        };

        let match;
        const elts = input.split(":");
        if ( (match = input.match(/^dummy:([^:]+):(.*)/)) ) {
            addCode(match[1]);
            switch (match[2].trim()) {
                case '':
                    value = "empty";
                    operator = "is";
                    no_operator = "is not";
                    break;
                case '__Others__':
                    value = "other";
                    operator = "is";
                    no_operator = "is not";
                    break;
                default:
                    value = esc(match[2]);
                    operator = "is";
                    no_operator = "is not";
            }
        } else if ( (match = input.match(/^(?:thsvd|hashvect):(.+):(\d+)$/)) ) {
            addCode(match[1]);
            addText('[text #' + match[2] + ']');
        } else if ( (match = input.match(/^unfold:([^:]+):(.*)$/)) ) {
            addCode(match[1]);
            addText('[element #' + match[2] + ']');
        } else if (input.startsWith("impact:")) {
            /*reg: impact:ft:all_target_values*/
            /*multi: impact:ft:target_value:<target_value>*/
            if (elts.length == 4) {
                addCode(elts[1]);
                addText('[impact #' + elts[3] + ']');
            } else {
                addCode(elts[1]);
                addText("[impact on target]");
            }
        } else if (input.startsWith("glmm:")) {
            /*reg: glmm:ft:all_target_values*/
            /*multi: glmm:ft:target_value:<target_value>*/
            if (elts.length == 4) {
                addCode(elts[1]);
                addText('[glmm #' + elts[3] + ']');
            } else {
                addCode(elts[1]);
                addText("[glmm on target]");
            }
        } else if (input.startsWith("frequency:")) {
            /*frequency: frequency:ft:frequency*/
            /*count: frequency:ft:count*/
            addCode(elts[1]);
            addText("[" + elts[2] + " encoded]");
        } else if (input.startsWith("ordinal:")) {
            /*lexicographic: ordinal:ft:lexicographic*/
            /*count: ordinal:ft:count*/
            addCode(elts[1]);
            addText("[ordinal encoded (" + elts[2] + ")]");
        } else if (input.startsWith("datetime_cyclical:")) {
            /* datetime_cyclical:ft:<period>:<cos|sin> */
            addCode(elts[1]);
            addText(`[${elts[2]} cycle (${elts[3]})]`);
        } else if ( (match = input.match(/^poly_int:(.*)$/)) ) {
            addCode(match[1]);
            addText("(computed)");
        } else if ( (match = input.match(/^pw_linear:(.*)$/)) ) {
            addCode(match[1]);
            addText("(computed)");
        } else if ( (match = input.match(/countvec:(.+):(.+)$/)) ) {
            addCode(match[1]);
            operator = 'contains';
            no_operator = "does not contain";
            value = match[2];
            type = "countvec";
        } else if ( (match = input.match(/tfidfvec:(.+):(.+):(.+)$/)) ) {
            addCode(match[1]);
            operator = 'contains';
            no_operator = "does not contain";
            value = match[3] + "(idf=" + match[2] + ")";
            rawValue = match[3];
            type = "tfidfvec";
        } else if(input.startsWith("hashing:")) {
            addCode(elts[1]);
            value = elts[2];
            operator = "hashes to";
            no_operator = "does not hash to";
            type = "hashing";
        } else if (input.startsWith("interaction")) {
            if (elts.length == 3) {
                addCode(elts[1]);
                addText("x");
                addCode(elts[2]);
            } else if (elts.length == 4) {
                addCode(elts[1]);
                addText("x");
                addCode(elts[2] + " = " + elts[3]);
            } else {
                addCode(elts[1] + " = " + elts[3]);
                addText("and");
                addCode(elts[2] + " = " + elts[4]);
            }
        } else {
            addCode(input);
        }

        return {
            elements: els,
            isCode: code,
            value: value,
            operator: operator,
            no_operator: no_operator,
            type : type,
            rawValue : rawValue
        };
    }

    var getAsHtmlString = function(feature) {
        var els = getAsElements(feature, true);
        var htmlArray = [];
        for (var i = 0; i < els.elements.length; i++) {
            if (els.isCode[i]) {
                htmlArray.push("<code>" + els.elements[i] + "</code>")
            } else {
                htmlArray.push(els.elements[i]);
            }
        }
        if (els.value != null) {
            htmlArray.push(els.operator, "<code>" + els.value + "</code>");
        }
        return htmlArray.join(" ");
    };

    var getAsTextElements = function(feature) {
        var els = getAsElements(feature);
        return {
            feature: els.elements.join(" "),
            operator: els.operator,
            no_operator: els.no_operator,
            value: els.value,
            type : els.type,
            rawValue : els.rawValue
        };
    };

    var getAsText = function(feature, negate) {
        if (negate === undefined) {
            negate = false;
        }
        var els = getAsElements(feature);
        if (els.value == null) {
            return els.elements.join(" ");
        } else {
            return els.elements.concat([negate ? els.no_operator : els.operator, els.value]).join(" ");
        }
    }

    return {
        getAsElements: getAsElements,
        getAsHtmlString: getAsHtmlString,
        getAsTextElements: getAsTextElements,
        getAsText: getAsText
    };

})

app.filter("getNameValueFromMLFeature", function(FeatureNameUtils) {
    return function(feature) {
        var els = FeatureNameUtils.getAsTextElements(feature);
        return {
            name: els.feature,
            value: els.value
        };
    }
});


app.factory("InfoMessagesUtils", function() {
    var svc = {
        /* Returns the first of the info messages with a given line, or null if there is none */
        getMessageAtLine: function(im, line) {
            if (!im || !im.messages) return null;
            for (var i = 0; i < im.messages.length; i++) {
                if (im.messages[i].line == line) return im.messages[i];
            }
            return null;
        },
        /* Filter the messages of all categories by line */
        filterForLine : function(im, line) {
            if (!im || !im.messages) return null;
            var fim = {};
            fim.messages = im.messages.filter(function(m) {return m.line == line;});
            fim.anyMessage = fim.messages.length > 0;
            fim.error = fim.messages.filter(function(m) {return m.severity == 'ERROR'}).length > 0;
            fim.warning = fim.messages.filter(function(m) {return m.severity == 'WARNING'}).length > 0;
            fim.maxSeverity = fim.error ? 'ERROR' : (fim.warning ? 'WARNING' : (fim.anyMessage ? 'INFO' : null));
            return fim;
        },
        getMessageAtColumn : function(im, column) {
            if (!im || !im.messages) return null;
            for (var i = 0; i < im.messages.length; i++) {
                if (im.messages[i].column == column) return im.messages[i];
            }
            return null;
        }
    }
    return svc;
});


app.factory("MessengerUtils", function($sanitize) {
    Messenger.options = {
        extraClasses: 'messenger-fixed messenger-on-bottom messenger-on-right',
        theme: 'dss'
    };

    const svc = {
        post: function(options) {
            let msg = null;
            options.actions = options.actions || {};
            if (options.showCloseButton) {
                options.actions.close = {
                    label: "Close",
                    action: function() {
                        msg.hide();
                    }
                };
                delete options.showCloseButton;
            }
            if (options.icon) {
                options.message = '<div style="width: 100%;"><div class="messenger-icon">' + $sanitize(options.icon) + '</div>' + $sanitize(options.message) + '</div>'
                delete options.icon;
            } else {
                options.message = '<div style="width: 100%;">' + $sanitize(options.message) + '</div>'
            }
            msg = Messenger().post(options);
        }
    }
    return svc;
});


// Front-end equivalent of StringNormalizationMode.java
app.factory("StringNormalizer", function() {

    var inCombiningDiatricalMarks = /[\u0300-\u036F]/g;
    var punct = /!"#\$%&'\(\)\*\+,-\.\/:;<=>\?@\[\]\^_`\{\|\}~/g;

    var svc = {
        get: function(stringNormalizationMode) {
            switch(stringNormalizationMode) {
                case 'EXACT':
                    return function(str) {
                        return str;
                    };

                case 'LOWERCASE':
                    return function(str) {
                        return str.toLowerCase();
                    };

                case 'NORMALIZED':
                default:
                    return function(str) {
                        return svc.normalize(str);
                    };
            }
        },

        normalize: function(str) {
            return str.normalize('NFD').replace(inCombiningDiatricalMarks, '');
        },

        removePunct: function(str) {
            return str.replace(punct, '');
        }
    };

    return svc;
});


app.service('HiveService', function($rootScope, Dialogs, ActivityIndicator, DataikuAPI, ToolBridgeService, CreateModalFromTemplate, $q) {
    this.convertToImpala = function(selectedRecipes) {
        var deferred = $q.defer();
        //TODO @flow need a dedicated modal or rather a generic confirm modal that can have errors in scope
        Dialogs.confirm($rootScope, "Convert recipes to Impala", `Are you sure you want to convert ${selectedRecipes.length} Hive recipes to Impala?`).then(function() {
            DataikuAPI.flow.recipes.massActions.convertToImpala(selectedRecipes, true)
                .success(function() {
                    deferred.resolve("converted");
                }).error(function(a,b,c) {
                    deferred.reject("conversion failed");
                    setErrorInScope.bind($rootScope)(a,b,c);
                });
        }, function() {deferred.reject("user cancelled");});
        return deferred.promise;
    };

    this.resynchronizeMetastore = function(selectedDatasets) {
        Dialogs.confirmPositive($rootScope,
            'Hive metastore resynchronization',
            'Are you sure you want to resynchronize datasets to the Hive metastore?')
        .then(function() {
            ActivityIndicator.waiting('Synchronizing Hive metastore...');
            DataikuAPI.datasets.synchronizeHiveMetastore(selectedDatasets).success(function(data) {
                if (data.anyMessage && (data.warning || data.error)) {
                    ActivityIndicator.hide();
                    Dialogs.infoMessagesDisplayOnly($rootScope, "Metastore synchronization", data);
                } else {
                    // nothing to show
                    ActivityIndicator.success('Hive metastore successfully synchronized');
                }
            }).error(function(data, status, headers) {
                ActivityIndicator.hide();
                setErrorInScope.call($rootScope, data, status, headers);
            });
        });
    };

    this.startChangeHiveEngine = function(selectedRecipes) {
        return CreateModalFromTemplate('/templates/recipes/fragments/hive-engine-modal.html', $rootScope, null, function(modalScope) {
            modalScope.options = {executionEngine: 'HIVESERVER2'};

            DataikuAPI.flow.recipes.massActions.startSetHiveEngine(selectedRecipes).success(function(data) {
                modalScope.messages = data;
                ToolBridgeService.emitRefreshView('HiveModeView');
            }).error(function(...args) {
                modalScope.fatalError = true;
                setErrorInScope.apply(modalScope, args);
            });

            modalScope.ok = function() {
                DataikuAPI.flow.recipes.massActions.setHiveEngine(selectedRecipes, modalScope.options.executionEngine)
                    .success(function() {
                        $rootScope.$emit('recipesHiveEngineUpdated');
                        modalScope.resolveModal();
                    })
                    .error(setErrorInScope.bind(modalScope));
            };
        });
    };
});


app.service('ImpalaService', function($rootScope, Dialogs, CreateModalFromTemplate, ToolBridgeService, DataikuAPI, $q) {

    this.convertToHive = function(selectedRecipes) {
        var deferred = $q.defer();
        //TODO @flow need a dedicated modal or rather a generic confirm modal that can have errors in scope
        Dialogs.confirm($rootScope, "Convert recipes to Hive", `Are you sure you want to convert ${selectedRecipes.length} Impala recipes to Hive?`).then(function() {
            DataikuAPI.flow.recipes.massActions.convertToHive(selectedRecipes, true)
                .success(function() {
                    deferred.resolve("converted");
                }).error(function(a,b,c) {
                    deferred.reject("conversion failed");
                    setErrorInScope.bind($rootScope)(a,b,c);
                });
        }, function() {deferred.reject("user cancelled");});
        return deferred.promise;
    };

    this.startChangeWriteMode = function(selectedRecipes) {
        return CreateModalFromTemplate('/templates/recipes/fragments/impala-write-flag-modal.html', $rootScope, null, function(modalScope) {
            modalScope.options = {runInStreamMode: true};

            DataikuAPI.flow.recipes.massActions.startSetImpalaWriteMode(selectedRecipes).success(function(data) {
                modalScope.messages = data;
                ToolBridgeService.emitRefreshView('ImpalaWriteModeView');
            }).error(function(...args) {
                modalScope.fatalError = true;
                setErrorInScope.apply(modalScope, args);
            });

            modalScope.ok = function() {
                DataikuAPI.flow.recipes.massActions.setImpalaWriteMode(selectedRecipes, modalScope.options.runInStreamMode)
                    .success(function() {
                        $rootScope.$emit('recipesImpalaWriteModeUpdated');
                        modalScope.resolveModal();
                    })
                    .error(setErrorInScope.bind(modalScope));
            };
        });
    };
});


app.service('SparkService', function($rootScope, CreateModalFromTemplate, DataikuAPI) {
    this.startChangeSparkConfig = function(selectedItems) {

        return CreateModalFromTemplate('/templates/recipes/fragments/spark-config-modal.html', $rootScope, null, function(modalScope) {
            modalScope.selectedRecipes = selectedItems.filter(it => it.type == 'RECIPE');
            modalScope.options = {};

            DataikuAPI.flow.recipes.massActions.startSetSparkConfig(modalScope.selectedRecipes).success(function(data) {
                modalScope.messages = data;
            }).error(function(...args) {
                modalScope.fatalError = true;
                setErrorInScope.apply(modalScope, args);
            });

            modalScope.ok = function() {
                DataikuAPI.flow.recipes.massActions.setSparkConfig(modalScope.selectedRecipes, modalScope.options.sparkConfig)
                    .success(function() {
                        $rootScope.$emit('recipesSparkConfigUpdated');
                        modalScope.resolveModal();
                    })
                    .error(setErrorInScope.bind(modalScope));
            };
        });
    };
});


app.service('PipelineService', function($rootScope, CreateModalFromTemplate, DataikuAPI) {
    this.startChangePipelineability = function(selectedItems, pipelineType) {
        return CreateModalFromTemplate('/templates/recipes/fragments/pipelineability-modal.html', $rootScope, null, function(modalScope) {
            modalScope.selectedRecipes = selectedItems.filter(it => it.type === 'RECIPE');

            modalScope.pipelineTypeText = (pipelineType === 'SPARK' ? 'Spark' : 'SQL');

            modalScope.options = {
                allowStart: true,
                allowMerge: true
            };

            DataikuAPI.flow.recipes.massActions.startSetPipelineability(modalScope.selectedRecipes, pipelineType).success(function(data) {
                modalScope.messages = data;
            }).error(function(...args) {
                modalScope.fatalError = true;
                setErrorInScope.apply(modalScope, args);
            });

            modalScope.ok = function() {
                DataikuAPI.flow.recipes.massActions.setPipelineability(modalScope.selectedRecipes, pipelineType, modalScope.options.allowStart, modalScope.options.allowMerge)
                    .success(function() {
                        modalScope.resolveModal();
                    })
                    .error(setErrorInScope.bind(modalScope));
            };
        });
    };
});


app.service('ColorPalettesService', function() {

    const svc = this;

    const DEFAULT_COLORS = [
        "#1ac2ab",
        "#0f6d82",
        "#FFD83D",
        "#de1ea5",
        "#9dd82b",
        "#28aadd",
        "#00a55a",
        "#d66b9b",
        "#77bec2",
        "#94be8e",
        "#123883",
        "#a088bd",
        "#c28e1a"
    ];

    svc.fixedColorsPalette = function(name, colors=DEFAULT_COLORS) {
        const colorMap = {};
        return function(key) {
            key = key + ''; //force conversion
            if (colorMap[key]) {
                return colorMap[key];
            }
            colorMap[key] = colors[Object.keys(colorMap).length % colors.length];
        };
    };

});


/*
 * TODO: finally this service is a bit of a duplicate of what CodeBasedEditorUtils was supposed to be. Would be good to merge both at some point...
 */
app.service('CodeMirrorSettingService', function($rootScope) {

    const INDENT_MORE_SHORTCUT = "Tab";
    const INDENT_LESS_SHORTCUT = "Shift-Tab";
    const FIND_SHORTCUT = "Ctrl-F";
    const REPLACE_SHORTCUT = "Ctrl-Alt-F";
    const JUMP_TO_LINE_SHORTCUT = "Ctrl-L";
    const TOGGLE_COMMENT_SHORTCUT_QWERTY = "Cmd-/";
    const TOGGLE_COMMENT_SHORTCUT_AZERTY = "Shift-Cmd-/";
    const TOGGLE_COMMENT_SHORTCUT_RSTUDIO = "Shift-Ctrl-C";
    const AUTOCOMPLETE_SHORTCUT = "Ctrl-Space";
    const FULL_SCREEN_SHORTCUT = "F11";

    this.getShortcuts = function() {
        return {
            "INDENT_MORE_SHORTCUT": INDENT_MORE_SHORTCUT,
            "INDENT_LESS_SHORTCUT": INDENT_LESS_SHORTCUT,
            "FIND_SHORTCUT": FIND_SHORTCUT,
            "REPLACE_SHORTCUT": REPLACE_SHORTCUT,
            "JUMP_TO_LINE_SHORTCUT": JUMP_TO_LINE_SHORTCUT,
            "TOGGLE_COMMENT_SHORTCUT": TOGGLE_COMMENT_SHORTCUT_QWERTY,
            "AUTOCOMPLETE_SHORTCUT": AUTOCOMPLETE_SHORTCUT,
            "FULL_SCREEN_SHORTCUT": FULL_SCREEN_SHORTCUT}
    }

    this.get = function(mimeType, options) {
        var extraKeys = {};

        if (!$rootScope.appConfig.userSettings.codeEditor || !$rootScope.appConfig.userSettings.codeEditor.keyMap || $rootScope.appConfig.userSettings.codeEditor.keyMap == "default") {
            extraKeys[INDENT_MORE_SHORTCUT] = "indentMore";
            extraKeys[INDENT_LESS_SHORTCUT] = "indentLess";
            extraKeys[FIND_SHORTCUT] = "find";
            extraKeys[REPLACE_SHORTCUT] = "replace";
            extraKeys[JUMP_TO_LINE_SHORTCUT] = "jumpToLine";
            extraKeys[TOGGLE_COMMENT_SHORTCUT_QWERTY] = "toggleComment";
            extraKeys[TOGGLE_COMMENT_SHORTCUT_AZERTY] = "toggleComment";
            extraKeys[TOGGLE_COMMENT_SHORTCUT_RSTUDIO] = "toggleComment";
            extraKeys[AUTOCOMPLETE_SHORTCUT] = this.showHint(mimeType, options && options.words ? options.words : []);
        }
        if (!options || !options.noFullScreen) {
            extraKeys[FULL_SCREEN_SHORTCUT] = function(cm) {
                if (cm.getOption("fullScreen")) {
                    cm.setOption("fullScreen", false);
                } else {
                    cm.setOption("fullScreen", !cm.getOption("fullScreen"));
                }
            };
        }


        var settings =  {
            mode: mimeType,
            theme: $rootScope.appConfig.userSettings.codeEditor && $rootScope.appConfig.userSettings.codeEditor.theme ? $rootScope.appConfig.userSettings.codeEditor.theme : 'default',

            //left column
            lineNumbers : true,
            foldGutter: true,
            gutters: ["CodeMirror-linenumbers", "CodeMirror-foldgutter"],

            //indentation
            indentUnit: $rootScope.appConfig.userSettings.codeEditor && $rootScope.appConfig.userSettings.codeEditor.indentUnit ? $rootScope.appConfig.userSettings.codeEditor.indentUnit : 4,
            tabSize: $rootScope.appConfig.userSettings.codeEditor && $rootScope.appConfig.userSettings.codeEditor.tabSize ? $rootScope.appConfig.userSettings.codeEditor.tabSize : 4,
            indentWithTabs: $rootScope.appConfig.userSettings.codeEditor ? $rootScope.appConfig.userSettings.codeEditor.indentWithTabs : false,

            //edition
            autoCloseBrackets: $rootScope.appConfig.userSettings.codeEditor ? $rootScope.appConfig.userSettings.codeEditor.autoCloseBrackets : true,
            autoCloseTags: $rootScope.appConfig.userSettings.codeEditor ? $rootScope.appConfig.userSettings.codeEditor.autoCloseTags : true,

            // readonly
            readOnly: options && options.readOnly,

            //code reading
            matchBrackets: true,
            matchTags: true,
            highlightSelectionMatches: true,
            styleSelectedText: true,
            styleActiveLine: true,

            keyMap: $rootScope.appConfig.userSettings.codeEditor && $rootScope.appConfig.userSettings.codeEditor.keyMap ? $rootScope.appConfig.userSettings.codeEditor.keyMap : 'default',
            extraKeys: extraKeys,
            onLoad: function(cm) {
                if ($rootScope.appConfig.userSettings.codeEditor && $rootScope.appConfig.userSettings.codeEditor.fontSize) {
                    $($(cm.getTextArea()).siblings('.CodeMirror')[0]).css('font-size', $rootScope.appConfig.userSettings.codeEditor.fontSize + 'px');
                }
                if(options && options.readOnly){
                    $($(cm.getTextArea()).siblings('.CodeMirror')[0]).css('opacity', '0.3');
                }
                if (options && options.onLoad && typeof(options.onLoad) == "function") {
                    options.onLoad(cm);
                }
            }
        };

        return settings;
    }

    this.showHint = function(mode, words) {
        return function(cm) {
            const modes = {
                'text/x-python': CodeMirror.hintWords.python,
                'text/css': CodeMirror.hintWords.css
            };

            CodeMirror.showHint(cm, function(editor) {
                const anyWordHint = CodeMirror.hint.anyword(cm);
                const recipeWords = words || [];
                const codeKeywords = CodeMirror.hintWords[mode] || modes[mode] || [];

                let combinedWords = [recipeWords, [' '], codeKeywords, [' '], anyWordHint && anyWordHint.list ? anyWordHint.list : []]
                    .reduce((a, b) => a.concat(b.filter(_ => _ === ' ' || !a.includes(_)))); // deduplicates

                /*
                    Filter functionality based off of https://github.com/amarnus/ng-codemirror-dictionary-hint/blob/master/lib/ng-codemirror-dictionary-hint.js
                */
                var cur = editor.getCursor();
                var curLine = editor.getLine(cur.line);
                var start = cur.ch;
                var end = start;
                while (end < curLine.length && /[\w$]/.test(curLine.charAt(end))) ++end;
                while (start && /[\w$]/.test(curLine.charAt(start - 1))) --start;
                var curWord = start !== end && curLine.slice(start, end);
                return {
                    list: (!curWord ? combinedWords : combinedWords.filter(_ => _.startsWith(curWord) && _ !== curWord)),
                    from: CodeMirror.Pos(cur.line, start),
                    to: CodeMirror.Pos(cur.line, end)
                }
            }, { completeSingle: false });
        };
    }
});

app.service('CodeEnvService', function(DataikuAPI, $stateParams, FutureProgressModal, Dialogs, Deprecation, CreateModalFromTemplate, $q, FutureWatcher, FeatureFlagsService) {
    const allInterpreters = [
        ["PYTHON36", "Python 3.6"],
        ["PYTHON37", "Python 3.7"],
        ["PYTHON38", "Python 3.8"],
        ["PYTHON39", "Python 3.9"],
        ["PYTHON310", "Python 3.10"],
        ["PYTHON311", "Python 3.11"],
        ["PYTHON312", "Python 3.12"],
        ["PYTHON313", "Python 3.13"],
        ["PYTHON314", "Python 3.14"],
        ["CUSTOM", "Custom (lookup in PATH)"]
    ];
    if (FeatureFlagsService.featureFlagEnabled('allowPython2CodeEnv')) {
        allInterpreters.unshift(["PYTHON27", "Python 2.7"])
    }

    const experimentalInterpreters = [
        "PYTHON314",
    ]

    const condaInterpreters = [
        ["PYTHON36", "Python 3.6"],
        ["PYTHON37", "Python 3.7"],
        ["PYTHON38", "Python 3.8"],
        ["PYTHON39", "Python 3.9"],
        ["PYTHON310", "Python 3.10"],
        ["PYTHON311", "Python 3.11"],
        ["PYTHON312", "Python 3.12"],
        ["PYTHON313", "Python 3.13"],
        ["PYTHON314", "Python 3.14"],
    ];
    if (FeatureFlagsService.featureFlagEnabled('allowPython2CodeEnv')) {
        condaInterpreters.unshift(["PYTHON27", "Python 2.7"])
    }

    const condaInterpretersWithDeprecationNotice = dkuDeepCopy(condaInterpreters);
    condaInterpretersWithDeprecationNotice.filter(interpreter => Deprecation.isPythonDeprecated(interpreter[0])).forEach(interpreter => interpreter[1] += " - Deprecated");

    let availableInterpreters;
    let enrichedInterpreters;

    function getAvailableInterpretersPromise() {
        const deferred = $q.defer();
        if (availableInterpreters) {
            deferred.resolve(availableInterpreters);
        } else {
            DataikuAPI.codeenvs.listAvailablePythonInterpreters().success(function (data) {
                availableInterpreters = data;
                // CUSTOM should always be available
                availableInterpreters.push("CUSTOM");
                deferred.resolve(availableInterpreters)
            });
        }
        return deferred.promise;
    }

    function enrichInterpreterDescription(availableInterpretersList) {
        const enrichedInterpreters = [];
        // make sure the list of available interpreters has been retrieved from backend
        for (const interpreter of allInterpreters) {
            const interpreterId = interpreter[0];
            const interpreterDesc = interpreter[1];
            if (!availableInterpretersList.includes(interpreterId)) {
                enrichedInterpreters.push([interpreterId, interpreterDesc + " - Not available"]);
            } else if (Deprecation.isPythonDeprecated(interpreterId)) {
                enrichedInterpreters.push([interpreterId, interpreterDesc + " - Deprecated"]);
            } else if (experimentalInterpreters.includes(interpreterId)) {
                enrichedInterpreters.push([interpreterId, interpreterDesc + " - Experimental"]);
            } else {
                enrichedInterpreters.push([interpreterId, interpreterDesc]);
            }
        }
        return enrichedInterpreters;
    }

    const svc = {
        pythonInterpreters: allInterpreters, // these interpreters can be used as default, but will not contain enrichments
        getPythonInterpreters: function() {
            if (enrichedInterpreters) {
                return $q.when(enrichedInterpreters);
            }
            return getAvailableInterpretersPromise()
                .then((interpreters) => {
                    enrichedInterpreters = enrichInterpreterDescription(interpreters);
                    return enrichedInterpreters;
                });
        },
        pythonCondaInterpreters: condaInterpretersWithDeprecationNotice,
        getPackagesThatAreBothRequiredAndInstalled : function(spec, actual) {
            if (!spec || !actual) {
                return new Set();
            }
            return svc.intersection(svc.getPackageNames(spec), svc.getPackageNames(actual));
        },
        intersection : function(setA, setB) {
            let intersection = new Set();
            for (let elt of setB) {
                if (setA.has(elt)) {
                    intersection.add(elt)
                }
            }
            return intersection;
        },
        difference : function(setA, setB) {
            let difference = new Set();
            for (let elt of setA) {
                if (!setB.has(elt)) {
                    difference.add(elt)
                }
            }
            return difference;
        },
        getPackagesThatAreRequired : function(codeEnv, packageSystem) {
            let spec = null;
            let mandatory = null;
            if (packageSystem == 'pip') {
                spec = codeEnv.specPackageList;
                mandatory = codeEnv.mandatoryPackageList;
            } else if (packageSystem == 'conda') {
                spec = codeEnv.specCondaEnvironment;
                mandatory = codeEnv.mandatoryCondaEnvironment;
            }
            let packages = !spec ? new Set() : svc.getPackageNames(spec);
            if (mandatory) {
                svc.getPackageNames(mandatory).forEach(p => packages.add(p));
            }
            return packages;
        },
        getPackageNames : function(listStr) {
            /** Gets the set of package names mentioned in a code env listing */
            let ret = new Set();
            for (let line of listStr.split("\n")) {
                // Ignore packages that are not clean package names
                if (line.indexOf("-e") >= 0 || line.indexOf("git+") >= 0 || line.indexOf("ssh") >= 0) continue;

                const chunks = line.split(/[\s>=<,[\]]+/);
                const packageName = chunks[0].trim().replaceAll('"',"");
                if (packageName.length > 0) {
                    ret.add(packageName.toLowerCase());
                }
            }
            return ret;
        },
        listLogs : function($scope){
            return DataikuAPI.admin.codeenvs.design.listLogs($scope.envLang, $stateParams.envName).then(function(response) {
                $scope.logs = response.data;
            }).catch(setErrorInScope.bind($scope));
        },
        makeCurrentDesc : function(desc) {
            return {
                yarnPythonBin: angular.copy(desc.yarnPythonBin),
                yarnRBin: angular.copy(desc.yarnRBin),
                owner: angular.copy(desc.owner),
                installJupyterSupport : desc.installJupyterSupport,
                installCorePackages : desc.installCorePackages,
                corePackagesSet : desc.corePackagesSet,
                envSettings : angular.copy(desc.envSettings),
                dockerImageResources: desc.dockerImageResources,
                updateResourcesApiNode: desc.updateResourcesApiNode,
                dockerfileAtStart: desc.dockerfileAtStart,
                dockerfileBeforePackages: desc.dockerfileBeforePackages,
                dockerfileAfterCondaPackages: desc.dockerfileAfterCondaPackages,
                dockerfileAfterPackages: desc.dockerfileAfterPackages,
                dockerfileAtEnd: desc.dockerfileAtEnd,
                containerCacheBustingLocation: desc.containerCacheBustingLocation,
                predefinedContainerHooks: angular.copy(desc.predefinedContainerHooks)
            };
        },
        makeCurrentSpec : function(spec) {
            return {
                specPackageList: angular.copy(spec.specPackageList),
                desc: svc.makeCurrentDesc(spec.desc),
                specCondaEnvironment: angular.copy(spec.specCondaEnvironment),
                permissions: angular.copy(spec.permissions),
                usableByAll: spec.usableByAll,
                allContainerConfs: angular.copy(spec.allContainerConfs),
                containerConfs: angular.copy(spec.containerConfs),
                allSparkKubernetesConfs: angular.copy(spec.allSparkKubernetesConfs),
                sparkKubernetesConfs: angular.copy(spec.sparkKubernetesConfs),
                rebuildDependentCodeStudioTemplates: angular.copy(spec.rebuildDependentCodeStudioTemplates),
                resourcesInitScript: angular.copy(spec.resourcesInitScript)
                };
        },
        refreshEnv : function($scope) {
            const getEnvPromise = DataikuAPI.admin.codeenvs.design.get($scope.envLang, $stateParams.envName).then(function(response) {
                const codeEnv = response.data;
                $scope.codeEnv = codeEnv;
                $scope.codeEnv.pythonInterpreterFriendlyName = svc.pythonInterpreterFriendlyName($scope.codeEnv.desc.pythonInterpreter);
                $scope.previousSpec = svc.makeCurrentSpec(codeEnv);
                $scope.previousPackagesSetForRemovalWarning['pip'] = svc.getPackagesThatAreBothRequiredAndInstalled(codeEnv.specPackageList, codeEnv.actualPackageList);
                $scope.previousPackagesSetForRemovalWarning['conda'] = svc.getPackagesThatAreBothRequiredAndInstalled(codeEnv.specCondaEnvironment, codeEnv.actualCondaEnvironment);
            }).catch(setErrorInScope.bind($scope));
            const listLogsPromise = svc.listLogs($scope);
            return $q.all([getEnvPromise, listLogsPromise]);
        },
        updateEnv : function($scope, envName, upgradeAllPackages, updateResources, forceRebuildEnv, envLang, callback){
            var updateSettings = {
                upgradeAllPackages: upgradeAllPackages,
                updateResources: $scope.uiState.updateResources,
                forceRebuildEnv: $scope.uiState.forceRebuildEnv
            }

            var buildTemplates = function(templatesToRebuild) {
                DataikuAPI.codeStudioTemplates.massBuild(templatesToRebuild)
                    .then(response => FutureProgressModal.show($scope, response.data, "Build Code Studio templates", undefined, 'static', 'false', true)
                        .then(result => {
                            if (result) {
                                Dialogs.infoMessagesDisplayOnly($scope, "Build Code Studio templates result", result.messages, result.futureLog);
                            }
                        })
                    ).catch(function(a,b,c){
                        setErrorInScope.bind($scope)(a,b,c);
                    });
            }

            var doUpdateEnv = function(templatesToRebuild) {
                DataikuAPI.admin.codeenvs.design.update($scope.envLang, $stateParams.envName, updateSettings).success(data => {
                    FutureProgressModal.show($scope, data, "Env update", undefined, 'static', false)
                        .then(result => {
                            if (result) {
                                function buildTemplatesIfNeeded() {
                                    if (templatesToRebuild) {
                                        buildTemplates(templatesToRebuild);
                                    }
                                };
                                if(result.messages.maxSeverity == 'ERROR') { // 'ERROR' also in case of abort
                                    Dialogs.infoMessagesDisplayOnly($scope, "Update Code Env result", result.messages, result.futureLog);
                                    // no template building if error
                                } else if(result.messages.maxSeverity == 'WARNING') {
                                    // warning is not serious enough to imply no template building
                                    Dialogs.infoMessagesDisplayOnly($scope, "Update Code Env result", result.messages, result.futureLog).then(buildTemplatesIfNeeded);
                                } else {
                                    buildTemplatesIfNeeded();
                                }
                            }
                            svc.refreshEnv($scope);
                            $scope.$broadcast('refreshCodeEnvResources', {envName: $scope.codeEnv.envName});
                            if (callback) {
                                callback();
                            }
                        })
                }).error(setErrorInScope.bind($scope));
            }

            if (["ASK", "ALL"].includes($scope.codeEnv.rebuildDependentCodeStudioTemplates)){
                let getOnlyGlobalUsages = true; // We only need usages in CodeStudio templates
                DataikuAPI.admin.codeenvs.design.listUsages($scope.codeEnv.envLang, $scope.codeEnv.envName, getOnlyGlobalUsages).success(usagesList => {
                    const templatesToRebuild = usagesList
                        .filter(u => u.envUsage === 'CODE_STUDIO_TEMPLATE' && u.projectKey === '__DKU_ANY_PROJECT__' && u.objectId !== 'OBJECT_NOT_ACCESSIBLE') // will be empty if you don't have the rights
                        .map(u => u.objectId);
                    if ($scope.codeEnv.rebuildDependentCodeStudioTemplates === 'ASK' && templatesToRebuild.length > 0){
                        CreateModalFromTemplate("/templates/admin/code-envs/design/code-studio-rebuild-modal.html", $scope, null, newScope => {
                            newScope.templatesToRebuild = templatesToRebuild;
                            newScope.rebuildAll = () => {
                                newScope.dismiss();
                                doUpdateEnv(templatesToRebuild);
                            }
                            newScope.rebuildNone = () => {
                                newScope.dismiss();
                                doUpdateEnv();
                            }
                        });
                    } else {
                        doUpdateEnv(templatesToRebuild);
                    }
                }).error(() => {
                    setErrorInScope.bind($scope);
                    doUpdateEnv();
                });
            } else {
                doUpdateEnv();
            }
        },
        pythonInterpreterFriendlyName: function(pythonInterpreter) {
            let interpreter = allInterpreters.filter(interpreter => interpreter[0] == pythonInterpreter);
            if (interpreter.length) {
                return interpreter[0][1];
            } else {
                return pythonInterpreter;
            }
        }
    }
    return svc;
});

app.service('TimingService', function($rootScope) {
    return {
        wrapInTimePrinter: function(prefix, fn) {
            return function() {
                const before = performance.now();
                const retval = fn.apply(this, arguments);
                const after = performance.now();
                // eslint-disable-next-line no-console
                console.info("Timing: " + prefix + ": " + (after - before) + "ms");
                return retval;
            }
        }
    };
});

app.service('PromiseService', function() {
    const svc = this;

    /**
     * Wrap a $q promise in a $http promise (in order to keep the .success and .error methods)
     */
    svc.qToHttp = function(p) {
        return {
            then: p.then,
            success: function (fn) {
                return svc.qToHttp(p.then(fn));
            },
            error: function (fn) {
                return svc.qToHttp(p.then(null, fn));
            }
        }
    }

})

app.service('ProjectStatusService', function(TaggingService, $rootScope)  {
    const svc = this;
    let projectStatusMap = {};

    svc.getProjectStatusColor = function(status) {
        if(projectStatusMap && projectStatusMap[status]) {
            return projectStatusMap[status];
        } else {
            return TaggingService.getDefaultColor(status);
        }
    }

    function computeProjectStatusMap() {
        projectStatusMap = {};
        if ($rootScope.appConfig && $rootScope.appConfig.projectStatusList) {
            $rootScope.appConfig.projectStatusList.forEach(function(projectStatus) {
                projectStatusMap[projectStatus.name] = projectStatus.color;
            });
        }
    }
    $rootScope.$watch('appConfig.projectStatusList', computeProjectStatusMap, true);
});


app.service('ProjectFolderService', function(DataikuAPI, $q) {
    const svc = this;
    const ROOT_ID = 'ROOT';

    /**
     * Maps from ProjectFolderSummary to browse node information consisting
     * - children - an array of folder ids
     * - pathElts - an array of folder ids, from the root down to and including this node
     * - exists - always true
     * - directory - always true
     */
    svc.buildBrowseNode = function(folder) {
        const folders = folder.children.map(f => angular.extend({}, f, { directory: true, fullPath: f.id }))
        const pathElts = treeToList(folder, item => item.parent);
        return({
            children: folders,
            pathElts: pathElts.map(f => angular.extend({}, f, { toString: () => f.id })),
            exists: true,
            directory: true,
        });
    };

    /*
     * Adds pathElts to ProjectFolderSummary, a string like /Sandbox/marketing/summer
     */
    svc.addPathToFolder = (folder) => {
        const pathElts = treeToList(folder, item => item.parent);
        return angular.extend({}, folder, { pathElts: pathElts.map(f => f.name).join('/') });
    };

    // Return (a promise which returns) a ProjectFolderSummary object for the given folderId with minimal info
    svc.getFolder = function(folderId) {
        const requestFolderId = folderId || ROOT_ID;
        return DataikuAPI.projectFolders.getSummary(requestFolderId, 1, true, true).then(resp => {
            return resp.data;
        });
    };

    // Returns a function which sets the given path on the root folder if necessary
    const setRootPath = function(rootPath) {
        return function(folder) {
            if (folder.id === ROOT_ID) {
                folder.pathElts = rootPath;
            }
            return folder;
        }
    }

    svc.getDefaultFolder = function(folderId) {
        const requestFolderId = folderId || ROOT_ID;
        return DataikuAPI.projectFolders.getDefaultFolderSummary(requestFolderId).then(resp => {
            return resp.data;
        });
    };

    svc.getBrowseNode = function(folderId) {
        return svc.getFolder(folderId).then(svc.buildBrowseNode);
    }

    svc.getFolderWithPath = function(folderId) {
        return svc.getFolder(folderId).then(svc.addPathToFolder).then(setRootPath('/'));
    }

    // Return (a promise which returns) a folder with a path (in pathElts) to be used in a folder-path-input
    // If the folder cannot be found (or is not visible to the user) we fallback to the root folder
    // rootFolderDisplayText is the text or path to display when we fallback to the root
    svc.getFolderFallbackRoot = function(folderId, rootFolderDisplayText) {
        const deferred = $q.defer();
        this.getFolderWithPath(folderId).then((folder) => {
            deferred.resolve(folder);
        }).catch(() => {
            this.getFolder('').then(setRootPath(rootFolderDisplayText)).then((folder) => {
                deferred.resolve(folder);
            });
        });
        return deferred.promise;
    }

    // Return (a promise which returns) a folder (ProjectFolderSummary) with a path (pathElts) for initialising folder creation or duplication dialogs
    // If we are in the root folder, currentFolderId will be an empty string - meaning we will fallback to the default configured
    svc.getDefaultFolderForNewProject = function(currentFolderId) {
         // set rootpath to '/' to match what is added in browseDoneFn in folderPathInput (when show-root-folder-path="true")
        return svc.getDefaultFolder(currentFolderId).then(svc.addPathToFolder).then(setRootPath('/'));
    }

    svc.isInRoot = function(folderId) {
        return folderId === ROOT_ID;
    }

    // Return true if a folder is the SANDBOX folder or any SUBFOLDER of it
    svc.isUnderSandbox = function(folder) {
        if (!folder) {
            return false;
        }

        return folder.id === 'SANDBOX' || !!(folder.parent && svc.isUnderSandbox(folder.parent));
    }

});

/**
 * Enhance fattable elements with dragging capabilities.
 * Mandatory class fat-draggable should be added for parent draggable zone.
 * Mandatory class fat-draggable__item should be added for each items that can be dragged.
 * Mandatory class fat-draggable__handler should be added to trigger drag. It should be a child of an item or the item itself.
 * Mandatory data-column-name attribute should be added at fat-draggable__item level to keep a reference to the column when DOM is being recycled.
 *
 * To enable drag on an element, call the setDraggable() method with the following options :
 *
 * @param {Object}              options                         - The available options
 * @param {HTMLElement}         options.element                 - (Mandatory) element containing the draggable items.
 * @param {String}              [options.axis="x"]              - Define the dragging axis. Default to horizontal dragging.
 * @param {Function}            [options.onDrop]                - Drop callback
 * @param {Function}            [options.onPlaceholderUpdate]   - Placeholder dimensions update. Use it to reshape / position the placeholder. Called with the placeholder dimensions
 * @param {ScrollBarProxy}      [options.scrollBar]             - Fattable scrollbar to be updated if necessary
 *
 * @example
 *
 * <div class="fat-draggable">
 *  <div class="fat-draggable__item" data-column-name="{{column.name}}">
 *       <i class="fat-draggable__handler"></i>
 *       ...
 *  </div>
 * </div>
 */
app.factory('FatDraggableService', function() {
    const MAIN_CLASSNAME = 'fat-draggable';
    const HANDLER_CLASSNAME = MAIN_CLASSNAME + '__handler';
    const ITEM_CLASSNAME = MAIN_CLASSNAME + '__item';
    const PLACEHOLDER_CLASSNAME = MAIN_CLASSNAME + '__placeholder';
    const BAR_CLASSNAME = MAIN_CLASSNAME + '__bar';
    const DRAGGING_CLASSNAME = MAIN_CLASSNAME + '--dragging';
    const DRAGGED_CLASSNAME = ITEM_CLASSNAME + '--dragged';
    const COLUMN_NAME_ATTRIBUTE = 'data-column-name';
    const BAR_THICKNESS = 2;
    const MINIMAL_MOVE_TO_DRAG = 10;
    let classNamesToIgnore;
    let scrollBar;
    let element;
    let axis = 'x';
    let axisClassname;
    let placeholderDOM = document.createElement('div');
    let barDOM = document.createElement('div');
    let draggedItem;
    let draggedColumnName;
    let disabledTarget;
    let draggedItemDimensions = {};
    let placeholderDimensions = {};
    let barDimensions = {};
    let elementDimensions = {};
    let hoveredItem;
    let hoveredColumnName;
    let hoveredItemDimensions = {};
    let dragging = false;
    let downing = false;
    let onDrop;
    let onPlaceholderUpdate;
    let cursorInitialPosition = -1;
    let cursorPosition = -1;
    let gap = -1;

    // Ensures requestAnimationFrame cross-browsers support
    window.requestAnimationFrame = window.requestAnimationFrame || window.mozRequestAnimationFrame || window.webkitRequestAnimationFrame || window.msRequestAnimationFrame;

    /* HELPERS */

    // Gets the draggable item matching the mouse target
    function getDraggableItem(target) {
        return target.closest('.' + ITEM_CLASSNAME);
    }

    function getColumnName(columnDOM) {
        if (columnDOM) {
            return columnDOM.getAttribute(COLUMN_NAME_ATTRIBUTE) || (columnDOM.firstElementChild && columnDOM.firstElementChild.getAttribute(COLUMN_NAME_ATTRIBUTE));
        }
    }

    function getColumnDOM(columnName) {
        if (columnName) {
            return element.querySelector('[' + COLUMN_NAME_ATTRIBUTE + '="' + columnName + '"]');
        }
    }

    // Returns true if we are dragging before (on top in y axis or on left on x axis) the item being dragged
    function isDraggingBefore() {
        return cursorPosition && draggedItemDimensions && (cursorPosition <= draggedItemDimensions[axis === 'y' ? 'top' : 'left']);
    }

    // Retrieves the item around the cursor position
    function getHoveredItem() {
        let items = element.querySelectorAll('.' + ITEM_CLASSNAME);
        let vertical = (axis === 'y');
        for (let i = 0; i < items.length; i++) {
            let currentItem = items[i];
            let itemDimensions = getBoundingClientRect(currentItem);
            if (vertical) {
                if (itemDimensions.top <= cursorPosition && cursorPosition < itemDimensions.top + itemDimensions.height) {
                    return currentItem;
                }
            } else {
                if (itemDimensions.left <= cursorPosition && cursorPosition < itemDimensions.left + itemDimensions.width) {
                    return currentItem;
                }
            }
        }
        return null;
    }

    // Sets hover style for the draggable item matching the mouse target
    function updateHoveredItem() {
        let draggingBefore = isDraggingBefore();
        let target = getHoveredItem();

        if (!target) {
            return;
        }
        if (hoveredItem && hoveredItem === target) {
            return;
        }

        hoveredItem = target;
        hoveredColumnName = getColumnName(hoveredItem);

        if (hoveredItem !== draggedItem) {
            hoveredItemDimensions = getBoundingClientRect(hoveredItem);
            elementDimensions = getBoundingClientRect(element);

            if (axis === 'y') {
                barDimensions.top = draggingBefore ? hoveredItemDimensions.top : hoveredItemDimensions.top + hoveredItemDimensions.height;
                barDimensions.height = BAR_THICKNESS;
                barDimensions.width = placeholderDimensions.width;
                barDimensions.left = elementDimensions.left;
            } else {
                barDimensions.left = draggingBefore ? hoveredItemDimensions.left : hoveredItemDimensions.left + hoveredItemDimensions.width;
                barDimensions.height = placeholderDimensions.height;
                barDimensions.width = BAR_THICKNESS;
                barDimensions.top = elementDimensions.top;
            }
        } else {
            barDimensions.width = 0;
            barDimensions.height = 0;
        }

        updateBarBoundingBox(barDimensions);
    }

    // Redraw the placeholder according the mouse position
    function updatePlaceholderBoundingBox(dimensions) {

        if (typeof onPlaceholderUpdate === 'function') {
            onPlaceholderUpdate(dimensions);
        }

        if (dimensions.top >= 0) {
            if (dimensions.top <= elementDimensions.top - placeholderDimensions.height / 2) {
                // If overflowing on top
                // Putting at top of parent element - half of the placeholder (for better auto scroll)
                dimensions.top = elementDimensions.top - placeholderDimensions.height / 2;
            } else if (dimensions.top + placeholderDimensions.height - placeholderDimensions.height / 2 >= elementDimensions.top + elementDimensions.height) {
                // If overflowing on bottom
                // Putting at bottom of parent element + half of the placeholder (for better auto scroll)
                dimensions.top = elementDimensions.top + elementDimensions.height - placeholderDimensions.height / 2;
            }
            placeholderDOM.style.top = dimensions.top + 'px';
        }
        if (dimensions.left >= 0) {
            if (dimensions.left <= elementDimensions.left - placeholderDimensions.width / 2) {
                // If overflowing on the left
                // Putting at left of parent element - half of the placeholder (for better auto scroll)
                dimensions.left = elementDimensions.left - placeholderDimensions.width / 2;
            } else if (dimensions.left + placeholderDimensions.width - placeholderDimensions.width / 2 >= elementDimensions.left + elementDimensions.width) {
                // If overflowing on the right
                // Putting at right of parent element + half of the placeholder (for better auto scroll)
                dimensions.left = elementDimensions.left + elementDimensions.width - placeholderDimensions.width / 2;
            }
            placeholderDOM.style.left = dimensions.left + 'px';
        }

        if (dimensions.height >= 0) {
            placeholderDOM.style.height = dimensions.height + 'px';
        }
        if (dimensions.width >= 0) {
            placeholderDOM.style.width = dimensions.width + 'px';
        }
    }

    // Wrap the placeholder position update in a callback for requestAnimationFrame()
    function placeholderDOMRedraw() {
        updatePlaceholderBoundingBox(axis === 'y' ? {top: (cursorPosition - gap)} : {left: (cursorPosition - gap)});
    }

    // Redraw the bar according the given dimensions
    function updateBarBoundingBox(dimensions) {
        if (dimensions.top >= 0) {
            if (dimensions.top <= elementDimensions.top) {
                dimensions.top = elementDimensions.top;
            } else if (dimensions.top >= elementDimensions.top + elementDimensions.height) {
                dimensions.top = elementDimensions.top + elementDimensions.height;
            }
            barDOM.style.top = dimensions.top + 'px';
        }
        if (dimensions.left >= 0) {
            if (dimensions.left <= elementDimensions.left) {
                dimensions.left = elementDimensions.left;
            } else if (dimensions.left >= elementDimensions.left + elementDimensions.width) {
                dimensions.left = elementDimensions.left + elementDimensions.width;
            }
            barDOM.style.left = dimensions.left + 'px';
        }
        if (dimensions.height >= 0) {
            barDOM.style.height = dimensions.height + 'px';
        }
        if (dimensions.width >= 0) {
            barDOM.style.width = dimensions.width + 'px';
        }
    }

    // Generic fatTable scroll update
    function updateScroll () {
        if (!scrollBar) {
            return;
        }
        if (axis === 'y') {
            let elementTop = getBoundingClientRect(element).top;
            let elementBottom = elementTop + element.offsetHeight;
            let placeholderTop= placeholderDOM.offsetTop;
            let placeholderBottom = placeholderTop + placeholderDOM.offsetHeight;

            if (placeholderBottom > elementBottom) {
                scrollBar.setScrollXY(element.scrollLeft, scrollBar.scrollTop + placeholderBottom - elementBottom);
            } else if (placeholderTop < elementTop) {
                scrollBar.setScrollXY(element.scrollLeft, scrollBar.scrollTop + placeholderTop - elementTop);
            }
        } else {
            let elementLeft = getBoundingClientRect(element).left;
            let elementRight = elementLeft + element.offsetWidth;
            let placeholderLeft = placeholderDOM.offsetLeft;
            let placeholderRight = placeholderLeft + placeholderDOM.offsetWidth;

            if (placeholderRight > elementRight) {
                scrollBar.setScrollXY(scrollBar.scrollLeft + placeholderRight - elementRight, element.scrollTop);
            } else if (placeholderLeft < elementLeft) {
                scrollBar.setScrollXY(scrollBar.scrollLeft + placeholderLeft - elementLeft, element.scrollTop);
            }
        }
    }

    // Clean every dragging-related things
    function reset() {
        document.body.contains(placeholderDOM) && document.body.removeChild(placeholderDOM);
        barDOM.style.top = '';
        barDOM.style.left = '';
        barDOM.style.height = '';
        barDOM.style.width = '';
        document.body.removeChild(barDOM);
        draggedItem && draggedItem.classList.remove(DRAGGED_CLASSNAME);
        document.body.classList.remove(DRAGGING_CLASSNAME);

        window.removeEventListener('mousemove', onMouseMove);
        window.removeEventListener('mouseup', onMouseUp);
        disabledTarget.removeEventListener('mouseup', disableClick);
        Mousetrap.unbind("esc", onEscape);

        dragging = false;
        downing = false;
        draggedItem = null;
        draggedColumnName = null;
        hoveredItem = null;
        hoveredColumnName = null;
        disabledTarget = null;
        draggedItemDimensions = {};
        placeholderDimensions = {};
        barDimensions = {};
        hoveredItemDimensions = {};
        elementDimensions = {};
        cursorPosition = -1;
        cursorInitialPosition = -1;
        gap = -1;
    }

    // Prevent the beginning of the drag
    function cancel() {
        if (dragging) {
            reset();
        } else {
            cursorInitialPosition = -1;
            downing = false;
            draggedItem = null;
            draggedColumnName = null;
            window.removeEventListener('mousemove', onMouseMove);
        }
    }

    /* EVENTS LISTENERS */
    function onMouseUp() {

        if (!dragging) {
            cancel();
            return;
        }
        if (typeof onDrop === 'function' && draggedColumnName && hoveredColumnName && draggedColumnName !== hoveredColumnName) {
            onDrop(draggedItem, hoveredItem, draggedColumnName, hoveredColumnName);
        }
        reset();
    }

    // When the dragging has started we do not want any click to be triggered on the element
    function disableClick(event) {
        event.stopImmediatePropagation();
        event.preventDefault();
        onMouseUp();
    }

    // If moving the placeholder, update its position according the given axis
    function onMouseMove(event) {

        // If not yet dragging, initiate drag, else update the placeholder position
        if (!dragging && downing) {
            if (axis === 'y') {
                cursorPosition = event.clientY;
                gap = cursorInitialPosition - draggedItemDimensions.y;
            } else {
                cursorPosition = event.clientX;
                gap = cursorInitialPosition - draggedItemDimensions.x;
            }

            // Do not start drag if the mouse has not moved enough
            if (Math.abs(cursorPosition - cursorInitialPosition) < MINIMAL_MOVE_TO_DRAG) {
                return;
            }

            dragging = true;
            // Bind mouseup for dragging end and click on target to prevent other listeners to be triggered
            window.addEventListener('mouseup', onMouseUp);
            disabledTarget = event.target;
            disabledTarget.addEventListener('click', disableClick);

            // Inject placeholder and bar in DOM and add the drag-related class names
            document.body.appendChild(placeholderDOM);
            document.body.appendChild(barDOM);
            document.body.classList.add(DRAGGING_CLASSNAME);
            draggedItem.classList.add(DRAGGED_CLASSNAME);

            placeholderDimensions = angular.copy(draggedItemDimensions);
            updatePlaceholderBoundingBox(placeholderDimensions);

            Mousetrap.bind("esc", onEscape);
        } else {
            cursorPosition = axis === 'y' ? event.clientY : event.clientX;
            requestAnimationFrame(placeholderDOMRedraw);
            updateScroll();
            // If the dragged column DOM from fattable has been removed try to re-fetch it
            if (!document.body.contains(draggedItem)) {
                let newDraggedItem = getColumnDOM(draggedColumnName);
                if (newDraggedItem) {
                    draggedItem = newDraggedItem;
                    draggedItem.classList.add(DRAGGED_CLASSNAME);
                }
            // If the dragged column DOM and column name are inconsistent, invalidate it and try to refetch it
            } else if (!draggedItem.getAttribute(COLUMN_NAME_ATTRIBUTE) || draggedItem.getAttribute(COLUMN_NAME_ATTRIBUTE) !== draggedColumnName) {
                draggedItem.classList.remove(DRAGGED_CLASSNAME);
                draggedItem = null;
                let newDraggedItem = getColumnDOM(draggedColumnName);
                if (newDraggedItem) {
                    draggedItem = newDraggedItem;
                    draggedItem.classList.add(DRAGGED_CLASSNAME);
                }
            }
        }

        updateHoveredItem();
    }

    // Press escape to stop the current drag
    function onEscape() {
        if (!dragging) {
            return;
        }
        reset();
    }

    function onMouseDown(event) {

        // Do not drag if the selected element has at least one class marking it as not-draggable
        try {
            classNamesToIgnore.forEach(className => {
                if (event.target.closest('.' + className)) {
                    throw new Error("Ignoring drag, element not draggable.");
                }
            });
        } catch (e) {
            return;
        }

        if (!event.target.closest('.' + HANDLER_CLASSNAME)) {
            return;
        }

        // Do not consider right click
        if (event.which === 3) {
            return;
        }

        downing = true;
        draggedItem = getDraggableItem(event.target);
        draggedColumnName = getColumnName(draggedItem);

        window.addEventListener('mousemove', onMouseMove);

        // If a click occurred, prevent dragging
        element.addEventListener('click', cancel);

        // Prevent native drag
        event.preventDefault();

        // Ensure mandatory class name are here
        element.classList.add(MAIN_CLASSNAME);
        element.classList.add(axisClassname);

        draggedItemDimensions = getBoundingClientRect(draggedItem);

        if (axis === 'y') {
            cursorInitialPosition = event.clientY;
        } else {
            cursorInitialPosition = event.clientX;
        }
    }

    return {

        setDraggable: function(options) {
            if (!options || !options.element) { return }
            element = options.element;
            onDrop = options.onDrop;
            onPlaceholderUpdate = options.onPlaceholderUpdate;
            scrollBar = options.scrollBar;
            classNamesToIgnore = options.classNamesToIgnore;

            element.classList.add(MAIN_CLASSNAME);

            axis = options.axis && options.axis === 'y' ? 'y' : 'x';
            axisClassname = axis === 'y' ? 'fat-draggable-y-axis' : 'fat-draggable-x-axis';
            element.classList.add(axisClassname);

            placeholderDOM.className = PLACEHOLDER_CLASSNAME;
            barDOM.className = BAR_CLASSNAME;

            // If clicking on a drag handler, retrieve dragged item data and attach mouse move
            element.addEventListener('mousedown', onMouseDown);
        }
    }
});

/**
 * Enhance fattable elements with resize capabilities.
 * Mandatory class fat-resizable__item should be added for each items that can be dragged.
 * Mandatory class fat-resizable__handler should be added to trigger resize. It should be a child of an item or the item itself.
 * Mandatory data-column-name and data-column-index attributes should be added at fat-resizable__item level to keep a reference to the column when DOM is being recycled.
 *
 * To enable resize on an element, call the setResizable() method with the following options :
 *
 * @param {Object}              options                         - The available options
 * @param {HTMLElement}         options.element                 - (Mandatory) element containing the resizable items.
 * @param {Function}            options.onDrop               - Drop callback. Returns an object containing resized column data: index, name and width.
 *
 * @example
 *
 * <div fat-resizable>
 * ...
 *  <div class="fat-resizable__item" data-column-name="{{column.name}}" data-column-index="{{columnIndex}}">
 *      ...
 *      <span class="fat-resizable__handler"></span>
 *      ...
 *  </div>
 * ...
 * </div>
 */
app.factory('FatResizableService', function () {
    const MAIN_CLASSNAME = 'fat-resizable';
    const ITEM_CLASSNAME = MAIN_CLASSNAME + '__item';
    const HANDLER_CLASSNAME = MAIN_CLASSNAME + '__handler';
    const BAR_CLASSNAME = MAIN_CLASSNAME + '__bar';
    const COLUMN_INDEX_ATTRIBUTE_NAME = 'data-column-index';
    const COLUMN_NAME_ATTRIBUTE_NAME = 'data-column-name';
    const BAR_THICKNESS = 4;
    const ITEM_MIN_WIDTH = 60;

    let element,
        onDrop,
        resizing = false,
        downing = false,
        cursorPosition = -1,
        barDimensions = {},
        elementDimensions,
        resizableItem = null,
        resizedItemDimensions = {},
        draggingLowerBound,
        disabledTarget,
        barDOM = document.createElement('div');

    /* HELPERS */

    // Gets the resizable item matching the mouse target
    function getResizableItem(target) {
        return target.closest('.' + ITEM_CLASSNAME);
    }

    function updateBar() {
        elementDimensions = elementDimensions || getBoundingClientRect(element);
        barDimensions.left = cursorPosition;
        barDimensions.top = elementDimensions.top;
        barDOM.style.top = barDimensions.top + 'px';
        barDOM.style.left = barDimensions.left + 'px';
        barDOM.style.height = barDimensions.height + 'px';
        barDOM.style.width = barDimensions.width + 'px';
    }

    // Clean every resizing-related things
    function reset() {
        barDOM.style.top = '';
        barDOM.style.left = '';
        barDOM.style.height = '';
        barDOM.style.width = '';
        document.body.removeChild(barDOM);

        window.removeEventListener('mousemove', onMouseMove);
        window.removeEventListener('mouseup', onMouseUp);
        disabledTarget.removeEventListener('mouseup', disableClick);
        Mousetrap.unbind("esc", onEscape);

        resizing = false;
        downing = false;
        barDimensions.left = 0;
        barDimensions.right = 0;
        cursorPosition = -1;
        draggingLowerBound = null;
        resizableItem = null;
        resizedItemDimensions = {};
    }

    // Prevent the beginning of the drag
    function cancel() {
        if (resizing) {
            reset();
        } else {
            downing = false;
            resizableItem = null;
            resizedItemDimensions = {};
            window.removeEventListener('mousemove', onMouseMove);
        }
    }

    /* EVENTS LISTENERS */

    function onMouseUp() {

        if (!resizing) {
            cancel();
            return;
        }

        if (typeof onDrop === 'function') {
            let resizedWidth = cursorPosition - resizedItemDimensions.x;
            resizedWidth = resizedWidth > ITEM_MIN_WIDTH ? resizedWidth : ITEM_MIN_WIDTH;
            onDrop({
                index: resizableItem.getAttribute(COLUMN_INDEX_ATTRIBUTE_NAME),
                name: resizableItem.getAttribute(COLUMN_NAME_ATTRIBUTE_NAME),
                width: resizedWidth
            });
        }
        reset();
    }

    // When the dragging has started we do not want any click to be triggered on the element
    function disableClick(event) {
        event.stopImmediatePropagation();
        event.preventDefault();
        onMouseUp();
    }

    function onMouseMove(event) {

        // If not yet dragging, initiate drag, else update the placeholder position
        if (!resizing && downing) {

            resizing = true;
            // Bind mouseup for dragging end and click on target to prevent other listeners to be triggered
            window.addEventListener('mouseup', onMouseUp);
            disabledTarget = event.target;
            disabledTarget.addEventListener('click', disableClick);

            // Inject bar in DOM
            document.body.appendChild(barDOM);

            Mousetrap.bind("esc", onEscape);
        } else {
            cursorPosition = event.clientX;
        }

        if (!draggingLowerBound) {
            draggingLowerBound = getBoundingClientRect(resizableItem).x
        }

        // Prevent resizing beyond the previous column
        if (cursorPosition <= draggingLowerBound + ITEM_MIN_WIDTH) { return }

        updateBar();
    }

    function onMouseDown(event) {

        if (!event.target.closest('.' + HANDLER_CLASSNAME)) {
            return;
        }

        // Do not consider right click
        if (event.which === 3) {
            return;
        }

        downing = true;
        resizableItem = getResizableItem(event.target);

        window.addEventListener('mousemove', onMouseMove);

        // If a click occurred, prevent dragging
        element.addEventListener('click', cancel);

        // Prevent native drag
        event.preventDefault();

        resizedItemDimensions = getBoundingClientRect(resizableItem);
    }

    // Press escape to stop the current drag
    function onEscape() {
        if (!resizing) {
            return;
        }
        reset();
    }

    return {

        setResizable: function(options) {
            if (!options || !options.element) { return }

            onDrop = options.onDrop;
            element = options.element;

            // Prepare vertical bar used as feedback while resizing
            barDimensions.height = options.barHeight;
            barDimensions.width = BAR_THICKNESS;
            barDOM.className = BAR_CLASSNAME;

            // If clicking on a drag handler, retrieve dragged item data and attach mouse move
            element.addEventListener('mousedown', onMouseDown);
        }
    }
})

app.service('FatTouchableService', function($timeout) {
    /**
     * Make a fattable/fatrepeat scrollable through touch interaction
     * @param scope: scope of the fattable/fatrepeat directive
     * @param element: DOM element of the fattable/fatrepeat directive
     * @param fattable: fattable object of the fattable/fatrepeat directive
     * @returns function to remove event listeners added by this function
     */
    this.setTouchable = function(scope, element, fattable) {
        /**
         * Return an object wrapping callbacks. This function will be called each time a touchstart is emmited in order
         * to give callbacks to the added the touchmove and touchend event listeners.
         */
        let getOnTouchCallbacks = (function() {
            /*
             * Callbacks
             */

            /**
             * Turns the touchMoveEvent passed in parameter into a scrollOrder to the fattable
             * @param event
             */
            let startScroll;
            let lastScroll;
            let startTouch;
            let lastTouch;
            function scrollFattable(touchMoveEvent) {
                fattable.scroll.dragging = true; //otherwise fattable behaves as user had scrolled using scrollbars, which triggers multiple reflows, which is bad for performances
                touchMoveEvent.preventDefault();
                touchMoveEvent.stopPropagation();
                let newTouch = touchMoveEvent.originalEvent.changedTouches[0];

                // Tracking for direction change
                function getTouchedDistance(t) {
                    return {
                        x: startTouch.screenX - t.screenX,
                        y: startTouch.screenY - t.screenY
                    }
                }
                let touchedDistance = getTouchedDistance(newTouch);
                let lastTouchedDistance = getTouchedDistance(lastTouch);
                if (Math.abs(lastTouchedDistance) - Math.abs(touchedDistance) > 0) {
                    startTouch = lastTouch;
                    startScroll = lastScroll;
                    touchedDistance = getTouchedDistance(newTouch);
                }
                // Scrolling
                requestAnimationFrame(_ => fattable.scroll.setScrollXY(startScroll.x + touchedDistance.x, startScroll.y + touchedDistance.y));
                // Updating memory
                lastTouch = touchMoveEvent.originalEvent.changedTouches[0];
                lastScroll = {
                    x: fattable.scroll.scrollLeft,
                    y: fattable.scroll.scrollTop
                }
            }

            /**
             * Keeps track of touch velocity in order to generate momentum when touchmove will stop.
             * (Tracking will stop at touchEnd)
             * @type {number}
             */
            const SPEED_FILTER = 0.8; // Arbitrary chosen constant
            let prevScroll;
            let velocity;
            let lastTimestamp;
            let keepTrackingVelocity;
            function trackVelocity() {
                // current scroll position
                let scroll = {
                    x: fattable.scroll.scrollLeft,
                    y: fattable.scroll.scrollTop
                };
                // scrolled distance since last track
                let delta = {
                    x: scroll.x - prevScroll.x,
                    y: scroll.y - prevScroll.y
                };
                // if scroll changed direction then we do not take previous velocity into account
                let prevVelocityCoeff = {
                    x: delta.x * velocity.x > 0 ? 0.2 : 0,
                    y: delta.y * velocity.y > 0 ? 0.2 : 0,
                };
                // computing velocity
                let timeStamp = Date.now();
                velocity.x = SPEED_FILTER * delta.x * 1000 / (1 + timeStamp - lastTimestamp) + prevVelocityCoeff.x * velocity.x;
                velocity.y = SPEED_FILTER * delta.y * 1000 / (1 + timeStamp - lastTimestamp) + prevVelocityCoeff.y * velocity.y;
                // updating memory
                lastTimestamp = timeStamp;
                prevScroll = scroll;
                if (keepTrackingVelocity) {
                    $timeout(trackVelocity, 10);
                }
            }

            /**
             * Generates a momentum animation when touch stops
             * @param event
             */
            function animateMomentum() {
                keepTrackingVelocity = false;
                let endTime = Date.now();
                // Momentum appears only if velocity was greater than 10px/s
                if (Math.abs(velocity.x) > 10 || Math.abs(velocity.y) > 10) {
                    // Detach event listeners when momentum ends
                    let onMomentumEnd = function () {
                        element.off('touchstart', stopMomentumAnimation);
                        element.off('touchmove', stopMomentumAnimation);
                        fattable.scroll.dragging = false;
                    }

                    // Stop momentum on new touchevent
                    let interruptedMomentum = false;
                    let stopMomentumAnimation = function () {
                        interruptedMomentum = true;
                        onMomentumEnd();
                    }
                    element.on('touchstart', stopMomentumAnimation);
                    element.on('touchmove', stopMomentumAnimation);

                    // Compute if scroll can go further with inertia
                    let canScroll = function (delta) {
                        let canScrollH = (delta.x < 0 && fattable.scroll.scrollLeft > 0) || (delta.x > 0 && fattable.scroll.scrollLeft < fattable.scroll.maxScrollHorizontal);
                        let canScrollV = (delta.y < 0 && fattable.scroll.scrollTop > 0) || (delta.y > 0 && fattable.scroll.scrollTop < fattable.scroll.maxScrollVertical);
                        return canScrollH || canScrollV;
                    }

                    /*
                     * MOMENTUM (all formulas come from this article: https://ariya.io/2013/11/javascript-kinetic-scrolling-part-2)
                     */

                    // additional distance to scroll while momentum
                    let amplitude = {
                        x: SPEED_FILTER * velocity.x,
                        y: SPEED_FILTER * velocity.y
                    }
                    let previousScroll = {x: 0, y: 0};
                    let autoScroll = function () {
                        const TIME_CONSTANT = 325; // arbitrary constant chosen experimentally
                        let elapsedSinceStop = Date.now() - endTime; // elapsed time since touchend
                        let exponentialDecay = Math.exp(-elapsedSinceStop / TIME_CONSTANT);
                        // where scroll should be at this time (due to inertia)
                        let scroll = {
                            x: amplitude.x * (1 - exponentialDecay),
                            y: amplitude.y * (1 - exponentialDecay)
                        }
                        // missing scroll distance (where scroll should be minus where scroll is now)
                        let delta = {
                            x: scroll.x - previousScroll.x,
                            y: scroll.y - previousScroll.y
                        }
                        // scrolling of missing scrolled distance
                        fattable.scroll.setScrollXY(fattable.scroll.scrollLeft + delta.x, fattable.scroll.scrollTop + delta.y);
                        previousScroll = scroll;
                        // momentum keeps going on until amplitude is almost reached or animation got interrupted by a new touchevent
                        if ((Math.abs(amplitude.x - scroll.x) > 0.5 || Math.abs(amplitude.y - scroll.y) > 0.5) && !interruptedMomentum && canScroll(delta)) {
                            requestAnimationFrame(autoScroll);
                        } else {
                            onMomentumEnd()
                        }
                    }
                    requestAnimationFrame(autoScroll);
                }
            }

            /*
             * Actual getOnTouchCallbacks function
             */
            return function (touchStartEvent) {
                // Initialization of scrollFattable variables
                startScroll = {
                    x: fattable.scroll.scrollLeft,
                    y: fattable.scroll.scrollTop
                };
                lastScroll = {
                    x: fattable.scroll.scrollLeft,
                    y: fattable.scroll.scrollTop
                };
                startTouch = touchStartEvent.originalEvent.changedTouches[0];
                lastTouch = touchStartEvent.originalEvent.changedTouches[0];

                // Initialization of trackVelocity variables
                prevScroll = {
                    x: fattable.scroll.scrollLeft,
                    y: fattable.scroll.scrollTop
                };
                velocity = {x: 0, y: 0};
                lastTimestamp = Date.now();
                keepTrackingVelocity = true;

                return {
                    onTouchStart: function (e) {
                        trackVelocity(e);
                    },
                    onTouchMove: function (e) {
                        scrollFattable(e);
                    },
                    onTouchEnd: function (e) {
                        animateMomentum(e);
                    }
                };
            }
        })();

        function onTouchStart(event) {
            let currentOnTouchCallbacks = getOnTouchCallbacks(event, element);
            currentOnTouchCallbacks.onTouchStart();
            element.on("touchmove", currentOnTouchCallbacks.onTouchMove);
            let onTouchEnd = function(event) {
                currentOnTouchCallbacks.onTouchEnd(event);
                element.off("touchmove", currentOnTouchCallbacks.onTouchMove);
                element.off("touchend", onTouchEnd);
            }
            element.on("touchend", onTouchEnd);
        }

        element.on("touchstart", onTouchStart);

        let removeOnDestroy = scope.$on('$destroy', function() {
            element.off("touchstart", onTouchStart);
        });

        return function() {
            removeOnDestroy();
            element.off("touchstart", onTouchStart)
        }
    }
});

app.service('ClipboardUtils', function($q, $rootScope, translate, ActivityIndicator) {
    const svc = this;

    /**
     * Copy some text into the clipboard and notify users of success/failure with an ActivityIndicator banner message.
     *
     * @param text text to copy to the clipbard. If empty, does nothing and immediately resolves the promise (successfully).
     * @param successMessage Localized message to display with ActivityIndicator upon success. Use null/undefined to use the default message.
     *          Use an empty string to prevent the service from displaying any message.
     * @param errorMessage Localized message to display with ActivityIndicator in case of error. Use null/undefined to use the default message.
     *          Use an empty string to prevent the service from displaying any message in case of error.
     * @return {Promise<void>|*} A promise that resolves when 'text' has been copied to the clipboard, or if an error occurred.
     */
    svc.copyToClipboard = function(text, successMessage, errorMessage) {
        function showSuccess(triggerDigest = true) {
            if (successMessage === undefined || successMessage === null) {
                successMessage = translate('GLOBAL.CLIPBOARD.COPY_SUCCESS', 'Copied to clipboard!');
            }
            if (successMessage) {
                ActivityIndicator.success(successMessage, 5000);
                if (triggerDigest) {  // Trigger an immediate digest as we are outside AngularJS
                    $rootScope.$apply();
                }
            }
        }
        function showError(triggerDigest = true) {
            if (errorMessage === undefined || errorMessage === null) {
                errorMessage = translate('GLOBAL.CLIPBOARD.COPY_ERROR', 'Failed to copy to the clipboard!');
            }
            if (errorMessage) {
                ActivityIndicator.error(errorMessage, 5000);
                if (triggerDigest) {  // Trigger an immediate digest as we are outside AngularJS
                    $rootScope.$apply();
                }
            }
        }
        if (!text) {
            return Promise.resolve();
        }
        if (!navigator.clipboard) { // In case we are not in a secure context (for example a self-signed certificate)
            let tempInput = document.createElement("textarea");
            tempInput.style = "position: absolute; left: -1000px; top: -1000px";
            tempInput.value = text;
            document.body.appendChild(tempInput);
            tempInput.select();
            try {
                // noinspection JSDeprecatedSymbols
                if (document.execCommand("copy")) {
                    showSuccess(false);
                    return Promise.resolve();
                } else {
                    showError(false);
                    return Promise.reject(new Error('execCommand copy failed'));
                }
            } catch (err) {
                showError(false);
                return Promise.reject(err);
            } finally {
                document.body.removeChild(tempInput);
            }
        }
        return navigator.clipboard.writeText(text).then(showSuccess, showError);
    };
    // for pasting into non-editable element
    // called after capturing ctrl + v keydown event
    svc.pasteFromClipboard = function(event, callback) {
        let tempInput = document.createElement("textarea");
        tempInput.style = 'position: absolute; left: -1000px; top: -1000px';
        document.body.appendChild(tempInput);
        tempInput.select();
        // delay to capture imput value
        window.setTimeout(function() {
            let data = tempInput.value;

            callback(data);

            document.body.removeChild(tempInput);

            if (event) {
                event.currentTarget.focus();
            }
        }, 100);
    }
});

app.factory('DetectUtils', function() {
    const svc = {
        getOS: function() {
            let browser = '';

            if (navigator.appVersion.indexOf("Win") !== -1){
                browser = 'windows';
            }

            if (navigator.appVersion.indexOf("Mac")!=-1){
                browser = 'macos';
            }

            if (navigator.appVersion.indexOf("X11")!=-1){
                browser = 'unix';
            }

            if (navigator.appVersion.indexOf("Linux")!=-1){
                browser = 'linux';
            }

            return browser;
        },
        isMac: function() {
            return svc.getOS() === 'macos';
        },
        isWindows: function() {
            return svc.getOS() === 'windows';
        },
        isUnix: function() {
            return svc.getOS() === 'unix';
        },
        isLinux: function() {
            return svc.getOS() === 'linux';
        }
    };

    return svc;
});

app.constant("GRAPHIC_EXPORT_OPTIONS", {
    fileTypes: ['PDF','JPEG','PNG'],
    orientationMap: {
        'LANDSCAPE': 'Landscape',
        'PORTRAIT': 'Portrait'
    },
    paperSizeMap: {
        'A4': 'A4',
        'A3': 'A3',
        'US_LETTER': 'US Letter',
        'LEDGER': 'Ledger (ANSI B)',
        'SCREEN_16_9': '16:9 (Computer screen)',
        'CUSTOM': 'Custom'
    },
    paperSizeMapPage: {
        'A4': 'A4',
        'A3': 'A3',
        'US_LETTER': 'US Letter',
        'LEDGER': 'Ledger (ANSI B)'
    },
    paperInchesMap: {
        'A4': 11.6929,
        'A3': 16.5354,
        'US_LETTER': 11,
        'LEDGER': 17,
        'SCREEN_16_9': 11
    },
    ratioMap: {
        'A4': Math.sqrt(2),
        'A3': Math.sqrt(2),
        'US_LETTER': 11 / 8.5,
        'LEDGER': 17 / 11,
        'SCREEN_16_9': 16 / 9,
        'CUSTOM': 16 / 9
    }
});

app.service('GraphicImportService', function(GRAPHIC_EXPORT_OPTIONS) {
    const svc = this;
    svc.computeHeight = function (width, paperSize, orientation) {
        if (orientation == "PORTRAIT") {
            return Math.round(width * GRAPHIC_EXPORT_OPTIONS.ratioMap[paperSize]);
        } else {
            return Math.round(width / GRAPHIC_EXPORT_OPTIONS.ratioMap[paperSize]);
        }
    };
});

app.service('StringUtils',  function() {
    return {
        transmogrify: function(name, usedNames, makeName, offset = 2, alwaysAddSuffix = false) {
            if (! (usedNames instanceof Set)) {
                usedNames = new Set(usedNames);
            }
            if (! usedNames.has(name) && !alwaysAddSuffix) {
                return name;
            }
            if (! makeName) {
                makeName = i => `${name} ${i}`;
            }

            let i = offset;
            while (usedNames.has(makeName(i, name))) {
                i++;
            }
        return makeName(i, name);
        },
        /**
         * Compute the real width (in pixels) that a given text will take once rendered in DOM for the given font.
         */
        getTextWidth(text, font = '12px Arial', canvas) {
            if (!canvas) {
                canvas = document.createElement('canvas');
            }
            const context = canvas.getContext('2d');
            context.font = font;
            return context.measureText(text).width || 0;
        },
        /**
         * Compute the real height (in pixels) that a given text will take once rendered in DOM for the given font.
         */
        getTextHeight(text, font = '12px Arial', canvas) {
            if (!canvas) {
                canvas = document.createElement('canvas');
            }
            const context = canvas.getContext('2d');
            context.font = font;
            const metrics = context.measureText(text);
            return metrics.actualBoundingBoxAscent + metrics.actualBoundingBoxDescent || 0;
        },
        /**
         * Return a string truncated such that its length when encoded in bytes does not exceed maxBytes.
         * Useful for WT1 mixpanel event param values which have a max length of 255 bytes in utf-8.
         * @param {string} text - string to truncate
         * @param {num} maxBytes - length in bytes to not be exceeded
         * @param {string} encodingScheme - optional, to indicate encoding if not utf-8
         * @returns text truncated as necessary
         */
        truncateTextToLengthInBytes(text, maxBytes, encodingScheme = 'utf-8') {
            const encoder = new TextEncoder(encodingScheme);
            const encoded = encoder.encode(text);
            if (encoded.length <= maxBytes) {
                return text;
            }

            // We need to truncate - but we need to make sure it is still a valid string
            // if it's invalid, keep slicing off bytes until it is valid
            const decoder = new TextDecoder(encodingScheme, {fatal : true});
            let trimmedEncoded = encoded.slice(0, maxBytes);
            let decodedText = null;
            while (!decodedText){
                try {
                    decodedText = decoder.decode(trimmedEncoded);
                } catch (e) {
                    trimmedEncoded = trimmedEncoded.slice(0, trimmedEncoded.length - 1);
                }
            };
            return decodedText;
        },
        //Same as src/app/utils/string-utils.ts
        normalizeTextForSearch(str) {
            return this.stripAccents(str ?? '').toLowerCase().trim();
        },
        stripAccents(str) {
            return str.normalize("NFD").replace(/[\u0300-\u036f]/g, "");
        },
    };
});

app.service('PermissionsService', function(Dialogs, Logger) {
    return {
        buildUnassignedGroups: function(item, allGroups) {
            if (!item || !allGroups) return;

            return allGroups.filter(function(groupName) {
                return item.permissions.every(perm => perm == null || perm.group !== groupName);
            });
        },
        buildUnassignedUsers: function(item, allUsers) {
            return allUsers.filter(user =>
                item.owner !== user.login && item.permissions.every(perm => perm == null || perm.user !== user.login));
        },
        transferOwnership: function(scope, item, itemName, ownerUiField="ownerLogin") {
            const ownerUi = scope.ui && scope.ui[ownerUiField];
            if (!ownerUi || !item || ownerUi === item.owner) return;
            const newOwnerDisplayName = scope.allUsers.find(user => user.login === ownerUi).displayName || ownerUi;

            Dialogs.confirm(scope, 'Ownership transfer',
                `Are you sure you want to transfer ${itemName} ownership to '${newOwnerDisplayName}' ?`).then(function() {
                Logger.info(`Transferring ${itemName} ownership to ${ownerUi}`);
                item.owner = ownerUi;
            },function() {
                scope.ui[ownerUiField] = item.owner;
            });
        }
    }
});

    app.service('FullScreenService', ['$location', '$state', function ($location, $state) {
        return {
            isFullScreen: () => $location.search().fullScreen && $location.search().fullScreen !== "false"
        }
    }]);

    app.service('MLUtilsService', function () {
        return {
            getMLCategory: (taskType, backendType, predictionType, savedModelType, proxyModelProtocol) => {
                // Keep aligned with SavedModel.getMLCategory in `SavedModel.java`
                switch (savedModelType) {
                    case 'MLFLOW_PYFUNC':
                        return 'mlflow';
                    case 'LLM_GENERIC':
                        return 'fine_tuning';
                    case 'PYTHON_AGENT':
                        return 'python_agent';
                    case 'PLUGIN_AGENT':
                        return 'plugin_agent';
                    case 'TOOLS_USING_AGENT':
                        return "tools_using_agent";
                    case 'RETRIEVAL_AUGMENTED_LLM':
                        return "retrieval_augmented_llm";
                    case 'PROXY_MODEL':
                        switch (proxyModelProtocol) {
                            case 'sagemaker':
                                return 'sagemaker';
                            case 'vertex-ai':
                                return 'vertex';
                            case 'azure-ml':
                                return 'azureml';
                            case 'databricks':
                                return 'databricks';
                        }
                    default:
                        break;
                }

                switch (taskType) {
                    case 'CLUSTERING':
                        return 'clustering';
                    case 'PREDICTION':
                        if (backendType === 'KERAS') {
                            return 'keras';
                        }
                        switch (predictionType) {
                            case 'TIMESERIES_FORECAST':
                                return 'timeseries';
                            case 'CAUSAL_REGRESSION':
                            case 'CAUSAL_BINARY_CLASSIFICATION':
                                return 'causal';
                            case 'DEEP_HUB_IMAGE_CLASSIFICATION':
                                return 'deephub_image_classification';
                            case 'DEEP_HUB_IMAGE_OBJECT_DETECTION':
                                return 'deephub_object_detection';
                        }
                    default:
                        return 'prediction';
                }
            },
        };
    });

    app.service('AgentToolService', function($rootScope, PluginsService, PluginConfigUtils) {
        const CUSTOM_PREFIX = 'Custom_';
        const pluginVisibilityFilter = PluginConfigUtils.shouldComponentBeVisible($rootScope.appConfig.loadedPlugins);
        const AGENT_TOOL_TYPES_UI_DATA = [
            {
                type: "InlinePython",
                typeInUI: "Custom Python",
                label: "Custom Python",
                description: "Implement your own custom tool in Python code",
                documentationLink: "/agents/tools/custom-tools",
                icon: "dku-icon-python-20",
                codeEnvSelectionSupported: true,
                containerExecutionSupported: true,
                testQuery: {},
                showStopDevKernelBtn: true,
                displayPriority: 60,
            },
            {
                type: "DatasetRowLookup",
                label: "Dataset Lookup",
                description: "Find a record in a dataset based on conditions on one or several columns",
                documentationLink: "/agents/tools/dataset-lookup",
                icon: "dku-icon-dataset-20",
                testQuery: {
                    "input": {
                        "filter": {
                            "column": "<COLUMN_NAME>",
                            "operator": "EQUALS",
                            "value": "<VALUE>"
                        }
                    }
                },
                showStopDevKernelBtn: false,
                displayPriority: 90,
            },
            {
                type: "VectorStoreSearch",
                label: "Knowledge Bank Search",
                description: "Search a Knowledge Bank for relevant information",
                documentationLink:"/agents/tools/knowledge-bank-search",
                icon: 'dku-icon-cards-stack-20',
                testQuery: {
                    "input": {
                        "searchQuery": "Enter search query here"
                    }
                },
                showStopDevKernelBtn: true,
                displayPriority: 100,
            },
            {
                type: "LLMMeshLLMQuery",
                label: "Query another agent or an LLM",
                description: "Send a request to an LLM or Agent registered in the LLM Mesh",
                documentationLink:"/agents/tools/llm-mesh-query",
                icon: 'dku-icon-ai-prompt-20',
                testQuery: {
                    "input": {
                        "question": "Ask your question here"
                    }
                },
                showStopDevKernelBtn: false,
                displayPriority: 70,
            },
            {
                type: "DatasetRowAppend",
                label: "Dataset Append",
                description: "Write a record to a Dataiku dataset",
                documentationLink:"/agents/tools/dataset-append",
                icon: "dku-icon-data-table-row-20",
                testQuery: {
                    "input": {
                        "record": {
                            "<COLUMN1_NAME>": "<COLUMN1_VALUE>",
                            "<COLUMN2_NAME>": "<COLUMN2_VALUE>"
                        }
                    }
                },
                showStopDevKernelBtn: false,
                displayPriority: 1,
            },
            {
                type: "DataikuReporter",
                label: "Send Message",
                description: "Send a message through a Dataiku Messaging Channel (email, Slack, Teams, ...)",
                icon: "dku-icon-data-user-email-20",
                testQuery: {
                    "input": {
                        "parameters": {
                            "message": "The message to send"
                        }
                    }
                },
                showStopDevKernelBtn: false,
                displayPriority: 1,
            },
            {
                type: "ClassicalPredictionModelPredict",
                label: "Model Prediction",
                description: "Predict a record with a Dataiku model",
                documentationLink:"/agents/tools/model-predict",
                icon: "dku-icon-automl-prediction-20",
                testQuery: {
                    "input": {
                        "record": {
                            "<FEATURE1_NAME>": "<FEATURE1_VALUE>",
                            "<FEATURE2_NAME>": "<FEATURE2_VALUE>"
                        }
                    }
                },
                showStopDevKernelBtn: false,
                displayPriority: 40,
            },
            {
                type: "GRELCalculator",
                typeInUI: "Calculator",
                label: "Calculator",
                description: "Perform arithmetic, trigonometry, boolean, date, and geometry calculations",
                documentationLink: "/agents/tools/calculator",
                icon: "dku-icon-calculator-20",
                testQuery: {
                    "input": {
                        "formula": "sqrt(441) * 2"
                    }
                },
                showStopDevKernelBtn: false,
                displayPriority: 1,
            },
            {
                type: "ApiEndpoint",
                label: "API Endpoint",
                description: "Use any API endpoint deployed with Dataiku",
                documentationLink: "/agents/tools/api-endpoint",
                icon: "dku-icon-api-service-deploy-20",
                testQuery: {},
                showStopDevKernelBtn: false,
                displayPriority: 1,
            },
            {
                type: "VirtualTypeForMCPClients",
                label: "MCP",
                description: "Access and use tools exposed by a remote/local MCP server",
                icon: "dku-icon-mcp-20",
                displayPriority: 50,
            },
            {
                type: "GenericStdioMCPClient",
                typeInUI: "Local MCP",
                label: "Run a local MCP server",
                description: "Use tools from any local MCP server",
                blueBoxDescription: "Configure and run a local MCP server in your environment. Set up the server manually or by pasting a JSON configuration",
                documentationLink: "/agents/tools/local-mcp",
                icon: "dku-icon-mcp-client-local-20",
                codeEnvSelectionSupported: true,
                containerExecutionSupported: true,
                testQuery: {
                    "input": {},
                    "subtoolName": "name of the subtool to use, must be enabled",
                },
                showStopDevKernelBtn: true,
                excludeFromNewAgentToolList: true,
            },
            {
                type: "RemoteMCPClient",
                typeInUI: "Remote MCP",
                label: "Use a remote MCP server",
                description: "Use tools from a remote MCP server",
                blueBoxDescription: "Access tools from a remote MCP server through an existing connection",
                documentationLink: "/agents/tools/remote-mcp",
                icon: "dku-icon-mcp-client-remote-20",
                codeEnvSelectionSupported: false,
                containerExecutionSupported: false,
                testQuery: {
                    "input": {},
                    "subtoolName": "name of the subtool to use, must be enabled",
                },
                showStopDevKernelBtn: false,
                excludeFromNewAgentToolList: true,
            }
        ];
        if ($rootScope.appConfig && $rootScope.appConfig.customAgentTools) {
            AGENT_TOOL_TYPES_UI_DATA.push(...$rootScope.appConfig.customAgentTools
                .map(agentTool => ({
                    type: CUSTOM_PREFIX + agentTool.desc.id,
                    label: agentTool.desc.meta?.label ?? "Custom agent tool " + agentTool.desc.id,
                    description: agentTool.desc.meta?.description ?? "",
                    documentationLink: "/agents/tools/index",
                    icon: agentTool.desc.meta?.icon ?? "dku-icon-puzzle-piece-20",
                    containerExecutionSupported: true,
                    showStopDevKernelBtn: true,
                    displayPriority: agentTool.desc.meta?.displayOrderRank ?? 0,
                })));
        }

        this.listAgentToolTypesUiData = function() {
            return AGENT_TOOL_TYPES_UI_DATA
                .filter(toolTypeUIData => !toolTypeUIData.excludeFromNewAgentToolList)
                .filter(toolTypeUIData => {
                    const customAgentToolDesc = this.getCustomAgentToolDesc(toolTypeUIData.type);
                    return !customAgentToolDesc || pluginVisibilityFilter(customAgentToolDesc)
                })
                .sort((a, b) => b.displayPriority - a.displayPriority);
        };

        this.getAgentToolTypesUiDataForType = function(agentToolType) {
            return AGENT_TOOL_TYPES_UI_DATA.find(toolTypeUiData => agentToolType === toolTypeUiData.type);
        };

        this.getAgentToolTypesUiData = function(agentTool) {
            return this.getAgentToolTypesUiDataForType(agentTool.type);
        };

        this.checkShowStopDevKernelBtn = function (agentToolType) {
            const toolUiData = AGENT_TOOL_TYPES_UI_DATA.find((item) => item.type === agentToolType);
            return toolUiData ? toolUiData.showStopDevKernelBtn : false;
        };

        this.getAgentToolQuickTestQueryForType = function(agentToolType) {
            const agentToolTypeUiData =  AGENT_TOOL_TYPES_UI_DATA.find(toolTypeUiData => agentToolType === toolTypeUiData.type);

            let testQuery = {"input": {}, "context": {}};
            if (agentToolTypeUiData) {
                Object.assign(testQuery, agentToolTypeUiData.testQuery);
            }

            return this.getAgentToolQuickTestQueryAsString(testQuery);
        };

        this.getAgentToolQuickTestQueryAsString = function(testQuery) {
            return JSON.stringify(testQuery, null, 2);
        };

        this.getCustomAgentToolDesc = function (agentToolType) {
            let customAgentToolDesc = null;
            if (agentToolType && agentToolType.startsWith(CUSTOM_PREFIX)) {
                customAgentToolDesc = $rootScope.appConfig.customAgentTools.findLast((c) => c.agentType === agentToolType.substring(CUSTOM_PREFIX.length));
            }
            return customAgentToolDesc;
        };

        this.getCustomAgentPluginId = function (agentToolType) {
            let pluginId = null;
            let customAgentToolDesc = this.getCustomAgentToolDesc(agentToolType);
            if (customAgentToolDesc) {
                const pluginDesc = PluginsService.getOwnerPluginDesc(customAgentToolDesc);
                if (pluginDesc && pluginDesc.id) {
                    pluginId = pluginDesc.id;
                }
            }
            return pluginId;
        };
    });

    app.service('LLMUtilsService', function() {
        // keep aligned with LLMStructuredRef.decodeId
        this.mapIdToType = (id) => {
            if (id.startsWith('agent:')) {
                return 'SAVED_MODEL_AGENT';
            } else if (id.startsWith('savedmodel:finetuned_openai:')) {
                return 'SAVED_MODEL_FINETUNED_OPENAI';
            } else if (id.startsWith('savedmodel:finetuned_azureopenai:')) {
                return 'SAVED_MODEL_FINETUNED_AZURE_OPENAI';
            } else if (id.startsWith('savedmodel:finetuned_huggingfacelocal:')) {
                return 'SAVED_MODEL_FINETUNED_HUGGINGFACE_TRANSFORMER';
            } else if (id.startsWith('savedmodel:finetuned_bedrock:')) {
                return 'SAVED_MODEL_FINETUNED_BEDROCK';
            } else if (id.startsWith('openai:')) {
                return 'OPENAI'
            } else if (id.startsWith('azureopenai-model:')) {
                return 'AZURE_OPENAI_MODEL';
            } else if (id.startsWith('azureopenai:')) {
                return 'AZURE_OPENAI_DEPLOYMENT';
            } else if (id.startsWith('cohere')) {
                return 'COHERE';
            } else if (id.startsWith('mistralai:')) {
                return 'MISTRALAI';
            } else if (id.startsWith('anthropic')) {
                return 'ANTHROPIC';
            } else if (id.startsWith('vertex:')) {
                return 'VERTEX';
            } else if (id.startsWith('bedrock')) {
                return 'BEDROCK';
            } else if (id.startsWith('mosaicml')) {
                return 'MOSAICML';
            } else if (id.startsWith('stabilityai')) {
                return 'STABILITYAI';
            } else if (id.startsWith('nvidia-nim')) {
                return 'NVIDIA_NIM';
            } else if (id.startsWith('huggingfaceapi:')) {
                return 'HUGGINGFACE_API';
            } else if (id.startsWith('huggingfacelocal:')) {
                return 'HUGGINGFACE_TRANSFORMER_LOCAL';
            } else if (id.startsWith('databricksllm:')) {
                return 'DATABRICKS';
            } else if (id.startsWith('snowflakecortex:')) {
                return 'SNOWFLAKE_CORTEX';
            } else if (id.startsWith('retrievalaugmented:')) {
                return 'RETRIEVAL_AUGMENTED';
            } else if (id.startsWith('retrieval-augmented-llm:')) {
                return 'RETRIEVAL_AUGMENTED';
            } else if (id.startsWith('custom:')) {
                return 'CUSTOM';
            } else if (id.startsWith('sagemaker-generic:')) {
                return 'SAGEMAKER_GENERICLLM';
            } else if (id.startsWith('azure-llm:')) {
                return 'AZURE_LLM';
            }

            return '';
        };
    })

    app.service('AgentToolEnrichmentService', function(DataikuAPI, TypeMappingService, DatasetUtils) {
        this.enrichAgentToolObject = function(objectData, scope) {
            //$scope.object = objectData.agentTool;
            objectData.summary = {
                usedByAgents: [],
                usesKnowledgeBank: undefined, // Knowledge Bank Search
                usesSavedModel: undefined, // Model Predict
                usesDataset: undefined, // Dataset Append and Dataset Lookup
            };

            // find agents that use this tool
            DataikuAPI.agentTools.getUsage(objectData.projectKey, objectData.id).success(data => {
                objectData.summary.usedByAgents = data.agents.map(agent => {
                    agent.computedIcon = TypeMappingService.mapSavedModelSubtypeToIcon(agent.type, agent.backendType, agent.predictionType, agent.savedModelType, null, 20);
                    return agent;
                });
            }).error(setErrorInScope.bind(scope));

            // load knowledge bank if any
            if (objectData.type === 'VectorStoreSearch' && objectData.params.knowledgeBankRef) {
                DataikuAPI.retrievableknowledge.get(objectData.projectKey, objectData.params.knowledgeBankRef).success(knowledgeBank => {
                    objectData.summary.usesKnowledgeBank = knowledgeBank;
                }).error(setErrorInScope.bind(scope));
            }

            // load saved model if any
            if (objectData.type === 'ClassicalPredictionModelPredict' && objectData.params.smRef) {
                DataikuAPI.savedmodels.get(objectData.projectKey, objectData.params.smRef).success(savedModel => {
                    objectData.summary.usesSavedModel = savedModel;
                    objectData.summary.usesSavedModel.computedIcon = TypeMappingService.mapSavedModelSubtypeToIcon(savedModel.type, savedModel.backendType, savedModel.predictionType, savedModel.savedModelType, null, 20);
                }).error(setErrorInScope.bind(scope));
            }

            // load dataset if any
            if ((objectData.type === 'DatasetRowAppend' || objectData.type === 'DatasetRowLookup') && objectData.params.datasetRef) {
                const datasetLoc = DatasetUtils.getLocFromSmart(objectData.projectKey, objectData.params.datasetRef);
                DataikuAPI.datasets.get(datasetLoc.projectKey, datasetLoc.name, objectData.projectKey).success(dataset => {
                    objectData.summary.usesDataset = dataset;
                }).error(setErrorInScope.bind(scope));
            }
        }
    });

    app.service('TypeMappingService', function(PluginsService, RecipeDescService, TYPE_MAPPING, LoggerProvider, DashboardUtils, WebAppsService, MLUtilsService, AgentToolService, LLMUtilsService) {

        this.mapRecipeTypeToName = RecipeDescService.getRecipeTypeName;

        /**
         * @param {string} type
         * @returns {string}
         */
        this.mapDatasetTypeToName = (type) => typeProcessor(type, TYPE_MAPPING.DATASET_TYPES, TYPE_MAPPING.CONVERSION_FIELD_NAME, LoggerProvider.getLogger('DisplayFilters'),
        defaultValueForDataset);

        /**
         * @param {string} type
         * @returns {string}
         */
        this.mapInsightTypeToIcon = (type, size) => this.iconProcessor(type, TYPE_MAPPING.DASHBOARDS_OR_INSIGHTS_TYPES, undefined, defaultValueForInsight(type), size);


        /**
         * @param {string} type
         * @param {number} size - Only used for new icons only, like data collections.
         * @returns {string}
         */
        this.mapTypeToIcon = (type, size) => this.iconProcessor(type, TYPE_MAPPING.ALL_TYPES, undefined, defaultValueForDataset, size);

        /**
         * @param {string} type
         * @returns {string}
         */
        this.mapOldIconTypeToModernIcon = (oldIcon, size = 16) => {
            // sometimes the input is not defined, make sure we don't make an error in that case
            if(oldIcon === undefined || oldIcon === null || oldIcon === '') {
                return oldIcon;
            }

            // Note: oldIcon may contain multiple classes
            const iconClasses = oldIcon.split(' ');

            // Find the first class that is a modern icon and add a size if missing
            const index = iconClasses.findIndex(iconClass => iconClass.startsWith('dku-icon'));
            if (index > -1) {
                const modernIcon = iconClasses[index];
                // check if icon is missing a size suffix
                if (!/(12|16|20|24|32|48)$/.test(modernIcon)) {
                    iconClasses[index] = modernIcon + '-' + size;
                }
            } else {
                // Check each icon for a modern equivalent and convert the first valid one -- otherwise just return oldIcon
                const mappedIconClasses = iconClasses.map(iconClass => TYPE_MAPPING.NEW_ICON_TYPE_MAPPING[iconClass]);
                const mappedIndex = mappedIconClasses.findIndex(iconClass => !!iconClass);
                if (mappedIndex === -1) {
                    if (window.devInstance) {
                        LoggerProvider.getLogger('icons-migration').warn(`Old icon "${oldIcon}" mapping to modern icon unsupported."`)
                    }
                } else {
                    iconClasses[mappedIndex] = `${mappedIconClasses[mappedIndex]}-${size}`;
                }
            }

            return iconClasses.join(' ');
        }

        /**
         * @param {string} subtype
         * @param {string} type
         * @param {number} size - Only used for new icons only, like data collections.
         * @returns {string}
         */
        this.mapSubtypeToIcon = (subtype, type, size, suffix) => {
            var subtypeFilters = {
                'INSIGHT': this.mapInsightTypeToIcon,
                'DATASET': this.mapDatasetTypeToIcon,
                'RECIPE': this.mapRecipeTypeToIcon,
                'WEB_APP': this.mapWebappTypeToIcon,
                'STREAMING_ENDPOINT': this.mapDatasetTypeToIcon,
                'SAVED_MODEL': this.mapSavedModelMLCategoryToIcon
            };
            if (subtype && type && subtypeFilters[type.toUpperCase()]) {
                return subtypeFilters[type.toUpperCase()](subtype, size, suffix);
            } else {
                return this.mapTypeToIcon(type, size);
            }
        }

        /**
         * @param {string} type
         * @param (boolean) forDetailView
         * @returns {string}
         */
        this.mapConnectionTypeToName = (type, forDetailView) => {
            const conversionField = forDetailView ? TYPE_MAPPING.CONVERSION_FIELD_NAME : TYPE_MAPPING.CONVERSION_FIELD_OTHER_NAME;
            return typeProcessor(type, TYPE_MAPPING.CONNECTION_TYPES, conversionField, undefined, defaultValue);
        }

        /**
         * @param {string} type
         * @returns {string}
         */
        this.mapConnectionTypeToNameForList = (type) => this.mapConnectionTypeToName(type, false);

        /**
         * @param {string} type
         * @returns {string}
         */
        this.mapConnectionTypeToNameForItem = (type) => this.mapConnectionTypeToName(type, true);

        /**
         * @param {string} type
         * @returns {string}
         */
        this.mapConnectionTypeToIcon = (type, size) => this.iconProcessor(type, TYPE_MAPPING.CONNECTION_TYPES, LoggerProvider.getLogger('connectionTypeToIcon'), defaultValue, size);

        /**
         * @param {string} type
         * @returns {string}
         */
        this.mapCredentialTypeToIcon = (type, size) => this.iconProcessor(type, TYPE_MAPPING.CREDENTIAL_TYPES, LoggerProvider.getLogger('credentialTypeToIcon'), defaultValue, size);

        /**
         * @param {string} type
         * @returns {string}
         */
        this.mapDatasetTypeToIcon = (type, size) => this.iconProcessor(type, TYPE_MAPPING.DATASET_TYPES, LoggerProvider.getLogger('datasetTypeToIcon'), defaultValueForDataset, size);

        /**
         * @param {string} type
         * @returns {string}
         */
        this.mapRecipeTypeToIcon = (type, size) => this.iconProcessor(type, TYPE_MAPPING.RECIPE_TYPES, undefined, defaultValueForRecipe, size);

        /**
         * @param {string} type
         * @returns {string}
         */
        this.mapModelTypeToIcon = (type, size) => this.iconProcessor(type, TYPE_MAPPING.ML_TYPES, undefined, defaultValueForRecipe, size);

        /**
         * @param {string} type
         * @returns {string}
         */
        this.mapRecipeTypeToLanguage = (type) => {
            function defaultToUndefined() {
                return undefined;
            }
            return typeProcessor(type,TYPE_MAPPING. RECIPE_TYPES, TYPE_MAPPING.CONVERSION_FIELD_LANGUAGE, undefined, defaultToUndefined);
        }

        /**
         * @param {string} type
         * @returns {string} one of TYPE_MAPPING.RECIPE_CATEGORIES.CODE or undefined if unknown or custom
         */
        this.mapRecipeTypeToCategory = (type) => {
            const recipeTypeObject = TYPE_MAPPING.RECIPE_TYPES[type];
            return recipeTypeObject ? recipeTypeObject.category : undefined;
        }


        /**
         * @param {string} type
         * @returns {string}
         */
        this.mapRecipeTypeToColorClass = (type) => {
            const customRecipeColors = ['red', 'pink', 'purple', 'blue', 'green', 'sky', 'yellow', 'orange', 'brown', 'gray'];

            if (!type) {
                return 'recipe-custom';
            }
            if (!RecipeDescService.isRecipeType(type) && type !== 'labeling') {
                return;
            }

            const loadedDesc = PluginsService.getRecipeLoadedDesc(type);
            if (loadedDesc) {
                let colorClass = 'recipe-custom';

                if (loadedDesc && loadedDesc.desc && loadedDesc.desc.meta) {
                    const iconColor = loadedDesc.desc.meta.iconColor;

                    if (customRecipeColors.indexOf(iconColor) > -1) {
                        colorClass = colorClass + '-' + iconColor;
                    }
                }
                return colorClass;
            }

            if (type.startsWith('App_')) {
                return 'app';
            }

            return 'recipe-' + typeProcessor(type, TYPE_MAPPING.RECIPE_TYPES, TYPE_MAPPING.CONVERSION_FIELD_CATEGORY, undefined, () => '');
        }

        /**
         * @param {string} type
         * @param {boolean} background
         * @returns {string}
         */
        this.mapTypeToColor = (type, background = false) => {
            const supportedTypes = [
                'project',
                'dataset',
                'streaming_endpoint',
                'recipe',
                'analysis',
                'notebook',
                'scenario',
                'saved_model',
                'model_evaluation_store',
                'genai_evaluation_store',
                'model_comparison',
                'genai_comparison',
                'prompt_studio',
                'agent_review',
                'retrievable_knowledge',
                'agent_tool',
                'managed_folder',
                'web_app',
                'report',
                'dashboard',
                'insight',
                'article',
                'labeling_task',
                'app'];


            if (!type) return '';

            const recipeColor = this.mapRecipeTypeToColor(type);
            if (recipeColor) {
                return recipeColor;
            }

            const stdType = getStandardType(type);
            if (stdType && supportedTypes.indexOf(stdType) >= 0) {
                return background ? 'universe-background ' + stdType : 'universe-color ' + stdType;
            }
            return '';
        }

        this.mapEvaluationStoreToClassColor = (subType) => {
            const color = subType == "LLM" || subType == "AGENT" ? 'genai_evaluation_store' : 'model_evaluation_store';
            return `universe-color ${color}`;
        }

        this.mapComparisonToClassColor = (subType) => {
            const color = subType == "LLM" || subType == "AGENT" ? 'genai_comparison' : 'model_comparison';
            return `universe-color ${color}`;
        }

        /**
         *
         * @param {string} subtype a subtype to be mapped.
         * @param {string} type the "parent" type.
         * @returns  {string}
         */
        this.mapSubtypeToColor = (subtype, type) => {
            const subtypeFilters = {
                'INSIGHT': this.mapInsightTypeToColor,
                'RECIPE': this.mapRecipeTypeToColor,
                'WORKSPACE_STORY': () => 'workspace-object-datasory-icon__color',
                'SAVED_MODEL': (subType) => this.mapSavedModelTypeToClassColor(subType, true),
                'MODEL_EVALUATION_STORE': (subType) => this.mapEvaluationStoreToClassColor(subType),
                'MODEL_COMPARISON': (subType) => this.mapComparisonToClassColor(subType),
              };
            if (subtypeFilters[type.toUpperCase()]) {
                return subtypeFilters[type.toUpperCase()](subtype);
            } else {
                return this.mapTypeToColor(type);
            }
        }

        /**
         *
         * @param {string} taskType
         * @param {string} predictionType
         * @returns {string}
         */
        this.mapMlTaskTypeToIcon = (taskType, predictionType) => {
            if (!taskType || !predictionType) {
                return;
            }
            if (taskType.toLowerCase() == 'clustering') {
                return 'icon-clustering';
            }
            // TODO @deepHub icons for new prediction types? Otherwise, fallback
            return 'icon-prediction-' +predictionType.toLowerCase();
        }

        /**
         *
         * @param {type} taskType
         * @returns {string}
         */
        // TODO @deepHub icons for new backend? Otherwise, fallback
        this.mapBackendTypeToIcon = (type, size) => 'icon-ml ' + this.iconProcessor(type ? type.toLowerCase() : '', TYPE_MAPPING.BACKEND_TYPES, undefined, defaultValue, size);

        /**
         *
         * @param {string} taskType
         * @param {string} backendType
         * @returns {string}
         */
        this.mapAnalysisTypeToIcon = (taskType, backendType, predictionType, size) => {
            const mlCategory = MLUtilsService.getMLCategory(taskType, backendType, predictionType);

            return this.iconProcessor(mlCategory, TYPE_MAPPING.VISUAL_ANALYSIS_TYPES, undefined, defaultValue, size);
        }

        /**
         *
         * @param {string} mlCategory
         * @param {number} size
         * @returns {string}
         */
        this.mapSavedModelMLCategoryToIcon = (mlCategory, size, suffix) => {
            return this.iconProcessor(mlCategory, TYPE_MAPPING.SAVED_MODEL_TYPES, undefined, defaultValue, size, suffix);
        }

        /**
         *
         * @param {string} taskType
         * @param {string} backendType
         * @param {string} predictionType
         * @param {string} savedModelType
         * @param {string} proxyModelProtocol
         * @param {number} size
         * @returns {string}
         */
        this.mapSavedModelSubtypeToIcon = (taskType, backendType, predictionType, savedModelType, proxyModelProtocol, size) => {
            const mlCategory = MLUtilsService.getMLCategory(taskType, backendType, predictionType, savedModelType, proxyModelProtocol);
            return this.iconProcessor(mlCategory, TYPE_MAPPING.SAVED_MODEL_TYPES, undefined, defaultValue, size);
        }

        this.mapSavedModelTypeToClassColor = (savedModelType, withUniverse) => {
            let classColor = withUniverse ? 'universe-color ' : ' ';
            // FINE_TUNING is technically an ML category and not a type (we already treat subtypes as ml categories in mapSubtypeToIcon)
            classColor += ['LLM_GENERIC', 'FINE_TUNING', 'PYTHON_AGENT', 'PLUGIN_AGENT', 'TOOLS_USING_AGENT', 'RETRIEVAL_AUGMENTED_LLM'].includes(savedModelType) ? 'llm-saved-model' : 'saved-model';
            return classColor;
        };

        this.mapSavedModelMLCategoryToColor = (mlCategory, withUniverse) => {
            let classColor = withUniverse ? 'universe-color ' : ' ';
            classColor += ['fine_tuning', 'python_agent', 'plugin_agent', 'tools_using_agent', 'retrieval_augmented_llm'].includes(mlCategory) ? 'llm-saved-model' : 'saved-model';
            return classColor;
        };

        this.mapAgentToolTypeToIcon = (agentToolType, size) => {
            const uiData = AgentToolService.getAgentToolTypesUiDataForType(agentToolType);
            if (uiData === undefined) {
                return "";
            }
            return uiData.icon.replace(/20$/, size);
        };

        this.mapLlmIdToIcon = (llmId, size) => {
            const llmType = LLMUtilsService.mapIdToType(llmId);
            return this.iconProcessor(llmType, TYPE_MAPPING.LLM_TYPES, undefined, defaultValueForLlm, size);
        };

        /**
         *
         * @param {string} type
         * @returns {string}
         */
        this.mapToFsProviderDisplayName = (type) => typeProcessor(type, TYPE_MAPPING.FS_PROVIDER_TYPES, TYPE_MAPPING.CONVERSION_FIELD_NAME, undefined, defaultValue);

        /**
         *
         * @param {string} type
         * @returns {string}
         */
        this.mapToNiceType = (type) => typeProcessor(type, TYPE_MAPPING.DATASET_TYPES, TYPE_MAPPING.CONVERSION_FIELD_NAME, undefined,
            this.mapConnectionTypeToNameForItem.bind(this))

        /**
         *
         * @param {string} type
         * @param {boolean} noBackground
         * @returns {string}
         */
        this.mapInsightTypeToColor = (type, noBackground = true) => {
            let color = '';
            switch (type) {
                case 'text':
                case 'iframe':
                case 'image':
                color = 'comments';
                break;
                default:
                color = (DashboardUtils.getInsightHandler(type) || {}).color;
                break;
            }

            return !noBackground ? 'universe-background insight-icon ' + color : color;
        }

        /**
         *
         * @param {string} type
         * @returns {string}
         */

        this.mapInsightTypeToDisplayableName = (type) => {
            var handler = DashboardUtils.getInsightHandler(type);
            if (!handler || !handler.name) return type;
            return handler.name;
        }

        this.mapWebappTypeToIcon = (type, size) => {
            return this.iconProcessor(type,  TYPE_MAPPING.WEBAPPS_TYPES, undefined, defaultValueForWebApp(type), size);
        }

        this.mapWebappTypeToColor = (type) => {
            if (WebAppsService.getBaseType(type) == type) {
                return 'notebook'; //native webapp => code color
            } else {
                return 'flow';
            }
        }

        this.mapWebappTypeToName = (type) => {
            return typeProcessor(type, TYPE_MAPPING.WEBAPPS_TYPES, TYPE_MAPPING.CONVERSION_FIELD_NAME, undefined,
                defaultValueForWebApp(type));
        }

        /**
         * @param {string} type
         * @returns {string}
         */
        this.mapRecipeTypeToColor = (type) => {
            if (!type) {
                return 'universe-color recipe-custom';
            }
            const colorClass = this.mapRecipeTypeToColorClass(type);
            if (!colorClass) {
                return '';
            }
            return 'universe-color ' + colorClass;
        }

        /**
         * @param {string} objectType
         * @returns {string}
         */
        this.mapChartTypeToColor = (objectType) => {
            if (objectType === 'dataset') {
                return 'dataset';
            } else if (objectType === 'analysis') {
                return 'analysis';
            }
            return 'chart';
        }

        /**
         *
         * @param {string} type
         * @returns {string}
         */
        const getStandardType = (type) => {
            if (!type) return '';
            if (type.endsWith('_NOTEBOOK')) {
                return 'notebook';
            }
            return type.toLowerCase();
        }

        /**
         *
         * @param {string} type
         * @param {cord<string, any>} types
         * @param {string} conversionField
         * @param {Logger} Logger
         * @param {Function} defaultValueFunction
         * @returns {string}
         */
        const typeProcessor = (type, types, conversionField, Logger, defaultValueFunction) => {
            if(!type) {
                return '';
            }
            const existingType = types[type.toLowerCase()];
            const result = existingType && existingType[conversionField];
            if (result !== undefined) {
                return result;
            }
            return defaultValueFunction(type, conversionField, Logger);
        }

        /**
         * Returns a function to be used by a mapXTypeToIcon filter.
         *
         * @param {string} type
         * @param {Record<string, any>} types
         * @param {Logger} Logger
         * @param {Function} defaultValueFunction
         * @param {number} size the size of the produced icon (only used for modern icons)
         * @returns {string}
         */
        this.iconProcessor = (type, types, Logger, defaultValueFunction, size, suffix = '') =>  {
            const newIconOnly = typeProcessor(type, types, 'newIconOnly', Logger, () => false);
            const icon = typeProcessor(type, types, TYPE_MAPPING.CONVERSION_FIELD_ICON, Logger, defaultValueFunction);

            if (newIconOnly) {
                if (!size) {
                    LoggerProvider.getLogger('icons-migration').warn(`New icon mapping used without size for type ${type}."`);
                    size = 16;
                }

                return icon + '-' + (suffix ? suffix + '-' : '') + size;
            }

            return icon;
        }

        /**
         *
         * @param {string} key
         * @param {string} conversionField
         * @param {Logger} Logger
         * @returns {string}
         */
        const defaultValue = (key, conversionField, Logger) => {
            const defaultPrefix = TYPE_MAPPING.CONVERSION_FIELD_ICON === conversionField ? 'icon-' : '';
            const result = defaultPrefix + key.toLowerCase();
            if (Logger !== undefined) {
                Logger.error('Unknown type: ' + key + '.Returning default value: ' + result);
            }
            return result;
        }


        const defaultValueForDataset = (originalKey, conversionField, Logger) => {
            const key = originalKey.toLowerCase();
            if (key.startsWith('custom') || key.startsWith('fsprovider_') || key.startsWith('sample_')) {
                if (TYPE_MAPPING.CONVERSION_FIELD_ICON === conversionField) {
                    return PluginsService.getDatasetIcon(originalKey);
                } else {
                    return PluginsService.getDatasetLabel(originalKey);
                }
            }
            return defaultValue(key, conversionField, Logger);
        }

        const defaultValueForRecipe = (originalKey, conversionField, Logger) => {
            const key = originalKey.toLowerCase();
            if (key.startsWith('custom')) {
                if (TYPE_MAPPING.CONVERSION_FIELD_ICON === conversionField) {
                    return PluginsService.getRecipeIcon(originalKey);
                }
            } else if (key.startsWith('app_')) {
                for (let ar of window.dkuAppConfig.appRecipes) {
                    if (ar.recipeType == originalKey) {
                        return ar.icon;
                    }
                }
            }
            return defaultValue(key, conversionField, Logger);
        }


        const defaultValueForWebApp = (type) => {
            return function(originalKey, conversionField, Logger) {
                if (conversionField === TYPE_MAPPING.CONVERSION_FIELD_ICON) {
                    return WebAppsService.getWebAppIcon(originalKey) || 'icon-code';
                } else if (conversionField === TYPE_MAPPING.CONVERSION_FIELD_NAME) {
                    return WebAppsService.getWebAppTypeName(originalKey) || type;
                } else {
                    return defaultValue(originalKey.toLowerCase(), conversionField, Logger);
                }
            };
        }

        const defaultValueForInsight = (type) => {
            return function() {
                return (DashboardUtils.getInsightHandler(type) || {}).icon;
            }
        }

        const defaultValueForLlm = () => {
            return function() {
                return 'dku-icon-question-circle-outline';
            }
        }
    });

    app.service('PrettyPrintDoubleService', function() {
        // regexp to find string values that are doubles in scientific notation
        const scientificNotationRegexp = /^[+-]?\d*(\.\d*)?[eE][+-]?\d+$/;

        // Keep this regex in sync with java class FrenchDoubleMeaning (note that we
        // removed the ? after the exponential term because we only want to detect
        // scientific notation numbers with an exponential)
        const frenchDoubleScientificNotationRegexp = /^[+-]?([0-9]{1,3} ([0-9]{3} )*)?[0-9]+(([eE][+-]?[0-9]+)$|,[0-9]+([eE][+-]?[0-9]+))$/

        // convert a double as non-scientific notation if exponent value is <= 15. Otherwise, rawValue is unchanged
        function avoidScientificNotation(rawValue) {
            let formatter = null;
            let doubleValue = null;

            if(rawValue && scientificNotationRegexp.test(rawValue)) {
                // Regular Decimal number
                // only apply on values the look like a scientific notation

                // we force en-US as a base in order to have consistent UI for all users
                formatter = new Intl.NumberFormat('en-US', {
                    numberingSystem: 'standard',
                    useGrouping: false,
                    minimumFractionDigits: 1,
                    maximumFractionDigits: 15,
                });

                doubleValue = parseFloat(rawValue); // might be nan for non-number cells
            } else if(rawValue && frenchDoubleScientificNotationRegexp.test(rawValue)) {
                // Decimal (comma), a.k.a FrenchDoubleMeaning
                // only apply on values the look like a scientific notation

                // we force fr-FR as a base in order to have consistent UI for all users
                formatter = new Intl.NumberFormat('fr-FR', {
                    numberingSystem: 'standard',
                    useGrouping: true,
                    style: "decimal",
                    minimumFractionDigits: 1,
                    maximumFractionDigits: 15,
                });

                doubleValue = parseFloat(rawValue.replace(/ /g, "").replace(/,/g,".")); // might be nan for non-number cells
            }

            if(formatter !== null && doubleValue !== null && !isNaN(doubleValue) && Math.abs(doubleValue)>=1e-15 && Math.abs(doubleValue)<1e16) {
                return formatter.format(doubleValue)
            }

            return rawValue;
        }

        // apply avoidScientificNotation to all columns that have a float or double type
        function patchFormulaPreview(previewTable, sampleType, columns) {
            previewTable.colNames.forEach((colName, colIdx) => {
                const colWithType = colIdx === 0 // first column is always output, but is not named
                    ? {type: sampleType}
                    : columns.find(col => col.name === colName);
                const colType = colWithType && colWithType.type;
                if(colType === 'double' || colType === 'float') {
                    previewTable.rows.forEach(row => {
                        row.data[colIdx] = avoidScientificNotation(row.data[colIdx]);
                    });
                }
            })
        }

        return {
            avoidScientificNotation,
            patchFormulaPreview,
        };
    })
    app.service('Deprecation', [function() {
        const deprecation = {};

        deprecation.isPythonDeprecated = (pythonInterpreter) => ["PYTHON27", "PYTHON34", "PYTHON35", "PYTHON36", "PYTHON37", "PYTHON38"].includes(pythonInterpreter);

        return deprecation;

    }]);
    app.service('PandasSupport', [function() {
        const service = {};

        const supportMatrix = {
            "LEGACY_PANDAS023": {
                "name": "Pandas 0.23 (legacy)",
                "interpreters": ["PYTHON27", "PYTHON34", "PYTHON35", "PYTHON36"]
            },
            "PANDAS10": {
                "name": "Pandas 1.0",
                "interpreters": ["PYTHON36"]
            },
            "PANDAS11": {
                "name": "Pandas 1.1",
                "interpreters": ["PYTHON36", "PYTHON37", "PYTHON38", "PYTHON39"]
            },
            "PANDAS12": {
                "name": "Pandas 1.2",
                "interpreters": ["PYTHON37", "PYTHON38", "PYTHON39"]
            },
            "PANDAS13": {
                "name": "Pandas 1.3",
                "interpreters": ["PYTHON37", "PYTHON38", "PYTHON39"]
            },
            "PANDAS14": {
                "name": "Pandas 1.4",
                "interpreters": ["PYTHON38", "PYTHON39", "PYTHON310"]
            },
            "PANDAS15": {
                "name": "Pandas 1.5",
                "interpreters": ["PYTHON38", "PYTHON39", "PYTHON310", "PYTHON311"]
            },
            "PANDAS20": {
                "name": "Pandas 2.0",
                "interpreters": ["PYTHON38", "PYTHON39", "PYTHON310", "PYTHON311"]
            },
            "PANDAS21": {
                "name": "Pandas 2.1",
                "interpreters": ["PYTHON39", "PYTHON310", "PYTHON311"]
            },
            "PANDAS22": {
                "name": "Pandas 2.2",
                "interpreters": ["PYTHON39", "PYTHON310", "PYTHON311", "PYTHON312", "PYTHON313"]
            },
            "PANDAS23": {
                "name": "Pandas 2.3",
                "interpreters": ["PYTHON39", "PYTHON310", "PYTHON311", "PYTHON312", "PYTHON313", "PYTHON314"]
            }
        };

        // returns the list of supported core package versions for a given interpreter
        // example: [ ["PANDAS22", "Pandas 2.2"], ["PANDAS23", "Pandas 2.3"] ]
        // also adds the currently selected core package version even if unsupported
        service.listSupportedVersions = (pythonInterpreter, selectedCorePackageVersion) => {
            let corePackageVersions = {};

            for (const [id, config] of Object.entries(supportMatrix)) {
                if (config.interpreters.includes(pythonInterpreter) || pythonInterpreter === "CUSTOM") {
                    corePackageVersions[id] = config.name;
                }
            }

            // add ghost item when needed
            if (!(selectedCorePackageVersion in corePackageVersions)) {
                if (selectedCorePackageVersion === "DEFAULT") {
                    corePackageVersions[selectedCorePackageVersion] = "Default"
                } else {
                    const name = supportMatrix[selectedCorePackageVersion]?.name ?? selectedCorePackageVersion;
                    corePackageVersions[selectedCorePackageVersion] = name + " (Unsupported)"
                }
            }

            return Object.entries(corePackageVersions);
        }

        return service;
    }]);
    app.service('PathUtils', [function() {
        const pathUtils = {};

        //set trailing slash
        pathUtils.makeT = (path) =>
            path && path.substr(-1) === '/' ? path : (path || '') + '/';

        //set no trailing slash
        pathUtils.makeNT = (path) =>
            (path || '').replace(/\/+$/, '');

        //set leading slash
        pathUtils.makeL = (path) =>
            path && path[0] === '/' ? path : '/' + (path || '');

        //set no leading slash
        pathUtils.makeNL = (path) =>
            (path || '').replace(/^\/+/, '');

        //set leading & trailing slash
        pathUtils.makeLT = (path) =>
            pathUtils.makeT(pathUtils.makeL(path));

        //set no leading & no trailing slash
        pathUtils.makeNLNT = (path) =>
            pathUtils.makeNT(pathUtils.makeNL(path));

        //concat path segments with no leading slash & no trailing slash
        pathUtils.concatNLNT =  (...paths) =>
           paths
               .filter(p => p)                        //remove undefined/null/empty entries
               .map(p => p.replace(/^\/+|\/+$/g, '')) //remove leading & ending slashes
               .filter(p => p.length>0)               //remove empty entries
               .join('/');

        //concat path segments with leading slash & no trailing slash
        pathUtils.concatLNT =  (...paths) =>
           '/' + pathUtils.concatNLNT(...paths);

        pathUtils.absPathFromUserPath = (usrPath, currentAbsPath) => {
            const path = (usrPath || '').trim();
            if (!path) return '/'; //discutable (legacy?) behavior, '' means '/'
            if (path[0] === '/') return pathUtils.makeLT(path);
            return pathUtils.concatLNT(currentAbsPath, path);
        };

        return pathUtils;
    }]);

app.service('ClipboardReadWriteService', function(ClipboardUtils, CreateModalFromTemplate) {
    const writeItemsToClipboard = function(items, separator = "\n") {
        let toCopy = "";
        if (items && items.length) {
            toCopy = items.join(separator);
        }
        ClipboardUtils.copyToClipboard(toCopy);
    }

    const readItemsFromNavigatorClipboard = async function() {
        if (!navigator.clipboard || !navigator.clipboard.readText) {
            throw new Error('Browser does not support the readText API');
        }

        try {
            const data = await navigator.clipboard.readText();
            if (!data) return;

            let items = data.trim().split(/[\r\n]+/);
            if (items && Array.isArray(items) && items.length > 0) {
                if (items.length === 1) {
                    items = items[0].split(",").map(e => e.trim());
                }
                return items;
            }
        } catch (error) {
            throw new Error('Failed to read from clipboard');
        }
    };

    const readItemsUsingPopup = async function(scope, popupHeader) {
        return new Promise((resolve, reject) => {
            let newScope = scope.$new();
            CreateModalFromTemplate("/templates/list-copy-paste-modal.html", newScope, 'PasteModalController', function(modalScope) {
                modalScope.header = popupHeader;

                // Override the onPasteText defined in the PasteModalController
                modalScope.onPasteText = function(event) {
                    let data = event.originalEvent.clipboardData.getData('text/plain');
                    if (!data) {
                        modalScope.uiState.hasError = true;
                        return;
                    }
                    try {
                        let stringArray = data.trim().split(/[\r\n]+/);
                        if (!stringArray || !Array.isArray(stringArray) || stringArray.length === 0) {
                            modalScope.uiState.hasError = true;
                            reject(new Error('Not a valid array'));
                            return;
                        }

                        if (stringArray.length === 1) {
                            stringArray = stringArray[0].split(",").map(e => e.trim());
                        }

                        let classes = stringArray.map((class_label, index) => ({ class_label, index }));
                        if (!modalScope.validateData(classes)) {
                            modalScope.uiState.hasError = true;
                            reject(new Error('Invalid data'));
                            return;
                        }

                        modalScope.uiState.editMode = false;
                        modalScope.uiState.hasError = false;
                        modalScope.uiState.items = classes;
                    } catch (error) {
                        modalScope.uiState.hasError = true;
                        reject(error);
                    }
                };

                modalScope.pasteItems = function() {
                    resolve(modalScope.uiState.items.map(item => item.class_label));
                };

                modalScope.validateData = (classes) => {
                    try {
                        return classes.every(aClass => aClass.class_label.length > 0);
                    } catch (e) {
                        return false;
                    }
                }
            });
        });
    }

    const readItemsFromClipboard = async function(scope, popupHeader) {
        try {
            return await readItemsFromNavigatorClipboard();
        } catch (_) {
            return await readItemsUsingPopup(scope, popupHeader)
        }
    }

    return {
        writeItemsToClipboard: writeItemsToClipboard,
        readItemsFromClipboard: readItemsFromClipboard
    }

});


app.service("FilePatternUtils", function(){

    //if there is a char like this in the glob string we need to replace - Does not include * or ? which mean something in GLOB
    const REGEXP_CHARS_PATTERN = /[/\-\\^$+.()|[\]{}]/g;

    //Replace with escaped match
    const REGEXP_CHARS_REPLACEMENT = '\\$&';

    // basically matches /**/ or **/ at the beginning but final slash not included in the capture and clever things so it is not inefficient
    const DEEP_WILDCARD_PATTERN = /(?<=^|\/)\*\*(?=$|\/)/g;

    const srv = {
        /**
         * Convert a glob string to a regexp to use on file paths.
         * It is not a global regexp, it is not stateful, and either matches the whole path string or not.
         * E.g. use `.test(path)` to see if this matches given path.
         *
         * Follows implemention in com.dataiku.dip.fs.FileSelectionRule.java
         *
         * @param {string} expr GLOB string
         * @returns {RegExp}
         * @throws exception if expr could not be compiled as a regexp
         */
        fullPathGlobRegExp: function(expr) {
            expr = expr.trim();
            DEEP_WILDCARD_PATTERN.lastIndex = 0;
            let specs = expr.replaceAll(/\/+/g, "/").split(DEEP_WILDCARD_PATTERN);
            let globRegExpString = "^";

            if (expr === "**" || expr === "**/*") {
                // Simplest to hard code these to match anything
                globRegExpString += ".*";
            } else {
                if (expr.startsWith("**/")) {
                    globRegExpString += "(?:^|.*?/)"; // can be any prefix path, including empty
                    specs = specs.slice(1);
                    if (specs.length > 0) {
                        specs[0] = specs[0].substring(1); // eat up the leading / from the next spec, / included above
                    }
                }

                for (let i = 0; i < specs.length; i++) {
                    let spec = specs[i];
                    if (i > 0) {
                        globRegExpString += ".*?(?<=\\/)";  // `**` is any char sequence denoting intermediate directories, including empty
                        spec = spec.substring(1); // eat up the leading / from the upcoming spec, / mandatory above
                    }

                    REGEXP_CHARS_PATTERN.lastIndex = 0;
                    globRegExpString += spec.replaceAll(REGEXP_CHARS_PATTERN, REGEXP_CHARS_REPLACEMENT)
                                        .replaceAll("?", "[^/]")     // `?` is any char but /
                                        .replaceAll("*", "[^/]*");  // `*` is any char sequence (including empty) without /   (NOTE: removed  ? from end of replacement that is in the backend)
                }

                if (expr.endsWith("/**")) {
                    // note this is .++ in backend, the extra + for possessive, but this is not possible in JS
                    globRegExpString += ".+";  // `**` is any char sequence
                }
            }

            globRegExpString += "$";

            // g flag not needed (and we'd would have to reset the lastIndex every time)
            return new RegExp(globRegExpString, "i");
        },

        /**
         * Convert a glob string to a regexp to use on file names (not whole paths just the name)
         * It is not a global regexp, it is not stateful, and either matches the whole name string or not.
         * E.g. use `.test(filename)` to see if this matches given name.
         *
         * Follows implemention in com.dataiku.dip.fs.FileSelectionRule.java - globChunkMatches()
         *
         * @param {string} expr GLOB string for a file name (not a path)
         * @returns {RegExp}
         * @throws exception if expr could not be compiled as a regexp
         */
        fileNameGlobRegExp: function(expr) {
            REGEXP_CHARS_PATTERN.lastIndex = 0;
            let globRegexString = "^"+ expr.replaceAll(REGEXP_CHARS_PATTERN, '\\$&')
                               .replaceAll("?", ".")    // `?` is any char
                               .replaceAll("*", ".*?")  // `*` is any char sequence (including empty)
                            + "$";

            // g flag not needed (and we'd would have to reset the lastIndex every time)
            return new RegExp(globRegexString, "i");
        },

        // Removes leading and trailing `/` slashes and collapses multiple to one where they are retained
        // Returns empty string if nothing left after this
        // Equivalent to PathUtils.slashes(path, false, false, true, "");
        normalizeSlashes: function (path) {
            const trimmedPath =  path.trim();
            if (!trimmedPath) return null;
            const collapsedPath  = path.trim().replaceAll(/\/+/g, "/");
            const slashAtStart = collapsedPath.startsWith("/");
            const slashAtEnd = collapsedPath.endsWith("/");

            if (slashAtStart) {
                return slashAtEnd ? collapsedPath.slice(1, -1) : collapsedPath.slice(1);
            } else {
                return slashAtEnd ?  collapsedPath.slice(0, -1) : collapsedPath;
            }
        },

        /**
         * @param {string} path - file path ending in file name -  must be normalised with normalizeSlashes
         * @returns file name at the end of the path with no slashes
         */
        extractFileNameFromPath: function (path) {
            if (!path) return null;
            return path.substring(path.lastIndexOf('/') + 1);
        }
    };
    return srv;
});

app.service('DataikuCloudService', function($q, DataikuAPI, $rootScope) {
    this.isDataikuCloud = function() {
        return $rootScope.appConfig.deploymentMode === 'CLOUD';
    };

    this.getLaunchpadUrl = function() {
        if (!this.isDataikuCloud()) {
            return;
        }

        const saasFrontendHook = window.dkuSaas || {}

        const host = saasFrontendHook.API_HOST || ''; // format is usually api-XXXXXXXX-YYYYYYY, though "api" can change
        const hostMatch = host.match(/[a-z-]*-.{8}-(.{8})/);
        const spaceId = hostMatch ?  (hostMatch.length > 1 ? hostMatch[1] : '') : '';

        return `${saasFrontendHook.URL_CONSOLE}/spaces/${spaceId}`;
    }

    this.getCloudInfo = function() {
        const info = {
            isDataikuCloud: this.isDataikuCloud(),
            isSpaceAdmin: $rootScope.appConfig.admin
        };
        const deferred = $q.defer();
        deferred.resolve(info);
        return deferred.promise;
    }
});

app.service('PuppeteerLoadedService', function (Debounce) {
    this.getDebouncedSetField = function($scope, $element, loadedStateField) {
        const setPuppeteerField = function() {
            const thisLoadedStateField = loadedStateField ? loadedStateField : "puppeteerHook_elementContentLoaded";

            $scope[thisLoadedStateField] = true;
            // in cases where we set a specific field, we set it also as an attribute on the element to be used by the puppeteer code css selector
            if (loadedStateField) {
                $element.attr(loadedStateField, true);
            }
        };

        return Debounce().withDelay(50,200).wrap(setPuppeteerField);
    };
});

//this is a singleton to share the rating feedback parameters for all AI features
app.factory("RatingFeedbackParams", () => {
    const state = {
        requestIdForFeedback: null,
        featureRated: null,
        showRatingFeedback: false,
    };

    return state;
});

app.factory("AvailableLanguages", ($rootScope) => {
    let languageList = [{id: 'en', label: 'English'}, {id: 'ja', label: '日本語'}, { id: 'fr', label: 'Français' }]
    languageList.sort((a, b) => a.label.localeCompare(b.label));
    return languageList;
});

app.service('ProfileService', function ($rootScope) {
    this.isTechnicalAccount = function () {
        return $rootScope?.appConfig?.userProfile?.profile === 'TECHNICAL_ACCOUNT';
    }
});

app.factory('AccessibleObjectsCacheService', function(DataikuAPI, $q) {
    const TYPE_WITHOUT_PROJECT_KEY = ['PROJECT', 'WORKSPACE', 'DATA_COLLECTION', 'APP'];
    const EMPTY_ARRAY = [];
    const EMPTY_CACHE_ENTRY = { value: EMPTY_ARRAY, promise: $q.when(EMPTY_ARRAY) }; // avoid having subsequent calls with the same param return a different ref

    const svc = {
        /**
         * Get a cached lazy accessible object getter. API query are done automatically on request, but shared between many calls with the same parameters.
         * Designed to share API call data between many object-picker instances or similar in order to avoid doing the same API queries multiple times
         *
         * Usage example to select many projects without doing one project list call per selector:
         * $scope.getAccessibleObjects = AccessibleObjectsCacheService.createCachedGetter('READ', setErrorInScope.bind($scope)).asValue;
         * (...)
         *  <div ng-for="item of manySelectors"
         *     type="PROJECT" object-picker="item.projectKey"
         *     input-available-objects="getAccessibleObjects('PROJECT')"
         *  ></div>
         *
         * @param {string} mode 'READ' or 'WRITE' to filter on readable or writeable objects (default 'READ')
         * @param {function} setErrorInScopeFunction If provided, used in API call catch statement to set API call error in the scope
         * @returns {object} Object containing two functions (keys `asValue` or `asPromise`)
         *
         * The asValue function:
         *      @param {string} type Type of the accessible objects to list (e.g. 'PROJECT', 'WORKSPACE', 'DATASET', 'WEB_APP', ...)
         *      @param {string} projectKey (optional for types that are not project objects)
         *      @returns {AccessibleObject[]} empty array while API query is pending, then the list of object once API call is finished (do not snapshot the returned value!)
         *
         * The asPromise function:
         *      @param {string} type Type of the accessible objects to list (e.g. 'PROJECT', 'WORKSPACE', 'DATASET', 'WEB_APP', ...)
         *      @param {string} projectKey (optional for types that are not project objects)
         *      @returns {Promise<AccessibleObject[]>} Promise of the list of objects (do not snapshot the returned value!)

         */
        createCachedGetter(mode = "READ", setErrorInScopeFunction = () => {}) {
            const cache = {};
            return {
                asValue: (type, projectKey) => getAccessibleObjects(type, projectKey, mode, cache, setErrorInScopeFunction).value,
                asPromise: (type, projectKey) => getAccessibleObjects(type, projectKey, mode, cache, setErrorInScopeFunction).promise,
            };
        },
    }

    return svc;

    //////////////

    function getAccessibleObjects(type, projectKey, mode, accessibleObjectsCache, setErrorInScopeFunction) {
        if (!type) {
            return EMPTY_CACHE_ENTRY;
        }

        if(TYPE_WITHOUT_PROJECT_KEY.includes(type)) {
            if (angular.isDefined(accessibleObjectsCache[type])) {
                return accessibleObjectsCache[type];
            }

            const listingPromise = DataikuAPI.taggableObjects.listAccessibleObjects(undefined, type, mode).then(function({data}) {
                accessibleObjectsCache[type].value = data;
                return data;
            }).catch((err) => {
                setErrorInScopeFunction(err);
                return EMPTY_ARRAY;
            });
            accessibleObjectsCache[type] = { value: EMPTY_ARRAY, promise: listingPromise };

            return accessibleObjectsCache[type];
        } else {
            if(!projectKey) {
                return EMPTY_CACHE_ENTRY;
            }
            if(!angular.isDefined(accessibleObjectsCache[type])) {
                accessibleObjectsCache[type] = {}
            }
            if(angular.isDefined(accessibleObjectsCache[type][projectKey])) {
                return accessibleObjectsCache[type][projectKey];
            }

            const listingPromise = DataikuAPI.taggableObjects.listAccessibleObjects(projectKey, type, mode).then(function({data}) {
                accessibleObjectsCache[type][projectKey].value = data;
                return data;
            }).catch((err) => {
                setErrorInScopeFunction(err);
                return EMPTY_ARRAY;
            });
            accessibleObjectsCache[type][projectKey] = { value: EMPTY_ARRAY, promise: listingPromise };

            return accessibleObjectsCache[type][projectKey]
        }
    }
});

app.service('SemanticVersionService', function () {
    const versionPartPattern = /^(\d+)(.*)$/;

    this.compareVersions = function(version1, version2) {
        const v1Parts = version1.split(".");
        const v2Parts = version2.split(".");
        const maxLength = Math.max(v1Parts.length, v2Parts.length);

        for (let i = 0; i < maxLength; i++) {
            const p1 = v1Parts[i] ?? "";
            const p2 = v2Parts[i] ?? "";
            const c = compareVersionPart(p1, p2);
            if (c !== 0) {
                return c;
            }
        }
        return 0;
    }

    function compareVersionPart(part1, part2) {
        const match1 = part1.match(versionPartPattern);
        const match2 = part2.match(versionPartPattern);

        // If both start with a number, numeric comparison first
        if (match1 && match2) {
            const num1 = parseInt(match1[1], 10);
            const num2 = parseInt(match2[1], 10);
            if (num1 !== num2) {
                return num1 > num2 ? 1 : -1;
            }
            // Same number, compare suffix
            const suffix1 = match1[2];
            const suffix2 = match2[2];

            if (suffix1 === suffix2) {
                return 0;
            }

            // Pure numerical version (released) is greater than version with suffix (pre-release)
            if (!suffix1 || !suffix2) {
                return !suffix1 ? 1 : -1;
            }

            // Both suffixes, compare lexicographically
            return suffix1.localeCompare(suffix2);
        }

        // If only one starts with a number, numeric one is greater
        return match1 ? 1 : match2 ? -1 : part1.localeCompare(part2);
    }
});

app.service('ProjectStandardsService', function() {
    const STATUS = {
        RUN_SUCCESS: 'RUN_SUCCESS',
        RUN_ERROR: 'RUN_ERROR',
    };

    function mapSeverity(result) {
        switch (result.status) {
            case STATUS.RUN_SUCCESS:
                return result.severity;
            default:
                return 0;
        }
    }

    function getProjectStandardsSummaryFromReport(projectStandardsRunReport) {
        const bundleChecksRunInfo = Object.values(projectStandardsRunReport.bundleChecksRunInfo);
        const worstSeverityNumber = Math.max(
                ...bundleChecksRunInfo.map(runInfo =>
                    mapSeverity(runInfo.result)
                ),
                0
            );
        return {
            worstSeverityNumber: worstSeverityNumber,
            checksInError: bundleChecksRunInfo.filter(c => c.result.status === STATUS.RUN_ERROR).length,
            checkCount: bundleChecksRunInfo.length,
        }
    }

    function getProjectStandardsSummaryFromRunReportSummary(projectStandardsRunReportSummary) {
        const checkCount = Object.values(projectStandardsRunReportSummary.nbOfChecksBySeverity).reduce((acc, val) => acc + val, 0)
                             + projectStandardsRunReportSummary.nbOfChecksNotApplicable
                             + projectStandardsRunReportSummary.nbOfChecksInError;
        const severitiesWithAtLeastOneCheck = Object.entries(projectStandardsRunReportSummary.nbOfChecksBySeverity)
                                      .filter(([key, value]) => value > 0)
                                      .map(([key, value]) => key);
        return {
            worstSeverityNumber:  Math.max(...severitiesWithAtLeastOneCheck, 0),
            checksInError: projectStandardsRunReportSummary.nbOfChecksInError,
            checkCount: checkCount,
        }
    }

    return {
        getProjectStandardsSummaryFromReport: getProjectStandardsSummaryFromReport,
        getProjectStandardsSummaryFromRunReportSummary: getProjectStandardsSummaryFromRunReportSummary,
        STATUS: STATUS
    }
});

/**
 * Utility for retrieving permission information (e.g. list of users, groups) required for populating permission
 * form and caching it, such that multiple consumers can leverage it without having to fetch the information
 * multiple times from the backend.
 *
 * This can be useful when you need to build multiple components that need this information on the same page, e.g.
 * in plugin presets.
 *
 * Can retrieve:
 *  * `getUsersAndGroups` which returns {allUsers: ..., allUsersLogin: ..., allGroups: ...}
 *  * `canAdminPlugin` which returns a boolean
 **/
app.factory('PermissionsDataFetcher', function(DataikuAPI, $q) {

    function _cachedDataFetcher(promiseFunction, containingScope) {
        let cache = {};
        let runningPromise = {};

        return function(arg = null) {
            const cacheKey = arg ? arg : "__DKU_CACHE_KEY";
            if (cache[cacheKey]) {
                return $q.when(cache[cacheKey].value);
            }

            // In case another consumer already triggered the promise but result is not there yet, and then not cached,
            // reusing the same promise rather than re-triggering the promise.
            if (runningPromise[cacheKey]) {
                return runningPromise[cacheKey].promise;
            }
            const deferred = $q.defer();
            runningPromise[cacheKey] = deferred;

            const promiseCall = arg ? promiseFunction(arg) : promiseFunction();
            promiseCall
                .then(function(result) {
                    // Wrapping result in a simple object to make sure it resolves to true when checking it,
                    // and not bother, even when it happens that `result` is the `false` boolean.
                    cache[cacheKey] = {value: result};
                    deferred.resolve(result);
                })
                .catch(setErrorInScope.bind(containingScope))
                .finally(function() {
                    runningPromise[cacheKey] = null;
                });

            return deferred.promise;
        }
    }

    return function(containingScope) {
        const usersAndGroupsPromise = () => {
            return $q.all([DataikuAPI.security.listUsers(),
                           DataikuAPI.security.listGroups(false)])
                .then(function(results) {
                    let users = results[0].data;
                    let groups = results[1].data;
                    let sortedUsers = users.sort((a, b) => a.displayName.localeCompare(b.displayName));
                    return {
                        allUsers: sortedUsers,
                        allUsersLogin: sortedUsers.map(user => '@' + user.login),
                        allGroups: groups
                    }
                });
        };
        const canAdminPluginPromise = (pluginId) => {
            return DataikuAPI.plugins.canAdminPlugin(pluginId).then(function(result) {
                return result.data;
            })
        };
        const usersAndGroupsFetcher = _cachedDataFetcher(usersAndGroupsPromise, containingScope);
        const canAdminPluginDataFetcher = _cachedDataFetcher(canAdminPluginPromise, containingScope);

        return {
            getUsersAndGroups: usersAndGroupsFetcher,
            canAdminPlugin: canAdminPluginDataFetcher
        }
    }
});

app.service('ConnectionsService', function() {
    this.getCredentialsError = function(errorMessage) {
        if (!errorMessage) {
            return null;
        }

        const regex = /User '(.*?)' does not have credentials for connection '(.*?)' to access (.*)/;
        const match = errorMessage.match(regex);
        if (match) {
            const username = match[1];
            const connectionName = match[2];
            const connectionType = match[3];
            if (connectionType === "RemoteMCP") {
                return {
                    payload: {
                        connectionName: connectionName,
                        user: username,
                        sameUser: true,
                    },
                    fixability: "USER_FILL_CREDENTIALS",
                    code: "ERR_CONNECTION_NO_CREDENTIALS",
                }
            }
        }
        return null;
    };

    this.searchConnectionsWithDescriptions = function(term, connection) { // connection can be either an instance of ConnectionTypeAndName, ConnectionUsability or ConnectionOption as defined in java, or ManagedDatasetConnection as defined in fetchManagedDatasetConnections() in controller_fragments.js
        // ConnectionTypeAndName or ConnectionUsability
        if (connection.type?.toLowerCase().includes(term.toLowerCase())) {
            return true;
        }
        if (connection.name?.toLowerCase().includes(term.toLowerCase())) {
            return true;
        }
        if (connection.description?.toLowerCase().includes(term.toLowerCase())) {
            return true;
        }

        // ConnectionOption
        if (connection.connectionType?.toLowerCase().includes(term.toLowerCase())) {
            return true;
        }
        if (connection.connectionName?.toLowerCase().includes(term.toLowerCase())) {
            return true;
        }
        if (connection.connectionDescription?.toLowerCase().includes(term.toLowerCase())) {
            return true;
        }

        // ManagedDatasetConnection
        if (connection.connection?.toLowerCase().includes(term.toLowerCase())) {
            return true;
        }

        return false;
    };

});

})();
