(function () {
  "use strict";

    angular.module("dataiku.nestedFilters").factory("Expressions", Expressions);

    function Expressions($stateParams, DataikuAPI, Logger, translate) {

        function inColName(value) {
            return "<span class=\"input-column-name\">" + sanitize(value) + "</span>";
        }
        function numLiteral(value) {
            return "<span class=\"num-literal\">" + sanitize(value) + "</span>";
        }
        function anumLiteral(value) {
            if (value === null || angular.isUndefined(value) || value.length === 0) {
                return '<span class="alphanum-literal">\'\'</span>';
            } else {
                return "<span class=\"alphanum-literal\">" + sanitize(value) + "</span>";
            }
        }

        /**
        An operator is defined by :

        Mandatory keys :
        name -> Links to the backend operator. Should be unique.
        label -> What is displayed to the user.
        group -> for prepare recipe "Create if then else" step categorization.
        params -> Type(s) of object(s) used as a secondary param(s) for this operator. can be "col" if operator takes a column as secondary param.
        enableFor / disableFor -> Used to know which type of input Column this operator will be available for
        repr -> Used in filter-nice-repr for a more compact representation of conditions

        Optional keys :
        meaning / approximateEquivalences -> Used to look for similar available operators when changing input column type
        disableForPrepare -> Whether this operator should not be available in a prepare Create if then else" step
        subParams -> In case of a list operator , type(s) of object(s) used as a secondary param(s) for its sub items
        valueMode -> Single value or multiple value
        **/
        var operators = [
        // Empty / Not empty group
        {
        name: "not empty",
        label: translate("FILTER.CONDITION.IS_DEFINED", "Is defined"),
        params: [],
        group: translate("FILTER.CONDITION.GROUP.EMPTY_NOT_EMPTY", "Empty / Not empty"),
        enableFor: ["string"],
        repr : function(cond) { return translate("FILTER.CONDITION.REPRESENTATION.IS_DEFINED", inColName(cond["input"]) + " is defined", { input: inColName(cond["input"]) }); },
        meaning: "is defined",
        valueMode: translate("FILTER.CONDITION.SINGLE_VALUE", "Single values")
        },
        {
        name: "is empty",
        label: translate("FILTER.CONDITION.IS_NOT_DEFINED", "Is not defined"),
        params: [],
        group: translate("FILTER.CONDITION.GROUP.EMPTY_NOT_EMPTY", "Empty / Not empty"),
        enableFor: ["string"],
        repr : function(cond) { return translate("FILTER.CONDITION.REPRESENTATION.IS_NOT_DEFINED", inColName(cond["input"]) + " is not defined", { input: inColName(cond["input"]) }); },
        meaning: "is not defined",
        valueMode: translate("FILTER.CONDITION.SINGLE_VALUE", "Single values")
        },
        // String operations group
        {
        name: "not empty string",
        label: translate("FILTER.CONDITION.IS_DEFINED", "Is defined"),
        params: [],
        group: translate("FILTER.CONDITION.GROUP.STRING_OPS", "String operations"),
        disableFor: ["string", "enum", "enum[]", "status"],
        repr : function(cond) { return translate("FILTER.CONDITION.REPRESENTATION.IS_DEFINED", inColName(cond["input"]) + " is defined", { input: inColName(cond["input"]) }); },
        meaning: "is defined",
        disableForPrepare: true,
        valueMode: translate("FILTER.CONDITION.SINGLE_VALUE", "Single values")
        },
        {
        name: "empty string",
        label: translate("FILTER.CONDITION.IS_NOT_DEFINED", "Is not defined"),
        params: [],
        group: translate("FILTER.CONDITION.GROUP.STRING_OPS", "String operations"),
        disableFor: ["string", "enum", "enum[]", "status"],
        repr : function(cond) { return translate("FILTER.CONDITION.REPRESENTATION.IS_NOT_DEFINED", inColName(cond["input"]) + " is not defined", { input: inColName(cond["input"]) }); },
        meaning: "is not defined",
        disableForPrepare: true,
        valueMode: translate("FILTER.CONDITION.SINGLE_VALUE", "Single values")
        },
        {
        name: "== [string]",
        label: translate("FILTER.CONDITION.EQUALS", "Equals"),
        params: ["string"],
        group: translate("FILTER.CONDITION.GROUP.STRING_OPS", "String operations"),
        disableFor: ["num", "date", "boolean", "enum", "enum[]", "status"],
        repr : function(cond) { return inColName(cond["input"]) + " == " + anumLiteral(cond["string"]); },
        meaning: "==",
        approximateEquivalences: ["is true", "== [column]"],
        valueMode: translate("FILTER.CONDITION.SINGLE_VALUE", "Single values")
        },
        {
        name: "== [string]i",
        label: translate("FILTER.CONDITION.EQUALS_CASE_INSENSITIVE", "Equals (case insensitive)"),
        params: ["string"],
        group: translate("FILTER.CONDITION.GROUP.STRING_OPS", "String operations"),
        disableFor: ["num", "date", "boolean", "enum", "enum[]", "status"],
        repr : function(cond) { return inColName(cond["input"]) + " == " + anumLiteral(cond["string"]) + " (insensitive)"; },
        meaning: "==",
        approximateEquivalences: ["is true", "== [column]"],
        valueMode: translate("FILTER.CONDITION.SINGLE_VALUE", "Single values")
        },
        {
        name: "!= [string]",
        label: translate("FILTER.CONDITION.IS_DIFFERENT_FROM", "Is different from"),
        params: ["string"],
        group: translate("FILTER.CONDITION.GROUP.STRING_OPS", "String operations"),
        disableFor: ["num", "date", "boolean", "enum", "enum[]", "status"],
        repr : function(cond) { return inColName(cond["input"]) + " != " + anumLiteral(cond["string"]); },
        meaning: "!=",
        approximateEquivalences: ["!= [column]"],
        valueMode: translate("FILTER.CONDITION.SINGLE_VALUE", "Single values")
        },
        {
        name: "== [NaNcolumn]",
        label: translate("FILTER.CONDITION.IS_THE_SAME_AS", "Is the same as"),
        params: ["col"],
        group: translate("FILTER.CONDITION.GROUP.STRING_OPS", "String operations"),
        disableFor: ["num", "enum", "enum[]", "status"],
        repr : function(cond) { return inColName(cond["input"]) + " == " + inColName(cond["col"]); },
        meaning: "== [column]",
        approximateEquivalences: ["=="],
        valueMode: translate("FILTER.CONDITION.SINGLE_VALUE", "Single values")
        },
        {
        name: "!= [NaNcolumn]",
        label: translate("FILTER.CONDITION.IS_DIFFERENT_FROM", "Is different from"),
        params: ["col"],
        group: translate("FILTER.CONDITION.GROUP.STRING_OPS", "String operations"),
        disableFor: ["num", "enum", "enum[]", "status"],
        repr : function(cond) { return inColName(cond["input"]) + " != " + inColName(cond["col"]); },
        meaning: "!= [column]",
        approximateEquivalences: ["!="],
        valueMode: translate("FILTER.CONDITION.SINGLE_VALUE", "Single values")
        },
        {
        name: "contains",
        label: translate("FILTER.CONDITION.CONTAINS", "Contains"),
        params: ["string"],
        group: translate("FILTER.CONDITION.GROUP.STRING_OPS", "String operations"),
        enableFor: ["string"],
        repr : function(cond){
        return translate("FILTER.CONDITION.REPRESENTATION.CONTAINS", inColName(cond["input"]) + " contains " + anumLiteral(cond["string"]), { input: inColName(cond["input"]), string: anumLiteral(cond["string"]) });
        },
        meaning: "contains",
        valueMode: translate("FILTER.CONDITION.SINGLE_VALUE", "Single values")
        },
        {
        name: "contains [string]i",
        label: translate("FILTER.CONDITION.CONTAINS_CASE_INSENSITIVE", "Contains (case insensitive)"),
        params: ["string"],
        group: translate("FILTER.CONDITION.GROUP.STRING_OPS", "String operations"),
        enableFor: ["string"],
        repr : function(cond){
        return translate("FILTER.CONDITION.REPRESENTATION.CONTAINS_CASE_INSENSITIVE", inColName(cond["input"]) + " contains " + anumLiteral(cond["string"]) + " (insensitive)", { input: inColName(cond["input"]), string: anumLiteral(cond["string"]) });
        },
        meaning: "contains",
        valueMode: translate("FILTER.CONDITION.SINGLE_VALUE", "Single values")
        },
        {
        name: "not contains",
        label: translate("FILTER.CONDITION.DOES_NOT_CONTAIN", "Does not contain"),
        params: ["string"],
        group: translate("FILTER.CONDITION.GROUP.STRING_OPS", "String operations"),
        enableFor: ["string"],
        repr : function(cond){
        return translate("FILTER.CONDITION.REPRESENTATION.DOES_NOT_CONTAIN", inColName(cond["input"]) + " does not contain " + anumLiteral(cond["string"]), { input: inColName(cond["input"]), string: anumLiteral(cond["string"]) });
        },
        meaning: "not contains",
        valueMode: translate("FILTER.CONDITION.SINGLE_VALUE", "Single values")
        },
        {
        name: "not contains [string]i",
        label: translate("FILTER.CONDITION.DOES_NOT_CONTAIN_CASE_INSENSITIVE", "Does not contain (case insensitive)"),
        params: ["string"],
        group: translate("FILTER.CONDITION.GROUP.STRING_OPS", "String operations"),
        enableFor: ["string"],
        repr : function(cond){
        return translate("FILTER.CONDITION.REPRESENTATION.DOES_NOT_CONTAIN_CASE_INSENSITIVE", inColName(cond["input"]) + " does not contain " + anumLiteral(cond["string"]) + " (insensitive)", { input: inColName(cond["input"]), string: anumLiteral(cond["string"]) });
        },
        meaning: "not contains",
        valueMode: translate("FILTER.CONDITION.SINGLE_VALUE", "Single values")
        },
        {
        name: "regex",
        label: translate("FILTER.CONDITION.MATCHES_REGEX", "Matches the regex"),
        params: ["string"],
        group: translate("FILTER.CONDITION.GROUP.STRING_OPS", "String operations"),
        enableFor: ["string"],
        repr : function(cond) { return inColName(cond["input"]) + " =~ /" + anumLiteral(cond["string"]) + "/"; },
        approximateEquivalences: ["=="], //debatable
        valueMode: translate("FILTER.CONDITION.SINGLE_VALUE", "Single values")
        },
        {
        name: "in [string]",
        label: translate("FILTER.CONDITION.IS_ANY_STRINGS", "Is any of the strings"),
        params: ["list"],
        subParams: ["string"],
        group: translate("FILTER.CONDITION.GROUP.STRING_OPS", "String operations"),
        enableFor: ["string"],
        repr : function(cond) { return translate("FILTER.CONDITION.REPRESENTATION.IS_ANY_STRINGS", inColName(cond["input"]) + " in " + anumLiteral((cond.items || []).map(i => i["string"]).join(',')), {input: inColName(cond["input"]), string: anumLiteral((cond.items || []).map(i => i["string"]).join(','))}); },
        approximateEquivalences: [],
        valueMode: translate("FILTER.CONDITION.MULTIPLE_VALUES", "Multiple values")
        },
        {
        name: "not in [string]",
        label: translate("FILTER.CONDITION.IS_NONE_STRINGS", "Is none of the strings"),
        params: ["list"],
        subParams: ["string"],
        group: translate("FILTER.CONDITION.GROUP.STRING_OPS", "String operations"),
        enableFor: ["string"],
        repr : function(cond) { return translate("FILTER.CONDITION.REPRESENTATION.IS_NONE_STRINGS", inColName(cond["input"]) + " not in " + anumLiteral((cond.items || []).map(i => i["string"]).join(',')), {input: inColName(cond["input"]), string: anumLiteral((cond.items || []).map(i => i["string"]).join(','))}); },
        approximateEquivalences: [],
        valueMode: translate("FILTER.CONDITION.MULTIPLE_VALUES", "Multiple values")
        },
        // Number operations group
        {
        name: "== [number]",
        label: "==",
        params: ["num"],
        group: translate("FILTER.CONDITION.GROUP.NUMBER_OPS", "Number operations"),
        enableFor: ["num"],
        repr : function(cond) { return inColName(cond["input"]) + " == " + numLiteral(cond["num"]); },
        meaning: "==",
        approximateEquivalences: ["is true", "== [column]"],
        valueMode: translate("FILTER.CONDITION.SINGLE_VALUE", "Single values")
        },
        {
        name: "!= [number]",
        label: "!=",
        params: ["num"],
        enableFor: ["num"],
        group: translate("FILTER.CONDITION.GROUP.NUMBER_OPS", "Number operations"),
        repr : function(cond) { return inColName(cond["input"]) + " != " + numLiteral(cond["num"]); },
        meaning: "!=",
        approximateEquivalences: ["!= [column]"],
        valueMode: translate("FILTER.CONDITION.SINGLE_VALUE", "Single values")
        },
        {
        name: ">  [number]",
        label: ">",
        params: ["num"],
        group: translate("FILTER.CONDITION.GROUP.NUMBER_OPS", "Number operations"),
        enableFor: ["num"],
        repr : function(cond) { return inColName(cond["input"]) + " > " + numLiteral(cond["num"]); },
        meaning: ">",
        approximateEquivalences: [">=", ">  [column]", ">= [column]"],
        valueMode: translate("FILTER.CONDITION.SINGLE_VALUE", "Single values")
        },
        {
        name: "<  [number]",
        label: "<",
        params: ["num"],
        group: translate("FILTER.CONDITION.GROUP.NUMBER_OPS", "Number operations"),
        enableFor: ["num"],
        repr : function(cond) { return inColName(cond["input"]) + " < " + numLiteral(cond["num"]); },
        meaning: "<",
        approximateEquivalences: ["<=", "<  [column]", "<= [column]"],
        valueMode: translate("FILTER.CONDITION.SINGLE_VALUE", "Single values")
        },
        {
        name: ">= [number]",
        label: ">=",
        params: ["num"],
        group: translate("FILTER.CONDITION.GROUP.NUMBER_OPS", "Number operations"),
        enableFor: ["num"],
        repr : function(cond) { return inColName(cond["input"]) + " >= " + numLiteral(cond["num"]); },
        meaning: ">=",
        approximateEquivalences: [ ">", ">= [column]", ">  [column]"],
        valueMode: translate("FILTER.CONDITION.SINGLE_VALUE", "Single values")
        },
        {
        name: "<= [number]",
        label: "<=",
        params: ["num"],
        group: translate("FILTER.CONDITION.GROUP.NUMBER_OPS", "Number operations"),
        enableFor: ["num"],
        repr : function(cond) { return inColName(cond["input"]) + " <= " + numLiteral(cond["num"]); },
        meaning: "<=",
        approximateEquivalences: ["<", "<= [column]", "<  [column]"],
        valueMode: translate("FILTER.CONDITION.SINGLE_VALUE", "Single values")
        },
        {
        name: ">< [number]",
        label: translate("FILTER.CONDITION.NUMBER_IS_BETWEEN", "Number is between"),
        params: ["num", "num2"],
        group: translate("FILTER.CONDITION.GROUP.NUMBER_OPS", "Number operations"),
        enableFor: ["num"],
        repr : function(cond) { return numLiteral(cond["num"]) + " <= " + inColName(cond["input"]) + " <= " + numLiteral(cond["num2"]); },
        approximateEquivalences: [],
        valueMode: translate("FILTER.CONDITION.SINGLE_VALUE", "Single values")
        },
        {
        name: "<> [number]",
        label: translate("FILTER.CONDITION.NUMBER_IS_NOT_BETWEEN", "Number is not between"),
        params: ["num", "num2"],
        group: translate("FILTER.CONDITION.GROUP.NUMBER_OPS", "Number operations"),
        enableFor: ["num"],
        repr : function(cond) { return inColName(cond["input"]) + " < " + numLiteral(cond["num"]) + translate("FILTER._OR_", " or ") + inColName(cond["input"]) + " > " + numLiteral(cond["num2"]); },
        approximateEquivalences: [],
        valueMode: translate("FILTER.CONDITION.SINGLE_VALUE", "Single values")
        },
        {
        name: "== [column]",
        label: "==",
        params: ["col"],
        group: translate("FILTER.CONDITION.GROUP.NUMBER_OPS", "Number operations"),
        enableFor: ["num"],
        repr : function(cond) { return inColName(cond["input"]) + " == " + inColName(cond["col"]); },
        meaning: "== [column]",
        approximateEquivalences: ["=="],
        valueMode: translate("FILTER.CONDITION.SINGLE_VALUE", "Single values")
        },
        {
        name: "!= [column]",
        label: "!=",
        params: ["col"],
        group: translate("FILTER.CONDITION.GROUP.NUMBER_OPS", "Number operations"),
        enableFor: ["num"],
        repr : function(cond) { return inColName(cond["input"]) + " != " + inColName(cond["col"]); },
        meaning: "!= [column]",
        approximateEquivalences: ["!="],
        valueMode: translate("FILTER.CONDITION.SINGLE_VALUE", "Single values")
        },
        {
        name: ">  [column]",
        label: ">",
        params: ["col"],
        group: translate("FILTER.CONDITION.GROUP.NUMBER_OPS", "Number operations"),
        enableFor: ["num"],
        repr : function(cond) { return inColName(cond["input"]) + " > " + inColName(cond["col"]); },
        meaning: ">  [column]",
        approximateEquivalences: [">= [column]", ">", ">="],
        valueMode: translate("FILTER.CONDITION.SINGLE_VALUE", "Single values")
        },
        {
        name: "<  [column]",
        label: "<",
        params: ["col"],
        group: translate("FILTER.CONDITION.GROUP.NUMBER_OPS", "Number operations"),
        enableFor: ["num"],
        repr : function(cond) { return inColName(cond["input"]) + " < " + inColName(cond["col"]); },
        meaning: "<  [column]",
        approximateEquivalences: ["<= [column]", "<", "<="],
        valueMode: translate("FILTER.CONDITION.SINGLE_VALUE", "Single values")
        },
        {
        name: ">= [column]",
        label: ">=",
        params: ["col"],
        group: translate("FILTER.CONDITION.GROUP.NUMBER_OPS", "Number operations"),
        enableFor: ["num"],
        repr : function(cond) { return inColName(cond["input"]) + " >= " + inColName(cond["col"]); },
        meaning: ">= [column]",
        approximateEquivalences: [">  [column]", ">=", ">"],
        valueMode: translate("FILTER.CONDITION.SINGLE_VALUE", "Single values")
        },
        {
        name: "<= [column]",
        label: "<=",
        params: ["col"],
        group: translate("FILTER.CONDITION.GROUP.NUMBER_OPS", "Number operations"),
        enableFor: ["num"],
        repr : function(cond) { return inColName(cond["input"]) + " <= " + inColName(cond["col"]); },
        meaning: "<= [column]",
        approximateEquivalences: ["<  [column]", "<=", "<"],
        valueMode: translate("FILTER.CONDITION.SINGLE_VALUE", "Single values")
        },
        {
        name: "in [number]",
        label: translate("FILTER.CONDITION.IS_ANY_NUMBERS", "Is any of the numbers"),
        params: ["list"],
        subParams: ["num"],
        group: translate("FILTER.CONDITION.GROUP.NUMBER_OPS", "Number operations"),
        enableFor: ["num"],
        repr : function(cond) { return translate("FILTER.CONDITION.REPRESENTATION.IS_ANY_NUMBERS", inColName(cond["input"]) + " in " + anumLiteral((cond.items || []).map(i => i["num"]).join(',')), {input: inColName(cond["input"]), num: anumLiteral((cond.items || []).map(i => i["num"]).join(','))}); },
        approximateEquivalences: [],
        valueMode: translate("FILTER.CONDITION.MULTIPLE_VALUES", "Multiple values")
        },
        {
        name: "not in [number]",
        label: translate("FILTER.CONDITION.IS_NONE_NUMBERS", "Is none of the numbers"),
        params: ["list"],
        subParams: ["num"],
        group: translate("FILTER.CONDITION.GROUP.NUMBER_OPS", "Number operations"),
        enableFor: ["num"],
        repr : function(cond) { return translate("FILTER.CONDITION.REPRESENTATION.IS_NONE_NUMBERS", inColName(cond["input"]) + " not in " + anumLiteral((cond.items || []).map(i => i["num"]).join(',')), {input: inColName(cond["input"]), num: anumLiteral((cond.items || []).map(i => i["num"]).join(','))}); },
        approximateEquivalences: [],
        valueMode: translate("FILTER.CONDITION.MULTIPLE_VALUES", "Multiple values")
        },
        // Date operations group
        {
        name: "== [date]",
        label: translate("FILTER.CONDITION.EQUALS_DATE", "Equals date"),
        params: ["date", "time", "unit"],
        group: translate("FILTER.CONDITION.GROUP.DATE_OPS", "Date operations"),
        enableFor: ["date"],
        repr : function(cond) { return inColName(cond["input"]) + " == " + numLiteral(angular.isUndefined(cond["date"]) ? "" : cond["date"]) + " " + numLiteral(angular.isUndefined(cond["time"]) ? "" : cond["time"]) + " (~ " + anumLiteral(cond["unit"]) + ")"; },
        meaning: "==",
        approximateEquivalences: ["is true", "== [column]"],
        valueMode: translate("FILTER.CONDITION.SINGLE_VALUE", "Single values")
        },
        {
        name: ">= [date]",
        label: translate("FILTER.CONDITION.IS_AFTER_OR_EQUAL", "Is after or equal"),
        params: ["date", "time"],
        group: translate("FILTER.CONDITION.GROUP.DATE_OPS", "Date operations"),
        enableFor: ["date"],
        repr : function(cond) {
            return inColName(cond["input"]) + " >= " + numLiteral(angular.isUndefined(cond["date"]) ? "" : cond["date"]) + " " + numLiteral(angular.isUndefined(cond["time"]) ? "" : cond["time"]) ;
            },
        meaning: ">=",
        approximateEquivalences: [">", ">= [column]", ">  [column]"],
        valueMode: translate("FILTER.CONDITION.SINGLE_VALUE", "Single values")
        },
        {
        name: "<  [date]",
        label: translate("FILTER.CONDITION.IS_STRICTLY_BEFORE", "Is strictly before"),
        params: ["date", "time"],
        group: translate("FILTER.CONDITION.GROUP.DATE_OPS", "Date operations"),
        enableFor: ["date"],
        repr : function(cond) { return inColName(cond["input"]) + " <" + numLiteral(angular.isUndefined(cond["date"]) ? "" : cond["date"]) + " " + numLiteral(angular.isUndefined(cond["time"]) ? "" : cond["time"]) ; },
        meaning: "<",
        approximateEquivalences: ["<=", "<  [column]", "<= [column]"],
        valueMode: translate("FILTER.CONDITION.SINGLE_VALUE", "Single values")
        },
        {
        name: ">< [date]",
        label: translate("FILTER.CONDITION.IS_BETWEEN", "Is between"),
        params: ["date", "date2", "time", "time2"],
        group: translate("FILTER.CONDITION.GROUP.DATE_OPS", "Date operations"),
        enableFor: ["date"],
        repr : function(cond) {
        return numLiteral(angular.isUndefined(cond["date"]) ? "" : cond["date"]) + " " + numLiteral(angular.isUndefined(cond["time"]) ? "" : cond["time"]) +
            " <= " +
            inColName(cond["input"]) +
            " < " +
            numLiteral(angular.isUndefined(cond["date2"]) ? "" : cond["date2"]) + " " + numLiteral(angular.isUndefined(cond["time2"]) ? "" : cond["time2"]);
        },
        approximateEquivalences: [">=", ">", ">= [column]", ">  [column]"], //debatable
        valueMode: translate("FILTER.CONDITION.SINGLE_VALUE", "Single values")
        },
        {
        name: "in [date]",
        label: translate("FILTER.CONDITION.IS_ANY_DATES", "Is any of the dates"),
        params: ["list"],
        subParams: ["date", "time", "unit"],
        group: translate("FILTER.CONDITION.GROUP.DATE_OPS", "Date operations"),
        enableFor: ["date"],
        repr : function(cond) { return translate("FILTER.CONDITION.REPRESENTATION.IS_ANY_DATES", inColName(cond["input"]) + " in " + anumLiteral((cond.items || []).map(i => (angular.isUndefined(i["date"]) ? "" : i["date"])  + " " + (angular.isUndefined(i["time"]) ? "" : i["time"])).join(',')), {input: inColName(cond["input"]), date: anumLiteral((cond.items || []).map(i => (angular.isUndefined(i["date"]) ? "" : i["date"])  + " " + (angular.isUndefined(i["time"]) ? "" : i["time"])).join(','))}); },
        approximateEquivalences: [],
        valueMode: translate("FILTER.CONDITION.MULTIPLE_VALUES", "Multiple values")
        },
        {
        name: "not in [date]",
        label: translate("FILTER.CONDITION.IS_NONE_DATES", "Is none of the dates"),
        params: ["list"],
        subParams: ["date", "time", "unit"],
        group: translate("FILTER.CONDITION.GROUP.DATE_OPS", "Date operations"),
        enableFor: ["date"],
        repr : function(cond) { return translate("FILTER.CONDITION.REPRESENTATION.IS_NONE_DATES", inColName(cond["input"]) + " not in " + anumLiteral((cond.items || []).map(i => (angular.isUndefined(i["date"]) ? "" : i["date"])  + " " + (angular.isUndefined(i["time"]) ? "" : i["time"])).join(',')), {input: inColName(cond["input"]), date: anumLiteral((cond.items || []).map(i => (angular.isUndefined(i["date"]) ? "" : i["date"])  + " " + (angular.isUndefined(i["time"]) ? "" : i["time"])).join(','))}); },
        approximateEquivalences: [],
        valueMode: translate("FILTER.CONDITION.MULTIPLE_VALUES", "Multiple values")
        },
        // Other group
        {
        name: "empty array",
        label: translate("FILTER.CONDITION.IS_EMPTY_ARRAY", "Is an empty array"),
        params: [],
        group: translate("FILTER.CONDITION.GROUP.OTHER", "Other"),
        enableFor: ["array"],
        repr : function(cond) { return translate("FILTER.CONDITION.REPRESENTATION.IS_EMPTY_ARRAY", inColName(cond["input"]) + " is an empty array", {input: inColName(cond["input"])}); },
        valueMode: translate("FILTER.CONDITION.SINGLE_VALUE", "Single values")
        },
        {
        name: "not empty array",
        label: translate("FILTER.CONDITION.IS_NOT_EMPTY_ARRAY", "Is not an empty array"),
        params: [],
        group: translate("FILTER.CONDITION.GROUP.OTHER", "Other"),
        enableFor: ["array"],
        repr : function(cond) { return translate("FILTER.CONDITION.REPRESENTATION.IS_NOT_EMPTY_ARRAY", inColName(cond["input"]) + " is not an empty array", {input: inColName(cond["input"])}); },
        valueMode: translate("FILTER.CONDITION.SINGLE_VALUE", "Single values")
        },
        {
        name: "array contains",
        label: translate("FILTER.CONDITION.ARRAY_CONTAINS_VALUE", "Array contains value"),
        params: ["string"],
        group: translate("FILTER.CONDITION.GROUP.OTHER", "Other"),
        enableFor: ["array"],
        repr : function(cond) { return translate("FILTER.CONDITION.REPRESENTATION.ARRAY_CONTAINS_VALUE", inColName(cond["input"]) + " contains " + anumLiteral(cond["string"]), {input: inColName(cond["input"]), string: anumLiteral(cond["string"])}); },
        meaning: "contains",
        valueMode: translate("FILTER.CONDITION.SINGLE_VALUE", "Single values")
        },
        {
        name: "array not contains",
        label: translate("FILTER.CONDITION.ARRAY_NOT_CONTAINS_VALUE", "Array does not contain value"),
        params: ["string"],
        group: translate("FILTER.CONDITION.GROUP.OTHER", "Other"),
        enableFor: ["array"],
        repr : function(cond) { return translate("FILTER.CONDITION.REPRESENTATION.ARRAY_NOT_CONTAINS_VALUE", inColName(cond["input"]) + " does not contain " + anumLiteral(cond["string"]), {input: inColName(cond["input"]), string: anumLiteral(cond["string"])}); },
        meaning: "not contains",
        valueMode: translate("FILTER.CONDITION.SINGLE_VALUE", "Single values")
        },
        {
        name: "true",
        label: translate("FILTER.CONDITION.IS_TRUE", "Is true"),
        params: [],
        group: translate("FILTER.CONDITION.GROUP.OTHER", "Other"),
        enableFor: ["boolean"],
        repr : function(cond) { return translate("FILTER.CONDITION.REPRESENTATION.IS_TRUE", inColName(cond["input"]) + " is true", {input: inColName(cond["input"])}); },
        meaning: "is true",
        approximateEquivalences: ["==", "== [column]"],
        valueMode: translate("FILTER.CONDITION.SINGLE_VALUE", "Single values")
        },
        {
        name: "false",
        label: translate("FILTER.CONDITION.IS_FALSE", "Is false"),
        params: [],
        group: translate("FILTER.CONDITION.GROUP.OTHER", "Other"),
        enableFor: ["boolean"],
        repr : function(cond) { return translate("FILTER.CONDITION.REPRESENTATION.IS_FALSE", inColName(cond["input"]) + " is false", {input: inColName(cond["input"])}); },
        meaning: "is false",
        approximateEquivalences: ["==", "== [column]"],
        valueMode: translate("FILTER.CONDITION.SINGLE_VALUE", "Single values")
        },
        {
        name: "geoWithin",
        label: translate("FILTER.CONDITION.WITHIN_GEOMETRY", "Within geometry"),
        params: ["string"], // For example: POLYGON((0 -90,0 90,360 90,360 -90,0 -90))
        group: translate("FILTER.CONDITION.GROUP.OTHER", "Other"),
        enableFor: ["geometry", "geopoint"],
        repr : function(cond) { return translate("FILTER.CONDITION.REPRESENTATION.WITHIN_GEOMETRY", inColName(cond["input"]) + " is within " + anumLiteral(cond["string"]), {input: inColName(cond["input"]), string: anumLiteral(cond["string"])}); },
        approximateEquivalences: [], //debatable
        valueMode: translate("FILTER.CONDITION.SINGLE_VALUE", "Single values")
        },
        {
        name: "geoContains",
        label: translate("FILTER.CONDITION.CONTAINS_GEOMETRY", "Contains geometry"),
        params: ["string"], // For example: POLYGON((0 -90,0 90,360 90,360 -90,0 -90))
        group: translate("FILTER.CONDITION.GROUP.OTHER", "Other"),
        enableFor: ["geometry"],
        repr : function(cond) { return translate("FILTER.CONDITION.REPRESENTATION.CONTAINS_GEOMETRY", inColName(cond["input"]) + " contains " + anumLiteral(cond["string"]), {input: inColName(cond["input"]), string: anumLiteral(cond["string"])}); },
        approximateEquivalences: [], //debatable
        valueMode: translate("FILTER.CONDITION.SINGLE_VALUE", "Single values")
        },
        // Enum operations group
        {
        name: "in [enum]",
        label: translate("FILTER.CONDITION.IS_ANY_ENUMS", "Is any of"),
        params: ["list"],
        subParams: ["enum"],
        group: translate("FILTER.CONDITION.GROUP.ENUM_OPS", "Enum operations"),
        enableFor: ["enum"],
        repr : function(cond) { return translate("FILTER.CONDITION.REPRESENTATION.IS_ANY_ENUMS", inColName(cond["input"]) + " in " + anumLiteral((cond.items || []).map(i => i["string"]).join(',')), {input: inColName(cond["input"]), string: anumLiteral((cond.items || []).map(i => i["string"]).join(','))}); },
        approximateEquivalences: [],
        valueMode: translate("FILTER.CONDITION.MULTIPLE_VALUES", "Multiple values"),
        disableForPrepare: true
        },
        {
        name: "not in [enum]",
        label: translate("FILTER.CONDITION.IS_NONE_ENUMS", "Is none of"),
        params: ["list"],
        subParams: ["enum"],
        group: translate("FILTER.CONDITION.GROUP.ENUM_OPS", "Enum operations"),
        enableFor: ["enum"],
        repr : function(cond) { return translate("FILTER.CONDITION.REPRESENTATION.IS_NONE_ENUMS", inColName(cond["input"]) + " not in " + anumLiteral((cond.items || []).map(i => i["string"]).join(',')), {input: inColName(cond["input"]), string: anumLiteral((cond.items || []).map(i => i["string"]).join(','))}); },
        approximateEquivalences: [],
        valueMode: translate("FILTER.CONDITION.MULTIPLE_VALUES", "Multiple values"),
        disableForPrepare: true
        },
        {
        name: "array contains any [enum]",
        label: translate("FILTER.CONDITION.ARRAY_CONTAINS_ANY_ENUM", "Contains any"),
        params: ["list"],
        subParams: ["enum"],
        group: translate("FILTER.CONDITION.GROUP.ENUM_OPS", "Enum operations"),
        enableFor: ["enum[]"],
        repr : function(cond) { return translate("FILTER.CONDITION.REPRESENTATION.IS_ANY_ENUMS", inColName(cond["input"]) + " contains any of " + anumLiteral((cond.items || []).map(i => i["string"]).join(',')), {input: inColName(cond["input"]), string: anumLiteral((cond.items || []).map(i => i["string"]).join(','))}); },
        approximateEquivalences: [],
        valueMode: translate("FILTER.CONDITION.MULTIPLE_VALUES", "Multiple values"),
        disableForPrepare: true
        },
        {
        name: "array contains none [enum]",
        label: translate("FILTER.CONDITION.ARRAY_CONTAINS_NONE_ENUM", "Contains none"),
        params: ["list"],
        subParams: ["enum"],
        group: translate("FILTER.CONDITION.GROUP.ENUM_OPS", "Enum operations"),
        enableFor: ["enum[]"],
        repr : function(cond) { return translate("FILTER.CONDITION.REPRESENTATION.IS_NONE_ENUMS", inColName(cond["input"]) + " contains none of " + anumLiteral((cond.items || []).map(i => i["string"]).join(',')), {input: inColName(cond["input"]), string: anumLiteral((cond.items || []).map(i => i["string"]).join(','))}); },
        approximateEquivalences: [],
        valueMode: translate("FILTER.CONDITION.MULTIPLE_VALUES", "Multiple values"),
        disableForPrepare: true
        },
        {
        name: "array contains all [enum]",
        label: translate("FILTER.CONDITION.ARRAY_CONTAINS_ALL_ENUM", "Contains all"),
        params: ["list"],
        subParams: ["enum"],
        group: translate("FILTER.CONDITION.GROUP.ENUM_OPS", "Enum operations"),
        enableFor: ["enum[]"],
        repr : function(cond) { return translate("FILTER.CONDITION.REPRESENTATION.CONTAINS_ALL_ENUMS", inColName(cond["input"]) + " contains all of " + anumLiteral((cond.items || []).map(i => i["string"]).join(',')), {input: inColName(cond["input"]), string: anumLiteral((cond.items || []).map(i => i["string"]).join(','))}); },
        approximateEquivalences: [],
        valueMode: translate("FILTER.CONDITION.MULTIPLE_VALUES", "Multiple values"),
        disableForPrepare: true
        },
        // Status operations group
        {
        name: "status no longer healthy",
        label: "Is no longer healthy",
        params: [],
        group: "Status evolution",
        enableFor: ["status"],
        repr : function(cond) { return translate("FILTER.CONDITION.STATUS_NO_LONGER_HEALTHY",  inColName(cond["input"]) + " no longer healthy", {input: inColName(cond["input"])}); },
        valueMode: translate("FILTER.CONDITION.STATUS_EVOLUTION", "Status evolution"),
        disableForPrepare: true,
        },
        {
        name: "status changes",
        label: "Changes",
        params: [],
        group: "Status evolution",
        enableFor: ["status"],
        repr : function(cond) { return translate("FILTER.CONDITION.STATUS_CHANGES",  inColName(cond["input"]) + " changes", {input: inColName(cond["input"])}); },
        valueMode: translate("FILTER.CONDITION.STATUS_EVOLUTION", "Status evolution"),
        disableForPrepare: true,
        },
        {
        name: "status becomes any of",
        label: "Becomes any of",
        params: ["list"],
        subParams: ["enum"],
        group: "Status evolution",
        enableFor: ["status"],
        repr : function(cond) { return translate("FILTER.CONDITION.STATUS_BECOMES_ANY_OF", inColName(cond["input"]) + "&#160; in &#160;" + anumLiteral((cond.items || []).map(i => i["string"]).join(',')), {input: inColName(cond["input"]), string: anumLiteral((cond.items || []).map(i => i["string"]).join(','))}); },
        valueMode: translate("FILTER.CONDITION.MULTIPLE_VALUES", "Multiple values"),
        disableForPrepare: true,
        }
        ];


        { //debug check for operator names
            var namesSoFar = [];
            for (var i = 0; i < operators.length; ++i) {
                var name = operators[i].name;
                if (namesSoFar.indexOf(name) >= 0) {
                    Logger.error("Duplicate operator name ("+name+")");
                }
                namesSoFar.push(name);
            }
        }

        var genericType = function(type) {
            type = (type||'').toLowerCase();
            if (["tinyint", "smallint", "int", "bigint", "float", "double"].indexOf(type) > -1) {
                return "num";
            }
            if (["date", "dateonly", "datetimenotz"].indexOf(type) > -1) {
                return "date";
            }
            return type;
        };

        var getDefaultOperator = function(type) { //for default selection in UI
            switch(genericType(type)) {
                case "array": return "not empty array";
                case "boolean": return "true";
                case "date": return ">= [date]";
                case "num": return "== [number]";
                case "string": return "contains";
                case "geometry": return "geoContains";
                case "geopoint": return "geoWithin";
                case "enum": return "in [enum]";
                case "enum[]": return "array contains any [enum]";
                case "status": return "status changes";
                default: return "not empty";
            }
        };

        const DATE_UNITS = [
            {name:"seconds",label:translate("SAMPLING.DATE_UNIT.SECOND", "second")},
            {name:"minutes",label:translate("SAMPLING.DATE_UNIT.MINUTE", "minute")},
            {name:"hours",label:translate("SAMPLING.DATE_UNIT.HOUR", "hour")},
            {name:"days",label:translate("SAMPLING.DATE_UNIT.DAY", "day")},
            {name:"weeks",label:translate("SAMPLING.DATE_UNIT.WEEK", "week")},
            {name:"months",label:translate("SAMPLING.DATE_UNIT.MONTH", "month")},
            {name:"years",label:translate("SAMPLING.DATE_UNIT.YEAR", "year")}
        ];

        var has = function(t,v) { if(!t) return false; return t.indexOf(v) > -1; };

        var getOperators = function(type) {
            if (!type) return operators;
            type = genericType(type);
            return operators
                .filter(function(op) {
                    return has(op.enableFor, type) || (!op.enableFor && !has(op.disableFor, type));
                })
                .map(function(op) {
                    return {name: op.name, label:op.label, params:op.params, subParams:op.subParams, meaning:op.meaning, approximateEquivalences: op.approximateEquivalences, valueMode: op.valueMode};
                });
        };

        var getOperatorsPrepareRecipe = function() {
            return operators.filter(function(op) {
                                return !(op.disableForPrepare);
                            })
                            .map(function(op) {
                                let appliesTo;
                                if (op.params.length === 0) {
                                    appliesTo = "none";
                                } else if (op.params.includes("col") || (angular.isDefined(op.subParams) && op.subParams.includes("col"))) {
                                    appliesTo = "column";
                                } else {
                                    appliesTo = "value";
                                }
                                return {name: op.name,
                                        label: op.label,
                                        params: op.params,
                                        subParams: op.subParams,
                                        meaning:op.meaning,
                                        approximateEquivalences: op.approximateEquivalences,
                                        repr: op.repr,
                                        appliesTo: appliesTo,
                                        group: op.group}
                            });
        }

        var getOperatorByName = function(name){
            return operators.find(op => op.name === name);
        };

        var validateExpression = function(expression, schema, recipe) {
            const stringifiedSchema = JSON.stringify(schema);
            if (recipe) {
                return DataikuAPI.flow.recipes.filter.validateExpression(expression,
                        stringifiedSchema, JSON.stringify(recipe));
            }
            return DataikuAPI.flow.recipes.filter.validateExpressionWithProjectKey(expression,
                    stringifiedSchema, $stateParams.projectKey);
        };

        var getOperatorLabelAppliesTo = function(label) {
            /**
                An operator label like "Equals" can correspond to several operators depending on what the operator is applied to,
                aka a column or a string input.
            **/
            let operatorsMatchingLabel = getOperatorsPrepareRecipe().filter(function(op) {
                return op.label === label;
            })
            return Array.from(new Set(operatorsMatchingLabel.map(op => op.appliesTo)));
        }

        var getOperatorsLabelsPrepareRecipe = function() {
            /**
                Gets a unique list of possible (label, group) tuples to display for the visual if step
            **/
            let ops = getOperatorsPrepareRecipe();
            let minimalOps = ops.map(function(op) {
                return {label: op.label, group: op.group};
            })
            const uniqueMinimalOps = minimalOps.filter((value, index) => {
                const _value = JSON.stringify(value);
                return index === minimalOps.findIndex(obj => {
                    return JSON.stringify(obj) === _value;
                });
            });
            return uniqueMinimalOps;
        }

        var findOperator = function(label, appliesTo) {
            return getOperatorsPrepareRecipe().find(op => op.label === label && op.appliesTo === appliesTo);
        }

        const service = {
            getOperators: getOperators,
            getDefaultOperator: getDefaultOperator,
            validateExpression: validateExpression,
            genericType: genericType,
            dateUnits: DATE_UNITS,
            getOperatorByName: getOperatorByName,
            getOperatorsPrepareRecipe: getOperatorsPrepareRecipe,
            getOperatorLabelAppliesTo: getOperatorLabelAppliesTo,
            getOperatorsLabelsPrepareRecipe: getOperatorsLabelsPrepareRecipe,
            findOperator: findOperator
        };
        return service;
    }
})();
