/* global describe, it, inject, expect, module, beforeEach */

describe('type mapping service', () => {

    function HistoryServiceMock() {
        return {};
    }
    function WT1Mock() {
        return {};
    }
    function StateUtilsMock() {
        return {};
    }
    function RecipeDescServiceMock() {
        return {};
    }
    function DashboardUtilsMock() {
        return {};
    }
    function WebAppsServiceMock() {
        return {};
    }

    beforeEach(() => {
        module('dataiku.constants');
        module('dataiku.services', ($provide) => {
            $provide.value('HistoryService', HistoryServiceMock);
            $provide.value('WT1', WT1Mock);
            $provide.value('StateUtils', StateUtilsMock);
            $provide.value('RecipeDescService', RecipeDescServiceMock);
            $provide.value('DashboardUtils', DashboardUtilsMock);
            $provide.value('WebAppsService', WebAppsServiceMock);
        });
        module('dataiku.mock');
    });

    let rootScopeForAppConfig;
    beforeEach(inject(function (_$rootScope_) {
        rootScopeForAppConfig = _$rootScope_;
        rootScopeForAppConfig.appConfig = {};
    }));

    let TypeMappingService;
    beforeEach(inject((_TypeMappingService_) => {
        TypeMappingService = _TypeMappingService_;
    }));

    const OBJECT_KEY = 'objectKey';
    const TYPE_KEY = 'typeKey';

    function buildCustomDefinition(objectKey, typeKey) {
        return {
            [OBJECT_KEY]: objectKey,
            [TYPE_KEY]: typeKey
        };
    }

    const CUSTOM_DATASETS = buildCustomDefinition('customDatasets', 'datasetType');
    const CUSTOM_PROVIDER_FS = buildCustomDefinition('customFSProviders', 'fsProviderType');
    const CUSTOM_SAMPLE_DATASET = buildCustomDefinition('customSampleDatasets', 'datasetType');

    function addMockedItemInCustoms(type, customToMock, icon, label, definedDesc, withOwnerPluginDesc) {
        let newMock = {
            [customToMock[TYPE_KEY]]: type
        };
        if (definedDesc) {
            Object.assign(newMock, {
                desc: {
                    meta: {
                        icon: icon,
                        label: label
                    }
                }
            });
        }
        const objectKey = customToMock[OBJECT_KEY];
        if (!(rootScopeForAppConfig.appConfig.hasOwnProperty(objectKey))) {
            rootScopeForAppConfig.appConfig[objectKey] = [];
        }

        if (withOwnerPluginDesc) {
            const ownerPluginId = 'ownerPluginId';
            Object.assign(newMock, {ownerPluginId: ownerPluginId});
            rootScopeForAppConfig.appConfig['loadedPlugins'] = [{id: ownerPluginId, icon: icon}];
        }
        rootScopeForAppConfig.appConfig[objectKey].push(newMock);
    }

    function addMockedLabelInCustoms(type, customToMock, value) {
        addMockedItemInCustoms(type, customToMock, undefined, value, true);
    }

    function addMockedIconInCustoms(type, customToMock, value) {
        addMockedItemInCustoms(type, customToMock, value, undefined, true);
    }

    function addMockedItemWithNoDescInCustoms(type, customToMock) {
        addMockedItemInCustoms(type, customToMock, undefined, undefined, false);
    }

    describe('mapDatasetTypeToName', function () {
        describe('no type provided', function () {
            it('returns empty', function () {
                expect(TypeMappingService.mapDatasetTypeToName('')).toBe('');
                expect(TypeMappingService.mapDatasetTypeToName(undefined)).toBe('');
            });
        });

        describe('type found in registered dataset types', function () {
            it('returns existing label', function () {
                expect(TypeMappingService.mapDatasetTypeToName('cachedhttp')).toBe('HTTP (with cache)');
            });
        });

        describe('type found with different case in registered dataset types', function () {
            it('returns existing label', function () {
                expect(TypeMappingService.mapDatasetTypeToName('CachedHTTP')).toBe('HTTP (with cache)');
            });
        });

        describe('type found but no label defined', function () {
            it('returns type', function () {
                expect(TypeMappingService.mapDatasetTypeToName('remotefiles')).toBe('remotefiles');
            });
        });

        describe('missing type', function () {
            describe('type not prefixed', function () {
                it('returns lower-cased type', function () {
                    expect(TypeMappingService.mapDatasetTypeToName('UnknownType')).toBe('unknowntype');
                });
            });

            describe('custom prefix', function () {
                describe('type in plugin dataset', function () {
                    it('returns plugin dataset label', function () {
                        addMockedLabelInCustoms('custom-KnownType', CUSTOM_DATASETS, 'custom-label');
                        expect(TypeMappingService.mapDatasetTypeToName('custom-KnownType')).toBe('custom-label');
                    });
                });
                describe('type not in plugin dataset but in FS provider', function () {
                    it('returns custom FS provider label', function () {
                        addMockedLabelInCustoms('custom-KnownType', CUSTOM_DATASETS, 'custom-label');
                        addMockedLabelInCustoms('custom-KnownFSType', CUSTOM_PROVIDER_FS, 'custom-fs-label');
                        expect(TypeMappingService.mapDatasetTypeToName('custom-KnownFSType')).toBe('custom-fs-label');
                    });
                });
                describe('type neither in plugin dataset nor in FS provider', function () {
                    it('returns datasetType if not found', function () {
                        addMockedLabelInCustoms('custom-KnownType', CUSTOM_DATASETS, 'custom-label');
                        addMockedLabelInCustoms('custom-KnownFSType', CUSTOM_PROVIDER_FS, 'custom-fs-label');
                        addMockedIconInCustoms('Sample-KnownType', CUSTOM_SAMPLE_DATASET, 'Sample-label');
                        expect(TypeMappingService.mapDatasetTypeToName('custom-UnknownType')).toBe('custom-UnknownType');
                    });
                });
                describe('type in custom', function () {
                    describe('no desc', function () {
                        it('returns type', function () {
                            addMockedItemWithNoDescInCustoms('custom-KnownTypeWithNoDesc', CUSTOM_DATASETS);
                            expect(TypeMappingService.mapDatasetTypeToName('custom-KnownTypeWithNoDesc')).toBe('custom-KnownTypeWithNoDesc');
                        });
                    });
                    it('returns plugin dataset label', function () {
                        addMockedLabelInCustoms('custom-KnownType', CUSTOM_DATASETS, 'custom-label');
                        expect(TypeMappingService.mapDatasetTypeToName('custom-KnownType')).toBe('custom-label');
                    });
                });
            });
            describe('Custom prefix with first upper case letter', function () {
                describe('type in plugin dataset', function () {
                    it('returns plugin dataset label', function () {
                        addMockedLabelInCustoms('Custom-KnownType-Prefix-Not-Case-Sensitive', CUSTOM_DATASETS, 'custom-label');
                        expect(TypeMappingService.mapDatasetTypeToName('Custom-KnownType-Prefix-Not-Case-Sensitive')).toBe('custom-label');
                    });
                });
            });

            describe('fsprovider_ prefix', function () {
                describe('type in plugin dataset', function () {
                    it('returns plugin dataset label no matter the prefix case', function () {
                        addMockedLabelInCustoms('custom-KnownType', CUSTOM_DATASETS, 'custom-label');
                        addMockedLabelInCustoms('fsprovider_KnownType-lowercase', CUSTOM_PROVIDER_FS,
                            'custom-fs-label-lc');
                        addMockedLabelInCustoms('FSPROVIDER_KnownType-uppercase', CUSTOM_PROVIDER_FS,
                            'custom-fs-label-uc');
                        expect(TypeMappingService.mapDatasetTypeToName('fsprovider_KnownType-lowercase')).toBe('custom-fs-label-lc');
                        expect(TypeMappingService.mapDatasetTypeToName('FSPROVIDER_KnownType-uppercase')).toBe('custom-fs-label-uc');
                    });
                });
            });
        });
    });
});