/*
 * It does not try to register in a CommonJS environment since jQuery is not
 * likely to run in those environments.
 */
(function(factory) {
    if (typeof define === 'function' && define.amd) {
    // AMD. Register as an anonymous module.
        define(['jquery', 'gridlist'], factory);
    } else {
        factory(jQuery, GridList);
    }
})(function($, GridList) {
    const DraggableGridList = function(element, options, draggableOptions, resizableOptions) {
        this._init(element, options, draggableOptions, resizableOptions);
    };

    // Minimum number of rows to add below the grid to allow dragging items to the bottom row when `addExtraGridSpaceBelow` is true
    const MIN_EXTRA_GRID_SPACE = 10;

    const DragAndDropManager = {
        draggedItem: null,
        targetGrid: null,
        targetSnapshot: null,
        sourceGrid: null,
        originalSize: null,
        dropZones: new Set()
    };

    DraggableGridList.prototype = {
        defaults: {
            lanes: 5,
            direction: 'horizontal',
            itemSelector: 'li[data-w]',
            itemIdAtribute: 'data-id',
            widthHeightRatio: 1,
            readOnly: false,
            dropZone: null,
            isChildGrid: false,
            itemSpacing: 0,
            cellSizeMatchCustomWidth: false,
            cellSizeMatchCustomHeight: false,
            numberOfCellsMatchingCustomHeight: false,
            numberOfCellsMatchingCustomWidth: false,
            addExtraGridSpaceBelow: true,
            autoStackUp: false,
            offsetHeight: 0,
            offsetWidth: 0
        },

        draggableDefaults: {
            zIndex: 3,
            scroll: false,
            containment: 'parent',
            disabled: false
        },

        resizableDefaults: {
            ghost: false,
            handles: 'all',
            disabled: false
        },

        setIsReadOnly: function(isReadOnly) {
            this.options.readOnly = isReadOnly;
            if (this.options.readOnly) {
                this._makeElementUndraggable(this.$items);
                this._makeElementUnresizable(this.$items);
            } else {
                const unlockedItems = this.$items.filter((_, element) => !this._getItemByElement(element).locked);
                this._makeElementDraggable(unlockedItems);
                this._makeElementResizable(unlockedItems);
            }
        },

        destroy: function() {
            DragAndDropManager.dropZones.delete(this.dropZone);
            this._removeEventListeners();
            this._safeDestroyPlugin(this.dropZone, 'droppable');
        },

        resize: function(lanes) {
            this._createGridSnapshot();
            this.gridList.resizeGrid(lanes);
            this.options.lanes = lanes;
            this._updateGridSnapshot();
            this.reflow();
        },

        toggleAutoStackUp: function(autoStackUp) {
            this._createGridSnapshot();
            this.gridList.toggleGapMode(!autoStackUp);
            this.options.autoStackUp = autoStackUp;
            this._updateGridSnapshot();
            this.reflow();
        },

        stackItemsUp: function() {
            this._createGridSnapshot();
            this.gridList.fillDirectionalGaps();
            this._updateGridSnapshot();
            this.reflow();
        },

        canStackItemsUp: function() {
            return this.gridList.hasDirectionalGaps();
        },

        captureItemSizeAndPositionSnapshot: function() {
            this.gridList.captureItemSizeAndPositionSnapshot();
        },

        positionPositionlessElements: function() {
            this._createGridSnapshot();
            this.gridList.positionPositionlessElements();
            this._updateGridSnapshot();

            this.reflow();
        },

        resizeItem: function(element, size) {
            /**
             * Resize an item.
             *
             * @param {Object} size
             * @param {Number} [size.w]
             * @param {Number} [size.h}
             */
            if (!this.options.readOnly) {
                this._createGridSnapshot();
                this.gridList.resizeItem(this._getItemByElement(element), size);
                this._updateGridSnapshot();
                this._updateWidestTallestItem();
                this.render();
            }
        },

        moveAndResizeItem: function(element, item) {
            /**
             * Move and resize an item.
             *
             * @param {Object} item
             * @param {Number} [item.x]
             * @param {Number} [item.y]
             * @param {Number} [item.w]
             * @param {Number} [item.h}
             */
            if (!this.options.readOnly) {
                this._createGridSnapshot();
                this.gridList.moveAndResizeItem(this._getItemByElement(element), item);
                this._updateGridSnapshot();
                this._updateWidestTallestItem();
                this.render();
            }
        },

        deleteItem: function(element) {
            if (!this.options.readOnly) {
                this._createGridSnapshot();
                const item = this._getItemByElement(element);
                this.gridList.deleteItem(item);
                this.$items = this.$items.not(element);
                this._updateGridSnapshot();
                this._updateWidestTallestItem();
                this.render();
            }
        },

        blockDragging: function(element) {
            if (!this.options.readOnly) {
                this._makeElementUndraggable(element);
            }
        },

        unblockDragging: function(element) {
            if (!this.options.readOnly) {
                this._makeElementDraggable(element);
            }
        },

        blockItemWhenHoveredByAnotherItem: function(element) {
            if (!this.options.readOnly) {
                const item = this._getItemByElement(element);
                item.lockedDirectMove = true;
                this._updateGridSnapshot();
            }
        },

        unblockItemWhenHoveredByAnotherItem: function(element) {
            if (!this.options.readOnly) {
                const item = this._getItemByElement(element);
                item.lockedDirectMove = false;
                this._updateGridSnapshot();
            }
        },

        lockItem: function(element) {
            if (!this.options.readOnly) {
                const item = this._getItemByElement(element);
                item.locked = true;
                this._updateGridSnapshot();
                this._makeElementUndraggable(element);
                this._makeElementUnresizable(element);
                this.render();
            }
        },

        unlockItem: function(element) {
            if (!this.options.readOnly) {
                const item = this._getItemByElement(element);
                item.locked = false;
                this._updateGridSnapshot();
                this._makeElementDraggable(element);
                this._makeElementResizable(element);
                this.render();
            }
        },

        _safeDestroyPlugin: function($els, plugin) {
            const dataKey = 'ui-' + plugin;
            $els.each(function() {
                const $el = $(this);
                if ($el.data(dataKey)) {
                    $el[plugin]('destroy');
                }
            });
        },

        _isItemOverlappingLockedPosition(item, newPosition) {
            const convertedNewPosition = this.gridList._getItemPosition(newPosition);
            return this.gridList.items.some(curr => curr.id !== item.id && curr.locked && this.gridList._arePositionsColliding(convertedNewPosition, this.gridList._getItemPosition(curr)));
        },

        _getElementRect: function(el) {
            const position = el.position(); // position relative to document
            return {
                x: position.left,
                y: position.top,
                w: el.outerWidth(),
                h: el.outerHeight()
            };
        },

        _mergeRects: function(rectA, rectB) {
            const x1 = Math.min(rectA.x, rectB.x);
            const y1 = Math.min(rectA.y, rectB.y);

            const x2 = Math.max(rectA.x + rectA.w, rectB.x + rectB.w);
            const y2 = Math.max(rectA.y + rectA.h, rectB.y + rectB.h);

            return {
                x: x1,
                y: y1,
                w: x2 - x1,
                h: y2 - y1
            };
        },

        _computeDirectionalOverlap: function(draggedItemStartRect, draggedItemCurrentRect, collidingItemRect) {
            // Create rectangle formed by the item's original position and its current position
            const dragRect = this._mergeRects(draggedItemStartRect, draggedItemCurrentRect);
            let yOver = Number.MAX_VALUE;
            let xOver = Number.MAX_VALUE;

            // Vertical overlap (top/bottom)
            if (draggedItemStartRect.y < collidingItemRect.y) { // from above
                yOver = ((dragRect.y + dragRect.h) - collidingItemRect.y) / collidingItemRect.h;
            } else if (draggedItemStartRect.y + draggedItemStartRect.h > collidingItemRect.y + collidingItemRect.h) { // from below
                yOver = ((collidingItemRect.y + collidingItemRect.h) - dragRect.y) / collidingItemRect.h;
            }

            // Horizontal overlap (left/right)
            if (draggedItemStartRect.x < collidingItemRect.x) { // from the left
                xOver = ((dragRect.x + dragRect.w) - collidingItemRect.x) / collidingItemRect.w;
            } else if (draggedItemStartRect.x + draggedItemStartRect.w > collidingItemRect.x + collidingItemRect.w) { // from the right
                xOver = ((collidingItemRect.x + collidingItemRect.w) - dragRect.x) / collidingItemRect.w;
            }

            return { xOver, yOver };
        },

        _computeOverlapThreshold: function(draggedItemCurrentRect, collidingItemRect, axis) {
            let sizeRatioOfDraggedItemOverCollidingOne;

            if (axis === 'x') {
                sizeRatioOfDraggedItemOverCollidingOne = draggedItemCurrentRect.w / collidingItemRect.w;
            } else if (axis === 'y') {
                sizeRatioOfDraggedItemOverCollidingOne = draggedItemCurrentRect.h / collidingItemRect.h;
            } else {
                throw new Error(`Invalid axis "${axis}" passed to computeAdaptiveCoverageThreshold`);
            }

            if (sizeRatioOfDraggedItemOverCollidingOne <= 0.25) {
                return 0.2;
            } else if (sizeRatioOfDraggedItemOverCollidingOne <= 0.5) {
                return 0.4;
            } else if (sizeRatioOfDraggedItemOverCollidingOne <= 1) {
                return 0.6;
            } else if (sizeRatioOfDraggedItemOverCollidingOne <= 1.5) {
                return 0.7;
            } else {
                return 0.8;
            }
        },

        _canMoveItemToPosition: function(item, newGridProsition, draggedItemStartRect) {
            const convertedNewPosition = this.gridList._getItemPosition(newGridProsition);
            const collidingItems = this.gridList.items.filter(curr => curr.id !== item.id && this.gridList._arePositionsColliding(convertedNewPosition, this.gridList._getItemPosition(curr)));

            if (collidingItems.length === 0) {
                return true;
            }

            let canMoveItemToPosition = false;
            for (const collidingItem of collidingItems) {
                if (collidingItem.locked || collidingItem.lockedDirectMove) {
                    return false;
                }

                const collidingItemRect = this._getElementRect(collidingItem.$element);
                const draggedItemCurrentRect = this._getElementRect(item.$element);
                const { xOver, yOver } = this._computeDirectionalOverlap(draggedItemStartRect, draggedItemCurrentRect, collidingItemRect);

                if (xOver < yOver) {
                    // Prefer horizontal movement
                    const coverageThreshold = this._computeOverlapThreshold(draggedItemCurrentRect, collidingItemRect, 'x');
                    if (xOver > coverageThreshold) {
                        canMoveItemToPosition = true;
                    }
                } else {
                    // Prefer vertical movement
                    const coverageThreshold = this._computeOverlapThreshold(draggedItemCurrentRect, collidingItemRect, 'y');
                    if (yOver > coverageThreshold) {
                        canMoveItemToPosition = true;
                    }
                }
            }

            return canMoveItemToPosition;
        },

        addItem: function(element) {
            if (!this.options.readOnly && !this._getItemByElement(element)) {
                this._createGridSnapshot();
                const item = this._generateItemFromElement(element);
                this.gridList.addItem(item);
                this.$items = this.$items.add(element);
                this._updateGridSnapshot();
                this.captureItemSizeAndPositionSnapshot();
                this._updateWidestTallestItem();
                if (!item.locked) {
                    this._makeElementDraggable(element);
                    this._makeElementResizable(element);
                }
                this.render();
            }
        },

        dragItemIn: function(draggedItem, targetItem, event) {
            if (this.options.readOnly || this._getItemByElement(draggedItem.$element)) {
                return;
            }
            this._maxGridCols = 9000;
            const coordinates = this._snapDraggedInItemPositionToGrid(draggedItem, targetItem);
            this.$element.append(draggedItem.$element);

            /**
             * Take a snapshot of the grid layout with all tiles at their original positions.
             * The newly added tile is placed at the end of the grid in this snapshot.
             * The snapshot will be used in the `_onDrag` method to try to restore tiles to their original positions during dragging.
             */
            const draggedItemWithNoPosition = {
                ...draggedItem,
                x: undefined,
                y: undefined
            };
            this.gridList.addItem(draggedItemWithNoPosition);
            this._createGridSnapshot();
            this.gridList.deleteItem(draggedItemWithNoPosition);

            // Now try to position the dragged item at the desired position moving the rest of the tiles
            const draggedItemTranslatedToGrid = {
                ...draggedItem,
                ...coordinates
            };
            this.gridList.addItem(draggedItemTranslatedToGrid);

            this.$items = this.$items.add(draggedItemWithNoPosition.$element);
            this._lastValidResizedItemPosition = {
                x: draggedItem.x,
                y: draggedItem.y,
                h: draggedItem.h,
                w: draggedItem.w
            };
            DragAndDropManager.draggedItem = draggedItemTranslatedToGrid;
            this._updateWidestTallestItem();
            this.gridList.generateGrid();
            this.render();
            this.captureItemSizeAndPositionSnapshot();

            /*
             * Ensure that the draggable's position is recalculated relative to the new container
             * without interrupting the current drag.
             */
            const draggableInstance = draggedItem.$element.data('ui-draggable');
            draggableInstance.offsetParent = this.$element;
            if (this.options.scrollingContainer != null) {
                draggableInstance.scrollParent = this.options.scrollingContainer;
                draggableInstance.scrollParentNotHidden = this.options.scrollingContainer;
            } else {
                draggableInstance.scrollParent = this.$element.scrollParent();
                draggableInstance.scrollParentNotHidden = draggableInstance.helper.scrollParent(false);
            }
            draggableInstance.overflowOffset = draggableInstance.scrollParentNotHidden.offset();
            draggableInstance._setContainment();
            draggableInstance._refreshOffsets(event);
            draggableInstance._cacheMargins();
        },

        dragItemOut: function(draggedItem, sourceSnapshot) {
            if (this.options.readOnly) {
                return;
            }
            // Need to retrieve the item again as snapshot updates perform a deep copies
            const item = this._getItemByElement(draggedItem.$element);
            this.gridList.deleteItem(item);
            this.$items = this.$items.not(item.$element);
            // Move back elements as they were except the dragged one
            GridList.cloneItems(sourceSnapshot, this.gridList.items);
            this._updateGridSnapshot();
            this.gridList.generateGrid();
            this._updateWidestTallestItem();
            this.render();
            this._removePositionHighlight();
            this.captureItemSizeAndPositionSnapshot();
        },

        reflow: function() {
            this._calculateCellSize();
            this.render();
        },

        render: function() {
            this._applySizeToItems();
            this._applyPositionToItems();
        },

        _bindMethod: function(fn) {
            /**
             * Bind prototype method to instance scope (similar to CoffeeScript's fat
             * arrow)
             */
            const that = this;
            return function() {
                return fn.apply(that, arguments);
            };
        },

        synchronizeWithDOM: function() {
            this._createGridSnapshot();
            this._removeEventListeners();
            this.$items = this.$element.children(this.options.itemSelector);
            this._synchronizeItemsWithDOM();
            if (!this.options.readOnly) {
                const unlockedItems = this.$items.filter((_, element) => {
                    const item = this._getItemByElement(element);
                    return item != null && !item.locked;
                });
                this._makeElementDraggable(unlockedItems);
                this._makeElementResizable(unlockedItems);
            }
            this._updateWidestTallestItem();
            this._updateGridSnapshot();
            this.render();
        },

        _synchronizeItemsWithDOM: function() {
            const itemById = new Map(this.gridList.items.map(item => [item.id, item]));
            this.$items.each((_, element) => {
                const $element = $(element);
                const id = $element.attr(this.options.itemIdAtribute);
                if (itemById.has(id)) {
                    const item = itemById.get(id);
                    item.$element = $element;
                } else {
                    const item = this._generateItemFromElement($element);
                    this.gridList.addItem(item);
                    this.captureItemSizeAndPositionSnapshot();
                    itemById.set(id, item);
                }
            });
        },

        _init: function(element, options, draggableOptions, resizableOptions) {
            this.options = $.extend({}, this.defaults, options);
            this.draggableOptions = $.extend({}, this.draggableDefaults, draggableOptions);
            this.resizableOptions = $.extend({}, this.resizableDefaults, resizableOptions);

            this.$element = $(element);
            /*
             * Read items and their meta data. Ignore other list elements (like the
             * position highlight)
             */
            this.$items = this.$element.children(this.options.itemSelector);
            const items = this._generateItemsFromDOM();
            this._initGridList(items);
            this._updateWidestTallestItem();

            // Used to highlight a position an element will land on upon drop
            this.$positionHighlight = this.$element.find('.position-highlight').hide();

            this.reflow();
            this._bindEvents();
            this._makeGridDroppable();
            if (!this.options.readOnly) {
                const unlockedItems = this.$items.filter((_, element) => {
                    const item = this._getItemByElement(element);
                    return item != null && !item.locked;
                });
                /*
                 * Init Draggable JQuery UI plugin for each of the list items
                 * http://api.jqueryui.com/draggable/
                 */
                this._makeElementDraggable(unlockedItems);
                this._makeElementResizable(unlockedItems);
            }
            this._lastValidResizedItemPosition = null;
            this.captureItemSizeAndPositionSnapshot();
            if (this.options.onLoad) {
                this.options.onLoad();
            }
        },

        _makeElementDraggable: function(element) {
            element.draggable(this.draggableOptions);
            this._addDraggingEventsToElement(element);

        },

        _makeElementUndraggable: function(element) {
            element.draggable({
                disabled: true
            });
            this._removeDraggingEventsFromElement(element);
        },

        _makeElementResizable: function(element) {
            element.resizable(this.resizableOptions);
            this._addResizeEventsToElement(element);
        },

        _makeElementUnresizable: function(element) {
            element.resizable({
                disabled: true
            });
            this._removeResizeEventsFromElement(element);
        },

        _getClosestDropZone: function(event) {
            let closest = null;
            let closestDepth = Infinity;

            DragAndDropManager.dropZones.forEach(dropZone => {
                const target = event.originalEvent.target;
                const dropZoneElement = dropZone.get(0);
                const depth = this._getDepthBetweenElements(target, dropZoneElement);
                if (depth < closestDepth) {
                    closest = dropZone;
                    closestDepth = depth;
                }
            });
            return closest;
        },

        _getDepthBetweenElements: function(child, ancestor) {
            let depth = 0;
            let current = child;
            while (current && current !== ancestor) {
                current = current.parentElement;
                depth++;
            }
            return depth;
        },

        _makeGridDroppable: function() {
            this.dropZone = this.options.dropZone != null ? this.options.dropZone : this.$element;
            DragAndDropManager.dropZones.add(this.dropZone);
            this.dropZone.droppable({
                accept: this.options.itemSelector,
                greedy: true,
                tolerance: 'pointer',
                out: ( event, ui ) => {
                    DragAndDropManager.sourceGrid = this;
                },
                over: ( event, ui ) =>{
                    if (DragAndDropManager.draggedItem != null
                            && DragAndDropManager.sourceGrid != null
                            && DragAndDropManager.sourceGrid != this
                            && DragAndDropManager.targetGrid != null
                            && DragAndDropManager.targetGrid !== this
                    ) {

                        /**
                         * Prevents the parent grid from receiving the tile when dragging a tile
                         * from one group tile to an adjacent group tile.
                         * This happens because the parent grid's `over` method is also triggered
                         * with a separate event object, making it impossible to stop.
                         */
                        if (DragAndDropManager.targetGrid.options.isChildGrid && this._getClosestDropZone(event) != this.dropZone) {
                            return;
                        }

                        const draggedItem = DragAndDropManager.draggedItem;
                        const sourceSnapshot = DragAndDropManager.targetSnapshot;
                        const sourceGrid = DragAndDropManager.targetGrid;
                        const targetGrid = this;

                        if (this.options.canDragItemOut != null) {
                            const canDragItemOut = sourceGrid.options.canDragItemOut(draggedItem, sourceGrid, targetGrid);
                            if (!canDragItemOut) {
                                return;
                            }
                        }

                        const targetSnapshot = GridList.cloneItems(targetGrid.gridList.items);

                        if (sourceGrid.options.onBeforeDragItemOut != null) {
                            sourceGrid.options.onBeforeDragItemOut(draggedItem, targetGrid);
                        }

                        if (targetGrid.options.readOnly) {
                            return;
                        }

                        sourceGrid.dragItemOut(draggedItem, sourceSnapshot);

                        if (sourceGrid.options.onAfterDragItemOut != null) {
                            sourceGrid.options.onAfterDragItemOut(draggedItem, targetGrid);
                        }

                        if (targetGrid.options.onBeforeDragItemIn != null) {
                            targetGrid.options.onBeforeDragItemIn(draggedItem, sourceGrid);
                        }

                        DragAndDropManager.targetGrid = targetGrid;
                        DragAndDropManager.targetSnapshot = targetSnapshot;

                        const targetItem = sourceGrid.gridList.items.find(item => item.$element.has(targetGrid.$element).length > 0);
                        const draggedItemWithOriginalSize = { ...draggedItem, ...DragAndDropManager.originalSize };
                        targetGrid.dragItemIn(draggedItemWithOriginalSize, targetItem, event);

                        if (targetGrid.options.onAfterDragItemIn != null) {
                            targetGrid.options.onAfterDragItemIn(draggedItem, sourceGrid);
                        }
                    }
                }
            });
        },

        _updateWidestTallestItem: function() {
            this._widestItem = Math.max.apply(
                null,
                this.gridList.items.map(function(item) {
                    return item.w;
                })
            );
            this._tallestItem = Math.max.apply(
                null,
                this.gridList.items.map(function(item) {
                    return item.h;
                })
            );
        },

        _initGridList: function(items) {
            /*
             * Create instance of GridList (decoupled lib for handling the grid
             * positioning and sorting post-drag and dropping)
             */
            this.gridList = new GridList(items, {
                lanes: this.options.lanes,
                direction: this.options.direction,
                allowDirectionalGaps: !this.options.autoStackUp
            });
        },

        _bindEvents: function() {
            this._onStart = this._bindMethod(this._onStart);
            this._onDrag = this._bindMethod(this._onDrag);
            this._onStop = this._bindMethod(this._onStop);
            this._onResize = this._bindMethod(this._onResize);
            this._onResizeStop = this._bindMethod(this._onResizeStop);
            this._onMouseEnter = this._bindMethod(this._onMouseEnter);
        },

        _addDraggingEventsToElement: function(element) {
            element.on('dragstart', this._onStart);
            element.on('drag', this._onDrag);
            element.on('dragstop', this._onStop);
        },

        _addResizeEventsToElement: function(element) {
            element.on('resizestart', this._onStart);
            element.on('resize', this._onResize);
            element.on('resizestop', this._onResizeStop);
        },

        _removeDraggingEventsFromElement: function(element) {
            element.off('dragstart', this._onStart);
            element.off('drag', this._onDrag);
            element.off('dragstop', this._onStop);
        },

        _removeResizeEventsFromElement: function(element) {
            element.off('resizestart', this._onStart);
            element.off('resize', this._onResize);
            element.off('resizestop', this._onResizeStop);
        },

        _removeEventListeners: function() {
            this._removeDraggingEventsFromElement(this.$items);
            this._removeResizeEventsFromElement(this.$items);
            this._safeDestroyPlugin(this.$items, 'draggable');
            this._safeDestroyPlugin(this.$items, 'resizable');
        },

        _onStart: function(event, ui) {
            const item = this._getItemByElement(ui.helper);
            DragAndDropManager.draggedItem = item;
            DragAndDropManager.targetGrid = this;
            DragAndDropManager.targetSnapshot = GridList.cloneItems(this.gridList.items).filter(curr => !ui.helper.is(curr.$element));
            DragAndDropManager.originalSize = { w: item.w, h: item.h };
            event.stopPropagation();

            /*
             * By default, jQuery UI Draggable uses `scrollParentNotHidden` to determine which parent element
             * should auto-scroll during a drag/resize. If our grid scrolling container has `overflow: hidden` (to hide the
             * scrollbar when not needed), jQuery might not detect it correctly and auto-scrolling could break.
             *
             * To ensure that auto-scrolling works even when the scrollbar is initially hidden (e.g. when
             * tiles are within bounds of a group tile), we explicitly set `scrollParentNotHidden` to our scrolling container.
             * This way, if the user drags a tile beyond the visible limit, scrolling still works as expected.
             */
            const draggableInstance = item.$element.data('ui-draggable');
            if (this.options.scrollingContainer != null && !this.options.scrollingContainer.is(draggableInstance.scrollParentNotHidden)) {
                draggableInstance.scrollParentNotHidden = this.options.scrollingContainer;
                draggableInstance.overflowOffset = draggableInstance.scrollParentNotHidden.offset();
            }
            /*
             * Create a deep copy of the items; we use them to revert the item
             * positions after each drag change, making an entire drag operation less
             * distructable
             */
            this._createGridSnapshot();

            /*
             * Since dragging actually alters the grid, we need to establish the number
             * of cols (+1 extra) before the drag starts
             */
            this._maxGridCols = 9000;

            // Need to save the current item size, so that when resizing it, if it encounters a locked element it keeps the last valid size and does not come back to the original size
            this._lastValidResizedItemPosition = {
                x: item.x,
                y: item.y,
                h: item.h,
                w: item.w
            };
            this._lastItemRect = this._getElementRect(item.$element);
        },

        _onDrag: function(event, ui) {
            if (DragAndDropManager.targetGrid != null && DragAndDropManager.targetGrid !== this) {
                DragAndDropManager.targetGrid._onDrag(event, ui);
                return;
            }

            let item = this._getItemByElement(ui.helper);
            const xyCoordinates = this._snapItemPositionToGrid(item);

            // When using containment = 'parent', jQuery only computes the containment box on dragstart, because our parent can change size during drag, we need to update this every time
            item.$element.data('ui-draggable')._setContainment();
            const newGridPosition = {
                x: xyCoordinates[0],
                y: xyCoordinates[1],
                w: item.w,
                h: item.h
            };
            if (this._dragPositionChanged(xyCoordinates) && this._canMoveItemToPosition(item, newGridPosition, this._lastItemRect)) {
                this._previousDragPosition = xyCoordinates;

                // Regenerate the grid with the positions from when the drag started
                GridList.cloneItems(this._items, this.gridList.items);
                this.gridList.generateGrid();

                this._lastItemRect = this._getElementRect(item.$element);

                /*
                 * Since the items list is a deep copy, we need to fetch the item
                 * corresponding to this drag action again
                 */
                item = this._getItemByElement(ui.helper);
                this.gridList.moveItemToPosition(item, xyCoordinates);

                // Visually update item positions
                this._applyPositionToItems();
                /*
                 * Collision solving algorithm might have change item reference when calling `moveItemToPosition`,
                 * and we need to get its updated position that might have been changed by packItemsOverGridDirection.
                 */
                item = this._getItemByElement(ui.helper);
                // Highlight shape
                this._highlightPositionForItem(item);
            }
        },

        _onStop: function(event, ui) {
            if (DragAndDropManager.targetGrid != null && DragAndDropManager.targetGrid !== this) {
                const sourceGrid = this;
                const targetGrid = DragAndDropManager.targetGrid;
                const draggedItem = DragAndDropManager.draggedItem;
                let dropOutData = null;
                if (sourceGrid.options.onDropItemOut != null) {
                    dropOutData = sourceGrid.options.onDropItemOut(draggedItem);
                }
                if (targetGrid.options.onDropItemIn != null) {
                    targetGrid.options.onDropItemIn(draggedItem, sourceGrid, dropOutData);
                }
                targetGrid._onStop(event, ui);
                return;
            }
            // Removing any rows at the bottom of the grid that are not used anymore
            this.gridList.generateGrid();

            this._updateGridSnapshot();
            this.captureItemSizeAndPositionSnapshot();
            this._previousDragPosition = null;
            this._lastValidResizedItemPosition = null;
            this._lastItemRect = null;

            DragAndDropManager.targetGrid = null;
            DragAndDropManager.draggedItem = null;
            DragAndDropManager.targetSnapshot = null;
            DragAndDropManager.sourceGrid = null;
            DragAndDropManager.originalSize = null;

            /*
             * HACK: jQuery.draggable removes this class after the dragstop callback,
             * and we need it removed before the drop, to re-enable CSS transitions
             */
            $(ui.helper).removeClass('ui-draggable-dragging');

            this._applyPositionToItems();
            this._removePositionHighlight();
        },

        _onResize: function(event, ui) {
            event.stopPropagation();

            // Regenerate the grid with the positions from when the resize started
            GridList.cloneItems(this._items, this.gridList.items);
            this.gridList.generateGrid();

            let w = Math.round((ui.size.width + this.options.itemSpacing) / this._cellWidth);
            let h = Math.round((ui.size.height + this.options.itemSpacing) / this._cellHeight);
            let x = Math.round(ui.position.left / this._cellWidth);
            let y = Math.round(ui.position.top / this._cellHeight);

            // Handle cases when the minimum size has been reached
            if (w <= 0) {
                w = 1;
                if (ui.position.left > ui.originalPosition.left) {
                    x = Math.floor(ui.position.left / this._cellWidth);
                }
            }

            if (h <= 0) {
                h = 1;
                if (ui.position.top > ui.originalPosition.top) {
                    y = Math.floor(ui.position.top / this._cellHeight);
                }
            }

            const overflow = x + w - this.options.lanes;
            if (overflow > 0) {
                x -= overflow;
                if (x < 0) {
                    w += x;
                }
            }
            const newPosition = {
                w: Math.min(w, this.options.lanes),
                h: h,
                x: Math.max(0, x),
                y: Math.max(0, y)
            };

            const item = this._getItemByElement(ui.element);
            if (!this._isItemOverlappingLockedPosition(item, newPosition)) {
                this.gridList.moveAndResizeItem(item, newPosition);
                this._lastValidResizedItemPosition = newPosition;
            } else {
                this.gridList.moveAndResizeItem(item, this._lastValidResizedItemPosition);
            }
            this._updateWidestTallestItem();
            this.render();
        },

        _onResizeStop: function(event, ui) {
            // Removing any rows at the bottom of the grid that are not used anymore
            this.gridList.generateGrid();

            this._onResize(event, ui);
            this._updateGridSnapshot();
            this.captureItemSizeAndPositionSnapshot();
            this._lastValidResizedItemPosition = null;
        },

        _generateItemsFromDOM: function() {
            /**
             * Generate the structure of items used by the GridList lib, using the DOM
             * data of the children of the targeted element. The items will have an
             * additional reference to the initial DOM element attached, in order to
             * trace back to it and re-render it once its properties are changed by the
             * GridList lib
             */

            const items = [];
            this.$items.each((_, element) => items.push(this._generateItemFromElement(element)));
            return items;
        },

        _generateItemFromElement: function(element) {
            return {
                $element: $(element),
                x: parseInt($(element).attr('data-x')),
                y: parseInt($(element).attr('data-y')),
                w: parseInt($(element).attr('data-w')),
                h: parseInt($(element).attr('data-h')),
                id: $(element).attr('data-id'),
                locked: ($(element).attr('data-locked')) === 'true',
                isDisplacing: ($(element).attr('data-is-displacing')) === 'true',
                minWidth: parseInt($(element).attr('data-min-width')),
                minHeight: parseInt($(element).attr('data-min-height'))
            };
        },

        _getItemByElement: function(element) {
            /*
             * XXX: this could be optimized by storing the item reference inside the
             * meta data of the DOM element
             */
            for (let i = 0; i < this.gridList.items.length; i++) {
                if (this.gridList.items[i].$element.is(element)) {
                    return this.gridList.items[i];
                }
            }
        },

        _calculateCellSize: function() {
            if (this.options.direction === 'horizontal') {
                this._cellHeight = Math.floor((this.$element.height() / this.options.lanes) * 100) / 100;
                this._cellWidth = this._cellHeight * this.options.widthHeightRatio;
            } else {
                this._cellWidth = Math.floor((this.$element.width() / this.options.lanes) * 100) / 100;
                this._cellHeight = this._cellWidth / this.options.widthHeightRatio;
            }
            if (this.options.heightToFontSizeRatio) {
                this._fontSize = this._cellHeight * this.options.heightToFontSizeRatio;
            }
            if (this.options.cellSizeMatchCustomWidth) {
                const cellNumber = this.options.numberOfCellsMatchingCustomWidth ?
                    this.options.numberOfCellsMatchingCustomWidth :
                    Math.round(this.options.cellSizeMatchCustomWidth / this._cellWidth);
                this._cellWidth = Math.floor((this.options.cellSizeMatchCustomWidth / cellNumber) * 100) / 100;
            }
            if (this.options.cellSizeMatchCustomHeight) {
                const cellNumber = this.options.numberOfCellsMatchingCustomHeight ?
                    this.options.numberOfCellsMatchingCustomHeight :
                    Math.round(this.options.cellSizeMatchCustomHeight / this._cellHeight);
                this._cellHeight = Math.floor((this.options.cellSizeMatchCustomHeight / cellNumber) * 100) / 100;
            }
            if (this.options.onCellSizeChange) {
                this.options.onCellSizeChange(this._cellWidth, this._cellHeight);
            }
        },

        _getItemWidth: function(item) {
            return item.w * this._cellWidth;
        },

        _getItemHeight: function(item) {
            return item.h * this._cellHeight;
        },

        _applySizeToItems: function() {
            for (let i = 0; i < this.gridList.items.length; i++) {
                this.gridList.items[i].$element.css({
                    width: this._getItemWidth(this.gridList.items[i]),
                    height: this._getItemHeight(this.gridList.items[i])
                });
            }
            if (this.options.heightToFontSizeRatio) {
                this.$items.css('font-size', this._fontSize);
            }
        },

        _applyPositionToItems: function() {
            // TODO: Implement group separators
            for (let i = 0; i < this.gridList.items.length; i++) {
                // Don't interfere with the positions of the dragged items
                if (this.gridList.items[i].move) {
                    continue;
                }
                this.gridList.items[i].$element.css({
                    left: this.gridList.items[i].x * this._cellWidth,
                    top: this.gridList.items[i].y * this._cellHeight
                });
            }
            /*
             * Update the width of the entire grid container with enough room on the
             * right to allow dragging items to the end of the grid.
             */
            if (this.options.direction === 'horizontal') {
                this.$element.width((this.gridList.grid.length + this.options.addExtraGridSpaceBelow ? Math.max(this._widestItem, MIN_EXTRA_GRID_SPACE) : 0) * this._cellWidth + this.options.offsetWidth);
            } else {
                this.$element.height(
                    (this.gridList.grid.length + (!this.options.readOnly && this.options.addExtraGridSpaceBelow ? Math.max(this._tallestItem, MIN_EXTRA_GRID_SPACE) : 0)) * this._cellHeight + this.options.offsetHeight
                );
            }
        },

        _dragPositionChanged: function(newPosition) {
            if (!this._previousDragPosition) {
                return true;
            }
            return newPosition[0] != this._previousDragPosition[0] || newPosition[1] != this._previousDragPosition[1];
        },

        _snapDraggedInItemPositionToGrid: function(item, targetItem) {
            let col;
            let row;
            let height = item.h;
            let width = item.w;

            const itemOffset = item.$element.offset();
            if (targetItem) {
                // Dragging into an item
                const targetItemOffset = targetItem.$element.offset();
                const gridPosition = this.$element.position();
                col = Math.round((itemOffset.left - targetItemOffset.left - gridPosition.left) / this._cellWidth);
                row = Math.round((itemOffset.top - targetItemOffset.top - gridPosition.top) / this._cellHeight);
            } else {
                // Dragging into main grid
                const gridOffset = this.$element.offset();
                col = Math.round((itemOffset.left - gridOffset.left) / this._cellWidth);
                row = Math.round((itemOffset.top - gridOffset.top) / this._cellHeight);
            }

            /*
             * Keep item position within the grid and don't let the item create more
             * than one extra column
             */
            col = Math.max(col, 0);
            row = Math.max(row, 0);

            if (this.options.direction === 'horizontal') {
                col = Math.min(col, this._maxGridCols);
                row = Math.min(row, this.options.lanes - item.h);
                height = Math.min(height, this.options.lanes);
            } else {
                col = Math.min(col, this.options.lanes - item.w);
                row = Math.min(row, this._maxGridCols);
                width = Math.min(width, this.options.lanes);
            }
            return { x: col, y: row, h: height, w: width };
        },

        _snapItemPositionToGrid: function(item) {
            const position = item.$element.position();

            position[0] -= this.$element.position().left;

            let col = Math.round(position.left / this._cellWidth),
                row = Math.round(position.top / this._cellHeight);

            /*
             * Keep item position within the grid and don't let the item create more
             * than one extra column
             */
            col = Math.max(col, 0);
            row = Math.max(row, 0);

            if (this.options.direction === 'horizontal') {
                col = Math.min(col, this._maxGridCols);
                row = Math.min(row, this.options.lanes - item.h);
            } else {
                col = Math.min(col, this.options.lanes - item.w);
                row = Math.min(row, this._maxGridCols);
            }

            return [col, row];
        },

        _highlightPositionForItem: function(item) {
            this.$positionHighlight
                .css({
                    width: this._getItemWidth(item),
                    height: this._getItemHeight(item),
                    left: item.x * this._cellWidth,
                    top: item.y * this._cellHeight
                })
                .show();
            if (this.options.heightToFontSizeRatio) {
                this.$positionHighlight.css('font-size', this._fontSize);
            }
        },

        _removePositionHighlight: function() {
            this.$positionHighlight.hide();
        },

        _createGridSnapshot: function() {
            this._items = GridList.cloneItems(this.gridList.items);
        },

        _updateGridSnapshot: function() {
            // Notify the user with the items that changed since the previous snapshot
            this._triggerOnChange();
            GridList.cloneItems(this.gridList.items, this._items);
        },

        _triggerOnChange: function() {
            if (typeof this.options.onChange != 'function') {
                return;
            }
            this.options.onChange.call(this, this.gridList.getChangedItems(this._items, '$element'));
        }
    };

    $.fn.gridList = function(options, draggableOptions, resizableOptions) {
        let instance, method, args;
        if (typeof options == 'string') {
            method = options;
            args = Array.prototype.slice.call(arguments, 1);
        }
        this.each(function() {
            instance = $(this).data('_gridList');
            /*
             * The plugin call be called with no method on an existing GridList
             * instance to re-initialize it
             */
            if (instance && !method) {
                instance.destroy();
                instance = null;
            }
            if (!instance) {
                instance = new DraggableGridList(this, options, draggableOptions, resizableOptions);
                $(this).data('_gridList', instance);
            }
            if (method) {
                instance[method].apply(instance, args);
            }
        });
        return instance;
    };

    $.fn.getGridListInstance = function() {
        return this.first().data('_gridList');
    };
});
