import logging

from dataiku.doctor.individual_explainer import IndividualExplainer
from dataiku.core.saved_model import BasePredictor

logger = logging.getLogger(__name__)


class ExternalModelIndividualExplainer(IndividualExplainer):
    """Designed to replace the classical Individual Explainer mainly to change the evaluation
    dataset.

    It replaces the train set used for explanations by the evaluation dataset (which is
    the only one available for external models).
    """

    def __init__(self, predictor, model_folder_context, per_feature, prediction_type, train_split_desc=None):
        """
        :type predictor: ExternalModelPredictor
        :type model_folder_context: dataiku.base.folder_context.FolderContext
        :type per_feature: dict
        :type prediction_type: str
        """
        # Training params are unknown for external models, but the kwarg is required
        if train_split_desc is None:
            train_split_desc = {"params": {}}
        else:
            train_split_desc["params"] = {}
        super(ExternalModelIndividualExplainer, self).__init__(
            predictor=predictor,
            model_folder_context=model_folder_context,
            preprocessing_folder_context=model_folder_context,
            split_folder_context=model_folder_context,
            train_split_desc=train_split_desc,
            per_feature=per_feature,
            is_ensemble=False,
            prediction_type=prediction_type,
            sample_weight_col=None
        )
        self._model_folder_context = model_folder_context

    def _get_train_set(self):
        """Method to retrieve the evaluation dataset in external models.

        It is likely to me the same as the one in model_information_handler
        as for classical DSS models.
        """
        raise NotImplementedError

    def _predict_and_get_transformed_df(self, observations_df):
        """No transformation for external models"""
        return None, self._predictor.predict(observations_df)

    def _get_features_to_column_indices_mapping_or_none(self, all_columns):
        logger.info("External model does not support features to column indices")
        return None


class ExternalModelPredictor(BasePredictor):

    """Base class for an external model to be usable with an PredictionExternalModelInformationHandler
    and enable partial depency analysis, subpopulation analysis and individual explanations.

    An example can be found in dataiku.external_ml.mlflow.predictor
    """

    def __init__(self, model_folder_context):
        self.params = self.load_params(model_folder_context)

        self._prediction_type = self.params.core_params["prediction_type"]
        self._individual_explainer = self.get_individual_explainer_class()(
            predictor=self,
            model_folder_context=self.params.model_folder_context,
            per_feature=self.params.preprocessing_params["per_feature"],
            prediction_type=self._prediction_type,
            train_split_desc=self.params.split_desc,
        )

        self.features = [feature["name"] for feature in self.params.model_meta["features"]]
        self.classes = [label["label"] for label in self.params.model_meta.get("classLabels", [])]
        self._model = None

    def get_individual_explainer_class(self):
        raise NotImplementedError

    @property
    def model(self):
        if self._model is None:
            self._model = self.load_model()
        return self._model

    def load_params(self, model_folder_context):
        """Should return a dataiku.core.saved_model.ModelParams"""
        raise NotImplementedError

    def load_model(self):
        raise NotImplementedError

    def predict(self, df, with_probas=True):
        """Predict scores and probabilities on a pandas.DataFrame

        The output should contain "prediction" and "proba_0", "proba_1", etc. when possible.
        """
        raise NotImplementedError
